/* **********************************************************
 * Copyright (c) 2022, 2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.io.IOException;
import java.util.Collection;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallback;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallbackFactory;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallbackParams;
import com.vmware.vapi.internal.protocol.common.http.impl.ChunkedTransferEncodingFrameDeserializer;
import com.vmware.vapi.internal.tracing.TracingSpan;
import com.vmware.vapi.internal.util.TracingUtil;
import com.vmware.vapi.protocol.common.http.HttpConstants;

/**
 * The main response consumer. Depending on the Content-Type HTTP header, it
 * delegates either to {@link SingleResponseConsumer} or to
 * {@link MultiResponseConsumer}.
 */
public class NioMainResponseConsumer extends NioDecoratorConsumer {

    private final ResponseCallbackFactory cbFactory;
    private final Collection<String> acceptTypes;
    private final AbortHandle abortHandle;
    private final ExecutionContext ctx;
    private final HttpContext httpContext;
    private final TracingSpan tracingSpan;
    private final int maxResponseSize;

    private static Logger logger = LoggerFactory
            .getLogger(NioMainResponseConsumer.class);

    public NioMainResponseConsumer(ResponseCallbackFactory cbFactory,
                                   ExecutionContext ctx,
                                   HttpContext httpCtx,
                                   Collection<String> acceptTypes,
                                   AbortHandle abortHandle,
                                   TracingSpan tracingSpan,
                                   int maxResponseSize) {
        this.cbFactory = cbFactory;
        this.ctx = ctx;
        this.acceptTypes = acceptTypes;
        this.abortHandle = abortHandle;
        this.httpContext = httpCtx;
        this.tracingSpan = tracingSpan;
        this.maxResponseSize = maxResponseSize;
    }

    @Override
    public void responseReceived(HttpResponse response)
            throws IOException, HttpException {
        TracingUtil.registerResponseIntoSpan(tracingSpan, response, httpContext);

        String contentType = getContentType(response);

        ResponseCallbackParams params = new ResponseCallbackParams(contentType);
        ResponseCallback cb = cbFactory.createResponseCallback(params);

        switch (contentType) {
        // TODO case HttpConstants.CONTENT_TYPE_CLEAN_STREAM_JSON:
        case HttpConstants.CONTENT_TYPE_STREAM_JSON:
            decorated = new NioStreamingResponseConsumer(new ChunkedTransferEncodingFrameDeserializer(maxResponseSize),
                                                         cb,
                                                         abortHandle,
                                                         ctx);
            break;
        case HttpConstants.CONTENT_TYPE_FRAMED:
            decorated = new NioMultiResponseConsumer(new ChunkedTransferEncodingFrameDeserializer(maxResponseSize),
                                                     cb);
            break;
        // TODO this is to be addressed when adding support for streaming with
        // clean JSON; throw error in the default case & enable content types
        // below
        // case HttpConstants.CONTENT_TYPE_JSON:
        // case HttpConstants.CONTENT_TYPE_CLEAN_JSON:
        // decorated = new NioSingleResponseConsumer(cb,
        // acceptTypes,
        // abortHandle);
        // break;
        default:
            decorated = new NioSingleResponseConsumer(cb,
                                                      acceptTypes,
                                                      abortHandle,
                                                      maxResponseSize);
        }

        decorated.responseReceived(response);
    }

    /**
     * Retrieves content type header value from the response. If the value is
     * not apparent an empty string is returned.
     *
     * @param response Http response
     * @return content type string
     */
    private String getContentType(HttpResponse response) {
        Header ct = response.getFirstHeader(HttpConstants.HEADER_CONTENT_TYPE);
        String contentType = "";
        if (ct != null) {
            contentType = ct.getValue();
        }
        logger.debug("Receiving HTTP response with content-type '"
                     + contentType + "'");

        return contentType;
    }
}