/* **********************************************************
 * Copyright (c) 2020, 2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.util.Collection;

import org.apache.http.HttpResponse;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallback;
import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheHttpUtil;
import com.vmware.vapi.internal.protocol.common.Util;
import com.vmware.vapi.internal.protocol.common.http.ApacheHttpClientExceptionTranslator;

/**
 * Consumer which expects the response to contain a single frame. Decorates the
 * default {@link CappedAsyncResponseConsumer} and intercepts
 * {@link #responseCompleted(HttpContext)}.
 */
public class NioSingleResponseConsumer extends NioDecoratorConsumer {

    private final ResponseCallback callback;
    private final Collection<String> acceptTypes;
    private final AbortHandle abortHandle;

    private static Logger logger = LoggerFactory
            .getLogger(NioSingleResponseConsumer.class);

    @Deprecated
    public NioSingleResponseConsumer(ResponseCallback callback,
                                     Collection<String> acceptTypes,
                                     AbortHandle abortHandle) {
        this(callback, acceptTypes, abortHandle, Integer.MAX_VALUE);
    }

    public NioSingleResponseConsumer(ResponseCallback callback,
                                     Collection<String> acceptTypes,
                                     AbortHandle abortHandle,
                                     int maxResponseSize) {
        this.callback = callback;
        this.decorated = new CappedAsyncResponseConsumer(maxResponseSize);
        this.acceptTypes = acceptTypes;
        this.abortHandle = abortHandle;
    }

    @Override
    public void responseCompleted(HttpContext context) {
        super.responseCompleted(context);
        HttpResponse httpResponse = super.getResult();

        if (Util.checkRequestAborted(abortHandle, callback)) {
            return;
        }

        try {
            ApacheHttpUtil.validateHttpResponse(httpResponse, acceptTypes);
            // TODO: Do we need TransportControl?
            callback.received(httpResponse.getEntity().getContent(),
                              CorrelatingClient.NO_OP_CONTROL);
        } catch (Exception ex) {
            logger.debug("Problem with HTTP response", ex);
            callback.failed(ApacheHttpClientExceptionTranslator
                    .translate(ex, abortHandle));
        }
    }
}
