/* **********************************************************
 * Copyright (c) 2014-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common;

import java.util.Collection;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

import org.apache.http.client.methods.HttpPost;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.ApplicationData;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.core.abort.AbortListener;
import com.vmware.vapi.internal.core.abort.RequestAbortedException;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallback;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallbackFactory;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.TransportControl;
import com.vmware.vapi.internal.protocol.client.rpc.http.Utils;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration.Protocol;
import com.vmware.vapi.protocol.common.http.HttpConstants;
import com.vmware.vapi.security.SessionSecurityContext;;

/**
 * Utility class for common protocol related utils.
 */
public class Util {
    private static final String COMMA = ",";
    private static Logger logger = LoggerFactory.getLogger(Util.class);

    /**
     * Check if request is aborted and if so set the {@link ResponseCallback}
     * result to error containing information about that.
     *
     * @param cb          the callback to handle the result if the request is
     *                    aborted; must not be null.
     * @param abortHandle the abort handle to be checked; must not be null.
     * @return            true iff the request is aborted, false otherwise.
     */
    public static boolean checkRequestAborted(AbortHandle abortHandle,
                                              ResponseCallback cb) {
        if (abortHandle != null && abortHandle.isAborted()) {
            cb.failed(new RequestAbortedException("Request aborted, "
                + "aborting execution."));
            return true;
        }

        return false;
    }

    /**
     * Check if request is aborted and if so set the {@link ResponseCallbackFactory}
     * result to error containing information about that.
     *
     * @param cbFactory          the factory to handle the result if the request is
     *                           aborted; must not be null.
     * @param abortHandle        the abort handle to be checked; must not be null.
     * @return                   true iff the request is aborted, false otherwise.
     */
    public static boolean checkRequestAborted(AbortHandle abortHandle,
                                              ResponseCallbackFactory cbFactory) {
        if (abortHandle != null && abortHandle.isAborted()) {
            cbFactory
                    .failed(new RequestAbortedException("Request aborted, "
                                                        + "aborting execution."));
            return true;
        }

        return false;
    }


    /**
     * Registers abort listener for the specified {@link HttpPost}. The Post
     * will be aborted (by invoking {@link HttpPost#abort()} if the request is
     * aborted.
     *
     * @param cbFactory   the response callback factory; must not be null.
     * @param post        the http post; must not be null.
     * @param abortHandle the abort handle; can be null.
     */
    public static void registerAbortListerner(final ResponseCallbackFactory cbFactory,
                                              final HttpPost post,
                                              AbortHandle abortHandle) {
        Validate.notNull(cbFactory);
        Validate.notNull(post);
        if (abortHandle == null) {
            return;
        }

        abortHandle.addAbortListener(new AbortListener() {
            @Override
            public void onAbort() {
                if (logger.isDebugEnabled()) {
                    logger.debug("Request aborted, aborting HTTP post "
                        + "execution.");
                }
                cbFactory.failed(new RequestAbortedException("Http request aborted."));
                post.abort();
            }
        });
    }

    /**
     * Registers abort listener for the specified
     * {@link CorrelatingClient.TransportControl}. The streaming will be
     * canceled {@link CorrelatingClient.TransportControl#cancel()} on the abort
     * event.
     */
    public static void registerStreamingAbortListerner(final TransportControl control,
                                                       AbortHandle abortHandle) {
        Validate.notNull(control);
        if (abortHandle == null) {
            return;
        }

        abortHandle.addAbortListener(new AbortListener() {
            @Override
            public void onAbort() {
                if (logger.isDebugEnabled()) {
                    logger.debug("Request aborted, aborting streaming.");
                }
                control.cancel();
            }
        });

        // Used for avoiding racing conditions - when abort is invoked between
        // the abort check and listener registration.
        if (abortHandle.isAborted()) {
            if (logger.isDebugEnabled()) {
                logger.debug("Request aborted, aborting streaming.");
            }
            control.cancel();
        }
    }

    /**
     * Utility method that adds HTTP headers used by the JSON-RPC protocol.
     *
     * @param post the http post; must not be {@code null}.
     * @param requestContentType MIME type of the request
     * @param acceptedTypes accepted MIME types
     * @param requestContentType; content type of the request.
     * @param protocol; protocol to send the request.
     * @param serviceId service identifier; must not be {@code null}.
     * @param operationId operation identifier; must not be {@code null}.
     * @param executionContext carries information auxiliary to the request
     *        which might not be represented on the wire; must not be {@code null}.
     */
    public static void addHeaders(final HttpPost post,
                                  String requestContentType,
                                  Collection<String> acceptedTypes,
                                  final Protocol protocol,
                                  String serviceId,
                                  String operationId,
                                  ExecutionContext executionContext) {
        Objects.requireNonNull(post);
        Objects.requireNonNull(requestContentType);
        Objects.requireNonNull(acceptedTypes);
        Objects.requireNonNull(serviceId);
        Objects.requireNonNull(operationId);
        Objects.requireNonNull(executionContext);
        addJsonRpcHeaders(post, requestContentType, acceptedTypes);
        if (protocol == Protocol.VAPI_JSON_RPC_1_1) {
            addJsonRpc11ExtraHeaders(post,
                                     serviceId,
                                     operationId,
                                     executionContext);
        }
    }

    /**
     * Utility method that adds HTTP headers used by vAPI JSON-RPC 1.0
     * (over HTTP) protocol.
     */
    private static void addJsonRpcHeaders(final HttpPost post,
                                          String requestContentType,
                                          Collection<String> acceptedTypes) {
        post.setHeader(HttpConstants.HEADER_CONTENT_TYPE, requestContentType);
        post.setHeader(HttpConstants.HEADER_ACCEPT,
                       Utils.join(acceptedTypes, COMMA));
    }

    /**
     * Utility method used in vAPI JSON-RPC 1.1 (over HTTP) protocol, where
     * target service and operation, authentication and other information is
     * added in HTTP headers
     *
     * @param post the http post; must not be null.
     * @param serviceId service identifier; must not be {@code null}.
     * @param operationId operation identifier; must not be {@code null}.
     * @param executionContext carries information auxiliary to the request
     *        which might not be represented on the wire; must not be {@code null}.
     */
    private static void addJsonRpc11ExtraHeaders(final HttpPost post,
                                                 String serviceId,
                                                 String operationId,
                                                 ExecutionContext executionContext) {
        post.setHeader(HttpConstants.HEADER_SERVICE_ID, serviceId);
        post.setHeader(HttpConstants.HEADER_OPERATION_ID, operationId);
        ApplicationData applicationData = executionContext
                .retrieveApplicationData();
        if (applicationData != null) {
            Map<String, String> wireData = applicationData.getAllProperties();
            for (Map.Entry<String, String> appEntry : wireData.entrySet()) {
                String value = appEntry.getValue();
                if (value == null) {
                    continue;
                }
                String headerName = appEntry.getKey();
                if (!ExecutionContext.ApplicationData.ACCEPT_LANGUAGE_KEY.equals(headerName)) {
                    headerName = HttpConstants.HEADER_APPLICATION_CONTEXT_PREFIX
                                  + headerName.toLowerCase(Locale.ENGLISH);
                }
                post.setHeader(headerName, value);
            }
        }
        SecurityContext securityContext = executionContext
                .retrieveSecurityContext();
        if (securityContext != null) {
            Object sessionId = securityContext
                    .getProperty(SessionSecurityContext.SESSION_ID_KEY);
            if (sessionId != null) {
                post.setHeader(HttpConstants.HEADER_SESSION_ID,
                               String.valueOf((char[]) sessionId));
            }
        }
    }
}