/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.http.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;

import com.vmware.vapi.internal.protocol.common.http.BinaryInput;
import com.vmware.vapi.internal.util.Validate;

/**
 * A deserializer which reads a single length field. The length field is encoded
 * as an ASCII string of hex digits which ends with a CRLF.
 * <p>
 * <i>Thread-safety:</i> not thread-safe.
 * </p>
 */
public final class LengthFieldDeserializer {

    /**
     * ASCII charset.
     */
    private static final Charset ASCII = Charset.forName("US-ASCII");

    /**
     * Hex radix as a number.
     */
    private static final int HEX_RADIX = 16;

    private final int maxAllowedHexDigits;

    private final ByteArrayOutputStream hexDigits;

    private boolean done;

    /**
     * Whether '\r' has been read for the current length field.
     */
    private boolean expectLf;

    public LengthFieldDeserializer(int maxAllowedHexDigits) {
        Validate.isTrue(maxAllowedHexDigits > 0);
        this.maxAllowedHexDigits = maxAllowedHexDigits;
        this.hexDigits = new ByteArrayOutputStream(maxAllowedHexDigits);
        this.done = false;
    }

    /**
     * Reads a length field from an input handle. Should not call this
     * method after it returns a non-null value, because the deserializer can be
     * used only for a single field.
     *
     * @param input handle to an input source
     * @return the whole field or <code>null</code> if there is not enough data
     *         to construct a whole field
     * @throws IOException I/O error or format error
     * @throws IllegalStateException if an attempt is made to use the
     *             deserializer to read multiple fields
     */
    public Integer read(BinaryInput input) throws IOException {
        if (done) {
            throw new IllegalStateException(
                    "Field already extracted. The deserializer can be used only for a single field.");
        }
        Integer result = null;
        while (result == null) {
            int b = input.read();

            if (b == BinaryInput.NO_DATA_TEMPORARY) {
                break;
            }

            if (b == BinaryInput.NO_DATA_PERMANENT) {
                if (hexDigits.size() == 0) {
                    // no more frames
                    done = true;
                    break;
                } else {
                    throw new IOException("Unexpected end of stream while reading frame length field.");
                }
            }

            if (expectLf) {
                assertLf(b);
                expectLf = false;
                result = parseHex(hexDigits.toByteArray());
                done = true;
            } else if (b == '\r') {
                if (hexDigits.size() > 0) {
                    expectLf = true;
                } else {
                    throw new IOException("Length field contains no digits.");
                }
            } else {
                assertHexDigit(b);
                hexDigits.write(b);
            }
        }
        return result;
    }

    private static void assertLf(int b) throws IOException {
        if (b != '\n') {
            throw new IOException("Expected line-feed ASCII character. Found " + b);
        }
    }

    private void assertHexDigit(int b) throws IOException {
        if (!isHexDigit((char) b)) {
            throw new IOException(
                    "Encountered invalid hex digit while reading frame length field: "
                            + b);
        }
        if (hexDigits.size() > maxAllowedHexDigits) {
            throw new IOException(
                    "The length field is too large. It has more than "
                            + maxAllowedHexDigits + " hex digits.");
        }
    }

    private static Integer parseHex(byte[] asciiHex) throws IOException {
        String hexString = new String(asciiHex, ASCII);
        Integer result = Integer.parseInt(hexString, HEX_RADIX);
        return result;
    }

    private static boolean isHexDigit(char c) {
        if ('0' <= c && c <= '9') {
            return true;
        }
        if ('a' <= c && c <= 'f') {
            return true;
        }
        if ('A' <= c && c <= 'F') {
            return true;
        }
        return false;
    }

}
