/* **********************************************************
 * Copyright 2013 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * AsyncHandleImpl.java --
 *
 *      AsyncHandle implementation for introspection services.
 */

package com.vmware.vapi.internal.provider.introspection;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ErrorValueException;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;

/**
 * <code>AsyncHandle</code> implementation for introspection services.
 */
abstract class IntrospectionAsyncHandle<T> extends AsyncHandle<T> {

    private static final Logger logger =
            LoggerFactory.getLogger(IntrospectionAsyncHandle.class);

    private final AsyncHandle<MethodResult> resultHandle;

    public IntrospectionAsyncHandle(AsyncHandle<MethodResult> resultHandle) {
        this.resultHandle = resultHandle;
    }

    /**
     * Converts the result of the handle from binding object to value.
     *
     * @param result result of the handle as binding object
     * @return result of the handle as value
     * @throws IntrospectionException format error coming from
     *          {@link IntrospectionDataFactory}
     * @throws ErrorValueException error from the introspection service
     *          contract (see the VMODL definition of each service)
     */
    protected abstract DataValue convert(T result);

    @Override
    public void updateProgress(DataValue progress) {
        resultHandle.updateProgress(progress);
    }

    @Override
    public void setResult(T result) {
        DataValue converted;
        try {
            converted = convert(result);
        } catch (ErrorValueException ex) {
            resultHandle.setResult(
                    MethodResult.newErrorResult(ex.getErrorValue()));
            return;
        } catch (RuntimeException ex) {
            logger.error(
                    "Could not serialize the result of an introspection operation",
                    ex);
            resultHandle.setError(ex);
            return;
        }
        if (converted == null) {
            // do nothing
            return;
        }
        resultHandle.setResult(MethodResult.newResult(converted));
    }

    @Override
    public void setError(RuntimeException error) {
        if (error instanceof ErrorValueException) {
            ErrorValueException ex = (ErrorValueException) error;
            resultHandle.setResult(MethodResult.newErrorResult(ex.getErrorValue()));
        } else {
            resultHandle.setError(error);
        }
    }
}
