/* **********************************************************
 * Copyright 2013-2014, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.provider.introspection;

import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.NOT_FOUND_ERROR_INFO_DEF;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_GET_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_GET_METHOD_OPERATION_ID_PARAM;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_GET_METHOD_SERVICE_ID_PARAM;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_INFO_DEF;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_LIST_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_LIST_METHOD_SERVICE_ID_PARAM;
import static com.vmware.vapi.provider.introspection.IntrospectionConstants.OPERATION_INTROSPECTION_SERVICE_ID;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.bindings.server.InvocationContext;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ErrorValueException;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.InterfaceDefinition;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodDefinition;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ListDefinition;
import com.vmware.vapi.data.StringDefinition;
import com.vmware.vapi.data.StructDefinition;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.provider.ApiMethodBasedApiInterface;
import com.vmware.vapi.provider.introspection.ApiIntrospection;
import com.vmware.vapi.provider.introspection.SyncApiIntrospection;
import com.vmware.vapi.std.BuiltInDataFactory;
import com.vmware.vapi.std.StandardDataFactory;

/**
 * This service provides operations to retrieve information about the
 * operations exposed by a vAPI Provider.
 */
public class OperationIntrospectionService extends ApiMethodBasedApiInterface {

    public OperationIntrospectionService(ApiIntrospection introspection) {
        super(OPERATION_INTROSPECTION_SERVICE_ID);
        Validate.notNull(introspection);
        registerMethod(new GetApiMethod(introspection));
        registerMethod(new ListApiMethod(introspection));
    }

    public OperationIntrospectionService(SyncApiIntrospection introspection) {
        this(new SyncToAsyncApiIntrospectionAdapter(introspection));
    }

    private static class GetApiMethod extends ApiIntrospectionMethodBase {
        private final static DataDefinition INPUT_DEF;
        private final static DataDefinition OUTPUT_DEF;

        static {
            Map<String, DataDefinition> fields =
                new HashMap<String, DataDefinition>();
            fields.put(OPERATION_GET_METHOD_SERVICE_ID_PARAM,
                    StringDefinition.getInstance());
            fields.put(OPERATION_GET_METHOD_OPERATION_ID_PARAM,
                    StringDefinition.getInstance());
            INPUT_DEF = new StructDefinition(
                    BuiltInDataFactory.OPERATION_INPUT_STRUCT_NAME,
                    fields);

            OUTPUT_DEF = OPERATION_INFO_DEF;
        }

        public GetApiMethod(ApiIntrospection introspection) {
            super(OPERATION_GET_METHOD_ID,
                  INPUT_DEF,
                  OUTPUT_DEF,
                  Collections.singleton(NOT_FOUND_ERROR_INFO_DEF),
                  introspection);
        }

        @Override
        public void invoke(InvocationContext invocationContext,
                           DataValue input,
                           AsyncHandle<MethodResult> asyncHandle) {
            final ExecutionContext executionContext =
                    invocationContext.getExecutionContext();
            StructValue inputStruct = (StructValue) input;
            final String ifaceId =
                    inputStruct.getString(OPERATION_GET_METHOD_SERVICE_ID_PARAM);
            final String methodId =
                    inputStruct.getString(OPERATION_GET_METHOD_OPERATION_ID_PARAM);
            final InterfaceIdentifier serviceId = new InterfaceIdentifier(ifaceId);
            MethodIdentifier method = new MethodIdentifier(serviceId, methodId);

            final AsyncHandle<InterfaceDefinition> ifaceCb;
            ifaceCb = new IntrospectionAsyncHandle<InterfaceDefinition>(
                    asyncHandle) {
                @Override
                public void updateProgress(DataValue progress) {
                    /*
                     * swallow progress updates, because this is the second
                     * invoke (i.e. getInterface()) and we might have already
                     * reported progress on the result handle for the first
                     * invoke (i.e. getMethod())
                     */
                }

                @Override
                protected DataValue convert(InterfaceDefinition result) {
                    if (result == null) {
                        // interface not available - report NotFound for service
                        Message errorMsg = MessageFactory
                                .getMessage(
                                        "vapi.introspection.operation.service.not_found",
                                        ifaceId);

                        throw new ErrorValueException(
                                StandardDataFactory.createErrorValueForMessages(
                                        NOT_FOUND_ERROR_INFO_DEF,
                                        Arrays.asList(errorMsg)));
                    } else {
                        // method not available - report NotFound for operation
                        Message errorMsg = MessageFactory.getMessage(
                                "vapi.introspection.operation.not_found",
                                methodId, ifaceId);

                        throw new ErrorValueException(
                                StandardDataFactory.createErrorValueForMessages(
                                        NOT_FOUND_ERROR_INFO_DEF,
                                        Arrays.asList(errorMsg)));
                    }
                }
            };

            AsyncHandle<MethodDefinition> methodCb;
            methodCb = new IntrospectionAsyncHandle<MethodDefinition>(
                    asyncHandle) {
                @Override
                protected DataValue convert(MethodDefinition result) {
                    if (result != null) {
                        return IntrospectionDataFactory.toOperationInfo(result);
                    } else {
                        /*
                         * could not find definition for method - determine what
                         * exactly is the reason - is the whole service name
                         * invalid or just the method name
                         */
                        introspection.getInterface(
                                executionContext, serviceId, ifaceCb);
                        /*
                         * return null from convert() to avoid setResult() here;
                         * setResult() would happen in the getInterface() call
                         * above
                         */
                        return null;
                    }
                }
            };

            introspection.getMethod(executionContext, method, methodCb);
        }
    }

    private static class ListApiMethod extends ApiIntrospectionMethodBase {
        private final static DataDefinition INPUT_DEF;
        private final static DataDefinition OUTPUT_DEF;

        static {
            INPUT_DEF = new StructDefinition(
                    BuiltInDataFactory.OPERATION_INPUT_STRUCT_NAME,
                    Collections.singletonMap(
                            OPERATION_LIST_METHOD_SERVICE_ID_PARAM,
                            (DataDefinition) StringDefinition.getInstance()));
            OUTPUT_DEF = new ListDefinition(StringDefinition.getInstance());
        }

        public ListApiMethod(ApiIntrospection introspection) {
            super(OPERATION_LIST_METHOD_ID,
                  INPUT_DEF,
                  OUTPUT_DEF,
                  Collections.singleton(NOT_FOUND_ERROR_INFO_DEF),
                  introspection);
        }

        @Override
        public void invoke(InvocationContext invocationContext,
                           DataValue input,
                           AsyncHandle<MethodResult> asyncHandle) {
            StructValue inputStruct = (StructValue) input;
            final String ifaceId =
                    inputStruct.getString(OPERATION_LIST_METHOD_SERVICE_ID_PARAM);

            ExecutionContext executionContext =
                    invocationContext.getExecutionContext();
            introspection.getInterface(
                    executionContext, new InterfaceIdentifier(ifaceId),
                new IntrospectionAsyncHandle<InterfaceDefinition>(asyncHandle) {
                    @Override
                    protected DataValue convert(InterfaceDefinition result) {
                        if (result != null) {
                            return IntrospectionDataFactory.toMethodIdList(
                                    result);
                        } else {
                            Message errorMsg = MessageFactory.getMessage(
                                    "vapi.introspection.operation.service.not_found",
                                    ifaceId);

                            throw new ErrorValueException(
                                    StandardDataFactory.createErrorValueForMessages(
                                            NOT_FOUND_ERROR_INFO_DEF,
                                            Arrays.asList(errorMsg)));
                        }
                    }
                });
        }
    }

}

