/* **********************************************************
 * Copyright (c) 2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.security;

import java.util.Collections;
import java.util.Map;

import com.vmware.vapi.core.ExecutionContext.SecurityContext;

/**
 * This class contains utility methods used by the runtime security code
 */
public final class SecurityUtil {

    /**
     * Casts down the passed object to the expected class if possible.
     *
     * @param o the object to be cast. can be null.
     * @param expectedClass the expected object type. must not be null.
     * @return the object value cast to the expected class. if the cast is not
     *         possible <code>null</code> will be returned.
     */
    public static <T> T narrowType(Object o, Class<T> expectedClass) {
        if (expectedClass.isInstance(o)) {
            return expectedClass.cast(o);
        }

        return null;
    }

    /**
     * Creates security context using the provided map of properties. Further
     * modifications of the specified map affect the returned instance.
     *
     * @param props contains the security context properties. can be
     *        <code>null</code>
     * @return {@link SecurityContext} containing the properties passed as
     *         parameter. must not be <code>null</code>
     */
    public static SecurityContext createDefaultSecurityContext(Map<String, Object> props) {
        final Map<String, Object> scProps = (props != null && !props.isEmpty())
                    ? Collections.unmodifiableMap(props)
                    : Collections.<String, Object>emptyMap();
        return new SecurityContext() {
            @Override
            public Object getProperty(String key) {
                return scProps.get(key);
            }

            @Override
            public Map<String, Object> getAllProperties() {
                return scProps;
            }
        };
    }
}
