/* **********************************************************
 * Copyright 2022-2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.tracing.otel;

import java.util.Objects;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.protocol.common.http.HttpConstants;
import com.vmware.vapi.tracing.Tracer;
import com.vmware.vapi.tracing.TracingLevel;

import io.opentelemetry.api.OpenTelemetry;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.context.Context;
import io.opentelemetry.context.propagation.TextMapPropagator;

/**
 * A {@link Tracer} backed by {@link io.opentelemetry.api.OpenTelemetry} to
 * create {@link OtelTracingSpan}s.
 */
public class OtelTracer implements Tracer {

    private static final String TRACER_NAME = "com.vmware.vapi";
    private static final Logger LOGGER = LoggerFactory
            .getLogger(OtelTracer.class);
    private static final short USER_AGENT_MAX_LEN = 128;

    private final TracingLevel tracingLevel;
    private final OpenTelemetry otel;
    private final io.opentelemetry.api.trace.Tracer tracer;

    /**
     * @param otel the opentelemetry instance. Cannot be {@code null}
     * @param traceingLevel the tracing level. Cannot be {@code null}
     */
    public OtelTracer(OpenTelemetry otel, TracingLevel tracingLevel) {
        this.otel = Objects.requireNonNull(otel);
        this.tracingLevel = Objects.requireNonNull(tracingLevel);
        tracer = otel.getTracer(TRACER_NAME);
    }

    @Override
    public OtelTracingSpan attachServerSpan(HttpServletRequest request) {
        TextMapPropagator propagator = otel.getPropagators()
                .getTextMapPropagator();
        Context context = propagator.extract(Context.current(),
                                             request,
                                             new TracingRequestAdapter());
        Span decorated = tracer.spanBuilder("")
                .setParent(context).setSpanKind(SpanKind.SERVER).startSpan();
        OtelTracingSpan span = new OtelTracingSpan(otel, decorated, tracingLevel);
        span.setAttribute(TracingAttributeKey.COMPONENT,
                          TracingAttributeKey.VAPI_COMPONENT_VALUE);

        String peerAddress = getPeerAddress(request);

        span.setAttribute(TracingAttributeKey.PEER_ADDRESS,
                                        peerAddress);
        span.setAttribute(TracingAttributeKey.IS_INTERNAL,
                                        isInternal(peerAddress));

        String agent = getUserAgentTrimmed(request);
        if (agent != null) {
            span.setAttribute(TracingAttributeKey.HTTP_USER_AGENT, agent);
        }
        LOGGER.trace("Started new span {}", decorated);
        return span;
    }

    @Override
    public OtelTracingSpan createClientSpan(String name) {
        Span decorated = tracer.spanBuilder("")
                .setSpanKind(SpanKind.CLIENT)
                .startSpan();
        OtelTracingSpan otelSpan = new OtelTracingSpan(otel,
                                                       decorated,
                                                       tracingLevel);
        otelSpan.updateName(name);
        otelSpan.setAttribute(TracingAttributeKey.COMPONENT,
                              TracingAttributeKey.VAPI_COMPONENT_VALUE);

        otelSpan.addEvent("user >> vAPI client",
                          TracingAttributeKey.THREAD_NAME,
                          Thread.currentThread().getName());
        LOGGER.trace("Started new span {}", decorated);
        return otelSpan;
    }

    static String getUserAgentTrimmed(HttpServletRequest request) {
        String agent = request.getHeader(HttpConstants.HEADER_USER_AGENT);
        if (agent == null) {
            return null;
        }
        if (agent.length() > USER_AGENT_MAX_LEN) {
            return agent.substring(0, USER_AGENT_MAX_LEN);
        }
        return agent;
    }

    static String getPeerAddress(HttpServletRequest request) {
        String forwardedFor = request.getHeader("X-Forwarded-For");

        if(forwardedFor == null) {
            return request.getRemoteAddr();
        }
        return forwardedFor;
    }

    static boolean isInternal(String peerAddress) {
        if (peerAddress == null) {
            return false;
        }

        String[] addresses = peerAddress.split(",");
        for(String address : addresses) {
            if(!isLocalAddress(address)) {
                return false;
            }
        }
        return true;
    }

    static boolean isLocalAddress(String ip) {
        return ip.startsWith("127.") || "::1".equals(ip);
    }

    /**
     * @return the decorated OpenTelemetry
     */
    public OpenTelemetry getOpenTelemetry() {
        return otel;
    }

    /**
     * @return the tracing level
     */
    public TracingLevel getTraceLevel() {
        return tracingLevel;
    }
}