/* **********************************************************
 * Copyright 2022-2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.tracing.otel;

import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.std.StandardDataFactory;

import io.opentelemetry.api.common.AttributeKey;

/**
 * A wrapper for an {@link AttributeKey}.
 *
 * <p>A typical scenario when this class will be used, is:</p>
 *
 * <blockquote><code><pre>
 * TracingSpan tracingSpan = .......;
 * tracingSpan.setAttribute(someTracingAttributreKey, value);
 * </pre></code></blockquote>
 *
 * <p>For definitions of common keys, see {@link TracingAttributeKey}./</p>
 */
// FIXME this class should not have a runtime dependency to oTel
// See https://jira.eng.vmware.com/browse/VAPI-6976
public class TracingAttributeKey<T> {

    public static final TracingAttributeKey<String> COMPONENT;
    public static final String VAPI_COMPONENT_VALUE = "VAPI";

    public static final TracingAttributeKey<String> HTTP_CONTENT_TYPE;

    public static final TracingAttributeKey<String> HTTP_METHOD;

    public static final TracingAttributeKey<Long> HTTP_STATUS_CODE;

    public static final TracingAttributeKey<String> HTTP_URL;

    public static final TracingAttributeKey<String> NET_HOST_NAME;

    public static final TracingAttributeKey<Long> NET_HOST_PORT;

    public static final TracingAttributeKey<String> THREAD_NAME;

    /**
     * Indicates what vAPI protocol was used, e.g. json-rpc, rest, json-rpc-1.1
     */
    public static final TracingAttributeKey<String> WIRE_PROTOCOL;

    public static final TracingAttributeKey<String> PEER_ADDRESS;

    public static final TracingAttributeKey<Boolean> IS_INTERNAL;

    /**
     * Indicates the type of IO that was used. For the possible values,
     * see {@link IoType}.
     */
    public static final TracingAttributeKey<String> IO;

    /**
     * The opId. Many VMware services assign an opId to an operation. As the
     * operation is carried out, possibly across multiple services, that opId
     * is passed from service to service, and logged, and serves as a way to
     * correlate the logs pertaining to that operation in each service.
     *
     * <p>The opId may become obsolete over time, as all services adopt
     * OpenTelemetry/OpenTracing, but while some services still don't support
     * OpenTelemetry/OpenTracing, store the opId as a span attribute, so it
     * will be available, if necessary.</p>
     */
    public static final TracingAttributeKey<String> OP_ID;

    /**
     * A boolean flag indicating whether the operation is a task. When a call
     * for a task operation is made, the server returns a string value
     * containing the task ID, and the server continues the processing of the
     * task. Later, through the task ID, the client can query the server about
     * the status of the task.
     *
     * @see #TASK_ID
     */
    public static final TracingAttributeKey<Boolean> TASK;

    /**
     * The ID of a task. When a call for a task operation is made, the server
     * returns a string value containing the task ID. We need to set that ID
     * as a span attribute.
     *
     *  @see #TASK
     */
    public static final TracingAttributeKey<String> TASK_ID;

    /**
     * Even through OpenTelemetry has its own semantic attributes for exceptions, we don't use them,
     * but we use the standard tags for VMware (see <a href=
     * "https://confluence.eng.vmware.com/display/VDT/How+to+integrate+a+new+service+with+VDT" >How
     * to integrate a new service with VDT</a>. The combined length of the tag key and tag value
     * should not exceed 255 characters, otherwise Wavefront will reject the whole span batch.
     * </p>
     *
     * <p>
     * Ultimately, if there's an error, we are supposed to add the following attributes (or, in
     * OpenTracing lingo, 'tags') to the span:
     * </p>
     *
     * <ul>
     * <li>{@code error}: a boolean flag indicating whether there was an error</li>
     * <li>{@code error.type}: a string indicating the type of the error (for details, see
     * {@link #ERROR_TYPE})</li>
     * <li>{@code error.what}: a string containing the error message</li>
     * </ul>
     *
     * @see #ERROR_TYPE
     * @see #ERROR_MESSAGE
     */
    public static final TracingAttributeKey<Boolean> ERROR;

    /**
     * See {@link #ERROR}.
     *
     * <p>In case of vAPI error responses (see {@link MethodResult#getError()}),
     * the error type will be the result {@link ErrorValue#getName()}.</p>
     *
     * <p>In case of Java exceptions, the error type will be the canonical class
     * name of the exception.</p>
     *
     * @see #ERROR
     * @see #ERROR_MESSAGE
     */
    public static final TracingAttributeKey<String> ERROR_TYPE;

    /**
     * See {@link #ERROR}.
     *
     * <p>In case of vAPI error responses (see {@link MethodResult#getError()}),
     * the error message will be extracted via {@link
     * StandardDataFactory#getMessagesFromErrorValue(com.vmware.vapi.data.StructValue)}.
     * If there are multiple error messages, the first one will be used.</p>
     *
     * <p>In case of Java exceptions, the error type will be the exception's
     * message.</p>
     *
     * @see #ERROR
     * @see #ERROR_TYPE
     */
    public static final TracingAttributeKey<String> ERROR_MESSAGE;

    public static final TracingAttributeKey<String> HTTP_USER_AGENT;

    static {
        if (OtelFeature.isEnabled()) {
            HTTP_CONTENT_TYPE = new TracingAttributeKey<>(AttributeKey
            .stringKey("http.response_content_type"));
            HTTP_METHOD = new TracingAttributeKey<>(AttributeKey.stringKey("http.method"));
            HTTP_STATUS_CODE = new TracingAttributeKey<>(AttributeKey.longKey("http.status_code"));
            HTTP_URL = new TracingAttributeKey<>(AttributeKey.stringKey("http.url"));
            HTTP_USER_AGENT = new TracingAttributeKey<>(AttributeKey.stringKey("http.user_agent"));
            NET_HOST_NAME = new TracingAttributeKey<>(AttributeKey.stringKey("net.host.name"));
            NET_HOST_PORT = new TracingAttributeKey<>(AttributeKey.longKey("net.host.port"));
            THREAD_NAME = new TracingAttributeKey<>(AttributeKey.stringKey("thread.name"));
            WIRE_PROTOCOL = new TracingAttributeKey<>(AttributeKey.stringKey("wire.protocol"));
            PEER_ADDRESS = new TracingAttributeKey<>(AttributeKey.stringKey("peer.address"));
            IS_INTERNAL = new TracingAttributeKey<>(AttributeKey.booleanKey("is_internal"));
            IO = new TracingAttributeKey<>(AttributeKey.stringKey("io"));
            OP_ID = new TracingAttributeKey<>(AttributeKey.stringKey("opId"));
            TASK = new TracingAttributeKey<>(AttributeKey.booleanKey("task"));
            TASK_ID = new TracingAttributeKey<>(AttributeKey.stringKey("task.id"));
            ERROR = new TracingAttributeKey<>(AttributeKey.booleanKey("error"));
            ERROR_TYPE = new TracingAttributeKey<>(AttributeKey.stringKey("error.type"));
            ERROR_MESSAGE = new TracingAttributeKey<>(AttributeKey.stringKey("error.what"));
            COMPONENT = new TracingAttributeKey<>(AttributeKey.stringKey("component"));
        } else {
            HTTP_CONTENT_TYPE = null;
            HTTP_METHOD = null;
            HTTP_STATUS_CODE = null;
            HTTP_URL = null;
            HTTP_USER_AGENT = null;
            NET_HOST_NAME = null;
            NET_HOST_PORT = null;
            THREAD_NAME = null;
            WIRE_PROTOCOL = null;
            PEER_ADDRESS = null;
            IS_INTERNAL = null;
            IO = null;
            OP_ID = null;
            TASK = null;
            TASK_ID = null;
            ERROR = null;
            ERROR_TYPE = null;
            ERROR_MESSAGE = null;
            COMPONENT = null;
        }
    }

    private final AttributeKey<T> _attrKey;

    private TracingAttributeKey(AttributeKey<T> attrKey) {
        assert attrKey != null;
        this._attrKey = attrKey;
    }

    public AttributeKey<T> getAttributeKey() {
        return _attrKey;
    }

    //-------------------------------------------------------------------------

    /** Contains possible values for the {@link TracingAttributeKey#IO} attribute */
    public enum IoType {
        /**
         * Blocking I/O
         */
        BIO,

        /**
         * Java NIO (non-blocking I/O)
         */
        NIO
    }
}
