/* **********************************************************
 * Copyright 2012-2013 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.util.async;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.data.DataValue;

/**
 * Utility for scenarios where an asynchronous operation needs to accumulate
 * the results of multiple other asynchronous operations. The accumulator
 * operation completes only after all the subsidiary operations have completed.
 * The results are accumulated in a {@link java.util.Set}.
 *
 * @param <T> the type of elements that would be collected in the accumulator
 */
public class SetAccumulator<T> {

    private final AsyncHandle<Set<T>> asyncHandle;

    /**
     * Total number of subsidiary operations.
     */
    private final int total;

    /**
     * Number of completed subsidiary operations.
     */
    private int received;

    private final Set<T> accumulator;

    /**
     * Creates an accumulator.
     *
     * @param asyncHandle  handle for the result of the accumulating operation;
     *                     this is the master async handle; slave async handles
     *                     for subsidiary operations have to be created via
     *                     {@link #createSlave()}
     * @param total        total number of subsidiary operations
     */
    public SetAccumulator(AsyncHandle<Set<T>> asyncHandle, int total) {
        this.asyncHandle = asyncHandle;
        this.total = total;
        this.received = 0;
        this.accumulator = new HashSet<T>();
        if (total == 0) {
            asyncHandle.setResult(accumulator);
        }
    }

    /**
     * Creates an async handle that would receive the result of a subsidiary
     * async operation. This method should be called exactly {@link #total} times.
     *
     * @return async handle for subsidiary operation
     */
    public AsyncHandle<Set<T>> createSlave() {
        return new AsyncHandle<Set<T>>() {
            @Override
            public void updateProgress(DataValue progress) {
                // ignore
            }

            @Override
            public void setResult(Set<T> result) {
                onResult(result);
            }

            @Override
            public void setError(RuntimeException error) {
                onError(error);
            }
        };
    }

    /**
     * Same as {@link #createSlave()} but for operations which produce a single
     * element rather than a set of elements.
     */
    public AsyncHandle<T> createSlaveForOneElement() {
        return new AsyncHandle<T>() {
            @Override
            public void updateProgress(DataValue progress) {
                // ignore
            }

            @Override
            public void setResult(T result) {
                Set<T> s;
                if (result != null) {
                    s = Collections.singleton(result);
                } else {
                    // avoid set with null element
                    s = Collections.emptySet();
                }
                onResult(s);
            }

            @Override
            public void setError(RuntimeException error) {
                onError(error);
            }
        };
    }

    /**
     * Notifies the accumulator that a subsidiary operation has completed.
     *
     * @param value  the result of the subsidiary operation
     */
    private void onResult(Set<T> value) {
        synchronized (accumulator) {
            if (received >= total) {
                return;
            }
            accumulator.addAll(value);
            subsidiaryOperationCompleted();
        }
    }

    /**
     * Notifies the accumulator that a subsidiary operation has completed with
     * an error.
     *
     * @param error  cause of death of a subsidiary operation
     */
    private void onError(RuntimeException error) {
        // ignore the error
        subsidiaryOperationCompleted();
    }

    private void subsidiaryOperationCompleted() {
        synchronized (accumulator) {
            ++received;
            if (received >= total) {
                asyncHandle.setResult(accumulator);
            }
        }
    }
}
