/* *************************************************************************
 * Copyright (c) 2019 VMware, Inc. All rights reserved. -- VMware Confidential
 * *************************************************************************/

package com.vmware.vapi.internal.util.time;

/**
 * API for simple timings. In order to start the chronometer, call
 * {@link #start()}. After that, you can call the output method
 * {@link #getTime()} or complete the session using {@link #stop()}.
 *
 * <p>This class is not thread-safe.</p>
 */
public class Chronometer {

    enum State {
         UNSTARTED,
         RUNNING,
         STOPPED
    }

    private State runningState = State.UNSTARTED;
    private long startTime;
    private long stopTime;

    /**
     * This method starts a new timing session, clearing any previous values.
     *
     * @throws IllegalStateException if the chronometer is already running.
     */
    public void start() {
        if (this.runningState == State.STOPPED) {
            throw new IllegalStateException("Stopwatch must be reset before being restarted. ");
        }
        if (this.runningState != State.UNSTARTED) {
            throw new IllegalStateException("Stopwatch already started. ");
        }
        this.stopTime = 0;
        this.startTime = System.nanoTime();
        this.runningState = State.RUNNING;
    }

    /**
     * This method ends a new timing session, allowing the time to be retrieved.
     *
     * @throws IllegalStateException if the chronometer can't be stopped in its
     *         current state.
     */
    public void stop() {
        if (this.runningState != State.RUNNING) {
            throw new IllegalStateException("Stopwatch is not running. ");
        }
        this.stopTime = System.nanoTime();
        this.runningState = State.STOPPED;
    }

    /**
     * Reset the chronometer so that it can be started again.
     */
    public void reset() {
        this.runningState = State.UNSTARTED;
    }

    /**
     * Get the time on the chronometer. In case the method is called after
     * {@link #stop()}, this method returns the difference between the start and
     * the stop times of the chronometer. Otherwise it returns the time between
     * the call of the current method and the {@link #start()}.
     *
     * @return the time in nanoseconds
     */
    public long getTime() {
        if (this.runningState == State.UNSTARTED) {
            return 0;
        } else if (this.runningState == State.STOPPED) {
            return this.stopTime - this.startTime;
        } else if (this.runningState == State.RUNNING) {
            return System.nanoTime() - this.startTime;
        }
        throw new RuntimeException("Illegal running state has occured. ");
    }
}
