/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * MessageFormatTemplateFormatter.java --
 *
 *      MessageFormat based implementation of TemplateFormatter
 */
package com.vmware.vapi.l10n;

import java.text.FieldPosition;
import java.text.Format;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

import com.vmware.vapi.internal.util.Validate;

/**
 * {@link MessageFormat} based implementation of
 * <code>TemplateFormatter</code>.
 */
public class MessageFormatTemplateFormatter implements TemplateFormatter {
    private MessageArgConverter argConverter;

    public MessageFormatTemplateFormatter() {
        argConverter = new MessageArgConverter();
    }

    @Override
    public String format(String msgTemplate, List<String> args, Locale locale) {
        return format(msgTemplate,args,locale, null);
    }

    @Override
    public String format(String msgTemplate,
                         List<String> args,
                         Locale locale,
                         TimeZone tz) {
        Validate.notNull(msgTemplate);
        Validate.notNull(args);
        Validate.notNull(locale);

        MessageFormat msgFormat = null;
        try {
            msgFormat = new MessageFormat(msgTemplate, locale);

            if(tz != null) {
                setTz(msgFormat, tz);
            }

            StringBuffer result = new StringBuffer();
            msgFormat.format(buildTypedArguments(args,
                                                 msgFormat.getFormatsByArgumentIndex()),
                             result,
                             new FieldPosition(0));

            return result.toString();
        } catch (IllegalArgumentException ex) {
            throw new LocalizationException("Invalid message template format", ex);
        }
    }

    Object[] buildTypedArguments(List<String> argValues,
                                        Format[] argFormats) {
        if (argValues == null) {
            return new Object[0];
        }

        Object[] result = new Object[argValues.size()];
        for (int i = 0; i < argValues.size(); i++) {
            result[i] = convertToTypedArg(argValues.get(i), argFormats[i]);
        }

        return result;
    }

    Object convertToTypedArg(String argValue, Format argFormat) {
        if (argFormat == null) {
            return argValue;
        }

        if (argFormat instanceof SimpleDateFormat) {
            Calendar cal = argConverter.toCalendarArgument(argValue);
            // MessageFormat needs a Date instance
            return cal.getTime();
        }
        if (argFormat instanceof NumberFormat) {
            try {
                return argConverter.toLong(argValue);
            } catch (LocalizationException ex) {
            }

            try {
                return argConverter.toDouble(argValue);
            } catch (LocalizationException ex) {
            }

            throw new LocalizationException(
                    "Value doesn't match expected format type");
        }

        // don't convert; keep original arg type (String)
        return argValue;
    }


    private void setTz(MessageFormat mf, TimeZone tz) {
        Object [] formats = mf.getFormats();
        for (int i = 0; i < formats.length; i++) {
            if (formats[i] instanceof SimpleDateFormat) {
                ((SimpleDateFormat)formats[i]).setTimeZone(tz);
            }
        }
    }
}
