/* **********************************************************
 * Copyright (c) 2012-2021, 2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol;

import java.net.ProxySelector;
import java.security.KeyStore;
import java.security.cert.CertStore;
import java.util.Objects;

import javax.net.ssl.SSLSocket;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.client.exception.TransportProtocolException;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.common.http.HttpConstants;

/**
 * This class is used for configuring the HTTP client that is used by the
 * runtime client. The class is immutable.
 */
public final class HttpConfiguration {
    /**
     * System property which is recognized by Bouncy Castle FIPS
     * crypto provider (bc-fips) and turns it into FIPS mode.
     */
    static final String BOUNCY_CASTLE_IN_FIPS_MODE =
            "org.bouncycastle.fips.approved_only";

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpConfiguration.class);

    private final LibType libraryType;
    private final SslConfiguration sslConfig;
    private final int ioThreadCount;
    private final int connectTimeout;
    private final int soTimeout;
    private final int maxConnections;
    private final int maxReponseSize;
    private final long keepAlivePeriod;
    private final HeadersProvider headersProvider;
    private final Protocol protocol;
    private final ProxyConfiguration proxyConfiguration;

    /**
     * Type of the third-party HTTP library which would be used by the
     * transport implementation.
     */
    public enum LibType {
        /**
         * Apache HTTP Client. This is blocking I/O implementation.
         */
        APACHE_HTTP_CLIENT,

        /**
         * Apache HTTP AsyncClient. This is non-blocking I/O implementation.
         */
        APACHE_HTTP_ASYNC_CLIENT,
    }

    /**
     * Enumerates the wire protocols supported by the vAPI client for invoking APIs.
     *
     * <p> vAPI JSON-RPC is a JSON-RPC based protocol supported by vAPI clients and servers.
     */
    public enum Protocol {
        /**
         * vAPI JSON-RPC 1.0 (over HTTP) protocol.
         *
         * <p> This is the default and recommended version in use. In this protocol,
         * all the information sent by the client is encapsulated in the JSON message
         * sent as HTTP request payload.
         */
        VAPI_JSON_RPC_1_0,

        /**
         * vAPI JSON-RPC 1.1 (over HTTP) protocol.
         *
         * <p> This version is enhancement over 1.0, where target service and operation,
         * authentication and other information is transported in HTTP headers.
         * This benefits intermediaries and proxies for making authentication or routing
         * decisions based on the HTTP headers only, and without need to parse the JSON payload.
         *
         * <p> vAPI servers which support vAPI JSON-RPC 1.1 are compatible with and
         * fully support 1.0 clients.
         *
         * <p> Even 1.0 server can handle 1.1 clients, but any changes made by intermediaries
         * to the HTTP header values will not be visible for the old 1.0 servers.
         */
        VAPI_JSON_RPC_1_1
    }
    private HttpConfiguration(LibType libraryType,
                              SslConfiguration sslConfig,
                              int ioThreadCount,
                              int connectTimeout,
                              int soTimeout,
                              int maxConnections,
                              int maxReponseSize,
                              long keepAlivePeriod,
                              HeadersProvider headersProvider,
                              Protocol protocol,
                              ProxyConfiguration proxyConfiguration) {
        this.libraryType = libraryType;
        this.sslConfig = sslConfig;
        this.ioThreadCount = ioThreadCount;
        this.connectTimeout = connectTimeout;
        this.soTimeout = soTimeout;
        this.maxConnections = maxConnections;
        this.keepAlivePeriod = keepAlivePeriod;
        this.headersProvider = headersProvider;
        this.protocol = protocol;
        this.proxyConfiguration = proxyConfiguration;
        this.maxReponseSize = maxReponseSize;
    }

    /**
     * @return the type of the HTTP library; cannot be {@code null}
     */
    public LibType getLibraryType() {
        return libraryType;
    }

    /**
     * @return the SSL configuration; cannot be {@code null}
     */
    public SslConfiguration getSslConfiguration() {
        return sslConfig;
    }

    /**
     * @return the number of I/O dispatch threads to be used by the I/O reactor
     */
    public int getIoThreadCount() {
        return ioThreadCount;
    }

    /**
     * @return the amount of time in milliseconds the client is going to wait for
     * establishing the connection
     */
    public int getConnectTimeout() {
        return connectTimeout;
    }

    /**
     * @return the amount of time in milliseconds the client is going to wait for
     *         a response before timing out
     */
    public int getSoTimeout() {
        return soTimeout;
    }

    /**
     * @return the maximum number of outstanding TCP connections
     */
    public int getMaxConnections() {
        return maxConnections;
    }

    /**
     * @return The effective maximum allowed size of the response content in bytes
     * @see {@link Builder#setMaxResponseSize(int)}
     */
    public int getMaxResponseSize() {
        return maxReponseSize;
    }

    /**
     * Maximum keep alive period for pooled/idle TCP connections. This value MUST be
     * lower than max idle time in the server.
     *
     * @return the maximum keep alive period for pooled/idle TCP connections
     */
    public long getKeepAlivePeriod() {
        return keepAlivePeriod;
    }

    /**
     * @return the callback instance for adding headers to all requests; may be
     *         {@code null}
     */
    public HeadersProvider getHeadersProvider() {
        return headersProvider;
    }

    /**
     * @return vAPI protocol; cannot be {@code null}
     */
    public Protocol getProtocol() {
        return protocol;
    }

    /**
     * @return a ProxyConfiguration instance; may be {@code null}.
     */
    public ProxyConfiguration getProxyConfiguration() {
        return proxyConfiguration;
    }

    /**
     * This class contains the SSL configuration. The class is immutable.
     */
    public static final class SslConfiguration {
        /**
         * Special instance which can be used to fully avoid TLS/SSL
         * initialization when used with
         * {@link HttpConfiguration.Builder#setSslConfiguration(SslConfiguration)}
         */
        public static final SslConfiguration SKIP_SSL_INITIALIZATION;

        private static final SslConfiguration DEFAULT;
        static final String[] DEFAULT_PROTOCOLS;

        static {
            SKIP_SSL_INITIALIZATION = new Builder().getConfig();

            // DEFAULT_PROTOCOLS must be initialized before the default
            // configuration; otherwise, the protocols in the default
            // configuration are set to null.
            DEFAULT_PROTOCOLS = new String[] { "TLSv1", "TLSv1.1", "TLSv1.2" };
            DEFAULT = new Builder().getConfig();
        }

        private final KeyStore trustStore;
        private final KeyStore keyStore;
        private final KeyStoreConfig keyStoreConfig;
        private final CertStore crlCertStore;
        private final String[] enabledProtocols;
        private final String[] enabledCipherSuites;
        private final boolean hostnameVerificationDisabled;
        private final boolean certificateValidationDisabled;

        /**
         * @deprecated Replaced by {@link Builder}.
         *
         * @param trustStore the keystore that contains the trusted
         *        certificates; may be {@code null}.
         */
        @Deprecated
        public SslConfiguration(KeyStore trustStore) {
            this(trustStore, null, null);
        }

        /**
         * @deprecated Replaced by {@link Builder}.
         *
         * @param trustStore the keystore that contains the trusted
         *        certificates; may be {@code null}.
         * @param keyStore the keystore that contains the client certificates.
         * @param keyStoreConfig the configuration for client certificates.
         */
        @Deprecated
        public SslConfiguration(KeyStore trustStore,
                                KeyStore keyStore,
                                KeyStoreConfig keyStoreConfig) {
            this(trustStore, keyStore, keyStoreConfig, null);
        }

        /**
         * @deprecated Replaced by {@link Builder}.
         *
         * @param trustStore the keystore that contains the trusted
         *        certificates; may be {@code null}.
         * @param keyStore the keystore that contains the client certificates.
         * @param keyStoreConfig the configuration for client certificates.
         * @param crlCertStore CRL CertStore that would be used for validating
         *        server certificates.
         */
        @Deprecated
        public SslConfiguration(KeyStore trustStore,
                                KeyStore keyStore,
                                KeyStoreConfig keyStoreConfig,
                                CertStore crlCertStore) {
            this(trustStore,
                 keyStore,
                 keyStoreConfig,
                 crlCertStore,
                 DEFAULT_PROTOCOLS,
                 null,
                 false,
                 false);
        }

        private SslConfiguration(KeyStore trustStore,
                                 KeyStore keyStore,
                                 KeyStoreConfig keyStoreConfig,
                                 CertStore crlCertStore,
                                 String[] enabledProtocols,
                                 String[] enabledCipherSuites,
                                 boolean hostnameVerificationDisabled,
                                 boolean certificateValidationDisabled) {
            this.trustStore = trustStore;
            this.keyStore = keyStore;
            this.keyStoreConfig = keyStoreConfig;
            this.crlCertStore = crlCertStore;
            this.enabledProtocols = enabledProtocols;
            this.enabledCipherSuites = enabledCipherSuites;
            this.hostnameVerificationDisabled = hostnameVerificationDisabled;
            this.certificateValidationDisabled = certificateValidationDisabled;
            validate();
        }

        private void validate() {
            if ((keyStore != null) ^ (keyStoreConfig != null)) {
                throw new IllegalStateException("Key-store and key-store-config"
                                         + " should both be set to non-null "
                                         + "values for the key-store to work.");
            }

            if (crlCertStore != null && trustStore == null) {
                throw new IllegalStateException("Custom CRL-cert-store cannot"
                        + " be configured along with the default trust-store.");
            }

            if (certificateValidationDisabled && trustStore != null) {
                throw new IllegalStateException("Certificate validation is "
                               + "disabled. Custom trust-store cannot be set.");
            }
        }

        /**
         * Returns the keystore that contains certificates of trusted parties or
         * Certificate Authorities trusted to identify other parties.
         *
         * @return the keystore with trusted certificates and CAs; cannot be
         *         {@code null}
         */
        public KeyStore getTrustStore() {
            return trustStore;
        }

        /**
         * Returns the keystore which contains the certificates used to
         * authenticate the client during an SSL handshake.
         *
         * @return the keystore with the client's certificates; may be
         *         {@code null}
         * @see #getKeyStoreConfig()
         */
        public KeyStore getKeyStore() {
            return keyStore;
        }

        /**
         * Returns the configuration parameters (alias and password) for the
         * {@link KeyStore} containing the client certificates for
         * authentication.
         *
         * @return the keystore configuration parameters; may be {@code null}
         * @see #getKeyStore()
         */
        public KeyStoreConfig getKeyStoreConfig() {
            return keyStoreConfig;
        }

        /**
         * Returns the certificate revocation list {@link CertStore} that would
         * be used for validating server certificates.
         *
         * @return the cert-store for retrieving the CRLs; may be {@code null}
         */
        public CertStore getCrlCertStore() {
            return crlCertStore;
        }

        /**
         * Returns the names of cryptographic protocols enabled for use by this
         * configuration.
         *
         * @return an array of protocol names; may be {@code null}
         */
        public String[] getEnabledProtocols() {
            return enabledProtocols;
        }

        /**
         * Returns the names of the SSL cipher suites enabled for use by this
         * configuration.
         *
         * @return an array of cipher suite names; may be {@code null}
         */
        public String[] getEnabledCipherSuites() {
            return enabledCipherSuites;
        }

        /**
         * @return {@code true} if the standard process of verifying the
         *         server's hostname is to be skipped; otherwise {@code false}
         */
        public boolean isHostnameVerificationDisabled() {
            return hostnameVerificationDisabled;
        }

        /**
         * @return {@code true} if the standard process of validating the
         *         server's certificate is to be skipped; otherwise
         *         {@code false}
         */
        public boolean isCertificateValidationDisabled() {
            return certificateValidationDisabled;
        }

        /**
         * Builder class for the {@link SslConfiguration}.
         */
        public static final class Builder {
            private KeyStore trustStore;
            private KeyStore keyStore;
            private KeyStoreConfig keyStoreConfig;
            private CertStore crlCertStore;
            private String[] enabledProtocols = DEFAULT_PROTOCOLS;
            private String[] enabledCipherSuites;
            private boolean hostnameVerificationDisabled;
            private boolean certificateValidationDisabled;

            public Builder() {
               this(null);
            }

            /**
             * @param trustStore the keystore that contains certificates of
             *        trusted parties or Certificate Authorities trusted to
             *        identify other parties; may be {@code null}, in which
             *        case, the default JRE trust-store would be used
             */
            public Builder(KeyStore trustStore) {
                this.trustStore = trustStore;
            }

            /**
             * Sets the {@link KeyStore} that contains certificates of trusted
             * parties or Certificate Authorities trusted to identify other
             * parties.
             *
             * <p>Default value is {@code null} which results in the default JRE
             * trust-store being used.
             *
             * @param trustStore the keystore that contains trusted certificates;
             *        may be {@code null}, in which case, the default JRE
             *        trust-store would be used
             * @return this instance
             */
            public Builder setTrustStore(KeyStore trustStore) {
                this.trustStore = trustStore;
                return this;
            }

            /**
             * Sets the {@link KeyStore} which contains the certificates used to
             * authenticate the client during an SSL handshake.
             *
             * <p>A {@link KeyStoreConfig} object specifying the alias and
             * password of the key should be provided through
             * {@link #setKeyStoreConfig(KeyStoreConfig)} for this to work.
             *
             * <p>Default value is {@code null} which results in no client
             * authentication provided when connecting to the server.
             *
             * @param keyStore the keystore that contains the client
             *        certificates; may be {@code null}.
             * @return this instance
             * @see #setKeyStoreConfig(KeyStoreConfig)
             */
            public Builder setKeyStore(KeyStore keyStore) {
                this.keyStore = keyStore;
                return this;
            }

            /**
             * Sets the configuration parameters for the {@link KeyStore}
             * provided with {@link #setKeyStore(KeyStore)}.
             *
             * <p>Default value is {@code null} which results in no client
             * authentication provided when connecting to the server.
             *
             * @param keyStoreConfig the configuration for client certificates.
             * @return this instance
             * @see #setKeyStore(KeyStore)
             */
            public Builder setKeyStoreConfig(KeyStoreConfig keyStoreConfig) {
                this.keyStoreConfig = keyStoreConfig;
                return this;
            }

            /**
             * Sets the certificate revocation list {@link CertStore} that would
             * be used for validating server certificates.
             *
             * <p>Default value is {@code null} which results in no validation
             * for the server certificates against CRLs.
             *
             * <p>Requires a custom trust-store to be set; otherwise an
             * {@link IllegalArgumentException} would be thrown upon invoking
             * {@link #getConfig()} of this instance.
             *
             * @param crlCertStore the cert-store to be used for retrieving the
             *        CRLs; may be {@code null}.
             * @return this instance
             */
            public Builder setCrlCertStore(CertStore crlCertStore) {
                this.crlCertStore = crlCertStore;
                return this;
            }

            /**
             * Sets the protocol versions enabled for use by this configuration.
             *
             * <p>The specified protocols must be among {@code SSLv2Hello},
             * {@code SSLv3}, {@code TLSv1}, {@code TLSv1.1} and {@code TLSv1.2}
             * ; otherwise, applying this configuration over an
             * {@link SSLSocket} later would fail.
             *
             * <p>Defaults to {@code TLSv1}, {@code TLSv1.1} and {@code TLSv1.2}.
             *
             * @param enabledProtocols Names of all the protocols to enable;
             *        must not be {@code null} or empty.
             * @return this instance
             */
            public Builder setEnabledProtocols(String[] enabledProtocols) {
                Validate.notEmpty(enabledProtocols);
                this.enabledProtocols = enabledProtocols;
                return this;
            }

            /**
             * Sets the cipher suites enabled for use by this configuration.
             *
             * <p>Default value is {@code null} which results in the default JRE
             * cipher-suites being enabled.
             *
             * @param enabledCipherSuites Names of all the cipher suites to
             *        enable.
             * @return this instance
             *
             * @see <a href=
             *      "https://docs.oracle.com/javase/8/docs/technotes/guides/security/SunProviders.html#SupportedCipherSuites">
             *      Default Enabled Cipher Suites in Java 8</a>
             */
            public Builder setEnabledCipherSuites(String[] enabledCipherSuites) {
                this.enabledCipherSuites = enabledCipherSuites;
                return this;
            }

            /**
             * Disables the verification of the server's hostname, thus exposing
             * the client to man-in-the-middle attacks.
             *
             * <p>By default the verification is enabled, providing for proper
             * SSL connectivity.
             *
             * <p>This option is provided to ease the process of initial
             * evaluation of the SDK or say to prepare code examples that are
             * easier to set-up. It <em>must not</em> be employed in a
             * production environment.
             *
             * <p>The runtime would log a warning upon each SSL-handshake when
             * the verification is disabled.
             *
             * @return this instance
             * @see <a href=
             *      "https://tools.ietf.org/html/rfc2818#section-3.1">Section
             *      3.1 of RFC 2818</a>
             */
            public Builder disableHostnameVerification() {
                this.hostnameVerificationDisabled = true;
                return this;
            }

            /**
             * Disables the validation of the server's certificate chain, thus
             * making the client trust all the endpoints it is connecting to
             * regardless of the certificate they provide.
             *
             * <p>By default the validation is enabled, allowing connections
             * only to trusted servers.
             *
             * <p>This option is provided to ease the process of initial
             * evaluation of the SDK or say to prepare code examples that are
             * easier to set-up. It <em>must not</em> be employed in a
             * production environment.
             *
             * <p>The runtime would log a warning upon each SSL-handshake when
             * the validation is disabled.
             *
             * <p>With the certificate validation disabled, if the trust-store
             * is set to a value other than {@code null}, an
             * {@link IllegalArgumentException} would be thrown upon invoking
             * {@link #getConfig()} of this instance.
             *
             * @return this instance
             */
            public Builder disableCertificateValidation() {
                this.certificateValidationDisabled = true;
                return this;
            }

            /**
             * Creates a configuration with all the settings set to this builder
             * instance.
             *
             * @return immutable {@link SslConfiguration}
             */
            public SslConfiguration getConfig() {
                return new SslConfiguration(trustStore,
                                            keyStore,
                                            keyStoreConfig,
                                            crlCertStore,
                                            enabledProtocols,
                                            enabledCipherSuites,
                                            hostnameVerificationDisabled,
                                            certificateValidationDisabled);
            }
        }
    }

    /**
     * This class contains the client certificates configuration. The class is
     * immutable.
     */
    public static final class KeyStoreConfig {
        private final String keyAlias;
        private final String keyStorePassword;

        public KeyStoreConfig(String keyAlias, String keyStorePassword) {
            Validate.notNull(keyAlias);
            Validate.notNull(keyStorePassword);
            this.keyAlias = keyAlias;
            this.keyStorePassword = keyStorePassword;
        }

        /**
         * @return the alias of the certificate to use
         */
        public String getKeyAlias() {
            return keyAlias;
        }

        /**
         * @return the keystore password
         */
        public String getKeyStorePassword() {
            return keyStorePassword;
        }
    }

    /**
     * Class contains client proxy configuration.
     * This class is immutable.
     */
    public static final class ProxyConfiguration{
        private final String host;
        private final int port;
        private final boolean useSystemProxyConfig;
        private final UserPassCredentials userPassCredentials;

        /**
         * Builder class for {@link ProxyConfiguration}.
         */
        public static final class Builder {
            private String host;
            private int port;
            private boolean useSystemProxyConfig = false;
            private UserPassCredentials userPassCredentials;

            private Builder(String host, int port) {
                this.host = host;
                this.port = port;
            }

            private Builder() {
                this.useSystemProxyConfig = true;
            }

            /**
             * Sets credentials for proxy authentication.
             * These would take precedence over system default credentials
             * set on default {@link java.net.Authenticator}.
             *
             * @return this instance
             */
            public Builder setUserPassCredentials(UserPassCredentials userPassCredentials) {
                this.userPassCredentials = userPassCredentials;
                return this;
            }

            /**
             * @return an http proxy configuration instance.
             */
            public ProxyConfiguration getConfig() {
                return new ProxyConfiguration(host,
                                              port,
                                              useSystemProxyConfig,
                                              userPassCredentials);
            }
        }

        private ProxyConfiguration(String host,
                                  int port,
                                  boolean useSystemProxyConfig,
                                  UserPassCredentials userPassCredentials) {

            this.host = host;
            this.port = port;
            this.useSystemProxyConfig = useSystemProxyConfig;
            this.userPassCredentials = userPassCredentials;
        }

        /**
         * Creates builder object for construction of
         * {@link ProxyConfiguration} instance with custom host, port.
         * <p>
         * Default schema(http) is assumed.
         * <p>
         * Mind that only explicit credentials set via
         * {@link Builder#setUserPassCredentials(UserPassCredentials)} method
         * are supported in this case.
         *
         * @param host is the proxy hostname or ip
         * @param port is the proxy port
         *
         * @return a builder instance
         */
        public static Builder custom(String host, int port) {
            return new HttpConfiguration.ProxyConfiguration.Builder(host, port);
        }

        /**
         * Creates builder object for construction of {@link ProxyConfiguration}
         * based on default {@link ProxySelector} configured on the JVM.
         * <p>
         * Mind that if the default {@link ProxySelector} is unable to select a
         * proxy host corresponding to the respective target url, request will be
         * sent directly to the target host without passing through proxy.
         * <p>
         * Credentials set via the default {@link java.net.Authenticator}
         * would be used in case no explicit credentials are supplied via
         * {@link Builder#setUserPassCredentials(UserPassCredentials)} method.
         * <p>
         * Please mind that this method is intended primarily for testing purposes
         * and its use in production code is discouraged, since it could lead to
         * exposing an HTTPS server authentication credentials via a plain text
         * network connection to an HTTP proxy if proxy asked for authentication.
         * See <a href="https://bugzilla.redhat.com/show_bug.cgi?id=1386103">OpenJDK:
         * exposure of server authentication credentials to proxy</a>
         *
         * @return builder instance
         */
        public static Builder systemDefault() {
            return new HttpConfiguration.ProxyConfiguration.Builder();
        }

        /**
         * @return the proxy hostname.
         */
        public String getHost() {
            return host;
        }

        /**
         * @return the proxy port.
         */
        public int getPort() {
            return port;
        }

        /**
         * @return {@code true} if system default proxy configuration is to be used.
         */
        public boolean usesSystemDefaultProxyConfiguration() {
            return useSystemProxyConfig;
        }

        /**
         * @return user and password.
         * Could return {@code null}.
         */
        public UserPassCredentials getUserPassCredentials() {
            return this.userPassCredentials;
        }
    }

    /**
     * Builder class for the {@link HttpConfiguration}.
     */
    public static final class Builder {

        private static final int DEFAULT_MAX_RESPONSE_SIZE = Integer.MAX_VALUE;
        private static final int DEFAULT_MAX_CONNECTIONS = 20;
        private static final long DEFAULT_KEEP_ALIVE_PERIOD = 45 * 1000;  // 45 sec
        private static final int DEFAULT_CONNECT_TIMEOUT = 15 * 1000;     // 15 sec
        private static final int DEFAULT_SOCKET_TIMEOUT = 0;    // never timeout

        private LibType libraryType =
                LibType.APACHE_HTTP_CLIENT;

        private Protocol protocol = Protocol.VAPI_JSON_RPC_1_0;

        private SslConfiguration sslConfig;
        private Integer ioThreadCount;
        private int connectTimeout = DEFAULT_CONNECT_TIMEOUT;
        private int soTimeout = DEFAULT_SOCKET_TIMEOUT;
        private int maxConnections = DEFAULT_MAX_CONNECTIONS;
        private int maxResponseSize = DEFAULT_MAX_RESPONSE_SIZE;
        private long keepAlivePeriod = DEFAULT_KEEP_ALIVE_PERIOD;
        private HeadersProvider headersProvider;
        private ProxyConfiguration proxyConfiguration;

        /**
         * Sets the type of the HTTP library to be used for execution of
         * API requests.
         *
         * <p>Default value is {@link LibType#APACHE_HTTP_CLIENT}.
         *
         * @param libraryType HTTP library flavor; must not be {@code null}
         * @return this instance
         */
        public Builder setLibraryType(LibType libraryType) {
            Validate.notNull(libraryType);
            this.libraryType = libraryType;
            return this;
        }

        /**
         * Sets the SSL configuration for executing request over HTTPs, including
         * trust store for validating server certificate, and optionally key store
         * for presenting client certificate (SSl mutual authentication).
         *
         * <p>This is not required, but if not set, the client will use the
         * default trust-store of the JRE to connect to API endpoints over HTTPs.
         *
         * <p>Note that the special instance {@link SslConfiguration#SKIP_SSL_INITIALIZATION}
         * may be used to completely disable the TLS/SSL initialization.
         *
         * <p>If the system property <b>org.bouncycastle.fips.approved_only</b> is set
         * to <b>true</b> (case insensitive) the TLS/SSL stack will not be initialized,
         * even if different configuration is provide here.
         *
         * @param sslConfig SSL related settings
         * @return this instance
         */
        public Builder setSslConfiguration(SslConfiguration sslConfig) {
            this.sslConfig = sslConfig;

            return this;
        }

        /**
         * Defines the number of I/O dispatch threads to be used by the I/O
         * reactor of the third-party HTTP library. Applies only to
         * the following libraries:
         * {@link LibType#APACHE_HTTP_ASYNC_CLIENT}.
         *
         * <p>Defaults to the number of the processors on the machine.
         *
         * @param ioThreadCount number of I/O threads; must be positive
         * @return this instance
         */
        public Builder setIoThreadCount(int ioThreadCount) {
            Validate.isTrue(ioThreadCount > 0);
            this.ioThreadCount = ioThreadCount;
            return this;
        }

        /**
         * Sets the amount of time the client is going to wait for establishing
         * a connection with the target API endpoint.
         *
         * <p>Value of 0 means never timeout (wait forever). Negative values are
         * not allowed.
         *
         * <p>The default value is 15 000 ms (15 seconds).
         *
         * @param connectTimeout time in milliseconds; must not be negative
         * @return this instance
         */
        public Builder setConnectTimeout(int connectTimeout) {
            Validate.isTrue(connectTimeout >= 0);
            this.connectTimeout = connectTimeout;
            return this;
        }

        /**
         * Sets the amount of time (in milliseconds) the client is going to wait
         * for a response before timing out. If the underlying transport
         * protocol is HTTP 1.1, this setting maps directly to
         * {@code SO_TIMEOUT} which impact the {@code Socket.read()} method.
         *
         * <p>Value of 0 means never time out (wait forever). Negative values
         * are not allowed.
         *
         * <p>The default value is 0 (never time out).
         *
         * @param soTimeout time in milliseconds; must not be negative
         * @return this instance
         */
        public Builder setSoTimeout(int soTimeout) {
            Validate.isTrue(soTimeout >= 0);
            this.soTimeout = soTimeout;
            return this;
        }

        /**
         * Sets the maximum number of outstanding TCP connections.
         *
         * <p>Default value is 20 connections.
         *
         * <p>Clients which execute large number of simultaneous API requests need to
         * consider increasing this value to avoid starvation for connections. It is inherent
         * for the HTTP protocol to occupy a TCP connection for the whole timespan of a
         * request/response execution (even if non-blocking I/O is used).
         *
         * @param maxConnections maximum number of connections; must be positive
         * @return this instance
         */
        public Builder setMaxConnections(int maxConnections) {
            Validate.isTrue(maxConnections > 0);
            this.maxConnections = maxConnections;
            return this;
        }

        /**
         * Set the maximum allowed size of the content of the response that the
         * client can consume in bytes. It is a helpful option having in mind
         * that the whole response is buffered in-memory. Response content that
         * is larger than the limit will cause a
         * {@link TransportProtocolException} at runtime.
         * <p>
         * The method guarantees that at least this size will be allowed.
         * However, the effective maximum size will be the lowest number which
         * is a power of two and is bigger than or equal to the number that is
         * set. That is, effective limit L = min(2^N), where L >=
         * maxResponseSize.
         * <p>
         * As an example, setting {@code 2000} bytes will produce an effective
         * limit of {@code 2^11} which is {@code 2048} bytes. Setting
         * {@code 4096} which is 2^12, will not lead to changes and the
         * effective limit will stay {@code 4096} bytes
         * <p>
         * {@see {@link HttpConfiguration#getMaxResponseSize()}} for the
         * effective limit set
         * <p>
         * When the response content type is
         * {@link HttpConstants#CONTENT_TYPE_STREAM_JSON} or
         * {@link HttpConstants#CONTENT_TYPE_FRAMED} {@code maxResponseSize}
         * designates the maximum size of a single frame
         * <p>
         * Default value is {@value #DEFAULT_MAX_RESPONSE_SIZE} bytes
         *
         * @param maxResponseSize The maximum allowed size of the response
         *        content in bytes.
         */
        public Builder setMaxResponseSize(int maxResponseSize) {
            Validate.isTrue(maxResponseSize > 0);
            this.maxResponseSize = computeLowestPower2(maxResponseSize);
            return this;
        }

        private int computeLowestPower2(int min) {
            int log2Min = (int) Math.ceil(Math.log(min) / Math.log(2));
            int lowestPower2 = (int) Math.pow(2, log2Min);
            if (lowestPower2 < min) {
                // integer overflow
                LOGGER.warn("The computed max response size number exceeds Integer.MAX_VALUE."
                            + "Setting it to Integer.MAX_VALUE instead.");
                return Integer.MAX_VALUE;
            }
            return lowestPower2;
        }

        /**
         * Sets the maximum time a connection is allowed to be kept alive. A
         * pooled (kept alive after finished reading a HTTP response) connection
         * that was idle for more than the specified period will be discarded
         * and closed (not reused anymore).
         *
         * <p>Non-positive value for this setting means that the connection can
         * be kept alive indefinitely.
         *
         * <p>It is highly recommended that a value lower than max idle timeout
         * on the server is used. Otherwise a race is possible between client
         * reusing a kept-alive connection for subsequent request and server
         * closing the same connection due to inactivity. In such cases the
         * client request might fail with network error (connection already
         * closed).
         *
         * <p>Default value is 45 000 ms (45 seconds).
         *
         * @param keepAlivePeriod the time period in milliseconds.
         * @return this instance
         */
        public Builder setKeepAlivePeriod(long keepAlivePeriod) {
            this.keepAlivePeriod = keepAlivePeriod;
            return this;
        }

        /**
         * Sets a callback instance that gets invoked upon each request.
         *
         * <p>The callback is expected to provide a list of
         * {@link Header headers} to be added to the request.
         *
         * <p>{@link Exception Exceptions} thrown from
         * the callback or from the methods of each {@link Header} will
         * propagate and cause the invocation to fail.
         *
         * <p>Headers {@link HttpConstants#RESERVED_HEADERS reserved} for the
         * runtime, that are being returned by the callback, will be skipped.
         *
         * @param headersProvider the callback instance
         * @return this instance
         */
        public Builder setHeadersProvider(HeadersProvider headersProvider) {
            this.headersProvider = headersProvider;
            return this;
        }

        /**
         * Sets the vAPI Protocol that is used by client to
         * invoke APIs on the server.
         *
         * <p> Default value is {@link Protocol#VAPI_JSON_RPC_1_0}.
         *
         * @param protocol vAPI Protocol; must not be {@code null}.
         * @return this instance
         */
        public Builder setProtocol(Protocol protocol) {
            Validate.notNull(protocol);
            this.protocol = protocol;
            return this;
        }

        /**
         * Sets the proxy configuration object used to configure proxy settings on the client.
         * @param proxyConfiguration is the ProxyConfiguration instance.
         * @return this instace.
         */
        public Builder setProxyConfiguration(ProxyConfiguration proxyConfiguration) {
            this.proxyConfiguration = proxyConfiguration;

            return this;
        }

        /**
         * Creates a configuration with all the settings set to this
         * builder instance.
         *
         * @return immutable {@link HttpConfiguration}
         * @throws IllegalStateException if the combination of settings is not supported
         */
        public HttpConfiguration getConfig() {
            validate();
            return new HttpConfiguration(libraryType,
                                         resolveSslConfig(sslConfig),
                                         resolveThreadCount(ioThreadCount),
                                         connectTimeout,
                                         soTimeout,
                                         maxConnections,
                                         maxResponseSize,
                                         keepAlivePeriod,
                                         headersProvider,
                                         protocol,
                                         proxyConfiguration);
        }

        private void validate() {
            if (ioThreadCount != null
                    && libraryType != LibType.APACHE_HTTP_ASYNC_CLIENT) {
                throw new IllegalStateException(
                        "IO thread count setting is not supported for transport based on library "
                                + libraryType);
            }
        }

        private SslConfiguration resolveSslConfig(SslConfiguration explicitConfig) {
            return explicitConfig != null ? explicitConfig
                                          : SslConfiguration.DEFAULT;
        }

        private int resolveThreadCount(Integer explicitConfig) {
            return explicitConfig == null ? Runtime.getRuntime().availableProcessors()
                                          : explicitConfig.intValue();
        }
    }

    /**
     * Callback for supplying HTTP headers per request.
     *
     * <p><i>Thread-safety:</i> The implementation must be thread-safe. The
     * callback method will be invoked simultaneously on different threads if
     * there are simultaneous vAPI invocations with this configuration.
     */
    public interface HeadersProvider {
        /**
         * Retrieves the headers to be added to an HTTP request.
         *
         * <p>
         * Note that headers with "vapi-ctx-" prefix and the Accept-Language
         * header are declared with precedence of the {@link HeadersProvider}
         * setting. When the same header names are provided both through the
         * {@link ExecutionContext.ApplicationData} and the
         * {@link HeadersProvider}, the values from the former will be neglected
         * in favor of the latter.
         *
         * @return the headers in the order in which they will be serialized on
         *         the wire; may be {@code null}, but the headers cannot contain
         *         {@code null} elements.
         */
        Iterable<Header> getHeaders();
    }

    /**
     * Represents an HTTP header field.
     */
    public interface Header {
        /**
         * Get the name of the header.
         *
         * @return the name of the header; never {@code null} nor empty
         */
        String getName();

        /**
         * Get the value of the header.
         *
         * @return the value of the header, may be {@code null}
         */
        String getValue();
    }

    /**
    * A basic implementation of {@link Header}.
    */
    public static class BasicHeader implements Header {
        private final String name;
        private final String value;

        /**
         * @param name the name of the header; cannot be {@code null} or empty
         * @param value the value of the header; may be {@code null}
         */
        public BasicHeader(String name, String value) {
            Objects.requireNonNull(name, "name cannot be null");
            if ("".equals(name)) {
               throw new IllegalArgumentException("name cannot be empty");
            }
            this.name = name;
            this.value = value;
        }

        @Override
        public String getName() {
            return name;
        }

        @Override
        public String getValue() {
            return value;
        }

        @Override
        public String toString() {
            return String.format("{0}/{1}", name, value);
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + name.hashCode();
            result = prime * result + ((value == null) ? 0 : value.hashCode());
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            BasicHeader other = (BasicHeader) obj;
            if (!name.equals(other.name))
                return false;
            if (value == null) {
                if (other.value != null)
                    return false;
            } else if (!value.equals(other.value))
                return false;
            return true;
        }
    }
}
