/* **********************************************************
 * Copyright (c) 2018 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.client.http;

import java.util.List;
import java.util.Objects;

import com.vmware.vapi.core.HttpResponse;
import com.vmware.vapi.core.HttpResponseAccessor;

/**
 * Extractor which captures particular singleton HTTP header from the raw
 * REST response.
 *
 * <p>This extractor assumes that the HTTP header occurs only once in the
 * HTTP response. If it occurs multiple times, the value of only one of the
 * instances will be returned.
 */
public class SingleHeaderExtractor implements HttpResponseAccessor {
    private final String headerName;
    private String singletonValue;

    /**
     * Constructor.
     *
     * @param headerName name of the header to be extracted; must not be
     * {@code null}.
     */
    public SingleHeaderExtractor(String headerName) {
        this.headerName = Objects.requireNonNull(headerName);
    }

    @Override
    public void access(HttpResponse response) {
        List<String> headerValues = response.getHeader(headerName);
        if (headerValues.size() > 0) {
            singletonValue = headerValues.get(0);
        }
    }

    /**
     * Returns the captured HTTP header value.
     *
     * @return the value of a single isntance of the specified HTTP header
     */
    public String getHeaderValue() {
        return this.singletonValue;
    }
}
