/* ************************************************************************
 * Copyright 2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * ************************************************************************/
package com.vmware.vapi.protocol.common.json;

import com.fasterxml.jackson.core.JsonFactory;
import com.vmware.vapi.CoreException;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.protocol.common.DataValueBuilder;
import com.vmware.vapi.internal.protocol.common.DirectDeserializer;
import com.vmware.vapi.internal.protocol.common.DirectSerializer;
import com.vmware.vapi.internal.protocol.common.SerializerConfig;
import com.vmware.vapi.internal.protocol.common.json.JsonDirectDeserializer;
import com.vmware.vapi.internal.protocol.common.json.JsonDirectSerializer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;

/**
 * Converter which encodes {@code DataValue} to and from simple JSON representation.
 *
 * <p>This is the JSON representation which is used in REST rendering of the APIs.
 */
public class JsonConverter {

    private final DirectSerializer directSerializer;
    private final DirectDeserializer directDeserializer;
    private final SerializerConfig serializerConfig;

    public JsonConverter() {
        JsonFactory factory = new JsonFactory();
        directSerializer = new JsonDirectSerializer(factory);
        directDeserializer = new JsonDirectDeserializer(factory);
        serializerConfig = new SerializerConfig(false);
    }

    /**
     * This method converts a JSON string representation into a concrete {@code DataValue} object.
     *
     * @param json a {@code String} representation of a JSON
     * @return Returns the concrete {@code DataValue}
     * @throws CoreException if the provided {@code InputStream} cannot be deserialized
     */
    public DataValue toDataValue(String json) {
        byte[] buf = json.getBytes(StandardCharsets.UTF_8);
        InputStream input = new ByteArrayInputStream(buf);

        return toDataValue(input);
    }

    /**
     * This method converts a JSON passed as an {@code InputStream} into a concrete {@code DataValue} object.
     *
     * @param input the {@code InputStream} we read the JSON from, which will be closed before the method returns
     *              or throws a {@code CoreException}
     * @return Returns the concrete {@code DataValue}
     * @throws CoreException if the provided {@code InputStream} cannot be deserialized
     */
    public DataValue toDataValue(InputStream input) {
        DataValueBuilder builder = new DataValueBuilder();

        try {
            directDeserializer.deserialize(builder, input);
        } catch (IOException e) {
            throw toVapiCoreException(e, "vapi.json.deserialize.ioerror");
        }

        return builder.getResult();
    }

    /**
     * This method converts the {@code DataValue} object to a JSON string.
     *
     * @param value a {@code DataValue} object we want to convert to JSON
     * @return the JSON string representation of a {@code DataValue}
     * @throws CoreException if the provided {@code DataValue} cannot be serialized
     */
    public String fromDataValue(DataValue value) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

        fromDataValue(value, outputStream);

        return new String(outputStream.toByteArray(), StandardCharsets.UTF_8);
    }

    /**
     * This method converts the {@code DataValue} Object to a {@code ByteArrayOutputStream}.
     *
     * @param value a {@code DataValue}
     * @param outputStream stream to write the serialized result to, which will be closed before the method returns
     *                     or throws a {@code CoreException}
     * @throws CoreException if the provided {@code DataValue} cannot be serialized
     */
    public void fromDataValue(DataValue value, OutputStream outputStream) {
        try {
            directSerializer.serialize(value, outputStream, serializerConfig);
        } catch (IOException e) {
            throw toVapiCoreException(e, "vapi.json.serialize.ioerror");
        }
    }

    private static CoreException toVapiCoreException(Exception ex, String id) {
        return new CoreException(MessageFactory.getMessage(id, ex.getMessage()), ex);
    }
}
