/* **********************************************************
 * Copyright (c) 2012, 2017, 2019, 2022-2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.protocol.server.rpc;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import com.vmware.vapi.internal.tracing.TracingSpan;

/**
 * Receiver of requests.
 */
public interface RequestReceiver {

    /*
     * This abstraction is roughly similar in purpose to javax.servlet.Servlet.
     * The transport context is similar to javax.servlet.AsyncContext.
     */

    /**
     * Transport handle which can be used to send responses or retrieve
     * transport specific information.
     */
    interface TransportContext {

        /**
         * Sends a response frame for the request which is associated with this
         * transport handle. If the response is flagged as final, the
         * handling of the request is considered complete and no more response
         * frames can be sent. For each request there must be exactly one
         * response frame flagged as final.
         *
         * @param response response frame; must not be <code>null</code>
         * @param responseLength length of the request as number of bytes; this
         *                       is just a hint to the implementation, pass
         *                       <code>-1</code> if the length is unknown
         * @param isFinal whether this is the final response frame for the
         *                request
         * @throws IOException transport error
         */
        void send(InputStream response, int responseLength, boolean isFinal)
                throws IOException;

        /**
         * Gets information about the client request.
         *
         * @return the request context.
         */
        RequestContext getRequestContext();

        /**
         * Sets a response header with the given name and value. If the header
         * has already been set, the new value overwrites the previous one.
         * Case-sensitivity of the header name depends on the underlying
         * transport (e.g. header names are case insensitive for HTTP 1.1).
         *
         * @param name the name of the header
         * @param value the header value
         */
        void setHeader(String name, String value);
    }

    /**
     * Auxiliary (optional) information about the request which triggered the
     * API invocation.
     */
    interface RequestContext {
        /**
         * Gets the identifier of the user agent.
         *
         * @return {@code String} representing the user agent, or {@code null}
         *         if one is not available in the context
         */
        String getUserAgent();

        /**
         * Gets the identifier of the vAPI session.
         *
         * @return {@code String} representing the session, or {@code null}
         *         if one is not available in the context
         */
        String getSession();

        /**
         * Gets the identifier of the service.
         *
         * @return {@code String} representing the service, or {@code null}
         *         if one is not available in the context
         */
        String getServiceId();

        /**
         * Gets the identifier of the operation.
         *
         * @return {@code String} representing the operation, or {@code null}
         *         if one is not available in the context
         */
        String getOperationId();

        /**
         * Gets the identifier of the accept language.
         *
         * @return {@code String} representing the accept language, or {@code null}
         *         if one is not available in the context
         */
        String getAcceptLanguage();

        /**
         * Gets the version of jsonrpc protocol.
         *
         * @return {@code String} jsonrpc or jsonrpc1.1
         */
        String getJsonRpcVersion();

        /**
         * Gets a mapping between all incoming auxiliary property names as keys
         * and their respective values.
         *
         * <p>In case the underlying transport allows for multiple values per
         * property (like the <tt>HTTP</tt> headers do), it is the implementation's
         * responsibility to combine the values into a single {@link String}
         * instance according to the transport's rules.</p>
         *
         * @return map of all properties; never {@code null}
         */
        Map<String, String> getAllProperties();

        /**
         * @return The tracing span associated with this request
         */
        default TracingSpan getTracingSpan() {
            return TracingSpan.NO_OP;
        };
    }

    /**
     * <p>
     * Notifies the request receiver that a new request is received.
     * </p>
     * <p>
     * There is no guarantee that the request is completely handled when this
     * method exits. The implementation is free to handle the request
     * asynchronously. Only after the {@link TransportContext#send} with
     * {@code true} value for the {@code isFinal} parameter is invoked, it is
     * certain that the request is completely processed and all responses have
     * been sent.
     * </p>
     *
     * @param request request frame; must not be <code>null</code>
     * @param transport transport handle which can be used to send responses;
     *                  this handle is strongly correlated with the request;
     *                  use it to send responses only for this particular
     *                  request; must not be <code>null</code>
     * @throws IOException propagate an error coming from the transport module
     *                     (e.g. thrown by the request stream or by the
     *                     transport handle)
     */
    void requestReceived(InputStream request, TransportContext transport)
            throws IOException;

}
