/* **********************************************************
 * Copyright 2009-2014 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.server.rpc.http;


/**
 * An HTTP server (host, port, protocol) triple plus connector specific
 * configuration
 */

public interface Endpoint {

   /** Describes the endpoint's protocol */
   public enum Protocol {HTTP, HTTPS};

   /** Describe the endpoint's I/O connector type */
   public enum EndpointType {
      BLOCKING,
      NON_BLOCKING_NIO,
      /* BLOCKING_NIO - not implemented */
   };

   /**
    * Returns the endpoint's port number.
    * @return The endpoint's port number.
    */
   public int getPort();

   /**
    * Returns the endpoint's host.
    * @return The endpoint's host.
    */
   public String getHost();

   /**
    * Returns the endpoint's protocol.
    * @return The endpoint's protocol.
    */
   public Protocol getProtocol();

   /**
    * Get the number of thread dedicated to accepting incoming connections.
    *
    * @return thread number
    */
   int getNumAcceptors();

   /**
    * Set the number of thread dedicated to accepting incoming connections.
    *
    * @param acceptors
    *           thread number. Defaults to 1
    */
   void setNumAcceptors(int acceptors);

   /**
    * Get number of connection requests that can be queued up before the
    * operating system starts to send rejections.
    *
    * @return max number of queued requests
    */
   int getAcceptQueueSize();

   /**
    * Set number of connection requests that can be queued up before the
    * operating system starts to send rejections.
    *
    * @param acceptQueueSize
    *           max number of queued requests. Defaults to 0. For the semantics
    *           see backlog parameter in
    *           {@link java.net.ServerSocket#bind(java.net.SocketAddress, int)}
    */
   void setAcceptQueueSize(int acceptQueueSize);

   /**
    * Set the maximum Idle time for a connection, which roughly translates
    * to the {@link java.net.Socket#setSoTimeout(int)} call but depends on
    * the application server used and how it is configured.
    * The max idle time is applied:<ul>
    * <li>When waiting for a new request to be received on a connection</li>
    * <li>When reading the headers and content of a request</li>
    * <li>When writing the headers and content of a response</li>
    * </ul>
    * <p>This value MUST be bigger than the keep alive period on the client.
    * <p>Default value is 200000 ms
    *
    * @param maxIdleTime The maxIdleTime to set.
    */
   public void setMaxIdleTime(int maxIdleTime);

   /**
    * Default valule is 200000 ms
    *
    * @return Returns the maxIdleTime.
    */
   public int getMaxIdleTime();

   /**
    * Set the endpoint connector I/O type (like blocking or  NIO).
    * Default is {@link EndpointType#BLOCKING}.
    *
    * @param type the I/O type to use
    */
   public void setEndpointType(EndpointType type);

   /**
    * Get the endpoint connector I/O type.
    *
    * @return <code>EndpointType</code> representing the type
    */
   public EndpointType getEndpointType();
}
