/* **********************************************************
 * Copyright 2012-2014, 2017 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.provider;

import static com.vmware.vapi.internal.util.TaskUtil.isTaskInvocation;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.ErrorValueFactory;
import com.vmware.vapi.bindings.server.InvocationContext;
import com.vmware.vapi.bindings.server.impl.InvocationContextImpl;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.InterfaceDefinition;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodDefinition;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.internal.provider.ApiTaskMethod;
import com.vmware.vapi.std.StandardDataFactory;

/**
 * Base {@link ApiInterface} implementation that uses {@link ApiMethod}
 */
public class ApiMethodBasedApiInterface implements ApiInterface {
    private final Map<MethodIdentifier, ApiMethod> methods;
    private final InterfaceIdentifier ifaceId;
    private final InterfaceDefinition definition;

    /**
     * Constructor.
     *
     * @param iface interface identifier
     */
    public ApiMethodBasedApiInterface(InterfaceIdentifier iface) {
        methods = new HashMap<MethodIdentifier, ApiMethod>();
        ifaceId = iface;
        definition = new InterfaceDefinition(ifaceId,
                Collections.unmodifiableSet(methods.keySet()));
    }

    /**
     * Constructor.
     *
     * @param iface interface name
     */
    public ApiMethodBasedApiInterface(String iface) {
        this(new InterfaceIdentifier(iface));
    }

    @Override
    public InterfaceIdentifier getIdentifier() {
        return ifaceId;
    }

    @Override
    public InterfaceDefinition getDefinition() {
        return definition;
    }

    @Override
    public MethodDefinition getMethodDefinition(MethodIdentifier methodId) {
        ApiMethod method = getMethod(methodId);
        if (method != null) {
            return method.getDefinition();
        }

        return null;
    }

    @Override
    public void invoke(ExecutionContext ctx,
                       MethodIdentifier methodId,
                       DataValue input,
                       AsyncHandle<MethodResult> asyncHandle) {
        ApiMethod method = getMethod(methodId);
        if (method != null) {
            InvocationContext invocationContext =
                    new InvocationContextImpl(ctx);
            method.invoke(invocationContext, input, asyncHandle);
        } else {
            String operationId = methodId.getName();
            String serviceId = methodId.getInterfaceIdentifier().getName();
            ErrorValue error = ErrorValueFactory.buildErrorValue(
                            StandardDataFactory.OPERATION_NOT_FOUND,
                            "vapi.method.input.invalid.method",
                            operationId,
                            serviceId);
            MethodResult result = MethodResult.newErrorResult(error);
            asyncHandle.setResult(result);
        }
    }

    /**
     * Get {@link ApiMethod} for the given {@link MethodIdentifier}
     * @param methodId method identifier
     *
     * @return {@code ApiMethod}
     */
    private ApiMethod getMethod(MethodIdentifier methodId) {
        return methods.get(methodId);
    }

    /**
     * Registers an <code>ApiMethod</code> with this <code>ApiInterface</code>.
     *
     * @param method the API method to register
     */
    protected void registerMethod(ApiMethod method) {
        MethodIdentifier methodId = method.getIdentifier();
        if (method.getDefinition().getTaskSupport() !=
                MethodDefinition.TaskSupport.TASK_ONLY) {
            methods.put(methodId, method);
        }
        if (method.getDefinition().getTaskSupport() !=
                MethodDefinition.TaskSupport.NONE) {
            ApiMethod taskMethod = new ApiTaskMethod(method);
            methods.put(taskMethod.getIdentifier(), taskMethod);
        }
    }
}
