/* **********************************************************
 * Copyright 2013, 2019, 2021 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.provider.introspection;

import java.util.Set;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.DecoratorApiProvider;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorDefinition;
import com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.util.async.DecoratorAsyncHandle;

/**
 * Decorator which augments the set of error definitions returned by the
 * operation introspection service.
 */
public final class ErrorAugmentingFilter extends DecoratorApiProvider {
    private static final String OPERATION_INTROSPECTION_SERVICE_NAME =
            IntrospectionDataFactory.OPERATION_GET_METHOD_ID
                    .getInterfaceIdentifier().getName();
    private static final String OPERATION_INTROSPECTION_GET_METHOD_NAME =
            IntrospectionDataFactory.OPERATION_GET_METHOD_ID.getName();

    private final Iterable<DataValue> extraErrors;

    /**
     * Creates a filter for a fixed set of error definitions.
     *
     * @param decoratedProvider decorated provider
     * @param extraErrors extra error definitions; must not be <code>null</code>
     */
    public ErrorAugmentingFilter(ApiProvider decoratedProvider,
            Set<ErrorDefinition> extraErrors) {
        super(decoratedProvider);
        Validate.notNull(extraErrors);
        this.extraErrors = IntrospectionDataFactory
                .toDataDefinitionList(extraErrors);
    }

    @Override
    public void invoke(String serviceId,
                       String operationId,
                       DataValue input,
                       ExecutionContext ctx,
                       AsyncHandle<MethodResult> asyncHandle) {

        if (OPERATION_INTROSPECTION_SERVICE_NAME.equals(serviceId)
              && OPERATION_INTROSPECTION_GET_METHOD_NAME.equals(operationId)) {
            asyncHandle = addErrorsToResponse(asyncHandle);
        }

        decoratedProvider.invoke(serviceId, operationId, input, ctx, asyncHandle);
    }

    private AsyncHandle<MethodResult> addErrorsToResponse(
                                        AsyncHandle<MethodResult> asyncHandle) {

        return new DecoratorAsyncHandle<MethodResult>(asyncHandle) {
            @Override
            public void setResult(MethodResult result) {
                if (result.success()) {
                    try {
                        IntrospectionDataFactory.addErrorsToOperationInfo(
                                result.getOutput(), extraErrors);
                    } catch (RuntimeException ex) {
                        decorated.setError(ex);
                        return;
                    }
                }
                decorated.setResult(result);
            }
        };
    }
}
