/* **********************************************************
 * Copyright 2012-2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.security;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.vmware.vapi.internal.util.Validate;

/**
 * Implementations of this interface will provide the authentication
 * configuration data
 */
public interface AuthenticationConfig {

    /**
     * @return the package authentication rules table. cannot be null.
     */
    Map<String, List<AuthnScheme>> getPackageAuthenticationRules();

    /**
     * @return the interface authentication rules table. cannot be null.
     */
    Map<String, List<AuthnScheme>> getIFaceAuthenticationRules();

    /**
     * @return the operation authentication rules table. cannot be null.
     */
    Map<String, List<AuthnScheme>> getOperationAuthenticationRules();

    /**
     * This class representations an authentication scheme
     */
    final class AuthnScheme {
        private static final AuthnScheme NO_AUTHN_SCHEME = new AuthnScheme(
                Collections.singletonList(StdSecuritySchemes.NO_AUTHN));
        // TODO is it necessary to keep a list here? currently the list could
        // contain only 0, 1 or 2 elements.
        private final List<String> authnSchemeList;

        /**
         * @param authnSchemeList the allowed authentication schemes for this
         *        instance. cannot be empty.
         */
        public AuthnScheme(List<String> authnSchemeList) {
            Validate.notNull(authnSchemeList);
            Validate.notEmpty(authnSchemeList);
            Validate.noNullElements(authnSchemeList);

            this.authnSchemeList = Collections
                    .unmodifiableList(authnSchemeList);
        }

        /**
         * @param userScheme cannot be null
         * @return true if the user scheme matches some of the allowed
         *         authentication schemes for this instance, false otherwise
         */
        public boolean isAllowed(AuthnScheme userScheme) {
            Validate.notNull(userScheme);

            for (String schemeId : authnSchemeList) {
                for (String userSchemeId : userScheme.authnSchemeList) {
                    if (schemeId.equalsIgnoreCase(userSchemeId)) {
                        return true;
                    }
                }
            }

            return false;
        }

        /**
         * @return the authentication scheme that represents a NoAuthentication
         *         scheme
         */
        public static AuthnScheme getNoAuthenticationScheme() {
            return NO_AUTHN_SCHEME;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("AuthenticationScheme[");
            for (String scheme : authnSchemeList) {
                sb.append(scheme).append(", ");
            }
            // authnSchemeList cannot be empty so at least one scheme will be
            // present i.e. one trailing ',' should be always deleted
            sb.delete(sb.length() - 2, sb.length());
            sb.append("]");
            return sb.toString();
        }

        @Override
        public int hashCode() {
            return authnSchemeList.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            AuthnScheme other = (AuthnScheme) obj;
            return authnSchemeList.equals(other.authnSchemeList);
        }
    }
}
