/* **********************************************************
 * Copyright 2013 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.security;

import java.util.List;

import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;

/**
 * Implementations of this interface should be able to verify the authentication
 * data provided in the security context against an identity source.
 */
public interface AuthenticationHandler {

    /**
     * Verifies the provided authentication data against the relevant identity
     * source.
     *
     * @param ctx the {@link SecurityContext} which carries the authentication
     *            data. Cannot be <code>null</code>.
     * @param asyncHandle Cannot be <code>null</code>
     */
    void authenticate(SecurityContext ctx, AsyncHandle<AuthenticationResult> asyncHandle);

    /**
     * @return list of the supported authentication scheme by this handler. A list of
     *         the standard authentication schemes can be found at
     *         {@link StdSecuritySchemes}. Cannot be <code>null</code> or empty.
     */
    List<String> supportedAuthenticationSchemes();

    /**
     * Implementations of this class will contain the needed authentication
     * result on a successful authentication.
     */
    abstract class AuthenticationResult {

        /**
         * @return the authenticated principal. cannot be null.
         */
        public abstract PrincipalId getUser();

        /**
         * @return the authenticated principal groups. can be null or empty.
         */
        public abstract List<PrincipalId> getGroups();

        /**
         * @return a security context that will be used to overwrite the context
         *         received from the wire. Returning <code>null</code> will mean
         *         that the wire context will be delivered to the service
         *         implementation.
         */
        public SecurityContext getSecurityContext() {
            return null;
        }
    }
}
