/* **********************************************************
 * Copyright 2022-2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.tracing;

import java.util.Objects;

import com.vmware.vapi.internal.tracing.otel.OtelTracer;

import io.opentelemetry.api.OpenTelemetry;

/**
 * This factory creates {@link Tracer} tracer instances.
 * In order to enable tracing in the vapi stack, a tracer must be configured.
 * <p>
 * Example usage in a vapi client:
 * <pre>{@code
 * Tracer tracer = new OpenTelemetryTracerBuilder(openTelemetry)
 *                 .setTracingLevel(TracingLevel.INFO)
 *                 .build();
 * ClientConfiguration clientConfig = new ClientConfiguration
 *                                    .Builder()
 *                                    .setTracer(tracer)
 *                                    .getConfig();
 * }
 * </pre>
 * Example usage in a vapi server:
 * <pre>{@code
 * Tracer tracer = new OpenTelemetryTracerBuilder(openTelemetry)
 *                 .setTracingLevel(TracingLevel.INFO)
 *                 .build();
 * SimpleConfigurator configurator = new SimpleConfigurator();
 * configurator.setTracer(tracer);
 * }
 * </pre>
 * @see {@link Tracer}
 */
public class OpenTelemetryTracerBuilder {
    private final OpenTelemetry oTel;
    private TracingLevel tracingLevel = TracingLevel.INFO;

    /**
     * Builds a new {@link Tracer}
     *
     * @return the configured tracer
     */
    public Tracer build() {
        return new OtelTracer(oTel, tracingLevel);
    }

    /**
     * @param openTelemetry the OpenTelemetry instance to use
     */
    public OpenTelemetryTracerBuilder(OpenTelemetry openTelemetry) {
        this.oTel = Objects.requireNonNull(openTelemetry, "openTelemetry");
    }

    /**
     * Sets the {@link TracingLevel} for the {@link Tracer} that will be built.
     * Default value is {@link TracingLevel#INFO}
     *
     * @param tracingLevel the tracing level to set
     *
     * @return the current object
     */
    public OpenTelemetryTracerBuilder setTracingLevel(TracingLevel tracingLevel) {
        this.tracingLevel = Objects.requireNonNull(tracingLevel,
                                                   "tracingLevel");
        return this;
    }
}
