# ***************************************************************************
# Copyright 2019 VMware, Inc.  All rights reserved.
# VMware Confidential
# ***************************************************************************

<#
.SYNOPSIS
    This script is a wrapper to run the setup.exe file silently.
    The script waits for the setup to complete before returning.
.DESCRIPTION
    This install script installs Windows BareMetal Server Host
    Agents Installer
.EXAMPLE
    Following are some examples to run the bootstrapper.

    Install:
        powershell c:\Users\root\Desktop\install_helper.ps1 -operation install -setupFile
        "C:\\VMware-NSX-3.0.0.25136500_baremetal-server_setup.exe" -installPath "C:\NSX"

    Uninstall:
        powershell c:\Users\root\Desktop\install_helper.ps1 -operation uninstall -setupFile
        "C:\\VMware-NSX-3.0.0.25136500_baremetal-server_setup.exe"
#>


param(
    # install, uninstall
    [Parameter(Mandatory=$true)][string]$operation,

    # path to setup.exe file
    [Parameter(Mandatory=$true)][string]$setupFile,

    #optional
    [string]$async = "false",

    # optional
    [string]$installPath = "",

    #optional
    [string]$logDir = [environment]::GetEnvironmentVariable("PROGRAMDATA") + "\VMware\NSX\Logs"
)


$NSX_DATA_DIR = [environment]::GetEnvironmentVariable("PROGRAMDATA") + "\VMware\NSX\Data"

$installFile = [io.path]::GetFileNameWithoutExtension($setupFile);

if (-Not (Test-Path $setupFile)) {
    echo "Failed to find $setupFile"
    exit 1
}


$version = (Get-Item $setupFile).VersionInfo.ProductVersion
$logFile = [io.path]::combine($logDir, "setup_" + $operation + "_$version.log")

if (-Not (Test-Path $logDir)) {
    New-Item $logDir -type directory -Force | Out-Null
}

$msiArgs = "/qn"

if ($installPath -ne "") {
    echo "Custom installPath $installPath"
    $msiArgs = $msiArgs + " INSTALLDIR=""$installPath"""
}
if ($operation -eq "install") {
    $args = @('/s', '/v', $msiArgs, '/l', "$logFile")
    & $setupFile $args
}
elseif ($operation -eq "uninstall") {
    $cmd = "/s /v ""/qn REMOVE=ALL"" /l $logFile"
    iex "& $setupFile $cmd"
}
else {
    echo "Invalid operation $operation"
    exit 1
}

if ($async -eq $false) {
    echo "Monitoring $installFile for $operation"
    $a = get-process $installFile
    $handle = $a.Handle

    do {
       echo "Waiting ....."

    } while ($a.waitforexit())

    if ($a.ExitCode -eq 0) {
        echo "Done $operation of $installFile"
    } else {
        echo "Failed to $operation. ExitCode=$($a.ExitCode)"
    }
    exit $a.ExitCode
} else {
    echo "Running silent $operation of $installFile in background"
}

# SIG # Begin signature block
# MIIoTQYJKoZIhvcNAQcCoIIoPjCCKDoCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUD8m6oWWaQ0/S8Ssj7J/DOlvH
# rXiggiF9MIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0B
# AQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz
# 7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS
# 5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
# bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfI
# SKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jH
# trHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14
# Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2
# h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt
# 6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPR
# iQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ER
# ElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4K
# Jpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAd
# BgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SS
# y4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAC
# hjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURS
# b290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRV
# HSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyh
# hyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO
# 0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo
# 8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++h
# UD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5x
# aiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMIIGrjCCBJag
# AwIBAgIQBzY3tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQG
# EwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNl
# cnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIw
# MzIzMDAwMDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUGA1UE
# ChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQg
# UlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyibaCw
# zIP5WvYRoUQVQl+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITaEfFz
# sbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2UKdPZ
# 7Gnf2ZCHRgB720RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7
# QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15/teP
# c5OsLDnipUjW8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCY
# OjgRs/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3NDzt9K
# oRxrOMUp88qqlnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6
# dSgkQe1CvwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmyMKVM
# 1+mYSlg+0wOI/rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4KbhPvbC
# dLI/Hgl27KtdRnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0zbEC
# AwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW2W1N
# hS9zKXaaL3WMaiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9P
# MA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEFBQcB
# AQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggr
# BgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1
# c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkwFzAI
# BgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7Zv
# mKlEIgF+ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfpPmDI
# 2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9Hd/ty
# dBTX/6tPiix6q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+ovHVP
# ulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG33irr9p6xeZmB
# o1aGqwpFyd/EjaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR8XKc
# 6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcTJM3c
# HXg65J6t5TRxktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0d
# KNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw8De/mADfIBZP
# J/tgZxahZrrdVcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLe
# Mt8EifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A+sDy
# Divl1vupL0QVSucTDh3bNzgaoSv27dZ8/DCCBrAwggSYoAMCAQICEAitQLJg0pxM
# n17Nqb2TrtkwDQYJKoZIhvcNAQEMBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoT
# DERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UE
# AxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIxMDQyOTAwMDAwMFoXDTM2
# MDQyODIzNTk1OVowaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2lnbmluZyBS
# U0E0MDk2IFNIQTM4NCAyMDIxIENBMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBANW0L0LQKK14t13VOVkbsYhC9TOM6z2Bl3DFu8SFJjCfpI5o2Fz16zQk
# B+FLT9N4Q/QX1x7a+dLVZxpSTw6hV/yImcGRzIEDPk1wJGSzjeIIfTR9TIBXEmtD
# mpnyxTsf8u/LR1oTpkyzASAl8xDTi7L7CPCK4J0JwGWn+piASTWHPVEZ6JAheEUu
# oZ8s4RjCGszF7pNJcEIyj/vG6hzzZWiRok1MghFIUmjeEL0UV13oGBNlxX+yT4Us
# SKRWhDXW+S6cqgAV0Tf+GgaUwnzI6hsy5srC9KejAw50pa85tqtgEuPo1rn3MeHc
# reQYoNjBI0dHs6EPbqOrbZgGgxu3amct0r1EGpIQgY+wOwnXx5syWsL/amBUi0nB
# k+3htFzgb+sm+YzVsvk4EObqzpH1vtP7b5NhNFy8k0UogzYqZihfsHPOiyYlBrKD
# 1Fz2FRlM7WLgXjPy6OjsCqewAyuRsjZ5vvetCB51pmXMu+NIUPN3kRr+21CiRshh
# WJj1fAIWPIMorTmG7NS3DVPQ+EfmdTCN7DCTdhSmW0tddGFNPxKRdt6/WMtyEClB
# 8NXFbSZ2aBFBE1ia3CYrAfSJTVnbeM+BSj5AR1/JgVBzhRAjIVlgimRUwcwhGug4
# GXxmHM14OEUwmU//Y09Mu6oNCFNBfFg9R7P6tuyMMgkCzGw8DFYRAgMBAAGjggFZ
# MIIBVTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBRoN+Drtjv4XxGG+/5h
# ewiIZfROQjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8B
# Af8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYIKwYBBQUHAQEEazBpMCQG
# CCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKG
# NWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290
# RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
# bS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMBwGA1UdIAQVMBMwBwYFZ4EMAQMw
# CAYGZ4EMAQQBMA0GCSqGSIb3DQEBDAUAA4ICAQA6I0Q9jQh27o+8OpnTVuACGqX4
# SDTzLLbmdGb3lHKxAMqvbDAnExKekESfS/2eo3wm1Te8Ol1IbZXVP0n0J7sWgUVQ
# /Zy9toXgdn43ccsi91qqkM/1k2rj6yDR1VB5iJqKisG2vaFIGH7c2IAaERkYzWGZ
# gVb2yeN258TkG19D+D6U/3Y5PZ7Umc9K3SjrXyahlVhI1Rr+1yc//ZDRdobdHLBg
# XPMNqO7giaG9OeE4Ttpuuzad++UhU1rDyulq8aI+20O4M8hPOBSSmfXdzlRt2V0C
# FB9AM3wD4pWywiF1c1LLRtjENByipUuNzW92NyyFPxrOJukYvpAHsEN/lYgggnDw
# zMrv/Sk1XB+JOFX3N4qLCaHLC+kxGv8uGVw5ceG+nKcKBtYmZ7eS5k5f3nqsSc8u
# pHSSrds8pJyGH+PBVhsrI/+PteqIe3Br5qC6/To/RabE6BaRUotBwEiES5ZNq0RA
# 443wFSjO7fEYVgcqLxDEDAhkPDOPriiMPMuPiAsNvzv0zh57ju+168u38HcT5uco
# P6wSrqUvImxB+YJcFWbMbA7KxYbD9iYzDAdLoNMHAmpqQDBISzSoUSC7rRuFCOJZ
# DW3KBVAr6kocnqX9oKcfBnTn8tZSkP2vhUgh+Vc7tJwD7YZF9LRhbr9o4iZghurI
# r6n+lB3nYxs6hlZ4TjCCBsIwggSqoAMCAQICEAVEr/OUnQg5pr/bP1/lYRYwDQYJ
# KoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2
# IFRpbWVTdGFtcGluZyBDQTAeFw0yMzA3MTQwMDAwMDBaFw0zNDEwMTMyMzU5NTla
# MEgxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjEgMB4GA1UE
# AxMXRGlnaUNlcnQgVGltZXN0YW1wIDIwMjMwggIiMA0GCSqGSIb3DQEBAQUAA4IC
# DwAwggIKAoICAQCjU0WHHYOOW6w+VLMj4M+f1+XS512hDgncL0ijl3o7Kpxn3GIV
# WMGpkxGnzaqyat0QKYoeYmNp01icNXG/OpfrlFCPHCDqx5o7L5Zm42nnaf5bw9Yr
# IBzBl5S0pVCB8s/LB6YwaMqDQtr8fwkklKSCGtpqutg7yl3eGRiF+0XqDWFsnf5x
# XsQGmjzwxS55DxtmUuPI1j5f2kPThPXQx/ZILV5FdZZ1/t0QoRuDwbjmUpW1R9d4
# KTlr4HhZl+NEK0rVlc7vCBfqgmRN/yPjyobutKQhZHDr1eWg2mOzLukF7qr2JPUd
# vJscsrdf3/Dudn0xmWVHVZ1KJC+sK5e+n+T9e3M+Mu5SNPvUu+vUoCw0m+PebmQZ
# BzcBkQ8ctVHNqkxmg4hoYru8QRt4GW3k2Q/gWEH72LEs4VGvtK0VBhTqYggT02ke
# fGRNnQ/fztFejKqrUBXJs8q818Q7aESjpTtC/XN97t0K/3k0EH6mXApYTAA+hWl1
# x4Nk1nXNjxJ2VqUk+tfEayG66B80mC866msBsPf7Kobse1I4qZgJoXGybHGvPrhv
# ltXhEBP+YUcKjP7wtsfVx95sJPC/QoLKoHE9nJKTBLRpcCcNT7e1NtHJXwikcKPs
# CvERLmTgyyIryvEoEyFJUX4GZtM7vvrrkTjYUQfKlLfiUKHzOtOKg8tAewIDAQAB
# o4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/
# BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcB
# MB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshvMB0GA1UdDgQWBBSltu8T
# 5+/N0GSh1VapZTGj3tXjSTBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0MDk2U0hBMjU2VGltZVN0
# YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAkBggrBgEFBQcwAYYYaHR0
# cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAChkxodHRwOi8vY2FjZXJ0
# cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0MDk2U0hBMjU2VGlt
# ZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4ICAQCBGtbeoKm1mBe8cI1P
# ijxonNgl/8ss5M3qXSKS7IwiAqm4z4Co2efjxe0mgopxLxjdTrbebNfhYJwr7e09
# SI64a7p8Xb3CYTdoSXej65CqEtcnhfOOHpLawkA4n13IoC4leCWdKgV6hCmYtld5
# j9smViuw86e9NwzYmHZPVrlSwradOKmB521BXIxp0bkrxMZ7z5z6eOKTGnaiaXXT
# UOREEr4gDZ6pRND45Ul3CFohxbTPmJUaVLq5vMFpGbrPFvKDNzRusEEm3d5al08z
# jdSNd311RaGlWCZqA0Xe2VC1UIyvVr1MxeFGxSjTredDAHDezJieGYkD6tSRN+9N
# UvPJYCHEVkft2hFLjDLDiOZY4rbbPvlfsELWj+MXkdGqwFXjhr+sJyxB0JozSqg2
# 1Llyln6XeThIX8rC3D0y33XWNmdaifj2p8flTzU8AL2+nCpseQHc2kTmOt44Owde
# OVj0fHMxVaCAEcsUDH6uvP6k63llqmjWIso765qCNVcoFstp8jKastLYOrixRoZr
# uhf9xHdsFWyuq69zOuhJRrfVf8y2OMDY7Bz1tqG4QyzfTkx9HmhwwHcK1ALgXGC7
# KP845VJa1qwXIiNO9OzTF/tQa/8Hdx9xl0RBybhG02wyfFgvZ0dl5Rtztpn5aywG
# Ru9BHvDwX+Db2a2QgESvgBBBijCCB7wwggWkoAMCAQICEAZAAlv/DEiXMpnUYyCD
# w3swDQYJKoZIhvcNAQELBQAwaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
# ZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2ln
# bmluZyBSU0E0MDk2IFNIQTM4NCAyMDIxIENBMTAeFw0yNDAzMjgwMDAwMDBaFw0y
# NzAzMzAyMzU5NTlaMIHEMRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGC
# NzwCAQITCERlbGF3YXJlMR0wGwYDVQQPDBRQcml2YXRlIE9yZ2FuaXphdGlvbjEQ
# MA4GA1UEBRMHNjYxMDExNzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCkNhbGlmb3Ju
# aWExETAPBgNVBAcTCFNhbiBKb3NlMRUwEwYDVQQKEwxCcm9hZGNvbSBJbmMxFTAT
# BgNVBAMTDEJyb2FkY29tIEluYzCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoC
# ggIBALrwL6MnEp/Gn1zUDp+CHBI2Oe9X5jWbLZeZMcQ+aTsI4VNbDVVEFkG2//NR
# etJK2Vb1CqqnjzCT4f4fmgzSov2OxRDrAAmmaO3fcXfG8qpzx6HNVktdQQAJAFdO
# jfQeduFtxpwoDJf2wYHIaoqVHpwcesR5tJYyjE47pM6BiDTrF95jPGnHweoxCM4q
# A0RuHhybVjH4RH/gGZaZhy9UdH0q1fAlbGbNQkJpLk4H2Wnwmo7n64Ae+ka1AQBv
# f3+H+/Lb3SKwJjDXr+5os7PkSlyG/l+iBMFOHijpJdYdqbTcV8w1VuZsxW5yJPN8
# 9IO3SLnFBZi1WTBVT40YThdS61fjNtqKeDndBgo0Br41b0WJZxqgTATWpO+TuKWy
# 5w/N2oKKRCkfJpoYLMegOG95IuIkqApGIOp57wpd9r+GQKhq30C0KKHJ3z32QnWR
# KVM71+RyLn8J0rCbBdAgqHo4+54qv44nr1c/HRX5FpowCSRW0Vvv9XB5wWYWCxcW
# QAry6U9oZjIgU4Wosyofxm4qItU/keTEOBKK0lxZtpJ/z1Mm7E2qHS9XWTiLBpCl
# yZBmE8rj2k2WPmPRzLDwH7FNg0PCcq/KhaCooBvZx/NzqkIuTYYksHBqQpnlgoLn
# Rd2E+VLKuqUtNvmi1lGhhYUYovVKrHGsGNGqDiqnVG+7FFwFAgMBAAGjggICMIIB
# /jAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4EFgQUv1lM
# xL8+GC8addmbdc6uYOglIh8wPQYDVR0gBDYwNDAyBgVngQwBAzApMCcGCCsGAQUF
# BwIBFhtodHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/BAQDAgeA
# MBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0wgaowU6BRoE+GTWh0dHA6
# Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5n
# UlNBNDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRwOi8vY3JsNC5kaWdp
# Y2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEz
# ODQyMDIxQ0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYBBQUHMAGGGGh0
# dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBcBggrBgEFBQcwAoZQaHR0cDovL2NhY2Vy
# dHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0TBAIwADANBgkqhkiG9w0BAQsFAAOC
# AgEAwuQz/kFkHWPbGONH0iTINi0uupL0jO0SvOXiiB/Z1e4CAEk/UQqb8Q3ig6L8
# EfEtbRNW5yUZndoRrf+fdSUfGqa8qb7h6UPmiHiwQ6bv4nsQJXQeht5mFGIuZ3KV
# rUypvHDGnnoBW+Eb/76d/4BXPD3lzrvVreRv7pVcwPNkrWpxSHrPIuSVn+9+MqKm
# vL0qAxGxd3zUj49Gv732JEj7mlSwXuLGZLi9OV02Hav9VEUou1e1N59nbRDnkR+7
# /w158l98P/4suYezMHctGWC4Vg5Qaifurx5AqLbQ8xbo9BvxrPiafox8DVjbH4vs
# cUOLL/9Uw7LkOpwo4darP81mvv9UQ+3LCgtseVTA0nIVjTCdCHO88KDrtikMRgRE
# OeCC2l45yEqUmiospAierrEMtO1yj7KdHogedZFzP6hiGbpN29FyY/uCKKo5soQV
# dDAuyNIeypGW6TYVDgOEbrf7yDIPispBWjJs/ShsTsaLZFgc5PSrA+izoaSVmtqG
# 39xqlrvCizWA0pckvsboFoJdBYLJqRPvpk4k50fMeMnaf7IGApZYYG/Xzqes/md+
# 5G7S8kQcbvhzLmhHB7e+Z7rO09DL1VKBKs/6DrttnrI977PT1Ps1zNbhpetdsxR1
# NvTbMpCwrxYQp9y/MGZnXXlLAy5/eDfgLnJGE6xYid+iyYAxggY6MIIGNgIBATB9
# MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8GA1UE
# AxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcgUlNBNDA5NiBTSEEz
# ODQgMjAyMSBDQTECEAZAAlv/DEiXMpnUYyCDw3swCQYFKw4DAhoFAKBwMBAGCisG
# AQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQB
# gjcCAQsxDjAMBgorBgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBS29zUR+bXciPc4
# JZNQ+K5hOZokTDANBgkqhkiG9w0BAQEFAASCAgADl1w/VDE0gSrdgRBdHb32yBOf
# SitFhtNn9U2/sRLOV8BqYfhqPqinJDAw/UdWblAd2pM5W2Zqymo6y45opoT71nqf
# f+9P2cMY20K3Fx693f3zsQBJlm4304hGCr+PgXAQhvKyZzMDoyqtHDvGQogOTUKW
# JB521zJr7aS56Y9hHcHZpwB5t6lpZFNHID7usmbxgDWK7+u+fi93n7cre/ovc9KF
# 1PAZnMDrFV8w79io1W82AHvc5pl1WWq56e4hoEaTUTuJsmdYqNjGfWv1yVtYdXSM
# fv97J0YoFytzLlu306wrL+bt1vIJB0tKHKCBEFdsP9iQrozw29VYaNA+ZpcuJ3eZ
# oF02LdQ/esRKfKjk6OpvH3MkUzkQFSO25NEhlipudLXIw1XopFM0B9OrPzP/UpiE
# uRojCPZMG+3cDf64EvfcRgTq/qCMh3gg7zm6QuooUyNqMyq3KeZTmCa7mvPldP7E
# 49qwejk3a73A7WQPK/KSUeuewqpLsUtzGTKDEpa9urRgGYA/IwDD7kKJQYKtDioc
# NYxm+N46tZIm23rCG23PWx3OocNMeyqpOy4a7iua8BDATrPoiaFWAJwozdktCA/R
# B59v5/p2GV2jnwpuJ3E4egdyomboiB/z3U+IrhnUn7PX3DlUosQAOw21j6KByKgy
# xe9mukLTLpdwo0JDr6GCAyAwggMcBgkqhkiG9w0BCQYxggMNMIIDCQIBATB3MGMx
# CzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMy
# RGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcg
# Q0ECEAVEr/OUnQg5pr/bP1/lYRYwDQYJYIZIAWUDBAIBBQCgaTAYBgkqhkiG9w0B
# CQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0yNDA4MTQxMjUwMDdaMC8G
# CSqGSIb3DQEJBDEiBCBDWVi6U8Ldo67tAbVXYnoFWbBLo89w3oi/iDWdG1K8YTAN
# BgkqhkiG9w0BAQEFAASCAgAxLyLpmOPAdIFNbbZ7md3WSCkIQGRAXX2TwTzF1SE1
# 4ceA/K0o4Z1fUqI1W3j37LeyrqWld10X97CbueARljfTtpey9vW0RKGr5Wo6wlIY
# aWDkM4i5q7B71Jjy5EboxaWLA0WWWvJU15hgpRPnWdQsqOKHEMDTnb/4Bf/di/zh
# tDTxom+RgpVplLX3fjdRv1VsjGGLUK5ml7HWBt0BPBg1aE8Qamdq6wmQH2eUbflF
# Z1AUy/kZ/flNEwT7exXzZckQt/8hvDGmQy1i/BJXyvuR7zWrG0NmQtx9/K/mcPsb
# 74ja2072ZcbIHT8mArelj211nmzel2t4q5qU1m98cPh63UCI5QgkqujwwwBOnYnM
# 4AE9oIuZY8rTtVHbGTE66QwAgz+MgFmO9RN4+4QZshBFnc4+vU0iABownUsmdaWG
# Sg9SmsZrd8W13rDthPOsgyPLIzreD9bbQ+oxE6VwNxyrvnCo9KgIB15p3U4gpYPc
# 0LNz0PlVhQSTG4VDjFEZcubl5Emt6SXzkpeM0nCZtXJFTlu3m8vBimUcS8tQwSka
# 7G81QD5ebCYXzl6IkVvzjd0soD7Mos8Qela169cGKwYsChRjhv8TAVTs4WiRh8mi
# SMhXfZbra5WkMuprg5qQHI0msiO+QUO72X8XXlu/xBisCyB1DnIvSHdtkEdinPNq
# oQ==
# SIG # End signature block
