# ***************************************************************************
# Copyright 2019 VMware, Inc.  All rights reserved.
# VMware Confidential
# ***************************************************************************

<#
.SYNOPSIS
    This script is a wrapper to run the setup.exe file silently.
    The script waits for the setup to complete before returning.
.DESCRIPTION
    This install script installs Windows BareMetal Server Host
    Agents Installer
.EXAMPLE
    Following are some examples to run the bootstrapper.

    Install:
        powershell c:\Users\root\Desktop\install_helper.ps1 -operation install -setupFile
        "C:\\VMware-NSX-3.0.0.25136500_baremetal-server_setup.exe" -installPath "C:\NSX"

    Uninstall:
        powershell c:\Users\root\Desktop\install_helper.ps1 -operation uninstall -setupFile
        "C:\\VMware-NSX-3.0.0.25136500_baremetal-server_setup.exe"
#>


param(
    # install, uninstall
    [Parameter(Mandatory=$true)][string]$operation,

    # path to setup.exe file
    [Parameter(Mandatory=$true)][string]$setupFile,

    #optional
    [string]$async = "false",

    # optional
    [string]$installPath = "",

    #optional
    [string]$logDir = [environment]::GetEnvironmentVariable("PROGRAMDATA") + "\VMware\NSX\Logs"
)


$NSX_DATA_DIR = [environment]::GetEnvironmentVariable("PROGRAMDATA") + "\VMware\NSX\Data"

$installFile = [io.path]::GetFileNameWithoutExtension($setupFile);

if (-Not (Test-Path $setupFile)) {
    echo "Failed to find $setupFile"
    exit 1
}


$version = (Get-Item $setupFile).VersionInfo.ProductVersion
$logFile = [io.path]::combine($logDir, "setup_" + $operation + "_$version.log")

if (-Not (Test-Path $logDir)) {
    New-Item $logDir -type directory -Force | Out-Null
}

$msiArgs = "/qn"

if ($installPath -ne "") {
    echo "Custom installPath $installPath"
    $msiArgs = $msiArgs + " INSTALLDIR=""$installPath"""
}
if ($operation -eq "install") {
    $args = @('/s', '/v', $msiArgs, '/l', "$logFile")
    & $setupFile $args
}
elseif ($operation -eq "uninstall") {
    $cmd = "/s /v ""/qn REMOVE=ALL"" /l $logFile"
    iex "& $setupFile $cmd"
}
else {
    echo "Invalid operation $operation"
    exit 1
}

if ($async -eq $false) {
    echo "Monitoring $installFile for $operation"
    $a = get-process $installFile
    $handle = $a.Handle

    do {
       echo "Waiting ....."

    } while ($a.waitforexit())

    if ($a.ExitCode -eq 0) {
        echo "Done $operation of $installFile"
    } else {
        echo "Failed to $operation. ExitCode=$($a.ExitCode)"
    }
    exit $a.ExitCode
} else {
    echo "Running silent $operation of $installFile in background"
}

# SIG # Begin signature block
# MIIpTgYJKoZIhvcNAQcCoIIpPzCCKTsCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUD8m6oWWaQ0/S8Ssj7J/DOlvH
# rXiggg50MIIGsDCCBJigAwIBAgIQCK1AsmDSnEyfXs2pvZOu2TANBgkqhkiG9w0B
# AQwFADBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVk
# IFJvb3QgRzQwHhcNMjEwNDI5MDAwMDAwWhcNMzYwNDI4MjM1OTU5WjBpMQswCQYD
# VQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lD
# ZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQwOTYgU0hBMzg0IDIwMjEg
# Q0ExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA1bQvQtAorXi3XdU5
# WRuxiEL1M4zrPYGXcMW7xIUmMJ+kjmjYXPXrNCQH4UtP03hD9BfXHtr50tVnGlJP
# DqFX/IiZwZHMgQM+TXAkZLON4gh9NH1MgFcSa0OamfLFOx/y78tHWhOmTLMBICXz
# ENOLsvsI8IrgnQnAZaf6mIBJNYc9URnokCF4RS6hnyzhGMIazMXuk0lwQjKP+8bq
# HPNlaJGiTUyCEUhSaN4QvRRXXegYE2XFf7JPhSxIpFaENdb5LpyqABXRN/4aBpTC
# fMjqGzLmysL0p6MDDnSlrzm2q2AS4+jWufcx4dyt5Big2MEjR0ezoQ9uo6ttmAaD
# G7dqZy3SvUQakhCBj7A7CdfHmzJawv9qYFSLScGT7eG0XOBv6yb5jNWy+TgQ5urO
# kfW+0/tvk2E0XLyTRSiDNipmKF+wc86LJiUGsoPUXPYVGUztYuBeM/Lo6OwKp7AD
# K5GyNnm+960IHnWmZcy740hQ83eRGv7bUKJGyGFYmPV8AhY8gyitOYbs1LcNU9D4
# R+Z1MI3sMJN2FKZbS110YU0/EpF23r9Yy3IQKUHw1cVtJnZoEUETWJrcJisB9IlN
# Wdt4z4FKPkBHX8mBUHOFECMhWWCKZFTBzCEa6DgZfGYczXg4RTCZT/9jT0y7qg0I
# U0F8WD1Hs/q27IwyCQLMbDwMVhECAwEAAaOCAVkwggFVMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0OBBYEFGg34Ou2O/hfEYb7/mF7CIhl9E5CMB8GA1UdIwQYMBaA
# FOzX44LScV1kTN8uZz/nupiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAK
# BggrBgEFBQcDAzB3BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9v
# Y3NwLmRpZ2ljZXJ0LmNvbTBBBggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4
# oDagNIYyaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJv
# b3RHNC5jcmwwHAYDVR0gBBUwEzAHBgVngQwBAzAIBgZngQwBBAEwDQYJKoZIhvcN
# AQEMBQADggIBADojRD2NCHbuj7w6mdNW4AIapfhINPMstuZ0ZveUcrEAyq9sMCcT
# Ep6QRJ9L/Z6jfCbVN7w6XUhtldU/SfQnuxaBRVD9nL22heB2fjdxyyL3WqqQz/WT
# auPrINHVUHmImoqKwba9oUgYftzYgBoRGRjNYZmBVvbJ43bnxOQbX0P4PpT/djk9
# ntSZz0rdKOtfJqGVWEjVGv7XJz/9kNF2ht0csGBc8w2o7uCJob054ThO2m67Np37
# 5SFTWsPK6Wrxoj7bQ7gzyE84FJKZ9d3OVG3ZXQIUH0AzfAPilbLCIXVzUstG2MQ0
# HKKlS43Nb3Y3LIU/Gs4m6Ri+kAewQ3+ViCCCcPDMyu/9KTVcH4k4Vfc3iosJocsL
# 6TEa/y4ZXDlx4b6cpwoG1iZnt5LmTl/eeqxJzy6kdJKt2zyknIYf48FWGysj/4+1
# 6oh7cGvmoLr9Oj9FpsToFpFSi0HASIRLlk2rREDjjfAVKM7t8RhWByovEMQMCGQ8
# M4+uKIw8y4+ICw2/O/TOHnuO77Xry7fwdxPm5yg/rBKupS8ibEH5glwVZsxsDsrF
# hsP2JjMMB0ug0wcCampAMEhLNKhRILutG4UI4lkNbcoFUCvqShyepf2gpx8GdOfy
# 1lKQ/a+FSCH5Vzu0nAPthkX0tGFuv2jiJmCG6sivqf6UHedjGzqGVnhOMIIHvDCC
# BaSgAwIBAgIQBkACW/8MSJcymdRjIIPDezANBgkqhkiG9w0BAQsFADBpMQswCQYD
# VQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lD
# ZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQwOTYgU0hBMzg0IDIwMjEg
# Q0ExMB4XDTI0MDMyODAwMDAwMFoXDTI3MDMzMDIzNTk1OVowgcQxEzARBgsrBgEE
# AYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMIRGVsYXdhcmUxHTAbBgNVBA8M
# FFByaXZhdGUgT3JnYW5pemF0aW9uMRAwDgYDVQQFEwc2NjEwMTE3MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKQ2FsaWZvcm5pYTERMA8GA1UEBxMIU2FuIEpvc2UxFTAT
# BgNVBAoTDEJyb2FkY29tIEluYzEVMBMGA1UEAxMMQnJvYWRjb20gSW5jMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAuvAvoycSn8afXNQOn4IcEjY571fm
# NZstl5kxxD5pOwjhU1sNVUQWQbb/81F60krZVvUKqqePMJPh/h+aDNKi/Y7FEOsA
# CaZo7d9xd8byqnPHoc1WS11BAAkAV06N9B524W3GnCgMl/bBgchqipUenBx6xHm0
# ljKMTjukzoGINOsX3mM8acfB6jEIzioDRG4eHJtWMfhEf+AZlpmHL1R0fSrV8CVs
# Zs1CQmkuTgfZafCajufrgB76RrUBAG9/f4f78tvdIrAmMNev7mizs+RKXIb+X6IE
# wU4eKOkl1h2ptNxXzDVW5mzFbnIk83z0g7dIucUFmLVZMFVPjRhOF1LrV+M22op4
# Od0GCjQGvjVvRYlnGqBMBNak75O4pbLnD83agopEKR8mmhgsx6A4b3ki4iSoCkYg
# 6nnvCl32v4ZAqGrfQLQoocnfPfZCdZEpUzvX5HIufwnSsJsF0CCoejj7niq/jiev
# Vz8dFfkWmjAJJFbRW+/1cHnBZhYLFxZACvLpT2hmMiBThaizKh/Gbioi1T+R5MQ4
# EorSXFm2kn/PUybsTaodL1dZOIsGkKXJkGYTyuPaTZY+Y9HMsPAfsU2DQ8Jyr8qF
# oKigG9nH83OqQi5NhiSwcGpCmeWCgudF3YT5Usq6pS02+aLWUaGFhRii9UqscawY
# 0aoOKqdUb7sUXAUCAwEAAaOCAgIwggH+MB8GA1UdIwQYMBaAFGg34Ou2O/hfEYb7
# /mF7CIhl9E5CMB0GA1UdDgQWBBS/WUzEvz4YLxp12Zt1zq5g6CUiHzA9BgNVHSAE
# NjA0MDIGBWeBDAEDMCkwJwYIKwYBBQUHAgEWG2h0dHA6Ly93d3cuZGlnaWNlcnQu
# Y29tL0NQUzAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMwgbUG
# A1UdHwSBrTCBqjBToFGgT4ZNaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lD
# ZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcmww
# U6BRoE+GTWh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRH
# NENvZGVTaWduaW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3JsMIGUBggrBgEFBQcB
# AQSBhzCBhDAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFwG
# CCsGAQUFBzAChlBodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRU
# cnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNydDAJBgNV
# HRMEAjAAMA0GCSqGSIb3DQEBCwUAA4ICAQDC5DP+QWQdY9sY40fSJMg2LS66kvSM
# 7RK85eKIH9nV7gIAST9RCpvxDeKDovwR8S1tE1bnJRmd2hGt/591JR8aprypvuHp
# Q+aIeLBDpu/iexAldB6G3mYUYi5ncpWtTKm8cMaeegFb4Rv/vp3/gFc8PeXOu9Wt
# 5G/ulVzA82StanFIes8i5JWf734yoqa8vSoDEbF3fNSPj0a/vfYkSPuaVLBe4sZk
# uL05XTYdq/1URSi7V7U3n2dtEOeRH7v/DXnyX3w//iy5h7Mwdy0ZYLhWDlBqJ+6v
# HkCottDzFuj0G/Gs+Jp+jHwNWNsfi+xxQ4sv/1TDsuQ6nCjh1qs/zWa+/1RD7csK
# C2x5VMDSchWNMJ0Ic7zwoOu2KQxGBEQ54ILaXjnISpSaKiykCJ6usQy07XKPsp0e
# iB51kXM/qGIZuk3b0XJj+4IoqjmyhBV0MC7I0h7KkZbpNhUOA4Rut/vIMg+KykFa
# Mmz9KGxOxotkWBzk9KsD6LOhpJWa2obf3GqWu8KLNYDSlyS+xugWgl0FgsmpE++m
# TiTnR8x4ydp/sgYCllhgb9fOp6z+Z37kbtLyRBxu+HMuaEcHt75nus7T0MvVUoEq
# z/oOu22esj3vs9PU+zXM1uGl612zFHU29NsykLCvFhCn3L8wZmddeUsDLn94N+Au
# ckYTrFiJ36LJgDGCGkQwghpAAgEBMH0waTELMAkGA1UEBhMCVVMxFzAVBgNVBAoT
# DkRpZ2lDZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENv
# ZGUgU2lnbmluZyBSU0E0MDk2IFNIQTM4NCAyMDIxIENBMQIQBkACW/8MSJcymdRj
# IIPDezAJBgUrDgMCGgUAoHAwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkD
# MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwIwYJ
# KoZIhvcNAQkEMRYEFLb3NRH5tdyI9zglk1D4rmE5miRMMA0GCSqGSIb3DQEBAQUA
# BIICAAOXXD9UMTSBKt2BEF0dvfbIE59KK0WG02f1Tb+xEs5XwGph+Go+qKckMDD9
# R1ZuUB3akzlbZmrKajrLjmimhPvWep9/70/ZwxjbQrcXHr3d/fOxAEmWbjfTiEYK
# v4+BcBCG8rJnMwOjKq0cO8ZCiA5NQpYkHnbXMmvtpLnpj2EdwdmnAHm3qWlkU0cg
# Pu6yZvGANYrv675+L3eftyt7+i9z0oXU8BmcwOsVXzDv2KjVbzYAe9zmmXVZarnp
# 7iGgRpNRO4myZ1io2MZ9a/XJW1h1dIx+/3snRigXK3MuW7fTrCsv5u3W8gkHS0oc
# oIEQV2w/2JCujPDb1Vho0D5mly4nd5mgXTYt1D96xEp8qOTo6m8fcyRTORAVI7bk
# 0SGWKm50tcjDVeikUzQH06s/M/9SmIS5GiMI9kwb7dwN/rgS99xGBOr+oIyHeCDv
# ObpC6ihTI2ozKrcp5lOYJrua8+V0/sTj2rB6OTdrvcDtZA8r8pJR657CqkuxS3MZ
# MoMSlr26tGAZgD8jAMPuQolBgq0OKhw1jGb43jq1kibbesIbbc9bHc6hw0x7Kqk7
# LhruK5rwEMBOs+iJoVYAnCjN2S0ID9EHn2/n+nYZXaOfCm4ncTh6B3KiZuiIH/Pd
# T4iuGdSfs9fcOVSixAA7DbWPooHIqDLF72a6QtMul3CjQkOvoYIXKjCCFyYGCisG
# AQQBgjcDAwExghcWMIIXEgYJKoZIhvcNAQcCoIIXAzCCFv8CAQMxDzANBglghkgB
# ZQMEAgEFADBoBgsqhkiG9w0BCRABBKBZBFcwVQIBAQYJYIZIAYb9bAcBMCEwCQYF
# Kw4DAhoFAAQU0lYi4MfkYp91wVmgEyh3m6hH3YUCEQDcFAqlLRJsNDj5sUV/TAtU
# GA8yMDI1MDEwNTA4MDk1MlqgghMDMIIGvDCCBKSgAwIBAgIQC65mvFq6f5WHxvnp
# BOMzBDANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGln
# aUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5
# NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTI0MDkyNjAwMDAwMFoXDTM1MTEy
# NTIzNTk1OVowQjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lDZXJ0MSAwHgYD
# VQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyNDCCAiIwDQYJKoZIhvcNAQEBBQAD
# ggIPADCCAgoCggIBAL5qc5/2lSGrljC6W23mWaO16P2RHxjEiDtqmeOlwf0KMCBD
# Er4IxHRGd7+L660x5XltSVhhK64zi9CeC9B6lUdXM0s71EOcRe8+CEJp+3R2O8oo
# 76EO7o5tLuslxdr9Qq82aKcpA9O//X6QE+AcaU/byaCagLD/GLoUb35SfWHh43rO
# H3bpLEx7pZ7avVnpUVmPvkxT8c2a2yC0WMp8hMu60tZR0ChaV76Nhnj37DEYTX9R
# eNZ8hIOYe4jl7/r419CvEYVIrH6sN00yx49boUuumF9i2T8UuKGn9966fR5X6kgX
# j3o5WHhHVO+NBikDO0mlUh902wS/Eeh8F/UFaRp1z5SnROHwSJ+QQRZ1fisD8UTV
# DSupWJNstVkiqLq+ISTdEjJKGjVfIcsgA4l9cbk8Smlzddh4EfvFrpVNnes4c16J
# idj5XiPVdsn5n10jxmGpxoMc6iPkoaDhi6JjHd5ibfdp5uzIXp4P0wXkgNs+CO/C
# acBqU0R4k+8h6gYldp4FCMgrXdKWfM4N0u25OEAuEa3JyidxW48jwBqIJqImd93N
# Rxvd1aepSeNeREXAu2xUDEW8aqzFQDYmr9ZONuc2MhTMizchNULpUEoA6Vva7b1X
# CB+1rxvbKmLqfY/M/SdV6mwWTyeVy5Z/JkvMFpnQy5wR14GJcv6dQ4aEKOX5AgMB
# AAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADAWBgNVHSUB
# Af8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1s
# BwEwHwYDVR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYDVR0OBBYEFJ9X
# LAN3DigVkGalY17uT5IfdqBbMFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwz
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1l
# U3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEBBIGDMIGAMCQGCCsGAQUFBzABhhho
# dHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9jYWNl
# cnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZU
# aW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcNAQELBQADggIBAD2tHh92mVvjOIQS
# R9lDkfYR25tOCB3RKE/P09x7gUsmXqt40ouRl3lj+8QioVYq3igpwrPvBmZdrlWB
# b0HvqT00nFSXgmUrDKNSQqGTdpjHsPy+LaalTW0qVjvUBhcHzBMutB6HzeledbDC
# zFzUy34VarPnvIWrqVogK0qM8gJhh/+qDEAIdO/KkYesLyTVOoJ4eTq7gj9UFAL1
# UruJKlTnCVaM2UeUUW/8z3fvjxhN6hdT98Vr2FYlCS7Mbb4Hv5swO+aAXxWUm3Wp
# ByXtgVQxiBlTVYzqfLDbe9PpBKDBfk+rabTFDZXoUke7zPgtd7/fvWTlCs30VAGE
# sshJmLbJ6ZbQ/xll/HjO9JbNVekBv2Tgem+mLptR7yIrpaidRJXrI+UzB6vAlk/8
# a1u7cIqV0yef4uaZFORNekUgQHTqddmsPCEIYQP7xGxZBIhdmm4bhYsVA6G2WgNF
# YagLDBzpmk9104WQzYuVNsxyoVLObhx3RugaEGru+SojW4dHPoWrUhftNpFC5H7Q
# EY7MhKRyrBe7ucykW7eaCuWBsBb4HOKRFVDcrZgdwaSIqMDiCLg4D+TPVgKx2EgE
# deoHNHT9l3ZDBD+XgbF+23/zBjeCtxz+dL/9NWR6P2eZRi7zcEO1xwcdcqJsyz/J
# ceENc2Sg8h3KeFUCS7tpFk7CrDqkMIIGrjCCBJagAwIBAgIQBzY3tyRUfNhHrP0o
# ZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGln
# aUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhE
# aWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIwMzIzMDAwMDAwWhcNMzcwMzIy
# MjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# OzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGlt
# ZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAxoY1
# BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyibaCwzIP5WvYRoUQVQl+kiPNo+n3z
# nIkLf50fng8zH1ATCyZzlm34V6gCff1DtITaEfFzsbPuK4CEiiIY3+vaPcQXf6sZ
# Kz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2UKdPZ7Gnf2ZCHRgB720RBidx8ald6
# 8Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7QKxfst5Kfc71ORJn7w6lY2zk
# psUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15/tePc5OsLDnipUjW8LAxE6lXKZYn
# LvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCYOjgRs/b2nuY7W+yB3iIU2YIq
# x5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3NDzt9KoRxrOMUp88qqlnNCaJ+2RrOd
# OqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6dSgkQe1CvwWcZklSUPRR8zZJ
# TYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmyMKVM1+mYSlg+0wOI/rOP015LdhJR
# k8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4KbhPvbCdLI/Hgl27KtdRnXiYKNYCQEo
# AA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0zbECAwEAAaOCAV0wggFZMBIGA1Ud
# EwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW2W1NhS9zKXaaL3WMaiCPnshvMB8G
# A1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9PMA4GA1UdDwEB/wQEAwIBhjAT
# BgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUHMAGG
# GGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggrBgEFBQcwAoY1aHR0cDovL2Nh
# Y2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYD
# VR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0
# VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9
# bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7ZvmKlEIgF+ZtbYIULhsBguEE0T
# zzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfpPmDI2AvlXFvXbYf6hCAlNDFnzbYS
# lm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9Hd/tydBTX/6tPiix6q4XNQ1/tYLaq
# T5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+ovHVPulr3qRCyXen/KFSJ8NWKcXZl
# 2szwcqMj+sAngkSumScbqyQeJsG33irr9p6xeZmBo1aGqwpFyd/EjaDnmPv7pp1y
# r8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR8XKc6UsCUqc3fpNTrDsdCEkPlM05
# et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcTJM3cHXg65J6t5TRxktcma+Q4c6um
# AU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0dKNPH+ejxmF/7K9h+8kaddSwe
# Jywm228Vex4Ziza4k9Tm8heZWcpw8De/mADfIBZPJ/tgZxahZrrdVcA6KYawmKAr
# 7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLeMt8EifAAzV3C+dAjfwAL5HYC
# JtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A+sDyDivl1vupL0QVSucTDh3bNzga
# oSv27dZ8/DCCBY0wggR1oAMCAQICEA6bGI750C3n79tQ4ghAGFowDQYJKoZIhvcN
# AQEMBQAwZTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcG
# A1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJl
# ZCBJRCBSb290IENBMB4XDTIyMDgwMTAwMDAwMFoXDTMxMTEwOTIzNTk1OVowYjEL
# MAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3
# LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAv+aQc2jeu+RdSjwwIjBp
# M+zCpyUuySE98orYWcLhKac9WKt2ms2uexuEDcQwH/MbpDgW61bGl20dq7J58soR
# 0uRf1gU8Ug9SH8aeFaV+vp+pVxZZVXKvaJNwwrK6dZlqczKU0RBEEC7fgvMHhOZ0
# O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs06wXGXuxbGrzryc/NrDRAX7F6Zu53
# yEioZldXn1RYjgwrt0+nMNlW7sp7XeOtyU9e5TXnMcvak17cjo+A2raRmECQecN4
# x7axxLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtVgkEy19sEcypukQF8IUzUvK4bA3Vd
# eGbZOjFEmjNAvwjXWkmkwuapoGfdpCe8oU85tRFYF/ckXEaPZPfBaYh2mHY9WV1C
# doeJl2l6SPDgohIbZpp0yt5LHucOY67m1O+SkjqePdwA5EUlibaaRBkrfsCUtNJh
# besz2cXfSwQAzH0clcOP9yGyshG3u3/y1YxwLEFgqrFjGESVGnZifvaAsPvoZKYz
# 0YkH4b235kOkGLimdwHhD5QMIR2yVCkliWzlDlJRR3S+Jqy2QXXeeqxfjT/JvNNB
# ERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFrb7GrhotPwtZFX50g/KEexcCPorF+
# CiaZ9eRpL5gdLfXZqbId5RsCAwEAAaOCATowggE2MA8GA1UdEwEB/wQFMAMBAf8w
# HQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiuHA9PMB8GA1UdIwQYMBaAFEXroq/0
# ksuCMS1Ri6enIZ3zbcgPMA4GA1UdDwEB/wQEAwIBhjB5BggrBgEFBQcBAQRtMGsw
# JAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcw
# AoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElE
# Um9vdENBLmNydDBFBgNVHR8EPjA8MDqgOKA2hjRodHRwOi8vY3JsMy5kaWdpY2Vy
# dC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMBEGA1UdIAQKMAgwBgYE
# VR0gADANBgkqhkiG9w0BAQwFAAOCAQEAcKC/Q1xV5zhfoKN0Gz22Ftf3v1cHvZqs
# oYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU9BNKei8ttzjv9P+Aufih9/Jy3iS8UgPI
# TtAq3votVs/59PesMHqai7Je1M/RQ0SbQyHrlnKhSLSZy51PpwYDE3cnRNTnf+hZ
# qPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4oVaO7KTVPeix3P0c2PR3WlxUjG/v
# oVA9/HYJaISfb8rbII01YBwCA8sgsKxYoA5AY8WYIsGyWfVVa88nq2x2zm8jLfR+
# cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNNn3O3AamfV6peKOK5lDGCA3YwggNy
# AgEBMHcwYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTsw
# OQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRpbWVT
# dGFtcGluZyBDQQIQC65mvFq6f5WHxvnpBOMzBDANBglghkgBZQMEAgEFAKCB0TAa
# BgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI1MDEw
# NTA4MDk1MlowKwYLKoZIhvcNAQkQAgwxHDAaMBgwFgQU29OF7mLb0j575PZxSFCH
# JNWGW0UwLwYJKoZIhvcNAQkEMSIEIMefeFCbfNQew8kgypqxEDm4BfANjWq7/rgz
# Ght4IXIHMDcGCyqGSIb3DQEJEAIvMSgwJjAkMCIEIHZ2n6jyYy8fQws6IzCu1lZ1
# /tdz2wXWZbkFk5hDj5rbMA0GCSqGSIb3DQEBAQUABIICALevE7xI8zmYGycbaKUi
# 3pTRlAFVjkKFLl6Quj+nGhFfh+2Oxxv5VHeHNRe8NEnaNOboDL0eqUOd2iBumDN7
# tLZvaBOPbHPFpuQUpPBQO7F1o6E6SABWwziXYBG3yy9ADdTwJrtOi7LBlJzbiZk0
# vgVOObfelJpSx6PkIu+RGYCS+pvWYYnB/CHwS/j5jmp3p77gEIFKt2lx2y5rsxeS
# 5aD95qDSUdpRqzAjI0Lv8q3dblXC7mhywv3ygg10dh9sg2xNqnSuO1/X+GHt4XdE
# wP0sCPjpadECZMZQcUw96PvnD3CqZBs990FgzSXrtG98C2zFIMpsG9/nvBFhfArJ
# 9yp2aDreoYY0w7aufwY0wNaJz8W/psUYrMFu5YQucSeOz6MRxSCkc2fcsHLyH9k0
# oVwb73gXlBOojNpmdn1ePUYy6z4o7RGutIH5srPyI0u3RShOoHl8PqMF3NwGaqfH
# C9HtBS/AkE18Y2L+6GCXmf9TxbHn9SAWpOxNAV0uhQJ5Bcqlo10V0PAjAQbSDkzJ
# cIK+UJXNw+xGb/Ei9gu1pilOYO95EONoufMT3nNfg7LE/0KC18m00SuCf9GQh3Ky
# V6jhsRyak1lJ1vkYHLVJkQ0HQLLMIcE8ZEKDY7TfJYSgmZ4QtQKrlQZ6aBiUziP9
# ErTz/yO0D6xH2WmFEsHtHdN3
# SIG # End signature block
