﻿# Copyright (c) Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Restricted
param (
    [string]$msiArgs,
    [string]$installerFilename,
    [string]$installDir,
    [string]$version,
    [string]$buildNumber,
    [string]$url,
    [string]$expectedFileSize,
    [string]$expectedChecksum,
    [string]$horizonRegKeyEncodedPath,
    [string]$lcmRegKeyEncodedPath,
    [string]$programData


)

$timeoutSeconds = 1800
$timestamp = Get-Date -Format "yyyy-MM-dd-HHmmss"
$tempDirectory = [System.IO.Path]::GetTempPath()
$logfolder = "\logs\"
$Logfile = Join-Path (Join-Path $programData $logfolder) "LifecycleManager-$timestamp.txt"
$exeFile = Join-Path $tempDirectory $installerFilename

$vdmRegKeyPath = [Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($horizonRegKeyEncodedPath))
$lcmKeyPath = [Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($lcmRegKeyEncodedPath))

<############################################################
 # function to log messages to file
 ############################################################>
function WriteLog {
    Param ([string]$LogString)
    $Stamp = (Get-Date).toString("yyyy/MM/dd HH:mm:ss")
    $LogMessage = "$Stamp $LogString"
    #  WriteLog $LogMessage
    Add-Content $Logfile -Value $LogMessage
}

<###################################################################
# function to set registry key value name . If key value name is not 
# present then we create key value and update the value data
####################################################################>
function setRegKeyValue {
    Param(
        $keyValueName,
        $keyValueData,
        $keyPath
    )
    try {
        if (Test-Path -Path $keyPath\$keyValueName) {
            Set-ItemProperty -Path $keyPath -Name $keyValueName -Value $keyValueData -Force -ErrorAction Stop
        }
        else {
            New-ItemProperty -Path $keyPath -Name $keyValueName -Value $keyValueData -Force -ErrorAction Stop
        }
    }
    catch {
        WriteLog "Failed to set registry key value: $keyValueName"
        throw "Failed to set registry key value: $keyValueName"
    }
}

<#############################################################
# function to read registry key value 
#############################################################>
function regKeyValueRead {
    param ( [string]$keyValueName,
        $keyPath)
    try {
        $value = Get-ItemPropertyValue -Path $keyPath -Name $keyValueName -ErrorAction Stop
        return $value    
    }
    catch [System.Management.Automation.ItemNotFoundException] { 
        WriteLog "Failed to read registry key value : $keyValueName"
        throw "Failed to read registry key value : $keyValueName"
    }
    catch {
        WriteLog "Failed to find registry key path $keyPath"
        throw "Failed to find registry key path $keyPath"
    }
}

<###########################################################################
# function to check ADAM Database service installed 
############################################################################>
############################################################################>
function isAdamServiceFound {
    Param([string]$vdmRegKeyPath)
    $adamServices = Get-Service | Where-Object { $_.Name -like "ADAM_*" }
    WriteLog "VDM Reg key path: " $vdmRegKeyPath
    WriteLog "ADAM services found : " $adamServices
    $oldLocalADAMService = "ADAM_VMwareVDMDS"
    $oldGlobalADAMService = "ADAM_VMwareVDMDSG"
    $newLocalADAMService = "ADAM_OmnissaHzeDS"
    $newGlobalADAMService = "ADAM_OmnissaHzeDSG"
    $oldRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\VMware, Inc.\VMware VDM\"
    $newRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Omnissa\Horizon\"
    if ($adamServices) {
        if ($vdmRegKeyPath -eq $oldRegKeyPath) {
            foreach ($service in $adamServices) {
                WriteLog "Service name:$($service.Name)"
                if ($oldLocalADAMService -eq $($service.Name) -or $oldGlobalADAMService -eq $($service.Name)) {
                    WriteLog "old ADMD service"
                    return $true
                }
            }
        } 
        elseif ($vdmRegKeyPath -eq $newRegKeyPath) {
            foreach ($service in $adamServices) {
                WriteLog  "Service name:$($service.Name)"
                if ($newLocalADAMService -eq $($service.Name) -or $newGlobalADAMService -eq $($service.Name)) {
                    WriteLog "new ADMD service"
                    return $true
                }
            }
        }
        
    }
    else {
        WriteLog  "No ADAM services found."
        return $false
    }
}

<############################################################
 # function to check if target server is upgradable
 ############################################################>

function checkIsInstallable {
    Param($version, $buildNumber, $vdmRegKeyPath, $lcmKeyPath)
    try {
        $currentVersion = Get-ItemProperty -Path $vdmRegKeyPath -Name "ProductVersion" -ErrorAction Stop | Select-Object -ExpandProperty ProductVersion
    }
    catch {
        WriteLog "Horizon Server installation not found"
    }
    $targetVersion = "$version.$buildNumber"
    if ((isAdamServiceFound $vdmRegKeyPath) -or ($null -ne $currentVersion)) {
        WriteLog "Instance of Horizon Server installation found on the specifed server"
        setRegKeyValue "Status" "Installation_Not_Supported" $lcmKeyPath
        setRegKeyValue "ErrorMessage" "Existing_HorizonServer_installation_found" $lcmKeyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $lcmKeyPath
        throw "Instance of Horizon Server installation found on the specifed server"
    }
    else {
        WriteLog "Initiating installation of Horizon Server $targetVersion"
        setRegKeyValue "Status" "Triggering_Install" $lcmKeyPath
    }
}

<############################################################
 # function to check build source and trigger download
 ############################################################>
function checkBuildPathAndDownload{
    Param($buildPath, $exeFile, $expectedFileSize, $timeoutSeconds, $keyPath)

    if ($buildPath -imatch "^https://") {
        WriteLog "Downloading build from webserver"
        downloadBuild $buildPath $exeFile $expectedFileSize $timeoutSeconds $keyPath
    } elseif ($buildPath -imatch "^smb://") {
        WriteLog "Copying build from SMB share"
        CopyFromSMBShare -smbPath $buildPath -destination $exeFile -expectedFileSize $expectedFileSize -keyPath $keyPath
    } else {
        WriteLog "Invalid path format. Only HTTPS and SMB paths are supported."
        setRegKeyValue "Status" "Build_Download_Failed" $keyPath
        setRegKeyValue "ErrorMessage" "Build download failed " $keyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $keyPath
        throw "Build_Download_Failed"
    }
}

<############################################################
# function to copy build from smb share on target server
############################################################>
function CopyFromSMBShare {
    param (
        [string]$smbPath,
        [string]$destination,
        [string]$expectedFileSize,
        [string]$keyPath)

    WriteLog "Copying from SMB share: $smbPath"
    setRegKeyValue "Status" "Build_Download_InProgress" $keyPath
    $smbPath = $smbPath -replace "^smb:", "" -replace "//","\\"
    try{
        Copy-Item -Path $smbPath -Destination $destination -Force -ErrorAction Stop

        # Check if file exists and verify size
        if (Test-Path $destination) {
            $actualSize = (Get-Item $destination).Length
            if ($actualSize -eq $expectedFileSize) {
                WriteLog "Copy successful. File size matches: $actualSize bytes"
                setRegKeyValue "Status" "Build_Download_Completed" $keyPath
            } else {
                WriteLog "Expected size $expectedFileSize , but got $actualSize"
                setRegKeyValue "Status" "Build_Download_Failed" $keyPath
                setRegKeyValue "ErrorMessage" "actual file size: $actualSize not equal to expected file size: $expectedFileSize" $keyPath
	        throw "Build_Download_Failed"
            }
        } else {
            WriteLog "File not found at destination"
            setRegKeyValue "Status" "Build_Download_Failed" $keyPath
            setRegKeyValue "ErrorMessage" "File not found at destination" $keyPath
            throw "Build_Download_Failed"
        }
    } catch {
        WriteLog "Error: $_"
        setRegKeyValue "Status" "Build_Download_Failed" $keyPath
        setRegKeyValue "ErrorMessage" "Build download failed " $keyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $keyPath
        throw "Build_Download_Failed"
    }
}
<############################################################
 # function to download build on target server
 ############################################################>
function downloadBuild {
    Param($url, $exeFile, $expectedFileSize, $timeoutSeconds, $keyPath)
    Add-Type -AssemblyName System.Net.Http
    try {        
        $retry = 3
        do {
            $httpClient = New-Object System.Net.Http.HttpClient
            $httpClient.Timeout = [System.TimeSpan]::FromSeconds($timeoutSeconds)
            WriteLog "Updating Status to Download InProgress"
            setRegKeyValue "Status" "Build_Download_InProgress" $keyPath
            $response = $httpClient.GetAsync($url).Result
            $responseStatusCode = [int]$response.StatusCode
            WriteLog "Response code: $responseStatusCode"
            WriteLog "IsSuccessStatusCode : $response.IsSuccessStatusCode"
            if ($response.IsSuccessStatusCode) {
                $content = $response.Content.ReadAsByteArrayAsync().Result
                [System.IO.File]::WriteAllBytes($exeFile, $content)
                $downloadedFileSize = (Get-Item $exeFile).length
                WriteLog "Downloaded file size: $downloadedFileSize"
                WriteLog "Expected file size: $expectedFileSize"
                if ($downloadedFileSize -eq $expectedFileSize) {
                    setRegKeyValue "Status" "Build_Download_Completed" $keyPath
                    WriteLog "Build downloaded successfully."
                    $retry = 0; 
                }
                else {
                    setRegKeyValue "Status" "Build_Download_Failed" $keyPath
                    WriteLog "Downloaded build size: $downloadedFileSize not equal to expected build size: $expectedFileSize."
                    setRegKeyValue "ErrorMessage" "Downloaded build size: $downloadedFileSize not equal to expected build size: $expectedFileSize" $keyPath
                    WriteLog "Clearing ControllerCS reg keyValue"
                    setRegKeyValue "ControllerCs" "" $keyPath
                    throw "Build_Download_Failed"
                }
            }
            else {
                setRegKeyValue "Status" "Build_Download_Failed" $keyPath
                setRegKeyValue "ErrorMessage" "Build download failed with error code : $responseStatusCode" $keyPath
                WriteLog "Build download failed with error "+ $response.Content $keyPath
                WriteLog "Error: HTTP status code "  $responseStatusCode
                WriteLog "Retry download"
                $httpClient.Dispose()
                $retry--
            }
        } while ($retry -gt 0)
        $downloadStatus = regKeyValueRead "Status" $keyPath
        WriteLog "Downlaod Status $downloadStatus"
        if ($downloadStatus -ne "Build_Download_Completed") {
            WriteLog "Clearing ControllerCS reg keyValue"
            setRegKeyValue "ControllerCs" "" $keyPath
            throw "Build_Download_Failed"
        }
    }
    catch {
        WriteLog "Error: $_"
        setRegKeyValue "Status" "Build_Download_Failed" $keyPath
        setRegKeyValue "ErrorMessage" "Build download failed " $keyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $keyPath
        throw "Build_Download_Failed"
    }
    finally {
        $httpClient.Dispose()
    }
}

<############################################################
 # function to perform checksum verification of installer
 ############################################################>
function checksumVerification {
    Param($exeFile, $expectedChecksum, $keyPath)
    setRegKeyValue "Status" "Checksum_Verification_InProgress" $keyPath
    WriteLog "Checksum Verification InProgress"
    if (Test-Path $exeFile -PathType Leaf) {
        $fileHash = Get-FileHash -Path $exeFile -Algorithm SHA256
        if ($fileHash.Hash -eq $expectedChecksum) {
            setRegKeyValue "Status" "Checksum_Verification_Completed" $keyPath
            WriteLog "Checksum is valid. File has not been tampered with."
        }
        else {
            setRegKeyValue "Status" "Checksum_Verification_Failed" $keyPath
            setRegKeyValue "ErrorMessage" "Checksum is not valid. The file may have been tampered with." $keyPath
            WriteLog "Checksum is not valid. The file may have been tampered with."
            WriteLog "Expected checksum: $expectedChecksum"
            WriteLog "Actual checksum: $($fileHash.Hash)"
            WriteLog "Clearing ControllerCS reg keyValue"
            setRegKeyValue "ControllerCs" "" $keyPath
            throw "Checksum is not valid. The file may have been tampered with."
        }
    }
    else {
        WriteLog "Installer file not found: $exeFile"
        setRegKeyValue "Status" "Checksum_Verification_Failed" $keyPath
        setRegKeyValue "ErrorMessage" "Installer file not found" $keyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $keyPath
        throw "Installer file not found: $exeFile"

    }

}

<############################################################
 # function to check is service in Running status
 ############################################################>
function isServiceRunning {
    param ($serviceName)
    $serviceStatus = Get-Service -Name $serviceName
    if ($serviceStatus.Status -eq "Running") {
        return $true
    }
    return $false
}

<#################################################################
 # function to perform horizon server installation on target server
 #################################################################>
function installServer {
    Param($exeFile , $msiArgs, $vdmRegKeyPath, $keyPath)
    setRegKeyValue "Status" "Installation_InProgress" $keyPath
    $Install = (Start-Process -Filepath $exeFile -Wait -ArgumentList "/s /v /qn $msiArgs" -PassThru)
    $Install.ExitCode
    if ($Install.ExitCode -ne 0) {
        setRegKeyValue "Status" "Installation_Failed" $keyPath
        setRegKeyValue "ErrorMessage" $Install.StandardError $keyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $keyPath
        throw "Installation_Failed"
    }
    else {
        setRegKeyValue "Status" "Installation_Success" $keyPath
        WriteLog "Successfully Installed Horizon Server" 
        setRegKeyValue "Status" "Post_Install_Check_InProgress" $keyPath
        if (isAdamServiceFound $vdmRegKeyPath) {
            $servicesToCheck = @('wsbroker', 'wstomcat', 'wstunnel', 'PCOIPSG', 'wsmsgbus', 'wsnm', 'VMBlastSG', 'wsdct')
            $allRunning = $true
            $notRunningServices = @()
            foreach ($serviceName in $servicesToCheck) {
                if (isServiceRunning -serviceName $serviceName) {
                    WriteLog "$serviceName is running."
                }
                else {
                    WriteLog "$serviceName is not running."
                    $allRunning = $false
                    $notRunningServices += $serviceName
                }
            } 
            if ($allRunning) {
                setRegKeyValue "Status" "Post_Installation_Check_Success" $keyPath
                WriteLog "Horizon Server Post Installation Check Successful"
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $keyPath
            }
            else {
                setRegKeyValue "Status" "Post_Installation_Check_Failed" $keyPath
                WriteLog "Horizon Server Post Installation Check failed"
                setRegKeyValue "ErrorMessage" "$notRunningServices is not Running" $keyPath
                WriteLog "$notRunningServices is not Running"
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $keyPath
            }
        }
        else {
            $enrollmentService = 'wsenroll'
            if (isServiceRunning -serviceName $enrollmentService) {
                WriteLog "$enrollmentService is running."
                setRegKeyValue "Status" "Post_Installation_Check_Success" $keyPath
                WriteLog "Horizon Server Post Installation Check Successful"
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $keyPath
            }
            else {
                setRegKeyValue "Status" "Post_Installation_Check_Failed" $keyPath
                WriteLog "Horizon Server Post Installation Check failed"
                setRegKeyValue "ErrorMessage" "$enrollmentService is not Running" $keyPath
                WriteLog "$enrollmentService is not Running"
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $keyPath
            }
        }
    }
}

$logPath = (Join-Path $programData $logfolder)
if (-not (Test-Path -Path $logPath -PathType Container)) {
    New-Item -Path $logPath -ItemType Directory -Force
}
WriteLog "Starting Horizon Server Install flow"
setRegKeyValue "Status" "Initiating_Server_Install" $lcmKeyPath
$sourceServer = regKeyValueRead -keyValueName "ControllerCs" $lcmKeyPath
WriteLog "ControllerCs : $sourceServer"
WriteLog "installerFilename : $installerFilename"
WriteLog "version : $version"
WriteLog "buildNumber: $buildNumber"
WriteLog "url : $url"
WriteLog "expectedFileSize : $expectedFileSize"
WriteLog "expectedChecksum : $expectedChecksum"
WriteLog "exeFile : $exeFile"
if (-not ([string]::IsNullOrEmpty($installDir))) {
    WriteLog "Installation Directory $installDir"
    $msiArgs = $msiArgs + " INSTALLDIR=" + '"' + $installDir + '"'
}

try {
    WriteLog "Create and Update process id registry key"
    setRegKeyValue "Pid" $PID $lcmKeyPath
            
    WriteLog "Verify the Horizon server can be installed on target server"
    checkIsInstallable $version $buildNumber $vdmRegKeyPath $lcmKeyPath
            
    WriteLog "Download Build"
    checkBuildPathAndDownload $url $exeFile $expectedFileSize $timeoutSeconds $lcmKeyPath

    WriteLog "Verify Checksum"
    checksumVerification $exeFile $expectedChecksum $lcmKeyPath

    WriteLog "Initiating the server installation"
    installServer $exeFile $msiArgs $vdmRegKeyPath $lcmKeyPath
}
catch {
    WriteLog " Installation failed. Reason - $_"
}
finally {
    $TaskName = "RunHorizonServerInstall"
    $TaskExists = Get-ScheduledTask -TaskName $TaskName -ErrorAction SilentlyContinue
    if ($TaskExists) {
        try {
            Unregister-ScheduledTask -TaskName $TaskName -Confirm:$false -ErrorAction Stop
            WriteLog "Scheduled task '$TaskName' deleted successfully."
        }
        catch {
            WriteLog "FAIL - Failed to delete task '$TaskName'. Error: $_"
        }
    }
}