﻿# Copyright (c) Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Restricted
param(
  [string]$targetServerFqdn,
  [string]$msiArgs,
  [string]$installDir,
  [string]$installerFilename,
  [string]$version,
  [string]$buildNumber,
  [string]$url,
  [string]$expectedFileSize,
  [string]$expectedChecksum,
  [string]$username,
  [string]$password,
  [string]$triggerMethodName = "AddSeconds", # Define triggermethod name (AddSeconds,AddMinutes),schedule task and deleted task intervals. 
  [int]$scheduleTaskAfter = 30               #In future release we can pass this values while invoking API.
)

$sourceServer = [System.Net.Dns]::GetHostEntry([System.Net.Dns]::GetHostName()).HostName

$statusValues = @("Horizon_Server_Install_Scheduled", "Horizon_Server_Upgrade_Scheduled", "Initiating_Server_Install_Flow", "Triggering_Install",
  "Triggering_Server_Upgrade_Flow", "Build_Download_InProgress", "Build_Download_Completed", "Checksum_Verification_InProgress", 
  "Checksum_Verification_Completed", "Installation_InProgress", "Installation_Success", "Post_Upgrade_Check_InProgress", 
  "Upgrade_InProgress", "Upgrade_Success", "Post_Upgrade_Check_Inprogress")
$tempDirectory = [System.IO.Path]::GetTempPath()
$logfolder = "logs\"
$global:vdmRegKeyPath = $null
$global:lcmRegKeyPath = $null
$global:programData = $null

<##################################################################
# function to schedule a install script execution on target server
###################################################################>
function scheduleTask {
  Param($serverFqdn,
    $msiArgs,
    $installDir,
    $sourceServer,
    $installerFilename,
    $version,
    $buildNumber,
    $url,
    $expectedFileSize,
    $expectedChecksum,
    $triggerMethodName,
    $scheduleTaskAfter,
    $username,
    $password)

  #Get location for powershell scripts on connection server
  if (Test-Path -Path 'HKLM:\SOFTWARE\Omnissa\Horizon\'){
      $localInstallPath = Get-ItemPropertyValue -Path 'HKLM:\SOFTWARE\Omnissa\Horizon\' -Name "ServerInstallPath"}
  elseif (Test-Path -Path 'HKLM:\SOFTWARE\Vmware Inc,\VDM\'){
      $localInstallPath = Get-ItemPropertyValue -Path 'HKLM:\SOFTWARE\Vmware Inc,\VDM\' -Name "ServerInstallPath"}
  $SourceSchedularFile = Join-Path (Join-Path $localInstallPath "lcm") "ScheduleInstallRunner.ps1"
  $SourceInstallRunnerFile = Join-Path (Join-Path $localInstallPath "lcm") "InstallRunner.ps1"
  $session = New-PSSession -ComputerName $serverFqdn
  Invoke-Command -Session $session -ScriptBlock {
    param($global:programData ) 
    if (-not (Test-Path -Path $global:programData)) {
      New-Item -Path $global:programData  -ItemType Directory
    }
  }-ArgumentList $global:programData 

  $DestinationSchedularFile = Join-Path $global:programData "ScheduleInstallRunner.ps1"
  $DestinationInstallRunnerFile = Join-Path $global:programData "InstallRunner.ps1"

  Copy-Item -Path $SourceSchedularFile -Destination $DestinationSchedularFile -ToSession $Session
  Copy-Item -Path $SourceInstallRunnerFile -Destination $DestinationInstallRunnerFile -ToSession $Session

  $securePassword = ConvertTo-SecureString $password -AsPlainText -Force
  try {
    Invoke-Command -Session $session -ScriptBlock {
      param($DestinationSchedularFile, $InstallRunnerFile, $sourceServer, $msiArgs, $installDir, $installerFilename, $version, $buildNumber, $url, $expectedFileSize, $expectedChecksum, $triggerMethodName, $scheduleTaskAfter, $username, $securePassword, $vdmRegKeyPath, $lcmRegKeyPath, $programData ) 
      & $using:DestinationSchedularFile -InstallRunnerFile $InstallRunnerFile -sourceServer $using:sourceServer -msiArgs $msiArgs -installDir $installDir -installerFilename $installerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -securePassword $using:securePassword -vdmRegKeyPath $using:vdmRegKeyPath -lcmRegKeyPath $using:lcmRegKeyPath -programData $using:programData
    } -ArgumentList $DestinationSchedularFile, $DestinationInstallRunnerFile, $sourceServer, $msiArgs, $installDir, $installerFilename, $version, $buildNumber, $url, $expectedFileSize, $expectedChecksum, $triggerMethodName, $scheduleTaskAfter, $username, $securePassword
  }
  catch {
    Write-Error "FAIL - An error occurred: $_"
    $exitCode = 1
  }
  Remove-PSSession $Session
}

<###################################################################
# function to set registry key value name . If key value name is not 
# present then we create key value and update the value data
####################################################################>
function setRegKeyValue {
  Param(
    [string]$serverFqdn,
    [string]$regKeyPath,
    [string]$keyValueName,
    [string]$keyValueData
  )
  
  $session = New-PSSession -ComputerName $serverFqdn
  
  Invoke-Command -Session $session -ScriptBlock {
    try {
      if (!(Test-Path -Path $using:regKeyPath)) {
        New-Item -Path $using:regKeyPath -Force
      }
      if (Test-Path -Path "$using:regKeyPath\$using:keyValueName") {
        Set-ItemProperty -Path "$using:regKeyPath" -Name $using:keyValueName -Value $using:keyValueData -Force -ErrorAction Stop
      }
      else {
        New-ItemProperty -Path "$using:regKeyPath" -Name $using:keyValueName -Value $using:keyValueData -Force -ErrorAction Stop
      }
    }
    catch {
      Write-Host "FAIL - Failed to set registry key value: $using:keyValueName"
      [System.Environment]::Exit(0)
    }
  } -ArgumentList $regKeyPath, $keyValueName, $keyValueData
  
  Remove-PSSession $session
}


<#############################################################
# function to check task timeout 
#############################################################>
function isTimeout {
  $currentDateTime = Get-Date
  $LastModifiedTime = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime"
  $timeDifference = New-TimeSpan -Start $LastModifiedTime -End $currentDateTime.ToString("yyyy-MM-dd HH:mm:ss")
  if ($timeDifference.TotalMinutes -ge 60) {
    return $true
  }
  return $false
}


<#############################################################
# function to read registry keyValue 
#############################################################>
function regKeyValueRead {
  param ( $serverFqdn,
    $regKeyPath,
    $keyValueName)
  $session = New-PSSession -ComputerName $serverFqdn
  Invoke-Command -Session $session -ScriptBlock {
    try {
      $value = Get-ItemPropertyValue -Path $using:regKeyPath -Name $using:keyValueName -ErrorAction Stop
      return $value
    }
    catch {
      Write-Host "FAIL - Failed to read registry key value:$keyValueName"
      [System.Environment]::Exit(0)
    }
  } -ArgumentList $serverFqdn, $regKeyPath, $keyValueName
  Remove-PSSession $session
}

<##############################################################
# function to delete registry keyValue
##############################################################>
function deleteRegKeyValueName {
  Param($serverFqdn,
    $keyValueName,
    $regKeyPath
  )
  $session = New-PSSession -ComputerName $serverFqdn
  Invoke-Command -Session $session -ScriptBlock {
    try {
      if (Get-ItemProperty -Path  $using:regKeyPath -Name  $using:keyValueName -ErrorAction Stop) {
        Remove-ItemProperty -Path  $using:regKeyPath -Name  $using:keyValueName -ErrorAction Stop
      }
    }
    catch {
      return 
    } 
        
  } -ArgumentList $serverFqdn, $regKeyPath, $keyValueName
  Remove-PSSession $session
}

<#############################################################
# function to get registry values from target server
#############################################################>
function checkKeyValueExists {
  Param( $serverFqdn,
    $regKeyPath,
    $keyValueName) 
  $session = New-PSSession -ComputerName $serverFqdn 
  Invoke-Command -Session $session -ScriptBlock {
    try {
      $keyProperties = Get-ItemProperty -Path $using:regKeyPath -ErrorAction Stop
      if ($keyProperties.PSObject.Properties.Name -contains $using:keyValueName) {
        return $true
      }
    }
    catch {
      return $false
    } 
     
  } -ArgumentList $regKeyPath, $keyValueName
  Remove-PSSession $session
} 

$inputVersion = [version]($version.Trim())
$baseVersion = [version]"8.14.0"
if ($inputVersion -ge $baseVersion) {
  $global:vdmRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Omnissa\Horizon\"
  $global:lcmRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Omnissa\Horizon\LifecycleManager\"
  $global:programData = "C:\ProgramData\Omnissa\Horizon"
}
elseif ($inputVersion -lt $baseVersion) {
  $global:vdmRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\VMware, Inc.\VMware VDM\"
  $global:lcmRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\VMware, Inc.\VMware VDM\LifecycleManager\"
  $global:programData = "C:\ProgramData\VMware\VDM"
}

if (checkKeyValueExists -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs") {
  $controllerKeyValue = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
  if (-not ([string]::IsNullOrEmpty($controllerKeyValue))) {
    Write-host "Controller key is not null :$controllerKeyValue"
    if (isTimeout) {
      write-host "Task timedout updating ControllerCs and LastModifiedTime"
      setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
      setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
      $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
      if ($invokingServer -eq $sourceServer) {
        write-host "Scheduling the task for timeout and controller server match"
        deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
        scheduleTask -serverFqdn $targetServerFqdn -msiArgs $msiArgs -installDir $installDir -sourceServer $sourceServer -installerFilename $installerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password -vdmRegKeyPath $global:vdmRegKeyPath -lcmRegKeyPath $global:lcmRegKeyPath -programData $global:programData
        scheduleTask -serverFqdn $targetServerFqdn -msiArgs $msiArgs -installDir $installDir -sourceServer $sourceServer -installerFilename $installerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password -vdmRegKeyPath $global:vdmRegKeyPath -lcmRegKeyPath $global:lcmRegKeyPath -programData $global:programData
      }
      else {
        write-host "FAIL - Execution of task scheduled by $invokingServer is in progress."
      }
    }
    else {
      $currentStatus = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "Status"
      $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
      if ($invokingServer -eq $sourceServer -and $statusValues -notcontains $currentStatus ) {
        write-host "Scheduling the task even when timeout not happenned and controller server match"
        deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
        scheduleTask -serverFqdn $targetServerFqdn -msiArgs $msiArgs -installDir $installDir -sourceServer $sourceServer -installerFilename $installerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password -vdmRegKeyPath $global:vdmRegKeyPath -lcmRegKeyPath $global:lcmRegKeyPath -programData $global:programData
      }
      elseif ($invokingServer -eq " " -and $statusValues -notcontains $currentStatus) {
        setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
        setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
        write-host "Scheduling the task even when timeout not happenned"
        deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
        scheduleTask -serverFqdn $targetServerFqdn -msiArgs $msiArgs -installDir $installDir -sourceServer $sourceServer -installerFilename $installerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password -vdmRegKeyPath $global:vdmRegKeyPath -lcmRegKeyPath $global:lcmRegKeyPath -programData $global:programData
      }
      else {
        write-host "FAIL - Execution of task scheduled by $invokingServer is in progress."
      } 
    }
  }
  else {
    setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
    setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
    $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
    if ( $invokingServer -eq $sourceServer ) {
      Write-host "Scheduling the task from Controller Server $invokingServer" 
      deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
      scheduleTask -serverFqdn $targetServerFqdn -msiArgs $msiArgs -installDir $installDir -sourceServer $sourceServer -installerFilename $installerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password -vdmRegKeyPath $global:vdmRegKeyPath -lcmRegKeyPath $global:lcmRegKeyPath -programData $global:programData
    }
    else { write-host "FAIL - Execution of task scheduled by $invokingServer is in progress." }
  }
}
else {
  setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
  setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
  $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
  if ( $invokingServer -eq $sourceServer ) {
    write-host "Scheduling the task from Controller Server $invokingServer"
    deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
    scheduleTask -serverFqdn $targetServerFqdn -msiArgs $msiArgs -installDir $installDir -sourceServer $sourceServer -installerFilename $installerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password -vdmRegKeyPath $global:vdmRegKeyPath -lcmRegKeyPath $global:lcmRegKeyPath -programData $global:programData
  }
      
}

