# Copyright (c) Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Restricted
param (
    [string]$sourceServer,
    [string]$InstallerFilename,
    [string]$version,
    [string]$buildNumber,
    [string]$url,
    [string]$expectedFileSize,
    [string]$expectedChecksum,
    [string]$horizonRegKeyEncodedPath,
    [string]$lcmRegKeyEncodedPath,
    [string]$programdata

)

$timeoutSeconds = 1800
$timestamp = Get-Date -Format "yyyy-MM-dd-HHmmss"
$tempDirectory = [System.IO.Path]::GetTempPath()
$logfolder = "\logs\"
$Logfile = Join-Path (Join-Path $programdata $logfolder) "LifecycleManager-$timestamp.txt"
$exeFile = Join-Path $tempDirectory $InstallerFilename
$horizonRegKeyPath = [Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($horizonRegKeyEncodedPath))
$lcmRegKeyPath = [Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($lcmRegKeyEncodedPath))

<#############################################################
 # function to log messages to file
 #############################################################>
function WriteLog {
    Param ([string]$LogString)
    $Stamp = (Get-Date).toString("yyyy/MM/dd HH:mm:ss")
    $LogMessage = "$Stamp $LogString"
    #  WriteLog $LogMessage
    Add-Content $Logfile -Value $LogMessage
}

<###################################################################
# function to set registry key value name . If key value name is not 
# present then we create key value and update the value data
####################################################################>
function setRegKeyValue {
    Param(
        $keyValueName,
        $keyValueData,
        $lcmRegKeyPath
    )
    try {
        if (Test-Path -Path (Join-Path $lcmRegKeyPath $keyValueName)) {
            write-host "true" $lcmRegKeyPath 
            Set-ItemProperty -Path $lcmRegKeyPath -Name $keyValueName -Value $keyValueData -Force -ErrorAction Stop
        }
        else {
            write-host "false" $lcmRegKeyPath
            New-ItemProperty -Path $lcmRegKeyPath -Name $keyValueName -Value $keyValueData -Force -ErrorAction Stop
        } 
    }
    catch {
        Write-Host "Failed to set registry key value: $keyValueName"
        throw "Failed to set registry key value: $keyValueName"
    }
}

<#############################################################
 # funcction to read registry key value
 #############################################################>
function regKeyValueRead {
    param ( [string]$keyValueName,
        $lcmRegKeyPath)
    try {
        $value = Get-ItemPropertyValue -Path $lcmRegKeyPath -Name $keyValueName -ErrorAction Stop
        return $value
    }
    catch [System.Management.Automation.ItemNotFoundException] { 
        WriteLog "Failed to read registry key value : $keyValueName"
        throw "Failed to read registry key value : $keyValueName"
    }
    catch {
        WriteLog "Failed to find registry key path $lcmRegKeyPath"
        throw "Failed to find registry key path $lcmRegKeyPath"
    }
}

<#############################################################
 # function to check is target server is upgradable
 #############################################################>

function checkIsUpgradable {
    Param($version, $buildNumber, $lcmRegKeyPath)
    $currentVersion = Get-ItemProperty -Path $horizonRegKeyPath -Name "ProductVersion" | Select-Object -ExpandProperty ProductVersion
    WriteLog $currentVersion
    if ($currentVersion) {
        WriteLog $currentVersion
        $targetVersion = "$version.$buildNumber"
        $referenceComponents = $currentVersion.Split('.')
        $referenceMajorMinor = $referenceComponents[0..2] -join '.'

        if ($version -gt $referenceMajorMinor) {
            WriteLog "The  $targetVersion is greater than the installed version $currentVersion.Triggering Upgrade flow"
            setRegKeyValue "Status" "Triggering_Server_Upgrade_Flow" $lcmRegKeyPath
        }
        elseif ($version -eq $referenceMajorMinor) {
            $referenceDecimal = [double]$referenceComponents[-1]

            if ($buildNumber -gt $referenceDecimal) {
                WriteLog "The $version is same as the installed version $referenceMajorMinor. But Build number is different, so performing build to build upgrade" 
                setRegKeyValue "Status" "Triggering_Server_Upgrade_Flow" $lcmRegKeyPath
            }
            elseif ($buildNumber -lt $referenceDecimal) {
                WriteLog "Given build version $targetVErsion is less than the installed version $currentVersion"
                setRegKeyValue "Status" "Upgrade_Not_Supported" $lcmRegKeyPath
                setRegKeyValue "ErrorMessage" "Build_version_less_than_installed_version" $lcmRegKeyPath
                setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
                throw "Given build version $targetVErsion is less than the installed version $currentVersion"
            }
            else {
                WriteLog "Give nbuild version $targetVErsion is equal to installed version $currentVersion"
                setRegKeyValue "Status" "Upgrade_Not_Supported" $lcmRegKeyPath
                setRegKeyValue "ErrorMessage" "Build_version_equal_to_installed_version" $lcmRegKeyPath
                setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
                throw "Give nbuild version $targetVErsion is equal to installed version $currentVersion"

            }
        }
        else {
            WriteLog "Given build version $targetVersion is less than the installed version $currentVersion"
            setRegKeyValue "Status" "Upgrade_Not_Supported" $lcmRegKeyPath
            setRegKeyValue "ErrorMessage" "Build_version_less_than_installed_version" $lcmRegKeyPath
            setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
            throw "Given build version $targetVersion is less than the installed version $currentVersion"
        }
    }
    else {
        WriteLog "Unable to detect Horizon Server installation on the specifed server"
        setRegKeyValue "Status" "Upgrade_Not_Supported" $lcmRegKeyPath
        setRegKeyValue "ErrorMessage" "HorizonServer_installation_not_found" $lcmRegKeyPath
        setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
        throw "Unable to detect Horizon Server installation on the specifed server"
    }

}

<###########################################################################
# function to check ADAM_VMwareVDMDS or ADAM_VMwareVDMDSG service installed 
############################################################################>
function isAdamServiceFound {
    Param([string]$horizonRegKeyPath)
    $adamServices = Get-Service | Where-Object { $_.Name -like "ADAM_*" }
    WriteLog "VDM Reg key path: " $horizonRegKeyPath
    WriteLog "ADAM services found : " $adamServices
    $oldLocalADAMService = "ADAM_VMwareVDMDS"
    $oldGlobalADAMService = "ADAM_VMwareVDMDSG"
    $newLocalADAMService = "ADAM_OmnissaHzeDS"
    $newGlobalADAMService = "ADAM_OmnissaHzeDSG"
    $oldRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\VMware, Inc.\VMware VDM"
    $newRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Omnissa\Horizon"
    if ($adamServices) {
        if ($horizonRegKeyPath -eq $oldRegKeyPath) {
            foreach ($service in $adamServices) {
                WriteLog "Service name:$($service.Name)"
                if ($oldLocalADAMService -eq $($service.Name) -or $oldGlobalADAMService -eq $($service.Name)) {
                    WriteLog "old ADMD service"
                    return $true
                }
            }
        } 
        elseif ($horizonRegKeyPath -eq $newRegKeyPath) {
            foreach ($service in $adamServices) {
                WriteLog  "Service name:$($service.Name)"
                if ($newLocalADAMService -eq $($service.Name) -or $newGlobalADAMService -eq $($service.Name)) {
                    WriteLog "new ADMD service"
                    return $true
                }
            }
        }
        
    }
    else {
        WriteLog  "No ADAM services found."
        return $false
    }
}

<############################################################
 # function to check build source and trigger download
 ############################################################>
function checkBuildPathAndDownload{
    Param($buildPath, $exeFile, $expectedFileSize, $timeoutSeconds, $keyPath)

    if ($buildPath -imatch "^https://") {
        WriteLog "Downloading build from webserver"
        downloadBuild $buildPath $exeFile $expectedFileSize $timeoutSeconds $keyPath
    } elseif ($buildPath -imatch "^smb://") {
        WriteLog "Copying build from SMB share"
        CopyFromSMBShare -smbPath $buildPath -destination $exeFile -expectedFileSize $expectedFileSize -keyPath $keyPath
    } else {
        WriteLog "Invalid path format. Only HTTPS and SMB paths are supported."
        setRegKeyValue "Status" "Build_Download_Failed" $keyPath
        setRegKeyValue "ErrorMessage" "Build download failed " $keyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $keyPath
        throw "Build_Download_Failed"
    }
}

<############################################################
# function to copy build from smb share on target server
############################################################>
function CopyFromSMBShare {
    param (
        [string]$smbPath,
        [string]$destination,
        [string]$expectedFileSize,
        [string]$keyPath)

    WriteLog "Copying from SMB share: $smbPath"
    setRegKeyValue "Status" "Build_Download_InProgress" $keyPath
    $smbPath = $smbPath -replace "^smb:", "" -replace "//","\\"
    try{
        Copy-Item -Path $smbPath -Destination $destination -Force -ErrorAction Stop

        # Check if file exists and verify size
        if (Test-Path $destination) {
            $actualSize = (Get-Item $destination).Length
            if ($actualSize -eq $expectedFileSize) {
                WriteLog "Copy successful. File size matches: $actualSize bytes"
                setRegKeyValue "Status" "Build_Download_Completed" $keyPath
            } else {
                WriteLog "Expected size $expectedFileSize , but got $actualSize"
                setRegKeyValue "Status" "Build_Download_Failed" $keyPath
                setRegKeyValue "ErrorMessage" "actual file size: $actualSize not equal to expected file size: $expectedFileSize" $keyPath
	        throw "Build_Download_Failed"
            }
        } else {
            WriteLog "File not found at destination"
            setRegKeyValue "Status" "Build_Download_Failed" $keyPath
            setRegKeyValue "ErrorMessage" "File not found at destination" $keyPath
            throw "Build_Download_Failed"
        }
    } catch {
        WriteLog "Error: $_"
        setRegKeyValue "Status" "Build_Download_Failed" $keyPath
        setRegKeyValue "ErrorMessage" "Build download failed " $keyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $keyPath
        throw "Build_Download_Failed"
    }
}

<#############################################################
 # function to download build on target server
 #############################################################>
function downloadBuild {
    Param($url, $exeFile, $expectedFileSize, $timeoutSeconds, $lcmRegKeyPath)
    Add-Type -AssemblyName System.Net.Http
    try {        
        $retry = 3
        do {
            $httpClient = New-Object System.Net.Http.HttpClient
            $httpClient.Timeout = [System.TimeSpan]::FromSeconds($timeoutSeconds)
            WriteLog "Updating Status to Download InProgress"
            setRegKeyValue "Status" "Build_Download_InProgress" $lcmRegKeyPath
            $response = $httpClient.GetAsync($url).Result
            $responseStatusCode = [int]$response.StatusCode
            WriteLog "Response code: $responseStatusCode"
            WriteLog "IsSuccessStatusCode : $response.IsSuccessStatusCode"
            if ($response.IsSuccessStatusCode) {
                $content = $response.Content.ReadAsByteArrayAsync().Result
                [System.IO.File]::WriteAllBytes($exeFile, $content)
                $downloadedFileSize = (Get-Item $exeFile).length
                WriteLog "Downloaded file size: $downloadedFileSize"
                WriteLog "Expected file size: $expectedFileSize"
                if ($downloadedFileSize -eq $expectedFileSize) {
                    setRegKeyValue "Status" "Build_Download_Completed" $lcmRegKeyPath
                    WriteLog "Build downloaded successfully."
                    $retry = 0; 
                }
                else {
                    setRegKeyValue "Status" "Build_Download_Failed" $lcmRegKeyPath
                    WriteLog "Downloaded build size: $downloadedFileSize not equal to expected build size: $expectedFileSize."
                    setRegKeyValue "ErrorMessage" "Downloaded build size: $downloadedFileSize not equal to expected build size: $expectedFileSize" $lcmRegKeyPath
                    WriteLog "Clearing ControllerCS reg keyValue"
                    setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
                    throw "Build download failed."
                }
            }
            else {
                setRegKeyValue "Status" "Build_Download_Failed" $lcmRegKeyPath
                setRegKeyValue "ErrorMessage" "Build download failed with error code : $responseStatusCode" $lcmRegKeyPath
                WriteLog "Build download failed with error "+ $response.Content $lcmRegKeyPath
                WriteLog "Error: HTTP status code "  $responseStatusCode
                WriteLog "Retry download"
                $httpClient.Dispose()
                $retry--
            }
        } while ($retry -gt 0)
        $downloadStatus = regKeyValueRead "Status" $lcmRegKeyPath
        WriteLog "Downlaod Status $downloadStatus"
        if ($downloadStatus -ne "Build_Download_Completed") {
            WriteLog "Clearing ControllerCS reg keyValue"
            setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
            throw "Build download failed."
        }
    }
    catch {
        WriteLog "Error: $_"
        setRegKeyValue "Status" "Build_Download_Failed" $lcmRegKeyPath
        setRegKeyValue "ErrorMessage" "Build download failed " $lcmRegKeyPath
        WriteLog "Clearing ControllerCS reg keyValue"
        setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
        throw "Build download failed."
    }
    finally {
        $httpClient.Dispose()
    }
}

<#############################################################
 # function to perfrom checksum verification of installer
 #############################################################>
function checksumVerification {
    Param($exeFile, $expectedChecksum, $lcmRegKeyPath)
    Set-ItemProperty -Path $lcmRegKeyPath -Name "Status" -Value "Checksum_Verification_InProgress"
    if (Test-Path $exeFile -PathType Leaf) {
        $fileHash = Get-FileHash -Path $exeFile -Algorithm SHA256
        if ($fileHash.Hash -eq $expectedChecksum) {
            setRegKeyValue "Status" "Checksum_Verification_Completed" $lcmRegKeyPath
            WriteLog "Checksum is valid. File has not been tampered with."
        }
        else {
            setRegKeyValue "Status" "Checksum_Verification_Failed" $lcmRegKeyPath
            setRegKeyValue "ErrorMessage" "Checksum is not valid. The file may have been tampered with." $lcmRegKeyPath
            setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
            WriteLog "Checksum is not valid. The file may have been tampered with."
            WriteLog "Expected checksum: $expectedChecksum"
            WriteLog "Actual checksum: $($fileHash.Hash)"
            throw "Checksum is not valid. The file may have been tampered with."
        }
    }
    else {
        WriteLog "Installer file not found: $exeFile"
        setRegKeyValue "Status" "Checksum_Verification_Failed" $lcmRegKeyPath
        setRegKeyValue "ErrorMessage" "Installer file not found" $lcmRegKeyPath
        setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
        throw "Installer file not found: $exeFile"

    }

}

<############################################################
 # function to check is service in Running status
 ############################################################>
function isServiceRunning {
    param ( $serviceName)
    $serviceStatus = Get-Service -Name $serviceName
    if ($serviceStatus.Status -eq "Running") {
        return $true
    }
    return $false
}


<#############################################################
 # funtion to perfrom horizon server upgrade on target server
#############################################################>
function upgradeServer {
    Param($exeFile, $lcmRegKeyPath, $horizonRegKeyPath)
    $MsiArgs = @(
        "/qn"
    ) 
    setRegKeyValue "Status" "Upgrade_InProgress" $lcmRegKeyPath
    $Install = (Start-Process -Filepath $exeFile -Wait -ArgumentList "/s /v""$MsiArgs" -PassThru)
    $Install.ExitCode
    if ($Install.ExitCode -ne '0') {
        setRegKeyValue "Status" "Upgrade_Failed" $lcmRegKeyPath
        setRegKeyValue "ErrorMessage" $Install.StandardError $lcmRegKeyPath
        setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
        WriteLog "The upgrade failed, remediate error and try again"
        throw "The upgrade failed, remediate error and try again"
    }
    else {
        if (isAdamServiceFound $horizonRegKeyPath) {
            $servicesToCheck = @('wsbroker', 'wstomcat', 'wstunnel', 'PCOIPSG', 'wsmsgbus', 'wsnm', 'VMBlastSG', 'wsdct')
            $allRunning = $true
            $notRunningServices = @()
            foreach ($serviceName in $servicesToCheck) {
                if (isServiceRunning -serviceName $serviceName) {
                    WriteLog "$serviceName is running."
                }
                else {
                    WriteLog "$serviceName is not running."
                    $allRunning = $false
                    $notRunningServices += $serviceName
                }
            } 
            if ($allRunning) {
                WriteLog "Successfully Upgraded Connection Server"
                setRegKeyValue "Status" "Upgrade_Success" $lcmRegKeyPath
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
            }
            else {
                WriteLog "Service $notRunningServices is not Running"
                setRegKeyValue "Status" "Post_Upgrade_Check_Failed" $lcmRegKeyPath
                setRegKeyValue "ErrorMessage" "$notRunningServices is not Running" $lcmRegKeyPath
                WriteLog "$notRunningServices is not Running"
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
            }
        }
        else {
            $enrollmentService = "wsenroll"
            if (isServiceRunning -serviceName $enrollmentService) {
                WriteLog "$enrollmentService is running."
                WriteLog "Successfully Upgraded Enrollment Server"
                setRegKeyValue "Status" "Upgrade_Success" $lcmRegKeyPath
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
            }
            else {
                WriteLog "$enrollmentService is not Running"
                setRegKeyValue "Status" "Post_Upgrade_Check_Failed" $lcmRegKeyPath
                setRegKeyValue "ErrorMessage" "$enrollmentService is not Running" $lcmRegKeyPath
                WriteLog "Clearing ControllerCS reg keyValue"
                setRegKeyValue "ControllerCs" "" $lcmRegKeyPath
            } 
        }
    }
}

$LogFolder = (Join-Path $programdata $logfolder)
if (-not (Test-Path -Path $LogFolder -PathType Container)) {
    New-Item -Path $LogFolder -ItemType Directory -Force
}
WriteLog "Starting Horizon Server Upgrade flow"
WriteLog $lcmRegKeyPath
$sourceServer = regKeyValueRead "ControllerCs" $lcmRegKeyPath
WriteLog "ControllerCs : $sourceServer"
WriteLog "InstallerFilename : $InstallerFilename"
WriteLog "version : $version"
WriteLog "buildNumber: $buildNumber"
WriteLog "url : $url"
WriteLog "expectedFileSize : $expectedFileSize"
WriteLog "expectedChecksum : $expectedChecksum"
WriteLog "exeFile : $exeFile"

try {
    WriteLog "Create and Update process id registry key"
    setRegKeyValue "Pid" $PID $lcmRegKeyPath

    WriteLog "Verify the target server is upgradable or not"
    checkIsUpgradable $version $buildNumber $lcmRegKeyPath

    WriteLog "Download Build"
    checkBuildPathAndDownload $url $exeFile $expectedFileSize $timeoutSeconds $lcmRegKeyPath

    WriteLog "Verify Checksum"
    checksumVerification $exeFile $expectedChecksum $lcmRegKeyPath

    WriteLog "Initiating the server upgrade"
    upgradeServer $exeFile $lcmRegKeyPath $horizonRegKeyPath
}
catch {
    WriteLog "Upgrade failed. Reason - $_"
}
finally {
    $TaskName = "RunHorizonServerUpgrade"
    $TaskExists = Get-ScheduledTask -TaskName $TaskName -ErrorAction SilentlyContinue
    if ($TaskExists) {
        try {
            Unregister-ScheduledTask -TaskName $TaskName -Confirm:$false -ErrorAction Stop
            WriteLog "Scheduled task '$TaskName' deleted successfully."
        }
        catch {
            WriteLog "FAIL - Failed to delete task '$TaskName'. Error: $_"
        }
    }
}