﻿# Copyright (c) Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Restricted
param(
  [string]$targetServerFqdn,
  [string]$InstallerFilename,
  [string]$version,
  [string]$buildNumber,
  [string]$url,
  [string]$expectedFileSize,
  [string]$expectedChecksum,
  [string]$username,
  [string]$password,
  [string]$triggerMethodName = "AddSeconds", # Define triggermethod name (AddSeconds,AddMinutes),schedule task and deleted task intervals. 
  [int]$scheduleTaskAfter = 30 #In future release we can pass this values while invoking API.
)

$sourceServer = [System.Net.Dns]::GetHostEntry([System.Net.Dns]::GetHostName()).HostName
$statusValues = @("Horizon_Server_Upgrade_Scheduled", "Horizon_Server_Install_Scheduled", "Initiating_Server_Install_Flow",
  "Triggering_Install", "Triggering_Server_Upgrade_Flow", "Build_Download_InProgress", "Build_Download_Completed",
  "Checksum_Verification_InProgress", "Checksum_Verification_Completed", "Upgrade_InProgress", "Upgrade_Success", 
  "Post_Upgrade_Check_Inprogress", "Installation_InProgress", "Installation_Success", "Post_Upgrade_Check_InProgress")
$tempDirectory = [System.IO.Path]::GetTempPath()
$global:Logfile = Join-Path $tempDirectory "LifecycleManager.log"
$global:vdmRegKeyPath = $null
$global:programdata = $null

<##############################################################
# function to determine the regKey path
###############################################################>
function getRemoteHorizonRegKeyPath {
  Param([string]$serverFqdn)

  $session = New-PSSession -ComputerName $serverFqdn
  $vmwareRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\VMware, Inc.\VMware VDM"
  $omnissaRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Omnissa\Horizon"
  try {
    # Invoke-Command and return the $regKeyPath from the remote session
    $regKeyPath = Invoke-Command -Session $session -ScriptBlock {
      $vmwareRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\VMware, Inc.\VMware VDM"
      $omnissaRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Omnissa\Horizon"
           
      if (Test-Path -Path $omnissaRegKeyPath) {
        return $omnissaRegKeyPath
      }
      elseif (Test-Path -Path $vmwareRegKeyPath) {
        return $vmwareRegKeyPath
      }
      else {
        return $null
      }
    }
     
    if (!$regKeyPath) {
      Write-Host "FAIL - Failed to get registry key value name"
      [System.Environment]::Exit(0)
    }
    elseif ($regKeyPath -eq $omnissaRegKeyPath) {
      $global:vdmRegKeyPath = $regKeyPath
      $global:lcmRegKeyPath = Join-Path $regKeyPath "\LifecycleManager"
      $global:programdata = "C:\ProgramData\Omnissa\Horizon"
    }
    elseif ($regKeyPath -eq $vmwareRegKeyPath) {
      $global:vdmRegKeyPath = $regKeyPath
      $global:lcmRegKeyPath = Join-Path $regKeyPath "\LifecycleManager"
      $global:programdata = "C:\ProgramData\VMware\VDM"
    }

  }
  catch {
    Write-Error "An error occurred: $_"
    [System.Environment]::Exit(0)
  }

  Remove-PSSession $session

  return $regKeyPath
}

<##############################################################
# function to determine local regKey path
###############################################################>
function getLocalHorizonRegKeyPath {
  $vmwareRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\VMware, Inc.\VMware VDM"
  $omnissaRegKeyPath = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Omnissa\Horizon"
  try {
    $regKeyPath = Invoke-Command -ScriptBlock {
      if (Test-Path -Path $omnissaRegKeyPath) {
        return $omnissaRegKeyPath
      }
      elseif (Test-Path -Path $vmwareRegKeyPath) {
        return $vmwareRegKeyPath
      }
      else {
        return $null
      }
    }

  }
  catch {
    Write-Error "FAIL - An error occurred: $_"
    [System.Environment]::Exit(0)
  }

  return $regKeyPath

}


<##############################################################
# function to log messages to file
###############################################################>
function WriteLog {
  Param ([string]$LogString)
  $Stamp = (Get-Date).toString("yyyy/MM/dd HH:mm:ss")
  $LogMessage = "$Stamp $LogString"
  Add-Content $Logfile -Value $LogMessage
}


<##################################################################
# function to schedule a install script execution on target server
##################################################################>
function scheduleTask {
  Param($serverFqdn,
    $sourceServer,
    $InstallerFilename,
    $version,
    $buildNumber,
    $url,
    $expectedFileSize,
    $expectedChecksum,
    $triggerMethodName,
    $scheduleTaskAfter,
    $username,
    $password)

  #Get location for powershell scripts on connection server
  if (Test-Path -Path 'HKLM:\SOFTWARE\Omnissa\Horizon\'){
      $localInstallPath = Get-ItemPropertyValue -Path 'HKLM:\SOFTWARE\Omnissa\Horizon\' -Name "ServerInstallPath"}
  elseif (Test-Path -Path 'HKLM:\SOFTWARE\Vmware Inc,\VDM\'){
      $localInstallPath = Get-ItemPropertyValue -Path 'HKLM:\SOFTWARE\Vmware Inc,\VDM\' -Name "ServerInstallPath"}
  $SourceSchedularFile = Join-Path (Join-Path $localInstallPath  "lcm" ) "ScheduleUpgradeRunner.ps1"
  $SourceUpgradeRunnerFile = Join-Path (Join-Path $localInstallPath  "lcm") "UpgradeRunner.ps1"

  $session = New-PSSession -ComputerName $serverFqdn
  # Get user temp directory location
  Invoke-Command -Session $session -ScriptBlock {
    param($programdata)
    if (-not (Test-Path -Path $programdata)) {
      write-host "no program data"
      New-Item -Path $programdata -ItemType Directory
    }
  }-ArgumentList $global:programdata

  $DestinationSchedularFile = Join-Path $global:programdata "ScheduleUpgradeRunner.ps1"
  $DestinationUpgradeRunnerFile = Join-Path $global:programdata "UpgradeRunner.ps1"

  Copy-Item -Path $SourceSchedularFile -Destination $DestinationSchedularFile -ToSession $Session
  Copy-Item -Path $SourceUpgradeRunnerFile -Destination $DestinationUpgradeRunnerFile -ToSession $Session

  $securePassword = ConvertTo-SecureString $password -AsPlainText -Force
  try {
    Invoke-Command -Session $session -ScriptBlock {
      param($DestinationSchedularFile, $UpgradeRunnerFile, $sourceServer, $InstallerFilename, $version, $buildNumber, $url, $expectedFileSize, $expectedChecksum, $triggerMethodName, $scheduleTaskAfter, $username, $securePassword, $vdmRegKeyPath, $lcmRegKeyPath, $programdata) 
      & $using:DestinationSchedularFile -UpgradeRunnerFile $UpgradeRunnerFile -sourceServer $using:sourceServer -InstallerFilename $InstallerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -securePassword $using:securePassword -vdmRegKeyPath $vdmRegKeyPath -lcmRegKeyPath $lcmRegKeyPath -programdata $programdata
    } -ArgumentList $DestinationSchedularFile, $DestinationUpgradeRunnerFile, $sourceServer, $InstallerFilename, $version, $buildNumber, $url, $expectedFileSize, $expectedChecksum, $triggerMethodName, $scheduleTaskAfter, $username, $securePassword, $global:vdmRegKeyPath, $global:lcmRegKeyPath, $global:programdata
  }
  catch {
    Write-Error "FAIL - An error occurred: $_"
    $exitCode = 1
  }
  Remove-PSSession $Session
}

<###################################################################
# function to set registry key value name . If key value name is not 
# present then we create key value and update the value data
####################################################################>
function setRegKeyValue {
  Param(
    [string]$serverFqdn,
    [string]$regKeyPath,
    [string]$keyValueName,
    [string]$keyValueData
  )
  
  $session = New-PSSession -ComputerName $serverFqdn
  
  Invoke-Command -Session $session -ScriptBlock {
    try {
      if (!(Test-Path -Path $using:regKeyPath)) {
        New-Item -Path $using:regKeyPath -Force
      }
      if (Test-Path -Path "$using:regKeyPath\$using:keyValueName") {
        Set-ItemProperty -Path "$using:regKeyPath" -Name $using:keyValueName -Value $using:keyValueData -Force -ErrorAction Stop
      }
      else {
        New-ItemProperty -Path "$using:regKeyPath" -Name $using:keyValueName -Value $using:keyValueData -Force -ErrorAction Stop
      }
    }
    catch {
      Write-Host "FAIL - Failed to set registry key value: $using:keyValueName"
      [System.Environment]::Exit(0)
    }
  } -ArgumentList $regKeyPath, $keyValueName, $keyValueData
  
  Remove-PSSession $session
}

<##############################################################
# function to check task timeout 
##############################################################>
function isTimeout {
  $currentDateTime = Get-Date
  $LastModifiedTime = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime"
  $timeDifference = New-TimeSpan -Start $LastModifiedTime -End $currentDateTime.ToString("yyyy-MM-dd HH:mm:ss")
  if ($timeDifference.TotalMinutes -ge 60) {
    return $true
  }
  return $false
}


<#############################################################
# function to read registry keyValue 
#############################################################>
function regKeyValueRead {
  param ( $serverFqdn,
    $regKeyPath,
    $keyValueName)
  $session = New-PSSession -ComputerName $serverFqdn
  Invoke-Command -Session $session -ScriptBlock {
    try {
      $value = Get-ItemPropertyValue -Path $using:regKeyPath -Name $using:keyValueName -ErrorAction Stop
      return $value
    }
    catch {
      Write-Host "FAIL - Failed to read registry key value: $using:keyValueName"
      [System.Environment]::Exit(0)
    }
  } -ArgumentList $serverFqdn, $regKeyPath, $keyValueName
  Remove-PSSession $session
}

<##############################################################
# function to delete registry keyValue
##############################################################>
function deleteRegKeyValueName {
  Param($serverFqdn,
    $keyValueName,
    $regKeyPath
  )
  $session = New-PSSession -ComputerName $serverFqdn
  Invoke-Command -Session $session -ScriptBlock {
    try {
      if (Get-ItemProperty -Path  $using:regKeyPath -Name  $using:keyValueName -ErrorAction Stop) {
        Remove-ItemProperty -Path  $using:regKeyPath -Name  $using:keyValueName -ErrorAction Stop
      }
    }
    catch {
      # We continue to execute script since we check the Status first
      # if the Status is not Failed the ErrorMessage Key is not read and ignored
      return 
    } 
  } -ArgumentList $serverFqdn, $regKeyPath, $keyValueName
  Remove-PSSession $session
}

<##############################################################
# function to get registry values from target server
##############################################################>
function checkParentKeyExists {
  Param( $serverFqdn,
    $regKeyPath,
    $keyValueName) 

  $session = New-PSSession -ComputerName $serverFqdn 
  Invoke-Command -Session $session -ScriptBlock {
    try {
      $keyProperties = Get-ItemProperty -Path $using:regKeyPath -ErrorAction Stop
      if ($keyProperties.PSObject.Properties.Name -contains $using:keyValueName) {
        return $true
      }
    }
    catch {
      return $false
    } 
     
  } -ArgumentList $regKeyPath, $valueName
  Remove-PSSession $session
} 


write-host "Identify Registry Keys and Folder Path"
getRemoteHorizonRegKeyPath -serverFqdn $targetServerFqdn

if (checkParentKeyExists -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs") {
  $controllerKeyValue = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
  if (-not ([string]::IsNullOrEmpty($controllerKeyValue))) {
    if (isTimeout) {
      setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
      setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
      $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
      if ($invokingServer -eq $sourceServer) {
        write-host "Scheduling the task for timeout and controller server match"
        deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
        scheduleTask -serverFqdn $targetServerFqdn -sourceServer $sourceServer -InstallerFilename $InstallerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password
      }
      else {
        write-host "FAIL - Execution of task scheduled by $invokingServer is in progress."
      }
    }
    else {
      $currentStatus = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "Status"
      $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
      write-host $invokingServer
      if ($invokingServer -eq $sourceServer -and $statusValues -notcontains $currentStatus ) {
        write-host "Scheduling the task even when timeout not happenned and controller server match"
        deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
        scheduleTask -serverFqdn $targetServerFqdn -sourceServer $sourceServer -InstallerFilename $InstallerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password
      }
      elseif ($invokingServer -eq " " -and $statusValues -notcontains $currentStatus) {
        setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
        setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
        write-host "Scheduling the task even when timeout not happenned"
        deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
        scheduleTask -serverFqdn $targetServerFqdn -sourceServer $sourceServer -InstallerFilename $InstallerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password
      }
      else {
        write-host "FAIL - Execution of task scheduled by $invokingServer is in progress."
      } 
    }
  }
  else {
    setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
    setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
    $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
    if ( $invokingServer -eq $sourceServer ) {
      Write-host "Scheduling the task from Controller Server $invokingServer"
      deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath 
      scheduleTask -serverFqdn $targetServerFqdn -sourceServer $sourceServer -InstallerFilename $InstallerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password
    }
    else { write-host "FAIL - Execution of task scheduled by $invokingServer is in progress." }
  }
}
else {
  Write-host "Creating RegKey"
  setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs" -keyValueData $sourceServer
  setRegKeyValue -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "LastModifiedTime" -keyValueData (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
  $invokingServer = regKeyValueRead -serverFqdn $targetServerFqdn -regKeyPath $global:lcmRegKeyPath -keyValueName "ControllerCs"
  if ( $invokingServer -eq $sourceServer ) {
    write-host "Scheduling the task from Controller Server $invokingServe"
    deleteRegKeyValueName $targetServerFqdn "ErrorMessage" $global:lcmRegKeyPath
    scheduleTask -serverFqdn $targetServerFqdn -sourceServer $sourceServer -InstallerFilename $InstallerFilename -version $version -buildNumber $buildNumber -url $url -expectedFileSize $expectedFileSize -expectedChecksum $expectedChecksum -triggerMethodName $triggerMethodName -scheduleTaskAfter $scheduleTaskAfter -username $username -password $password
  }
      
}