#
# Copyright (c) 2022-2024 Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Public
#

<#
    .SYNOPSIS
     Sample script supported on powershell 7 to deploy a UAG virtual appliance to Microsoft Azure
    .EXAMPLE
     .\uagdeployaz_ps7.ps1 uag1.ini
#>

param([string]$iniFile = "uag.ini", [string] $rootPwd, [string] $adminPwd, [string] $ceipEnabled,
     [string] $awAPIServerPwd, [string] $awTunnelGatewayAPIServerPwd, [string] $awCGAPIServerPwd, [string] $awSEGAPIServerPwd,[string] $newAdminUserPwd)

<#
    Az module v11+ onwards, default SecurityType 'TrustedLaunch' is applied while creating VM in Azure which on previous
    versions were defaulted to 'Standard' unless explicitly passed as parameter in New-AzVMConfig or New-AzVM command.
    Since UAG is deployed on Azure with VM size Standard_A4_V2 which belongs to an unsupported family w.r.t
    TrustedLaunch, deployment will fail. Hence explicitly passing SecurityType as 'Standard'
#>
$securityType = "Standard"

#
# Function to create an Azure Network Interface Card (NIC)
#

function CreateNIC {
    Param ($settings, $nic)

    $virtualNetworkName = $settings.Azure.virtualNetworkName
    $subnetName = $settings.Azure.("subnetName"+$nic)
    $resourceGroupName = $settings.Azure.resourceGroupName
    $nicName = $settings.General.name+"-eth"+$nic
    $publicIPAddressName = $settings.Azure.("publicIPAddressName"+$nic)
    $networkSecurityGroupName = $settings.Azure.("networkSecurityGroupName"+$nic)

    $vNetResourceGroupName = $settings.Azure.vNetResourceGroupName
    if($vNetResourceGroupName.Length -le 0)
    {
        $vNetResourceGroupName = $resourceGroupName
    }
    $vNet = Get-AzVirtualNetwork -Name $virtualNetworkName -ResourceGroupName $vNetResourceGroupName -ErrorAction Ignore -WarningAction SilentlyContinue

    if ($publicIPAddressName.length -gt 0) {
        $publicIpAddrResourceGroupName=$settings.Azure.publicIpAddrResourceGroupName
        if( $publicIpAddrResourceGroupName.Length -le 0)
        {
            $publicIpAddrResourceGroupName = $resourceGroupName
        }
        $pip=Get-AzPublicIpAddress -Name $publicIPAddressName -ResourceGroupName $publicIpAddrResourceGroupName -ErrorAction Ignore -WarningAction SilentlyContinue

        $pipParam = @{
            "PublicIpAddressId"=$pip.Id
        }
    } else {
        $pipParam = @{}
    }

    if ($networkSecurityGroupName.length -gt 0) {
        $nsgResourceGroupName=$settings.Azure.nsgResourceGroupName
        if($nsgResourceGroupName.Length -le 0)
        {
            $nsgResourceGroupName = $resourceGroupName
        }
        $nsg=Get-AzNetworkSecurityGroup -Name $networkSecurityGroupName -ResourceGroupName $nsgResourceGroupName -ErrorAction Ignore -WarningAction SilentlyContinue

        $nsgParam = @{
            "NetworkSecurityGroupId"=$nsg.Id
        }
    } else {
        $nsgParam = @{}
    }

    $subnetId = $vnet.Subnets[0].Id

    if ($subnetName.Length -gt 0) {
        $sn = Get-AzVirtualNetworkSubnetConfig -Name $subnetName -VirtualNetwork $vnet -ErrorAction Ignore -WarningAction SilentlyContinue
        if ($sn) {
            $subnetId = $sn.Id
        }
    }

    $newnic = New-AzNetworkInterface -Name $nicName -ResourceGroupName $resourceGroupName -Location $settings.Azure.location -Force -SubnetId $subnetId @pipParam @nsgParam -WarningAction SilentlyContinue

    If([string]::IsNullOrEmpty($newnic)) {
        $msg = $error[0]
        WriteErrorString "Error: Failed to create NIC$nic - $msg"
        Exit
    }

    $newnic
}

#
# Function to validate Azure network settings from values specified in the .INI file
#

function ValidateNetworkSettings {
    Param ($settings, $nic)

    $virtualNetworkName = $settings.Azure.virtualNetworkName
    $subnetName = $settings.Azure.("subnetName"+$nic)
    $resourceGroupName = $settings.Azure.resourceGroupName

    if ($virtualNetworkName.length -gt 0)
    {
        $vNetResourceGroupName = $settings.Azure.vNetResourceGroupName
        if($vNetResourceGroupName.Length -le 0)
        {
            $vNetResourceGroupName = $resourceGroupName
        }
        $vNet = Get-AzVirtualNetwork -Name $virtualNetworkName -ResourceGroupName $vNetResourceGroupName -ErrorAction Ignore -WarningAction SilentlyContinue

        If ( [string]::IsNullOrEmpty($vNet))
        {
            $msg = $error[0]
            WriteErrorString "Error: [Azure] virtualNetworkName ($virtualNetworkName) not found"
            Exit
        }
    } else {
        WriteErrorString "Error: [Azure] virtualNetworkName not specified"
        Exit
    }

    if ($subnetName.Length -gt 0) {
        $sn = Get-AzVirtualNetworkSubnetConfig -Name $subnetName -VirtualNetwork $vnet -ErrorAction Ignore -WarningAction SilentlyContinue
        if (!$sn) {
            WriteErrorString "Error: [Azure] subnetName$nic ($subnetName) not found in virtual network $virtualNetworkName"
            Exit
       }
    }

    $publicIPName = $settings.Azure.("publicIPAddressName"+$nic)

    if ($publicIPName.length -gt 0) {
        $publicIpAddrResourceGroupName=$settings.Azure.publicIpAddrResourceGroupName
        if( $publicIpAddrResourceGroupName.Length -le 0)
        {
            $publicIpAddrResourceGroupName = $resourceGroupName
        }

        $pip=Get-AzPublicIpAddress -Name $publicIPName -ResourceGroupName $publicIpAddrResourceGroupName -ErrorAction Ignore -WarningAction SilentlyContinue

        If([string]::IsNullOrEmpty($pip)) {
            WriteErrorString "Error: [Azure] publicIPAddressName$nic ($publicIPName) not found"
            Exit
        }

        if ($pip.ipConfiguration.length -gt 0) {
            WriteErrorString "Error: [Azure] publicIPAddressName$nic ($publicIPName) is already in use"
            Exit
        }
    }

    $networkSecurityGroupName = $settings.Azure.("networkSecurityGroupName"+$nic)

    if ($networkSecurityGroupName.length -gt 0) {
        $nsgResourceGroupName=$settings.Azure.nsgResourceGroupName
        if($nsgResourceGroupName.Length -le 0)
        {
            $nsgResourceGroupName = $resourceGroupName
        }
        $nsg=Get-AzNetworkSecurityGroup -Name $networkSecurityGroupName -ResourceGroupName $nsgResourceGroupName -ErrorAction Ignore -WarningAction SilentlyContinue

        If([string]::IsNullOrEmpty($nsg)) {
            WriteErrorString "Error: [Azure] networkSecurityGroupName$nic ($networkSecurityGroupName) not found"
            Exit
        }
    }
}

#
# Generate pseudo random password that meets the required Azure complexity rules
#

function GenerateAzureRandomPassword {
    #GeneratePassword function does not work on azure Az module, commented below logic and  written password generation logic to support on powershell 7 with Az module
    #add-type -AssemblyName System.Web
    #$pwd = [System.Web.Security.Membership]::GeneratePassword(12,3)
    #$pwd = "Passw0rd!"+$pwd
	#Write-Host "Password is $pwd"
    #$pwd
	$length = 12
	$charSet = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789{}[]@#$%^()'.ToCharArray()
	$rng = New-Object System.Security.Cryptography.RNGCryptoServiceProvider
	$bytes = New-Object byte[]($length)
	$rng.GetBytes($bytes)
	$result = New-Object char[]($length)
	for ($i = 0 ; $i -lt $length ; $i++) {
	$result[$i] = $charSet[$bytes[$i]%$charSet.Length]
	}
	return "Password!"+(-join $result)

}

function DeleteExistingUAGResources {
    Param ($settings, $uagName)

    $resourceGroupName = $settings.Azure.resourceGroupName
    $storageAccName = $settings.Azure.storageAccountName
    $storageContext = (Get-AzStorageAccount | Where-Object{$_.StorageAccountName -Contains $storageAccName}).Context

    $VMInfo = Get-AzVM  -Name $uagName -ResourceGroupName $resourceGroupName -DisplayHint Expand -ErrorAction Ignore

    If ($VMInfo.Name) {
        $OsDiskUri =  $VMInfo.StorageProfile.OsDisk.Vhd.Uri
        write-host "Existing UAG VM $uagName and associated resources will be deleted"
        #get Managed disk Id of VM
	    $managedDskId = $VMInfo.StorageProfile.OsDisk.ManagedDisk

        $resourceParams = @{
            'Name' = $uagName
            'ResourceType' = 'Microsoft.Compute/virtualMachines'
            'ResourceGroupName' = $resourceGroupName
        }

        $vmResource = Get-AzResource @resourceParams -ErrorAction Ignore

        if ($vmResource) {
            $vmId = $vmResource.Properties.VmId
            $diagContainerName = ('bootdiagnostics-{0}-{1}' -f $VMInfo.Name.ToLower(), $vmId)
        }

	    $rm = Remove-AzVM -ResourceGroupName $resourceGroupName -Name $uagName -Force

        if ($rm) {
            $status = $rm.Status
	        write-host "Deleted UAG VM $uagName - Status=$status"
        }

        #
        #  Delete the VM's disk.
        #
        if($managedDskId) {
	     #ensure keeps same disk name while creating it.
	     $osDiskName = $uagName + "-osdisk"
         #removing existing OSDisk
	     $removedOSDisk = Remove-AzDisk -ResourceGroupName  $resourceGroupName -DiskName  $osDiskName -Force -InformationAction Continue -ErrorAction Ignore
	     if ($removedOSDisk) {
          $status = $removedOSDisk.Status
	      write-host "Deleted UAG VM $uagName OS Disk $osDiskName - Status=$status"
        }
      } elseif ($OsDiskUri) {
        $OsDisk = Split-Path -Leaf $OsDiskUri
        $diskStorageContainer = $settings.Azure.diskStorageContainer.ToLower()
        $out = Remove-AzStorageBlob -Context $storageContext -Container $diskStorageContainer -Blob $OsDisk -Force -InformationAction Continue -ErrorAction Ignore

        $diskStorageContainer2 = Split-Path (split-path $OsDiskUri -Parent) -Leaf
        $out = Remove-AzStorageBlob -Context $storageContext -Container $diskStorageContainer2 -Blob $OsDisk -Force -InformationAction Continue -ErrorAction Ignore
     }

        #
        # Delete the boot diagnostics folder
        #

        if ($diagContainerName) {
            $diagnosticParam = @{
            'ResourceGroupName' = $resourceGroupName
            'Name' =  $storageAccName
            }
            Get-AzStorageAccount @diagnosticParam | Get-AzStorageContainer | where {$_.Name-eq $diagContainerName} |  Remove-AzStorageContainer -Force
         }
    }

    #
    # Delete the NICs
    #

    $out = Remove-AzNetworkInterface -ResourceGroupName $resourceGroupName -Name $($uagName+"-eth0")  -Force
    $out = Remove-AzNetworkInterface -ResourceGroupName $resourceGroupName -Name $($uagName+"-eth1")  -Force
    $out = Remove-AzNetworkInterface -ResourceGroupName $resourceGroupName -Name $($uagName+"-eth2")  -Force

}

#
# Load the dependent UAG PowerShell Module
#
if (-not (Get-InstalledModule -Name Az)) {
    Write-host "Error: Powershell module Az not found.
    Please look into details https://docs.microsoft.com/en-us/powershell/azure/install-az-ps?view=azps-8.2.0 for Az module installation.
    Alternatively Run the command 'Install-Module -Name Az -AllowClobber -Force' to install Az module and retry"
   Exit
}
$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptDir  = Split-Path -Parent $ScriptPath
$uagDeployModule=$ScriptDir+"\uagdeploy.psm1"


if (!(Test-path $uagDeployModule)) {
    Write-host "Error: PowerShell Module $uagDeployModule not found." -foregroundcolor red -backgroundcolor black
    Exit
}



import-module $uagDeployModule -Force -ArgumentList $awAPIServerPwd, $awTunnelGatewayAPIServerPwd, $awCGAPIServerPwd, $awSEGAPIServerPwd



Write-host "Unified Access Gateway (UAG) virtual appliance Microsoft Azure deployment script"

Write-host "Note: Please make sure to use the latest script that came with the build"

if (!(Test-path $iniFile)) {
    WriteErrorString "Error: Configuration file ($iniFile) not found."
    Exit
}

$settings = ImportIni $iniFile

$uagName=$settings.General.name

#
# Login if needed
#

Write-Host -NoNewline "Validating Azure subscription .."

try {
    $out=Get-AzSubscription -ErrorAction Ignore
    }

catch {
    Connect-AzAccount
}

Write-Host -NoNewline "."

if (!$out) {
    try {
        $out=Get-AzSubscription -ErrorAction Ignore
        }

    catch {
        WriteErrorString "Error: Failed to log in to Azure."
        Exit
    }
}

Write-Host -NoNewline "."

if ($settings.Azure.subscriptionID -gt 0) {

    try {
        $out=Set-AzContext -Subscription $settings.Azure.subscriptionID
    }

    catch {
        WriteErrorString "Error: Specified subscriptionID not found."
        Exit
    }
} else {
     WriteErrorString "Error: [Azure] subscriptionID not specified."
     Exit
}

Write-Host ". OK"

$deploymentOption=GetDeploymentSettingOption $settings

if ($uagName.length -gt 32) {
    WriteErrorString "Error: Virtual machine name must be no more than 32 characters in length"
    Exit
}

if (!$uagName) {
    WriteErrorString "Error: [General] name not specified"
    Exit
}

$osLoginUsername = ReadOsLoginUsername $settings
if ($osLoginUsername.length -eq 0) {
    $osLoginUsername = "root"
}

if ($settings.General.dsComplianceOS -eq "true") {
    updatePasswordPolicyForDsComplianceOS $settings
}

if (!$rootPwd) {
    $rootPwd = GetRootPwd $uagName $settings $osLoginUsername
}

if (!$adminPwd) {
    $adminPwd = GetAdminPwd $uagName $settings
}

if (!$ceipEnabled) {
    $ceipEnabled = GetCeipEnabled $uagName
}

$settingsJSON=GetJSONSettings $settings $newAdminUserPwd

SetUp

$ovfFile = "$env:HOME/$uagName.cfg"

if ($Env:OS -like 'win*') {
  $ovfFile = "${env:APPDATA}\VMware\$uagName.cfg"
}

([string[]]("deploymentOption="+"$deploymentOption")) | Set-Content -Path $ovfFile

$dns=$settings.General.dns
if ($dns.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("DNS="+"$dns"))
}

if ($osLoginUsername -ne "root") {
	[IO.File]::AppendAllLines($ovfFile, [string[]]("osLoginUsername="+"$osLoginUsername"))
}

$osMaxLoginLimit = ReadOsMaxLoginLimit $settings
if ($osMaxLoginLimit.length -gt 0) {
	[IO.File]::AppendAllLines($ovfFile, [string[]]("osMaxLoginLimit="+"$osMaxLoginLimit"))
}

$rootPasswordExpirationDays=$settings.General.rootPasswordExpirationDays
if ($rootPasswordExpirationDays.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("rootPasswordExpirationDays="+"$rootPasswordExpirationDays"))
}

$passwordPolicyMinLen=$settings.General.passwordPolicyMinLen
if ($passwordPolicyMinLen.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyMinLen="+"$passwordPolicyMinLen"))
}

$passwordPolicyMinClass=$settings.General.passwordPolicyMinClass
if ($passwordPolicyMinClass.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyMinClass="+"$passwordPolicyMinClass"))
}

$passwordPolicyDifok=$settings.General.passwordPolicyDifok
if ($passwordPolicyDifok.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyDifok="+"$passwordPolicyDifok"))
}

$passwordPolicyUnlockTime=$settings.General.passwordPolicyUnlockTime
if ($passwordPolicyUnlockTime.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyUnlockTime="+"$passwordPolicyUnlockTime"))
}

$passwordPolicyFailedLockout=$settings.General.passwordPolicyFailedLockout
if ($passwordPolicyFailedLockout.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyFailedLockout="+"$passwordPolicyFailedLockout"))
}


$adminPasswordFailedLockoutCount=$settings.General.adminPasswordPolicyFailedLockoutCount
if ($adminPasswordFailedLockoutCount.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPasswordPolicyFailedLockoutCount="+"$adminPasswordFailedLockoutCount"))
}

$adminPasswordMinLen=$settings.General.adminPasswordPolicyMinLen
if ($adminPasswordMinLen.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPasswordPolicyMinLen="+"$adminPasswordMinLen"))
}

$adminPasswordLockoutTime=$settings.General.adminPasswordPolicyUnlockTime
if ($adminPasswordLockoutTime.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPasswordPolicyUnlockTime="+"$adminPasswordLockoutTime"))
}

$adminSessionIdleTimeoutMinutes=$settings.General.adminSessionIdleTimeoutMinutes
if ($adminSessionIdleTimeoutMinutes.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminSessionIdleTimeoutMinutes="+"$adminSessionIdleTimeoutMinutes"))
}

$adminMaxConcurrentSessions = ValidateAdminMaxConcurrentSessions $settings
if ($adminMaxConcurrentSessions.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminMaxConcurrentSessions="+"$adminMaxConcurrentSessions"))
}

$rootSessionIdleTimeoutSeconds = ValidateRootSessionIdleTimeoutSeconds $settings
if ($rootSessionIdleTimeoutSeconds.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("rootSessionIdleTimeoutSeconds="+"$rootSessionIdleTimeoutSeconds"))
}

$commandsFirstBoot = ValidateCustomBootTimeCommands $settings "commandsFirstBoot"
if ($commandsFirstBoot.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("commandsFirstBoot="+"$commandsFirstBoot"))
}
$commandsEveryBoot = ValidateCustomBootTimeCommands $settings "commandsEveryBoot"
if ($commandsEveryBoot.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("commandsEveryBoot="+"$commandsEveryBoot"))
}

$defaultGateway=$settings.General.defaultGateway
if ($defaultGateway.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("defaultGateway="+"$defaultGateway"))
}

$v6DefaultGateway=$settings.General.v6DefaultGateway
if ($v6defaultGateway.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("v6defaultGateway="+"$v6defaultGateway"))
}

$forwardrules=$settings.General.forwardrules
if ($forwardrules.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("forwardrules="+"$forwardrules"))
}

$routes0=$settings.General.routes0
if ($routes0.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes0="+"$routes0"))
}

$routes1=$settings.General.routes1
if ($routes1.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes1="+"$routes1"))
}

$routes2=$settings.General.routes2
if ($routes2.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes2="+"$routes2"))
}

$policyRouteGateway0=$settings.General.policyRouteGateway0
if ($policyRouteGateway0.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway0="+"$policyRouteGateway0"))
}

$policyRouteGateway1=$settings.General.policyRouteGateway1
if ($policyRouteGateway1.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway1="+"$policyRouteGateway1"))
}

$policyRouteGateway2=$settings.General.policyRouteGateway2
if ($policyRouteGateway2.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway2="+"$policyRouteGateway2"))
}

if ($ceipEnabled -eq $true) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("ceipEnabled=true"))
}

if ($settings.General.dsComplianceOS -eq "true") {
	[IO.File]::AppendAllLines($ovfFile, [string[]]("dsComplianceOS=true"))
}

if ($settings.General.tlsPortSharingEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("tlsPortSharingEnabled=true"))
}

if ($settings.General.sshEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshEnabled=true"))
}

if ($settings.General.sshPasswordAccessEnabled -eq "false") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshPasswordAccessEnabled=false"))
}

if ($settings.General.sshKeyAccessEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshKeyAccessEnabled=true"))
}

$sshBannerText=ReadLoginBannerText $settings
if ($sshBannerText.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshLoginBannerText=" + "$sshBannerText"))
}

$sshInterface = validateSSHInterface $settings
if (($sshInterface.length -gt 0)) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshInterface=" + "$sshInterface"))
}

$sshPort = $settings.General.sshPort
if ($sshPort.length -gt 0 -and ($sshPort -match '^[0-9]+$')) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshPort=" + "$sshPort"))
}

$secureRandomSrc=ReadSecureRandomSource $settings
if ($secureRandomSrc.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("secureRandomSource=" + "$secureRandomSrc"))
}

[IO.File]::AppendAllLines($ovfFile, [string[]]("rootPassword="+"$rootPwd"))

if ($adminPwd.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPassword="+"$adminPwd"))
}

$enabledAdvancedFeatures = $settings.General.enabledAdvancedFeatures
if($enabledAdvancedFeatures.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("enabledAdvancedFeatures="+"$enabledAdvancedFeatures"))
}

$gatewaySpec = getGatewaySpec $settings
if($gatewaySpec.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("gatewaySpec="+"$gatewaySpec"))
}

# Adding ConfigData related propertiesFile
$configURL = $settings.General.configURL
if($configURL.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configURL="+"$configURL"))
}

$configKey = $settings.General.configKey
if($configKey.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configKey="+"$configKey"))
}

$configURLThumbprints = $settings.General.configURLThumbprints
if($configURLThumbprints.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configURLThumbprints="+"$configURLThumbprints"))
}

$configURLHttpProxy = $settings.General.configURLHttpProxy
if($configURLHttpProxy.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configURLHttpProxy="+"$configURLHttpProxy"))
}

$adminCsrSubject = $settings.General.adminCsrSubject
if($adminCsrSubject.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminCsrSubject="+"$adminCsrSubject"))
}

$adminCsrSAN = $settings.General.adminCsrSAN
if($adminCsrSAN.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminCsrSAN="+"$adminCsrSAN"))
}

$additionalDeploymentMetadata = $settings.General.additionalDeploymentMetadata
if($additionalDeploymentMetadata.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("additionalDeploymentMetadata="+"$additionalDeploymentMetadata"))
}

$imageURI = $settings.Azure.imageURI
$zone = $settings.Azure.zone
#imageURI is mandatory in the absence of zone
if (!$zone -and $imageURI.length -eq 0) {
    WriteErrorString "Error: [Azure] imageURI not found"
    [IO.File]::Delete($ovfFile)
    Exit
}

$location = $settings.Azure.location

if ($location.length -gt 0) {
    $res =  Get-AzResourceProvider -Location $settings.Azure.location -ProviderNameSpace Microsoft.Compute
    If([string]::IsNullOrEmpty($res)) {
        WriteErrorString "Error: [Azure] location ($location) not found"
        $locations = Get-AzResourceProvider -ProviderNameSpace Microsoft.Compute
        $locationNames = $locations[0].Locations
        WriteErrorString "Specify a location from the following list:"
        for ($i=0; $i -lt $locations[0].Locations.Count; $i++) {
            write-host $locations[0].Locations[$i]
        }
        [IO.File]::Delete($ovfFile)
        Exit
    }
} else {
    WriteErrorString "Error: [Azure] location not specified"
    [IO.File]::Delete($ovfFile)
    Exit
}

#default managed storage account type
$managedStorageAccountType = 'Standard_LRS'
#managed disk image creation  settings
$managedImageName = $settings.Azure.managedImageName
#zone validation
if ($zone.length -gt 0) {
	try {
		 $zoneNum = [int]$zone
	} catch {
    	WriteErrorString "Error: $zone in the section $iniSection is not an integer"
		Exit
	}
    if (!$managedImageName) {
        WriteErrorString "Parameter managedImageName is required in ini file for uag deployment with azure zone"
        Exit
    }
    #removing whitespaces and converts to lowercase
    $locationName=$location -replace '\s+', ''
    $locationName=$locationName.ToLower()
	#validate if zone belongs given azure location.
	 $sku = Get-AzComputeResourceSku | Where-Object { $_.Locations -contains $locationName -and $_.LocationInfo.Zones -ne $null }
     $zoneFound = "false"
     for ($i=0; $i -lt $sku.LocationInfo.Count; $i++) {
            if($sku.LocationInfo[$i].Zones.Contains($zone)){
               $zoneFound  = "true"
               break
            }
      }
	 if ($zoneFound -eq "false") {
      WriteErrorString "Error: zone $zone is not supported on location $location"
      Exit
    }
}

$resourceGroupName = $settings.Azure.resourceGroupName
if ($resourceGroupName.Length -gt 0)
{
    $out = Get-AzResourceGroup -Name $resourceGroupName -ErrorAction SilentlyContinue -WarningAction SilentlyContinue

    if ($out.ResourceId.Length -eq 0)
    {
        $out = New-AzResourceGroup -Name $resourceGroupName -Location $location

        $out
        if ($out.ResourceId.Length -eq 0) {
            WriteErrorString "Error: [Azure] resourceGroupName ($resourceGroupName) not found and could not be created"
            [IO.File]::Delete($ovfFile)
            Exit
        }
    }
} else {
     WriteErrorString "Error: [Azure] resourceGroupName not specified."
    [IO.File]::Delete($ovfFile)
     Exit
}
#check if managed image is found with name
if ($managedImageName) {
    $managedImage = Get-AzImage -ImageName $managedImageName  -ResourceGroupName $resourceGroupName -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
    if ($managedImage.Id.Length -eq 0)
    {
        WriteErrorString "Error: Managed Image with name $managedImageName not found in specified resourceGroup $resourceGroupName."
        [IO.File]::Delete($ovfFile)
        Exit
    }
    Write-Host "Managed Image with name $managedImageName is found in specified resourceGroup $resourceGroupName."
}

$storageAccountName = $settings.Azure.storageAccountName

if ($storageAccountName.length -gt 0) {
    $storageAcc = Get-AzStorageAccount -ResourceGroupName $resourceGroupName -Name $storageAccountName -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
    If($storageAcc.Id.Length -eq 0) {
        $storageAcc = New-AzStorageAccount -ResourceGroupName $resourceGroupName -Name $storageAccountName -Location $location -SkuName Standard_LRS -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
        if ($storageAcc.Id.Length -eq 0) {
            $msg = $error[0]
            WriteErrorString "Error: [Azure] storageAccountName ($storageAccountName) not found and could not be created - $msg"
            [IO.File]::Delete($ovfFile)
            Exit
        }
    }
} else {
    WriteErrorString "Error: [Azure] storageAccountName not specified"
    [IO.File]::Delete($ovfFile)
    Exit
}
$diskStorageContainer = $settings.Azure.diskStorageContainer.ToLower()
#storage container name is not mandatory from ini for zone deployment
if(!$zone) {
    if ($diskStorageContainer.length -gt 0) {
        $container = Get-AzRmStorageContainer -Name $diskStorageContainer -ResourceGroupName $resourceGroupName -StorageAccountName $storageAccountName -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
        If ($container.Name.Length -eq 0) {
            $container = New-AzRmStorageContainer -Name $diskStorageContainer -ResourceGroupName $resourceGroupName -StorageAccountName $storageAccountName -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
            if ($container.Name.Length -eq 0) {
                $msg = $error[0]
                WriteErrorString "Error: [Azure] diskStorageContainer ($diskStorageContainer) not found and could not be created - $msg"
                [IO.File]::Delete($ovfFile)
                Exit
            }
        }
    }
    else {
        WriteErrorString "Error: [Azure] diskStorageContainer not specified"
        [IO.File]::Delete($ovfFile)
        Exit
    }
}

DeleteExistingUAGResources $settings $uagName

$vmSize = $settings.Azure.vmSize
if ($vmSize.length -gt 0) {
    write-host "Deploying $uagName as $vmSize"
} else {
    $vmSize = "Standard_A4_v2"
}

#
# Set up the VM object
#

Write-Host -NoNewline "Creating network interfaces .."

$vm = New-AzVMConfig -VMName $uagName -VMSize $vmSize -SecurityType $securityType

switch -Wildcard ($deploymentOption) {

    'onenic*' {
        ValidateNetworkSettings $settings "0"
        $eth0 = CreateNIC $settings "0"
        $vm = Add-AzVMNetworkInterface -VM $vm -Id $eth0.Id -Primary
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $customConfigEntry0 = GetCustomConfigEntry $settings "0"
        if ($customConfigEntry0.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry0))
        }
    }
    'twonic*' {
        ValidateNetworkSettings $settings "0"
        ValidateNetworkSettings $settings "1"
        $eth0 = CreateNIC $settings "0"
        $vm = Add-AzVMNetworkInterface -VM $vm -Id $eth0.Id -Primary
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $eth1 = CreateNIC $settings "1"
        $vm = Add-AzVMNetworkInterface -VM $vm -Id $eth1.Id
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode1=DHCPV4+DHCPV6"))
        $customConfigEntry0 = GetCustomConfigEntry $settings "0"
        if ($customConfigEntry0.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry0))
        }
        $customConfigEntry1 = GetCustomConfigEntry $settings "1"
        if ($customConfigEntry1.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry1))
        }
    }
    'threenic*' {
        ValidateNetworkSettings $settings "0"
        ValidateNetworkSettings $settings "1"
        ValidateNetworkSettings $settings "2"
        $eth0 = CreateNIC $settings "0"
        $vm = Add-AzVMNetworkInterface -VM $vm -Id $eth0.Id -Primary
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $eth1 = CreateNIC $settings "1"
        $vm = Add-AzVMNetworkInterface -VM $vm -Id $eth1.Id
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode1=DHCPV4+DHCPV6"))
        $eth2 = CreateNIC $settings "2"
        $vm = Add-AzVMNetworkInterface -VM $vm -Id $eth2.Id
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode2=DHCPV4+DHCPV6"))
        $customConfigEntry0 = GetCustomConfigEntry $settings "0"
        if ($customConfigEntry0.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry0))
        }
        $customConfigEntry1 = GetCustomConfigEntry $settings "1"
        if ($customConfigEntry1.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry1))
        }
        $customConfigEntry2 = GetCustomConfigEntry $settings "2"
        if ($customConfigEntry2.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry2))
        }
    }
    default {
        WriteErrorString "Error: Invalid deploymentOption ($deploymentOption)."
        [IO.File]::Delete($ovfFile)
        Exit
    }
}

Write-Host ". OK"

Write-Host -NoNewline "Creating disk image .."

$pwd = GenerateAzureRandomPassword

$securePassword = ConvertTo-SecureString $pwd -AsPlainText -Force
$credential = New-Object System.Management.Automation.PSCredential ("azureuser", $securePassword)

[IO.File]::AppendAllLines($ovfFile, [string[]]("settingsJSON="+"$settingsJSON"))

$ovfProperties = Get-Content -Raw $ovfFile

$customData = [System.Convert]::ToBase64String([System.Text.Encoding]::ASCII.GetBytes($ovfProperties))

if ($zone) {
    Write-Host "UAG $uagName deployment with zone $zone"
    $managedImage = Get-AzImage -ImageName $managedImageName  -ResourceGroupName $resourceGroupName -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
    $vm = Set-AzVMSourceImage -VM $vm -Id $managedImage.Id
    $diskName = $uagName + "-osdisk"
    Write-Host "Managed disk storage type: $managedStorageAccountType"
    $vm = Set-AzVMOSDisk -VM $vm -Name $diskName -StorageAccountType $managedStorageAccountType -DiskSizeInGB '40' -CreateOption FromImage -Caching ReadWrite

} else {

  $guid = [guid]::NewGuid()
  $diskName = $($uagName+"-"+$guid+"-disk1.vhd")

  $osDiskUri = '{0}{1}/{2}' -f $storageAcc.PrimaryEndpoints.Blob.ToString(), $diskStorageContainer, $diskName

  #
  # Associate the starter disk with the VM object
  #
  $vm =  Set-AzVMOSDisk -VM $vm -Name $diskName -VhdUri $osDiskUri -CreateOption fromImage -DiskSizeInGB 40 -SourceImageUri $imageURI -Linux
}

#
# Associate the VM settings with VM object. CustomData is base64 encoded OVF properties pushed to the VM.
#
$vm = Set-AzVMOperatingSystem -VM $vm -Linux -Credential $Credential -ComputerName $uagName -CustomData $customData

Write-Host ". OK"

Write-Host -NoNewline "Creating virtual appliance .."

if($zone) {
    $newvm = New-AzVM -ResourceGroupName $resourceGroupName -Location $location -VM $vm -Zone $zone -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
    if (!$newvm)
    {
        Write-Host ". FAILED"
        $msg = $error[0]
        WriteErrorString "Error: Failed to create $uagName VM on zone $zone ($msg)."
        [IO.File]::Delete($ovfFile)
        Exit
    }

} else {
  $newvm = New-AzVM -ResourceGroupName $resourceGroupName -Location $location -VM $vm -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
  if(!$newvm) {
    Write-Host ". FAILED"
    $msg = $error[0]
    WriteErrorString "Error: Failed to create $uagName VM ($msg)."
    [IO.File]::Delete($ovfFile)
    Exit
  }
}
Write-Host ". OK"

write-host "Deployed $uagName successfully to Azure resource group $resourceGroupName"

[IO.File]::Delete($ovfFile)

