#
# Copyright (c) 2018-2024 Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Public
#

<#
    .SYNOPSIS
     Sample Powershell script to deploy a UAG virtual appliance to Amazon AWS EC2.
    .EXAMPLE
     .\uagdeployec2.ps1 uag1.ini
#>

param([string]$iniFile = "uag.ini", [string] $rootPwd, [string] $adminPwd, [string] $ceipEnabled,
      [string] $awAPIServerPwd, [string] $awTunnelGatewayAPIServerPwd, [string] $awCGAPIServerPwd, [string] $awSEGAPIServerPwd,[string] $newAdminUserPwd)

function SetNICDeleteOnTermination {
    Param ($settings, $eth)

    if ($eth.NetworkInterfaceId.length -gt 0) {
        $net=Get-EC2NetworkInterface -NetworkInterfaceId $eth.NetworkInterfaceId
        if ($net.Attachment.AttachmentId.length -gt 0) {
            Edit-EC2NetworkInterfaceAttribute -NetworkInterfaceId $eth.NetworkInterfaceId -Attachment_DeleteOnTermination $true -Attachment_AttachmentId $net.Attachment.AttachmentId
        }
    }
}

function CreateNIC {
    Param ($settings, $nic, $description)

    $privateIPAddress = $settings.AmazonEC2.("privateIPAddress"+$nic)

    $ipv6AddressCount = $settings.AmazonEC2.("ipv6AddressCount"+$nic)
    $subnetId = $settings.AmazonEC2.("subnetId"+$nic)
    $securityGroupId =  $settings.AmazonEC2.("securityGroupId"+$nic)
    $newnic = "New-EC2NetworkInterface -SubnetId $subnetId -Group $securityGroupId"
    if($privateIPAddress.Length -gt 0) {
        $newnic = "$newnic -PrivateIPAddress $privateIPAddress"
    }
    if($ipv6AddressCount.Length -gt 0) {
        $newnic = "$newnic -Ipv6AddressCount $ipv6AddressCount"
    }
    $newnic = Invoke-Expression $newnic

    If([string]::IsNullOrEmpty($newnic)) {
        $msg = $error[0]
        WriteErrorString "Error: Failed to create NIC$nic - $msg"
        Exit
    }

    New-EC2Tag -Region $awsRegion -Tag @( @{ Key = "Name" ; Value = "$uagName-eth$nic"}) -Resource $newnic.NetworkInterfaceId -Force

    $e = Edit-EC2NetworkInterfaceAttribute -NetworkInterfaceId $newnic.NetworkInterfaceId -Description "$uagName-eth$nic ($description)"

    if ($settings.AmazonEC2.("publicIPId"+$nic).length -gt 0) {
        $r = Register-EC2Address -AllocationId $settings.AmazonEC2.("publicIPId"+$nic) -NetworkInterfaceId $newnic.NetworkInterfaceId
        If([string]::IsNullOrEmpty($r)) {
            $msg = $error[0]
            WriteErrorString "Error: Failed to register address - $msg"
            Exit
        }
    }

    $eth = new-object Amazon.EC2.Model.InstanceNetworkInterfaceSpecification
    $eth.NetworkInterfaceId = $newnic.NetworkInterfaceId
    $eth.DeviceIndex = $nic
    $eth.DeleteOnTermination = $false

    $eth
}

function terminateExistingInstance {
    Param ($settings)

    $statePending="0"
    $stateRunning="16"
    $stateTerminated="48"
    $stateStopped="80"

    $uagName=$settings.General.name

    $i=(Get-EC2Instance -Filter @(@{name="tag:Name";values=$uagName};@{name="instance-state-code";values=$statePending,$stateRunning,$stateStopped})).Instances.InstanceId

    if ($i.Count -eq 1) {
        write-host "Terminating existing instance of $uagName"

        $r=Remove-EC2Instance -InstanceId $i -Force

        while ($true) {
            $j=Get-EC2Instance -InstanceId $i
            if ($j.Instances[0].State.Code -eq $stateTerminated) {
                write-host "Existing instance of $uagName terminated successfully"
                break
            }
            sleep 2
        }
    }
}

function ValidateNetworkSettings {
    Param ($settings, $nic)

    $subnetID = $settings.AmazonEC2.("subnetId"+$nic)

    if ($subnetID.length -gt 0) {

        $subnet = Get-EC2Subnet -SubnetId $subnetID
        If([string]::IsNullOrEmpty($subnet)) {
            $msg = $error[0]
            WriteErrorString "Error: [AmazonEC2] subnetID$nic ($subnetID) not found"
            Exit
        }
    } else {
        WriteErrorString "Error: [AmazonEC2] subnetID$nic not specified"
        Exit
    }

    $publicIPId = $settings.AmazonEC2.("publicIPId"+$nic)

    if ($publicIPId.length -gt 0) {

        $ipObj = Get-EC2Address -AllocationId $publicIPId

        If([string]::IsNullOrEmpty($ipObj)) {
            WriteErrorString "Error: [AmazonEC2] publicIPId$nic ($publicIPId) not found"
            Exit
        }

        $publicIP = $ipObj.PublicIp

        if ($ipObj.InstanceId.length -gt 0) {
            WriteErrorString "Error: [AmazonEC2] publicIPId$nic ($publicIPId - $publicIP) is already in use by another instance"
            Exit
        }
    }

    $securityGroupId = $settings.AmazonEC2.("securityGroupId"+$nic)

    if ($securityGroupId.length -gt 0) {
        $sg = Get-EC2SecurityGroup -GroupId $securityGroupId
        If([string]::IsNullOrEmpty($sg)) {
            WriteErrorString "Error: [AmazonEC2] securityGroupId$nic ($securityGroupId) not found"
            Exit
        }

        if ($sg.VpcId -ne $subnet.VpcId) {
            WriteErrorString "Error: [AmazonEC2] securityGroupId$nic ($securityGroupId) is not in the same VPC as the specified subnet"
            Exit
        }
    }
}

#
# Load the dependent PowerShell Module
#

$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptDir  = Split-Path -Parent $ScriptPath
$uagDeployModule=$ScriptDir+"\uagdeploy.psm1"

if (!(Test-path $uagDeployModule)) {
    Write-host "Error: PowerShell Module $uagDeployModule not found." -foregroundcolor red -backgroundcolor black
    Exit
}

import-module $uagDeployModule -Force -ArgumentList $awAPIServerPwd, $awTunnelGatewayAPIServerPwd, $awCGAPIServerPwd, $awSEGAPIServerPwd

if (-not (Get-Module -ListAvailable -Name "AWSPowerShell")) {
    WriteErrorString "Error: Powershell module AWSPowerShell not found. Run the command 'Install-Module -Name AWSPowerShell' and retry"
    Exit
}

$os = [environment]::OSVersion.Platform
if ($os -notlike 'win*')
{
    if (-not(Get-Module -ListAvailable -Name "AWS.Tools.Common"))
    {
        WriteErrorString "Error: AWS powershell module  AWS.Tools.Common not found. Run the command 'Install-Module -Name AWS.Tools.Common' and retry"
        Exit
    }
    if (-not(Get-Module -ListAvailable -Name "AWS.Tools.EC2"))
    {
        WriteErrorString "Error: AWS powershell module  AWS.Tools.EC2 not found. Run the command 'Install-Module -Name AWS.Tools.EC2' and retry"
        Exit
    }
}

Write-host "Unified Access Gateway (UAG) virtual appliance Amazon AWS EC2 deployment script"

Write-host "Note: Please make sure to use the latest script that came with the build"

if (!(Test-path $iniFile)) {
    WriteErrorString "Error: Configuration file ($iniFile) not found."
    Exit
}

$settings = ImportIni $iniFile

$uagName=$settings.General.name

#
# Assign and validate network settings
#

$deploymentOption=GetDeploymentSettingOption $settings

if ($uagName.length -gt 32) {
    WriteErrorString "Error: Virtual machine name must be no more than 32 characters in length"
    Exit
}

if (!$uagName) {
    WriteErrorString "Error: [General] Name not specified"
    Exit
}

$osLoginUsername = ReadOsLoginUsername $settings
if ($osLoginUsername.length -eq 0) {
    $osLoginUsername = "root"
}

if ($settings.General.dsComplianceOS -eq "true") {
    updatePasswordPolicyForDsComplianceOS $settings
}

if (!$rootPwd) {
    $rootPwd = GetRootPwd $uagName $settings $osLoginUsername
}

if (!$adminPwd) {
    $adminPwd = GetAdminPwd $uagName $settings
}

if (!$ceipEnabled) {
    $ceipEnabled = GetCeipEnabled $uagName
}

$settingsJSON=GetJSONSettings $settings $newAdminUserPwd

SetUp

$ovfFile = "$env:HOME/$uagName.cfg"

if ($Env:OS -like 'win*') {
  $ovfFile = "${env:APPDATA}\VMware\$uagName.cfg"
}


[IO.File]::WriteAllLines($ovfFile, [string[]]("deploymentOption="+"$deploymentOption"))

$dns=$settings.General.dns
if ($dns.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("DNS="+"$dns"))
}

$defaultGateway=$settings.General.defaultGateway
if ($defaultGateway.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("defaultGateway="+"$defaultGateway"))
}

$v6DefaultGateway=$settings.General.v6DefaultGateway
if ($v6defaultGateway.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("v6defaultGateway="+"$v6defaultGateway"))
}

$forwardrules=$settings.General.forwardrules
if ($forwardrules.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("forwardrules="+"$forwardrules"))
}

$routes0=$settings.General.routes0
if ($routes0.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes0="+"$routes0"))
}

$routes1=$settings.General.routes1
if ($routes1.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes1="+"$routes1"))
}

$routes2=$settings.General.routes2
if ($routes2.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes2="+"$routes2"))
}

$policyRouteGateway0=$settings.General.policyRouteGateway0
if ($policyRouteGateway0.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway0="+"$policyRouteGateway0"))
}

$policyRouteGateway1=$settings.General.policyRouteGateway1
if ($policyRouteGateway1.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway1="+"$policyRouteGateway1"))
}

$policyRouteGateway2=$settings.General.policyRouteGateway2
if ($policyRouteGateway2.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway2="+"$policyRouteGateway2"))
}

if ($osLoginUsername.length -ne "root") {
	[IO.File]::AppendAllLines($ovfFile, [string[]]("osLoginUsername="+"$osLoginUsername"))
}

$osMaxLoginLimit = ReadOsMaxLoginLimit $settings
if ($osMaxLoginLimit.length -gt 0) {
	[IO.File]::AppendAllLines($ovfFile, [string[]]("osMaxLoginLimit="+"$osMaxLoginLimit"))
}

$rootPasswordExpirationDays=$settings.General.rootPasswordExpirationDays
if ($rootPasswordExpirationDays.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("rootPasswordExpirationDays="+"$rootPasswordExpirationDays"))
}

$passwordPolicyMinLen=$settings.General.passwordPolicyMinLen
if ($passwordPolicyMinLen.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyMinLen="+"$passwordPolicyMinLen"))
}

$passwordPolicyMinClass=$settings.General.passwordPolicyMinClass
if ($passwordPolicyMinClass.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyMinClass="+"$passwordPolicyMinClass"))
}

$passwordPolicyDifok=$settings.General.passwordPolicyDifok
if ($passwordPolicyDifok.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyDifok="+"$passwordPolicyDifok"))
}

$passwordPolicyUnlockTime=$settings.General.passwordPolicyUnlockTime
if ($passwordPolicyUnlockTime.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyUnlockTime="+"$passwordPolicyUnlockTime"))
}

$passwordPolicyFailedLockout=$settings.General.passwordPolicyFailedLockout
if ($passwordPolicyFailedLockout.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("passwordPolicyFailedLockout="+"$passwordPolicyFailedLockout"))
}

$adminPasswordFailedLockoutCount=$settings.General.adminPasswordPolicyFailedLockoutCount
if ($adminPasswordFailedLockoutCount.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPasswordPolicyFailedLockoutCount="+"$adminPasswordFailedLockoutCount"))
}

$adminPasswordMinLen=$settings.General.adminPasswordPolicyMinLen
if ($adminPasswordMinLen.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPasswordPolicyMinLen="+"$adminPasswordMinLen"))
}

$adminPasswordLockoutTime=$settings.General.adminPasswordPolicyUnlockTime
if ($adminPasswordLockoutTime.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPasswordPolicyUnlockTime="+"$adminPasswordLockoutTime"))
}

$adminSessionIdleTimeoutMinutes=$settings.General.adminSessionIdleTimeoutMinutes
if ($adminSessionIdleTimeoutMinutes.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminSessionIdleTimeoutMinutes="+"$adminSessionIdleTimeoutMinutes"))
}

$adminMaxConcurrentSessions = ValidateAdminMaxConcurrentSessions $settings
if ($adminMaxConcurrentSessions.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminMaxConcurrentSessions="+"$adminMaxConcurrentSessions"))
}

$rootSessionIdleTimeoutSeconds = ValidateRootSessionIdleTimeoutSeconds $settings
if ($rootSessionIdleTimeoutSeconds.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("rootSessionIdleTimeoutSeconds="+"$rootSessionIdleTimeoutSeconds"))
}

if ($ceipEnabled -eq $true) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("ceipEnabled=true"))
}

if ($settings.General.dsComplianceOS -eq "true") {
	[IO.File]::AppendAllLines($ovfFile, [string[]]("dsComplianceOS=true"))
}

if ($settings.General.tlsPortSharingEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("tlsPortSharingEnabled=true"))
}

if ($settings.General.sshEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshEnabled=true"))
}

if ($settings.General.sshPasswordAccessEnabled -eq "false") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshPasswordAccessEnabled=false"))
}

if ($settings.General.sshKeyAccessEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshKeyAccessEnabled=true"))
}

$sshBannerText=ReadLoginBannerText $settings
if ($sshBannerText.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshLoginBannerText=" + "$sshBannerText"))
}

$sshInterface = validateSSHInterface $settings
if (($sshInterface.length -gt 0)) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshInterface=" + "$sshInterface"))
}

$sshPort = $settings.General.sshPort
if ($sshPort.length -gt 0 -and ($sshPort -match '^[0-9]+$')) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshPort=" + "$sshPort"))
}

$secureRandomSrc=ReadSecureRandomSource $settings
if ($secureRandomSrc.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("secureRandomSource=" + "$secureRandomSrc"))
}

[IO.File]::AppendAllLines($ovfFile, [string[]]("rootPassword="+"$rootPwd"))

if ($adminPwd.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPassword="+"$adminPwd"))
}

$enabledAdvancedFeatures=$settings.General.enabledAdvancedFeatures
if ($enabledAdvancedFeatures.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("enabledAdvancedFeatures="+"$enabledAdvancedFeatures"))
}

$gatewaySpec = getGatewaySpec $settings
if ($gatewaySpec.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("gatewaySpec="+"$gatewaySpec"))
}

$commandsFirstBoot = ValidateCustomBootTimeCommands $settings "commandsFirstBoot"
if ($commandsFirstBoot.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("commandsFirstBoot="+"$commandsFirstBoot"))
}
$commandsEveryBoot = ValidateCustomBootTimeCommands $settings "commandsEveryBoot"
if ($commandsEveryBoot.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("commandsEveryBoot="+"$commandsEveryBoot"))
}

$configURL=$settings.General.configURL
if ($configURL.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configURL="+"$configURL"))
}

$configKey=$settings.General.configKey
if ($configURL.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configKey="+"$configKey"))
}

$configURLThumbprints=$settings.General.configURLThumbprints
if ($configURLThumbprints.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configURLThumbprints="+"$configURLThumbprints"))
}

$configURLHttpProxy=$settings.General.configURLHttpProxy
if ($configURLHttpProxy.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("configURLHttpProxy="+"$configURLHttpProxy"))
}

$adminCsrSubject=$settings.General.adminCsrSubject
if ($adminCsrSubject.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminCsrSubject="+"$adminCsrSubject"))
}

$adminCsrSAN=$settings.General.adminCsrSAN
if ($adminCsrSAN.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminCsrSAN="+"$adminCsrSAN"))
}

$additionalDeploymentMetadata = $settings.General.additionalDeploymentMetadata
if($additionalDeploymentMetadata.length -gt 0){
    [IO.File]::AppendAllLines($ovfFile, [string[]]("additionalDeploymentMetadata="+"$additionalDeploymentMetadata"))
}

$awsamiId = $settings.AmazonEC2.amiId

if ($awsamiId.length -eq 0) {
   WriteErrorString "Error: [AmazonEC2] amiId not found"
   Exit
}

$ErrorActionPreference = "SilentlyContinue"

$credentialProfileName = $settings.AmazonEC2.credentialProfileName

if ($credentialProfileName.length -gt 0) {
    $cred = Get-AWSCredential -ProfileName $credentialProfileName
    If([string]::IsNullOrEmpty($cred)) {
        WriteErrorString "Error: [AmazonEC2] credentialProfileName ($credentialProfileName) not found. To set a named credential profile, run the command:"
        WriteErrorString "Set-AWSCredential -AccessKey <accesskey> -SecretKey <secretkey> -StoreAs $credentialProfileName"
        Exit
    }
    $subNets = get-ec2subnet -ProfileName $credentialProfileName
    If([string]::IsNullOrEmpty($subNets)) {
        WriteErrorString "Error: [AmazonEC2] credentialProfileName ($credentialProfileName) is invalid. To set a named credential profile, run the command:"
        WriteErrorString "Set-AWSCredential -AccessKey <accesskey> -SecretKey <secretkey> -StoreAs $credentialProfileName"
        Exit
    }
} else {
    $credentialProfileName = "default"
    $subNets = get-ec2subnet -ProfileName $credentialProfileName
    If([string]::IsNullOrEmpty($subNets)) {
        WriteErrorString "Error: Default credential profile is not set or is invalid. To set a default credential profile, run the command:"
        WriteErrorString "Set-AWSCredential -AccessKey <accesskey> -SecretKey <secretkey> -StoreAs default"
        Exit
    }
}

$awsInstanceType = $settings.AmazonEC2.instanceType
if ($awsInstanceType.length -gt 0) {
    write-host "Deploying $uagName as $awsInstanceType instance type"
} else {
    $awsInstanceType = "c4.large"
}

$awsRegion = $settings.AmazonEC2.region
if ($awsRegion.length -gt 0) {
    $region = Get-EC2Region -Region $awsRegion
    If([string]::IsNullOrEmpty($region)) {
        WriteErrorString "Error: [AmazonEC2] region ($awsRegion) not found"
        $regions = Get-EC2Region
        $regionNames = $regions.RegionName
        WriteErrorString "Specify a region from the following list - $regionNames"
        Exit
    }
    Set-DefaultAWSRegion $awsRegion
} else {
    WriteErrorString "Error: [AmazonEC2] region not specified"
    Exit
}

Initialize-AWSDefaultConfiguration -ProfileName $credentialProfileName -Region $awsRegion

terminateExistingInstance $settings

$ami = Get-EC2Image -ImageId $awsamiId -ErrorAction SilentlyContinue

If([string]::IsNullOrEmpty($ami)) {
    WriteErrorString "Error: AMI image $awsamiId not found in region $awsRegion"
    Exit
}

switch -Wildcard ($deploymentOption) {

    'onenic*' {
        ValidateNetworkSettings $settings "0"
        $eth0 = CreateNIC $settings "0" "Internet, Management and Backend"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $NetworkInterfaces = $eth0
        $customConfigEntry0 = GetCustomConfigEntry $settings "0"
        if ($customConfigEntry0.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry0))
        }
    }
    'twonic*' {
        ValidateNetworkSettings $settings "0"
        ValidateNetworkSettings $settings "1"
        $eth0 = CreateNIC $settings "0" "Internet"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $eth1 = CreateNIC $settings "1" "Management and Backend"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode1=DHCPV4+DHCPV6"))
        $NetworkInterfaces = $eth0,$eth1
        $customConfigEntry0 = GetCustomConfigEntry $settings "0"
        if ($customConfigEntry0.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry0))
        }
        $customConfigEntry1 = GetCustomConfigEntry $settings "1"
        if ($customConfigEntry1.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry1))
        }
    }
    'threenic*' {
        ValidateNetworkSettings $settings "0"
        ValidateNetworkSettings $settings "1"
        ValidateNetworkSettings $settings "2"
        $eth0 = CreateNIC $settings "0" "Internet"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $eth1 = CreateNIC $settings "1" "Management"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode1=DHCPV4+DHCPV6"))
        $eth2 = CreateNIC $settings "2" "Backend"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode2=DHCPV4+DHCPV6"))
        $NetworkInterfaces = $eth0,$eth1,$eth2
        $customConfigEntry0 = GetCustomConfigEntry $settings "0"
        if ($customConfigEntry0.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry0))
        }
        $customConfigEntry1 = GetCustomConfigEntry $settings "1"
        if ($customConfigEntry1.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry1))
        }
        $customConfigEntry2 = GetCustomConfigEntry $settings "2"
        if ($customConfigEntry2.length -gt 0) {
            [IO.File]::AppendAllLines($ovfFile, [string[]]($customConfigEntry2))
        }
    }
    default {
        WriteErrorString "Error: Invalid deploymentOption ($deploymentOption)."
        Exit
    }
}

[IO.File]::AppendAllLines($ovfFile, [string[]]("settingsJSON="+"$settingsJSON"))

$ovfProperties = Get-Content -Raw $ovfFile

$UserData = [System.Convert]::ToBase64String([System.Text.Encoding]::ASCII.GetBytes($ovfProperties))

$InstanceParams = @{
    ImageId = $awsamiId
    NetworkInterface = $NetworkInterfaces
    MinCount = "1"
    MaxCount = "1"
    InstanceType = $awsInstanceType
    UserData = $UserData
}

$i = New-EC2Instance @InstanceParams

If([string]::IsNullOrEmpty($i)) {
    $msg = $error[0]
    WriteErrorString "Error: Failed to deploy UAG - $msg"
    [IO.File]::Delete($ovfFile)
    Exit
}

$instance = $i.Instances[0]
$instanceId = $instance.InstanceId

New-EC2Tag -Region $awsRegion -Tag @( @{ Key = "Name" ; Value = "$uagName"})-Resource $instanceId -Force

SetNICDeleteOnTermination $settings $eth0
SetNICDeleteOnTermination $settings $eth1
SetNICDeleteOnTermination $settings $eth2

write-host "$uagName Deployed successfully to AWS EC2 region $awsRegion - InstanceId=$instanceId"

[IO.File]::Delete($ovfFile)
