# Copyright (c) 2024 Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Public

<#
    .SYNOPSIS
     Sample Powershell script to deploy a UAG virtual appliance to Nutanix.
    .EXAMPLE
     .\uagdeployntnx.ps1 -iniFile uag15-ntnx.ini
#>

param([string]$iniFile = "uag.ini", [string] $rootPwd, [string] $adminPwd, [string] $ceipEnabled, [string] $prismCentralPassword,
     [string] $awAPIServerPwd, [string] $awTunnelGatewayAPIServerPwd, [string] $awCGAPIServerPwd, [string] $awSEGAPIServerPwd,[string] $newAdminUserPwd)

$nic_map = @{ onenic = 1 ; twonic = 2; threenic = 3}

# UAG size mapping
$size_map = @{
    "Standard" = @{
        "vcpu" = 2
        "ram" = 4096
    }
    "Large" = @{
        "vcpu" = 4
        "ram" = 16384
    }
    "XL" = @{
        "vcpu" = 8
        "ram" = 32768
    }
}

# Function for validating the mandatory configurations are present or not
function validateNutanixBasicSettings {
    param($settings)

    if([string]::IsNullOrEmpty($settings.Nutanix.prismCentralUserName)) {
        WriteErrorString "Error: Prism central username can't be null."
        CleanExit
    }

    if([string]::IsNullOrEmpty($settings.Nutanix.prismCentralIP)) {
        WriteErrorString "Error: Prism Central IP can't be null."
        CleanExit
    }

}

# Function for validating project details in which UAG is going to be deployed.
function validateNutanixProjectDetails {
    param($settings)

    if([string]::IsNullOrEmpty($settings.Nutanix.project)) {
        return "";
    }
    $projectUUID = ""
    $requestBody = @{
        kind = "project"
    }
    $requestBodyJson = ,$requestBody | ConvertTo-Json
    $listOfProjectsUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/api/nutanix/v3/projects/list"
    $listOfProjects = invokeRequest -method "POST" -url $listOfProjectsUrl -base64AuthInfo $base64AuthInfo -requestBody $requestBodyJson | ConvertFrom-Json
    if ($listOfProjects.entities) {
        $projectList = $listOfProjects.entities
        $projectName = $settings.Nutanix.project
        $matchedProject = $projectList | Where-Object { $_.metadata.project_reference.name -eq $projectName }
        if ($matchedProject) {
            $projectUUID = $matchedProject.metadata.uuid
            Write-Host "Project $projectName found with UUID : $projectUUID"
        } else {
            WriteErrorString "Error : Project $projectName' not found. Please update the project name or leave it empty and try again."
            CleanExit
        }
    } else {
        WriteErrorString "Error : Project list is empty. Please create the project in Prism Central before doing the deployment or leave project field empty."
        CleanExit
    }
    return $projectUUID;
}

# Function for validating cluster details in which UAG is going to be deployed.
function validateNutanixClusterDetails {
    param($settings)

    if([string]::IsNullOrEmpty($settings.Nutanix.cluster)) {
        WriteErrorString "Error: Cluster name can't be null."
        CleanExit
    }

    $clusterUUID = ""
    $requestBody = @{
        kind = "cluster"
    }
    $requestBodyJson = ,$requestBody | ConvertTo-Json
    $listOfClustersUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/api/nutanix/v3/clusters/list"
    $listOfClusters = invokeRequest -method "POST" -url $listOfClustersUrl -base64AuthInfo $base64AuthInfo -requestBody $requestBodyJson | ConvertFrom-Json
    if ($listOfClusters.entities) {
        $clusterList = $listOfClusters.entities
        $clusterName = $settings.Nutanix.cluster
        $matchedCluster = $clusterList | Where-Object { $_.spec.name -eq $clusterName }
        if ($matchedCluster) {
            $clusterUUID = $matchedCluster.metadata.uuid
            Write-Host "Cluster $clusterName found with UUID : $clusterUUID"
        } else {
            WriteErrorString "Error : Cluster $clusterName not found. Please update the cluster name and try again."
            CleanExit
        }
    } else {
        WriteErrorString "Error : Cluster list is empty. Please register Prism Element in Prism Central before doing the deployment. "
        CleanExit
    }
    return $clusterUUID
}

# Function for validating vmdk image details which is going to be used for creating UAG VM.
function validateNutanixImageDetails {
    param($settings)

    if([string]::IsNullOrEmpty($settings.Nutanix.image)) {
        WriteErrorString "Error: UAG vmdk image name can't be null."
        CleanExit
    }

    $imageUUID = ""
    $requestBody = @{
        kind = "image"
    }
    $requestBodyJson = ,$requestBody | ConvertTo-Json
    $listOfImagesUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/api/nutanix/v3/images/list"
    $listOfImages = invokeRequest -method "POST" -url $listOfImagesUrl -base64AuthInfo $base64AuthInfo -requestBody $requestBodyJson | ConvertFrom-Json
    if ($listOfImages.entities) {
        $imageList = $listOfImages.entities
        $imageName = $settings.Nutanix.image
        $matchedImage = $imageList | Where-Object { $_.spec.name -eq $imageName }
        if ($matchedImage) {
            $imageUUID = $matchedImage.metadata.uuid
            Write-Host "Image $imageName found with UUID : $imageUUID"
        } else {
            WriteErrorString "Error : Image $imageName not found. Please update UAG vmdk image name and try again."
            CleanExit
        }
    } else {
        WriteErrorString "Error : Image list is empty. Please upload UAG vmdk image to Nutanix before doing the deployment."
        CleanExit
    }
    return $imageUUID
}

# Function for validating network and subnet details in which UAG is going to be deployed.
function validateNutanixSubnetDetails {
    param($settings)

    $requestBody = @{
        kind = "subnet"
    }
    $requestBodyJson = ,$requestBody | ConvertTo-Json
    $listOfSubnetsUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/api/nutanix/v3/subnets/list"
    $listOfSubnets = invokeRequest -method "POST" -url $listOfSubnetsUrl -base64AuthInfo $base64AuthInfo -requestBody $requestBodyJson | ConvertFrom-Json

    $no_of_nics = getNicsCount($deploymentOption)

    $arrayList = New-Object System.Collections.ArrayList

    if ($listOfSubnets.entities) {
        $subnetList = $listOfSubnets.entities
        for ($i = 0; $i -lt $no_of_nics; $i++) {
            $subnetRef = "subnet" + $i
            if([string]::IsNullOrEmpty($settings.Nutanix.$subnetRef)) {
                WriteErrorString "Error: $subnetRef can't be null."
                CleanExit
            }
            $matchedSubnet = $subnetList | Where-Object { $_.spec.name -eq $settings.Nutanix.$subnetRef -and $_.status.cluster_reference.name -eq $settings.Nutanix.cluster }
            if ($matchedSubnet) {
                $subnetUUID = $matchedSubnet.metadata.uuid
                $arrayList.Add($subnetUUID) | Out-Null
                Write-Host "Subnet $subnetRef found with UUID : $subnetUUID"
            } else {
                WriteErrorString "Error : Subnet $subnetRef not found. Please update the $subnetRef name and try again."
                CleanExit
            }
        }
    } else {
        WriteErrorString "Error : Subnet list is empty. Please create no of subnets corresponding to deploymentOption before doing the deployment."
        CleanExit
    }
    return ,$arrayList.ToArray([string]);
}

# Function for validating Storage Container details in which UAG is going to be created.
function validateNutanixStorageContainerDetails {
    param($settings)

    $containerUuid = ""

    $listOfContainersUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/PrismGateway/services/rest/v1/containers"
    $listOfContainers = invokeRequest -method "GET" -url $listOfContainersUrl -base64AuthInfo $base64AuthInfo | ConvertFrom-Json
    if ($listOfContainers.entities) {
        $containerList = $listOfContainers.entities
        $containerName = $settings.Nutanix.storageContainer
        $matchedContainer = $containerList | Where-Object { $_.name -eq $containerName -and $_.clusterUuid -eq $clusterUUID }
        if ($matchedContainer) {
            $containerUuid = $matchedContainer.containerUuid
            Write-Host "Storage Container $containerName found with UUID : $containerUuid"
        } else {
            WriteErrorString "Error : Storage Container $containerName not found. Please update the storage container name and try again."
            CleanExit
        }
    } else {
        WriteErrorString "Error : Storage Container list is empty. Please create storage container before doing the deployment."
        CleanExit
    }
    return $containerUuid;
}

# Function for validating and updating the network information in userdata
function validateNicSettings() {
    param($settings)

    $no_of_nics = getNicsCount($deploymentOption)
    $userDataPayload = ""
    for ($i = 1; $i -le $no_of_nics; $i++) {

        $ipModeLabel = "ipMode" + ($i - 1)
        $ipMode = $settings.General.$ipModeLabel

        $ipLabel = "ip" + ($i - 1)
        $ip=$settings.General.$ipLabel

        $netmaskLabel = "netmask" + ($i - 1)
        $netmask=$settings.General.$netmaskLabel

        if (($ip.length -gt 0) -and ($netmask.length -eq 0)) {
            WriteErrorString "Error: missing value General $netmaskLabel."
            Exit
        }

        if ($ipMode.length -eq 0) {
            $ipMode = "DHCPV4"
            if (($ip.length -gt 0)) {
                $ipMode = "STATICV4"
            }
        }
        # Add the below config to user-data (ip0, ipMode0, netmask0)
        if($ipMode.Contains("STATIC")) {
            $userDataPayload += [string]"$ipModeLabel="+"$ipMode"+"`n"
            $userDataPayload += [string]"$ipLabel="+"$ip"+"`n"
            $userDataPayload += [string]"$netmaskLabel="+"$netmask"+"`n"
        } else {
            $userDataPayload += [string]"$ipModeLabel="+"$ipMode"+"`n"
        }
    }
    return $userDataPayload;
}

# Function for creating the network interface payload
# In Nutanix NICs are automatically created and destroyed when a VM is created or destroyed.
# Size of nic_list is equal to the no of nics that are going to be created.
# Once VM is created, in UAG we will do the network settings again according to the data present in INI file.
function createNicPayload() {
    param($settings, $subnetUuidList)

    $nicList = [System.Collections.Generic.List[PSObject]]::new()
    $no_of_nics = getNicsCount($deploymentOption)
    for ($i = 0; $i -lt $no_of_nics; $i++) {
        $nicList.Add([PSCustomObject]@{
            is_connected = $true
            subnet_reference = [PSCustomObject]@{
                kind = "subnet"
                uuid = $subnetUuidList[$i]
            }
        })
    }

    return ,$nicList
}

# This is a general method for all kinds of Prism Central API invocation.
function invokeRequest() {
    param (
        [Parameter(Mandatory = $true)]
        [String]$method,

        [Parameter(Mandatory = $true)]
        [String]$url,

        [Parameter(Mandatory = $true)]
        $base64AuthInfo,

        [Parameter(Mandatory = $false)]
        $requestBody
    )

    $request = [System.Net.WebRequest]::Create($url)
    $request.Method = $method
    $request.ContentType = "application/json"
    $request.Headers["Authorization"] = "Basic $base64AuthInfo"

    if($requestBody -ne $null) {
        $byteArray = [System.Text.Encoding]::UTF8.GetBytes($requestBody)
        $request.ContentLength = $byteArray.Length
        $requestStream = $request.GetRequestStream()
        $requestStream.Write($byteArray, 0, $byteArray.Length)
        $requestStream.Close()
    }

    try {
        $response = $request.GetResponse()
        $responseStream = $response.GetResponseStream()
        $reader = New-Object System.IO.StreamReader($responseStream)
        $responseBody = $reader.ReadToEnd()
        $reader.Close()
        $responseStream.Close()
        return $responseBody
    } catch [System.Net.WebException] {
        $errorResponse = $_.Exception.Response
        if ($errorResponse -ne $null) {
            $responseStream = $errorResponse.GetResponseStream()
            $reader = New-Object System.IO.StreamReader($responseStream)
            $errorMessage = $reader.ReadToEnd()
            $reader.Close()
            $responseStream.Close()
            try {
                $errorDetails = $errorMessage | ConvertFrom-Json
                $error = $errorDetails.message_list[0].message
                WriteErrorString "Error Details: $error"
            } catch {
                WriteErrorString "Error Message: $errorMessage"
            }
        } else {
            WriteErrorString "An unknown error occurred."
        }
        CleanExit
    }
}

# Function to get the UAG size
function getUagSize() {
    param($key)
    $deployment_details  = $key -split '-'
    if ($deployment_details.Count -gt 1) {
        return $deployment_details[1]
    } else {
        return "Standard"
    }
}

# Function to get No of NICs
function getNicsCount() {
    param($deploymentOption)

    $deployment_details  = $deploymentOption -split '-'
    $no_of_nics = 0
    foreach ($entry in $nic_map.GetEnumerator()) {
        if ($entry.Key.ToLower() -eq $deployment_details[0]) {
            $no_of_nics = $entry.Value
        }
    }
    return $no_of_nics
}

# ------------- Execution starts here -------------------

$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptDir  = Split-Path -Parent $ScriptPath
$apDeployModule=$ScriptDir+"\uagdeploy.psm1"

if (!(Test-path $apDeployModule)) {
    Write-host "Error: PowerShell Module $apDeployModule not found." -foregroundcolor red -backgroundcolor black
    Exit
}

import-module $apDeployModule -Force -ArgumentList $awAPIServerPwd, $awTunnelGatewayAPIServerPwd, $awCGAPIServerPwd, $awSEGAPIServerPwd

Write-host "Unified Access Gateway (UAG) virtual appliance deployment script"

Write-host "Note: Please make sure to use the latest script that came with the build"

if (!(Test-path $iniFile)) {
    WriteErrorString "Error: Configuration file ($iniFile) not found."
    Exit
}

# ----------------- Import Ini ---------------------------------

$settings = ImportIni $iniFile
$settingsJSON = GetJSONSettings $settings $newAdminUserPwd

# -------------- Prism Central User Credentials --------------------
$prismCentralUserName=$settings.Nutanix.prismCentralUserName
if (!$prismCentralPassword) {
    $secure_password = Read-Host "Enter Prism Central Password for user '$prismCentralUserName'" -AsSecureString
    $binary_string = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($secure_password)
    $plain_text_password = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($binary_string)
    $prismCentralPassword = $plain_text_password
}

# HTTP Basic Authorization header
$base64AuthInfo = [Convert]::ToBase64String([Text.Encoding]::ASCII.GetBytes(("${prismCentralUserName}:${prismCentralPassword}")))

# ----------------- input validation start ----------------------

$settings = ImportIni $iniFile
$uagName = $settings.General.name
$deploymentOption=GetDeploymentSettingOption $settings

# ----------------- Validate Nutanix Project, Cluster, Image, Network Details --------------

# Get Prism Central Port
# If not specified in INI file, then fallback to default port 9440.
$prismCentralPort = 9440
if(-Not ([string]::IsNullOrEmpty($settings.Nutanix.prismCentralPort))) {
    $prismCentralPort = [Int]$settings.Nutanix.prismCentralPort
}

validateNutanixBasicSettings $settings
$clusterUUID = validateNutanixClusterDetails $settings
$projectUUID = validateNutanixProjectDetails $settings
$imageUUID = validateNutanixImageDetails $settings
$containerUuid = validateNutanixStorageContainerDetails $settings
$subnetUuidList = validateNutanixSubnetDetails $settings

if ($uagName.length -gt 32) {
    WriteErrorString "Error: Virtual machine name must be no more than 32 characters in length"
    Exit
}

if (!$uagName) {
    WriteErrorString "Error: [General] name not specified"
    Exit
}

$osLoginUsername = ReadOsLoginUsername $settings
if ($osLoginUsername.length -eq 0) {
    $osLoginUsername = "root"
}

if ($settings.General.dsComplianceOS -eq "true") {
    updatePasswordPolicyForDsComplianceOS $settings
}

if (!$rootPwd) {
    $rootPwd = GetRootPwd $uagName $settings $osLoginUsername
}

if (!$adminPwd) {
    $adminPwd = GetAdminPwd $uagName $settings
}

if (!$ceipEnabled) {
    $ceipEnabled = GetCeipEnabled $uagName
}

# ------------ input validation end ----------------------

# ------------ user-data creation start ------------------

$userDataPayload = ""

$dns=$settings.General.DNS
if ($dns.length -gt 0) {
    $userDataPayload += [string]"DNS="+"$dns"+"`n"
}

if ($osLoginUsername -ne "root") {
    $userDataPayload += [string]"osLoginUsername="+"$osLoginUsername"+"`n"
}

$osMaxLoginLimit = ReadOsMaxLoginLimit $settings
if ($osMaxLoginLimit.length -gt 0) {
    $userDataPayload += [string]"osMaxLoginLimit="+"$osMaxLoginLimit"+"`n"
}

$rootPasswordExpirationDays=$settings.General.rootPasswordExpirationDays
if ($rootPasswordExpirationDays.length -gt 0) {
    $userDataPayload += [string]"rootPasswordExpirationDays="+"$rootPasswordExpirationDays"+"`n"
}

$passwordPolicyMinLen=$settings.General.passwordPolicyMinLen
if ($passwordPolicyMinLen.length -gt 0) {
    $userDataPayload += [string]"passwordPolicyMinLen="+"$passwordPolicyMinLen"+"`n"
}

$passwordPolicyMinClass=$settings.General.passwordPolicyMinClass
if ($passwordPolicyMinClass.length -gt 0) {
    $userDataPayload += [string]"passwordPolicyMinClass="+"$passwordPolicyMinClass"+"`n"
}

$passwordPolicyDifok=$settings.General.passwordPolicyDifok
if ($passwordPolicyDifok.length -gt 0) {
    $userDataPayload += [string]"passwordPolicyDifok="+"$passwordPolicyDifok"+"`n"
}

$passwordPolicyUnlockTime=$settings.General.passwordPolicyUnlockTime
if ($passwordPolicyUnlockTime.length -gt 0) {
    $userDataPayload += [string]"passwordPolicyUnlockTime="+"$passwordPolicyUnlockTime"+"`n"
}

$passwordPolicyFailedLockout=$settings.General.passwordPolicyFailedLockout
if ($passwordPolicyFailedLockout.length -gt 0) {
    $userDataPayload += [string]"passwordPolicyFailedLockout="+"$passwordPolicyFailedLockout"+"`n"
}

$adminPasswordFailedLockoutCount=$settings.General.adminPasswordPolicyFailedLockoutCount
if ($adminPasswordFailedLockoutCount.length -gt 0){
    $userDataPayload += [string]"adminPasswordPolicyFailedLockoutCount="+"$adminPasswordFailedLockoutCount"+"`n"
}

$adminPasswordMinLen=$settings.General.adminPasswordPolicyMinLen
if ($adminPasswordMinLen.length -gt 0){
    $userDataPayload += [string]"adminPasswordPolicyMinLen="+"$adminPasswordMinLen"+"`n"
}

$adminPasswordLockoutTime=$settings.General.adminPasswordPolicyUnlockTime
if ($adminPasswordLockoutTime.length -gt 0){
    $userDataPayload += [string]"adminPasswordPolicyUnlockTime="+"$adminPasswordLockoutTime"+"`n"
}

$adminSessionIdleTimeoutMinutes=$settings.General.adminSessionIdleTimeoutMinutes
if ($adminSessionIdleTimeoutMinutes.length -gt 0) {
    $userDataPayload += [string]"adminSessionIdleTimeoutMinutes="+"$adminSessionIdleTimeoutMinutes"+"`n"
}

$adminMaxConcurrentSessions = ValidateAdminMaxConcurrentSessions $settings
if ($adminMaxConcurrentSessions.length -gt 0) {
    $userDataPayload += [string]"adminMaxConcurrentSessions="+"$adminMaxConcurrentSessions"+"`n"
}

$rootSessionIdleTimeoutSeconds = ValidateRootSessionIdleTimeoutSeconds $settings
if ($rootSessionIdleTimeoutSeconds.length -gt 0) {
    $userDataPayload += [string]"rootSessionIdleTimeoutSeconds="+"$rootSessionIdleTimeoutSeconds"+"`n"
}

$commandsFirstBoot = ValidateCustomBootTimeCommands $settings "commandsFirstBoot"
if ($commandsFirstBoot.length -gt 0) {
    $userDataPayload += [string]"commandsFirstBoot="+"$commandsFirstBoot"+"`n"
}
$commandsEveryBoot = ValidateCustomBootTimeCommands $settings "commandsEveryBoot"
if ($commandsEveryBoot.length -gt 0) {
    $userDataPayload += [string]"commandsEveryBoot="+"$commandsEveryBoot"+"`n"
}

$defaultGateway=$settings.General.defaultGateway
if ($defaultGateway.length -gt 0) {
    $userDataPayload += [string]"defaultGateway="+"$defaultGateway"+"`n"
}

$v6DefaultGateway=$settings.General.v6DefaultGateway
if ($v6defaultGateway.length -gt 0) {
    $userDataPayload += [string]"v6defaultGateway="+"$v6DefaultGateway"+"`n"
}

$forwardrules=$settings.General.forwardrules
if ($forwardrules.length -gt 0) {
    $userDataPayload += [string]"forwardrules="+"$forwardrules"+"`n"
}

$routes0=$settings.General.routes0
if ($routes0.length -gt 0) {
    $userDataPayload += [string]"routes0="+"$routes0"+"`n"
}

$routes1=$settings.General.routes1
if ($routes1.length -gt 0) {
    $userDataPayload += [string]"routes1="+"$routes1"+"`n"
}

$routes2=$settings.General.routes2
if ($routes2.length -gt 0) {
    $userDataPayload += [string]"routes2="+"$routes2"+"`n"
}

$policyRouteGateway0=$settings.General.policyRouteGateway0
if ($policyRouteGateway0.length -gt 0) {
    $userDataPayload += [string]"policyRouteGateway0="+"$policyRouteGateway0"+"`n"
}

$policyRouteGateway1=$settings.General.policyRouteGateway1
if ($policyRouteGateway1.length -gt 0) {
    $userDataPayload += [string]"policyRouteGateway1="+"$policyRouteGateway1"+"`n"
}

$policyRouteGateway2=$settings.General.policyRouteGateway2
if ($policyRouteGateway2.length -gt 0) {
    $userDataPayload += [string]"policyRouteGateway2="+"$policyRouteGateway2"+"`n"
}

if ($ceipEnabled -eq $true) {
    $userDataPayload += [string]"ceipEnabled=true"+"`n"
}

if ($settings.General.dsComplianceOS -eq "true") {
    $userDataPayload += [string]"dsComplianceOS=true"+"`n"
}

if ($settings.General.tlsPortSharingEnabled -eq "true") {
    $userDataPayload += [string]"tlsPortSharingEnabled=true"+"`n"
}

if ($settings.General.sshEnabled -eq "true") {
    $userDataPayload += [string]"sshEnabled=true"+"`n"
}

if ($settings.General.sshPasswordAccessEnabled -eq "false") {
    $userDataPayload += [string]"sshPasswordAccessEnabled=false"+"`n"
}

if ($settings.General.sshKeyAccessEnabled -eq "true") {
    $userDataPayload += [string]"sshKeyAccessEnabled=true"+"`n"
}

$sshBannerText=ReadLoginBannerText $settings
if ($sshBannerText.length -gt 0) {
    $userDataPayload += [string]"sshLoginBannerText="+"$sshBannerText"+"`n"
}

$sshInterface = validateSSHInterface $settings
if (($sshInterface.length -gt 0)) {
    $userDataPayload += [string]"sshInterface="+"$sshInterface"+"`n"
}

$sshPort = $settings.General.sshPort
if ($sshPort.length -gt 0 -and ($sshPort -match '^[0-9]+$')) {
    $userDataPayload += [string]"sshPort="+"$sshPort"+"`n"
}

$secureRandomSrc=ReadSecureRandomSource $settings
if ($secureRandomSrc.length -gt 0) {
    $userDataPayload += [string]"secureRandomSource="+"$secureRandomSrc"+"`n"
}

$userDataPayload += [string]"rootPassword="+"$rootPwd"+"`n"

if ($adminPwd.length -gt 0) {
    $userDataPayload += [string]"adminPassword="+"$adminPwd"+"`n"
}

$enabledAdvancedFeatures = $settings.General.enabledAdvancedFeatures
if($enabledAdvancedFeatures.length -gt 0){
    $userDataPayload += [string]"enabledAdvancedFeatures="+"$enabledAdvancedFeatures"+"`n"
}

$gatewaySpec = getGatewaySpec $settings
if ($gatewaySpec.length -gt 0) {
    $userDataPayload += [string]"gatewaySpec="+"$gatewaySpec"+"`n"
}

# Adding ConfigData related propertiesFile
$configURL = $settings.General.configURL
if($configURL.length -gt 0){
    $userDataPayload += [string]"configURL="+"$configURL"+"`n"
}

$configKey = $settings.General.configKey
if($configKey.length -gt 0){
    $userDataPayload += [string]"configKey="+"$configKey"+"`n"
}

$configURLThumbprints = $settings.General.configURLThumbprints
if($configURLThumbprints.length -gt 0){
    $userDataPayload += [string]"configURLThumbprints="+"$configURLThumbprints"+"`n"
}

$configURLHttpProxy = $settings.General.configURLHttpProxy
if($configURLHttpProxy.length -gt 0){
    $userDataPayload += [string]"configURLHttpProxy="+"$configURLHttpProxy"+"`n"
}

$adminCsrSubject = $settings.General.adminCsrSubject
if($adminCsrSubject.length -gt 0){
    $userDataPayload += [string]"adminCsrSubject="+"$adminCsrSubject"+"`n"
}

$adminCsrSAN = $settings.General.adminCsrSAN
if($adminCsrSAN.length -gt 0){
    $userDataPayload += [string]"adminCsrSAN="+"$adminCsrSAN"+"`n"
}

$additionalDeploymentMetadata = $settings.General.additionalDeploymentMetadata
if($additionalDeploymentMetadata.length -gt 0){
    $userDataPayload += [string]"additionalDeploymentMetadata="+"$additionalDeploymentMetadata"+"`n"
}

$networkSettingsPayload = validateNicSettings $Settings

$userDataPayload += [string]$networkSettingsPayload

$userDataPayload += [string]"settingsJSON="+"$settingsJSON"

# Encode the final json to Base64 in order to pass in user_data
$base64EncodedUserData = stringToBase64($userDataPayload)

# ----------- user-data creation end ------------------------

# ----------- API calls start ------------------------

$requestBody = @{
    kind = "vm"
}
$requestBodyJson = ,$requestBody | ConvertTo-Json

# Step - 1 : Check if VM with same name exist or not. If yes delete the existing VM.
$getListOfVmUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/api/nutanix/v3/vms/list"
$getListOfVmResponse = invokeRequest -method "POST" -url $getListOfVmUrl -base64AuthInfo $base64AuthInfo -requestBody $requestBodyJson | ConvertFrom-Json
$vmUuid=""
if ($getListOfVmResponse.entities) {
    $vmList = $getListOfVmResponse.entities
    $vmNameToFind = $settings.General.name
    $matchedVm = $vmList | Where-Object { $_.spec.name -eq $vmNameToFind -and $_.spec.cluster_reference.name -eq $settings.Nutanix.cluster }
    if ($matchedVm) {
        $vmUuid = $matchedVm.metadata.uuid
        Write-Host "VM '$vmNameToFind' found with UUID : '$vmUuid' already exists. Hence deleting it and creating a new one."
    } else {
        Write-Host "VM with name '$vmNameToFind' not found. Hence creating a new one without deletion."
    }
}

# if VM exists, Delete the VM
if (-not ([string]::IsNullOrEmpty($vmUuid))) {
    $deleteVmUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/api/nutanix/v3/vms/" + $vmUuid
    $deleteVMResponse = invokeRequest -method "DELETE" -url $deleteVmUrl -base64AuthInfo $base64AuthInfo | ConvertFrom-Json
    Write-Host "VM found with name '$vmNameToFind' deleted."
    Write-Host "Waiting for 15s in order to complete the deletion."
    Start-Sleep 15
}

# Prepare the payload for create VM request

# Customer will give only UAG size. Internally we will map the ram and cpu.
$uag_size = getUagSize $deploymentOption
$memory = $size_map[$uag_size]["ram"]
$socket = $size_map[$uag_size]["vcpu"]

$nicList = createNicPayload $settings $subnetUuidList

$vmObject = [PSCustomObject]@{
    spec = [PSCustomObject]@{
        name = $settings.General.name
        resources = [PSCustomObject]@{
            num_sockets = $socket
            memory_size_mib = $memory
            hardware_clock_timezone = "UTC"
            disk_list = @([PSCustomObject]@{
                device_properties = [PSCustomObject]@{
                    device_type = "DISK"
                    disk_address = [PSCustomObject]@{
                        adapter_type = "SCSI"
                        device_index = 0
                    }
                }
                disk_size_bytes = 21474836480
                storage_config = [PSCustomObject]@{
                    storage_container_reference = [PSCustomObject]@{
                        kind = "storage_container"
                        uuid = $containerUuid
                    }
                }
                data_source_reference = [PSCustomObject]@{
                    kind = "image"
                    uuid = $imageUUID
                }
            })
            boot_config = [PSCustomObject]@{
                boot_type = "LEGACY"
                boot_device_order_list = @("CDROM", "DISK", "NETWORK")
            }
            nic_list = $nicList
            guest_customization = [PSCustomObject]@{
                cloud_init = [PSCustomObject]@{
                    user_data = $base64EncodedUserData
                }
            }
            power_state = $settings.Nutanix.vmState
        }
        cluster_reference = [PSCustomObject]@{
            uuid = $clusterUUID
            kind = "cluster"
        }
    }
    metadata = [PSCustomObject]@{
        kind = "vm"
        project_reference = [PSCustomObject]@{
            uuid = "$projectUUID"
            kind = "project"
        }
    }
}

if ([string]::IsNullOrEmpty($projectUUID)) {
    $metadata = $vmObject.metadata
    $metadata.PSObject.Properties.Remove( 'project_reference' )
    Write-Host "Default Nutanix Project selected for deployment."
}

$vmObjectJson = $vmObject | ConvertTo-Json -depth 10
#Write-Host "VM Creation payload : " + $vmObjectJson
$vmCreationUrl = "https://" + $settings.Nutanix.prismCentralIP + ":" + $prismCentralPort + "/api/nutanix/v3/vms"
$vmCreationResponse = invokeRequest -method "POST" -url $vmCreationUrl -base64AuthInfo $base64AuthInfo -requestBody $vmObjectJson | ConvertFrom-Json
Write-Host "VM with name '$vmNameToFind' created."