#
# Copyright (c) 2023-2024 Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Public
#

param([string]$iniFile = "uag.ini", [string] $rootPwd, [string] $adminPwd, [switch] $disableVerification, [switch] $noSSLVerify, [string] $ceipEnabled, [string] $ocPassword,
       [string] $awAPIServerPwd, [string] $awTunnelGatewayAPIServerPwd, [string] $awCGAPIServerPwd, [string] $awSEGAPIServerPwd,
       [string] $newAdminUserPwd)

$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptDir  = Split-Path -Parent $ScriptPath
$apDeployModule=$ScriptDir+"\uagdeploy.psm1"
#Load app module
if (!(Test-path $apDeployModule)) {
	Write-Error "Error: PowerShell Module $apDeployModule not found." -foregroundcolor red -backgroundcolor black
	Exit
}
import-module $apDeployModule -Force -ArgumentList $awAPIServerPwd, $awTunnelGatewayAPIServerPwd, $awCGAPIServerPwd, $awSEGAPIServerPwd

Write-host "Unified Access Gateway (UAG) virtual appliance deployment script"

Write-host "Note: Please make sure to use the latest script that came with the build"
if ($PSVersionTable.PSVersion.Major -lt "7") {
    WriteErrorString "Powershell version detected is less than 7, please use powershell greater than version 7 . Exiting"
   Exit
}  

#Initialize
$jsonHeader="application/*+json;version=36.3"
$xmlHeader="application/*+xml;version=36.3"
#Base64 encoded string of ovf template.
#This script will modify the template reading values from ini file.
#The structure of this template is similar to the ovf template that is shipped as part of the build.
#The modified template will provided to vcloud api for creation of UAG VM
$baseTemplateString = "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"
$baseTemplateString = [System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String($baseTemplateString))
$xmlDoc = New-Object System.Xml.XmlDocument
$xmlDoc.LoadXml($baseTemplateString)


#defining required functions
#function to change values in product section of the template
function changeAttributeValueInProductSection {
    param (
        [Parameter(Position = 0, Mandatory = $true)]
        [String]$key,
        
        [Parameter(Position = 1, Mandatory = $true)]
        [String]$keyValue,

        [Parameter(Position = 2, Mandatory = $true)]
        [String]$setValue

    )

   foreach ($childNode in $productSection.ChildNodes) {
    if ($childNode.HasAttribute($key) -and $childNode.GetAttribute($key) -eq $keyValue) {
            $childNode.SetAttribute("ovf:value", $setValue)
       break;
    }
    }
}

#function which removed network section of the template
function removeNetworkConnection {
    $childNodes=$networkSection.ChildNodes
    for ($i = $childNodes.Count - 1; $i -ge 0; $i--) {
        $node = $childNodes[$i]
        if ($node.HasAttribute("network")) {
             $networkSection.RemoveChild($node) | Out-Null
        }
    }
}

#Adds a network connection element in the template
function createANeworkConnection {
    param (
        [Parameter(Position = 0, Mandatory = $true)]
        [int]$interface,

        [Parameter(Position = 1, Mandatory = $false)]
        [String]$ip,

        [Parameter(Position = 2, Mandatory = $true)]
        [String]$network

    )

    $networkConnectionNode = $xmlDoc.CreateElement("root","NetworkConnection","http://www.vmware.com/vcloud/v1.5")
    $networkConnectionNode.SetAttribute("network",$network)

    $networkConnectionIndexElement = $xmlDoc.CreateElement("root","NetworkConnectionIndex","http://www.vmware.com/vcloud/v1.5")
    $networkConnectionIndexElement.InnerText=$interface

    $ipAddressAllocationElement = $xmlDoc.CreateElement("root","IpAddressAllocationMode","http://www.vmware.com/vcloud/v1.5")



    $isConnectedElement = $xmlDoc.CreateElement("root","IsConnected","http://www.vmware.com/vcloud/v1.5")
    $isConnectedElement.InnerText="true"

    $networkConnectionNode.AppendChild($networkConnectionIndexElement) | Out-Null
    if (-not($ip -eq $null -Or $ip -eq "")) {
        $ipAddressAllocationElement.InnerText="MANUAL"
        $ipAddressElement = $xmlDoc.CreateElement("root","IpAddress","http://www.vmware.com/vcloud/v1.5")
        $ipAddressElement.InnerText =$ip
        $networkConnectionNode.AppendChild($ipAddressElement) | Out-Null
    }
    else {
        $ipAddressAllocationElement.InnerText="DHCP"
    }
    $networkConnectionNode.AppendChild($isConnectedElement) | Out-Null
    $networkConnectionNode.AppendChild($ipAddressAllocationElement) | Out-Null
    $networkSection.AppendChild($networkConnectionNode) | Out-Null
}


function GetNetOptionsForVCloud {
    Param ($settings, $nic)


    $ipModeLabel = "ipMode" + $nic
    $ipMode = $settings.General.$ipModeLabel

    $ipLabel = "ip" + $nic
    $ip=$settings.General.$ipLabel

    $netmaskLabel = "netmask" + $nic
    $netmask=$settings.General.$netmaskLabel

    $v6ipLabel = "v6ip" + $nic
    $v6ip=$settings.General.$v6ipLabel

    $v6ipprefixLabel = "v6ipprefix" + $nic
    $v6ipprefix=$settings.General.$v6ipprefixLabel

    if (($ip.length -gt 0) -and ($netmask.length -eq 0)) {
        WriteErrorString "Error: missing value $netmaskLabel."
        Exit
    }
    if (($v6ip.length -gt 0) -and ($v6ipprefix.length -eq 0)) {
        WriteErrorString "Error: missing value $v6ipprefixLabel."
        Exit
    }

    if ($ipMode.length -eq 0) {

        $ipMode = "DHCPV4"

        if (($ip.length -gt 0) -and ($v6ip.length -eq 0)) {
            $ipMode = "STATICV4"
        }

        if (($ip.length -eq 0) -and ($v6ip.length -gt 0)) {
            $ipMode = "STATICV6"
        }

        if (($ip.length -gt 0) -and ($v6ip.length -gt 0)) {
            $ipMode = "STATICV4+STATICV6"
        }
    }
    $ipMode
}

function getvCloudPassword {
    param( [Parameter(Mandatory=$true)] $vCloudUrl)
    
    try{
    while(! $valid)
    {
            $vCloudPassword = Read-Host -assecurestring "Enter $vCloudUrl password "
            $vCloudPassword = ConvertFromSecureToPlain $vCloudPassword
            if(! $vCloudPassword.Length -gt 0)
        {
            WriteErrorString "Enter some values"
            $valid = 0;
        }else {
            $valid = 1;
        }
    }
    }catch{
        #If try catch is not added, the prompt to enter the password will be shown infinitely untill valid value is provided.
        #If user wants to exit pressing Ctrl+c , it wont work. So this block is added
        WriteErrorString "Problem encountered while accepting vcloud password ,so Exiting."
        Exit
    }
    $vCloudPassword
}

#function to validate the response
function validateReponse()
{
    param(
        [Parameter(Mandatory = $true)]
        [hashtable]$responseObj,

        [Parameter(Mandatory = $true)]
        [String]$responseCode,

        [Parameter(Mandatory = $true)]
        [String]$source,

        [Parameter(Mandatory = $false)]
        [String]$failureMessage

    )
    if(!($responseObj["statusCode"] -eq $responseCode))
    {
        WriteErrorString "Problem in invoking request for $source, so Exiting."
        if($failureMessage.Length -gt 0)
        {
            WriteErrorString $failureMessage
        }
        Exit
    }  
}

#fucntion to invoke api call
function invokeRequest()
{
    param (
        [Parameter(Mandatory = $true)]
        [String]$method,
        
        [Parameter(Mandatory = $true)]
        [String]$url1,

        [Parameter(Mandatory = $false)]
        [String]$authToken,

        [Parameter(Mandatory = $false)]
        [String]$acceptHeader,

        [Parameter(Mandatory = $false)]
        [String]$contentTypeHeader,

        [Parameter(Mandatory = $false)]
        [String]$requestBody
    )

   
    try {
        $responseMap = @{}
        $webRequest = [System.Net.WebRequest]::Create($url1)
        $webRequest.Method = $method
        if($authToken.Length -gt 0)
        {
            $webRequest.Headers.Add("Authorization", $authToken)
        }
        if($acceptHeader.Length-gt 0)
        {
            $webRequest.Accept = $acceptHeader
        }
        if($contentTypeHeader.Length-gt 0)
        {
            $webRequest.ContentType = $contentTypeHeader
        }
    
        if($requestBody.Length -gt 0)
        {
            $content = [System.Text.Encoding]::UTF8.GetBytes($requestBody)
            $webRequest.ContentLength = $content.Length
            $requestStream = $webRequest.GetRequestStream()
            $requestStream.Write($content, 0, $content.length)
        }else {
            $webRequest.ContentLength = 0
        }
        [System.Net.WebResponse] $response = $webRequest.GetResponse();
        $responseMap["header"] = $response.Headers
        $responseMap["statusCode"] = $response.StatusCode
        $streamReader = New-Object System.IO.StreamReader($response.GetResponseStream())
        $responseContent = $streamReader.ReadToEnd()
        $responseMap["body"] = $responseContent; 
    }catch [System.Net.WebException] {
            $errorResponse = $_.Exception.Response
            if($null -eq $errorResponse)
            {
            WriteErrorString "Check connectivity with vcloud. Exiting"
               Exit
            }
            if ($errorResponse.StatusCode -ge [System.Net.HttpStatusCode]::BadRequest) {
                $errorBody = $errorResponse.GetResponseStream()
                $reader = New-Object System.IO.StreamReader($errorBody)
                $errorContent = $reader.ReadToEnd()
                $responseMap["header"] = $errorResponse.Headers
                $responseMap["statusCode"] =$errorResponse.StatusCode
                $responseMap["body"] = $errorContent
                if($errorContent.Length -gt 0)
                {
                    WriteErrorString "Invoke request failed due to $errorContent"
                }   
                $reader.Close()
            } 
    }catch{     
        WriteErrorString "Invoke request failed due to  $_.Exception.Response"
    } finally {
        if($null -ne $streamReader)
        {
            $streamReader.Close()
        }

        if($null -ne $response)
        {
            $response.Close()
        }   
    } 
    $responseMap
}

#Provides xml payload to power of the VM
function getPowerOffPayload()
{

    $powerOffPayloadString='<root:UndeployVAppParams xmlns:root="http://www.vmware.com/vcloud/v1.5"><root:UndeployPowerAction>powerOff</root:UndeployPowerAction></root:UndeployVAppParams>'
    $powerOffPayload = New-Object System.Xml.XmlDocument
    $powerOffPayload.LoadXml($powerOffPayloadString)
    $powerOffPayload.OuterXml
}

function waitForTaskToComplete()
{
    param (
        [Parameter(Mandatory = $true)]
        [String]$xmlString,

        [Parameter(Mandatory = $true)]
        [String]$task
    )

    $xml1 = New-Object System.Xml.XmlDocument
    $xml1.LoadXml($xmlString)
    $href = $xml1.Task.href
    $counter=0;
    while ($true) {
        $output = invokeRequest -Method "Get" -url1 $href -acceptHeader $jsonHeader -authToken $token
        validateReponse -responseObj $output -responseCode "OK" -source "wait Fot Task to complete function" -failureMessage "Task to $task , did not give OK response"
        $jsonObject = $output["body"] | ConvertFrom-Json
        $status = $jsonObject.status
        if($status -eq "success")
        {
            break
        }elseif($status -eq "error" -or $status -eq "canceled" -or $status -eq "aborted")
        {
            WriteErrorString "Exiting the program , because task to $task returned $status"
            Exit
        }
        else
        {
            Start-Sleep 5
            if($counter -gt 30)
            {
                WriteErrorString "Problem in getting response from task $task"
                Exit
            }
        }
    }
}
#Provides payload for uploading the build
function getOvfUploadUrlXml()
{
    param(
        [Parameter(Mandatory = $true)]
        [String]$ovfUrl,
    
        [Parameter(Mandatory = $true)]
        [String]$templateName
    )
    $uploadBaseXmlString='<root:UploadVAppTemplateParams xmlns:root="http://www.vmware.com/vcloud/v1.5" name="UAG23.06-1" sourceHref="ovflink"><root:Description>UAG23.06</root:Description></root:UploadVAppTemplateParams>'
    $uploadXmlDoc = New-Object System.Xml.XmlDocument
    $uploadXmlDoc.LoadXml($uploadBaseXmlString)
    $uploadXmlDoc = $uploadXmlDoc.DocumentElement
    $uploadXmlDoc.SetAttribute("sourceHref",$ovfUrl)
    $uploadXmlDoc.SetAttribute("name",$templateName)
    $uploadXmlDoc.Description = $templateName
    $uploadXmlDoc.OuterXml
}

function updateBuildInfoInProductSection {
    param (
        [Parameter(Position = 0, Mandatory = $true)]
        [String]$nodeName,

        [Parameter(Position = 1, Mandatory = $true)]
        [String]$value

    )
    $childNodes=$productSection.ChildNodes
    for ($i = $childNodes.Count - 1; $i -ge 0; $i--) {

        $node = $childNodes[$i]

        if($node.Name -eq $nodeName)
        {
            $node.InnerText = $value
        }
    }
}

function updateOvfProperties()
{
    #Update Description section
    $descriptionElement.SetAttribute("href",$vmHref)
    $descriptionElement.SetAttribute("id",$vmId)

 if($apName.Length -gt 0)
 {
    $descriptionElement.SetAttribute("name",$apName)
 }
#configure network
#Network Change
 #remove network connection node
  removeNetworkConnection
 #add network connection node
    switch ($deploymentOption) {

        'onenic' {
            $ipMode0 = GetNetOptionsForVCloud $settings "0"
            $ipMode0=$ipMode0.Replace("--prop:ipMode0=","")
            createANeworkConnection -interface "0" -ip $ip0 -network $netInternet
        }
        'twonic' {
            $ipMode0 = GetNetOptionsForVCloud $settings "0"
            $ipMode0=$ipMode0.Replace("--prop:ipMode0=","")
            $ipMode1 = GetNetOptionsForVCloud $settings "1"
            $ipMode1=$ipMode1.Replace("--prop:ipMode1=","")
            createANeworkConnection -interface "0" -ip $ip0 -network $netInternet
            createANeworkConnection -interface "1" -ip $ip1 -network $netManagementNetwork
        }
        'threenic' {
            $ipMode0 = GetNetOptionsForVCloud $settings "0"
            $ipMode0=$ipMode0.Replace("--prop:ipMode0=","")
            $ipMode1 = GetNetOptionsForVCloud $settings "1"
            $ipMode1=$ipMode1.Replace("--prop:ipMode1=","")
            $ipMode2 = GetNetOptionsForVCloud $settings "2"
            $ipMode2=$ipMode2.Replace("--prop:ipMode2=","")
            createANeworkConnection -interface "0" -ip $ip0 -network $netInternet
            createANeworkConnection -interface "1" -ip $ip1 -network $netManagementNetwork
            createANeworkConnection -interface "2" -ip $ip2 -network $netBackendNetwork
        }
    }
    
#Change values in Product Section
if($rootPwd.Length -gt 0)
{
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "rootPassword" -setValue $rootPwd
}

if($adminPwd.Length -gt 0)
{
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminPassword" -setValue $adminPwd
}

if ($dns.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "DNS" -setValue $dns
}

if ($rootPasswordExpirationDays.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "rootPasswordExpirationDays" -setValue $rootPasswordExpirationDays
}

if ($passwordPolicyMinLen.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "passwordPolicyMinLen" -setValue $passwordPolicyMinLen
}

if ($passwordPolicyMinClass.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "passwordPolicyMinClass" -setValue $passwordPolicyMinClass
}

if ($passwordPolicyUnlockTime.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "passwordPolicyMinClass" -setValue $passwordPolicyUnlockTime
}

if ($passwordPolicyFailedLockout.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "passwordPolicyFailedLockout" -setValue $passwordPolicyFailedLockout
}

#Admin Password policy settings
if ($adminPasswordFailedLockoutCount.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminPasswordPolicyFailedLockoutCount" -setValue $passwordPolicyFailedLockout

if ($adminPasswordMinLen.length -gt 0){
            changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminPasswordPolicyMinLen" -setValue $adminPasswordMinLen
}

if ($adminPasswordLockoutTime.length -gt 0){
            changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminPasswordPolicyUnlockTime" -setValue $adminPasswordLockoutTime
}
}

if ($adminSessionIdleTimeoutMinutes.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminSessionIdleTimeoutMinutes" -setValue $adminSessionIdleTimeoutMinutes
}

$adminMaxConcurrentSessions = ValidateAdminMaxConcurrentSessions $settings
if ($adminMaxConcurrentSessions.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminMaxConcurrentSessions" -setValue $adminMaxConcurrentSessions
}

$rootSessionIdleTimeoutSeconds = ValidateRootSessionIdleTimeoutSeconds $settings
if ($rootSessionIdleTimeoutSeconds.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "rootSessionIdleTimeoutSeconds" -setValue $rootSessionIdleTimeoutSeconds
}

$commandsFirstBoot = ValidateCustomBootTimeCommands $settings "commandsFirstBoot"
if ($commandsFirstBoot.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "commandsFirstBoot" -setValue $commandsFirstBoot
}

$commandsEveryBoot = ValidateCustomBootTimeCommands $settings "commandsEveryBoot"
if ($commandsEveryBoot.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "commandsEveryBoot" -setValue $commandsEveryBoot
}

if ($defaultGateway.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "defaultGateway" -setValue $defaultGateway
}

if ($v6DefaultGateway.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "v6DefaultGateway" -setValue $v6DefaultGateway
}

if ($forwardrules.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "forwardrules" -setValue $forwardrules
}

if ($routes0.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "routes0" -setValue $routes0
}

if ($routes1.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "routes1" -setValue $routes1
}

if ($routes2.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "routes2" -setValue $routes2
}

if ($policyRouteGateway0.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "policyRouteGateway0" -setValue $policyRouteGateway0
}

if ($policyRouteGateway1.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "policyRouteGateway1" -setValue $policyRouteGateway1
}

if ($policyRouteGateway2.length -gt 0)
{
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "policyRouteGateway2" -setValue $policyRouteGateway2
}

if ($ceipEnabled -eq $false) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "ceipEnabled" -setValue $ceipEnabled
}

if ($settings.General.dsComplianceOS -eq "true") {
    updatePasswordPolicyForDsComplianceOS $settings
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "dsComplianceOS" -setValue "true"
}

if ($settings.General.tlsPortSharingEnabled -eq "true") {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "tlsPortSharingEnabled" -setValue "true"
}

if ($settings.General.sshEnabled -eq "true") {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "sshEnabled" -setValue "true"
}

if ($settings.General.sshPasswordAccessEnabled -eq "false") {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "sshPasswordAccessEnabled" -setValue "true"
}

if ($settings.General.sshKeyAccessEnabled -eq "true") {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "sshKeyAccessEnabled" -setValue "true"
}

$sshBannerText=ReadLoginBannerText $settings
if ($sshBannerText.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "sshLoginBannerText" -setValue $sshBannerText
}

$sshInterface = validateSSHInterface $settings
if (($sshInterface.length -gt 0)) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "sshInterface" -setValue $sshInterface
}

$sshPort = $settings.General.sshPort
if ($sshPort.length -gt 0 -and ($sshPort -match '^[0-9]+$')) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "sshPort" -setValue $sshPort
}

$secureRandomSource=ReadSecureRandomSource $settings
if ($secureRandomSource.length -gt 0) {
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "secureRandomSource" -setValue $secureRandomSource
}

if($enabledAdvancedFeatures.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "enabledAdvancedFeatures" -setValue $enabledAdvancedFeatures
}

$gatewaySpec = getGatewaySpec $settings
if($gatewaySpec.length -gt 0){
    changeAttributeValueInProductSection -key "ovf:key" -keyValue "gatewaySpec" -setValue $gatewaySpec
}

if($configURL.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "configURL" -setValue $configURL
}

if($configKey.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "configKey" -setValue $configKey
}

if($configURLThumbprints.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "configURLThumbprints" -setValue $configURLThumbprints
}

if($configURLHttpProxy.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "configURLHttpProxy" -setValue $configURLHttpProxy
}

if($adminCsrSubject.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminCsrSubject" -setValue $adminCsrSubject
}

if($adminCsrSAN.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "adminCsrSAN" -setValue $adminCsrSAN
}

if($additionalDeploymentMetadata.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "additionalDeploymentMetadata" -setValue $additionalDeploymentMetadata
}

if($netMask0.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "netMask0" -setValue $netMask0
}

if($netMask1.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "netMask1" -setValue $netMask1
}

if($netMask2.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "netMask2" -setValue $netMask2
}

if($ipMode0.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "ipMode0" -setValue $ipMode0
}

if($ipMode1.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "ipMode1" -setValue $ipMode1
}

if($ipMode2.length -gt 0){
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "ipMode2" -setValue $ipMode2
}

if($jsonString.length -gt 0)
{
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "settingsJSON" -setValue $jsonString
}

if($ip0.length -gt 0)
{
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "ip0" -setValue $ip0
}

if($ip1.length -gt 0)
{
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "ip1" -setValue $ip1
}

if($ip2.length -gt 0)
{
        changeAttributeValueInProductSection -key "ovf:key" -keyValue "ip2" -setValue $ip2
    }

}
if (!(Test-path $iniFile)) {
    WriteErrorString "Error: Configuration file ($iniFile) not found."
    Exit
}

#read Ini file
#General Section
$settings = ImportIni $iniFile
$apName=$settings.General.name
$deploymentOption=$settings.General.deploymentOption
$dns=$settings.General.dns
$defaultGateway=$settings.General.defaultGateway
$v6DefaultGateway=$settings.General.v6DefaultGateway
$forwardrules=$settings.General.forwardrules
$netInternet=$settings.General.netInternet
$ip0=$settings.General.ip0
$routes0=$settings.General.routes0
$policyRouteGateway0=$settings.General.policyRouteGateway0
$netmask0=$settings.General.netmask0
$netManagementNetwork=$settings.General.netManagementNetwork
$ip1=$settings.General.ip1
$routes1=$settings.General.routes1
$policyRouteGateway1=$settings.General.policyRouteGateway1
$netmask1=$settings.General.netmask1
$netBackendNetwork=$settings.General.netBackendNetwork
$ip2=$settings.General.ip2
$routes2=$settings.General.routes2
$policyRouteGateway2=$settings.General.policyRouteGateway2
$netmask2=$settings.General.netmask2
$rootPasswordExpirationDays=$settings.General.rootPasswordExpirationDays
$passwordPolicyMinLen=$settings.General.passwordPolicyMinLen
$passwordPolicyMinClass=$settings.General.passwordPolicyMinClass
$passwordPolicyUnlockTime=$settings.General.passwordPolicyUnlockTime
$passwordPolicyFailedLockout=$settings.General.passwordPolicyFailedLockout
$enabledAdvancedFeatures=$settings.General.enabledAdvancedFeatures
$adminPasswordMinLen=$settings.General.adminPasswordPolicyMinLen
$adminPasswordLockoutTime=$settings.General.adminPasswordPolicyUnlockTime
$adminPasswordFailedLockoutCount=$settings.General.adminPasswordPolicyFailedLockoutCount
$adminSessionIdleTimeoutMinutes=$settings.General.adminSessionIdleTimeoutMinutes
$configURL = $settings.General.configURL
$configKey = $settings.General.configKey
$configURLThumbprints = $settings.General.configURLThumbprints
$configURLHttpProxy = $settings.General.configURLHttpProxy
$adminCsrSubject = $settings.General.adminCsrSubject
$adminCsrSAN = $settings.General.adminCsrSAN
$additionalDeploymentMetadata = $settings.General.additionalDeploymentMetadata
#vCloud Section
$vcHost=$settings.vCloud.vcHost
$vcUsername=$settings.vCloud.vcUsername
$vcVdcName=$settings.vCloud.vcVdcName
$vAppName=$settings.vCloud.vcVappName
if (!$ocPassword) {
    $ocPassword = getvCloudPassword -vCloudUrl $vcHost
}
$vcOrgId=$settings.vCloud.vcOrgId
$catalog=$settings.vCloud.catalog
#$ocVdc=$settings.vCloud.ocVdc
$uagTemplate=$settings.vCloud.uagTemplate
$uploadOvfUrl=$settings.vCloud.uploadOvfUrl

## Ini read completed ###
if (!$apName) {
    $apName = GetAPName
}

$osLoginUsername = ReadOsLoginUsername $settings
if ($osLoginUsername.length -eq 0) {
    $osLoginUsername = "root"
}

if ($settings.General.dsComplianceOS -eq "true") {
    updatePasswordPolicyForDsComplianceOS $settings
}

if (!$rootPwd) {
    $rootPwd = GetRootPwd  $apName $settings $osLoginUsername
}

if (!$adminPwd) {
    $adminPwd = GetAdminPwd $apName $settings
}

if (!$ceipEnabled) {
    $ceipEnabled = GetCeipEnabled $apname
}
$settingsJSON=GetJSONSettings $settings $newAdminUserPwd
$jsonString=GetSettingsJSONProperty $settingsJSON

$jsonString=$jsonString.Replace('\"',"""").Replace("prop:settingsJSON=","")
#Write-Host "Json string is $jsonString"

#login and get token
$base64Encoded = [System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($vcUsername+"@"+$vcOrgId+":"+$ocPassword))
$basicAuthHeader="Basic"+" "+$base64Encoded
$url="https://$vcHost/api/sessions"
$output = invokeRequest -Method "Post" -url1 $url -acceptHeader $jsonHeader -authToken $basicAuthHeader
$token=$output["header"]["X-VMWARE-VCLOUD-ACCESS-TOKEN"]
if (! $token.Length -gt 0 )
{
    WriteErrorString "Check vcloud  username , password and org values provided are valid in vCloud Section of ini. Exiting"
    Exit
}
$token="Bearer $token"
#Write-Host "token is $token"

#get catalog id
$url="https://$vcHost/api/query?type=catalog"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token
validateReponse -responseObj $output -responseCode "OK" -source "getting catalog id"
$jsonObject = $output["body"] | ConvertFrom-Json
$catalogId=($jsonObject.record | Where-Object {$_.name -eq $catalog}).href
if($catalogId.Length -eq 0)
{
    WriteErrorString "Problem in getting catalog id, check vCloud:catalog section in ini value, Exiting"
    Exit
}
$catalogId=$catalogId.Split('/')[-1]
Write-Host "Catalog id is $catalogId"

#check if template is already present. Upload if not present
Write-Host "Checking if $uagTemplate is already present"
$url="https://$vcHost/api/catalog/$catalogId"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token
validateReponse -responseObj $output -responseCode "OK" -source "getting catalog information" -failureMessage "Check valid catalog is provided"
$jsonObject = $output["body"] | ConvertFrom-Json
$catalogItemsId = ($jsonObject.catalogItems.catalogItem | Where-Object { $_.name -eq $uagTemplate }).href
if([string]::IsNullOrEmpty($catalogItemsId) )
{
    if($uploadOvfUrl.Length -gt 0)
    {
        Write-Host "Template  $uagTemplate not found so uploading"
        $url="https://$vcHost/api/catalog/$catalogId/action/upload"
        $uploadBody=getOvfUploadUrlXml -ovfUrl $uploadOvfUrl -templateName $uagTemplate
        $output = invokeRequest -Method "Post" -url1 $url -acceptHeader $jsonHeader -ContentType $xmlHeader -requestBody $uploadBody -authToken $token 
        validateReponse -responseObj $output -responseCode "Created" -source "uploading ova" -failureMessage "Check if ova template is already present"
    }else{
        Write-Host "Provide uploadOvfUrl in ini"
        Exit 
    }     
}

#get catalog items id
Write-Host "Invoking request to get catalog items id"
$url="https://$vcHost/api/catalog/$catalogId"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token
validateReponse -responseObj $output -responseCode "OK" -source "getting catalog information" -failureMessage "Check valid catalogId is provided"
$jsonObject = $output["body"] | ConvertFrom-Json
$catalogItemsId = ($jsonObject.catalogItems.catalogItem | Where-Object { $_.name -eq $uagTemplate }).href
$catalogItemsId=$catalogItemsId.Split('/')[-1]
Write-Host "Catalog items id is $catalogItemsId"

#get Vapp template URL ID
Write-Host "Invoking request to get template url id"
$url="https://$vcHost/api/catalogItem/$catalogItemsId"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token
validateReponse -responseObj $output -responseCode "OK" -source "getting template url id"
$jsonObject = $output["body"] | ConvertFrom-Json
$vappTemplateUrlId = ($jsonObject.entity | Where-Object { $_.name -eq $uagTemplate }).href
$vappTemplateUrlId=$vappTemplateUrlId.Split('/')[-1]
Write-Host "Template URL id is $vappTemplateUrlId"

#Check vm template upload was successful, if yes obtain its id and add it in the xml
Write-Host "Checking if ovf template is already present"
$url="https://${vcHost}/api/vAppTemplate/$vappTemplateUrlId"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token
validateReponse -responseObj $output -responseCode "OK" -source "getting template info"
$counter=0;
while ($true) {
    $output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token
    $jsonObject = $output["body"] | ConvertFrom-Json
    #If task object is not present, then ovf upload was successful
    $flag=($jsonObject.tasks)
    if([string]::IsNullOrEmpty($flag))
    {
        Start-Sleep 1;
        break
    }
    $counter++
    Write-Host "Waiting for ovf upload to complete.Please wait ....."
    Start-Sleep 30;
    if($counter -gt 50)
    {
        WriteErrorString "Upload not successful after 10 minutes so , existing. Please check whether upload status in vcloud"
       Exit
    }
}
$vmHref = ($jsonObject.children.vm | Where-Object { $_.name -eq "EUC_Unified_Access_Gateway" }).href
$vmId = ($jsonObject.children.vm | Where-Object { $_.name -eq "EUC_Unified_Access_Gateway" }).id
#$xmlDoc.Save($modifiedFile)

#Write-Host "vmHref name is $vmHref"
#Write-Host "vmId name is $vmId"

#getting vApp id
$filterValue=[System.Web.HttpUtility]::UrlEncode("((vdcName==${vcVdcName}))")
$url="https://${vcHost}/api/query?type=vApp&format=records&filter=${filterValue}"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token 
$jsonObject = $output["body"] | ConvertFrom-Json
Write-Host "vApp name is $vAppName"
$vAppId = ($jsonObject.record | Where-Object { $_.name -eq $vAppName }).href
if($vAppId.Length -eq 0)
{
    WriteErrorString "Problem in getting vApp id, check vCloud:vcVappName section in ini "
    Exit
}
$vAppId=$vAppId.Split('/')[-1]
Write-Host "vApp id is $vAppId"

#If VM with same name already exist , power off and delete it
$filterValue=[System.Web.HttpUtility]::UrlEncode("((containerName==${vAppName});(name==${apName});(vdcName==${vcVdcName}))")
$url="https://${vcHost}/api/query?type=vm&format=records&filter=${filterValue}"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token 
$jsonObject = $output["body"] | ConvertFrom-Json
$vmName = $jsonObject.record.name
if($vmName.Length -gt 0 )
{
    Write-Host "Vm with same name already exist , so deleting it and creating a new one"
    $vmId=($jsonObject.record.href).Split('/')[-1]
    Write-Host "vm id is $vmId"
    if($vmId.Length -gt 0){
        $poweredOffStatus = $jsonObject.record.status
        if($poweredOffStatus -ne "POWERED_OFF" -And $poweredOffStatus -ne "FAILED_CREATION")
        {
            Write-Host "Powering off the VM"
            $url="https://${vcHost}/api/vApp/$vmId/action/undeploy"
            $powerOffBody=getPowerOffPayload
            $output = invokeRequest -Method "Post" -url1 $url -acceptHeader $xmlHeader -ContentType $xmlHeader -requestBody $powerOffBody -authToken $token
            validateReponse -responseObj $output -responseCode "Accepted" -source "powering off VM" -failureMessage "Not able to power off already existing VM, check vcloud logs"
            waitForTaskToComplete $output["body"] "power off VM"
        }
        Write-Host "Deleting the VM"
        $url="https://${vcHost}/api/vApp/${vmId}"
        $output = invokeRequest -Method "Delete" -url1 $url -acceptHeader $xmlHeader -authToken $token
        validateReponse -responseObj $output -responseCode "Accepted" -source "Deleting VM" -failureMessage "Not able to delete already existing VM, check vcloud logs"
        waitForTaskToComplete $output["body"] "Delete the VM"
    }else{
        WriteErrorString "Not able to get vm id while deleting existing VM, so Exiting"
        Exit
    }
}

Write-Host "Getting product section from Vapp template"
#We need to add ProductSection in reCompose xml.
#So make a call to get vapp template which will give ProductSection.
#get ProductSection and add it recompose xml at required place
$rootElement = $xmlDoc.DocumentElement
$output = invokeRequest -Method "Get" -url1 $vmHref -acceptHeader $xmlHeader -authToken $token
$xmlTemplate = New-Object System.Xml.XmlDocument
$xmlTemplate.LoadXml($output["body"])
$productSectionTemplate=$xmlTemplate.DocumentElement.ProductSection
$targetSection=$rootElement.SourcedItem.InstantiationParams

if ($productSectionTemplate.Count -eq 0)
{
    WriteErrorString "Not able to get product section from template, please check the ovf file"
    exit
}
foreach ($section in $productSectionTemplate) {

    $importedSection = $xmlDoc.ImportNode($section, $true)
    $targetSection.AppendChild($importedSection) | Out-Null
}

#Make changes to ovf properties present in recompose xml
$descriptionElement=$rootElement.SourcedItem.Source
$networkSection=$rootElement.SourcedItem.InstantiationParams.NetworkConnectionSection
$productSection=$rootElement.SourcedItem.InstantiationParams.ProductSection
updateOvfProperties
#$xmlDoc.OuterXml


#create VM using the xml payload created earlier
Write-Host "Invoking request to create VM"
$url="https://${vcHost}/api/vApp/$vAppId/action/recomposeVApp"
$output = invokeRequest -Method "Post" -url1 $url -acceptHeader $xmlHeader -ContentType $xmlHeader -authToken $token -requestBody $xmlDoc.OuterXml
validateReponse -responseObj $output -responseCode "Accepted" -source "creating VM" -failureMessage "Problem while creating the VM"
waitForTaskToComplete $output["body"] "Create VM"

#power on VM
$filterValue=[System.Web.HttpUtility]::UrlEncode("((containerName==${vAppName});(name==${apName});(vdcName==${vcVdcName}))")
$url="https://${vcHost}/api/query?type=vm&format=records&filter=${filterValue}"
$output = invokeRequest -Method "Get" -url1 $url -acceptHeader $jsonHeader -authToken $token 
$jsonObject = $output["body"] | ConvertFrom-Json
$vmId=($jsonObject.record.href).Split('/')[-1]
Write-Host "Sleeping for 20 seconds , before powering on VM"
Start-Sleep 20
$url="https://${vcHost}/api/vApp/${vmId}/power/action/powerOn"
$output = invokeRequest -Method "Post" -url1 $url -acceptHeader $jsonHeader -authToken $token
validateReponse -responseObj $output -responseCode "Accepted" -source "powering on VM"
Write-Host "Creation of VM completed"