/*
 *  linux/arch/i386/kernel/vmnix.c
 *
 *  Copyright 2000 VMware, Inc.  All rights reserved. -- VMware Confidential
 */

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/stddef.h>
#include <linux/unistd.h>
#include <linux/version.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/kallsyms.h>
#include <asm/uaccess.h>
#include <asm/semaphore.h>

#define __KERNEL_SYSCALLS__
#include <asm/io.h>

#include <linux/proc_fs.h>
#include <linux/file.h>

#include <asm/page.h>
#include <asm/pgtable.h>
#include <linux/smp.h>
#include <asm/e820.h>
#include <linux/irq.h>
#include <asm/vmnix.h>
#include <asm/fixmap.h>

#ifdef SMP
#error "Please compile your kernel UP "
#endif

#ifndef CONFIG_VMNIX
#error "CONFIG_VMNIX must be defined"
#endif



unsigned vmnixVersion = VMNIX_VERSION;
unsigned long long vmnix_mem_size;
char vmnixPCIInfo[512];

/*
 * local functions
 */

static void vmnix_save_cpci_cmdline(const char *cmdLine);

/*
 * A lock for mutex access to the e820 structs after boot
 */
rwlock_t e820_lock  __cacheline_aligned = RW_LOCK_UNLOCKED;

/*
 * vmnix state
 */

VMnix_Init vmnixInitBlock;
VMnix_VTable vmnixVTable;
unsigned long long vmnixTotalRAM;
struct e820map  e820vmk;

static void  
vmnix_print_memory_map_vmk(char *who)
{
   int i;

   for (i = 0; i < e820vmk.nr_map; i++) {
      printk(" %s: %016Lx - %016Lx ", who,
             e820vmk.map[i].addr,
             e820vmk.map[i].addr + e820vmk.map[i].size);
      switch (e820vmk.map[i].type) {
      case E820_RAM:	
         printk("(usable)\n");
         break;
      case E820_RESERVED:
         printk("(reserved)\n");
         break;
      case E820_ACPI:
         printk("(ACPI data)\n");
         break;
      case E820_NVS:
         printk("(ACPI NVS)\n");
         break;
      default:	
         printk("type %lu\n", e820vmk.map[i].type);
         break;
      }
   }
}

static unsigned long long
vmnix_compute_ram_end(struct e820map *e820p)
{
   int i;

   unsigned long long ram_end = 0;

   for (i = 0; i < e820p->nr_map; i++) {
      struct e820entry *map_ptr = &e820p->map[i];
      unsigned long long start = map_ptr->addr;
      unsigned long long size = map_ptr->size;
      unsigned long long end = start + size;
      unsigned long type = map_ptr->type;

      /* Overflow in 64 bits? Ignore the memory map. */
      if (start > end)
	 return 0; 

      if ((type == E820_RAM) && (end > ram_end)) {
           ram_end = end;
      }
   }

   if (ram_end < 0xffffffff) {
      printk(KERN_DEBUG "E820: reported memory end %Lx\n", ram_end);
      return ram_end;
   } else {
      /* didn't find a usable region above 1 MB */
      return 0;
   }
}

/*
 *----------------------------------------------------------------------
 *
 * vmnix_setup --
 *
 *      Early init from the setup routine.
 * 
 * Results: 
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
vmnix_setup(struct e820map *e820,
            const char *cmdLine)
{
   int i, j;
   unsigned long long linuxMemEnd = vmnix_compute_ram_end(e820);

   vmnix_save_cpci_cmdline(cmdLine);
   printk("VMNIX: linux MA=[0x%x,0x%Lx)\n", 0, linuxMemEnd);

   j = 0;
   for (i = 0; i < e820vmk.nr_map; i++) {
      if (e820vmk.map[i].type == E820_RAM) {
         unsigned long long start = e820vmk.map[i].addr;
         unsigned long long size = e820vmk.map[i].size;
         unsigned long long end; 
	 
         /* Discard memory regions that are smaller than a full page */
         if ( size < PAGE_SIZE) {
            continue;
         }

         end = start + size;
         if (end < linuxMemEnd) {
            continue;
         }
         if (start < linuxMemEnd) {
            start = linuxMemEnd;
         }
         vmnixInitBlock.vmkMem[j].startMPN = start >> 12;
         vmnixInitBlock.vmkMem[j].endMPN = (end >> 12) - 1;
         printk("VMNIX: vmkernel MPN=[0x%x,0x%x]\n",
                vmnixInitBlock.vmkMem[j].startMPN,
                vmnixInitBlock.vmkMem[j].endMPN);
         j++;
         if (j == MAX_VMNIX_MEM_RANGES) {
            printk("VMNIX: e820(%d) has more memory entries than vmkernel can handle\n", e820vmk.nr_map);
            break;
         } else {
         }
      }
   }
   vmnixInitBlock.nextMPN = vmnixInitBlock.vmkMem[0].startMPN;

   /*  mark the end if fewer than max entries used */
   if (j < MAX_VMNIX_MEM_RANGES) {
      vmnixInitBlock.vmkMem[j].startMPN = 0;
   }

   vmnix_print_memory_map_vmk("vmk");
}

/*
 * Temporarily use the virtual area starting from FIX_IO_APIC_BASE_0
 * to map the target physical address.
 * !!! CANNOT be called after smp_init() because the ioapic area is no longer
 * free.
 */
unsigned char *
vmnix_maptemp(unsigned int phys, unsigned int size) // derived from acpitable.c
{
   unsigned int offset;
   int idx;
   unsigned int mapped_size;

   /*
    * set_fixmap will map the page containing 'phys', no need
    * to align it, however we must adjust the 'size' to reflect
    * that we are mapping at the first page boundary before 'phys'.
    */
   offset = phys & (PAGE_SIZE-1);
   size += offset;

   /*
    * Map starting at FIX_IO_APIC_BASE_0 till we run out of space or
    * fulfill the request.
    * BEWARE: fixed pages are indexed from the top down so we need
    * to map likewise.
    */
   for (idx = FIX_IO_APIC_BASE_END, mapped_size = 0;
              (idx >= FIX_IO_APIC_BASE_0) && (mapped_size < size);
                    idx--, mapped_size += PAGE_SIZE, phys += PAGE_SIZE) {
      set_fixmap(idx, phys);
   }

   if (mapped_size >= size) {
      return ((unsigned char *)fix_to_virt(FIX_IO_APIC_BASE_END))+offset;
   } else {
      printk("VMNIX: Not enough space at FIX_IO_APIC_BASE_0\n");
      return NULL;
   }
}

/*
 *----------------------------------------------------------------------
 *
 * vmnix_alloc_lowmem --
 *
 *      Called very early to give us a chance to alloc memory in the
 *      real mode range for the AP processor start code.
 * 
 * Results: 
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

unsigned long
vmnix_alloc_lowmem(unsigned long start_low_mem)
{
   unsigned physAddr;
   
   /*
    * This memory buffer is setup by vmnix because it has
    * to be in a physical address accessable by real mode.
    */
   
   physAddr = virt_to_phys((void *)start_low_mem);

   if (physAddr >= 0x9F000) {
      panic("vmnix_alloc_lowmem: "
            "Insufficient low memory for vmnix AP realmode code 0x%lx.\n",
            start_low_mem);
   }

   return start_low_mem + PAGE_SIZE;
}


/*
 *----------------------------------------------------------------------
 *
 * sys_vmnix --
 *
 *      
 * 
 * Results: 
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

asmlinkage int
sys_vmnix(unsigned cmd,
	  char *inBuffer,
	  unsigned inBufferLength,
	  char *outBuffer,
	  unsigned outBufferLength)
{
   if (vmnixVTable.syscall) {
      return vmnixVTable.syscall(cmd, inBuffer, inBufferLength,
                                 outBuffer, outBufferLength);
   } else {
      return -ENOSYS;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * vmnix_idle --
 *
 *      
 *
 * Results: 
 *      None.
 *      
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
vmnix_idle(void (*defaultIdle)(void))
{
   if (vmnixVTable.idle) {
      vmnixVTable.idle();
   } else {
      while (!current->need_resched)
         defaultIdle();
   }
}


/*
 *----------------------------------------------------------------------
 *
 * vmnix_lost_ticks --
 *
 *      
 *
 * Results: 
 *      None.
 *      
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

unsigned long
vmnix_lost_ticks(struct pt_regs *regs)
{
   if (vmnixVTable.lostTicks) {
      return vmnixVTable.lostTicks(regs);
   }

   return 1;
}


/*
 *----------------------------------------------------------------------
 *
 * vmnix_sym_dump --
 *
 *      Dump the symbols for all modules in a format usable for gdb.
 *
 * Results: 
 *      None.
 *      
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
int 
vmnix_sym_dump(void *t, const char *modName, const char * sectionName, 
               ElfW(Addr) start, ElfW(Addr) end, ElfW(Word) size)
{
   const char **token = t;

   if (*token != modName) {
      *token = modName;
      printk("\n%-10s", modName);
   }
   if (strcmp(sectionName, ".text") == 0) {
      printk("%#x ", start);
   } else if ((strcmp(sectionName, ".data") == 0) ||
              (strcmp(sectionName, ".bss") == 0)) {
      printk("-s %s %#x ", sectionName, start);
   }
   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * vmnix_panic --
 *
 *      
 *
 * Results: 
 *      None.
 *      
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

extern void log_get_buffer(char **logBuf, int *logEnd, int *logBufLen);
void
vmnix_panic(const char *msg, struct pt_regs *regs)
{
   static int inPanic = 0;

   if (!inPanic) {
      char *token = NULL;
      inPanic = 1;
      /* 
       * Dumping out the module load addreses seems scary, so only do it
       * on the first panic
       */
      kallsyms_sections(&token, vmnix_sym_dump);
      printk("\n");
   }


   if (vmnixVTable.panic) {
      static char eipBuf[512];
      char *logBuf;
      int logBufLen, logEnd;
      const char *msgBuf = msg;

      /* Add eip for the first panic to the msg */
      if (regs && !inPanic) {
         int msgLen;

         msgLen = snprintf(eipBuf, sizeof eipBuf, "%s: EIP is at ", msg);
         if (msgLen < sizeof eipBuf) {
            lookup_symbol(regs->eip, eipBuf + msgLen, (sizeof eipBuf) - msgLen);
            msgBuf = eipBuf;
         }
      }

      log_get_buffer(&logBuf, &logEnd, &logBufLen);
      vmnixVTable.panic(msgBuf, regs, logBuf, logEnd, logBufLen);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * vmnix_console_callback --
 *
 * 	Some Alt-Function key has been pressed and a console switch may
 * 	happen. We need to check if the console coming up is maybe
 * 	a vmkernel one.
 *
 * 	If the vmkernel handles it, it will eventually have to let COS
 * 	update its data so to distinguish that second pass, the nr
 * 	is offset when calling set_console() in modules/vmnix/vga.c/
 * 	VGA_HandleInterrupt(). If that's the case, we remove the offset
 * 	and let COS handle it.
 *
 * Results:
 * 	1 if the vmkernel can handle it, 0 otherwise
 *
 * Side Effects:
 * 	None.
 *
 *----------------------------------------------------------------------
 */

int
vmnix_console_callback(int *nr)
{
   if (*nr >= MAX_NR_CONSOLES) {
      /*
       * The switch has been checked by vmkernel already so it
       * can proceed. Adjust to the real console number.
       */
      *nr -= MAX_NR_CONSOLES;
      return 0;
   }
   
   if (vmnixVTable.consoleCallback) {
      return vmnixVTable.consoleCallback(*nr);
   } else {
      return 0;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * vmnix_settimeofday --
 *
 * Results:
 * 	None.
 *
 * Side Effects:
 * 	Set vmkernel's time of day.
 *
 *----------------------------------------------------------------------
 */

void
vmnix_settimeofday(long long tod)
{
   if (vmnixVTable.setTimeOfDay) {
      vmnixVTable.setTimeOfDay(tod);
   }
   
   return;
}


/*
* cache the cpci= kernel parameter for later parsing in pci.c
* */
static void 
vmnix_save_cpci_cmdline(const char *cmdLine)
{
   const char *p = cmdLine;
   int nChars = 0;
   const char *s;

   vmnixPCIInfo[0] = 0;

   while (*p != 0 && nChars++ < sizeof(vmnixPCIInfo)) {
      if (strncmp(p, "cpci=", 5) == 0) {
         s = p + 5;
         while (*p != ' ' && *p != 0) {
            p++;
         }
         memcpy(vmnixPCIInfo, s, p - s);
         vmnixPCIInfo[p - s] = 0;
         break;
      }
      p++;
   }
   vmnixPCIInfo[sizeof(vmnixPCIInfo) -1] = '\0';
}




void 
vmnix_limit_vmkernel_regions(unsigned long long size)
{
   int i;
   unsigned long long current_size = 0;

   for (i = 0; i < e820vmk.nr_map; i++) {
      if (e820vmk.map[i].type == E820_RAM) {
         current_size += e820vmk.map[i].size;
         if (current_size >= size) {
            e820vmk.map[i].size -= current_size-size;
            e820vmk.nr_map = i + 1;
            return;
         }
      }
   }
}

static void
vmnix_add_memory_region(struct e820map *e820p,
                                    unsigned long long start,
                                    unsigned long long size, int type)
{
	int x = e820p->nr_map;

	if (x == E820MAX) {
	    printk(KERN_ERR "Ooops! Too many entries in the memory map!\n");
	    return;
	}

	e820p->map[x].addr = start;
	e820p->map[x].size = size;
	e820p->map[x].type = type;
	e820p->nr_map++;
} 


void
vmnix_add_vmkernel_memory_region(unsigned long long start,
                                 unsigned long long size, int type)
{
   vmnix_add_memory_region(&e820vmk, start, size, type);
}

static unsigned long long
vmnix_compute_total_ram(struct e820map *e820p)
{
   int i;

   unsigned long long totalRAM = 0;

   for (i = 0; i < e820p->nr_map; i++) {
      struct e820entry *map_ptr = &e820p->map[i];
      unsigned long type = map_ptr->type;

      if (type == E820_RAM) {
           totalRAM += map_ptr->size;
      }
   }

   return totalRAM;
}

void
vmnix_setup_early(struct e820map *srcE820)
{
   int i;

   e820vmk.nr_map = 0;
   for (i = 0; i < srcE820->nr_map; i++) {
      vmnix_add_memory_region(&e820vmk, srcE820->map[i].addr,
                              srcE820->map[i].size,
                              srcE820->map[i].type);
   }

   vmnixTotalRAM = vmnix_compute_total_ram(srcE820);
}

void
vmnix_clear_vmk_e820_map(void)
{
   e820vmk.nr_map = 0;
}

unsigned long long 
vmnix_get_total_ram(void)
{
   return vmnixTotalRAM;
}

unsigned int
vmnix_allow_net_params(char *devName)
{
   if (vmnixVTable.allowNetParams) {
      return vmnixVTable.allowNetParams(devName);
   }

   return 0;
}

void
vmnix_printk(unsigned char alert, char *msg, int bufLen, unsigned char toggle)
{
   if (vmnixVTable.printk) {
      vmnixVTable.printk(alert, msg, bufLen, toggle);
   }
}
