/**
 * mptsas_sas_io_unit_pg0
 *
 * obtaining SAS_IO_UNIT page 0
 *
 * @ioc
 * @port_info
 *
 **/
static int
mptsas_sas_io_unit_pg0(MPT_ADAPTER *ioc, struct mptsas_portinfo *port_info)
{
	ConfigExtendedPageHeader_t hdr;
	CONFIGPARMS cfg;
	SasIOUnitPage0_t *buffer;
	dma_addr_t dma_handle;
	int error, i;

	hdr.PageVersion = MPI_SASIOUNITPAGE0_PAGEVERSION;
	hdr.ExtPageLength = 0;
	hdr.PageNumber = 0;
	hdr.Reserved1 = 0;
	hdr.Reserved2 = 0;
	hdr.PageType = MPI_CONFIG_PAGETYPE_EXTENDED;
	hdr.ExtPageType = MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT;

	cfg.cfghdr.ehdr = &hdr;
	cfg.physAddr = -1;
	cfg.pageAddr = 0;
	cfg.action = MPI_CONFIG_ACTION_PAGE_HEADER;
	cfg.dir = 0;	/* read */
	cfg.timeout = 10;

	error = mpt_config(ioc, &cfg);
	if (error)
		goto out;
	if (!hdr.ExtPageLength) {
		error = -ENXIO;
		goto out;
	}

	buffer = pci_alloc_consistent(ioc->pcidev, hdr.ExtPageLength * 4,
					    &dma_handle);
	if (!buffer) {
		error = -ENOMEM;
		goto out;
	}

	cfg.physAddr = dma_handle;
	cfg.action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;

	error = mpt_config(ioc, &cfg);
	if (error)
		goto out_free_consistent;

	port_info->num_phys = buffer->NumPhys;
	port_info->phy_info = kmalloc(port_info->num_phys *
		sizeof(*port_info->phy_info),GFP_KERNEL);
	if (!port_info->phy_info) {
		error = -ENOMEM;
		goto out_free_consistent;
	}

	if (port_info->num_phys)
		port_info->handle =
		    le16_to_cpu(buffer->PhyData[0].ControllerDevHandle);
	for (i = 0; i < port_info->num_phys; i++) {
		port_info->phy_info[i].phy_id = i;
		port_info->phy_info[i].port_id =
		    buffer->PhyData[i].Port;
		port_info->phy_info[i].negotiated_link_rate =
		    buffer->PhyData[i].NegotiatedLinkRate;
		port_info->phy_info[i].portinfo = port_info;
		port_info->phy_info[i].port_flags =
		    buffer->PhyData[i].PortFlags;
	}

 out_free_consistent:
	pci_free_consistent(ioc->pcidev, hdr.ExtPageLength * 4,
			    buffer, dma_handle);
 out:
	return error;
}

/**
 * mptsas_sas_device_pg0
 *
 * obtaining SAS_DEVICE page 0
 *
 * @ioc
 * device_info
 *
 **/
static int
mptsas_sas_device_pg0(MPT_ADAPTER *ioc, struct mptsas_devinfo *device_info,
		u32 form, u32 form_specific)
{
	ConfigExtendedPageHeader_t hdr;
	CONFIGPARMS cfg;
	SasDevicePage0_t *buffer;
	dma_addr_t dma_handle;
	u64 sas_address;
	int error=0;

	hdr.PageVersion = MPI_SASDEVICE0_PAGEVERSION;
	hdr.ExtPageLength = 0;
	hdr.PageNumber = 0;
	hdr.Reserved1 = 0;
	hdr.Reserved2 = 0;
	hdr.PageType = MPI_CONFIG_PAGETYPE_EXTENDED;
	hdr.ExtPageType = MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE;

	cfg.cfghdr.ehdr = &hdr;
	cfg.pageAddr = form + form_specific;
	cfg.physAddr = -1;
	cfg.action = MPI_CONFIG_ACTION_PAGE_HEADER;
	cfg.dir = 0;	/* read */
	cfg.timeout = 10;

	memset(device_info, 0, sizeof(struct mptsas_devinfo));
	error = mpt_config(ioc, &cfg);
	if (error)
		goto out;
	if (!hdr.ExtPageLength) {
		error = -ENXIO;
		goto out;
	}

	buffer = pci_alloc_consistent(ioc->pcidev, hdr.ExtPageLength * 4,
				      &dma_handle);
	if (!buffer) {
		error = -ENOMEM;
		goto out;
	}

	cfg.physAddr = dma_handle;
	cfg.action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;

	error = mpt_config(ioc, &cfg);
	if (error)
		goto out_free_consistent;

	device_info->handle = le16_to_cpu(buffer->DevHandle);
	device_info->handle_parent = le16_to_cpu(buffer->ParentDevHandle);
	device_info->handle_enclosure =
	    le16_to_cpu(buffer->EnclosureHandle);
	device_info->slot = le16_to_cpu(buffer->Slot);
	device_info->phy_id = buffer->PhyNum;
	device_info->port_id = buffer->PhysicalPort;
	device_info->id = buffer->TargetID;
	device_info->channel = buffer->Bus;
	memcpy(&sas_address, &buffer->SASAddress, sizeof(u64));
	device_info->sas_address = le64_to_cpu(sas_address);
	device_info->device_info =
	    le32_to_cpu(buffer->DeviceInfo);

 out_free_consistent:
	pci_free_consistent(ioc->pcidev, hdr.ExtPageLength * 4,
			    buffer, dma_handle);
 out:
	return error;
}

/**
 *	mptsas_get_number_hotspares - returns num hot spares in this ioc
 *	@ioc: Pointer to MPT_ADAPTER structure
 *
 *	Return: number of hotspares
 *
 **/
static int
mptsas_get_number_hotspares(MPT_ADAPTER *ioc)
{
	ConfigPageHeader_t	 hdr;
	CONFIGPARMS		 cfg;
	IOCPage5_t		 *buffer = NULL;
	dma_addr_t		 dma_handle;
	int			 data_sz;
	int			 rc;

	memset(&hdr, 0, sizeof(ConfigPageHeader_t));
	memset(&cfg, 0, sizeof(CONFIGPARMS));

	rc = 0;
	data_sz = 0;
	hdr.PageNumber = 5;
	hdr.PageType = MPI_CONFIG_PAGETYPE_IOC;
	cfg.cfghdr.hdr = &hdr;
	cfg.physAddr = -1;
	cfg.action = MPI_CONFIG_ACTION_PAGE_HEADER;
	cfg.timeout = 10;

	if ((rc = mpt_config(ioc, &cfg)) != 0)
		goto get_ioc_pg5;

	if (hdr.PageLength == 0)
		goto get_ioc_pg5;

	data_sz = hdr.PageLength * 4;
	buffer = (IOCPage5_t *) pci_alloc_consistent(ioc->pcidev,
		data_sz, &dma_handle);
	if (!buffer)
		goto get_ioc_pg5;

	memset((u8 *)buffer, 0, data_sz);
	cfg.physAddr = dma_handle;
	cfg.action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;

	if ((rc = mpt_config(ioc, &cfg)) != 0)
		goto get_ioc_pg5;

	rc = buffer->NumHotSpares;

 get_ioc_pg5:

	if (buffer)
		pci_free_consistent(ioc->pcidev, data_sz,
		    (u8 *) buffer, dma_handle);

	return rc;
}

/**
 *	mptsas_get_ioc_pg5 - ioc Page 5 hot spares
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@pIocPage5: ioc page 5
 *
 *	Return: 0 for success
 *	-ENOMEM if no memory available
 *		-EPERM if not allowed due to ISR context
 *		-EAGAIN if no msg frames currently available
 *		-EFAULT for non-successful reply or no reply (timeout)
 **/
static int
mptsas_get_ioc_pg5(MPT_ADAPTER *ioc, IOCPage5_t *iocPage5)
{
	ConfigPageHeader_t	 hdr;
	CONFIGPARMS		 cfg;
	IOCPage5_t		 *buffer = NULL;
	dma_addr_t		 dma_handle;
	int			 data_sz;
	int			 rc;

	memset(&hdr, 0, sizeof(ConfigPageHeader_t));
	memset(&cfg, 0, sizeof(CONFIGPARMS));

	rc = 0;
	data_sz = 0;
	hdr.PageNumber = 5;
	hdr.PageType = MPI_CONFIG_PAGETYPE_IOC;
	cfg.cfghdr.hdr = &hdr;
	cfg.physAddr = -1;
	cfg.action = MPI_CONFIG_ACTION_PAGE_HEADER;
	cfg.timeout = 10;

	if ((rc = mpt_config(ioc, &cfg)) != 0)
		goto get_ioc_pg5;

	if (hdr.PageLength == 0) {
		rc = -EFAULT;
		goto get_ioc_pg5;
	}

	data_sz = hdr.PageLength * 4;
	buffer = (IOCPage5_t *) pci_alloc_consistent(ioc->pcidev,
		data_sz, &dma_handle);
	if (!buffer) {
		rc = -ENOMEM;
		goto get_ioc_pg5;
	}

	memset((u8 *)buffer, 0, data_sz);
	cfg.physAddr = dma_handle;
	cfg.action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;

	if ((rc = mpt_config(ioc, &cfg)) != 0)
		goto get_ioc_pg5;

	memcpy(iocPage5, buffer, sizeof(*iocPage5));

 get_ioc_pg5:

	if (buffer)
		pci_free_consistent(ioc->pcidev, data_sz,
		    (u8 *) buffer, dma_handle);

	return rc;
}

/**
 * mptsas_add_device_component
 *
 * @ioc
 * @channel - fw mapped id's
 * @id
 * @sas_address
 * @device_info
 *
 **/
static void
mptsas_add_device_component(MPT_ADAPTER *ioc, u8 channel, u8 id,
	u64 sas_address, u32 device_info)
{
	struct sas_device_info	*sas_info, *next;

	down(&ioc->sas_device_info_mutex);

	/*
	 * Delete all matching sas_address's out of tree
	 */
	list_for_each_entry_safe(sas_info, next, &ioc->sas_device_info_list, list) {
		if (sas_info->sas_address != sas_address)
			continue;
		list_del(&sas_info->list);
		kfree(sas_info);
	}

	/*
	 * If there is a matching channel/id, then swap out with new target info
	 */
	list_for_each_entry(sas_info, &ioc->sas_device_info_list, list) {
		if (sas_info->fw.channel == channel && sas_info->fw.id == id)
			goto initialize_data;
	}

	if (!(sas_info = kmalloc(sizeof(*sas_info), GFP_KERNEL)))
		goto out;
	memset(sas_info, 0, sizeof(*sas_info));

	/*
	 * mapping - is for compatibility with drivers supporting sas transport layer
	 */
	sas_info->fw.id = id;
	sas_info->fw.channel = channel;
	sas_info->os.id = id;
	sas_info->os.channel = channel;
	list_add_tail(&sas_info->list, &ioc->sas_device_info_list);

 initialize_data:

	sas_info->sas_address = sas_address;
	sas_info->device_info = device_info;
	sas_info->is_cached = 0;
	sas_info->is_logical_volume = 0;
	devtprintk((KERN_INFO "%s: adding channel=%d id=%d "
	    "sas_address=0x%llX\n", __FUNCTION__, channel, id, sas_address));

 out:
	up(&ioc->sas_device_info_mutex);
	return;
}

/**
 * mptsas_add_device_component_single
 *
 * @ioc
 * @channel
 * @id
 *
 **/
static void
mptsas_add_device_component_single(MPT_ADAPTER *ioc, u8 channel, u8 id)
{
	struct mptsas_devinfo sas_device;
	int rc;
#if defined(__VMKERNEL_MODULE__)
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice	*pTarget;
#endif

	rc = mptsas_sas_device_pg0(ioc, &sas_device,
	    (MPI_SAS_DEVICE_PGAD_FORM_BUS_TARGET_ID <<
	     MPI_SAS_DEVICE_PGAD_FORM_SHIFT),
	    (channel << 8) + id);
	if (rc)
		return;

	mptsas_add_device_component(ioc, sas_device.channel,
	    sas_device.id, sas_device.sas_address, sas_device.device_info);

#if defined(__VMKERNEL_MODULE__)
	pMptTarget = ioc->Target_List[sas_device.channel];
	if(pMptTarget) {
		pTarget = (VirtDevice *)pMptTarget->Target[sas_device.id];
		if (pTarget) {
			if(sas_device.device_info &
					MPI_SAS_DEVICE_INFO_SATA_DEVICE) {
				ioc->sh->max_sectors = 128;
				pTarget->device_type |= MPT_TARGET_DEVICE_TYPE_SATA;
			}
		}
	}
#endif
}

/**
 * mptsas_add_device_component_hotspare
 *
 * Handle adding hotspares into the list
 *
 * @ioc
 *
 **/
static void
mptsas_add_device_component_hotspare(MPT_ADAPTER *ioc)
{
	int		num_hotpares;
	IOCPage5_t 	*iocPage5;
	RaidPhysDiskPage0_t	phys_disk;
	int 		i;

	iocPage5 = NULL;
	num_hotpares = mptsas_get_number_hotspares(ioc);
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
	if (num_hotpares <= 0)
#else
	if (!num_hotpares)
#endif
		goto out;

	iocPage5 = kmalloc(offsetof(IOCPage5_t,HotSpare) +
	    num_hotpares * sizeof(IOC_5_HOT_SPARE), GFP_KERNEL);
	if (!iocPage5)
		goto out;
	memset(iocPage5, 0, sizeof(*iocPage5));
	if (mptsas_get_ioc_pg5(ioc, iocPage5) != 0)
		goto out;
	for(i = 0; i < num_hotpares; i++) {
		mpt_raid_phys_disk_pg0(ioc,
		    iocPage5->HotSpare[i].PhysDiskNum, &phys_disk );
		mptsas_add_device_component_single(ioc,
		    phys_disk.PhysDiskBus, phys_disk.PhysDiskID);
	}
 out:
	kfree(iocPage5);

}

/**
 * mptsas_add_device_component_ir
 *
 * Handle Integrated RAID, adding each individual device to list
 *
 * @ioc
 * @channel
 * @id
 *
 **/
static void
mptsas_add_device_component_ir(MPT_ADAPTER *ioc, u8 channel, u8 id)
{
	CONFIGPARMS			cfg;
	ConfigPageHeader_t		hdr;
	dma_addr_t			dma_handle;
	pRaidVolumePage0_t		buffer = NULL;
	int				i;
	RaidPhysDiskPage0_t 		phys_disk;
	struct sas_device_info		*sas_info;

	memset(&cfg, 0 , sizeof(CONFIGPARMS));
	memset(&hdr, 0 , sizeof(ConfigPageHeader_t));
	hdr.PageType = MPI_CONFIG_PAGETYPE_RAID_VOLUME;
	cfg.pageAddr = (channel << 8) + id;
	cfg.cfghdr.hdr = &hdr;
	cfg.action = MPI_CONFIG_ACTION_PAGE_HEADER;

	if (mpt_config(ioc, &cfg) != 0)
		goto out;

	if (!hdr.PageLength)
		goto out;

	buffer = pci_alloc_consistent(ioc->pcidev, hdr.PageLength * 4,
	    &dma_handle);

	if (!buffer)
		goto out;

	cfg.physAddr = dma_handle;
	cfg.action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;

	if (mpt_config(ioc, &cfg) != 0)
		goto out;

	if (!buffer->NumPhysDisks)
		goto out;

	/*
	 * Adding entry for hidden components
	 */
	for (i = 0; i < buffer->NumPhysDisks; i++) {

		if(mpt_raid_phys_disk_pg0(ioc,
		    buffer->PhysDisk[i].PhysDiskNum, &phys_disk) != 0)
			continue;

		mptsas_add_device_component_single(ioc, phys_disk.PhysDiskBus,
		    phys_disk.PhysDiskID);
	}

	/*
	 * Adding entry for logical volume in list
	 */
	list_for_each_entry(sas_info, &ioc->sas_device_info_list, list) {
		if (sas_info->fw.channel == channel && sas_info->fw.id ==  id)
			goto initialize_data;
	}

	if (!(sas_info = kmalloc(sizeof(*sas_info), GFP_KERNEL)))
		goto out;
	memset(sas_info, 0, sizeof(*sas_info));

	sas_info->fw.id = id;
	sas_info->fw.channel = channel; /* channel zero */
	down(&ioc->sas_device_info_mutex);
	list_add_tail(&sas_info->list, &ioc->sas_device_info_list);
	up(&ioc->sas_device_info_mutex);

 initialize_data:

	sas_info->os.id = id;
	sas_info->os.channel = channel;
	sas_info->sas_address = 0;
	sas_info->device_info = 0;
	sas_info->is_logical_volume = 1;
	sas_info->is_cached = 0;

	devtprintk((KERN_INFO "%s: adding volume at channel=%d id=%d\n",
	    __FUNCTION__, channel, id));

	mptsas_add_device_component_hotspare(ioc);
 out:
	if (buffer)
		pci_free_consistent(ioc->pcidev, hdr.PageLength * 4, buffer,
		    dma_handle);
}


/**
 * mptsas_del_device_component
 *
 * Once a device has been removed, we mark the
 * entry in the list as being cached
 *
 * @ioc
 * @channel - os mapped id's
 * @id
 *
 **/
static void
mptsas_del_device_component(MPT_ADAPTER *ioc, u8 channel, u8 id)
{
	struct sas_device_info	*sas_info, *next;

	/*
	 * Set is_cached flag
	 */
	list_for_each_entry_safe(sas_info, next, &ioc->sas_device_info_list, list) {
		if (sas_info->os.channel == channel && sas_info->os.id == id) {
			sas_info->is_cached = 1;
			devtprintk((KERN_INFO
			    "%s: deleting channel=%d id=%d "
			    "sas_address=0x%llX\n", __FUNCTION__, channel, id,
			    sas_info->sas_address));
		}
	}
}

/**
 * mptsas_del_device_components
 *
 * Cleaning the list
 *
 * @ioc
 *
 **/
static void
mptsas_del_device_components(MPT_ADAPTER *ioc)
{
	struct sas_device_info	*sas_info, *next;

	down(&ioc->sas_device_info_mutex);
	list_for_each_entry_safe(sas_info, next, &ioc->sas_device_info_list, list) {
		list_del(&sas_info->list);
		kfree(sas_info);
	}
	up(&ioc->sas_device_info_mutex);
}

/**
 * mptsas_find_vdevice
 *
 * @ioc
 * @channel
 * @id
 *
 **/
static VirtDevice *
mptsas_find_vdevice(MPT_ADAPTER *ioc, u8 channel, u8 id)
{
	struct _MPT_DEVICE *pMptTarget;

	if (id >= ioc->DevicesPerBus || channel >= ioc->NumberOfBuses)
		return NULL;

	pMptTarget = ioc->Target_List[channel];
	return pMptTarget->Target[id];
}

/**
 * mptscsih_sas_persist_clear_table
 *
 *
 * @ioc
 *
 **/
static void
mptsas_sas_persist_clear_table(void * arg)
{
	MPT_ADAPTER *ioc = (MPT_ADAPTER *)arg;

	mptbase_sas_persist_operation(ioc, MPI_SAS_OP_CLEAR_NOT_PRESENT);
}

/**
 * mptsas_hotplug_print
 *
 *
 * @ioc
 * @hot_plug_info
 * @msg_string
 *
 **/
static void
mptsas_hotplug_print(MPT_ADAPTER *ioc, struct mptsas_hotplug_event *hot_plug_info,  u32 lun, u8 * msg_string)
{
	char *ds;
	u32 	id = hot_plug_info->id;
	u32 	channel = hot_plug_info->channel;

	if ( id >= ioc->DevicesPerBus ) {
		printk(MYIOC_s_WARN_FMT "%s: Invalid id=%d, DevicesPerBus=%d\n",
		    ioc->name, __FUNCTION__, id, ioc->DevicesPerBus);
		return;
	}

	if ( channel >= ioc->NumberOfBuses ) {
		printk(MYIOC_s_WARN_FMT
		    "%s: Invalid channel=%d, NumberOfBuses=%d\n",
		    ioc->name, __FUNCTION__, channel, ioc->NumberOfBuses);
		return;
	}

	if (hot_plug_info->device_info &
	    MPI_SAS_DEVICE_INFO_SSP_TARGET)
		ds = "sas ";
	else if (hot_plug_info->device_info &
	    MPI_SAS_DEVICE_INFO_STP_TARGET)
		ds = "stp ";
	else if (hot_plug_info->device_info &
	    MPI_SAS_DEVICE_INFO_SATA_DEVICE)
		ds = "sata ";
	else
		ds = "";

	printk(MYIOC_s_INFO_FMT
	    "%s %sdevice, channel %d, id %d, lun %d,"
	    "  phy %d\n", ioc->name, msg_string, ds,
	    channel, id, lun,
	    hot_plug_info->phy_id);
}

/*
 * mptsas_remove_target - try to remove a target and all its devices
 *
 * In newer kernels there is scsi_remove_target(), which does
 * the same.
 */
static void
mptsas_remove_target(MPT_ADAPTER *ioc, struct mptsas_hotplug_event *hot_plug_info)
{
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0))
	struct _MPT_DEVICE *pMptTarget;
#endif
	u32 channel, id;

	id = hot_plug_info->id;
	if ( id >= ioc->DevicesPerBus ) {
		printk(MYIOC_s_WARN_FMT "%s: Invalid id=%d, DevicesPerBus=%d\n",
		    ioc->name, __FUNCTION__, id, ioc->DevicesPerBus);
		return;
	}

	channel = hot_plug_info->channel;
	if ( channel >= ioc->NumberOfBuses ) {
		printk(MYIOC_s_WARN_FMT
		    "%s: Invalid channel=%d, NumberOfBuses=%d\n",
		    ioc->name, __FUNCTION__, channel, ioc->NumberOfBuses);
		return;
	}

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0))
	pMptTarget = ioc->Target_List[channel];
	if (pMptTarget->Target[id]) {
		kfree(pMptTarget->Target[id]);
		pMptTarget->Target[id] = NULL;
	}
#endif

	mptsas_hotplug_print(ioc, hot_plug_info, 0, "removing");
}


/*
 * mptsas_scan_target - scan a target id, possibly including all LUNs on the
 *     target.
 *
 * In newer kernels there is scsi_scan_target(), which does the same.
 *
 */
static void
mptsas_scan_target(MPT_ADAPTER *ioc, struct mptsas_hotplug_event *hot_plug_info)
{
	MPT_SCSI_HOST	*hd = (MPT_SCSI_HOST *)ioc->sh->hostdata;
	u32 	channel, id;

	id = hot_plug_info->id;
	channel = hot_plug_info->channel;

	mptscsih_initTarget(hd, channel, id, 0 /*lun*/, NULL, 0);
}

static void
mptsas_device_not_responding(MPT_ADAPTER *ioc, u8 channel, u8 id)
{
	MPT_SCSI_HOST	*hd = (MPT_SCSI_HOST *)ioc->sh->hostdata;
	MPT_FRAME_HDR	*mf;
	SEPRequest_t *SEPMsg;
	static VirtDevice *pTarget;

	pTarget = mptsas_find_vdevice(ioc, channel, id);
	if (!pTarget)
		return;

	pTarget->tflags &= ~MPT_TARGET_FLAGS_TLR_DONE;
	pTarget->tflags |= MPT_TARGET_FLAGS_DELETED;
	if (pTarget->tflags & MPT_TARGET_FLAGS_LED_ON) {
		/* Get a MF for this command.
		*/
		if ((mf = mpt_get_msg_frame(ScsiScanDvCtx, ioc)) == NULL) {
			dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
			    ioc->name,__FUNCTION__));
			} else {
			SEPMsg = (SEPRequest_t *)mf;
			SEPMsg->Function = MPI_FUNCTION_SCSI_ENCLOSURE_PROCESSOR;
			SEPMsg->Bus = channel;
			SEPMsg->TargetID = id;
			SEPMsg->Action = MPI_SEP_REQ_ACTION_WRITE_STATUS;
			SEPMsg->SlotStatus = MPI_SEP_REQ_SLOTSTATUS_UNCONFIGURED;
			pTarget->tflags &= ~MPT_TARGET_FLAGS_LED_ON;
			devtprintk((MYIOC_s_WARN_FMT "Sending SEP UNCONFIGURED cmd bus=%d id=%d\n",
				ioc->name, channel, id));
			mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);
		}
	}

	if (mptscsih_TMHandler(hd, MPI_SCSITASKMGMT_TASKTYPE_TARGET_RESET,
	    channel, id, 0, 0, NO_SLEEP) < 0) {
	  /* The TM request failed!
		 */
		printk(MYIOC_s_WARN_FMT "Error processing target reset TaskMgmt request\n",
			hd->ioc->name);
		ioc->tmState = TM_STATE_NONE;
	}
}

static void
mptsas_smart_fill_log(MPT_ADAPTER *ioc, EVENT_DATA_SAS_DEVICE_STATUS_CHANGE *sas_event_data)
{
	int idx;

	idx = ioc->eventContext % MPTCTL_EVENT_LOG_SIZE;
	ioc->events[idx].event = MPI_EVENT_SAS_DEVICE_STATUS_CHANGE;
//	ioc->events[idx].eventContext =
//	     le32_to_cpu(pEvReply->EventContext);
	ioc->events[idx].data[0] =
	    (MPI_EVENT_SAS_DEV_STAT_RC_SMART_DATA << 16) ||
	    (sas_event_data->Bus << 8) || sas_event_data->TargetID;
	ioc->events[idx].data[1] =
	    (sas_event_data->ASCQ << 8) || sas_event_data->ASC;
	ioc->eventContext++;
}

/**
 * mptsas_initialize_work
 *
 *
 * @ioc
 *
 **/
static void
mptsas_initialize_work(void *arg)
{
	MPT_ADAPTER *ioc = (MPT_ADAPTER *)arg;
	struct mptsas_portinfo	port_info;
	int ii;

	if (!mptsas_sas_io_unit_pg0(ioc, &port_info))
		ioc->num_ports = port_info.num_phys;

	/*
	 * Handling Inactive Volumes
	 */
	if (!ioc->ir_firmware ||
	    !ioc->raid_data.pIocPg2 ||
	    !ioc->raid_data.pIocPg2->NumActiveVolumes)
	return;

	for (ii = 0; ii < ioc->raid_data.pIocPg2->NumActiveVolumes; ii++)
		mptsas_add_device_component_ir(ioc,
		    ioc->raid_data.pIocPg2->RaidVolume[ii].VolumeBus,
		    ioc->raid_data.pIocPg2->RaidVolume[ii].VolumeID);
}

/**
 * mptsas_initialize
 *
 *
 * @ioc
 *
 **/
static void
mptsas_initialize(MPT_ADAPTER *ioc)
{
	MPT_INIT_WORK(&ioc->mptsas_initialize_work,
	    mptsas_initialize_work,(void *)ioc);
	SCHEDULE_TASK(&ioc->mptsas_initialize_work);
}

/**
 * mptsas_hotplug_work
 *
 *
 * @hot_plug_info
 *
 **/
static void
mptsas_hotplug_work(void *arg)
{
	struct mptsas_hotplug_event *hot_plug_info = arg;
	MPT_ADAPTER 		*ioc = hot_plug_info->ioc;
	u32 			id = hot_plug_info->id;
	u32 			channel = hot_plug_info->channel;

	dhotpprintk((MYIOC_s_WARN_FMT "Entering %s for channel=%d id=%d\n",
		ioc->name,__FUNCTION__, channel, id));

	down(&ioc->hot_plug_semaphore);

	/* If there has been a change to raid, then we need to
	 * refresh the config raid data
	 */
	if (hot_plug_info->refresh_raid_config_pages)
		mpt_findImVolumes(ioc);

	switch  (hot_plug_info->event_type) {
	case MPTSAS_DEL_DEVICE:
		mptsas_del_device_component(ioc, channel, id);
		if (hot_plug_info->refresh_raid_config_pages)
			mptsas_add_device_component_hotspare(ioc);
		dhotpprintk((MYIOC_s_WARN_FMT
		    "MPTSAS_DEL_DEVICE: channel=%d id=%d\n",
		    ioc->name, channel, id));
		mptsas_remove_target(ioc, hot_plug_info);
		break;

	case MPTSAS_ADD_DEVICE:
		if (ioc->raid_data.isRaid & (1 << id))
			mptsas_add_device_component_ir(ioc, channel, id);
		else
			mptsas_add_device_component_single(ioc, channel, id);
		dhotpprintk((MYIOC_s_WARN_FMT
		    "MPTSAS_ADD_DEVICE: channel=%d id=%d\n",
		    ioc->name, channel, id));
		mptsas_scan_target(ioc, hot_plug_info);
#if defined(__VMKERNEL_MODULE__)
		vmk_scsi_state_change(ioc->sh);
#endif
		break;
	case MPTSAS_ADD_INACTIVE_VOLUME:
		dhotpprintk((MYIOC_s_WARN_FMT
		    "MPTSAS_ADD_INACTIVE_VOLUME: channel=%d id=%d\n",
		    ioc->name, channel, id));
		mptsas_add_device_component_ir(ioc, channel, id);
		break;
	case MPTSAS_PHYSDISK_ADD:
		mptsas_add_device_component_single(ioc, channel, id);
		break;
	case MPTSAS_INIT_TARGET:
		if (ioc->raid_data.isRaid & (1 << id))
			mptsas_add_device_component_ir(ioc, channel, id);
		else
			mptsas_add_device_component_single(ioc, channel, id);
		break;
	default:
		dhotpprintk((MYIOC_s_WARN_FMT
		    "Unknown hot_plug event_type=%x: channel=%d id=%d "
		    " skipping\n", ioc->name, hot_plug_info->event_type,
		    channel, id));
		goto out;
	}

 out:
	dhotpprintk((MYIOC_s_WARN_FMT "%s: kfree hot_plug_info=%p\n",
	    ioc->name,__FUNCTION__, hot_plug_info));
	kfree(hot_plug_info);
	up(&ioc->hot_plug_semaphore);
}

/**
 * mptsas_send_sas_event
 *
 *
 * @ioc
 * @sas_event_data
 *
 **/
static void
mptsas_send_sas_event(MPT_ADAPTER *ioc,
		EVENT_DATA_SAS_DEVICE_STATUS_CHANGE *sas_event_data)
{
	struct mptsas_hotplug_event *ev;
	u32 device_info = le32_to_cpu(sas_event_data->DeviceInfo);
	u64 sas_address;

	if ((device_info &
	     (MPI_SAS_DEVICE_INFO_SSP_TARGET |
	      MPI_SAS_DEVICE_INFO_STP_TARGET |
	      MPI_SAS_DEVICE_INFO_SATA_DEVICE )) == 0)
		return;

	if (sas_event_data->ReasonCode ==
		    MPI_EVENT_SAS_DEV_STAT_RC_NO_PERSIST_ADDED) {
		MPT_INIT_WORK(&ioc->mptsas_persistTask,
		    mptsas_sas_persist_clear_table,(void *)ioc);
		SCHEDULE_TASK(&ioc->mptsas_persistTask);
		return;
	}

	switch (sas_event_data->ReasonCode) {
/*	case MPI_EVENT_SAS_DEV_STAT_RC_NOT_RESPONDING:
		mptsas_device_not_responding(ioc, sas_event_data->Bus,
		    sas_event_data->TargetID); */

	case MPI_EVENT_SAS_DEV_STAT_RC_ADDED:
		ev = kmalloc(sizeof(*ev), GFP_ATOMIC);
		if (!ev) {
			printk(KERN_WARNING "mptsas: lost hotplug event\n");
			break;
		}

		memset(ev, 0, sizeof(*ev));
		ev->ioc = ioc;
		ev->handle = le16_to_cpu(sas_event_data->DevHandle);
		ev->parent_handle =
		    le16_to_cpu(sas_event_data->ParentDevHandle);
		ev->channel = sas_event_data->Bus;
		ev->id = sas_event_data->TargetID;
		ev->phy_id = sas_event_data->PhyNum;
		memcpy(&sas_address, &sas_event_data->SASAddress,
		    sizeof(u64));
		ev->sas_address = le64_to_cpu(sas_address);
		ev->device_info = device_info;

		if (sas_event_data->ReasonCode &
		    MPI_EVENT_SAS_DEV_STAT_RC_ADDED)
			ev->event_type = MPTSAS_ADD_DEVICE;
		else
			ev->event_type = MPTSAS_DEL_DEVICE;
		MPT_INIT_WORK(&ev->work, mptsas_hotplug_work, ev);
		SCHEDULE_TASK(&ev->work);
		break;

	case MPI_EVENT_SAS_DEV_STAT_RC_NO_PERSIST_ADDED:
	/*
	 * Persistent table is full.
	 */
		MPT_INIT_WORK(&ioc->mptsas_persistTask,
		    mptsas_sas_persist_clear_table,(void *)ioc);
		SCHEDULE_TASK(&ioc->mptsas_persistTask);
		break;

	case MPI_EVENT_SAS_DEV_STAT_RC_SMART_DATA:
		mptsas_smart_fill_log(ioc, sas_event_data);
		break;

	case MPI_EVENT_SAS_DEV_STAT_RC_INTERNAL_DEVICE_RESET:
	/* TODO */
	default:
		break;
	}
}

/**
 * mptsas_send_raid_event
 *
 *
 * @ioc
 * @raid_event_data
 *
 **/
static void
mptsas_send_raid_event(MPT_ADAPTER *ioc,
		EVENT_DATA_RAID *raid_event_data)
{
	struct mptsas_hotplug_event *ev;
	int status = le32_to_cpu(raid_event_data->SettingsStatus);
	int state = (status >> 8) & 0xff;

	if (ioc->bus_type != SAS)
		return;

	ev = kmalloc(sizeof(*ev), GFP_ATOMIC);
	if (!ev) {
		printk(KERN_WARNING "mptsas: lost hotplug event\n");
		return;
	}

	memset(ev,0,sizeof(struct mptsas_hotplug_event));
	ev->ioc = ioc;
	ev->id = raid_event_data->VolumeID;
	ev->channel = raid_event_data->VolumeBus;
	ev->refresh_raid_config_pages = 1;

	devtprintk((KERN_INFO MYNAM ": VolumeID=%d Reason=%x received\n",
	    ev->id, raid_event_data->ReasonCode));
	switch (raid_event_data->ReasonCode) {
	case MPI_EVENT_RAID_RC_PHYSDISK_DELETED:
		ev->event_type = MPTSAS_ADD_DEVICE;
		break;
	case MPI_EVENT_RAID_RC_PHYSDISK_CREATED:
		ev->event_type = MPTSAS_DEL_DEVICE;
		break;
	case MPI_EVENT_RAID_RC_PHYSDISK_STATUS_CHANGED:
		switch (state) {
		case MPI_PD_STATE_ONLINE:
		case MPI_PD_STATE_NOT_COMPATIBLE:
			ev->event_type = MPTSAS_PHYSDISK_ADD;
			break;
		case MPI_PD_STATE_MISSING:
		case MPI_PD_STATE_OFFLINE_AT_HOST_REQUEST:
		case MPI_PD_STATE_FAILED_AT_HOST_REQUEST:
		case MPI_PD_STATE_OFFLINE_FOR_ANOTHER_REASON:
			ev->event_type = MPTSAS_DEL_DEVICE;
			break;
		default:
			devtprintk((KERN_INFO MYNAM
			    ": ignoring this event! %d\n", __LINE__));
			return;
		}
		break;
	case MPI_EVENT_RAID_RC_VOLUME_DELETED:
		ev->event_type = MPTSAS_DEL_DEVICE;
		break;
	case MPI_EVENT_RAID_RC_VOLUME_CREATED:
		ev->skip_report_luns = 1;
		ev->event_type = MPTSAS_ADD_DEVICE;
		break;
	case MPI_EVENT_RAID_RC_VOLUME_STATUS_CHANGED:
		switch (state) {
/*		case MPI_RAIDVOL0_STATUS_STATE_FAILED:
		case MPI_RAIDVOL0_STATUS_STATE_MISSING:
			ev->event_type = MPTSAS_DEL_DEVICE;
			break; */
		case MPI_RAIDVOL0_STATUS_STATE_OPTIMAL:
		case MPI_RAIDVOL0_STATUS_STATE_DEGRADED:
			ev->skip_report_luns = 1;
			ev->event_type = MPTSAS_ADD_DEVICE;
			break;
		default:
			devtprintk((KERN_INFO MYNAM
			    ": ignoring this event! %d\n", __LINE__));
			return;
		}
		break;
	default:
		devtprintk((KERN_INFO MYNAM
		    ": ignoring this event! %d\n", __LINE__));
		return;
	}
	MPT_INIT_WORK(&ev->work, mptsas_hotplug_work, ev);
	SCHEDULE_TASK(&ev->work);
}

/*
 * mptsas_send_ir2_event
 *
 * This handle exposing hidden disk when an inactive raid volume is added
 */
static void
mptsas_send_ir2_event(MPT_ADAPTER *ioc, PTR_MPI_EVENT_DATA_IR2 ir2_data)
{
	struct mptsas_hotplug_event *ev;

	if (ir2_data->ReasonCode !=
	    MPI_EVENT_IR2_RC_FOREIGN_CFG_DETECTED)
		return;

	ev = kmalloc(sizeof(*ev), GFP_ATOMIC);
	if (!ev)
		return;
	memset(ev, 0, sizeof(*ev));
	ev->ioc = ioc;
	ev->id = ir2_data->TargetID;
	ev->channel = ir2_data->Bus;
	ev->refresh_raid_config_pages = 1;
	ev->event_type = MPTSAS_ADD_INACTIVE_VOLUME;
	MPT_INIT_WORK(&ev->work, mptsas_hotplug_work, ev);
	SCHEDULE_TASK(&ev->work);
};

/**
 * mptsas_event_process
 *
 *
 * @ioc
 * @reply
 *
 **/
static int
mptsas_event_process(MPT_ADAPTER *ioc, EventNotificationReply_t *reply)
{
	int rc=1;
	u8 event = le32_to_cpu(reply->Event) & 0xFF;

	dhotpprintk((MYIOC_s_WARN_FMT "Entering %s\n",
		    ioc->name,__FUNCTION__));

	if (!ioc->sh)
		goto out;

	switch (event) {
	case MPI_EVENT_SAS_DEVICE_STATUS_CHANGE:
		ioc->csmi_change_count++;
		mptsas_send_sas_event(ioc,
			(EVENT_DATA_SAS_DEVICE_STATUS_CHANGE *)reply->Data);
		break;
	case MPI_EVENT_INTEGRATED_RAID:
		ioc->csmi_change_count++;
		mptsas_send_raid_event(ioc, (EVENT_DATA_RAID *)reply->Data);
		break;
	case MPI_EVENT_IR2:
		ioc->csmi_change_count++;
		mptsas_send_ir2_event(ioc,
		    (PTR_MPI_EVENT_DATA_IR2)reply->Data);
		break;
	case MPI_EVENT_PERSISTENT_TABLE_FULL:
		MPT_INIT_WORK(&ioc->mptsas_persistTask,
		    mptsas_sas_persist_clear_table,(void *)ioc);
		SCHEDULE_TASK(&ioc->mptsas_persistTask);
		break;
	}
 out:

	return rc;
}

