/*
 *  linux/drivers/message/fusion/mptscsih.c
 *      For use with LSI Logic PCI chip/adapter(s)
 *      running LSI Logic Fusion MPT (Message Passing Technology) firmware.
 *
 *  Copyright (c) 1999-2007 LSI Logic Corporation
 *  (mailto:mpt_linux_developer@lsil.com)
 *  For support, send a description of issues to: support@lsil.com.
 *
 *  $Id: mptscsih.c,v 1.1.2.4 2003/05/07 14:08:34 Exp $
 */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    NO WARRANTY
    THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR
    CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT
    LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,
    MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is
    solely responsible for determining the appropriateness of using and
    distributing the Program and assumes all risks associated with its
    exercise of rights under this Agreement, including but not limited to
    the risks and costs of program errors, damage to or loss of data,
    programs or equipment, and unavailability or interruption of operations.

    DISCLAIMER OF LIABILITY
    NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY
    DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND
    ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
    USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED
    HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/kdev_t.h>
#include <linux/blkdev.h>
#include <linux/blk.h>		/* for io_request_lock (spinlock) decl */
#include <linux/delay.h>	/* for mdelay */
#include <linux/interrupt.h>	/* needed for in_interrupt() proto */
#include <linux/reboot.h>	/* notifier code */
#include "scsi.h"
#include "hosts.h"
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,32)
#if defined(CONFIG_DISKDUMP) || defined(CONFIG_DISKDUMP_MODULE)
#include "scsi_dump.h"
#endif
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,45)
#include "sd.h"
#endif

#include "linux_compat.h"	/* linux-2.2.x (vs. -2.4.x) tweaks */
#include "mptbase.h"
#include "mptscsih.h"
#include "mptsas.h"
#ifdef __VMKERNEL_MODULE__
#include "vmklinux_dist.h"
#include "scsi_transport_sas.h"
#endif
#if (defined(CONFIG_VMNIX) || defined(__VMKERNEL_MODULE__))
#define MPT_GET_PCIDEV(x) ((struct pci_dev *) x->pcidev)
#endif


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
#define my_NAME		"Fusion MPT SCSI Host driver"
#define my_VERSION	MPT_LINUX_VERSION_COMMON
#define MYNAM		"mptscsih"

MODULE_AUTHOR(MODULEAUTHOR);
MODULE_DESCRIPTION(my_NAME);
MODULE_LICENSE("GPL");

extern int mptbase_can_queue;
extern int mpt_ScsiDoneCtx;

/* Set string for command line args from insmod */
char *mptscsih = NULL;
static int mpt_dv = 1;
MODULE_PARM(mpt_dv, "i");
MODULE_PARM_DESC(mpt_dv, " DV Algorithm: enhanced = 1, basic = 0 (default=1)");
static int mpt_width = 1;
MODULE_PARM(mpt_width, "i");
MODULE_PARM_DESC(mpt_width, " Max Bus Width: wide = 1, narrow = 0 (default=1)");
static int mpt_factor = 0x08;
MODULE_PARM(mpt_factor, "h");
MODULE_PARM_DESC(mpt_factor, " Sync Factor (default=0x08)");
static int mpt_saf_te = 0;
MODULE_PARM(mpt_saf_te, "i");
MODULE_PARM_DESC(mpt_saf_te, " Force enabling SEP Processor: enable=1 (default=0)");
static int mpt_pt_clear = 0;
MODULE_PARM(mpt_pt_clear, "i");
MODULE_PARM_DESC(mpt_pt_clear, " Clear persistency table: enable=1 (default=0)");
static int mpt_pq_filter = 0;
MODULE_PARM(mpt_pq_filter, "i");
MODULE_PARM_DESC(mpt_pq_filter, " Enable peripheral qualifier filter: enable=1 (default=0)");

static int mpt_qas_disable = 0;
MODULE_PARM(mpt_qas_disable, "i");
MODULE_PARM_DESC(mpt_qas_disable, " Disable QAS Support (default=0)");

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
#ifdef MY_PANIC_PROCESSOR_LOOP
char 		panic_msg[30];
#endif

typedef struct _BIG_SENSE_BUF {
	u8		data[MPT_SENSE_BUFFER_ALLOC];
} BIG_SENSE_BUF;

#define OEM_TLR_COMMAND			0xC2
#define MPT_SCANDV_GOOD			(0x00000000) /* must be 0 */
#define MPT_SCANDV_DID_RESET		(0x00000001)
#define MPT_SCANDV_SENSE		(0x00000002)
#define MPT_SCANDV_SOME_ERROR		(0x00000004)
#define MPT_SCANDV_SELECTION_TIMEOUT	(0x00000008)
#define MPT_SCANDV_ISSUE_SENSE		(0x00000010)
#define MPT_SCANDV_FALLBACK		(0x00000020)

#define MPT_SCANDV_MAX_RETRIES		(10)

#define MPT_ICFLAG_BUF_CAP	0x01	/* ReadBuffer Read Capacity format */
#define MPT_ICFLAG_ECHO		0x02	/* ReadBuffer Echo buffer format */
#define MPT_ICFLAG_EBOS		0x04	/* ReadBuffer Echo buffer has EBOS */
#define MPT_ICFLAG_PHYS_DISK	0x08	/* Any SCSI IO but do Phys Disk Format */
#define MPT_ICFLAG_TAGGED_CMD	0x10	/* Do tagged IO */
#define MPT_ICFLAG_DID_RESET	0x20	/* Bus Reset occurred with this command */
#define MPT_ICFLAG_RESERVED	0x40	/* Reserved has been issued */

typedef struct _internal_cmd {
	char		*data;		/* data pointer */
	dma_addr_t	data_dma;	/* data dma address */
	int		size;		/* transfer size */
	u8		cmd;		/* SCSI Op Code */
	u8		bus;		/* bus number */
	u8		id;		/* SCSI ID (virtual) */
	u8		lun;
	u8		flags;		/* Bit Field - See above */
	u8		physDiskNum;	/* Phys disk number, -1 else */
	u8		rsvd2;
	u8		rsvd;
} INTERNAL_CMD;

typedef struct _negoparms {
	u8 width;
	u8 offset;
	u8 factor;
	u8 flags;
} NEGOPARMS;

typedef struct _dv_parameters {
	NEGOPARMS	 max;
	NEGOPARMS	 now;
	u8		 cmd;
	u8		 id;
	u16		 pad1;
} DVPARAMETERS;


/*
 *  Other private/forward protos...
 */
#ifdef MPT_DEBUG_QCMD_DEPTH
static void mptscsih_scsi_done(MPT_ADAPTER *ioc, Scsi_Cmnd *SCpnt);
#endif
static int	mptscsih_io_done(MPT_ADAPTER *ioc, MPT_FRAME_HDR *mf, MPT_FRAME_HDR *r);
static void	mptscsih_report_queue_full(Scsi_Cmnd *sc, SCSIIOReply_t *pScsiReply, SCSIIORequest_t *pScsiReq);
static int	mptscsih_taskmgmt_complete(MPT_ADAPTER *ioc, MPT_FRAME_HDR *mf, MPT_FRAME_HDR *r);

static int	mptscsih_AddSGE(MPT_ADAPTER *ioc, Scsi_Cmnd *SCpnt,
				 SCSIIORequest_t *pReq, int req_idx);
static void	mptscsih_freeChainBuffers(MPT_ADAPTER *ioc, int req_idx);
static void	copy_sense_data(Scsi_Cmnd *sc, MPT_SCSI_HOST *hd, MPT_FRAME_HDR *mf, SCSIIOReply_t *pScsiReply);
static MPT_FRAME_HDR *mptscsih_search_pendingQ(MPT_ADAPTER *ioc, Scsi_Cmnd * sc);

static int	mptscsih_TMHandler(MPT_SCSI_HOST *hd, u8 type, u8 bus, u8 id, u8 lun, int ctx2abort, int sleepFlag);

static int	mptscsih_ioc_reset(MPT_ADAPTER *ioc, int post_reset);
static int	mptscsih_event_process(MPT_ADAPTER *ioc, EventNotificationReply_t *pEvReply);
static void	mptscsih_initTarget(MPT_SCSI_HOST *hd, int bus, int id, u8 lun, char *data, int dlen);
static void	mptscsih_setTargetNegoParms(MPT_SCSI_HOST *hd, VirtDevice *pTarget, char byte56);
static void	mptscsih_setDevicePage1Flags (u8 width, u8 factor, u8 offset, int *requestedPtr, int *configurationPtr, u8 flags);
static void	mptscsih_no_negotiate(MPT_SCSI_HOST *hd, int id);
static int	mptscsih_writeSDP1(MPT_SCSI_HOST *hd, int portnum, int id, int flags);
static int	mptscsih_writeIOCPage4(MPT_ADAPTER *ioc, int id, int bus);
static int	mptscsih_readFCDevicePage0(MPT_SCSI_HOST *hd, int id);
static int	mptscsih_writeFCPortPage3(MPT_SCSI_HOST *hd, int id);
static int	mptscsih_sendIOCInit(MPT_SCSI_HOST *hd);
static int	mptscsih_scandv_complete(MPT_ADAPTER *ioc, MPT_FRAME_HDR *mf, MPT_FRAME_HDR *r);
static void	mptscsih_timer_expired(unsigned long data);
static void	mptscsih_taskmgmt_timeout(unsigned long data);
static int	mptscsih_do_cmd(MPT_SCSI_HOST *hd, INTERNAL_CMD *iocmd);
static int	mptscsih_synchronize_cache(MPT_SCSI_HOST *hd, int portnum);

#ifdef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
static int	mptscsih_do_raid(MPT_SCSI_HOST *hd, u8 action, INTERNAL_CMD *io);
static void	mptscsih_domainValidation(void *arg);
static int	mptscsih_is_phys_disk(MPT_ADAPTER *ioc, int channel, int id);
static void	mptscsih_qas_check(MPT_SCSI_HOST *hd, int id);
static int	mptscsih_doDv(MPT_SCSI_HOST *hd, int bus, int id);
static void	mptscsih_dv_parms(MPT_SCSI_HOST *hd, DVPARAMETERS *dv,void *pPage);
static void	mptscsih_fillbuf(char *buffer, int size, int index, int width);
#endif
static int	mptscsih_halt(struct notifier_block *nb, ulong event, void *buf);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,32)
#if defined(CONFIG_DISKDUMP) || defined(CONFIG_DISKDUMP_MODULE)
static int	mptscsih_sanity_check(Scsi_Device *SDev);
static int	mptscsih_quiesce(Scsi_Device *SDev);
static void	mptscsih_poll(Scsi_Device *SDev);
#endif
#endif
static void
mpt_IssueTLR(void *arg);

/*
 *	Reboot Notification
 */
static struct notifier_block mptscsih_notifier = {
	mptscsih_halt, NULL, 0
};

/*
 *	Private data...
 */

static int	mpt_scsi_hosts = 0;

static int	ScsiTaskCtx = -1;
static int	ScsiScanDvCtx = -1; /* Used only for bus scan and dv */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,28)
static struct proc_dir_entry proc_mpt_scsihost =
{
	.low_ino =	PROC_SCSI_MPT,
	.namelen =	8,
	.name =		"mptscsih",
	.mode =		S_IFDIR | S_IRUGO | S_IXUGO,
	.nlink =	2,
};
#endif

#define SNS_LEN(scp)	sizeof((scp)->sense_buffer)

#ifdef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
/*
 * Domain Validation task structure
 */
static spinlock_t dvtaskQ_lock = SPIN_LOCK_UNLOCKED;
static int dvtaskQ_active = 0;
#endif

/*
 * Wait Queue setup
 */
static DECLARE_WAIT_QUEUE_HEAD (scandv_waitq);
static int scandv_wait_done = 1;
#ifdef __VMKERNEL_MODULE__

static void
mpt2xx_get_target_sas_address(struct scsi_target *starget);

static struct sas_function_template mpt2xx_transport_functions = {
        .get_starget_sas_address = mpt2xx_get_target_sas_address,
};
#endif

#include "mptsas.c"
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *  Private inline routines...
 */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*  Return absolute SCSI data direction:
 *     1 = _DATA_OUT
 *     0 = _DIR_NONE
 *    -1 = _DATA_IN
 *
 * Changed to use the default data
 * direction and the defines set up in the
 * 2.4 kernel series
 *     1 = _DATA_OUT	changed to SCSI_DATA_WRITE (1)
 *     0 = _DIR_NONE	changed to SCSI_DATA_NONE (3)
 *    -1 = _DATA_IN	changed to SCSI_DATA_READ (2)
 * If the direction is unknown, fall through to original code.
 *
 * Mid-layer bug fix(): sg interface generates the wrong data
 * direction in some cases. Set the direction the hard way for
 * the most common commands.
 */
static inline int
mptscsih_io_direction(Scsi_Cmnd *cmd)
{
	switch (cmd->cmnd[0]) {
	case WRITE_6:
	case WRITE_10:
	case 0x8A:		/* WRITE_16 */
		return SCSI_DATA_WRITE;
		break;
	case READ_6:
	case READ_10:
	case 0x88:		/* READ_16 */
		return SCSI_DATA_READ;
		break;
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	if (cmd->sc_data_direction != SCSI_DATA_UNKNOWN)
		return cmd->sc_data_direction;
#endif
	switch (cmd->cmnd[0]) {
	/*  _DATA_OUT commands	*/
	case WRITE_6:		case WRITE_10:		case WRITE_12:
	case WRITE_LONG:	case WRITE_SAME:	case WRITE_BUFFER:
	case WRITE_VERIFY:	case WRITE_VERIFY_12:
	case COMPARE:		case COPY:		case COPY_VERIFY:
	case SEARCH_EQUAL:	case SEARCH_HIGH:	case SEARCH_LOW:
	case SEARCH_EQUAL_12:	case SEARCH_HIGH_12:	case SEARCH_LOW_12:
	case MODE_SELECT:	case MODE_SELECT_10:	case LOG_SELECT:
	case SEND_DIAGNOSTIC:	case CHANGE_DEFINITION: case UPDATE_BLOCK:
	case SET_WINDOW:	case MEDIUM_SCAN:	case SEND_VOLUME_TAG:
	case REASSIGN_BLOCKS:
	case PERSISTENT_RESERVE_OUT:
	case 0xea:
	case 0xa3:
		return SCSI_DATA_WRITE;

	/*  No data transfer commands  */
	case SEEK_6:		case SEEK_10:
	case RESERVE:		case RELEASE:
	case TEST_UNIT_READY:
	case START_STOP:
	case ALLOW_MEDIUM_REMOVAL:
		return SCSI_DATA_NONE;

	/*  Conditional data transfer commands	*/
	case FORMAT_UNIT:
		if (cmd->cmnd[1] & 0x10)	/* FmtData (data out phase)? */
			return SCSI_DATA_WRITE;
		else
			return SCSI_DATA_NONE;

	case VERIFY:
		if (cmd->cmnd[1] & 0x02)	/* VERIFY:BYTCHK (data out phase)? */
			return SCSI_DATA_WRITE;
		else
			return SCSI_DATA_NONE;

	case RESERVE_10:
		if (cmd->cmnd[1] & 0x03)	/* RESERVE:{LongID|Extent} (data out phase)? */
			return SCSI_DATA_WRITE;
		else
			return SCSI_DATA_NONE;

	/*  Must be data _IN!  */
	default:
		return SCSI_DATA_READ;
	}
} /* mptscsih_io_direction() */

#ifdef MY_PANIC_PROCESSOR_LOOP
static void my_panic(const char * fmt, ...)
{
	static char buf[1024];
	va_list args;

	va_start(args, fmt);
	vsnprintf(buf, sizeof(buf), fmt, args);
	va_end(args);

	printk(KERN_EMERG "LSI Kernel panic: %s\n",buf);
//	smp_send_stop();

	for (;;)
		;
}
#endif

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_add_chain - Place a 32 bit chain SGE at address pAddr.
 *	@pAddr: virtual address for SGE
 *	@next: nextChainOffset value (u32's)
 *	@length: length of next SGL segment
 *	@dma_addr: Physical address
 *
 */
static inline void
mptscsih_add_chain(char *pAddr, u8 next, u16 length, dma_addr_t dma_addr)
{
		SGEChain32_t *pChain = (SGEChain32_t *) pAddr;
		pChain->Length = cpu_to_le16(length);
		pChain->Flags = MPI_SGE_FLAGS_CHAIN_ELEMENT;
		pChain->NextChainOffset = next;
		pChain->Address = cpu_to_le32(dma_addr);
}

/**
 *	mptscsih_add_chain_64_bit - Place a 64 bit chain SGE at address pAddr.
 *	@pAddr: virtual address for SGE
 *	@next: nextChainOffset value (u64's)
 *	@length: length of next SGL segment
 *	@dma_addr: Physical address
 *
 */
static inline void
mptscsih_add_chain_64_bit(char *pAddr, u8 next, u16 length, dma_addr_t dma_addr)
{
		SGEChain64_t *pChain = (SGEChain64_t *) pAddr;
		u32 tmp = dma_addr & 0xFFFFFFFF;

		pChain->Length = cpu_to_le16(length);
		pChain->Flags = (MPI_SGE_FLAGS_CHAIN_ELEMENT | 
				 MPI_SGE_FLAGS_64_BIT_ADDRESSING);

		pChain->NextChainOffset = next;

		pChain->Address.Low = cpu_to_le32(tmp);
		tmp = (u32) ((u64)dma_addr >> 32);
		pChain->Address.High = cpu_to_le32(tmp);
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_getFreeChainBuffer - Function to get a free chain
 *	from the MPT_SCSI_HOST FreeChainQ.
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@req_idx: Index of the SCSI IO request frame. (output)
 *
 *	return SUCCESS or FAILED
 */
static inline int
mptscsih_getFreeChainBuffer(MPT_ADAPTER *ioc, int *retIndex)
{
	MPT_FRAME_HDR *chainBuf;
	unsigned long flags;
	int rc;
	int chain_idx;

	dsgprintk((MYIOC_s_WARN_FMT "getFreeChainBuffer called\n",
			ioc->name));
	SPIN_LOCK_IRQSAVE(&ioc->FreeQlock, flags);
	if (!Q_IS_EMPTY(&ioc->FreeChainQ)) {
		int offset;

		chainBuf = ioc->FreeChainQ.head;
		Q_DEL_ITEM(&chainBuf->u.frame.linkage);
		offset = (u8 *)chainBuf - (u8 *)ioc->ChainBuffer;
		chain_idx = offset / ioc->req_sz;
		rc = SUCCESS;
		dsgprintk((MYIOC_s_ERR_FMT "getFreeChainBuffer chainBuf=%p ChainBuffer=%p offset=%d chain_idx=%d\n",
			ioc->name, chainBuf, ioc->ChainBuffer, offset, chain_idx));
	} else {
		rc = FAILED;
		chain_idx = MPT_HOST_NO_CHAIN;
		dfailprintk((MYIOC_s_WARN_FMT "getFreeChainBuffer failed\n",
			ioc->name));
	}
	SPIN_UNLOCK_IRQRESTORE(&ioc->FreeQlock, flags);

	*retIndex = chain_idx;

	return rc;
} /* mptscsih_getFreeChainBuffer() */

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_AddSGE - Add a SGE (plus chain buffers) to the
 *	SCSIIORequest_t Message Frame.
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@SCpnt: Pointer to Scsi_Cmnd structure
 *	@pReq: Pointer to SCSIIORequest_t structure
 *
 *	Returns ...
 */
static int
mptscsih_AddSGE(MPT_ADAPTER *ioc, Scsi_Cmnd *SCpnt,
		SCSIIORequest_t *pReq, int req_idx)
{
	char 	*psge;
	char	*chainSge;
	struct scatterlist *sg;
	int	 frm_sz;
	int	 sges_left, sg_done;
	int	 chain_idx = MPT_HOST_NO_CHAIN;
	int	 sgeOffset;
	int	 numSgeSlots, numSgeThisFrame;
	u32	 sgflags, sgdir, thisxfer = 0;
	int	 chain_dma_off = 0;
	int	 newIndex;
	int	 ii;
	dma_addr_t v2;
	u32	RequestNB;

	sgdir = le32_to_cpu(pReq->Control) & MPI_SCSIIO_CONTROL_DATADIRECTION_MASK;
	if (sgdir == MPI_SCSIIO_CONTROL_WRITE)  {
		sgdir = MPT_TRANSFER_HOST_TO_IOC;
	} else {
		sgdir = MPT_TRANSFER_IOC_TO_HOST;
	}

	psge = (char *) &pReq->SGL;
	frm_sz = ioc->req_sz;

	/* Map the data portion, if any.
	 * sges_left  = 0 if no data transfer.
	 */
	if ( (sges_left = SCpnt->use_sg) ) {
#ifndef __VMKERNEL_MODULE__
		sges_left = pci_map_sg(ioc->pcidev,
			       (struct scatterlist *) SCpnt->request_buffer,
 			       SCpnt->use_sg,
			       SCpnt->sc_data_direction);
#endif  /*! __VMKERNEL_MODULE__ */
		if (sges_left == 0)
			return FAILED;
	} else if (SCpnt->request_bufflen) {
#ifdef __VMKERNEL_MODULE__
                SCpnt->SCp.dma_handle = SCpnt->request_bufferMA;
#else
		SCpnt->SCp.dma_handle = pci_map_single(ioc->pcidev,
				      SCpnt->request_buffer,
				      SCpnt->request_bufflen,
				      SCpnt->sc_data_direction);
#endif
		dsgprintk((MYIOC_s_WARN_FMT "SG: non-SG for %p, len=%d\n",
				ioc->name, SCpnt, SCpnt->request_bufflen));
		ioc->add_sge((char *) &pReq->SGL,
			0xD1000000|sgdir|SCpnt->request_bufflen,
			SCpnt->SCp.dma_handle);

		return SUCCESS;
	}

	/* Handle the SG case.
	 */
	sg = (struct scatterlist *) SCpnt->request_buffer;
	sg_done  = 0;
	sgeOffset = sizeof(SCSIIORequest_t) - sizeof(SGE_IO_UNION);
	chainSge = NULL;

	/* Prior to entering this loop - the following must be set
	 * current MF:  sgeOffset (bytes)
	 *              chainSge (Null if original MF is not a chain buffer)
	 *              sg_done (num SGE done for this MF)
	 */

nextSGEset:
	numSgeSlots = ((frm_sz - sgeOffset) / (sizeof(u32) + ioc->sge_size) );
	numSgeThisFrame = (sges_left < numSgeSlots) ? sges_left : numSgeSlots;

	sgflags = MPT_SGE_FLAGS_SIMPLE_ELEMENT | sgdir;

	/* Get first (num - 1) SG elements
	 * Skip any SG entries with a length of 0
	 * NOTE: at finish, sg and psge pointed to NEXT data/location positions
	 */
	for (ii=0; ii < (numSgeThisFrame-1); ii++) {
		thisxfer = sg_dma_len(sg);
		if (thisxfer == 0) {
			sg ++; /* Get next SG element from the OS */
			sg_done++;
			continue;
		}

		v2 = sg_dma_address(sg);
		ioc->add_sge(psge, sgflags | thisxfer, v2);

		sg++;		/* Get next SG element from the OS */
		psge += (sizeof(u32) + ioc->sge_size);
		sgeOffset += (sizeof(u32) + ioc->sge_size);
		sg_done++;
	}

	if (numSgeThisFrame == sges_left) {
		/* Add last element, end of buffer and end of list flags.
		 */
		sgflags |= MPT_SGE_FLAGS_LAST_ELEMENT |
				MPT_SGE_FLAGS_END_OF_BUFFER |
				MPT_SGE_FLAGS_END_OF_LIST;

		/* Add last SGE and set termination flags.
		 * Note: Last SGE may have a length of 0 - which should be ok.
		 */
		thisxfer = sg_dma_len(sg);

		v2 = sg_dma_address(sg);
		ioc->add_sge(psge, sgflags | thisxfer, v2);
		/*
		sg++;
		psge += (sizeof(u32) + sizeof(dma_addr_t));
		*/
		sgeOffset += (sizeof(u32) + ioc->sge_size);
		sg_done++;

		if (chainSge) {
			/* The current buffer is a chain buffer,
			 * but there is not another one.
			 * Update the chain element
			 * Offset and Length fields.
			 */
			if (ioc->sge_size == sizeof(u32)) {
				mptscsih_add_chain((char *)chainSge, 0, sgeOffset, ioc->ChainBufferDMA + chain_dma_off);
			} else {
				mptscsih_add_chain_64_bit((char *)chainSge, 0, sgeOffset, ioc->ChainBufferDMA + chain_dma_off);
			}
		} else {
			/* The current buffer is the original MF
			 * and there is no Chain buffer.
			 */
			pReq->ChainOffset = 0;
			RequestNB = (((sgeOffset - 1) >> ioc->NBShiftFactor)  + 1) & 0x03;
			dsgprintk((MYIOC_s_WARN_FMT "Single Buffer RequestNB=%x, sgeOffset=%d\n", ioc->name, RequestNB, sgeOffset));
			ioc->RequestNB[req_idx] = RequestNB;
		}
	} else {
		/* At least one chain buffer is needed.
		 * Complete the first MF
		 *  - last SGE element, set the LastElement bit
		 *  - set ChainOffset (words) for orig MF
		 *             (OR finish previous MF chain buffer)
		 *  - update MFStructPtr ChainIndex
		 *  - Populate chain element
		 * Also
		 * Loop until done.
		 */

		dsgprintk((MYIOC_s_WARN_FMT "SG: Chain Required! sg done %d\n",
				ioc->name, sg_done));

		/* Set LAST_ELEMENT flag for last non-chain element
		 * in the buffer. Since psge points at the NEXT
		 * SGE element, go back one SGE element, update the flags
		 * and reset the pointer. (Note: sgflags & thisxfer are already
		 * set properly).
		 */
		if (sg_done) {
			u32 *ptmp = (u32 *) (psge - (sizeof(u32) + ioc->sge_size));
			sgflags = le32_to_cpu(*ptmp);
			sgflags |= MPT_SGE_FLAGS_LAST_ELEMENT;
			*ptmp = cpu_to_le32(sgflags);
		}

		if (chainSge) {
			/* The current buffer is a chain buffer.
			 * chainSge points to the previous Chain Element.
			 * Update its chain element Offset and Length (must
			 * include chain element size) fields.
			 * Old chain element is now complete.
			 */
			u8 nextChain = (u8) (sgeOffset >> 2);
			sgeOffset += (sizeof(u32) + ioc->sge_size);
			if (ioc->sge_size == sizeof(u32)) {
				mptscsih_add_chain((char *)chainSge, nextChain, sgeOffset, ioc->ChainBufferDMA + chain_dma_off);
			} else {
				mptscsih_add_chain_64_bit((char *)chainSge, nextChain, sgeOffset, ioc->ChainBufferDMA + chain_dma_off);
			}
		} else {
			/* The original MF buffer requires a chain buffer -
			 * set the offset.
			 * Last element in this MF is a chain element.
			 */
			pReq->ChainOffset = (u8) (sgeOffset >> 2);
			RequestNB = (((sgeOffset - 1) >> ioc->NBShiftFactor)  + 1) & 0x03;
			dsgprintk((MYIOC_s_ERR_FMT "Chain Buffer Needed, RequestNB=%x sgeOffset=%d\n", ioc->name, RequestNB, sgeOffset));
			ioc->RequestNB[req_idx] = RequestNB;
		}

		sges_left -= sg_done;


		/* NOTE: psge points to the beginning of the chain element
		 * in current buffer. Get a chain buffer.
		 */
		if ((mptscsih_getFreeChainBuffer(ioc, &newIndex)) == FAILED) {
			dfailprintk((MYIOC_s_WARN_FMT "getFreeChainBuffer FAILED SCSI cmd=%02x (%p)\n",
 			ioc->name, pReq->CDB[0], SCpnt));
			return FAILED;
		}

		/* Update the tracking arrays.
		 * If chainSge == NULL, update ReqToChain, else ChainToChain
		 */
		if (chainSge) {
			ioc->ChainToChain[chain_idx] = newIndex;
		} else {
			ioc->ReqToChain[req_idx] = newIndex;
		}
		chain_idx = newIndex;
		chain_dma_off = ioc->req_sz * chain_idx;

		/* Populate the chainSGE for the current buffer.
		 * - Set chain buffer pointer to psge and fill
		 *   out the Address and Flags fields.
		 */
		chainSge = (char *) psge;
		dsgprintk((KERN_INFO "  Current buff @ %p (index 0x%x)",
				psge, req_idx));

		/* Start the SGE for the next buffer
		 */
		psge = (char *) (ioc->ChainBuffer + chain_dma_off);
		sgeOffset = 0;
		sg_done = 0;

		dsgprintk((KERN_INFO "  Chain buff @ %p (index 0x%x)\n",
				psge, chain_idx));

		/* Start the SGE for the next buffer
		 */

		goto nextSGEset;
	}

	return SUCCESS;
} /* mptscsih_AddSGE() */

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_io_done - Main SCSI IO callback routine registered to
 *	Fusion MPT (base) driver
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@mf: Pointer to original MPT request frame
 *	@r: Pointer to MPT reply frame (NULL if TurboReply)
 *
 *	This routine is called from mpt.c::mpt_interrupt() at the completion
 *	of any SCSI IO request.
 *	This routine is registered with the Fusion MPT (base) driver at driver
 *	load/init time via the mpt_register() API call.
 *
 *	Returns 1 indicating alloc'd request frame ptr should be freed.
 */
static int
mptscsih_io_done(MPT_ADAPTER *ioc, MPT_FRAME_HDR *mf, MPT_FRAME_HDR *mr)
{
	Scsi_Cmnd	*sc;
	MPT_SCSI_HOST	*hd;
	SCSIIORequest_t	*pScsiReq;
	SCSIIOReply_t	*pScsiReply;
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,5,0)
	unsigned long	 flags;
#endif
	u16		 req_idx;
#ifdef MPT_DEBUG_MSG_FRAME
	u16		 req_idx_MR;
#endif

	hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;

	req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
#ifdef MPT_DEBUG_MSG_FRAME
	req_idx_MR = (mr != NULL) ?
	    le16_to_cpu(mr->u.frame.hwhdr.msgctxu.fld.req_idx) : req_idx;
	if ((req_idx != req_idx_MR) ||
	    (mf->u.frame.linkage.arg1 == 0xdeadbeaf)) {
		printk(MYIOC_s_ERR_FMT "Received a mf that was already freed\n",
		    ioc->name);
		printk (MYIOC_s_ERR_FMT
		    "req_idx=%x req_idx_MR=%x mf=%p mr=%p sc=%p\n",
		    ioc->name, req_idx, req_idx_MR, mf, mr, 
		    ioc->ScsiLookup[req_idx_MR]);
		return 0;
	}
#endif

	sc = ioc->ScsiLookup[req_idx];
	ioc->ScsiLookup[req_idx] = NULL;
	if (sc == NULL) {
		MPIHeader_t *hdr = (MPIHeader_t *)mf;

		/* Remark: writeSDP1 will use the mpt_ScsiDoneCtx
		 * If a SCSI I/O cmd, device disabled by OS and
		 * completion done. Cannot touch sc struct. Just free mem.
		 */
		if (hdr->Function == MPI_FUNCTION_SCSI_IO_REQUEST)
			printk(MYIOC_s_ERR_FMT "NULL ScsiCmd ptr!\n", ioc->name);

		if (hdr->Function == MPI_FUNCTION_CONFIG) {
			Config_t		*pReq = (Config_t *)mf;
			ConfigReply_t	    	*pRep = (ConfigReply_t *)mr;
			FCDevicePage0_t		*pFCDevice0;
			struct _MPT_DEVICE	*pMptTarget;
			VirtDevice		*pTarget;
			int			 bus, id;
			u32			pageAddress = le32_to_cpu(pReq->PageAddress);

			dcprintk((MYIOC_s_INFO_FMT
				"config request for %08x/%08x done with status %04x!\n",
				ioc->name,
				le32_to_cpu(*(U32 *)&pReq->Header),
				pageAddress,
				le16_to_cpu(pRep->IOCStatus)));
			if (le16_to_cpu(pRep->IOCStatus) == MPI_IOCSTATUS_SUCCESS &&
			    pReq->Header.PageType == MPI_CONFIG_PAGETYPE_FC_DEVICE &&
			    pReq->Header.PageNumber == 0) {
				pFCDevice0 = (FCDevicePage0_t *)(pReq + 1);
				bus = (pageAddress &
					MPI_FC_DEVICE_PGAD_BT_BUS_MASK) >> 
					MPI_FC_DEVICE_PGAD_BT_BUS_SHIFT;
				id = pageAddress &
					MPI_FC_DEVICE_PGAD_BT_TID_MASK;
				pMptTarget = ioc->Target_List[bus];
				pTarget = (VirtDevice *)pMptTarget->Target[id];
				dcprintk((MYIOC_s_INFO_FMT
					"  bus=%d id=%d is WWPN = %08x%08x, WWNN = %08x%08x\n",
					ioc->name, bus, id,
					le32_to_cpu(pFCDevice0->WWPN.High),
					le32_to_cpu(pFCDevice0->WWPN.Low),
					le32_to_cpu(pFCDevice0->WWNN.High),
					le32_to_cpu(pFCDevice0->WWNN.Low)));
				if (pTarget) {
					pTarget->WWPN = pFCDevice0->WWPN;
					pTarget->WWNN = pFCDevice0->WWNN;
				}
			}
		}

		mptscsih_freeChainBuffers(ioc, req_idx);
		return 1;
	}
	if ((unsigned char *)mf != sc->host_scribble)
	{
		mptscsih_freeChainBuffers(ioc, req_idx);
		return 1;
	}


	sc->result = DID_OK << 16;		/* Set default reply as OK */
	pScsiReq = (SCSIIORequest_t *) mf;
	pScsiReply = (SCSIIOReply_t *) mr;

#ifdef MPT_DEBUG_MSG_FRAME
	if((ioc->facts.MsgVersion >= MPI_VERSION_01_05) && pScsiReply){
		dmfprintk((MYIOC_s_WARN_FMT
			"ScsiDone (mf=%p,mr=%p,sc=%p,idx=%d,task-tag=%d)\n",
			ioc->name, mf, mr, sc, req_idx, pScsiReply->TaskTag));
	}else{
		dmfprintk((MYIOC_s_WARN_FMT
			"ScsiDone (mf=%p,mr=%p,sc=%p,idx=%d)\n",
			ioc->name, mf, mr, sc, req_idx));
	}
#endif

	if (pScsiReply == NULL) {
		/* special context reply handling */

		/* If regular Inquiry cmd - save inquiry data
		 */
		if (pScsiReq->CDB[0] == INQUIRY) {
			int	 dlen;
#ifdef __VMKERNEL_MODULE__
			char *buf;
			dlen = le32_to_cpu(pScsiReq->DataLength);
			buf = vmk_phys_to_kmap(sc->request_bufferMA, dlen);
			mptscsih_initTarget(hd,
				sc->channel,
				sc->target,
				pScsiReq->LUN[1],
				buf,
				dlen);
			vmk_phys_to_kmap_free(buf);
#else
			dlen = le32_to_cpu(pScsiReq->DataLength);
			mptscsih_initTarget(hd,
					sc->channel,
					sc->target,
					pScsiReq->LUN[1],
					sc->buffer,
					dlen);
#endif
		}
	} else {
		u32	 xfer_cnt;
		u16	 ioc_status;
		u8	 scsi_state, scsi_status;
		struct _MPT_DEVICE	*pMptTarget;
		VirtDevice	*pTarget;
		int	 bus, id;

		ioc_status = le16_to_cpu(pScsiReply->IOCStatus) & MPI_IOCSTATUS_MASK;
		scsi_state = pScsiReply->SCSIState;
		scsi_status = pScsiReply->SCSIStatus;
		xfer_cnt = le32_to_cpu(pScsiReply->TransferCount);
		sc->resid = sc->request_bufflen - xfer_cnt;

		dreplyprintk((KERN_NOTICE "Reply ha=%d id=%d lun=%d:\n"
			"IOCStatus=%04xh SCSIState=%02xh SCSIStatus=%02xh\n"
			"resid=%d bufflen=%d xfer_cnt=%d\n",
			ioc->id, pScsiReq->TargetID, pScsiReq->LUN[1],
			ioc_status, scsi_state, scsi_status, sc->resid, 
			sc->request_bufflen, xfer_cnt));

		if (scsi_state & MPI_SCSI_STATE_AUTOSENSE_VALID)
			copy_sense_data(sc, hd, mf, pScsiReply);

		/*
		 *  Look for + dump FCP ResponseInfo[]!
		 */
		if (scsi_state & MPI_SCSI_STATE_RESPONSE_INFO_VALID) {
			printk(KERN_NOTICE "  FCP_ResponseInfo=%08xh\n",
			le32_to_cpu(pScsiReply->ResponseInfo));
		}

		switch(ioc_status) {
		case MPI_IOCSTATUS_BUSY:			/* 0x0002 */
			/* CHECKME!
			 * Maybe: DRIVER_BUSY | SUGGEST_RETRY | DID_SOFT_ERROR (retry)
			 * But not: DID_BUS_BUSY lest one risk
			 * killing interrupt handler:-(
			 */
			sc->result = STS_BUSY;
#if 1 || defined(__VMWARE__)
			/* Increment the IOC status BUSY counter for this target if present */
			bus = pScsiReq->Bus;
			id = pScsiReq->TargetID;
			pMptTarget = ioc->Target_List[bus];
			pTarget = (VirtDevice *)pMptTarget->Target[id];
			if ( pTarget )
				pTarget->busy_count++;
#endif
			break;

		case MPI_IOCSTATUS_SCSI_INVALID_BUS:		/* 0x0041 */
		case MPI_IOCSTATUS_SCSI_INVALID_TARGETID:	/* 0x0042 */
			sc->result = DID_BAD_TARGET << 16;
			break;

		case MPI_IOCSTATUS_SCSI_DEVICE_NOT_THERE:	/* 0x0043 */
			/* Spoof to SCSI Selection Timeout! */
			sc->result = DID_NO_CONNECT << 16;

			bus = pScsiReq->Bus;
			id = pScsiReq->TargetID;
			if ( ioc->bus_type == SPI ) {
				if (hd->sel_timeout[id] < 0xFFFF)
					hd->sel_timeout[pScsiReq->TargetID]++;
			}

			pMptTarget = ioc->Target_List[bus];
			pTarget = (VirtDevice *)pMptTarget->Target[id];

			if ( pTarget ) {
				if (pTarget->tflags & MPT_TARGET_FLAGS_LED_ON) {
					MPT_FRAME_HDR		*mf;
					SEPRequest_t *SEPMsg;
					/* Get a MF for this command.
	 				*/
					if ((mf = mpt_get_msg_frame(ScsiScanDvCtx, ioc)) == NULL) {
						dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
						    ioc->name,__FUNCTION__));
					} else {
						SEPMsg = (SEPRequest_t *)mf;
						SEPMsg->Function = MPI_FUNCTION_SCSI_ENCLOSURE_PROCESSOR;
						SEPMsg->Bus = pTarget->bus;
						SEPMsg->TargetID = pTarget->id;
						SEPMsg->Action = MPI_SEP_REQ_ACTION_WRITE_STATUS;
						SEPMsg->SlotStatus = MPI_SEP_REQ_SLOTSTATUS_UNCONFIGURED;
						pTarget->tflags &= ~MPT_TARGET_FLAGS_LED_ON;
						devtprintk((MYIOC_s_WARN_FMT "Sending SEP UNCONFIGURED cmd id=%d bus=%d\n",
							ioc->name, SEPMsg->TargetID, SEPMsg->Bus));
						mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);
					}
				}
				if ( ioc->bus_type != FC ) {
					dinitprintk((KERN_INFO "Target structure (bus=%d id=%d) @ %p Removed due to NOT_THERE\n",
						bus, id, pTarget));
					pMptTarget->Target[id] = NULL;
				}
			}
			break;

		case MPI_IOCSTATUS_SCSI_TASK_TERMINATED:	/* 0x0048 */
			/* DID_RESET causes a retry but does not bump the 
			 * retries counter in the sc structure */
			sc->result = DID_RESET << 16;
			dreplyprintk((MYIOC_s_WARN_FMT "TASK_TERMINATED: "
				"id=%d "
				"IOCStatus=%04x SCSIState=%02x\n"
				"SCSIStatus=%02x "
				"sc->result=%08x retries=%d sc=%p\n",
				ioc->name, pScsiReq->TargetID, 
				le16_to_cpu(pScsiReply->IOCStatus),
				scsi_state, 
				scsi_status, sc->result, 
				sc->retries, sc));
			break;

		case MPI_IOCSTATUS_SCSI_IOC_TERMINATED:		/* 0x004B */
			if ( ioc->bus_type == SAS ) {
				u16	 status = le16_to_cpu(pScsiReply->IOCStatus);
				u32	 log_info = le32_to_cpu(mr->u.reply.IOCLogInfo);
// 				sc->result = DID_RESET << 16;
				sc->result = DID_SOFT_ERROR << 16;
				if (status & MPI_IOCSTATUS_FLAG_LOG_INFO_AVAILABLE) {
					if ((log_info & 0xFFFF0000) == 
						SAS_LOGINFO_NEXUS_LOSS) {
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
						/* start a path switch at nexus loss */
						sc->result = (DID_NO_CONNECT << 16);
#else	
						sc->result = (DID_BUS_BUSY << 16);
#endif
					}
				}
				derrprintk((KERN_NOTICE "IOC_TERMINATED: "
					"ha=%d id=%d lun=%d "
					"IOCStatus=%04x SCSIState=%02x\n"
					"SCSIStatus=%02x LogInfo=%08x "
					"sc->result=%08x sc=%p\n",
					ioc->id, pScsiReq->TargetID, 
					pScsiReq->LUN[1], status, scsi_state, 
					scsi_status, log_info, sc->result, sc));
				break;	
			}  /* allow non-SAS & non-NEXUS_LOSS to drop into below code */

		case MPI_IOCSTATUS_SCSI_EXT_TERMINATED:		/* 0x004C */
			/* Linux handles an unsolicited DID_RESET better
			 * than an unsolicited DID_ABORT.
			 */
			sc->result = DID_RESET << 16;

			/* GEM Workaround. */
			if (ioc->bus_type == SPI)
				mptscsih_no_negotiate(hd, sc->target);
			break;

		case MPI_IOCSTATUS_SCSI_RESIDUAL_MISMATCH:	/* 0x0049 */
			sc->resid = sc->request_bufflen - xfer_cnt;
			if ( (xfer_cnt == 0) || (sc->underflow > xfer_cnt)) {
				sc->result = DID_SOFT_ERROR << 16;
			} else {
				/* Sufficient data transfer occurred */
				sc->result = (DID_OK << 16) | scsi_status;
			}
			dreplyprintk((KERN_NOTICE "RESIDUAL_MISMATCH: result=%x on id=%d xfer_cnt=%d underflow=%d\n", 
				sc->result, sc->device->id, xfer_cnt, sc->underflow));
			break;

		case MPI_IOCSTATUS_SCSI_DATA_UNDERRUN:		/* 0x0045 */
			/*
			 *  Do upfront check for valid SenseData and give it
			 *  precedence!
			 */
			sc->result = (DID_OK << 16) | scsi_status;
			if (scsi_state & MPI_SCSI_STATE_AUTOSENSE_VALID) {
				/* Have already saved the status and sense data
				 */
				;
			} else {
				if (xfer_cnt < sc->underflow) {
					sc->result = DID_SOFT_ERROR << 16;
				}
				if (scsi_state & (MPI_SCSI_STATE_AUTOSENSE_FAILED | MPI_SCSI_STATE_NO_SCSI_STATUS)) {
					/* What to do?
				 	*/
					sc->result = DID_SOFT_ERROR << 16;
				}
				else if (scsi_state & MPI_SCSI_STATE_TERMINATED) {
					/*  Not real sure here either...  */
					sc->result = DID_RESET << 16;
				}
			}

			dreplyprintk((KERN_NOTICE "  sc->underflow={report ERR if < %02xh bytes xfer'd}\n",
					sc->underflow));
			dreplyprintk((KERN_NOTICE "  ActBytesXferd=%02xh\n", xfer_cnt));
			/* Report Queue Full
			 */
			if (scsi_status == MPI_SCSI_STATUS_TASK_SET_FULL)
				mptscsih_report_queue_full(sc, pScsiReply, pScsiReq);

			if ( (pScsiReq->CDB[0] == INQUIRY) && xfer_cnt ) {
#ifdef __VMKERNEL_MODULE__
				char *buf;
				buf = vmk_phys_to_kmap(sc->request_bufferMA,
					xfer_cnt);
				mptscsih_initTarget(hd,
					sc->channel,
					sc->target,
					pScsiReq->LUN[1],
					buf,
					xfer_cnt);
				vmk_phys_to_kmap_free(buf);
#else
				mptscsih_initTarget(hd,
						sc->channel,
						sc->target,
						pScsiReq->LUN[1],
						sc->buffer,
						xfer_cnt);
#endif
			}
			break;

		case MPI_IOCSTATUS_SCSI_DATA_OVERRUN:		/* 0x0044 */
			sc->resid = 0;
		case MPI_IOCSTATUS_SCSI_RECOVERED_ERROR:	/* 0x0040 */
		case MPI_IOCSTATUS_SUCCESS:			/* 0x0000 */
			scsi_status = pScsiReply->SCSIStatus;
			sc->result = (DID_OK << 16) | scsi_status;
			if (scsi_state == 0) {
				;
			} else if (scsi_state & MPI_SCSI_STATE_AUTOSENSE_VALID) {
				/*
				 * If running against circa 200003dd 909 MPT f/w,
				 * may get this (AUTOSENSE_VALID) for actual TASK_SET_FULL
				 * (QUEUE_FULL) returned from device! --> get 0x0000?128
				 * and with SenseBytes set to 0.
				 */
				if (pScsiReply->SCSIStatus == MPI_SCSI_STATUS_TASK_SET_FULL)
					mptscsih_report_queue_full(sc, pScsiReply, pScsiReq);
			}
			else if (scsi_state &
			         (MPI_SCSI_STATE_AUTOSENSE_FAILED | MPI_SCSI_STATE_NO_SCSI_STATUS)
			   ) {
				/*
				 * What to do?
				 */
				sc->result = DID_SOFT_ERROR << 16;
			}
			else if (scsi_state & MPI_SCSI_STATE_TERMINATED) {
				/*  Not real sure here either...  */
				sc->result = DID_RESET << 16;
			}
			else if (scsi_state & MPI_SCSI_STATE_QUEUE_TAG_REJECTED) {
				/* Device Inq. data indicates that it supports
				 * QTags, but rejects QTag messages.
				 * This command completed OK.
				 *
				 * Not real sure here either so do nothing...  */
			}

			if (sc->result == MPI_SCSI_STATUS_TASK_SET_FULL)
				mptscsih_report_queue_full(sc, pScsiReply, pScsiReq);

			/* Add handling of:
			 * Reservation Conflict, Busy,
			 * Command Terminated, CHECK
			 */

			/* If regular Inquiry cmd - save inquiry data
			 */
			if ( (sc->result == (DID_OK << 16))
			     && xfer_cnt
			     && pScsiReq->CDB[0] == INQUIRY ) {
#ifdef __VMKERNEL_MODULE__
				char *buf;
				buf = vmk_phys_to_kmap(sc->request_bufferMA,
					xfer_cnt);
				mptscsih_initTarget(hd,
					sc->channel,
					sc->target,
					pScsiReq->LUN[1],
					buf,
					xfer_cnt);
				vmk_phys_to_kmap_free(buf);
#else
				mptscsih_initTarget(hd,
						sc->channel,
						sc->target,
						pScsiReq->LUN[1],
						sc->buffer,
						xfer_cnt);
#endif
			}
			break;

		case MPI_IOCSTATUS_SCSI_PROTOCOL_ERROR:		/* 0x0047 */
			sc->result = DID_SOFT_ERROR << 16;
			break;

		case MPI_IOCSTATUS_INVALID_FUNCTION:		/* 0x0001 */
		case MPI_IOCSTATUS_INVALID_SGL:			/* 0x0003 */
		case MPI_IOCSTATUS_INTERNAL_ERROR:		/* 0x0004 */
		case MPI_IOCSTATUS_RESERVED:			/* 0x0005 */
		case MPI_IOCSTATUS_INSUFFICIENT_RESOURCES:	/* 0x0006 */
		case MPI_IOCSTATUS_INVALID_FIELD:		/* 0x0007 */
		case MPI_IOCSTATUS_INVALID_STATE:		/* 0x0008 */
		case MPI_IOCSTATUS_SCSI_IO_DATA_ERROR:		/* 0x0046 */
		case MPI_IOCSTATUS_SCSI_TASK_MGMT_FAILED:	/* 0x004A */
		default:
			/*
			 * What to do?
			 */
			sc->result = DID_SOFT_ERROR << 16;
			break;

		}	/* switch(ioc_status) */

		dreplyprintk((KERN_NOTICE "  sc->result is %08xh\n", sc->result));
	} /* end of address reply case */

#ifndef __VMKERNEL_MODULE__
	/* Unmap the DMA buffers, if any. */
	if (sc->use_sg) {
		pci_unmap_sg(ioc->pcidev, (struct scatterlist *) sc->request_buffer,
			    sc->use_sg, sc->sc_data_direction);
	} else if (sc->request_bufflen) {
		pci_unmap_single(ioc->pcidev, sc->SCp.dma_handle,
				sc->request_bufflen, sc->sc_data_direction);
	}
#endif

	/* Free Chain buffers */
	mptscsih_freeChainBuffers(ioc, req_idx);

	sc->host_scribble = NULL;

        MPT_HOST_LOCK(flags);
#ifdef MPT_DEBUG_QCMD_DEPTH
	mptscsih_scsi_done(ioc,sc);
#else
	sc->scsi_done(sc);		/* Issue the command callback */
#endif
        MPT_HOST_UNLOCK(flags);

	return 1;
}

static void
mptscsih_reset_timeouts (MPT_SCSI_HOST *hd)
{
	MPT_ADAPTER *ioc = hd->ioc;
	Scsi_Cmnd	*SCpnt;
	int		 ii;
	int		 max = hd->ioc->req_depth;

//	for (ii= 0; ii < max; ii++) {
	for (ii= max; ii < 0; ii--) {
		if ((SCpnt = ioc->ScsiLookup[ii]) != NULL) {
#ifdef MPT_DEBUG_RESET
			MPT_FRAME_HDR	*mf;
			mf = MPT_INDEX_2_MFPTR(hd->ioc, ii);
			drsprintk((MYIOC_s_WARN_FMT "resetting SCpnt=%p mf=%p ii=%d timeout + 60MPT_HZ\n",
				(hd && hd->ioc) ? hd->ioc->name : "ioc?", 
				SCpnt, mf, ii));
			DBG_DUMP_TIMER_REQUEST_FRAME(mf)
#endif
			mod_timer(&SCpnt->eh_timeout, jiffies + MPT_JIFFY*60);
		}
	}
}

/*
 *	mptscsih_flush_running_cmds - For each command found, search
 *		Scsi_Host instance taskQ and reply to OS.
 *		Called only if recovering from a FW reload.
 *	@hd: Pointer to a SCSI HOST structure
 *
 *	Returns: None.
 *
 *	Must be called while new I/Os are being queued.
 */
static void
mptscsih_flush_running_cmds(MPT_SCSI_HOST *hd)
{
	MPT_ADAPTER *ioc = hd->ioc;
	Scsi_Cmnd	*SCpnt;
	MPT_FRAME_HDR	*mf;
	int		 ii;
	int		 max = ioc->req_depth;

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,5,0)
//	unsigned long	 flags;
#endif

	drsprintk((KERN_INFO MYNAM ": flush_ScsiLookup called\n"));
	for (ii= 0; ii < max; ii++) {
		if ((SCpnt = ioc->ScsiLookup[ii]) != NULL) {

			/* Command found.
			 */

			/* Null ScsiLookup index
			 */
			ioc->ScsiLookup[ii] = NULL;

			mf = MPT_INDEX_2_MFPTR(ioc, ii);
			dmfprintk(( "flush: ScsiDone (mf=%p,sc=%p)\n",
					mf, SCpnt));

			/* Free Chain buffers */
			mptscsih_freeChainBuffers(ioc, ii);

			/* Free Message frame */
			mpt_free_msg_frame(ioc, mf);

			if ((unsigned char *)mf != SCpnt->host_scribble)
			{
				continue;
			}

#ifndef __VMKERNEL_MODULE__
			/* Set status, free OS resources (SG DMA buffers)
			 * Free driver resources (chain, msg buffers)
			 */
			if (SCpnt->use_sg) {
				pci_unmap_sg(ioc->pcidev,
					(struct scatterlist *) SCpnt->request_buffer,
					SCpnt->use_sg,
					SCpnt->sc_data_direction);
			} else if (SCpnt->request_bufflen) {
				pci_unmap_single(ioc->pcidev,
					SCpnt->SCp.dma_handle,
					SCpnt->request_bufflen,
					SCpnt->sc_data_direction);
			}
#endif

			SCpnt->result = DID_RESET << 16;
			SCpnt->host_scribble = NULL;

			drsprintk((MYIOC_s_WARN_FMT "flush SCpnt=%p mf=%p ii=%d result=%x\n",
				ioc->name, SCpnt, mf, ii, SCpnt->result));

/*			drsprintk((MYIOC_s_WARN_FMT "Issuing MPT_HOST_LOCK\n",
				ioc->name));

                        MPT_HOST_LOCK(flags); */
			drsprintk((MYIOC_s_WARN_FMT "returning SC=%p to mid-layer\n",
				ioc->name, SCpnt));

#ifdef MPT_DEBUG_QCMD_DEPTH
			mptscsih_scsi_done(ioc,SCpnt);
#else
			if (SCpnt->scsi_done) {
				SCpnt->scsi_done(SCpnt);	/* Issue the command callback */
				drsprintk((MYIOC_s_WARN_FMT "SC=%p to mid-layer callback completed\n",
					ioc->name, SCpnt));
			} else {
				drsprintk((MYIOC_s_WARN_FMT "SC=%p has no scsi_done callback address\n",
					ioc->name, SCpnt));
			}
#endif

/* 			drsprintk((MYIOC_s_WARN_FMT "Issuing MPT_HOST_UNLOCK\n",
				ioc->name));

                        MPT_HOST_UNLOCK(flags);
			drsprintk((MYIOC_s_WARN_FMT "MPT_HOST_UNLOCK completed\n",
				ioc->name)); */
		}
	}
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *  Hack! It might be nice to report if a device is returning QUEUE_FULL
 *  but maybe not each and every time...
 */
static long last_queue_full = 0;

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_report_queue_full - Report QUEUE_FULL status returned
 *	from a SCSI target device.
 *	@sc: Pointer to Scsi_Cmnd structure
 *	@pScsiReply: Pointer to SCSIIOReply_t
 *	@pScsiReq: Pointer to original SCSI request
 *
 *	This routine periodically reports QUEUE_FULL status returned from a
 *	SCSI target device.  It reports this to the console via kernel
 *	printk() API call, not more than once every 10 seconds.
 */
static void
mptscsih_report_queue_full(Scsi_Cmnd *sc, SCSIIOReply_t *pScsiReply, SCSIIORequest_t *pScsiReq)
{
	long time = jiffies;

	if (time - last_queue_full > 10 * MPT_HZ) {
		char *ioc_str = "ioc?";

		if (sc->host != NULL && sc->host->hostdata != NULL)
			ioc_str = ((MPT_SCSI_HOST *)sc->host->hostdata)->ioc->name;
		dprintk((MYIOC_s_WARN_FMT "Device (%d:%d:%d) reported QUEUE_FULL!\n",
				ioc_str, 0, sc->target, sc->lun));
		last_queue_full = time;
	}
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
static int BeenHereDoneThat = 0;
static char *info_kbuf = NULL;

/*  SCSI host fops start here...  */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_detect - Register MPT adapter(s) as SCSI host(s) with
 *	linux scsi mid-layer.
 *	@tpnt: Pointer to Scsi_Host_Template structure
 *
 *	(linux Scsi_Host_Template.detect routine)
 *
 *	Returns number of SCSI host adapters that were successfully
 *	registered with the linux scsi mid-layer via the scsi_register()
 *	API call.
 */
int
mptscsih_detect(Scsi_Host_Template *tpnt)
{
	struct Scsi_Host	*sh;
	MPT_SCSI_HOST		*hd;
	MPT_ADAPTER		*ioc;
	MPT_REQUEST_Q		*freeQ;
	unsigned long		 flags;
	int			 sz, ii;
	int			 ioc_cap;
	u8			*mem;
#ifdef __VMKERNEL_MODULE__
        if (!vmk_set_module_version("%s", my_VERSION)) {
           return 0;
        }
#endif
#ifdef CONFIG_FUSION_VMWARE_SINGLE_MODULE_2xx
        /*
         * Actually do the deferred initialization.  This is necessary in
         * VMNIX as well in the case that someone rmmod's the protocol module
         * and then insmod's it again.
         */
        fusion_init();
#endif

	if (! BeenHereDoneThat++) {
		show_mptmod_ver(my_NAME, my_VERSION);

		mpt_ScsiDoneCtx = mpt_register(mptscsih_io_done, MPTSCSIH_DRIVER);
		ScsiTaskCtx = mpt_register(mptscsih_taskmgmt_complete, MPTSCSIH_DRIVER);
		ScsiScanDvCtx = mpt_register(mptscsih_scandv_complete, MPTSCSIH_DRIVER);

#if (defined(__VMKERNEL_MODULE__) && !defined(MPTSCSIH_DISABLE_DOMAIN_VALIDATION))
                spin_lock_init(&dvtaskQ_lock);
#endif
#ifdef __VMKERNEL_MODULE__
                spin_lock_init(&scandv_waitq.lock);
#endif

		if (mpt_event_register(mpt_ScsiDoneCtx, mptscsih_event_process) == 0) {
			devtprintk((KERN_INFO MYNAM ": Registered for IOC event notifications\n"));
		} else {
			/* FIXME! */
		}

		if (mpt_reset_register(mpt_ScsiDoneCtx, mptscsih_ioc_reset) == 0) {
			dinitprintk((KERN_INFO MYNAM ": Registered mptscsih_ioc_reset notification mpt_ScsiDoneCtx=%d\n", mpt_ScsiDoneCtx));
		} else {
			dinitprintk((KERN_INFO MYNAM ": Register of mptscsih_ioc_reset notification mpt_ScsiDoneCtx=%d FAILED\n", mpt_ScsiDoneCtx));
		}
	}
	dinitprintk((KERN_INFO MYNAM ": mpt_scsih_detect()\n"));

	list_for_each_entry(ioc, &ioc_list, list) {
		/*  Added sanity check on readiness of the MPT adapter.
		 */
		if (ioc->last_state != MPI_IOC_STATE_OPERATIONAL) {
			printk(MYIOC_s_WARN_FMT "Skipping because it's not operational!\n",
					ioc->name);
			continue;
		}

		if (!ioc->active) {
			printk(MYIOC_s_WARN_FMT "Skipping because it's disabled!\n",
					ioc->name);
			continue;
		}


		/*  Sanity check - ensure at least 1 port is INITIATOR capable
		 */
		ioc_cap = 0;
		for (ii=0; ii < ioc->facts.NumberOfPorts; ii++) {
			if (ioc->pfacts[ii].ProtocolFlags & MPI_PORTFACTS_PROTOCOL_INITIATOR)
				ioc_cap ++;
		}

		if (!ioc_cap) {
			printk(MYIOC_s_WARN_FMT "Skipping ioc=%p because SCSI Initiator mode is NOT enabled!\n",
					ioc->name, ioc);
			continue;
		}


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
		tpnt->proc_dir = &proc_mpt_scsihost;
#endif
		tpnt->proc_info = mptscsih_proc_info;
#ifdef __VMKERNEL_MODULE__
		sh = vmk_scsi_register(tpnt, sizeof(MPT_SCSI_HOST),
				MPT_GET_PCIDEV(ioc)->bus->number,
				MPT_GET_PCIDEV(ioc)->devfn);
#else
		sh = scsi_register(tpnt, sizeof(MPT_SCSI_HOST));
#endif
		if (sh != NULL) {
#ifdef __VMKERNEL_MODULE__
			u64 host_sas_address, temp;
                        vmk_scsi_register_uinfo(sh, MPT_GET_PCIDEV(ioc)->bus->number,
                                                MPT_GET_PCIDEV(ioc)->devfn, ioc);
                        /* Start the sas transport */
                        if (ioc->bus_type == SAS) {
                                if (vmk_sas_attach_transport(sh, &mpt2xx_transport_functions) != SUCCESS){
                                        printk(MYIOC_s_WARN_FMT"mptscsih_detect: vmk_sas_attach_transport: failed!\n",
                                               ioc->name);

                                        scsi_unregister(sh);
                                        // fatal error
                                        continue;
                                }
				temp = ioc->sas_port_WWID[0].High;
#if 1 || defined(__VMWARE__)
				host_sas_address = (temp << 32) | ioc->sas_port_WWID[0].Low;
#else
				host_sas_address = (temp << 32) || ioc->sas_port_WWID[0].Low;
#endif
				sas_host_sas_address(sh) =  host_sas_address;
                        }
#endif
			SPIN_LOCK_IRQSAVE(&ioc->FreeQlock, flags);
#if defined(CONFIG_VMNIX) && !defined(__VMKERNEL_MODULE__)
			sh->bus = MPT_GET_PCIDEV(ioc)->bus->number;
			sh->devfn = MPT_GET_PCIDEV(ioc)->devfn;
			sh->devid = (void *)sh->hostdata;
#endif
			ioc->TaskCtx = ScsiTaskCtx;
			sh->io_port = 0;
			sh->n_io_port = 0;
			sh->irq = 0;

			/* Yikes!  This is important!
			 * Otherwise, by default, linux
			 * only scans target IDs 0-7!
			 * pfactsN->MaxDevices unreliable
			 * (not supported in early
			 *	versions of the FW).
			 * max_id = 1 + actual max id,
			 * max_lun = 1 + actual last lun,
			 *	see hosts.h :o(
			 */
			if ( mptbase_can_queue < ioc->req_depth )
				sh->can_queue = mptbase_can_queue;
			else
				sh->can_queue = ioc->req_depth;
			dinitprintk((MYIOC_s_WARN_FMT
				"mptbase_can_queue=%d req_depth=%d can_queue=%d\n",
				ioc->name, mptbase_can_queue, ioc->req_depth, 
				sh->can_queue));
#if 1 || defined(__VMWARE__)
			if (ioc->bus_type == SPI) {
				sh->max_id = MPT_MAX_SCSI_DEVICES;
				sh->max_lun = MPT_SCSI_LAST_LUN + 1;
			} else {
				sh->max_id = ioc->DevicesPerBus;
				sh->max_lun = MPT_LAST_LUN + 1;
			}
#else
			sh->max_id = ioc->DevicesPerBus;
			sh->max_lun = MPT_LAST_LUN + 1;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,7)
			sh->max_sectors = MPT_SCSI_MAX_SECTORS;
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20) || defined CONFIG_HIGHIO
			sh->highmem_io = 1;
#endif
			/* MPI uses {port, bus, id, lun}, but logically maps
			 * devices on different ports to different buses, i.e.,
			 * bus 1 may be the 2nd bus on port 0 or the 1st bus on port 1.
			 * Map bus to channel, ignore port number in SCSI....
			 *	hd->port = 0;
			 */
			sh->max_channel = ioc->NumberOfBuses - 1;
			sh->this_id = ioc->pfacts[0].PortSCSIID;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,44)
			/* OS entry to allow host drivers to force
			 * a queue depth on a per device basis.
			 */
			sh->select_queue_depths = mptscsih_select_queue_depths;
#endif
			/* Required entry.
			 */
			sh->unique_id = ioc->id;

			sh->sg_tablesize = ioc->sg_tablesize;

			/* Set the pci device pointer in Scsi_Host structure.
			 */
			scsi_set_pci_device(sh, ioc->pcidev);

			SPIN_UNLOCK_IRQRESTORE(&ioc->FreeQlock, flags);

			hd = (MPT_SCSI_HOST *) sh->hostdata;
			hd->ioc = ioc;

			/* SCSI needs Scsi_Cmnd lookup table!
			 * (with size equal to req_depth*PtrSz!)
			 */
			sz = ioc->req_depth * sizeof(void *);
			mem = kmalloc(sz, GFP_ATOMIC);
			if (mem == NULL) {
				dinitprintk((MYIOC_s_WARN_FMT "ScsiLookup allocation failed\n",
					ioc->name));
				goto done;
			}

			memset(mem, 0, sz);
			ioc->ScsiLookup = (struct scsi_cmnd **) mem;

			dinitprintk((MYIOC_s_WARN_FMT "ScsiLookup @ %p, sz=%d\n",
				 ioc->name, ioc->ScsiLookup, sz));

			mem = kmalloc(sz, GFP_ATOMIC);
			if (mem == NULL) {
				dinitprintk((MYIOC_s_WARN_FMT "PendingScsi allocation failed\n",
					ioc->name));
				goto done;
			}

			memset(mem, 0, sz);
			ioc->PendingScsi = (struct scsi_cmnd **) mem;

			for (ii=0; ii < ioc->NumberOfBuses; ii++) {
				/* Allocate memory for the device structures.
				 * A non-Null pointer at an offset
				 * indicates a device exists.
				 */
				sz = ioc->DevicesPerBus * sizeof(void *);
				mem = kmalloc(sz, GFP_ATOMIC);
				if (mem == NULL) {
					dinitprintk((KERN_INFO
				 	 " Unable to allocate Target_List bus=%d sz=%d\n", ii, sz));
					goto done;
				}

				memset(mem, 0, sz);
				ioc->Target_List[ii] = (struct _MPT_DEVICE *) mem;

				dinitprintk((KERN_INFO
				  " %s: For Bus=%d, Target_List=%p sz=%d\n", 
					ioc->name, ii, mem, sz));
			}

			dinitprintk((MYIOC_s_WARN_FMT "PendingScsi @ %p, sz=%d\n",
				 ioc->name, ioc->PendingScsi, sz));

			/* Allocate memory for free and doneQ's
			 */
			sz = sh->can_queue * sizeof(MPT_REQUEST_Q);
			mem = kmalloc(sz, GFP_ATOMIC);
			if (mem == NULL) {
				dinitprintk((MYIOC_s_WARN_FMT "memQ allocation failed\n",
					ioc->name));
				goto done;
			}

			memset(mem, 0xFF, sz);
			hd->memQ = mem;

			/* Initialize the free and pending Qs.
			 */
			Q_INIT(&ioc->freeQ, MPT_REQUEST_Q);
			Q_INIT(&ioc->pendingQ, MPT_REQUEST_Q);
			Q_INIT(&ioc->TMpendingQ, MPT_REQUEST_Q);
			spin_lock_init(&ioc->freeQlock);

			for (ii=0; ii < sh->can_queue; ii++) {
				freeQ = (MPT_REQUEST_Q *) mem;
				Q_ADD_TAIL(&ioc->freeQ.head, freeQ, MPT_REQUEST_Q);
				mem += sizeof(MPT_REQUEST_Q);
			}

			/* Initialize this Scsi_Host
			 * internal task Q.
			 */
			Q_INIT(&hd->taskQ, MPT_FRAME_HDR);
			hd->taskQcnt = 0;

			/* Clear the TM flags
			 */
			ioc->tmState = TM_STATE_NONE;
			dtmprintk((MYIOC_s_WARN_FMT "%s: tmState set to TM_STATE_NONE!\n", 
				ioc->name, __FUNCTION__));
			hd->resetPending = 0;
			hd->abortSCpnt = NULL;
			ioc->tmPtr = NULL;

			/* Clear the pointer used to store
			 * single-threaded commands, i.e., those
			 * issued during a bus scan, dv and
			 * configuration pages.
			 */
			hd->cmdPtr = NULL;

			/* Attach the SCSI Host to the IOC structure
			 */
			ioc->sh = sh;

			/* Initialize this SCSI Hosts' timers
			 * To use, set the timer expires field
			 * and ADD_TIMER
			 */
			init_timer(&hd->timer);
			hd->timer.data = (unsigned long) hd;
			hd->timer.function = mptscsih_timer_expired;

			init_timer(&ioc->TMtimer);
			ioc->TMtimer.data = (unsigned long) hd;
			ioc->TMtimer.function = mptscsih_taskmgmt_timeout;
			hd->qtag_tick = jiffies;

			if (ioc->bus_type == SAS) {
				/* Update with the driver setup
				 * values.
				 */
				ioc->spi_data.ptClear =
				    mpt_pt_clear;

				ioc->spi_data.Saf_Te = mpt_saf_te;

				if(ioc->spi_data.ptClear==1) {
					mptbase_sas_persist_operation(
					    ioc, MPI_SAS_OP_CLEAR_ALL_PERSISTENT);
				}

				init_MUTEX(&ioc->hot_plug_semaphore);
				INIT_LIST_HEAD(&ioc->sas_device_info_list);
				init_MUTEX(&ioc->sas_device_info_mutex);
				mptsas_initialize(ioc);
			} else {
				if (ioc->bus_type == SPI) {
					/* Update with the driver setup
					 * values.
					 */
					if (ioc->spi_data.maxBusWidth > mpt_width)
						ioc->spi_data.maxBusWidth = mpt_width;
					if (ioc->spi_data.minSyncFactor < mpt_factor)
						ioc->spi_data.minSyncFactor = mpt_factor;

					if (ioc->spi_data.minSyncFactor == MPT_ASYNC)
						ioc->spi_data.maxSyncOffset = 0;

					ioc->spi_data.Saf_Te = mpt_saf_te;

					hd->negoNvram = 0;
#ifndef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
					hd->negoNvram = MPT_SCSICFG_USE_NVRAM;
#endif
					ioc->spi_data.forceDv = 0;
					if (mpt_qas_disable)  /* Is command-line QAS Disable requested? */
						ioc->spi_data.noQas |= MPT_TARGET_NO_NEGO_QAS;
					else
						ioc->spi_data.noQas = 0;
//					if (hd->negoNvram == 0) {
						for (ii=0; ii < ioc->facts.MaxDevices; ii++)
							ioc->spi_data.dvStatus[ii] |= MPT_SCSICFG_DV_NOT_DONE;
//					}

					dinitprintk((MYIOC_s_WARN_FMT
						"dv %x width %x factor %x saf_te %x pt_clear %x\n",
						ioc->name, mpt_dv,
						mpt_width,
						mpt_factor,
						mpt_saf_te,
						mpt_pt_clear));
				}
			}
			mpt_scsi_hosts++;
		}
	}

done:

	if (mpt_scsi_hosts > 0) {
		register_reboot_notifier(&mptscsih_notifier);
#ifdef CONFIG_FUSION_VMWARE_SINGLE_MODULE_2xx
		mptctl_init();
#endif
	} else {
		mpt_reset_deregister(mpt_ScsiDoneCtx);
		dprintk((KERN_INFO MYNAM ": Deregistered for IOC reset notifications\n"));

		mpt_event_deregister(mpt_ScsiDoneCtx);
		dprintk((KERN_INFO MYNAM ": Deregistered for IOC event notifications\n"));

		mpt_deregister(ScsiScanDvCtx);
		mpt_deregister(ScsiTaskCtx);
		mpt_deregister(mpt_ScsiDoneCtx);

		if (info_kbuf != NULL)
			kfree(info_kbuf);
#ifdef __VMKERNEL_MODULE__
#ifndef MPTSCSIH_DISABLE_DOMAIN_VALIDATION
			spin_lock_destroy(&dvtaskQ_lock);
#endif
			spin_lock_destroy(&scandv_waitq.lock);
#endif
#if (!defined(__VMKERNEL_MODULE__MPT_UNLINKED__) && !defined(__VMNIX__MPT_UNLINKED__))
			fusion_exit();
#endif
	}

	return mpt_scsi_hosts;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_release - Unregister SCSI host from linux scsi mid-layer
 *	@host: Pointer to Scsi_Host structure
 *
 *	(linux Scsi_Host_Template.release routine)
 *	This routine releases all resources associated with the SCSI host
 *	adapter.
 *
 *	Returns 0 for success.
 */
int
mptscsih_release(struct Scsi_Host *host)
{
	MPT_ADAPTER	*ioc;
	MPT_SCSI_HOST	*hd;
	int 		 count;
	unsigned long	 flags;

	hd = (MPT_SCSI_HOST *) host->hostdata;

#ifdef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
	/* Check DV thread active */
	count = 10 * MPT_HZ;
	SPIN_LOCK_IRQSAVE(&dvtaskQ_lock, flags);
	if (dvtaskQ_active) {
		SPIN_UNLOCK_IRQRESTORE(&dvtaskQ_lock, flags);
		while(dvtaskQ_active && --count) {
			set_current_state(TASK_INTERRUPTIBLE);
			schedule_timeout(1);
		}
	} else {
		SPIN_UNLOCK_IRQRESTORE(&dvtaskQ_lock, flags);
	}
	if (!count)
		printk(KERN_ERR MYNAM ": ERROR - DV thread still active!\n");
#if defined(MPT_DEBUG_DV) || defined(MPT_DEBUG_DV_TINY)
	else
		printk(KERN_ERR MYNAM ": DV thread orig %d, count %d\n", (int)(10 * MPT_HZ), count);
#endif
#endif

	unregister_reboot_notifier(&mptscsih_notifier);

	if (hd != NULL) {
		struct _MPT_DEVICE	*pMptTarget;
		VirtDevice	*pTarget;
		int sz1, sz3;
		int szQ = 0;
		int bus, id;

		ioc = hd->ioc;
		/* Synchronize disk caches
		 */
		(void) mptscsih_synchronize_cache(hd, 0);

		sz1 = sz3 = 0;

		if (ioc->ScsiLookup != NULL) {
			sz1 = ioc->req_depth * sizeof(void *);
			kfree(ioc->ScsiLookup);
			ioc->ScsiLookup = NULL;
			kfree(ioc->PendingScsi);
			ioc->PendingScsi = NULL;
			dexitprintk((KERN_INFO MYNAM "%s: Free'd ScsiLookup, PendingScsi (%d) memory\n",
				ioc->name, sz1));
		}

		if (hd->memQ != NULL) {
			szQ = host->can_queue * sizeof(MPT_REQUEST_Q);
			kfree(hd->memQ);
			hd->memQ = NULL;
			dexitprintk((KERN_INFO MYNAM "%s: Free'd memQ (%d) memory\n", ioc->name, szQ));
		}

		for (bus = 0; bus < ioc->NumberOfBuses; bus++) {
			if ((pMptTarget = ioc->Target_List[bus])) {
				for (id = 0; id < ioc->DevicesPerBus; id++) {
					if ((pTarget = pMptTarget->Target[id])) {
						dexitprintk((KERN_INFO MYNAM ": %s free bus=%d id=%d pTarget=%p\n",
	 						ioc->name, bus, id, pTarget));
						kfree (pTarget);
						pMptTarget->Target[id] = NULL;
					}
				}
				dexitprintk((KERN_INFO MYNAM ": %s free bus=%d pMptTarget=%p\n",
	 				ioc->name, bus, pMptTarget));
				kfree (pMptTarget);
				ioc->Target_List[bus] = NULL;
			}
		}
		if (ioc->bus_type == SAS)
			mptsas_del_device_components(ioc);
	}
#if !defined(CONFIG_VMNIX)
	/* NULL the Scsi_Host pointer
	 */
	hd->ioc->sh = NULL;
	scsi_unregister(host);
#endif

	if (mpt_scsi_hosts) {
		if (--mpt_scsi_hosts == 0) {
			mpt_reset_deregister(mpt_ScsiDoneCtx);
			dexitprintk((KERN_INFO MYNAM ": Deregistered for IOC reset notifications\n"));

			mpt_event_deregister(mpt_ScsiDoneCtx);
			dexitprintk((KERN_INFO MYNAM ": Deregistered for IOC event notifications\n"));

			mpt_deregister(ScsiScanDvCtx);
			mpt_deregister(ScsiTaskCtx);
			mpt_deregister(mpt_ScsiDoneCtx);

			if (info_kbuf != NULL)
				kfree(info_kbuf);
#ifdef __VMKERNEL_MODULE__
#ifndef MPTSCSIH_DISABLE_DOMAIN_VALIDATION
			spin_lock_destroy(&dvtaskQ_lock);
#endif
			spin_lock_destroy(&scandv_waitq.lock);
#endif
#if (!defined(__VMKERNEL_MODULE__MPT_UNLINKED__) && !defined(__VMNIX__MPT_UNLINKED__))
			mptctl_exit();
			fusion_exit();
#endif

		}
	}

	return 0;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_halt - Process the reboot notification
 *	@nb: Pointer to a struct notifier_block (ignored)
 *	@event: event (SYS_HALT, SYS_RESTART, SYS_POWER_OFF)
 *	@buf: Pointer to a data buffer (ignored)
 *
 *	This routine called if a system shutdown or reboot is to occur.
 *
 *	Return NOTIFY_DONE if this is something other than a reboot message.
 *		NOTIFY_OK if this is a reboot message.
 */
static int
mptscsih_halt(struct notifier_block *nb, ulong event, void *buf)
{
	MPT_ADAPTER *ioc;
	MPT_SCSI_HOST *hd;

	dexitprintk((KERN_INFO MYNAM " %s\n", __FUNCTION__));

	/* Ignore all messages other than reboot message
	 */
	if ((event != SYS_RESTART) && (event != SYS_HALT)
		&& (event != SYS_POWER_OFF))
		return (NOTIFY_DONE);

	list_for_each_entry(ioc, &ioc_list, list) {
		/* Flush the cache of this adapter
		 */
		if (ioc->sh) {
			hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;
			if (hd) {
				mptscsih_synchronize_cache(hd, 0);
			}
		}
	}

	unregister_reboot_notifier(&mptscsih_notifier);
	return NOTIFY_OK;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_info - Return information about MPT adapter
 *	@SChost: Pointer to Scsi_Host structure
 *
 *	(linux Scsi_Host_Template.info routine)
 *
 *	Returns pointer to buffer where information was written.
 */
const char *
mptscsih_info(struct Scsi_Host *SChost)
{
	MPT_SCSI_HOST *h;
	int size = 0;

	if (info_kbuf == NULL)
		if ((info_kbuf = kmalloc(0x1000 /* 4Kb */, GFP_KERNEL)) == NULL)
			return info_kbuf;

	h = (MPT_SCSI_HOST *)SChost->hostdata;
	info_kbuf[0] = '\0';
	if (h) {
		mpt_print_ioc_summary(h->ioc, info_kbuf, &size, 0, 0);
		info_kbuf[size-1] = '\0';
	}

	return info_kbuf;
}

struct info_str {
	char *buffer;
	int   length;
	int   offset;
	int   pos;
};

static void copy_mem_info(struct info_str *info, char *data, int len)
{
	if (info->pos + len > info->length)
		len = info->length - info->pos;

	if (info->pos + len < info->offset) {
		info->pos += len;
		return;
	}

	if (info->pos < info->offset) {
	        data += (info->offset - info->pos);
	        len  -= (info->offset - info->pos);
	}

	if (len > 0) {
                memcpy(info->buffer + info->pos, data, len);
                info->pos += len;
	}
}

static int copy_info(struct info_str *info, char *fmt, ...)
{
	va_list args;
	char buf[81];
	int len;

	va_start(args, fmt);
	len = vsprintf(buf, fmt, args);
	va_end(args);

	copy_mem_info(info, buf, len);
	return len;
}

static int mptscsih_host_info(MPT_ADAPTER *ioc, char *pbuf, off_t offset, int len)
{
	struct info_str info;

	info.buffer	= pbuf;
	info.length	= len;
	info.offset	= offset;
	info.pos	= 0;

	copy_info(&info, "%s: %s, ", ioc->name, ioc->prod_name);
	copy_info(&info, "%s%08xh, ", MPT_FW_REV_MAGIC_ID_STRING, ioc->facts.FWVersion.Word);
	copy_info(&info, "Ports=%d, ", ioc->facts.NumberOfPorts);
	copy_info(&info, "MaxQ=%d\n", ioc->req_depth);

	return ((info.pos > info.offset) ? info.pos - info.offset : 0);
}


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_proc_info - Return information about MPT adapter
 *
 *	(linux Scsi_Host_Template.info routine)
 *
 * 	buffer: if write, user data; if read, buffer for user
 * 	length: if write, return length;
 * 	offset: if write, 0; if read, the current offset into the buffer from
 * 		the previous read.
 * 	hostno: scsi host number
 *	func:   if write = 1; if read = 0
 */
int
mptscsih_proc_info(char *buffer, char **start, off_t offset,
			int length, int hostno, int func)
{
	MPT_ADAPTER	*ioc;
	MPT_SCSI_HOST	*hd = NULL;
	int size = 0;

	dprintk(("Called mptscsih_proc_info: hostno=%d, func=%d\n", hostno, func));
	dprintk(("buffer %p, start=%p (%p) offset=%ld length = %d\n",
			buffer, start, *start, offset, length));

	list_for_each_entry(ioc, &ioc_list, list) {
		if ((ioc->sh) && (ioc->sh->host_no == hostno)) {
			hd = (MPT_SCSI_HOST *)ioc->sh->hostdata;
			break;
		}
	}
	if ((ioc == NULL) || (ioc->sh == NULL) || (hd == NULL))
		return 0;

	if (func) {
//		size = mptscsih_user_command(ioc, buffer, length);
	} else {
		if (start)
			*start = buffer;

		size = mptscsih_host_info(ioc, buffer, offset, length);
	}

	return size;
}


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
#ifdef MPT_DEBUG_QCMD_DEPTH
static void mptscsih_scsi_done(MPT_ADAPTER *ioc, Scsi_Cmnd *SCpnt)
{
	ioc->qcmd_depth--;
	if (SCpnt->scsi_done) {
		SCpnt->scsi_done(SCpnt);
	}
}
#endif

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_qcmd - Primary Fusion MPT SCSI initiator IO start routine.
 *	@SCpnt: Pointer to Scsi_Cmnd structure
 *	@done: Pointer SCSI mid-layer IO completion function
 *
 *	(linux Scsi_Host_Template.queuecommand routine)
 *	This is the primary SCSI IO start routine.  Create a MPI SCSIIORequest
 *	from a linux Scsi_Cmnd request and send it to the IOC.
 *
 *	Returns 0. (rtn value discarded by linux scsi mid-layer)
 */
int
mptscsih_qcmd(Scsi_Cmnd *SCpnt, void (*done)(Scsi_Cmnd *))
{
	MPT_SCSI_HOST		*hd;
	MPT_ADAPTER		*ioc;
	MPT_FRAME_HDR		*mf;
	SCSIIORequest_t		*pScsiReq;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;
	MPT_REQUEST_Q		*buffer;
	unsigned long		 flags;
	int	 bus, id;
	int	 lun;
	int	 datadir;
	u32	 datalen;
	u32	 scsictl;
	u32	 cmd_len;
	int	 my_idx;
	int	 ii;

	hd = (MPT_SCSI_HOST *) SCpnt->host->hostdata;
	ioc = hd->ioc;
	bus = SCpnt->channel;
	id = SCpnt->target;
	lun = SCpnt->lun;
	SCpnt->scsi_done = done;

	pMptTarget = ioc->Target_List[bus];
	pTarget = (VirtDevice *)pMptTarget->Target[id];

#ifdef MPT_DEBUG_QCMD_DEPTH
	ioc->qcmd_depth++;
	if(ioc->qcmd_depth > ioc->sh->can_queue-5 )
		printk("qcmd_depth=%d on %s\n",ioc->qcmd_depth, ioc->name);
#endif

	if ( pTarget ) {
	    	if (pTarget->tflags & MPT_TARGET_FLAGS_DELETED) {
			dsprintk((MYIOC_s_WARN_FMT
				"qcmd: id=%d is DELETED\n",
				ioc->name, id));
			SCpnt->result = DID_NO_CONNECT << 16;
#ifdef MPT_DEBUG_QCMD_DEPTH
			mptscsih_scsi_done(ioc,SCpnt);
#else
			SCpnt->scsi_done(SCpnt);
#endif
			return 0;
		}
#if defined(__VMKERNEL_MODULE__)
		/*
		 * Return success for SCSI Reserve/Release commands
		 * for SATA disks
		 */
		if ( ((SCpnt->cmnd[0] == RESERVE) || (SCpnt->cmnd[0] == RELEASE)) &&
			(pTarget->device_type & MPT_TARGET_DEVICE_TYPE_SATA) ) {
			SCpnt->result =  DID_OK << 16;
			SCpnt->scsi_done(SCpnt);
			return 0;	
		}
#endif
		if ( lun > pTarget->last_lun ) {
			dsprintk((MYIOC_s_WARN_FMT
				"qcmd: lun=%d > last_lun=%d on id=%d\n",
				ioc->name, lun, pTarget->last_lun, id));
			SCpnt->result = DID_BAD_TARGET << 16;
#ifdef MPT_DEBUG_QCMD_DEPTH
			mptscsih_scsi_done(ioc,SCpnt);
#else
			SCpnt->scsi_done(SCpnt);
#endif
			return 0;
		}
		/* Default to untagged. Once a target structure has been
		 * allocated, use the Inquiry data to determine if device
		 * supports tagged.
	 	*/
		if (pTarget->tflags & MPT_TARGET_FLAGS_Q_YES) {
			dioprintk((MYIOC_s_WARN_FMT "qcmd: CDB=%02x id=%d lun=%d tflags has Q_YES \n",
				ioc->name, SCpnt->cmnd[0], id, lun));
			scsictl =  MPI_SCSIIO_CONTROL_SIMPLEQ;
		} else {
			scsictl = MPI_SCSIIO_CONTROL_UNTAGGED;
			dioprintk((MYIOC_s_WARN_FMT "qcmd: CDB=%02x id=%d lun=%d tflags not Q_YES, sending Untagged \n",
				ioc->name, SCpnt->cmnd[0], id, lun));
		}
	} else {
		scsictl = MPI_SCSIIO_CONTROL_UNTAGGED;
		dioprintk((MYIOC_s_WARN_FMT "qcmd: CDB=%02x id=%d lun=%d Null pTarget, sending Untagged\n",
			ioc->name, SCpnt->cmnd[0], id, lun));
	}

	/*
	 *  Put together a MPT SCSI request...
	 */
	if ((mf = mpt_get_msg_frame(mpt_ScsiDoneCtx, ioc)) == NULL) {
		dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
		    ioc->name,__FUNCTION__));
#ifdef MPT_DEBUG_QCMD_DEPTH
		dfailprintk((MYIOC_s_WARN_FMT "QueueCmd: qcmd_depth=%d\n",
				ioc->name, ioc->qcmd_depth));
#endif
#ifdef MFCNT
		dfailprintk((MYIOC_s_WARN_FMT "QueueCmd mfcnt=%d!!\n",
				ioc->name, ioc->mfcnt));
#endif
		goto did_error;
	}

	pScsiReq = (SCSIIORequest_t *) mf;

	my_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);

	/*
	 *  The scsi layer should be handling this stuff
	 *  (In 2.3.x it does -DaveM)
	 */

	/*    TUR's being issued with scsictl=0x02000000 (DATA_IN)!
	 *    Seems we may receive a buffer (datalen>0) even when there
	 *    will be no data transfer!  GRRRRR...
	 */
	datadir = mptscsih_io_direction(SCpnt);
	if (datadir == SCSI_DATA_READ) {
		datalen = SCpnt->request_bufflen;
		scsictl |= MPI_SCSIIO_CONTROL_READ;	/* DATA IN  (host<--ioc<--dev) */
	} else if (datadir == SCSI_DATA_WRITE) {
		datalen = SCpnt->request_bufflen;
		scsictl |= MPI_SCSIIO_CONTROL_WRITE;	/* DATA OUT (host-->ioc-->dev) */
	} else {
		datalen = 0;
		scsictl |= MPI_SCSIIO_CONTROL_NODATATRANSFER;
	}

	/* Use the above information to set up the message frame
	 */
	pScsiReq->TargetID = (u8) id;
	pScsiReq->Bus = (u8) bus;
	pScsiReq->ChainOffset = 0;
	pScsiReq->Function = MPI_FUNCTION_SCSI_IO_REQUEST;
	pScsiReq->CDBLength = SCpnt->cmd_len;
	pScsiReq->SenseBufferLength = MPT_SENSE_BUFFER_SIZE;
	pScsiReq->Reserved = 0;
	pScsiReq->MsgFlags = mpt_msg_flags();
	pScsiReq->LUN[0] = 0;
	pScsiReq->LUN[1] = lun;
	pScsiReq->LUN[2] = 0;
	pScsiReq->LUN[3] = 0;
	pScsiReq->LUN[4] = 0;
	pScsiReq->LUN[5] = 0;
	pScsiReq->LUN[6] = 0;
	pScsiReq->LUN[7] = 0;
	pScsiReq->Control = cpu_to_le32(scsictl);

	/*
	 *  Write SCSI CDB into the message
	 */
	cmd_len = SCpnt->cmd_len;
	for (ii=0; ii < cmd_len; ii++)
		pScsiReq->CDB[ii] = SCpnt->cmnd[ii];

	for (ii=cmd_len; ii < 16; ii++)
		pScsiReq->CDB[ii] = 0;

	/* DataLength */
	pScsiReq->DataLength = cpu_to_le32(datalen);
	/* SenseBuffer low address */
	pScsiReq->SenseBufferLowAddr = cpu_to_le32(ioc->sense_buf_low_dma
					   + (my_idx * MPT_SENSE_BUFFER_ALLOC));

	/* Now add the SG list
	 * Always have a SGE even if null length.
	 */
	if (datalen == 0) {
		/* Add a NULL SGE */
		ioc->add_sge((char *)&pScsiReq->SGL, MPT_SGE_FLAGS_SSIMPLE_READ | 0,
			(dma_addr_t) -1);
	} else {
		/* Add a 32 or 64 bit SGE */
		if ( mptscsih_AddSGE(ioc, SCpnt, pScsiReq, my_idx) != SUCCESS ) {
			dfailprintk((MYIOC_s_WARN_FMT "QueueCmd, SCpnt=%p insufficient Chain Buffers!!\n",
					ioc->name, SCpnt));
			mptscsih_freeChainBuffers(ioc, my_idx);
			mpt_free_msg_frame(ioc, mf);
			goto did_error;
		}
	}
	SCpnt->host_scribble = (unsigned char *)mf;

	if (hd->resetPending) {
		/* Prevent new commands from being issued.
		 */
		dpendprintk((MYIOC_s_WARN_FMT "QueueCmd, SCpnt=%p id=%d resetPending!!\n",
			ioc->name, SCpnt, id));
		goto queueCmd;
	}

	if (ioc->tmState != TM_STATE_NONE) {
		/* Prevent new commands from being issued.
		 */
		dpendprintk((MYIOC_s_WARN_FMT "QueueCmd, SCpnt=%p id=%d tmPending!!\n",
			ioc->name, SCpnt, id));
		goto queueCmd;
	}

#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
	/* revisionID for VM is 1 and this change should exclude VM */
	if (( ioc->bus_type == SPI ) && ( ioc->revisionID > 1 )) {
#else
	if ( ioc->bus_type == SPI ) {
#endif
		if (ioc->spi_data.dvStatus[id] & MPT_SCSICFG_DV_IN_PROGRESS) {
			/* Prevent new commands from being issued while DV is 
			 * running on this ID.
			 */
			dpendprintk((MYIOC_s_WARN_FMT "QueueCmd, SCpnt=%p id=%d mf=%p DV in progress!!\n",
				ioc->name, SCpnt, id, mf));
			goto queueCmd;
		}
	}

	ioc->ScsiLookup[my_idx] = SCpnt;

	mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);
	dmfprintk((MYIOC_s_WARN_FMT "Issued SCSI cmd=%p id=%d idx=%x\n",
			ioc->name, SCpnt, id, my_idx));
	dioprintk((MYIOC_s_WARN_FMT "Issued SCSI cmd=%p id=%d idx=%x\n",
			ioc->name, SCpnt, id, my_idx));
	DBG_DUMP_REQUEST_FRAME(mf)

	return 0;

did_error:
	dfailprintk((MYIOC_s_WARN_FMT "did_error: cmd=%p\n",
		ioc->name, SCpnt));
	/* Just wish OS to issue a retry */
	SCpnt->result = (DID_BUS_BUSY << 16);
#ifdef MPT_DEBUG_QCMD_DEPTH
	mptscsih_scsi_done(ioc,SCpnt);
#else
	SCpnt->scsi_done(SCpnt);
#endif
	return 0;

queueCmd:
	/* Place this command on the pendingQ if possible */
	SPIN_LOCK_IRQSAVE(&ioc->freeQlock, flags);
	ioc->PendingScsi[my_idx] = SCpnt;
	if (!Q_IS_EMPTY(&ioc->freeQ)) {
		buffer = ioc->freeQ.head;
		Q_DEL_ITEM(buffer);

		/* Save the mf pointer
		 */
		buffer->argp = (void *)mf;
#ifdef __VMWARE__
		SCpnt->host_scribble = (unsigned char *)mf;
#endif

		/* Add to the pendingQ
		 */
		Q_ADD_TAIL(&ioc->pendingQ.head, buffer, MPT_REQUEST_Q);
		SPIN_UNLOCK_IRQRESTORE(&ioc->freeQlock, flags);
		return 0;
	} else {
		dpendprintk((MYIOC_s_WARN_FMT "Pending cmd=%p idx %d freeQ is EMPTY!!!\n",
			ioc->name, SCpnt, my_idx));
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
		ioc->PendingScsi[my_idx] = NULL;
#endif
		SPIN_UNLOCK_IRQRESTORE(&ioc->freeQlock, flags);
		SCpnt->result = (DID_BUS_BUSY << 16);
#ifdef MPT_DEBUG_QCMD_DEPTH
		mptscsih_scsi_done(ioc,SCpnt);
#else
		SCpnt->scsi_done(SCpnt);
#endif
		return 0;
	}
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_freeChainBuffers - Function to free chain buffers associated
 *	with a SCSI IO request
 *	@hd: Pointer to the MPT_SCSI_HOST instance
 *	@req_idx: Index of the SCSI IO request frame.
 *
 *	Called if SG chain buffer allocation fails and mptscsih callbacks.
 *	No return.
 */
static void
mptscsih_freeChainBuffers(MPT_ADAPTER *ioc, int req_idx)
{
	MPT_FRAME_HDR *chain;
	unsigned long flags;
	int chain_idx;
	int next;

	/* Get the first chain index and reset
	 * tracker state.
	 */
	chain_idx = ioc->ReqToChain[req_idx];
	ioc->ReqToChain[req_idx] = MPT_HOST_NO_CHAIN;

	while (chain_idx != MPT_HOST_NO_CHAIN) {

		/* Save the next chain buffer index */
		next = ioc->ChainToChain[chain_idx];

		/* Free this chain buffer and reset
		 * tracker
		 */
		ioc->ChainToChain[chain_idx] = MPT_HOST_NO_CHAIN;

		chain = (MPT_FRAME_HDR *) (ioc->ChainBuffer
					+ (chain_idx * ioc->req_sz));
		SPIN_LOCK_IRQSAVE(&ioc->FreeQlock, flags);
		Q_ADD_TAIL(&ioc->FreeChainQ.head,
					&chain->u.frame.linkage, MPT_FRAME_HDR);
		SPIN_UNLOCK_IRQRESTORE(&ioc->FreeQlock, flags);

		dsgprintk((MYIOC_s_WARN_FMT "FreeChainBuffers (index %d)\n",
				ioc->name, chain_idx));

		/* handle next */
		chain_idx = next;
	}
	return;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	Reset Handling
 */

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_TMHandler - Generic handler for SCSI Task Management.
 *	Fall through to mpt_HardResetHandler if: not operational, too many
 *	failed TM requests or handshake failure.
 *
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@type: Task Management type
 *	@target: Logical Target ID for reset (if appropriate)
 *	@lun: Logical Unit for reset (if appropriate)
 *	@ctx2abort: Context for the task to be aborted (if appropriate)
 *	@sleepFlag: If set, use udelay instead of schedule in handshake code.
 *
 *	Remark: Currently invoked from a non-interrupt thread (_bh).
 *
 *	Remark: At most 1 SCSI TaskMgmt function per IOC
 *	will be active.
 *
 *	Returns 0 for SUCCESS or -1 if FAILED.
 */
static int
mptscsih_TMHandler(MPT_SCSI_HOST *hd, u8 type, u8 bus, u8 id, u8 lun, int ctx2abort, int sleepFlag)
{
	MPT_ADAPTER	*ioc;
	int		 rc = -1;
	u32		 ioc_raw_state;
	unsigned long	 flags;

	ioc = hd->ioc;
	dtmprintk((MYIOC_s_WARN_FMT "TMHandler Entered!\n", ioc->name));

	// SJR - CHECKME - Can we avoid this here?
	// (mpt_HardResetHandler has this check...)
	SPIN_LOCK_IRQSAVE(&ioc->diagLock, flags);
	if ((ioc->ioc_reset_in_progress) || (ioc->alt_ioc && ioc->alt_ioc->ioc_reset_in_progress)) {
		dtmprintk((MYIOC_s_WARN_FMT "%s: diagPending=%d!\n", 
			__FUNCTION__, ioc->name, ioc->ioc_reset_in_progress));
		if (ioc->alt_ioc) {
			dtmprintk((MYIOC_s_WARN_FMT "%s: alt_ioc diagPending=%d!\n", 
			__FUNCTION__, ioc->alt_ioc->name, ioc->alt_ioc->ioc_reset_in_progress));
			dtmprintk((MYIOC_s_WARN_FMT "TMHandler Failed!\n", ioc->name));
		}
		SPIN_UNLOCK_IRQRESTORE(&ioc->diagLock, flags);
		return FAILED;
	}
	SPIN_UNLOCK_IRQRESTORE(&ioc->diagLock, flags);

	/* Is operational?
	 */
	ioc_raw_state = mpt_GetIocState(ioc, 0);

	if ((ioc_raw_state & MPI_IOC_STATE_MASK) == MPI_IOC_STATE_OPERATIONAL) {
		/* Issue the Task Mgmt request.
		 */
		if (hd->hard_resets < -1)
			hd->hard_resets++;
		rc = mptscsih_IssueTaskMgmt(hd, type, bus, id, lun, ctx2abort, sleepFlag);
		if (rc) {
			ioc->tmState = TM_STATE_NONE;
			printk(MYIOC_s_WARN_FMT "Issue of TaskMgmt=%x failed!\n", hd->ioc->name, type);
		} else {
			dtmprintk((MYIOC_s_WARN_FMT "Issue of TaskMgmt=%x Successful!\n", 
				 hd->ioc->name, type));
			mpt_post_pendingQ_commands(ioc);
		}
	} else {
		printk(MYIOC_s_WARN_FMT
			"TM Handler for type=%x: IOC Not operational (0x%x)!\n",
			ioc->name, type, ioc_raw_state);
		printk(KERN_WARNING " Issuing HardReset!!\n");
#ifdef MY_PANIC_PROCESSOR_LOOP
		sprintf(panic_msg, "%s: TM Handler for type=%x: IOC Not operational (0x%x)!\n",
			ioc->name, type, ioc_raw_state);
		my_panic (panic_msg);
#endif

#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
/*
 * Under SAS controller + SATA disks with sg3 util format running, controller is 
 * not responding. Abort kicks in and realizes that controller is in
 * FAULTY/NON_OPERATIONAL state. As a result, hard reset is started as part of abort. 
 * Before calling driver abort, SCSI mid-layer acquires drverLock in vmkernel
 * implementation and hostLock for Linux implementation.
 * Calling HardReset with CAN_SLEEP implies schedule_timeout will be invoked.
 * As such, a PSOD will occur due to disabling interrupt. 
 * Current value of sleepFlag is NO_SLEEP so just pass sleepFlag to HardReset.
 * Busy-waiting on hard reset will cause "losing HB" warning and this is expected for
 * hard reset. Eventually, controller is able to recover without PSOD.
 */
		if (mpt_HardResetHandler(ioc, sleepFlag) < 0) {
#else
		if (mpt_HardResetHandler(ioc, CAN_SLEEP) < 0) {
#endif
			printk((KERN_WARNING "TMHandler: HardReset FAILED!!\n"));
		}
		return FAILED;
	}

	/*
	 * Check IOCStatus from TM reply message
	 */
	if (hd->tm_iocstatus == MPI_IOCSTATUS_SUCCESS ||
	    hd->tm_iocstatus == MPI_IOCSTATUS_SCSI_TASK_TERMINATED ||
	    hd->tm_iocstatus == MPI_IOCSTATUS_SCSI_IOC_TERMINATED)
		rc = 0;
	else
		rc = FAILED;

	dtmprintk((MYIOC_s_INFO_FMT "TMHandler rc = %d tm_iocstatus=%08x!\n", 
		ioc->name, rc, hd->tm_iocstatus));

	return rc;
}


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_abort - Abort linux Scsi_Cmnd routine, new_eh variant
 *	@SCpnt: Pointer to Scsi_Cmnd structure, IO to be aborted
 *
 *	(linux Scsi_Host_Template.eh_abort_handler routine)
 *
 *	Returns SUCCESS or FAILED.
 */
int
mptscsih_abort(Scsi_Cmnd * SCpnt)
{
	MPT_SCSI_HOST	*hd = (MPT_SCSI_HOST *)SCpnt->host->hostdata;
	MPT_ADAPTER	*ioc = hd->ioc;
	MPT_FRAME_HDR	*mf;
	u32		 ctx2abort;
	int		 scpnt_idx;
	unsigned long	 sn = SCpnt->serial_number;

	printk(KERN_WARNING MYNAM ": %s: Attempting task abort on id=%d! (sc=%p)\n",
	       ioc->name, SCpnt->target, SCpnt);


	if (hd->resetPending) {
		printk(KERN_WARNING MYNAM ": %s: resetPending!\n",
	       		ioc->name);
		return FAILED;
	}

	if (hd->timeouts < -1)
		hd->timeouts++;

	/* If this command is pended, then timeout/hang occurred
	 * during DV. Post command and flush pending Q
	 * and then following up with the reset request.
	 */
	if ( (mf = mptscsih_search_pendingQ(ioc, SCpnt)) != NULL) {
		/* Cmd was in pendingQ.
		 */
		SCpnt->result = DID_RESET << 16;
		nehprintk((KERN_INFO MYNAM ": %s: mptscsih_abort: "
			   "Command was in the pendingQ! (sc=%p)\n",
			   ioc->name, SCpnt));
#ifdef __VMWARE__
		SCpnt->host_scribble = NULL;
#endif

#ifdef __VMKERNEL_MODULE__
		/*  VMKernel Requires all commands to be completed with scsi_done
		 */
		SCpnt->scsi_done(SCpnt);

#endif

#ifdef __VMWARE__
		mpt_free_msg_frame(ioc, mf);
#endif

		return SUCCESS;
	}

	/* Find this command
	 */
	if ((scpnt_idx = SCPNT_TO_LOOKUP_IDX(SCpnt)) < 0) {
		/* Cmd not found in ScsiLookup.
		 */
#ifdef __VMKERNEL_MODULE__
		SCpnt->result = DID_ERROR << 16;
#else
		SCpnt->result = DID_RESET << 16;
#endif
		nehprintk((KERN_INFO MYNAM ": %s: mptscsih_abort: "
			   "Command not in the active list! (sc=%p)\n",
			   ioc->name, SCpnt));
#ifdef __VMKERNEL_MODULE__
		return SCSI_ABORT_NOT_RUNNING;
#else
		return SUCCESS;
#endif
	} else {
		mf = MPT_INDEX_2_MFPTR(ioc, scpnt_idx);
	}

	/* Most important!  Set TaskMsgContext to SCpnt's MsgContext!
	 * (the IO to be ABORT'd)
	 *
	 * NOTE: Since we do not byteswap MsgContext, we do not
	 *	 swap it here either.  It is an opaque cookie to
	 *	 the controller, so it does not matter. -DaveM
	 */
	ctx2abort = mf->u.frame.hwhdr.msgctxu.MsgContext;

	printk(KERN_WARNING MYNAM ": %s: Abort ctx2abort=%x\n",
	       ioc->name, ctx2abort);

	hd->abortSCpnt = SCpnt;
	if (mptscsih_TMHandler(hd, MPI_SCSITASKMGMT_TASKTYPE_ABORT_TASK,
	                       SCpnt->channel, SCpnt->target, SCpnt->lun, ctx2abort, NO_SLEEP)
		< 0) {

		/* The TM request failed!
		 */
		printk(MYIOC_s_WARN_FMT "Error processing abort task TaskMgmt request\n",
			ioc->name);

		ioc->tmState = TM_STATE_NONE;

		return FAILED;
	}
	if (SCPNT_TO_LOOKUP_IDX(SCpnt) == scpnt_idx &&
	    SCpnt->serial_number == sn) {
		return FAILED;
	}

	return SUCCESS;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_dev_reset - Perform a SCSI TARGET_RESET!  new_eh variant
 *	@SCpnt: Pointer to Scsi_Cmnd structure, IO which reset is due to
 *
 *	(linux Scsi_Host_Template.eh_dev_reset_handler routine)
 *
 *	Returns SUCCESS or FAILED.
 */
int
mptscsih_dev_reset(Scsi_Cmnd * SCpnt)
{
	MPT_SCSI_HOST	*hd = (MPT_SCSI_HOST *)SCpnt->host->hostdata;
	MPT_ADAPTER	*ioc = hd->ioc;

	if (hd->resetPending)
		return FAILED;

	printk(KERN_WARNING MYNAM ": %s: Attempting target reset on id=%d! (sc=%p)\n",
	       ioc->name, SCpnt->target, SCpnt);

	if (mptscsih_TMHandler(hd, MPI_SCSITASKMGMT_TASKTYPE_TARGET_RESET,
	                       SCpnt->channel, SCpnt->target, 0, 0, NO_SLEEP)
		< 0){
		/* The TM request failed!
		 */
		printk(MYIOC_s_WARN_FMT "Error processing target reset TaskMgmt request\n",
			ioc->name);
		ioc->tmState = TM_STATE_NONE;
		return FAILED;
	}
	return SUCCESS;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_bus_reset - Perform a SCSI BUS_RESET!	new_eh variant
 *	@SCpnt: Pointer to Scsi_Cmnd structure, IO which reset is due to
 *
 *	(linux Scsi_Host_Template.eh_bus_reset_handler routine)
 *
 *	Returns SUCCESS or FAILED.
 */
int
mptscsih_bus_reset(Scsi_Cmnd * SCpnt)
{
	MPT_SCSI_HOST	*hd = (MPT_SCSI_HOST *)SCpnt->host->hostdata;
	MPT_ADAPTER	*ioc = hd->ioc;

	printk(KERN_WARNING MYNAM ": %s: Attempting bus reset due to id=%d! (sc=%p)\n",
	       ioc->name, SCpnt->target, SCpnt);

	if (hd->timeouts < -1)
		hd->timeouts++;

	/* We are now ready to execute the task management request. */
	if (mptscsih_TMHandler(hd, MPI_SCSITASKMGMT_TASKTYPE_RESET_BUS,
	                       SCpnt->channel, 0, 0, 0, NO_SLEEP)
	    < 0){

		/* The TM request failed!
		 */
		printk(MYIOC_s_WARN_FMT "Error processing bus reset TaskMgmt request\n",
			ioc->name);
		ioc->tmState = TM_STATE_NONE;
		return FAILED;
	}
	return SUCCESS;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_host_reset - Perform a SCSI host adapter RESET!
 *	@SCpnt: Pointer to Scsi_Cmnd structure, IO which reset is due to
 *
 *	(linux Scsi_Host_Template.eh_host_reset_handler routine)
 *
 *	Returns SUCCESS or FAILED.
 */
int
mptscsih_host_reset(Scsi_Cmnd *SCpnt)
{
	MPT_SCSI_HOST	*hd = (MPT_SCSI_HOST *)SCpnt->host->hostdata;
	MPT_ADAPTER	*ioc = hd->ioc;
	int              status = SUCCESS;

	printk(KERN_WARNING MYNAM ": %s: Attempting host reset due to id=%d! (sc=%p)\n",
	       ioc->name, SCpnt->target, SCpnt);

	/*  If our attempts to reset the host failed, then return a failed
	 *  status.  The host will be taken off line by the SCSI mid-layer.
	 */
	if (mpt_HardResetHandler(hd->ioc, CAN_SLEEP) < 0){
		status = FAILED;
		printk(KERN_WARNING MYNAM ": %s: host_reset: HardReset FAILED\n",
			ioc->name);
	} else {
		/*  Make sure tmState is cleared.
		 */
		ioc->tmState = TM_STATE_NONE;
	}


	drsprintk( ( KERN_INFO MYNAM ": mptscsih_host_reset: "
		     "Status = %s\n",
		     (status == SUCCESS) ? "SUCCESS" : "FAILED" ) );

	return status;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_taskmgmt_complete - Registered with Fusion MPT base driver
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@mf: Pointer to SCSI task mgmt request frame
 *	@mr: Pointer to SCSI task mgmt reply frame
 *
 *	This routine is called from mptbase.c::mpt_interrupt() at the completion
 *	of any SCSI task management request.
 *	This routine is registered with the MPT (base) driver at driver
 *	load/init time via the mpt_register() API call.
 *
 *	Returns 1 indicating alloc'd request frame ptr should be freed.
 */
static int
mptscsih_taskmgmt_complete(MPT_ADAPTER *ioc, MPT_FRAME_HDR *mf, MPT_FRAME_HDR *mr)
{
	SCSITaskMgmtReply_t	*pScsiTmReply;
	SCSITaskMgmt_t		*pScsiTmReq;
	MPT_SCSI_HOST		*hd;
	unsigned long		 flags;
	u16			 iocstatus;
	u8			 tmType;

	dtmprintk((MYIOC_s_WARN_FMT "TaskMgmt completed (mf=%p mr=%p)\n",
			ioc->name, mf, mr));

	SPIN_LOCK_IRQSAVE(&ioc->freeQlock, flags);
	ioc->tmState = TM_STATE_NONE;
	SPIN_UNLOCK_IRQRESTORE(&ioc->freeQlock, flags);

	if (ioc->sh) {
		/* Depending on the thread, a timer is activated for
		 * the TM request.  Delete this timer on completion of TM.
		 * Decrement count of outstanding TM requests.
		 */
		hd = (MPT_SCSI_HOST *)ioc->sh->hostdata;
		if (ioc->tmPtr) {
			DEL_TIMER(&ioc->TMtimer);
		}
		dtmprintk((MYIOC_s_WARN_FMT "taskQcnt (%d)\n",
			ioc->name, hd->taskQcnt));
	} else {
		dtmprintk((MYIOC_s_WARN_FMT "TaskMgmt Complete: NULL Scsi Host Ptr\n",
			ioc->name));
		return 1;
	}

	if (mr == NULL) {
		dtmprintk((MYIOC_s_WARN_FMT "ERROR! NULL TaskMgmt Reply: TMRequest=%p\n",
			ioc->name, mf));
		return 1;
	} else {
		pScsiTmReply = (SCSITaskMgmtReply_t*)mr;
		pScsiTmReq = (SCSITaskMgmt_t*)mf;

		/* Figure out if this was ABORT_TASK, TARGET_RESET, or BUS_RESET! */
		tmType = pScsiTmReq->TaskType;

		dtmprintk((KERN_INFO "  TaskType = %d, TerminationCount=%d\n",
				tmType, le32_to_cpu(pScsiTmReply->TerminationCount)));
		DBG_DUMP_TM_REPLY_FRAME((u32 *)pScsiTmReply);

		iocstatus = le16_to_cpu(pScsiTmReply->IOCStatus) & MPI_IOCSTATUS_MASK;
		hd->tm_iocstatus = iocstatus;

		dtmprintk((MYIOC_s_WARN_FMT "  SCSI TaskMgmt (%d) IOCStatus=%04x IOCLogInfo=%08x\n", 
			ioc->name, tmType, iocstatus, le32_to_cpu(pScsiTmReply->IOCLogInfo))); 
		/* Error?  (anything non-zero?) */
		if (iocstatus) {
			/* clear flags and continue.
			 */
			if (tmType == MPI_SCSITASKMGMT_TASKTYPE_ABORT_TASK)
				hd->abortSCpnt = NULL;

			/* If an internal command is present
			 * or the TM failed - reload the FW.
			 * FC FW may respond FAILED to an ABORT
			 */
			if (tmType == MPI_SCSITASKMGMT_TASKTYPE_RESET_BUS) {
				if ((hd->cmdPtr) ||
				    (iocstatus == MPI_IOCSTATUS_SCSI_TASK_MGMT_FAILED)) {
					if (mpt_HardResetHandler(ioc, CAN_SLEEP) < 0) {
						printk((KERN_WARNING
							" taskmgmt_complete: HardReset FAILED!!\n"));
					}
				}
			}
		} else {
			dtmprintk((MYIOC_s_WARN_FMT " TaskMgmt SUCCESS\n", ioc->name));
			SPIN_LOCK_IRQSAVE(&ioc->freeQlock, flags);
			if (!Q_IS_EMPTY(&ioc->TMpendingQ)) {
				MPT_REQUEST_Q	*buffer;
				buffer = ioc->TMpendingQ.head;
				pScsiTmReq = (SCSITaskMgmt_t *)buffer->argp;
				dtmprintk((MYIOC_s_WARN_FMT "mpt_send_special_message from TMpendingQ MsgContext=%08x type=%d\n",
					ioc->name, pScsiTmReq->TaskMsgContext,
					pScsiTmReq->TaskType));

				Q_DEL_ITEM(buffer);

				/* clear the arg pointer
				 */
				buffer->argp = NULL;

				/* Add to the freeQ
				 */
				Q_ADD_TAIL(&ioc->freeQ.head, buffer, MPT_REQUEST_Q);
				SPIN_UNLOCK_IRQRESTORE(&ioc->freeQlock, flags);
				DBG_DUMP_TM_REQUEST_FRAME((u32 *)pScsiTmReq);

				if ((mpt_send_special_message(
					ScsiTaskCtx, ioc, sizeof(SCSITaskMgmt_t),
					(u32*)pScsiTmReq, 10, NO_SLEEP)) < 0) {
					dtmprintk((MYIOC_s_ERR_FMT "from TMpendingQ send_handshake FAILED! pScsiTMReq=%p\n", ioc->name, pScsiTmReq));
					ioc->tmPtr = NULL;
					mpt_free_msg_frame(ioc, 
						(MPT_FRAME_HDR *)pScsiTmReq);
					printk(KERN_WARNING " Issuing HardReset!!\n");
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
					if (mpt_HardResetHandler(ioc, NO_SLEEP) < 0) {
#else
					if (mpt_HardResetHandler(ioc, CAN_SLEEP) < 0) {
#endif
						printk((KERN_WARNING "IssueTaskMgmt: HardReset FAILED!!\n"));
					}
				}
				return 1;
			}
			SPIN_UNLOCK_IRQRESTORE(&ioc->freeQlock, flags);
			hd->abortSCpnt = NULL;
		}
	}

	return 1;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	This is anyones guess quite frankly.
 */
int
mptscsih_bios_param(Disk * disk, kdev_t dev, int *ip)
{
	unsigned capacity = disk->capacity;
	unsigned size;

	size = capacity;
	ip[0] = 64;				/* heads			*/
	ip[1] = 32;				/* sectors			*/
	if ((ip[2] = size >> 11) > 1024) {	/* cylinders, test for big disk */
		ip[0] = 255;			/* heads			*/
		ip[1] = 63;			/* sectors			*/
		ip[2] = size / (255 * 63);	/* cylinders			*/
	}
	return 0;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	OS entry point to adjust the queue_depths on a per-device basis.
 *	Called once per device the bus scan. Use it to force the queue_depth
 *	member to 1 if a device does not support Q tags.
 */
void
mptscsih_select_queue_depths(struct Scsi_Host *sh, Scsi_Device *sdList)
{
	struct scsi_device	*device;
	MPT_ADAPTER		*ioc;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;
	MPT_SCSI_HOST		*hd;

	for (device = sdList; device != NULL; device = device->next) {

		if (device->host != sh)
			continue;

		hd = (MPT_SCSI_HOST *) sh->hostdata;
		ioc = hd->ioc;
		if (hd == NULL)
			continue;

		if (device->id >= ioc->DevicesPerBus) {
			/* error case, should never happen */
			device->queue_depth = 1;
			dinitprintk((MYIOC_s_WARN_FMT
				 "max_id=%d: scsi%d: Lun=%d: queue_depth=%d\n",
				 hd->ioc->name, ioc->DevicesPerBus, device->id,  device->lun, device->queue_depth));
			continue;
		} else {
			pMptTarget = ioc->Target_List[device->channel];
			pTarget = (VirtDevice *)pMptTarget->Target[device->id];
		}

		if (pTarget) {
			device->queue_depth = MPT_SCSI_CMD_PER_DEV_HIGH;
			if (pTarget->tflags & MPT_TARGET_FLAGS_VALID_INQUIRY) {
				if (!(pTarget->tflags & MPT_TARGET_FLAGS_Q_YES)) {

					device->queue_depth = 1;
					dinitprintk((MYIOC_s_WARN_FMT
			 			"no Q: scsi%d: bus=%d id=%d lun=%d: queue_depth=%d tflags=%x\n",
			 			hd->ioc->name, device->id, pTarget->bus, pTarget->id, device->lun, device->queue_depth, pTarget->tflags));
				} else {
					if ( hd->ioc->bus_type == SPI ) {
						if (((pTarget->inq_data[0] & 0x1f) == 0x00) && (pTarget->minSyncFactor > MPT_ULTRA160 )){
							device->queue_depth = MPT_SCSI_CMD_PER_DEV_LOW;
						}
					}
					dinitprintk((MYIOC_s_WARN_FMT
			 		"scsi%d: bus=%d id=%d lun=%d: queue_depth=%d tflags=%x\n",
		 			hd->ioc->name, device->id, pTarget->bus, pTarget->id, device->lun, device->queue_depth, pTarget->tflags));
				}
				if ((pTarget->inq_data[0] & 0x1f) == SCSI_TYPE_TAPE) {
					device->queue_depth = 1;
				}
			} else {
				/* error case - No Inq. Data */
				device->queue_depth = 1;
				dinitprintk((MYIOC_s_WARN_FMT
			 		"no Inq Data: scsi%d: bus=%d id=%d lun=%d: queue_depth=%d tflags=%x\n",
		 			hd->ioc->name, device->id, pTarget->bus, pTarget->id, device->lun, device->queue_depth, pTarget->tflags));
			}
		} else {
			/* Driver doesn't know about this device.
			 * Kernel may generate a "Dummy Lun 0" which
			 * may become a real Lun if a
			 * "scsi add-single-device" command is executed
			 * while the driver is active (hot-plug a
			 * device).  LSI Raid controllers need
			 * queue_depth set to DEV_HIGH for this reason.
			 */
			device->queue_depth = MPT_SCSI_CMD_PER_DEV_HIGH;
			dinitprintk((MYIOC_s_WARN_FMT
				 "No Driver Device: bus=%d id=%d: Lun=%d: queue_depth=%d\n",
				 hd->ioc->name, device->channel, device->id, device->lun, device->queue_depth));
		}
	}
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *  Private routines...
 */

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* Utility function to copy sense data from the scsi_cmnd buffer
 * to the FC and SCSI target structures.
 *
 */
static void
copy_sense_data(Scsi_Cmnd *sc, MPT_SCSI_HOST *hd, MPT_FRAME_HDR *mf, SCSIIOReply_t *pScsiReply)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice	*pTarget;
	SCSIIORequest_t	*pReq;
	u32		 sense_count = le32_to_cpu(pScsiReply->SenseCount);
	int		 bus, id;

	/* Get target structure
	 */
	pReq = (SCSIIORequest_t *) mf;
	bus = (int) pReq->Bus;
	id = (int) pReq->TargetID;
	pMptTarget = ioc->Target_List[bus];
	pTarget = (VirtDevice *)pMptTarget->Target[id];

	if (sense_count) {
#ifdef MPT_DEBUG_SENSE_DATA
		u8 skey, asc, ascq;
#endif
		u8 *sense_data;
		int req_index;

		/* Copy the sense received into the scsi command block. */
		req_index = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
		sense_data = ((u8 *)hd->ioc->sense_buf_pool + (req_index * MPT_SENSE_BUFFER_ALLOC));
		memcpy(sc->sense_buffer, sense_data, SNS_LEN(sc));

#ifdef MPT_DEBUG_SENSE_DATA
		skey = sense_data[2] & 0x0F;
		asc = sense_data[12];
		ascq = sense_data[13];
		printk(MYIOC_s_WARN_FMT 
			" Id=%d Check Cond SenseKey=%02x ASC=%02x ASCQ=%02x\n",
			hd->ioc->name, index, skey, asc, ascq);
#endif

		/* Log SMART data (asc = 0x5D, non-IM case only) if required.
		 */
		if ((hd->ioc->events) && (hd->ioc->eventTypes & (1 << MPI_EVENT_SCSI_DEVICE_STATUS_CHANGE))) {
			if ((sense_data[12] == 0x5D) && (pTarget->raidVolume == 0)) {
				int idx;
				MPT_ADAPTER *ioc = hd->ioc;

				idx = ioc->eventContext % MPTCTL_EVENT_LOG_SIZE;
				ioc->events[idx].event = MPI_EVENT_SCSI_DEVICE_STATUS_CHANGE;
				ioc->events[idx].eventContext = ioc->eventContext;

				ioc->events[idx].data[0] = (pReq->LUN[1] << 24) ||
					(MPI_EVENT_SCSI_DEV_STAT_RC_SMART_DATA << 16) ||
					(pReq->Bus << 8) || pReq->TargetID;

				ioc->events[idx].data[1] = (sense_data[13] << 8) || sense_data[12];

				ioc->eventContext++;
				/* OEM Specific to light the fault light */
				if (ioc->vendorID == 0x1014) {
					MPT_FRAME_HDR		*mf;
					SEPRequest_t *SEPMsg;
					/* Get a MF for this command.
	 				*/
					if ((mf = mpt_get_msg_frame(ScsiScanDvCtx, hd->ioc)) == NULL) {
						dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
						    ioc->name,__FUNCTION__));
					}
					SEPMsg = (SEPRequest_t *)mf;
					SEPMsg->Function = MPI_FUNCTION_SCSI_ENCLOSURE_PROCESSOR;
					SEPMsg->Bus = pReq->Bus;
					SEPMsg->TargetID = pReq->TargetID;
					SEPMsg->Action = MPI_SEP_REQ_ACTION_WRITE_STATUS;
					SEPMsg->SlotStatus = MPI_SEP_REQ_SLOTSTATUS_PREDICTED_FAULT;
					pTarget->tflags |= MPT_TARGET_FLAGS_LED_ON;
					devtprintk((MYIOC_s_WARN_FMT "Sending SEP LED_ON cmd id=%d bus=%d\n",
						ioc->name, SEPMsg->TargetID, SEPMsg->Bus));
					mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);
				}
			}
		}
	} else {
		dprintk((MYIOC_s_WARN_FMT "Hmmm... SenseData len=0! (?)\n",
				hd->ioc->name));
	}

}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

/* see mptscsih.h */

#ifdef MPT_SCSIHOST_NEED_ENTRY_EXIT_HOOKUPS
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,32)
#if defined(CONFIG_DISKDUMP) || defined(CONFIG_DISKDUMP_MODULE)
	static struct scsi_dump_ops mptscsih_dump_ops = {
		.sanity_check	= mptscsih_sanity_check,
		.quiesce	= mptscsih_quiesce,
		.poll		= mptscsih_poll
	};
	static Scsi_Host_Template_dump driver_template_dump = MPT_SCSIHOST;
#define driver_template	(driver_template_dump.hostt)
#else
	static Scsi_Host_Template driver_template = MPT_SCSIHOST;
#endif
#else
	static Scsi_Host_Template driver_template = MPT_SCSIHOST;
#endif
#	include "scsi_module.c"
#endif

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* Search the pendingQ for a command with specific index.
 * If found, delete and return mf pointer
 * If not found, return NULL
 */
static MPT_FRAME_HDR *
mptscsih_search_pendingQ(MPT_ADAPTER *ioc, Scsi_Cmnd * sc)
{
	unsigned long	 flags;
	MPT_REQUEST_Q	*buffer;
	MPT_FRAME_HDR	*mf = NULL;
	u16		 req_idx;

	SPIN_LOCK_IRQSAVE(&ioc->freeQlock, flags);
	if (!Q_IS_EMPTY(&ioc->pendingQ)) {
		buffer = ioc->pendingQ.head;
		do {
			mf = (MPT_FRAME_HDR *) buffer->argp;
			req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
			if (sc == ioc->PendingScsi[req_idx]) {
				dpendprintk((MYIOC_s_WARN_FMT ": search_pendingQ mf=%p found\n", ioc->name, mf));
				Q_DEL_ITEM(buffer);

				/* clear the arg pointer
				 */
				buffer->argp = NULL;

				/* Add to the freeQ
				 */
				Q_ADD_TAIL(&ioc->freeQ.head, buffer, MPT_REQUEST_Q);
				break;
			}
			mf = NULL;
		} while ((buffer = buffer->forw) != (MPT_REQUEST_Q *) &ioc->pendingQ);
	}
	SPIN_UNLOCK_IRQRESTORE(&ioc->freeQlock, flags);
	return mf;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
static int
mptscsih_ioc_reset(MPT_ADAPTER *ioc, int reset_phase)
{
	MPT_SCSI_HOST	*hd;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice	*pTarget;
	unsigned long	 flags;
	int		 ii, bus, id;
	int		 n;

	drsprintk((KERN_INFO MYNAM
			": IOC %s_reset routed to SCSI host driver!\n",
			reset_phase==MPT_IOC_SETUP_RESET ? "setup" : (
			reset_phase==MPT_IOC_PRE_RESET ? "pre" : "post")));

	/* If a FW reload request arrives after base installed but
	 * before all scsi hosts have been attached, then an alt_ioc
	 * may have a NULL sh pointer.
	 */
	if ((ioc->sh == NULL) || (ioc->sh->hostdata == NULL))
		return 0;
	else
		hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;

	if (reset_phase == MPT_IOC_SETUP_RESET) {
		drsprintk((MYIOC_s_WARN_FMT "Setup-Diag Reset\n", ioc->name));
		/* Clean Up:
		 * 1. Set Hard Reset Pending Flag
		 */
		hd->resetPending = 1;

		/* 2. Reset timeouts on all running commands
		 */
		mptscsih_reset_timeouts (hd);

	} else if (reset_phase == MPT_IOC_PRE_RESET) {
		drsprintk((MYIOC_s_WARN_FMT "Pre-Diag Reset\n", ioc->name));

		/* 2. Flush running commands
		 *	Clean ScsiLookup (and associated memory)
		 *	AND clean mytaskQ
		 */

		/* 2b. Reply to OS all known outstanding I/O commands.
		 */
		mptscsih_flush_running_cmds(hd);

		/* 2c. If there was an internal command that
		 * has not completed, configuration or io request,
		 * free these resources.
		 */
		if (hd->cmdPtr) {
			DEL_TIMER(&hd->timer);
			mpt_free_msg_frame(ioc, hd->cmdPtr);
		}

		/* 2d. If a task management has not completed,
		 * free resources associated with this request.
		 */
		if (ioc->tmPtr) {
			DEL_TIMER(&ioc->TMtimer);
			mpt_free_msg_frame(ioc, ioc->tmPtr);
		}

		drsprintk((MYIOC_s_WARN_FMT "Pre-Reset complete.\n", ioc->name));
	} else {
		drsprintk((MYIOC_s_WARN_FMT "Post-Diag Reset\n", ioc->name));

		if (ioc->bus_type == FC) {
			n = 0;
			for (bus = 0; bus < ioc->NumberOfBuses; bus++) {
				pMptTarget = ioc->Target_List[bus];
				for (id=0; id < ioc->DevicesPerBus; id++) {
					pTarget = pMptTarget->Target[id];
					if (pTarget) {
						dsprintk((MYIOC_s_INFO_FMT
							"bus=%d id=%d is known to be WWPN %08x%08x, WWNN %08x%08x\n",
							ioc->name, bus, id,
							le32_to_cpu(pTarget->WWPN.High),
							le32_to_cpu(pTarget->WWPN.Low),
							le32_to_cpu(pTarget->WWNN.High),
							le32_to_cpu(pTarget->WWNN.Low)));
						mptscsih_writeFCPortPage3(hd, id);
						n++;
					}
				}
			}

			if (n) {
				mptscsih_sendIOCInit(hd);
			}
		}

		 /* Init all control structures.
		 */

		/* ScsiLookup initialization
		 */
		for (ii=0; ii < hd->ioc->req_depth; ii++)
			ioc->ScsiLookup[ii] = NULL;

		/* 2. tmPtr clear
		 */
		if (ioc->tmPtr) {
			ioc->tmPtr = NULL;
		}

		/* 3. Renegotiate to all devices, if SCSI
		 */

		if (ioc->bus_type == SPI) {
			dnegoprintk(("writeSDP1: ALL_IDS USE_NVRAM\n"));
			mptscsih_writeSDP1(hd, 0, 0, MPT_SCSICFG_ALL_IDS | MPT_SCSICFG_USE_NVRAM);
		}

		/* 4. Enable new commands to be posted
		 */
		SPIN_LOCK_IRQSAVE(&ioc->freeQlock, flags);
		hd->resetPending = 0;
		ioc->tmState = TM_STATE_NONE;
		dtmprintk((MYIOC_s_WARN_FMT "%s: tmState set to TM_STATE_NONE!\n", 
			ioc->name, __FUNCTION__));
		SPIN_UNLOCK_IRQRESTORE(&ioc->freeQlock, flags);

		/* 5. If there was an internal command,
		 * wake this process up.
		 */
		if (hd->cmdPtr) {
			/*
			 * Wake up the original calling thread
			 */
			hd->pLocal = &hd->localReply;
			hd->pLocal->completion = MPT_SCANDV_DID_RESET;
			scandv_wait_done = 1;
			WAKE_UP(&scandv_waitq);
			hd->cmdPtr = NULL;
		}

		/* 6. Set flag to force DV and re-read IOC Page 3
		 */
		if (ioc->bus_type == SPI) {
			ioc->spi_data.forceDv = MPT_SCSICFG_NEED_DV |
			    MPT_SCSICFG_RELOAD_IOC_PG3;
			drsprintk(("Set reload IOC Pg3 Flag\n"));
		}

		mpt_post_pendingQ_commands(ioc);

		drsprintk((MYIOC_s_WARN_FMT "Post-Reset complete.\n", ioc->name));
	}

	return 1;		/* currently means nothing really */
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
static int
mptscsih_event_process(MPT_ADAPTER *ioc, EventNotificationReply_t *pEvReply)
{
	MPT_SCSI_HOST *hd;
	u8 event = le32_to_cpu(pEvReply->Event) & 0xFF;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;
	int	 bus, id;

//	devtprintk((MYIOC_s_WARN_FMT "MPT event (=%02Xh) routed to SCSI host driver!\n",
//			ioc->name, event));

	if (ioc->bus_type == SAS)
		return mptsas_event_process(ioc, pEvReply);

	switch (event) {
	case MPI_EVENT_UNIT_ATTENTION:			/* 03 */
		/* FIXME! */
		break;
	case MPI_EVENT_IOC_BUS_RESET:			/* 04 */
	case MPI_EVENT_EXT_BUS_RESET:			/* 05 */
		hd = NULL;
		if (ioc->sh) {
			hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;
			if (hd && (ioc->bus_type == SPI) && (hd->soft_resets < -1))
				hd->soft_resets++;
		}
		break;
	case MPI_EVENT_LOGOUT:				/* 09 */
		/* FIXME! */
		break;

		/*
		 *  CHECKME! Don't think we need to do
		 *  anything for these, but...
		 */
	case MPI_EVENT_RESCAN:				/* 06 */
	case MPI_EVENT_LINK_STATUS_CHANGE:		/* 07 */
	case MPI_EVENT_LOOP_STATE_CHANGE:		/* 08 */
		/*
		 *  CHECKME!  Falling thru...
		 */
		break;

	case MPI_EVENT_INTEGRATED_RAID:			/* 0B */
	{
#ifdef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
                pMpiEventDataRaid_t pRaidEventData =
                    (pMpiEventDataRaid_t) pEvReply->Data;
		hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;
	        bus = pRaidEventData->VolumeBus;
	        id = pRaidEventData->VolumeID;

                /* Domain Validation Needed */
	        if (ioc->bus_type == SPI &&
        	                pRaidEventData->ReasonCode ==
        	                MPI_EVENT_RAID_RC_DOMAIN_VAL_NEEDED) {

				pMptTarget = ioc->Target_List[bus];
				pTarget = (VirtDevice *)pMptTarget->Target[id];
				if (pTarget) {
        				ScsiCfgData	*pSpi;
					ddvprintk((KERN_INFO " Raid Event: DV Scheduled for bus=%d id=%d pTarget=%p\n",
						bus, id, pTarget));
					pSpi = &ioc->spi_data;
					pSpi->dvStatus[id] |= MPT_SCSICFG_NEED_DV;

					pSpi->forceDv |= MPT_SCSICFG_RELOAD_IOC_PG3;
					MPT_INIT_WORK(&pTarget->dvTask, mptscsih_domainValidation, (void *) pTarget);
					SCHEDULE_TASK(&pTarget->dvTask);
				} else {
					ddvprintk((KERN_INFO " Raid Event: DV Requested for bus=%d id=%d but pTarget is NULL\n",
						bus, id));
				}
        	        }
#endif

#if defined(MPT_DEBUG_EVENTS)
		u32 *m = (u32 *)pEvReply;
		int ii;
		int n = (int)pEvReply->MsgLength;
		printk("Raid Event Reply: MessageLength=%d:\n", n);
		for (ii=0; ii < n; ii++) {
			if (ii && ((ii%8)==0))
				printk("\n");
			printk(" %08x", le32_to_cpu(m[ii]));
		}
		printk("\n");
#endif
		break;
	}

	default:
		devtprintk((KERN_INFO "  Ignoring event (=%02Xh)\n", event));
		break;
	}

	return 1;		/* currently means nothing really */
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_initTarget - Target, LUN alloc/free functionality.
 *	@hd: Pointer to MPT_SCSI_HOST structure
 *	@bus: Bus number
 *	@id: SCSI id
 *	@lun: SCSI LUN id
 *	@data: Pointer to data
 *	@dlen: Number of INQUIRY bytes
 *
 *	NOTE: It's only SAFE to call this routine if data points to
 *	sane & valid STANDARD INQUIRY data!
 *
 *	Allocate and initialize memory for this target.
 *	Save inquiry data.
 *
 */
static void
mptscsih_initTarget(MPT_SCSI_HOST *hd, int bus, int id, u8 lun, char *data, int dlen)
{
        MPT_ADAPTER     *ioc = hd->ioc;
	int		indexed_lun, lun_index;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice	*pTarget;
	ScsiCfgData	*pSpi;
	char		data_56;

	dinitprintk((MYIOC_s_WARN_FMT "initTarget bus=%d id=%d lun=%d hd=%p\n",
			ioc->name, bus, id, lun, hd));

	/*
	 * If the peripheral qualifier filter is enabled and the target
	 * reports a Peripheral Qualifier (upper 3 bits of byte 0) of 0x1
	 * (Target is capable of supporting the specified peripheral
	 * device type on this logical unit; however, the physical device is
	 * not currently connected to this logical unit), the Peripherial
	 * Qualifier will be forced to 0x3 (Target is not capable of 
	 * supporting a physical device on this logical unit).  This is to 
	 * work around a bug in the mid-layer in some distributions in which 
	 * the mid-layer will continue to try to communicate to the LUN and 
	 * eventually create a dummy LUN.
	 */
	if (mpt_pq_filter && dlen && (data[0] & 0x20))
		data[0] |= 0x40;

	/* Is LUN supported? If so, upper 2 bits will be 0
	* in first byte of inquiry data.
	*/
	if (dlen && (data[0] & 0xe0))
		return;

	pSpi = &ioc->spi_data;
	if ( bus >= ioc->NumberOfBuses) {
		printk(MYIOC_s_WARN_FMT "initTarget Invalid bus=%d id=%d lun=%d NumberOfBuses=%d\n",
			ioc->name, bus, id, lun, ioc->NumberOfBuses);
		return;
	}
	pMptTarget = ioc->Target_List[bus];
	if ( id >= ioc->DevicesPerBus) {
		printk(MYIOC_s_WARN_FMT "initTarget bus=%d Invalid id=%d lun=%d DevicesPerBus=%d\n",
			ioc->name, bus, id, lun, ioc->DevicesPerBus);
		return;
	}
	pTarget = (VirtDevice *)pMptTarget->Target[id];
	if (pTarget == NULL) {
		if ((pTarget = kmalloc(sizeof(VirtDevice), GFP_ATOMIC)) == NULL) {
			printk(MYIOC_s_ERR_FMT "initTarget kmalloc(%d) FAILED!\n",
					ioc->name, (int)sizeof(VirtDevice));
			return;
		}
		memset(pTarget, 0, sizeof(VirtDevice));
		pTarget->tflags = MPT_TARGET_FLAGS_Q_YES;
		pTarget->ioc = ioc;
		pTarget->id = id;
		pTarget->bus = bus;
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
		if (ioc->bus_type == SPI) {
			pTarget->last_lun = MPT_SCSI_LAST_LUN;
		} else {
			pTarget->last_lun = MPT_LAST_LUN;
		}
#else
		pTarget->last_lun = MPT_LAST_LUN;
#endif
		pTarget->raidVolume = 0;
		pMptTarget->Target[id] = pTarget;
		if (ioc->bus_type == FC) {
			mptscsih_readFCDevicePage0(hd, id);
		} else {
			if (ioc->bus_type == SPI) {
				pSpi->dvStatus[id] |= MPT_SCSICFG_NEED_DV;
				if ( (dlen && data[0] == SCSI_TYPE_PROC) && (ioc->spi_data.Saf_Te) ) {
		       			/* Treat all Processors as SAF-TE if
				 	 * command line option is set */
					pTarget->tflags |= MPT_TARGET_FLAGS_SAF_TE_ISSUED;
					mptscsih_writeIOCPage4(ioc, id, bus);
				}
			}
			if (ioc->raid_data.isRaid & (1 << id)) {
				pTarget->raidVolume = 1;
				dinitprintk((KERN_INFO "RAID Volume @ id %d\n", id));
			}
		}

		if (ioc->bus_type == SAS) {
			struct mptsas_hotplug_event *ev;

			ev = kmalloc(sizeof(*ev), GFP_ATOMIC);
			if (ev) {
				memset(ev, 0, sizeof(*ev));
				ev->ioc = ioc;
				ev->id = id;
				ev->channel = bus;
				if (ioc->raid_data.isRaid & (1 << id))
					ev->refresh_raid_config_pages = 1;
				ev->event_type = MPTSAS_INIT_TARGET;
				MPT_INIT_WORK(&ev->work, mptsas_hotplug_work, ev);
				SCHEDULE_TASK(&ev->work);
			}
		}
		dinitprintk((KERN_INFO "  *NEW* Target structure (bus=%d id=%d) @ %p\n",
			bus, id, pTarget));
//		printk(KERN_INFO "  *NEW* Target structure (bus=%d id=%d) @ %p\n",
//			bus, id, pTarget);
	} else {
	 	pTarget->tflags &= ~MPT_TARGET_FLAGS_DELETED;
	}

	lun_index = (lun >> 5);  /* 32 luns per lun_index */
	indexed_lun = (lun % 32);
	pTarget->luns[lun_index] |= (1 << indexed_lun);

	if ( dlen == 0 )
		return;  /* No Inquiry data */

#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
        if((pTarget->scsi_level == 0) && (data[2] & 0x07)) {
           pTarget->scsi_level = data[2] & 0x07;
        }
        /*
         * scsi layer support sparseluns scan regardless scsi level. However,
         * controller will hang if lun scan is beyond lun 8 on a scsi2 device.
         * So, we keep track max_lun for each target and reject a scsi
         * command if the lun is over the max lun limit. 
         */
        if((ioc->bus_type == SPI) &&
           (pTarget->scsi_level > 0) &&
           (pTarget->scsi_level <= SCSI_LEVEL2)) {
           pTarget->last_lun = MPT_SCSI2_LAST_LUN;
        }
#endif
	if ( (ioc->bus_type == SPI) && (data[0] == SCSI_TYPE_PROC) &&
		!(pTarget->tflags & MPT_TARGET_FLAGS_SAF_TE_ISSUED )) {
		if ( dlen > 49 ) {
//			pTarget->tflags |= MPT_TARGET_FLAGS_VALID_INQUIRY;
			if ( data[44] == 'S' &&
			     data[45] == 'A' &&
			     data[46] == 'F' &&
			     data[47] == '-' &&
			     data[48] == 'T' &&
			     data[49] == 'E' ) {
				dinitprintk((KERN_INFO " Saf_Te Processor at id %d\n",
					id));
				pTarget->tflags |= MPT_TARGET_FLAGS_SAF_TE_ISSUED;
				mptscsih_writeIOCPage4(ioc, id, bus);
			}
		}
	} else if (ioc->bus_type == SAS) {
		if ( (data[7] & 0x02) == 0) {
			pTarget->tflags &= ~MPT_TARGET_FLAGS_Q_YES;
			dinitprintk((MYIOC_s_WARN_FMT
				"SAS device not Queuing bus=%d id=%d data[7]=%02x dlen=%d\n",
				hd->ioc->name, pTarget->bus, pTarget->id, data[7], dlen));
		}

		if ((data[0] == TYPE_TAPE)) {
			dinitprintk((KERN_INFO "SAS TAPE id=%d\n",
				id));
			if (ioc->facts.IOCCapabilities & 
				MPI_IOCFACTS_CAPABILITY_TLR ) {
				dinitprintk((KERN_INFO "SAS TLR Capable\n"));
				if ((pTarget->tflags & MPT_TARGET_FLAGS_TLR_DONE) == 0) {
					if ( data[8]  == 'H' &&
					     data[9]  == 'P' &&
					     data[10] == ' ' &&
					     data[11] == ' ' &&
					     data[12] == ' ' &&
					     data[13] == ' ' &&
					     data[14] == ' ' &&
					     data[15] == ' ' ) {
						dinitprintk((KERN_INFO "Issue TLR to SAS TAPE id=%d\n",
							id));
						MPT_INIT_WORK(&pTarget->TLRTask, mpt_IssueTLR, (void *) pTarget);
						SCHEDULE_TASK(&pTarget->TLRTask);
						pTarget->tflags |= MPT_TARGET_FLAGS_TLR_DONE;
					}
				}
			}
		}
	}
	if ( dlen > 8 ) {
		memcpy (pTarget->inq_data, data, 8);
	} else {
		memcpy (pTarget->inq_data, data, dlen);
	}

	if (ioc->bus_type == SPI) {
		if (!(pTarget->tflags & MPT_TARGET_FLAGS_VALID_INQUIRY)) {
			pTarget->tflags |= MPT_TARGET_FLAGS_VALID_INQUIRY;

			data_56 = 0x00;  /* Default to no Ultra 160 or 320 capabilities if Inq data length is < 57 */
			if (dlen > 56) {
				if ( (!(pTarget->tflags & MPT_TARGET_FLAGS_VALID_56))) {
				/* Update the target capabilities
				 */
					data_56 = data[56];
					pTarget->tflags |= MPT_TARGET_FLAGS_VALID_56;
				}
			}
			mptscsih_setTargetNegoParms(hd, pTarget, data_56);
			if ((pSpi->dvStatus[id] & MPT_SCSICFG_NEED_DV) ||
				(ioc->spi_data.forceDv & MPT_SCSICFG_NEED_DV)) {
//				unsigned long lflags;
				/* Schedule DV if necessary */
//				SPIN_LOCK_IRQSAVE(&dvtaskQ_lock, lflags);
//				if (!dvtaskQ_active) {
//					dvtaskQ_active = 1;
//					SPIN_UNLOCK_IRQRESTORE(&dvtaskQ_lock, lflags);
					ddvprintk((MYIOC_s_WARN_FMT " DV Scheduled for id=%d\n",
						ioc->name, id));
//					printk(KERN_INFO " DV Scheduled for id %d\n",
//						id);
					MPT_INIT_WORK(&pTarget->dvTask, mptscsih_domainValidation, (void *) pTarget);
					SCHEDULE_TASK(&pTarget->dvTask);
//				} else {
//					SPIN_UNLOCK_IRQRESTORE(&dvtaskQ_lock, lflags);
//				}
				ioc->spi_data.forceDv &= ~MPT_SCSICFG_NEED_DV;
			}
		} else {
			/* Initial Inquiry may not request enough data bytes to
			 * obtain byte 57.  DV will; if target doesn't return 
			 * at least 57 bytes, data[56] will be zero. */
			if (dlen > 56) {
				if ( (!(pTarget->tflags & MPT_TARGET_FLAGS_VALID_56))) {
				/* Update the target capabilities
				 */
					data_56 = data[56];
					pTarget->tflags |= MPT_TARGET_FLAGS_VALID_56;
					mptscsih_setTargetNegoParms(hd, pTarget, data_56);
				}
			}
		}
	} else {
		pTarget->tflags |= MPT_TARGET_FLAGS_VALID_INQUIRY;
	}
}

static void
mpt_IssueTLR(void *arg)
{
	VirtDevice	*pTarget=(VirtDevice *)arg;
	MPT_SCSI_HOST	*hd;
	MPT_ADAPTER		*ioc;
	INTERNAL_CMD		 iocmd;
	int			 lun, indexed_lun, lun_index;

	ioc = pTarget->ioc;
	hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;
	iocmd.id = pTarget->id;
	iocmd.bus = pTarget->bus;
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
	int last_lun;
	if (ioc->bus_type == SPI) {
		last_lun = MPT_SCSI_LAST_LUN;
	} else {
		last_lun = MPT_LAST_LUN;
	}
	for (lun=0; lun <= last_lun; lun++) {
#else
	for (lun=0; lun <= MPT_LAST_LUN; lun++) {
#endif
		/* If LUN present, issue the command
		 */
		lun_index = (lun >> 5);  /* 32 luns per lun_index */
		indexed_lun = (lun % 32);
		if (pTarget->luns[lun_index] & (1<<indexed_lun)) {
			iocmd.lun = lun;
			goto issueTLR;
		}
	}
	printk(MYIOC_s_INFO_FMT "mpt_IssueTLR: Unable find a lun on id=%d\n",
		ioc->name, iocmd.id);
	return;
issueTLR:
	iocmd.flags = 0;
	iocmd.physDiskNum = -1;
	iocmd.rsvd = iocmd.rsvd2 = 0;
	iocmd.cmd = OEM_TLR_COMMAND;
	iocmd.data_dma = -1;
	iocmd.data = NULL;
	iocmd.size = 0;
	printk(MYIOC_s_INFO_FMT "mpt_IssueTLR do_cmd=%x id=%d\n",
		ioc->name, iocmd.cmd, iocmd.id);
	if (mptscsih_do_cmd(hd, &iocmd) < 0) {
		if (mptscsih_do_cmd(hd, &iocmd) < 0) {
			printk(MYIOC_s_INFO_FMT "Unable to set TLR on id=%d\n",
				hd->ioc->name, iocmd.id);
		}
	}
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *  Update the target negotiation parameters based on the
 *  the Inquiry data, adapter capabilities, and NVRAM settings.
 *
 */
static void 
mptscsih_setTargetNegoParms(MPT_SCSI_HOST *hd, VirtDevice *pTarget, char byte56)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	ScsiCfgData *pspi_data = &hd->ioc->spi_data;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice	*loop_pTarget;
	int  id = (int) pTarget->id;
	int  nvram;
	int ii;
	u8 width = MPT_NARROW;
	u8 factor = MPT_ASYNC;
	u8 offset = 0;
	u8 version, nfactor;
	u8 noQas = 1;

	pTarget->negoFlags = pspi_data->noQas;

	/* noQas == 0 => device supports QAS. Need byte 56 of Inq to determine
	 * support. If available, default QAS to off and allow enabling.
	 * If not available, default QAS to on, turn off for non-disks.
	 */

	/* Set flags based on Inquiry data
	 */
	version = pTarget->inq_data[2] & 0x07;
	if (version < 2) {
		width = 0;
		factor = MPT_ULTRA2;
		offset = pspi_data->maxSyncOffset;
		pTarget->tflags &= ~MPT_TARGET_FLAGS_Q_YES;
		dinitprintk((MYIOC_s_WARN_FMT
			"device version=%d bus=%d id=%d: inq_data[2]=%02x set to not Queuing\n",
			hd->ioc->name, version, pTarget->bus, pTarget->id, 
			pTarget->inq_data[2]));
	} else {
		if (pTarget->inq_data[7] & 0x20) {
			width = 1;
		}

		if (pTarget->inq_data[7] & 0x10) {
			factor = pspi_data->minSyncFactor;
			if (pTarget->tflags & MPT_TARGET_FLAGS_VALID_56) {
				/* bits 2 & 3 show Clocking support */
				if ((byte56 & 0x0C) == 0)
					factor = MPT_ULTRA2;
				else {
					if ((byte56 & 0x03) == 0)
						factor = MPT_ULTRA160;
					else {
						factor = MPT_ULTRA320;
						if (byte56 & 0x02)
						{
							ddvprintk((KERN_INFO "Enabling QAS due to byte56=%02x on id=%d!\n", byte56, id));
							noQas = 0;
						}
						if (pTarget->inq_data[0] == SCSI_TYPE_TAPE) {
							if (byte56 & 0x01)
								pTarget->negoFlags |= MPT_TAPE_NEGO_IDP;
						}
					}
				}
			} else {
				ddvprintk((KERN_INFO "Ultra 80 on id=%d due to ~TARGET_FLAGS_VALID_56!\n", id));
				factor = MPT_ULTRA2;
			}
				
			offset = pspi_data->maxSyncOffset;

			/* If RAID, never disable QAS
			 * else if non RAID, do not disable
			 *   QAS if bit 1 is set
			 * bit 1 QAS support, non-raid only
			 * bit 0 IU support
			 */
			if (pTarget->raidVolume == 1) {
				noQas = 0;
			}
		} else {
			factor = MPT_ASYNC;
			offset = 0;
		}
	}

	if ( (pTarget->inq_data[7] & 0x02) == 0) {
		pTarget->tflags &= ~MPT_TARGET_FLAGS_Q_YES;
	}

	/* Update tflags based on NVRAM settings. (SCSI only)
	 */
	if (pspi_data->nvram && (pspi_data->nvram[id] != MPT_HOST_NVRAM_INVALID)) {
		nvram = pspi_data->nvram[id];
		nfactor = (nvram & MPT_NVRAM_SYNC_MASK) >> 8;

		if (width)
			width = nvram & MPT_NVRAM_WIDE_DISABLE ? 0 : 1;

		if (offset > 0) {
			/* Ensure factor is set to the
			 * maximum of: adapter, nvram, inquiry
			 */
			if (nfactor) {
				if (nfactor < pspi_data->minSyncFactor )
					nfactor = pspi_data->minSyncFactor;

				factor = max(factor, nfactor);
				if (factor == MPT_ASYNC)
					offset = 0;
			} else {
				offset = 0;
				factor = MPT_ASYNC;
			}
		} else {
			factor = MPT_ASYNC;
		}
	}

	/* Make sure data is consistent
	 */
	if ((!width) && (factor < MPT_ULTRA2)) {
		factor = MPT_ULTRA2;
	}

	/* Save the data to the target structure.
	 */
	pTarget->minSyncFactor = factor;
	pTarget->maxOffset = offset;
	pTarget->maxWidth = width;

	pTarget->tflags |= MPT_TARGET_FLAGS_VALID_NEGO;

	/* Disable unused features.
	 */
	if (!width)
		pTarget->negoFlags |= MPT_TARGET_NO_NEGO_WIDE;

	if (!offset)
		pTarget->negoFlags |= MPT_TARGET_NO_NEGO_SYNC;

	if ( factor > MPT_ULTRA320 )
		noQas = 0;

	/* GEM, processor WORKAROUND
	 */
	if ((pTarget->inq_data[0] == SCSI_TYPE_PROC) || (pTarget->inq_data[0] > 0x08)) {
		pTarget->negoFlags |= (MPT_TARGET_NO_NEGO_WIDE | MPT_TARGET_NO_NEGO_SYNC);
		pspi_data->dvStatus[id] |= MPT_SCSICFG_BLK_NEGO;
	} else {
		if (noQas && (pspi_data->noQas == 0)) {
			pspi_data->noQas |= MPT_TARGET_NO_NEGO_QAS;
			pTarget->negoFlags |= MPT_TARGET_NO_NEGO_QAS;

			/* Disable QAS in a mixed configuration case
	 		*/
			pMptTarget = ioc->Target_List[0];
			for (ii = 0; ii < id; ii++) {
				loop_pTarget = pMptTarget->Target[id];
				if ((loop_pTarget)) {
					ddvprintk((KERN_INFO "Disabling QAS on id=%d due to noQas=%02x on id=%d!\n", ii, noQas, id));
					loop_pTarget->negoFlags |= MPT_TARGET_NO_NEGO_QAS;
					mptscsih_writeSDP1(hd, 0, ii, loop_pTarget->negoFlags);
				}
			}
		}
	}

	/* Write SDP1 on this I/O to this target */
	if (pspi_data->dvStatus[id] & MPT_SCSICFG_NEGOTIATE) {
		ddvprintk((KERN_INFO "MPT_SCSICFG_NEGOTIATE on id=%d!\n", id));
		mptscsih_writeSDP1(hd, 0, id, hd->negoNvram);
		pspi_data->dvStatus[id] &= ~MPT_SCSICFG_NEGOTIATE;
	} else if (pspi_data->dvStatus[id] & MPT_SCSICFG_BLK_NEGO) {
		ddvprintk((KERN_INFO "MPT_SCSICFG_BLK_NEGO on id=%d!\n", id));
		mptscsih_writeSDP1(hd, 0, id, MPT_SCSICFG_BLK_NEGO);
		pspi_data->dvStatus[id] &= ~MPT_SCSICFG_BLK_NEGO;
	}
}


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 * If no Target, bus reset on 1st I/O. Set the flag to
 * prevent any future negotiations to this device.
 */
static void mptscsih_no_negotiate(MPT_SCSI_HOST *hd, int id)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;

	pMptTarget = ioc->Target_List[0];
	pTarget = (VirtDevice *)pMptTarget->Target[id];
	if (pTarget == NULL)
		hd->ioc->spi_data.dvStatus[id] |= MPT_SCSICFG_BLK_NEGO;
	return;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *  SCSI Config Page functionality ...
 */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_setDevicePage1Flags  - add Requested and Configuration fields flags
 *	based on width, factor and offset parameters.
 *	@width: bus width
 *	@factor: sync factor
 *	@offset: sync offset
 *	@requestedPtr: pointer to requested values (updated)
 *	@configurationPtr: pointer to configuration values (updated)
 *	@flags: flags to block WDTR or SDTR negotiation
 *
 *	Return: None.
 *
 *	Remark: Called by writeSDP1 and _dv_params
 */
static void
mptscsih_setDevicePage1Flags (u8 width, u8 factor, u8 offset, int *requestedPtr, int *configurationPtr, u8 flags)
{
	u8 nowide = flags & MPT_TARGET_NO_NEGO_WIDE;
	u8 nosync = flags & MPT_TARGET_NO_NEGO_SYNC;

	*configurationPtr = 0;
	*requestedPtr = width ? MPI_SCSIDEVPAGE1_RP_WIDE : 0;
	*requestedPtr |= (offset << 16) | (factor << 8);

	if (width && offset && !nowide && !nosync) {
		if (factor < MPT_ULTRA160) {
			*requestedPtr |= (MPI_SCSIDEVPAGE1_RP_IU | MPI_SCSIDEVPAGE1_RP_DT);
			if ((flags & MPT_TARGET_NO_NEGO_QAS) == 0)
				*requestedPtr |= MPI_SCSIDEVPAGE1_RP_QAS;
			if (flags & MPT_TAPE_NEGO_IDP)
				*requestedPtr |= MPI_SCSIDEVPAGE1_RP_IDP;
		} else if (factor < MPT_ULTRA2) {
			*requestedPtr |= MPI_SCSIDEVPAGE1_RP_DT;
		}
	}

	if (nowide)
		*configurationPtr |= MPI_SCSIDEVPAGE1_CONF_WDTR_DISALLOWED;

	if (nosync)
		*configurationPtr |= MPI_SCSIDEVPAGE1_CONF_SDTR_DISALLOWED;

	return;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_writeSDP1  - write SCSI Device Page 1
 *	@hd: Pointer to a SCSI Host Strucutre
 *	@portnum: IOC port number
 *	@id: writeSDP1 for single ID
 *	@flags: MPT_SCSICFG_ALL_IDS, MPT_SCSICFG_USE_NVRAM, MPT_SCSICFG_BLK_NEGO
 *
 *	Return: -EFAULT if read of config page header fails
 *		or 0 if success.
 *
 *	Remark: If a target has been found, the settings from the
 *		target structure are used, else the device is set
 *		to async/narrow.
 *
 *	Remark: Called during init and after a FW reload.
 *	Remark: We do not wait for a return, write pages sequentially.
 */
static int
mptscsih_writeSDP1(MPT_SCSI_HOST *hd, int portnum, int id, int flags)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	Config_t		*pReq;
	SCSIDevicePage1_t	*pData;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;
	MPT_FRAME_HDR		*mf;
	dma_addr_t		 dataDma;
	u16			 req_idx;
	u32			 frameOffset;
	u32			 requested, configuration, flagsLength;
	int			 ii, nvram;
	int			 loop_id = 0;
	u8			 width;
	u8			 factor;
	u8			 offset;
	u8			 bus = 0;
	u8			 negoFlags;
	u8			 maxwidth, maxoffset, maxfactor;

	if (ioc->spi_data.sdp1length == 0)
		return 0;

	if (flags & MPT_SCSICFG_ALL_IDS) {
		loop_id = 0;
	} else {
		loop_id = id;
	}

	for (; loop_id < ioc->DevicesPerBus; loop_id++) {
		if (loop_id == ioc->pfacts[portnum].PortSCSIID)
			continue;

		/* Use NVRAM to get adapter and target maximums
		 * Data over-riden by target structure information, if present
		 */
		maxwidth = ioc->spi_data.maxBusWidth;
		maxoffset = ioc->spi_data.maxSyncOffset;
		maxfactor = ioc->spi_data.minSyncFactor;
		if (ioc->spi_data.nvram && (ioc->spi_data.nvram[loop_id] != MPT_HOST_NVRAM_INVALID)) {
			nvram = ioc->spi_data.nvram[loop_id];

			if (maxwidth)
				maxwidth = nvram & MPT_NVRAM_WIDE_DISABLE ? 0 : 1;

			if (maxoffset > 0) {
				maxfactor = (nvram & MPT_NVRAM_SYNC_MASK) >> 8;
				if (maxfactor == 0) {
					/* Key for async */
					maxfactor = MPT_ASYNC;
					maxoffset = 0;
				} else if (maxfactor < ioc->spi_data.minSyncFactor) {
					maxfactor = ioc->spi_data.minSyncFactor;
				}
			} else
				maxfactor = MPT_ASYNC;
		}

		/* Set the negotiation flags.
		 */
		negoFlags = ioc->spi_data.noQas;
		if (!maxwidth)
			negoFlags |= MPT_TARGET_NO_NEGO_WIDE;

		if (!maxoffset)
			negoFlags |= MPT_TARGET_NO_NEGO_SYNC;

		if (flags & MPT_SCSICFG_USE_NVRAM) {
			width = maxwidth;
			factor = maxfactor;
			offset = maxoffset;
		} else {
			width = 0;
			factor = MPT_ASYNC;
			offset = 0;
			//negoFlags = 0;
			//negoFlags = MPT_TARGET_NO_NEGO_SYNC;
		}

		/* If id is not a raid volume, get the updated
		 * transmission settings from the target structure.
		 */
		pMptTarget = ioc->Target_List[0];
		pTarget = (VirtDevice *)pMptTarget->Target[loop_id];
		if (pTarget && !pTarget->raidVolume) {
			width = pTarget->maxWidth;
			factor = pTarget->minSyncFactor;
			offset = pTarget->maxOffset;
			negoFlags |= pTarget->negoFlags;
		}

#ifdef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
		/* Force to async and narrow if DV has not been executed
		 * for this ID
		 */
		if ((hd->ioc->spi_data.dvStatus[loop_id] & MPT_SCSICFG_DV_NOT_DONE) != 0) {
			width = 0;
			factor = MPT_ASYNC;
			offset = 0;
		}
#endif

		if (flags & MPT_SCSICFG_BLK_NEGO)
			negoFlags |= MPT_TARGET_NO_NEGO_WIDE | MPT_TARGET_NO_NEGO_SYNC;

		mptscsih_setDevicePage1Flags(width, factor, offset,
					&requested, &configuration, negoFlags);
		dnegoprintk(("writeSDP1: id=%d width=%d factor=%x offset=%x negoFlags=%x request=%x config=%x\n",
			loop_id, width, factor, offset, negoFlags, requested, configuration));

		/* Get a MF for this command.
		 */
		if ((mf = mpt_get_msg_frame(mpt_ScsiDoneCtx, ioc)) == NULL) {
			dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
			    ioc->name,__FUNCTION__));
			return -EAGAIN;
		}

		ddvprintk((MYIOC_s_WARN_FMT "writeSDP1 (mf=%p, id=%d, req=0x%x, cfg=0x%x)\n",
			hd->ioc->name, mf, loop_id, requested, configuration));


		/* Set the request and the data pointers.
		 * Request takes: 36 bytes (32 bit SGE)
		 * SCSI Device Page 1 requires 16 bytes
		 * 40 + 16 <= size of SCSI IO Request = 56 bytes
		 * and MF size >= 64 bytes.
		 * Place data at end of MF.
		 */
		pReq = (Config_t *)mf;

		req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
		frameOffset = ioc->req_sz - sizeof(SCSIDevicePage1_t);

		pData = (SCSIDevicePage1_t *)((u8 *) mf + frameOffset);
		dataDma = ioc->req_frames_dma + (req_idx * ioc->req_sz) + frameOffset;

		/* Complete the request frame (same for all requests).
		 */
		pReq->Action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
		pReq->Reserved = 0;
		pReq->ChainOffset = 0;
		pReq->Function = MPI_FUNCTION_CONFIG;
		pReq->ExtPageLength = 0;
		pReq->ExtPageType = 0;
		pReq->MsgFlags = 0;
		for (ii=0; ii < 8; ii++) {
			pReq->Reserved2[ii] = 0;
		}
		pReq->Header.PageVersion = ioc->spi_data.sdp1version;
		pReq->Header.PageLength = ioc->spi_data.sdp1length;
		pReq->Header.PageNumber = 1;
		pReq->Header.PageType = MPI_CONFIG_PAGETYPE_SCSI_DEVICE;
		pReq->PageAddress = cpu_to_le32(loop_id | (bus << 8 ));

		/* Add a SGE to the config request.
		 */
		flagsLength = MPT_SGE_FLAGS_SSIMPLE_WRITE | ioc->spi_data.sdp1length * 4;

		ioc->add_sge((char *)&pReq->PageBufferSGE, flagsLength, dataDma);

		/* Set up the common data portion
		 */
		pData->Header.PageVersion = pReq->Header.PageVersion;
		pData->Header.PageLength = pReq->Header.PageLength;
		pData->Header.PageNumber = pReq->Header.PageNumber;
		pData->Header.PageType = pReq->Header.PageType;
		pData->RequestedParameters = cpu_to_le32(requested);
		pData->Reserved = 0;
		pData->Configuration = cpu_to_le32(configuration);

		if ( pTarget ) {
			if ( requested & MPI_SCSIDEVPAGE1_RP_IU ) {
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
				if (ioc->bus_type == SPI) {
					pTarget->last_lun = MPT_SCSI_LAST_LUN;
				} else {
					pTarget->last_lun = MPT_LAST_LUN;
				}
#else
				pTarget->last_lun = MPT_LAST_LUN;
#endif
			} else {
				pTarget->last_lun = MPT_NON_IU_LAST_LUN;
			}
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
                        /*
                         * limit to 8 luns per target for scsi2 devices.
                         */
                        if(pTarget->ioc &&
                           (pTarget->ioc->bus_type == SPI) &&
                           (pTarget->scsi_level > 0) &&
                           (pTarget->scsi_level <= SCSI_LEVEL2)) {
                                pTarget->last_lun = MPT_SCSI2_LAST_LUN;
			}
#endif

			dsprintk((MYIOC_s_WARN_FMT
				"writeSDP1: last_lun=%d on id=%d\n",
				ioc->name, pTarget->last_lun, loop_id));
		}

		dcprintk((MYIOC_s_WARN_FMT
			"write SDP1: id %d pgaddr 0x%x req 0x%x config 0x%x\n",
				ioc->name, loop_id, (loop_id | (bus<<8)),
				requested, configuration));

		mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);
	}

	return 0;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_writeIOCPage4  - write IOC Page 4
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@id: write IOC Page4 for this ID & Bus
 *
 *	Return: -EAGAIN if unable to obtain a Message Frame
 *		or 0 if success.
 *
 *	Remark: We do not wait for a return, write pages sequentially.
 */
static int
mptscsih_writeIOCPage4(MPT_ADAPTER *ioc, int id, int bus)
{
	Config_t		*pReq;
	IOCPage4_t		*IOCPage4Ptr;
	MPT_FRAME_HDR		*mf;
	dma_addr_t		 dataDma;
	u16			 req_idx;
	u32			 frameOffset;
	u32			 flagsLength;
	int			 ii;

	/* Get a MF for this command.
	 */
	if ((mf = mpt_get_msg_frame(mpt_ScsiDoneCtx, ioc)) == NULL) {
		dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
		    ioc->name,__FUNCTION__));
		return -EAGAIN;
	}

	/* Set the request and the data pointers.
	 * Place data at end of MF.
	 */
	pReq = (Config_t *)mf;

	req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
	frameOffset = ioc->req_sz - sizeof(IOCPage4_t);

	/* Complete the request frame (same for all requests).
	 */
	pReq->Action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
	pReq->Reserved = 0;
	pReq->ChainOffset = 0;
	pReq->Function = MPI_FUNCTION_CONFIG;
	pReq->ExtPageLength = 0;
	pReq->ExtPageType = 0;
	pReq->MsgFlags = 0;
	for (ii=0; ii < 8; ii++) {
		pReq->Reserved2[ii] = 0;
	}

       	IOCPage4Ptr = ioc->spi_data.pIocPg4;
       	dataDma = ioc->spi_data.IocPg4_dma;
       	ii = IOCPage4Ptr->ActiveSEP++;
       	IOCPage4Ptr->SEP[ii].SEPTargetID = id;
       	IOCPage4Ptr->SEP[ii].SEPBus = bus;
       	pReq->Header = IOCPage4Ptr->Header;
	pReq->PageAddress = cpu_to_le32(id | (bus << 8 ));

	/* Add a SGE to the config request.
	 */
	flagsLength = MPT_SGE_FLAGS_SSIMPLE_WRITE | 
		(IOCPage4Ptr->Header.PageLength + ii) * 4;

	ioc->add_sge((char *)&pReq->PageBufferSGE, flagsLength, dataDma);

	dcprintk((MYIOC_s_WARN_FMT
		"writeIOCPage4: MaxSEP=%d ActiveSEP=%d id=%d bus=%d\n",
			ioc->name, IOCPage4Ptr->MaxSEP, IOCPage4Ptr->ActiveSEP, id, bus));

	mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);

	return 0;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_readFCDevicePage0  - read FC Device Page 0
 *	@hd: Pointer to a SCSI Host Structure
 *	@id: read FC Device Page 0 for this target ID
 *
 *	Return: -EAGAIN if unable to obtain a Message Frame
 *		or 0 if success.
 *
 *	Remark: We do not wait for a return, read pages sequentially.
 */
static int
mptscsih_readFCDevicePage0(MPT_SCSI_HOST *hd, int id)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	Config_t		*pReq;
	MPT_FRAME_HDR		*mf;
	dma_addr_t		 dataDma;
	u16			 req_idx;
	u32			 frameOffset;
	u32			 flagsLength;
	int			 ii;

	/* Get a MF for this command.
	 */
	if ((mf = mpt_get_msg_frame(mpt_ScsiDoneCtx, ioc)) == NULL) {
		dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
		    ioc->name,__FUNCTION__));
		return -EAGAIN;
	}

	/* Set the request and the data pointers.
	 * Place data at end of MF.
	 */
	pReq = (Config_t *)mf;

	req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
	frameOffset = sizeof(Config_t);

	dataDma = ioc->req_frames_dma + (req_idx * ioc->req_sz) + frameOffset;

	/* Complete the request frame (same for all requests).
	 */
	pReq->Action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
	pReq->Reserved = 0;
	pReq->ChainOffset = 0;
	pReq->Function = MPI_FUNCTION_CONFIG;
	pReq->ExtPageLength = 0;
	pReq->ExtPageType = 0;
	pReq->MsgFlags = 0;
	for (ii=0; ii < 8; ii++) {
		pReq->Reserved2[ii] = 0;
	}
	pReq->Header.PageVersion = MPI_FC_DEVICE_PAGE0_PAGEVERSION;
	pReq->Header.PageLength = sizeof(FCDevicePage0_t) / 4;
	pReq->Header.PageNumber = 0;
       	pReq->Header.PageType = MPI_CONFIG_PAGETYPE_FC_DEVICE;
	pReq->PageAddress = cpu_to_le32(MPI_FC_DEVICE_PGAD_FORM_BUS_TID |
					id);

	/* Add a SGE to the config request.
	 */
	flagsLength = MPT_SGE_FLAGS_SSIMPLE_READ | sizeof(FCDevicePage0_t);
	ioc->add_sge((char *)&pReq->PageBufferSGE, flagsLength, dataDma);

	drsprintk((MYIOC_s_INFO_FMT "readFCDevicePage0: target=%d\n", ioc->name, id));

	mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);

	return 0;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_writeFCPortPage3  - write FC Port Page 3
 *	@hd: Pointer to a SCSI Host Structure
 *	@id: write FC Port Page 3 for this target ID
 *
 *	Return: -EAGAIN if unable to obtain a Message Frame
 *		or 0 if success.
 *
 *	Remark: We do not wait for a return, write pages sequentially.
 */
static int
mptscsih_writeFCPortPage3(MPT_SCSI_HOST *hd, int id)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	Config_t		*pReq;
	FCPortPage3_t		*FCPort3;
	MPT_FRAME_HDR		*mf;
	dma_addr_t		 dataDma;
	u16			 req_idx;
	u32			 frameOffset;
	u32			 flagsLength;
	int			 ii;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;

	/* Get a MF for this command.
	 */
	if ((mf = mpt_get_msg_frame(mpt_ScsiDoneCtx, ioc)) == NULL) {
		dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
		    ioc->name,__FUNCTION__));
		return -EAGAIN;
	}

	/* Set the request and the data pointers.
	 * Place data at end of MF.
	 */
	pReq = (Config_t *)mf;

	req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
	frameOffset = sizeof(Config_t);

	FCPort3 = (FCPortPage3_t *)((u8 *)mf + frameOffset);
	dataDma = ioc->req_frames_dma + (req_idx * ioc->req_sz) + frameOffset;

	/* Complete the request frame (same for all requests).
	 */
	pReq->Action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
	pReq->Reserved = 0;
	pReq->ChainOffset = 0;
	pReq->Function = MPI_FUNCTION_CONFIG;
	pReq->ExtPageLength = 0;
	pReq->ExtPageType = 0;
	pReq->MsgFlags = 0;
	for (ii=0; ii < 8; ii++) {
		pReq->Reserved2[ii] = 0;
	}
	pReq->Header.PageVersion = MPI_FCPORTPAGE3_PAGEVERSION;
	pReq->Header.PageLength = sizeof(FCPortPage3_t) / 4;
	pReq->Header.PageNumber = 3;
       	pReq->Header.PageType = MPI_CONFIG_PAGETYPE_FC_PORT |
				MPI_CONFIG_PAGEATTR_PERSISTENT;
	pReq->PageAddress = cpu_to_le32(MPI_FC_PORT_PGAD_FORM_INDEX |
					id);

	pMptTarget = ioc->Target_List[0];
	pTarget = (VirtDevice *)pMptTarget->Target[id];

	FCPort3->Header.PageVersion = MPI_FCPORTPAGE3_PAGEVERSION;
	FCPort3->Header.PageLength = sizeof(FCPortPage3_t) / 4;
	FCPort3->Header.PageNumber = 3;
       	FCPort3->Header.PageType = MPI_CONFIG_PAGETYPE_FC_PORT |
				   MPI_CONFIG_PAGEATTR_PERSISTENT;
       	FCPort3->Entry[0].PhysicalIdentifier.WWN.WWPN = pTarget->WWPN;
       	FCPort3->Entry[0].PhysicalIdentifier.WWN.WWNN = pTarget->WWNN;
       	FCPort3->Entry[0].TargetID = pTarget->id;
       	FCPort3->Entry[0].Bus = pTarget->bus;
	FCPort3->Entry[0].Flags = cpu_to_le16(MPI_PERSISTENT_FLAGS_ENTRY_VALID);

	/* Add a SGE to the config request.
	 */
	flagsLength = MPT_SGE_FLAGS_SSIMPLE_READ | sizeof(FCPortPage3_t);
	ioc->add_sge((char *)&pReq->PageBufferSGE, flagsLength, dataDma);

	drsprintk((MYIOC_s_INFO_FMT "writeFCPortPage3: target=%d\n", ioc->name, id));

	mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);

	return 0;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_sendIOCInit  - send IOC Init
 *	@hd: Pointer to a SCSI Host Structure
 *
 *	Return: -EAGAIN if unable to obtain a Message Frame
 *		or 0 if success.
 *
 *	Remark: We do not wait for a return, just dump and run.
 */
static int
mptscsih_sendIOCInit(MPT_SCSI_HOST *hd)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	IOCInit_t		*pReq;
	MPT_FRAME_HDR		*mf;
	u16			 req_idx;

	/* Get a MF for this command.
	 */
	if ((mf = mpt_get_msg_frame(mpt_ScsiDoneCtx, ioc)) == NULL) {
		dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
		    ioc->name,__FUNCTION__));
		return -EAGAIN;
	}

	/* Set the request pointer.
	 */
	pReq = (IOCInit_t *)mf;

	req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
	ioc->RequestNB[req_idx] = 0;

	/* Complete the request frame (same for all requests).
	 */
	pReq->WhoInit = MPI_WHOINIT_HOST_DRIVER;
	pReq->Reserved = 0;
	pReq->ChainOffset = 0;
	pReq->Function = MPI_FUNCTION_IOC_INIT;
	pReq->Flags = 0;
	pReq->MaxDevices = ioc->DevicesPerBus;
	pReq->MaxBuses = ioc->NumberOfBuses;
	pReq->MsgFlags = 0;
	pReq->ReplyFrameSize = cpu_to_le16(ioc->reply_sz);
	pReq->Reserved1[0] = 0;
	pReq->Reserved1[1] = 0;
	pReq->HostMfaHighAddr = ioc->facts.CurrentHostMfaHighAddr;
	pReq->SenseBufferHighAddr = ioc->facts.CurrentSenseBufferHighAddr;
	pReq->ReplyFifoHostSignalingAddr = ioc->facts.ReplyFifoHostSignalingAddr;
	if (ioc->facts.Flags & MPI_IOCFACTS_FLAGS_REPLY_FIFO_HOST_SIGNAL) {
		pReq->Flags |= MPI_IOCINIT_FLAGS_REPLY_FIFO_HOST_SIGNAL;
	}
	pReq->HostPageBufferSGE = ioc->facts.HostPageBufferSGE;
	pReq->MsgVersion = cpu_to_le16(MPI_VERSION);
	pReq->HeaderVersion = cpu_to_le16(MPI_HEADER_VERSION);

	drsprintk((MYIOC_s_INFO_FMT "sendIOCInit\n", ioc->name));

	mpt_put_msg_frame(mpt_ScsiDoneCtx, ioc, mf);

	return 0;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_taskmgmt_timeout - Call back for timeout on a
 *	task management request.
 *	@data: Pointer to MPT_SCSI_HOST recast as an unsigned long
 *
 */
static void mptscsih_taskmgmt_timeout(unsigned long data)
{
	MPT_SCSI_HOST *hd = (MPT_SCSI_HOST *) data;
	MPT_ADAPTER	*ioc=hd->ioc;

	dtmprintk((KERN_INFO MYNAM ": %s: mptscsih_taskmgmt_timeout: "
		   "TM request timed out!\n", ioc->name));

	ioc->tmState = TM_STATE_NONE;

	/* Delete the timer that triggered this callback.
	 * Remark: DEL_TIMER checks to make sure timer is active
	 * before deleting.
	 */
	DEL_TIMER(&ioc->TMtimer);

	/* Call the reset handler. Already had a TM request
	 * timeout - so issue a diagnostic reset
	 */
	if (mpt_HardResetHandler(ioc, CAN_SLEEP) < 0) {
		printk((KERN_WARNING " taskmgmt_timeout: HardReset FAILED!!\n"));
	}
	else {
		/* Because we have reset the IOC, no TM requests can be
		 * pending.  So let's make sure the tmState flag is reset.
		 */
		nehprintk((KERN_INFO MYNAM
			   ": %s: mptscsih_taskmgmt_timeout\n",
			   ioc->name));
	}

	return;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *  Bus Scan and Domain Validation functionality ...
 */

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
 *	mptscsih_scandv_complete - Scan and DV callback routine registered
 *	to Fustion MPT (base) driver.
 *
 *	@ioc: Pointer to MPT_ADAPTER structure
 *	@mf: Pointer to original MPT request frame
 *	@mr: Pointer to MPT reply frame (NULL if TurboReply)
 *
 *	This routine is called from mpt.c::mpt_interrupt() at the completion
 *	of any SCSI IO request.
 *	This routine is registered with the Fusion MPT (base) driver at driver
 *	load/init time via the mpt_register() API call.
 *
 *	Returns 1 indicating alloc'd request frame ptr should be freed.
 *
 *	Remark: Sets a completion code and (possibly) saves sense data
 *	in the IOC member localReply structure.
 *	Used ONLY for DV and other internal commands.
 */
static int
mptscsih_scandv_complete(MPT_ADAPTER *ioc, MPT_FRAME_HDR *mf, MPT_FRAME_HDR *mr)
{
	MPT_SCSI_HOST	*hd;
	SCSIIORequest_t *pReq;
	int		 completionCode;
	u16		 req_idx;

	if ((mf == NULL) ||
	    (mf >= MPT_INDEX_2_MFPTR(ioc, ioc->req_depth))) {
		printk(MYIOC_s_ERR_FMT
			"ScanDvComplete, %s req frame ptr! (=%p)\n",
				ioc->name, mf?"BAD":"NULL", (void *) mf);
		goto wakeup;
	}

	hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;
	DEL_TIMER(&hd->timer);
	req_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);
	ioc->ScsiLookup[req_idx] = NULL;
	pReq = (SCSIIORequest_t *) mf;

	if (mf != hd->cmdPtr) {
		printk(MYIOC_s_WARN_FMT "ScanDvComplete (mf=%p, cmdPtr=%p, idx=%d)\n",
				ioc->name, (void *)mf, (void *) hd->cmdPtr, req_idx);
	}
	hd->cmdPtr = NULL;

	ddvprintk((MYIOC_s_WARN_FMT "ScanDvComplete (mf=%p,mr=%p,idx=%d)\n",
			ioc->name, mf, mr, req_idx));

	hd->pLocal = &hd->localReply;
	hd->pLocal->scsiStatus = 0;

	/* If target struct exists, clear sense valid flag.
	 */
	if (mr == NULL) {
		completionCode = MPT_SCANDV_GOOD;
	} else {
		SCSIIOReply_t	*pReply;
		u16		 status;
		u8		 scsi_status;

		pReply = (SCSIIOReply_t *) mr;

		status = le16_to_cpu(pReply->IOCStatus) & MPI_IOCSTATUS_MASK;
		scsi_status = pReply->SCSIStatus;

		ddvprintk((KERN_NOTICE "  IOCStatus=%04xh, SCSIState=%02xh, SCSIStatus=%02xh, IOCLogInfo=%08xh\n",
			     status, pReply->SCSIState, scsi_status,
			     le32_to_cpu(pReply->IOCLogInfo)));

		switch(status) {

		case MPI_IOCSTATUS_SCSI_DEVICE_NOT_THERE:	/* 0x0043 */
			completionCode = MPT_SCANDV_SELECTION_TIMEOUT;
			break;

		case MPI_IOCSTATUS_SCSI_IO_DATA_ERROR:		/* 0x0046 */
		case MPI_IOCSTATUS_SCSI_TASK_TERMINATED:	/* 0x0048 */
		case MPI_IOCSTATUS_SCSI_IOC_TERMINATED:		/* 0x004B */
		case MPI_IOCSTATUS_SCSI_EXT_TERMINATED:		/* 0x004C */
			completionCode = MPT_SCANDV_DID_RESET;
			break;

		case MPI_IOCSTATUS_SCSI_DATA_UNDERRUN:		/* 0x0045 */
		case MPI_IOCSTATUS_SCSI_RECOVERED_ERROR:	/* 0x0040 */
		case MPI_IOCSTATUS_SUCCESS:			/* 0x0000 */
			if (pReply->Function == MPI_FUNCTION_CONFIG) {
				ConfigReply_t *pr = (ConfigReply_t *)mr;
				completionCode = MPT_SCANDV_GOOD;
				hd->pLocal->header.PageVersion = pr->Header.PageVersion;
				hd->pLocal->header.PageLength = pr->Header.PageLength;
				hd->pLocal->header.PageNumber = pr->Header.PageNumber;
				hd->pLocal->header.PageType = pr->Header.PageType;

			} else if (pReply->Function == MPI_FUNCTION_RAID_ACTION) {
				/* If the RAID Volume request is successful,
				 * return GOOD, else indicate that
				 * some type of error occurred.
				 */
				MpiRaidActionReply_t	*pr = (MpiRaidActionReply_t *)mr;
				if (le16_to_cpu(pr->ActionStatus) == MPI_RAID_ACTION_ASTATUS_SUCCESS)
					completionCode = MPT_SCANDV_GOOD;
				else
					completionCode = MPT_SCANDV_SOME_ERROR;

			} else if (pReply->SCSIState & MPI_SCSI_STATE_AUTOSENSE_VALID) {
				u8		*sense_data;
				int		 sz;

				/* save sense data in global structure
				 */
				completionCode = MPT_SCANDV_SENSE;
				hd->pLocal->scsiStatus = scsi_status;
				sense_data = ((u8 *)hd->ioc->sense_buf_pool +
					(req_idx * MPT_SENSE_BUFFER_ALLOC));

				sz = min_t(int, pReq->SenseBufferLength,
							SCSI_STD_SENSE_BYTES);
				memcpy(hd->pLocal->sense, sense_data, sz);

				ddvprintk((KERN_NOTICE "  Check Condition, sense ptr %p\n",
						sense_data));
			} else if (pReply->SCSIState & MPI_SCSI_STATE_AUTOSENSE_FAILED) {
				if (pReq->CDB[0] == CMD_Inquiry)
					completionCode = MPT_SCANDV_ISSUE_SENSE;
				else
					completionCode = MPT_SCANDV_DID_RESET;
			}
			else if (pReply->SCSIState & MPI_SCSI_STATE_NO_SCSI_STATUS)
				completionCode = MPT_SCANDV_DID_RESET;
			else if (pReply->SCSIState & MPI_SCSI_STATE_TERMINATED)
				completionCode = MPT_SCANDV_DID_RESET;
			else {
				hd->pLocal->scsiStatus = scsi_status;
				completionCode = MPT_SCANDV_GOOD;
			}
			break;

		case MPI_IOCSTATUS_SCSI_PROTOCOL_ERROR:		/* 0x0047 */
			if (pReply->SCSIState & MPI_SCSI_STATE_TERMINATED)
				completionCode = MPT_SCANDV_DID_RESET;
			else
				completionCode = MPT_SCANDV_SOME_ERROR;
			break;

		default:
			completionCode = MPT_SCANDV_SOME_ERROR;
			break;

		}	/* switch(status) */

		ddvprintk((KERN_NOTICE "  completionCode set to %08xh\n",
				completionCode));
	} /* end of address reply case */

	hd->pLocal->completion = completionCode;

	/* MF and RF are freed in mpt_interrupt
	 */
wakeup:
	/* Free Chain buffers (will never chain) in scan or dv */
	//mptscsih_freeChainBuffers(ioc, req_idx);

	/*
	 * Wake up the original calling thread
	 */
	scandv_wait_done = 1;
	WAKE_UP(&scandv_waitq);

	return 1;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_timer_expired - Call back for timer process.
 *	Used only for dv functionality.
 *	@data: Pointer to MPT_SCSI_HOST recast as an unsigned long
 *
 */
static void mptscsih_timer_expired(unsigned long data)
{
	MPT_SCSI_HOST *hd = (MPT_SCSI_HOST *) data;
	MPT_ADAPTER	*ioc=hd->ioc;
	MPIHeader_t *cmd = (MPIHeader_t *)hd->cmdPtr;

	ddvprintk((MYIOC_s_WARN_FMT "DV Timer Expired! cmdPtr=%p\n", 
		ioc->name, hd->cmdPtr));

	hd->pLocal->completion = MPT_SCANDV_FALLBACK;

	if (cmd->Function == MPI_FUNCTION_SCSI_IO_REQUEST) {
		SCSIIORequest_t	*pScsiReq = (SCSIIORequest_t *) hd->cmdPtr;
		int	 id = pScsiReq->TargetID;

		/* Desire to issue a task management request here.
		 * TM requests MUST be single threaded.
		 * If old eh code and no TM current, issue request.
		 * If new eh code, do nothing. Wait for OS cmd timeout
		 *	for bus reset.
		 */
		ddvprintk((MYIOC_s_WARN_FMT "DV Cmd Timeout: id=%d CDB=%02x,  Attempting Target Reset\n", 
			ioc->name, id, pScsiReq->CDB[0]));

		if (mptscsih_TMHandler(hd, MPI_SCSITASKMGMT_TASKTYPE_TARGET_RESET,
                       0, id, 0, 0, NO_SLEEP) < 0){
			/* The TM request failed!
			 */
			printk(MYIOC_s_WARN_FMT "Error processing target reset TaskMgmt request\n",
				ioc->name);
			ioc->tmState = TM_STATE_NONE;
		}
		ioc->spi_data.dvStatus[id] |= MPT_SCSICFG_NEED_DV;
	} else {
		/* Perform a FW reload */
		if (mpt_HardResetHandler(hd->ioc, CAN_SLEEP) < 0) {
			printk(MYIOC_s_WARN_FMT "DV timer_expired: HardReset FAILED!\n", ioc->name);
		}
	}

	ddvprintk((MYIOC_s_WARN_FMT "DV Timer Expired Complete!\n", ioc->name));
}

#ifdef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_do_raid - Format and Issue a RAID volume request message.
 *	@hd: Pointer to scsi host structure
 *	@action: What do be done.
 *	@id: Logical target id.
 *	@bus: Target locations bus.
 *
 *	Returns: < 0 on a fatal error
 *		0 on success
 *
 *	Remark: Wait to return until reply processed by the ISR.
 */
static int
mptscsih_do_raid(MPT_SCSI_HOST *hd, u8 action, INTERNAL_CMD *io)
{
	MPT_ADAPTER *ioc = hd->ioc;
	MpiRaidActionRequest_t	*pReq;
	MPT_FRAME_HDR		*mf;

	if (in_interrupt()) {
		dprintk((MYIOC_s_WARN_FMT "Internal raid request not allowed in ISR context!\n",
       				ioc->name));
		return -EPERM;
	}

	/* Get and Populate a free Frame
	 */
	if ((mf = mpt_get_msg_frame(ScsiScanDvCtx, ioc)) == NULL) {
		dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
		    ioc->name,__FUNCTION__));
		return -EAGAIN;
	}
	pReq = (MpiRaidActionRequest_t *)mf;
	pReq->Action = action;
	pReq->Reserved1 = 0;
	pReq->ChainOffset = 0;
	pReq->Function = MPI_FUNCTION_RAID_ACTION;
	pReq->VolumeID = io->id;
	pReq->VolumeBus = io->bus;
	pReq->PhysDiskNum = io->physDiskNum;
	pReq->MsgFlags = 0;
	pReq->Reserved2 = 0;
	pReq->ActionDataWord = 0; /* Reserved for this action */
	//pReq->ActionDataSGE = 0;

	ioc->add_sge((char *)&pReq->ActionDataSGE,
		MPT_SGE_FLAGS_SSIMPLE_READ | 0, (dma_addr_t) -1);

	hd->pLocal = NULL;
	hd->timer.expires = jiffies + MPT_JIFFY*10; /* 10 second timeout */
	scandv_wait_done = 0;

	/* Save cmd pointer, for resource free if timeout or
	 * FW reload occurs
	 */
	hd->cmdPtr = mf;

	ddvprintk((MYIOC_s_WARN_FMT "RAID Volume action %x id %d\n",
			ioc->name, action, io->id));

	ADD_TIMER(&hd->timer);
	mpt_put_msg_frame(ScsiScanDvCtx, ioc, mf);
	WAIT_EVENT(scandv_waitq, scandv_wait_done);

	if ((hd->pLocal == NULL) || (hd->pLocal->completion != MPT_SCANDV_GOOD))
		return -1;

	return 0;
}
#endif /* ~MPTSCSIH_ENABLE_DOMAIN_VALIDATION */

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_do_cmd - Do internal command.
 *	@hd: MPT_SCSI_HOST pointer
 *	@io: INTERNAL_CMD pointer.
 *
 *	Issue the specified internally generated command and do command
 *	specific cleanup. For bus scan / DV only.
 *	NOTES: If command is Inquiry and status is good,
 *	initialize a target structure, save the data
 *
 *	Remark: Single threaded access only.
 *
 *	Return:
 *		< 0 if an illegal command or no resources
 *
 *		   0 if good
 *
 *		 > 0 if command complete but some type of completion error.
 */
static int
mptscsih_do_cmd(MPT_SCSI_HOST *hd, INTERNAL_CMD *io)
{
	MPT_ADAPTER *ioc = hd->ioc;
	MPT_FRAME_HDR	*mf;
	SCSIIORequest_t	*pScsiReq;
	SCSIIORequest_t	 ReqCopy;
	int		 my_idx, ii, dir;
	int		 rc, cmdTimeout;
	u8		 cmdLen;
	u8		 CDB[]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	u8		 cmd = io->cmd;

	if (in_interrupt()) {
		dfailprintk((MYIOC_s_WARN_FMT "Internal SCSI IO request not allowed in ISR context!\n",
       				ioc->name));
		return -EPERM;
	}


	/* Set command specific information
	 */
	switch (cmd) {
	case CMD_Inquiry:
		cmdLen = 6;
		dir = MPI_SCSIIO_CONTROL_READ;
		CDB[0] = cmd;
		CDB[4] = io->size;
		cmdTimeout = 10;
		break;

	case CMD_TestUnitReady:
		cmdLen = 6;
		dir = MPI_SCSIIO_CONTROL_NODATATRANSFER;
		cmdTimeout = 10;
		break;

	case CMD_StartStopUnit:
		cmdLen = 6;
		dir = MPI_SCSIIO_CONTROL_NODATATRANSFER;
		CDB[0] = cmd;
		CDB[4] = 1;	/*Spin up the disk */
		cmdTimeout = 15;
		break;

	case CMD_RequestSense:
		cmdLen = 6;
		CDB[0] = cmd;
		CDB[4] = io->size;
		dir = MPI_SCSIIO_CONTROL_READ;
		cmdTimeout = 10;
		break;

	case CMD_ReadBuffer:
		cmdLen = 10;
		dir = MPI_SCSIIO_CONTROL_READ;
		CDB[0] = cmd;
		if (io->flags & MPT_ICFLAG_ECHO) {
			CDB[1] = 0x0A;
		} else {
			CDB[1] = 0x02;
		}

		if (io->flags & MPT_ICFLAG_BUF_CAP) {
			CDB[1] |= 0x01;
		}
		CDB[6] = (io->size >> 16) & 0xFF;
		CDB[7] = (io->size >>  8) & 0xFF;
		CDB[8] = io->size & 0xFF;
		cmdTimeout = 10;
		break;

	case CMD_WriteBuffer:
		cmdLen = 10;
		dir = MPI_SCSIIO_CONTROL_WRITE;
		CDB[0] = cmd;
		if (io->flags & MPT_ICFLAG_ECHO) {
			CDB[1] = 0x0A;
		} else {
			CDB[1] = 0x02;
		}
		CDB[6] = (io->size >> 16) & 0xFF;
		CDB[7] = (io->size >>  8) & 0xFF;
		CDB[8] = io->size & 0xFF;
		cmdTimeout = 10;
		break;

	case CMD_Reserve6:
		cmdLen = 6;
		dir = MPI_SCSIIO_CONTROL_NODATATRANSFER;
		CDB[0] = cmd;
		cmdTimeout = 10;
		break;

	case CMD_Release6:
		cmdLen = 6;
		dir = MPI_SCSIIO_CONTROL_NODATATRANSFER;
		CDB[0] = cmd;
		cmdTimeout = 10;
		break;

	case CMD_SynchronizeCache:
		cmdLen = 10;
		dir = MPI_SCSIIO_CONTROL_NODATATRANSFER;
		CDB[0] = cmd;
//		CDB[1] = 0x02;	/* set immediate bit */
		cmdTimeout = 10;
		break;

	case OEM_TLR_COMMAND:
		CDB[0] = cmd;
		CDB[1] = 0x01;
		cmdLen = 6;
		dir = MPI_SCSIIO_CONTROL_READ;
		cmdTimeout = 10;
		break;

	default:
		/* Error Case */
		dfailprintk((MYIOC_s_WARN_FMT "Invalid cmd=%x\n",
		    ioc->name, cmd));
		return -EFAULT;
	}

	/* Get and Populate a free Frame
	 */
	if ((mf = mpt_get_msg_frame(ScsiScanDvCtx, ioc)) == NULL) {
		dfailprintk((MYIOC_s_WARN_FMT "%s: no msg frames!!\n",
		    ioc->name,__FUNCTION__));
		return -EBUSY;
	}

	pScsiReq = (SCSIIORequest_t *) mf;

	/* Get the request index */
	my_idx = le16_to_cpu(mf->u.frame.hwhdr.msgctxu.fld.req_idx);

	if (io->flags & MPT_ICFLAG_PHYS_DISK) {
		pScsiReq->TargetID = io->physDiskNum;
		pScsiReq->Bus = 0;
		pScsiReq->ChainOffset = 0;
		pScsiReq->Function = MPI_FUNCTION_RAID_SCSI_IO_PASSTHROUGH;
	} else {
		pScsiReq->TargetID = io->id;
		pScsiReq->Bus = io->bus;
		pScsiReq->ChainOffset = 0;
		pScsiReq->Function = MPI_FUNCTION_SCSI_IO_REQUEST;
	}

	pScsiReq->CDBLength = cmdLen;
	pScsiReq->SenseBufferLength = MPT_SENSE_BUFFER_SIZE;

	pScsiReq->Reserved = 0;

	pScsiReq->MsgFlags = mpt_msg_flags();
	/* MsgContext set in mpt_get_msg_frame call  */

	for (ii=0; ii < 8; ii++)
		pScsiReq->LUN[ii] = 0;
	pScsiReq->LUN[1] = io->lun;

	if (io->flags & MPT_ICFLAG_TAGGED_CMD) {
		pScsiReq->Control = cpu_to_le32(dir | MPI_SCSIIO_CONTROL_SIMPLEQ);
		ddvprintk((MYIOC_s_NOTE_FMT
			"DV: CDB=%02x id=%d sent Tagged\n",
			hd->ioc->name, CDB[0], io->id));
	} else {
		pScsiReq->Control = cpu_to_le32(dir | 
					MPI_SCSIIO_CONTROL_UNTAGGED);
		ddvprintk((MYIOC_s_NOTE_FMT
			"DV: CDB=%02x id=%d sent Untagged\n",
			hd->ioc->name, CDB[0], io->id));
	}

	if (cmd == CMD_RequestSense) {
		pScsiReq->Control = cpu_to_le32(dir | MPI_SCSIIO_CONTROL_UNTAGGED);
		ddvprintk((MYIOC_s_WARN_FMT "Untagged! 0x%2x\n",
			ioc->name, cmd));
	}

	for (ii=0; ii < 16; ii++)
		pScsiReq->CDB[ii] = CDB[ii];

	pScsiReq->DataLength = cpu_to_le32(io->size);
	pScsiReq->SenseBufferLowAddr = cpu_to_le32(ioc->sense_buf_low_dma
					   + (my_idx * MPT_SENSE_BUFFER_ALLOC));

	if (dir == MPI_SCSIIO_CONTROL_READ) {
		ioc->add_sge((char *) &pScsiReq->SGL,
			MPT_SGE_FLAGS_SSIMPLE_READ | io->size,
			io->data_dma);
	} else if (dir == MPI_SCSIIO_CONTROL_WRITE) {
		ioc->add_sge((char *) &pScsiReq->SGL,
			MPT_SGE_FLAGS_SSIMPLE_WRITE | io->size,
			io->data_dma);
	} else {
		/* Add a NULL SGE */
		ioc->add_sge((char *)&pScsiReq->SGL, 
			MPT_SGE_FLAGS_SSIMPLE_READ,
			(dma_addr_t) -1);
	}

	/* The ISR will free the request frame, but we need
	 * the information to initialize the target. Duplicate.
	 */
	memcpy(&ReqCopy, pScsiReq, sizeof(SCSIIORequest_t));

	/* Issue this command after:
	 *	finish init
	 *	add timer
	 * Wait until the reply has been received
	 *  ScsiScanDvCtx callback function will
	 *	set hd->pLocal;
	 *	set scandv_wait_done and call WAKE_UP
	 */
	hd->pLocal = NULL;
	hd->timer.expires = jiffies + MPT_JIFFY*cmdTimeout;
	scandv_wait_done = 0;

	/* Save cmd pointer, for resource free if timeout or
	 * FW reload occurs
	 */
	hd->cmdPtr = mf;

	ddvprintk((MYIOC_s_WARN_FMT "Sending Command 0x%x for (%d:%d:%d)\n",
			ioc->name, cmd, io->bus, io->id, io->lun));

	ADD_TIMER(&hd->timer);
	mpt_put_msg_frame(ScsiScanDvCtx, ioc, mf);
	WAIT_EVENT(scandv_waitq, scandv_wait_done);

	if (hd->pLocal) {
		rc = hd->pLocal->completion;
		hd->pLocal->skip = 0;

		/* Always set fatal error codes in some cases.
		 */
		if (rc == MPT_SCANDV_SELECTION_TIMEOUT) {
			dfailprintk((MYIOC_s_WARN_FMT "_do_cmd=%x: SELECTION_TIMEOUT!!!\n",
				ioc->name, cmd));
			rc = -ENXIO;
		} else if (rc == MPT_SCANDV_SOME_ERROR) {
			dfailprintk((MYIOC_s_WARN_FMT "_do_cmd=%x: SOME_ERROR!!!\n",
				ioc->name, cmd));
			rc =  -rc;
		}
	} else {
		rc = -EFAULT;
		/* This should never happen. */
		dfailprintk((MYIOC_s_WARN_FMT "_do_cmd=%x: Null pLocal!!!\n",
				ioc->name, cmd));
	}

	return rc;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_synchronize_cache - Send SYNCHRONIZE_CACHE to all disks.
 *	@hd: Pointer to MPT_SCSI_HOST structure
 *	@portnum: IOC port number
 *
 *	Uses the ISR, but with special processing.
 *	MUST be single-threaded.
 *
 *	Return: 0 on completion
 */
static int
mptscsih_synchronize_cache(MPT_SCSI_HOST *hd, int portnum)
{
	MPT_ADAPTER		*ioc= hd->ioc;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;
	SCSIDevicePage1_t	*pcfg1Data = NULL;
	INTERNAL_CMD		 iocmd;
	CONFIGPARMS		 cfg;
	dma_addr_t		 cfg1_dma_addr = -1;
	ConfigPageHeader_t	 header1;
	int			 bus = 0;
	int			 id = 0;
	int			 lun;
	int			 indexed_lun, lun_index;
	int			 hostId = ioc->pfacts[portnum].PortSCSIID;
	int			 max_id;
	int			 requested, configuration, data;
	int			 doConfig = 0;
	u8			 flags, factor;

	dexitprintk((KERN_INFO MYNAM "%s: %s\n",
		 		ioc->name, __FUNCTION__));
	max_id = ioc->sh->max_id - 1;

	/* Following parameters will not change
	 * in this routine.
	 */
	iocmd.cmd = CMD_SynchronizeCache;
	iocmd.flags = 0;
	iocmd.physDiskNum = -1;
	iocmd.data = NULL;
	iocmd.data_dma = -1;
	iocmd.size = 0;
	iocmd.rsvd = iocmd.rsvd2 = 0;

	/* Skip the host
	 */
	if (id == hostId)
		id++;

	/* Write SDP1 for all SCSI devices
	 * Alloc memory and set up config buffer
	 */
	if (ioc->bus_type == SPI) {
		if (ioc->spi_data.sdp1length > 0) {
			pcfg1Data = (SCSIDevicePage1_t *)pci_alloc_consistent(ioc->pcidev,
					 ioc->spi_data.sdp1length * 4, &cfg1_dma_addr);

			if (pcfg1Data != NULL) {
				doConfig = 1;
				header1.PageVersion = ioc->spi_data.sdp1version;
				header1.PageLength = ioc->spi_data.sdp1length;
				header1.PageNumber = 1;
				header1.PageType = MPI_CONFIG_PAGETYPE_SCSI_DEVICE;
				cfg.cfghdr.hdr = &header1;
				cfg.physAddr = cfg1_dma_addr;
				cfg.action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
				cfg.dir = 1;
				cfg.timeout = 0;
			}
		}
	}

	/* loop through all devices on this port
	 */
	while (bus < ioc->NumberOfBuses) {
		iocmd.bus = bus;
		iocmd.id = id;
		pMptTarget = ioc->Target_List[bus];
		pTarget = (VirtDevice *)pMptTarget->Target[id];

		if (doConfig) {


			/* Set the negotiation flags */
			if (pTarget && !pTarget->raidVolume) {
				flags = pTarget->negoFlags;
			} else {
				flags = ioc->spi_data.noQas;
				if (ioc->spi_data.nvram && (ioc->spi_data.nvram[id] != MPT_HOST_NVRAM_INVALID)) {
					data = ioc->spi_data.nvram[id];
	
					if (data & MPT_NVRAM_WIDE_DISABLE)
						flags |= MPT_TARGET_NO_NEGO_WIDE;

					factor = (data & MPT_NVRAM_SYNC_MASK) >> MPT_NVRAM_SYNC_SHIFT;
					if ((factor == 0) || (factor == MPT_ASYNC))
						flags |= MPT_TARGET_NO_NEGO_SYNC;
				}
			}
	
			/* Force to async, narrow */
			mptscsih_setDevicePage1Flags(0, MPT_ASYNC, 0, &requested,
					&configuration, flags);
			dnegoprintk(("syncronize cache: id=%d width=0 factor=MPT_ASYNC offset=0 negoFlags=%x request=%x config=%x\n",
				id, flags, requested, configuration));
			pcfg1Data->RequestedParameters = cpu_to_le32(requested);
			pcfg1Data->Reserved = 0;
			pcfg1Data->Configuration = cpu_to_le32(configuration);
			cfg.pageAddr = (bus<<8) | id;
			mpt_config(ioc, &cfg);
		}

		/* If target Ptr NULL or if this target is NOT a disk, skip.
		 * Per e-mail suggestion from John Lohmeyer 4/26/2005 
		 */
		if (pTarget && (pTarget->inq_data[0] == SCSI_TYPE_DAD)) {
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
			int last_lun;
			if (ioc->bus_type == SPI) {
				last_lun = MPT_SCSI_LAST_LUN;
			} else {
				last_lun = MPT_LAST_LUN;
			}
			for (lun=0; lun <= last_lun; lun++) {
#else
			for (lun=0; lun <= MPT_LAST_LUN; lun++) {
#endif
				/* If LUN present, issue the command
				 */
				lun_index = (lun >> 5);  /* 32 luns per lun_index */
				indexed_lun = (lun % 32);
				if (pTarget->luns[lun_index] & (1<<indexed_lun)) {
					iocmd.lun = lun;
					(void) mptscsih_do_cmd(hd, &iocmd);
				}
			}
		}

		/* get next relevant device */
		id++;

		if (id == hostId)
			id++;

		if (id > max_id) {
			id = 0;
			bus++;
		}
	}

	if (pcfg1Data) {
		pci_free_consistent(ioc->pcidev, header1.PageLength * 4, pcfg1Data, cfg1_dma_addr);
	}

	return 0;
}

#ifdef MPTSCSIH_ENABLE_DOMAIN_VALIDATION
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_domainValidation - Top level handler for domain validation.
 *	@hd: Pointer to MPT_SCSI_HOST structure.
 *
 *	Uses the ISR, but with special processing.
 *	Called from schedule, should not be in interrupt mode.
 *	While thread alive, do dv for all devices needing dv
 *
 *	Return: None.
 */
static void
mptscsih_domainValidation(void *arg)
{
	VirtDevice	*pTarget=(VirtDevice *)arg;
	MPT_SCSI_HOST	*hd;
	MPT_ADAPTER	*ioc;
	unsigned long	 flags;
	int 		 id, dvStatus;
	int		 ii;

	SPIN_LOCK_IRQSAVE(&dvtaskQ_lock, flags);
	dvtaskQ_active = 1;
	SPIN_UNLOCK_IRQRESTORE(&dvtaskQ_lock, flags);

	ioc = pTarget->ioc;
	id = pTarget->id;
//	set_current_state(TASK_INTERRUPTIBLE);
//	schedule_timeout(MPT_HZ/4);

	hd = (MPT_SCSI_HOST *) ioc->sh->hostdata;

	if ((ioc->spi_data.forceDv & MPT_SCSICFG_RELOAD_IOC_PG3) != 0) {
		mpt_read_ioc_pg_3(ioc);
		if (ioc->raid_data.pIocPg3) {
			Ioc3PhysDisk_t *pPDisk = ioc->raid_data.pIocPg3->PhysDisk;
			int	numPDisk = ioc->raid_data.pIocPg3->NumPhysDisks;

			while (numPDisk) {
				if (ioc->spi_data.dvStatus[pPDisk->PhysDiskID] & MPT_SCSICFG_DV_NOT_DONE)
					ioc->spi_data.dvStatus[pPDisk->PhysDiskID] |= MPT_SCSICFG_NEED_DV;

				pPDisk++;
				numPDisk--;
			}
		}
		ioc->spi_data.forceDv &= ~MPT_SCSICFG_RELOAD_IOC_PG3;
	}

	dvStatus = ioc->spi_data.dvStatus[id];

	if (dvStatus & MPT_SCSICFG_NEED_DV) {
		ioc->spi_data.dvStatus[id] &= ~MPT_SCSICFG_NEED_DV;
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
		if (ioc->revisionID > 1) {
			/* revisionID for VM is 1 and this change should exclude VM */
			ioc->spi_data.dvStatus[id] |= MPT_SCSICFG_DV_IN_PROGRESS;
		}
#else
		ioc->spi_data.dvStatus[id] |= MPT_SCSICFG_DV_IN_PROGRESS;
#endif

//		set_current_state(TASK_INTERRUPTIBLE);
//		schedule_timeout(MPT_HZ/4);

		/* If hidden phys disk, block IO's to all
		 *	raid volumes
		 * else, process normally
		 */
		if (ioc->raid_data.isRaid & (1 << id)) {
			Ioc3PhysDisk_t *pPDisk =  ioc->raid_data.pIocPg3->PhysDisk;
			int numPDisk = ioc->raid_data.pIocPg3->NumPhysDisks;
			while (numPDisk) {
				ii = pPDisk->PhysDiskID;

				ddvprintk((KERN_INFO "doDv for PhysDiskNum=%d PhysDiskID=%d numPDisk=%d\n",
					pPDisk->PhysDiskNum, ii, numPDisk));
				if (mptscsih_doDv(hd, 0, ii) == 1) {
					/* Untagged device was busy, try again
					 */
					ioc->spi_data.dvStatus[ii] |= MPT_SCSICFG_NEED_DV;
				}
				pPDisk++;
				numPDisk--;
			}
		} else {
			ddvprintk((KERN_INFO "doDv for id=%d\n",
				id));
			if (mptscsih_doDv(hd, 0, id) == 1) {
				/* Untagged device was busy, try again
				 */
				ioc->spi_data.dvStatus[id] |= MPT_SCSICFG_NEED_DV;
			}
		}

		/* Post OS IOs that were pended while
		 * DV running.
		 */
		mpt_post_pendingQ_commands(ioc);

		if (ioc->spi_data.noQas)
			mptscsih_qas_check(hd, id);
	} else {
		ddvprintk((KERN_INFO "~NEED_DV dvStatus=%x for id %d\n",
			dvStatus, id));
	}

	SPIN_LOCK_IRQSAVE(&dvtaskQ_lock, flags);
	dvtaskQ_active = 0;
	SPIN_UNLOCK_IRQRESTORE(&dvtaskQ_lock, flags);

	return;
}

/* Search IOC page 3 to determine if this is hidden physical disk
 */
static int
mptscsih_is_phys_disk(MPT_ADAPTER *ioc, int channel, int id)
{
	struct inactive_raid_component_info *component_info;
	int i;
	int rc = 0;

	if (!ioc->raid_data.pIocPg3)
		goto out;
	for (i = 0; i < ioc->raid_data.pIocPg3->NumPhysDisks; i++) {
		if ((id == ioc->raid_data.pIocPg3->PhysDisk[i].PhysDiskID) &&
		    (channel == ioc->raid_data.pIocPg3->PhysDisk[i].PhysDiskBus)) {
			rc = 1;
			goto out;
		}
	}

	/*
	 * Check inactive list for matching phys disks
	 */
	if (list_empty(&ioc->raid_data.inactive_list))
		goto out;

	down(&ioc->raid_data.inactive_list_mutex);
	list_for_each_entry(component_info,
	    &ioc->raid_data.inactive_list, list) {
		if ((component_info->d.PhysDiskID == id) &&
		    (component_info->d.PhysDiskBus == channel))
			rc = 1;
	}
	up(&ioc->raid_data.inactive_list_mutex);

 out:
	return rc;
}

/* Write SDP1 if no QAS has been enabled
 */
static void mptscsih_qas_check(MPT_SCSI_HOST *hd, int id)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice *pTarget;
	int ii;

	for (ii=0; ii < ioc->facts.MaxDevices; ii++) {
		if (ii == id)
			continue;

		if ((hd->ioc->spi_data.dvStatus[ii] & MPT_SCSICFG_DV_NOT_DONE) != 0)
			continue;

		pMptTarget = ioc->Target_List[0];
		pTarget = (VirtDevice *)pMptTarget->Target[ii];

		if (pTarget) {
			if (!pTarget->raidVolume) {
				if ((pTarget->negoFlags & hd->ioc->spi_data.noQas) == 0) {
					pTarget->negoFlags |= hd->ioc->spi_data.noQas;
					dnegoprintk(("mptscsih_qas_check: writeSDP1: id=%d negoFlags=%d\n", id, pTarget->negoFlags));
					mptscsih_writeSDP1(hd, 0, ii, MPT_SCSICFG_USE_NVRAM);
				}
			} else {
				if (mptscsih_is_phys_disk(hd->ioc, pTarget->bus, ii) == 1) {
					dnegoprintk(("mptscsih_qas_check: is_phys_disk writeSDP1: id=%d SCSICFG_USE_NVRAM\n", ii));
					mptscsih_writeSDP1(hd, 0, ii, MPT_SCSICFG_USE_NVRAM);
				}
			}
		}
	}
}



#define MPT_GET_NVRAM_VALS	0x01
#define MPT_UPDATE_MAX		0x02
#define MPT_SET_MAX		0x04
#define MPT_SET_MIN		0x08
#define MPT_FALLBACK		0x10
#define MPT_SAVE		0x20

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/**
 *	mptscsih_doDv - Perform domain validation to a target.
 *	@hd: Pointer to MPT_SCSI_HOST structure.
 *	@portnum: IOC port number.
 *	@id: Physical ID of this target
 *
 *	Uses the ISR, but with special processing.
 *	MUST be single-threaded.
 *	Test will exit if target is at async & narrow.
 *
 *	Return: None.
 */
static int
mptscsih_doDv(MPT_SCSI_HOST *hd, int bus, int id)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;
	SCSIDevicePage1_t	*pcfg1Data;
	SCSIDevicePage0_t	*pcfg0Data;
	u8			*pbuf1;
	u8			*pbuf2;
	u8			*pDvBuf;
	dma_addr_t		 dvbuf_dma = -1;
	dma_addr_t		 buf1_dma = -1;
	dma_addr_t		 buf2_dma = -1;
	dma_addr_t		 cfg1_dma_addr = -1;
	dma_addr_t		 cfg0_dma_addr = -1;
	ConfigPageHeader_t	 header1;
	ConfigPageHeader_t	 header0;
	DVPARAMETERS		 dv;
	INTERNAL_CMD		 iocmd;
	CONFIGPARMS		 cfg;
	int			 dv_alloc = 0;
	int			 rc, sz = 0;
	int			 bufsize = 0;
	int			 dataBufSize = 0;
	int			 echoBufSize = 0;
	int			 notDone;
	int			 patt;
	int			 repeat;
	int			 retcode = 0;
	int			 nfactor =  MPT_ULTRA320;
	char			 firstPass = 1;
	char			 doFallback = 0;
	char			 readPage0;
	char			 lun;
	char			 inq0 = 0;

	if (ioc->spi_data.sdp1length == 0)
		return 0;

	if (ioc->spi_data.sdp0length == 0)
		return 0;

	/* If multiple buses are used, require that the initiator
	 * id be the same on all buses.
	 */
	if (id == ioc->pfacts[0].PortSCSIID)
		return 0;

	lun = 0;
	ddvprintk((MYIOC_s_NOTE_FMT
			"DV Started: bus=%d id=%d dv @ %p\n",
			ioc->name, bus, id, &dv));
//	printk(MYIOC_s_NOTE_FMT
//			"DV Started: bus=%d id=%d dv @ %p\n",
//			ioc->name, bus, id, &dv);

	/* Prep DV structure
	 */
	memset (&dv, 0, sizeof(DVPARAMETERS));
	dv.id = id;

	/* Populate tmax with the current maximum
	 * transfer parameters for this target.
	 * Exit if narrow and async.
	 */
	dv.cmd = MPT_GET_NVRAM_VALS;
	mptscsih_dv_parms(hd, &dv, NULL);

	/* Prep SCSI IO structure
	 */
	iocmd.id = id;
	iocmd.bus = bus;
	iocmd.lun = lun;
	iocmd.flags = 0;
	iocmd.physDiskNum = -1;
	iocmd.rsvd = iocmd.rsvd2 = 0;

	pMptTarget = ioc->Target_List[bus];
	pTarget = (VirtDevice *)pMptTarget->Target[id];

	/* Use tagged commands if possible.
	 */
	if (pTarget) {
		if (pTarget->tflags & MPT_TARGET_FLAGS_Q_YES) {
			iocmd.flags |= MPT_ICFLAG_TAGGED_CMD;
			ddvprintk((MYIOC_s_NOTE_FMT
				"DV: bus=%d id=%d Tagged\n",
				ioc->name, bus, id));
		} else {
			ddvprintk((MYIOC_s_NOTE_FMT
				"DV: bus=%d id=%d Untagged\n",
				ioc->name, bus, id));
			if (hd->ioc->facts.FWVersion.Word < 0x01000600) {
				ddvprintk((MYIOC_s_NOTE_FMT
				"DV bus=%d id=%d FWVersion=%08x < 01000600!\n",
				ioc->name, bus, id, 
				hd->ioc->facts.FWVersion.Word));

				return 0;
			}

			if ((hd->ioc->facts.FWVersion.Word >= 0x01010000) &&
				(hd->ioc->facts.FWVersion.Word < 0x01010B00)) {
				ddvprintk((MYIOC_s_NOTE_FMT
				"DV bus=%d id=%d 01010000 <= FWVersion=%08x < 01010B00!\n",
				ioc->name, bus, id, 
				hd->ioc->facts.FWVersion.Word));
				return 0;
			}
		}
	}

	/* Prep cfg structure
	 */
	cfg.pageAddr = (bus<<8) | id;
	cfg.cfghdr.hdr = NULL;

	/* Prep SDP0 header
	 */
	header0.PageVersion = ioc->spi_data.sdp0version;
	header0.PageLength = ioc->spi_data.sdp0length;
	header0.PageNumber = 0;
	header0.PageType = MPI_CONFIG_PAGETYPE_SCSI_DEVICE;

	/* Prep SDP1 header
	 */
	header1.PageVersion = ioc->spi_data.sdp1version;
	header1.PageLength = ioc->spi_data.sdp1length;
	header1.PageNumber = 1;
	header1.PageType = MPI_CONFIG_PAGETYPE_SCSI_DEVICE;

	if (header0.PageLength & 1)
		dv_alloc = (header0.PageLength * 4) + 4;

	dv_alloc +=  (2048 + (header1.PageLength * 4));

	pDvBuf = pci_alloc_consistent(ioc->pcidev, dv_alloc, &dvbuf_dma);
	if (pDvBuf == NULL)
		return 0;

	sz = 0;
	pbuf1 = (u8 *)pDvBuf;
	buf1_dma = dvbuf_dma;
	sz +=1024;

	pbuf2 = (u8 *) (pDvBuf + sz);
	buf2_dma = dvbuf_dma + sz;
	sz +=1024;

	pcfg0Data = (SCSIDevicePage0_t *) (pDvBuf + sz);
	cfg0_dma_addr = dvbuf_dma + sz;
	sz += header0.PageLength * 4;

	/* 8-byte alignment
	 */
	if (header0.PageLength & 1)
		sz += 4;

	pcfg1Data = (SCSIDevicePage1_t *) (pDvBuf + sz);
	cfg1_dma_addr = dvbuf_dma + sz;

	/* Skip this ID? Set cfg.cfghdr.hdr to force config page write
	 */
	{
		ScsiCfgData *pspi_data = &hd->ioc->spi_data;
		if (pspi_data->nvram && (pspi_data->nvram[id] != MPT_HOST_NVRAM_INVALID)) {
			/* Set the factor from nvram */
			nfactor = (pspi_data->nvram[id] & MPT_NVRAM_SYNC_MASK) >> 8;
			if (nfactor < pspi_data->minSyncFactor )
				nfactor = pspi_data->minSyncFactor;
	
			if (!(pspi_data->nvram[id] & MPT_NVRAM_ID_SCAN_ENABLE) ||
				(pspi_data->PortFlags == MPI_SCSIPORTPAGE2_PORT_FLAGS_OFF_DV) ) {
	
				ddvprintk((MYIOC_s_NOTE_FMT "DV Skipped: bus, id, lun (%d, %d, %d)\n",
					ioc->name, bus, id, lun));
	
				dv.cmd = MPT_SET_MAX;
				mptscsih_dv_parms(hd, &dv, (void *)pcfg1Data);
				cfg.cfghdr.hdr = &header1;

				/* Save the final negotiated settings to
				 * SCSI device page 1.
				 */
				cfg.physAddr = cfg1_dma_addr;
				cfg.action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
				cfg.dir = 1;
				mpt_config(hd->ioc, &cfg);
				goto target_done;
			}
		} else {
			ddvprintk((MYIOC_s_NOTE_FMT
				"DV bus=%d id=%d nvram[id]=%x == MPT_HOST_NVRAM_INVALID!\n",
				ioc->name, bus, id,
				pspi_data->nvram[id]));
		}
	}

	/* Finish iocmd inititialization - hidden or visible disk? */
	if (ioc->raid_data.pIocPg3) {
		/* Search IOC page 3 for matching id
		 */
		Ioc3PhysDisk_t *pPDisk =  ioc->raid_data.pIocPg3->PhysDisk;
		int		numPDisk = ioc->raid_data.pIocPg3->NumPhysDisks;

		while (numPDisk) {
			if (pPDisk->PhysDiskID == id) {
				/* match */
				iocmd.flags |= MPT_ICFLAG_PHYS_DISK;
				iocmd.physDiskNum = pPDisk->PhysDiskNum;

				/* Quiesce the IM
				 */
				if (mptscsih_do_raid(hd, MPI_RAID_ACTION_QUIESCE_PHYS_IO, &iocmd) < 0) {
					ddvprintk((MYIOC_s_ERR_FMT "RAID Queisce FAILED!\n", ioc->name));
					goto target_done;
				}
//				break;
			}
			pPDisk++;
			numPDisk--;
		}
	}

	/* Basic Test.
	 * Async & Narrow - Inquiry
	 * Async & Narrow - Inquiry
	 * Maximum transfer rate - Inquiry
	 * Compare buffers:
	 *	If compare, test complete.
	 *	If miscompare and first pass, repeat
	 *	If miscompare and not first pass, fall back and repeat
	 */
	hd->pLocal = NULL;
	readPage0 = 0;
	sz = SCSI_MAX_INQUIRY_BYTES;
	rc = MPT_SCANDV_GOOD;
	while (1) {
		retcode = 0;
		ddvprintk((MYIOC_s_NOTE_FMT "DV: Start Basic test on id=%d\n", 
			ioc->name, id));
		dv.cmd = MPT_SET_MIN;
		mptscsih_dv_parms(hd, &dv, (void *)pcfg1Data);

		cfg.cfghdr.hdr = &header1;
		cfg.physAddr = cfg1_dma_addr;
		cfg.action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
		cfg.dir = 1;
		if (mpt_config(hd->ioc, &cfg) != 0) {
			ddvprintk((MYIOC_s_NOTE_FMT "DV: mpt_config failed on id=%d\n", ioc->name, id));
			goto target_done;
		}

		/* Wide - narrow - wide workaround case
		 */
		if ((rc == MPT_SCANDV_ISSUE_SENSE) && dv.max.width) {
			/* Send an untagged command to reset disk Qs corrupted
			 * when a parity error occurs on a Request Sense.
			 */
			if ((hd->ioc->facts.FWVersion.Word >= 0x01000600) ||
				((hd->ioc->facts.FWVersion.Word >= 0x01010000) &&
				(hd->ioc->facts.FWVersion.Word < 0x01010B00)) ) {

				iocmd.cmd = CMD_RequestSense;
				iocmd.data_dma = buf1_dma;
				iocmd.data = pbuf1;
				iocmd.size = 0x12;
				if (mptscsih_do_cmd(hd, &iocmd) < 0)
					goto target_done;
				else {
					if (hd->pLocal == NULL)
						goto target_done;
					rc = hd->pLocal->completion;
					if ((rc == MPT_SCANDV_GOOD) || (rc == MPT_SCANDV_SENSE)) {
						dv.max.width = 0;
						doFallback = 0;
					} else
						goto target_done;
				}
			} else
				goto target_done;
		}

		iocmd.cmd = CMD_Inquiry;
		iocmd.data_dma = buf1_dma;
		iocmd.data = pbuf1;
		iocmd.size = sz;
		memset(pbuf1, 0x00, sz);
		if (mptscsih_do_cmd(hd, &iocmd) < 0)
			goto target_done;
		else {
			if (hd->pLocal == NULL)
				goto target_done;
			rc = hd->pLocal->completion;
			if (rc == MPT_SCANDV_GOOD) {
				if (hd->pLocal->scsiStatus == STS_BUSY) {
					if ((iocmd.flags & MPT_ICFLAG_TAGGED_CMD) == 0)
						retcode = 1;
					else
						retcode = 0;

					goto target_done;
				}
			} else if  (rc == MPT_SCANDV_SENSE) {
				;
			} else {
				/* If first command doesn't complete
				 * with a good status or with a check condition,
				 * exit.
				 */
				goto target_done;
			}
		}

		/* Reset the size for disks
		 */
		inq0 = (*pbuf1) & 0x1F;
		if ((inq0 == 0) && pTarget && !pTarget->raidVolume) {
			sz = 0x40;
			iocmd.size = sz;
		}

		/* Another GEM workaround. Check peripheral device type,
		 * if PROCESSOR, quit DV.
		 */
		if (inq0 == SCSI_TYPE_PROC) { 
			goto target_done;
		}

		if (inq0 > 0x08)
			goto target_done;

		if (mptscsih_do_cmd(hd, &iocmd) < 0)
			goto target_done;

		if (sz == 0x40) {
			if ((pTarget->maxWidth == 1) && (pTarget->maxOffset) && (nfactor < 0x0A)
				&& (pTarget->minSyncFactor > 0x09)) {
				if ((pbuf1[56] & 0x04) == 0)
					;
				else if ((pbuf1[56] & 0x01) == 1) {
					pTarget->minSyncFactor = nfactor > MPT_ULTRA320 ? nfactor : MPT_ULTRA320;
				} else {
					pTarget->minSyncFactor = nfactor > MPT_ULTRA160 ? nfactor : MPT_ULTRA160;
				}

				dv.max.factor = pTarget->minSyncFactor;

				if ((pbuf1[56] & 0x02) == 0) {
					pTarget->negoFlags |= MPT_TARGET_NO_NEGO_QAS;
					hd->ioc->spi_data.noQas = MPT_TARGET_NO_NEGO_QAS;
					ddvprintk((MYIOC_s_NOTE_FMT "DV: Start Basic noQas on id=%d due to pbuf1[56]=%x\n", ioc->name, id, pbuf1[56]));
				}
			}
		}

		if (doFallback)
			dv.cmd = MPT_FALLBACK;
		else
			dv.cmd = MPT_SET_MAX;

		mptscsih_dv_parms(hd, &dv, (void *)pcfg1Data);
		if (mpt_config(hd->ioc, &cfg) != 0)
			goto target_done;

		if ((!dv.now.width) && (!dv.now.offset))
			goto target_done;

		iocmd.cmd = CMD_Inquiry;
		iocmd.data_dma = buf2_dma;
		iocmd.data = pbuf2;
		iocmd.size = sz;
		memset(pbuf2, 0x00, sz);
		if (mptscsih_do_cmd(hd, &iocmd) < 0)
			goto target_done;
		else if (hd->pLocal == NULL)
			goto target_done;
		else {
			/* Save the return code.
			 * If this is the first pass,
			 * read SCSI Device Page 0
			 * and update the target max parameters.
			 */
			rc = hd->pLocal->completion;
			doFallback = 0;
			if (rc == MPT_SCANDV_GOOD) {
				if (!readPage0) {
					u32 sdp0_info;
					u32 sdp0_nego;

					cfg.cfghdr.hdr = &header0;
					cfg.physAddr = cfg0_dma_addr;
					cfg.action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
					cfg.dir = 0;

					if (mpt_config(hd->ioc, &cfg) != 0)
						goto target_done;

					sdp0_info = le32_to_cpu(pcfg0Data->Information) & 0x0E;
					sdp0_nego = (le32_to_cpu(pcfg0Data->NegotiatedParameters) & 0xFF00 ) >> 8;

					/* Quantum and Fujitsu workarounds.
					 * Quantum: PPR U320 -> PPR reply with Ultra2 and wide
					 * Fujitsu: PPR U320 -> Msg Reject and Ultra2 and wide
					 * Resetart with a request for U160.
					 */
					if ((dv.now.factor == MPT_ULTRA320) && (sdp0_nego == MPT_ULTRA2)) {
							doFallback = 1;
					} else {
						dv.cmd = MPT_UPDATE_MAX;
						mptscsih_dv_parms(hd, &dv, (void *)pcfg0Data);
						/* Update the SCSI device page 1 area
						 */
						pcfg1Data->RequestedParameters = pcfg0Data->NegotiatedParameters;
						readPage0 = 1;
					}
				}

				/* Quantum workaround. Restart this test will the fallback
				 * flag set.
				 */
				if (doFallback == 0) {
					if (memcmp(pbuf1, pbuf2, sz) != 0) {
						if (!firstPass)
							doFallback = 1;
					} else {
						ddvprintk((MYIOC_s_NOTE_FMT "DV:Inquiry compared id=%d\n", ioc->name, id));
						hd->ioc->spi_data.dvStatus[id] &= ~MPT_SCSICFG_DV_NOT_DONE;
						break;	/* test complete */
					}
				}


			} else if (rc == MPT_SCANDV_ISSUE_SENSE)
				doFallback = 1;	/* set fallback flag */
			else if ((rc == MPT_SCANDV_DID_RESET) ||
				 (rc == MPT_SCANDV_SENSE) || 
				 (rc == MPT_SCANDV_FALLBACK))
				doFallback = 1;	/* set fallback flag */
			else
				goto target_done;

			firstPass = 0;
		}
	}
	ddvprintk((MYIOC_s_NOTE_FMT "DV: Basic test on id=%d completed OK.\n", ioc->name, id));

	if (mpt_dv == 0)
		goto target_done;

	inq0 = (*pbuf1) & 0x1F;

	/* Continue only for disks
	 */
	if (inq0 != 0)
		goto target_done;

	ddvprintk((MYIOC_s_NOTE_FMT "DV: bus, id, lun (%d, %d, %d) PortFlags=%x\n",
			ioc->name, bus, id, lun, ioc->spi_data.PortFlags));
	if ( ioc->spi_data.PortFlags == MPI_SCSIPORTPAGE2_PORT_FLAGS_BASIC_DV_ONLY ) {
		ddvprintk((MYIOC_s_NOTE_FMT "DV Basic Only: bus, id, lun (%d, %d, %d) PortFlags=%x\n",
			ioc->name, bus, id, lun, ioc->spi_data.PortFlags));
		goto target_done;
	}

	/* Start the Enhanced Test.
	 * 0) issue TUR to clear out check conditions
	 * 1) read capacity of echo (regular) buffer
	 * 2) reserve device
	 * 3) do write-read-compare data pattern test
	 * 4) release
	 * 5) update nego parms to target struct
	 */
	cfg.cfghdr.hdr = &header1;
	cfg.physAddr = cfg1_dma_addr;
	cfg.action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
	cfg.dir = 1;

	iocmd.cmd = CMD_TestUnitReady;
	iocmd.data_dma = -1;
	iocmd.data = NULL;
	iocmd.size = 0;
	notDone = 1;
	while (notDone) {
		if (mptscsih_do_cmd(hd, &iocmd) < 0)
			goto target_done;

		if (hd->pLocal == NULL)
			goto target_done;

		rc = hd->pLocal->completion;
		if (rc == MPT_SCANDV_GOOD)
			notDone = 0;
		else if (rc == MPT_SCANDV_SENSE) {
			u8 skey = hd->pLocal->sense[2] & 0x0F;
			u8 asc = hd->pLocal->sense[12];
			u8 ascq = hd->pLocal->sense[13];
			ddvprintk((MYIOC_s_WARN_FMT
				"SenseKey:ASC:ASCQ = (%x:%02x:%02x)\n",
				ioc->name, skey, asc, ascq));

			if (skey == SK_UNIT_ATTENTION)
				notDone++; /* repeat */
			else if ((skey == SK_NOT_READY) &&
					(asc == 0x04)&&(ascq == 0x01)) {
				/* wait then repeat */
				mdelay (2000);
				notDone++;
			} else if ((skey == SK_NOT_READY) && (asc == 0x3A)) {
				/* no medium, try read test anyway */
				notDone = 0;
			} else {
				/* All other errors are fatal.
				 */
				ddvprintk((MYIOC_s_WARN_FMT "DV: fatal error.",
						ioc->name));
				goto target_done;
			}
		} else
			goto target_done;
	}

	iocmd.cmd = CMD_ReadBuffer;
	iocmd.data_dma = buf1_dma;
	iocmd.data = pbuf1;
	iocmd.size = 4;
	iocmd.flags |= MPT_ICFLAG_BUF_CAP;

	dataBufSize = 0;
	echoBufSize = 0;
	for (patt = 0; patt < 2; patt++) {
		if (patt == 0)
			iocmd.flags |= MPT_ICFLAG_ECHO;
		else
			iocmd.flags &= ~MPT_ICFLAG_ECHO;

		notDone = 1;
		while (notDone) {
			bufsize = 0;

			/* If not ready after 8 trials,
			 * give up on this device.
			 */
			if (notDone > 8)
				goto target_done;

			if (mptscsih_do_cmd(hd, &iocmd) < 0)
				goto target_done;
			else if (hd->pLocal == NULL)
				goto target_done;
			else {
				rc = hd->pLocal->completion;
				ddvprintk(("ReadBuffer Comp Code %d", rc));
				ddvprintk(("  buff: %0x %0x %0x %0x\n",
					pbuf1[0], pbuf1[1], pbuf1[2], pbuf1[3]));

				if (rc == MPT_SCANDV_GOOD) {
					notDone = 0;
					if (iocmd.flags & MPT_ICFLAG_ECHO) {
						bufsize =  ((pbuf1[2] & 0x1F) <<8) | pbuf1[3];
						if (pbuf1[0] & 0x01)
							iocmd.flags |= MPT_ICFLAG_EBOS;
					} else {
						bufsize =  pbuf1[1]<<16 | pbuf1[2]<<8 | pbuf1[3];
					}
				} else if (rc == MPT_SCANDV_SENSE) {
					u8 skey = hd->pLocal->sense[2] & 0x0F;
					u8 asc = hd->pLocal->sense[12];
					u8 ascq = hd->pLocal->sense[13];
					ddvprintk((MYIOC_s_WARN_FMT
						"SenseKey:ASC:ASCQ = (%x:%02x:%02x)\n",
						ioc->name, skey, asc, ascq));
					if (skey == SK_ILLEGAL_REQUEST) {
						notDone = 0;
					} else if (skey == SK_UNIT_ATTENTION) {
						notDone++; /* repeat */
					} else if ((skey == SK_NOT_READY) &&
						(asc == 0x04)&&(ascq == 0x01)) {
						/* wait then repeat */
						mdelay (2000);
						notDone++;
					} else {
						/* All other errors are fatal.
						 */
						ddvprintk((MYIOC_s_WARN_FMT "DV: fatal error.",
							ioc->name));
						goto target_done;
					}
				} else {
					/* All other errors are fatal
					 */
					goto target_done;
				}
			}
		}

		if (iocmd.flags & MPT_ICFLAG_ECHO)
			echoBufSize = bufsize;
		else
			dataBufSize = bufsize;
	}
	sz = 0;
	iocmd.flags &= ~MPT_ICFLAG_BUF_CAP;

	/* Use echo buffers if possible,
	 * Exit if both buffers are 0.
	 */
	if (echoBufSize > 0) {
		iocmd.flags |= MPT_ICFLAG_ECHO;
		if (dataBufSize > 0)
			bufsize = min(echoBufSize, dataBufSize);
		else
			bufsize = echoBufSize;
	} else if (dataBufSize == 0)
		goto target_done;

	ddvprintk((MYIOC_s_WARN_FMT "%s Buffer Capacity %d\n", ioc->name,
		(iocmd.flags & MPT_ICFLAG_ECHO) ? "Echo" : " ", bufsize));

	/* Data buffers for write-read-compare test max 1K.
	 */
	sz = min(bufsize, 1024);

	/* --- loop ----
	 * On first pass, always issue a reserve.
	 * On additional loops, only if a reset has occurred.
	 * iocmd.flags indicates if echo or regular buffer
	 */
	for (patt = 0; patt < 4; patt++) {
		ddvprintk(("Pattern %d\n", patt));
		if ((iocmd.flags & MPT_ICFLAG_RESERVED) && (iocmd.flags & MPT_ICFLAG_DID_RESET)) {
			iocmd.cmd = CMD_TestUnitReady;
			iocmd.data_dma = -1;
			iocmd.data = NULL;
			iocmd.size = 0;
			if (mptscsih_do_cmd(hd, &iocmd) < 0)
				goto target_done;

			iocmd.cmd = CMD_Release6;
			iocmd.data_dma = -1;
			iocmd.data = NULL;
			iocmd.size = 0;
			if (mptscsih_do_cmd(hd, &iocmd) < 0)
				goto target_done;
			else if (hd->pLocal == NULL)
				goto target_done;
			else {
				rc = hd->pLocal->completion;
				ddvprintk(("Release rc %d\n", rc));
				if (rc == MPT_SCANDV_GOOD)
					iocmd.flags &= ~MPT_ICFLAG_RESERVED;
				else
					goto target_done;
			}
			iocmd.flags &= ~MPT_ICFLAG_RESERVED;
		}
		iocmd.flags &= ~MPT_ICFLAG_DID_RESET;

		if (iocmd.flags & MPT_ICFLAG_EBOS)
			goto skip_Reserve;

		repeat = 5;
		while (repeat && (!(iocmd.flags & MPT_ICFLAG_RESERVED))) {
			iocmd.cmd = CMD_Reserve6;
			iocmd.data_dma = -1;
			iocmd.data = NULL;
			iocmd.size = 0;
			if (mptscsih_do_cmd(hd, &iocmd) < 0)
				goto target_done;
			else if (hd->pLocal == NULL)
				goto target_done;
			else {
				rc = hd->pLocal->completion;
				if (rc == MPT_SCANDV_GOOD) {
					iocmd.flags |= MPT_ICFLAG_RESERVED;
				} else if (rc == MPT_SCANDV_SENSE) {
					/* Wait if coming ready
					 */
					u8 skey = hd->pLocal->sense[2] & 0x0F;
					u8 asc = hd->pLocal->sense[12];
					u8 ascq = hd->pLocal->sense[13];
					ddvprintk((MYIOC_s_WARN_FMT
						"DV: Reserve Failed: ", ioc->name));
					ddvprintk(("SenseKey:ASC:ASCQ = (%x:%02x:%02x)\n",
							skey, asc, ascq));

					if ((skey == SK_NOT_READY) && (asc == 0x04)&&
									(ascq == 0x01)) {
						/* wait then repeat */
						mdelay (2000);
						notDone++;
					} else {
						ddvprintk((MYIOC_s_WARN_FMT
							"DV: Reserved Failed.", ioc->name));
						goto target_done;
					}
				} else {
					ddvprintk((MYIOC_s_WARN_FMT "DV: Reserved Failed.",
							 ioc->name));
					goto target_done;
				}
			}
		}

skip_Reserve:
		mptscsih_fillbuf(pbuf1, sz, patt, 1);
		iocmd.cmd = CMD_WriteBuffer;
		iocmd.data_dma = buf1_dma;
		iocmd.data = pbuf1;
		iocmd.size = sz;
		if (mptscsih_do_cmd(hd, &iocmd) < 0)
			goto target_done;
		else if (hd->pLocal == NULL)
			goto target_done;
		else {
			rc = hd->pLocal->completion;
			if (rc == MPT_SCANDV_GOOD)
				;		/* Issue read buffer */
			else if (rc == MPT_SCANDV_DID_RESET) {
				/* If using echo buffers, reset to data buffers.
				 * Else do Fallback and restart
				 * this test (re-issue reserve
				 * because of bus reset).
				 */
				if ((iocmd.flags & MPT_ICFLAG_ECHO) && (dataBufSize >= bufsize)) {
					iocmd.flags &= ~MPT_ICFLAG_ECHO;
				} else {
					dv.cmd = MPT_FALLBACK;
					mptscsih_dv_parms(hd, &dv, (void *)pcfg1Data);

					if (mpt_config(hd->ioc, &cfg) != 0)
						goto target_done;

					if ((!dv.now.width) && (!dv.now.offset))
						goto target_done;
				}

				iocmd.flags |= MPT_ICFLAG_DID_RESET;
				patt = -1;
				continue;
			} else if (rc == MPT_SCANDV_SENSE) {
				/* Restart data test if UA, else quit.
				 */
				u8 skey = hd->pLocal->sense[2] & 0x0F;
				ddvprintk((MYIOC_s_WARN_FMT
					"SenseKey:ASC:ASCQ = (%x:%02x:%02x)\n", ioc->name, skey,
					hd->pLocal->sense[12], hd->pLocal->sense[13]));
				if (skey == SK_UNIT_ATTENTION) {
					patt = -1;
					continue;
				} else if (skey == SK_ILLEGAL_REQUEST) {
					if (iocmd.flags & MPT_ICFLAG_ECHO) {
						if (dataBufSize >= bufsize) {
							iocmd.flags &= ~MPT_ICFLAG_ECHO;
							patt = -1;
							continue;
						}
					}
					goto target_done;
				}
				else
					goto target_done;
			} else {
				/* fatal error */
				goto target_done;
			}
		}

		iocmd.cmd = CMD_ReadBuffer;
		iocmd.data_dma = buf2_dma;
		iocmd.data = pbuf2;
		iocmd.size = sz;
		if (mptscsih_do_cmd(hd, &iocmd) < 0)
			goto target_done;
		else if (hd->pLocal == NULL)
			goto target_done;
		else {
			rc = hd->pLocal->completion;
			if (rc == MPT_SCANDV_GOOD) {
				 /* If buffers compare,
				  * go to next pattern,
				  * else, do a fallback and restart
				  * data transfer test.
				  */
				if (memcmp (pbuf1, pbuf2, sz) == 0) {
					; /* goto next pattern */
				} else {
					/* Miscompare with Echo buffer, go to data buffer,
					 * if that buffer exists.
					 * Miscompare with Data buffer, check first 4 bytes,
					 * some devices return capacity. Exit in this case.
					 */
					if (iocmd.flags & MPT_ICFLAG_ECHO) {
						if (dataBufSize >= bufsize)
							iocmd.flags &= ~MPT_ICFLAG_ECHO;
						else
							goto target_done;
					} else {
						if (dataBufSize == (pbuf2[1]<<16 | pbuf2[2]<<8 | pbuf2[3])) {
							/* Argh. Device returning wrong data.
							 * Quit DV for this device.
							 */
							goto target_done;
						}

						/* Had an actual miscompare. Slow down.*/
						dv.cmd = MPT_FALLBACK;
						mptscsih_dv_parms(hd, &dv, (void *)pcfg1Data);

						if (mpt_config(hd->ioc, &cfg) != 0)
							goto target_done;

						if ((!dv.now.width) && (!dv.now.offset))
							goto target_done;
					}

					patt = -1;
					continue;
				}
			} else if (rc == MPT_SCANDV_DID_RESET) {
				/* Do Fallback and restart
				 * this test (re-issue reserve
				 * because of bus reset).
				 */
				dv.cmd = MPT_FALLBACK;
				mptscsih_dv_parms(hd, &dv, (void *)pcfg1Data);

				if (mpt_config(hd->ioc, &cfg) != 0)
					 goto target_done;

				if ((!dv.now.width) && (!dv.now.offset))
					goto target_done;

				iocmd.flags |= MPT_ICFLAG_DID_RESET;
				patt = -1;
				continue;
			} else if (rc == MPT_SCANDV_SENSE) {
				/* Restart data test if UA, else quit.
				 */
				u8 skey = hd->pLocal->sense[2] & 0x0F;
				ddvprintk((MYIOC_s_WARN_FMT
					"SenseKey:ASC:ASCQ = (%x:%02x:%02x)\n", ioc->name, skey,
					hd->pLocal->sense[12], hd->pLocal->sense[13]));
				if (skey == SK_UNIT_ATTENTION) {
					patt = -1;
					continue;
				}
				else
					goto target_done;
			} else {
				/* fatal error */
				goto target_done;
			}
		}

	} /* --- end of patt loop ---- */

target_done:
	if (iocmd.flags & MPT_ICFLAG_RESERVED) {
		iocmd.cmd = CMD_Release6;
		iocmd.data_dma = -1;
		iocmd.data = NULL;
		iocmd.size = 0;
		if (mptscsih_do_cmd(hd, &iocmd) < 0)
			printk(MYIOC_s_WARN_FMT "DV: Release failed. id %d",
					ioc->name, id);
		else if (hd->pLocal) {
			if (hd->pLocal->completion == MPT_SCANDV_GOOD)
				iocmd.flags &= ~MPT_ICFLAG_RESERVED;
		} else {
			printk(MYIOC_s_WARN_FMT "DV: Release failed. id %d",
						ioc->name, id);
		}
	}


	/* Set if cfg1_dma_addr contents is valid
	 */
	if ((cfg.cfghdr.hdr != NULL) && (retcode == 0)){
		/* If disk, not U320, disable QAS
		 */
		if ((inq0 == 0) && (dv.now.factor > MPT_ULTRA320)) {
			hd->ioc->spi_data.noQas = MPT_TARGET_NO_NEGO_QAS;
			ddvprintk((MYIOC_s_NOTE_FMT "noQas set due to id=%d has factor=%x\n", ioc->name, id, dv.now.factor));
		}

		dv.cmd = MPT_SAVE;
		mptscsih_dv_parms(hd, &dv, (void *)pcfg1Data);

		/* Double writes to SDP1 can cause problems,
		 * skip save of the final negotiated settings to
		 * SCSI device page 1.
		 *
		cfg.cfghdr.hdr = &header1;
		cfg.physAddr = cfg1_dma_addr;
		cfg.action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
		cfg.dir = 1;
		mpt_config(hd->ioc, &cfg);
		 */
	}

	/* If this is a RAID Passthrough, enable internal IOs
	 */
	if (iocmd.flags & MPT_ICFLAG_PHYS_DISK) {
		if (mptscsih_do_raid(hd, MPI_RAID_ACTION_ENABLE_PHYS_IO, &iocmd) < 0)
			ddvprintk((MYIOC_s_ERR_FMT "RAID Enable FAILED!\n", ioc->name));
	}

	/* Done with the DV scan of the current target
	 */
	if (pDvBuf)
		pci_free_consistent(ioc->pcidev, dv_alloc, pDvBuf, dvbuf_dma);
#if 1 || defined(__VMKERNEL_MODULE__) || defined(CONFIG_VMNIX)
	ioc->spi_data.dvStatus[id] &= ~MPT_SCSICFG_DV_NOT_DONE;
	if (ioc->revisionID > 1) {
		/* revisionID for VM is 1 and this change should exclude VM */
		ioc->spi_data.dvStatus[id] &= ~MPT_SCSICFG_DV_IN_PROGRESS;
	}
#else
	ioc->spi_data.dvStatus[id] &= ~(MPT_SCSICFG_DV_NOT_DONE | MPT_SCSICFG_DV_IN_PROGRESS);
#endif

	ddvprintk((MYIOC_s_WARN_FMT "DV Done id=%d\n",
			ioc->name, id));

	return retcode;
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_dv_parms - perform a variety of operations on the
 *	parameters used for negotiation.
 *	@hd: Pointer to a SCSI host.
 *	@dv: Pointer to a structure that contains the maximum and current
 *		negotiated parameters.
 */
static void
mptscsih_dv_parms(MPT_SCSI_HOST *hd, DVPARAMETERS *dv,void *pPage)
{
	MPT_ADAPTER		*ioc = hd->ioc;
	struct _MPT_DEVICE	*pMptTarget;
	VirtDevice		*pTarget;
	SCSIDevicePage0_t	*pPage0;
	SCSIDevicePage1_t	*pPage1;
	int			val = 0, data, configuration;
	u8			width = 0;
	u8			offset = 0;
	u8			factor = 0;
	u8			negoFlags = 0;
	u8			cmd = dv->cmd;
	u8			id = dv->id;

	switch (cmd) {
	case MPT_GET_NVRAM_VALS:
		ddvprintk((MYIOC_s_NOTE_FMT "Getting id=%d NVRAM: ",
			ioc->name, id));
		/* Get the NVRAM values and save in tmax
		 * If not an LVD bus, the adapter minSyncFactor has been
		 * already throttled back.
		 */
		negoFlags = ioc->spi_data.noQas;
		pMptTarget = ioc->Target_List[0];
		pTarget = (VirtDevice *)pMptTarget->Target[id];
		if (pTarget && !pTarget->raidVolume) {
			width = pTarget->maxWidth;
			offset = pTarget->maxOffset;
			factor = pTarget->minSyncFactor;
			negoFlags |= pTarget->negoFlags;
			ddvprintk(("non-RAID device: "));
		} else {
			if (hd->ioc->spi_data.nvram && (hd->ioc->spi_data.nvram[id] != MPT_HOST_NVRAM_INVALID)) {
				data = hd->ioc->spi_data.nvram[id];
				width = data & MPT_NVRAM_WIDE_DISABLE ? 0 : 1;
				if ((offset = hd->ioc->spi_data.maxSyncOffset) == 0)
					factor = MPT_ASYNC;
				else {
					factor = (data & MPT_NVRAM_SYNC_MASK) >> MPT_NVRAM_SYNC_SHIFT;
					if ((factor == 0) || (factor == MPT_ASYNC)){
						factor = MPT_ASYNC;
						offset = 0;
					}
				}
				ddvprintk(("NVRAM data: "));
			} else {
				width = MPT_NARROW;
				offset = 0;
				factor = MPT_ASYNC;
				ddvprintk(("MPT_HOST_NVRAM_INVALID: "));
			}

			/* Set the negotiation flags */
			if (!width)
				negoFlags |= MPT_TARGET_NO_NEGO_WIDE;

			if (!offset)
				negoFlags |= MPT_TARGET_NO_NEGO_SYNC;
		}

		/* limit by adapter capabilities */
		width = min(width, hd->ioc->spi_data.maxBusWidth);
		offset = min(offset, hd->ioc->spi_data.maxSyncOffset);
		factor = max(factor, hd->ioc->spi_data.minSyncFactor);

		/* Check Consistency */
		if (offset && (factor < MPT_ULTRA2) && !width)
			factor = MPT_ULTRA2;

		dv->max.width = width;
		dv->max.offset = offset;
		dv->max.factor = factor;
		dv->max.flags = negoFlags;
		ddvprintk((" id=%d width=%d factor=%x offset=%x negoFlags=%x\n",
				id, width, factor, offset, negoFlags));
		break;

	case MPT_UPDATE_MAX:
		ddvprintk((MYIOC_s_NOTE_FMT
			"Updating with SDP0 Data: ", hd->ioc->name));
		/* Update tmax values with those from Device Page 0.*/
		pPage0 = (SCSIDevicePage0_t *) pPage;
		if (pPage0) {
			val = le32_to_cpu(pPage0->NegotiatedParameters);
			dv->max.width = val & MPI_SCSIDEVPAGE0_NP_WIDE ? 1 : 0;
			dv->max.offset = (val&MPI_SCSIDEVPAGE0_NP_NEG_SYNC_OFFSET_MASK) >> 16;
			dv->max.factor = (val&MPI_SCSIDEVPAGE0_NP_NEG_SYNC_PERIOD_MASK) >> 8;
		}

		dv->now.width = dv->max.width;
		dv->now.offset = dv->max.offset;
		dv->now.factor = dv->max.factor;
		ddvprintk(("id=%d width=%d factor=%x offset=%x negoFlags=%x\n",
				id, dv->now.width, dv->now.factor, dv->now.offset, dv->now.flags));
		break;

	case MPT_SET_MAX:
		ddvprintk((MYIOC_s_NOTE_FMT "Setting Max: ",
								hd->ioc->name));
		/* Set current to the max values. Update the config page.*/
		dv->now.width = dv->max.width;
		dv->now.offset = dv->max.offset;
		dv->now.factor = dv->max.factor;
		dv->now.flags = dv->max.flags;

		pPage1 = (SCSIDevicePage1_t *)pPage;
		if (pPage1) {
			mptscsih_setDevicePage1Flags (dv->now.width, dv->now.factor,
				dv->now.offset, &val, &configuration, dv->now.flags);
			pPage1->RequestedParameters = cpu_to_le32(val);
			pPage1->Reserved = 0;
			pPage1->Configuration = cpu_to_le32(configuration);
		}

		ddvprintk(("id=%d width=%d factor=%x offset=%x negoFlags=%x request=%x configuration=%x\n",
				id, dv->now.width, dv->now.factor, dv->now.offset, dv->now.flags, val, configuration));
		break;

	case MPT_SET_MIN:
		ddvprintk((MYIOC_s_NOTE_FMT "Setting Min: ",
								hd->ioc->name));
		/* Set page to asynchronous and narrow
		 * Do not update now, breaks fallback routine. */
		width = MPT_NARROW;
		offset = 0;
		factor = MPT_ASYNC;
		negoFlags = dv->max.flags;

		pPage1 = (SCSIDevicePage1_t *)pPage;
		if (pPage1) {
			mptscsih_setDevicePage1Flags (width, factor,
				offset, &val, &configuration, negoFlags);
			pPage1->RequestedParameters = cpu_to_le32(val);
			pPage1->Reserved = 0;
			pPage1->Configuration = cpu_to_le32(configuration);
		}
		ddvprintk(("id=%d width=%d factor=%x offset=%x request=%x config=%x negoFlags=%x\n",
				id, width, factor, offset, val, configuration, negoFlags));
		break;

	case MPT_FALLBACK:
		ddvprintk((MYIOC_s_NOTE_FMT
			"Fallback: Start: offset %d, factor %x, width %d \n",
				hd->ioc->name, dv->now.offset,
				dv->now.factor, dv->now.width));
		width = dv->now.width;
		offset = dv->now.offset;
		factor = dv->now.factor;
		if ((offset) && (dv->max.width)) {
			if (factor < MPT_ULTRA160)
				factor = MPT_ULTRA160;
			else if (factor < MPT_ULTRA2) {
				factor = MPT_ULTRA2;
				width = MPT_WIDE;
			} else if ((factor == MPT_ULTRA2) && width) {
				width = MPT_NARROW;
			} else if (factor < MPT_ULTRA) {
				factor = MPT_ULTRA;
				width = MPT_WIDE;
			} else if ((factor == MPT_ULTRA) && width) {
				width = MPT_NARROW;
			} else if (factor < MPT_FAST) {
				factor = MPT_FAST;
				width = MPT_WIDE;
			} else if ((factor == MPT_FAST) && width) {
				width = MPT_NARROW;
			} else if (factor < MPT_SCSI) {
				factor = MPT_SCSI;
				width = MPT_WIDE;
			} else if ((factor == MPT_SCSI) && width) {
				width = MPT_NARROW;
			} else {
				factor = MPT_ASYNC;
				offset = 0;
			}

		} else if (offset) {
			width = MPT_NARROW;
			if (factor < MPT_ULTRA)
				factor = MPT_ULTRA;
			else if (factor < MPT_FAST)
				factor = MPT_FAST;
			else if (factor < MPT_SCSI)
				factor = MPT_SCSI;
			else {
				factor = MPT_ASYNC;
				offset = 0;
			}

		} else {
			width = MPT_NARROW;
			factor = MPT_ASYNC;
		}
		dv->max.flags |= MPT_TARGET_NO_NEGO_QAS;
		dv->max.flags &= ~MPT_TAPE_NEGO_IDP;

		dv->now.width = width;
		dv->now.offset = offset;
		dv->now.factor = factor;
		dv->now.flags = dv->max.flags;

		pPage1 = (SCSIDevicePage1_t *)pPage;
		if (pPage1) {
			mptscsih_setDevicePage1Flags (width, factor, offset, &val,
						&configuration, dv->now.flags);

			pPage1->RequestedParameters = cpu_to_le32(val);
			pPage1->Reserved = 0;
			pPage1->Configuration = cpu_to_le32(configuration);
		}

		ddvprintk(("Finish: id=%d offset=%d factor=%x width=%d request=%x config=%x\n",
			     id, dv->now.offset, dv->now.factor, dv->now.width, val, configuration));
		break;

	case MPT_SAVE:
		ddvprintk((MYIOC_s_NOTE_FMT
			"Saving to Target structure: ", hd->ioc->name));
		ddvprintk(("id=%d width=%x factor=%x offset=%d negoFlags=%x\n",
			     id, dv->now.width, dv->now.factor, dv->now.offset, dv->now.flags));

		/* Save these values to target structures
		 * or overwrite nvram (phys disks only).
		 */

		pMptTarget = ioc->Target_List[0];
		pTarget = (VirtDevice *)pMptTarget->Target[id];
		if (pTarget && !pTarget->raidVolume ) {
			pTarget->maxWidth = dv->now.width;
			pTarget->maxOffset = dv->now.offset;
			pTarget->minSyncFactor = dv->now.factor;
			pTarget->negoFlags = dv->now.flags;
		} else {
			/* Preserv all flags, use
			 * read-modify-write algorithm
			 */
			if (hd->ioc->spi_data.nvram) {
				data = hd->ioc->spi_data.nvram[id];

				if (dv->now.width)
					data &= ~MPT_NVRAM_WIDE_DISABLE;
				else
					data |= MPT_NVRAM_WIDE_DISABLE;

				if (!dv->now.offset)
					factor = MPT_ASYNC;

				data &= ~MPT_NVRAM_SYNC_MASK;
				data |= (dv->now.factor << MPT_NVRAM_SYNC_SHIFT) & MPT_NVRAM_SYNC_MASK;

				hd->ioc->spi_data.nvram[id] = data;
			}
		}
		break;
	}
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*	mptscsih_fillbuf - fill a buffer with a special data pattern
 *		cleanup. For bus scan only.
 *
 *	@buffer: Pointer to data buffer to be filled.
 *	@size: Number of bytes to fill
 *	@index: Pattern index
 *	@width: bus width, 0 (8 bits) or 1 (16 bits)
 */
static void
mptscsih_fillbuf(char *buffer, int size, int index, int width)
{
	char *ptr = buffer;
	int ii;
	char byte;
	short val;

	switch (index) {
	case 0:

		if (width) {
			/* Pattern:  0000 FFFF 0000 FFFF
			 */
			for (ii=0; ii < size; ii++, ptr++) {
				if (ii & 0x02)
					*ptr = 0xFF;
				else
					*ptr = 0x00;
			}
		} else {
			/* Pattern:  00 FF 00 FF
			 */
			for (ii=0; ii < size; ii++, ptr++) {
				if (ii & 0x01)
					*ptr = 0xFF;
				else
					*ptr = 0x00;
			}
		}
		break;

	case 1:
		if (width) {
			/* Pattern:  5555 AAAA 5555 AAAA 5555
			 */
			for (ii=0; ii < size; ii++, ptr++) {
				if (ii & 0x02)
					*ptr = 0xAA;
				else
					*ptr = 0x55;
			}
		} else {
			/* Pattern:  55 AA 55 AA 55
			 */
			for (ii=0; ii < size; ii++, ptr++) {
				if (ii & 0x01)
					*ptr = 0xAA;
				else
					*ptr = 0x55;
			}
		}
		break;

	case 2:
		/* Pattern:  00 01 02 03 04 05
		 * ... FE FF 00 01..
		 */
		for (ii=0; ii < size; ii++, ptr++)
			*ptr = (char) ii;
		break;

	case 3:
		if (width) {
			/* Wide Pattern:  FFFE 0001 FFFD 0002
			 * ...  4000 DFFF 8000 EFFF
			 */
			byte = 0;
			for (ii=0; ii < size/2; ii++) {
				/* Create the base pattern
				 */
				val = (1 << byte);
				/* every 64 (0x40) bytes flip the pattern
				 * since we fill 2 bytes / iteration,
				 * test for ii = 0x20
				 */
				if (ii & 0x20)
					val = ~(val);

				if (ii & 0x01) {
					*ptr = (char)( (val & 0xFF00) >> 8);
					ptr++;
					*ptr = (char)(val & 0xFF);
					byte++;
					byte &= 0x0F;
				} else {
					val = ~val;
					*ptr = (char)( (val & 0xFF00) >> 8);
					ptr++;
					*ptr = (char)(val & 0xFF);
				}

				ptr++;
			}
		} else {
			/* Narrow Pattern:  FE 01 FD 02 FB 04
			 * .. 7F 80 01 FE 02 FD ...  80 7F
			 */
			byte = 0;
			for (ii=0; ii < size; ii++, ptr++) {
				/* Base pattern - first 32 bytes
				 */
				if (ii & 0x01) {
					*ptr = (1 << byte);
					byte++;
					byte &= 0x07;
				} else {
					*ptr = (char) (~(1 << byte));
				}

				/* Flip the pattern every 32 bytes
				 */
				if (ii & 0x20)
					*ptr = ~(*ptr);
			}
		}
		break;
	}
}
#endif /* ~MPTSCSIH_ENABLE_DOMAIN_VALIDATION */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,32)
#if defined(CONFIG_DISKDUMP) || defined(CONFIG_DISKDUMP_MODULE)
static int
mptscsih_sanity_check(Scsi_Device *SDev)
{
	MPT_SCSI_HOST *hd;
	MPT_ADAPTER *ioc;

	hd = (MPT_SCSI_HOST *) SDev->host->hostdata;
	if (hd == NULL)
		return -ENXIO;
	ioc = hd->ioc;

	/* message frame freeQ is busy */
	if (spin_is_locked(&ioc->FreeQlock))
		return -EBUSY;

	return 0;
}

static int
mptscsih_quiesce(Scsi_Device *SDev)
{
	MPT_HOST_LOCK_INIT();
	return 0;
}

static void
mptscsih_poll(Scsi_Device *SDev)
{
	MPT_SCSI_HOST *hd;
	u32 intstat;

	hd = (MPT_SCSI_HOST *) SDev->host->hostdata;
	if (!hd)
		return;

	/* check interrupt pending */
	mpt_poll_interrupt(hd->ioc);
}
#endif
#endif

#ifdef __VMKERNEL_MODULE__

#define to_mpt_scsi_host(x)          ((MPT_SCSI_HOST *) (x)->hostdata)

/*
 *----------------------------------------------------------------------------
 *
 *  mpt2xx_get_sas_device --
 *     Returns the mpt target SAS device corresponding to the scsi target
 *
 *  Results:
 *
 *  Side effects:
 *
 *
 *----------------------------------------------------------------------------
 */
static struct sas_device_info *
mpt2xx_get_sas_device(struct scsi_target *starget)
{
	MPT_SCSI_HOST	     *hd = to_mpt_scsi_host(starget->sh);
        MPT_ADAPTER          *ioc = hd->ioc;
        struct sas_device_info    *sasDevice;

        if (ioc->bus_type != SAS) {
           printk(MYIOC_s_WARN_FMT " Not a SAS controller in mpt2xx_get_sas_device \n", ioc->name);
           return NULL;
        }

        if (starget->id >= ioc->DevicesPerBus) {
           printk(MYIOC_s_WARN_FMT "mpt2xx_get_virt_device channel=%d Invalid id=%d DevicesPerBus=%d\n",
                                    ioc->name, starget->channel, starget->id, ioc->DevicesPerBus);
           return NULL;
        }
#if 0 && !defined(__VMWARE__)
	down(&ioc->sas_device_info_mutex);
#endif
        list_for_each_entry(sasDevice, &ioc->sas_device_info_list, list) {
           if ((starget->channel == sasDevice->os.channel) &&
               (starget->id == sasDevice->os.id)) {
#if 0 && !defined(__VMWARE__)
			up(&ioc->sas_device_info_mutex);
#endif
			return sasDevice;
           }
        }
#if 0 && !defined(__VMWARE__)
	up(&ioc->sas_device_info_mutex);
#endif
        return NULL; 
}

/*
 *----------------------------------------------------------------------------
 *
 *  mpt2xx_get_target_sas_address --
 *    Returns the target SAS address
 *
 *  Results:
 *
 *  Side effects:
 *
 *
 *----------------------------------------------------------------------------
 */
static void
mpt2xx_get_target_sas_address(struct scsi_target *starget)
{
        struct sas_device_info    *sasDevice;
        

        sasDevice = mpt2xx_get_sas_device(starget);

        if (sasDevice) {
           sas_starget_sas_address(starget) = sasDevice->sas_address;
           return;
        }

	sas_starget_sas_address(starget) = -1;
}
#endif
