#include "3c90x.h"

#ifdef __VMKERNEL_MODULE__
#include "vmklinux_dist.h"
#endif
#include "3c90x_stress.h"
#ifdef NICE_SUPPORT
static char* version = 
        "3Com 3c90x Ethernet Driver Version 1.0.2 2001 with B NIC Extension (NICE) <linux_drivers@3com.com>\n";
#else
static char *version =
        "3Com 3c90x Ethernet Driver Version 1.0.2 2001 <linux_drivers@3com.com>\n";
#endif  /* NICE_SUPPORT */

/*
   3Com EtherLink 10/100 PCI (3C90x) Linux Network Driver, Copyright (c) 1999
   3Com Corporation. All rights reserved.
   3Com Linux Network Driver software is distributed as is, without any warranty
   of any kind, either express or implied as further specified in the GNU Public
   License. This software may be used and distributed according to the terms of
   the GNU Public License, located in the file LICENSE.

   3Com and EtherLink are registered trademarks of 3Com Corporation. Linux is a
   registered trademarks of Linus Torvalds.


   Credit
   ------
   Special thanks goes to Donald Becker for providing the skeleton driver outline
   used in this driver, and for the hard work he has put into the 3c59x EtherLink
   The 3Com 3c90x driver works in cooperatioon with his 3c59x driver.

   skeleton.c: A network driver out line for Linux.
   Copyright 1993 Uninted States Government as represented by the Director, National
   Security Agency.

   Donald Becker may be reached as becker@CESDIS.gsfc.nasa.gov, or C/O Center of
   Excellence in Space Data and Information Sciences Code 930.5, Goddard Space Flight
   Center, Greenbelt MD 20771
   http://cesdis.gsfc.nasa.gov/linux/


   This is 3Com's EtherLink PCI driver for Linux.  It provides support for the
   3c90x and 3c980 network adapters listed here:

    EtherLink 10/100 PCI NICs
    3C905C Family and 3C920 ASICs EtherLink 10/100 PCI including
    the -TX and -TX-M
    3C905B Family and 3C918 ASICs EtherLink 10/100 PCI including
    the -TX -TX-M and -TX-NM
    3C905B-COMBO  EtherLink 10/100 PCI COMBO
    3C905B-T4     EtherLink 10/100 PCI T4

    EtherLink Server 10/100 PCI NICs
    3C982-TXM    EtherLink Server 10/100 Dual Head PCI
    3C980C-TX    EtherLink Server 10/100 PCI
    3C980B-TX    EtherLink Server 10/100 PCI
    3C980-TX     EtherLink Server 10/100 PCI

    EtherLink 100 PCI NIC
    3C905B-FX     EtherLink 100 PCI Fiber

    EtherLink 10 PCI NICs
    3C900B-TPO    EtherLink 10 PCI TPO
    3C900B-TPC    EtherLink 10 PCI TPC
    3C900B-COMBO  EtherLink 10 PCI COMBO
    3C900B-FL     EtherLink 10 PCI Fiber

    E-mail Support:

    - USA or Canada: 3COM_US_NIC_FAMILY@3COM.COM
    - Mexico and Latin America: AMI_HD@3com.com
    - Brazil: br-nicsupport@3com.com
    - Europe, Middle East and Africa: European_Technical_Support@3com.com
    - Asia Pacific Rim: apr_technical_support@3com.com

    URL: http://support.3com.com/infodeli/tools/nic/linux.htm


    Compile command :

       gcc -c 3c90x.c -O2 -Wall -Wstrict-prototypes -fomit-frame-pointer \
           -fno-strength-reduce -pipe -m486 -malign-loops=2 \
           -malign-jumps=2 -malign-functions=2 -DCPU=486 -DMODULE -D__KERNEL__

        + Add -D__SMP__ to the command line for SMP support
        + Add -DMODVERSIONS to the command line if your kernel was built with
          symbol versioning (RedHat, etc.)



    c-indent-level: 4
    c-basic-offset: 4
    tab-width: 8
 */



#include <linux/string.h>

#if 1 // __VMKERNEL_MODULE__
#define MAX_UNITS 32
#else
#define MAX_UNITS 8
#endif

#if defined(MODULE) && (LINUX_VERSION_CODE >= 0x20115)

MODULE_AUTHOR("3Com Corporation <linux_drivers@3com.com>");
/* MODULE_DESCRIPTION("EtherLink PCI Driver"); */

#ifdef NICE_SUPPORT
    MODULE_DESCRIPTION( "3Com 3c90x Ethernet Driver Version 1.0.2 2001 with B NIC Extension (NICE) <linux_drivers@3com.com>\n" );
#else
    MODULE_DESCRIPTION( "3Com 3c90x Ethernet Driver Version 1.0.2 2001 <linux_drivers@3com.com>\n" );
#endif

#ifdef DEBUG
MODULE_PARM(debug, "i");
#endif


MODULE_PARM(switchdelay, "1-" __MODULE_STRING(MAX_UNITS) "i");
MODULE_PARM(downpoll, "1-" __MODULE_STRING(MAX_UNITS) "i");
MODULE_PARM(flowcontrol, "1-" __MODULE_STRING(MAX_UNITS) "i");
MODULE_PARM(media_select, "1-" __MODULE_STRING(MAX_UNITS) "i");
MODULE_PARM(full_duplex, "1-" __MODULE_STRING(MAX_UNITS) "i");

#endif // MODULE && LINUX_VERSION_CODE

char kernel_version [] = UTS_RELEASE;

static const INT MTU = 1500;
PCHAR ProductName = "3Com EtherLink PCI NIC\n";

static UINT tc90x_Index = 0;

#ifndef NOSTATIC
static
#endif
UCHAR BroadcastAddr[] = {0xff,0xff,0xff,0xff,0xff,0xff};

// Global variable for waittimer
TIMER WaitTimer;
BOOLEAN InWaitTimer;
BOOLEAN DCConverterEnabledState_g;
ULONG TimeOutCount = 0;
USHORT volatile MediaStatus_g = 0;
BOOLEAN PhyResponding_g;
USHORT volatile PhyStatus_g;
UINT volatile DownListPointer_g;
UINT volatile UpListPointer_g;
UINT volatile portValue_g;
UINT volatile dmaControl_g;


PDEVICE RootNICDevice = NULL;

#ifdef __VMKERNEL_MODULE__
/*
 * we use this as a callback to allow the receive clustering
 * code to turn interrupts off when it is in polling mode
 */
int tc90x_enable_intr(PDEVICE Device, int enable)
{
   PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
    
    if(enable) {
       pAdapter->intr_enabled = 1;
       NIC_UNMASK_ALL_INTERRUPT(pAdapter);
    } else {
       pAdapter->intr_enabled = 0;
       NIC_MASK_ALL_INTERRUPT(pAdapter);
    }
    return 0;
}

#define pci_read_config_byte(_dev, _where, _val)                                                 \
              pcibios_read_config_byte(_dev->bus->number, _dev->devfn, _where, _val);      


#define pci_read_config_dword(_dev, _where, _val)                                                \
              pcibios_read_config_dword(_dev->bus->number, _dev->devfn, _where, _val);

#define pci_read_config_word(_dev, _where, _val)                                                 \
              pcibios_read_config_word(_dev->bus->number, _dev->devfn, _where, _val);

#define pci_write_config_byte(_dev, _where, _val)                                                 \
              pcibios_write_config_byte(_dev->bus->number, _dev->devfn, _where, _val);

#define pci_write_config_dword(_dev, _where, _val)                                               \
              pcibios_write_config_dword(_dev->bus->number, _dev->devfn, _where, _val);

#define pci_write_config_word(_dev, _where, _val)                                                \
              pcibios_write_config_word(_dev->bus->number, _dev->devfn, _where, _val);

#endif // __VMKERNEL_MODULE__

#ifdef MODULE
#if 0

/*++

Routine Name:

    init_module

Routine Description:

    This routine finds the adapter using the Linux calls

Arguments:

        Device - Pointer to the device structure.

Return Value:

        -ENODEV if no adapter found
        noAdapterFound if adapter(s) found

--*/

INT
init_module(VOID)
{
        NIC_STATUS nicStatus;
        DBGPRINT_INIT((KERN_CRIT "init_module: IN\n"));

        nicStatus = tc90xbc_ScanDevices(0);

        if (NIC_STATUS_SUCCESS == nicStatus) {

                DBGPRINT_FUNCTION((KERN_CRIT "init_module: OUT\n"));
		printk(KERN_INFO "%s", version);
                return 0;

        }
        DBGPRINT_ERROR((KERN_CRIT "Scan Devices failed\n"));
        DBGPRINT_FUNCTION((KERN_CRIT "init_module: OUT\n"));
        return -ENODEV;

}

#endif
#else

/*++

Routine Name:

    tc90xbc_probe

Routine Description:

    This routine finds the adapter using the Linux calls

Arguments:

        Device - Pointer to the device structure.

Return Value:

        -ENODEV if no adapter found
        noAdapterFound if adapter(s) found

--*/

INT
tc90xbc_probe(
        IN PDEVICE Device
        )
{
        static int scanned = 0;
        NIC_STATUS nicStatus;

        DBGPRINT_INIT((KERN_CRIT "tc90xbc_probe: IN\n"));

        if(scanned++)
                return -ENODEV;
        printk(KERN_INFO "%s", version);

        nicStatus = tc90xbc_ScanDevices(Device);

        if (NIC_STATUS_SUCCESS == nicStatus) {

                DBGPRINT_FUNCTION((
                        KERN_CRIT"NICScanDevices: OUT-success\n"));
                return 0;
        }
        else {

                DBGPRINT_ERROR((
                        KERN_CRIT "NICScanDevices returned error\n"));
                return -ENODEV;
        }
}

#endif

static INT tc90x_init(VOID);
static VOID tc90x_cleanup(VOID);


/*++

Routine Name:

    init_module

Routine Description:

    This routine finds the adapter using the Linux calls

Arguments:

        void:

Return Value:

        -ENODEV if no adapter found
        0 on success

--*/


int init_module(void)
{
#ifdef __VMKERNEL_MODULE__
   if (!vmk_set_module_version("%s", version)) {
      return -ENODEV;
   }
#endif
  return tc90x_init();
}
void cleanup_module(void)
{
  tc90x_cleanup();
}

static INT tc90x_init(VOID)
{
        NIC_STATUS nicStatus;
        DBGPRINT_INIT((KERN_CRIT "tc90x_init: IN\n"));

        nicStatus = tc90xbc_ScanDevices(0);

        if (NIC_STATUS_SUCCESS == nicStatus) {

                DBGPRINT_FUNCTION((KERN_CRIT "init_module: OUT\n"));
                printk(version);
                return 0;

        }
        DBGPRINT_ERROR((KERN_CRIT "Scan Devices failed\n"));
        DBGPRINT_FUNCTION((KERN_CRIT "tc90x_init: OUT\n"));
        return -ENODEV;

}

/*++

Routine Name:

    cleanup_module

Routine Description:

    This routine releases the resources.

Arguments:

        None

Return Value:

        None

--*/


static VOID tc90x_cleanup(VOID)
{
        PDEVICE nextDevice;
        PNIC_INFORMATION pAdapter;

        DBGPRINT_FUNCTION(("tc90x_cleanup: IN\n"));

        while (RootNICDevice) {

                nextDevice = ((PNIC_INFORMATION)
                              (RootNICDevice->priv))->NextDevice;

                unregister_netdev(RootNICDevice);
                pAdapter = (PNIC_INFORMATION)RootNICDevice->priv;
                tc90x_FreeAdapterResources(pAdapter);
                kfree(RootNICDevice);
                kfree(pAdapter);
                RootNICDevice = nextDevice;
        }

        DBGPRINT_FUNCTION(("tc90x_cleanup: OUT\n"));
}



NIC_STATUS
tc90xbc_ScanDevices(
        IN PDEVICE Device
        )
{
        PNIC_INFORMATION pAdapter = NULL;
        struct pci_dev* pdev = NULL;
        INT ioBaseAddress = 0;
        USHORT vendorId, deviceId;
        INT interruptVector, noAdapterFound = 0;

#if LINUX_VERSION_CODE < 0x20193
        USHORT pciCommand;
        UCHAR pciBus, pciDeviceFunction;
        static INT pciIndex = 0;
#endif

        UCHAR cacheLineSize, revisionId;
        USHORT powerManagementControl;

        DBGPRINT_INIT((KERN_CRIT "tc90xbc_ScanDevices: IN\n"));

#if LINUX_VERSION_CODE < 0x20193

        for (; pciIndex < 0xff; pciIndex++) {

                if (pcibios_find_class(
                        PCI_CLASS_NETWORK_ETHERNET << 8,
                        pciIndex,
                        &pciBus,
                        &pciDeviceFunction
                        ) != PCIBIOS_SUCCESSFUL)
                        break;

                pcibios_read_config_word(
                        pciBus,
                        pciDeviceFunction,
                        PCI_VENDOR_ID,
                        &vendorId);

                pcibios_read_config_word(
                        pciBus,
                        pciDeviceFunction,
                        PCI_DEVICE_ID,
                        &deviceId);

#else
        while ((pdev = pci_find_class(PCI_CLASS_NETWORK_ETHERNET<<8, pdev))) {

                vendorId = pdev->vendor;
                deviceId = pdev->device;
#endif

                if (vendorId != NIC_VENDOR_ID)
                        continue;

                switch (deviceId) {

                        case NIC_PCI_DEVICE_ID_9055:
                                DBGPRINT_INIT(("10/100 Base-TX NIC found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_9058:
                                DBGPRINT_INIT(("10/100 COMBO Deluxe board found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_9004:
                                DBGPRINT_INIT(("10Base-T TPO NIC found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_9005:
                                DBGPRINT_INIT(("10Base-T/10Base-2/AUI Combo found\n"));

                        case NIC_PCI_DEVICE_ID_9006:
                                DBGPRINT_INIT(("10Base-T/10Base-2/TPC found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_900A:
                                DBGPRINT_INIT(("10Base-FL NIC found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_905A:
                                DBGPRINT_INIT(("100Base-Fx NIC found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_9200:
                                DBGPRINT_INIT(("Tornado NIC found\n"));
                                break;
#ifdef BRIDGEPORT_SUPPORT
			case NIC_PCI_DEVICE_ID_9201:
				DBGPRINT_INIT(("Revolution NIC found\n"));
				break;
#endif
                        case NIC_PCI_DEVICE_ID_9800:
                                DBGPRINT_INIT(("10/100 Base-TX NIC(Python-H) found\n"));
                                break;


                        case NIC_PCI_DEVICE_ID_9805:
                                DBGPRINT_INIT(("10/100 Base-TX NIC(Python-T) found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_4500:
                                DBGPRINT_INIT(("10/100 Base-TX NIC(Home Network) found\n"));
                                break;

                        case NIC_PCI_DEVICE_ID_7646:
                                DBGPRINT_INIT(("10/100 Base-TX NIC(SOHO) found\n"));
                                break;

                        default:
                                DBGPRINT_INIT(("UnSupported NIC found\n"));
                                continue;
                }
                //
                // Initialize the ether device.
                //
                Device = init_etherdev(Device, 0);
                DBGPRINT_INIT((
                        KERN_CRIT "3Com device %s\n",Device->name));
                Device->priv = kmalloc(sizeof(NIC_INFORMATION), GFP_KERNEL);
                memset(Device->priv, 0, sizeof(NIC_INFORMATION));
                pAdapter = (PNIC_INFORMATION)Device->priv;
                pAdapter->Device = Device;
                //
                // Save the NIC index.
                //
                pAdapter->Index = tc90x_Index++;

#if LINUX_VERSION_CODE < 0x20193

                pcibios_read_config_word(
                        pciBus,
                        pciDeviceFunction,
                        PCI_COMMAND,
                        &pciCommand);

                if (!(pciCommand & PCI_COMMAND_MASTER)) {

                        DBGPRINT_INIT(("Enabling Bus Matering\n"));
                        pciCommand |= PCI_COMMAND_MASTER;
                }
                else {
                        DBGPRINT_INIT(("Bus Mastering enabled by BIOS\n"));
                }
#else
                pci_set_master(pdev);

#endif
                {
#if ((LINUX_VERSION_CODE >= 0x20155) && (LINUX_VERSION_CODE < 0x20193))
                        pdev = pci_find_slot( pciBus, pciDeviceFunction);
                        ioBaseAddress = pdev->base_address[0];
                        interruptVector = pdev->irq;

#elif ((LINUX_VERSION_CODE >= 0x20193) && (LINUX_VERSION_CODE <= 0x20312))
                        ioBaseAddress = pdev->base_address[0];
                        interruptVector = pdev->irq;

#elif LINUX_VERSION_CODE > 0x20312
                        ioBaseAddress = pdev->resource[0].start;
                        interruptVector = pdev->irq;

#else
                        u32 pciIoAddress;
                        u8 pciIrqLine;

                        pcibios_read_config_byte(
                                pciBus,
                                pciDeviceFunction,
                                PCI_INTERRUPT_LINE,
                                &pciIrqLine);

                        pcibios_read_config_dword(
                                pciBus,
                                pciDeviceFunction,
                                PCI_BASE_ADDRESS_0,
                                &pciIoAddress);

                        ioBaseAddress = pciIoAddress;
                        interruptVector = pciIrqLine;
#endif
                }
                ioBaseAddress &= ~1;
                //
                //Check that IO range is free.
                //
                if (check_region(ioBaseAddress, 0x40)) {

                        DBGPRINT_ERROR(("NICScan: check_region failed\n"));
                        continue;
                }
                DBGPRINT_INIT(("Irq = %x , IoAddress = %x\n",
                                        interruptVector,
                                        ioBaseAddress));

#if LINUX_VERSION_CODE < 0x20193

                pcibios_read_config_byte(
                        pciBus,
                        pciDeviceFunction,
                        PCI_REVISION_ID,
                        &revisionId);

                pcibios_read_config_byte(
                        pciBus,
                        pciDeviceFunction,
                        PCI_CACHE_LINE_SIZE,
                        &cacheLineSize);
#else
                pci_read_config_byte(pdev, PCI_REVISION_ID, &revisionId);
                pci_read_config_byte(pdev, PCI_CACHE_LINE_SIZE, &cacheLineSize);

#endif

                powerManagementControl = PCI_PME_STATUS | PCI_POWER_STATE_D0;


#if LINUX_VERSION_CODE < 0x20193

                pcibios_write_config_word(
                        pciBus,
                        pciDeviceFunction,
                        PCI_POWER_CONTROL,
                        powerManagementControl);

                pcibios_write_config_dword(
                        pciBus,
                        pciDeviceFunction,
                        PCI_BASE_ADDRESS_0,
                        ioBaseAddress);
                pcibios_write_config_byte(
                        pciBus,
                        pciDeviceFunction,
                        PCI_INTERRUPT_LINE,
                        interruptVector);

                pcibios_write_config_byte(
                        pciBus,
                        pciDeviceFunction,
                        PCI_CACHE_LINE_SIZE,
                        cacheLineSize);

                pcibios_write_config_word(
                        pciBus,
                        pciDeviceFunction,
                        PCI_COMMAND,
                        pciCommand);
#else

                pci_write_config_word(pdev, PCI_POWER_CONTROL,
                        powerManagementControl);

                pci_write_config_dword(pdev, PCI_BASE_ADDRESS_0, ioBaseAddress);

                pci_write_config_byte(pdev, PCI_INTERRUPT_LINE,
                        interruptVector);

                pci_write_config_byte(pdev, PCI_CACHE_LINE_SIZE, cacheLineSize);

                /* pci_write_config_word(pdev, PCI_COMMAND, pciCommand); */

#endif

                pAdapter->Hardware.CacheLineSize = cacheLineSize * 4;

                if ((pAdapter->Hardware.CacheLineSize % 0x10) ||
                    (!pAdapter->Hardware.CacheLineSize)) {

                        DBGPRINT_ERROR((
                                "tc90xbc_Scan: Cacheline size wrong\n"));

                        pAdapter->Hardware.CacheLineSize = 0x20;
                }
                //
                // Save the variables in adapter structure
                //
#if LINUX_VERSION_CODE >= 0x20343
                pAdapter->pciDevice = pdev;
                pdev->dma_mask = 0xffffffff; /* Mask for making sure
                                                                                        that dma address will be
                                                                    within 32 bits */
#endif
                pAdapter->IoBaseAddress = ioBaseAddress;
                pAdapter->PCI.IoBaseAddress = ioBaseAddress;
                pAdapter->PCI.InterruptVector = interruptVector;

                pAdapter->Hardware.RevisionId = revisionId;
                pAdapter->Hardware.DeviceId = deviceId;

#ifdef __SMP__
                spin_lock_init(&pAdapter->SpinLock_m); // set multicast
                spin_lock_init(&pAdapter->SpinLock_send);
                spin_lock_init(&pAdapter->SpinLock_int);
                spin_lock_init(&pAdapter->SpinLock_misc); //recv mode/close/timer
#endif
                //
                // Fill the device structure
                //
                if (tc90x_FillDeviceStructure(pAdapter) != NIC_STATUS_SUCCESS) {

                        DBGPRINT_ERROR((
                                KERN_CRIT "FillDeviceStructure failed\n"));
                        tc90x_FreeAdapterResources(pAdapter);
                        continue;
                }

                if (tc90x_ReadCommandLineChanges(pAdapter) !=
                    NIC_STATUS_SUCCESS) {

                        DBGPRINT_ERROR((
                                KERN_CRIT "ReadCommandLineChanges failed\n"
                                ));
                        tc90x_FreeAdapterResources(pAdapter);
                        continue;
                }
                //
                // Allocate Shared memory
                //
                if (tc90x_AllocateSharedMemory(pAdapter) != NIC_STATUS_SUCCESS) {

                        DBGPRINT_ERROR((
                                "tc90x_AllocateSharedMemory failed\n"
                                ));
                        tc90x_FreeAdapterResources(pAdapter);
                        continue;
                }
                //
                // Reserve the io range.
                //
                request_region(
                        Device->base_addr,
                        0x80,
                        ProductName);

                pAdapter->ResourcesReserved |= NIC_IO_PORT_REGISTERED;
                //
                // Set the root device and the next device.
                //
                ((PNIC_INFORMATION)(Device->priv))->NextDevice = RootNICDevice;
                ((PNIC_INFORMATION)(Device->priv))->Device = Device;
                RootNICDevice = Device;

#ifdef __VMKERNEL_MODULE__
                if(vmk_net_register_dev(Device, pdev, TRUE) != 0) {
                   tc90x_FreeAdapterResources(pAdapter);
                   continue;                   
                }
#endif // __VMKERNEL_MODULE__

                Device = 0;
                noAdapterFound++;
        }


        DBGPRINT_FUNCTION((KERN_CRIT "tc90xbc_ScanDevices: OUT\n"));

        if (noAdapterFound) {

                DBGPRINT_INITIALIZE(("NoAdapter = %x\n", noAdapterFound));
                return NIC_STATUS_SUCCESS;
        }
        else
                return NIC_STATUS_FAILURE;
}



/*++

Routine Name:

    tc90x_FillDeviceStructure

Routine Description:

    This routine fills the device structure

Arguments:

        Device - Pointer to the device structure.
        IoBaseAddress - IoBase address for the adapter.
        Irq - Irq of the adapter.


Return Value:

        -ENODEV if no adapter found
        noAdapterFound if adapter(s) found

--*/

NIC_STATUS
tc90x_FillDeviceStructure(
        IN PNIC_INFORMATION Adapter
        )
{

        PNIC_INFORMATION pAdapter = Adapter;
        PDEVICE device = pAdapter->Device;

        DBGPRINT_INIT(("FillDeviceStructure: IN\n"));
        device->base_addr = pAdapter->PCI.IoBaseAddress;
        device->irq = pAdapter->PCI.InterruptVector;
        device->mtu = MTU;
        //
        // Set the routine addresses
        //
        device->open = &NICOpen;
        device->hard_start_xmit  = &NICSendPacket;
        device->stop = &NICClose;
        device->get_stats = &NICGetStatistics;
        device->do_ioctl = &NICIoctl;
	device->set_mac_address = &NICSetMacAddress;
#ifdef NEW_MULTICAST
        device->set_multicast_list = &NICSetReceiveMode;
#else
        device->set_multicast_list = &NICSetMulticastList;
#endif
#if LINUX_VERSION_CODE >= 0x20343
       device->tx_timeout = &NICTxTimeOut;
       device->watchdog_timeo = TX_TIMEOUT;
#endif
#ifdef __VMKERNEL_MODULE__
       device->enable_intr = &tc90x_enable_intr;
#endif // __VMKERNEL_MODULE__
        DBGPRINT_INIT(("FillDeviceStructure: OUT\n"));
        return NIC_STATUS_SUCCESS;
}

/*++

Routine Name:

    NICTxTimeOut

Routine Description:

     TxTimeout handler that is invoked by the OS when the transmission
     duration exceeds the threshold. It resets the transmitter engine,
     updates statistics and enables the networking queue.

Arguments:

        Device - Pointer to the device structure.

Return Value:
         VOID
--*/

#if LINUX_VERSION_CODE >= 0x20343

VOID
NICTxTimeOut(
        IN PDEVICE Device
        )

{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
        if (tc90x_ResetAndEnableTransmitter(pAdapter) !=
                        NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("NICTxTimeOut: TxReset failed\n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_HUNG;
                return;
        }
        pAdapter->Statistics.TxHWErrors++;
        pAdapter->txing = 0;
        netif_start_queue(Device);
}
#endif


/*++

Routine Name:

    NICOpen

Routine Description:

    This routine opens the interface.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        0 if SUCCESS
        -ENODEV if FAILURE

--*/


INT
NICOpen(
        IN PDEVICE Device
        )

{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;

        DBGPRINT_FUNCTION(("New NICOpen: IN\n"));
        //
        // Initialize general wait timer used during driver init
        //
        init_timer(&WaitTimer);
        WaitTimer.data = (ULONG)Device;
        WaitTimer.function = &WaitTimerHandler;
        pAdapter->WaitCases = NONE;
        add_timer(&WaitTimer);
        InWaitTimer = TRUE;
        pAdapter->ResourcesReserved |= WAIT_TIMER_REGISTERED;

#if LINUX_VERSION_CODE >= 0x20200
        pAdapter->SpinLock_int = (spinlock_t) SPIN_LOCK_UNLOCKED;
        pAdapter->SpinLock_send = (spinlock_t) SPIN_LOCK_UNLOCKED;
        pAdapter->SpinLock_misc = (spinlock_t) SPIN_LOCK_UNLOCKED;
        pAdapter->SpinLock_m = (spinlock_t) SPIN_LOCK_UNLOCKED;
#endif
        //
        // Register IOBase and Irq with the OS
        //
        if (tc90x_RegisterAdapter(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("NICOpen: RegisterAdapter failed\n"));
                DBGPRINT_FUNCTION(("NICOpen: Out with ERROR\n"));
                tc90x_FreeAdapterResources(pAdapter);
                return -ENODEV;
        }

	GlobalReset (pAdapter, COMMAND_GLOBAL_RESET |
		GLOBAL_RESET_MASK_TP_AUI_RESET |
		GLOBAL_RESET_MASK_ENDEC_RESET |
		GLOBAL_RESET_MASK_AISM_RESET |
		GLOBAL_RESET_MASK_SMB_RESET |
		GLOBAL_RESET_MASK_VCO_RESET);

        if (tc90x_GetAdapterProperties(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("NICOpen: GetAdapterProperties failed\n"));
                DBGPRINT_FUNCTION(("NICOpen: Out with ERROR\n"));
                tc90x_FreeAdapterResources(pAdapter);
                return -ENODEV;
        }

        if (tc90x_BasicInitializeAdapter(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("NICOpen: BasicInitializeAdapter failed\n"));
                DBGPRINT_FUNCTION(("NICOpen: Out with error\n"));
                tc90x_FreeAdapterResources(pAdapter);
                return -ENODEV;
        }

#ifndef __VMKERNEL_MODULE__
        if (tc90x_TestAdapter(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("NICOpen: TestAdapter failed\n"));
                DBGPRINT_FUNCTION(("NICOpen: Out with error\n"));
                tc90x_FreeAdapterResources(pAdapter);
                return -ENODEV;
        }
#endif



        if (tc90x_SetupMedia(Device) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("NICOpen: SetupMedia failed\n"));
                DBGPRINT_FUNCTION(("NICOpen: Out with error\n"));
                tc90x_FreeAdapterResources(pAdapter);
                return -ENODEV;
        }

        if (tc90x_SoftwareWork(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "NICOpen: EnableSoftwareWork failed\n"));
                DBGPRINT_FUNCTION(("NICOpen: Out with error\n"));
                tc90x_FreeAdapterResources(pAdapter);
                return -ENODEV;
        }

        if (tc90x_StartAdapter(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("NICOpen: StartAdapter failed\n"));
                DBGPRINT_FUNCTION(("NICOpen: Out with error\n"));
                tc90x_FreeAdapterResources(pAdapter);
                return -ENODEV;
        }

#if LINUX_VERSION_CODE < 0x20343

        Device->tbusy = 0;
        Device->interrupt = 0;
        Device->start = 1;
#else
        pAdapter->txing = 0;
        /* pAdapter->interrupt = 0; */
        pAdapter->start = 1;
#endif
        //
        // Initialize the timer.
        //
        pAdapter->Resources.TimerInterval = 100;
        init_timer(&pAdapter->Resources.Timer);
        pAdapter->Resources.Timer.expires = RUN_AT(HZ/10);
        pAdapter->Resources.Timer.data = (ULONG)Device;
        pAdapter->Resources.Timer.function = &NICTimer;
        add_timer(&pAdapter->Resources.Timer);
        //
        // Initialize task queue for hosterr task, just in case
        //
        pAdapter->Resources.hostErr_task.routine = ReStartAdapter;
        pAdapter->Resources.hostErr_task.data = (void *)&pAdapter;
        pAdapter->Resources.hostErr_task.sync = 0;
        //
        // Timer has been registered.
        //
        pAdapter->ResourcesReserved |= NIC_TIMER_REGISTERED;

        /* We are now ready to accept transmit requeusts from
         * the queueing layer of the networking.
         */
        /* 2.3.99 changes */
#if LINUX_VERSION_CODE >= 0x20343
        netif_start_queue(Device);
#endif

        MOD_INC_USE_COUNT;

        DBGPRINT_FUNCTION(("NICOpen:  OUT with SUCCESS\n"));
        return  0;

}

/*++

Routine Name:

    NICClose

Routine Description:

    This routine closes the interface.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        -ENODEV if no adapter found
        noAdapterFound if adapter(s) found

--*/
INT
NICClose(
        IN PDEVICE Device
        )
{

        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
        PDEVICE device = pAdapter->Device;
#ifdef __VMKERNEL_MODULE__
        ULONG flags;
#endif // __VMKERNEL_MODULE__
        DBGPRINT_FUNCTION(("NICClose: IN\n"));

#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_lock_irqsave(&pAdapter->SpinLock_misc, flags); // __VMKERNEL_MODULE__
#else
        spin_lock(&pAdapter->SpinLock_misc);
#endif // __VMKERNEL_MODULE__

#endif

#if LINUX_VERSION_CODE >= 0x20343
        pAdapter->start = 0;
        pAdapter->txing = 1;
        netif_stop_queue(Device);
#else
	Device->start = 0;
	Device->tbusy = 1;
#endif
        //
        // Disable transmit and receive.
        //
        NIC_COMMAND(pAdapter, COMMAND_TX_DISABLE);
        NIC_COMMAND(pAdapter, COMMAND_RX_DISABLE);
        //
        // Wait for the transmit in progress to go off.
        //
        NIC_COMMAND(pAdapter,COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);
        MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);
        NIC_DELAY(10);

        if(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS)
        {       pAdapter->WaitCases = CHECK_TRANSMIT_IN_PROGRESS;
                TimeOutCount = jiffies + HZ;  //max = 1s
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100);
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies) ;
                if(!(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS))
                        DBGPRINT_ERROR(("NICClose: Adapter is not responding\n"));
        }
        // Issue Global reset.
        //
        GlobalReset (pAdapter,
                COMMAND_GLOBAL_RESET |
                GLOBAL_RESET_MASK_TP_AUI_RESET |
                GLOBAL_RESET_MASK_ENDEC_RESET |
                GLOBAL_RESET_MASK_AISM_RESET |
                GLOBAL_RESET_MASK_SMB_RESET |
                GLOBAL_RESET_MASK_VCO_RESET);

        NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_RESET);
        NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET);
    //
        // Mask and acknowledge all interrupts.
        //
        NIC_MASK_ALL_INTERRUPT(pAdapter);
        NIC_ACKNOWLEDGE_ALL_INTERRUPT(pAdapter);
        tc90x_CleanupSendLogic(Device);
        //
        // Unregister the interrupt handler.
        //
        if (pAdapter->ResourcesReserved & NIC_INTERRUPT_REGISTERED) {

                DBGPRINT_INITIALIZE(("Releasing interrupt\n"));
                free_irq(device->irq, device);
                pAdapter->ResourcesReserved &= ~NIC_INTERRUPT_REGISTERED;
        }
        //
        // Unregister the timer handler.
        //
        if (pAdapter->ResourcesReserved & NIC_TIMER_REGISTERED) {
                DBGPRINT_INITIALIZE((KERN_CRIT "Releasing Timer\n"));
                if (del_timer(&pAdapter->Resources.Timer)) {
                        DBGPRINT_INITIALIZE(("Timer already queued\n"));
                }
                pAdapter->ResourcesReserved &= ~NIC_TIMER_REGISTERED;
        }
        if (pAdapter->ResourcesReserved & WAIT_TIMER_REGISTERED) {
                DBGPRINT_INITIALIZE((KERN_CRIT "Releasing WaitTimer\n"));
                if (del_timer(&WaitTimer))
                        DBGPRINT_INITIALIZE(("WaitTimer already queued\n"));
                pAdapter->ResourcesReserved &= ~WAIT_TIMER_REGISTERED;
        }
        MOD_DEC_USE_COUNT;

#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_unlock_irqrestore(&pAdapter->SpinLock_misc, flags); // __VMKERNEL_MODULE__
#else
        spin_unlock(&pAdapter->SpinLock_misc); // __VMKERNEL_MODULE__
#endif // __VMKERNEL_MODULE__

#endif

        DBGPRINT_FUNCTION(("NICClose: OUT\n"));
        return 0;
}


/*++

Routine Name:

    cleanup_module

Routine Description:

    This routine releases the resources.

Arguments:

        None

Return Value:

        None

--*/

#ifdef MODULE
#if 0

VOID
cleanup_module(
        VOID
        )
{
        PDEVICE nextDevice;
        PNIC_INFORMATION pAdapter;

        DBGPRINT_FUNCTION(("cleanup_module: IN\n"));

        while (RootNICDevice) {

                nextDevice = ((PNIC_INFORMATION)
                              (RootNICDevice->priv))->NextDevice;

                unregister_netdev(RootNICDevice);
                pAdapter = (PNIC_INFORMATION)RootNICDevice->priv;
                tc90x_FreeAdapterResources(pAdapter);
                kfree(RootNICDevice);
                kfree(pAdapter);
                RootNICDevice = nextDevice;
        }

        DBGPRINT_FUNCTION(("cleanup_module: OUT\n"));
}
#endif
#endif






/*++

Routine Name:

        MainAutoSelectionRoutine

Routine Description:

        If autoselection is set, determine the connector and link speed
        by trying the various transceiver types.

Arguments:

        IN PNIC_INFORMATION Adapter
        IN USHORT Options

Return Value:

        VOID

--*/

VOID
tc90x_MainAutoSelectionRoutine(
        IN PNIC_INFORMATION Adapter,
        IN USHORT Options
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        CONNECTOR_TYPE NotUsed;
        USHORT index;

        DBGPRINT_FUNCTION(("MainAutoSelectionRoutine: IN \n"));

        pAdapter->Hardware.Connector = CONNECTOR_UNKNOWN;
        //
        // Try 100MB Connectors
        //
        if ((Options & MEDIA_OPTIONS_100BASETX_AVAILABLE) ||
            (Options & MEDIA_OPTIONS_10BASET_AVAILABLE) ||
            (Options & MEDIA_OPTIONS_MII_AVAILABLE)) {
                //
                // For 10Base-T and 100Base-TX, select autonegotiation
                // instead of autoselect before calling trymii
                //
                if ((Options & MEDIA_OPTIONS_100BASETX_AVAILABLE) ||
                    (Options & MEDIA_OPTIONS_10BASET_AVAILABLE)) {
                                pAdapter->Hardware.Connector =
                                        CONNECTOR_AUTONEGOTIATION;
                }
                else
                                pAdapter->Hardware.Connector = CONNECTOR_MII;

                DBGPRINT_INITIALIZE(("Trying MII\n"));

                if (!tc90x_TryMII(pAdapter, Options))
                        pAdapter->Hardware.Connector = CONNECTOR_UNKNOWN;
        }
        //
        // Transceiver available is 100Base-FX
        //
        if ((Options & MEDIA_OPTIONS_100BASEFX_AVAILABLE) &&
                (pAdapter->Hardware.Connector == CONNECTOR_UNKNOWN)) {
                DBGPRINT_INITIALIZE(("Trying 100BFX\n"));
                if (tc90x_TryLinkBeat(pAdapter, CONNECTOR_100BASEFX)) {
                        pAdapter->Hardware.Connector = CONNECTOR_100BASEFX;
                        pAdapter->Hardware.LinkSpeed = LINK_SPEED_100;
                }
        }
        //
        // Transceiver available is 10AUI
        //
        if ((Options & MEDIA_OPTIONS_10AUI_AVAILABLE) &&
                (pAdapter->Hardware.Connector == CONNECTOR_UNKNOWN)) {
                DBGPRINT_INITIALIZE(("Trying 10AUI\n"));
                tc90x_SetupConnector(pAdapter, CONNECTOR_10AUI, &NotUsed);
                //
                // Try to loopback packet
                //
                for (index = 0; index < 3; index++) {
                        if (TestPacket(pAdapter)) {
                                pAdapter->Hardware.Connector = CONNECTOR_10AUI;
                                pAdapter->Hardware.LinkSpeed = LINK_SPEED_10;
                                DBGPRINT_INITIALIZE(("Found AUI\n"));
                                break;
                        }
                }
                if (index == 3)
                        DBGPRINT_INITIALIZE(("Unable to find AUI\n"));
        }
        //
        // Transceiver available is 10Base-2
        //
        if ((Options & MEDIA_OPTIONS_10BASE2_AVAILABLE) &&
                (pAdapter->Hardware.Connector == CONNECTOR_UNKNOWN)) {
                DBGPRINT_INITIALIZE(("Trying 10BASEB2\n"));
                //
                // Set up the connector
                //
                tc90x_SetupConnector(pAdapter, CONNECTOR_10BASE2, &NotUsed);
                //
                // Try to loopback packet
                //
                for (index = 0; index < 3; index++) {
                        if (TestPacket(pAdapter)) {
                                pAdapter->Hardware.Connector = CONNECTOR_10BASE2;
                                pAdapter->Hardware.LinkSpeed = LINK_SPEED_10;

                                DBGPRINT_INITIALIZE(("Found 10Base2 \n"));
                                break;
                        }
                }
                if (index == 3)
                        DBGPRINT_INITIALIZE(("Unable to find 10Base2\n"));
                //
                // Disable DC converter
                //
                NIC_COMMAND(pAdapter, COMMAND_DISABLE_DC_CONVERTER);
                //
                // Check if DC convertor has been disabled
                //
                tc90x_CheckDCConverter(pAdapter, FALSE);
        }
        //
        // Nothing left to try!
        //
        if (pAdapter->Hardware.Connector == CONNECTOR_UNKNOWN) {

                pAdapter->Hardware.Connector = pAdapter->Hardware.ConfigConnector;
                pAdapter->Hardware.LinkSpeed = LINK_SPEED_10;
                DBGPRINT_INITIALIZE(("AutoSelection failed. Using default.\n"));
                DBGPRINT_INITIALIZE(("Connector: %x\n",pAdapter->Hardware.Connector));
                pAdapter->Hardware.LinkState = LINK_DOWN_AT_INIT;
        }

        tc90x_SetupConnector(
                pAdapter,
                pAdapter->Hardware.Connector,
                &NotUsed);

        DBGPRINT_FUNCTION(("MainAutoSelectionRoutine: OUT \n"));
}



/*++

Routine Name:

        tc90x_CheckDCConverter

Routine Description:


Arguments:

        IN PNIC_INFORMATION Adapter
        IN BOOLEAN EnabledState

Return Value:

    BOOLEAN

--*/

BOOLEAN
tc90x_CheckDCConverter (
        IN PNIC_INFORMATION Adapter,
        IN BOOLEAN EnabledState
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        NIC_COMMAND(pAdapter,COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);
        MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);
        NIC_DELAY(1000);

        if( ((EnabledState) && !(MediaStatus_g & MEDIA_STATUS_DC_CONVERTER_ENABLED) ) ||
                ((!EnabledState) && (MediaStatus_g & MEDIA_STATUS_DC_CONVERTER_ENABLED)) )
        {
                DCConverterEnabledState_g = EnabledState;  /* for waittimer */
                pAdapter->WaitCases = CHECK_DC_CONVERTER;
                TimeOutCount = jiffies + 3; // 30ms
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100);
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies) ;
                if( (EnabledState && !(MediaStatus_g & MEDIA_STATUS_DC_CONVERTER_ENABLED)) ||
                    (!EnabledState && (MediaStatus_g & MEDIA_STATUS_DC_CONVERTER_ENABLED))
                  )
                {       pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                        DBGPRINT_ERROR(("ConfigureDCConverter: Timeout setting DC Converter \n"));
                        return FALSE;
                }
        }
        return TRUE;
}


/*++

Routine Name:

        tc90x_SetupConnector

Routine Description:

        Setup new transceiver type in InternalConfig. Determine whether to
        set JabberGuardEnable, enableSQEStats and linkBeatEnable in MediaStatus.
        Determine if the coax transceiver also needs to be enabled/disabled.

Arguments:

        IN PNIC_INFORMATION Adapter
        IN CONNECTOR_TYPE NewConnector
        OUT PCONNECTOR_TYPE OldConnector

Return Value:

        VOID

--*/

VOID
tc90x_SetupConnector(
        IN PNIC_INFORMATION Adapter,
        IN CONNECTOR_TYPE NewConnector,
        OUT PCONNECTOR_TYPE OldConnector
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
    UINT InternalConfig = 0;
    UINT OldInternalConfig = 0;
    USHORT MediaStatus = 0;

        NIC_COMMAND(
                pAdapter,
        COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        InternalConfig = NIC_READ_PORT_UINT(
                                pAdapter,
                                INTERNAL_CONFIG_REGISTER);
        OldInternalConfig = InternalConfig;
        //
        // Save old choice
        //
        *OldConnector =
                (InternalConfig & INTERNAL_CONFIG_TRANSCEIVER_MASK) >> 20;
        //
        // Program the MII registers if forcing the configuration to 10/100BaseT.
        //
        if ((NewConnector == CONNECTOR_10BASET) ||
                (NewConnector == CONNECTOR_100BASETX)){
                //
                // Clear transceiver type and change to new transceiver type.
                //
                InternalConfig &= ~(INTERNAL_CONFIG_TRANSCEIVER_MASK);
                InternalConfig |= (CONNECTOR_AUTONEGOTIATION << 20);
                //
                // Update the internal config register. Only do this if the value has
                //changed to avoid dropping link.
                //
                if (OldInternalConfig != InternalConfig) {
                        NIC_WRITE_PORT_UINT(
                                pAdapter,
                                INTERNAL_CONFIG_REGISTER,
                                InternalConfig);
                }
                //
                // Force the MII registers to the correct settings.
                //
                if ( !CheckMIIConfiguration(
                                pAdapter,
                                (USHORT)((NewConnector == CONNECTOR_100BASETX)
                                                        ? MEDIA_OPTIONS_100BASETX_AVAILABLE
                                                        : MEDIA_OPTIONS_10BASET_AVAILABLE)) ) {
                        //
                        // If the forced configuration didn't work, check the results and see why.
                        //
                        CheckMIIAutoNegotiationStatus(pAdapter);
                        return;
                }
        }
        else {
                //
                // Clear transceiver type and change to new transceiver type
                //
                InternalConfig = InternalConfig & (~INTERNAL_CONFIG_TRANSCEIVER_MASK);
                InternalConfig |= (NewConnector << 20);
                //
                // Update the internal config register. Only do this if the value has
                // changed to avoid dropping link.
                //
                if (OldInternalConfig != InternalConfig) {
                        NIC_WRITE_PORT_UINT(
                                pAdapter,
                                INTERNAL_CONFIG_REGISTER,
                                InternalConfig);
                }
        }
    //
    // Determine whether to set enableSQEStats and linkBeatEnable
    // Automatically set JabberGuardEnable in MediaStatus register.
        //
        NIC_COMMAND(
                        pAdapter,
                        COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        MediaStatus = NIC_READ_PORT_USHORT(
                                        pAdapter,
                                        MEDIA_STATUS_REGISTER);

        MediaStatus &= ~(
                MEDIA_STATUS_SQE_STATISTICS_ENABLE |
                MEDIA_STATUS_LINK_BEAT_ENABLE |
                MEDIA_STATUS_JABBER_GUARD_ENABLE);

        MediaStatus |= MEDIA_STATUS_JABBER_GUARD_ENABLE;

    if (NewConnector == CONNECTOR_10AUI)
                MediaStatus |= MEDIA_STATUS_SQE_STATISTICS_ENABLE;

        if (NewConnector == CONNECTOR_AUTONEGOTIATION)
                MediaStatus |= MEDIA_STATUS_LINK_BEAT_ENABLE;
        else {
                if ((NewConnector == CONNECTOR_10BASET) ||
                        (NewConnector == CONNECTOR_100BASETX) ||
                        (NewConnector == CONNECTOR_100BASEFX)) {
                        if (!pAdapter->Hardware.LinkBeatDisable)
                                MediaStatus |= MEDIA_STATUS_LINK_BEAT_ENABLE;
        }
        }
        NIC_WRITE_PORT_USHORT(pAdapter, MEDIA_STATUS_REGISTER, MediaStatus);

        DBGPRINT_INITIALIZE((
                "tc90x_SetupConnector: MediaStatus = %x \n",MediaStatus));
        //
        // If configured for coax we must start the internal transceiver.
        // If not, we stop it (in case the configuration changed across a
        // warm boot).
        //
    if (NewConnector == CONNECTOR_10BASE2) {
                NIC_COMMAND(pAdapter, COMMAND_ENABLE_DC_CONVERTER);
                //
                // Check if DC convertor has been enabled
                //
                tc90x_CheckDCConverter(pAdapter, TRUE);
        }
        else {
                NIC_COMMAND(pAdapter, COMMAND_DISABLE_DC_CONVERTER);
                //
                // Check if DC convertor has been disabled
                //
                tc90x_CheckDCConverter(pAdapter, FALSE);
        }
}


/*++

Routine Name:

        tc90x_TryMII

Routine Description:

    Used to detect if 10Base-T, 100Base-TX, or external MII is available.

Arguments:

    IN pAdapter pAdapter - Pointer to the adapter structure
    IN CONNECTOR_TYPE newconnector

Return Value:
        BOOLEAN

--*/

BOOLEAN
tc90x_TryMII(
        IN PNIC_INFORMATION Adapter,
        IN USHORT MediaOptions
        )
{
    PNIC_INFORMATION pAdapter = Adapter;
    BOOLEAN Handles100Mbit = FALSE;
    //CONNECTOR_TYPE NotUsed;
    //USHORT PhyControl;
    //BOOLEAN PhyResponding;
    //USHORT PhyStatus;

    DBGPRINT_FUNCTION(("tc90x_TryMII: IN \n"));
    //
    // First see if there's anything connected to the MII
    //
    if (!FindMIIPhy(pAdapter)) {
                DBGPRINT_ERROR(("tc90x_TryMII: FindMIIPhy failed \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return FALSE;
    }
    //
    // Nowhere is it written that the register must be latched, and since
    // reset is the last bit out, the contents might not be valid.  read
    // it one more time.
    //
    /*PhyResponding_g = ReadMIIPhy(pAdapter, MII_PHY_CONTROL, &PhyControl);
    if (!PhyResponding_g) {

        DBGPRINT_ERROR(("tc90x_TryMII: Phy not responding (1) \n"));
        pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
        return FALSE;
    }*/
    //
    // Now we can read the status and try to figure out what's out there.
    //
    PhyResponding_g = ReadMIIPhy(pAdapter, MII_PHY_STATUS, (PUSHORT)&PhyStatus_g);
    if (!PhyResponding_g) {
        DBGPRINT_ERROR(("tc90x_TryMII: Phy not responding (2) \n"));
        pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
        return FALSE;
    }

    if ((PhyStatus_g & MII_STATUS_AUTO) &&
            (PhyStatus_g & MII_STATUS_EXTENDED)) {
        //
        //  If it is capable of auto negotiation, see if it has
        // been done already.
        //
        DBGPRINT_INITIALIZE(("MII Capable of Autonegotiation\n"));
        //
        // Check the current MII auto-negotiation state and see if we need to
        // start auto-neg over.
        //
        if (!CheckMIIConfiguration(pAdapter, MediaOptions))
                return FALSE;
        //
        // See if link is up...
        //
    PhyResponding_g = ReadMIIPhy(pAdapter, MII_PHY_STATUS, (PUSHORT)&PhyStatus_g);
    if (!PhyResponding_g) {
        DBGPRINT_ERROR(("tc90x_TryMII: Phy not responding (3) \n"));
        pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
        return FALSE;
    }
        if (PhyStatus_g & MII_STATUS_LINK_UP) {
                if (!GetLinkSpeed(pAdapter, &Handles100Mbit)) {
                                DBGPRINT_ERROR(("TryMII: Unknown link speed \n"));
                                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                                return FALSE;
                }
                if (Handles100Mbit) {
                        DBGPRINT_INITIALIZE(("TryMII: Link speed set to 100\n"));
                        pAdapter->Hardware.LinkSpeed = LINK_SPEED_100;
                }
                else {
                        DBGPRINT_INITIALIZE(("TryMII: Link speed set to 10\n"));
                        pAdapter->Hardware.LinkSpeed = LINK_SPEED_10;
                }
                return TRUE;

        } else {
                //
                // Assume 10Mbit if no link
                //
                if (PhyStatus_g & MII_STATUS_100MB_MASK) {
                        DBGPRINT_INITIALIZE((
                        "TryMII: Link speed defaulted to 100 \n"));
                        pAdapter->Hardware.LinkSpeed = LINK_SPEED_100;
                }
                else {
                        DBGPRINT_INITIALIZE((
                        "TryMII: Link speed defaulted to 10 \n"));
                        pAdapter->Hardware.LinkSpeed = LINK_SPEED_10;
                }
                return TRUE;
    }

        }
    return FALSE;
}


/*++

Routine Name:

        tc90x_TryLinkBeat

Routine Description:

        Download a self-directed packet. If successful, 100BASE-FX is available.

Arguments:

        IN PNIC_INFORMATION Adapter
        IN CONNECTOR_TYPE NewConnector

Return Value:

        BOOLEAN

--*/

BOOLEAN
tc90x_TryLinkBeat(
        IN PNIC_INFORMATION Adapter,
        IN CONNECTOR_TYPE NewConnector
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        USHORT MediaStatus = 0;
        BOOLEAN retval = FALSE;
        CONNECTOR_TYPE NotUsed;

        DBGPRINT_FUNCTION(("tc90x_TryLinkBeat: IN \n"));
        //
        // Go quiet for 1.5 seconds to get any N-Way hub into a receptive
        // state to sense the new link speed.  We go quiet by switching over
        // to 10BT and disabling Linkbeat.
        //
        pAdapter->Hardware.LinkBeatDisable = TRUE;
        tc90x_SetupConnector(pAdapter, CONNECTOR_10BASET, &NotUsed);
        //
        // Delay 1.5 seconds
        //
        TimeOutCount = jiffies + 15*HZ/10;
        while(TimeOutCount > jiffies) ;

        NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_RESET | TX_RESET_MASK_NETWORK_RESET);
        NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET | RX_RESET_MASK_NETWORK_RESET);
        //
        // Set up for TP transceiver
        //
        tc90x_SetupConnector(pAdapter, NewConnector, &NotUsed);

        // delay 5 milliseconds
        TimeOutCount = jiffies + 1; //make it 10ms
        while(TimeOutCount > jiffies) ;
        //
        // We need to send a test packet to clear the
        // the Partition if for some reason it's partitioned
        // (i.e., we were testing 100mb before.)
        // Download a 20 byte packet into the TxFIFO, from us, to us
        //
        if (!DownloadSelfDirected(pAdapter)) {
                return FALSE;
        }
        //
        // Acknowledge the down complete
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_ACKNOWLEDGE_INTERRUPT + INTSTATUS_DOWN_COMPLETE);
        //
        // Check MediaStatus for linkbeat indication
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        MediaStatus = NIC_READ_PORT_USHORT(pAdapter, MEDIA_STATUS_REGISTER);

        if (MediaStatus & MEDIA_STATUS_LINK_DETECT) {
                retval = TRUE;
        }

        NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_RESET | TX_RESET_MASK_NETWORK_RESET);
        NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET | RX_RESET_MASK_NETWORK_RESET);

        DBGPRINT_INITIALIZE(("tc90x_TryLinkBeat: OUT with success\n"));
        return retval;
}

/*++
Routine Name:

        DownloadSelfDirected

Routine Description:

    Download a 20 byte packet into the TxFIFO, from us, to us.

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        BOOLEAN

--*/

BOOLEAN
DownloadSelfDirected(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        PDPD_LIST_ENTRY dpdVirtual;
        UINT PacketBufPhysAddr;
    UCHAR *PacketBuffer;
    PUCHAR pbuffer;
        //
        // Set allocated buffer for transmit
        //
        PacketBuffer = (PUCHAR)pAdapter->TestBufferVirtual[0];
        PacketBufPhysAddr = pAdapter->TestBufferPhysical[0];
        //
    // Fill data in the buffer
        //
        pbuffer = pAdapter->StationAddress;
        memcpy(PacketBuffer, pbuffer, 6);       // destination is me
        memcpy(PacketBuffer + 6, pbuffer, 6);   // source is me
        *(UINT *)(PacketBuffer + 12) = 0;
        *(UINT *)(PacketBuffer + 16) = 0;
        //
        // Create a single DPD
        //
        dpdVirtual = (PDPD_LIST_ENTRY)pAdapter->TestDPDVirtual[0];

        dpdVirtual->DownNextPointer = 0;
        dpdVirtual->FrameStartHeader = 20 | FSH_ROUND_UP_DEFEAT;
        dpdVirtual->SGList[0].Address = PacketBufPhysAddr;
        dpdVirtual->SGList[0].Count = 20 | 0x80000000;
        //
        // Download DPD
        //
        NIC_COMMAND_WAIT(pAdapter, COMMAND_DOWN_STALL);

        NIC_WRITE_PORT_UINT(
                pAdapter,
                DOWN_LIST_POINTER_REGISTER,
                dpdVirtual->DPDPhysicalAddress);

        NIC_COMMAND(pAdapter, COMMAND_DOWN_UNSTALL);
        //
        // Check if the DPD is done with
    //
        DownListPointer_g = NIC_READ_PORT_UINT(pAdapter,DOWN_LIST_POINTER_REGISTER);

        NIC_DELAY(100);

        if(DownListPointer_g == dpdVirtual->DPDPhysicalAddress)
        {
                pAdapter->WaitCases = CHECK_DOWNLOAD_SELFDIRECTED;
                TimeOutCount = jiffies + 3*HZ;  //max = 3s
                //WaitTimer.expires = RUN_AT(HZ/100);
                //add_timer(&WaitTimer);
                //InWaitTimer = TRUE;
                while(TimeOutCount > jiffies) ;
                if(DownListPointer_g != dpdVirtual->DPDPhysicalAddress)
                {       pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                        DBGPRINT_ERROR(("DownloadSelfDirected: DPD not finished\n"));
                        return FALSE;
                }
        }
        return TRUE;
}

BOOLEAN
CheckTransmitInProgress(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        MediaStatus_g = 0;

        NIC_COMMAND(pAdapter,COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);

        NIC_DELAY(10);

        if(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS)
        {       pAdapter->WaitCases = CHECK_TRANSMIT_IN_PROGRESS;
                TimeOutCount = jiffies + HZ;  //max = 1s
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100);
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies) ;
                if(!(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS))
                {       pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                        DBGPRINT_ERROR(("CheckTransmitInProgress: Transmit still in progress\n"));
                        return FALSE;
                }
        }
        return TRUE;
}


/*++

Routine Name:

        TestPacket

Routine Description:

    This function is called by TryLoopback to determine if a packet can
        successfully be loopbacked for 10Base-2 and AUI.

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        BOOLEAN

--*/

BOOLEAN
TestPacket(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN ReturnValue = FALSE;
        USHORT MacControl = 0;
        UINT PacketStatus = 0;

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        MacControl = NIC_READ_PORT_USHORT(
                                        pAdapter,
                                        MAC_CONTROL_REGISTER);
        //
        // Enable full duplex
        //
        NIC_WRITE_PORT_USHORT(
                pAdapter,
                MAC_CONTROL_REGISTER,
                (USHORT)(MacControl | MAC_CONTROL_FULL_DUPLEX_ENABLE));
        //
        // Write UpListPointer to UpListPointer register and unstall
        //
        NIC_COMMAND_WAIT(pAdapter, COMMAND_UP_STALL);

        NIC_WRITE_PORT_UINT(
                pAdapter,
                UP_LIST_POINTER_REGISTER,
                pAdapter->HeadUPDVirtual->UPDPhysicalAddress);

        NIC_COMMAND(pAdapter, COMMAND_UP_UNSTALL);

        //
        // Enable receive and transmit and setup our packet filter
        //
        NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_ENABLE);
        NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_ENABLE);
        NIC_COMMAND_WAIT(pAdapter, COMMAND_SET_RX_FILTER +
                         RX_FILTER_INDIVIDUAL);

        //
        // Create single DPD and download
        //
        if (!DownloadSelfDirected(pAdapter)) {

                return FALSE;
        }
        //
        // Check if transmit is still in progress
        //
        if (!CheckTransmitInProgress(pAdapter))
                return FALSE;
        //
        // Reset the transmitter to get rid of any TxStatus we haven't seen yet
        //
        NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_RESET | TX_RESET_MASK_NETWORK_RESET);
        //
        // Check UpListPtr to see if it has changed to see if upload complete
        //
        UpListPointer_g = NIC_READ_PORT_UINT(pAdapter,UP_LIST_POINTER_REGISTER);

        NIC_DELAY(100);

        if(UpListPointer_g == pAdapter->HeadUPDVirtual->UPDPhysicalAddress)
        {       pAdapter->WaitCases = AUTONEG_TEST_PACKET;
                TimeOutCount = jiffies + HZ; //max =1s
                while(TimeOutCount > jiffies) ;
                if(UpListPointer_g != pAdapter->HeadUPDVirtual->UPDPhysicalAddress)
                {       DBGPRINT_ERROR(("TestPacket: UPD not finished\n"));
                        return FALSE;
                }
        }
    //
    // Check RxStatus. If we've got a packet without any errors, this
    // connector is okay.
    //
        PacketStatus = pAdapter->HeadUPDVirtual->UpPacketStatus;

        if (!(PacketStatus & UP_PACKET_STATUS_ERROR) &&
                (PacketStatus & UP_PACKET_STATUS_COMPLETE)) {

                ReturnValue = TRUE;             // Received a good packet
        }
        //
        // The following cleans up after the test we just ran
        //
        NIC_WRITE_PORT_UINT(pAdapter, UP_LIST_POINTER_REGISTER, 0);
        NIC_WRITE_PORT_UINT(pAdapter, DOWN_LIST_POINTER_REGISTER, 0);
        pAdapter->HeadUPDVirtual->UpPacketStatus = 0;
        //
        // Reset the receiver to wipe anything we haven't seen yet
        //
        NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET | RX_RESET_MASK_NETWORK_RESET);
        NIC_COMMAND(
                pAdapter,
                COMMAND_ACKNOWLEDGE_INTERRUPT +
                INTSTATUS_ACKNOWLEDGE_ALL);

        //
    // Get out of loopback mode
    //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        MacControl = NIC_READ_PORT_USHORT(pAdapter, MAC_CONTROL_REGISTER);

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                MAC_CONTROL_REGISTER,
                (USHORT)(MacControl & ~MAC_CONTROL_FULL_DUPLEX_ENABLE));

        return ReturnValue;
}


/*++

Routine Name:

        GetLinkSpeed

Routine Description:

        Determine from the MII AutoNegotiationAdvertisement and
        AutoNegotiationPartnerAbility registers whether the
        current linkspeed is 10Mbits or 100Mbits.

Arguments:

        IN PNIC_INFORMATION Adapter,
        OUT PBOOLEAN handles100Mbitptr

Return Value:

        BOOLEAN

--*/

BOOLEAN
GetLinkSpeed(
        IN PNIC_INFORMATION Adapter,
        OUT PBOOLEAN handles100Mbitptr
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN PhyResponding;
        USHORT PhyAnlpar;
        USHORT PhyAner;
        USHORT PhyAnar;
        USHORT PhyStatus;

    PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANER, &PhyAner);
    if (!PhyResponding)
                return FALSE;

    PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANLPAR, &PhyAnlpar);
    if (!PhyResponding)
                return FALSE;

    PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANAR, &PhyAnar);
    if (!PhyResponding)
                return FALSE;

        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_STATUS, &PhyStatus);
        if (!PhyResponding)
                return FALSE;
        //
        // Check to see if we've completed auto-negotiation.
        //
        if (!(PhyStatus & MII_STATUS_AUTO_DONE))
                return FALSE;

        if ((PhyAnar & MII_ANAR_100TXFD) &&
                (PhyAnlpar & MII_ANLPAR_100TXFD)) {
                pAdapter->Hardware.MIIPhyUsed = MII_100TXFD;
                *handles100Mbitptr = TRUE;
                pAdapter->Hardware.FullDuplexEnable = TRUE;

        }
    else if ((PhyAnar & MII_ANAR_100TX) && (PhyAnlpar & MII_ANLPAR_100TX)) {
                pAdapter->Hardware.MIIPhyUsed = MII_100TX ;
                *handles100Mbitptr = TRUE;
                pAdapter->Hardware.FullDuplexEnable = FALSE;
    }
    else if ((PhyAnar & MII_ANAR_10TFD) && (PhyAnlpar & MII_ANLPAR_10TFD)) {
                pAdapter->Hardware.MIIPhyUsed = MII_10TFD ;
                pAdapter->Hardware.FullDuplexEnable = TRUE;
        *handles100Mbitptr = FALSE;
        }
        else if ((PhyAnar & MII_ANAR_10T) && (PhyAnlpar & MII_ANLPAR_10T)) {
                pAdapter->Hardware.MIIPhyUsed = MII_10T ;
                pAdapter->Hardware.FullDuplexEnable = FALSE;
                *handles100Mbitptr = FALSE;
    }
        else if (!(PhyAner & MII_ANER_LPANABLE)) {
        //
        // Link partner is not capable of auto-negotiation. Fall back to 10HD.
        //
        pAdapter->Hardware.MIIPhyUsed = MII_10T ;
        pAdapter->Hardware.FullDuplexEnable = FALSE;
        *handles100Mbitptr = FALSE;
        }
        else
                return FALSE;

    return TRUE;
}






/*++

Routine Name:

    tc90x_FreeAdapterResources.

Routine Description:

    This routine frees up the adapter resources.

Arguments:

        Device - Pointer to the device structure.

Return Value:

    None

--*/

VOID
tc90x_FreeAdapterResources(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        PDEVICE device = pAdapter->Device;
        PUPD_LIST_ENTRY currentUPDVirtual = pAdapter->HeadUPDVirtual;


        DBGPRINT_FUNCTION(("FreeAdapterResources: IN\n"));

        if (pAdapter->ResourcesReserved & NIC_INTERRUPT_REGISTERED) {

                DBGPRINT_INITIALIZE(("Releasing interrupt\n"));
                free_irq(device->irq, device);
                pAdapter->ResourcesReserved &= ~NIC_INTERRUPT_REGISTERED;

        }

        if (pAdapter->ResourcesReserved & WAIT_TIMER_REGISTERED) {

                DBGPRINT_INITIALIZE((KERN_CRIT "Releasing WaitTimer\n"));
                if (del_timer(&WaitTimer))
                        /* Timer has already been queued. */
                        DBGPRINT_ERROR(("WaitTimer already queued\n"));
                pAdapter->ResourcesReserved &= ~WAIT_TIMER_REGISTERED;
        }

        if (pAdapter->ResourcesReserved & NIC_TIMER_REGISTERED) {

                DBGPRINT_INITIALIZE((KERN_CRIT "Releasing Timers\n"));
                if (del_timer(&pAdapter->Resources.Timer)) {
                        //
                        // Timer has already been queued.
                        //
                        DBGPRINT_ERROR(("Timer already queued\n"));
                }
                pAdapter->ResourcesReserved &= ~NIC_TIMER_REGISTERED;
                if (del_timer(&WaitTimer))
                        DBGPRINT_ERROR(("WaitTimer already queued\n"));
        }

        if (pAdapter->ResourcesReserved & NIC_SHARED_MEMORY_ALLOCATED) {

                DBGPRINT_INITIALIZE((KERN_CRIT "Releasing memory\n"));

                currentUPDVirtual = pAdapter->HeadUPDVirtual;
                //
                // Release the SKBs allocated
                //
                while (1) {

                        if (currentUPDVirtual->SocketBuffer)
#if LINUX_VERSION_CODE >= 0x20343
                                pci_unmap_single(pAdapter->pciDevice,
                                                currentUPDVirtual->SGList[0].Address,
                                                currentUPDVirtual->SocketBuffer->len,
                                                PCI_DMA_FROMDEVICE);

#endif

#if LINUX_VERSION_CODE >= 0x20200
                                dev_kfree_skb(
                                        currentUPDVirtual->SocketBuffer
                                        );
#else
				VMWARE_STRESS_RETAIN_BUFFER(skb, NO_CONTINUE_CMD,
                                dev_kfree_skb(
                                        currentUPDVirtual->SocketBuffer,
                                        GFP_ATOMIC
                                        ));

#endif
                        currentUPDVirtual = currentUPDVirtual->Next;
                        if (currentUPDVirtual == pAdapter->HeadUPDVirtual)
                                        break;
                }

#if LINUX_VERSION_CODE < 0x20343
                kfree(pAdapter->Resources.SharedMemoryVirtual);
#else
                pci_free_consistent(pAdapter->pciDevice,
                                    pAdapter->Resources.SharedMemorySize,
                                    pAdapter->Resources.SharedMemoryVirtual,
                                    (dma_addr_t)pAdapter->Resources.SharedMemoryPhysical);
#endif
                pAdapter->ResourcesReserved &= ~NIC_SHARED_MEMORY_ALLOCATED;
        }


        if (pAdapter->ResourcesReserved & NIC_IO_PORT_REGISTERED) {

                DBGPRINT_INITIALIZE((KERN_CRIT "Releasing IO port region\n"));
                release_region(device->base_addr, 0x80);
                pAdapter->ResourcesReserved &= ~NIC_IO_PORT_REGISTERED;
        }
        DBGPRINT_FUNCTION(("FreeAdapterResources: OUT\n"));
}


/*++

RoutineName:

    tc90x_RegisterAdapter.

Routine Description:

    This routine registers the adapter resources with Linux.

Arguments:

        Device - Pointer to the device structure.

Return Value:

    NIC_STATUS_SUCCESS if the adapter resources could be registered.
    NIC_STATUS_FAILURE if the adapter resources could not be registered.

--*/

NIC_STATUS
tc90x_RegisterAdapter(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        PDEVICE device = pAdapter->Device;

        DBGPRINT_FUNCTION(("RegisterAdapter: IN\n"));
        //
        // Use the now-standard shared IRQ implementation.
        //
        DBGPRINT_INITIALIZE(("SA_SHIRQ  registering IRQ %x\n", device->irq));
        if (request_irq(
                device->irq,
                &NICInterrupt,
                SA_SHIRQ,
                device->name,
                device)) {

                DBGPRINT_ERROR(("RegisterAdapter: IRQ registration failed\n"));
                return NIC_STATUS_FAILURE;

        }

        pAdapter->ResourcesReserved|= NIC_INTERRUPT_REGISTERED;

        DBGPRINT_FUNCTION(("RegisterAdapter: OUT\n"));
        return NIC_STATUS_SUCCESS;
}

/*++

Routine Name:

    tc90x_GetAdapterProperties.

Routine Description:

    This routine sets up the adapter.

Arguments:

        Device - Pointer to the device structure.

Return Value:

    NIC_STATUS_SUCCESS if the adapter could be set up successfully.
    NIC_STATUS_FAILURE if the adpater could not be set up successfully.

--*/

NIC_STATUS
tc90x_GetAdapterProperties(
        IN PNIC_INFORMATION Adapter
        )

{
        PNIC_INFORMATION pAdapter = Adapter;
        PDEVICE device = pAdapter->Device;

        NIC_STATUS nicStatus;
        USHORT eepromValue;
        USHORT intStatus;
        COMPATABILITY_WORD compatability;
        SOFTWARE_INFORMATION_1 information1;
        SOFTWARE_INFORMATION_2 information2;
        CAPABILITIES_WORD capabilities;
        UCHAR value, index;

        DBGPRINT_FUNCTION(("GetAdapterProperties: IN \n"));
    //
    // Check the ASIC type. Udp checksum should not be used in
    // pre-Tornado cards-  JRR
    //

    if ((pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_9055) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_9004) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_9005) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_9006) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_9058) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_900A) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_905A) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_4500) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_7646) ||
        (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_9800)) {

        pAdapter->Hardware.BitsInHashFilter = 0x40;
    }
    else {

        pAdapter->Hardware.BitsInHashFilter = 0x100;
        pAdapter->Hardware.UDPChecksumErrDone = TRUE;
    }

        pAdapter->Hardware.Status = HARDWARE_STATUS_WORKING;
        pAdapter->Hardware.LinkSpeed = LINK_SPEED_100;

        //
        // Make sure we can see the adapter.  Set up for window 7, and make
        // sure the window number gets reflected in status.
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_7);

        intStatus = NIC_READ_PORT_USHORT(
                        pAdapter,
                        INTSTATUS_COMMAND_REGISTER);

        DBGPRINT_INITIALIZE(("intStatus =%x\n", intStatus));
        DBGPRINT_INITIALIZE(("ioBase =%x\n", (INT)pAdapter->IoBaseAddress));

        if ((intStatus & REGISTER_WINDOW_MASK) !=
            (REGISTER_WINDOW_7 << 13)) {

                DBGPRINT_ERROR(("SetupAdapter: Window selection failure\n"));
                DBGPRINT_INITIALIZE(("SetupAdapter: Out with error\n"));
                return NIC_STATUS_FAILURE;
        }
        //
        // ----------------- Read the compatability level ----------
        //

        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_COMPATABILITY_WORD,
                        (PUSHORT)&compatability);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_INITIALIZE((
                        "GetAdapterProperties: compatability read failed\n"));
                return NIC_STATUS_FAILURE;
        }
        //
        // Check the Failure level.
        //

        if (compatability.FailureLevel > EEPROM_COMPATABILITY_LEVEL) {

                DBGPRINT_ERROR((
                        "GetAdapterProperties: Incompatible level\n"));
                DBGPRINT_INITIALIZE((
                        "GetAdapterProperties: Out with error\n"));
                return NIC_STATUS_FAILURE;
        }
        //
        // Check the warning level.
        //

        if (compatability.WarningLevel > EEPROM_COMPATABILITY_LEVEL) {

                DBGPRINT_ERROR((
                        "GetAdapterProperties: Wrong down compatability level\n"
                        ));
        }
    //
        // ----------------- Read the software information 1 -------
        //
        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_SOFTWARE_INFORMATION_1,
                        (PUSHORT)&information1);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_INITIALIZE((
                        "GetAdapterProperties: EEPROM s/w info1 read failed\n"));
                return NIC_STATUS_FAILURE;

        }
        if (information1.LinkBeatDisable) {

                DBGPRINT_INITIALIZE(("s/w information1 - Link beat disable\n"));
                pAdapter->Hardware.LinkBeatDisable = TRUE;

        }
        if (pAdapter->Hardware.DuplexCommandOverride == FALSE) {
                if (information1.FullDuplexMode) {

                        DBGPRINT_INITIALIZE(("s/w information1 - Full duplex enable\n"));
                        pAdapter->Hardware.FullDuplexEnable = TRUE;
                }
                else {
                        DBGPRINT_INITIALIZE(("s/w information 1 - Full duplex disabled\n"));
                pAdapter->Hardware.FullDuplexEnable = FALSE;
                }
        }

        switch (information1.OptimizeFor) {

        case EEPROM_OPTIMIZE_FOR_THROUGHPUT:

                        DBGPRINT_INITIALIZE(("sw info1 - optimize throughput\n"));
                        pAdapter->Hardware.OptimizeForThroughput = TRUE;
                        break;

                case EEPROM_OPTIMIZE_FOR_CPU:

                        DBGPRINT_INITIALIZE(("s/w info1 - optimize CPU\n"));
                        pAdapter->Hardware.OptimizeForCPU = TRUE;
                        break;

                case EEPROM_OPTIMIZE_NORMAL:

                        DBGPRINT_INITIALIZE(("s/w info1 - optimize Normal\n"));
                        pAdapter->Hardware.OptimizeNormal = TRUE;
                        break;

                default:
                        DBGPRINT_ERROR((
                        "GetAdapterProperties: Wrong optimization level\n"));
                        return NIC_STATUS_FAILURE;
                        break;

        }

        // ----------------- Read the capabilities information -----
        //
        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_CAPABILITIES_WORD,
                        (PUSHORT)&capabilities
                        );

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_INITIALIZE((
                "GetAdapterProprties: EEPROM s/w capabilities read failed\n"));

                return NIC_STATUS_FAILURE;
        }

        if (capabilities.SupportsPowerManagement) {

                DBGPRINT_INITIALIZE(("Adapter supports power management\n"));
                pAdapter->Hardware.SupportsPowerManagement = TRUE;
        }
        //
        // ----------------- Read the software information 2 -------
        //

        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_SOFTWARE_INFORMATION_2,
                        (PUSHORT)&information2);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "GetAdapterProperties: ReadEEPROM , SWINFO2 failed\n"));
                DBGPRINT_INITIALIZE((
                        "GetAdapterProperties: Out with error\n"));
                return NIC_STATUS_FAILURE;

        }

        if (information2.BroadcastRxErrDone){

                DBGPRINT_INITIALIZE(("Adapter has BroadcastRxErrDone\n"));
                pAdapter->Hardware.BroadcastErrDone = TRUE;
        }

        if (information2.MWIErrDone) {

                DBGPRINT_INITIALIZE(("Adapter has MWIErrDone\n"));
                pAdapter->Hardware.MWIErrDone = TRUE;
        }

        if (information2.WOLConnectorPresent){

                DBGPRINT_INITIALIZE(("WOL is connected\n"));
                pAdapter->Hardware.WOLConnectorPresent = TRUE;
        }

        if (information2.AutoResetToD0) {

                DBGPRINT_INITIALIZE(("Auto reset to D0 bit on\n"));
                pAdapter->Hardware.AutoResetToD0 = TRUE;
        }
        //
        // ----------------- Read the OEM station address ----------
        //

        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_OEM_NODE_ADDRESS_WORD_0,
                        &eepromValue);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "GetAdapterProperties: EEPROM read word 0 failed\n"));
                DBGPRINT_INITIALIZE((
                        "GetAdapterProperties: Out with error\n"));
                return NIC_STATUS_FAILURE;

        }

        pAdapter->PermanentAddress[0] = HIBYTE(eepromValue);
        pAdapter->PermanentAddress[1] = LOBYTE(eepromValue);

        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_OEM_NODE_ADDRESS_WORD_1,
                        &eepromValue);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "GetAdapterProperties: EEPROM read word 1 failed\n"));
                DBGPRINT_INITIALIZE((
                        "GetAdapterProperties: Out with error\n"));
                return NIC_STATUS_FAILURE;

        }

        pAdapter->PermanentAddress[2] = HIBYTE(eepromValue);
        pAdapter->PermanentAddress[3] = LOBYTE(eepromValue);

        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_OEM_NODE_ADDRESS_WORD_2,
                        &eepromValue);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "GetAdapterProperties: EEPROM read word 2 failed\n"));
                        DBGPRINT_INITIALIZE((
                        "GetAdapterProperties: Out with error\n"));
                return NIC_STATUS_FAILURE;

        }

        pAdapter->PermanentAddress[4] = HIBYTE(eepromValue);
        pAdapter->PermanentAddress[5] = LOBYTE(eepromValue);

        //
        // If the station address has not been overriden, fill the permanent
        // address into it.
        //

        value = pAdapter->StationAddress[0] |
                pAdapter->StationAddress[1] |
                pAdapter->StationAddress[2] |
                pAdapter->StationAddress[3] |
                pAdapter->StationAddress[4] |
                pAdapter->StationAddress[5];

        //
        // If the station address has not been overriden, set this value
        // in the station address.
        //

        if (!value) {

                for (index=0; index < 6; index++)
                        pAdapter->StationAddress[index] =
                                pAdapter->PermanentAddress[index];


        }

        for (index=0; index < 6; index++)
                device->dev_addr[index] = pAdapter->StationAddress[index];

        DBGPRINT_FUNCTION(("GetAdapterProperties: OUT \n"));
        return NIC_STATUS_SUCCESS;

}


/*++

Routine Name:

    tc90x_BasicInitializeAdapter.

Routine Description:

    This routine does the basic initialize of the adapter. It does
    not set the media specific stuff.

Arguments:

    MiniportAdapterContext - Pointer to the adapter structure.

Return Value:

    NIC_STATUS_SUCCESS if the initialization succeeds.
    NIC_STATUS_FAILUTE if the initialization fails.

--*/

NIC_STATUS
tc90x_BasicInitializeAdapter(
        IN PNIC_INFORMATION Adapter
        )

{

        PNIC_INFORMATION pAdapter = Adapter;
        USHORT stationTemp, macControl;
        NIC_STATUS nicStatus;

        DBGPRINT_FUNCTION(("BasicInitializeAdapter: In\n"));
        //
        // ----------------- Tx Engine handling --------------------
        //

        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_DISABLE);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "BasicInitializeAdapter: COMMAND_TX_DISABLE failed\n"));
                DBGPRINT_FUNCTION((
                        "BasicInitializeAdapter: Out with error\n"));
                return NIC_STATUS_FAILURE;

        }

#ifdef SERR_NEEDED
    // Down stall the adapter and wait for 100 milliseconds for
    // any pending PCI retry to be over.
    NIC_COMMAND_WAIT(pAdapter, COMMAND_DOWN_STALL);
#endif

        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_RESET | TX_RESET_MASK_NETWORK_RESET);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "BasicInitializeAdapter: COMMAND_TX_RESET failed\n"));
                DBGPRINT_FUNCTION((
                        "GetAdapterProperties: Out with error\n"));
                return NIC_STATUS_FAILURE;
        }
    //
        // ----------------- Rx engine handling --------------------
        //

        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_DISABLE);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "BasicInitializeAdapter: Rx disable failed\n"));
                DBGPRINT_FUNCTION((
                        "BasicInitializeAdapter: Out with error\n"));
                return NIC_STATUS_FAILURE;
    }

#ifdef SERR_NEEDED
    // Up stall the adapter and wait for 100 milliseconds for
    // any pending PCI retry to be over.
    NIC_COMMAND_WAIT(pAdapter, COMMAND_UP_STALL);
#endif

        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET | RX_RESET_MASK_NETWORK_RESET);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "BasicInitializeAdapter: Rx reset failed\n"
                        ));
                DBGPRINT_FUNCTION((
                        "BasicInitializeAdapter: Out with error\n"
                        ));
                return NIC_STATUS_FAILURE;
        }
        //
        // Take care of the interrupts.
        //
        NIC_ACKNOWLEDGE_ALL_INTERRUPT(pAdapter);
        NIC_COMMAND(pAdapter, COMMAND_STATISTICS_DISABLE);

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_6);
        //
        // Clear the statistics from the hardware.
        //
        NIC_READ_PORT_UCHAR(pAdapter, CARRIER_LOST_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, SQE_ERRORS_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, MULTIPLE_COLLISIONS_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, SINGLE_COLLISIONS_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, LATE_COLLISIONS_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, RX_OVERRUNS_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, FRAMES_TRANSMITTED_OK_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, FRAMES_RECEIVED_OK_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, FRAMES_DEFERRED_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, UPPER_FRAMES_OK_REGISTER);
        NIC_READ_PORT_USHORT(pAdapter, BYTES_RECEIVED_OK_REGISTER);
        NIC_READ_PORT_USHORT(pAdapter, BYTES_TRANSMITTED_OK_REGISTER);

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        NIC_READ_PORT_UCHAR(pAdapter, BAD_SSD_REGISTER);
        NIC_READ_PORT_UCHAR(pAdapter, UPPER_BYTES_OK_REGISTER);
        //
        // Program the station address.
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_2);

        stationTemp = pAdapter->StationAddress[1] << 8;
        stationTemp |= pAdapter->StationAddress[0];

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                STATION_ADDRESS_LOW_REGISTER,
                stationTemp);

        stationTemp = pAdapter->StationAddress[3] << 8;
        stationTemp |= pAdapter->StationAddress[2];

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                STATION_ADDRESS_MID_REGISTER,
                stationTemp);

        stationTemp = pAdapter->StationAddress[5] << 8;
        stationTemp |= pAdapter->StationAddress[4];

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                STATION_ADDRESS_HIGH_REGISTER,
                stationTemp);

        NIC_WRITE_PORT_USHORT(pAdapter, 0x6, 0);
        NIC_WRITE_PORT_USHORT(pAdapter, 0x8, 0);
        NIC_WRITE_PORT_USHORT(pAdapter, 0xA, 0);
        NIC_COMMAND(pAdapter, COMMAND_STATISTICS_ENABLE);
        //
        // Clear the mac control register.
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        macControl = NIC_READ_PORT_USHORT(pAdapter, MAC_CONTROL_REGISTER);
        macControl &= 0x1;
        NIC_WRITE_PORT_USHORT(pAdapter, MAC_CONTROL_REGISTER, macControl);

        DBGPRINT_FUNCTION((
                "BasicInitializeAdapter: Out with success\n"));

        return NIC_STATUS_SUCCESS;

}

/*++

Routine Name:

        tc90x_AllocateSharedMemory.

Routine Description:

        This routine allocates the shared memory

Arguments:

        Device - Pointer to the device structure

Return Value:

        NIC_STATUS_SUCCESS if memory allocations succeeds
        NIC_STATUS_FAILURE if memory allocation fails

--*/

NIC_STATUS
tc90x_AllocateSharedMemory(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        PDEVICE device = pAdapter->Device;

        UINT updMemoryForOne, totalUPDMemory;
        UINT dpdMemoryForOne, totalDPDMemory;

        UINT rxMemoryForOne;

        UINT cacheLineSize, count, alignment;

        ULONG memoryBaseVirtual;
        dma_addr_t memoryBasePhysical;
        ULONG updMemoryVirtualStart;
        UINT updMemoryPhysicalStart;
        ULONG dpdMemoryVirtualStart;
        UINT dpdMemoryPhysicalStart;

        UINT currentUPDPhysical, previousUPDPhysical;
        UINT firstUPDPhysical = 0;

        PUPD_LIST_ENTRY currentUPDVirtual = NULL;
        PUPD_LIST_ENTRY firstUPDVirtual = NULL;
        PUPD_LIST_ENTRY previousUPDVirtual = NULL;

        PUCHAR dataPointer;

        PDPD_LIST_ENTRY currentDPDVirtual = NULL;
        PDPD_LIST_ENTRY headDPDVirtual = NULL;
        PDPD_LIST_ENTRY previousDPDVirtual = NULL;
        UINT currentDPDPhysical = 0;

        ULONG testMemoryVirtualStart;
        UINT testMemoryPhysicalStart;
        UINT totalTestMemory;

        DBGPRINT_FUNCTION(("tc90x_AllocateSharedMemory: In\n"));

        cacheLineSize = pAdapter->Hardware.CacheLineSize;
        //
        // UPD structure memory requirement.
        //
        updMemoryForOne = sizeof(UPD_LIST_ENTRY) + cacheLineSize;
        totalUPDMemory = pAdapter->Resources.ReceiveCount * updMemoryForOne;
        //
        // Receive buffer requirement.
        //
        rxMemoryForOne = ETHERNET_MAXIMUM_FRAME_SIZE + cacheLineSize;
        //
        // DPD structure memory requirement.
        //
        dpdMemoryForOne = sizeof(DPD_LIST_ENTRY) + cacheLineSize;
        totalDPDMemory = pAdapter->Resources.SendCount * dpdMemoryForOne;
        //
        // Calculate the test memory required.
        //
        totalTestMemory =  MAXIMUM_TEST_BUFFERS *
                          (dpdMemoryForOne + rxMemoryForOne);

        pAdapter->Resources.SharedMemorySize    = totalUPDMemory +
                                                  totalDPDMemory +
                                                                                                totalTestMemory;
        //
        // Allocate the memory
        //
#if LINUX_VERSION_CODE < 0x20343
        pAdapter->Resources.SharedMemoryVirtual =
                        (PUCHAR) kmalloc(
                                        pAdapter->Resources.SharedMemorySize,
                                        GFP_KERNEL);

        //
        // Zero out the memory.
        //
        memset(
                pAdapter->Resources.SharedMemoryVirtual,
                0,
                pAdapter->Resources.SharedMemorySize);
    //
        // -------------- Carve out the regions ---------------
        //
        memoryBasePhysical = LIMIT_TO_32BIT_ADDR(virt_to_bus(
                                pAdapter->Resources.SharedMemoryVirtual));
#else
        pAdapter->Resources.SharedMemoryVirtual
                = (PUCHAR)pci_alloc_consistent(pAdapter->pciDevice,
                                pAdapter->Resources.SharedMemorySize,
                                &memoryBasePhysical);

#endif
        pAdapter->ResourcesReserved |= NIC_SHARED_MEMORY_ALLOCATED;
        pAdapter->Resources.SharedMemoryPhysical = (VOID *)memoryBasePhysical;
        memoryBaseVirtual = (ULONG)pAdapter->Resources.SharedMemoryVirtual;

        DBGPRINT_INITIALIZE(("memoryBaseVirtual=%lu, memoryBasePhysical=%x\n",
                             memoryBaseVirtual,
                             (INT)memoryBasePhysical));
        //
        // Virtual addresses of the regions.
        //
        updMemoryVirtualStart = memoryBaseVirtual;
        dpdMemoryVirtualStart = updMemoryVirtualStart + totalUPDMemory;
        testMemoryVirtualStart = dpdMemoryVirtualStart + totalDPDMemory;
        //
        // Physical addresses of the regions.
        //
        updMemoryPhysicalStart = memoryBasePhysical;
        dpdMemoryPhysicalStart = updMemoryPhysicalStart + totalUPDMemory;
        testMemoryPhysicalStart = dpdMemoryPhysicalStart + totalDPDMemory;
        //
        // -------------- Make the receive structures --------------
        //
        for (count = 0; count < pAdapter->Resources.ReceiveCount; count++) {

                currentUPDPhysical = updMemoryPhysicalStart +
                                     count * updMemoryForOne;

                alignment = cacheLineSize -
                            (currentUPDPhysical % cacheLineSize );
                currentUPDPhysical += alignment;

                currentUPDVirtual = (PUPD_LIST_ENTRY)(
                                    updMemoryVirtualStart +
                                    alignment +
                                    count * updMemoryForOne );
        //
                // Store the physical address of this UPD in the UPD itself.
                //
                currentUPDVirtual->UPDPhysicalAddress = currentUPDPhysical;

                if (0 == count) {
                        //
                        // Store the virtual and physical address of the
                        // first UPD.
                        firstUPDVirtual = currentUPDVirtual;
                        firstUPDPhysical = currentUPDPhysical;

                }
                else {
                        //
                        // Put the links in the UPDs.
                        //
                        previousUPDVirtual->Next = currentUPDVirtual;
                        previousUPDVirtual->UpNextPointer = currentUPDPhysical;
                        currentUPDVirtual->Previous = previousUPDVirtual;

                }
                //
                // Allocate the skb per UPD
                //
                currentUPDVirtual->SocketBuffer =
                                        DEV_ALLOC_SKB(
                                                ETHERNET_MAXIMUM_FRAME_SIZE +
                                                2);
        //
                // Make the current UPD as the previous one.
                //
                if (currentUPDVirtual->SocketBuffer == 0) {

                        DBGPRINT_ERROR((
                                "Socket buffer allocation failed\n"));
                        return NIC_STATUS_FAILURE;
                }
                else {

                        currentUPDVirtual->SocketBuffer->dev = device;
                        //
                        // Align IP on 16 byte boundaries
                        //
                        skb_reserve(
                                currentUPDVirtual->SocketBuffer,
                                2);

                        dataPointer = skb_put(
                                        currentUPDVirtual->SocketBuffer,
                                        ETHERNET_MAXIMUM_FRAME_SIZE );
                }

                currentUPDVirtual->RxBufferVirtual = dataPointer;

#if LINUX_VERSION_CODE < 0x20343
                currentUPDVirtual->SGList[0].Address
                        = LIMIT_TO_32BIT_ADDR(virt_to_bus(dataPointer));

#else
                currentUPDVirtual->SGList[0].Address
                        = pci_map_single(pAdapter->pciDevice, dataPointer,
                                         ETHERNET_MAXIMUM_FRAME_SIZE,
                                         PCI_DMA_FROMDEVICE);

#endif
                currentUPDVirtual->SGList[0].Count =
                                ETHERNET_MAXIMUM_FRAME_SIZE | 0x80000000;

                previousUPDVirtual = currentUPDVirtual;
                previousUPDPhysical = currentUPDPhysical;

        }
        //
        // Link the first and last UPDs.
        //

        currentUPDVirtual->Next = firstUPDVirtual;
        currentUPDVirtual->UpNextPointer = firstUPDPhysical;
        firstUPDVirtual->Previous = currentUPDVirtual;
        //
        // Save the address of the first UPD in the adapter structure.
        //
        pAdapter->HeadUPDVirtual = firstUPDVirtual;

        //
        // --------------- Carve out DPD structures -----------------
        //

        for (count=0; count < pAdapter->Resources.SendCount; count++) {

                currentDPDPhysical = dpdMemoryPhysicalStart +
                                     count * dpdMemoryForOne;

                alignment = cacheLineSize -
                            (currentDPDPhysical % cacheLineSize);
                currentDPDPhysical += alignment;

                currentDPDVirtual = (PDPD_LIST_ENTRY)(
                                    dpdMemoryVirtualStart +
                                    count * dpdMemoryForOne +
                                    alignment);
                //
                // Save the physical address of this DPD in the DPD itself.
                //
                currentDPDVirtual->DPDPhysicalAddress = currentDPDPhysical;

                if (0 == count) {

                        headDPDVirtual = currentDPDVirtual;

                }
                else {

                        previousDPDVirtual->Next = currentDPDVirtual;
                        currentDPDVirtual->Previous = previousDPDVirtual;
                }
                previousDPDVirtual = currentDPDVirtual;
        }

        //
        // Link head and tail.
        //
        headDPDVirtual->Previous = currentDPDVirtual;
        currentDPDVirtual->Next = headDPDVirtual;
        //
        // Point head and tail to this DPD.
        //
        pAdapter->HeadDPDVirtual = headDPDVirtual;
        pAdapter->TailDPDVirtual = headDPDVirtual;

        //
        // -------------- Test DPD and test buffer -----------------
        //

        for ( count = 0; count < MAXIMUM_TEST_BUFFERS; count++ ) {

                alignment = cacheLineSize -
                            (testMemoryPhysicalStart % cacheLineSize);
                pAdapter->TestDPDVirtual[count] = testMemoryVirtualStart +
                                                  alignment;
                pAdapter->TestDPDPhysical[count] = testMemoryPhysicalStart +
                                                   alignment;

                testMemoryVirtualStart += dpdMemoryForOne;
                testMemoryPhysicalStart += dpdMemoryForOne;

                alignment = cacheLineSize -
                            (testMemoryPhysicalStart % cacheLineSize);

                pAdapter->TestBufferVirtual[count] =
                                testMemoryVirtualStart + alignment;
                pAdapter->TestBufferPhysical[count] =
                                testMemoryPhysicalStart + alignment;
                //
                // Save the physical address of the DPD, in the DPD.
                //
                ((PDPD_LIST_ENTRY)
                (pAdapter->TestDPDVirtual[count]))->DPDPhysicalAddress =
                                        pAdapter->TestDPDPhysical[count];

                testMemoryVirtualStart += rxMemoryForOne;
                testMemoryPhysicalStart += rxMemoryForOne;

        }

        DBGPRINT_FUNCTION(("tc90x_AllocateSharedMemory: Out\n"));
        return NIC_STATUS_SUCCESS;
}

/*++

Routine Name:

    TestAdapter.

Routine Description:

    This routine tests the functionality of the adapter.

Arguments:

    MiniportAdapterContext - Pointer to the adapter structure.

Return Value:

    NIC_STATUS_SUCCESS if the adapter is functioning properly.
    NIC_STATUS_FAILURE if the adapter is not functioning properly.

--*/

NIC_STATUS
tc90x_TestAdapter(
        IN PNIC_INFORMATION Adapter
        )

{
        PNIC_INFORMATION pAdapter = Adapter;
        NIC_STATUS nicStatus;
        PDPD_LIST_ENTRY dpdVirtual;
        PUPD_LIST_ENTRY updVirtual;
        PUCHAR sourceBufferVirtual, destinationBufferVirtual;
        UINT sourceBufferPhysical;
        UINT  count;
        USHORT networkDiagnosticsValue;

        DBGPRINT_FUNCTION(("TestAdapter: IN \n"));
        //
        // Select the network diagnostics window.
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        //
        // Read the network diagnostics register.
        //
        networkDiagnosticsValue = NIC_READ_PORT_USHORT(
                                        pAdapter,
                                        NETWORK_DIAGNOSTICS_REGISTER);
        //
        // Enable loop back on the adapter.
        //

        networkDiagnosticsValue |= BIT_12;
        NIC_WRITE_PORT_USHORT(
                pAdapter,
                NETWORK_DIAGNOSTICS_REGISTER,
                networkDiagnosticsValue);

        NIC_COMMAND(pAdapter, COMMAND_TX_ENABLE);
        NIC_COMMAND(pAdapter, COMMAND_RX_ENABLE);
        //
        // Write the address to the UpListPointer register.
        //

        NIC_WRITE_PORT_UINT(
                pAdapter,
                UP_LIST_POINTER_REGISTER,
                pAdapter->HeadUPDVirtual->UPDPhysicalAddress);

        NIC_COMMAND(pAdapter, COMMAND_UP_UNSTALL);
        //
        // Use the buffer in the second UPD to make the packet.
        //
        sourceBufferVirtual = (PUCHAR)pAdapter->TestBufferVirtual[0];
        sourceBufferPhysical = pAdapter->TestBufferPhysical[0];

        for (count=0; count < ETHERNET_MAXIMUM_FRAME_SIZE; count++) {

        *sourceBufferVirtual++ = (UCHAR)count;
    }

        dpdVirtual = (PDPD_LIST_ENTRY)pAdapter->TestDPDVirtual[0];
        dpdVirtual->FrameStartHeader = ETHERNET_MAXIMUM_FRAME_SIZE;
        dpdVirtual->DownNextPointer = 0;
        dpdVirtual->SGList[0].Address = sourceBufferPhysical;
        dpdVirtual->SGList[0].Count = ETHERNET_MAXIMUM_FRAME_SIZE |
                                        0x80000000;

        NIC_COMMAND_WAIT(pAdapter, COMMAND_DOWN_STALL);
        //
        // Write the down list pointer register.
        //
        NIC_WRITE_PORT_UINT(
                pAdapter,
                DOWN_LIST_POINTER_REGISTER,
                dpdVirtual->DPDPhysicalAddress);

        NIC_COMMAND(pAdapter, COMMAND_DOWN_UNSTALL);
        //
        // Check that packet has been picked up by the hardware.
        //
        portValue_g = NIC_READ_PORT_UINT(pAdapter,DOWN_LIST_POINTER_REGISTER);
        NIC_DELAY(10);

        if(portValue_g == dpdVirtual->DPDPhysicalAddress)
        {       pAdapter->WaitCases = CHECK_DOWNLOAD_STATUS;
                TimeOutCount = jiffies + HZ;
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100); //max=1s
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies) ;
                /* DBGPRINT_INIT(("portValue_g = %x\n", portValue_g));  */
                if(portValue_g == dpdVirtual->DPDPhysicalAddress)
                   {    DBGPRINT_ERROR(("Packet not picked up by the hardware\n"));
                        DBGPRINT_INITIALIZE(("TestAdapter: Out with error\n"));
                        return NIC_STATUS_FAILURE;
                }
        }
        //
        // Check the upload information.
        //
        portValue_g = NIC_READ_PORT_UINT(pAdapter,UP_LIST_POINTER_REGISTER);

        NIC_DELAY(10);

        if(portValue_g == pAdapter->HeadUPDVirtual->UPDPhysicalAddress)
        {       pAdapter->WaitCases = CHECK_UPLOAD_STATUS;
                TimeOutCount = jiffies + HZ;
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100); // max=1s
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies) ;
                if(portValue_g == pAdapter->HeadUPDVirtual->UPDPhysicalAddress)
                {       DBGPRINT_ERROR(("Packet not uploaded by adapter\n"));
                        DBGPRINT_INITIALIZE(("TestAdapter: Out with error\n"));
                        return NIC_STATUS_FAILURE;
                }
        }
        //
        // Check the contents of the packet.
        //
        updVirtual = pAdapter->HeadUPDVirtual;
        destinationBufferVirtual = (PUCHAR)updVirtual->RxBufferVirtual;

        for (count=0; count < ETHERNET_MAXIMUM_FRAME_SIZE; count++) {

                if (*(destinationBufferVirtual + count) != (UCHAR)count)
                        break;
        }

        if (ETHERNET_MAXIMUM_FRAME_SIZE != count) {

        DBGPRINT_ERROR((
                        "TestAdapter: Receive buffer contents not ok\n"));
                DBGPRINT_INITIALIZE(("TestAdapter: Out with error\n"));
        return NIC_STATUS_FAILURE;

        }

        pAdapter->HeadUPDVirtual->UpPacketStatus = 0;

        NIC_WRITE_PORT_UINT(pAdapter, UP_LIST_POINTER_REGISTER, 0);
        NIC_ACKNOWLEDGE_ALL_INTERRUPT(pAdapter);
        //
        // Issue transmit and receive reset here.
        //
        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_TX_RESET | TX_RESET_MASK_NETWORK_RESET);

        if (nicStatus != NIC_STATUS_SUCCESS) {

        DBGPRINT_ERROR(("TestAdapter: Transmit reset failed\n"));
                DBGPRINT_INITIALIZE(("TestAdapter: Out with error\n"));
        return NIC_STATUS_FAILURE;

        }

        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET | RX_RESET_MASK_NETWORK_RESET);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("TestAdapter: Receiver reset failed\n"));
                DBGPRINT_INITIALIZE(("TestAdapter: Out with error\n"));
                return NIC_STATUS_FAILURE;

        }
        //
        // Clear the loop back bit in network diagnostics.
        //
        networkDiagnosticsValue &= ~BIT_12;

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                NETWORK_DIAGNOSTICS_REGISTER,
                networkDiagnosticsValue);

        DBGPRINT_FUNCTION(("TestAdapter: OUT\n"));
        return NIC_STATUS_SUCCESS;
}


/*++

Routine Name:

    StartAdapter.

Routine Description:

    This routine starts the adapter.

Arguments:

    MiniportAdapterContext - Pointer to the adapter structure.

Return Value:

    None.

--*/

NIC_STATUS
tc90x_StartAdapter(
        IN PNIC_INFORMATION Adapter
        )
{

        PNIC_INFORMATION pAdapter = Adapter;


        NIC_STATUS nicStatus;

        PDPD_LIST_ENTRY headDPDVirtual;

        USHORT diagnostics;
        UINT dmaControl;

        DBGPRINT_FUNCTION(("StartAdapter: In\n"));


        //
        // Enable upper bytes counting in diagnostics register.
        //

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        diagnostics = NIC_READ_PORT_USHORT(
                        pAdapter,
            NETWORK_DIAGNOSTICS_REGISTER);

        diagnostics |= NETWORK_DIAGNOSTICS_UPPER_BYTES_ENABLE;

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                NETWORK_DIAGNOSTICS_REGISTER,
                diagnostics);
        //
        // Enable counter speed in DMA control.
        //
        dmaControl = NIC_READ_PORT_UINT(
                        pAdapter,
            DMA_CONTROL_REGISTER);

        if (100000000 == pAdapter->Hardware.LinkSpeed)
                dmaControl |= DMA_CONTROL_COUNTER_SPEED;

        NIC_WRITE_PORT_UINT(
                pAdapter,
                DMA_CONTROL_REGISTER,
                dmaControl);
        //
        // ------------ Give download structures to the adapter -----
        //
        // Stall the download engine.
        //
        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_DOWN_STALL);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("StartAdapter: down stall failed\n"));
                DBGPRINT_INITIALIZE(("StartAdapter: Out with error\n"));
                return NIC_STATUS_FAILURE;

        }
    //
        // Use the head DPD to mark it as dummy.
        //
        headDPDVirtual = pAdapter->HeadDPDVirtual;
        headDPDVirtual->FrameStartHeader = FSH_DPD_EMPTY;
        //
        // Now move head and tail to next one.
        //
        pAdapter->HeadDPDVirtual = headDPDVirtual->Next;
        pAdapter->TailDPDVirtual = headDPDVirtual->Next;
        //
        // Write the first DPD address to the hardware.
        //
        NIC_WRITE_PORT_UINT(
                pAdapter,
                DOWN_LIST_POINTER_REGISTER,
                headDPDVirtual->DPDPhysicalAddress);
        //
        // Enable down polling on the hardware.
        //
        NIC_WRITE_PORT_UCHAR(
                pAdapter,
                DOWN_POLL_REGISTER,
                (UCHAR)pAdapter->Resources.DownPollRate);
        //
        // Unstall the download engine.
        //
        NIC_COMMAND(pAdapter, COMMAND_DOWN_UNSTALL);
        //
        // ------------ Give upload structures to the adapter -------
        //
    //
        // Stall the upload engine.
        //
        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_UP_STALL);

        if (nicStatus != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("StartAdapter: up stall failed\n"));
                DBGPRINT_INITIALIZE(("StartAdapter: Out with error\n"));
                return NIC_STATUS_FAILURE;
        }
        //
        // Give the address of the first UPD to the adapter.
        //

        NIC_WRITE_PORT_UINT(
                pAdapter,
                UP_LIST_POINTER_REGISTER,
                pAdapter->HeadUPDVirtual->UPDPhysicalAddress);
        //
        // Write the up poll register.
        //
        NIC_WRITE_PORT_UCHAR(
                pAdapter,
                UP_POLL_REGISTER,
                8);
        //
        // Unstall the download engine.
        //
        NIC_COMMAND(pAdapter, COMMAND_UP_UNSTALL);
        //
        // Enable the statistics back.
        //
        NIC_COMMAND(pAdapter, COMMAND_STATISTICS_ENABLE);
        //
        // Acknowledge any pending interrupts.
        //
        NIC_ACKNOWLEDGE_ALL_INTERRUPT(pAdapter);
        //
        // Enable indication for all interrupts.
        //
        NIC_ENABLE_ALL_INTERRUPT_INDICATION(pAdapter);
        //
        // Enable all interrupts to the host.
        //
#ifdef __VMKERNEL_MODULE__
	pAdapter->intr_enabled = 1;
#endif // __VMKERNEL_MODULE__
        NIC_UNMASK_ALL_INTERRUPT(pAdapter);
        //
        // Enable the transmit and receive engines.
        //
        NIC_COMMAND(pAdapter, COMMAND_RX_ENABLE);
        NIC_COMMAND(pAdapter, COMMAND_TX_ENABLE);
        //
        // Delay three seconds, only some switches need this,
        // default is no delay, user can enable this delay in command line
        //
        if(pAdapter->DelayStart == TRUE) {
                TimeOutCount = jiffies + 3*HZ;
                while(TimeOutCount > jiffies) ;
        }
        DBGPRINT_FUNCTION(("StartAdapter: Out\n"));
        return NIC_STATUS_SUCCESS;
}

/*++

Routine Name:

    ReStartAdapter.

Routine Description:

    This routine starts the adapter.

Arguments:

    MiniportAdapterContext - Pointer to the adapter structure.

Return Value:

    None.

--*/

VOID
ReStartAdapter(
        //IN PNIC_INFORMATION Adapter
        PVOID Adapter
        )
{

        PNIC_INFORMATION pAdapter = Adapter;
        UINT internalConfig = pAdapter->keepForGlobalReset;

        DBGPRINT_INTERRUPT((KERN_CRIT "ReStartAdapter after global reset: IN\n"));
        //
        // Delay for 1 second, might not be that long since some has been comsumed with task queue
        TimeOutCount = jiffies + HZ;
        while(TimeOutCount > jiffies) ;
        //
        // Mask all the interrupts.
        //
        NIC_MASK_ALL_INTERRUPT(pAdapter);
        //
        // Enable indication for all interrupts.
        //
        NIC_ENABLE_ALL_INTERRUPT_INDICATION(pAdapter);
        //
        // Write the internal config back.
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        NIC_WRITE_PORT_UINT(
                pAdapter,
                INTERNAL_CONFIG_REGISTER,
                internalConfig);
        //
    // Set the adapter for operation
    //
        if (tc90x_GetAdapterProperties(pAdapter) != NIC_STATUS_SUCCESS)
                DBGPRINT_INTERRUPT((KERN_CRIT "GetAdapterProperties failed\n"));

        if (tc90x_BasicInitializeAdapter(pAdapter) != NIC_STATUS_SUCCESS)
                DBGPRINT_INTERRUPT((KERN_CRIT "BasicInitialize failed\n"));

        if (tc90x_SetupMedia(pAdapter->Device) != NIC_STATUS_SUCCESS)
                DBGPRINT_INTERRUPT((KERN_CRIT "SetupMedia failed\n"));

        if (tc90x_SoftwareWork(pAdapter) != NIC_STATUS_SUCCESS)
                DBGPRINT_INTERRUPT((KERN_CRIT "SoftwareWork failed\n"));
        //
        // Countdown timer is cleared by reset. So write it back.
        //
        tc90x_SetCountDownTimer(pAdapter);

        pAdapter->DelayStart = FALSE; // no need to delay for switch here

        if (tc90x_StartAdapter(pAdapter) != NIC_STATUS_SUCCESS)
                DBGPRINT_INTERRUPT((KERN_CRIT "ReStartAdapter: StartAdapter failed\n"));

        DBGPRINT_INTERRUPT((KERN_CRIT "ReStartAdapter after global reset: OUT\n"));

        return;
}


UCHAR firstTime = 0;

/*++

Routine Name:

    NICSendPacket

Routine Description:

    This routine sends the packet

Arguments:

        SocketBuffer - Pointer to the socket buffer
        Device - Pointer to the device structure

Return Value:

        0 if packet could be sent
        1 if packet could not be sent

--*/

INT
NICSendPacket(
        IN PSKB SocketBuffer,
        IN PDEVICE Device
        )
{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
        PDPD_LIST_ENTRY dpdVirtual;
        UINT packetLength, flags;
#ifdef DEBUG
        PDEVICE device = pAdapter->Device;
#endif
#if LINUX_VERSION_CODE < 0x20343
        if (test_and_set_bit(0, (void*)&Device->tbusy) != 0)
        {       DBGPRINT_SEND(("SendPacket: %s Device busy\n",device->name));
                return 1;
        }
#else
        if (test_and_set_bit(0, (void*)&pAdapter->txing) != 0)
        {       DBGPRINT_SEND(("SendPacket: %s Device busy\n",device->name));
                return 1;
        }

#endif
	//spin_lock_irqsave(&pAdapter->SpinLock_send, flags);
        if (pAdapter->TailDPDVirtual->Next == pAdapter->HeadDPDVirtual)
        {       DBGPRINT_SEND(("SendPacket: %s DPDring full\n",device->name));
                pAdapter->DPDRingFull = TRUE;
                //spin_unlock_irqrestore(&pAdapter->SpinLock_send, flags);
#if LINUX_VERSION_CODE >= 0x20343
                netif_stop_queue(Device);
#endif
                return 1;
        }
        /* Get the free DPD from the DPD ring */
        dpdVirtual = pAdapter->TailDPDVirtual;
        dpdVirtual->FrameStartHeader = 0;
#ifdef __VMKERNEL_MODULE__
        dpdVirtual->SGList[0].Address = SocketBuffer->headMA;
#else
#if LINUX_VERSION_CODE < 0x20343
        dpdVirtual->SGList[0].Address = LIMIT_TO_32BIT_ADDR(virt_to_bus(
                                        SocketBuffer->data));

#else
        dpdVirtual->SGList[0].Address
                = pci_map_single(pAdapter->pciDevice, SocketBuffer->data,
                                 SocketBuffer->len, PCI_DMA_TODEVICE);

#endif
#endif
        packetLength = SocketBuffer->len;
        dpdVirtual->SGList[0].Count = packetLength | 0x80000000;
        dpdVirtual->FrameStartHeader |= (ULONG)(FSH_ROUND_UP_DEFEAT);
        dpdVirtual->SocketBuffer = SocketBuffer;
        dpdVirtual->PacketLength = SocketBuffer->len;
        dpdVirtual->DownNextPointer = 0;
	VMWARE_STRESS_ONLY_CORRUPT_BUFFER(NET_HW_CORRUPT_TX,
					  SocketBuffer, 0);
	VMWARE_STRESS_ONLY_FAIL_BUFFER(NET_HW_FAIL_HARD_TX,
				       SocketBuffer,DEV_FREE_SKB,
				       {clear_bit(0, (void*)&pAdapter->txing); return 0;});


#ifdef __VMKERNEL_MODULE__
        spin_lock_irqsave(&pAdapter->SpinLock_int, flags); 
#else
        save_flags(flags);
        cli();
#endif
        NIC_COMMAND_WAIT(pAdapter, COMMAND_DOWN_STALL);
        pAdapter->BytesInDPDQueue += packetLength;
        pAdapter->TailDPDVirtual->Previous->DownNextPointer =
                               dpdVirtual->DPDPhysicalAddress;
        pAdapter->TailDPDVirtual = dpdVirtual->Next;
        NIC_COMMAND(pAdapter, COMMAND_DOWN_UNSTALL);
        tc90x_SetCountDownTimer(pAdapter);
#ifdef __VMKERNEL_MODULE__
        spin_unlock_irqrestore(&pAdapter->SpinLock_int, flags);
#else
        restore_flags(flags);
#endif


        //spin_unlock_irqrestore(&pAdapter->SpinLock_send, flags);
#if LINUX_VERSION_CODE < 0x20343
        clear_bit(0, (void*)&Device->tbusy);
#else
        clear_bit(0, (void*)&pAdapter->txing);
#endif
        Device->trans_start = jiffies;
        return 0;
}


/*++

Routine Name:

        tc90x_TxCompleteEvent.

Routine Description:

        This routine handles the Tx complete event.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        None

--*/

VOID
tc90x_TxCompleteEvent(
        IN PNIC_INFORMATION Adapter
        )

{
        PNIC_INFORMATION pAdapter = Adapter;

        UCHAR txStatus;

        DBGPRINT_ERROR((KERN_CRIT "TxCompleteEvent: IN\n"));

        LOG_LABEL(pAdapter, "TCE>");

        txStatus = NIC_READ_PORT_UCHAR(pAdapter, TX_STATUS_REGISTER);
        NIC_WRITE_PORT_UCHAR(pAdapter, TX_STATUS_REGISTER, txStatus);

        if (txStatus & TX_STATUS_HWERROR) {
                //
                // Transmit HWError recovery.
                //
                DBGPRINT_SEND(("TxCompleteEvent: TxHWError\n"));
                pAdapter->Statistics.TxHWErrors++;
                if (tc90x_ResetAndEnableTransmitter(pAdapter) !=
                                NIC_STATUS_SUCCESS) {

                        DBGPRINT_ERROR(("TxCompleteEvent: TxReset failed\n"));
                        pAdapter->Hardware.Status = HARDWARE_STATUS_HUNG;
                        return;

                }

        }
        else if (txStatus & TX_STATUS_JABBER) {

                DBGPRINT_ERROR(("TxCompleteEvent: Jabber\n"));
                pAdapter->Statistics.TxJabberError++;

                if (tc90x_ResetAndEnableTransmitter(pAdapter) !=
                                NIC_STATUS_SUCCESS) {

                        DBGPRINT_ERROR(("TxCompleteEvent: TxReset failed\n"));
                        pAdapter->Hardware.Status = HARDWARE_STATUS_HUNG;
                        return;
                }
        }
        else if (txStatus & TX_STATUS_MAXIMUM_COLLISION) {

                DBGPRINT_ERROR(("TxCompleteEvent: Maximum collision\n"));
                pAdapter->Statistics.TxMaximumCollisions++;
                NIC_COMMAND(pAdapter, COMMAND_TX_ENABLE);
        }
        else {

                if (txStatus != 0 ) {

                        DBGPRINT_ERROR((KERN_CRIT "TxCompleteEvent: Unknown error\n"));
                        pAdapter->Statistics.TxUnknownError++;
                        NIC_COMMAND(pAdapter, COMMAND_TX_ENABLE);
                }
        }
        DBGPRINT_ERROR((KERN_CRIT "TxCompleteEvent: OUT\n"));
}


/*++

Routine Name:

        tc90x_ResetAndEnableTransmitter

Routine Description:

        This routine resets the transmitter.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        NIC_STATUS_SUCCESS if reset is successful.
        NIC_STATUS_FAILURE if reset failed.

--*/


NIC_STATUS
tc90x_ResetAndEnableTransmitter(
        IN PNIC_INFORMATION Adapter
        )

{
        PNIC_INFORMATION pAdapter = Adapter;
        NIC_STATUS nicStatus;

        NIC_COMMAND(pAdapter, COMMAND_TX_DISABLE);
        //
        // Wait for the transmit to go quiet.
        //
        NIC_COMMAND(pAdapter,COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);

        NIC_DELAY(10);

        if(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS)
        {       pAdapter->WaitCases = CHECK_TRANSMIT_IN_PROGRESS;
                TimeOutCount = jiffies + HZ;
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100); //max = 1s
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies);
                if(!(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS))
                {       DBGPRINT_ERROR((
                         "tc90x_ResetAndEnableTransmitter: media status is hung\n"));
                        return NIC_STATUS_FAILURE;
                }
        }

#ifdef SERR_NEEDED
    //
    // Issue down stall and delay 100 miliseconds for PCI retries to be over
    //
    NIC_COMMAND_WAIT(pAdapter, COMMAND_DOWN_STALL);
        TimeOutCount = jiffies + HZ/10;
        while(TimeOutCount > jiffies) ;
#endif
        //
        // Wait for download engine to stop
        //
        dmaControl_g = NIC_READ_PORT_UINT(pAdapter,DMA_CONTROL_REGISTER);
        NIC_DELAY(10);

        if(dmaControl_g & DMA_CONTROL_DOWN_IN_PROGRESS)
        {       pAdapter->WaitCases = CHECK_DMA_CONTROL;
                TimeOutCount = jiffies + HZ;
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100); //max = 1s
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies);
                if(!(dmaControl_g & DMA_CONTROL_DOWN_IN_PROGRESS))
                {       DBGPRINT_ERROR((
                        "tc90x_ResetAndEnableTransmitter: DMAControl hung\n"));
                        return NIC_STATUS_FAILURE;
                }
        }

        nicStatus = NIC_COMMAND_WAIT(
                      pAdapter,
                      COMMAND_TX_RESET |
                      TX_RESET_MASK_DOWN_RESET );

        if (nicStatus != NIC_STATUS_SUCCESS) {

        DBGPRINT_ERROR((
            "tc90x_ResetAndEnableTransmitter: Tx reset failed\n"));
        return NIC_STATUS_FAILURE;

    }

        NIC_COMMAND(pAdapter, COMMAND_TX_ENABLE);

        return NIC_STATUS_SUCCESS;
}



/*++

Routine Name:

        tc90x_CleanupSendLogic

Routine Description:

        This routines cleans the send logic.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        None

--*/

VOID
tc90x_CleanupSendLogic(
        IN PDEVICE Device
    )

{
    PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
    PDPD_LIST_ENTRY headDPDVirtual;

    DBGPRINT_SEND(("tc90x_CleanupSendLogic: IN\n"));
    //
    // Now clean up the DPD ring.
    //
    headDPDVirtual = pAdapter->HeadDPDVirtual;
    //
    // This is to take care of hardware raise condition.
    //
    pAdapter->TailDPDVirtual->FrameStartHeader = 0;

    while (1) {

        if (headDPDVirtual == pAdapter->TailDPDVirtual)
            break;
        //
        // Complete all the packets.
        //
        pAdapter->BytesInDPDQueue -= headDPDVirtual->PacketLength;

#if LINUX_VERSION_CODE >= 0x20343
        pci_unmap_single(pAdapter->pciDevice,
                         headDPDVirtual->SGList[0].Address,
                         headDPDVirtual->SocketBuffer->len,
                         PCI_DMA_TODEVICE);
#endif
	VMWARE_STRESS_RETAIN_BUFFER(headDPDVirtual->SocketBuffer,
				    NO_CONTINUE_CMD,
        DEV_FREE_SKB(headDPDVirtual->SocketBuffer));

        headDPDVirtual->SocketBuffer = NULL;
        headDPDVirtual->FrameStartHeader = 0;

        headDPDVirtual = headDPDVirtual->Next;
    }
    //
    // Update the head to point to this DPD now.
    //
    pAdapter->HeadDPDVirtual = headDPDVirtual;
    //
    // Initialize all DPDs.
    //
    headDPDVirtual = pAdapter->HeadDPDVirtual;

    while (1) {

        headDPDVirtual->DownNextPointer = 0;
        headDPDVirtual->SocketBuffer = NULL;
        headDPDVirtual->FrameStartHeader = 0;
        headDPDVirtual = headDPDVirtual->Next;
        if (headDPDVirtual == pAdapter->HeadDPDVirtual)
            break;
    }

    DBGPRINT_SEND(("tc90x_CleanupSendLogic: OUT\n"));
}





/*++

Routine Name:

        tc90x_UpCompleteEvent

Routine Description:

        This routine handles the receive event.

Arguments:

        Adapter - Pointer to the adapter structure

Return Value:

        None

--*/

VOID
tc90x_UpCompleteEvent(
        IN PNIC_INFORMATION Adapter
        )

{
        PNIC_INFORMATION pAdapter = Adapter;
        PDEVICE device = pAdapter->Device;
        PUPD_LIST_ENTRY currentUPDVirtual = pAdapter->HeadUPDVirtual;

        PSKB socketBuffer;

        UINT upPacketStatus;
        UINT frameLength;
        PUCHAR dataPointer;
        PETH_ADDR EthAddr;

        DBGPRINT_RECEIVE(("UpCompleteEvent: IN \n"));
        LOG_LABEL(pAdapter, "UCE>");


        while (1) {
                //
                // If done with all UPDs break.
                //
                upPacketStatus = currentUPDVirtual->UpPacketStatus;

                if (!(upPacketStatus & UP_PACKET_STATUS_COMPLETE)) {
                        break;
                }
                //
                // Get the frame length from the UPD.
                //
                frameLength = currentUPDVirtual->UpPacketStatus & 0x1FFF;
                //
                // Check if there is any error bit set.
                //
		VMWARE_STRESS_ONLY_FAIL_BUFFER(NET_HW_FAIL_RX,
					       NO_SKB_NEEDED, NO_KFREE_SKB_NEEDED,
					       { currentUPDVirtual->UpPacketStatus = 0;
					         currentUPDVirtual = currentUPDVirtual->Next;
						 continue;});

                if (upPacketStatus & UP_PACKET_STATUS_ERROR) {

                        if ((frameLength < ETHERNET_MINIMUM_FRAME_SIZE) ||
                            (upPacketStatus & UP_PACKET_STATUS_OVERRUN) ||
                            (upPacketStatus & UP_PACKET_STATUS_ALIGNMENT_ERROR) ||
                            (upPacketStatus & UP_PACKET_STATUS_CRC_ERROR) ||
                            (upPacketStatus & UP_PACKET_STATUS_OVERSIZE_FRAME)) {

                                if (upPacketStatus & UP_PACKET_STATUS_RUNT_FRAME) {
                                        DBGPRINT_ERROR(("UpCompleteEvent: Runt\n"));
                                         LOG_LABEL(pAdapter, "RUNT");
                                }
                                if (upPacketStatus & UP_PACKET_STATUS_ALIGNMENT_ERROR) {
                                        LOG_LABEL(pAdapter, "ALGN");
                                        DBGPRINT_ERROR(("UpCompleteEvent: Alignment\n"));
                                         pAdapter->Statistics.RxAlignmentError++;
                                }
                                 if (upPacketStatus & UP_PACKET_STATUS_CRC_ERROR) {
                                         LOG_LABEL(pAdapter, "CRC ");
                                         DBGPRINT_ERROR(("UpCompleteEvent: Crc\n"));
                                         pAdapter->Statistics.RxBadCRCError++;

                                 }
                                 if (upPacketStatus & UP_PACKET_STATUS_OVERSIZE_FRAME){
                                         LOG_LABEL(pAdapter, "OVSZ");
                                         DBGPRINT_ERROR(("UpCompleteEvent: Oversize\n"));
                                         pAdapter->Statistics.RxOversizeError++;

                                 }
                        //
                        // Discard this packet and move on.
                        //
                        currentUPDVirtual->UpPacketStatus = 0;
                        currentUPDVirtual = currentUPDVirtual->Next;
                        continue;
                        }
                        else {
                                pAdapter->Statistics.RxFramesOk++;
                                pAdapter->Statistics.RxBytesOk += frameLength;
                         }
                 }
                //
                // Try to allocate SKB
                //
                socketBuffer = DEV_ALLOC_SKB(
                                        ETHERNET_MAXIMUM_FRAME_SIZE + 2 +
                                        pAdapter->Hardware.CacheLineSize);

                if (socketBuffer != 0) {
                        socketBuffer->dev = device;
                        //
                        // Align IP on 16 byte boundaries
                        //
                        skb_reserve(socketBuffer, 2);
                        dataPointer = skb_put(
                                        socketBuffer,
                                        ETHERNET_MAXIMUM_FRAME_SIZE );
#if LINUX_VERSION_CODE < 0x20343
                        currentUPDVirtual->SGList[0].Address = LIMIT_TO_32BIT_ADDR(virt_to_bus(dataPointer));

#else
                        pci_unmap_single(pAdapter->pciDevice,
                                        currentUPDVirtual->SGList[0].Address,
                                        currentUPDVirtual->SocketBuffer->len,
                                        PCI_DMA_FROMDEVICE);

                        currentUPDVirtual->SGList[0].Address
                                = pci_map_single(pAdapter->pciDevice,
                                                 dataPointer,
                                                 ETHERNET_MAXIMUM_FRAME_SIZE,
                                                 PCI_DMA_FROMDEVICE);

#endif
			VMWARE_STRESS_ONLY_CORRUPT_BUFFER(NET_HW_CORRUPT_RX,
							  currentUPDVirtual->SocketBuffer, 40);
                        currentUPDVirtual->SGList[0].Count =
                                        ETHERNET_MAXIMUM_FRAME_SIZE | 0x80000000;

                        currentUPDVirtual->RxBufferVirtual = dataPointer;

                        currentUPDVirtual->SocketBuffer->protocol =
                                eth_type_trans(
                                        currentUPDVirtual->SocketBuffer,
                                        device);

                        SetRxTcpIpChecksumOffloadFlagsInSocketBuffer(
                                pAdapter,
                                currentUPDVirtual->SocketBuffer,
                                currentUPDVirtual->UpPacketStatus);
#ifdef __VMKERNEL_MODULE__
                        currentUPDVirtual->SocketBuffer->len = frameLength; 
#endif
                        
                        /*
                         * VMWARE FIX: Moved this code block above netif_rx
                         * to avoid touching the pkt after a free.
                         */
                        
                        //
                        // Check for Multicast
                        //
                        
                        EthAddr = (PETH_ADDR)(currentUPDVirtual->SocketBuffer->data);
                        if( (EthAddr->Addr[0] & ETH_MULTICAST_BIT) &&
                            !(COMPARE_MACS(EthAddr, BroadcastAddr)) )
                           pAdapter->Statistics.Rx_MulticastPkts++;
                        // END VMWARE FIX

#ifdef NICE_SUPPORT
			if( pAdapter->nice_rx )
				pAdapter->nice_rx(currentUPDVirtual->SocketBuffer, pAdapter->nice_ctx);
			else
				netif_rx(currentUPDVirtual->SocketBuffer);
#else  /* !NICE_SUPPORT */
				netif_rx(currentUPDVirtual->SocketBuffer);
#endif /* NICE_SUPPORT */

                        device->last_rx = jiffies;

			/****** phl *****/
                        currentUPDVirtual->SocketBuffer = socketBuffer;
                }
                else {

#ifndef __VMKERNEL_MODULE__
                        printk( KERN_CRIT "SKB allocation failed\n");
#endif
                        DBGPRINT_ERROR((
                                "UpCompleteEvent: SKB allocation failed\n"));

                }
                currentUPDVirtual->UpPacketStatus = 0;
                currentUPDVirtual = currentUPDVirtual->Next;
    }
        pAdapter->HeadUPDVirtual = currentUPDVirtual;

        DBGPRINT_RECEIVE(("UpCompleteEvent: OUT \n"));
        LOG_LABEL(pAdapter, "UCE<");
}


/*++

Routine Name:

        tc90x_ResetAndEnableReceiver.

Routine Description:

        This routine resets the receiver.

Arguments:

        Adapter - Pointer to the adapter structure

Return Value:

        NIC_STATUS_SUCCESS if reset succeeds.

--*/

NIC_STATUS
tc90x_ResetAndEnableReceiver(
        IN PNIC_INFORMATION Adapter
        )

{

    PNIC_INFORMATION pAdapter = Adapter;

    NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_DISABLE);
    NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET | RX_RESET_MASK_NETWORK_RESET);
    NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_ENABLE);

    return NIC_STATUS_SUCCESS;

}




/*++

Routine Name:

    NICIoctl

Routine Description:

    This routine is IOCTL handler.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        -ENODEV if no adapter found
        noAdapterFound if adapter(s) found

--*/

INT NICIoctl( IN PDEVICE Device,
                                         IN PIFREQ Request,
                                         IN INT command )
{
        BOOLEAN PhyResponding;
        long ioaddr;
        u16 *data;
        USHORT phy;
        PNIC_INFORMATION pAdapter;

        DBGPRINT_IOCTL(("NICIoctl: IN\n"));

        pAdapter = (PNIC_INFORMATION)Device->priv;
        ioaddr = Device->base_addr;
        data = (u16 *)&Request->ifr_data;
        phy = pAdapter->Hardware.phys;

        switch(command) {

        case SIOCDEVPRIVATE:            /* Get the address of the PHY in use. */
                data[0] = phy;
                /* Fall Through */

        case SIOCDEVPRIVATE+1:          /* Read the specified MII register. */

	  /* per B, change data[0] to date[1] */
                PhyResponding = ReadMIIPhy(pAdapter, data[1], (PUSHORT)&data[3]);
                if (!PhyResponding)
                        DBGPRINT_ERROR(("IOCTL-ReadPhy: Phy not responding"));
                return 0;

        case SIOCDEVPRIVATE+2:          /* Write the specified MII register */
                if (!suser())
                        return -EPERM;
	  /* per B, change data[0] to date[1] */
                WriteMIIPhy(pAdapter, data[1], data[2]);
                return 0;

#ifdef NICE_SUPPORT
        case SIOCNICE:
                {
                        struct nice_req* nrq;
                        nrq = (struct nice_req*)&Request->ifr_ifru;
                        if( nrq->cmd == NICE_CMD_QUERY_SUPPORT ) {
                                nrq->nrq_magic = NICE_DEVICE_MAGIC;
                                nrq->nrq_support_rx = 1;
                                return 0;
                        }
                        else if( nrq->cmd == NICE_CMD_SET_RX ) {
                                pAdapter->nice_rx = nrq->nrq_rx;
                                pAdapter->nice_ctx = nrq->nrq_ctx;
                                return 0;
                        }
                        else if( nrq->cmd == NICE_CMD_GET_RX ) {
                                nrq->nrq_rx = pAdapter->nice_rx;
                                nrq->nrq_ctx = pAdapter->nice_ctx;
                                return 0;
                        }
                        else {
                                return -EOPNOTSUPP;
                        }
                        break;
                }
#endif /* NICE_SUPPORT */

        default:
                return -EOPNOTSUPP;
        }
}

/*++

Routine Name:

    NICSetMacAddress

Routine Description:

    This routine override Station (MAC) address 

Arguments:

        Device - Pointer to the device structure.

Return Value:

        -ENODEV if the device is busy

--*/

INT NICSetMacAddress( IN PDEVICE Device, PVOID SockPtr)
{
struct sockaddr *addrp = SockPtr;
PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
USHORT index;

#if (LINUX_VERSION_CODE >= 0x20343)
if (pAdapter->start)
	return ENODEV;
#else
if (Device->start)
	return ENODEV;
#endif
for (index=0; index < 6; index++)
	pAdapter->StationAddress[index] = addrp->sa_data[index];
for (index=0; index < 6; index++)
	Device->dev_addr[index] = addrp->sa_data[index];
return 0;
} 

/*++

Routine Name:

        NICSetReceiveMode

Routine Description:

        This routine sets receive mode

Arguments:

        Device - Pointer to the device structure.

Return Value:

        -ENODEV if no adapter found
        noAdapterFound if adapter(s) found

--*/

VOID
NICSetReceiveMode(
        IN PDEVICE Device
        )
{

        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
        UINT count, hardwareReceiveFilter = 0;
        UCHAR flowControlAddress[ETHERNET_ADDRESS_SIZE];
        UCHAR broadcastAddress[ETHERNET_ADDRESS_SIZE];
        UCHAR address[ETHERNET_ADDRESS_SIZE];
        UINT numberMulticast, bitsInHashFilter, index;
        PDEV_MC_LIST multicastList;
#ifdef __VMKERNEL_MODULE__
        ULONG flags; // __VMKERNEL_MODULE__
#endif

        DBGPRINT_FUNCTION(("NICSetReceiveMode: In\n"));

#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_lock_irqsave(&pAdapter->SpinLock_misc, flags); // __VMKERNEL_MODULE__
#else
        spin_lock(&pAdapter->SpinLock_misc);
#endif // __VMKERNEL_MODULE__

#endif

        bitsInHashFilter = pAdapter->Hardware.BitsInHashFilter;
        //
        // Check if Promiscuous mode to be enabled.
        //
        if (Device->flags & IFF_PROMISC) {

                DBGPRINT_INITIALIZE(("IFF_PROMISC mode \n"));
                hardwareReceiveFilter |= RX_FILTER_PROMISCUOUS;
        }
        else if (Device->flags & IFF_ALLMULTI) {
                //
                // Check if ALL_MULTI mode to be enabled.
                //
                DBGPRINT_INITIALIZE(("IFF_ALLMULTI\n"));
                hardwareReceiveFilter |= RX_FILTER_INDIVIDUAL;
                hardwareReceiveFilter |= RX_FILTER_BROADCAST;
                hardwareReceiveFilter |= RX_FILTER_ALL_MULTICAST;
        }
        else  if (Device->flags & IFF_MULTICAST) {
                //
                // Check if hash multicast to be enabled.
                //
                DBGPRINT_INITIALIZE(("IFF_MULTICAST\n"));
                hardwareReceiveFilter |= RX_FILTER_INDIVIDUAL;
                hardwareReceiveFilter |= RX_FILTER_BROADCAST;
                hardwareReceiveFilter |= RX_FILTER_MULTICAST_HASH;
        }
        else {
                //
                // OS does not want to enable multicast.
                //
                DBGPRINT_INITIALIZE((
                        "Setting filter individual and broadcast\n"));
                hardwareReceiveFilter |= RX_FILTER_INDIVIDUAL;
                hardwareReceiveFilter |= RX_FILTER_BROADCAST;
        }
        //
        // Write the Rx filter
        //
        NIC_COMMAND(
                pAdapter,
                (USHORT)(COMMAND_SET_RX_FILTER | hardwareReceiveFilter));
        //
        // Clear the hash filter.
        //
        for (count=0; count < bitsInHashFilter; count++) {

                NIC_COMMAND(
                        pAdapter,
                        (USHORT)(COMMAND_SET_HASH_FILTER_BIT | count));
        }
        //
        // Set the hash filter.
        //
        numberMulticast = Device->mc_count;
        multicastList = Device->mc_list;

        DBGPRINT_INITIALIZE(("mcCount = %x\n", (INT)numberMulticast));

        for (count=0; count < numberMulticast; count++) {

                DBGPRINT_INITIALIZE(("Multicast = "));

                for (index=0; index < 6; index++)  {
                        address[index] = multicastList->dmi_addr[index];
                        DBGPRINT_INITIALIZE(("%x", address[index]));
                }

                DBGPRINT_INITIALIZE(("\n"));

                NIC_COMMAND(
                        pAdapter,
                        (USHORT)
                        (COMMAND_SET_HASH_FILTER_BIT |
                         0x400 |
                        tc90x_HashAddress(address)));

                multicastList = multicastList->next;
        }
        //
        // If receive filter is not promiscuos or multicast, enable
        // hash multicast for receiving the flow control packets and
        // for the broadcast.
        //
        if (!((hardwareReceiveFilter & RX_FILTER_PROMISCUOUS) ||
              (hardwareReceiveFilter & RX_FILTER_ALL_MULTICAST))) {

                hardwareReceiveFilter |= RX_FILTER_MULTICAST_HASH;
                //
                // Set the flow control enable
                //
                if (pAdapter->Hardware.FlowControlEnable &&
                    pAdapter->Hardware.FlowControlSupported &&
                    pAdapter->Hardware.FullDuplexEnable) {

                        DBGPRINT_INITIALIZE(("Setting flow control bit\n"));
                        //
                        // Set the flow control address bit
                        //
                        flowControlAddress[0] = NIC_FLOW_CONTROL_ADDRESS_0;
                        flowControlAddress[1] = NIC_FLOW_CONTROL_ADDRESS_1;
                        flowControlAddress[2] = NIC_FLOW_CONTROL_ADDRESS_2;
                        flowControlAddress[3] = NIC_FLOW_CONTROL_ADDRESS_3;
                        flowControlAddress[4] = NIC_FLOW_CONTROL_ADDRESS_4;
                        flowControlAddress[5] = NIC_FLOW_CONTROL_ADDRESS_5;

                        NIC_COMMAND(
                                pAdapter,
                                (USHORT)(COMMAND_SET_HASH_FILTER_BIT |
                                0x0400 |
                                tc90x_HashAddress(flowControlAddress)));
                }
                //
                // If there is a broadcast error, write value for broadcast.
                //
                if (FALSE == pAdapter->Hardware.BroadcastErrDone) {

                        DBGPRINT_INITIALIZE(("Broadcast Err Done\n"));
                        for (count=0; count < ETHERNET_ADDRESS_SIZE; count++)
                                broadcastAddress[count] = 0xff;

                        NIC_COMMAND(
                                pAdapter,
                                (USHORT)
                                (COMMAND_SET_HASH_FILTER_BIT |
                                 0x400 |
                                tc90x_HashAddress(broadcastAddress)) );
                }
        }
#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_unlock_irqrestore(&pAdapter->SpinLock_misc, flags);
#else
        spin_unlock(&pAdapter->SpinLock_misc);
#endif // __VMKERNEL_MODULE__

#endif

        DBGPRINT_FUNCTION(("NICSetReceiveMode: Out\n"));
}


/*++

Routine Name:

    NICGetStatistics

Routine Description:

    This routine gets the statistics.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        -ENODEV if no adapter found
        noAdapterFound if adapter(s) found

--*/

PENET_STATISTICS
NICGetStatistics(
        IN PDEVICE Device
        )
{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
        PNIC_STATISTICS statistics = &pAdapter->Statistics;
        PENET_STATISTICS enetStatistics = &pAdapter->EnetStatistics;
        ULONG flags;

#if 0
        if(pAdapter->start) {
#endif
#ifdef __VMKERNEL_MODULE__
           spin_lock_irqsave(&pAdapter->SpinLock_int, flags);
#else
           save_flags(flags);
           cli();
#endif

        enetStatistics->rx_packets      = statistics->RxFramesOk;
        enetStatistics->tx_packets      = statistics->TxFramesOk;

#if LINUX_VERSION_CODE > 0x20027
        enetStatistics->rx_bytes        = statistics->RxBytesOk;
        enetStatistics->tx_bytes        = statistics->TxBytesOk;
#endif

        enetStatistics->rx_errors       = statistics->RxBadCRCError +
                                          statistics->RxOverruns +
                                          statistics->RxAlignmentError + statistics->RxOversizeError;

        enetStatistics->tx_errors       = statistics->TxHWErrors +
                                          statistics->TxMaximumCollisions +
                                          statistics->TxJabberError +
                                          statistics->TxUnknownError;

        enetStatistics->rx_dropped      = statistics->RxOverruns;
        enetStatistics->multicast       = statistics->Rx_MulticastPkts;
        enetStatistics->collisions      = statistics->TxMaximumCollisions;
        enetStatistics->rx_over_errors  = statistics->RxOversizeError;
        enetStatistics->rx_crc_errors   = statistics->RxBadCRCError;
        enetStatistics->rx_frame_errors = statistics->RxAlignmentError;
        enetStatistics->rx_fifo_errors = statistics->RxOverruns;
        enetStatistics->tx_aborted_errors = statistics->TxUnknownError;
        enetStatistics->tx_carrier_errors = statistics->TxCarrierLost;
        enetStatistics->tx_heartbeat_errors = statistics->TxSQEErrors;

        enetStatistics->rx_missed_errors = 0;
        enetStatistics->tx_dropped      = 0;
        enetStatistics->rx_length_errors= 0;
        enetStatistics->tx_fifo_errors  = 0;
        enetStatistics->tx_window_errors = 0;

#ifdef __VMKERNEL_MODULE__
        spin_unlock_irqrestore(&pAdapter->SpinLock_int, flags);
#else
        restore_flags(flags);
#endif
#if 0
        }
#endif

        return enetStatistics;
}


/*++

Routine Name:

        tc90x_HashAddress.

Routine Description:

        This routine returns a bit corresponding to the hash address.

Arguments:

        MulticastAddress - Mutlicast address to be hashed.

Return Value:

        Hash Value

--*/

USHORT
tc90x_HashAddress(
        IN PUCHAR Address
        )
{

        UINT crc, carry, count, bit;
        UCHAR thisByte;
        //
        // Intialize CRC.
        //
        crc = 0xffffffff;

        for (count = 0; count < 6; count++) {
        thisByte = Address[count];
        for ( bit = 0; bit < 8; bit++) {
                carry = ((crc & 0x80000000) ? 1 : 0) ^
                                                        (thisByte & 0x01);
                crc <<= 1;
                thisByte >>= 1;
                if (carry)
                        crc  = (crc ^ 0x04c11db6) | carry;
        }

    }
    return (USHORT)(crc & 0x000003FF) ;
}

/*++

Routine Name:

        tc90x_SetMulticastAddresses.

Routine Description:

        This routine sets up the multicast list on the adapter.

Arguments:

        Device - Pointer to the device structure

Return Value:

        NIC_STATUS_SUCCESS if the addresses could be set.
        NIC_STATUS_FAILURE if the addresses could not be set.

--*/

NIC_STATUS
tc90x_SetMulticastAddresses(
        IN PNIC_INFORMATION Adapter
        )
{
    PNIC_INFORMATION pAdapter = Adapter;
    UCHAR FlowControlAddress[ETHERNET_ADDRESS_SIZE];
#ifdef __VMKERNEL_MODULE__
    ULONG flags;
#endif // __VMKERNEL_MODULE__

    DBGPRINT_FUNCTION(("SetMulticastAddresses: IN\n"));
#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
    spin_lock_irqsave(&pAdapter->SpinLock_m, flags);
#else
    spin_lock(&pAdapter->SpinLock_m);
#endif // __VMKERNEL_MODULE__

#endif
        //
        // Clear all bits in the hash filter, then write all multicast bits back
        //
        tc90x_InitializeHashFilter(pAdapter);

        if ((pAdapter->Hardware.FlowControlEnable &&
            pAdapter->Hardware.FlowControlSupported) &&
            (pAdapter->Hardware.FullDuplexEnable)) {
                //
                // Set the flow control address bit
                //
                FlowControlAddress[0] = NIC_FLOW_CONTROL_ADDRESS_0;
                FlowControlAddress[1] = NIC_FLOW_CONTROL_ADDRESS_1;
                FlowControlAddress[2] = NIC_FLOW_CONTROL_ADDRESS_2;
                FlowControlAddress[3] = NIC_FLOW_CONTROL_ADDRESS_3;
                FlowControlAddress[4] = NIC_FLOW_CONTROL_ADDRESS_4;
                FlowControlAddress[5] = NIC_FLOW_CONTROL_ADDRESS_5;

                NIC_COMMAND(
                        pAdapter,
                        (USHORT)(COMMAND_SET_HASH_FILTER_BIT |
                        0x0400 |
                        tc90x_HashAddress(FlowControlAddress)) );
        }

#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_unlock_irqrestore(&pAdapter->SpinLock_m, flags);
#else
        spin_unlock(&pAdapter->SpinLock_m);
#endif // __VMKERNEL_MODULE__

#endif

    DBGPRINT_FUNCTION(("SetMulticastAddresses: OUT\n"));

        return NIC_STATUS_SUCCESS;
}


/*++

Routine Name:

        InitializeHashFilter

Routine Description:

        Clear all bits in hash filter and setup the multicast address bit.

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        VOID

--*/

VOID
tc90x_InitializeHashFilter(
        IN PNIC_INFORMATION Adapter
        )
{

        PNIC_INFORMATION pAdapter= Adapter;
        PDEVICE device = pAdapter->Device;
        UINT count;
        UINT bitsInHashFilter;
        UINT numberMulticast;
        PDEV_MC_LIST multicastList;
        UCHAR address[ETHERNET_ADDRESS_SIZE];
        UINT index;
        //
        // Clear all bits in the hash filter, then write all multicast bits back
        //
        bitsInHashFilter = pAdapter->Hardware.BitsInHashFilter;

        for (count = 0; count < bitsInHashFilter; count++ ) {

                NIC_COMMAND(
                                pAdapter,
                                (USHORT)(COMMAND_SET_HASH_FILTER_BIT | count));
        }

        numberMulticast = device->mc_count;
        multicastList = device->mc_list;

        DBGPRINT_INITIALIZE(("mcCount = %x\n", (INT)numberMulticast));

        for (count=0; count < numberMulticast; count++) {

                DBGPRINT_INITIALIZE(("Multicast = "));

                for (index=0; index < 6; index++)  {
                        address[index] = multicastList->dmi_addr[index];
                        DBGPRINT_INITIALIZE(("%x", address[index]));

                }
                DBGPRINT_INITIALIZE(("\n"));

                NIC_COMMAND(
                        pAdapter,
                        (USHORT)
                        (COMMAND_SET_HASH_FILTER_BIT |
                         0x400 |
                        tc90x_HashAddress(address)) );

                multicastList = multicastList->next;
        }
}



/*++

Routine Name:

    tc90x_SetupMedia.

Routine Description:

        This routine checks whether autoselection is specified.  If it
        does, it calls MainAutoSelectionRoutine else for non-autoselect
        case, calls ProgramMII.

Arguments:

        Device - Pointer to the device structure.

Return Value:

    NIC_STATUS_SUCCESS if the media could be set up.

--*/

NIC_STATUS
tc90x_SetupMedia(
        IN PDEVICE Device
    )

{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
        USHORT OptionAvailable = 0;
        UINT InternalConfig = 0;
        USHORT InternalConfig0 = 0;
        USHORT InternalConfig1 = 0;
        NIC_STATUS nicStatus;
        USHORT MacControl = 0;
        CONNECTOR_TYPE NotUsed;
        BOOLEAN Handles100Mbit = FALSE;
	BOOLEAN PhyResponding = FALSE;
	USHORT PhyOUI = 0;
	USHORT Mii_Add = 0;

        DBGPRINT_FUNCTION(("SetupMedia: In\n"));

        pAdapter->Hardware.AutoSelect = FALSE;
        pAdapter->Hardware.ConfigConnector = CONNECTOR_UNKNOWN;
        pAdapter->Hardware.Connector = CONNECTOR_UNKNOWN;
        //
        // Assumption made here for Cyclone, Hurricane, and Tornado
        // adapters have the same fixed PHY address.  For other PHY
        // address values, this needs to be changed.
        //
        pAdapter->Hardware.phys = MII_PHY_ADDRESS;
        pAdapter->Hardware.MIIReadCommand = MII_PHY_ADDRESS | 0x6000;
        pAdapter->Hardware.MIIWriteCommand = MII_PHY_ADDRESS | 0x5002;

//sfy 1/16/01 for Ext-Phy detect 
	if (OptionAvailable & MEDIA_OPTIONS_MII_AVAILABLE) {
		for(Mii_Add = 0; Mii_Add < 32; Mii_Add++) {
			pAdapter->Hardware.MIIReadCommand = 0x6000 | (Mii_Add << 7);
			pAdapter->Hardware.MIIWriteCommand = 0x5002 | (Mii_Add << 7);
		    PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_OUI, &PhyOUI);
		    if (PhyResponding && PhyOUI) {
				DBGPRINT_INITIALIZE(("New MIIReadCommand is %x\n", 
					pAdapter->Hardware.MIIReadCommand));
				break;
			}
		}
	}
//sfy 
        // If this is a 10mb Lightning card, assume that the 10FL bit is
        // set in the media options register
        //
        if (pAdapter->Hardware.DeviceId == NIC_PCI_DEVICE_ID_900A) {
                DBGPRINT_INITIALIZE((
                        "SetupMedia: 10BaseFL force Media Option \n"));
                OptionAvailable = MEDIA_OPTIONS_10BASEFL_AVAILABLE;
        }
        else {
                //
                // Read the MEDIA OPTIONS to see what connectors are available
                //
                NIC_COMMAND(
                        pAdapter,
                    COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);
                OptionAvailable = NIC_READ_PORT_USHORT(
                                                        pAdapter,
                                                        MEDIA_OPTIONS_REGISTER);
        }
        //
        // Read the internal config through EEPROM since reset
        // invalidates the normal register value.
        //
        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_INTERNAL_CONFIG_WORD_0,
                        (PUSHORT)&InternalConfig0);

        if (nicStatus != NIC_STATUS_SUCCESS) {
                DBGPRINT_ERROR((
                        "SetupMedia: InternalConfig 0 read failed\n"));
                return NIC_STATUS_FAILURE;
        }

        nicStatus = tc90x_ReadEEPROM(
                        pAdapter,
                        EEPROM_INTERNAL_CONFIG_WORD_1,
                        (PUSHORT)&InternalConfig1);

        if (nicStatus != NIC_STATUS_SUCCESS) {
                DBGPRINT_ERROR((
                        "SetupMedia: InternalConfig 1 read failed\n"));
                return NIC_STATUS_FAILURE;
        }

        InternalConfig = InternalConfig0 | (InternalConfig1 <<16);

        DBGPRINT_INITIALIZE((
                "SetupMedia: InternalConfig %x\n", (INT)InternalConfig));

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);
        //
        // Read the current value of the InternalConfig register. If it's different
        // from the EEPROM values, than write it out using the EEPROM values.
        // This is done since a global reset may invalidate the register value on
        // some ASICs. Also, writing to InternalConfig may reset the PHY on some ASICs.
        //
        if (InternalConfig != NIC_READ_PORT_UINT(
                                pAdapter,
                                INTERNAL_CONFIG_REGISTER)) {
                NIC_WRITE_PORT_UINT(
                        pAdapter,
                        INTERNAL_CONFIG_REGISTER,
                        InternalConfig);
        }
        //
        // Get the connector to use.
        //
        if (pAdapter->Hardware.ConfigConnector == CONNECTOR_UNKNOWN) {
                pAdapter->Hardware.ConfigConnector =
                (InternalConfig & INTERNAL_CONFIG_TRANSCEIVER_MASK) >> 20;
                pAdapter->Hardware.Connector =
                                pAdapter->Hardware.ConfigConnector;
                if (InternalConfig & INTERNAL_CONFIG_AUTO_SELECT)
                        pAdapter->Hardware.AutoSelect = TRUE;
                ProcessMediaOverrides(pAdapter, OptionAvailable);
        }
        //
        // If auto selection of connector was specified, do it now...
        //
        if (pAdapter->Hardware.AutoSelect) {
                DBGPRINT_INITIALIZE(("SetupMedia: Autoselect set\n"));
                NIC_COMMAND(pAdapter, COMMAND_STATISTICS_DISABLE);
                tc90x_MainAutoSelectionRoutine(pAdapter, OptionAvailable);
        }
        else {
            //
            // MII connector needs to be initialized and the data rates
            // set up even in the non-autoselect case
            //
                DBGPRINT_INITIALIZE(("SetupMedia: Adapter in forced-mode\n"));

                if ((pAdapter->Hardware.Connector == CONNECTOR_MII) ||
                    (pAdapter->Hardware.Connector ==
                     CONNECTOR_AUTONEGOTIATION)) {
                        ProgramMII(pAdapter, CONNECTOR_MII);
                }
                else {
                        if ((pAdapter->Hardware.Connector ==
                             CONNECTOR_100BASEFX) ||
                             (pAdapter->Hardware.Connector ==
                             CONNECTOR_100BASETX)) {
                                pAdapter->Hardware.LinkSpeed = LINK_SPEED_100;
                        }
                        else {
                                pAdapter->Hardware.LinkSpeed = LINK_SPEED_10;
                        }
                }
                tc90x_SetupConnector(
                        pAdapter,
                        pAdapter->Hardware.Connector,
                        &NotUsed);
        }
        //
        // Check link speed and duplex settings before doing anything else.
        // If the call succeeds, we know the link is up, so we'll update the
        // link state.
        //
        if (GetLinkSpeed(pAdapter, &Handles100Mbit)) {
                pAdapter->Hardware.LinkSpeed =
                        (Handles100Mbit) ? LINK_SPEED_100 : LINK_SPEED_10;
                pAdapter->Hardware.LinkState = LINK_UP;
        }
        else
                pAdapter->Hardware.LinkState = LINK_DOWN_AT_INIT;
        //
        // Set up duplex mode
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        MacControl = NIC_READ_PORT_USHORT(pAdapter, MAC_CONTROL_REGISTER);

        if (pAdapter->Hardware.FullDuplexEnable) {
                //
                // Set Full duplex in MacControl register
                //
                MacControl |= MAC_CONTROL_FULL_DUPLEX_ENABLE;
                DBGPRINT_INITIALIZE(("Changed link to full duplex\n"));
                //
                // Since we're switching to full duplex, enable flow control.
                //
                if (pAdapter->Hardware.FlowControlSupported) {
                        DBGPRINT_INITIALIZE(("SetupMedia: flow Control support is on!\n"));
                        MacControl |=  MAC_CONTROL_FLOW_CONTROL_ENABLE;
                        pAdapter->Hardware.FlowControlEnable = TRUE;
                        tc90x_SetMulticastAddresses(pAdapter);
                }
        }
        else {
                //
                // Set Half duplex in MacControl register
                //
                MacControl &= ~MAC_CONTROL_FULL_DUPLEX_ENABLE;
                DBGPRINT_INITIALIZE(("Changed link to half duplex\n"));
                //
                // Since we're switching to half duplex, disable flow control
                //
                if (pAdapter->Hardware.FlowControlSupported) {
                        MacControl &= ~ MAC_CONTROL_FLOW_CONTROL_ENABLE;
                        pAdapter->Hardware.FlowControlEnable = FALSE;
                        tc90x_SetMulticastAddresses(pAdapter);
                }
        }

        NIC_WRITE_PORT_USHORT(pAdapter,MAC_CONTROL_REGISTER,MacControl);

        if (tc90x_ResetAndEnableTransmitter(pAdapter) != NIC_STATUS_SUCCESS) {
                        DBGPRINT_INITIALIZE((
                                "SetupMedia: Reset transmitter failed\n"));
                        DBGPRINT_FUNCTION(("SetupMedia: Out with error\n"));
                        return NIC_STATUS_FAILURE;
        }

        if (tc90x_ResetAndEnableReceiver(pAdapter) != NIC_STATUS_SUCCESS) {
                        DBGPRINT_INITIALIZE((
                                "SetupMedia: Reset receiver failed\n"));
                        DBGPRINT_FUNCTION((
                                "SetupMedia: Out with error\n"));
                        return NIC_STATUS_FAILURE;
        }
        //
        // This is for advertisement of flow control.  We only need to
        // call this if the adapter is using flow control, in Autoselect
        // mode and not a Tornado board.
        //
        if ((pAdapter->Hardware.AutoSelect &&
             pAdapter->Hardware.FlowControlEnable) &&
             (pAdapter->Hardware.DeviceId != NIC_PCI_DEVICE_ID_9200) &&
	     (pAdapter->Hardware.DeviceId != NIC_PCI_DEVICE_ID_9201) &&
             (pAdapter->Hardware.DeviceId != NIC_PCI_DEVICE_ID_9805)) {
                tc90x_FlowControl(pAdapter);
        }
        DBGPRINT_FUNCTION(("SetupMedia: Out\n"));
        return NIC_STATUS_SUCCESS;
}

/*++

Routine Name:

        ProcessMediaOverrides

Routine Description:

        Change the connector and duplex if values are present in command line.

Arguments:

        IN PNIC_INFORMATION Adapter
        IN USHORT OptionAvailable

Return Value:

        VOID

--*/

VOID
ProcessMediaOverrides(
        IN PNIC_INFORMATION Adapter,
        IN USHORT OptionAvailable
        )
{
        PNIC_INFORMATION pAdapter= Adapter;
        UINT InternalConfig = 0;
        UINT OldInternalConfig = 0;

        NIC_COMMAND(pAdapter,
                                COMMAND_SELECT_REGISTER_WINDOW |
                                REGISTER_WINDOW_3);

        InternalConfig = NIC_READ_PORT_UINT(
                                                pAdapter,
                                                INTERNAL_CONFIG_REGISTER);

        OldInternalConfig = InternalConfig;

        if (pAdapter->Hardware.MediaOverride == MEDIA_AUTO_SELECT)
                InternalConfig |= INTERNAL_CONFIG_AUTO_SELECT;
        else
                InternalConfig &= ~INTERNAL_CONFIG_AUTO_SELECT;
        //
        // Write to the InternalConfig register only if it's changed.
        //
        if (OldInternalConfig != InternalConfig) {
                NIC_WRITE_PORT_UINT(pAdapter,
                                                        INTERNAL_CONFIG_REGISTER,
                                                        InternalConfig);
        }

        switch (pAdapter->Hardware.MediaOverride) {

                case MEDIA_AUTO_SELECT:
                        pAdapter->Hardware.AutoSelect = TRUE;
                        break;

            case MEDIA_10BASE_T:
                        if (OptionAvailable & MEDIA_OPTIONS_10BASET_AVAILABLE) {
                                pAdapter->Hardware.Connector = CONNECTOR_10BASET;
                                pAdapter->Hardware.AutoSelect = FALSE;
                        }
                        break;

                case MEDIA_10AUI:
                        if (OptionAvailable & MEDIA_OPTIONS_10AUI_AVAILABLE) {
                                pAdapter->Hardware.Connector = CONNECTOR_10AUI;
                                pAdapter->Hardware.AutoSelect = FALSE;
                        }
                        break;

                case MEDIA_10BASE_2:
                        if (OptionAvailable & MEDIA_OPTIONS_10BASE2_AVAILABLE) {
                                pAdapter->Hardware.Connector = CONNECTOR_10BASE2;
                                pAdapter->Hardware.AutoSelect = FALSE;
                        }
                        break;

                case MEDIA_100BASE_TX:
                        if (OptionAvailable &
                            MEDIA_OPTIONS_100BASETX_AVAILABLE) {
                                pAdapter->Hardware.Connector = CONNECTOR_100BASETX;
                                pAdapter->Hardware.AutoSelect = FALSE;
                        }
                        break;

                case MEDIA_100BASE_FX:
                        if (OptionAvailable &
                            MEDIA_OPTIONS_100BASEFX_AVAILABLE) {
                                pAdapter->Hardware.Connector = CONNECTOR_100BASEFX;
                                pAdapter->Hardware.AutoSelect = FALSE;
                        }
                        break;

                case MEDIA_10BASE_FL:
                        if (OptionAvailable &
                            MEDIA_OPTIONS_10BASEFL_AVAILABLE) {
                                pAdapter->Hardware.Connector = CONNECTOR_10AUI;
                                pAdapter->Hardware.AutoSelect = FALSE;

                        }
                        break;

            case MEDIA_NONE:
                        break;
        }
}

/*++

Routine Name:

        TickMediaHandler

Routine Description:

        Adjust linkspeed and duplex in tick handler

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        VOID

--*/

VOID
tc90x_TickMediaHandler(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;

        if ((pAdapter->Hardware.Connector == CONNECTOR_AUTONEGOTIATION) ||
            (pAdapter->Hardware.Connector == CONNECTOR_10BASET) ||
            (pAdapter->Hardware.Connector == CONNECTOR_100BASETX)) {
                //
                // TP case
                //
                CheckTPLinkState(pAdapter);
        }
        else if (pAdapter->Hardware.Connector == CONNECTOR_100BASEFX) {
                //
                // FX case
                //
                CheckFXLinkState(pAdapter);
        }
}


/*++

Routine Name:

        CheckTPLinkState

Routine Description:

        Determine whether to notify the operating system if link is lost
        or restored. For autonegotiation case, made adjustments to duplex
        and speed if necessary.

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        VOID

--*/

VOID
CheckTPLinkState(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN handles100Mbit = FALSE;
        USHORT PhyStatus = 0;
        BOOLEAN OldFullDuplex;
        UINT OldLinkSpeed;

        if (ReadMIIPhy(pAdapter, MII_PHY_STATUS, &PhyStatus)) {

                if (!(PhyStatus & MII_STATUS_LINK_UP)) {
                        //
                        // If OS doesn't know link was lost, go
                        // ahead and notify
                        //
                        if (pAdapter->Hardware.LinkState == LINK_UP)
                                IndicateToOSLinkStateChange(pAdapter);
                }
                else {
                        //
                        // If OS doesn't know link was restored, go
                        // ahead and notify
                        //
                        if (pAdapter->Hardware.LinkState != LINK_UP)
                                IndicateToOSLinkStateChange(pAdapter);

                        if (((PhyStatus & MII_STATUS_AUTO) &&
                                (PhyStatus & MII_STATUS_EXTENDED)) &&
                                (pAdapter->Hardware.Connector == CONNECTOR_AUTONEGOTIATION)) {

                                OldLinkSpeed = pAdapter->Hardware.LinkSpeed;
                                OldFullDuplex = pAdapter->Hardware.FullDuplexEnable;
                                //
                                // Capable of autonegotiation
                                //
                                if (GetLinkSpeed(pAdapter, &handles100Mbit)) {
                                        if (handles100Mbit)
                                                pAdapter->Hardware.LinkSpeed = LINK_SPEED_100;
                                        else
                                                pAdapter->Hardware.LinkSpeed = LINK_SPEED_10;
                                        //
                                        // Set up for new speed if speed changed. Set
                                        // counterSpeed bit in DmaCtrl register.
                                        //
                                        if (pAdapter->Hardware.LinkSpeed != OldLinkSpeed)
                                                tc90x_SetupNewSpeed(pAdapter);
                                        //
                                        // Set up for new duplex mode if duplex changed.
                                        //
                                        if (pAdapter->Hardware.FullDuplexEnable != OldFullDuplex)
                                                tc90x_SetupNewDuplex(pAdapter);
                                }
                                else
                                        DBGPRINT_INITIALIZE(("Failed to get link speed\n"));
                        }
                }
        }
}

/*++

Routine Name:

        CheckFXLinkState

Routine Description:

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        VOID

--*/

VOID
CheckFXLinkState(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        USHORT MediaStatus = 0;

        NIC_COMMAND(
                pAdapter,
        COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        MediaStatus = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);

    if (!(MediaStatus & MEDIA_STATUS_LINK_DETECT)) {
                if (pAdapter->Hardware.LinkState == LINK_UP)
                        IndicateToOSLinkStateChange(pAdapter);
    }
    else {
       if (pAdapter->Hardware.LinkState != LINK_UP)
               IndicateToOSLinkStateChange(pAdapter);
    }
}

/*++

Routine Name:

        tc90x_SetupNewDuplex

Routine Description:

        Setup new duplex in MacControl register.

Arguments:

        IN PNIC_INFORMATION Adapter


Return Value:

        NIC_STATUS

--*/
NIC_STATUS
tc90x_SetupNewDuplex(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        USHORT MacControl = 0;

        DBGPRINT_INITIALIZE(("SetupNewDuplex: IN\n"));

        NIC_COMMAND(pAdapter, COMMAND_RX_DISABLE);
        NIC_COMMAND(pAdapter, COMMAND_TX_DISABLE);

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);
        //
        // Wait for transmit to go quiet
        //
        MediaStatus_g = 0;
        MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);
        NIC_DELAY(10);

        if(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS)
        {       pAdapter->WaitCases = CHECK_TRANSMIT_IN_PROGRESS;
                TimeOutCount = jiffies + HZ;
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100); //max = 1s
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies);
                if(!(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS))
                {       DBGPRINT_ERROR((
                        "tc90x_SetupNewDuplex: Packet not picked up by hardware"));
                        return NIC_STATUS_FAILURE;
                }
        }
        //
        // Wait for receive to go quiet
        //
        MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);

        NIC_DELAY(10);

        if(MediaStatus_g &  MEDIA_STATUS_CARRIER_SENSE)
        {       pAdapter->WaitCases = CHECK_CARRIER_SENSE;
                TimeOutCount = jiffies + HZ;
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/100); //max = 1s
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies) ;
                if(!(MediaStatus_g &  MEDIA_STATUS_CARRIER_SENSE))
                {       pAdapter->Hardware.Status = HARDWARE_STATUS_HUNG;
                        DBGPRINT_ERROR(("tc90x_SetupNewDuplex: Packet not uploaded by hardware"));
                        return NIC_STATUS_FAILURE;
                }
        }

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        MacControl = NIC_READ_PORT_USHORT(pAdapter, MAC_CONTROL_REGISTER);

        if (pAdapter->Hardware.FullDuplexEnable) {
                //
                // Set Full duplex in MacControl register
                //
                MacControl |= MAC_CONTROL_FULL_DUPLEX_ENABLE;
                DBGPRINT_INITIALIZE(("Changed link to full duplex\n"));
                //
                // Since we're switching to full duplex, enable flow control.
                //
                if (pAdapter->Hardware.FlowControlSupported) {

                        MacControl |=  MAC_CONTROL_FLOW_CONTROL_ENABLE;
                        pAdapter->Hardware.FlowControlEnable = TRUE;

                        tc90x_SetMulticastAddresses(pAdapter);
                }
        }
    else {
                //
                // Set Half duplex in MacControl register
                //
                MacControl &= ~MAC_CONTROL_FULL_DUPLEX_ENABLE;
                DBGPRINT_INITIALIZE(("Changed link to half duplex\n"));
                //
                // Since we're switching to half duplex, disable flow control
                //
                if (pAdapter->Hardware.FlowControlEnable &&
                        pAdapter->Hardware.FlowControlSupported) {

                        MacControl &= ~ MAC_CONTROL_FLOW_CONTROL_ENABLE;
                        pAdapter->Hardware.FlowControlEnable = FALSE;
                        tc90x_SetMulticastAddresses(pAdapter);
                }
        }
        NIC_WRITE_PORT_USHORT(pAdapter, MAC_CONTROL_REGISTER, MacControl);
        NIC_DELAY(20);

        /* must do Tx/Rx reset and wait at least 3s */
//      NIC_COMMAND(pAdapter,COMMAND_TX_RESET);
//      NIC_COMMAND(pAdapter, COMMAND_RX_RESET);

//      TimeOutCount = jiffies + 3*HZ;
//      while(TimeOutCount > jiffies) ;

        NIC_COMMAND(pAdapter, COMMAND_RX_ENABLE);
        NIC_COMMAND(pAdapter, COMMAND_TX_ENABLE);

        DBGPRINT_INITIALIZE(("SetupNewDuplex: OUT\n"));

        return NIC_STATUS_SUCCESS;
}


/*++

Routine Name:

        tc90x_SetupNewSpeed

Routine Description:

        Sets the counter speed in the DMA control register. Clear bit
        for 10Mbps or set the bit for 100Mbps.

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        VOID

--*/

VOID
tc90x_SetupNewSpeed(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        UINT DmaControl = 0;

        DBGPRINT_INITIALIZE(("SetupNewSpeed: IN\n"));

        DmaControl = NIC_READ_PORT_UINT(
                                        pAdapter,
                                        DMA_CONTROL_REGISTER);

        if (pAdapter->Hardware.LinkSpeed == LINK_SPEED_100) {

                DmaControl |= DMA_CONTROL_COUNTER_SPEED;
        }
        else {

                DmaControl &= ~DMA_CONTROL_COUNTER_SPEED;
        }

        NIC_WRITE_PORT_UINT(
                pAdapter,
                DMA_CONTROL_REGISTER,
                DmaControl);

        DBGPRINT_INITIALIZE(("Changed link speed to %d bps\n",
                pAdapter->Hardware.LinkSpeed));

        DBGPRINT_INITIALIZE(("SetupNewSpeed: OUT\n"));
}


/*++

Routine Name:

        IndicateToOSLinkStateChange

Routine Description:

        Notify to NDI wrapper the change in link state.

Arguments:

        IN PNIC_INFORMATION Adapter

Return Value:

        VOID

--*/

VOID
IndicateToOSLinkStateChange(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;

        if (pAdapter->Hardware.LinkState != LINK_DOWN) {
                DBGPRINT_INITIALIZE(("Link Lost...\n"));
                pAdapter->Hardware.LinkState = LINK_DOWN;
                //(pAdapter->Device)->flags &= ~(IFF_UP|IFF_RUNNING);
        }
        else {
                DBGPRINT_INITIALIZE(("Link Regained...\n"));
                pAdapter->Hardware.LinkState = LINK_UP;
                //(pAdapter->Device)->flags |= (IFF_UP|IFF_RUNNING);
        }
}






/*++

Routine Name:

        NICInterrupt

Routine Description:

        This routine handles the interrupt.

Arguments:

        Irq
        DeviceId
        Registers

Return Value:

        None

--*/

VOID
NICInterrupt IRQ(
        INT Irq,
        PVOID DeviceId,
        PTREGS Registers
        )

{
        //
        // Use the now-standard shared IRQ implementation.
        //
        PDEVICE Device = DeviceId;
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Device->priv;
        USHORT intStatus = 0;
        UCHAR loopCount = 2;
        BOOLEAN countDownTimerEventCalled = FALSE;
#ifdef __VMKERNEL_MODULE__
        ULONG flags;
#endif // __VMKERNEL_MODULE__

#if defined(__i386__) /* SMP - Becker's fix */

#if LINUX_VERSION_CODE < 0x20343
        /* A lock to prevent simultaneous entry bug on Intel SMP machines. */
        if (test_and_set_bit(0, (void*)&Device->interrupt)) {
                DBGPRINT_ERROR(("NICInterrupt: %s simultaneous entry of an interrupt handler.\n",
                                        Device->name));
                /* Avoid halting machine. */
                Device->interrupt = 0;
                return ;
        }
#else
#if 0
        if (test_and_set_bit(0, (void*)&pAdapter->interrupt)) {
                DBGPRINT_ERROR(("NICInterrupt: %s simultaneous entry of an interrupt handler.\n",
                                        Device->name));
                /* Avoid halting machine. */
                pAdapter->interrupt = 0;
                return ;
        }
#endif

#endif
#else
#if LINUX_VERSION_CODE < 0x20343
        if (Device->interrupt) {
                DBGPRINT_ERROR(("NICInterrupt: %s Re-enter the interrupt handler.\n", Device->name));
                return;
        }
#else
#if 0
        if (pAdapter->interrupt) {
                DBGPRINT_ERROR(("NICInterrupt: %s Re-enter the interrupt handler.\n", Device->name));
                return;
        }
#endif

#endif

#endif /* end fix */

#if LINUX_VERSION_CODE < 0x20343
        Device->interrupt = 1;
#else
        /* pAdapter->interrupt = 1; */
#endif

#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_lock_irqsave(&pAdapter->SpinLock_int, flags);
#else
        spin_lock(&pAdapter->SpinLock_int);
#endif // __VMKERNEL_MODULE__

#endif

        intStatus = NIC_READ_PORT_UCHAR(
                        pAdapter,
                        INTSTATUS_COMMAND_REGISTER);

        if (!(intStatus & INTSTATUS_INTERRUPT_LATCH))
        {

#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
           spin_unlock_irqrestore(&pAdapter->SpinLock_int, flags);
#else 
           spin_unlock(&pAdapter->SpinLock_int);
#endif // __VMKERNEL_MODULE__

#endif
#if LINUX_VERSION_CODE < 0x20343
                /* Do we need to have i386 fix over here too...*/
                Device->interrupt = 0;
#else
                /* pAdapter->interrupt = 0; */
#endif
                return;
        }
        else
        {
                /* Mask all the interrupts */
                NIC_MASK_ALL_INTERRUPT(pAdapter);
                NIC_COMMAND(
                                pAdapter,
                                COMMAND_ACKNOWLEDGE_INTERRUPT |
                                ACKNOWLEDGE_INTERRUPT_LATCH);
        }
        while (loopCount--)
        {
                /* Read the interrupt status register. */
                intStatus = NIC_READ_PORT_USHORT(pAdapter,
                                INTSTATUS_COMMAND_REGISTER);

                intStatus &= INTSTATUS_INTERRUPT_MASK;

                if (!intStatus) break;

                if (intStatus & INTSTATUS_HOST_ERROR)
                {       printk( KERN_CRIT "HostError ");
                        DBGPRINT_ERROR((
                                                "NICInterrupt: HostError event happened.\n"));
                        tc90x_HostErrorEvent(pAdapter);
                }
                if (intStatus & INTSTATUS_UPDATE_STATISTICS)
                {       // interrupt is cleared by reading  statistics.
                        tc90x_UpdateStatisticsEvent(pAdapter);
                }

                if (intStatus & INTSTATUS_UP_COMPLETE)
                { NIC_COMMAND(
                                pAdapter,
                                COMMAND_ACKNOWLEDGE_INTERRUPT |
                                ACKNOWLEDGE_UP_COMPLETE);

                tc90x_UpCompleteEvent(pAdapter);
                }
                if (intStatus & INTSTATUS_INTERRUPT_REQUESTED)
                {       NIC_COMMAND(
                                pAdapter,
                                COMMAND_ACKNOWLEDGE_INTERRUPT |
                                ACKNOWLEDGE_INTERRUPT_REQUESTED);

                tc90x_CountDownTimerEvent(pAdapter);
                countDownTimerEventCalled = TRUE;
                }

                if (intStatus & INTSTATUS_TX_COMPLETE)
                {       tc90x_TxCompleteEvent(pAdapter);
                }

                if ((intStatus & INTSTATUS_RX_COMPLETE) ||
                                (intStatus & INTSTATUS_LINK_EVENT) ||
                                (intStatus & INTSTATUS_DOWN_COMPLETE))
                {       DBGPRINT_ERROR((
                                        "NICInterrupt: Unknown interrupt\n"));
                DBGPRINT_ERROR((
                                        "NICInterrupt: IntStatus =%x\n", intStatus));
                }

                if (FALSE == countDownTimerEventCalled)
                {       tc90x_CountDownTimerEvent(pAdapter);
                        countDownTimerEventCalled = TRUE;
                }

        }
        NIC_UNMASK_ALL_INTERRUPT(pAdapter);

#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_unlock_irqrestore(&pAdapter->SpinLock_int, flags);
#else
        spin_unlock(&pAdapter->SpinLock_int);
#endif // __VMKERNEL_MODULE__

#endif

        // Becker's fix?
#if defined(__i386__)
#if LINUX_VERSION_CODE < 0x20343
        clear_bit(0, (void*)&Device->interrupt);
#else
        /* clear_bit(0, (void*)&pAdapter->interrupt); */
#endif
#else
#if LINUX_VERSION_CODE < 0x20343
        Device->interrupt = 0;
#else
        /* pAdapter->interrupt = 0; */
#endif
#endif
	VMWARE_STRESS_ONLY_STOP_QUEUE(Device, return);
        return;
}


/*++

Routine Name:

        tc90x_HostErrorEvent.

Routine Description:

        This routine handles the host error.

Arguments:

        Device - Pointer to the device structure.

Return Value:

        None

--*/

VOID
tc90x_HostErrorEvent(IN PNIC_INFORMATION Adapter)
{
        PNIC_INFORMATION pAdapter = Adapter;
        TASKQ   HostErrTask;

        HostErrTask = pAdapter->Resources.hostErr_task;

        DBGPRINT_INTERRUPT((KERN_CRIT "tc90x_HostErrorEvent: IN \n"));
        //
        // Read the internal config.
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        pAdapter->keepForGlobalReset = NIC_READ_PORT_UINT(
                                                                pAdapter,
                                                                INTERNAL_CONFIG_REGISTER);

        DBGPRINT_INTERRUPT((KERN_CRIT "Adapter does global reset and restart \n"));
        //
        // Issue Global reset. I will mask the updown reset so that
        // I don't have to set the UpPoll, DownPoll, UpListPointer
        // and DownListPointer.
        //
        NIC_COMMAND_WAIT(
                pAdapter,
                COMMAND_GLOBAL_RESET |
                GLOBAL_RESET_MASK_NETWORK_RESET |
                GLOBAL_RESET_MASK_TP_AUI_RESET |
                GLOBAL_RESET_MASK_ENDEC_RESET |
                GLOBAL_RESET_MASK_AISM_RESET |
                GLOBAL_RESET_MASK_SMB_RESET |
                GLOBAL_RESET_MASK_VCO_RESET |
                GLOBAL_RESET_MASK_UP_DOWN_RESET
                );


        // run task in contex of a process so that we can sleep at start

#ifdef __VMKERNEL_MODULE__
        queue_task(&HostErrTask, &tq_scheduler);      
#else

#if LINUX_VERSION_CODE >= 0x20400
        schedule_task(&HostErrTask);
#else
	queue_task(&HostErrTask, &tq_scheduler);      
#endif

#endif // __VMKERNEL_MODULE__

        DBGPRINT_INTERRUPT((KERN_CRIT "tc90x_HostErrorEvent: OUT\n"));

        return;
}


/*++

Routine Name:

        tc90x_UpdateStatisticsEvent.

Routine Description:

        This routine handles the update statistics interrupt.

Arguments:

        Device - Pointer to the device structure.

Return Value:

    None

--*/

VOID
tc90x_UpdateStatisticsEvent(
        IN PNIC_INFORMATION Adapter
        )

{

        PNIC_INFORMATION pAdapter = Adapter;
        PNIC_STATISTICS statistics = &pAdapter->Statistics;
        USHORT rxPackets, txPackets, highPackets;
        USHORT rxBytes, txBytes, highBytes;

        DBGPRINT_INTERRUPT((KERN_CRIT "tc90x_UpdateStatisticsEvent: IN\n"));
        //
        // Change the window.
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_6);

        statistics->TxSQEErrors += NIC_READ_PORT_UCHAR(
                               pAdapter,
                               SQE_ERRORS_REGISTER);

        statistics->TxMultipleCollisions += NIC_READ_PORT_UCHAR(
                                        pAdapter,
                                        MULTIPLE_COLLISIONS_REGISTER);

        statistics->TxSingleCollisions += NIC_READ_PORT_UCHAR(
                                      pAdapter,
                                      SINGLE_COLLISIONS_REGISTER);

        statistics->RxOverruns += NIC_READ_PORT_UCHAR(
                              pAdapter,
                              RX_OVERRUNS_REGISTER);

        statistics->TxCarrierLost += NIC_READ_PORT_UCHAR(
                                 pAdapter,
                                 CARRIER_LOST_REGISTER);

        statistics->TxLateCollisions += NIC_READ_PORT_UCHAR(
                                    pAdapter,
                                    LATE_COLLISIONS_REGISTER);

        statistics->TxFramesDeferred += NIC_READ_PORT_UCHAR(
                                    pAdapter,
                                    FRAMES_DEFERRED_REGISTER);
        rxPackets = NIC_READ_PORT_UCHAR(
                        pAdapter,
                        FRAMES_RECEIVED_OK_REGISTER);

        txPackets = NIC_READ_PORT_UCHAR(
                        pAdapter,
                        FRAMES_TRANSMITTED_OK_REGISTER);

        highPackets = NIC_READ_PORT_UCHAR(
                                                pAdapter,
                        UPPER_FRAMES_OK_REGISTER);

        rxPackets += ((highPackets & 0x03) << 8);
        txPackets += ((highPackets & 0x30) << 4);

        if (pAdapter->Hardware.SQEDisable)
        statistics->TxSQEErrors += txPackets;

        statistics->RxFramesOk += rxPackets;
        statistics->TxFramesOk += txPackets;

        rxBytes = NIC_READ_PORT_USHORT(
                    pAdapter,
                    BYTES_RECEIVED_OK_REGISTER);

        txBytes = NIC_READ_PORT_USHORT(
                    pAdapter,
                    BYTES_TRANSMITTED_OK_REGISTER);

        NIC_COMMAND(
            pAdapter,
            COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        highBytes = NIC_READ_PORT_UCHAR(
                        pAdapter,
                        UPPER_BYTES_OK_REGISTER);

        rxBytes += ((highBytes & 0x0F) << 8);
        txBytes += ((highBytes & 0xF0) << 4);

        statistics->RxBytesOk += rxBytes;
        statistics->TxBytesOk += txBytes;

        statistics->RxBadSSD += NIC_READ_PORT_UCHAR(
                                    pAdapter,
                                    BAD_SSD_REGISTER);

        DBGPRINT_INTERRUPT((KERN_CRIT "tc90x_UpdateStatisticsEvent: OUT\n"));
}

/*++

Routine Name:

        tc90x_CountDownTimerEvent.

Routine Description:

        This routine handles the interrupt requested event.

Arguments:

        Device - Pointer to the device structure

Return Value:

        None

--*/

VOID
tc90x_CountDownTimerEvent(
        IN PNIC_INFORMATION Adapter
        )

{
        PNIC_INFORMATION pAdapter = Adapter;
        PDPD_LIST_ENTRY headDPDVirtual;
        PDEVICE device = pAdapter->Device;

        DBGPRINT_INTERRUPT((KERN_CRIT "tc90x_CountDownTimerEvent: IN\n"));
        headDPDVirtual = pAdapter->HeadDPDVirtual;
        //
        // This clears the FSH_DPD_EMPTY and a raise condition of
        // hardware.
        //
        pAdapter->TailDPDVirtual->FrameStartHeader = 0;
        while (1)
        {
                if (!(headDPDVirtual->FrameStartHeader & FSH_DOWN_COMPLETE))
                        break;
                ASSERT(headDPDVirtual->Packet != NULL);

#if LINUX_VERSION_CODE < 0x20343
                DEV_FREE_SKB(headDPDVirtual->SocketBuffer);
#else
                pci_unmap_single(pAdapter->pciDevice,
                                 headDPDVirtual->SGList[0].Address,
                                 headDPDVirtual->SocketBuffer->len,
                                 PCI_DMA_TODEVICE);
		VMWARE_STRESS_RETAIN_BUFFER(headDPDVirtual->SocketBuffer,
					    NO_CONTINUE_CMD,
                DEV_FREE_SKB_IRQ(headDPDVirtual->SocketBuffer));
#endif
                headDPDVirtual->SocketBuffer = NULL;

                /* Clear the down complete bit in the frame start header. */
                headDPDVirtual->FrameStartHeader = 0;
                pAdapter->BytesInDPDQueue -= headDPDVirtual->PacketLength;
                headDPDVirtual = headDPDVirtual->Next;
                ASSERT(pAdapter->HeadDPDVirtual != NULL);
        }
        pAdapter->HeadDPDVirtual = headDPDVirtual;
        if (pAdapter->BytesInDPDQueue)
                tc90x_SetCountDownTimer(pAdapter);

        /* If DPD ring is full, run the bottom half */
#if LINUX_VERSION_CODE < 0x20343
        if ((device->tbusy) && (pAdapter->DPDRingFull==TRUE))
        {       DBGPRINT_SEND(("CountdownTimer: set RingFull false,mark bh\n"));
                pAdapter->DPDRingFull = FALSE;
                clear_bit(0, (void*)&device->tbusy);
                mark_bh(NET_BH);
        }
#else
        if ((pAdapter->txing) && (pAdapter->DPDRingFull==TRUE))
        {       DBGPRINT_SEND(("CountdownTimer: set RingFull false,mark bh\n"));
                pAdapter->DPDRingFull = FALSE;
                clear_bit(0, (void*)&pAdapter->txing);
                netif_wake_queue(device);
        }
#endif
        DBGPRINT_INTERRUPT((KERN_CRIT "tc90x_CountDownTimerEvent: OUT\n"));
}




/*++

Routine Name:

        NICTimer

Routine Description:

        This is the tick handler.

Arguments:

Return Value:

    None.

--*/




VOID
NICTimer(
        IN ULONG Data
        )
{
        PDEVICE device =  (PDEVICE)Data;
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)device->priv;
        pAdapter->InTimer = TRUE;

        if ((pAdapter->Statistics.UpdateInterval +=
             pAdapter->Resources.TimerInterval) > 1000) {

                pAdapter->Statistics.UpdateInterval = 0;
                tc90x_UpdateStatisticsEvent(pAdapter);
        }
        //
        // Check every five seconds for media changed speed or duplex
        //
        if ((pAdapter->Hardware.UpdateInterval +=
                pAdapter->Resources.TimerInterval) > 5000) {

                pAdapter->Hardware.UpdateInterval = 0;
                tc90x_TickMediaHandler(pAdapter);
    }

        pAdapter->Resources.Timer.expires = RUN_AT(HZ/10);
        add_timer(&pAdapter->Resources.Timer);
        pAdapter->InTimer = FALSE;
}


VOID
WaitTimerHandler(
        IN ULONG Data
        )
{
        PDEVICE device =  (PDEVICE)Data;
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)device->priv;
        INT NextTick = 0;
        PDPD_LIST_ENTRY dpdVirtual;
#ifdef __VMKERNEL_MODULE__
        ULONG flags;
#endif // __VMKERNEL_MODULE__

        DBGPRINT_FUNCTION(("In WaitTimer-Time=%x\n",(int)jiffies));
#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_lock_irqsave(&pAdapter->SpinLock_misc, flags);
#else
        spin_lock(&pAdapter->SpinLock_misc);
#endif // __VMKERNEL_MODULE__

#endif
        switch (pAdapter->WaitCases)
        {
        case CHECK_DOWNLOAD_STATUS:
                DBGPRINT_INITIALIZE(("CHECK_DOWNLOAD_STATUS\n"));
                dpdVirtual = (PDPD_LIST_ENTRY)pAdapter->TestDPDVirtual[0];
                /* DBGPRINT_INIT(("portValue_g = %x\n", portValue_g)); */
                if(portValue_g == dpdVirtual->DPDPhysicalAddress)
                {       portValue_g = NIC_READ_PORT_UINT(pAdapter,DOWN_LIST_POINTER_REGISTER);
                        NextTick = HZ/10;
                }
                /* DBGPRINT_INIT(("portValue_g = %x\n", portValue_g)); */
                break;

        case CHECK_UPLOAD_STATUS:
                DBGPRINT_INITIALIZE(("CHECK_UPLOAD_STATUS\n"));
                if(portValue_g == pAdapter->HeadUPDVirtual->UPDPhysicalAddress)
                {       portValue_g = NIC_READ_PORT_UINT(pAdapter,UP_LIST_POINTER_REGISTER);
                        NextTick = HZ/10;
                }
                break;

        case CHECK_DC_CONVERTER:
                DBGPRINT_INITIALIZE(("CHECK_DC_CONVERTER\n"));
                if ( ((DCConverterEnabledState_g) && !(MediaStatus_g & MEDIA_STATUS_DC_CONVERTER_ENABLED) ) ||
                         ((!DCConverterEnabledState_g) && (MediaStatus_g & MEDIA_STATUS_DC_CONVERTER_ENABLED)) )
                {       MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);
                        NextTick = HZ/10;
                }
                break;

        case CHECK_PHY_STATUS:
                DBGPRINT_INITIALIZE(("CHECK_PHY_STATUS\n"));
                if(!(PhyStatus_g & MII_STATUS_AUTO_DONE) )
                {       PhyResponding_g = ReadMIIPhy(pAdapter, MII_PHY_STATUS, (PUSHORT)&PhyStatus_g);
                        NextTick = HZ/10;
                }
                break;

        case CHECK_TRANSMIT_IN_PROGRESS:
                DBGPRINT_INITIALIZE(("CHECK_TRANSMIT_IN_PROGRESS\n"));
                if(MediaStatus_g & MEDIA_STATUS_TX_IN_PROGRESS)
                {       MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);
                        NextTick = HZ/10;
                }
                break;

        case CHECK_DOWNLOAD_SELFDIRECTED:
                DBGPRINT_INITIALIZE(("CHECK_DOWNLOAD_SELFDIRECTED\n"));
                dpdVirtual = (PDPD_LIST_ENTRY)pAdapter->TestDPDVirtual[0];
                if(DownListPointer_g == dpdVirtual->DPDPhysicalAddress)
                {       DownListPointer_g = NIC_READ_PORT_UINT(pAdapter,DOWN_LIST_POINTER_REGISTER);
                        NextTick = HZ/10;
                }
                break;

        case AUTONEG_TEST_PACKET:
                DBGPRINT_INITIALIZE(("AUTONEG_TEST_PACKET\n"));
                if(UpListPointer_g == pAdapter->HeadUPDVirtual->UPDPhysicalAddress)
                {       UpListPointer_g = NIC_READ_PORT_UINT(pAdapter,UP_LIST_POINTER_REGISTER);
                        NextTick = HZ/10;
                }
                break;

        case CHECK_DMA_CONTROL:
                DBGPRINT_INITIALIZE(("CHECK_DMA_CONTROL\n"));
                if(dmaControl_g & DMA_CONTROL_DOWN_IN_PROGRESS)
                {       dmaControl_g = NIC_READ_PORT_UINT(pAdapter,DMA_CONTROL_REGISTER);
                        NextTick = HZ/10;
                }
                break;

        case CHECK_CARRIER_SENSE:
                DBGPRINT_INITIALIZE(("CHECK_CARRIER_SENSE\n"));
                if(MediaStatus_g & MEDIA_STATUS_CARRIER_SENSE)
                {       MediaStatus_g = NIC_READ_PORT_USHORT(pAdapter,MEDIA_STATUS_REGISTER);
                        NextTick = HZ/10;
                }
                break;

        default:
                break;
        }
        if(NextTick)
        {       WaitTimer.expires = RUN_AT(NextTick);
                add_timer(&WaitTimer);
        }
        else
                InWaitTimer = FALSE;
#if LINUX_VERSION_CODE >= 0x20200
#ifdef __VMKERNEL_MODULE__
        spin_unlock_irqrestore(&pAdapter->SpinLock_misc, flags);
#else
        spin_unlock(&pAdapter->SpinLock_misc);
#endif // __VMKERNEL_MODULE__

#endif
        DBGPRINT_FUNCTION(("Out WaitTimer\n"));
}





/*++

Routine Name:

        FindMIIPhy

Routine Description:

        Search for any PHY that is not known.

Arguments:

        Adapter - Pointer to the adapter structure.

Return Value:

        BOOLEAN

--*/

BOOLEAN
FindMIIPhy(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Adapter;
        USHORT MediaOptions = 0;
        USHORT PhyManagement = 0;
        UCHAR index;
        //
        // Read the MEDIA OPTIONS to see what connectors are available
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_3);

        MediaOptions = NIC_READ_PORT_USHORT(pAdapter,MEDIA_OPTIONS_REGISTER);

        if ((MediaOptions & MEDIA_OPTIONS_MII_AVAILABLE) ||
            (MediaOptions & MEDIA_OPTIONS_100BASET4_AVAILABLE) ) {
                //
        // Drop everything, so we are not driving the data, and run the
        // clock through 32 cycles in case the PHY is trying to tell us
        // something. Then read the data line, since the PHY's pull-up
        // will read as a 1 if it's present.
        //
                NIC_COMMAND(
                        pAdapter,
                        COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

                NIC_WRITE_PORT_USHORT(pAdapter,PHYSICAL_MANAGEMENT_REGISTER,0);

                for (index = 0; index < 32; index++) {

                        NIC_DELAY(1);

                        NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                PHY_CLOCK);

                        NIC_DELAY(1);

                        NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                0);
                }

                PhyManagement = NIC_READ_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER);

                if ( PhyManagement & PHY_DATA1) {

                        return TRUE;
                }
                else {
                        return FALSE;
                }
    }
        return TRUE;
}



/*++

Routine Name:

        SendMIIPhyPreamble

Routine Description:

        Establishes the synchronization for each MII transaction. This
        is done by sending thirty-two "1" bits.

Arguments:

        Adapter - Pointer to the adapter structure.

Return Value:

        VOID

--*/

VOID
SendMIIPhyPreamble(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Adapter;
    UCHAR index;
    //
    // Set up and send the preamble, a sequence of 32 "1" bits
        //
        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                PHYSICAL_MANAGEMENT_REGISTER,
                PHY_WRITE);

        for (index = 0; index < 32; index++) {

                NIC_WRITE_PORT_USHORT(
                        pAdapter,
                        PHYSICAL_MANAGEMENT_REGISTER,
                        PHY_WRITE | PHY_DATA1
                        );

                NIC_WRITE_PORT_USHORT(
                        pAdapter,
                        PHYSICAL_MANAGEMENT_REGISTER,
                        PHY_WRITE | PHY_DATA1 | PHY_CLOCK);

                NIC_DELAY(1);

                NIC_WRITE_PORT_USHORT(
                        pAdapter,
                        PHYSICAL_MANAGEMENT_REGISTER,
                        PHY_WRITE);

                NIC_DELAY(1);
        }
}


/*++

Routine Name:

        WriteMIIPhy

Routine Description:

  Writes to a particular MII PHY register given the proper offset.

Arguments:

        IN PNIC_INFORMATION Adapter
        IN USHORT RegAddr
        IN USHORT Output

Return Value:

        VOID

--*/

VOID
WriteMIIPhy(
        IN PNIC_INFORMATION Adapter,
        IN USHORT RegAddr,
        IN USHORT Output
        )
{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Adapter;
        UINT index,index2;
    USHORT writecmd[2];

        writecmd[0] = pAdapter->Hardware.MIIWriteCommand;
        writecmd[1] = 0;

        SendMIIPhyPreamble(pAdapter);
    //
    // Bits 2..6 of the command word specify the register.
    //
    writecmd[0] |= (RegAddr & 0x1F) << 2;
    writecmd[1] = Output;

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

        for (index2 = 0; index2 < 2; index2++) {

                for (index = 0x8000; index; index >>= 1) {

                        if (writecmd[index2] & index) {

                                NIC_WRITE_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER,
                                        PHY_WRITE | PHY_DATA1);

                                NIC_WRITE_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER,
                                        PHY_WRITE | PHY_DATA1 | PHY_CLOCK);
                        }
                else {

                                NIC_WRITE_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER,
                                        PHY_WRITE);

                                NIC_WRITE_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER,
                                        PHY_WRITE | PHY_CLOCK);
                        }
                        NIC_DELAY(1);

                        NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                PHY_WRITE);

                        NIC_DELAY(1);
                }
        }
        //
        // OK now give it a couple of clocks with nobody driving.
        //
        NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                0);

        for (index = 0; index < 2; index++) {

                        NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                PHY_CLOCK);

                        NIC_DELAY(1);

                        NIC_WRITE_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER,
                                        0);

                        NIC_DELAY(1);
        }
}


/*++

Routine Name:

        ReadMIIPhy

Routine Description:

        Reads a particular MII PHY register given the proper offset.

Arguments:

        IN PNIC_INFORMATION Adapter
        IN USHORT RegisterAddress
        OUT PUSHORT pInput

Return Value:
        BOOLEAN

--*/

BOOLEAN
ReadMIIPhy(
        IN PNIC_INFORMATION Adapter,
        IN USHORT RegisterAddress,
        OUT PUSHORT pInput
        )
{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Adapter;
        USHORT PhyManagement = 0;
        USHORT ReadCommand;
        UINT index;

        ReadCommand = pAdapter->Hardware.MIIReadCommand;

        SendMIIPhyPreamble(pAdapter);
        //
        // Bits 2..6 of the command word specify the register.
        //
        ReadCommand |= (RegisterAddress & 0x1F) << 2;

        for (index = 0x8000; index > 2; index >>= 1) {

                NIC_COMMAND(
                        pAdapter,
                        COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

                if (ReadCommand & index) {

                        NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                PHY_WRITE | PHY_DATA1);

                        NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                PHY_WRITE | PHY_DATA1 | PHY_CLOCK);
                }
                else {
                        NIC_WRITE_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER,
                                        PHY_WRITE);

                                NIC_WRITE_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER,
                                        PHY_WRITE | PHY_CLOCK);
                }
                NIC_DELAY(1);

                NIC_WRITE_PORT_USHORT(
                        pAdapter,
                        PHYSICAL_MANAGEMENT_REGISTER,
                        PHY_WRITE);

                NIC_DELAY(1);
        }
        //
    // Now run one clock with nobody driving.
    //
        NIC_WRITE_PORT_USHORT(
                pAdapter,
                PHYSICAL_MANAGEMENT_REGISTER,
                0);

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                PHYSICAL_MANAGEMENT_REGISTER,
                PHY_CLOCK);

        NIC_DELAY(1);

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                PHYSICAL_MANAGEMENT_REGISTER,
                0);

        NIC_DELAY(1);
        //
    // Now run one clock, expecting the PHY to be driving a 0 on the data
    // line.  If we read a 1, it has to be just his pull-up, and he's not
    // responding.
    //
        PhyManagement = NIC_READ_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER);

        if (PhyManagement & PHY_DATA1) {
                        return FALSE;
        }
        //
        // We think we are in sync.  Now we read 16 bits of data from the PHY.
        //
        for (index = 0x8000; index; index >>= 1) {
                //
                // Shift input up one to make room
                //
                NIC_COMMAND(
                        pAdapter,
                        COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

                NIC_WRITE_PORT_USHORT(
                        pAdapter,
                        PHYSICAL_MANAGEMENT_REGISTER,
                        PHY_CLOCK);

                NIC_DELAY(1);

                NIC_WRITE_PORT_USHORT(
                        pAdapter,
                        PHYSICAL_MANAGEMENT_REGISTER,
                        0);

                NIC_DELAY(1);

                PhyManagement = NIC_READ_PORT_USHORT(
                                        pAdapter,
                                        PHYSICAL_MANAGEMENT_REGISTER);

                if (PhyManagement & PHY_DATA1)
                        *pInput |= index;
                else
                        *pInput &= ~index;
        }
    //
    // OK now give it a couple of clocks with nobody driving.
    //
        NIC_WRITE_PORT_USHORT(
                pAdapter,
                PHYSICAL_MANAGEMENT_REGISTER,
                0);

    for (index = 0; index < 2; index++) {

                NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                PHY_CLOCK);

                NIC_DELAY(1);

                NIC_WRITE_PORT_USHORT(
                                pAdapter,
                                PHYSICAL_MANAGEMENT_REGISTER,
                                0);

                NIC_DELAY(1);
        }
        return TRUE;
}


/*++

Routine Name:

        MIIMediaOverride

Routine Description:

        MII values need to be updated based on what was set in the
        registry.

Arguments:

        IN PNIC_INFORMATION Adapter
        IN USHORT PhyModes
        OUT PUSHORT MiiType

Return Value:
        BOOLEAN

--*/

BOOLEAN
MIIMediaOverride(
        IN PNIC_INFORMATION Adapter,
        IN USHORT PhyModes,
        OUT PUSHORT MiiType
        )
{
        PNIC_INFORMATION pAdapter = Adapter;

        switch(pAdapter->Hardware.MediaOverride) {

                case MEDIA_10BASE_T:

                        if (( PhyModes & MII_STATUS_10TFD ) &&
                                ( pAdapter->Hardware.FullDuplexEnable == TRUE ))
                                *MiiType = MIISELECT_10BT;

                        else if ( PhyModes & MII_STATUS_10T )
                                *MiiType = MIISELECT_10BT;

                        else
                                return FALSE;

                        break;

                case MEDIA_100BASE_TX:

                        if (( PhyModes & MII_STATUS_100TXFD ) &&
                                ( pAdapter->Hardware.FullDuplexEnable == TRUE ))
                                *MiiType = MIISELECT_100BTX;

                        else if ( PhyModes & MII_STATUS_100TX )
                                *MiiType = MIISELECT_100BTX;

                        else
                                return FALSE;

                        break;
        }

        return TRUE;
}


/*++

Routine Name:

        ProgramMII

Routine Description:

        Setup the necessary MII registers with values either
        read from the EEPROM or from command line

Arguments:

        IN PNIC_INFORMATION Adapter
        IN CONNECTOR_TYPE NewConnector

Return Value:
        BOOLEAN

--*/

BOOLEAN
ProgramMII(
        IN PNIC_INFORMATION Adapter,
        IN CONNECTOR_TYPE NewConnector
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN PhyResponding;
        USHORT PhyControl;
        USHORT PhyStatus;
        NIC_STATUS status;
        USHORT MiiType=0, PhyModes;
        //
        // First see if there's anything connected to the MII
        //
        if (!FindMIIPhy(pAdapter))
                return FALSE;
        // Nowhere is it written that the register must be latched, and since
        // reset is the last bit out, the contents might not be valid.  read
        // it one more time.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_CONTROL, &PhyControl);
        //
        // Now we can read the status and try to figure out what's out there.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_STATUS, &PhyStatus);
        if (!PhyResponding)
                return FALSE;
        //
        // Reads the miiSelect field in EEPROM. Program MII as the default.
        //
        status = tc90x_ReadEEPROM(
                                pAdapter,
                                EEPROM_SOFTWARE_INFORMATION_3,
                                &MiiType);
        //
        // If an override is present AND the transceiver type is available
        // on the card, that type will be used.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_STATUS, &PhyModes);
        if (!PhyResponding)
                return FALSE;

        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_CONTROL, &PhyControl);
        if (!PhyResponding)
                return FALSE;

        if (!MIIMediaOverride(pAdapter, PhyModes, &MiiType))
                return FALSE;
        //
        // If full duplex selected, set it in PhyControl.
        //
        if (pAdapter->Hardware.FullDuplexEnable)
                PhyControl |= MII_CONTROL_FULL_DUPLEX;
        else
                PhyControl &= ~MII_CONTROL_FULL_DUPLEX;

        PhyControl &= ~MII_CONTROL_ENABLE_AUTO;

        if (((MiiType & MIITXTYPE_MASK) == MIISELECT_100BTX) ||
            ((MiiType & MIITXTYPE_MASK) == MIISELECT_100BTX_ANE)) {
                PhyControl |= MII_CONTROL_100MB;
                WriteMIIPhy(pAdapter, MII_PHY_CONTROL, PhyControl);
                //delay 600 milliseconds
                TimeOutCount = jiffies + 6*HZ/10;
                while(TimeOutCount > jiffies) ;
                pAdapter->Hardware.LinkSpeed = 100000000L;
                DBGPRINT_INITIALIZE(("ProgramMII() Set to 100M\n"));
                return TRUE;
        }
        else if (((MiiType & MIITXTYPE_MASK ) == MIISELECT_10BT) ||
                 ((MiiType & MIITXTYPE_MASK ) == MIISELECT_10BT_ANE)) {
                PhyControl &= ~MII_CONTROL_100MB;
                WriteMIIPhy(pAdapter, MII_PHY_CONTROL, PhyControl);
                //delay 600 milliseconds
                TimeOutCount = jiffies + 6*HZ/10;
                while(TimeOutCount > jiffies) ;
                pAdapter->Hardware.LinkSpeed = 10000000L;
                DBGPRINT_INITIALIZE(("ProgramMII() Set to 10M\n"));
                return TRUE;
        }

        PhyControl &= ~MII_CONTROL_100MB;
        WriteMIIPhy(pAdapter, MII_PHY_CONTROL, PhyControl);
                //delay 600 milliseconds
                TimeOutCount = jiffies + 6*HZ/10;
                while(TimeOutCount > jiffies) ;

        pAdapter->Hardware.LinkSpeed = 10000000L;
        DBGPRINT_INITIALIZE(("ProgramMII() Defaults to 10M\n"));
        return FALSE;
}

/*++

Routine Name:

        CheckMIIConfiguration

Routine Description:

    Sets up the tranceiver through MII registers. This will first check on the current connection
        state as shown by the MII registers. If the current state matches what the media options support,
        then the link is kept. If not, the registers will be configured in the proper manner and auto-negotiation
        will be restarted.

        Since this function is called for forced xcvr configurations, it assumes that the xcvr type has
        been verified as supported by the NIC.

Arguments:

    IN pAdapter pAdapter - Pointer to the adapter structure
    IN USHORT MediaOptions - Value of MediaOptions register passed by caller.

Return Value:
        BOOLEAN

--*/

BOOLEAN
CheckMIIConfiguration(
        IN PNIC_INFORMATION Adapter,
    IN USHORT MediaOptions
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN PhyResponding;
        USHORT PhyControl;
        USHORT PhyStatus;
        USHORT PhyAnar;
        USHORT tempAnar;
        //
        // Check to see if auto-negotiation has completed. Check the results
        // in the control and status registers.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_CONTROL, &PhyControl);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("CheckMIIConfiguration: Phy not responding (1) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return FALSE;
        }
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_STATUS, &PhyStatus);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("CheckMIIConfiguration: Phy not responding (2) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return FALSE;
        }
        if (!((PhyControl & MII_CONTROL_ENABLE_AUTO) && (PhyStatus & MII_STATUS_AUTO_DONE))) {
                //
                // Auto-negotiation did not complete, so start it over using the new settings.
                //
                if (!ConfigureMII(pAdapter,MediaOptions))
                        return FALSE;
        }
        //
        // Auto-negotiation has completed. Check the results against the ANAR and ANLPAR
        // registers to see if we need to restart auto-neg.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANAR, &PhyAnar);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("CheckMIIConfiguration: Phy not responding (3) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return FALSE;
        }
        //
        // Check to see what we negotiated with the link partner. First, let's make
        // sure that the ANAR is set properly based on the media options defined.
        //
        tempAnar = 0;
        if (MediaOptions & MEDIA_OPTIONS_100BASETX_AVAILABLE) {
                if (pAdapter->Hardware.AutoSelect) {
                        tempAnar |= MII_ANAR_100TXFD | MII_ANAR_100TX;
                }
                else {
                        if (pAdapter->Hardware.FullDuplexEnable)
                                tempAnar |= MII_ANAR_100TXFD;
                        else
                                tempAnar |= MII_ANAR_100TX;
                }
        }
        if (MediaOptions & MEDIA_OPTIONS_10BASET_AVAILABLE) {
                if (pAdapter->Hardware.AutoSelect)
                        tempAnar |= MII_ANAR_10TFD | MII_ANAR_10T;
                else {
                        if (pAdapter->Hardware.FullDuplexEnable)
                                tempAnar |= MII_ANAR_10TFD;
                        else
                                tempAnar |= MII_ANAR_10T;
                }
        }
        if ( pAdapter->Hardware.FullDuplexEnable &&
                pAdapter->Hardware.FlowControlSupported )
                tempAnar |= MII_ANAR_FLOWCONTROL;

        if ((PhyAnar & MII_ANAR_MEDIA_MASK) == tempAnar) {
                //
                // The negotiated configuration hasn't changed.
                // So, return and don't restart auto-negotiation.
                //
                return TRUE;
        }
        //
        // Check the media settings.
        //
        if (MediaOptions & MEDIA_OPTIONS_100BASETX_AVAILABLE) {
                //
                // Check 100BaseTX settings.
                //
                if ((PhyAnar & MII_ANAR_MEDIA_100_MASK) != (tempAnar & MII_ANAR_MEDIA_100_MASK)) {
                DBGPRINT_INITIALIZE(("CheckMIIConfiguration: Re-Initiating autonegotiation...\n"));
                        return ConfigureMII(pAdapter,MediaOptions);
                }
        }
        if (MediaOptions & MEDIA_OPTIONS_10BASET_AVAILABLE) {
                //
                // Check 10BaseT settings.
                //
                if ((PhyAnar & MII_ANAR_MEDIA_10_MASK) != (tempAnar & MII_ANAR_MEDIA_10_MASK)) {
                DBGPRINT_INITIALIZE(("CheckMIIConfiguration: Re-Initiating autonegotiation...\n"));
                        return ConfigureMII(pAdapter,MediaOptions);
                }
        }
        return TRUE;
}

/*++

Routine Name: ConfigureMII

Routine Description:

    Used to setup the configuration for 10Base-T and 100Base-TX.

Arguments:

    IN pAdapter - Pointer to the adapter structure
        IN MediaOptions - Media options that will define how to set up MII registers.

Return Value:
        BOOLEAN

--*/

BOOLEAN
ConfigureMII(
        IN PNIC_INFORMATION Adapter,
        IN USHORT MediaOptions
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN PhyResponding;
        USHORT PhyControl;
        USHORT PhyAnar;
        ULONG TimeOutCount;

        DBGPRINT_INITIALIZE(("ConfigureMII: IN \n"));
        //
        // Nowhere is it written that the register must be latched, and since
        // reset is the last bit out, the contents might not be valid.  read
        // it one more time.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_CONTROL, &PhyControl);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("ConfigureMII: Phy not responding (1) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return FALSE;
        }
        //
        // Also, read the ANAR register and clear out it's current settings.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANAR, &PhyAnar);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("ConfigureMII: Phy not responding (2) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return FALSE;
        }
        //
        // Set up speed and duplex settings in MII Control and ANAR register.
        //
        PhyAnar &= ~(
                        MII_ANAR_100TXFD|
                        MII_ANAR_100TX  |
                        MII_ANAR_10TFD  |
                        MII_ANAR_10T);
        //
        // Set up duplex.
        //
        if (pAdapter->Hardware.FullDuplexEnable)
                PhyControl |= MII_CONTROL_FULL_DUPLEX;
        else
                PhyControl &= ~(MII_CONTROL_FULL_DUPLEX);
        //
        // Set up flow control.
        // NOTE: On some NICs, such as Tornado, this will be hardwired to be set.
        // Clearing it will have no effect.
        //
        if (pAdapter->Hardware.FlowControlSupported)
                PhyAnar |= MII_ANAR_FLOWCONTROL;
        else
                PhyAnar &= ~(MII_ANAR_FLOWCONTROL);
        //
        // Set up the media options. For duplex settings, if we're set to auto-select
        // then enable both half and full-duplex settings. Otherwise, go by what's
        // been enabled for duplex mode.
        //
        if (MediaOptions & MEDIA_OPTIONS_100BASETX_AVAILABLE){
                if (pAdapter->Hardware.AutoSelect)
                        PhyAnar |= (MII_ANAR_100TXFD | MII_ANAR_100TX);
                else {
                        if (pAdapter->Hardware.FullDuplexEnable)
                                PhyAnar |= MII_ANAR_100TXFD;
                        else
                                PhyAnar |= MII_ANAR_100TX;
                }
        }
        if (MediaOptions & MEDIA_OPTIONS_10BASET_AVAILABLE){
                if (pAdapter->Hardware.AutoSelect)
                        PhyAnar |= (MII_ANAR_10TFD | MII_ANAR_10T);
                else {
                        if (pAdapter->Hardware.FullDuplexEnable)
                                PhyAnar |= MII_ANAR_10TFD;
                        else
                                PhyAnar |= MII_ANAR_10T;
                }
        }
        //
        // Enable and start auto-negotiation
        //
        PhyControl |= (MII_CONTROL_ENABLE_AUTO | MII_CONTROL_START_AUTO);
        //
        // Write the MII registers back.
        //
        WriteMIIPhy(pAdapter, MII_PHY_ANAR, PhyAnar);
        WriteMIIPhy(pAdapter, MII_PHY_CONTROL, PhyControl);
        //
        // Wait for auto-negotiation to finish.
        //
        PhyStatus_g = 0;
        PhyResponding_g = ReadMIIPhy(pAdapter,
                                        MII_PHY_STATUS,
                                        (PUSHORT)&PhyStatus_g);
        NIC_DELAY(1000);
        if (!PhyResponding_g) {
                        DBGPRINT_ERROR(("ConfigureMII: Phy not responding (3) \n"));
                        pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                        return FALSE;
        }
        if (!(PhyStatus_g & MII_STATUS_AUTO_DONE) )
        {       
#ifdef __VMKERNEL_MODULE__
                TimeOutCount = jiffies + (HZ/50); // run max = 20ms
#else
                TimeOutCount = jiffies + (3*HZ); // run max = 3s
#endif
                pAdapter->WaitCases = CHECK_PHY_STATUS;
                if(!InWaitTimer)
                {       WaitTimer.expires = RUN_AT(HZ/10);
                        add_timer(&WaitTimer);
                        InWaitTimer = TRUE;
                }
                while(TimeOutCount > jiffies) ;
                pAdapter->WaitCases = NONE;
                if(!(PhyStatus_g & MII_STATUS_AUTO_DONE))
                {       DBGPRINT_ERROR(("ConfigureMII: Autonegotiation not done \n"));
                        pAdapter->Hardware.LinkState = LINK_DOWN_AT_INIT;
                        return FALSE;
                }
        }
        pAdapter->Hardware.LinkState = LINK_UP;
        return TRUE;
}

/*++

Routine Name:

        CheckMIIAutoNegotiationStatus

Routine Description:

        Since this function is called for forced xcvr configurations, it assumes that the xcvr type has
        been verified as supported by the NIC.

Arguments:

    IN pAdapter pAdapter - Pointer to the adapter structure

Return Value:
        None

--*/

VOID
CheckMIIAutoNegotiationStatus(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN PhyResponding;
        USHORT PhyStatus;
        USHORT PhyAnar;
        USHORT PhyAnlpar;
        //
        // Check to see if auto-negotiation has completed. Check the results in the status
        // registers.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_STATUS, &PhyStatus);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("CheckMIIAutoNegotiationStatus: Phy not responding (1) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return;
        }
        if (PhyStatus & MII_STATUS_LINK_UP)
                return; // We have a valid link, so get out!
        //
        // Check to see why auto-negotiation or parallel detection has failed. We'll do this
        // by comparing the advertisement registers between the NIC and the link partner.
        //
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANAR, &PhyAnar);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("CheckMIIAutoNegotiationStatus: Phy not responding (2) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return;
        }
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANLPAR, &PhyAnlpar);
        if (!PhyResponding) {
                DBGPRINT_ERROR(("CheckMIIAutoNegotiationStatus: Phy not responding (3) \n"));
                pAdapter->Hardware.Status = HARDWARE_STATUS_FAILURE;
                return;
        }
        //
        // Now, compare what was advertised between the NIC and it's link partner.
        // If the media bits don't match, then write an error log entry.
        //
        if ((PhyAnar & MII_ANAR_MEDIA_MASK) != (PhyAnlpar & MII_ANAR_MEDIA_MASK))
                DBGPRINT_ERROR(("CheckMIIAutoNegotiationStatus: Incompatible configuration \n"));
}



/*++

Routine Name :

        tc90x_CheckIfEEPROMBusy

Routine Description:

        This routine checks if the EEPROM is busy

Arguments:

        PDEVICE - Pointer to the device structure

Return Value:

        NIC_STATUS_SUCCESS
        NIC_STATUS_FAILURE

--*/

NIC_STATUS
tc90x_CheckIfEEPROMBusy(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        USHORT command = 0;
        ULONG count;

        count = jiffies + HZ;
        do {
                command = NIC_READ_PORT_USHORT(
                                pAdapter,
                                EEPROM_COMMAND_REGISTER);
                NIC_DELAY(10);
        } while ( (command & EEPROM_BUSY_BIT) &&  (count > jiffies) );

        if (count < jiffies) {
                DBGPRINT_ERROR(("tc90x_CheckIfEEPROMBusy: command timeout"));
                return NIC_STATUS_FAILURE;
        }
        return NIC_STATUS_SUCCESS;
}

/*++

Routine Name:

        tc90x_ReadEEPROM

Routine Description:

        This routine reads from the EEPROM

Arguments:

        PDEVICE - Pointer to the device structure
        EEPROMAddress - EEPROM register to read
        Contents - Buffer where contents of the location are returned

Return Value:

        NIC_STATUS_SUCCESS
        NIC_STATUS_FAILURE

--*/

NIC_STATUS
tc90x_ReadEEPROM(
        IN PNIC_INFORMATION Adapter,
        IN USHORT EEPROMAddress,
        OUT PUSHORT Contents
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        USHORT lowerOffset = 0;
        USHORT upperOffset = 0;

        if (EEPROMAddress > 0x003F) {

                lowerOffset = EEPROMAddress & 0x003F;
                upperOffset = (EEPROMAddress & 0x03C0) << 2;
                EEPROMAddress = upperOffset | lowerOffset;
        }

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_0);
        //
        // Check if EEPROM is busy
        //
        if (tc90x_CheckIfEEPROMBusy(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR(("tc90x_ReadEEPROM: EEPROM is busy\n"));
                return NIC_STATUS_FAILURE;

        }
        //
        // Issue the read eeprom data command
        //
        NIC_WRITE_PORT_USHORT(
                pAdapter,
                EEPROM_COMMAND_REGISTER,
                (USHORT)(EEPROM_COMMAND_READ + (USHORT)EEPROMAddress));
        //
        // Check if EEPROM is busy
        //
        if (tc90x_CheckIfEEPROMBusy(pAdapter) != NIC_STATUS_SUCCESS) {

                DBGPRINT_ERROR((
                        "tc90x_ReadEEPROM: EEPROM is busy after command.\n"));

                return NIC_STATUS_FAILURE;
        }
        //
        // Save value read from eeprom
        //
        *Contents = NIC_READ_PORT_USHORT(
                        pAdapter,
                        EEPROM_DATA_REGISTER);

        return NIC_STATUS_SUCCESS;
}


/*++

Routine Name :

        WriteEEPROM

Routine Description:

        This routine writes to the EEPROM

Arguments:

        PDEVICE - Pointer to the device structure
        EEPROMAddress - EEPROM register to write
        Data - Data to write

Return Value:

        NIC_STATUS_SUCCESS
        NIC_STATUS_FAILURE

--*/

NIC_STATUS
tc90x_WriteEEPROM(
        IN PNIC_INFORMATION Adapter,
        IN USHORT EEPROMAddress,
        IN USHORT Data
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        USHORT lowerOffset = 0;
        USHORT upperOffset = 0;
        USHORT saveAddress;

        saveAddress = EEPROMAddress;

        if (EEPROMAddress > 0x003F) {

                lowerOffset = EEPROMAddress & 0x003F;
                upperOffset = (EEPROMAddress & 0x03C0) << 2;
                EEPROMAddress = upperOffset | lowerOffset;
        }

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_0);

        //
        // Issue erase register command prior to writing
        //
        NIC_WRITE_PORT_USHORT(
                pAdapter,
                EEPROM_COMMAND_REGISTER,
                (USHORT)EEPROM_WRITE_ENABLE);

        if (tc90x_CheckIfEEPROMBusy(pAdapter) != NIC_STATUS_SUCCESS){

                DBGPRINT_ERROR((
                        "WriteEEPROM: Write enable, EEPROM is busy\n"));

                return NIC_STATUS_FAILURE;
        }

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                EEPROM_COMMAND_REGISTER,
                (USHORT)(EEPROM_ERASE_REGISTER + (USHORT)EEPROMAddress));

        if (tc90x_CheckIfEEPROMBusy(pAdapter) != NIC_STATUS_SUCCESS){

                DBGPRINT_ERROR((
                        "WriteEEPROM: Erase Register, EEPROM is busy\n"));

                return NIC_STATUS_FAILURE;
        }
        //
        // Load data to be written to the eeprom
        //
        NIC_WRITE_PORT_USHORT(pAdapter, EEPROM_DATA_REGISTER, Data);

//      DBGPRINT_INITIALIZE((
//              "WriteEeprom: Writing value %x at %x \n" ,Data, saveAddress));

        if (tc90x_CheckIfEEPROMBusy(pAdapter) != NIC_STATUS_SUCCESS){

                DBGPRINT_ERROR((
                        "WriteEEPROM: Write data, EEPROM is busy\n"
                        ));
                return NIC_STATUS_FAILURE;
        }


        //
        // Issue the write eeprom data command
        //
        NIC_WRITE_PORT_USHORT(
                pAdapter,
                EEPROM_COMMAND_REGISTER,
                (USHORT)EEPROM_WRITE_ENABLE);

        if (tc90x_CheckIfEEPROMBusy(pAdapter) != NIC_STATUS_SUCCESS){

                DBGPRINT_ERROR(("WriteEEPROM: EEPROM is busy\n"));
                return NIC_STATUS_FAILURE;
        }

        NIC_WRITE_PORT_USHORT(
                pAdapter,
                EEPROM_COMMAND_REGISTER,
                (USHORT)(EEPROM_WRITE_REGISTER + (USHORT)EEPROMAddress));

        if (tc90x_CheckIfEEPROMBusy(pAdapter) != NIC_STATUS_SUCCESS){

                DBGPRINT_ERROR((
                        "WriteEEPROM: Write register, EEPROM is busy\n"));
                return NIC_STATUS_FAILURE;
        }
        return NIC_STATUS_SUCCESS;
}



/*++

Routine Name :

        CalculateEEPROMChecksum1

Routine Description:

        Calculates the EEPROM checksum #1 from offset 0 to 0x1F.

Arguments:

        PDEVICE - Pointer to the device structure

Return Value:

        Checksum #1 value

--*/

USHORT
tc90x_CalculateEEPROMChecksum1(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        NIC_STATUS status;
        USHORT checksum=0;
        USHORT value=0;
        UCHAR index;

        DBGPRINT_FUNCTION(("CalculateChecksum1: IN \n"));

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_0);

        for (index = EEPROM_NODE_ADDRESS_WORD_0;
             index < EEPROM_CHECKSUM_1;
             index++) {

                status = tc90x_ReadEEPROM(pAdapter, index, &value);

                if (status == NIC_STATUS_FAILURE) {

                        DBGPRINT_ERROR((
                                "CalculateEEPROMChecksum1: Read failure\n"));

                        return NIC_STATUS_FAILURE;
                }

                checksum ^= (USHORT)LOBYTE(value);
                checksum ^= (USHORT)HIBYTE(value);
        }

        DBGPRINT_FUNCTION(("CalculateChecksum1: OUT \n"));

        return((USHORT)checksum);
}





NIC_STATUS
tc90x_SoftwareWork(
        IN PNIC_INFORMATION Adapter
        )

{
        PNIC_INFORMATION pAdapter = Adapter;
        USHORT SoftwareInformation2 = 0;
        UINT DmaControl = 0;
        USHORT NetDiag = 0;
        USHORT Contents = 0;

        DBGPRINT_FUNCTION(("SoftwareWork: IN\n"));
        // Additional work#1

        tc90x_ReadEEPROM(
                pAdapter,
                EEPROM_SOFTWARE_INFORMATION_2,
                &Contents);

        if (!(Contents & ENABLE_MWI_WORK)) {

                DmaControl = NIC_READ_PORT_UINT(
                                pAdapter,
                                DMA_CONTROL_REGISTER);

                NIC_WRITE_PORT_UINT(
                        pAdapter,
                        DMA_CONTROL_REGISTER,
                        (ULONG)(DMA_CONTROL_DEFEAT_MWI | DmaControl));
        }

        // Additional work#2

        NIC_COMMAND(
                pAdapter,
                COMMAND_SELECT_REGISTER_WINDOW | REGISTER_WINDOW_4);

    NetDiag = NIC_READ_PORT_USHORT(pAdapter, NETWORK_DIAGNOSTICS_REGISTER);

        if ((((NetDiag & NETWORK_DIAGNOSTICS_ASIC_REVISION) >> 4) == 1) &&
            (((NetDiag & NETWORK_DIAGNOSTICS_ASIC_REVISION_LOW) >> 1) < 4)) {
                pAdapter->Hardware.HurricaneEarlyRevision = TRUE;
                DBGPRINT_INITIALIZE(("Hurricane Early board\n"));
        tc90x_HurricaneEarlyRevision(pAdapter);
    }

        SoftwareInformation2 = 0;
        ((PSOFTWARE_INFORMATION_2)&SoftwareInformation2)->D3Work = 1;

        if (Contents & SoftwareInformation2) {
                DBGPRINT_INITIALIZE(("Enable D3 work \n"));
                pAdapter->Hardware.D3Work = TRUE;
        }


        // Additional work#3

        pAdapter->Hardware.DontSleep = FALSE;

        SoftwareInformation2 = 0;
        ((PSOFTWARE_INFORMATION_2)
        &SoftwareInformation2)->WOLConnectorPresent = 1;

        ((PSOFTWARE_INFORMATION_2)&SoftwareInformation2)->AutoResetToD0 = 1;

        DBGPRINT_INITIALIZE(("SoftwareInfo2 : %x\n", SoftwareInformation2));

        if (!(Contents & SoftwareInformation2)) {
                DBGPRINT_INITIALIZE(("Don't sleep is TRUE \n"));
                pAdapter->Hardware.DontSleep = TRUE;
        }

        DBGPRINT_FUNCTION(("SoftwareWork: OUT\n"));
        return NIC_STATUS_SUCCESS;
}

VOID
tc90x_FlowControl(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = (PNIC_INFORMATION)Adapter;
        BOOLEAN PhyResponding;
        USHORT PhyAnar;
        USHORT PhyControl;
        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_ANAR, &PhyAnar);
        PhyAnar |= MII_ANAR_FLOWCONTROL;
        WriteMIIPhy(pAdapter, MII_PHY_ANAR, PhyAnar);

        PhyResponding = ReadMIIPhy(pAdapter, MII_PHY_CONTROL, &PhyControl);
        PhyControl |= MII_CONTROL_START_AUTO;
        WriteMIIPhy(pAdapter, MII_PHY_CONTROL, PhyControl);
}

VOID
tc90x_HurricaneEarlyRevision(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        BOOLEAN PhyResponding;
        USHORT PhyRegisterValue;

        PhyResponding = ReadMIIPhy(pAdapter,
                                   MII_PHY_REGISTER_24,
                                   &PhyRegisterValue);

        if (!PhyResponding)
                DBGPRINT_ERROR(("CaneRev-ReadMIIPhy: Phy not responding\n"));

        PhyRegisterValue |= MII_PHY_REGISTER_24_PVCIRC;

        WriteMIIPhy(
                pAdapter,
                MII_PHY_REGISTER_24,
                PhyRegisterValue);
}

/* NIC_STATUS
RxResetAndWork(IN PNIC_INFORMATION Adapter)
{
        PNIC_INFORMATION pAdapter = Adapter;
        NIC_STATUS nicStatus;

        nicStatus = NIC_COMMAND_WAIT(pAdapter, COMMAND_RX_RESET);

        if (nicStatus != NIC_STATUS_SUCCESS) {
                DBGPRINT_ERROR(("RxResetAndWork: Rx reset failed\n"));
                return NIC_STATUS_FAILURE;
        }
        // not all work needed after Rx reset. Refine later
        if (tc90x_SoftwareWork(pAdapter) != NIC_STATUS_SUCCESS) {
                DBGPRINT_ERROR(("RxResetAndWork: SoftwareWork failed\n"));
                return NIC_STATUS_FAILURE;
        }
        return NIC_STATUS_SUCCESS;
} */





#ifdef DEBUG
static UINT debug = DEBUG_ERROR | DEBUG_INITIALIZE | DEBUG_FUNCTION | DEBUG_IOCTL | DEBUG_GET_STATISTICS | DEBUG_SEND | DEBUG_RECEIVE | DEBUG_INTERRUPT ;
#endif

#ifdef __VMKERNEL_MODULE__
static INT switchdelay[MAX_UNITS] =         { [0 ... MAX_UNITS-1 ] = 0 };
static INT media_select[MAX_UNITS] =        { [0 ... MAX_UNITS-1 ] = MEDIA_NONE };
static INT flowcontrol[MAX_UNITS] =         { [0 ... MAX_UNITS-1 ] = 0x1 };
static INT downpoll[MAX_UNITS] =            { [0 ... MAX_UNITS-1 ] = 0x8 };
static INT full_duplex[MAX_UNITS] =         { [0 ... MAX_UNITS-1 ] = -1 };

static UINT tc90x_SendCount[MAX_UNITS] =    { [0 ... MAX_UNITS-1 ] = 0x40 };
static UINT tc90x_ReceiveCount[MAX_UNITS] = { [0 ... MAX_UNITS-1 ] = 0x40 };
#else
static INT switchdelay[] = {0, 0, 0, 0, 0, 0, 0, 0};
static INT media_select[MAX_UNITS] = {MEDIA_NONE, };
static INT flowcontrol[MAX_UNITS] = {0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1};
static INT downpoll[MAX_UNITS] = {0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8};
static INT full_duplex[MAX_UNITS] = {-1, -1, -1, -1, -1, -1, -1, -1};

static UINT tc90x_SendCount[] = { 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40 };
static UINT tc90x_ReceiveCount[] = { 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40 };
#endif

NIC_STATUS
tc90x_ReadCommandLineChanges(
        IN PNIC_INFORMATION Adapter
        )
{
        PNIC_INFORMATION pAdapter = Adapter;
        INT index = pAdapter->Index;
        USHORT  mediatype;

        DBGPRINT_INIT(("ReadCommandLineChanges: IN index=%x\n",index));

        if (tc90x_SendCount[index] < NIC_MINIMUM_SEND_COUNT ||
            tc90x_SendCount[index] > NIC_MAXIMUM_SEND_COUNT) {

                DBGPRINT_ERROR(("ProcessOverride: Using default value\n"));
                pAdapter->Resources.SendCount = NIC_DEFAULT_SEND_COUNT;
        }
        else  {
                DBGPRINT_INITIALIZE(("SendCount = %x\n",(INT)tc90x_SendCount[index]));
                pAdapter->Resources.SendCount = tc90x_SendCount[index];
        }

        if (tc90x_ReceiveCount[index] < NIC_MINIMUM_RECEIVE_COUNT ||
            tc90x_ReceiveCount[index] > NIC_MAXIMUM_RECEIVE_COUNT) {

                DBGPRINT_ERROR(("ReadCommandLineChanges: Using default receive\n"));
                pAdapter->Resources.ReceiveCount = NIC_DEFAULT_RECEIVE_COUNT;

        }
        else  {
                pAdapter->Resources.ReceiveCount = tc90x_ReceiveCount[index];
        }

        DBGPRINT_INITIALIZE((
                "ReceiveCount=%x\n", (INT)pAdapter->Resources.ReceiveCount));

        if ( (index < MAX_UNITS)  && (flowcontrol[index] <= 0) ){

                DBGPRINT_INITIALIZE(("User disables FlowControl\n"));
                pAdapter->Hardware.FlowControlSupported = FALSE;
                pAdapter->Hardware.FlowControlEnable = FALSE;
        }
        else {

                DBGPRINT_INITIALIZE(("FlowControl is enabled by default\n"));
                pAdapter->Hardware.FlowControlSupported = TRUE;
                pAdapter->Hardware.FlowControlEnable = TRUE ;
        }

        if ( (index < MAX_UNITS)  && (downpoll[index] == 0x40) ){

                DBGPRINT_INITIALIZE(("DownPollRate is 64\n"));
                pAdapter->Resources.DownPollRate = 0x40;
        }
        else {

                DBGPRINT_INITIALIZE(("DownPollRate is 8 by default\n"));
                pAdapter->Resources.DownPollRate = 0x8;
        }

        if ( (index < MAX_UNITS)  && (switchdelay[index] > 0) ){

                DBGPRINT_INITIALIZE(("User enables delay for switch\n"));
                pAdapter->DelayStart = TRUE;
        }
        else
                pAdapter->DelayStart = FALSE;

                /* Possible media selections:
                MEDIA_NONE              0
                MEDIA_10BASE_T          1
                MEDIA_10AUI             2
                MEDIA_10BASE_2          3
                MEDIA_100BASE_TX        4
                MEDIA_100BASE_FX        5
                MEDIA_10BASE_FL         6
                MEDIA_AUTO_SELECT       7
                */
                if (index < MAX_UNITS)
                        mediatype = media_select[index];
                else
                        mediatype = MEDIA_NONE;

                if ((mediatype > 0) && (mediatype <= 7))
                {       pAdapter->Hardware.MediaOverride = mediatype;
                        DBGPRINT_INITIALIZE(("User selects Media = %i\n", mediatype));
                }
                else
                        pAdapter->Hardware.MediaOverride = MEDIA_NONE;

                if (pAdapter->Hardware.MediaOverride != MEDIA_AUTO_SELECT)
                {       if ( (index < MAX_UNITS)  &&  (full_duplex[index] > 0) )
                        {       DBGPRINT_INITIALIZE(("NotAutoSelect: User enables full duplex\n"));
                                if ( (pAdapter->Hardware.MediaOverride != MEDIA_10AUI) &&
                                        (pAdapter->Hardware.MediaOverride != MEDIA_10BASE_2) )
                                                pAdapter->Hardware.FullDuplexEnable = TRUE;
                                pAdapter->Hardware.DuplexCommandOverride = TRUE;
                        }
                        else if ( (index < MAX_UNITS)  &&  (full_duplex[index] == 0) )
                        {       DBGPRINT_INITIALIZE(("NotAutoSelect: User disables full duplex\n"));
                                pAdapter->Hardware.FullDuplexEnable = FALSE;
                                pAdapter->Hardware.DuplexCommandOverride = TRUE;
                        }
                        else
                                pAdapter->Hardware.DuplexCommandOverride = FALSE;
                }

        DBGPRINT_INITIALIZE(("ReadCommandLineChanges: OUT\n"));
        return NIC_STATUS_SUCCESS;
}
NIC_STATUS GlobalReset (
	IN  PNIC_INFORMATION pAdapter,
        IN USHORT Command
	)
{
        ULONG count;
        USHORT value;

    NIC_WRITE_PORT_USHORT(
	    pAdapter,
	    INTSTATUS_COMMAND_REGISTER,
	    Command
	    );

    for (count=0; count < 100000; count++) {
	value = NIC_READ_PORT_USHORT(pAdapter, INTSTATUS_COMMAND_REGISTER);
	if (!(value & INTSTATUS_COMMAND_IN_PROGRESS)) {
		value = NIC_READ_PORT_USHORT(pAdapter, INTSTATUS_COMMAND_REGISTER);
		if (!(value & INTSTATUS_COMMAND_IN_PROGRESS))
			break;
	}
	NIC_DELAY(10);
    }

    DBGPRINT_INITIALIZE(("count is :%ld\n", count));
    if (count == 100000) {
		DBGPRINT_ERROR(("WaitAsicReady: timeout\n"));
		return NIC_STATUS_FAILURE;
    }

    
    for (count=0; count < 100000; count++) {
		value = NIC_READ_PORT_USHORT(pAdapter, EEPROM_DATA_REGISTER);
		DBGPRINT_INITIALIZE(("EEPROM Data Register:%04X\n", value));
		if (value == 0xe000)
			break;
		NIC_DELAY(10);
    }

    DBGPRINT_INITIALIZE(("count of EEPROM Data is :%ld\n", count));
    if (count == 100000) {
		DBGPRINT_ERROR(("WaitAsicReady: timeout\n"));
    }
    return NIC_STATUS_SUCCESS;
}
