/* bnx2x_netq.c: bnx2x netq driver.
 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#if (LINUX_VERSION_CODE >= 0x020600) /* BNX2X_UPSTREAM */
#include <linux/device.h>  /* for dev_info() */
#endif
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#if (LINUX_VERSION_CODE >= 0x020600) /* BNX2X_UPSTREAM */
#include <linux/dma-mapping.h>
#endif
#include <linux/bitops.h>
#include <linux/irq.h>
#include <linux/delay.h>
#include <asm/byteorder.h>
#include <linux/time.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#ifdef NETIF_F_HW_VLAN_TX
	#include <linux/if_vlan.h>
#endif
#ifndef __VMKERNEL_MODULE__
#include <net/ip.h>
#if (LINUX_VERSION_CODE < 0x020600) /* ! BNX2X_UPSTREAM */
#include <net/ipv6.h>
#endif
#include <net/tcp.h>
#endif /* !__VMKERNEL_MODULE__ */
#include <net/checksum.h>
#if (LINUX_VERSION_CODE > 0x020607) /* BNX2X_UPSTREAM */
#include <net/ip6_checksum.h>
#endif
#if (LINUX_VERSION_CODE >= 0x020600) /* BNX2X_UPSTREAM */
#include <linux/workqueue.h>
#endif
#include <linux/prefetch.h>

#ifdef __VMKERNEL_MODULE__
/* Definitions for ESX35 */
#include <asm/checksum.h>

	#ifndef LINUX_VERSION_CODE
	#define LINUX_VERSION_CODE KERNEL_VERSION(2, 4, 25)
	#endif /* LINUX_VERSION_CODE */

#endif /* __VMKERNEL_MODULE__ */

#if (LINUX_VERSION_CODE < 0x020618) /* ! BNX2X_UPSTREAM */
#include "bnx2x_compat.h"
#endif

#include "bnx2x_reg.h"
#include "bnx2x_fw_defs.h"
#include "bnx2x_hsi.h"
#include "bnx2x_link.h"
#include "bnx2x.h"

int bnx2x_sp_post(struct bnx2x *bp, int command, int cid,
			 u32 data_hi, u32 data_lo, int common);
void bnx2x_set_storm_rx_mode(struct bnx2x *bp);

#if defined(__VMKERNEL_MODULE__) && defined(__VMKNETDDI_QUEUEOPS__)
static void
bnx2x_remove_rx_filter_e1h(struct bnx2x *bp, u16 qid)
{
	struct bnx2x_fastpath *fp = &bp->fp[qid];
	struct mac_configuration_cmd_e1h *config =
		(struct mac_configuration_cmd_e1h *)bnx2x_sp(bp, mac_config);

	/* CAM allocation for E1H
	 * unicasts: by func number
	 * multicast: 20+FUNC*20, 20 each
	 */
	config->hdr.length_6b = 1;
	config->hdr.offset = BP_FUNC(bp)*4 + qid + 4;
	config->hdr.client_id = BP_CL_ID(bp);
	config->hdr.reserved1 = 0;

	/* primary MAC */
	config->config_table[0].msb_mac_addr = 0xffff;
	config->config_table[0].middle_mac_addr = 0xffff;
	config->config_table[0].lsb_mac_addr = 0xffff;
	config->config_table[0].client_id = BP_L_ID(bp) + qid;
	config->config_table[0].vlan_id = 0;
	config->config_table[0].e1hov_id = cpu_to_le16(bp->e1hov);
	config->config_table[0].flags = MAC_CONFIGURATION_ENTRY_E1H_ACTION_TYPE;

	bnx2x_sp_post(bp, RAMROD_CMD_ID_ETH_SET_MAC, 0,
		      U64_HI(bnx2x_sp_mapping(bp, mac_config)),
		      U64_LO(bnx2x_sp_mapping(bp, mac_config)), 0);

	fp->mac_filter_addr = NULL;
	fp->rx_queue_active = 0;
	return;
}

static void
bnx2x_remove_rx_filter_e1(struct bnx2x *bp, u16 qid)
{
	struct mac_configuration_cmd *config = bnx2x_sp(bp, mac_config);
	struct bnx2x_fastpath *fp = &bp->fp[qid];

	config->hdr.length_6b = 1;
	config->hdr.offset = (BP_PORT(bp) ? 31 : 0) + qid;
	config->hdr.client_id = BP_CL_ID(bp);
	config->hdr.reserved1 = 0;

	config->config_table[0].cam_entry.msb_mac_addr = 0xffff;
	config->config_table[0].cam_entry.middle_mac_addr = 0xffff;
	config->config_table[0].cam_entry.lsb_mac_addr = 0xffff;
	config->config_table[0].cam_entry.flags = cpu_to_le16(BP_PORT(bp));
	config->config_table[0].target_table_entry.flags =
					TSTORM_CAM_TARGET_TABLE_ENTRY_ACTION_TYPE;
	config->config_table[0].target_table_entry.client_id = qid;
	config->config_table[0].target_table_entry.vlan_id = 0;

	bnx2x_sp_post(bp, RAMROD_CMD_ID_ETH_SET_MAC, 0,
		      U64_HI(bnx2x_sp_mapping(bp, mac_config)),
		      U64_LO(bnx2x_sp_mapping(bp, mac_config)), 0);

	fp->mac_filter_addr = NULL;
	fp->rx_queue_active = 0;
	return;
}

static int
bnx2x_remove_rx_filter(vmknetddi_queueop_remove_rx_filter_args_t *args)
{
	struct bnx2x *bp = netdev_priv(args->netdev);
	u16 qid = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	u16 fid = VMKNETDDI_QUEUEOPS_FILTERID_VAL(args->filterid);
	struct bnx2x_fastpath *fp = &bp->fp[qid];
	int rc = VMKNETDDI_QUEUEOPS_ERR;

	if (!VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid))
		goto out;
	if (qid > bp->num_queues)
		goto out;

	/* Only support one Mac filter per queue */
	if (fid != 0 || fp->rx_queue_active == 0)
		goto out;

	if (CHIP_IS_E1H(bp))
		bnx2x_remove_rx_filter_e1h(bp, qid);
	else
		bnx2x_remove_rx_filter_e1(bp, qid);
	rc = VMKNETDDI_QUEUEOPS_OK;

out:
	return rc;
}
static void
bnx2x_apply_rx_filter_e1h(struct bnx2x *bp, u16 queueid)
{
	struct mac_configuration_cmd_e1h *config =
		(struct mac_configuration_cmd_e1h *)bnx2x_sp(bp, mac_config);
	struct bnx2x_fastpath *fp = &bp->fp[queueid];
	config->hdr.length_6b = 1;
	config->hdr.offset = BP_FUNC(bp)*4 + queueid + 4;
	config->hdr.client_id = BP_CL_ID(bp);
	config->hdr.reserved1 = 0;

	config->config_table[0].msb_mac_addr =
					swab16(*(u16 *)&fp->mac_filter_addr[0]);
	config->config_table[0].middle_mac_addr =
					swab16(*(u16 *)&fp->mac_filter_addr[2]);
	config->config_table[0].lsb_mac_addr =
					swab16(*(u16 *)&fp->mac_filter_addr[4]);
	config->config_table[0].client_id = BP_L_ID(bp) + queueid;
	config->config_table[0].vlan_id = 0;
	config->config_table[0].e1hov_id = cpu_to_le16(bp->e1hov);
	config->config_table[0].flags = 0;

	bnx2x_sp_post(bp, RAMROD_CMD_ID_ETH_SET_MAC, 0,
		      U64_HI(bnx2x_sp_mapping(bp, mac_config)),
		      U64_LO(bnx2x_sp_mapping(bp, mac_config)), 0);

	fp->rx_queue_active = 1;
	return;
}
static void
bnx2x_apply_rx_filter_e1(struct bnx2x *bp, u16 queueid)
{
	struct mac_configuration_cmd *config = bnx2x_sp(bp, mac_config);
	struct bnx2x_fastpath *fp = &bp->fp[queueid];

	config->hdr.length_6b = 1;
	config->hdr.offset = (BP_PORT(bp) ? 31 : 0) + queueid;
	config->hdr.client_id = BP_CL_ID(bp);
	config->hdr.reserved1 = 0;

	config->config_table[0].cam_entry.msb_mac_addr =
					swab16(*(u16 *)&fp->mac_filter_addr[0]);
	config->config_table[0].cam_entry.middle_mac_addr =
					swab16(*(u16 *)&fp->mac_filter_addr[2]);
	config->config_table[0].cam_entry.lsb_mac_addr =
					swab16(*(u16 *)&fp->mac_filter_addr[4]);
	config->config_table[0].cam_entry.flags = cpu_to_le16(BP_PORT(bp));
	config->config_table[0].target_table_entry.flags = 0;
	config->config_table[0].target_table_entry.client_id = queueid;
	config->config_table[0].target_table_entry.vlan_id = 0;

	bnx2x_sp_post(bp, RAMROD_CMD_ID_ETH_SET_MAC, 0,
		      U64_HI(bnx2x_sp_mapping(bp, mac_config)),
		      U64_LO(bnx2x_sp_mapping(bp, mac_config)), 0);

	fp->rx_queue_active = 1;
	return;
}

static int
bnx2x_apply_rx_filter(vmknetddi_queueop_apply_rx_filter_args_t *args)
{
	struct bnx2x *bp = netdev_priv(args->netdev);
	u16 queueid = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	struct bnx2x_fastpath *fp;
	int rc = VMKNETDDI_QUEUEOPS_ERR;

	if (!VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid))
		goto out;

	if (vmknetddi_queueops_get_filter_class(&args->filter)
					!= VMKNETDDI_QUEUEOPS_FILTER_MACADDR)
		goto out;

	if (queueid > bp->num_queues)
		goto out;

	fp = &bp->fp[queueid];
	if (!fp->rx_queue_allocated || fp->rx_queue_active)
		goto out;

	fp->mac_filter_addr = (void *)vmknetddi_queueops_get_filter_macaddr(&args->filter);

	if (CHIP_IS_E1H(bp))
		bnx2x_apply_rx_filter_e1h(bp, queueid);
	else
		bnx2x_apply_rx_filter_e1(bp, queueid);
	args->filterid = VMKNETDDI_QUEUEOPS_MK_FILTERID(0);
	rc = VMKNETDDI_QUEUEOPS_OK;

out:
	return rc;
}

static int bnx2x_get_default_queue(vmknetddi_queueop_get_default_queue_args_t *args)
{
	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		args->queueid = VMKNETDDI_QUEUEOPS_MK_RX_QUEUEID(0);
		return VMKNETDDI_QUEUEOPS_OK;
	}
	else {
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
bnx2x_get_queue_vector(vmknetddi_queueop_get_queue_vector_args_t *args)
{
	int qid;
	struct net_device *netdev = args->netdev;
	struct bnx2x *bp = netdev_priv(netdev);
	int rc = VMKNETDDI_QUEUEOPS_OK;

	qid = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	if (qid > bp->num_queues) {
		rc = VMKNETDDI_QUEUEOPS_ERR;
		goto out;
	}
	args->vector = bp->msix_table[qid].vector;

out:
	return rc;
}
static int
bnx2x_free_rx_queue(struct net_device *netdev, vmknetddi_queueops_queueid_t qid)
{
	struct bnx2x *bp = netdev_priv(netdev);
	u16 index = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(qid);
	struct bnx2x_fastpath *fp;
	if (index > bp->num_queues) {
		return VMKNETDDI_QUEUEOPS_ERR;
	}
	fp = &bp->fp[index];
	if (fp->rx_queue_allocated != TRUE) {
		return VMKNETDDI_QUEUEOPS_ERR;
	}
	fp->rx_queue_allocated = FALSE;
	bp->n_rx_queues_allocated--;
	return VMKNETDDI_QUEUEOPS_OK;
}

static int
bnx2x_free_queue(vmknetddi_queueop_free_queue_args_t *args)
{
	int rc = VMKNETDDI_QUEUEOPS_ERR;

	if (VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid)) {
		rc = bnx2x_free_rx_queue(args->netdev, args->queueid);
	}
	return rc;
}

static int
bnx2x_alloc_rx_queue(struct net_device *netdev, vmknetddi_queueops_queueid_t *p_qid)
{
	int i;
	struct bnx2x *bp = netdev_priv(netdev);

	if (bp->n_rx_queues_allocated >= bp->num_queues) {
		return VMKNETDDI_QUEUEOPS_ERR;
	}
	for_each_nondefault_queue(bp, i) {
		struct bnx2x_fastpath *fp = &bp->fp[i];
		if (!fp->rx_queue_allocated) {
			fp->rx_queue_allocated = 1;
			bp->n_rx_queues_allocated++;
			*p_qid = VMKNETDDI_QUEUEOPS_MK_RX_QUEUEID(fp->index);
			return VMKNETDDI_QUEUEOPS_OK;
		}
	}
	BNX2X_ERR("bnx2x_alloc_rx_queue: no free rx queues found!\n");
	return VMKNETDDI_QUEUEOPS_ERR;
}

static int
bnx2x_alloc_queue(vmknetddi_queueop_alloc_queue_args_t *args)
{
	int rc = VMKNETDDI_QUEUEOPS_ERR;

	/* Only RX queue is supported in ESX3.5 */
	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		rc = bnx2x_alloc_rx_queue(args->netdev, &args->queueid);
	}
	return rc;
}


static int
bnx2x_get_filter_count(vmknetddi_queueop_get_filter_count_args_t *args)
{
	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		/* Only support 1 Mac filter per queue */
		args->count = 1;
		return VMKNETDDI_QUEUEOPS_OK;
	}
	else {
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
bnx2x_get_queue_count(vmknetddi_queueop_get_queue_count_args_t *args)
{
	struct bnx2x *bp = netdev_priv(args->netdev);
	int rc = VMKNETDDI_QUEUEOPS_ERR;

	/* workaround for packets duplicated */
	if (bp->num_queues > 1) {
		bp->netq_enabled = 1;
		bnx2x_set_storm_rx_mode(bp);
	}
	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		args->count = bp->num_queues - 1;
		rc = VMKNETDDI_QUEUEOPS_OK;
	}
	return rc;
}
static int
bnx2x_get_netqueue_features(vmknetddi_queueop_get_features_args_t *args)
{
	/* Only supporting RX queue this version */
	args->features = VMKNETDDI_QUEUEOPS_FEATURE_NONE;
	args->features |= VMKNETDDI_QUEUEOPS_FEATURE_RXQUEUES;

	return VMKNETDDI_QUEUEOPS_OK;

}

static int bnx2x_get_netqueue_version(vmknetddi_queueop_get_version_args_t *args)
{
	return vmknetddi_queueops_version(args);
}

int bnx2x_netqueue_ops(vmknetddi_queueops_op_t op, void *args)
{
	int rc =  VMKNETDDI_QUEUEOPS_ERR;
	rtnl_lock();
	switch (op) {
		case VMKNETDDI_QUEUEOPS_OP_GET_VERSION:
		rc = bnx2x_get_netqueue_version(
			(vmknetddi_queueop_get_version_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_FEATURES:
		rc = bnx2x_get_netqueue_features(
			(vmknetddi_queueop_get_features_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_QUEUE_COUNT:
		rc = bnx2x_get_queue_count(
			(vmknetddi_queueop_get_queue_count_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_FILTER_COUNT:
		rc = bnx2x_get_filter_count(
			(vmknetddi_queueop_get_filter_count_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_ALLOC_QUEUE:
		rc = bnx2x_alloc_queue(
			(vmknetddi_queueop_alloc_queue_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_FREE_QUEUE:
		rc = bnx2x_free_queue(
			(vmknetddi_queueop_free_queue_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_QUEUE_VECTOR:
		rc = bnx2x_get_queue_vector(
			(vmknetddi_queueop_get_queue_vector_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_DEFAULT_QUEUE:
		rc = bnx2x_get_default_queue(
			(vmknetddi_queueop_get_default_queue_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_APPLY_RX_FILTER:
		rc = bnx2x_apply_rx_filter(
			(vmknetddi_queueop_apply_rx_filter_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_REMOVE_RX_FILTER:
		rc = bnx2x_remove_rx_filter(
			(vmknetddi_queueop_remove_rx_filter_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_STATS:
		rc = VMKNETDDI_QUEUEOPS_ERR;
		break;

		default:
		rc = VMKNETDDI_QUEUEOPS_ERR;
	}

	rtnl_unlock();
	return rc;
}
#endif /* defined(__VMKERNEL_MODULE__) && defined(__VMKNETDDI_QUEUEOPS__) */
