/*
 * E100 stress specific macros
 *
 */

#ifndef _E100_STRESS_H_
#define _E100_STRESS_H_
#ifdef __VMKERNEL_MODULE__
#include "linux_stress.h"
#include "e100.h"
#include "e100_ucode.h"
#include "e100_config.h"
#include "e100_phy.h"

/*
 * VMWARE_STRESS_ONLY_STOP_QUEUE
 *
 * The macro takes two parameters. The first one is the 
 * device for which queuing is to be disabled. The second
 * parameter is a hack to specify where control should
 * be transferred when the stress counter is hit. Typically,
 * it is either a continue or a return statement.
 *
 */
#define VMWARE_STRESS_ONLY_STOP_QUEUE(dev, CONTINUE_CMD)                   \
if (VMK_STRESS_DEBUG_COUNTER(NET_HW_STOP_QUEUE)) {                         \
   netif_stop_queue(dev);                                                  \
   CONTINUE_CMD;                                                           \
} 


/*
 * LinuxStress_StressIntrRate
 *
 * This is the e100 specific interrupt rate control function.
 * This is invoked through a tasklet that is scheduled in the
 * e100 interrupt handler whenever the interrupt rate is changed.
 *
 */

static
void LinuxStress_StressIntrRate(unsigned long data)
{
	struct e100_private *bdp = (struct e100_private *)data;
   
	bdp->params.IntDelay = VMK_STRESS_DEBUG_VALUE(NET_HW_INTR_RATE);
	// disable waiting for small frames
	bdp->params.b_params &= ~PRM_BUNDLE_SMALL;
	// set bundling based on the interrupt delay
	if (bdp->params.IntDelay < 0x200) {
		bdp->params.BundleMax = 0x1;
	} else if (bdp->params.IntDelay < 0x600) {
		bdp->params.BundleMax = 0x6;
	} else if (bdp->params.IntDelay < 0x800) {
		bdp->params.BundleMax = 0x8;
	} else {
		bdp->params.BundleMax = 0xffff; // very large value
		bdp->params.b_params |= PRM_BUNDLE_SMALL;
	}
	// if interrupt delay is zero (disabled), reset
	// BundleMax to default value
	if (!bdp->params.IntDelay) {
		bdp->params.BundleMax = 0x6;
	}
	bdp->params.b_params |= PRM_UCODE;
	printk(KERN_ALERT "Changing Interrupt Rate: delay = %u, bundleMax = %u\n",
	       bdp->params.IntDelay, bdp->params.BundleMax);
	if (e100_load_microcode(bdp)) {
		bdp->flags |= DF_UCODE_LOADED;
	} else {
		printk(KERN_ALERT "Failed to install microcode\n");
	}
}
#else
/* These are generic definitions when __VMKERNEL_MODULE__ is not defined.
 * Some of them are from linux_stress.h. The reason the default cases
 * are defined here is that linux_stress.h is not in the include
 * path when __VMKERNEL_MODULE__ is not defined.
 */
#define VMWARE_STRESS_RETAIN_BUFFER(skb, CONTINUE_CMD, ELSE_CMD) ELSE_CMD;
#define VMWARE_STRESS_ONLY_STOP_QUEUE(dev, CONTINUE_CMD)
#define VMWARE_STRESS_ONLY_SET_INTR_RATE(param, compare)
#define VMWARE_STRESS_ONLY_CORRUPT_BUFFER(COUNTER, skb, len)
#define VMWARE_STRESS_ONLY_FAIL_BUFFER(COUNTER, skb, FREE_SKB, CONTINUE_CMD)
#endif // __VMKERNEL_MODULE__
#define NO_ELSE_CMD
#define NO_CONTINUE_CMD
#endif // _E100_STRESS_H_
