/*
 * forcedeth: Ethernet driver for NVIDIA nForce media access controllers.
 *
 * Note: This driver is a cleanroom reimplementation based on reverse
 *      engineered documentation written by Carl-Daniel Hailfinger
 *      and Andrew de Quincey. It's neither supported nor endorsed
 *      by NVIDIA Corp. Use at your own risk.
 *
 * NVIDIA, nForce and other NVIDIA marks are trademarks or registered
 * trademarks of NVIDIA Corporation in the United States and other
 * countries.
 *
 * Copyright (C) 2003,4,5 Manfred Spraul
 * Copyright (C) 2004 Andrew de Quincey (wol support)
 * Copyright (C) 2004 Carl-Daniel Hailfinger (invalid MAC handling, insane
 *		IRQ rate fixes, bigendian fixes, cleanups, verification)
 * Copyright (c) 2004 NVIDIA Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Changelog:
 * 	0.01: 05 Oct 2003: First release that compiles without warnings.
 * 	0.02: 05 Oct 2003: Fix bug for nv_drain_tx: do not try to free NULL skbs.
 * 			   Check all PCI BARs for the register window.
 * 			   udelay added to mii_rw.
 * 	0.03: 06 Oct 2003: Initialize dev->irq.
 * 	0.04: 07 Oct 2003: Initialize np->lock, reduce handled irqs, add printks.
 * 	0.05: 09 Oct 2003: printk removed again, irq status print tx_timeout.
 * 	0.06: 10 Oct 2003: MAC Address read updated, pff flag generation updated,
 * 			   irq mask updated
 * 	0.07: 14 Oct 2003: Further irq mask updates.
 * 	0.08: 20 Oct 2003: rx_desc.Length initialization added, nv_alloc_rx refill
 * 			   added into irq handler, NULL check for drain_ring.
 * 	0.09: 20 Oct 2003: Basic link speed irq implementation. Only handle the
 * 			   requested interrupt sources.
 * 	0.10: 20 Oct 2003: First cleanup for release.
 * 	0.11: 21 Oct 2003: hexdump for tx added, rx buffer sizes increased.
 * 			   MAC Address init fix, set_multicast cleanup.
 * 	0.12: 23 Oct 2003: Cleanups for release.
 * 	0.13: 25 Oct 2003: Limit for concurrent tx packets increased to 10.
 * 			   Set link speed correctly. start rx before starting
 * 			   tx (nv_start_rx sets the link speed).
 * 	0.14: 25 Oct 2003: Nic dependant irq mask.
 * 	0.15: 08 Nov 2003: fix smp deadlock with set_multicast_list during
 * 			   open.
 * 	0.16: 15 Nov 2003: include file cleanup for ppc64, rx buffer size
 * 			   increased to 1628 bytes.
 * 	0.17: 16 Nov 2003: undo rx buffer size increase. Substract 1 from
 * 			   the tx length.
 * 	0.18: 17 Nov 2003: fix oops due to late initialization of dev_stats
 * 	0.19: 29 Nov 2003: Handle RxNoBuf, detect & handle invalid mac
 * 			   addresses, really stop rx if already running
 * 			   in nv_start_rx, clean up a bit.
 * 	0.20: 07 Dec 2003: alloc fixes
 * 	0.21: 12 Jan 2004: additional alloc fix, nic polling fix.
 *	0.22: 19 Jan 2004: reprogram timer to a sane rate, avoid lockup
 *			   on close.
 *	0.23: 26 Jan 2004: various small cleanups
 *	0.24: 27 Feb 2004: make driver even less anonymous in backtraces
 *	0.25: 09 Mar 2004: wol support
 *	0.26: 03 Jun 2004: netdriver specific annotation, sparse-related fixes
 *	0.27: 19 Jun 2004: Gigabit support, new descriptor rings,
 *			   added CK804/MCP04 device IDs, code fixes
 *			   for registers, link status and other minor fixes.
 *	0.28: 21 Jun 2004: Big cleanup, making driver mostly endian safe
 *	0.29: 31 Aug 2004: Add backup timer for link change notification.
 *	0.30: 25 Sep 2004: rx checksum support for nf 250 Gb. Add rx reset
 *			   into nv_close, otherwise reenabling for wol can
 *			   cause DMA to kfree'd memory.
 *	0.31: 14 Nov 2004: ethtool support for getting/setting link
 *			   capabilities.
 *	0.32: 16 Apr 2005: RX_ERROR4 handling added.
 *	0.33: 16 May 2005: Support for MCP51 added.
 *	0.34: 18 Jun 2005: Add DEV_NEED_LINKTIMER to all nForce nics.
 *	0.35: 26 Jun 2005: Support for MCP55 added.
 *	0.36: 28 Jun 2005: Add jumbo frame support.
 *	0.37: 10 Jul 2005: Additional ethtool support, cleanup of pci id list
 *	0.38: 16 Jul 2005: tx irq rewrite: Use global flags instead of
 *			   per-packet flags.
 *	0.39: 18 Jul 2005: Add 64bit descriptor support.
 *	0.40: 19 Jul 2005: Add support for mac address change.
 *	0.41: 30 Jul 2005: Write back original MAC in nv_close instead
 *			   of nv_remove
 *	0.42: 06 Aug 2005: Fix lack of link speed initialization
 *			   in the second (and later) nv_open call
 *	0.43: 10 Aug 2005: Add support for tx checksum.
 *	0.44: 20 Aug 2005: Add support for scatter gather and segmentation.
 *	0.45: 18 Sep 2005: Remove nv_stop/start_rx from every link check
 *	0.46: 20 Oct 2005: Add irq optimization modes.
 *	0.47: 26 Oct 2005: Add phyaddr 0 in phy scan.
 *	0.48: 24 Dec 2005: Disable TSO, bugfix for pci_map_single
 *	0.49: 10 Dec 2005: Fix tso for large buffers.
 *	0.50: 20 Jan 2006: Add 8021pq tagging support.
 *	0.51: 20 Jan 2006: Add 64bit consistent memory allocation for rings.
 *	0.52: 20 Jan 2006: Add MSI/MSIX support.
 *	0.53: 19 Mar 2006: Fix init from low power mode and add hw reset.
 *	0.54: 21 Mar 2006: Fix spin locks for multi irqs and cleanup.
 *	0.55: 22 Mar 2006: Add flow control (pause frame).
 *	0.56: 22 Mar 2006: Additional ethtool and moduleparam support.
 *	0.57: 14 May 2006: Moved mac address writes to nv_probe and nv_remove.
 *	0.58: 20 May 2006: Optimized rx and tx data paths.
 *	0.59: 31 May 2006: Added support for sideband management unit.
 *	0.60: 31 May 2006: Added support for recoverable error.
 *
 * Known bugs:
 * We suspect that on some hardware no TX done interrupts are generated.
 * This means recovery from netif_stop_queue only happens if the hw timer
 * interrupt fires (100 times/second, configurable with NVREG_POLL_DEFAULT)
 * and the timer is active in the IRQMask, or if a rx packet arrives by chance.
 * If your hardware reliably generates tx done interrupts, then you can remove
 * DEV_NEED_TIMERIRQ from the driver_data flags.
 * DEV_NEED_TIMERIRQ will not harm you on sane hardware, only generating a few
 * superfluous timer interrupts from the nic.
 */
#define FORCEDETH_VERSION		"0.60-Driver Package V1.25"
#define DRV_NAME			"forcedeth"

#include <linux/module.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/ethtool.h>
#include <linux/timer.h>
#include <linux/skbuff.h>
#include <linux/mii.h>
#include <linux/random.h>
#include <linux/init.h>
#include <linux/if_vlan.h>
#include <linux/rtnetlink.h>
#include <linux/version.h>
#ifdef __VMKERNEL_MODULE__  // add reboot notifier for vmkernel
#include <linux/reboot.h>
#endif

#define RHES3  		0
#define SLES9	 	1
#define RHES4		2
#define SUSE10		3 
#define	FEDORA5		4 

 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,13)
#define NVVER FEDORA5		
#elif LINUX_VERSION_CODE > KERNEL_VERSION(2,6,9)
#define NVVER SUSE10		
#elif LINUX_VERSION_CODE > KERNEL_VERSION(2,6,6)
#define NVVER RHES4	
#elif LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
#define NVVER SLES9	
#else
#define NVVER RHES3  	
#endif

#if NVVER > RHES3
#include <linux/dma-mapping.h>
#else
//#include <linux/forcedeth-compat.h>
#include "forcedeth-compat.h"
#endif

#include <asm/irq.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#ifdef __VMKERNEL_MODULE__
#ifndef IRQ_HANDLED
#define irqreturn_t void
#define IRQ_HANDLED
#define IRQ_NONE
#endif

#define IRQ_RETVAL(i) i
#endif // __VMKERNEL_MODULE__

#if 0
#define dprintk			printk
#else
#define dprintk(x...)		do { } while (0)
#endif

#ifdef __VMKERNEL_MODULE__
#define spin_lock_irq(_lck)                                                  \
   {                                                                         \
      unsigned long __flags;                                                 \
      spin_lock_irqsave(_lck, __flags);

#define spin_unlock_irq(_lck)                                                \
      spin_unlock_irqrestore(_lck, __flags);                                 \
   }
#endif // __VMKERNEL_MODULE__

/* it should add in pci_ids.h */
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_12
#define PCI_DEVICE_ID_NVIDIA_NVENET_12 0x0268 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_13
#define PCI_DEVICE_ID_NVIDIA_NVENET_13 0x0269 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_14
#define PCI_DEVICE_ID_NVIDIA_NVENET_14 0x0372 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_15
#define PCI_DEVICE_ID_NVIDIA_NVENET_15 0x0373 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_16
#define PCI_DEVICE_ID_NVIDIA_NVENET_16 0x03E5
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_17
#define PCI_DEVICE_ID_NVIDIA_NVENET_17 0x03E6 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_18
#define PCI_DEVICE_ID_NVIDIA_NVENET_18 0x03EE 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_19
#define PCI_DEVICE_ID_NVIDIA_NVENET_19 0x03EF 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_20
#define PCI_DEVICE_ID_NVIDIA_NVENET_20 0x0450 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_21
#define PCI_DEVICE_ID_NVIDIA_NVENET_21 0x0451 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_22
#define PCI_DEVICE_ID_NVIDIA_NVENET_22 0x0452 
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_NVENET_23
#define PCI_DEVICE_ID_NVIDIA_NVENET_23 0x0453 
#endif

/* it should add in mii.h */
#ifndef ADVERTISE_1000HALF
#define ADVERTISE_1000HALF	0x0100
#endif
#ifndef ADVERTISE_1000FULL
#define ADVERTISE_1000FULL	0x0200
#endif
#ifndef ADVERTISE_PAUSE_CAP
#define ADVERTISE_PAUSE_CAP	0x0400
#endif
#ifndef ADVERTISE_PAUSE_ASYM
#define ADVERTISE_PAUSE_ASYM	0x0800
#endif
#ifndef MII_CTRL1000
#define MII_CTRL1000		0x09 
#endif
#ifndef MII_STAT1000
#define MII_STAT1000		0x0A 
#endif
#ifndef LPA_1000FULL
#define LPA_1000FULL		0x0800
#endif
#ifndef LPA_1000HALF
#define LPA_1000HALF		0x0400 
#endif
#ifndef LPA_PAUSE_CAP
#define LPA_PAUSE_CAP		0x0400
#endif
#ifndef LPA_PAUSE_ASYM
#define LPA_PAUSE_ASYM		0x0800
#endif
#ifndef BMCR_SPEED1000
#define BMCR_SPEED1000		0x0040	/* MSB of Speed (1000)         */
#endif

#ifndef NETDEV_TX_OK
#define NETDEV_TX_OK 		0	/* driver took care of packet */
#endif

#ifndef NETDEV_TX_BUSY
#define NETDEV_TX_BUSY 		1    /* driver tx path was busy*/
#endif

#ifndef DMA_39BIT_MASK
#define DMA_39BIT_MASK		0x0000007fffffffffULL    
#endif

#ifndef __iomem
#define __iomem 
#endif

/* rx/tx mac addr + type + vlan + align + slack*/
#ifndef RX_NIC_BUFSIZE	
#define RX_NIC_BUFSIZE		(ETH_DATA_LEN + 64)
#endif
/* even more slack */
#ifndef RX_ALLOC_BUFSIZE	
#define RX_ALLOC_BUFSIZE	(ETH_DATA_LEN + 128)
#endif

#ifndef __VMKERNEL_MODULE__
#if NVVER < RHES4
struct msix_entry {
	u16 vector;	/* kernel uses to write allocated vector */
	u16 entry;	/* driver uses to specify entry, OS writes */
};
#endif
#endif

/*
 * Hardware access:
 */

#ifdef __VMKERNEL_MODULE__
#define DEV_NEED_TIMERIRQ       0x0000  /* set the timer irq flag in the irq mask */
#else
#define DEV_NEED_TIMERIRQ       0x0001  /* set the timer irq flag in the irq mask */
#endif

#define DEV_NEED_LINKTIMER	0x0002	/* poll link settings. Relies on the timer irq */
#define DEV_HAS_LARGEDESC	0x0004	/* device supports jumbo frames and needs packet format 2 */
#define DEV_HAS_HIGH_DMA        0x0008  /* device supports 64bit dma */
#define DEV_HAS_CHECKSUM        0x0010  /* device supports tx and rx checksum offloads */
#define DEV_HAS_VLAN            0x0020  /* device supports vlan tagging and striping */
#define DEV_HAS_MSI             0x0040  /* device supports MSI */
#define DEV_HAS_MSI_X           0x0080  /* device supports MSI-X */
#define DEV_HAS_POWER_CNTRL     0x0100  /* device supports power savings */
#define DEV_HAS_PAUSEFRAME_TX   0x0200  /* device supports tx pause frames */
#define DEV_HAS_STATISTICS      0x0400  /* device supports hw statistics */
#define DEV_HAS_TEST_EXTENDED   0x0800  /* device supports extended diagnostic test */
#define DEV_HAS_MGMT_UNIT       0x1000  /* device supports management unit */
#define NVIDIA_ETHERNET_ID(deviceid,nv_driver_data) {\
		.vendor = PCI_VENDOR_ID_NVIDIA, \
		.device = deviceid, \
		.subvendor = PCI_ANY_ID, \
		.subdevice = PCI_ANY_ID, \
		.driver_data = nv_driver_data, \
		},

enum {
	NvRegIrqStatus = 0x000,
#define NVREG_IRQSTAT_MIIEVENT	0x040
#define NVREG_IRQSTAT_MASK		0x81ff
	NvRegIrqMask = 0x004,
#define NVREG_IRQ_RX_ERROR		0x0001
#define NVREG_IRQ_RX			0x0002
#define NVREG_IRQ_RX_NOBUF		0x0004
#define NVREG_IRQ_TX_ERR		0x0008
#define NVREG_IRQ_TX_OK			0x0010
#define NVREG_IRQ_TIMER			0x0020
#define NVREG_IRQ_LINK			0x0040
#define NVREG_IRQ_RX_FORCED		0x0080
#define NVREG_IRQ_TX_FORCED		0x0100
#define NVREG_IRQ_RECOVER_ERROR		0x8000
#define NVREG_IRQMASK_THROUGHPUT	0x00df
#define NVREG_IRQMASK_CPU		0x0040
#define NVREG_IRQ_TX_ALL		(NVREG_IRQ_TX_ERR|NVREG_IRQ_TX_OK|NVREG_IRQ_TX_FORCED)
#define NVREG_IRQ_RX_ALL		(NVREG_IRQ_RX_ERROR|NVREG_IRQ_RX|NVREG_IRQ_RX_NOBUF|NVREG_IRQ_RX_FORCED)
#define NVREG_IRQ_OTHER			(NVREG_IRQ_TIMER|NVREG_IRQ_LINK|NVREG_IRQ_RECOVER_ERROR)

#define NVREG_IRQ_UNKNOWN	(~(NVREG_IRQ_RX_ERROR|NVREG_IRQ_RX|NVREG_IRQ_RX_NOBUF|NVREG_IRQ_TX_ERR| \
					NVREG_IRQ_TX_OK|NVREG_IRQ_TIMER|NVREG_IRQ_LINK|NVREG_IRQ_RX_FORCED| \
					NVREG_IRQ_TX_FORCED|NVREG_IRQ_RECOVER_ERROR))

	NvRegUnknownSetupReg6 = 0x008,
#define NVREG_UNKSETUP6_VAL		3

/*
 * NVREG_POLL_DEFAULT is the interval length of the timer source on the nic
 * NVREG_POLL_DEFAULT=97 would result in an interval length of 1 ms
 */
	NvRegPollingInterval = 0x00c,
#define NVREG_POLL_DEFAULT_THROUGHPUT	970
#define NVREG_POLL_DEFAULT_CPU	13
	NvRegMSIMap0 = 0x020,
	NvRegMSIMap1 = 0x024,
	NvRegMSIIrqMask = 0x030,
#define NVREG_MSI_VECTOR_0_ENABLED 0x01
	NvRegMisc1 = 0x080,
#define NVREG_MISC1_PAUSE_TX	0x01
#define NVREG_MISC1_HD		0x02
#define NVREG_MISC1_FORCE	0x3b0f3c

	NvRegMacReset = 0x3c,
#define NVREG_MAC_RESET_ASSERT	0x0F3
	NvRegTransmitterControl = 0x084,
#define NVREG_XMITCTL_START	0x01
#define NVREG_XMITCTL_MGMT_ST	0x40000000
#define NVREG_XMITCTL_SYNC_MASK		0x000f0000
#define NVREG_XMITCTL_SYNC_NOT_READY	0x0
#define NVREG_XMITCTL_SYNC_PHY_INIT	0x00040000
#define NVREG_XMITCTL_MGMT_SEMA_MASK	0x00000f00
#define NVREG_XMITCTL_MGMT_SEMA_FREE	0x0
#define NVREG_XMITCTL_HOST_SEMA_MASK	0x0000f000
#define NVREG_XMITCTL_HOST_SEMA_ACQ	0x0000f000
#define NVREG_XMITCTL_HOST_LOADED	0x00004000
#define NVREG_XMITCTL_TX_PATH_EN	0x01000000
	NvRegTransmitterStatus = 0x088,
#define NVREG_XMITSTAT_BUSY	0x01

	NvRegPacketFilterFlags = 0x8c,
#define NVREG_PFF_PAUSE_RX	0x08
#define NVREG_PFF_ALWAYS	0x7F0000
#define NVREG_PFF_PROMISC	0x80
#define NVREG_PFF_MYADDR	0x20
#define NVREG_PFF_LOOPBACK	0x10

	NvRegOffloadConfig = 0x90,
#define NVREG_OFFLOAD_HOMEPHY	0x601
#define NVREG_OFFLOAD_NORMAL	RX_NIC_BUFSIZE
	NvRegReceiverControl = 0x094,
#define NVREG_RCVCTL_START	0x01
#define NVREG_RCVCTL_RX_PATH_EN	0x01000000
	NvRegReceiverStatus = 0x98,
#define NVREG_RCVSTAT_BUSY	0x01

	NvRegRandomSeed = 0x9c,
#define NVREG_RNDSEED_MASK	0x00ff
#define NVREG_RNDSEED_FORCE	0x7f00
#define NVREG_RNDSEED_FORCE2	0x2d00
#define NVREG_RNDSEED_FORCE3	0x7400

	NvRegTxDeferral = 0xA0,
#define NVREG_TX_DEFERRAL_DEFAULT      	0x15050f
#define NVREG_TX_DEFERRAL_RGMII_10_100	0x16070f
#define NVREG_TX_DEFERRAL_RGMII_1000	0x14050f
	NvRegRxDeferral = 0xA4,
#define NVREG_RX_DEFERRAL_DEFAULT	0x16
	NvRegMacAddrA = 0xA8,
	NvRegMacAddrB = 0xAC,
	NvRegMulticastAddrA = 0xB0,
#define NVREG_MCASTADDRA_FORCE	0x01
	NvRegMulticastAddrB = 0xB4,
	NvRegMulticastMaskA = 0xB8,
	NvRegMulticastMaskB = 0xBC,

	NvRegPhyInterface = 0xC0,
#define PHY_RGMII		0x10000000

	NvRegTxRingPhysAddr = 0x100,
	NvRegRxRingPhysAddr = 0x104,
	NvRegRingSizes = 0x108,
#define NVREG_RINGSZ_TXSHIFT 0
#define NVREG_RINGSZ_RXSHIFT 16
	NvRegTransmitPoll = 0x10c,
#define NVREG_TRANSMITPOLL_MAC_ADDR_REV	0x00008000
	NvRegLinkSpeed = 0x110,
#define NVREG_LINKSPEED_FORCE 0x10000
#define NVREG_LINKSPEED_10	1000
#define NVREG_LINKSPEED_100	100
#define NVREG_LINKSPEED_1000	50
#define NVREG_LINKSPEED_MASK	(0xFFF)
	NvRegUnknownSetupReg5 = 0x130,
#define NVREG_UNKSETUP5_BIT31	(1<<31)
	NvRegTxWatermark = 0x13c,
#define NVREG_TX_WM_DESC1_DEFAULT	0x0200010
#define NVREG_TX_WM_DESC2_3_DEFAULT	0x1e08000
#define NVREG_TX_WM_DESC2_3_1000	0xfe08000
	NvRegTxRxControl = 0x144,
#define NVREG_TXRXCTL_KICK	0x0001
#define NVREG_TXRXCTL_BIT1	0x0002
#define NVREG_TXRXCTL_BIT2	0x0004
#define NVREG_TXRXCTL_IDLE	0x0008
#define NVREG_TXRXCTL_RESET	0x0010
#define NVREG_TXRXCTL_RXCHECK	0x0400
#define NVREG_TXRXCTL_DESC_1	0
#define NVREG_TXRXCTL_DESC_2	0x002100
#define NVREG_TXRXCTL_DESC_3	0xc02200
#define NVREG_TXRXCTL_VLANSTRIP 0x00040
#define NVREG_TXRXCTL_VLANINS	0x00080
	NvRegTxRingPhysAddrHigh = 0x148,
	NvRegRxRingPhysAddrHigh = 0x14C,
	NvRegTxPauseFrame = 0x170,
#define NVREG_TX_PAUSEFRAME_DISABLE	0x1ff0080
#define NVREG_TX_PAUSEFRAME_ENABLE	0x0c00030
	NvRegMIIStatus = 0x180,
#define NVREG_MIISTAT_ERROR		0x0001
#define NVREG_MIISTAT_LINKCHANGE	0x0008
#define NVREG_MIISTAT_MASK		0x000f
#define NVREG_MIISTAT_MASK2		0x000f
	NvRegMIIMask = 0x184,
#define NVREG_MII_LINKCHANGE		0x0008

	NvRegAdapterControl = 0x188,
#define NVREG_ADAPTCTL_START	0x02
#define NVREG_ADAPTCTL_LINKUP	0x04
#define NVREG_ADAPTCTL_PHYVALID	0x40000
#define NVREG_ADAPTCTL_RUNNING	0x100000
#define NVREG_ADAPTCTL_PHYSHIFT	24
	NvRegMIISpeed = 0x18c,
#define NVREG_MIISPEED_BIT8	(1<<8)
#define NVREG_MIIDELAY	5
	NvRegMIIControl = 0x190,
#define NVREG_MIICTL_INUSE	0x08000
#define NVREG_MIICTL_WRITE	0x00400
#define NVREG_MIICTL_ADDRSHIFT	5
	NvRegMIIData = 0x194,
	NvRegWakeUpFlags = 0x200,
#define NVREG_WAKEUPFLAGS_VAL		0x7770
#define NVREG_WAKEUPFLAGS_BUSYSHIFT	24
#define NVREG_WAKEUPFLAGS_ENABLESHIFT	16
#define NVREG_WAKEUPFLAGS_D3SHIFT	12
#define NVREG_WAKEUPFLAGS_D2SHIFT	8
#define NVREG_WAKEUPFLAGS_D1SHIFT	4
#define NVREG_WAKEUPFLAGS_D0SHIFT	0
#define NVREG_WAKEUPFLAGS_ACCEPT_MAGPAT		0x01
#define NVREG_WAKEUPFLAGS_ACCEPT_WAKEUPPAT	0x02
#define NVREG_WAKEUPFLAGS_ACCEPT_LINKCHANGE	0x04
#define NVREG_WAKEUPFLAGS_ENABLE	0x1111

	NvRegPatternCRC = 0x204,
#define NV_UNKNOW_VAL  0x01
	NvRegPatternMask = 0x208,
	NvRegPowerCap = 0x268,
#define NVREG_POWERCAP_D3SUPP	(1<<30)
#define NVREG_POWERCAP_D2SUPP	(1<<26)
#define NVREG_POWERCAP_D1SUPP	(1<<25)
	NvRegPowerState = 0x26c,
#define NVREG_POWERSTATE_POWEREDUP	0x8000
#define NVREG_POWERSTATE_VALID		0x0100
#define NVREG_POWERSTATE_MASK		0x0003
#define NVREG_POWERSTATE_D0		0x0000
#define NVREG_POWERSTATE_D1		0x0001
#define NVREG_POWERSTATE_D2		0x0002
#define NVREG_POWERSTATE_D3		0x0003
	NvRegTxCnt = 0x280,
	NvRegTxZeroReXmt = 0x284,
	NvRegTxOneReXmt = 0x288,
	NvRegTxManyReXmt = 0x28c,
	NvRegTxLateCol = 0x290,
	NvRegTxUnderflow = 0x294,
	NvRegTxLossCarrier = 0x298,
	NvRegTxExcessDef = 0x29c,
	NvRegTxRetryErr = 0x2a0,
	NvRegRxFrameErr = 0x2a4,
	NvRegRxExtraByte = 0x2a8,
	NvRegRxLateCol = 0x2ac,
	NvRegRxRunt = 0x2b0,
	NvRegRxFrameTooLong = 0x2b4,
	NvRegRxOverflow = 0x2b8,
	NvRegRxFCSErr = 0x2bc,
	NvRegRxFrameAlignErr = 0x2c0,
	NvRegRxLenErr = 0x2c4,
	NvRegRxUnicast = 0x2c8,
	NvRegRxMulticast = 0x2cc,
	NvRegRxBroadcast = 0x2d0,
	NvRegTxDef = 0x2d4,
	NvRegTxFrame = 0x2d8,
	NvRegRxCnt = 0x2dc,
	NvRegTxPause = 0x2e0,
	NvRegRxPause = 0x2e4,
	NvRegRxDropFrame = 0x2e8,

	NvRegVlanControl = 0x300,
#define NVREG_VLANCONTROL_ENABLE	0x2000
	NvRegMSIXMap0 = 0x3e0,
	NvRegMSIXMap1 = 0x3e4,
	NvRegMSIXIrqStatus = 0x3f0,

	NvRegPowerState2 = 0x600,
#define NVREG_POWERSTATE2_POWERUP_MASK		0x0F11
#define NVREG_POWERSTATE2_POWERUP_REV_A3	0x0001
};

/* Big endian: should work, but is untested */
struct ring_desc {
	u32 PacketBuffer;
	u32 FlagLen;
};

struct ring_desc_ex {
	u32 PacketBufferHigh;
	u32 PacketBufferLow;
	u32 TxVlan;
	u32 FlagLen;
};

typedef union _ring_type {
	struct ring_desc* orig;
	struct ring_desc_ex* ex;
} ring_type;

#define FLAG_MASK_V1 0xffff0000
#define FLAG_MASK_V2 0xffffc000
#define LEN_MASK_V1 (0xffffffff ^ FLAG_MASK_V1)
#define LEN_MASK_V2 (0xffffffff ^ FLAG_MASK_V2)

#define NV_TX_LASTPACKET	(1<<16)
#define NV_TX_RETRYERROR	(1<<19)
#define NV_TX_FORCED_INTERRUPT	(1<<24)
#define NV_TX_DEFERRED		(1<<26)
#define NV_TX_CARRIERLOST	(1<<27)
#define NV_TX_LATECOLLISION	(1<<28)
#define NV_TX_UNDERFLOW		(1<<29)
#define NV_TX_ERROR		(1<<30) /* logical OR of all errors */
#define NV_TX_VALID		(1<<31)

#define NV_TX2_LASTPACKET	(1<<29)
#define NV_TX2_RETRYERROR	(1<<18)
#define NV_TX2_FORCED_INTERRUPT	(1<<30)
#define NV_TX2_DEFERRED		(1<<25)
#define NV_TX2_CARRIERLOST	(1<<26)
#define NV_TX2_LATECOLLISION	(1<<27)
#define NV_TX2_UNDERFLOW	(1<<28)
/* error and valid are the same for both */
#define NV_TX2_ERROR		(1<<30) /* logical OR of all errors */
#define NV_TX2_VALID		(1<<31)
#define NV_TX2_TSO		(1<<28)
#define NV_TX2_TSO_SHIFT	14
#define NV_TX2_TSO_MAX_SHIFT	14
#define NV_TX2_TSO_MAX_SIZE	(1<<NV_TX2_TSO_MAX_SHIFT)
#define NV_TX2_CHECKSUM_L3	(1<<27)
#define NV_TX2_CHECKSUM_L4	(1<<26)

#define NV_TX3_VLAN_TAG_PRESENT (1<<18)

#define NV_RX_DESCRIPTORVALID	(1<<16)
#define NV_RX_MISSEDFRAME	(1<<17)
#define NV_RX_SUBSTRACT1	(1<<18)
#define NV_RX_ERROR1		(1<<23)
#define NV_RX_ERROR2		(1<<24)
#define NV_RX_ERROR3		(1<<25)
#define NV_RX_ERROR4		(1<<26)
#define NV_RX_CRCERR		(1<<27)
#define NV_RX_OVERFLOW		(1<<28)
#define NV_RX_FRAMINGERR	(1<<29)
#define NV_RX_ERROR		(1<<30) /* logical OR of all errors */
#define NV_RX_AVAIL		(1<<31)

#define NV_RX2_CHECKSUMMASK	(0x1C000000)
#define NV_RX2_CHECKSUMOK1	(0x10000000)
#define NV_RX2_CHECKSUMOK2	(0x14000000)
#define NV_RX2_CHECKSUMOK3	(0x18000000)
#define NV_RX2_DESCRIPTORVALID	(1<<29)
#define NV_RX2_SUBSTRACT1	(1<<25)
#define NV_RX2_ERROR1		(1<<18)
#define NV_RX2_ERROR2		(1<<19)
#define NV_RX2_ERROR3		(1<<20)
#define NV_RX2_ERROR4		(1<<21)
#define NV_RX2_CRCERR		(1<<22)
#define NV_RX2_OVERFLOW		(1<<23)
#define NV_RX2_FRAMINGERR	(1<<24)
/* error and avail are the same for both */
#define NV_RX2_ERROR		(1<<30) /* logical OR of all errors */
#define NV_RX2_AVAIL		(1<<31)

#define NV_RX3_VLAN_TAG_PRESENT (1<<16)
#define NV_RX3_VLAN_TAG_MASK	(0x0000FFFF)

/* Miscelaneous hardware related defines: */
#define NV_PCI_REGSZ_VER1      	0x270
#define NV_PCI_REGSZ_VER2      	0x604

/* various timeout delays: all in usec */
#define NV_TXRX_RESET_DELAY	4
#define NV_TXSTOP_DELAY1	10
#define NV_TXSTOP_DELAY1MAX	500000
#define NV_TXSTOP_DELAY2	100
#define NV_RXSTOP_DELAY1	10
#define NV_RXSTOP_DELAY1MAX	500000
#define NV_RXSTOP_DELAY2	100
#define NV_SETUP5_DELAY		5
#define NV_SETUP5_DELAYMAX	50000
#define NV_POWERUP_DELAY	5
#define NV_POWERUP_DELAYMAX	5000
#define NV_MIIBUSY_DELAY	50
#define NV_MIIPHY_DELAY	10
#define NV_MIIPHY_DELAYMAX	10000
#define NV_MAC_RESET_DELAY	64

#define NV_WAKEUPPATTERNS	5
#define NV_WAKEUPMASKENTRIES	4

/* General driver defaults */
#define NV_WATCHDOG_TIMEO	(5*HZ)

#define RX_RING_DEFAULT		128
#define TX_RING_DEFAULT		64
#define RX_RING_MIN		RX_RING_DEFAULT
#define TX_RING_MIN		TX_RING_DEFAULT
#define RING_MAX_DESC_VER_1	1024
#define RING_MAX_DESC_VER_2_3	16384
/* 
 * Difference between the get and put pointers for the tx ring.
 * This is used to throttle the amount of data outstanding in the
 * tx ring.
 */
#define TX_LIMIT_DIFFERENCE	1

/* rx/tx mac addr + type + vlan + align + slack*/
#define NV_RX_HEADERS		(64)
/* even more slack. */
#define NV_RX_ALLOC_PAD		(64)

/* maximum mtu size */
#define NV_PKTLIMIT_1	ETH_DATA_LEN	/* hard limit not known */
#define NV_PKTLIMIT_2	9100	/* Actual limit according to NVidia: 9202 */

#define OOM_REFILL	(1+HZ/20)
#define POLL_WAIT	(1+HZ/100)
#define LINK_TIMEOUT	(3*HZ)
#define STATS_INTERVAL	(10*HZ)

/* 
 * desc_ver values:
 * The nic supports three different descriptor types:
 * - DESC_VER_1: Original
 * - DESC_VER_2: support for jumbo frames.
 * - DESC_VER_3: 64-bit format.
 */
#define DESC_VER_1	1
#define DESC_VER_2	2
#define DESC_VER_3	3

/* PHY defines */
#define PHY_OUI_MARVELL	0x5043
#define PHY_OUI_CICADA	0x03f1
#define PHY_OUI_VITESSE	0x01c1
#define PHYID1_OUI_MASK	0x03ff
#define PHYID1_OUI_SHFT	6
#define PHYID2_OUI_MASK	0xfc00
#define PHYID2_OUI_SHFT	10
#define PHYID2_MODEL_MASK		0x03f0
#define PHY_MODEL_MARVELL_E3016		0x220
#define PHY_MARVELL_E3016_INITMASK	0x0300
#define PHY_CICADA_INIT1	0x0f000
#define PHY_CICADA_INIT2	0x0e00
#define PHY_CICADA_INIT3	0x01000
#define PHY_CICADA_INIT4	0x0200
#define PHY_CICADA_INIT5	0x0004
#define PHY_CICADA_INIT6	0x02000
#define PHY_VITESSE_INIT_REG1	0x1f
#define PHY_VITESSE_INIT_REG2	0x10
#define PHY_VITESSE_INIT_REG3	0x11
#define PHY_VITESSE_INIT_REG4	0x12
#define PHY_VITESSE_INIT_MSK1	0xc
#define PHY_VITESSE_INIT_MSK2	0x0180
#define PHY_VITESSE_INIT1	0x52b5
#define PHY_VITESSE_INIT2	0xaf8a
#define PHY_VITESSE_INIT3	0x8
#define PHY_VITESSE_INIT4	0x8f8a
#define PHY_VITESSE_INIT5	0xaf86
#define PHY_VITESSE_INIT6	0x8f86
#define PHY_VITESSE_INIT7	0xaf82
#define PHY_VITESSE_INIT8	0x0100
#define PHY_VITESSE_INIT9	0x8f82
#define PHY_VITESSE_INIT10	0x0

#define PHY_GIGABIT	0x0100

#define PHY_TIMEOUT	0x1
#define PHY_ERROR	0x2

#define PHY_100	0x1
#define PHY_1000	0x2
#define PHY_HALF	0x100

#define NV_PAUSEFRAME_RX_CAPABLE 0x0001
#define NV_PAUSEFRAME_TX_CAPABLE 0x0002
#define NV_PAUSEFRAME_RX_ENABLE  0x0004
#define NV_PAUSEFRAME_TX_ENABLE  0x0008
#define NV_PAUSEFRAME_RX_REQ     0x0010
#define NV_PAUSEFRAME_TX_REQ     0x0020
#define NV_PAUSEFRAME_AUTONEG    0x0040

/* MSI/MSI-X defines */
#define NV_MSI_X_MAX_VECTORS  8
#define NV_MSI_X_VECTORS_MASK 0x000f
#define NV_MSI_CAPABLE        0x0010
#define NV_MSI_X_CAPABLE      0x0020
#define NV_MSI_ENABLED        0x0040
#define NV_MSI_X_ENABLED      0x0080

#define NV_MSI_X_VECTOR_ALL   0x0
#define NV_MSI_X_VECTOR_RX    0x0
#define NV_MSI_X_VECTOR_TX    0x1
#define NV_MSI_X_VECTOR_OTHER 0x2

/* statistics */
#define NV_STATS_COUNT_SW 10
   
struct nv_ethtool_str {
	char name[ETH_GSTRING_LEN];
};

static const struct nv_ethtool_str nv_estats_str[] = {
	{ "tx_dropped" },
	{ "tx_fifo_errors" },
	{ "tx_carrier_errors" },
	{ "tx_packets" },
	{ "tx_bytes" },
	{ "rx_crc_errors" },
	{ "rx_over_errors" },
	{ "rx_errors_total" },
	{ "rx_packets" },
	{ "rx_bytes" },
	
	/* hardware counters */
	{ "tx_zero_rexmt" },
	{ "tx_one_rexmt" },
	{ "tx_many_rexmt" },
	{ "tx_late_collision" },
	{ "tx_excess_deferral" },
	{ "tx_retry_error" },
	{ "rx_frame_error" },
	{ "rx_extra_byte" },
	{ "rx_late_collision" },
	{ "rx_runt" },
	{ "rx_frame_too_long" },
	{ "rx_frame_align_error" },
	{ "rx_length_error" },
	{ "rx_unicast" },
	{ "rx_multicast" },
	{ "rx_broadcast" },
	{ "tx_deferral" },
	{ "tx_pause" },
	{ "rx_pause" },
	{ "rx_drop_frame" }
};

struct nv_ethtool_stats {
	u64 tx_dropped;
	u64 tx_fifo_errors;
	u64 tx_carrier_errors;
	u64 tx_packets;
	u64 tx_bytes;
	u64 rx_crc_errors;
	u64 rx_over_errors;
	u64 rx_errors_total;
	u64 rx_packets;
	u64 rx_bytes;

	/* hardware counters */
	u64 tx_zero_rexmt;
	u64 tx_one_rexmt;
	u64 tx_many_rexmt;
	u64 tx_late_collision;
	u64 tx_excess_deferral;
	u64 tx_retry_error;
	u64 rx_frame_error;
	u64 rx_extra_byte;
	u64 rx_late_collision;
	u64 rx_runt;
	u64 rx_frame_too_long;
	u64 rx_frame_align_error;
	u64 rx_length_error;
	u64 rx_unicast;
	u64 rx_multicast;
	u64 rx_broadcast;
	u64 tx_deferral;
	u64 tx_pause;
	u64 rx_pause;
	u64 rx_drop_frame;
};

/* diagnostics */
#define NV_TEST_COUNT_BASE 3
#define NV_TEST_COUNT_EXTENDED 4

static const struct nv_ethtool_str nv_etests_str[] = {
	{ "link      (online/offline)" },
	{ "register  (offline)       " },
	{ "interrupt (offline)       " },
	{ "loopback  (offline)       " }
};

struct register_test {
	u32 reg;
	u32 mask;
};

static const struct register_test nv_registers_test[] = {
	{ NvRegUnknownSetupReg6, 0x01 },
	{ NvRegMisc1, 0x03c },
	{ NvRegOffloadConfig, 0x03ff },
	{ NvRegMulticastAddrA, 0xffffffff },
	{ NvRegTxWatermark, 0x0ff },
	{ NvRegWakeUpFlags, 0x07777 },
	{ 0,0 }
};

struct nv_skb_map {
	struct sk_buff *skb;
	dma_addr_t dma;
	unsigned int dma_len;
};

/*
 * SMP locking:
 * All hardware access under dev->priv->lock, except the performance
 * critical parts:
 * - rx is (pseudo-) lockless: it relies on the single-threading provided
 *	by the arch code for interrupts.
 * - tx setup is lockless: it relies on dev->xmit_lock. Actual submission
 *	needs dev->priv->lock :-(
 * - set_multicast_list: preparation lockless, relies on dev->xmit_lock.
 */

/* in dev: base, irq */
struct fe_priv {

	/* fields used in fast path are grouped together 
	   for better cache performance
	*/
	spinlock_t lock;
	void __iomem *base;
	struct pci_dev *pci_dev;
	u32 txrxctl_bits;
	int stop_tx;
	int need_linktimer;
	unsigned long link_timeout;
	u32 irqmask;
	u32 msi_flags;

	unsigned int rx_buf_sz;
	struct vlan_group *vlangrp;
	int tx_ring_size;
	int rx_csum;

	/*
	 * rx specific fields in fast path
	 */
	ring_type get_rx __attribute__((aligned(L1_CACHE_BYTES)));
	ring_type put_rx, first_rx, last_rx;
	struct nv_skb_map *get_rx_ctx, *put_rx_ctx;
	struct nv_skb_map *first_rx_ctx, *last_rx_ctx;
	
	/*
	 * tx specific fields in fast path
	 */
	ring_type get_tx __attribute__((aligned(L1_CACHE_BYTES)));
	ring_type put_tx, first_tx, last_tx;
	struct nv_skb_map *get_tx_ctx, *put_tx_ctx;
	struct nv_skb_map *first_tx_ctx, *last_tx_ctx;

	struct nv_skb_map *rx_skb;
	struct nv_skb_map *tx_skb;

	/* General data:
	 * Locking: spin_lock(&np->lock); */
	struct net_device_stats stats;
	struct nv_ethtool_stats estats;
	int in_shutdown;
	u32 linkspeed;
	int duplex;
	int autoneg;
	int fixed_mode;
	int phyaddr;
	int wolenabled;
	unsigned int phy_oui;
	unsigned int phy_model;
	u16 gigabit;
	int intr_test;
	int recover_error;

	/* General data: RO fields */
	dma_addr_t ring_addr;
	u32 orig_mac[2];
	u32 desc_ver;
	u32 vlanctl_bits;
	u32 driver_data;
	u32 register_size;
	u32 mac_in_use;

	/* rx specific fields.
	 * Locking: Within irq hander or disable_irq+spin_lock(&np->lock);
	 */
	ring_type rx_ring;
	unsigned int pkt_limit;
	struct timer_list oom_kick;
	struct timer_list nic_poll;
	struct timer_list stats_poll;
	u32 nic_poll_irq;
	int rx_ring_size;
	u32 rx_len_errors;
	/*
	 * tx specific fields.
	 */
	ring_type tx_ring;
	u32 tx_flags;
	int tx_limit_start;
	int tx_limit_stop;


	/* msi/msi-x fields */
	struct msix_entry msi_x_entry[NV_MSI_X_MAX_VECTORS];

	/* flow control */
	u32 pause_flags;
};

#ifdef __VMKERNEL_MODULE__
/*
 * Flag to enable the timer irq so that we can prevent the wathdog
 * from firing due to missing TX done IRQ's. See "Known bugs:" above.
 */
static int vmk_enable_timer_irq = 0;
static int count = 0;
MODULE_PARM(vmk_enable_timer_irq, "i");
MODULE_PARM_DESC(vmk_enable_timer_irq, "VMWare enable the timer irq on select hardware in order to prevent missing TX done irq from setting off watchdog.");
#endif

/*
 * Maximum number of loops until we assume that a bit in the irq mask
 * is stuck. Overridable with module param.
 */
static int max_interrupt_work = 5;

/*
 * Optimization can be either throuput mode or cpu mode
 * 
 * Throughput Mode: Every tx and rx packet will generate an interrupt.
 * CPU Mode: Interrupts are controlled by a timer.
 */
enum {
	NV_OPTIMIZATION_MODE_THROUGHPUT, 
	NV_OPTIMIZATION_MODE_CPU
};

static int optimization_mode = NV_OPTIMIZATION_MODE_THROUGHPUT;

/*
 * Poll interval for timer irq
 *
 * This interval determines how frequent an interrupt is generated.
 * The is value is determined by [(time_in_micro_secs * 100) / (2^10)]
 * Min = 0, and Max = 65535
 */
static int poll_interval = -1;

/*
 * MSI interrupts
 */
enum {
	NV_MSI_INT_DISABLED,
	NV_MSI_INT_ENABLED
};

#ifdef CONFIG_PCI_MSI 
static int msi = NV_MSI_INT_ENABLED;
#else
static int msi = NV_MSI_INT_DISABLED;
#endif

/*
 * MSIX interrupts
 */
enum {
	NV_MSIX_INT_DISABLED, 
	NV_MSIX_INT_ENABLED
};

#ifdef CONFIG_PCI_MSI 
static int msix = NV_MSIX_INT_ENABLED;
#else
static int msix = NV_MSIX_INT_DISABLED;
#endif
/*
 * PHY Speed and Duplex
 */
enum {
	NV_SPEED_DUPLEX_AUTO,
	NV_SPEED_DUPLEX_10_HALF_DUPLEX,
	NV_SPEED_DUPLEX_10_FULL_DUPLEX,
	NV_SPEED_DUPLEX_100_HALF_DUPLEX,
	NV_SPEED_DUPLEX_100_FULL_DUPLEX,
	NV_SPEED_DUPLEX_1000_FULL_DUPLEX
};
static int speed_duplex = NV_SPEED_DUPLEX_AUTO;

/*
 * PHY autonegotiation
 */
static int autoneg = AUTONEG_ENABLE;

/*
 * Scatter gather
 */
enum {
	NV_SCATTER_GATHER_DISABLED,
	NV_SCATTER_GATHER_ENABLED
};
static int scatter_gather = NV_SCATTER_GATHER_ENABLED;

/*
 * TCP Segmentation Offload (TSO)
 */
enum {
	NV_TSO_DISABLED,
	NV_TSO_ENABLED
};
static int tso_offload = NV_TSO_ENABLED;

/*
 * MTU settings
 */
static int mtu = ETH_DATA_LEN;

/*
 * Tx checksum offload
 */
enum {
	NV_TX_CHECKSUM_DISABLED, 
	NV_TX_CHECKSUM_ENABLED 
};
static int tx_checksum_offload = NV_TX_CHECKSUM_ENABLED;

/*
 * Rx checksum offload
 */
enum {
	NV_RX_CHECKSUM_DISABLED, 
	NV_RX_CHECKSUM_ENABLED 
};
static int rx_checksum_offload = NV_RX_CHECKSUM_ENABLED;

/*
 * Tx ring size
 */
static int tx_ring_size = TX_RING_DEFAULT;

/*
 * Rx ring size
 */
static int rx_ring_size = RX_RING_DEFAULT;

/*
 * Tx flow control
 */
enum {
	NV_TX_FLOW_CONTROL_DISABLED, 
	NV_TX_FLOW_CONTROL_ENABLED
};
static int tx_flow_control = NV_TX_FLOW_CONTROL_ENABLED;

/*
 * Rx flow control
 */
enum {
	NV_RX_FLOW_CONTROL_DISABLED, 
	NV_RX_FLOW_CONTROL_ENABLED
};
static int rx_flow_control = NV_RX_FLOW_CONTROL_ENABLED;

/*
 * DMA 64bit
 */
enum {
	NV_DMA_64BIT_DISABLED,
	NV_DMA_64BIT_ENABLED
};
static int dma_64bit = NV_DMA_64BIT_ENABLED;

/*
 * Wake On Lan
 */
enum {
	NV_WOL_DISABLED,
	NV_WOL_ENABLED
};
static int wol = NV_WOL_DISABLED;

/*
 * Tagging 802.1pq
 */
enum {
	NV_8021PQ_DISABLED,
	NV_8021PQ_ENABLED
};

static int tagging_8021pq = NV_8021PQ_ENABLED;

#if NVVER < RHES4
static inline unsigned long nv_msecs_to_jiffies(const unsigned int m)
{
#if HZ <= 1000 && !(1000 % HZ)
        return (m + (1000 / HZ) - 1) / (1000 / HZ);
#elif HZ > 1000 && !(HZ % 1000)
        return m * (HZ / 1000);
#else
        return (m * HZ + 999) / 1000;
#endif
}
#endif

static void nv_msleep(unsigned int msecs)
{
#if NVVER > SLES9 
	msleep(msecs);
#else
	unsigned long timeout = nv_msecs_to_jiffies(msecs);

	while (timeout) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		timeout = schedule_timeout(timeout);
	}
#endif
}

static inline struct fe_priv *get_nvpriv(struct net_device *dev)
{
#if NVVER > RHES3 
	return netdev_priv(dev);
#else
	return (struct fe_priv *) dev->priv;
#endif
}

static void __init quirk_nforce_network_class(struct pci_dev *pdev)
{
	/* Some implementations of the nVidia network controllers
	 * show up as bridges, when we need to see them as network
	 * devices.
	 */

	/* If this is already known as a network ctlr, do nothing. */
	if ((pdev->class >> 8) == PCI_CLASS_NETWORK_ETHERNET)
		return;

	if ((pdev->class >> 8) == PCI_CLASS_BRIDGE_OTHER) {
		char    c;

		/* Clearing bit 6 of the register at 0xf8
		 * selects Ethernet device class
		 */
		pci_read_config_byte(pdev, 0xf8, &c);
		c &= 0xbf;
		pci_write_config_byte(pdev, 0xf8, c);

		/* sysfs needs pdev->class to be set correctly */
		pdev->class &= 0x0000ff;
		pdev->class |= (PCI_CLASS_NETWORK_ETHERNET << 8);
	}
}

static inline u8 __iomem *get_hwbase(struct net_device *dev)
{
	return ((struct fe_priv *)get_nvpriv(dev))->base;
}

static inline void pci_push(u8 __iomem *base)
{
	/* force out pending posted writes */
	readl(base);
}

static inline u32 nv_descr_getlength(struct ring_desc *prd, u32 v)
{
	return le32_to_cpu(prd->FlagLen)
		& ((v == DESC_VER_1) ? LEN_MASK_V1 : LEN_MASK_V2);
}

static inline u32 nv_descr_getlength_ex(struct ring_desc_ex *prd, u32 v)
{
	return le32_to_cpu(prd->FlagLen) & LEN_MASK_V2;
}

static int reg_delay(struct net_device *dev, int offset, u32 mask, u32 target,
				int delay, int delaymax, const char *msg)
{
	u8 __iomem *base = get_hwbase(dev);

	pci_push(base);
	do {
		udelay(delay);
		delaymax -= delay;
		if (delaymax < 0) {
			if (msg)
				printk(msg);
			return 1;
		}
	} while ((readl(base + offset) & mask) != target);
	return 0;
}

#define NV_SETUP_RX_RING 0x01
#define NV_SETUP_TX_RING 0x02

static void setup_hw_rings(struct net_device *dev, int rxtx_flags)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);

	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
		if (rxtx_flags & NV_SETUP_RX_RING) {
			writel((u32) cpu_to_le64(np->ring_addr), base + NvRegRxRingPhysAddr);
		}
		if (rxtx_flags & NV_SETUP_TX_RING) {
			writel((u32) cpu_to_le64(np->ring_addr + np->rx_ring_size*sizeof(struct ring_desc)), base + NvRegTxRingPhysAddr);
		}
	} else {
		if (rxtx_flags & NV_SETUP_RX_RING) {
			writel((u32) cpu_to_le64(np->ring_addr), base + NvRegRxRingPhysAddr);
			writel((u32) (cpu_to_le64(np->ring_addr) >> 32), base + NvRegRxRingPhysAddrHigh);
		}
		if (rxtx_flags & NV_SETUP_TX_RING) {
			writel((u32) cpu_to_le64(np->ring_addr + np->rx_ring_size*sizeof(struct ring_desc_ex)), base + NvRegTxRingPhysAddr);
			writel((u32) (cpu_to_le64(np->ring_addr + np->rx_ring_size*sizeof(struct ring_desc_ex)) >> 32), base + NvRegTxRingPhysAddrHigh);
		}
	}
}

static void free_rings(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
		if(np->rx_ring.orig)
			pci_free_consistent(np->pci_dev, sizeof(struct ring_desc) * (np->rx_ring_size + np->tx_ring_size),
					    np->rx_ring.orig, np->ring_addr);
	} else {
		if (np->rx_ring.ex)
			pci_free_consistent(np->pci_dev, sizeof(struct ring_desc_ex) * (np->rx_ring_size + np->tx_ring_size),
					    np->rx_ring.ex, np->ring_addr);
	}
	if (np->rx_skb)
		kfree(np->rx_skb);
	if (np->tx_skb)
		kfree(np->tx_skb);	
}

static int using_multi_irqs(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (!(np->msi_flags & NV_MSI_X_ENABLED) ||
	    ((np->msi_flags & NV_MSI_X_ENABLED) && 
	     ((np->msi_flags & NV_MSI_X_VECTORS_MASK) == 0x1)))
		return 0;
	else
		return 1;
}

static void nv_enable_irq(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (!using_multi_irqs(dev)) {
		if (np->msi_flags & NV_MSI_X_ENABLED)
			enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector);
		else
			enable_irq(dev->irq);
	} else {
		enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector);
		enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector);
		enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_OTHER].vector);
	}
}

static void nv_disable_irq(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (!using_multi_irqs(dev)) {
		if (np->msi_flags & NV_MSI_X_ENABLED)
			disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector);
		else
			disable_irq(dev->irq);
	} else {
		disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector);
		disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector);
		disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_OTHER].vector);
	}
}

/* In MSIX mode, a write to irqmask behaves as XOR */
static void nv_enable_hw_interrupts(struct net_device *dev, u32 mask)
{
	u8 __iomem *base = get_hwbase(dev);

	writel(mask, base + NvRegIrqMask);
}

static void nv_disable_hw_interrupts(struct net_device *dev, u32 mask)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);

	if (np->msi_flags & NV_MSI_X_ENABLED) {
		writel(mask, base + NvRegIrqMask);
	} else {
		if (np->msi_flags & NV_MSI_ENABLED)
			writel(0, base + NvRegMSIIrqMask);
		writel(0, base + NvRegIrqMask);
	}
}

#define MII_READ	(-1)
/* mii_rw: read/write a register on the PHY.
 *
 * Caller must guarantee serialization
 */
static int mii_rw(struct net_device *dev, int addr, int miireg, int value)
{
	u8 __iomem *base = get_hwbase(dev);
	u32 reg;
	int retval;

	writel(NVREG_MIISTAT_MASK, base + NvRegMIIStatus);

	reg = readl(base + NvRegMIIControl);
	if (reg & NVREG_MIICTL_INUSE) {
		writel(NVREG_MIICTL_INUSE, base + NvRegMIIControl);
		udelay(NV_MIIBUSY_DELAY);
	}

	reg = (addr << NVREG_MIICTL_ADDRSHIFT) | miireg;
	if (value != MII_READ) {
		writel(value, base + NvRegMIIData);
		reg |= NVREG_MIICTL_WRITE;
	}
	writel(reg, base + NvRegMIIControl);

	if (reg_delay(dev, NvRegMIIControl, NVREG_MIICTL_INUSE, 0,
			NV_MIIPHY_DELAY, NV_MIIPHY_DELAYMAX, NULL)) {
		dprintk(KERN_DEBUG "%s: mii_rw of reg %d at PHY %d timed out.\n",
				dev->name, miireg, addr);
		retval = -1;
	} else if (value != MII_READ) {
		/* it was a write operation - fewer failures are detectable */
		dprintk(KERN_DEBUG "%s: mii_rw wrote 0x%x to reg %d at PHY %d\n",
				dev->name, value, miireg, addr);
		retval = 0;
	} else if (readl(base + NvRegMIIStatus) & NVREG_MIISTAT_ERROR) {
		dprintk(KERN_DEBUG "%s: mii_rw of reg %d at PHY %d failed.\n",
				dev->name, miireg, addr);
		retval = -1;
	} else {
		retval = readl(base + NvRegMIIData);
		dprintk(KERN_DEBUG "%s: mii_rw read from reg %d at PHY %d: 0x%x.\n",
				dev->name, miireg, addr, retval);
	}

	return retval;
}

static int phy_reset(struct net_device *dev, u32 bmcr_setup)
{
	struct fe_priv *np = get_nvpriv(dev);
	u32 miicontrol;
	unsigned int tries = 0;

	miicontrol = BMCR_RESET | bmcr_setup;
	if (mii_rw(dev, np->phyaddr, MII_BMCR, miicontrol)) {
		return -1;
	}

	/* wait for 500ms */
	nv_msleep(500);

	/* must wait till reset is deasserted */
	while (miicontrol & BMCR_RESET) {
		nv_msleep(10);
		miicontrol = mii_rw(dev, np->phyaddr, MII_BMCR, MII_READ);
		/* FIXME: 100 tries seem excessive */
		if (tries++ > 100)
			return -1;
	}
	return 0;
}

static int phy_init(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 phyinterface, phy_reserved, mii_status, mii_control, mii_control_1000,reg;

	/* phy errata for E3016 phy */
	if (np->phy_model == PHY_MODEL_MARVELL_E3016) {
		reg = mii_rw(dev, np->phyaddr, MII_NCONFIG, MII_READ);
		reg &= ~PHY_MARVELL_E3016_INITMASK;
		if (mii_rw(dev, np->phyaddr, MII_NCONFIG, reg)) {
			printk(KERN_INFO "%s: phy write to errata reg failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}
	}

	/* set advertise register */
	reg = mii_rw(dev, np->phyaddr, MII_ADVERTISE, MII_READ);
	reg &= ~(ADVERTISE_ALL | ADVERTISE_100BASE4 | ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM);
	if (speed_duplex == NV_SPEED_DUPLEX_AUTO)
		reg |= (ADVERTISE_10HALF|ADVERTISE_10FULL|ADVERTISE_100HALF|ADVERTISE_100FULL);
	if (speed_duplex == NV_SPEED_DUPLEX_10_HALF_DUPLEX)
		reg |= ADVERTISE_10HALF;
	if (speed_duplex == NV_SPEED_DUPLEX_10_FULL_DUPLEX)
		reg |= ADVERTISE_10FULL;
	if (speed_duplex == NV_SPEED_DUPLEX_100_HALF_DUPLEX)
		reg |= ADVERTISE_100HALF;
	if (speed_duplex == NV_SPEED_DUPLEX_100_FULL_DUPLEX)
		reg |= ADVERTISE_100FULL;
	if (np->pause_flags & NV_PAUSEFRAME_RX_REQ) /* for rx we set both advertisments but disable tx pause */
		reg |= ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM;
	if (np->pause_flags & NV_PAUSEFRAME_TX_REQ)
		reg |= ADVERTISE_PAUSE_ASYM;
	np->fixed_mode = reg;

	if (mii_rw(dev, np->phyaddr, MII_ADVERTISE, reg)) {
		printk(KERN_INFO "%s: phy write to advertise failed.\n", pci_name(np->pci_dev));
		return PHY_ERROR;
	}

	/* get phy interface type */
	phyinterface = readl(base + NvRegPhyInterface);

	/* see if gigabit phy */
	mii_status = mii_rw(dev, np->phyaddr, MII_BMSR, MII_READ);
	if (mii_status & PHY_GIGABIT) {
		np->gigabit = PHY_GIGABIT;
		mii_control_1000 = mii_rw(dev, np->phyaddr, MII_CTRL1000, MII_READ);
		mii_control_1000 &= ~ADVERTISE_1000HALF;
		if (phyinterface & PHY_RGMII && 
		    (speed_duplex == NV_SPEED_DUPLEX_AUTO || 
		     (speed_duplex == NV_SPEED_DUPLEX_1000_FULL_DUPLEX && autoneg == AUTONEG_ENABLE)))
			mii_control_1000 |= ADVERTISE_1000FULL;
		else {
			if (speed_duplex == NV_SPEED_DUPLEX_1000_FULL_DUPLEX && autoneg == AUTONEG_DISABLE)
				printk(KERN_INFO "%s: 1000mpbs full only allowed with autoneg\n", pci_name(np->pci_dev));
			mii_control_1000 &= ~ADVERTISE_1000FULL;
		}
		if (mii_rw(dev, np->phyaddr, MII_CTRL1000, mii_control_1000)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}
	}
	else
		np->gigabit = 0;

	mii_control = mii_rw(dev, np->phyaddr, MII_BMCR, MII_READ);
	if (autoneg == AUTONEG_DISABLE){
		np->pause_flags &= ~(NV_PAUSEFRAME_RX_ENABLE | NV_PAUSEFRAME_TX_ENABLE);
		if (np->pause_flags & NV_PAUSEFRAME_RX_REQ)
			np->pause_flags |= NV_PAUSEFRAME_RX_ENABLE;
		if (np->pause_flags & NV_PAUSEFRAME_TX_REQ)
			np->pause_flags |= NV_PAUSEFRAME_TX_ENABLE;
		mii_control &= ~(BMCR_ANENABLE|BMCR_SPEED100|BMCR_SPEED1000|BMCR_FULLDPLX);
		if (reg & (ADVERTISE_10FULL|ADVERTISE_100FULL))
			mii_control |= BMCR_FULLDPLX;
		if (reg & (ADVERTISE_100HALF|ADVERTISE_100FULL))
			mii_control |= BMCR_SPEED100;
	} else {
		mii_control |= BMCR_ANENABLE;
	}

	/* reset the phy and setup BMCR 
	 * (certain phys need reset at same time new values are set) */
	if (phy_reset(dev, mii_control)) {
		printk(KERN_INFO "%s: phy reset failed\n", pci_name(np->pci_dev));
		return PHY_ERROR;
	}

	/* phy vendor specific configuration */
	if ((np->phy_oui == PHY_OUI_CICADA) && (phyinterface & PHY_RGMII) ) {
		phy_reserved = mii_rw(dev, np->phyaddr, MII_RESV1, MII_READ);
		phy_reserved &= ~(PHY_CICADA_INIT1 | PHY_CICADA_INIT2);
		phy_reserved |= (PHY_CICADA_INIT3 | PHY_CICADA_INIT4);
		if (mii_rw(dev, np->phyaddr, MII_RESV1, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}
		phy_reserved = mii_rw(dev, np->phyaddr, MII_NCONFIG, MII_READ);
		phy_reserved |= PHY_CICADA_INIT5;
		if (mii_rw(dev, np->phyaddr, MII_NCONFIG, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}
	}
	if (np->phy_oui == PHY_OUI_CICADA) {
		phy_reserved = mii_rw(dev, np->phyaddr, MII_SREVISION, MII_READ);
		phy_reserved |= PHY_CICADA_INIT6;
		if (mii_rw(dev, np->phyaddr, MII_SREVISION, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}
	}
	if (np->phy_oui == PHY_OUI_VITESSE) {
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG1, PHY_VITESSE_INIT1)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG2, PHY_VITESSE_INIT2)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		phy_reserved = mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG4, MII_READ);
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG4, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		phy_reserved = mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG3, MII_READ);
		phy_reserved &= ~PHY_VITESSE_INIT_MSK1;
		phy_reserved |= PHY_VITESSE_INIT3;
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG3, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG2, PHY_VITESSE_INIT4)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG2, PHY_VITESSE_INIT5)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		phy_reserved = mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG4, MII_READ);
		phy_reserved &= ~PHY_VITESSE_INIT_MSK1;
		phy_reserved |= PHY_VITESSE_INIT3;
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG4, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		phy_reserved = mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG3, MII_READ);
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG3, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG2, PHY_VITESSE_INIT6)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG2, PHY_VITESSE_INIT7)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		phy_reserved = mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG4, MII_READ);
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG4, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		phy_reserved = mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG3, MII_READ);
		phy_reserved &= ~PHY_VITESSE_INIT_MSK2;
		phy_reserved |= PHY_VITESSE_INIT8;
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG3, phy_reserved)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG2, PHY_VITESSE_INIT9)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
		if (mii_rw(dev, np->phyaddr, PHY_VITESSE_INIT_REG1, PHY_VITESSE_INIT10)) {
			printk(KERN_INFO "%s: phy init failed.\n", pci_name(np->pci_dev));
			return PHY_ERROR;
		}		
	}
	/* some phys clear out pause advertisment on reset, set it back */
	mii_rw(dev, np->phyaddr, MII_ADVERTISE, reg);

	/* restart auto negotiation */
	if (autoneg == AUTONEG_ENABLE) {
		mii_control = mii_rw(dev, np->phyaddr, MII_BMCR, MII_READ);
		mii_control |= (BMCR_ANRESTART | BMCR_ANENABLE);
		if (mii_rw(dev, np->phyaddr, MII_BMCR, mii_control)) {
			return PHY_ERROR;
		}
	}

	return 0;
}

static void nv_start_rx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 rx_ctrl = readl(base + NvRegReceiverControl);

	dprintk(KERN_DEBUG "%s: nv_start_rx\n", dev->name);
	/* Already running? Stop it. */
	if ((readl(base + NvRegReceiverControl) & NVREG_RCVCTL_START) && !np->mac_in_use) {
		rx_ctrl &= ~NVREG_RCVCTL_START;
		writel(rx_ctrl, base + NvRegReceiverControl);
		pci_push(base);
	}
	writel(np->linkspeed, base + NvRegLinkSpeed);
	pci_push(base);
        rx_ctrl |= NVREG_RCVCTL_START;
        if (np->mac_in_use)
		rx_ctrl &= ~NVREG_RCVCTL_RX_PATH_EN;
	writel(rx_ctrl, base + NvRegReceiverControl);
	dprintk(KERN_DEBUG "%s: nv_start_rx to duplex %d, speed 0x%08x.\n",
				dev->name, np->duplex, np->linkspeed);
	pci_push(base);
}

static void nv_stop_rx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 rx_ctrl = readl(base + NvRegReceiverControl);

	dprintk(KERN_DEBUG "%s: nv_stop_rx\n", dev->name);
	if (!np->mac_in_use)
		rx_ctrl &= ~NVREG_RCVCTL_START;
	else
		rx_ctrl |= NVREG_RCVCTL_RX_PATH_EN;
	writel(rx_ctrl, base + NvRegReceiverControl);
	reg_delay(dev, NvRegReceiverStatus, NVREG_RCVSTAT_BUSY, 0,
			NV_RXSTOP_DELAY1, NV_RXSTOP_DELAY1MAX,
			KERN_INFO "nv_stop_rx: ReceiverStatus remained busy");

	udelay(NV_RXSTOP_DELAY2);
	if (!np->mac_in_use)
	writel(0, base + NvRegLinkSpeed);
}

static void nv_start_tx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 tx_ctrl = readl(base + NvRegTransmitterControl);

	dprintk(KERN_DEBUG "%s: nv_start_tx\n", dev->name);
	tx_ctrl |= NVREG_XMITCTL_START;
	if (np->mac_in_use)
		tx_ctrl &= ~NVREG_XMITCTL_TX_PATH_EN;
	writel(tx_ctrl, base + NvRegTransmitterControl);
	pci_push(base);
}

static void nv_stop_tx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 tx_ctrl = readl(base + NvRegTransmitterControl);

	dprintk(KERN_DEBUG "%s: nv_stop_tx\n", dev->name);
	if (!np->mac_in_use)
		tx_ctrl &= ~NVREG_XMITCTL_START;
	else
		tx_ctrl |= NVREG_XMITCTL_TX_PATH_EN;
	writel(tx_ctrl, base + NvRegTransmitterControl);
	reg_delay(dev, NvRegTransmitterStatus, NVREG_XMITSTAT_BUSY, 0,
			NV_TXSTOP_DELAY1, NV_TXSTOP_DELAY1MAX,
			KERN_INFO "nv_stop_tx: TransmitterStatus remained busy");

	udelay(NV_TXSTOP_DELAY2);
	if (!np->mac_in_use)
		writel(readl(base + NvRegTransmitPoll) & NVREG_TRANSMITPOLL_MAC_ADDR_REV, base + NvRegTransmitPoll);
}

static void nv_txrx_reset(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);

	dprintk(KERN_DEBUG "%s: nv_txrx_reset\n", dev->name);
	writel(NVREG_TXRXCTL_BIT2 | NVREG_TXRXCTL_RESET | np->txrxctl_bits, base + NvRegTxRxControl);
	pci_push(base);
	udelay(NV_TXRX_RESET_DELAY);
}

static void nv_mac_reset(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);

	dprintk(KERN_DEBUG "%s: nv_mac_reset\n", dev->name);
	writel(NVREG_TXRXCTL_BIT2 | NVREG_TXRXCTL_RESET | np->txrxctl_bits, base + NvRegTxRxControl);
	pci_push(base);
	writel(NVREG_MAC_RESET_ASSERT, base + NvRegMacReset);
	pci_push(base);
	udelay(NV_MAC_RESET_DELAY);
	writel(0, base + NvRegMacReset);
	pci_push(base);
	udelay(NV_MAC_RESET_DELAY);
	writel(NVREG_TXRXCTL_BIT2 | np->txrxctl_bits, base + NvRegTxRxControl);
	pci_push(base);
}

#if NVVER < SLES9
static int nv_ethtool_ioctl(struct net_device *dev, void *useraddr)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 *base = get_hwbase(dev);
	u32 ethcmd;

	if (copy_from_user(&ethcmd, useraddr, sizeof (ethcmd)))
		return -EFAULT;

	switch (ethcmd) {
		case ETHTOOL_GDRVINFO:
			{
				struct ethtool_drvinfo info = { ETHTOOL_GDRVINFO };
				strcpy(info.driver, "forcedeth");
				strcpy(info.version, FORCEDETH_VERSION);
				strcpy(info.bus_info, pci_name(np->pci_dev));
				if (copy_to_user(useraddr, &info, sizeof (info)))
					return -EFAULT;
				return 0;
			}
		case ETHTOOL_GLINK:
			{
				struct ethtool_value edata = { ETHTOOL_GLINK };

				edata.data = !!netif_carrier_ok(dev);

				if (copy_to_user(useraddr, &edata, sizeof(edata)))
					return -EFAULT;
				return 0;
			}
		case ETHTOOL_GWOL:
			{
				struct ethtool_wolinfo wolinfo;
				memset(&wolinfo, 0, sizeof(wolinfo));
				wolinfo.supported = WAKE_MAGIC;

				spin_lock_irq(&np->lock);
				if (np->wolenabled)
					wolinfo.wolopts = WAKE_MAGIC;
				spin_unlock_irq(&np->lock);

				if (copy_to_user(useraddr, &wolinfo, sizeof(wolinfo)))
					return -EFAULT;
				return 0;
			}
		case ETHTOOL_SWOL:
			{
				struct ethtool_wolinfo wolinfo;
				if (copy_from_user(&wolinfo, useraddr, sizeof(wolinfo)))
					return -EFAULT;

				spin_lock_irq(&np->lock);
				if (wolinfo.wolopts == 0) {
					writel(0, base + NvRegWakeUpFlags);
					np->wolenabled = NV_WOL_DISABLED;
				}
				if (wolinfo.wolopts & WAKE_MAGIC) {
					writel(NVREG_WAKEUPFLAGS_ENABLE, base + NvRegWakeUpFlags);
					np->wolenabled = NV_WOL_ENABLED;
				}
				spin_unlock_irq(&np->lock);
				return 0;
			}

		default:
			break;
	}

	return -EOPNOTSUPP;
}

/*
 * nv_ioctl: dev->do_ioctl function
 * Called with rtnl_lock held.
 */
static int nv_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	switch(cmd) {
		case SIOCETHTOOL:
			return nv_ethtool_ioctl(dev, rq->ifr_data);

		default:
			return -EOPNOTSUPP;
	}
}
#endif

/*
 * nv_alloc_rx: fill rx ring entries.
 * Return 1 if the allocations for the skbs failed and the
 * rx engine is without Available descriptors
 */
static inline int nv_alloc_rx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	struct ring_desc* less_rx;
	struct sk_buff *skb;

	less_rx = np->get_rx.orig;
	if (less_rx-- == np->first_rx.orig)
		less_rx = np->last_rx.orig;

	while (np->put_rx.orig != less_rx) {
#ifdef __VMKERNEL_MODULE__
		/* dev_alloc_skb will alloc 16 extra bytes, which we already do 
		 * in vmklinux do_alloc_skb, so it is redundant here.
		 */
		skb = alloc_skb(np->rx_buf_sz + NV_RX_ALLOC_PAD, GFP_ATOMIC);
#else
		skb = dev_alloc_skb(np->rx_buf_sz + NV_RX_ALLOC_PAD);
#endif // __VMKERNEL_MODULE__
		if (skb) {
			skb->dev = dev;
			np->put_rx_ctx->skb = skb;
#ifdef __VMKERNEL_MODULE__
			np->put_rx_ctx->dma = skb->headMA;
#else
			np->put_rx_ctx->dma = pci_map_single(np->pci_dev, skb->data,
							     skb->end-skb->data, PCI_DMA_FROMDEVICE);
#endif // __VMKERNEL_MODULE__
			np->put_rx_ctx->dma_len = skb->end-skb->data;
			np->put_rx.orig->PacketBuffer = cpu_to_le32(np->put_rx_ctx->dma);
			wmb();
			np->put_rx.orig->FlagLen = cpu_to_le32(np->rx_buf_sz | NV_RX_AVAIL);
			if (unlikely(np->put_rx.orig++ == np->last_rx.orig))
				np->put_rx.orig = np->first_rx.orig;
			if (unlikely(np->put_rx_ctx++ == np->last_rx_ctx))
				np->put_rx_ctx = np->first_rx_ctx;
		} else {
			return 1;
		}
	}
	return 0;
}

static inline int nv_alloc_rx_optimized(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	struct ring_desc_ex* less_rx;
	struct sk_buff *skb;

	less_rx = np->get_rx.ex;
	if (less_rx-- == np->first_rx.ex)
		less_rx = np->last_rx.ex;

	while (np->put_rx.ex != less_rx) {
#ifdef __VMKERNEL_MODULE__
		/* dev_alloc_skb will alloc 16 extra bytes, which we already do 
		 * in vmklinux do_alloc_skb, so it is redundant here.
		 */
		skb = alloc_skb(np->rx_buf_sz + NV_RX_ALLOC_PAD, GFP_ATOMIC);
#else
		skb = dev_alloc_skb(np->rx_buf_sz + NV_RX_ALLOC_PAD);
#endif // __VMKERNEL_MODULE__
		if (skb) {
			skb->dev = dev;
			np->put_rx_ctx->skb = skb;
#ifdef __VMKERNEL_MODULE__
			np->put_rx_ctx->dma = skb->headMA;
#else
			np->put_rx_ctx->dma = pci_map_single(np->pci_dev, skb->data,
							     skb->end-skb->data, PCI_DMA_FROMDEVICE);
#endif // __VMKERNEL_MODULE__
			np->put_rx_ctx->dma_len = skb->end-skb->data;
			np->put_rx.ex->PacketBufferHigh = cpu_to_le64(np->put_rx_ctx->dma) >> 32;
			np->put_rx.ex->PacketBufferLow = cpu_to_le64(np->put_rx_ctx->dma) & 0x0FFFFFFFF;			
			wmb();
			np->put_rx.ex->FlagLen = cpu_to_le32(np->rx_buf_sz | NV_RX2_AVAIL);
			if (unlikely(np->put_rx.ex++ == np->last_rx.ex))
				np->put_rx.ex = np->first_rx.ex;
			if (unlikely(np->put_rx_ctx++ == np->last_rx_ctx))
				np->put_rx_ctx = np->first_rx_ctx;
		} else {
			return 1;
		}
	}
	return 0;

}

static void nv_do_rx_refill(unsigned long data)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	int retcode;

	if (!using_multi_irqs(dev)) {
		if (np->msi_flags & NV_MSI_X_ENABLED)
			disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector);
		else
			disable_irq(dev->irq);
	} else {
		disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector);
	}

	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
		retcode = nv_alloc_rx(dev);
	else
		retcode = nv_alloc_rx_optimized(dev);
	if (retcode) {
		spin_lock_irq(&np->lock);
		if (!np->in_shutdown)
			mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
		spin_unlock_irq(&np->lock);
	}
	if (!using_multi_irqs(dev)) {
		if (np->msi_flags & NV_MSI_X_ENABLED)
			enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector);
		else
			enable_irq(dev->irq);
	} else {
		enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector);
	}
}

static void nv_init_rx(struct net_device *dev) 
{
	struct fe_priv *np = get_nvpriv(dev);
	int i;

	np->get_rx = np->put_rx = np->first_rx = np->rx_ring;
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
		np->last_rx.orig = &np->rx_ring.orig[np->rx_ring_size-1];
	else
		np->last_rx.ex = &np->rx_ring.ex[np->rx_ring_size-1];
	np->get_rx_ctx = np->put_rx_ctx = np->first_rx_ctx = np->rx_skb;
	np->last_rx_ctx = &np->rx_skb[np->rx_ring_size-1];

	for (i = 0; i < np->rx_ring_size; i++) {
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
			np->rx_ring.orig[i].FlagLen = 0;
			np->rx_ring.orig[i].PacketBuffer = 0;
		} else {
			np->rx_ring.ex[i].FlagLen = 0;
			np->rx_ring.ex[i].TxVlan = 0;
			np->rx_ring.ex[i].PacketBufferHigh = 0;
			np->rx_ring.ex[i].PacketBufferLow = 0;
		}
		np->rx_skb[i].skb = NULL;
		np->rx_skb[i].dma = 0;
	}
}

static void nv_init_tx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	int i;

	np->get_tx = np->put_tx = np->first_tx = np->tx_ring;
	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
		np->last_tx.orig = &np->tx_ring.orig[np->tx_ring_size-1];
	else
		np->last_tx.ex = &np->tx_ring.ex[np->tx_ring_size-1];
	np->get_tx_ctx = np->put_tx_ctx = np->first_tx_ctx = np->tx_skb;
	np->last_tx_ctx = &np->tx_skb[np->tx_ring_size-1];

	for (i = 0; i < np->tx_ring_size; i++) {
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
			np->tx_ring.orig[i].FlagLen = 0;
			np->tx_ring.orig[i].PacketBuffer = 0;
		} else {
			np->tx_ring.ex[i].FlagLen = 0;
			np->tx_ring.ex[i].TxVlan = 0;
			np->tx_ring.ex[i].PacketBufferHigh = 0;
			np->tx_ring.ex[i].PacketBufferLow = 0;
		}
		np->tx_skb[i].skb = NULL;
		np->tx_skb[i].dma = 0;
	}
}

static int nv_init_ring(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	nv_init_tx(dev);
	nv_init_rx(dev);
	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
		return nv_alloc_rx(dev);
	else
		return nv_alloc_rx_optimized(dev);
}

static int nv_release_txskb(struct net_device *dev, unsigned int skbnr)
{
	struct fe_priv *np = get_nvpriv(dev);

	dprintk(KERN_INFO "%s: nv_release_txskb for skbnr %d\n",
		dev->name, skbnr);

	if (np->tx_skb[skbnr].dma) {
#ifdef __VMKERNEL_MODULE__
	 /* VMkernel doesn't do pci_map...so there is no need to do
	  * a unmap.
	  */
#else
		pci_unmap_page(np->pci_dev, np->tx_skb[skbnr].dma,
			       np->tx_skb[skbnr].dma_len,
			       PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
		np->tx_skb[skbnr].dma = 0;
	}
	if (np->tx_skb[skbnr].skb) {
		dev_kfree_skb_any(np->tx_skb[skbnr].skb);
		np->tx_skb[skbnr].skb = NULL;
		return 1;
	} else {
		return 0;
	}
}

static void nv_drain_tx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	unsigned int i;
	
	for (i = 0; i < np->tx_ring_size; i++) {
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
			np->tx_ring.orig[i].FlagLen = 0;
			np->tx_ring.orig[i].PacketBuffer = 0;
		} else {
			np->tx_ring.ex[i].FlagLen = 0;
			np->tx_ring.ex[i].TxVlan = 0;
			np->tx_ring.ex[i].PacketBufferHigh = 0;
			np->tx_ring.ex[i].PacketBufferLow = 0;
		}
		if (nv_release_txskb(dev, i))
			np->stats.tx_dropped++;
	}
}

static void nv_drain_rx(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	int i;
	for (i = 0; i < np->rx_ring_size; i++) {
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
			np->rx_ring.orig[i].FlagLen = 0;
			np->rx_ring.orig[i].PacketBuffer = 0;
		} else {
			np->rx_ring.ex[i].FlagLen = 0;
			np->rx_ring.ex[i].TxVlan = 0;
			np->rx_ring.ex[i].PacketBufferHigh = 0;
			np->rx_ring.ex[i].PacketBufferLow = 0;
		}
		wmb();
		if (np->rx_skb[i].skb) {
#ifndef __VMKERNEL_MODULE__
			pci_unmap_single(np->pci_dev, np->rx_skb[i].dma,
						np->rx_skb[i].skb->end-np->rx_skb[i].skb->data,
						PCI_DMA_FROMDEVICE);
#endif
			dev_kfree_skb(np->rx_skb[i].skb);
			np->rx_skb[i].skb = NULL;
		}
	}
}

static void drain_ring(struct net_device *dev)
{
	nv_drain_tx(dev);
	nv_drain_rx(dev);
}

/*
 * nv_start_xmit: dev->hard_start_xmit function
 * Called with dev->xmit_lock held.
 */
static int nv_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u32 tx_flags = 0;
	u32 tx_flags_extra = (np->desc_ver == DESC_VER_1 ? NV_TX_LASTPACKET : NV_TX2_LASTPACKET);
	unsigned int fragments = skb_shinfo(skb)->nr_frags;
	unsigned int i;
	u32 offset = 0;
	u32 bcnt;
	u32 size = skb->len-skb->data_len;
	u32 entries = (size >> NV_TX2_TSO_MAX_SHIFT) + ((size & (NV_TX2_TSO_MAX_SIZE-1)) ? 1 : 0);
	u32 empty_slots;
	struct ring_desc* put_tx;
	struct ring_desc* start_tx;
	struct ring_desc* prev_tx;
	struct nv_skb_map* prev_tx_ctx;
	unsigned long flags;

	/* add fragments to entries count */
	for (i = 0; i < fragments; i++) {
		entries += (skb_shinfo(skb)->frags[i].size >> NV_TX2_TSO_MAX_SHIFT) +
			   ((skb_shinfo(skb)->frags[i].size & (NV_TX2_TSO_MAX_SIZE-1)) ? 1 : 0);
	}

   	spin_lock_irqsave(&np->lock, flags);
	empty_slots = (u32)(np->tx_ring_size - ((np->tx_ring_size + (np->put_tx_ctx - np->get_tx_ctx)) % np->tx_ring_size));
	if (unlikely(empty_slots <= entries)) {
		netif_stop_queue(dev);
		np->stop_tx = 1;
		spin_unlock_irqrestore(&np->lock, flags);
		return NETDEV_TX_BUSY;
	}
	spin_unlock_irqrestore(&np->lock, flags);

	start_tx = put_tx = np->put_tx.orig;

	/* setup the header buffer */
	do {
		prev_tx = put_tx;
		prev_tx_ctx = np->put_tx_ctx;
		bcnt = (size > NV_TX2_TSO_MAX_SIZE) ? NV_TX2_TSO_MAX_SIZE : size;
#ifdef __VMKERNEL_MODULE__
		np->put_tx_ctx->dma = skb->headMA + offset;
#else
		np->put_tx_ctx->dma = pci_map_single(np->pci_dev, skb->data + offset, bcnt,
						PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
		np->put_tx_ctx->dma_len = bcnt;
		put_tx->PacketBuffer = cpu_to_le32(np->put_tx_ctx->dma);
		put_tx->FlagLen = cpu_to_le32((bcnt-1) | tx_flags);

		tx_flags = np->tx_flags;
		offset += bcnt;
		size -= bcnt;
		if (unlikely(put_tx++ == np->last_tx.orig))
			put_tx = np->first_tx.orig;
		if (unlikely(np->put_tx_ctx++ == np->last_tx_ctx))
			np->put_tx_ctx = np->first_tx_ctx;
	} while(size);

	/* setup the fragments */
	for (i = 0; i < fragments; i++) {
		skb_frag_t *frag = &skb_shinfo(skb)->frags[i];
		u32 size = frag->size;
		offset = 0;

		do {
			prev_tx = put_tx;
			prev_tx_ctx = np->put_tx_ctx;
			bcnt = (size > NV_TX2_TSO_MAX_SIZE) ? NV_TX2_TSO_MAX_SIZE : size;

#ifdef __VMKERNEL_MODULE__
			np->put_tx_ctx->dma = page_to_phys(frag->page) + frag->page_offset + offset;
#else
			np->put_tx_ctx->dma = pci_map_page(np->pci_dev, frag->page, frag->page_offset+offset, bcnt,
							   PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
			np->put_tx_ctx->dma_len = bcnt;

			put_tx->PacketBuffer = cpu_to_le32(np->put_tx_ctx->dma);
			put_tx->FlagLen = cpu_to_le32((bcnt-1) | tx_flags);
			offset += bcnt;
			size -= bcnt;
			if (unlikely(put_tx++ == np->last_tx.orig))
				put_tx = np->first_tx.orig;
			if (unlikely(np->put_tx_ctx++ == np->last_tx_ctx))
				np->put_tx_ctx = np->first_tx_ctx;
		} while (size);
	}

	/* set last fragment flag  */
	prev_tx->FlagLen |= cpu_to_le32(tx_flags_extra);

	/* save skb in this slot's context area */
	prev_tx_ctx->skb = skb;

#ifdef NETIF_F_TSO
	if (skb_shinfo(skb)->tso_size)
		tx_flags_extra = NV_TX2_TSO | (skb_shinfo(skb)->tso_size << NV_TX2_TSO_SHIFT);
	else
#endif
	tx_flags_extra = (skb->ip_summed == CHECKSUM_HW ? (NV_TX2_CHECKSUM_L3|NV_TX2_CHECKSUM_L4) : 0);

	spin_lock_irqsave(&np->lock, flags);	
	start_tx->FlagLen |= cpu_to_le32(tx_flags | tx_flags_extra);
	np->put_tx.orig = put_tx;
	spin_unlock_irqrestore(&np->lock, flags);

	dev->trans_start = jiffies;
	writel(NVREG_TXRXCTL_KICK|np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);
	return NETDEV_TX_OK;
}

static int nv_start_xmit_optimized(struct sk_buff *skb, struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u32 tx_flags = 0;
	u32 tx_flags_extra;
	unsigned int fragments = skb_shinfo(skb)->nr_frags;
	unsigned int i;
	u32 offset = 0;
	u32 bcnt;
	u32 size = skb->len-skb->data_len;
	u32 empty_slots;
	struct ring_desc_ex* put_tx;
	struct ring_desc_ex* start_tx;
	struct ring_desc_ex* prev_tx;
	struct nv_skb_map* prev_tx_ctx;
	unsigned long flags;

	u32 entries = (size >> NV_TX2_TSO_MAX_SHIFT) + ((size & (NV_TX2_TSO_MAX_SIZE-1)) ? 1 : 0);

	/* add fragments to entries count */
	for (i = 0; i < fragments; i++) {
		entries += (skb_shinfo(skb)->frags[i].size >> NV_TX2_TSO_MAX_SHIFT) +
			   ((skb_shinfo(skb)->frags[i].size & (NV_TX2_TSO_MAX_SIZE-1)) ? 1 : 0);
	}

	spin_lock_irqsave(&np->lock, flags);
	empty_slots = (u32)(np->tx_ring_size - ((np->tx_ring_size + (np->put_tx_ctx - np->get_tx_ctx)) % np->tx_ring_size));
	if (unlikely(empty_slots <= entries)) {
		netif_stop_queue(dev);
		np->stop_tx = 1;
		spin_unlock_irqrestore(&np->lock, flags);
		return NETDEV_TX_BUSY;
	}
   	spin_unlock_irqrestore(&np->lock, flags);

	start_tx = put_tx = np->put_tx.ex;

	/* setup the header buffer */
	do {
		prev_tx = put_tx;
		prev_tx_ctx = np->put_tx_ctx;
		bcnt = (size > NV_TX2_TSO_MAX_SIZE) ? NV_TX2_TSO_MAX_SIZE : size;
#ifdef __VMKERNEL_MODULE__
		np->put_tx_ctx->dma = skb->headMA + offset;
#else
		np->put_tx_ctx->dma = pci_map_single(np->pci_dev, skb->data + offset, bcnt,
						PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
		np->put_tx_ctx->dma_len = bcnt;
		put_tx->PacketBufferHigh = cpu_to_le64(np->put_tx_ctx->dma) >> 32;
		put_tx->PacketBufferLow = cpu_to_le64(np->put_tx_ctx->dma) & 0x0FFFFFFFF;
		put_tx->FlagLen = cpu_to_le32((bcnt-1) | tx_flags);

		tx_flags = NV_TX2_VALID;
		offset += bcnt;
		size -= bcnt;
		if (unlikely(put_tx++ == np->last_tx.ex))
			put_tx = np->first_tx.ex;
		if (unlikely(np->put_tx_ctx++ == np->last_tx_ctx))
			np->put_tx_ctx = np->first_tx_ctx;
	} while(size);
	/* setup the fragments */
	for (i = 0; i < fragments; i++) {
		skb_frag_t *frag = &skb_shinfo(skb)->frags[i];
		u32 size = frag->size;
		offset = 0;

		do {
			prev_tx = put_tx;
			prev_tx_ctx = np->put_tx_ctx;
			bcnt = (size > NV_TX2_TSO_MAX_SIZE) ? NV_TX2_TSO_MAX_SIZE : size;

#ifdef __VMKERNEL_MODULE__
			np->put_tx_ctx->dma = page_to_phys(frag->page) + frag->page_offset + offset;
#else
			np->put_tx_ctx->dma = pci_map_page(np->pci_dev, frag->page, frag->page_offset+offset, bcnt,
							   PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
			np->put_tx_ctx->dma_len = bcnt;

			put_tx->PacketBufferHigh = cpu_to_le64(np->put_tx_ctx->dma) >> 32;
			put_tx->PacketBufferLow = cpu_to_le64(np->put_tx_ctx->dma) & 0x0FFFFFFFF;
			put_tx->FlagLen = cpu_to_le32((bcnt-1) | tx_flags);
			offset += bcnt;
			size -= bcnt;
			if (unlikely(put_tx++ == np->last_tx.ex))
				put_tx = np->first_tx.ex;
			if (unlikely(np->put_tx_ctx++ == np->last_tx_ctx))
				np->put_tx_ctx = np->first_tx_ctx;
		} while (size);
	}

	/* set last fragment flag  */
	prev_tx->FlagLen |= cpu_to_le32(NV_TX2_LASTPACKET);

	/* save skb in this slot's context area */
	prev_tx_ctx->skb = skb;

#ifdef NETIF_F_TSO
	if (skb_shinfo(skb)->tso_size)
		tx_flags_extra = NV_TX2_TSO | (skb_shinfo(skb)->tso_size << NV_TX2_TSO_SHIFT);
	else
#endif
	tx_flags_extra = (skb->ip_summed == CHECKSUM_HW ? (NV_TX2_CHECKSUM_L3|NV_TX2_CHECKSUM_L4) : 0);

	/* vlan tag */
	if (likely(!np->vlangrp)) {
		start_tx->TxVlan = 0;
	} else {
		if (vlan_tx_tag_present(skb))
			start_tx->TxVlan = cpu_to_le32(NV_TX3_VLAN_TAG_PRESENT | vlan_tx_tag_get(skb));
		else
			start_tx->TxVlan = 0;
	}

	spin_lock_irqsave(&np->lock, flags);
	/* set tx flags */
	start_tx->FlagLen |= cpu_to_le32(tx_flags | tx_flags_extra);
	np->put_tx.ex = put_tx;
	spin_unlock_irqrestore(&np->lock, flags);

	dev->trans_start = jiffies;
	writel(NVREG_TXRXCTL_KICK|np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);
	return NETDEV_TX_OK;

}

/*
 * nv_tx_done: check for completed packets, release the skbs.
 *
 * Caller must own np->lock.
 */
static inline void nv_tx_done(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u32 Flags;
	struct ring_desc* orig_get_tx = np->get_tx.orig;
	struct ring_desc* put_tx = np->put_tx.orig;

	while ((np->get_tx.orig != put_tx) &&
	       !((Flags = le32_to_cpu(np->get_tx.orig->FlagLen)) & NV_TX_VALID)) {

#ifdef __VMKERNEL_MODULE__
	 /* VMkernel doesn't do pci_map...so there is no need to do
	  * a unmap.
	  */
#else
		pci_unmap_page(np->pci_dev, np->get_tx_ctx->dma,
			       np->get_tx_ctx->dma_len,
			       PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
		np->get_tx_ctx->dma = 0;

		if (np->desc_ver == DESC_VER_1) {
			if (Flags & NV_TX_LASTPACKET) {
				if (Flags & NV_TX_ERROR) {
					if (Flags & NV_TX_UNDERFLOW)
						np->stats.tx_fifo_errors++;
					if (Flags & NV_TX_CARRIERLOST)
						np->stats.tx_carrier_errors++;
					np->stats.tx_errors++;
				} else {
					np->stats.tx_packets++;
					np->stats.tx_bytes += np->get_tx_ctx->skb->len;
				}
				dev_kfree_skb_any(np->get_tx_ctx->skb);
				np->get_tx_ctx->skb = NULL;

			}
		} else {
			if (Flags & NV_TX2_LASTPACKET) {
				if (Flags & NV_TX2_ERROR) {
					if (Flags & NV_TX2_UNDERFLOW)
						np->stats.tx_fifo_errors++;
					if (Flags & NV_TX2_CARRIERLOST)
						np->stats.tx_carrier_errors++;
					np->stats.tx_errors++;
				} else {
					np->stats.tx_packets++;
					np->stats.tx_bytes += np->get_tx_ctx->skb->len;
				}				
				dev_kfree_skb_any(np->get_tx_ctx->skb);
				np->get_tx_ctx->skb = NULL;
			}
		}

		if (unlikely(np->get_tx.orig++ == np->last_tx.orig))
			np->get_tx.orig = np->first_tx.orig;
		if (unlikely(np->get_tx_ctx++ == np->last_tx_ctx))
			np->get_tx_ctx = np->first_tx_ctx;
	}
	if (unlikely((np->stop_tx == 1) && (np->get_tx.orig != orig_get_tx))) {
		np->stop_tx = 0;
		netif_wake_queue(dev);
	}
}

static inline void nv_tx_done_optimized(struct net_device *dev, int max_work)
{
	struct fe_priv *np = get_nvpriv(dev);
	u32 Flags;
	struct ring_desc_ex* orig_get_tx = np->get_tx.ex;
	struct ring_desc_ex* put_tx = np->put_tx.ex;

	while ((np->get_tx.ex != put_tx) &&
	       !((Flags = le32_to_cpu(np->get_tx.ex->FlagLen)) & NV_TX_VALID) &&
	       (max_work-- > 0)) {

#ifdef __VMKERNEL_MODULE__
	 /* VMkernel doesn't do pci_map...so there is no need to do
	  * a unmap.
	  */
#else
		pci_unmap_page(np->pci_dev, np->get_tx_ctx->dma,
			       np->get_tx_ctx->dma_len,
			       PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
		np->get_tx_ctx->dma = 0;

		if (Flags & NV_TX2_LASTPACKET) {
			if (!(Flags & NV_TX2_ERROR)) {
				np->stats.tx_packets++;
			}
			dev_kfree_skb_any(np->get_tx_ctx->skb);
			np->get_tx_ctx->skb = NULL;
		}
		
		if (unlikely(np->get_tx.ex++ == np->last_tx.ex))
			np->get_tx.ex = np->first_tx.ex;
		if (unlikely(np->get_tx_ctx++ == np->last_tx_ctx))
			np->get_tx_ctx = np->first_tx_ctx;
	}
	if (unlikely((np->stop_tx == 1) && (np->get_tx.ex != orig_get_tx))) {
		np->stop_tx = 0;
		netif_wake_queue(dev);
	}
}

/*
 * nv_tx_timeout: dev->tx_timeout function
 * Called with dev->xmit_lock held.
 */
static void nv_tx_timeout(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 status;

	if (!netif_running(dev))
		return;

	if (np->msi_flags & NV_MSI_X_ENABLED)
		status = readl(base + NvRegMSIXIrqStatus) & NVREG_IRQSTAT_MASK;
	else
		status = readl(base + NvRegIrqStatus) & NVREG_IRQSTAT_MASK;

	printk(KERN_INFO "%s: Got tx_timeout. irq: %08x\n", dev->name, status);

	{
		int i;

		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
			printk(KERN_INFO "%s: Ring at %lx: get %lx put %lx\n",
			       dev->name, (unsigned long)np->tx_ring.orig,
			       (unsigned long)np->get_tx.orig, (unsigned long)np->put_tx.orig);
		} else {
			printk(KERN_INFO "%s: Ring at %lx: get %lx put %lx\n",
			       dev->name, (unsigned long)np->tx_ring.ex,
			       (unsigned long)np->get_tx.ex, (unsigned long)np->put_tx.ex);
		}
		printk(KERN_INFO "%s: Dumping tx registers\n", dev->name);
		for (i=0;i<=np->register_size;i+= 32) {
			printk(KERN_INFO "%3x: %08x %08x %08x %08x %08x %08x %08x %08x\n",
					i,
					readl(base + i + 0), readl(base + i + 4),
					readl(base + i + 8), readl(base + i + 12),
					readl(base + i + 16), readl(base + i + 20),
					readl(base + i + 24), readl(base + i + 28));
		}
		printk(KERN_INFO "%s: Dumping tx ring\n", dev->name);
		for (i=0;i<np->tx_ring_size;i+= 4) {
			if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
				printk(KERN_INFO "%03x: %08x %08x // %08x %08x // %08x %08x // %08x %08x\n",
				       i, 
				       le32_to_cpu(np->tx_ring.orig[i].PacketBuffer),
				       le32_to_cpu(np->tx_ring.orig[i].FlagLen),
				       le32_to_cpu(np->tx_ring.orig[i+1].PacketBuffer),
				       le32_to_cpu(np->tx_ring.orig[i+1].FlagLen),
				       le32_to_cpu(np->tx_ring.orig[i+2].PacketBuffer),
				       le32_to_cpu(np->tx_ring.orig[i+2].FlagLen),
				       le32_to_cpu(np->tx_ring.orig[i+3].PacketBuffer),
				       le32_to_cpu(np->tx_ring.orig[i+3].FlagLen));
			} else {
				printk(KERN_INFO "%03x: %08x %08x %08x // %08x %08x %08x // %08x %08x %08x // %08x %08x %08x\n",
				       i, 
				       le32_to_cpu(np->tx_ring.ex[i].PacketBufferHigh),
				       le32_to_cpu(np->tx_ring.ex[i].PacketBufferLow),
				       le32_to_cpu(np->tx_ring.ex[i].FlagLen),
				       le32_to_cpu(np->tx_ring.ex[i+1].PacketBufferHigh),
				       le32_to_cpu(np->tx_ring.ex[i+1].PacketBufferLow),
				       le32_to_cpu(np->tx_ring.ex[i+1].FlagLen),
				       le32_to_cpu(np->tx_ring.ex[i+2].PacketBufferHigh),
				       le32_to_cpu(np->tx_ring.ex[i+2].PacketBufferLow),
				       le32_to_cpu(np->tx_ring.ex[i+2].FlagLen),
				       le32_to_cpu(np->tx_ring.ex[i+3].PacketBufferHigh),
				       le32_to_cpu(np->tx_ring.ex[i+3].PacketBufferLow),
				       le32_to_cpu(np->tx_ring.ex[i+3].FlagLen));
			}
		}
	}

	nv_disable_irq(dev);
	spin_lock_irq(&np->lock);

	/* 1) stop tx engine */
	nv_stop_tx(dev);

	/* 2) check that the packets were not sent already: */
	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
	nv_tx_done(dev);
	else
		nv_tx_done_optimized(dev, np->tx_ring_size);

	/* 3) if there are dead entries: clear everything */
	if (np->get_tx_ctx != np->put_tx_ctx) {
		printk(KERN_DEBUG "%s: tx_timeout: dead entries!\n", dev->name);
		nv_drain_tx(dev);
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
			np->get_tx.orig = np->put_tx.orig = np->first_tx.orig;
		else
			np->get_tx.ex = np->put_tx.ex = np->first_tx.ex;
		np->get_tx_ctx = np->put_tx_ctx = np->first_tx_ctx;
		setup_hw_rings(dev, NV_SETUP_TX_RING);
		netif_wake_queue(dev);
	}

	/* 4) restart tx engine */
	nv_start_tx(dev);
	spin_unlock_irq(&np->lock);
	nv_enable_irq(dev);
}

/*
 * Called when the nic notices a mismatch between the actual data len on the
 * wire and the len indicated in the 802 header
 */
static int nv_getlen(struct net_device *dev, void *packet, int datalen)
{
	int hdrlen;	/* length of the 802 header */
	int protolen;	/* length as stored in the proto field */

	/* 1) calculate len according to header */
	if ( ((struct vlan_ethhdr *)packet)->h_vlan_proto == __constant_htons(ETH_P_8021Q)) {
		protolen = ntohs( ((struct vlan_ethhdr *)packet)->h_vlan_encapsulated_proto );
		hdrlen = VLAN_HLEN;
	} else {
		protolen = ntohs( ((struct ethhdr *)packet)->h_proto);
		hdrlen = ETH_HLEN;
	}
	dprintk(KERN_DEBUG "%s: nv_getlen: datalen %d, protolen %d, hdrlen %d\n",
				dev->name, datalen, protolen, hdrlen);
	if (protolen > ETH_DATA_LEN)
		return datalen; /* Value in proto field not a len, no checks possible */

	protolen += hdrlen;
	/* consistency checks: */
	if (datalen > ETH_ZLEN) {
		if (datalen >= protolen) {
			/* more data on wire than in 802 header, trim of
			 * additional data.
			 */
			dprintk(KERN_DEBUG "%s: nv_getlen: accepting %d bytes.\n",
					dev->name, protolen);
			return protolen;
		} else {
			/* less data on wire than mentioned in header.
			 * Discard the packet.
			 */
			dprintk(KERN_DEBUG "%s: nv_getlen: discarding long packet.\n",
					dev->name);
			return -1;
		}
	} else {
		/* short packet. Accept only if 802 values are also short */
		if (protolen > ETH_ZLEN) {
			dprintk(KERN_DEBUG "%s: nv_getlen: discarding short packet.\n",
					dev->name);
			return -1;
		}
		dprintk(KERN_DEBUG "%s: nv_getlen: accepting %d bytes.\n",
				dev->name, datalen);
		return datalen;
	}
}

static inline void nv_rx_process(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u32 Flags;
	struct sk_buff *skb;
	int len;

	while((np->get_rx.orig != np->put_rx.orig) &&
	      !((Flags = le32_to_cpu(np->get_rx.orig->FlagLen)) & NV_RX_AVAIL)) {

#ifndef __VMKERNEL_MODULE__
		pci_unmap_single(np->pci_dev, np->get_rx_ctx->dma,
				 np->get_rx_ctx->dma_len,
				 PCI_DMA_FROMDEVICE);
#endif

		skb = np->get_rx_ctx->skb;
		np->get_rx_ctx->skb = NULL;

		{
			int j;
			dprintk(KERN_DEBUG "Dumping packet (flags 0x%x).",Flags);
			for (j=0; j<64; j++) {
				if ((j%16) == 0)
					dprintk("\n%03x:", j);
				dprintk(" %02x", ((unsigned char*)skb->data)[j]);
			}
			dprintk("\n");
		}

		if (np->desc_ver == DESC_VER_1) {

			if (likely(Flags & NV_RX_DESCRIPTORVALID)) {
				len = Flags & LEN_MASK_V1;
				if (unlikely(Flags & NV_RX_ERROR)) {
					if (Flags & NV_RX_ERROR4) {
						len = nv_getlen(dev, skb->data, len);
						if (len < 0) {
							np->stats.rx_errors++;
							dev_kfree_skb(skb);
							goto next_pkt;
						}
					}
					/* framing errors are soft errors */
					else if (Flags & NV_RX_FRAMINGERR) {
						if (Flags & NV_RX_SUBSTRACT1) {
							len--;
						}
					}
					/* the rest are hard errors */
					else {
						if (Flags & NV_RX_MISSEDFRAME)
							np->stats.rx_missed_errors++;
						if (Flags & NV_RX_CRCERR)
							np->stats.rx_crc_errors++;
						if (Flags & NV_RX_OVERFLOW)
							np->stats.rx_over_errors++;
						np->stats.rx_errors++;
						dev_kfree_skb(skb);
						goto next_pkt;
					}
				}
			} else {
				dev_kfree_skb(skb);
				goto next_pkt;
			}
		} else {
			if (likely(Flags & NV_RX2_DESCRIPTORVALID)) {
				len = Flags & LEN_MASK_V2;
				if (unlikely(Flags & NV_RX2_ERROR)) {
					if (Flags & NV_RX2_ERROR4) {
						len = nv_getlen(dev, skb->data, len);
						if (len < 0) {
							np->stats.rx_errors++;
							dev_kfree_skb(skb);
							goto next_pkt;
						}
					}
					/* framing errors are soft errors */
					else if (Flags & NV_RX2_FRAMINGERR) {
						if (Flags & NV_RX2_SUBSTRACT1) {
							len--;
						}
					}
					/* the rest are hard errors */
					else {
						if (Flags & NV_RX2_CRCERR)
							np->stats.rx_crc_errors++;
						if (Flags & NV_RX2_OVERFLOW)
							np->stats.rx_over_errors++;
						np->stats.rx_errors++;
						dev_kfree_skb(skb);
						goto next_pkt;
					}
				}
				if ((Flags & NV_RX2_CHECKSUMMASK) == NV_RX2_CHECKSUMOK2)/*ip and tcp */ {
					skb->ip_summed = CHECKSUM_UNNECESSARY;
				} else {
					if ((Flags & NV_RX2_CHECKSUMMASK) == NV_RX2_CHECKSUMOK1 ||
					    (Flags & NV_RX2_CHECKSUMMASK) == NV_RX2_CHECKSUMOK3) {
						skb->ip_summed = CHECKSUM_UNNECESSARY;
					}
				}
			} else {
				dev_kfree_skb(skb);
				goto next_pkt;
			}
		}

		/* got a valid packet - forward it to the network core */
		skb_put(skb, len);
		skb->protocol = eth_type_trans(skb, dev);
		netif_rx(skb);
		dev->last_rx = jiffies;
		np->stats.rx_packets++;
		np->stats.rx_bytes += len;
next_pkt:
		if (unlikely(np->get_rx.orig++ == np->last_rx.orig))
			np->get_rx.orig = np->first_rx.orig;
		if (unlikely(np->get_rx_ctx++ == np->last_rx_ctx))
			np->get_rx_ctx = np->first_rx_ctx;
	}
}

static inline int nv_rx_process_optimized(struct net_device *dev, int max_work)
{
	struct fe_priv *np = get_nvpriv(dev);
	u32 Flags;
	u32 vlanflags = 0;
	u32 rx_processed_cnt = 0;
	struct sk_buff *skb;
	int len;

	while((np->get_rx.ex != np->put_rx.ex) &&
	      !((Flags = le32_to_cpu(np->get_rx.ex->FlagLen)) & NV_RX2_AVAIL) &&
	      (rx_processed_cnt++ < max_work)) {

#ifndef __VMKERNEL_MODULE__
		pci_unmap_single(np->pci_dev, np->get_rx_ctx->dma,
				 np->get_rx_ctx->dma_len,
				 PCI_DMA_FROMDEVICE);
#endif

		skb = np->get_rx_ctx->skb;
		np->get_rx_ctx->skb = NULL;

		/* look at what we actually got: */
		if (likely(Flags & NV_RX2_DESCRIPTORVALID)) {
			len = Flags & LEN_MASK_V2;
			if (unlikely(Flags & NV_RX2_ERROR)) {
				if (Flags & NV_RX2_ERROR4) {
					len = nv_getlen(dev, skb->data, len);
					if (len < 0) {
						np->rx_len_errors++;
						dev_kfree_skb(skb);
						goto next_pkt;
					}
				}
				/* framing errors are soft errors */
				else if (Flags & NV_RX2_FRAMINGERR) {
					if (Flags & NV_RX2_SUBSTRACT1) {
						len--;
					}
				}
				/* the rest are hard errors */
				else {
					dev_kfree_skb(skb);
					goto next_pkt;
				}
			}

			if (likely(np->rx_csum)) {
				if (likely((Flags & NV_RX2_CHECKSUMMASK) == NV_RX2_CHECKSUMOK2)) {
					/*ip and tcp */
					skb->ip_summed = CHECKSUM_UNNECESSARY;
				} else {
					if ((Flags & NV_RX2_CHECKSUMMASK) == NV_RX2_CHECKSUMOK1 ||
					    (Flags & NV_RX2_CHECKSUMMASK) == NV_RX2_CHECKSUMOK3) {
						skb->ip_summed = CHECKSUM_UNNECESSARY;
					}
				}
			}

			/* got a valid packet - forward it to the network core */
			skb_put(skb, len);
			skb->protocol = eth_type_trans(skb, dev);
			prefetch(skb->data);

			if (likely(!np->vlangrp)) {
				netif_rx(skb);
			} else {
				vlanflags = le32_to_cpu(np->get_rx.ex->PacketBufferLow);
				if (vlanflags & NV_RX3_VLAN_TAG_PRESENT)
					vlan_hwaccel_rx(skb, np->vlangrp, vlanflags & NV_RX3_VLAN_TAG_MASK);
				else
					netif_rx(skb);
			}

			dev->last_rx = jiffies;
			np->stats.rx_packets++;
			np->stats.rx_bytes += len;
		} else {
			dev_kfree_skb(skb);
		}
next_pkt:
		if (unlikely(np->get_rx.ex++ == np->last_rx.ex))
			np->get_rx.ex = np->first_rx.ex;
		if (unlikely(np->get_rx_ctx++ == np->last_rx_ctx))
			np->get_rx_ctx = np->first_rx_ctx;
	}
	return rx_processed_cnt;
}

static void set_bufsize(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (dev->mtu <= ETH_DATA_LEN)
		np->rx_buf_sz = ETH_DATA_LEN + NV_RX_HEADERS;
	else
		np->rx_buf_sz = dev->mtu + NV_RX_HEADERS;
}

/*
 * nv_change_mtu: dev->change_mtu function
 * Called with dev_base_lock held for read.
 */
static int nv_change_mtu(struct net_device *dev, int new_mtu)
{
	struct fe_priv *np = get_nvpriv(dev);
	int old_mtu;

	if (new_mtu < 64 || new_mtu > np->pkt_limit)
		return -EINVAL;

	old_mtu = dev->mtu;
	dev->mtu = new_mtu;

	/* return early if the buffer sizes will not change */
	if (old_mtu <= ETH_DATA_LEN && new_mtu <= ETH_DATA_LEN)
		return 0;
	if (old_mtu == new_mtu)
		return 0;

	/* synchronized against open : rtnl_lock() held by caller */
	if (netif_running(dev)) {
		u8 __iomem *base = get_hwbase(dev);
		/*
		 * It seems that the nic preloads valid ring entries into an
		 * internal buffer. The procedure for flushing everything is
		 * guessed, there is probably a simpler approach.
		 * Changing the MTU is a rare event, it shouldn't matter.
		 */
		nv_disable_irq(dev);
		spin_lock_bh(&dev->xmit_lock);
		spin_lock(&np->lock);
		/* stop engines */
		nv_stop_rx(dev);
		nv_stop_tx(dev);
		nv_txrx_reset(dev);
		/* drain rx queue */
		nv_drain_rx(dev);
		nv_drain_tx(dev);
		/* reinit driver view of the rx queue */
		set_bufsize(dev);
		if (nv_init_ring(dev)) {
			if (!np->in_shutdown)
				mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
		}
		/* reinit nic view of the rx queue */
		writel(np->rx_buf_sz, base + NvRegOffloadConfig);
		setup_hw_rings(dev, NV_SETUP_RX_RING | NV_SETUP_TX_RING);
		writel( ((np->rx_ring_size-1) << NVREG_RINGSZ_RXSHIFT) + ((np->tx_ring_size-1) << NVREG_RINGSZ_TXSHIFT),
			base + NvRegRingSizes);
		pci_push(base);
		writel(NVREG_TXRXCTL_KICK|np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);
		pci_push(base);

		/* restart rx engine */
		nv_start_rx(dev);
		nv_start_tx(dev);
		spin_unlock(&np->lock);
		spin_unlock_bh(&dev->xmit_lock);
		nv_enable_irq(dev);
	}
	return 0;
}

static void nv_copy_mac_to_hw(struct net_device *dev)
{
	u8 __iomem *base = get_hwbase(dev);
	u32 mac[2];

	mac[0] = (dev->dev_addr[0] << 0) + (dev->dev_addr[1] << 8) +
			(dev->dev_addr[2] << 16) + (dev->dev_addr[3] << 24);
	mac[1] = (dev->dev_addr[4] << 0) + (dev->dev_addr[5] << 8);

	writel(mac[0], base + NvRegMacAddrA);
	writel(mac[1], base + NvRegMacAddrB);
}

/*
 * nv_set_mac_address: dev->set_mac_address function
 * Called with rtnl_lock() held.
 */
static int nv_set_mac_address(struct net_device *dev, void *addr)
{
	struct fe_priv *np = get_nvpriv(dev);
	struct sockaddr *macaddr = (struct sockaddr*)addr;

	if(!is_valid_ether_addr(macaddr->sa_data))
		return -EADDRNOTAVAIL;

	/* synchronized against open : rtnl_lock() held by caller */
	memcpy(dev->dev_addr, macaddr->sa_data, ETH_ALEN);

	if (netif_running(dev)) {
		spin_lock_bh(&dev->xmit_lock);
		spin_lock_irq(&np->lock);

		/* stop rx engine */
		nv_stop_rx(dev);

		/* set mac address */
		nv_copy_mac_to_hw(dev);

		/* restart rx engine */
		nv_start_rx(dev);
		spin_unlock_irq(&np->lock);
		spin_unlock_bh(&dev->xmit_lock);
	} else {
		nv_copy_mac_to_hw(dev);
	}
	return 0;
}

/*
 * nv_set_multicast: dev->set_multicast function
 * Called with dev->xmit_lock held.
 */
static void nv_set_multicast(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 addr[2];
	u32 mask[2];
	u32 pff = readl(base + NvRegPacketFilterFlags) & NVREG_PFF_PAUSE_RX;

	memset(addr, 0, sizeof(addr));
	memset(mask, 0, sizeof(mask));

	if (dev->flags & IFF_PROMISC) {
		dprintk(KERN_INFO "%s: Promiscuous mode enabled.\n", dev->name);
		pff |= NVREG_PFF_PROMISC;
	} else {
		pff |= NVREG_PFF_MYADDR;

		if (dev->flags & IFF_ALLMULTI || dev->mc_list) {
			u32 alwaysOff[2];
			u32 alwaysOn[2];

			alwaysOn[0] = alwaysOn[1] = alwaysOff[0] = alwaysOff[1] = 0xffffffff;
			if (dev->flags & IFF_ALLMULTI) {
				alwaysOn[0] = alwaysOn[1] = alwaysOff[0] = alwaysOff[1] = 0;
			} else {
				struct dev_mc_list *walk;

				walk = dev->mc_list;
				while (walk != NULL) {
					u32 a, b;
					a = le32_to_cpu(*(u32 *) walk->dmi_addr);
					b = le16_to_cpu(*(u16 *) (&walk->dmi_addr[4]));
					alwaysOn[0] &= a;
					alwaysOff[0] &= ~a;
					alwaysOn[1] &= b;
					alwaysOff[1] &= ~b;
					walk = walk->next;
				}
			}
			addr[0] = alwaysOn[0];
			addr[1] = alwaysOn[1];
			mask[0] = alwaysOn[0] | alwaysOff[0];
			mask[1] = alwaysOn[1] | alwaysOff[1];
		}
	}
	addr[0] |= NVREG_MCASTADDRA_FORCE;
	pff |= NVREG_PFF_ALWAYS;
	spin_lock_irq(&np->lock);
	nv_stop_rx(dev);
	writel(addr[0], base + NvRegMulticastAddrA);
	writel(addr[1], base + NvRegMulticastAddrB);
	writel(mask[0], base + NvRegMulticastMaskA);
	writel(mask[1], base + NvRegMulticastMaskB);
	writel(pff, base + NvRegPacketFilterFlags);
	dprintk(KERN_INFO "%s: reconfiguration for multicast lists.\n",
		dev->name);
	nv_start_rx(dev);
	spin_unlock_irq(&np->lock);
}

static void nv_update_pause(struct net_device *dev, u32 pause_flags)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);

	np->pause_flags &= ~(NV_PAUSEFRAME_TX_ENABLE | NV_PAUSEFRAME_RX_ENABLE);

	if (np->pause_flags & NV_PAUSEFRAME_RX_CAPABLE) {
		u32 pff = readl(base + NvRegPacketFilterFlags) & ~NVREG_PFF_PAUSE_RX;
		if (pause_flags & NV_PAUSEFRAME_RX_ENABLE) {
			writel(pff|NVREG_PFF_PAUSE_RX, base + NvRegPacketFilterFlags);
			np->pause_flags |= NV_PAUSEFRAME_RX_ENABLE;
		} else {
			writel(pff, base + NvRegPacketFilterFlags);
		}
	}
	if (np->pause_flags & NV_PAUSEFRAME_TX_CAPABLE) {
		u32 regmisc = readl(base + NvRegMisc1) & ~NVREG_MISC1_PAUSE_TX;
		if (pause_flags & NV_PAUSEFRAME_TX_ENABLE) {
			writel(NVREG_TX_PAUSEFRAME_ENABLE,  base + NvRegTxPauseFrame);
			writel(regmisc|NVREG_MISC1_PAUSE_TX, base + NvRegMisc1);
			np->pause_flags |= NV_PAUSEFRAME_TX_ENABLE;
		} else {
			writel(NVREG_TX_PAUSEFRAME_DISABLE,  base + NvRegTxPauseFrame);
			writel(regmisc, base + NvRegMisc1);			
		}
	}
}

/**
 * nv_update_linkspeed: Setup the MAC according to the link partner
 * @dev: Network device to be configured
 *
 * The function queries the PHY and checks if there is a link partner.
 * If yes, then it sets up the MAC accordingly. Otherwise, the MAC is
 * set to 10 MBit HD.
 *
 * The function returns 0 if there is no link partner and 1 if there is
 * a good link partner.
 */
static int nv_update_linkspeed(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	int adv = 0;
	int lpa = 0;
	int adv_lpa, adv_pause, lpa_pause;
	int newls = np->linkspeed;
	int newdup = np->duplex;
	int mii_status;
	int retval = 0;
	u32 control_1000, status_1000, phyreg, pause_flags, txreg;

	/* BMSR_LSTATUS is latched, read it twice:
	 * we want the current value.
	 */
	mii_rw(dev, np->phyaddr, MII_BMSR, MII_READ);
	mii_status = mii_rw(dev, np->phyaddr, MII_BMSR, MII_READ);

	if (!(mii_status & BMSR_LSTATUS)) {
		dprintk(KERN_DEBUG "%s: no link detected by phy - falling back to 10HD.\n",
				dev->name);
		newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
		newdup = 0;
		retval = 0;
		goto set_speed;
	}

	if (np->autoneg == AUTONEG_DISABLE) {
		dprintk(KERN_DEBUG "%s: nv_update_linkspeed: autoneg off, PHY set to 0x%04x.\n",
				dev->name, np->fixed_mode);
		if (np->fixed_mode & LPA_100FULL) {
			newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_100;
			newdup = 1;
		} else if (np->fixed_mode & LPA_100HALF) {
			newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_100;
			newdup = 0;
		} else if (np->fixed_mode & LPA_10FULL) {
			newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
			newdup = 1;
		} else {
			newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
			newdup = 0;
		}
		retval = 1;
		goto set_speed;
	}
	/* check auto negotiation is complete */
	if (!(mii_status & BMSR_ANEGCOMPLETE)) {
		/* still in autonegotiation - configure nic for 10 MBit HD and wait. */
		newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
		newdup = 0;
		retval = 0;
		dprintk(KERN_DEBUG "%s: autoneg not completed - falling back to 10HD.\n", dev->name);
		goto set_speed;
	}

	adv = mii_rw(dev, np->phyaddr, MII_ADVERTISE, MII_READ);
	lpa = mii_rw(dev, np->phyaddr, MII_LPA, MII_READ);
	dprintk(KERN_DEBUG "%s: nv_update_linkspeed: PHY advertises 0x%04x, lpa 0x%04x.\n",
				dev->name, adv, lpa);
	retval = 1;
	if (np->gigabit == PHY_GIGABIT) {
		control_1000 = mii_rw(dev, np->phyaddr, MII_CTRL1000, MII_READ);
		status_1000 = mii_rw(dev, np->phyaddr, MII_STAT1000, MII_READ);

		if ((control_1000 & ADVERTISE_1000FULL) &&
			(status_1000 & LPA_1000FULL)) {
			dprintk(KERN_DEBUG "%s: nv_update_linkspeed: GBit ethernet detected.\n",
				dev->name);
			newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_1000;
			newdup = 1;
			goto set_speed;
		}
	}

	/* FIXME: handle parallel detection properly */
	adv_lpa = lpa & adv;
	if (adv_lpa & LPA_100FULL) {
		newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_100;
		newdup = 1;
	} else if (adv_lpa & LPA_100HALF) {
		newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_100;
		newdup = 0;
	} else if (adv_lpa & LPA_10FULL) {
		newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
		newdup = 1;
	} else if (adv_lpa & LPA_10HALF) {
		newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
		newdup = 0;
	} else {
		dprintk(KERN_DEBUG "%s: bad ability %04x - falling back to 10HD.\n", dev->name, adv_lpa);
		newls = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
		newdup = 0;
	}

set_speed:
	if (np->duplex == newdup && np->linkspeed == newls)
		return retval;

	dprintk(KERN_INFO "%s: changing link setting from %d/%d to %d/%d.\n",
			dev->name, np->linkspeed, np->duplex, newls, newdup);

	np->duplex = newdup;
	np->linkspeed = newls;

	if (np->gigabit == PHY_GIGABIT) {
		phyreg = readl(base + NvRegRandomSeed);
		phyreg &= ~(0x3FF00);
		if ((np->linkspeed & 0xFFF) == NVREG_LINKSPEED_10)
			phyreg |= NVREG_RNDSEED_FORCE3;
		else if ((np->linkspeed & 0xFFF) == NVREG_LINKSPEED_100)
			phyreg |= NVREG_RNDSEED_FORCE2;
		else if ((np->linkspeed & 0xFFF) == NVREG_LINKSPEED_1000)
			phyreg |= NVREG_RNDSEED_FORCE;
		writel(phyreg, base + NvRegRandomSeed);
	}

	phyreg = readl(base + NvRegPhyInterface);
	phyreg &= ~(PHY_HALF|PHY_100|PHY_1000);
	if (np->duplex == 0)
		phyreg |= PHY_HALF;
	if ((np->linkspeed & NVREG_LINKSPEED_MASK) == NVREG_LINKSPEED_100)
		phyreg |= PHY_100;
	else if ((np->linkspeed & NVREG_LINKSPEED_MASK) == NVREG_LINKSPEED_1000)
		phyreg |= PHY_1000;
	writel(phyreg, base + NvRegPhyInterface);

	if (phyreg & PHY_RGMII) {
		if ((np->linkspeed & NVREG_LINKSPEED_MASK) == NVREG_LINKSPEED_1000)
			txreg = NVREG_TX_DEFERRAL_RGMII_1000;
		else
			txreg = NVREG_TX_DEFERRAL_RGMII_10_100;
	} else {
		txreg = NVREG_TX_DEFERRAL_DEFAULT;
	}
	writel(txreg, base + NvRegTxDeferral);

	if (np->desc_ver == DESC_VER_1) {
		txreg = NVREG_TX_WM_DESC1_DEFAULT;
	} else {
		if ((np->linkspeed & NVREG_LINKSPEED_MASK) == NVREG_LINKSPEED_1000)
			txreg = NVREG_TX_WM_DESC2_3_1000;
		else
			txreg = NVREG_TX_WM_DESC2_3_DEFAULT;
	}
	writel(txreg, base + NvRegTxWatermark);
	writel(NVREG_MISC1_FORCE | ( np->duplex ? 0 : NVREG_MISC1_HD),
		base + NvRegMisc1);
	pci_push(base);
	writel(np->linkspeed, base + NvRegLinkSpeed);
	pci_push(base);

	pause_flags = 0;
	/* setup pause frame */
	if (np->duplex != 0) {
		if (np->autoneg && np->pause_flags & NV_PAUSEFRAME_AUTONEG) {
			adv_pause = adv & (ADVERTISE_PAUSE_CAP| ADVERTISE_PAUSE_ASYM);
			lpa_pause = lpa & (LPA_PAUSE_CAP| LPA_PAUSE_ASYM);

			switch (adv_pause) {
			case (ADVERTISE_PAUSE_CAP):
				if (lpa_pause & LPA_PAUSE_CAP) {
					pause_flags |= NV_PAUSEFRAME_RX_ENABLE;
					if (np->pause_flags & NV_PAUSEFRAME_TX_REQ)
						pause_flags |= NV_PAUSEFRAME_TX_ENABLE;
				}
				break;
			case (ADVERTISE_PAUSE_ASYM):
				if (lpa_pause == (LPA_PAUSE_CAP| LPA_PAUSE_ASYM))
				{
					pause_flags |= NV_PAUSEFRAME_TX_ENABLE;
				}
				break;
			case (ADVERTISE_PAUSE_CAP| ADVERTISE_PAUSE_ASYM):
				if (lpa_pause & LPA_PAUSE_CAP)
				{
					pause_flags |=  NV_PAUSEFRAME_RX_ENABLE;
					if (np->pause_flags & NV_PAUSEFRAME_TX_REQ)
						pause_flags |= NV_PAUSEFRAME_TX_ENABLE;
				}
				if (lpa_pause == LPA_PAUSE_ASYM)
				{
					pause_flags |= NV_PAUSEFRAME_RX_ENABLE;
				} 
				break;
			}
		} else {
			pause_flags = np->pause_flags;
		}
	}
	nv_update_pause(dev, pause_flags);

	return retval;
}

static void nv_linkchange(struct net_device *dev)
{
	if (nv_update_linkspeed(dev)) {
		if (!netif_carrier_ok(dev)) {
			netif_carrier_on(dev);
			printk(KERN_INFO "%s: link up.\n", dev->name);
			nv_start_rx(dev);
		}
	} else {
		if (netif_carrier_ok(dev)) {
			netif_carrier_off(dev);
			printk(KERN_INFO "%s: link down.\n", dev->name);
			nv_stop_rx(dev);
		}
	}
}

static void nv_link_irq(struct net_device *dev)
{
	u8 __iomem *base = get_hwbase(dev);
	u32 miistat;

	miistat = readl(base + NvRegMIIStatus);
	writel(NVREG_MIISTAT_MASK, base + NvRegMIIStatus);
	dprintk(KERN_INFO "%s: link change irq, status 0x%x.\n", dev->name, miistat);

	if (miistat & (NVREG_MIISTAT_LINKCHANGE))
		nv_linkchange(dev);
	dprintk(KERN_DEBUG "%s: link change notification done.\n", dev->name);
}

static irqreturn_t nv_nic_irq(int foo, void *data, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 events;
#if 1 || defined(__VMKERNEL_MODULE__)
	u32 irq_mask;
#endif
	int i;

	dprintk(KERN_DEBUG "%s: nv_nic_irq\n", dev->name);

	for (i=0; ; i++) {
		if (!(np->msi_flags & NV_MSI_X_ENABLED)) {
			events = readl(base + NvRegIrqStatus) & NVREG_IRQSTAT_MASK;
			writel(NVREG_IRQSTAT_MASK, base + NvRegIrqStatus);
		} else {
			events = readl(base + NvRegMSIXIrqStatus) & NVREG_IRQSTAT_MASK;
			writel(NVREG_IRQSTAT_MASK, base + NvRegMSIXIrqStatus);
		}
#if 1 || defined(__VMKERNEL_MODULE__)
		/*
		 * Read the irq mask from the device and use it to check if the
		 * interrupt needs to be serviced. nv_disable_hw_interrupts()
		 * clears the mask register when the device is not MSI-X capable
		 * but leaves the np->irqmask unmodified.
		 * So np->irqmask and the actually masked interrupts can be
		 * different. Not reading from the device can cause PSODs when
		 * the NVidia Ethernet device shares an irq with another device.
		 * PR144391.
		 */
		if (np->msi_flags & NV_MSI_X_ENABLED) {
			irq_mask = readl(base + NvRegIrqMask);
		} else {
			if (np->msi_flags & NV_MSI_ENABLED)
				irq_mask = readl(base + NvRegMSIIrqMask);
			else  
				irq_mask = readl(base + NvRegIrqMask);  
		}
#endif // __VMKERNEL_MODULE__
		pci_push(base);
		dprintk(KERN_DEBUG "%s: irq: %08x\n", dev->name, events);
#if 1 || defined(__VMKERNEL_MODULE__)
		if (!(events & np->irqmask & irq_mask)) 
			break;
#else // __VMKERNEL_MODULE__ 
		if (!(events & np->irqmask))
			break;
#endif // __VMKERNEL_MODULE__ 
#ifdef __VMKERNEL_MODULE__
		vmk_net_inc_dev_intrcount(dev);
#endif // __VMKERNEL_MODULE__

		nv_tx_done(dev);
		
		nv_rx_process(dev);
		if (nv_alloc_rx(dev)) {
			spin_lock(&np->lock);
			if (!np->in_shutdown)
				mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
			spin_unlock(&np->lock);
		}
		
		if (events & NVREG_IRQ_LINK) {
			spin_lock(&np->lock);
			nv_link_irq(dev);
			spin_unlock(&np->lock);
		}
		if (np->need_linktimer && time_after(jiffies, np->link_timeout)) {
			spin_lock(&np->lock);
			nv_linkchange(dev);
			spin_unlock(&np->lock);
			np->link_timeout = jiffies + LINK_TIMEOUT;
		}
		if (events & (NVREG_IRQ_TX_ERR)) {
			dprintk(KERN_DEBUG "%s: received irq with events 0x%x. Probably TX fail.\n",
						dev->name, events);
		}
		if (events & (NVREG_IRQ_UNKNOWN)) {
			printk(KERN_DEBUG "%s: received irq with unknown events 0x%x. Please report\n",
						dev->name, events);
		}
		if (i > max_interrupt_work) {
			spin_lock(&np->lock);
			/* disable interrupts on the nic */
			if (!(np->msi_flags & NV_MSI_X_ENABLED))
				writel(0, base + NvRegIrqMask);
			else
				writel(np->irqmask, base + NvRegIrqMask);
			pci_push(base);

			if (!np->in_shutdown) {
				np->nic_poll_irq = np->irqmask;
				mod_timer(&np->nic_poll, jiffies + POLL_WAIT);
			}
			printk(KERN_DEBUG "%s: too many iterations (%d) in nv_nic_irq.\n", dev->name, i);
			spin_unlock(&np->lock);
			break;
		}

	}
	dprintk(KERN_DEBUG "%s: nv_nic_irq completed\n", dev->name);

	return IRQ_RETVAL(i);
}

#define TX_WORK_PER_LOOP  64
#define RX_WORK_PER_LOOP  64
static irqreturn_t nv_nic_irq_optimized(int foo, void *data, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 events;
#if 1 || defined(__VMKERNEL_MODULE__)
	u32 irq_mask;
#endif // __VMKERNEL_MODULE__
	int i = 1;

	do {
		if (!(np->msi_flags & NV_MSI_X_ENABLED)) {
			events = readl(base + NvRegIrqStatus) & NVREG_IRQSTAT_MASK;
			writel(NVREG_IRQSTAT_MASK, base + NvRegIrqStatus);
		} else {
			events = readl(base + NvRegMSIXIrqStatus) & NVREG_IRQSTAT_MASK;
			writel(NVREG_IRQSTAT_MASK, base + NvRegMSIXIrqStatus);
		}	
#if 1 || defined(__VMKERNEL_MODULE__)
		/*
		 * Read the irq mask from the device and use it to check if the
		 * interrupt needs to be serviced. nv_disable_hw_interrupts()
		 * clears the mask register when the device is not MSI-X capable
		 * but leaves the np->irqmask unmodified.
		 * So np->irqmask and the actually masked interrupts can be 
		 * different. Not reading from the device can cause PSODs when
		 * the NVidia Ethernet device shares an irq with another device.
		 * PR144391.
		 */
		if (np->msi_flags & NV_MSI_X_ENABLED) { 
			irq_mask = readl(base + NvRegIrqMask);
		} else {
			if (np->msi_flags & NV_MSI_ENABLED)
				irq_mask = readl(base + NvRegMSIIrqMask);
			else   
				irq_mask = readl(base + NvRegIrqMask); 
		}
		if (events & np->irqmask & irq_mask) { 
#else // __VMKERNEL_MODULE__
		if (events & np->irqmask) {
#endif // __VMKERNEL_MODULE__ 

			spin_lock(&np->lock);  
			nv_tx_done_optimized(dev, TX_WORK_PER_LOOP);
			spin_unlock(&np->lock);

			if (nv_rx_process_optimized(dev, RX_WORK_PER_LOOP)) {
				if (unlikely(nv_alloc_rx_optimized(dev))) {
					spin_lock(&np->lock);
					if (!np->in_shutdown)
						mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
					spin_unlock(&np->lock);
				}
			}
			if (unlikely(events & NVREG_IRQ_LINK)) {
				spin_lock(&np->lock);
				nv_link_irq(dev);
				spin_unlock(&np->lock);
			}
			if (unlikely(np->need_linktimer && time_after(jiffies, np->link_timeout))) {
				spin_lock(&np->lock);
				nv_linkchange(dev);
				spin_unlock(&np->lock);
				np->link_timeout = jiffies + LINK_TIMEOUT;
			}
			if (unlikely(events & NVREG_IRQ_RECOVER_ERROR)) {
				spin_lock(&np->lock);
				/* disable interrupts on the nic */
				if (!(np->msi_flags & NV_MSI_X_ENABLED))
					writel(0, base + NvRegIrqMask);
				else
					writel(np->irqmask, base + NvRegIrqMask);
				pci_push(base);

				if (!np->in_shutdown) {
					np->nic_poll_irq = np->irqmask;
					np->recover_error = 1;
					mod_timer(&np->nic_poll, jiffies + POLL_WAIT);
				}
				spin_unlock(&np->lock);
				break;
			}
		} else
			break;
	}
	while (i++ <= max_interrupt_work);

	return IRQ_RETVAL(i);
}

static irqreturn_t nv_nic_irq_tx(int foo, void *data, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 events;
	int i;
	unsigned long flags;

	dprintk(KERN_DEBUG "%s: nv_nic_irq_tx\n", dev->name);

	for (i=0; ; i++) {
		events = readl(base + NvRegMSIXIrqStatus) & NVREG_IRQ_TX_ALL;
		writel(NVREG_IRQ_TX_ALL, base + NvRegMSIXIrqStatus);
		dprintk(KERN_DEBUG "%s: tx irq: %08x\n", dev->name, events);
		if (!(events & np->irqmask))
			break;

		spin_lock_irqsave(&np->lock, flags);
		nv_tx_done_optimized(dev, TX_WORK_PER_LOOP);
		spin_unlock_irqrestore(&np->lock, flags);
		
		if (events & (NVREG_IRQ_TX_ERR)) {
			dprintk(KERN_DEBUG "%s: received irq with events 0x%x. Probably TX fail.\n",
						dev->name, events);
		}
		if (i > max_interrupt_work) {
			spin_lock_irq(&np->lock);
			/* disable interrupts on the nic */
			writel(NVREG_IRQ_TX_ALL, base + NvRegIrqMask);
			pci_push(base);

			if (!np->in_shutdown) {
				np->nic_poll_irq |= NVREG_IRQ_TX_ALL;
				mod_timer(&np->nic_poll, jiffies + POLL_WAIT);
			}
			printk(KERN_DEBUG "%s: too many iterations (%d) in nv_nic_irq_tx.\n", dev->name, i);
			spin_unlock_irq(&np->lock);
			break;
		}

	}
	dprintk(KERN_DEBUG "%s: nv_nic_irq_tx completed\n", dev->name);

	return IRQ_RETVAL(i);
}

static irqreturn_t nv_nic_irq_rx(int foo, void *data, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 events;
	int i;

	dprintk(KERN_DEBUG "%s: nv_nic_irq_rx\n", dev->name);

	for (i=0; ; i++) {
		events = readl(base + NvRegMSIXIrqStatus) & NVREG_IRQ_RX_ALL;
		writel(NVREG_IRQ_RX_ALL, base + NvRegMSIXIrqStatus);
		dprintk(KERN_DEBUG "%s: rx irq: %08x\n", dev->name, events);
		if (!(events & np->irqmask))
			break;
		
		if (nv_rx_process_optimized(dev, RX_WORK_PER_LOOP)) {
			if (unlikely(nv_alloc_rx_optimized(dev))) {
			spin_lock_irq(&np->lock);
			if (!np->in_shutdown)
				mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
			spin_unlock_irq(&np->lock);
			}
		}
		
		if (i > max_interrupt_work) {
			spin_lock_irq(&np->lock);
			/* disable interrupts on the nic */
			writel(NVREG_IRQ_RX_ALL, base + NvRegIrqMask);
			pci_push(base);

			if (!np->in_shutdown) {
				np->nic_poll_irq |= NVREG_IRQ_RX_ALL;
				mod_timer(&np->nic_poll, jiffies + POLL_WAIT);
			}
			printk(KERN_DEBUG "%s: too many iterations (%d) in nv_nic_irq_rx.\n", dev->name, i);
			spin_unlock_irq(&np->lock);
			break;
		}

	}
	dprintk(KERN_DEBUG "%s: nv_nic_irq_rx completed\n", dev->name);

	return IRQ_RETVAL(i);
}

static irqreturn_t nv_nic_irq_other(int foo, void *data, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 events;
	int i;

	dprintk(KERN_DEBUG "%s: nv_nic_irq_other\n", dev->name);

	for (i=0; ; i++) {
		events = readl(base + NvRegMSIXIrqStatus) & NVREG_IRQ_OTHER;
		writel(NVREG_IRQ_OTHER, base + NvRegMSIXIrqStatus);
		dprintk(KERN_DEBUG "%s: irq: %08x\n", dev->name, events);
		if (!(events & np->irqmask))
			break;
		
		if (events & NVREG_IRQ_LINK) {
			spin_lock_irq(&np->lock);
			nv_link_irq(dev);
			spin_unlock_irq(&np->lock);
		}
		if (np->need_linktimer && time_after(jiffies, np->link_timeout)) {
			spin_lock_irq(&np->lock);
			nv_linkchange(dev);
			spin_unlock_irq(&np->lock);
			np->link_timeout = jiffies + LINK_TIMEOUT;
		}
		if (events & NVREG_IRQ_RECOVER_ERROR) {
			spin_lock_irq(&np->lock);
			/* disable interrupts on the nic */
			writel(NVREG_IRQ_OTHER, base + NvRegIrqMask);
			pci_push(base);
			
			if (!np->in_shutdown) {
				np->nic_poll_irq |= NVREG_IRQ_OTHER;
				np->recover_error = 1;
				mod_timer(&np->nic_poll, jiffies + POLL_WAIT);
			}
			spin_unlock_irq(&np->lock);
			break;
		}
		if (events & (NVREG_IRQ_UNKNOWN)) {
			printk(KERN_DEBUG "%s: received irq with unknown events 0x%x. Please report\n",
						dev->name, events);
		}
		if (i > max_interrupt_work) {
			spin_lock_irq(&np->lock);
			/* disable interrupts on the nic */
			writel(NVREG_IRQ_OTHER, base + NvRegIrqMask);
			pci_push(base);

			if (!np->in_shutdown) {
				np->nic_poll_irq |= NVREG_IRQ_OTHER;
				mod_timer(&np->nic_poll, jiffies + POLL_WAIT);
			}
			printk(KERN_DEBUG "%s: too many iterations (%d) in nv_nic_irq_other.\n", dev->name, i);
			spin_unlock_irq(&np->lock);
			break;
		}

	}
	dprintk(KERN_DEBUG "%s: nv_nic_irq_other completed\n", dev->name);

	return IRQ_RETVAL(i);
}

static irqreturn_t nv_nic_irq_test(int foo, void *data, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 events;

	dprintk(KERN_DEBUG "%s: nv_nic_irq_test\n", dev->name);

	if (!(np->msi_flags & NV_MSI_X_ENABLED)) {
		events = readl(base + NvRegIrqStatus) & NVREG_IRQSTAT_MASK;
		writel(NVREG_IRQ_TIMER, base + NvRegIrqStatus);
	} else {
		events = readl(base + NvRegMSIXIrqStatus) & NVREG_IRQSTAT_MASK;
		writel(NVREG_IRQ_TIMER, base + NvRegMSIXIrqStatus);
	}
	pci_push(base);
	dprintk(KERN_DEBUG "%s: irq: %08x\n", dev->name, events);
	if (!(events & NVREG_IRQ_TIMER))
		return IRQ_RETVAL(0);
	
	spin_lock(&np->lock);
	np->intr_test = 1;
	spin_unlock(&np->lock);
		
	dprintk(KERN_DEBUG "%s: nv_nic_irq_test completed\n", dev->name);

	return IRQ_RETVAL(1);
}

#ifdef CONFIG_PCI_MSI
static void set_msix_vector_map(struct net_device *dev, u32 vector, u32 irqmask)
{
	u8 __iomem *base = get_hwbase(dev);
	int i;
	u32 msixmap = 0;

	/* Each interrupt bit can be mapped to a MSIX vector (4 bits).
	 * MSIXMap0 represents the first 8 interrupts and MSIXMap1 represents
	 * the remaining 8 interrupts.
	 */
	for (i = 0; i < 8; i++) {
		if ((irqmask >> i) & 0x1) {
			msixmap |= vector << (i << 2);
		}
	}
	writel(readl(base + NvRegMSIXMap0) | msixmap, base + NvRegMSIXMap0);

	msixmap = 0;
	for (i = 0; i < 8; i++) {
		if ((irqmask >> (i + 8)) & 0x1) {
			msixmap |= vector << (i << 2);
		}
	}
	writel(readl(base + NvRegMSIXMap1) | msixmap, base + NvRegMSIXMap1);
}
#endif

static int nv_request_irq(struct net_device *dev, int intr_test)
{
	struct fe_priv *np = get_nvpriv(dev);
	int ret = 1;

#if NVVER > SLES9
	u8 __iomem *base = get_hwbase(dev);
	int i;

	if (np->msi_flags & NV_MSI_X_CAPABLE) {
		for (i = 0; i < (np->msi_flags & NV_MSI_X_VECTORS_MASK); i++) {
			np->msi_x_entry[i].entry = i;
		}
		if ((ret = pci_enable_msix(np->pci_dev, np->msi_x_entry, (np->msi_flags & NV_MSI_X_VECTORS_MASK))) == 0) {
			np->msi_flags |= NV_MSI_X_ENABLED;
			if (optimization_mode == NV_OPTIMIZATION_MODE_THROUGHPUT && !intr_test) {
				/* Request irq for rx handling */
				if (request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector, &nv_nic_irq_rx, SA_SHIRQ, dev->name, dev) != 0) {
					printk(KERN_INFO "forcedeth: request_irq failed for rx %d\n", ret);
					pci_disable_msix(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_err;
				}
				/* Request irq for tx handling */
				if (request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector, &nv_nic_irq_tx, SA_SHIRQ, dev->name, dev) != 0) {
					printk(KERN_INFO "forcedeth: request_irq failed for tx %d\n", ret);
					pci_disable_msix(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_free_rx;
				}
				/* Request irq for link and timer handling */
				if (request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_OTHER].vector, &nv_nic_irq_other, SA_SHIRQ, dev->name, dev) != 0) {
					printk(KERN_INFO "forcedeth: request_irq failed for link %d\n", ret);
					pci_disable_msix(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_free_tx;
				}
				/* map interrupts to their respective vector */
				writel(0, base + NvRegMSIXMap0);
				writel(0, base + NvRegMSIXMap1);
				set_msix_vector_map(dev, NV_MSI_X_VECTOR_RX, NVREG_IRQ_RX_ALL);
				set_msix_vector_map(dev, NV_MSI_X_VECTOR_TX, NVREG_IRQ_TX_ALL);
				set_msix_vector_map(dev, NV_MSI_X_VECTOR_OTHER, NVREG_IRQ_OTHER);
			} else {
				/* Request irq for all interrupts */
				if ((!intr_test && np->desc_ver == DESC_VER_3 &&
				     request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector, &nv_nic_irq_optimized, SA_SHIRQ, dev->name, dev) != 0) ||
				    (!intr_test && np->desc_ver != DESC_VER_3 &&
				     request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector, &nv_nic_irq, SA_SHIRQ, dev->name, dev) != 0) ||
				    (intr_test &&
				     request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector, &nv_nic_irq_test, SA_SHIRQ, dev->name, dev) != 0)) {
					printk(KERN_INFO "forcedeth: request_irq failed %d\n", ret);
					pci_disable_msix(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_err;
				}

				/* map interrupts to vector 0 */
				writel(0, base + NvRegMSIXMap0);
				writel(0, base + NvRegMSIXMap1);
			}
		}
	}
	if (ret != 0 && np->msi_flags & NV_MSI_CAPABLE) {
		if ((ret = pci_enable_msi(np->pci_dev)) == 0) {
			np->msi_flags |= NV_MSI_ENABLED;
			if ((!intr_test && np->desc_ver == DESC_VER_3 &&
			     request_irq(np->pci_dev->irq, &nv_nic_irq_optimized, SA_SHIRQ, dev->name, dev) != 0) ||
			    (!intr_test && np->desc_ver != DESC_VER_3 &&
			     request_irq(np->pci_dev->irq, &nv_nic_irq, SA_SHIRQ, dev->name, dev) != 0) ||
			    (intr_test && request_irq(np->pci_dev->irq, &nv_nic_irq_test, SA_SHIRQ, dev->name, dev) != 0)) {
				printk(KERN_INFO "forcedeth: request_irq failed %d\n", ret);
				pci_disable_msi(np->pci_dev);
				np->msi_flags &= ~NV_MSI_ENABLED;
				goto out_err;
			}

			/* map interrupts to vector 0 */
			writel(0, base + NvRegMSIMap0);
			writel(0, base + NvRegMSIMap1);
			/* enable msi vector 0 */
			writel(NVREG_MSI_VECTOR_0_ENABLED, base + NvRegMSIIrqMask);
		}
	}
#else
#ifdef CONFIG_PCI_MSI
	u8 __iomem *base = get_hwbase(dev);
	int i;

	if (np->msi_flags & NV_MSI_X_CAPABLE) {
		for (i = 0; i < (np->msi_flags & NV_MSI_X_VECTORS_MASK); i++) {
			np->msi_x_entry[i].entry = i;
		}
		if ((ret = pci_enable_msi(np->pci_dev)) == 0) {
			np->msi_flags |= NV_MSI_X_ENABLED;
			if (optimization_mode == NV_OPTIMIZATION_MODE_THROUGHPUT && !intr_test) {
				msi_alloc_vectors(np->pci_dev,(int *)np->msi_x_entry,2);
				/* Request irq for rx handling */
				if (request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector, &nv_nic_irq_rx, SA_SHIRQ, dev->name, dev) != 0) {
					printk(KERN_INFO "forcedeth: request_irq failed for rx %d\n", ret);
					pci_disable_msi(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_err;
				}
				/* Request irq for tx handling */
				if (request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector, &nv_nic_irq_tx, SA_SHIRQ, dev->name, dev) != 0) {
					printk(KERN_INFO "forcedeth: request_irq failed for tx %d\n", ret);
					pci_disable_msi(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_free_rx;
				}
				/* Request irq for link and timer handling */
				if (request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_OTHER].vector, &nv_nic_irq_other, SA_SHIRQ, dev->name, dev) != 0) {
					printk(KERN_INFO "forcedeth: request_irq failed for link %d\n", ret);
					pci_disable_msi(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_free_tx;
				}
				/* map interrupts to their respective vector */
				writel(0, base + NvRegMSIXMap0);
				writel(0, base + NvRegMSIXMap1);
				set_msix_vector_map(dev, NV_MSI_X_VECTOR_RX, NVREG_IRQ_RX_ALL);
				set_msix_vector_map(dev, NV_MSI_X_VECTOR_TX, NVREG_IRQ_TX_ALL);
				set_msix_vector_map(dev, NV_MSI_X_VECTOR_OTHER, NVREG_IRQ_OTHER);
			} else {
				/* Request irq for all interrupts */
				if ((!intr_test && np->desc_ver == DESC_VER_3 &&
				     request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector, &nv_nic_irq_optimized, SA_SHIRQ, dev->name, dev) != 0) ||
				    (!intr_test && np->desc_ver != DESC_VER_3 &&
				     request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector, &nv_nic_irq, SA_SHIRQ, dev->name, dev) != 0) ||
				    (intr_test &&
				     request_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector, &nv_nic_irq_test, SA_SHIRQ, dev->name, dev) != 0)) {
					printk(KERN_INFO "forcedeth: request_irq failed %d\n", ret);
					pci_disable_msi(np->pci_dev);
					np->msi_flags &= ~NV_MSI_X_ENABLED;
					goto out_err;
				}

				/* map interrupts to vector 0 */
				writel(0, base + NvRegMSIXMap0);
				writel(0, base + NvRegMSIXMap1);
			}
		}
	}
	if (ret != 0 && np->msi_flags & NV_MSI_CAPABLE) {

		if ((ret = pci_enable_msi(np->pci_dev)) == 0) {
			np->msi_flags |= NV_MSI_ENABLED;
			if ((!intr_test && np->desc_ver == DESC_VER_3 &&
			     request_irq(np->pci_dev->irq, &nv_nic_irq_optimized, SA_SHIRQ, dev->name, dev) != 0) ||
			    (!intr_test && np->desc_ver != DESC_VER_3 &&
			     request_irq(np->pci_dev->irq, &nv_nic_irq, SA_SHIRQ, dev->name, dev) != 0) ||
			    (intr_test && request_irq(np->pci_dev->irq, &nv_nic_irq_test, SA_SHIRQ, dev->name, dev) != 0)) {
				printk(KERN_INFO "forcedeth: request_irq failed %d\n", ret);
				pci_disable_msi(np->pci_dev);
				np->msi_flags &= ~NV_MSI_ENABLED;
				goto out_err;
			}

			/* map interrupts to vector 0 */
			writel(0, base + NvRegMSIMap0);
			writel(0, base + NvRegMSIMap1);
			/* enable msi vector 0 */
			writel(NVREG_MSI_VECTOR_0_ENABLED, base + NvRegMSIIrqMask);
		}
	}
#endif
#endif
	if (ret != 0) {
		if ((!intr_test  && np->desc_ver == DESC_VER_3 && 
		     request_irq(np->pci_dev->irq, &nv_nic_irq_optimized, SA_SHIRQ, dev->name, dev) != 0) ||
		    (!intr_test  && np->desc_ver != DESC_VER_3 &&
		     request_irq(np->pci_dev->irq, &nv_nic_irq, SA_SHIRQ, dev->name, dev) != 0) || 
		    (intr_test && request_irq(np->pci_dev->irq, &nv_nic_irq_test, SA_SHIRQ, dev->name, dev) != 0))
			goto out_err;
			
	}

	return 0;

#if NVVER > SLES9
out_free_tx:
	free_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector, dev);
out_free_rx:
	free_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector, dev);
#else
#ifdef CONFIG_PCI_MSI	
out_free_tx:
	free_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector, dev);
out_free_rx:
	free_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector, dev);
#endif
#endif
out_err:
	return 1;
}

#if NVVER > SLES9
static void nv_free_irq(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	int i;
	
	if (np->msi_flags & NV_MSI_X_ENABLED) {
		for (i = 0; i < (np->msi_flags & NV_MSI_X_VECTORS_MASK); i++) {
			free_irq(np->msi_x_entry[i].vector, dev);
		}
		pci_disable_msix(np->pci_dev);
		np->msi_flags &= ~NV_MSI_X_ENABLED;
	} else {
		free_irq(np->pci_dev->irq, dev);
		if (np->msi_flags & NV_MSI_ENABLED) {
			pci_disable_msi(np->pci_dev);
			np->msi_flags &= ~NV_MSI_ENABLED;
		}
	}
}
#else
static void nv_free_irq(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	
#ifdef CONFIG_PCI_MSI		
	int i;

	if (np->msi_flags & NV_MSI_X_ENABLED) {
		for (i = 0; i < (np->msi_flags & NV_MSI_X_VECTORS_MASK); i++) {
			free_irq(np->msi_x_entry[i].vector, dev);
		}
		pci_disable_msi(np->pci_dev);
		np->msi_flags &= ~NV_MSI_X_ENABLED;
	} else {
		free_irq(np->pci_dev->irq, dev);

		if (np->msi_flags & NV_MSI_ENABLED) {
			pci_disable_msi(np->pci_dev);
			np->msi_flags &= ~NV_MSI_ENABLED;
		}
	}
#else
	free_irq(np->pci_dev->irq, dev);
#endif
	
}
#endif 

static void nv_do_nic_poll(unsigned long data)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 mask = 0;

	/*
	 * First disable irq(s) and then
	 * reenable interrupts on the nic, we have to do this before calling
	 * nv_nic_irq because that may decide to do otherwise
	 */

	if (!using_multi_irqs(dev)) {
		if (np->msi_flags & NV_MSI_X_ENABLED)
			disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector);
		else
			disable_irq(dev->irq);
		mask = np->irqmask;
	} else {
		if (np->nic_poll_irq & NVREG_IRQ_RX_ALL) {
			disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector);
			mask |= NVREG_IRQ_RX_ALL;
		}
		if (np->nic_poll_irq & NVREG_IRQ_TX_ALL) {
			disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector);
			mask |= NVREG_IRQ_TX_ALL;
		}
		if (np->nic_poll_irq & NVREG_IRQ_OTHER) {
			disable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_OTHER].vector);
			mask |= NVREG_IRQ_OTHER;
		}
	}
	np->nic_poll_irq = 0;

	if (np->recover_error) {
		np->recover_error = 0;
		printk(KERN_INFO "forcedeth: MAC in recoverable error state\n");
		if (netif_running(dev)) {
			spin_lock_bh(&dev->xmit_lock);
			spin_lock(&np->lock);
			/* stop engines */
			nv_stop_rx(dev);
			nv_stop_tx(dev);
			nv_txrx_reset(dev);
			/* drain rx queue */
			nv_drain_rx(dev);
			nv_drain_tx(dev);
			/* reinit driver view of the rx queue */
			set_bufsize(dev);
			if (nv_init_ring(dev)) {
				if (!np->in_shutdown)
					mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
			}
			/* reinit nic view of the rx queue */
			writel(np->rx_buf_sz, base + NvRegOffloadConfig);
			setup_hw_rings(dev, NV_SETUP_RX_RING | NV_SETUP_TX_RING);
			writel( ((np->rx_ring_size-1) << NVREG_RINGSZ_RXSHIFT) + ((np->tx_ring_size-1) << NVREG_RINGSZ_TXSHIFT),
				base + NvRegRingSizes);
			pci_push(base);
			writel(NVREG_TXRXCTL_KICK|np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);
			pci_push(base);

			/* restart rx engine */
			nv_start_rx(dev);
			nv_start_tx(dev);
			spin_unlock(&np->lock);
			spin_unlock_bh(&dev->xmit_lock);
		}
	}
	/* FIXME: Do we need synchronize_irq(dev->irq) here? */
	
	writel(mask, base + NvRegIrqMask);
	pci_push(base);

	if (!using_multi_irqs(dev)) {
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
		nv_nic_irq((int) 0, (void *) data, (struct pt_regs *) NULL);
		else
			nv_nic_irq_optimized((int) 0, (void *) data, (struct pt_regs *) NULL);
		if (np->msi_flags & NV_MSI_X_ENABLED)
			enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_ALL].vector);
		else
			enable_irq(dev->irq);
	} else {
		if (np->nic_poll_irq & NVREG_IRQ_RX_ALL) {
			nv_nic_irq_rx((int) 0, (void *) data, (struct pt_regs *) NULL);
			enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_RX].vector);
		}
		if (np->nic_poll_irq & NVREG_IRQ_TX_ALL) {
			nv_nic_irq_tx((int) 0, (void *) data, (struct pt_regs *) NULL);
			enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_TX].vector);
		}
		if (np->nic_poll_irq & NVREG_IRQ_OTHER) {
			nv_nic_irq_other((int) 0, (void *) data, (struct pt_regs *) NULL);
			enable_irq(np->msi_x_entry[NV_MSI_X_VECTOR_OTHER].vector);
		}
	}
}

#if NVVER > RHES3
#ifdef CONFIG_NET_POLL_CONTROLLER
static void nv_poll_controller(struct net_device *dev)
{
	nv_do_nic_poll((unsigned long) dev);
}
#endif
#else
static void nv_poll_controller(struct net_device *dev)
{
	nv_do_nic_poll((unsigned long) dev);
}
#endif

static void nv_do_stats_poll(unsigned long data)
{
	struct net_device *dev = (struct net_device *) data;
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);

	spin_lock_irq(&np->lock);
	
	np->estats.tx_dropped = np->stats.tx_dropped;
	if (np->driver_data & DEV_HAS_STATISTICS) {
		np->estats.tx_packets += readl(base + NvRegTxFrame);
		np->estats.tx_fifo_errors += readl(base + NvRegTxUnderflow);
		np->estats.tx_carrier_errors += readl(base + NvRegTxLossCarrier);
		np->estats.tx_bytes += readl(base + NvRegTxCnt);
		np->estats.rx_bytes += readl(base + NvRegRxCnt);
		np->estats.rx_crc_errors += readl(base + NvRegRxFCSErr);
		np->estats.rx_over_errors += readl(base + NvRegRxOverflow);

		np->estats.tx_zero_rexmt += readl(base + NvRegTxZeroReXmt);
		np->estats.tx_one_rexmt += readl(base + NvRegTxOneReXmt);
		np->estats.tx_many_rexmt += readl(base + NvRegTxManyReXmt);
		np->estats.tx_late_collision += readl(base + NvRegTxLateCol);
		np->estats.tx_excess_deferral += readl(base + NvRegTxExcessDef);
		np->estats.tx_retry_error += readl(base + NvRegTxRetryErr);
		np->estats.rx_frame_error += readl(base + NvRegRxFrameErr);
		np->estats.rx_extra_byte += readl(base + NvRegRxExtraByte);
		np->estats.rx_late_collision += readl(base + NvRegRxLateCol);
		np->estats.rx_runt += readl(base + NvRegRxRunt);
		np->estats.rx_frame_too_long += readl(base + NvRegRxFrameTooLong);
		np->estats.rx_frame_align_error += readl(base + NvRegRxFrameAlignErr);
		np->estats.rx_length_error += readl(base + NvRegRxLenErr);
		np->estats.rx_unicast += readl(base + NvRegRxUnicast);
		np->estats.rx_multicast += readl(base + NvRegRxMulticast);
		np->estats.rx_broadcast += readl(base + NvRegRxBroadcast);
		np->estats.tx_deferral += readl(base + NvRegTxDef);
		np->estats.tx_pause += readl(base + NvRegTxPause);
		np->estats.rx_pause += readl(base + NvRegRxPause);
		np->estats.rx_drop_frame += readl(base + NvRegRxDropFrame);
		np->estats.rx_packets = 
			np->estats.rx_unicast + 
			np->estats.rx_multicast + 
			np->estats.rx_broadcast;
		np->estats.rx_errors_total = 
			np->estats.rx_crc_errors +
			np->estats.rx_over_errors +
			np->estats.rx_frame_error +
			(np->estats.rx_frame_align_error - np->estats.rx_extra_byte) +
			np->estats.rx_late_collision +
			np->estats.rx_runt +
			np->estats.rx_frame_too_long +
			np->rx_len_errors;

		/* copy to net_device stats */
		np->stats.tx_packets = np->estats.tx_packets;
		np->stats.tx_fifo_errors = np->estats.tx_fifo_errors;
		np->stats.tx_carrier_errors = np->estats.tx_carrier_errors;
		np->stats.tx_bytes = np->estats.tx_bytes;
		np->stats.rx_bytes = np->estats.rx_bytes;
		np->stats.rx_crc_errors = np->estats.rx_crc_errors;
		np->stats.rx_over_errors = np->estats.rx_over_errors;
		np->stats.rx_packets = np->estats.rx_packets;
		np->stats.rx_errors = np->estats.rx_errors_total;
		
	} else {
		np->estats.tx_packets = np->stats.tx_packets;
		np->estats.tx_fifo_errors = np->stats.tx_fifo_errors;
		np->estats.tx_carrier_errors = np->stats.tx_carrier_errors;
		np->estats.tx_bytes = np->stats.tx_bytes;
		np->estats.rx_bytes = np->stats.rx_bytes;
		np->estats.rx_crc_errors = np->stats.rx_crc_errors;
		np->estats.rx_over_errors = np->stats.rx_over_errors;
		np->estats.rx_packets = np->stats.rx_packets;
		np->estats.rx_errors_total = np->stats.rx_errors;
	}

	if (!np->in_shutdown && netif_running(dev))
		mod_timer(&np->stats_poll, jiffies + STATS_INTERVAL);
	spin_unlock_irq(&np->lock);
}

/*
 * nv_get_stats: dev->get_stats function
 * Get latest stats value from the nic.
 * Called with read_lock(&dev_base_lock) held for read -
 * only synchronized against unregister_netdevice.
 */
static struct net_device_stats *nv_get_stats(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	/* It seems that the nic always generates interrupts and doesn't
	 * accumulate errors internally. Thus the current values in np->stats
	 * are already up to date.
	 */
	nv_do_stats_poll((unsigned long)dev);
	return &np->stats;
}

static void nv_get_drvinfo(struct net_device *dev, struct ethtool_drvinfo *info)
{
	struct fe_priv *np = get_nvpriv(dev);
	strcpy(info->driver, "forcedeth");
	strcpy(info->version, FORCEDETH_VERSION);
	strcpy(info->bus_info, pci_name(np->pci_dev));
}

static void nv_get_wol(struct net_device *dev, struct ethtool_wolinfo *wolinfo)
{
	struct fe_priv *np = get_nvpriv(dev);
	wolinfo->supported = WAKE_MAGIC;

	spin_lock_irq(&np->lock);
	if (np->wolenabled)
		wolinfo->wolopts = WAKE_MAGIC;
	spin_unlock_irq(&np->lock);
}

static int nv_set_wol(struct net_device *dev, struct ethtool_wolinfo *wolinfo)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 flags = 0;

	if (wolinfo->wolopts == 0) {
		np->wolenabled = 0;
	} else if (wolinfo->wolopts & WAKE_MAGIC) {
		np->wolenabled = 1;
		flags = NVREG_WAKEUPFLAGS_ENABLE;
	}
	if (netif_running(dev)) {
		spin_lock_irq(&np->lock);
		writel(flags, base + NvRegWakeUpFlags);
		spin_unlock_irq(&np->lock);
	}
	return 0;
}

static int nv_get_settings(struct net_device *dev, struct ethtool_cmd *ecmd)
{
	struct fe_priv *np = get_nvpriv(dev);
	int adv;

	spin_lock_irq(&np->lock);
	ecmd->port = PORT_MII;
	if (!netif_running(dev)) {
		/* We do not track link speed / duplex setting if the
		 * interface is disabled. Force a link check */
		if (nv_update_linkspeed(dev)) {
			if (!netif_carrier_ok(dev))
				netif_carrier_on(dev);
		} else {
			if (netif_carrier_ok(dev))
				netif_carrier_off(dev);
		}
	}

	if (netif_carrier_ok(dev)) {
		switch(np->linkspeed & (NVREG_LINKSPEED_MASK)) {
		case NVREG_LINKSPEED_10:
			ecmd->speed = SPEED_10;
			break;
		case NVREG_LINKSPEED_100:
			ecmd->speed = SPEED_100;
			break;
		case NVREG_LINKSPEED_1000:
			ecmd->speed = SPEED_1000;
			break;
		}
		ecmd->duplex = DUPLEX_HALF;
		if (np->duplex)
			ecmd->duplex = DUPLEX_FULL;
	} else {
		ecmd->speed = -1;
		ecmd->duplex = -1;
	}

	ecmd->autoneg = np->autoneg;

	ecmd->advertising = ADVERTISED_MII;
	if (np->autoneg) {
		ecmd->advertising |= ADVERTISED_Autoneg;
		adv = mii_rw(dev, np->phyaddr, MII_ADVERTISE, MII_READ);
		if (adv & ADVERTISE_10HALF)
			ecmd->advertising |= ADVERTISED_10baseT_Half;
		if (adv & ADVERTISE_10FULL)
			ecmd->advertising |= ADVERTISED_10baseT_Full;
		if (adv & ADVERTISE_100HALF)
			ecmd->advertising |= ADVERTISED_100baseT_Half;
		if (adv & ADVERTISE_100FULL)
			ecmd->advertising |= ADVERTISED_100baseT_Full;
		if (np->gigabit == PHY_GIGABIT) {
			adv = mii_rw(dev, np->phyaddr, MII_CTRL1000, MII_READ);
			if (adv & ADVERTISE_1000FULL)
				ecmd->advertising |= ADVERTISED_1000baseT_Full;
		}
	}
	ecmd->supported = (SUPPORTED_Autoneg |
		SUPPORTED_10baseT_Half | SUPPORTED_10baseT_Full |
		SUPPORTED_100baseT_Half | SUPPORTED_100baseT_Full |
		SUPPORTED_MII);
	if (np->gigabit == PHY_GIGABIT)
		ecmd->supported |= SUPPORTED_1000baseT_Full;

	ecmd->phy_address = np->phyaddr;
	ecmd->transceiver = XCVR_EXTERNAL;

	/* ignore maxtxpkt, maxrxpkt for now */
	spin_unlock_irq(&np->lock);
	return 0;
}

static int nv_set_settings(struct net_device *dev, struct ethtool_cmd *ecmd)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (ecmd->port != PORT_MII)
		return -EINVAL;
	if (ecmd->transceiver != XCVR_EXTERNAL)
		return -EINVAL;
	if (ecmd->phy_address != np->phyaddr) {
		/* TODO: support switching between multiple phys. Should be
		 * trivial, but not enabled due to lack of test hardware. */
		return -EINVAL;
	}
	if (ecmd->autoneg == AUTONEG_ENABLE) {
		u32 mask;

		mask = ADVERTISED_10baseT_Half | ADVERTISED_10baseT_Full |
			  ADVERTISED_100baseT_Half | ADVERTISED_100baseT_Full;
		if (np->gigabit == PHY_GIGABIT)
			mask |= ADVERTISED_1000baseT_Full;

		if ((ecmd->advertising & mask) == 0)
			return -EINVAL;

	} else if (ecmd->autoneg == AUTONEG_DISABLE) {
		/* Note: autonegotiation disable, speed 1000 intentionally
		 * forbidden - noone should need that. */

		if (ecmd->speed != SPEED_10 && ecmd->speed != SPEED_100)
			return -EINVAL;
		if (ecmd->duplex != DUPLEX_HALF && ecmd->duplex != DUPLEX_FULL)
			return -EINVAL;
	} else {
		return -EINVAL;
	}

	netif_carrier_off(dev);
	if (netif_running(dev)) {
		nv_disable_irq(dev);
		spin_lock_bh(&dev->xmit_lock);
		spin_lock(&np->lock);
		/* stop engines */
		nv_stop_rx(dev);
		nv_stop_tx(dev);
		spin_unlock(&np->lock);
		spin_unlock_bh(&dev->xmit_lock);
	}

	if (ecmd->autoneg == AUTONEG_ENABLE) {
		int adv, bmcr;

		np->autoneg = 1;

		/* advertise only what has been requested */
		adv = mii_rw(dev, np->phyaddr, MII_ADVERTISE, MII_READ);
		adv &= ~(ADVERTISE_ALL | ADVERTISE_100BASE4 | ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM);
		if (ecmd->advertising & ADVERTISED_10baseT_Half)
			adv |= ADVERTISE_10HALF;
		if (ecmd->advertising & ADVERTISED_10baseT_Full)
			adv |= ADVERTISE_10FULL;
		if (ecmd->advertising & ADVERTISED_100baseT_Half)
			adv |= ADVERTISE_100HALF;
		if (ecmd->advertising & ADVERTISED_100baseT_Full)
			adv |= ADVERTISE_100FULL;
		if (np->pause_flags & NV_PAUSEFRAME_RX_REQ)  /* for rx we set both advertisments but disable tx pause */
			adv |=  ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM;
		if (np->pause_flags & NV_PAUSEFRAME_TX_REQ)
			adv |=  ADVERTISE_PAUSE_ASYM;
		mii_rw(dev, np->phyaddr, MII_ADVERTISE, adv);

		if (np->gigabit == PHY_GIGABIT) {
			adv = mii_rw(dev, np->phyaddr, MII_CTRL1000, MII_READ);
			adv &= ~ADVERTISE_1000FULL;
			if (ecmd->advertising & ADVERTISED_1000baseT_Full)
				adv |= ADVERTISE_1000FULL;
			mii_rw(dev, np->phyaddr, MII_CTRL1000, adv);
		}
		if (netif_running(dev))
			printk(KERN_INFO "%s: link down.\n", dev->name);
		bmcr = mii_rw(dev, np->phyaddr, MII_BMCR, MII_READ);
		if (np->phy_model == PHY_MODEL_MARVELL_E3016) {
			bmcr |= BMCR_ANENABLE;
			/* reset the phy in order for settings to stick,
			 * and cause autoneg to start */
			if (phy_reset(dev, bmcr)) {
				printk(KERN_INFO "%s: phy reset failed\n", dev->name);
				return -EINVAL;
			}
		} else {
		bmcr |= (BMCR_ANENABLE | BMCR_ANRESTART);
		mii_rw(dev, np->phyaddr, MII_BMCR, bmcr);
		}
	} else {
		int adv, bmcr;

		np->autoneg = 0;

		adv = mii_rw(dev, np->phyaddr, MII_ADVERTISE, MII_READ);
		adv &= ~(ADVERTISE_ALL | ADVERTISE_100BASE4 | ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM);
		if (ecmd->speed == SPEED_10 && ecmd->duplex == DUPLEX_HALF)
			adv |= ADVERTISE_10HALF;
		if (ecmd->speed == SPEED_10 && ecmd->duplex == DUPLEX_FULL)
			adv |= ADVERTISE_10FULL;
		if (ecmd->speed == SPEED_100 && ecmd->duplex == DUPLEX_HALF)
			adv |= ADVERTISE_100HALF;
		if (ecmd->speed == SPEED_100 && ecmd->duplex == DUPLEX_FULL)
			adv |= ADVERTISE_100FULL;
		np->pause_flags &= ~(NV_PAUSEFRAME_AUTONEG|NV_PAUSEFRAME_RX_ENABLE|NV_PAUSEFRAME_TX_ENABLE);
		if (np->pause_flags & NV_PAUSEFRAME_RX_REQ) {/* for rx we set both advertisments but disable tx pause */
			adv |=  ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM;
			np->pause_flags |= NV_PAUSEFRAME_RX_ENABLE;
		}
		if (np->pause_flags & NV_PAUSEFRAME_TX_REQ) {
			adv |=  ADVERTISE_PAUSE_ASYM;
			np->pause_flags |= NV_PAUSEFRAME_TX_ENABLE;
		}
		mii_rw(dev, np->phyaddr, MII_ADVERTISE, adv);
		np->fixed_mode = adv;

		if (np->gigabit == PHY_GIGABIT) {
			adv = mii_rw(dev, np->phyaddr, MII_CTRL1000, MII_READ);
			adv &= ~ADVERTISE_1000FULL;
			mii_rw(dev, np->phyaddr, MII_CTRL1000, adv);
		}

		bmcr = mii_rw(dev, np->phyaddr, MII_BMCR, MII_READ);
		bmcr &= ~(BMCR_ANENABLE|BMCR_SPEED100|BMCR_SPEED1000|BMCR_FULLDPLX);
		if (np->fixed_mode & (ADVERTISE_10FULL|ADVERTISE_100FULL))
			bmcr |= BMCR_FULLDPLX;
		if (np->fixed_mode & (ADVERTISE_100HALF|ADVERTISE_100FULL))
			bmcr |= BMCR_SPEED100;
		if (np->phy_oui == PHY_OUI_MARVELL) {
			/* reset the phy in order for forced mode settings to stick */
			if (phy_reset(dev, bmcr)) {
				printk(KERN_INFO "%s: phy reset failed\n", dev->name);
				return -EINVAL;
			}
		}
 
		/* PR 178879: Force unlink/uplink */
		mii_rw(dev, np->phyaddr, MII_BMCR, bmcr);
		if (netif_running(dev)) {
			/* Wait a bit and then reconfigure the nic. */
			udelay(10);
			nv_linkchange(dev);
		}
	}

	if (netif_running(dev)) {
		nv_start_rx(dev);
		nv_start_tx(dev);
		nv_enable_irq(dev);
	}

	return 0;
}

#define FORCEDETH_REGS_VER	1

static int nv_get_regs_len(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	return np->register_size;
}

static void nv_get_regs(struct net_device *dev, struct ethtool_regs *regs, void *buf)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u32 *rbuf = buf;
	int i;

	regs->version = FORCEDETH_REGS_VER;
	spin_lock_irq(&np->lock);
	for (i = 0;i <= np->register_size/sizeof(u32); i++)
		rbuf[i] = readl(base + i*sizeof(u32));
	spin_unlock_irq(&np->lock);
}

static int nv_nway_reset(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	int ret;

	if (np->autoneg) {
		int bmcr;

		netif_carrier_off(dev);
		if (netif_running(dev)) {
			nv_disable_irq(dev);
			spin_lock_bh(&dev->xmit_lock);
			spin_lock(&np->lock);
			/* stop engines */
			nv_stop_rx(dev);
			nv_stop_tx(dev);
			spin_unlock(&np->lock);
			spin_unlock_bh(&dev->xmit_lock);
			printk(KERN_INFO "%s: link down.\n", dev->name);
		}

		bmcr = mii_rw(dev, np->phyaddr, MII_BMCR, MII_READ);
		if (np->phy_model == PHY_MODEL_MARVELL_E3016) {
			bmcr |= BMCR_ANENABLE;
			/* reset the phy in order for settings to stick*/
			if (phy_reset(dev, bmcr)) {
				printk(KERN_INFO "%s: phy reset failed\n", dev->name);
				return -EINVAL;
			}
		} else {
			bmcr |= (BMCR_ANENABLE | BMCR_ANRESTART);
			mii_rw(dev, np->phyaddr, MII_BMCR, bmcr);
		}

		if (netif_running(dev)) {
			nv_start_rx(dev);
			nv_start_tx(dev);
			nv_enable_irq(dev);
		}
		ret = 0;
	} else {
		ret = -EINVAL;
	}

	return ret;
}

static void nv_get_ringparam(struct net_device *dev, struct ethtool_ringparam* ring)
{
	struct fe_priv *np = get_nvpriv(dev);

	ring->rx_max_pending = (np->desc_ver == DESC_VER_1) ? RING_MAX_DESC_VER_1 : RING_MAX_DESC_VER_2_3;
	ring->rx_mini_max_pending = 0;
	ring->rx_jumbo_max_pending = 0;
	ring->tx_max_pending = (np->desc_ver == DESC_VER_1) ? RING_MAX_DESC_VER_1 : RING_MAX_DESC_VER_2_3;

	ring->rx_pending = np->rx_ring_size;
	ring->rx_mini_pending = 0;
	ring->rx_jumbo_pending = 0;
	ring->tx_pending = np->tx_ring_size;
}

static int nv_set_ringparam(struct net_device *dev, struct ethtool_ringparam* ring)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	u8 *rxtx_ring, *rx_skbuff, *tx_skbuff;
	dma_addr_t ring_addr;

	if (ring->rx_pending < RX_RING_MIN ||
	    ring->tx_pending < TX_RING_MIN ||
	    ring->rx_mini_pending != 0 ||
	    ring->rx_jumbo_pending != 0 ||
	    (np->desc_ver == DESC_VER_1 && 
	     (ring->rx_pending > RING_MAX_DESC_VER_1 || 
	      ring->tx_pending > RING_MAX_DESC_VER_1)) ||
	    (np->desc_ver != DESC_VER_1 && 
	     (ring->rx_pending > RING_MAX_DESC_VER_2_3 || 
	      ring->tx_pending > RING_MAX_DESC_VER_2_3))) {
		return -EINVAL;
	}

	/* allocate new rings */
	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
#ifdef __VMKERNEL_MODULE__
		/*
		 * We just do this for now, since pci_alloc_consistent
		 * in VMkernel ends up allocating full pages which makes
		 * this driver grab more than 3MB of memory.
		 * When this is changed, remember to also change the 
		 * corresponding freeing. */
		rxtx_ring = kmalloc(sizeof(struct ring_desc) * (ring->rx_pending + ring->tx_pending),
			GFP_ATOMIC);
		if (rxtx_ring) {
			ring_addr = virt_to_bus(rxtx_ring);
		}
#else
		rxtx_ring = pci_alloc_consistent(np->pci_dev,
					    sizeof(struct ring_desc) * (ring->rx_pending + ring->tx_pending),
					    &ring_addr);
#endif // __VMKERNEL_MODULE__
	} else {
#ifdef __VMKERNEL_MODULE__
		/*
		 * We just do this for now, since pci_alloc_consistent
		 * in VMkernel ends up allocating full pages which makes
		 * this driver grab more than 3MB of memory.
		 * When this is changed, remember to also change the 
		 * corresponding freeing. */
		rxtx_ring = kmalloc(sizeof(struct ring_desc) * (ring->rx_pending + ring->tx_pending),
			GFP_ATOMIC);
		if (rxtx_ring){
			ring_addr = virt_to_bus(rxtx_ring);
		}
#else
		rxtx_ring = pci_alloc_consistent(np->pci_dev,
					    sizeof(struct ring_desc_ex) * (ring->rx_pending + ring->tx_pending),
					    &ring_addr);
#endif // __VMKERNEL_MODULE__

	}
	rx_skbuff = kmalloc(sizeof(struct nv_skb_map) * ring->rx_pending, GFP_KERNEL);
	tx_skbuff = kmalloc(sizeof(struct nv_skb_map) * ring->tx_pending, GFP_KERNEL);

	if (!rxtx_ring || !rx_skbuff || !tx_skbuff) {
		/* fall back to old rings */
		if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
			if(rxtx_ring)
#ifdef __VMKERNEL_MODULE__
				kfree(rxtx_ring);
#else
				pci_free_consistent(np->pci_dev, sizeof(struct ring_desc) * (ring->rx_pending + ring->tx_pending),
						    rxtx_ring, ring_addr);
#endif // __VMKERNEL_MODULE__
		} else {
			if (rxtx_ring)
#ifdef __VMKERNEL_MODULE__
				kfree(rxtx_ring);
#else
				pci_free_consistent(np->pci_dev, sizeof(struct ring_desc_ex) * (ring->rx_pending + ring->tx_pending),
						    rxtx_ring, ring_addr);
#endif // __VMKERNEL_MODULE__
		}
		if (rx_skbuff)
			kfree(rx_skbuff);
		if (tx_skbuff)
			kfree(tx_skbuff);
		goto exit;
	}

	if (netif_running(dev)) {
		nv_disable_irq(dev);
		spin_lock_bh(&dev->xmit_lock);
		spin_lock(&np->lock);
		/* stop engines */
		nv_stop_rx(dev);
		nv_stop_tx(dev);
		nv_txrx_reset(dev);
		/* drain queues */
		nv_drain_rx(dev);
		nv_drain_tx(dev);
		/* delete queues */
		free_rings(dev);
	}
	
	/* set new values */
	np->rx_ring_size = ring->rx_pending;
	np->tx_ring_size = ring->tx_pending;
	np->tx_limit_stop =np->tx_ring_size - TX_LIMIT_DIFFERENCE;
	np->tx_limit_start =np->tx_ring_size - TX_LIMIT_DIFFERENCE - 1;
	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
		np->rx_ring.orig = (struct ring_desc*)rxtx_ring;
		np->tx_ring.orig = &np->rx_ring.orig[np->rx_ring_size];
	} else {
		np->rx_ring.ex = (struct ring_desc_ex*)rxtx_ring;
		np->tx_ring.ex = &np->rx_ring.ex[np->rx_ring_size];
	}
	np->rx_skb = (struct nv_skb_map*)rx_skbuff;
	np->tx_skb = (struct nv_skb_map*)tx_skbuff;
	np->ring_addr = ring_addr;
	
	memset(np->rx_skb, 0, sizeof(struct nv_skb_map) * np->rx_ring_size);
	memset(np->tx_skb, 0, sizeof(struct nv_skb_map) * np->tx_ring_size);

	if (netif_running(dev)) {
		/* reinit driver view of the queues */
		set_bufsize(dev);
		if (nv_init_ring(dev)) {
			if (!np->in_shutdown)
				mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
		}
		
		/* reinit nic view of the queues */
		writel(np->rx_buf_sz, base + NvRegOffloadConfig);
		setup_hw_rings(dev, NV_SETUP_RX_RING | NV_SETUP_TX_RING);
		writel( ((np->rx_ring_size-1) << NVREG_RINGSZ_RXSHIFT) + ((np->tx_ring_size-1) << NVREG_RINGSZ_TXSHIFT),
			base + NvRegRingSizes);
		pci_push(base);
		writel(NVREG_TXRXCTL_KICK|np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);
		pci_push(base);
		
		/* restart engines */
		nv_start_rx(dev);
		nv_start_tx(dev);
		spin_unlock(&np->lock);
		spin_unlock_bh(&dev->xmit_lock);
		nv_enable_irq(dev);
	}
	return 0;
exit:
	return -ENOMEM;
}

static void nv_get_pauseparam(struct net_device *dev, struct ethtool_pauseparam* pause)
{
	struct fe_priv *np = get_nvpriv(dev);

	pause->autoneg = (np->pause_flags & NV_PAUSEFRAME_AUTONEG) != 0;
	pause->rx_pause = (np->pause_flags & NV_PAUSEFRAME_RX_ENABLE) != 0;
	pause->tx_pause = (np->pause_flags & NV_PAUSEFRAME_TX_ENABLE) != 0;
}

static int nv_set_pauseparam(struct net_device *dev, struct ethtool_pauseparam* pause)
{
	struct fe_priv *np = get_nvpriv(dev);
	int adv, bmcr;

	if ((!np->autoneg && np->duplex == 0) ||
	    (np->autoneg && !pause->autoneg && np->duplex == 0)) {
		printk(KERN_INFO "%s: can not set pause settings when forced link is in half duplex.\n", 
		       dev->name);
		return -EINVAL;
	}
	if (pause->tx_pause && !(np->pause_flags & NV_PAUSEFRAME_TX_CAPABLE)) {
		printk(KERN_INFO "%s: hardware does not support tx pause frames.\n", dev->name);
		return -EINVAL;
	}

	netif_carrier_off(dev);
	if (netif_running(dev)) {
		nv_disable_irq(dev);
		spin_lock_bh(&dev->xmit_lock);
		spin_lock(&np->lock);
		/* stop engines */
		nv_stop_rx(dev);
		nv_stop_tx(dev);
		spin_unlock(&np->lock);
		spin_unlock_bh(&dev->xmit_lock);
	}

	np->pause_flags &= ~(NV_PAUSEFRAME_RX_REQ|NV_PAUSEFRAME_TX_REQ);
	if (pause->rx_pause)
		np->pause_flags |= NV_PAUSEFRAME_RX_REQ;
	if (pause->tx_pause)
		np->pause_flags |= NV_PAUSEFRAME_TX_REQ;

	if (np->autoneg && pause->autoneg) {
		np->pause_flags |= NV_PAUSEFRAME_AUTONEG;

		adv = mii_rw(dev, np->phyaddr, MII_ADVERTISE, MII_READ);
		adv &= ~(ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM);
		if (np->pause_flags & NV_PAUSEFRAME_RX_REQ) /* for rx we set both advertisments but disable tx pause */
			adv |=  ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM;
		if (np->pause_flags & NV_PAUSEFRAME_TX_REQ)
			adv |=  ADVERTISE_PAUSE_ASYM;
		mii_rw(dev, np->phyaddr, MII_ADVERTISE, adv);

		if (netif_running(dev))
			printk(KERN_INFO "%s: link down.\n", dev->name);
		bmcr = mii_rw(dev, np->phyaddr, MII_BMCR, MII_READ);
		bmcr |= (BMCR_ANENABLE | BMCR_ANRESTART);
		mii_rw(dev, np->phyaddr, MII_BMCR, bmcr);
	} else {
		np->pause_flags &= ~(NV_PAUSEFRAME_AUTONEG|NV_PAUSEFRAME_RX_ENABLE|NV_PAUSEFRAME_TX_ENABLE);
		if (pause->rx_pause)
			np->pause_flags |= NV_PAUSEFRAME_RX_ENABLE;
		if (pause->tx_pause)
			np->pause_flags |= NV_PAUSEFRAME_TX_ENABLE;

		if (!netif_running(dev))
			nv_update_linkspeed(dev);
		else
			nv_update_pause(dev, np->pause_flags);
	}

	if (netif_running(dev)) {
		nv_start_rx(dev);
		nv_start_tx(dev);
		nv_enable_irq(dev);
	}
	return 0;
}

static u32 nv_get_rx_csum(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	return (np->rx_csum) != 0;
}

static int nv_set_rx_csum(struct net_device *dev, u32 data)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	int retcode = 0;

	if (np->driver_data & DEV_HAS_CHECKSUM) {

		if (data) {
			np->rx_csum = 1;
			np->txrxctl_bits |= NVREG_TXRXCTL_RXCHECK;
		} else {
			np->rx_csum = 0;
			/* vlan is dependent on rx checksum offload */
			if (!(np->vlanctl_bits & NVREG_VLANCONTROL_ENABLE))
				np->txrxctl_bits &= ~NVREG_TXRXCTL_RXCHECK;
		}

		if (netif_running(dev)) {
			spin_lock_irq(&np->lock);
			writel(np->txrxctl_bits, base + NvRegTxRxControl);
			spin_unlock_irq(&np->lock);
		}
	} else {
		return -EINVAL;
	}

	return retcode;
}

#ifdef NETIF_F_TSO
static int nv_set_tso(struct net_device *dev, u32 data)
{
	struct fe_priv *np = get_nvpriv(dev);
	
	if (np->driver_data & DEV_HAS_CHECKSUM){
#if NVVER < SUSE10 
		if(data){
			if(ethtool_op_get_sg(dev)==0)
					return -EINVAL;
		}
#endif
		return ethtool_op_set_tso(dev, data);
	}else
		return -EINVAL;
}
#endif

static int nv_set_sg(struct net_device *dev, u32 data)
{
	struct fe_priv *np = get_nvpriv(dev);
	
	if (np->driver_data & DEV_HAS_CHECKSUM){
#if NVVER < SUSE10
		if(data){
			if(ethtool_op_get_tx_csum(dev)==0)
					return -EINVAL;
		}
#ifdef NETIF_F_TSO
		if(!data)
		    	/* set tso off */
				nv_set_tso(dev,data);   
#endif
#endif
		return ethtool_op_set_sg(dev, data);
	}else
		return -EINVAL;
}

static int nv_set_tx_csum(struct net_device *dev, u32 data)
{
	struct fe_priv *np = get_nvpriv(dev);

#if NVVER < SUSE10
	/* set sg off if tx off */
	if(!data)
		nv_set_sg(dev,data);
#endif	
	if (np->driver_data & DEV_HAS_CHECKSUM)
#if NVVER > RHES4 
		return ethtool_op_set_tx_hw_csum(dev, data);
#else
	{
			if (data)
				dev->features |= NETIF_F_IP_CSUM;
			else
				dev->features &= ~NETIF_F_IP_CSUM;
			return 0;
	}
#endif
	else
		return -EINVAL;
}

static int nv_get_stats_count(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (np->driver_data & DEV_HAS_STATISTICS)
		return (sizeof(struct nv_ethtool_stats)/sizeof(u64));
	else
		return NV_STATS_COUNT_SW;
}

static void nv_get_ethtool_stats(struct net_device *dev, struct ethtool_stats *estats, u64 *buffer)
{
	struct fe_priv *np = get_nvpriv(dev);

	/* update stats */
	nv_do_stats_poll((unsigned long)dev);

	memcpy(buffer, &np->estats, nv_get_stats_count(dev)*sizeof(u64));
}

static int nv_self_test_count(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);

	if (np->driver_data & DEV_HAS_TEST_EXTENDED)
		return NV_TEST_COUNT_EXTENDED;
	else
		return NV_TEST_COUNT_BASE;
}

static int nv_link_test(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	int mii_status;

	mii_rw(dev, np->phyaddr, MII_BMSR, MII_READ);
	mii_status = mii_rw(dev, np->phyaddr, MII_BMSR, MII_READ);

	/* check phy link status */
	if (!(mii_status & BMSR_LSTATUS))
		return 0;
	else
		return 1;
}

static int nv_register_test(struct net_device *dev)
{
	u8 __iomem *base = get_hwbase(dev);
	int i = 0;
	u32 orig_read, new_read;

	do {
		orig_read = readl(base + nv_registers_test[i].reg);

		/* xor with mask to toggle bits */
		orig_read ^= nv_registers_test[i].mask;

		writel(orig_read, base + nv_registers_test[i].reg);

		new_read = readl(base + nv_registers_test[i].reg);

		if ((new_read & nv_registers_test[i].mask) != (orig_read & nv_registers_test[i].mask))
			return 0;

		/* restore original value */
		orig_read ^= nv_registers_test[i].mask;
		writel(orig_read, base + nv_registers_test[i].reg);

	} while (nv_registers_test[++i].reg != 0);

	return 1;
}

static int nv_interrupt_test(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	int ret = 1;
	int testcnt;
	u32 save_msi_flags, save_poll_interval = 0;

	if (netif_running(dev)) {
		/* free current irq */
		nv_free_irq(dev);
		save_poll_interval = readl(base+NvRegPollingInterval);
	}

	/* flag to test interrupt handler */
	np->intr_test = 0;

	/* setup test irq */
	save_msi_flags = np->msi_flags;
	np->msi_flags &= ~NV_MSI_X_VECTORS_MASK;
	np->msi_flags |= 0x001; /* setup 1 vector */
	if (nv_request_irq(dev, 1))
		return 0;

	/* setup timer interrupt */
	writel(NVREG_POLL_DEFAULT_CPU, base + NvRegPollingInterval);
	writel(NVREG_UNKSETUP6_VAL, base + NvRegUnknownSetupReg6);

	nv_enable_hw_interrupts(dev, NVREG_IRQ_TIMER);

	/* wait for at least one interrupt */
	nv_msleep(100);

	spin_lock_irq(&np->lock);

	/* flag should be set within ISR */
	testcnt = np->intr_test;
	if (!testcnt)
		ret = 2;

	nv_disable_hw_interrupts(dev, NVREG_IRQ_TIMER);
	if (!(np->msi_flags & NV_MSI_X_ENABLED))
		writel(NVREG_IRQSTAT_MASK, base + NvRegIrqStatus);
	else
		writel(NVREG_IRQSTAT_MASK, base + NvRegMSIXIrqStatus);

	spin_unlock_irq(&np->lock);

	nv_free_irq(dev);

	np->msi_flags = save_msi_flags;

	if (netif_running(dev)) {
		writel(save_poll_interval, base + NvRegPollingInterval);
		writel(NVREG_UNKSETUP6_VAL, base + NvRegUnknownSetupReg6);
		/* restore original irq */
		if (nv_request_irq(dev, 0))
			return 0;
	}

	return ret;
}

static int nv_loopback_test(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	struct sk_buff *tx_skb, *rx_skb;
	dma_addr_t test_dma_addr;
	u32 tx_flags_extra = (np->desc_ver == DESC_VER_1 ? NV_TX_LASTPACKET : NV_TX2_LASTPACKET);
	u32 Flags;
	int len, i, pkt_len;
	u8 *pkt_data;
	u32 filter_flags = 0;
	u32 misc1_flags = 0;
	int ret = 1;

	if (netif_running(dev)) {
		nv_disable_irq(dev);
		filter_flags = readl(base + NvRegPacketFilterFlags);
		misc1_flags = readl(base + NvRegMisc1);
	} else {
		nv_txrx_reset(dev);
	}

	/* reinit driver view of the rx queue */
	set_bufsize(dev);
	nv_init_ring(dev);

	/* setup hardware for loopback */
	writel(NVREG_MISC1_FORCE, base + NvRegMisc1);
	writel(NVREG_PFF_ALWAYS | NVREG_PFF_LOOPBACK, base + NvRegPacketFilterFlags);

	/* reinit nic view of the rx queue */
	writel(np->rx_buf_sz, base + NvRegOffloadConfig);
	setup_hw_rings(dev, NV_SETUP_RX_RING | NV_SETUP_TX_RING);
	writel( ((np->rx_ring_size-1) << NVREG_RINGSZ_RXSHIFT) + ((np->tx_ring_size-1) << NVREG_RINGSZ_TXSHIFT),
		base + NvRegRingSizes);
	pci_push(base);

	/* restart rx engine */
	nv_start_rx(dev);
	nv_start_tx(dev);

	/* setup packet for tx */
	pkt_len = ETH_DATA_LEN;
#ifdef __VMKERNEL_MODULE__
	tx_skb = alloc_skb(pkt_len, GFP_ATOMIC);
#else
	tx_skb = dev_alloc_skb(pkt_len);
#endif // __VMKERNEL_MODULE__
	pkt_data = skb_put(tx_skb, pkt_len);
	for (i = 0; i < pkt_len; i++)
		pkt_data[i] = (u8)(i & 0xff);
#ifdef __VMKERNEL_MODULE__
	test_dma_addr = tx_skb->headMA;
#else
	test_dma_addr = pci_map_single(np->pci_dev, tx_skb->data,
				       tx_skb->end-tx_skb->data, PCI_DMA_FROMDEVICE);
#endif // __VMKERNEL_MODULE__

	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
		np->tx_ring.orig[0].PacketBuffer = cpu_to_le32(test_dma_addr);
		np->tx_ring.orig[0].FlagLen = cpu_to_le32((pkt_len-1) | np->tx_flags | tx_flags_extra);
	} else {
		np->tx_ring.ex[0].PacketBufferHigh = cpu_to_le64(test_dma_addr) >> 32;
		np->tx_ring.ex[0].PacketBufferLow = cpu_to_le64(test_dma_addr) & 0x0FFFFFFFF;
		np->tx_ring.ex[0].FlagLen = cpu_to_le32((pkt_len-1) | np->tx_flags | tx_flags_extra);
	}
	writel(NVREG_TXRXCTL_KICK|np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);
	pci_push(get_hwbase(dev));

	nv_msleep(500);

	/* check for rx of the packet */
	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
		Flags = le32_to_cpu(np->rx_ring.orig[0].FlagLen);
		len = nv_descr_getlength(&np->rx_ring.orig[0], np->desc_ver);
			
	} else {
		Flags = le32_to_cpu(np->rx_ring.ex[0].FlagLen);
		len = nv_descr_getlength_ex(&np->rx_ring.ex[0], np->desc_ver);
	}

	if (Flags & NV_RX_AVAIL) {
		ret = 0;
	} else if (np->desc_ver == DESC_VER_1) {
		if (Flags & NV_RX_ERROR)
			ret = 0;
	} else {
		if (Flags & NV_RX2_ERROR) {
			ret = 0;
		}
	}

	if (ret) {		
		if (len != pkt_len) {
			ret = 0;
			dprintk(KERN_DEBUG "%s: loopback len mismatch %d vs %d\n", 
				dev->name, len, pkt_len);
		} else {
			rx_skb = np->rx_skb[0].skb;
			for (i = 0; i < pkt_len; i++) {
				if (rx_skb->data[i] != (u8)(i & 0xff)) {
					ret = 0;
					dprintk(KERN_DEBUG "%s: loopback pattern check failed on byte %d\n", 
						dev->name, i);
					break;
				}
			}
		}
	} else {
		dprintk(KERN_DEBUG "%s: loopback - did not receive test packet\n", dev->name);
	}

#ifdef __VMKERNEL_MODULE__
	 /* VMkernel doesn't do pci_map...so there is no need to do
	  * a unmap.
	  */
#else
	pci_unmap_page(np->pci_dev, test_dma_addr,
		       tx_skb->end-tx_skb->data,
		       PCI_DMA_TODEVICE);
#endif // __VMKERNEL_MODULE__
	dev_kfree_skb_any(tx_skb);
	
	/* stop engines */
	nv_stop_rx(dev);
	nv_stop_tx(dev);
	nv_txrx_reset(dev);
	/* drain rx queue */
	nv_drain_rx(dev);
	nv_drain_tx(dev);

	if (netif_running(dev)) {
		writel(misc1_flags, base + NvRegMisc1);
		writel(filter_flags, base + NvRegPacketFilterFlags);
		nv_enable_irq(dev);
	}

	return ret;
}

static void nv_self_test(struct net_device *dev, struct ethtool_test *test, u64 *buffer)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	int result;
	memset(buffer, 0, nv_self_test_count(dev)*sizeof(u64));

	if (!nv_link_test(dev)) {
		test->flags |= ETH_TEST_FL_FAILED;
		buffer[0] = 1;
	}

	if (test->flags & ETH_TEST_FL_OFFLINE) {
		if (netif_running(dev)) {
			netif_stop_queue(dev);
			spin_lock_bh(&dev->xmit_lock);
			spin_lock_irq(&np->lock);
			nv_disable_hw_interrupts(dev, np->irqmask);
			if (!(np->msi_flags & NV_MSI_X_ENABLED)) {
				writel(NVREG_IRQSTAT_MASK, base + NvRegIrqStatus);
			} else {
				writel(NVREG_IRQSTAT_MASK, base + NvRegMSIXIrqStatus);
			}
			/* stop engines */
			nv_stop_rx(dev);
			nv_stop_tx(dev);
			nv_txrx_reset(dev);
			/* drain rx queue */
			nv_drain_rx(dev);
			nv_drain_tx(dev);
			spin_unlock_irq(&np->lock);
			spin_unlock_bh(&dev->xmit_lock);
		}

		if (!nv_register_test(dev)) {
			test->flags |= ETH_TEST_FL_FAILED;
			buffer[1] = 1;
		}

		result = nv_interrupt_test(dev);
		if (result != 1) {
			test->flags |= ETH_TEST_FL_FAILED;
			buffer[2] = 1;
		}
		if (result == 0) {
			/* bail out */
			return;
		}

		if (!nv_loopback_test(dev)) {
			test->flags |= ETH_TEST_FL_FAILED;
			buffer[3] = 1;
		}

		if (netif_running(dev)) {
			/* reinit driver view of the rx queue */
			set_bufsize(dev);
			if (nv_init_ring(dev)) {
				if (!np->in_shutdown)
					mod_timer(&np->oom_kick, jiffies + OOM_REFILL);
			}
			/* reinit nic view of the rx queue */
			writel(np->rx_buf_sz, base + NvRegOffloadConfig);
			setup_hw_rings(dev, NV_SETUP_RX_RING | NV_SETUP_TX_RING);
			writel( ((np->rx_ring_size-1) << NVREG_RINGSZ_RXSHIFT) + ((np->tx_ring_size-1) << NVREG_RINGSZ_TXSHIFT),
				base + NvRegRingSizes);
			pci_push(base);
			writel(NVREG_TXRXCTL_KICK|np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);
			pci_push(base);
			/* restart rx engine */
			nv_start_rx(dev);
			nv_start_tx(dev);
			netif_start_queue(dev);
			nv_enable_hw_interrupts(dev, np->irqmask);
		}
	}
}

static void nv_get_strings(struct net_device *dev, u32 stringset, u8 *buffer)
{
	switch (stringset) {
	case ETH_SS_STATS:
		memcpy(buffer, &nv_estats_str, nv_get_stats_count(dev)*sizeof(struct nv_ethtool_str));
		break;
	case ETH_SS_TEST:
		memcpy(buffer, &nv_etests_str, nv_self_test_count(dev)*sizeof(struct nv_ethtool_str));
		break;
	}
}

static struct ethtool_ops ops = {
	.get_drvinfo = nv_get_drvinfo,
	.get_link = ethtool_op_get_link,
	.get_wol = nv_get_wol,
	.set_wol = nv_set_wol,
	.get_settings = nv_get_settings,
	.set_settings = nv_set_settings,
	.get_regs_len = nv_get_regs_len,
	.get_regs = nv_get_regs,
	.nway_reset = nv_nway_reset,
#if NVVER > SUSE10
	.get_perm_addr = ethtool_op_get_perm_addr,
#endif
	.get_ringparam = nv_get_ringparam,
	.set_ringparam = nv_set_ringparam,
	.get_pauseparam = nv_get_pauseparam,
	.set_pauseparam = nv_set_pauseparam,
	.get_rx_csum = nv_get_rx_csum,
	.set_rx_csum = nv_set_rx_csum,
	.get_tx_csum = ethtool_op_get_tx_csum,
	.set_tx_csum = nv_set_tx_csum,
	.get_sg = ethtool_op_get_sg,
	.set_sg = nv_set_sg,
#ifdef NETIF_F_TSO
	.get_tso = ethtool_op_get_tso,
	.set_tso = nv_set_tso,
#endif
	.get_strings = nv_get_strings,
	.get_stats_count = nv_get_stats_count,
	.get_ethtool_stats = nv_get_ethtool_stats,
	.self_test_count = nv_self_test_count,
	.self_test = nv_self_test,
};

static void nv_vlan_rx_register(struct net_device *dev, struct vlan_group *grp)
{
	struct fe_priv *np = get_nvpriv(dev);

	spin_lock_irq(&np->lock);

	/* save vlan group */
	np->vlangrp = grp;

	if (grp) {
		/* enable vlan on MAC */
		np->txrxctl_bits |= NVREG_TXRXCTL_VLANSTRIP | NVREG_TXRXCTL_VLANINS;
		np->txrxctl_bits |= NVREG_TXRXCTL_RXCHECK;
	} else {
		/* disable vlan on MAC */
		np->txrxctl_bits &= ~NVREG_TXRXCTL_VLANSTRIP;
		np->txrxctl_bits &= ~NVREG_TXRXCTL_VLANINS;
		if (!np->rx_csum)
			np->txrxctl_bits &= ~NVREG_TXRXCTL_RXCHECK;
	}

	writel(np->txrxctl_bits, get_hwbase(dev) + NvRegTxRxControl);

	spin_unlock_irq(&np->lock);
};

static void nv_vlan_rx_kill_vid(struct net_device *dev, unsigned short vid)
{
	/* nothing to do */
};

/* The mgmt unit and driver use a semaphore to access the phy during init */
static int nv_mgmt_acquire_sema(struct net_device *dev)
{
	u8 __iomem *base = get_hwbase(dev);
	int i;
	u32 tx_ctrl, mgmt_sema;

	for (i = 0; i < 10; i++) {
		mgmt_sema = readl(base + NvRegTransmitterControl) & NVREG_XMITCTL_MGMT_SEMA_MASK;
		if (mgmt_sema == NVREG_XMITCTL_MGMT_SEMA_FREE) {
                        dprintk(KERN_INFO "forcedeth: nv_mgmt_acquire_sema: sema is free\n");
			break;
                }
		nv_msleep(500);
	}

	if (mgmt_sema != NVREG_XMITCTL_MGMT_SEMA_FREE) {
                dprintk(KERN_INFO "forcedeth: nv_mgmt_acquire_sema: sema is not free\n");
		return 0;
        }

	for (i = 0; i < 2; i++) {
		tx_ctrl = readl(base + NvRegTransmitterControl);
		tx_ctrl |= NVREG_XMITCTL_HOST_SEMA_ACQ;
		writel(tx_ctrl, base + NvRegTransmitterControl);

		/* verify that semaphore was acquired */
		tx_ctrl = readl(base + NvRegTransmitterControl);
		if (((tx_ctrl & NVREG_XMITCTL_HOST_SEMA_MASK) == NVREG_XMITCTL_HOST_SEMA_ACQ) &&
		    ((tx_ctrl & NVREG_XMITCTL_MGMT_SEMA_MASK) == NVREG_XMITCTL_MGMT_SEMA_FREE)) {
                        dprintk(KERN_INFO "forcedeth: nv_mgmt_acquire_sema: acquired sema\n");
			return 1;
		} else
			udelay(50);
	}

        dprintk(KERN_INFO "forcedeth: nv_mgmt_acquire_sema: exit\n");
	return 0;
}

/* Indicate to mgmt unit whether driver is loaded or not */
static void nv_mgmt_driver_loaded(struct net_device *dev, int loaded)
{
	u8 __iomem *base = get_hwbase(dev);
	u32 tx_ctrl;

	tx_ctrl = readl(base + NvRegTransmitterControl);
	if (loaded)
		tx_ctrl |= NVREG_XMITCTL_HOST_LOADED;
	else
		tx_ctrl &= ~NVREG_XMITCTL_HOST_LOADED;
	writel(tx_ctrl, base + NvRegTransmitterControl);
}

static int nv_open(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);
	int ret = 1;
	int oom, i;

	dprintk(KERN_DEBUG "nv_open: begin\n");

	/* erase previous misconfiguration */
	if (np->driver_data & DEV_HAS_POWER_CNTRL)
		nv_mac_reset(dev);
	/* stop adapter: ignored, 4.3 seems to be overkill */
	writel(NVREG_MCASTADDRA_FORCE, base + NvRegMulticastAddrA);
	writel(0, base + NvRegMulticastAddrB);
	writel(0, base + NvRegMulticastMaskA);
	writel(0, base + NvRegMulticastMaskB);
	writel(0, base + NvRegPacketFilterFlags);

	nv_stop_tx(dev);
	nv_stop_rx(dev);

	writel(0, base + NvRegAdapterControl);

	if (np->pause_flags & NV_PAUSEFRAME_TX_CAPABLE)
		writel(NVREG_TX_PAUSEFRAME_DISABLE,  base + NvRegTxPauseFrame);

	/* initialize descriptor rings */
	set_bufsize(dev);
	oom = nv_init_ring(dev);

	nv_txrx_reset(dev);
	writel(0, base + NvRegUnknownSetupReg6);

	np->in_shutdown = 0;

	/* give hw rings */
	setup_hw_rings(dev, NV_SETUP_RX_RING | NV_SETUP_TX_RING);
	writel( ((np->rx_ring_size-1) << NVREG_RINGSZ_RXSHIFT) + ((np->tx_ring_size-1) << NVREG_RINGSZ_TXSHIFT),
		base + NvRegRingSizes);

	/* continue setup */
	writel(np->linkspeed, base + NvRegLinkSpeed);
	if (np->desc_ver == DESC_VER_1)
		writel(NVREG_TX_WM_DESC1_DEFAULT, base + NvRegTxWatermark);
	else
		writel(NVREG_TX_WM_DESC2_3_DEFAULT, base + NvRegTxWatermark);
	writel(np->txrxctl_bits, base + NvRegTxRxControl);
	writel(np->vlanctl_bits, base + NvRegVlanControl);
	pci_push(base);
	writel(NVREG_TXRXCTL_BIT1|np->txrxctl_bits, base + NvRegTxRxControl);
	reg_delay(dev, NvRegUnknownSetupReg5, NVREG_UNKSETUP5_BIT31, NVREG_UNKSETUP5_BIT31,
			NV_SETUP5_DELAY, NV_SETUP5_DELAYMAX,
			KERN_INFO "open: SetupReg5, Bit 31 remained off\n");

	writel(0, base + NvRegMIIMask);
	writel(NVREG_IRQSTAT_MASK, base + NvRegIrqStatus);
	writel(NVREG_MIISTAT_MASK2, base + NvRegMIIStatus);

	/* continue setup */
	writel(NVREG_MISC1_FORCE | NVREG_MISC1_HD, base + NvRegMisc1);
	writel(readl(base + NvRegTransmitterStatus), base + NvRegTransmitterStatus);
	writel(NVREG_PFF_ALWAYS, base + NvRegPacketFilterFlags);
	writel(np->rx_buf_sz, base + NvRegOffloadConfig);

	writel(readl(base + NvRegReceiverStatus), base + NvRegReceiverStatus);
	get_random_bytes(&i, sizeof(i));
	writel(NVREG_RNDSEED_FORCE | (i&NVREG_RNDSEED_MASK), base + NvRegRandomSeed);
	writel(NVREG_TX_DEFERRAL_DEFAULT, base + NvRegTxDeferral);
	writel(NVREG_RX_DEFERRAL_DEFAULT, base + NvRegRxDeferral);
	if (poll_interval == -1) {
		if (optimization_mode == NV_OPTIMIZATION_MODE_THROUGHPUT)
			writel(NVREG_POLL_DEFAULT_THROUGHPUT, base + NvRegPollingInterval);
		else
			writel(NVREG_POLL_DEFAULT_CPU, base + NvRegPollingInterval);
	}
	else
		writel(poll_interval & 0xFFFF, base + NvRegPollingInterval);
	writel(NVREG_UNKSETUP6_VAL, base + NvRegUnknownSetupReg6);
	writel((np->phyaddr << NVREG_ADAPTCTL_PHYSHIFT)|NVREG_ADAPTCTL_PHYVALID|NVREG_ADAPTCTL_RUNNING,
			base + NvRegAdapterControl);
	writel(NVREG_MIISPEED_BIT8|NVREG_MIIDELAY, base + NvRegMIISpeed);
	writel(NVREG_MII_LINKCHANGE, base + NvRegMIIMask);
	if (np->wolenabled)
		writel(NVREG_WAKEUPFLAGS_ENABLE , base + NvRegWakeUpFlags);

	i = readl(base + NvRegPowerState);
	if ( (i & NVREG_POWERSTATE_POWEREDUP) == 0)
		writel(NVREG_POWERSTATE_POWEREDUP|i, base + NvRegPowerState);

	pci_push(base);
	udelay(10);
	writel(readl(base + NvRegPowerState) | NVREG_POWERSTATE_VALID, base + NvRegPowerState);

	nv_disable_hw_interrupts(dev, np->irqmask);
	pci_push(base);
	writel(NVREG_MIISTAT_MASK2, base + NvRegMIIStatus);
	writel(NVREG_IRQSTAT_MASK, base + NvRegIrqStatus);
	pci_push(base);

	if (nv_request_irq(dev, 0)) {
		goto out_drain;
	}

	/* ask for interrupts */
	nv_enable_hw_interrupts(dev, np->irqmask);

	spin_lock_irq(&np->lock);
	writel(NVREG_MCASTADDRA_FORCE, base + NvRegMulticastAddrA);
	writel(0, base + NvRegMulticastAddrB);
	writel(0, base + NvRegMulticastMaskA);
	writel(0, base + NvRegMulticastMaskB);
	writel(NVREG_PFF_ALWAYS|NVREG_PFF_MYADDR, base + NvRegPacketFilterFlags);
	/* One manual link speed update: Interrupts are enabled, future link
	 * speed changes cause interrupts and are handled by nv_link_irq().
	 */
	{
		u32 miistat;
		miistat = readl(base + NvRegMIIStatus);
		writel(NVREG_MIISTAT_MASK, base + NvRegMIIStatus);
		dprintk(KERN_INFO "startup: got 0x%08x.\n", miistat);
	}
	/* set linkspeed to invalid value, thus force nv_update_linkspeed
	 * to init hw */
	np->linkspeed = 0;
	ret = nv_update_linkspeed(dev);
	nv_start_rx(dev);
	nv_start_tx(dev);
	netif_start_queue(dev);
	if (ret) {
		netif_carrier_on(dev);
	} else {
		printk("%s: no link during initialization.\n", dev->name);
		netif_carrier_off(dev);
	}
	if (oom)
		mod_timer(&np->oom_kick, jiffies + OOM_REFILL);

	/* start statistics timer */
	mod_timer(&np->stats_poll, jiffies + STATS_INTERVAL);

	spin_unlock_irq(&np->lock);

	return 0;
out_drain:
	drain_ring(dev);
	return ret;
}

static int nv_close(struct net_device *dev)
{
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base;

	spin_lock_irq(&np->lock);
	np->in_shutdown = 1;
	spin_unlock_irq(&np->lock);

#ifdef __VMKERNEL_MODULE__
	vmk_synchronize_irq(dev->irq);
#elif NVVER > RHES3
	synchronize_irq(dev->irq);
#else
	synchronize_irq();
#endif //__VMKERNEL_MODULE_

	del_timer_sync(&np->oom_kick);
	del_timer_sync(&np->nic_poll);
	del_timer_sync(&np->stats_poll);

	netif_stop_queue(dev);
	spin_lock_irq(&np->lock);
	nv_stop_tx(dev);
	nv_stop_rx(dev);
	nv_txrx_reset(dev);

	/* disable interrupts on the nic or we will lock up */
	base = get_hwbase(dev);
	nv_disable_hw_interrupts(dev, np->irqmask);
	pci_push(base);
	dprintk(KERN_INFO "%s: Irqmask is zero again\n", dev->name);

	spin_unlock_irq(&np->lock);

	nv_free_irq(dev);

	drain_ring(dev);

#if 1 || __VMKERNEL_MODULE__ // fix PR 130686
	if (np->wolenabled) {
		writel(NVREG_PFF_ALWAYS|NVREG_PFF_MYADDR,
                       base + NvRegPacketFilterFlags);
		nv_start_rx(dev);
	}
#else
	if (np->wolenabled)
		nv_start_rx(dev);
#endif

	/* FIXME: power down nic */

	return 0;
}

static int __devinit nv_probe(struct pci_dev *pci_dev, const struct pci_device_id *id)
{
	struct net_device *dev;
	struct fe_priv *np;
	unsigned long addr;
	u8 __iomem *base;
	int err, i;
	u32 powerstate, phystate_orig = 0, phystate, txreg;
	int phyinitialized = 0;

	/* modify network device class id */	
	quirk_nforce_network_class(pci_dev);
	dev = alloc_etherdev(sizeof(struct fe_priv));
	err = -ENOMEM;
	if (!dev)
		goto out;

	np = get_nvpriv(dev);
	np->pci_dev = pci_dev;
	spin_lock_init(&np->lock);
	SET_MODULE_OWNER(dev);
	SET_NETDEV_DEV(dev, &pci_dev->dev);

	init_timer(&np->oom_kick);
	np->oom_kick.data = (unsigned long) dev;
	np->oom_kick.function = &nv_do_rx_refill;	/* timer handler */
	init_timer(&np->nic_poll);
	np->nic_poll.data = (unsigned long) dev;
	np->nic_poll.function = &nv_do_nic_poll;	/* timer handler */
	init_timer(&np->stats_poll);
	np->stats_poll.data = (unsigned long) dev;
	np->stats_poll.function = &nv_do_stats_poll;	/* timer handler */

	err = pci_enable_device(pci_dev);
	if (err) {
		printk(KERN_INFO "forcedeth: pci_enable_dev failed (%d) for device %s\n",
				err, pci_name(pci_dev));
		goto out_free;
	}

	pci_set_master(pci_dev);

	err = pci_request_regions(pci_dev, DRV_NAME);
	if (err < 0)
		goto out_disable;

	if (id->driver_data & (DEV_HAS_VLAN|DEV_HAS_MSI_X|DEV_HAS_POWER_CNTRL|DEV_HAS_STATISTICS))
		np->register_size = NV_PCI_REGSZ_VER2;
	else
		np->register_size = NV_PCI_REGSZ_VER1;

	err = -EINVAL;
	addr = 0;
	for (i = 0; i < DEVICE_COUNT_RESOURCE; i++) {
		dprintk(KERN_DEBUG "%s: resource %d start %p len %ld flags 0x%08lx.\n",
				pci_name(pci_dev), i, (void*)pci_resource_start(pci_dev, i),
				pci_resource_len(pci_dev, i),
				pci_resource_flags(pci_dev, i));
		if (pci_resource_flags(pci_dev, i) & IORESOURCE_MEM &&
				pci_resource_len(pci_dev, i) >= np->register_size) {
			addr = pci_resource_start(pci_dev, i);
			break;
		}
	}
	if (i == DEVICE_COUNT_RESOURCE) {
		printk(KERN_INFO "forcedeth: Couldn't find register window for device %s.\n",
					pci_name(pci_dev));
		goto out_relreg;
	}

	/* copy of driver data */
	np->driver_data = id->driver_data;

	/* handle different descriptor versions */
	if (id->driver_data & DEV_HAS_HIGH_DMA) {
		/* packet format 3: supports 40-bit addressing */
		np->desc_ver = DESC_VER_3;
		np->txrxctl_bits = NVREG_TXRXCTL_DESC_3;
		if (dma_64bit) {
			if (pci_set_dma_mask(pci_dev, DMA_39BIT_MASK)) {
				printk(KERN_INFO "forcedeth: 64-bit DMA failed, using 32-bit addressing for device %s.\n",
				       pci_name(pci_dev));
			} else {
				dev->features |= NETIF_F_HIGHDMA;
				printk(KERN_INFO "forcedeth: using HIGHDMA\n");
			}
#if NVVER > RHES3
			if (pci_set_consistent_dma_mask(pci_dev, DMA_39BIT_MASK)) {
				printk(KERN_INFO "forcedeth: 64-bit DMA (consistent) failed, using 32-bit ring buffers for device %s.\n",
				       pci_name(pci_dev));
			}
#endif
		}
	} else if (id->driver_data & DEV_HAS_LARGEDESC) {
		/* packet format 2: supports jumbo frames */
		np->desc_ver = DESC_VER_2;
		np->txrxctl_bits = NVREG_TXRXCTL_DESC_2;
	} else {
		/* original packet format */
		np->desc_ver = DESC_VER_1;
		np->txrxctl_bits = NVREG_TXRXCTL_DESC_1;
	}

	np->pkt_limit = NV_PKTLIMIT_1;
	if (id->driver_data & DEV_HAS_LARGEDESC)
		np->pkt_limit = NV_PKTLIMIT_2;
	if (mtu > np->pkt_limit) {
		printk(KERN_INFO "forcedeth: MTU value of %d is too large. Setting to maximum value of %d\n",
		       mtu, np->pkt_limit);
		dev->mtu = np->pkt_limit;
	} else {
		dev->mtu = mtu;
	}

	if (id->driver_data & DEV_HAS_CHECKSUM) {
		if (rx_checksum_offload) {
			np->rx_csum = 1;
			np->txrxctl_bits |= NVREG_TXRXCTL_RXCHECK;
		}

		if (tx_checksum_offload)	
#if NVVER > RHES4
			dev->features |= NETIF_F_HW_CSUM;
#else
			dev->features |= NETIF_F_IP_CSUM;
#endif

		if (scatter_gather)
			dev->features |= NETIF_F_SG;
#ifdef NETIF_F_TSO
		if (tso_offload)
			dev->features |= NETIF_F_TSO;
#endif
 	}

	np->vlanctl_bits = 0;
	if (id->driver_data & DEV_HAS_VLAN && tagging_8021pq) {
			np->vlanctl_bits = NVREG_VLANCONTROL_ENABLE;
			dev->features |= NETIF_F_HW_VLAN_RX | NETIF_F_HW_VLAN_TX;
			dev->vlan_rx_register = nv_vlan_rx_register;
			dev->vlan_rx_kill_vid = nv_vlan_rx_kill_vid;
		/* vlan needs rx checksum support, so force it */
		np->txrxctl_bits |= NVREG_TXRXCTL_RXCHECK;
	}

	np->msi_flags = 0;
	if ((id->driver_data & DEV_HAS_MSI) && msi) {
		np->msi_flags |= NV_MSI_CAPABLE;
	}
	if ((id->driver_data & DEV_HAS_MSI_X) && msix) {
		np->msi_flags |= NV_MSI_X_CAPABLE;
	}
	
	np->pause_flags = NV_PAUSEFRAME_RX_CAPABLE;
	if (rx_flow_control == NV_RX_FLOW_CONTROL_ENABLED)
		np->pause_flags |= NV_PAUSEFRAME_RX_REQ;
	if (id->driver_data & DEV_HAS_PAUSEFRAME_TX) {
		np->pause_flags |= NV_PAUSEFRAME_TX_CAPABLE;
		if (tx_flow_control == NV_TX_FLOW_CONTROL_ENABLED)
			np->pause_flags |= NV_PAUSEFRAME_TX_REQ;
	}
	if (autoneg == AUTONEG_ENABLE) {
		np->pause_flags |= NV_PAUSEFRAME_AUTONEG;
	} else if (speed_duplex == NV_SPEED_DUPLEX_1000_FULL_DUPLEX) {
		printk(KERN_INFO "forcedeth: speed_duplex of 1000 full can not enabled if autoneg is disabled\n");
		goto out_relreg;
	}

	err = -ENOMEM;
	np->base = ioremap(addr, np->register_size);
	if (!np->base)
		goto out_relreg;
	dev->base_addr = (unsigned long)np->base;

	dev->irq = pci_dev->irq;

	if (np->desc_ver == DESC_VER_1) {
		if (rx_ring_size > RING_MAX_DESC_VER_1) {
			printk(KERN_INFO "forcedeth: rx_ring_size of %d is too large. Setting to maximum of %d\n",
			       rx_ring_size, RING_MAX_DESC_VER_1);
			rx_ring_size = RING_MAX_DESC_VER_1;
		}
		if (tx_ring_size > RING_MAX_DESC_VER_1) {
			printk(KERN_INFO "forcedeth: tx_ring_size of %d is too large. Setting to maximum of %d\n",
			       tx_ring_size, RING_MAX_DESC_VER_1);
			tx_ring_size = RING_MAX_DESC_VER_1;
		}
	} else {
		if (rx_ring_size > RING_MAX_DESC_VER_2_3) {
			printk(KERN_INFO "forcedeth: rx_ring_size of %d is too large. Setting to maximum of %d\n",
			       rx_ring_size, RING_MAX_DESC_VER_2_3);
			rx_ring_size = RING_MAX_DESC_VER_2_3;
		}
		if (tx_ring_size > RING_MAX_DESC_VER_2_3) {
			printk(KERN_INFO "forcedeth: tx_ring_size of %d is too large. Setting to maximum of %d\n",
			       tx_ring_size, RING_MAX_DESC_VER_2_3);
			tx_ring_size = RING_MAX_DESC_VER_2_3;
		}
	}
	np->rx_ring_size = rx_ring_size;
	np->tx_ring_size = tx_ring_size;
	np->tx_limit_stop = tx_ring_size - TX_LIMIT_DIFFERENCE;
	np->tx_limit_start = tx_ring_size - TX_LIMIT_DIFFERENCE - 1;

	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2) {
		np->rx_ring.orig = pci_alloc_consistent(pci_dev,
					sizeof(struct ring_desc) * (np->rx_ring_size + np->tx_ring_size),
					&np->ring_addr);
		if (!np->rx_ring.orig)
			goto out_unmap;
		np->tx_ring.orig = &np->rx_ring.orig[np->rx_ring_size];
	} else {
		np->rx_ring.ex = pci_alloc_consistent(pci_dev,
					sizeof(struct ring_desc_ex) * (np->rx_ring_size + np->tx_ring_size),
					&np->ring_addr);
		if (!np->rx_ring.ex)
			goto out_unmap;
		np->tx_ring.ex = &np->rx_ring.ex[np->rx_ring_size];
	}
	np->rx_skb = kmalloc(sizeof(struct nv_skb_map) * np->rx_ring_size, GFP_KERNEL);
	np->tx_skb = kmalloc(sizeof(struct nv_skb_map) * np->tx_ring_size, GFP_KERNEL);
	if (!np->rx_skb || !np->tx_skb)
		goto out_freering;
	memset(np->rx_skb, 0, sizeof(struct nv_skb_map) * np->rx_ring_size);
	memset(np->tx_skb, 0, sizeof(struct nv_skb_map) * np->tx_ring_size);

	dev->open = nv_open;
	dev->stop = nv_close;
	if (np->desc_ver == DESC_VER_1 || np->desc_ver == DESC_VER_2)
		dev->hard_start_xmit = nv_start_xmit;
	else
		dev->hard_start_xmit = nv_start_xmit_optimized;
	dev->get_stats = nv_get_stats;
	dev->change_mtu = nv_change_mtu;
	dev->set_mac_address = nv_set_mac_address;
	dev->set_multicast_list = nv_set_multicast;

#if NVVER < SLES9
	dev->do_ioctl = nv_ioctl;
#endif

#if NVVER > RHES3
#ifdef CONFIG_NET_POLL_CONTROLLER
	dev->poll_controller = nv_poll_controller;
#endif
#else
	dev->poll_controller = nv_poll_controller;
#endif

	SET_ETHTOOL_OPS(dev, &ops);
	dev->tx_timeout = nv_tx_timeout;
	dev->watchdog_timeo = NV_WATCHDOG_TIMEO;

	pci_set_drvdata(pci_dev, dev);

	/* read the mac address */
	base = get_hwbase(dev);
	np->orig_mac[0] = readl(base + NvRegMacAddrA);
	np->orig_mac[1] = readl(base + NvRegMacAddrB);

	/* check the workaround bit for correct mac address order */
	txreg = readl(base + NvRegTransmitPoll);
	if (txreg & NVREG_TRANSMITPOLL_MAC_ADDR_REV) {
		/* mac address is already in correct order */
		dev->dev_addr[0] = (np->orig_mac[0] >>  0) & 0xff;
		dev->dev_addr[1] = (np->orig_mac[0] >>  8) & 0xff;
		dev->dev_addr[2] = (np->orig_mac[0] >> 16) & 0xff;
		dev->dev_addr[3] = (np->orig_mac[0] >> 24) & 0xff;
		dev->dev_addr[4] = (np->orig_mac[1] >>  0) & 0xff;
		dev->dev_addr[5] = (np->orig_mac[1] >>  8) & 0xff;
	} else {
	dev->dev_addr[0] = (np->orig_mac[1] >>  8) & 0xff;
	dev->dev_addr[1] = (np->orig_mac[1] >>  0) & 0xff;
	dev->dev_addr[2] = (np->orig_mac[0] >> 24) & 0xff;
	dev->dev_addr[3] = (np->orig_mac[0] >> 16) & 0xff;
	dev->dev_addr[4] = (np->orig_mac[0] >>  8) & 0xff;
	dev->dev_addr[5] = (np->orig_mac[0] >>  0) & 0xff;
		/* set permanent address to be correct aswell */
		np->orig_mac[0] = (dev->dev_addr[0] << 0) + (dev->dev_addr[1] << 8) +
			(dev->dev_addr[2] << 16) + (dev->dev_addr[3] << 24);
		np->orig_mac[1] = (dev->dev_addr[4] << 0) + (dev->dev_addr[5] << 8);
		writel(txreg|NVREG_TRANSMITPOLL_MAC_ADDR_REV, base + NvRegTransmitPoll);
	}
#if NVVER > SUSE10
	memcpy(dev->perm_addr, dev->dev_addr, dev->addr_len);

	if (!is_valid_ether_addr(dev->perm_addr)) {
#else
	if (!is_valid_ether_addr(dev->dev_addr)) {		
#endif	
		/*
		 * Bad mac address. At least one bios sets the mac address
		 * to 01:23:45:67:89:ab
		 */
		printk(KERN_ERR "%s: Invalid Mac address detected: %02x:%02x:%02x:%02x:%02x:%02x\n",
			pci_name(pci_dev),
			dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2],
			dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5]);
		printk(KERN_ERR "Please complain to your hardware vendor. Switching to a random MAC.\n");
		dev->dev_addr[0] = 0x00;
		dev->dev_addr[1] = 0x00;
		dev->dev_addr[2] = 0x6c;
		get_random_bytes(&dev->dev_addr[3], 3);
	}

	dprintk(KERN_DEBUG "%s: MAC Address %02x:%02x:%02x:%02x:%02x:%02x\n", pci_name(pci_dev),
			dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2],
			dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5]);
	/* set mac address */
	nv_copy_mac_to_hw(dev);

	/* disable WOL */
	writel(0, base + NvRegWakeUpFlags);
	np->wolenabled = wol;

	if (id->driver_data & DEV_HAS_POWER_CNTRL) {
		u8 revision_id;
		pci_read_config_byte(pci_dev, PCI_REVISION_ID, &revision_id);

		/* take phy and nic out of low power mode */
		powerstate = readl(base + NvRegPowerState2);
		powerstate &= ~NVREG_POWERSTATE2_POWERUP_MASK;
		if ((id->device == PCI_DEVICE_ID_NVIDIA_NVENET_12 ||
		     id->device == PCI_DEVICE_ID_NVIDIA_NVENET_13) &&
		    revision_id >= 0xA3)
			powerstate |= NVREG_POWERSTATE2_POWERUP_REV_A3;
		writel(powerstate, base + NvRegPowerState2);
	}

	if (np->desc_ver == DESC_VER_1) {
		np->tx_flags = NV_TX_VALID;
	} else {
		np->tx_flags = NV_TX2_VALID;
	}
	if (optimization_mode == NV_OPTIMIZATION_MODE_THROUGHPUT) {
		np->irqmask = NVREG_IRQMASK_THROUGHPUT;
		if (np->msi_flags & NV_MSI_X_CAPABLE) /* set number of vectors */
			np->msi_flags |= 0x0003;
	} else {
		np->irqmask = NVREG_IRQMASK_CPU;
		if (np->msi_flags & NV_MSI_X_CAPABLE) /* set number of vectors */
			np->msi_flags |= 0x0001;
	}

#ifndef __VMKERNEL_MODULE__
   if (id->driver_data & DEV_NEED_TIMERIRQ) {
#else
   if ((id->driver_data & DEV_NEED_TIMERIRQ) || (vmk_enable_timer_irq == 1)) {
      printk(KERN_INFO "WARNING: forcedeth loaded with vmk_enable_timer_irq = 1. \
         Performance impact.\n");
#endif
      np->irqmask |= NVREG_IRQ_TIMER;
   }
	if (id->driver_data & DEV_NEED_LINKTIMER) {
		dprintk(KERN_INFO "%s: link timer on.\n", pci_name(pci_dev));
		np->need_linktimer = 1;
		np->link_timeout = jiffies + LINK_TIMEOUT;
	} else {
		dprintk(KERN_INFO "%s: link timer off.\n", pci_name(pci_dev));
		np->need_linktimer = 0;
	}

	/* clear phy state and temporarily halt phy interrupts */
	writel(0, base + NvRegMIIMask);
	phystate = readl(base + NvRegAdapterControl);
	if (phystate & NVREG_ADAPTCTL_RUNNING) {
		phystate_orig = 1;
		phystate &= ~NVREG_ADAPTCTL_RUNNING;
		writel(phystate, base + NvRegAdapterControl);
	}
	writel(NVREG_MIISTAT_MASK, base + NvRegMIIStatus);

	if (id->driver_data & DEV_HAS_MGMT_UNIT) {
		writel(NV_UNKNOW_VAL, base + NvRegPatternCRC);
	       	pci_push(base);
		nv_msleep(500);
		/* management unit running on the mac? */
		np->mac_in_use = readl(base + NvRegTransmitterControl) & NVREG_XMITCTL_MGMT_ST;
		if (np->mac_in_use) {
			u32 mgmt_sync;
			dprintk(KERN_INFO "forcedeth: probe: mac in use\n");
			/* management unit setup the phy already? */
			mgmt_sync = readl(base + NvRegTransmitterControl) & NVREG_XMITCTL_SYNC_MASK;
			if (mgmt_sync == NVREG_XMITCTL_SYNC_NOT_READY) {
				dprintk(KERN_INFO "forcedeth: probe: sync not ready\n");
				if (!nv_mgmt_acquire_sema(dev)) {
					dprintk(KERN_INFO "forcedeth: probe: could not acquire sema\n");
					for (i = 0; i < 5000; i++) {
						nv_msleep(1);
						mgmt_sync = readl(base + NvRegTransmitterControl) & NVREG_XMITCTL_SYNC_MASK;
						if (mgmt_sync == NVREG_XMITCTL_SYNC_NOT_READY)
							continue;
						if (mgmt_sync == NVREG_XMITCTL_SYNC_PHY_INIT) {
							dprintk(KERN_INFO "forcedeth: probe: phy inited by SMU 1\n");
							phyinitialized = 1;
						}
						break;
						dprintk(KERN_INFO "forcedeth: probe: breaking out of loop\n");
					}
				} else {
					/* we need to init the phy */
					dprintk(KERN_INFO "forcedeth: probe: we need to init phy 1\n");
				}
			} else if (mgmt_sync == NVREG_XMITCTL_SYNC_PHY_INIT) {
				dprintk(KERN_INFO "forcedeth: probe: phy inited by SMU 2\n");
				/* phy is inited by SMU */
				phyinitialized = 1;
			} else {
				/* we need to init the phy */
				dprintk(KERN_INFO "forcedeth: probe: we need to init phy 2\n");
			}
		} else
			dprintk(KERN_INFO "forcedeth: probe: mac not in use\n");
	}

	/* find a suitable phy */
	for (i = 1; i <= 32; i++) {
		int id1, id2;
		int phyaddr = i & 0x1F;

		spin_lock_irq(&np->lock);
		id1 = mii_rw(dev, phyaddr, MII_PHYSID1, MII_READ);
		spin_unlock_irq(&np->lock);
		if (id1 < 0 || id1 == 0xffff)
			continue;
		spin_lock_irq(&np->lock);
		id2 = mii_rw(dev, phyaddr, MII_PHYSID2, MII_READ);
		spin_unlock_irq(&np->lock);
		if (id2 < 0 || id2 == 0xffff)
			continue;

		np->phy_model = id2 & PHYID2_MODEL_MASK;
		id1 = (id1 & PHYID1_OUI_MASK) << PHYID1_OUI_SHFT;
		id2 = (id2 & PHYID2_OUI_MASK) >> PHYID2_OUI_SHFT;
		dprintk(KERN_DEBUG "%s: open: Found PHY %04x:%04x at address %d.\n",
			pci_name(pci_dev), id1, id2, phyaddr);
		np->phyaddr = phyaddr;
		np->phy_oui = id1 | id2;
		break;
	}
	if (i == 33) {
		printk(KERN_INFO "%s: open: Could not find a valid PHY.\n",
		       pci_name(pci_dev));
		goto out_error;
	}
	
	if (!phyinitialized) {		
	/* reset it */
	phy_init(dev);
	} else {
		/* see if gigabit phy */
		u32 mii_status = mii_rw(dev, np->phyaddr, MII_BMSR, MII_READ);
		if (mii_status & PHY_GIGABIT) {
			np->gigabit = PHY_GIGABIT;
		}
	}
	if (id->driver_data & DEV_HAS_MGMT_UNIT) {
		nv_mgmt_driver_loaded(dev, 1);
	}

	/* set default link speed settings */
	np->linkspeed = NVREG_LINKSPEED_FORCE|NVREG_LINKSPEED_10;
	np->duplex = 0;
	np->autoneg = autoneg;

	err = register_netdev(dev);
	if (err) {
		printk(KERN_INFO "forcedeth: unable to register netdev: %d\n", err);
		goto out_error;
	}
	printk(KERN_INFO "%s: forcedeth.c: subsystem: %05x:%04x bound to %s\n",
			dev->name, pci_dev->subsystem_vendor, pci_dev->subsystem_device,
			pci_name(pci_dev));

	return 0;

out_error:
	if (phystate_orig)
		writel(phystate|NVREG_ADAPTCTL_RUNNING, base + NvRegAdapterControl);
	if (np->mac_in_use)
		nv_mgmt_driver_loaded(dev, 0);
	pci_set_drvdata(pci_dev, NULL);
out_freering:
	free_rings(dev);
out_unmap:
	iounmap(get_hwbase(dev));
out_relreg:
	pci_release_regions(pci_dev);
out_disable:
	pci_disable_device(pci_dev);
out_free:
	free_netdev(dev);
out:
	return err;
}

static void __devexit nv_remove(struct pci_dev *pci_dev)
{
	struct net_device *dev = pci_get_drvdata(pci_dev);
	struct fe_priv *np = get_nvpriv(dev);
	u8 __iomem *base = get_hwbase(dev);

	unregister_netdev(dev);
	/* special op: write back the misordered MAC address - otherwise
	 * the next nv_probe would see a wrong address.
	 */
	writel(np->orig_mac[0], base + NvRegMacAddrA);
	writel(np->orig_mac[1], base + NvRegMacAddrB);
	if (np->mac_in_use)
		nv_mgmt_driver_loaded(dev, 0);

	/* free all structures */
	free_rings(dev);
	iounmap(get_hwbase(dev));
	pci_release_regions(pci_dev);
	pci_disable_device(pci_dev);
	free_netdev(dev);
	pci_set_drvdata(pci_dev, NULL);
}

static struct pci_device_id pci_tbl[] = {
	/* nForce Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_1,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER)
	/* nForce2 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_2,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER)
	/* nForce3 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_3,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER)
	/* nForce3 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_4,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM)
	/* nForce3 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_5,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM)
	/* nForce3 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_6,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM)
	/* nForce3 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_7,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM)
	/* CK804 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_8,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA)
	/* CK804 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_9,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA)
	/* MCP04 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_10,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA)
	/* MCP04 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_11,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA)
	/* MCP51 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_12,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL)
	/* MCP51 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_13,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL)
	/* MCP55 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_14,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA|DEV_HAS_VLAN|DEV_HAS_MSI|DEV_HAS_MSI_X|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_POWER_CNTRL)
	/* MCP55 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_15,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA|DEV_HAS_VLAN|DEV_HAS_MSI|DEV_HAS_MSI_X|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_POWER_CNTRL)
	/* MCP61 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_16, DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT)
	/* MCP61 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_17,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT)
	/* MCP61 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_18, DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT)
	/* MCP61 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_19, DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT)
	/* MCP65 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_20, DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT)
	/* MCP65 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_21, DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT )
	/* MCP65 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_22,DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT)
	/* MCP65 Ethernet Controller */
	NVIDIA_ETHERNET_ID(PCI_DEVICE_ID_NVIDIA_NVENET_23, DEV_NEED_TIMERIRQ|DEV_NEED_LINKTIMER|DEV_HAS_LARGEDESC|DEV_HAS_CHECKSUM|DEV_HAS_HIGH_DMA|DEV_HAS_POWER_CNTRL|DEV_HAS_MSI|DEV_HAS_PAUSEFRAME_TX|DEV_HAS_STATISTICS|DEV_HAS_TEST_EXTENDED|DEV_HAS_MGMT_UNIT)
	{0,},
};

#ifdef __VMKERNEL_MODULE__
static int nv_notify_reboot(struct notifier_block *nb, unsigned long event, void *p);

struct notifier_block nv_notifier_reboot = {
        .notifier_call  = nv_notify_reboot,
        .next           = NULL,
        .priority       = 0
};
#endif

static struct pci_driver driver = {
	.name = "forcedeth",
	.id_table = pci_tbl,
	.probe = nv_probe,
	.remove = __devexit_p(nv_remove),
};

#ifdef __VMKERNEL_MODULE__
/*
 * The vmkernel currently does not close network devices before
 * powering off the host, so we add a reboot notifier to handle
 * powering off the NIC and enabling WoL if requested.  This follows
 * the style of the e100 and e1000 drivers we are currently using.
 * However, the latest versions of Linux 2.6 NIC drivers don't have
 * such notifiers.
 */
static int
nv_notify_reboot(struct notifier_block *nb, unsigned long event, void *p)
{
        struct pci_dev *pdev = NULL;
	
        switch(event) {
        case SYS_DOWN:
        case SYS_HALT:
        case SYS_POWER_OFF:
		while ((pdev = pci_find_device(PCI_ANY_ID, PCI_ANY_ID, pdev)) != NULL) {
                        if (pci_dev_driver(pdev) == &driver) {
                                struct net_device *dev = pci_get_drvdata(pdev);
                                if (dev) {
                                        dev_close(dev);
                                }
			}
		}
        }
        return NOTIFY_DONE;
}
#endif

static int __init init_nic(void)
{
#ifdef __VMKERNEL_MODULE__
        int ret;

        if (!vmk_set_module_version("%s", FORCEDETH_VERSION)) {
           return -ENODEV;
        }
	printk(KERN_INFO "forcedeth.c: Reverse Engineered nForce ethernet driver. Version %s.\n", FORCEDETH_VERSION);
	ret = pci_module_init(&driver);
        if (ret >= 0) {
                register_reboot_notifier(&nv_notifier_reboot);
        }
        return ret;
#else
	printk(KERN_INFO "forcedeth.c: Reverse Engineered nForce ethernet driver. Version %s.\n", FORCEDETH_VERSION);
	return pci_module_init(&driver);
#endif // __VMKERNEL_MODULE__
}

static void __exit exit_nic(void)
{
#ifdef __VMKERNEL_MODULE__
        unregister_reboot_notifier(&nv_notifier_reboot);
#endif
	pci_unregister_driver(&driver);
}

#if NVVER > SLES9
module_param(max_interrupt_work, int, 0);
MODULE_PARM_DESC(max_interrupt_work, "forcedeth maximum events handled per interrupt");
module_param(optimization_mode, int, 0);
MODULE_PARM_DESC(optimization_mode, "In throughput mode (0), every tx & rx packet will generate an interrupt. In CPU mode (1), interrupts are controlled by a timer.");
module_param(poll_interval, int, 0);
MODULE_PARM_DESC(poll_interval, "Interval determines how frequent timer interrupt is generated by [(time_in_micro_secs * 100) / (2^10)]. Min is 0 and Max is 65535.");
module_param(msi, int, 0);
MODULE_PARM_DESC(msi, "MSI interrupts are enabled by setting to 1 and disabled by setting to 0.");
module_param(msix, int, 0);
MODULE_PARM_DESC(msix, "MSIX interrupts are enabled by setting to 1 and disabled by setting to 0.");

module_param(speed_duplex, int, 0);
MODULE_PARM_DESC(speed_duplex, "PHY speed and duplex settings. Auto = 0, 10mbps half = 1, 10mbps full = 2, 100mbps half = 3, 100mbps full = 4, 1000mbps full = 5.");
module_param(autoneg, int, 0);
MODULE_PARM_DESC(autoneg, "PHY autonegotiate is enabled by setting to 1 and disabled by setting to 0.");
module_param(scatter_gather, int, 0);
MODULE_PARM_DESC(scatter_gather, "Scatter gather is enabled by setting to 1 and disabled by setting to 0.");
module_param(tso_offload, int, 0);
MODULE_PARM_DESC(tso_offload, "TCP Segmentation offload is enabled by setting to 1 and disabled by setting to 0.");
module_param(mtu, int, 0);
MODULE_PARM_DESC(mtu, "MTU value. Maximum value of 1500 or 9100 depending on hardware.");
module_param(tx_checksum_offload, int, 0);
MODULE_PARM_DESC(tx_checksum_offload, "Tx checksum offload is enabled by setting to 1 and disabled by setting to 0.");
module_param(rx_checksum_offload, int, 0);
MODULE_PARM_DESC(rx_checksum_offload, "Rx checksum offload is enabled by setting to 1 and disabled by setting to 0.");
module_param(tx_ring_size, int, 0);
MODULE_PARM_DESC(tx_ring_size, "Tx ring size. Maximum value of 1024 or 16384 depending on hardware.");
module_param(rx_ring_size, int, 0);
MODULE_PARM_DESC(rx_ring_size, "Rx ring size. Maximum value of 1024 or 16384 depending on hardware.");
module_param(tx_flow_control, int, 0);
MODULE_PARM_DESC(tx_flow_control, "Tx flow control is enabled by setting to 1 and disabled by setting to 0.");
module_param(rx_flow_control, int, 0);
MODULE_PARM_DESC(rx_flow_control, "Rx flow control is enabled by setting to 1 and disabled by setting to 0.");
module_param(dma_64bit, int, 0);
MODULE_PARM_DESC(dma_64bit, "High DMA is enabled by setting to 1 and disabled by setting to 0.");
module_param(wol, int, 0);
MODULE_PARM_DESC(wol, "Wake-On-Lan is enabled by setting to 1 and disabled by setting to 0.");
module_param(tagging_8021pq, int, 0);
MODULE_PARM_DESC(tagging_8021pq, "802.1pq tagging is enabled by setting to 1 and disabled by setting to 0.");
#else
MODULE_PARM(max_interrupt_work, "i");
MODULE_PARM_DESC(max_interrupt_work, "forcedeth maximum events handled per interrupt");
MODULE_PARM(optimization_mode, "i");
MODULE_PARM_DESC(optimization_mode, "In throughput mode (0), every tx & rx packet will generate an interrupt. In CPU mode (1), interrupts are controlled by a timer.");
MODULE_PARM(poll_interval, "i");
MODULE_PARM_DESC(poll_interval, "Interval determines how frequent timer interrupt is generated by [(time_in_micro_secs * 100) / (2^10)]. Min is 0 and Max is 65535.");
#ifdef CONFIG_PCI_MSI
MODULE_PARM(msi, "i");
MODULE_PARM_DESC(msi, "MSI interrupts are enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(msix, "i");
MODULE_PARM_DESC(msix, "MSIX interrupts are enabled by setting to 1 and disabled by setting to 0.");
#endif
MODULE_PARM(speed_duplex, "i");
MODULE_PARM_DESC(speed_duplex, "PHY speed and duplex settings. Auto = 0, 10mbps half = 1, 10mbps full = 2, 100mbps half = 3, 100mbps full = 4, 1000mbps full = 5.");
MODULE_PARM(autoneg, "i");
MODULE_PARM_DESC(autoneg, "PHY autonegotiate is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(scatter_gather, "i");
MODULE_PARM_DESC(scatter_gather, "Scatter gather is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(tso_offload, "i");
MODULE_PARM_DESC(tso_offload, "TCP Segmentation offload is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(mtu, "i");
MODULE_PARM_DESC(mtu, "MTU value. Maximum value of 1500 or 9100 depending on hardware.");
MODULE_PARM(tx_checksum_offload, "i");
MODULE_PARM_DESC(tx_checksum_offload, "Tx checksum offload is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(rx_checksum_offload, "i");
MODULE_PARM_DESC(rx_checksum_offload, "Rx checksum offload is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(tx_ring_size, "i");
MODULE_PARM_DESC(tx_ring_size, "Tx ring size. Maximum value of 1024 or 16384 depending on hardware.");
MODULE_PARM(rx_ring_size, "i");
MODULE_PARM_DESC(rx_ring_size, "Rx ring size. Maximum value of 1024 or 16384 depending on hardware.");
MODULE_PARM(tx_flow_control, "i");
MODULE_PARM_DESC(tx_flow_control, "Tx flow control is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(rx_flow_control, "i");
MODULE_PARM_DESC(rx_flow_control, "Rx flow control is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(dma_64bit, "i");
MODULE_PARM_DESC(dma_64bit, "High DMA is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(wol, "i");
MODULE_PARM_DESC(wol, "Wake-On-Lan is enabled by setting to 1 and disabled by setting to 0.");
MODULE_PARM(tagging_8021pq, "i");
MODULE_PARM_DESC(tagging_8021pq, "802.1pq tagging is enabled by setting to 1 and disabled by setting to 0.");
#endif
MODULE_AUTHOR("Manfred Spraul <manfred@colorfullife.com>");
MODULE_DESCRIPTION("Reverse Engineered nForce ethernet driver");
MODULE_LICENSE("GPL");

MODULE_DEVICE_TABLE(pci, pci_tbl);

module_init(init_nic);
module_exit(exit_nic);
