/******************************************************************************
 * s2io.c: A Linux PCI-X Ethernet driver for Neterion 10GbE Server NIC        *
 * Copyright(c) 2002-2005 Neterion Inc.                                       *
 *                                                                            *
 * This software may be used and distributed according to the terms of        *
 * the GNU General Public License (GPL), incorporated herein by reference.    *
 * Drivers based on or derived from this code fall under the GPL and must     *
 * retain the authorship, copyright and license notice.  This file is not     *
 * a complete program and may only be used when the entire operating          *
 * system is licensed under the GPL.                                          *
 * See the file COPYING in this distribution for more information.            *
 *                                                                            *
 * Credits:                                                                   *
 * Jeff Garzik		: For pointing out the improper error condition       *
 *			  check in the s2io_xmit routine and also some        *
 *			  issues in the Tx watch dog function. Also for       *
 *			  patiently answering all those innumerable           *
 *			  questions regaring the 2.6 porting issues.          *
 * Stephen Hemminger	: Providing proper 2.6 porting mechanism for some     *
 *			  macros available only in 2.6 Kernel.                *
 * Francois Romieu	: For pointing out all code part that were            *
 *			  deprecated and also styling related comments.       *
 * Grant Grundler	: For helping me get rid of some Architecture         *
 *			  dependent code.                                     *
 * Christopher Hellwig	: Some more 2.6 specific issues in the driver.        *
 *                                                                            *
 * The module loadable parameters that are supported by the driver and a brief*
 * explanation of all the variables.                                         *
 *                                                                            *
 * rx_ring_num : This can be used to program the number of receive rings used *
 *     in the driver.                                                         *
 * rx_ring_sz:  This defines the number of receive blocks each ring can have. *
 *     This is an array of size 8.                                            *
 * rx_ring_mode: This defines the operation mode of all 8 rings. Possible     *
 *     values are 1, 2, and 3, corresponds to 1-buffer, 2-buffer and 3-buffer *
 *     respectively. Default value is '1'.                                    *
 * tx_fifo_num: This defines the number of Tx FIFOs thats used int the driver.*
 * tx_fifo_len: This is an array of 8. Each element defines the number of     *
 *     Tx descriptors that can be associated with each corresponding FIFO     *
 * intr_type: This defines the type of interrupt. The values can be 0(INTA),  *
 *	   2(MSI_X). Default value is '0(INTA)'									  *
 * lro: Specifies whether to enable Large Receive Offload (LRO) or not.       *
 *     Possible values '1' for enable '0' for disable. Default is '0'         *
 * lro_max_pkts: This parameter defines maximum number of packets can be      *
 *     aggregated as a single large packet                                    *
 ******************************************************************************/
#include<linux/module.h>
#include<linux/types.h>
#include<linux/errno.h>
#include<linux/ioport.h>
#include<linux/pci.h>
#include<linux/kernel.h>
#include<linux/netdevice.h>
#include<linux/etherdevice.h>
#include<linux/skbuff.h>
#include<linux/init.h>
#include<linux/delay.h>
#include<linux/stddef.h>
#include<linux/ioctl.h>
#include<linux/timex.h>
#include<linux/sched.h>
#include<linux/ethtool.h>
#include<asm/system.h>
#include<asm/uaccess.h>
#include<linux/version.h>
#include<asm/io.h>
#include<linux/proc_fs.h>
#include<linux/if_vlan.h>
#include<linux/ip.h>
#include<linux/tcp.h>
#include<asm/checksum.h>
#include <net/sock.h>

/* local include */
#include "spdm_cfg.h"
#include "kcompat.h"
#include "s2io.h"
#include "regs.h"
#include "util.h"

#ifndef __VMKERNEL_MODULE__
#include<net/tcp.h>
#endif

#define DRV_VERSION "2.1.0.11070"

/* VENDOR and DEVICE ID of XENA. */
#ifndef PCI_VENDOR_ID_S2IO
#define PCI_VENDOR_ID_S2IO      0x17D5
#define PCI_DEVICE_ID_S2IO_WIN  0x5731
#define PCI_DEVICE_ID_S2IO_UNI  0x5831
#endif

#ifndef PCI_DEVICE_ID_HERC_WIN
#define PCI_DEVICE_ID_HERC_WIN  0x5732
#define PCI_DEVICE_ID_HERC_UNI  0x5832
#endif

int rxd_size[4] = {32,48,48,64};
int rxd_count[4] = {127,85,85,63};

static inline int RXD_IS_UP2DT(RxD_t *rxdp)
{
	int ret;

	ret = ((!(rxdp->Control_1 & RXD_OWN_XENA)) &&
		(GET_RXD_MARKER(rxdp->Control_2) != THE_RXD_MARK));

	return ret;
}


/* S2io Driver name & version. */
static char s2io_driver_name[] = "s2io";
static char s2io_driver_version[] = DRV_VERSION;

/*
 * Cards with following subsystem_id have a link state indication
 * problem, 600B, 600C, 600D, 640B, 640C and 640D.
 * macro below identifies these cards given the subsystem_id.
 */
#define CARDS_WITH_FAULTY_LINK_INDICATORS(dev_type, subid) \
	(dev_type == XFRAME_I_DEVICE) ?			\
		((((subid >= 0x600B) && (subid <= 0x600D)) || \
		 ((subid >= 0x640B) && (subid <= 0x640D))) ? 1 : 0) : 0

#define LINK_IS_UP(val64) (!(val64 & (ADAPTER_STATUS_RMAC_REMOTE_FAULT | \
				      ADAPTER_STATUS_RMAC_LOCAL_FAULT)))

/* Ethtool related variables and Macros. */
#ifndef SET_ETHTOOL_OPS
static int s2io_ethtool(struct net_device *dev, struct ifreq *rq);
#endif
static char s2io_gstrings[][ETH_GSTRING_LEN] = {
	"Register test\t(offline)",
	"Eeprom test\t(offline)",
	"Link test\t(online)",
	"RLDRAM test\t(offline)",
	"BIST Test\t(offline)"
};

#ifdef ETHTOOL_GSTATS
static char ethtool_stats_keys[][ETH_GSTRING_LEN] = {
	{"tmac_frms"},
	{"tmac_data_octets"},
	{"tmac_drop_frms"},
	{"tmac_mcst_frms"},
	{"tmac_bcst_frms"},
	{"tmac_pause_ctrl_frms"},
	{"tmac_ttl_octets"},
	{"tmac_ucst_frms"},
	{"tmac_nucst_frms"},
	{"tmac_any_err_frms"},
	{"tmac_ttl_less_fb_octets"},
	{"tmac_vld_ip_octets"},
	{"tmac_vld_ip"},
	{"tmac_drop_ip"},
	{"tmac_icmp"},
	{"tmac_rst_tcp"},
	{"tmac_tcp"},
	{"tmac_udp"},
	{"rmac_vld_frms"},
	{"rmac_data_octets"},
	{"rmac_fcs_err_frms"},
	{"rmac_drop_frms"},
	{"rmac_vld_mcst_frms"},
	{"rmac_vld_bcst_frms"},
	{"rmac_in_rng_len_err_frms"},
	{"rmac_out_rng_len_err_frms"},
	{"rmac_long_frms"},
	{"rmac_pause_ctrl_frms"},
	{"rmac_unsup_ctrl_frms"},
	{"rmac_ttl_octets"},
	{"rmac_accepted_ucst_frms"},
	{"rmac_accepted_nucst_frms"},
	{"rmac_discarded_frms"},
	{"rmac_drop_events"},
	{"rmac_ttl_less_fb_octets"},
	{"rmac_ttl_frms"},
	{"rmac_usized_frms"},
	{"rmac_osized_frms"},
	{"rmac_frag_frms"},
	{"rmac_jabber_frms"},
	{"rmac_ttl_64_frms"},
	{"rmac_ttl_65_127_frms"},
	{"rmac_ttl_128_255_frms"},
	{"rmac_ttl_256_511_frms"},
	{"rmac_ttl_512_1023_frms"},
	{"rmac_ttl_1024_1518_frms"},
	{"rmac_ip"},
	{"rmac_ip_octets"},
	{"rmac_hdr_err_ip"},
	{"rmac_drop_ip"},
	{"rmac_icmp"},
	{"rmac_tcp"},
	{"rmac_udp"},
	{"rmac_err_drp_udp"},
	{"rmac_xgmii_err_sym"},
	{"rmac_frms_q0"},
	{"rmac_frms_q1"},
	{"rmac_frms_q2"},
	{"rmac_frms_q3"},
	{"rmac_frms_q4"},
	{"rmac_frms_q5"},
	{"rmac_frms_q6"},
	{"rmac_frms_q7"},
	{"rmac_full_q0"},
	{"rmac_full_q1"},
	{"rmac_full_q2"},
	{"rmac_full_q3"},
	{"rmac_full_q4"},
	{"rmac_full_q5"},
	{"rmac_full_q6"},
	{"rmac_full_q7"},
	{"rmac_pause_cnt"},
	{"rmac_xgmii_data_err_cnt"},
	{"rmac_xgmii_ctrl_err_cnt"},
	{"rmac_accepted_ip"},
	{"rmac_err_tcp"},
	{"rd_req_cnt"},
	{"new_rd_req_cnt"},
	{"new_rd_req_rtry_cnt"},
	{"rd_rtry_cnt"},
	{"wr_rtry_rd_ack_cnt"},
	{"wr_req_cnt"},
	{"new_wr_req_cnt"},
	{"new_wr_req_rtry_cnt"},
	{"wr_rtry_cnt"},
	{"wr_disc_cnt"},
	{"rd_rtry_wr_ack_cnt"},
	{"txp_wr_cnt"},
	{"txd_rd_cnt"},
	{"txd_wr_cnt"},
	{"rxd_rd_cnt"},
	{"rxd_wr_cnt"},
	{"txf_rd_cnt"},
	{"rxf_wr_cnt"},
	{"rmac_ttl_1519_4095_frms"},
	{"rmac_ttl_4096_8191_frms"},
	{"rmac_ttl_8192_max_frms"},
	{"rmac_ttl_gt_max_frms"},
	{"rmac_osized_alt_frms"},
	{"rmac_jabber_alt_frms"},
	{"rmac_gt_max_alt_frms"},
	{"rmac_vlan_frms"},
	{"rmac_len_discard"},
	{"rmac_fcs_discard"},
	{"rmac_pf_discard"},
	{"rmac_da_discard"},
	{"rmac_red_discard"},
	{"rmac_rts_discard"},
	{"rmac_ingm_full_discard"},
	{"link_fault_cnt"},
	{"\n DRIVER STATISTICS"},
	{"single_bit_ecc_errs"},
	{"double_bit_ecc_errs"},
	{"parity_err_cnt"},
	{"serious_err_cnt"},
	{"soft_reset_cnt"},
	{"fifo_0_full_cnt"},
	{"fifo_1_full_cnt"},
	{"fifo_2_full_cnt"},
	{"fifo_3_full_cnt"},
	{"fifo_4_full_cnt"},
	{"fifo_5_full_cnt"},
	{"fifo_6_full_cnt"},
	{"fifo_7_full_cnt"},
	{"ring_0_full_cnt"},
	{"ring_1_full_cnt"},
	{"ring_2_full_cnt"},
	{"ring_3_full_cnt"},
	{"ring_4_full_cnt"},
	{"ring_5_full_cnt"},
	{"ring_6_full_cnt"},
	{"ring_7_full_cnt"},
	{"ring_0_mem_fail_cnt"},
	{"ring_1_mem_fail_cnt"},
	{"ring_2_mem_fail_cnt"},
	{"ring_3_mem_fail_cnt"},
	{"ring_4_mem_fail_cnt"},
	{"ring_5_mem_fail_cnt"},
	{"ring_6_mem_fail_cnt"},
	{"ring_7_mem_fail_cnt"},
	("alarm_transceiver_temp_high"),
	("alarm_transceiver_temp_low"),
	("alarm_laser_bias_current_high"),
	("alarm_laser_bias_current_low"),
	("alarm_laser_output_power_high"),
	("alarm_laser_output_power_low"),
	("warn_transceiver_temp_high"),
	("warn_transceiver_temp_low"),
	("warn_laser_bias_current_high"),
	("warn_laser_bias_current_low"),
	("warn_laser_output_power_high"),
	("warn_laser_output_power_low"),
	("lro_aggregated_pkts"),
	("lro_flush_both_count"),
	("lro_out_of_sequence_pkts"),
	("lro_flush_due_to_max_pkts"),
	("lro_avg_aggr_pkts"),
};

#define S2IO_STAT_LEN sizeof(ethtool_stats_keys)/ ETH_GSTRING_LEN
#define S2IO_STAT_STRINGS_LEN S2IO_STAT_LEN * ETH_GSTRING_LEN
#endif

#define S2IO_TEST_LEN	sizeof(s2io_gstrings) / ETH_GSTRING_LEN
#define S2IO_STRINGS_LEN	S2IO_TEST_LEN * ETH_GSTRING_LEN

#define S2IO_TIMER_CONF(timer, handle, arg, exp)		\
			init_timer(&timer);			\
			timer.function = handle;		\
			timer.data = (unsigned long) arg;	\
			mod_timer(&timer, (jiffies + exp))	\

/* Add the vlan */
static void s2io_vlan_rx_register(struct net_device *dev,
					struct vlan_group *grp)
{
	nic_t *nic = dev->priv;
	int i;
	unsigned long flags[MAX_TX_FIFOS];
	struct mac_info *mac_control = &nic->mac_control;
	struct config_param *config = &nic->config;

	for (i = 0; i < config->tx_fifo_num; i++)
		spin_lock_irqsave(&mac_control->fifos[i].tx_lock, flags[i]);

	nic->vlgrp = grp;
	for (i = config->tx_fifo_num - 1; i >= 0; i--)
		spin_unlock_irqrestore(&mac_control->fifos[i].tx_lock, flags[i]);
}

/* Unregister the vlan */
static void s2io_vlan_rx_kill_vid(struct net_device *dev, unsigned long vid)
{
	int i;
	nic_t *nic = dev->priv;
	unsigned long flags[MAX_TX_FIFOS];
	struct mac_info *mac_control = &nic->mac_control;
	struct config_param *config = &nic->config;

	for (i = 0; i < config->tx_fifo_num; i++)
		spin_lock_irqsave(&mac_control->fifos[i].tx_lock, flags[i]);

	if (nic->vlgrp)
		nic->vlgrp->vlan_devices[vid] = NULL;

	for (i = config->tx_fifo_num - 1; i >= 0; i--)
		spin_unlock_irqrestore(&mac_control->fifos[i].tx_lock, flags[i]);
}

/*
 * Constants to be programmed into the Xena's registers, to configure
 * the XAUI.
 */

#define	END_SIGN	0x0

static u64 herc_act_dtx_cfg[] = {
	/* Set address */
	0x8000051536750000ULL, 0x80000515367500E0ULL,
	/* Write data */
	0x8000051536750004ULL, 0x80000515367500E4ULL,
	/* Set address */
	0x80010515003F0000ULL, 0x80010515003F00E0ULL,
	/* Write data */
	0x80010515003F0004ULL, 0x80010515003F00E4ULL,
	/* Set address */
	0x801205150D440000ULL, 0x801205150D4400E0ULL,
	/* Write data */
	0x801205150D440004ULL, 0x801205150D4400E4ULL,
	/* Set address */
	0x80020515F2100000ULL, 0x80020515F21000E0ULL,
	/* Write data */
	0x80020515F2100004ULL, 0x80020515F21000E4ULL,
	/* Done */
	END_SIGN
};

static u64 xena_dtx_cfg[] = {
	/* Set address */
	0x8000051500000000ULL, 0x80000515000000E0ULL,
	/* Write data */
	0x80000515D9350004ULL, 0x80000515D93500E4ULL,
	/* Set address */
	0x8001051500000000ULL, 0x80010515000000E0ULL,
	/* Write data */
	0x80010515001E0004ULL, 0x80010515001E00E4ULL,
	/* Set address */
	0x8002051500000000ULL, 0x80020515000000E0ULL,
	/* Write data */
	0x80020515F2100004ULL, 0x80020515F21000E4ULL,
	END_SIGN
};


/*
 * Constants for Fixing the MacAddress problem seen mostly on
 * Alpha machines.
 */
static u64 fix_mac[] = {
	0x0060000000000000ULL, 0x0060600000000000ULL,
	0x0040600000000000ULL, 0x0000600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0060600000000000ULL,
	0x0020600000000000ULL, 0x0000600000000000ULL,
	0x0040600000000000ULL, 0x0060600000000000ULL,
	END_SIGN
};

MODULE_LICENSE("GPL");
#ifdef MODULE_VERSION
MODULE_VERSION(DRV_VERSION);
#endif

/* Rx Steering parameter list explanation
 * rx_steering_type: There are 2 ways to steer Rx'ed frames to correct rings
 *	on the host either RTH or L4 Port based steering. This parameter can
 *	be used to choose any one scheme. To use RTH, set this parameter to 2
 *	or for L4 Port steering set it to 1.
 *
 * num_rth_ports: for port based steering, this parameter can be used to
 *      specify the number of ports being configured.
 *
 * rth_ports: The specefic ports whicch will be routed to programmed rings. A
 *	maximum of 'MAX_STEERABLE_PORTS' ports can be configured.
 *
 * port_type: This parameter can be used to specify if the ports being
 *	scanned by the H/W is a Destination port or source port. Default is
 *	destination port.
 *
 * rth_fn_and_mask: This parameter can be used to define
 *	i. Input to the hash function (Least significant byte).
 *		1 is IPV4 Src and dst address and TCP src abd Dst ports
 *		2 is IPV4 Src and dst address and UDP src abd Dst ports
 *		3 is IPV4 Src and dst address
 *		4 is IPV6 Src and dst address and TCP src abd Dst ports
 *		5 is IPV6 Src and dst address and UDP src abd Dst ports
 *		6 is IPV6 Src and dst address
 *		7 is extended IPV6 Src and dst address and TCP src abd Dst
 *		ports
 *		8 is extended IPV6 Src and dst address and UDP src abd Dst
 *		ports
 *		9 is extended IPV6 Src and dst address
 *	ii. Mask any of the six parameters used for RTH calculation mentioned
 *		above(bit 9 to bit 14). The bits 9 to 14 represent IPV4 SA,
 *		IPV4 DA, IPV6 SA, IPV6 DA, L4 SP, L4 DP in that order.
 *		Masking any field will make it all '1's.
 */

/* Module Loadable parameters. */
S2IO_PARM_INT(mmrbc, S2IO_PCI_BIOS_DEFAULTS);
S2IO_PARM_INT(most, S2IO_PCI_BIOS_DEFAULTS);
S2IO_PARM_INT(latency, S2IO_PCI_BIOS_DEFAULTS);
S2IO_PARM_INT(tx_fifo_num, 1);
S2IO_PARM_INT(rx_ring_num, 1);
S2IO_PARM_INT(rx_ring_mode, 1);
S2IO_PARM_INT(use_continuous_tx_intrs, 1);
S2IO_PARM_INT(rmac_pause_time, 0x100);
S2IO_PARM_INT(mc_pause_threshold_q0q3, 187);
S2IO_PARM_INT(mc_pause_threshold_q4q7, 187);
S2IO_PARM_INT(shared_splits, 0);
S2IO_PARM_INT(tmac_util_period, 5);
S2IO_PARM_INT(rmac_util_period, 5);
S2IO_PARM_INT(bimodal, 0);
S2IO_PARM_INT(l3l4hdr_size, 128);
/* Frequency of Rx desc syncs expressed as power of 2 */
S2IO_PARM_INT(rxsync_frequency, 3);
/* Interrupt type. Values can be 0(INTA), 2(MSI_X) */
S2IO_PARM_INT(intr_type, 0);
/* 0 is no steering, 1 is Port steering, 2 is RTH */
S2IO_PARM_INT(rx_steering_type, NO_STEERING);
S2IO_PARM_INT(num_rth_ports, MAX_STEERABLE_PORTS);
S2IO_PARM_INT(rth_fn_and_mask, 0x1);
S2IO_PARM_INT(port_type, DP);
S2IO_PARM_INT(lro, 0);
S2IO_PARM_INT(lro_max_pkts, 0xFFFF);
#ifndef CONFIG_S2IO_NAPI
S2IO_PARM_INT(indicate_max_pkts, 0);
#endif

#ifdef __VMKERNEL_MODULE__
S2IO_PARM_INT(enable_netq, 0);
#endif

static unsigned int tx_fifo_len[MAX_TX_FIFOS] =
    {DEFAULT_FIFO_0_LEN, [1 ...(MAX_TX_FIFOS - 1)] = DEFAULT_FIFO_1_7_LEN};
static unsigned int rx_ring_sz[MAX_RX_RINGS] =
    {[0 ...(MAX_RX_RINGS - 1)] = SMALL_BLK_CNT};
static unsigned int rts_frm_len[MAX_RX_RINGS] =
    {[0 ...(MAX_RX_RINGS - 1)] = 0 };
static int rth_ports[MAX_STEERABLE_PORTS] =
			{[0 ...(MAX_STEERABLE_PORTS - 1)] = 0 };
#ifdef module_param_array
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,10))
module_param_array(tx_fifo_len, uint, NULL, 0);
module_param_array(rx_ring_sz, uint, NULL, 0);
module_param_array(rts_frm_len, uint, NULL, 0);
#else
module_param_array(tx_fifo_len, int, tx_fifo_num, 0);
module_param_array(rx_ring_sz, int, rx_ring_num, 0);
module_param_array(rts_frm_len, int, rx_ring_num, 0);
#endif
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,10))
module_param_array(rth_ports, uint, NULL, 0);
#else
module_param_array(rth_ports, int, num_rth_ports, 0);
#endif
#else
MODULE_PARM(tx_fifo_len, "1-" __MODULE_STRING(8) "i");
MODULE_PARM(rx_ring_sz, "1-" __MODULE_STRING(8) "i");
MODULE_PARM(rts_frm_len, "1-" __MODULE_STRING(8) "i");
MODULE_PARM(rth_ports, "1-" __MODULE_STRING(MAX_STEERABLE_PORTS) "i");
#endif

/*
 * S2IO device table.
 * This table lists all the devices that this driver supports.
 */
static struct pci_device_id s2io_tbl[] __devinitdata = {
	{PCI_VENDOR_ID_S2IO, PCI_DEVICE_ID_S2IO_WIN,
	 PCI_ANY_ID, PCI_ANY_ID},
	{PCI_VENDOR_ID_S2IO, PCI_DEVICE_ID_S2IO_UNI,
	 PCI_ANY_ID, PCI_ANY_ID},
	{PCI_VENDOR_ID_S2IO, PCI_DEVICE_ID_HERC_WIN,
         PCI_ANY_ID, PCI_ANY_ID},
        {PCI_VENDOR_ID_S2IO, PCI_DEVICE_ID_HERC_UNI,
         PCI_ANY_ID, PCI_ANY_ID},
	{0,}
};

MODULE_DEVICE_TABLE(pci, s2io_tbl);

static struct pci_driver s2io_driver = {
      .name = "S2IO",
      .id_table = s2io_tbl,
      .probe = s2io_init_nic,
      .remove = __devexit_p(s2io_rem_nic),
};

#if defined(__VMKERNEL_MODULE__) && defined(__VMKNETDDI_QUEUEOPS__)
static int s2io_netqueue_ops(vmknetddi_queueops_op_t op, void *args);
#endif

/* A simplifier macro used both by init and free shared_mem Fns(). */
#define TXD_MEM_PAGE_CNT(len, per_each) ((len+per_each - 1) / per_each)

static int spdm_data_processor(spdm_cfg_t *usr_info, nic_t *sp)
{
	int ret;

	if (sp->device_type == XFRAME_I_DEVICE) {
		usr_info->ret = SPDM_XENA_IF;
		return FAILURE;
	}

	if (sp->device_initialized == FALSE) {
		usr_info->ret = SPDM_HW_UNINITIALIZED;
		return FAILURE;
	}

	if (usr_info->ret == SPDM_GET_CFG_DATA) {/* Retrieve info */
		u8 *data = kmalloc(MAX_SPDM_ENTRIES_SIZE, GFP_KERNEL);
		if (!data) {
			usr_info->ret = SPDM_TABLE_MALLOC_FAIL;
			return FAILURE;
		}

		ret = spdm_extract_table(data, sp);
		if (ret != FAILURE) {
			memcpy(usr_info->data, data,
				     (sp->spdm_entry * 0x40));
			usr_info->data_len= ret;
			usr_info->ret = SPDM_CONF_SUCCESS;
			ret = SUCCESS;
		} else {
			usr_info->ret = SPDM_TABLE_ACCESS_FAILED;
			ret = FAILURE;
		}

		kfree(data);
		return ret;
	}

	if (!usr_info->dip || !usr_info->sip || !usr_info->sprt ||
							!usr_info->dprt) {
		usr_info->ret = SPDM_INCOMPLETE_SOCKET;
		return FAILURE;
	}

	if ((ret = spdm_configure(sp, usr_info)) == SUCCESS) {
		usr_info->ret = SPDM_CONF_SUCCESS;
	}

	return SUCCESS;
}

/**
 * init_shared_mem - Allocation and Initialization of Memory
 * @nic: Device private variable.
 * Description: The function allocates all the memory areas shared
 * between the NIC and the driver. This includes Tx descriptors,
 * Rx descriptors and the statistics block.
 */

static int init_shared_mem(struct s2io_nic *nic)
{
	u32 size;
	void *tmp_v_addr, *tmp_v_addr_next;
	dma_addr_t tmp_p_addr, tmp_p_addr_next;
	RxD_block_t *pre_rxd_blk = NULL;
	int i, j, blk_cnt, rx_sz, tx_sz;
	int lst_size, lst_per_page;
	struct net_device *dev = nic->dev;
	unsigned long tmp;
	buffAdd_t *ba;

	mac_info_t *mac_control;
	struct config_param *config;

	mac_control = &nic->mac_control;
	config = &nic->config;


	/* Allocation and initialization of TXDLs in FIOFs */
	size = 0;
	for (i = 0; i < config->tx_fifo_num; i++) {
		size += config->tx_cfg[i].fifo_len;

		/*
		 * Legal values are from 2 to 8192
		 */
		if (config->tx_cfg[i].fifo_len < 2){
			DBG_PRINT(ERR_DBG, "s2io: Invalid fifo len (%d) ", size);
			DBG_PRINT(ERR_DBG, "for fifo %d\n", i);
			DBG_PRINT(ERR_DBG, "s2io: Legal values for fifo len "
				"are 2 to 8192\n");
			return -EINVAL;
		}
	}

	if (size > MAX_AVAILABLE_TXDS) {
		DBG_PRINT(ERR_DBG, "s2io: Requested TxDs too high, ");
		DBG_PRINT(ERR_DBG, "Requested: %d, max supported: 8192\n", size);
		return -EINVAL;
	}

	lst_size = (sizeof(TxD_t) * config->max_txds);
	tx_sz = lst_size * size;
	lst_per_page = PAGE_SIZE / lst_size;

	for (i = 0; i < config->tx_fifo_num; i++) {
		int fifo_len = config->tx_cfg[i].fifo_len;
		int list_holder_size = fifo_len * sizeof(list_info_hold_t);

		mac_control->fifos[i].list_info = kmalloc(list_holder_size,
							  GFP_KERNEL);
		if (!mac_control->fifos[i].list_info) {
			DBG_PRINT(INFO_DBG,
				  "Malloc failed for list_info\n");
			return -ENOMEM;
		}
		memset(mac_control->fifos[i].list_info, 0, list_holder_size);
	}

#ifdef CONFIG_AMD_FIX
	for (i = 0; i < config->tx_fifo_num; i++) {
		unsigned long tmp;
		int fifo_len = config->tx_cfg[i].fifo_len;

		nic->amd_tmp_org[i] = kmalloc((fifo_len * AMD_BUF_ALIGN) +
					      AMD_BUF_ALIGN - 1, GFP_KERNEL);
		if (!nic->amd_tmp_org[i]) {
			return -ENOMEM;
		}
		memset(nic->amd_tmp_org[i], 0, (fifo_len * AMD_BUF_ALIGN) +
		       AMD_BUF_ALIGN - 1);
	       tmp = (unsigned long) nic->amd_tmp_org[i];
	       tmp += (AMD_BUF_ALIGN - 1);
	       tmp &= ~((unsigned long)(AMD_BUF_ALIGN - 1));
	       nic->amd_tmp[i] = (u8 *)tmp;
	}
#endif
	for (i = 0; i < config->tx_fifo_num; i++) {
		int page_num = TXD_MEM_PAGE_CNT(config->tx_cfg[i].fifo_len,
						lst_per_page);
		mac_control->fifos[i].tx_curr_put_info.offset = 0;
		mac_control->fifos[i].tx_curr_put_info.fifo_len =
		    config->tx_cfg[i].fifo_len - 1;
		mac_control->fifos[i].tx_curr_get_info.offset = 0;
		mac_control->fifos[i].tx_curr_get_info.fifo_len =
		    config->tx_cfg[i].fifo_len - 1;
		mac_control->fifos[i].fifo_no = i;
		mac_control->fifos[i].nic = nic;
		mac_control->fifos[i].max_txds = MAX_SKB_FRAGS + 1;
#ifdef CONFIG_AMD_FIX
		mac_control->fifos[i].max_txds++;
#endif
#ifdef NETIF_F_UFO
		mac_control->fifos[i].max_txds++;
#endif

		for (j = 0; j < page_num; j++) {
			int k = 0;
			dma_addr_t tmp_p;
			void *tmp_v;
			tmp_v = pci_alloc_consistent(nic->pdev,
						     PAGE_SIZE, &tmp_p);
			if (!tmp_v) {
				DBG_PRINT(INFO_DBG,
					  "pci_alloc_consistent ");
				DBG_PRINT(INFO_DBG, "failed for TxDL\n");
				return -ENOMEM;
			}
			/* If we got a zero DMA address(can happen on
			 * certain platforms like PPC), reallocate.
			 * Store virtual address of page we don't want,
			 * to be freed later.
			 */
			if (!tmp_p) {
				mac_control->zerodma_virt_addr = tmp_v;
				DBG_PRINT(INIT_DBG,
				"%s: Zero DMA address for TxDL. ", dev->name);
				DBG_PRINT(INIT_DBG,
				"Virtual address %p\n", tmp_v);
				tmp_v = pci_alloc_consistent(nic->pdev,
						     PAGE_SIZE, &tmp_p);
				if (!tmp_v) {
					DBG_PRINT(INFO_DBG,
					  "pci_alloc_consistent ");
					DBG_PRINT(INFO_DBG, "failed for TxDL\n");
					return -ENOMEM;
				}
			}
			while (k < lst_per_page) {
				int l = (j * lst_per_page) + k;
				if (l == config->tx_cfg[i].fifo_len)
					break;
				mac_control->fifos[i].list_info[l].
					list_virt_addr = tmp_v + (k * lst_size);
				mac_control->fifos[i].list_info[l].
					list_phy_addr = tmp_p + (k * lst_size);
				k++;
			}
		}
	}
#ifdef NETIF_F_UFO
	nic->ufo_in_band_v = kmalloc((sizeof(u64) * size), GFP_KERNEL);
	if (!nic->ufo_in_band_v) {
		 return -ENOMEM;
	}
	memset(nic->ufo_in_band_v, 0, size);
#endif

	/* Allocation and initialization of RXDs in Rings */
	size = 0;
	for (i = 0; i < config->rx_ring_num; i++) {
		if (config->rx_cfg[i].num_rxd %
		    (rxd_count[nic->rxd_mode] + 1)) {
			DBG_PRINT(ERR_DBG, "%s: RxD count of ", dev->name);
			DBG_PRINT(ERR_DBG, "Ring%d is not a multiple of ",
				  i);
			DBG_PRINT(ERR_DBG, "RxDs per Block");
			return FAILURE;
		}
		size += config->rx_cfg[i].num_rxd;
		mac_control->rings[i].block_count =
			config->rx_cfg[i].num_rxd /
			(rxd_count[nic->rxd_mode] + 1 );
		mac_control->rings[i].pkt_cnt = config->rx_cfg[i].num_rxd -
			mac_control->rings[i].block_count;
	}
	if (nic->rxd_mode == RXD_MODE_1)
		size = (size * (sizeof(RxD1_t)));
	else
		size = (size * (sizeof(RxD3_t)));
	rx_sz = size;

	for (i = 0; i < config->rx_ring_num; i++) {
		mac_control->rings[i].rx_curr_get_info.block_index = 0;
		mac_control->rings[i].rx_curr_get_info.offset = 0;
		mac_control->rings[i].rx_curr_get_info.ring_len =
		    config->rx_cfg[i].num_rxd - 1;
		mac_control->rings[i].rx_curr_put_info.block_index = 0;
		mac_control->rings[i].rx_curr_put_info.offset = 0;
		mac_control->rings[i].rx_curr_put_info.ring_len =
		    config->rx_cfg[i].num_rxd - 1;
		mac_control->rings[i].nic = nic;
		mac_control->rings[i].ring_no = i;

		blk_cnt = config->rx_cfg[i].num_rxd /
				(rxd_count[nic->rxd_mode] + 1);
		/*  Allocating all the Rx blocks */
		for (j = 0; j < blk_cnt; j++) {
			rx_block_info_t *rx_blocks;
			int l;

			rx_blocks = &mac_control->rings[i].rx_blocks[j];
			size = SIZE_OF_BLOCK; //size is always page size
			tmp_v_addr = pci_alloc_consistent(nic->pdev, size,
							  &tmp_p_addr);
			if (tmp_v_addr == NULL) {
				/*
				 * In case of failure, free_shared_mem()
				 * is called, which should free any
				 * memory that was alloced till the
				 * failure happened.
				 */
				rx_blocks->block_virt_addr = tmp_v_addr;
				return -ENOMEM;
			}
			memset(tmp_v_addr, 0, size);
			rx_blocks->block_virt_addr = tmp_v_addr;
			rx_blocks->block_dma_addr = tmp_p_addr;
			rx_blocks->rxds = kmalloc(sizeof(rxd_info_t)*
						  rxd_count[nic->rxd_mode],
						  GFP_KERNEL);
			if (!rx_blocks->rxds) {
				return -ENOMEM;
			}

			for (l=0; l<rxd_count[nic->rxd_mode];l++) {
				rx_blocks->rxds[l].virt_addr =
					rx_blocks->block_virt_addr +
					(rxd_size[nic->rxd_mode] * l);
				rx_blocks->rxds[l].dma_addr =
					rx_blocks->block_dma_addr +
					(rxd_size[nic->rxd_mode] * l);
			}
		}
		/* Interlinking all Rx Blocks */
		for (j = 0; j < blk_cnt; j++) {
			tmp_v_addr = mac_control->rings[i].rx_blocks[j].
				block_virt_addr;
			tmp_v_addr_next = mac_control->rings[i].
				rx_blocks[(j + 1) % blk_cnt].block_virt_addr;
			tmp_p_addr = mac_control->rings[i].rx_blocks[j].
				block_dma_addr;
			tmp_p_addr_next =
				mac_control->rings[i].rx_blocks[(j + 1) %
					      blk_cnt].block_dma_addr;

			pre_rxd_blk = (RxD_block_t *) tmp_v_addr;
			pre_rxd_blk->reserved_2_pNext_RxD_block =
			    (unsigned long) tmp_v_addr_next;
			pre_rxd_blk->pNext_RxD_Blk_physical =
			    (u64) tmp_p_addr_next;
		}
	}

	if (nic->rxd_mode >= RXD_MODE_3A) {
		/*
		 * Allocation of Storages for buffer addresses in 2BUFF mode
		 * and the buffers as well.
		 */
		for (i = 0; i < config->rx_ring_num; i++) {
			blk_cnt = config->rx_cfg[i].num_rxd /
			   (rxd_count[nic->rxd_mode]+ 1);
			mac_control->rings[i].ba =
				kmalloc((sizeof(buffAdd_t *) * blk_cnt),
				     GFP_KERNEL);
			if (!mac_control->rings[i].ba)
				return -ENOMEM;
			for (j = 0; j < blk_cnt; j++) {
				int k = 0;
				mac_control->rings[i].ba[j] =
					kmalloc((sizeof(buffAdd_t) *
						(rxd_count[nic->rxd_mode] + 1)),
						GFP_KERNEL);
				if (!mac_control->rings[i].ba[j])
					return -ENOMEM;
				while (k != rxd_count[nic->rxd_mode]) {
					ba = &mac_control->rings[i].ba[j][k];

					ba->ba_0_org = (void *) kmalloc
					    (BUF0_LEN + ALIGN_SIZE, GFP_KERNEL);
					if (!ba->ba_0_org)
						return -ENOMEM;
					tmp = (unsigned long)ba->ba_0_org;
					tmp += ALIGN_SIZE;
					tmp &= ~((unsigned long) ALIGN_SIZE);
					ba->ba_0 = (void *) tmp;

					ba->ba_1_org = (void *) kmalloc
					    (BUF1_LEN + ALIGN_SIZE, GFP_KERNEL);
					if (!ba->ba_1_org)
						return -ENOMEM;
					tmp = (unsigned long) ba->ba_1_org;
					tmp += ALIGN_SIZE;
					tmp &= ~((unsigned long) ALIGN_SIZE);
					ba->ba_1 = (void *) tmp;
					k++;
				}
			}
		}
	}

	/* Allocation and initialization of Statistics block */
	size = sizeof(StatInfo_t);
	mac_control->stats_mem = pci_alloc_consistent
	    (nic->pdev, size, &mac_control->stats_mem_phy);

	if (!mac_control->stats_mem) {
		/*
		 * In case of failure, free_shared_mem() is called, which
		 * should free any memory that was alloced till the
		 * failure happened.
		 */
		return -ENOMEM;
	}
	mac_control->stats_mem_sz = size;

	tmp_v_addr = mac_control->stats_mem;
	mac_control->stats_info = (StatInfo_t *) tmp_v_addr;
	memset(tmp_v_addr, 0, size);
#ifdef SNMP_SUPPORT
	nic->nMemorySize = tx_sz + rx_sz + mac_control->stats_mem_sz;
#endif

	DBG_PRINT(INIT_DBG, "%s:Ring Mem PHY: 0x%llx\n", dev->name,
		  (unsigned long long) tmp_p_addr);

	return SUCCESS;
}

/**
 * free_shared_mem - Free the allocated Memory
 * @nic:  Device private variable.
 * Description: This function is to free all memory locations allocated by
 * the init_shared_mem() function and return it to the kernel.
 */

static void free_shared_mem(struct s2io_nic *nic)
{
	int i, j, blk_cnt, size;
	void *tmp_v_addr;
	dma_addr_t tmp_p_addr;
	mac_info_t *mac_control;
	struct config_param *config;
	int lst_size, lst_per_page;
	struct net_device *dev = nic->dev;

	if (!nic)
		return;

	mac_control = &nic->mac_control;
	config = &nic->config;

	lst_size = (sizeof(TxD_t) * config->max_txds);
	lst_per_page = PAGE_SIZE / lst_size;

	for (i = 0; i < config->tx_fifo_num; i++) {
		int page_num = TXD_MEM_PAGE_CNT(config->tx_cfg[i].fifo_len,
						lst_per_page);
		for (j = 0; j < page_num; j++) {
			int mem_blks = (j * lst_per_page);
			if (!mac_control->fifos[i].list_info)
				return;
			if (!mac_control->fifos[i].list_info[mem_blks].
				 list_virt_addr)
				break;
			pci_free_consistent(nic->pdev, PAGE_SIZE,
					    mac_control->fifos[i].
					    list_info[mem_blks].
					    list_virt_addr,
					    mac_control->fifos[i].
					    list_info[mem_blks].
					    list_phy_addr);
		}
		/* If we got a zero DMA address during allocation,
		 * free the page now
		 */
		if (mac_control->zerodma_virt_addr) {
			pci_free_consistent(nic->pdev, PAGE_SIZE,
					    mac_control->zerodma_virt_addr,
					    (dma_addr_t)0);
			DBG_PRINT(INIT_DBG,
				"%s: Freeing TxDL with zero DMA addr. ",
				dev->name);
			DBG_PRINT(INIT_DBG, "Virtual address %p\n",
				mac_control->zerodma_virt_addr);
		}
		kfree(mac_control->fifos[i].list_info);
#ifdef CONFIG_AMD_FIX
		kfree(nic->amd_tmp_org[i]);
#endif
	}

	size = SIZE_OF_BLOCK;
	for (i = 0; i < config->rx_ring_num; i++) {
		blk_cnt = mac_control->rings[i].block_count;
		for (j = 0; j < blk_cnt; j++) {
			tmp_v_addr = mac_control->rings[i].rx_blocks[j].
				block_virt_addr;
			tmp_p_addr = mac_control->rings[i].rx_blocks[j].
				block_dma_addr;
			if (tmp_v_addr == NULL)
				break;
			pci_free_consistent(nic->pdev, size,
					    tmp_v_addr, tmp_p_addr);
			if (NULL == mac_control->rings[i].rx_blocks[j].rxds)
				break;

			kfree(mac_control->rings[i].rx_blocks[j].rxds);
		}
	}

	if (nic->rxd_mode >= RXD_MODE_3A) {
		/* Freeing buffer storage addresses in 2BUFF mode. */
		for (i = 0; i < config->rx_ring_num; i++) {
			blk_cnt = config->rx_cfg[i].num_rxd /
			    (rxd_count[nic->rxd_mode] + 1);
			for (j = 0; j < blk_cnt; j++) {
				int k = 0;
				if (!mac_control->rings[i].ba[j])
					continue;
				while (k != rxd_count[nic->rxd_mode]) {
					buffAdd_t *ba =
						&mac_control->rings[i].ba[j][k];
					if(ba == NULL)
						break;
					kfree(ba->ba_0_org);
					kfree(ba->ba_1_org);
					k++;
				}
				kfree(mac_control->rings[i].ba[j]);
			}
			if(mac_control->rings[i].ba)
				kfree(mac_control->rings[i].ba);
		}
	}

	if (mac_control->stats_mem) {
		pci_free_consistent(nic->pdev,
				    mac_control->stats_mem_sz,
				    mac_control->stats_mem,
				    mac_control->stats_mem_phy);
	}
#ifdef NETIF_F_UFO
	if (nic->ufo_in_band_v)
		kfree(nic->ufo_in_band_v);
#endif
}

/**
 * s2io_verify_pci_mode -
 */

static int s2io_verify_pci_mode(nic_t *nic)
{
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	register u64 val64 = 0;
	int     mode;

	val64 = readq(&bar0->pci_mode);
	mode = (u8)GET_PCI_MODE(val64);

	if ( val64 & PCI_MODE_UNKNOWN_MODE)
		return -1;      /* Unknown PCI mode */
	return mode;
}

#define NEC_VENID   0x1033
#define NEC_DEVID   0x0125
int s2io_on_nec_bridge(struct pci_dev *s2io_pdev)
{
	struct pci_dev *tdev = NULL;
	while ((tdev = pci_find_device(PCI_ANY_ID, PCI_ANY_ID, tdev)) != NULL) {
		if ((tdev->vendor == NEC_VENID) && (tdev->device == NEC_DEVID)){
			if (tdev->bus == s2io_pdev->bus->parent)
				return 1;
		}
	}
	return 0;
}


int bus_speed[8] = {33, 133, 133, 200, 266, 133, 200, 266};
/**
 * s2io_print_pci_mode -
 */
static int s2io_print_pci_mode(nic_t *nic)
{
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	register u64 val64 = 0;
	int	mode;
	struct config_param *config = &nic->config;

	val64 = readq(&bar0->pci_mode);
	mode = (u8)GET_PCI_MODE(val64);

	if ( val64 & PCI_MODE_UNKNOWN_MODE)
		return -1;	/* Unknown PCI mode */

	config->bus_speed = bus_speed[mode];

	if (s2io_on_nec_bridge(nic->pdev)) {
		DBG_PRINT(ERR_DBG, "%s: Device is on PCI-E bus\n",
							nic->dev->name);
		return mode;
	}


	if (val64 & PCI_MODE_32_BITS) {
		DBG_PRINT(ERR_DBG, "%s: Device is on 32 bit ", nic->dev->name);
	} else {
		DBG_PRINT(ERR_DBG, "%s: Device is on 64 bit ", nic->dev->name);
	}

	switch(mode) {
		case PCI_MODE_PCI_33:
			DBG_PRINT(ERR_DBG, "33MHz PCI bus\n");
			break;
		case PCI_MODE_PCI_66:
			DBG_PRINT(ERR_DBG, "66MHz PCI bus\n");
			break;
		case PCI_MODE_PCIX_M1_66:
			DBG_PRINT(ERR_DBG, "66MHz PCIX(M1) bus\n");
			break;
		case PCI_MODE_PCIX_M1_100:
			DBG_PRINT(ERR_DBG, "100MHz PCIX(M1) bus\n");
			break;
		case PCI_MODE_PCIX_M1_133:
			DBG_PRINT(ERR_DBG, "133MHz PCIX(M1) bus\n");
			break;
		case PCI_MODE_PCIX_M2_66:
			DBG_PRINT(ERR_DBG, "133MHz PCIX(M2) bus\n");
			break;
		case PCI_MODE_PCIX_M2_100:
			DBG_PRINT(ERR_DBG, "200MHz PCIX(M2) bus\n");
			break;
		case PCI_MODE_PCIX_M2_133:
			DBG_PRINT(ERR_DBG, "266MHz PCIX(M2) bus\n");
			break;
		default:
			return -1;	/* Unsupported bus speed */
	}

	return mode;
}

/**
 * s2io_rth_configure -
 */
static int s2io_rth_configure(nic_t *nic)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
        register u64 val64 = 0;
        struct config_param *config;
        int buckets, i, ring = 0, cnt = 0;
	u8 mask, fn_type;

	config = &nic->config;

        /* Enabled enhanced RTS steering */
        val64 = readq(&bar0->rts_ctrl);
        val64 |= RTS_CTRL_ENHANCED;
        writeq(val64, &bar0->rts_ctrl);

	if (config->rth_enable == PORT_STEERING) {
		for (i=0, ring = 0; i<MAX_STEERABLE_PORTS; i++) {
			int port = config->steer_ports[i];
			if (!port)
				break;
			ring = i % config->rx_ring_num;
			val64 = BIT(7) | vBIT(port,8,16) | vBIT(ring,37,3) |
				BIT(63);
			if (config->port_type == SP)
				val64 = BIT(47);
			writeq(val64, &bar0->rts_pn_cam_data);
			val64 = BIT(7) | BIT(15) | vBIT(i,24,8);
			writeq(val64, &bar0->rts_pn_cam_ctrl);
			mdelay(5);
		}
	} else {
		int rth_sz = ((config->rx_ring_num + 1)/2);
		buckets = config->rx_ring_num;

	        for (i=0; i<buckets; i++) {
		        val64 = RTS_RTH_MAP_MEM_DATA_ENTRY_EN |
				RTS_RTH_MAP_MEM_DATA_(ring);
	                writeq(val64, &bar0->rts_rth_map_mem_data);

		        val64 = RTS_RTH_MAP_MEM_CTRL_WE |
				RTS_RTH_MAP_MEM_CTRL_STROBE |
				RTS_RTH_MAP_MEM_CTRL_OFFSET(i);
			writeq(val64, &bar0->rts_rth_map_mem_ctrl);

			do {
				val64 = readq(&bar0->rts_rth_map_mem_ctrl);
				if (val64 & RTS_RTH_MAP_MEM_CTRL_STROBE) {
					cnt++;
					msleep(10);
					continue;
				}
				break;
			} while(cnt < 5);
			if (cnt == 5)
				return FAILURE;
	                ring ++;
			ring %= config->rx_ring_num;
		}

		/*
		 * Mask all parameters as per user's input.
		 */
		mask = (u8)((config->rth_fn_and_mask >> 8) & 0xFF);
		for (i=0; i<6; i++) {
			if (!((mask >> i) & 0x1))
				continue;
			switch(i) {
				case 0:
				val64 = readq(&bar0->rts_rth_hash_mask_n[4]);
				val64 |= RTS_RTH_HASH_MASK_IPV4_SA(0xFFFFFFFF);
				writeq(val64, &bar0->rts_rth_hash_mask_n[4]);
				break;
				case 1:
				val64 = readq(&bar0->rts_rth_hash_mask_n[4]);
				val64 |= RTS_RTH_HASH_MASK_IPV4_DA(0xFFFFFFFF);
				writeq(val64, &bar0->rts_rth_hash_mask_n[4]);
				break;
				case 2:
				val64 = 0xFFFFFFFFFFFFFFFFULL;
				writeq(val64, &bar0->rts_rth_hash_mask_n[0]);
				val64 = 0xFFFFFFFFFFFFFFFFULL;
				writeq(val64, &bar0->rts_rth_hash_mask_n[1]);
				break;
				case 3:
				val64 = 0xFFFFFFFFFFFFFFFFULL;
				writeq(val64, &bar0->rts_rth_hash_mask_n[2]);
				val64 = 0xFFFFFFFFFFFFFFFFULL;
				writeq(val64, &bar0->rts_rth_hash_mask_n[3]);
				break;
				case 4:
				val64 = readq(&bar0->rts_rth_hash_mask_5);
				val64 |= RTS_RTH_HASH_MASK_L4_SP(0xFFFF);
				writeq(val64, &bar0->rts_rth_hash_mask_5);
				break;
				case 5:
				val64 = readq(&bar0->rts_rth_hash_mask_5);
				val64 |= RTS_RTH_HASH_MASK_L4_DP(0xFFFF);
				writeq(val64, &bar0->rts_rth_hash_mask_5);
				break;
			}
		}

		/*
		 * Set the RTH function type as per user's input and enable RTH.
		 */
		fn_type = (u8)(config->rth_fn_and_mask & 0xFF);
		switch(fn_type) {
			case 1:
				val64 = RTS_RTH_TCP_IPV4_EN;
				break;
			case 2:
				val64 = RTS_RTH_UDP_IPV4_EN;
				break;
			case 3:
				val64 = RTS_RTH_IPV4_EN;
				break;
			case 4:
				val64 = RTS_RTH_TCP_IPV6_EN;
				break;
			case 5:
				val64 = RTS_RTH_UDP_IPV6_EN;
				break;
			case 6:
				val64 = RTS_RTH_IPV6_EN;
				break;
			case 7:
				val64 = RTS_RTH_TCP_IPV6_EX_EN;
				break;
			case 8:
				val64 = RTS_RTH_UDP_IPV6_EX_EN;
				break;
			case 9:
				val64 = RTS_RTH_IPV6_EX_EN;
				break;
			default:
				val64 = RTS_RTH_TCP_IPV4_EN;
				break;
		}
	        val64 |= RTS_RTH_EN | RTS_RTH_BUCKET_SIZE(rth_sz);
	        writeq(val64, &bar0->rts_rth_cfg);
	}

	return 0;
}

/**
 * spdm_extract_table -
 */
static int spdm_extract_table(void *data, nic_t *nic)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
	u64 val64, table_content;
	int line, entry = 0;

	while (entry < nic->spdm_entry) {
		u64 *tmp = (u64 *)((u8 *)data + (0x40 * entry));
		for (line = 0; line < 8; line++, tmp++) {
			val64 = RTS_RTH_SPDM_MEM_CTRL_OFFSET(entry) |
				RTS_RTH_SPDM_MEM_CTRL_LINE_SEL(line) |
				RTS_RTH_SPDM_MEM_CTRL_STROBE;
			writeq(val64, &bar0->rts_rth_spdm_mem_ctrl);
			if (wait_for_cmd_complete(&bar0->rts_rth_spdm_mem_ctrl,
			      RTS_RTH_SPDM_MEM_CTRL_STROBE,
			  		S2IO_BIT_RESET) == FAILURE)
				return FAILURE;
			table_content = readq(&bar0->rts_rth_spdm_mem_data);
			if (!line && !table_content)
				goto end;
			*tmp = table_content;
		}
		entry++;
	}
end:
	return entry;
}

/**
 * s2io_program_spdm_table -
 */
static int s2io_program_spdm_table(spdm_cfg_t *info, int entry, nic_t *nic)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
	u64 val64;
	unsigned long tmp;
	int ring;
	u32 start_tbl, entry_offset;
	spdm_entry_t element;
	void *element_addr;
	u16 sprt, dprt;
	u32 sip, dip, hash;

	ring = info->t_queue;
	entry = nic->spdm_entry;

	val64 = readq(&bar0->spdm_bir_offset);
	start_tbl = (u32)(val64 >> 32);
	start_tbl *= 8;
	entry_offset = entry * sizeof(spdm_entry_t);

	element_addr = (void *)((u8 *)bar0 + start_tbl + entry_offset);
	tmp = (unsigned long)element_addr;

	sprt = info->sprt;
	dprt = info->dprt;
	sip = info->sip;
	dip = info->dip;
	element.port_n_entry_control_0 = SPDM_PGM_L4_SRC_PORT(sprt) |
			SPDM_PGM_L4_DST_PORT(dprt) |
			SPDM_PGM_TARGET_QUEUE(ring);
	if (info->sprt)
		element.port_n_entry_control_0 |= SPDM_PGM_IS_TCP |
			SPDM_PGM_IS_IPV4;
	writeq(element.port_n_entry_control_0, element_addr);
	tmp += 8;
	element_addr = (void *)tmp;

	element.ip.ipv4_sa_da = sip;
	element.ip.ipv4_sa_da <<= 32;
	element.ip.ipv4_sa_da |= dip;
	writeq(element.ip.ipv4_sa_da, element_addr);

	tmp += 8;
	element_addr = (void *)tmp;
	writeq(0, element_addr);
	tmp += 8;
	element_addr = (void *)tmp;
	writeq(0, element_addr);
	tmp += 8;
	element_addr = (void *)tmp;
	writeq(0, element_addr);
	tmp += 8;
	element_addr = (void *)tmp;
	writeq(0, element_addr);
	tmp += 8;
	element_addr = (void *)tmp;
	writeq(0, element_addr);
	tmp += 8;
	element_addr = (void *)tmp;

	hash = info->hash;
	element.hash_n_entry_control_1 = SPDM_PGM_HASH(hash) |
					SPDM_PGM_ENABLE_ENTRY;
	writeq(element.hash_n_entry_control_1, element_addr);
	msleep(20);

	return 0;
}

/**
 * spdm_configure -
 */
static int spdm_configure(nic_t *nic, spdm_cfg_t *info)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
	struct net_device *dev = nic->dev;
	u64 val64;
	int ret;

	val64 = readq(&bar0->spdm_bir_offset);
	if (!(val64 & (vBIT(3, 0, 2)))) {
		s2io_program_spdm_table(info, nic->spdm_entry, nic);
		nic->spdm_entry++;
		if (nic->spdm_entry == MAX_SUPPORTED_SPDM_ENTRIES)
			nic->spdm_entry = 0;
		ret = SUCCESS;
	} else {
		DBG_PRINT(ERR_DBG,"SPDM table of %s is not in BAR0!!\n",
			  dev->name);
		info->ret = SPDM_TABLE_UNKNOWN_BAR;
		ret = FAILURE;
	}

	return ret;
}

/**
 *	rts_ds_steer - Receive traffic steering based on IPv4 or IPv6 TOS
 *	or Traffic class respectively.
 *	@nic: device peivate variable
 *	Description: The function configures the receive steering to
 *	desired receive ring.
 *	Return Value:  SUCCESS on success and
 *	'-1' on failure (endian settings incorrect).
 */
static int rts_ds_steer(struct s2io_nic *nic, u8 ds_codepoint, u8 ring)
{
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	register u64 val64 = 0;

	if (ds_codepoint > 63)
		return FAILURE;

	val64 = RTS_DS_MEM_DATA(ring);

	writeq(val64, &bar0->rts_ds_mem_data);

	val64 = RTS_DS_MEM_CTRL_WE |
			RTS_DS_MEM_CTRL_STROBE_NEW_CMD|
			RTS_DS_MEM_CTRL_OFFSET(ds_codepoint);

	writeq(val64, &bar0->rts_ds_mem_ctrl);

	return wait_for_cmd_complete(&bar0->rts_ds_mem_ctrl,
				RTS_DS_MEM_CTRL_STROBE_CMD_BEING_EXECUTED,
				S2IO_BIT_RESET);
}

/**
 *  init_nic - Initialization of hardware
 *  @nic: device peivate variable
 *  Description: The function sequentially configures every block
 *  of the H/W from their reset values.
 *  Return Value:  SUCCESS on success and
 *  '-1' on failure (endian settings incorrect).
 */

static int init_nic(struct s2io_nic *nic)
{
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	struct net_device *dev = nic->dev;
	register u64 val64 = 0;
	void __iomem *add;
	u32 time;
	int i, j;
	mac_info_t *mac_control;
	struct config_param *config;
	int dtx_cnt = 0;
	unsigned long long mem_share;
	int mem_size;

	mac_control = &nic->mac_control;
	config = &nic->config;

	/* to set the swapper control on the card */
	if(s2io_set_swapper(nic)) {
		DBG_PRINT(ERR_DBG,"ERROR: Setting Swapper failed\n");
		return -EIO;
	}

	/*
	 * Herc requires EOI to be removed from reset before XGXS, so..
	 */
	if (nic->device_type & XFRAME_II_DEVICE) {
		val64 = 0xA500000000ULL;
		writeq(val64, &bar0->sw_reset);
		msleep(500);
		val64 = readq(&bar0->sw_reset);
	}

	/* Remove XGXS from reset state */
	writeq(0, &bar0->sw_reset);
	msleep(500);
	val64 = readq(&bar0->sw_reset);

	/* Ensure that it's safe to access registers by checking
	 * RIC_RUNNING bit is reset. Check is valid only for XframeII.
	 */
	if (nic->device_type == XFRAME_II_DEVICE) {
		for (i=0; i < 50; i++) {
			val64 = readq(&bar0->adapter_status);
			if (!(val64 & ADAPTER_STATUS_RIC_RUNNING))
				break;
			msleep(10);
		}
		if (i == 50)
			return -ENODEV;
	}

	/*  Enable Receiving broadcasts */
	add = &bar0->mac_cfg;
	val64 = readq(&bar0->mac_cfg);
	val64 |= MAC_RMAC_BCAST_ENABLE;
	writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
	writel((u32) val64, add);
	writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
	writel((u32) (val64 >> 32), (add + 4));

	/* Read registers in all blocks */
	val64 = readq(&bar0->mac_int_mask);
	val64 = readq(&bar0->mc_int_mask);
	val64 = readq(&bar0->xgxs_int_mask);

	/*  Set MTU */
	val64 = dev->mtu;
	writeq(vBIT(val64, 2, 14), &bar0->rmac_max_pyld_len);

	if (nic->device_type & XFRAME_II_DEVICE) {
		while (herc_act_dtx_cfg[dtx_cnt] != END_SIGN) {
			SPECIAL_REG_WRITE(herc_act_dtx_cfg[dtx_cnt],
					  &bar0->dtx_control, UF);
			if (dtx_cnt & 0x1)
				msleep(1); /* Necessary!! */
			dtx_cnt++;
		}
	} else {
		while (xena_dtx_cfg[dtx_cnt] != END_SIGN) {
			SPECIAL_REG_WRITE(xena_dtx_cfg[dtx_cnt],
					  &bar0->dtx_control, UF);
			val64 = readq(&bar0->dtx_control);
			dtx_cnt++;
		}
	}

	/*  Tx DMA Initialization */
	val64 = 0;
	writeq(val64, &bar0->tx_fifo_partition_0);
	writeq(val64, &bar0->tx_fifo_partition_1);
	writeq(val64, &bar0->tx_fifo_partition_2);
	writeq(val64, &bar0->tx_fifo_partition_3);

	for (i = 0, j = 0; i < config->tx_fifo_num; i++) {
		val64 |=
			vBIT(config->tx_cfg[i].fifo_len - 1, ((j * 32) + 19),
			 13) | vBIT(config->tx_cfg[i].fifo_priority,
				    ((j * 32) + 5), 3);

		if (i == (config->tx_fifo_num - 1)) {
			if (i % 2 == 0)
				i++;
		}

		switch (i) {
		case 1:
			writeq(val64, &bar0->tx_fifo_partition_0);
			val64 = 0;
			j = 0;
			break;
		case 3:
			writeq(val64, &bar0->tx_fifo_partition_1);
			val64 = 0;
			j = 0;
			break;
		case 5:
			writeq(val64, &bar0->tx_fifo_partition_2);
			val64 = 0;
			j = 0;
			break;
		case 7:
			writeq(val64, &bar0->tx_fifo_partition_3);
			val64 = 0;
			j = 0;
			break;
		default:
			j++;
			break;
		}
	}

	/* Enable Tx FIFO partition 0. */
	val64 = readq(&bar0->tx_fifo_partition_0);
	val64 |= BIT(0);	/* To enable the FIFO partition. */
	writeq(val64, &bar0->tx_fifo_partition_0);

	/*
	 * Disable 4 PCCs for Xena1, 2 and 3 as per H/W bug
	 * SXE-008 TRANSMIT DMA ARBITRATION ISSUE.
	 */
	if ((nic->device_type == XFRAME_I_DEVICE) &&
		(get_xena_rev_id(nic->pdev) < 4))
		writeq(PCC_ENABLE_FOUR, &bar0->pcc_enable);

	val64 = readq(&bar0->tx_fifo_partition_0);
	DBG_PRINT(INIT_DBG, "Fifo partition at: 0x%p is: 0x%llx\n",
		  &bar0->tx_fifo_partition_0, (unsigned long long) val64);

	/*
	 * Initialization of Tx_PA_CONFIG register to ignore packet
	 * integrity checking.
	 */
	val64 = readq(&bar0->tx_pa_cfg);
	val64 |= TX_PA_CFG_IGNORE_FRM_ERR | TX_PA_CFG_IGNORE_SNAP_OUI |
	    TX_PA_CFG_IGNORE_LLC_CTRL | TX_PA_CFG_IGNORE_L2_ERR;
	writeq(val64, &bar0->tx_pa_cfg);

	/* Rx DMA intialization. */
	val64 = 0;
	for (i = 0; i < config->rx_ring_num; i++) {
		val64 |=
		    vBIT(config->rx_cfg[i].ring_priority, (5 + (i * 8)),
			 3);
	}
	writeq(val64, &bar0->rx_queue_priority);

	/*
	 * Allocating equal share of memory to all the
	 * configured Rings.
	 */
	val64 = 0;
	if (nic->device_type & XFRAME_II_DEVICE)
		mem_size = 32;
	else
		mem_size = 64;

	for (i = 0; i < config->rx_ring_num; i++) {
		switch (i) {
		case 0:
			mem_share = (mem_size / config->rx_ring_num +
				     mem_size % config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q0_SZ(mem_share);
			continue;
		case 1:
			mem_share = (mem_size / config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q1_SZ(mem_share);
			continue;
		case 2:
			mem_share = (mem_size / config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q2_SZ(mem_share);
			continue;
		case 3:
			mem_share = (mem_size / config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q3_SZ(mem_share);
			continue;
		case 4:
			mem_share = (mem_size / config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q4_SZ(mem_share);
			continue;
		case 5:
			mem_share = (mem_size / config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q5_SZ(mem_share);
			continue;
		case 6:
			mem_share = (mem_size / config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q6_SZ(mem_share);
			continue;
		case 7:
			mem_share = (mem_size / config->rx_ring_num);
			val64 |= RX_QUEUE_CFG_Q7_SZ(mem_share);
			continue;
		}
	}
	writeq(val64, &bar0->rx_queue_cfg);

	/*
	 * Filling Tx round robin registers
	 * as per the number of FIFOs
	 */
	switch (config->tx_fifo_num) {
	case 1:
		val64 = 0x0;
		writeq(val64, &bar0->tx_w_round_robin_0);
		writeq(val64, &bar0->tx_w_round_robin_1);
		writeq(val64, &bar0->tx_w_round_robin_2);
		writeq(val64, &bar0->tx_w_round_robin_3);
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	case 2:
		val64 = 0x0001000100010001ULL;
		writeq(val64, &bar0->tx_w_round_robin_0);
		writeq(val64, &bar0->tx_w_round_robin_1);
		writeq(val64, &bar0->tx_w_round_robin_2);
		writeq(val64, &bar0->tx_w_round_robin_3);
		val64 = 0x0001000100000000ULL;
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	case 3:
		val64 = 0x0001020001020001ULL;
		writeq(val64, &bar0->tx_w_round_robin_0);
		val64 = 0x0200010200010200ULL;
		writeq(val64, &bar0->tx_w_round_robin_1);
		val64 = 0x0102000102000102ULL;
		writeq(val64, &bar0->tx_w_round_robin_2);
		val64 = 0x0001020001020001ULL;
		writeq(val64, &bar0->tx_w_round_robin_3);
		val64 = 0x0200010200000000ULL;
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	case 4:
		val64 = 0x0001020300010203ULL;
		writeq(val64, &bar0->tx_w_round_robin_0);
		writeq(val64, &bar0->tx_w_round_robin_1);
		writeq(val64, &bar0->tx_w_round_robin_2);
		writeq(val64, &bar0->tx_w_round_robin_3);
		val64 = 0x0001020300000000ULL;
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	case 5:
		val64 = 0x0001020304000102ULL;
		writeq(val64, &bar0->tx_w_round_robin_0);
		val64 = 0x0304000102030400ULL;
		writeq(val64, &bar0->tx_w_round_robin_1);
		val64 = 0x0102030400010203ULL;
		writeq(val64, &bar0->tx_w_round_robin_2);
		val64 = 0x0400010203040001ULL;
		writeq(val64, &bar0->tx_w_round_robin_3);
		val64 = 0x0203040000000000ULL;
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	case 6:
		val64 = 0x0001020304050001ULL;
		writeq(val64, &bar0->tx_w_round_robin_0);
		val64 = 0x0203040500010203ULL;
		writeq(val64, &bar0->tx_w_round_robin_1);
		val64 = 0x0405000102030405ULL;
		writeq(val64, &bar0->tx_w_round_robin_2);
		val64 = 0x0001020304050001ULL;
		writeq(val64, &bar0->tx_w_round_robin_3);
		val64 = 0x0203040500000000ULL;
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	case 7:
		val64 = 0x0001020304050600ULL;
		writeq(val64, &bar0->tx_w_round_robin_0);
		val64 = 0x0102030405060001ULL;
		writeq(val64, &bar0->tx_w_round_robin_1);
		val64 = 0x0203040506000102ULL;
		writeq(val64, &bar0->tx_w_round_robin_2);
		val64 = 0x0304050600010203ULL;
		writeq(val64, &bar0->tx_w_round_robin_3);
		val64 = 0x0405060000000000ULL;
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	case 8:
		val64 = 0x0001020304050607ULL;
		writeq(val64, &bar0->tx_w_round_robin_0);
		writeq(val64, &bar0->tx_w_round_robin_1);
		writeq(val64, &bar0->tx_w_round_robin_2);
		writeq(val64, &bar0->tx_w_round_robin_3);
		val64 = 0x0001020300000000ULL;
		writeq(val64, &bar0->tx_w_round_robin_4);
		break;
	}

	/* Filling the Rx round robin registers as per the
	 * number of Rings and steering based on QoS.
	*/

	/*
	 * Classic mode: Specify which queues a frame with a particular
	 * 		QoS value may be steered to
	 * Enhanced mode: Maps QoS to a priority level which is
	 * 		then mapped to a queue
	*/

	switch (config->rx_ring_num) {
	case 1:
		val64 = 0x0;
		writeq(val64, &bar0->rx_w_round_robin_0);
		writeq(val64, &bar0->rx_w_round_robin_1);
		writeq(val64, &bar0->rx_w_round_robin_2);
		writeq(val64, &bar0->rx_w_round_robin_3);
		writeq(val64, &bar0->rx_w_round_robin_4);

		if(rx_steering_type) {
			val64 = 0x0180018001800180ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0180018001800180ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x00ULL;
		} else
			val64 = 0x8080808080808080ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	case 2:
		val64 = 0x0001000100010001ULL;
		writeq(val64, &bar0->rx_w_round_robin_0);
		writeq(val64, &bar0->rx_w_round_robin_1);
		writeq(val64, &bar0->rx_w_round_robin_2);
		writeq(val64, &bar0->rx_w_round_robin_3);
		val64 = 0x0001000100000000ULL;
		writeq(val64, &bar0->rx_w_round_robin_4);

		if(rx_steering_type) {
			val64 = 0x0180018001800180ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0140014001400140ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x0000000001010101ULL;
		} else
			val64 = 0x8080808040404040ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	case 3:
		val64 = 0x0001020001020001ULL;
		writeq(val64, &bar0->rx_w_round_robin_0);
		val64 = 0x0200010200010200ULL;
		writeq(val64, &bar0->rx_w_round_robin_1);
		val64 = 0x0102000102000102ULL;
		writeq(val64, &bar0->rx_w_round_robin_2);
		val64 = 0x0001020001020001ULL;
		writeq(val64, &bar0->rx_w_round_robin_3);
		val64 = 0x0200010200000000ULL;
		writeq(val64, &bar0->rx_w_round_robin_4);

		if(rx_steering_type) {
			val64 = 0x0180018001800140ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0140014001200120ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x0000000101010202ULL;
		} else
			val64 = 0x8080804040402020ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	case 4:
		val64 = 0x0001020300010203ULL;
		writeq(val64, &bar0->rx_w_round_robin_0);
		writeq(val64, &bar0->rx_w_round_robin_1);
		writeq(val64, &bar0->rx_w_round_robin_2);
		writeq(val64, &bar0->rx_w_round_robin_3);
		val64 = 0x0001020300000000ULL;
		writeq(val64, &bar0->rx_w_round_robin_4);

		if(rx_steering_type){
			val64 = 0x0180018001400140ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0120012001100110ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x0000010102020303ULL;
		} else
			val64 = 0x8080404020201010ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	case 5:
		val64 = 0x0001020304000102ULL;
		writeq(val64, &bar0->rx_w_round_robin_0);
		val64 = 0x0304000102030400ULL;
		writeq(val64, &bar0->rx_w_round_robin_1);
		val64 = 0x0102030400010203ULL;
		writeq(val64, &bar0->rx_w_round_robin_2);
		val64 = 0x0400010203040001ULL;
		writeq(val64, &bar0->rx_w_round_robin_3);
		val64 = 0x0203040000000000ULL;
		writeq(val64, &bar0->rx_w_round_robin_4);

		if(rx_steering_type) {
			val64 = 0x0180018001400140ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0120012001100108ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x0000010102020304ULL;
		} else
			val64 = 0x8080404020201008ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	case 6:
		val64 = 0x0001020304050001ULL;
		writeq(val64, &bar0->rx_w_round_robin_0);
		val64 = 0x0203040500010203ULL;
		writeq(val64, &bar0->rx_w_round_robin_1);
		val64 = 0x0405000102030405ULL;
		writeq(val64, &bar0->rx_w_round_robin_2);
		val64 = 0x0001020304050001ULL;
		writeq(val64, &bar0->rx_w_round_robin_3);
		val64 = 0x0203040500000000ULL;
		writeq(val64, &bar0->rx_w_round_robin_4);

		if(rx_steering_type) {
			val64 = 0x0180018001400140ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0120011001080104ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x0000010102030405ULL;
		} else
			val64 = 0x8080404020100804ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	case 7:
		val64 = 0x0001020304050600ULL;
		writeq(val64, &bar0->rx_w_round_robin_0);
		val64 = 0x0102030405060001ULL;
		writeq(val64, &bar0->rx_w_round_robin_1);
		val64 = 0x0203040506000102ULL;
		writeq(val64, &bar0->rx_w_round_robin_2);
		val64 = 0x0304050600010203ULL;
		writeq(val64, &bar0->rx_w_round_robin_3);
		val64 = 0x0405060000000000ULL;
		writeq(val64, &bar0->rx_w_round_robin_4);
		if(rx_steering_type) {
			val64 = 0x0180018001400120ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0110010801040102ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x0000010203040506ULL;
		} else
			val64 = 0x8080402010080402ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	case 8:
		val64 = 0x0001020304050607ULL;
		writeq(val64, &bar0->rx_w_round_robin_0);
		writeq(val64, &bar0->rx_w_round_robin_1);
		writeq(val64, &bar0->rx_w_round_robin_2);
		writeq(val64, &bar0->rx_w_round_robin_3);
		val64 = 0x0001020300000000ULL;
		writeq(val64, &bar0->rx_w_round_robin_4);

		if(rx_steering_type) {
			val64 = 0x0180014001200110ULL;
			writeq(val64, &bar0->rts_p0_p3_map);
			val64 = 0x0108010401020101ULL;
			writeq(val64, &bar0->rts_p4_p7_map);
			val64 = 0x0001020304050607ULL;
		} else
			val64 = 0x8040201008040201ULL;
		writeq(val64, &bar0->rts_qos_steering);

		break;
	}

	/* UDP Fix */
	val64 = 0;
	for (i = 0; i < 8; i++)
		writeq(val64, &bar0->rts_frm_len_n[i]);

	/* Set the default rts frame length for the rings configured */
	val64 = MAC_RTS_FRM_LEN_SET(dev->mtu+22);
	for (i = 0 ; i < config->rx_ring_num ; i++)
		writeq(val64, &bar0->rts_frm_len_n[i]);

	/* Set the frame length for the configured rings
	 * desired by the user
	 */
	for (i = 0; i < config->rx_ring_num; i++) {
		/* If rts_frm_len[i] == 0 then it is assumed that user not
		 * specified frame length steering.
		 * If the user provides the frame length then program
		 * the rts_frm_len register for those values or else
		 * leave it as it is.
		 */
		if (rts_frm_len[i] != 0) {
			writeq(MAC_RTS_FRM_LEN_SET(rts_frm_len[i]),
				&bar0->rts_frm_len_n[i]);
		}
	}

	/*
	 * Disable differentiated services steering logic
	 */
	for (i = 0; i < 64; i++)
		if (rts_ds_steer(nic, i, 0) == FAILURE){
			DBG_PRINT(ERR_DBG, "%s: failed rts ds steering", dev->name);
			DBG_PRINT(ERR_DBG, "set on codepoint %d\n", i);
			return -ENODEV;
		}

	/* Program statistics memory */
	writeq(mac_control->stats_mem_phy, &bar0->stat_addr);

	if (nic->device_type == XFRAME_II_DEVICE) {
		val64 = STAT_BC(0x320);
		writeq(val64, &bar0->stat_byte_cnt);
	}

	/*
	 * Initializing the sampling rate for the device to calculate the
	 * bandwidth utilization.
	 */
	val64 = MAC_TX_LINK_UTIL_VAL(tmac_util_period) |
	    MAC_RX_LINK_UTIL_VAL(rmac_util_period);
	writeq(val64, &bar0->mac_link_util);


	/*
	 * Initializing the Transmit and Receive Traffic Interrupt
	 * Scheme.
	 */
	/*
	 * TTI Initialization. Default Tx timer gets us about
	 * 250 interrupts per sec. Continuous interrupts are enabled
	 * by default.
	 */
	if (nic->device_type == XFRAME_II_DEVICE) {
		int count = (nic->config.bus_speed * 125)/2;
		val64 = TTI_DATA1_MEM_TX_TIMER_VAL(count);
	} else {

		val64 = TTI_DATA1_MEM_TX_TIMER_VAL(0x2078);
	}
	val64 |= TTI_DATA1_MEM_TX_URNG_A(0xA) |
	    TTI_DATA1_MEM_TX_URNG_B(0x10) |
	    TTI_DATA1_MEM_TX_URNG_C(0x30) | TTI_DATA1_MEM_TX_TIMER_AC_EN;
	writeq(val64, &bar0->tti_data1_mem);

	val64 = TTI_DATA2_MEM_TX_UFC_A(0x10) |
	    TTI_DATA2_MEM_TX_UFC_B(0x20) |
	    TTI_DATA2_MEM_TX_UFC_C(0x40) | TTI_DATA2_MEM_TX_UFC_D(0x80);
	writeq(val64, &bar0->tti_data2_mem);

	val64 = TTI_CMD_MEM_WE | TTI_CMD_MEM_STROBE_NEW_CMD;
	writeq(val64, &bar0->tti_command_mem);

	/*
	 * Once the operation completes, the Strobe bit of the command
	 * register will be reset. We poll for this particular condition
	 * We wait for a maximum of 500ms for the operation to complete,
	 * if it's not complete by then we return error.
	 */
	time = 0;
	while (TRUE) {
		val64 = readq(&bar0->tti_command_mem);
		if (!(val64 & TTI_CMD_MEM_STROBE_NEW_CMD)) {
			break;
		}
		if (time > 10) {
			DBG_PRINT(ERR_DBG, "%s: TTI init Failed\n",
				  dev->name);
			return -ENODEV;
		}
		msleep(50);
		time++;
	}

	if (nic->config.bimodal) {
		int k = 0;
		for (k = 0; k < config->rx_ring_num; k++) {
			val64 = TTI_CMD_MEM_WE | TTI_CMD_MEM_STROBE_NEW_CMD;
			val64 |= TTI_CMD_MEM_OFFSET(0x38+k);
			writeq(val64, &bar0->tti_command_mem);

		/*
		 * Once the operation completes, the Strobe bit of the command
		 * register will be reset. We poll for this particular condition
		 * We wait for a maximum of 500ms for the operation to complete,
		 * if it's not complete by then we return error.
		*/
			time = 0;
			while (TRUE) {
				val64 = readq(&bar0->tti_command_mem);
				if (!(val64 & TTI_CMD_MEM_STROBE_NEW_CMD)) {
					break;
				}
				if (time > 10) {
					DBG_PRINT(ERR_DBG,
						"%s: TTI init Failed\n",
					dev->name);
					return -ENODEV;
				}
				time++;
				msleep(50);
			}
		}
	} else {

		/* RTI Initialization */
		if (nic->device_type == XFRAME_II_DEVICE) {
			/*
			 * Programmed to generate Apprx 500 Intrs per
			 * second
			 */
			int count = (nic->config.bus_speed * 125)/4;
			val64 = RTI_DATA1_MEM_RX_TIMER_VAL(count);
		} else {
			val64 = RTI_DATA1_MEM_RX_TIMER_VAL(0xFFF);
		}
		val64 |= RTI_DATA1_MEM_RX_URNG_A(0xA) |
		    RTI_DATA1_MEM_RX_URNG_B(0x10) |
		    RTI_DATA1_MEM_RX_URNG_C(0x30) | RTI_DATA1_MEM_RX_TIMER_AC_EN;

		writeq(val64, &bar0->rti_data1_mem);

		val64 = RTI_DATA2_MEM_RX_UFC_A(0x1) |
		    RTI_DATA2_MEM_RX_UFC_B(0x2);
		if (nic->intr_type == MSI_X)
		    val64 |= RTI_DATA2_MEM_RX_UFC_C(0x20) | \
				RTI_DATA2_MEM_RX_UFC_D(0x40);
		else
		    val64 |= RTI_DATA2_MEM_RX_UFC_C(0x40) | \
				RTI_DATA2_MEM_RX_UFC_D(0x80);
		writeq(val64, &bar0->rti_data2_mem);

		for (i = 0; i < config->rx_ring_num; i++) {
			val64 = RTI_CMD_MEM_WE | RTI_CMD_MEM_STROBE_NEW_CMD
					| RTI_CMD_MEM_OFFSET(i);
			writeq(val64, &bar0->rti_command_mem);

			/*
			 * Once the operation completes, the Strobe bit of the
			 * command register will be reset. We poll for this
			 * particular condition. We wait for a maximum of 500ms
			 * for the operation to complete, if it's not complete
			 * by then we return error.
			 */
			time = 0;
			while (TRUE) {
				val64 = readq(&bar0->rti_command_mem);
				if (!(val64 & RTI_CMD_MEM_STROBE_NEW_CMD)) {
					break;
				}
				if (time > 10) {
					DBG_PRINT(ERR_DBG, "%s: RTI init Failed\n",
						  dev->name);
					return -ENODEV;
				}
				time++;
				msleep(50);
			}
		}
	}

	/* Configure RTH */
	if (config->rth_enable & RX_STEERING_SUPPORT) {
		s2io_rth_configure(nic);
	}

	/*
	 * Initializing proper values as Pause threshold into all
	 * the 8 Queues on Rx side.
	 */
	writeq(0xffbbffbbffbbffbbULL, &bar0->mc_pause_thresh_q0q3);
	writeq(0xffbbffbbffbbffbbULL, &bar0->mc_pause_thresh_q4q7);

	/* Disable RMAC PAD STRIPPING */
	add = &bar0->mac_cfg;
	val64 = readq(&bar0->mac_cfg);
	val64 &= ~(MAC_CFG_RMAC_STRIP_PAD);
	writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
	writel((u32) (val64), add);
	writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
	writel((u32) (val64 >> 32), (add + 4));
	val64 = readq(&bar0->mac_cfg);

	/* Enable FCS stripping by adapter */
	add = &bar0->mac_cfg;
	val64 = readq(&bar0->mac_cfg);
	val64 |= MAC_CFG_RMAC_STRIP_FCS;
	if (nic->device_type == XFRAME_II_DEVICE)
		writeq(val64, &bar0->mac_cfg);
	else {
		writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
		writel((u32) (val64), add);
		writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
		writel((u32) (val64 >> 32), (add + 4));
	}

	/*
	 * Set the time value to be inserted in the pause frame
	 * generated by xena.
	 */
	val64 = readq(&bar0->rmac_pause_cfg);
	val64 &= ~(RMAC_PAUSE_HG_PTIME(0xffff));
	val64 |= RMAC_PAUSE_HG_PTIME(nic->mac_control.rmac_pause_time);
	writeq(val64, &bar0->rmac_pause_cfg);

	/*
	 * Set the Threshold Limit for Generating the pause frame
	 * If the amount of data in any Queue exceeds ratio of
	 * (mac_control.mc_pause_threshold_q0q3 or q4q7)/256
	 * pause frame is generated
	 */
	val64 = 0;
	for (i = 0; i < 4; i++) {
		val64 |=
		    (((u64) 0xFF00 | nic->mac_control.
		      mc_pause_threshold_q0q3)
		     << (i * 2 * 8));
	}
	writeq(val64, &bar0->mc_pause_thresh_q0q3);

	val64 = 0;
	for (i = 0; i < 4; i++) {
		val64 |=
		    (((u64) 0xFF00 | nic->mac_control.
		      mc_pause_threshold_q4q7)
		     << (i * 2 * 8));
	}
	writeq(val64, &bar0->mc_pause_thresh_q4q7);

	/*
	 * TxDMA will stop Read request if the number of read split has
	 * exceeded the limit pointed by shared_splits
	 */
	val64 = readq(&bar0->pic_control);
	val64 |= PIC_CNTL_SHARED_SPLITS(shared_splits);
	writeq(val64, &bar0->pic_control);

	if (nic->config.bus_speed == 266) {
		writeq(TXREQTO_VAL(0x7f) | TXREQTO_EN, &bar0->txreqtimeout);
		writeq(0x0, &bar0->read_retry_delay);
		writeq(0x0, &bar0->write_retry_delay);
	}

	/*
	 * Programming the Herc to split every write transaction
	 * that does not start on an ADB to reduce disconnects.
	 */
	if (nic->device_type == XFRAME_II_DEVICE) {
		val64 = FAULT_BEHAVIOUR | EXT_REQ_EN |
				 MISC_LINK_STABILITY_PRD(3);
		writeq(val64, &bar0->misc_control);
		val64 = readq(&bar0->pic_control2);
		val64 &= ~(BIT(13)|BIT(14)|BIT(15));
		writeq(val64, &bar0->pic_control2);
	}
	if (strstr(nic->product_name, "CX4")) {
		val64 = TMAC_AVG_IPG(0x17);
		writeq(val64, &bar0->tmac_avg_ipg);
	}

	nic->spdm_entry = 0;
	nic->device_initialized = TRUE;
	return SUCCESS;
}

#define LINK_UP_DOWN_INTERRUPT		1
#define MAC_RMAC_ERR_TIMER		2

int s2io_link_fault_indication(nic_t *nic)
{
	if (nic->intr_type != INTA)
		return MAC_RMAC_ERR_TIMER;
	if (nic->device_type == XFRAME_II_DEVICE)
		return LINK_UP_DOWN_INTERRUPT;
	else
		return MAC_RMAC_ERR_TIMER;
}

/**
 *  en_dis_able_nic_intrs - Enable or Disable the interrupts
 *  @nic: device private variable,
 *  @mask: A mask indicating which Intr block must be modified and,
 *  @flag: A flag indicating whether to enable or disable the Intrs.
 *  Description: This function will either disable or enable the interrupts
 *  depending on the flag argument. The mask argument can be used to
 *  enable/disable any Intr block.
 *  Return Value: NONE.
 */

static void en_dis_able_nic_intrs(struct s2io_nic *nic, u16 mask, int flag)
{
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	register u64 intr_mask = 0, temp64 = 0;

	/*  Top level interrupt classification */
	/*  PIC Interrupts */
	if (mask & TX_PIC_INTR) {
		/*  Enable PIC Intrs in the general intr mask register */
		intr_mask |= TXPIC_INT_M;
		if (flag == ENABLE_INTRS) {
			/*
			 * If Hercules adapter enable GPIO otherwise
			 * disable all PCIX, Flash, MDIO, IIC and GPIO
			 * interrupts for now.
			 * TODO
			 */
			if (s2io_link_fault_indication(nic) ==
					LINK_UP_DOWN_INTERRUPT ) {
				temp64 = readq(&bar0->pic_int_mask);
				temp64 &= ~((u64) PIC_INT_GPIO);
				writeq(temp64, &bar0->pic_int_mask);

				temp64 = readq(&bar0->gpio_int_mask);
				temp64 &= ~((u64) GPIO_INT_MASK_LINK_UP);
				writeq(temp64, &bar0->gpio_int_mask);
			} else {
				writeq(DISABLE_ALL_INTRS, &bar0->pic_int_mask);
			}
			/*
			 * No MSI Support is available presently, so TTI and
			 * RTI interrupts are also disabled.
			 */
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable PIC Intrs in the general
			 * intr mask register
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->pic_int_mask);
		}
	}

	/*  DMA Interrupts */
	/*  Enabling/Disabling Tx DMA interrupts */
	if (mask & TX_DMA_INTR) {
		/* Enable TxDMA Intrs in the general intr mask register */
		intr_mask |= TXDMA_INT_M;
		if (flag == ENABLE_INTRS) {
			/*
			 * Keep all interrupts other than PFC interrupt
			 * and PCC interrupt disabled in DMA level.
			 */
			temp64 = DISABLE_ALL_INTRS & ~(TXDMA_PFC_INT_M |
						      TXDMA_PCC_INT_M);
			writeq(temp64, &bar0->txdma_int_mask);
			/*
			 * Enable only the MISC error 1 interrupt in PFC block
			 */
			temp64 = DISABLE_ALL_INTRS & (~PFC_MISC_ERR_1);
			writeq(temp64, &bar0->pfc_err_mask);
			/*
			 * Enable only the FB_ECC error interrupt in PCC block
			 */
			temp64 = DISABLE_ALL_INTRS & (~PCC_FB_ECC_ERR);
			writeq(temp64, &bar0->pcc_err_mask);
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable TxDMA Intrs in the general intr mask
			 * register
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->txdma_int_mask);
			writeq(DISABLE_ALL_INTRS, &bar0->pfc_err_mask);
		}
	}

	/*  Enabling/Disabling Rx DMA interrupts */
	if (mask & RX_DMA_INTR) {
		/*  Enable RxDMA Intrs in the general intr mask register */
		intr_mask |= RXDMA_INT_M;
		if (flag == ENABLE_INTRS) {
			/*
			 * All RxDMA block interrupts are disabled for now
			 * TODO
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->rxdma_int_mask);
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable RxDMA Intrs in the general intr mask
			 * register
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->rxdma_int_mask);
		}
	}

	/*  MAC Interrupts */
	/*  Enabling/Disabling MAC interrupts */
	if (mask & (TX_MAC_INTR | RX_MAC_INTR)) {
		intr_mask |= TXMAC_INT_M | RXMAC_INT_M;
		if (flag == ENABLE_INTRS) {
			/*
			 * All MAC block error interrupts are disabled for now
			 * TODO
			 */
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable MAC Intrs in the general intr mask register
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->mac_int_mask);
			writeq(DISABLE_ALL_INTRS,
			       &bar0->mac_rmac_err_mask);
		}
	}

	/*  XGXS Interrupts */
	if (mask & (TX_XGXS_INTR | RX_XGXS_INTR)) {
		intr_mask |= TXXGXS_INT_M | RXXGXS_INT_M;
		if (flag == ENABLE_INTRS) {
			/*
			 * All XGXS block error interrupts are disabled for now
			 * TODO
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->xgxs_int_mask);
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable MC Intrs in the general intr mask register
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->xgxs_int_mask);
		}
	}

	/*  Memory Controller(MC) interrupts */
	if (mask & MC_INTR) {
		intr_mask |= MC_INT_M;
		if (flag == ENABLE_INTRS) {
			/*
			 * Enable all MC Intrs.
			 */
			writeq(0x0, &bar0->mc_int_mask);
			writeq(0x0, &bar0->mc_err_mask);
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable MC Intrs in the general intr mask register
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->mc_int_mask);
		}
	}


	/*  Tx traffic interrupts */
	if (mask & TX_TRAFFIC_INTR) {
		intr_mask |= TXTRAFFIC_INT_M;
		if (flag == ENABLE_INTRS) {
			/*
			 * Enable all the Tx side interrupts
			 * writing 0 Enables all 64 TX interrupt levels
			 */
			writeq(0x0, &bar0->tx_traffic_mask);
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable Tx Traffic Intrs in the general intr mask
			 * register.
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->tx_traffic_mask);
		}
	}

	/*  Rx traffic interrupts */
	if (mask & RX_TRAFFIC_INTR) {
		intr_mask |= RXTRAFFIC_INT_M;
		if (flag == ENABLE_INTRS) {
			/* writing 0 Enables all 8 RX interrupt levels */
			writeq(0x0, &bar0->rx_traffic_mask);
		} else if (flag == DISABLE_INTRS) {
			/*
			 * Disable Rx Traffic Intrs in the general intr mask
			 * register.
			 */
			writeq(DISABLE_ALL_INTRS, &bar0->rx_traffic_mask);
		}
	}
	temp64 = readq(&bar0->general_int_mask);
	if (flag == ENABLE_INTRS)
		temp64 &= ~((u64) intr_mask);
	else
		temp64 = DISABLE_ALL_INTRS;
	writeq(temp64, &bar0->general_int_mask);
}

static int check_prc_pcc_state(u64 val64, int flag, int rev_id, int herc)
{
	int ret = 0;

	if (flag == FALSE) {
		if ((!herc && (rev_id >= 4)) || herc) {
			if (!(val64 & ADAPTER_STATUS_RMAC_PCC_IDLE) &&
			    ((val64 & ADAPTER_STATUS_RC_PRC_QUIESCENT) ==
			     ADAPTER_STATUS_RC_PRC_QUIESCENT)) {
				ret = 1;
			}
		}else {
			if (!(val64 & ADAPTER_STATUS_RMAC_PCC_FOUR_IDLE) &&
			    ((val64 & ADAPTER_STATUS_RC_PRC_QUIESCENT) ==
			     ADAPTER_STATUS_RC_PRC_QUIESCENT)) {
				ret = 1;
			}
		}
	} else {
		if ((!herc && (rev_id >= 4)) || herc) {
			if (((val64 & ADAPTER_STATUS_RMAC_PCC_IDLE) ==
			     ADAPTER_STATUS_RMAC_PCC_IDLE) &&
			    (!(val64 & ADAPTER_STATUS_RC_PRC_QUIESCENT) ||
			     ((val64 & ADAPTER_STATUS_RC_PRC_QUIESCENT) ==
			      ADAPTER_STATUS_RC_PRC_QUIESCENT))) {
				ret = 1;
			}
		} else {
			if (((val64 & ADAPTER_STATUS_RMAC_PCC_FOUR_IDLE) ==
			     ADAPTER_STATUS_RMAC_PCC_FOUR_IDLE) &&
			    (!(val64 & ADAPTER_STATUS_RC_PRC_QUIESCENT) ||
			     ((val64 & ADAPTER_STATUS_RC_PRC_QUIESCENT) ==
			      ADAPTER_STATUS_RC_PRC_QUIESCENT))) {
				ret = 1;
			}
		}
	}

	return ret;
}
/**
 *  verify_xena_quiescence - Checks whether the H/W is ready
 *  @val64 :  Value read from adapter status register.
 *  @flag : indicates if the adapter enable bit was ever written once
 *  before.
 *  Description: Returns whether the H/W is ready to go or not. Depending
 *  on whether adapter enable bit was written or not the comparison
 *  differs and the calling function passes the input argument flag to
 *  indicate this.
 *  Return: 1 If xena is quiescence
 *          0 If Xena is not quiescence
 */

static int verify_xena_quiescence(nic_t *sp, u64 val64, int flag)
{
	int ret = 0, herc;
	u64 tmp64 = ~((u64) val64);
	int rev_id = get_xena_rev_id(sp->pdev);

	herc = (sp->device_type == XFRAME_II_DEVICE);
	if (!
	    (tmp64 &
	     (ADAPTER_STATUS_TDMA_READY | ADAPTER_STATUS_RDMA_READY |
	      ADAPTER_STATUS_PFC_READY | ADAPTER_STATUS_TMAC_BUF_EMPTY |
	      ADAPTER_STATUS_PIC_QUIESCENT | ADAPTER_STATUS_MC_DRAM_READY |
	      ADAPTER_STATUS_MC_QUEUES_READY | ADAPTER_STATUS_M_PLL_LOCK |
	      ADAPTER_STATUS_P_PLL_LOCK))) {
		ret = check_prc_pcc_state(val64, flag, rev_id, herc);
	}

	return ret;
}

/**
 * fix_mac_address -  Fix for Mac addr problem on Alpha platforms
 * @sp: Pointer to device specifc structure
 * Description :
 * New procedure to clear mac address reading  problems on Alpha platforms
 *
 */

void fix_mac_address(nic_t * sp)
{
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64;
	int i = 0;

	while (fix_mac[i] != END_SIGN) {
		writeq(fix_mac[i++], &bar0->gpio_control);
		udelay(10);
		val64 = readq(&bar0->gpio_control);
	}
}

/**
 *  start_nic - Turns the device on
 *  @nic : device private variable.
 *  Description:
 *  This function actually turns the device on. Before this  function is
 *  called,all Registers are configured from their reset states
 *  and shared memory is allocated but the NIC is still quiescent. On
 *  calling this function, the device interrupts are cleared and the NIC is
 *  literally switched on by writing into the adapter control register.
 *  Return Value:
 *  SUCCESS on success and -1 on failure.
 */

static int start_nic(struct s2io_nic *nic)
{
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	struct net_device *dev = nic->dev;
	register u64 val64 = 0;
	u16 subid, i;
	mac_info_t *mac_control;
	struct config_param *config;

	mac_control = &nic->mac_control;
	config = &nic->config;

	/*  PRC Initialization and configuration */
	for (i = 0; i < config->rx_ring_num; i++) {
		writeq((u64) mac_control->rings[i].rx_blocks[0].block_dma_addr,
		       &bar0->prc_rxd0_n[i]);

		val64 = readq(&bar0->prc_ctrl_n[i]);
		if (nic->config.bimodal)
			val64 |= PRC_CTRL_BIMODAL_INTERRUPT;
		if (nic->rxd_mode == RXD_MODE_1)
			val64 |= PRC_CTRL_RC_ENABLED;
		else
			val64 |= PRC_CTRL_RC_ENABLED | PRC_CTRL_RING_MODE_3;
		if (nic->device_type == XFRAME_II_DEVICE)
			val64 |= PRC_CTRL_GROUP_READS;
		val64 &= ~PRC_CTRL_RXD_BACKOFF_INTERVAL(0xFFFFFF);
		val64 |= PRC_CTRL_RXD_BACKOFF_INTERVAL(0x1000);
		writeq(val64, &bar0->prc_ctrl_n[i]);
	}

	if (nic->rxd_mode == RXD_MODE_3B) {
		/* Enabling 2 buffer mode by writing into Rx_pa_cfg reg. */
		val64 = readq(&bar0->rx_pa_cfg);
		val64 |= RX_PA_CFG_IGNORE_L2_ERR;
		writeq(val64, &bar0->rx_pa_cfg);
	}

	/*
	 * Enabling MC-RLDRAM. After enabling the device, we timeout
	 * for around 100ms, which is approximately the time required
	 * for the device to be ready for operation.
	 */
	val64 = readq(&bar0->mc_rldram_mrs);
	val64 |= MC_RLDRAM_QUEUE_SIZE_ENABLE | MC_RLDRAM_MRS_ENABLE;
	SPECIAL_REG_WRITE(val64, &bar0->mc_rldram_mrs, UF);
	val64 = readq(&bar0->mc_rldram_mrs);

	msleep(100);	/* Delay by around 100 ms. */

	/* Enabling ECC Protection. */
	val64 = readq(&bar0->adapter_control);
	val64 &= ~ADAPTER_ECC_EN;
	writeq(val64, &bar0->adapter_control);

	/*
	 * Verify if the device is ready to be enabled, if so enable
	 * it.
	 */
	val64 = readq(&bar0->adapter_status);
	if (!verify_xena_quiescence(nic, val64, nic->device_enabled_once)) {
		DBG_PRINT(ERR_DBG, "%s: device is not ready, ", dev->name);
		DBG_PRINT(ERR_DBG, "Adapter status reads: 0x%llx\n",
			  (unsigned long long) val64);
		return FAILURE;
	}

	/*
	 * With some switches, link might be already up at this point.
	 * Because of this weird behavior, when we enable laser,
	 * we may not get link. We need to handle this. We cannot
	 * figure out which switch is misbehaving. So we are forced to
	 * make a global change.
	 */

	/* Enabling Laser. */
	val64 = readq(&bar0->adapter_control);
	val64 |= ADAPTER_EOI_TX_ON;
	writeq(val64, &bar0->adapter_control);

	if (s2io_link_fault_indication(nic) == MAC_RMAC_ERR_TIMER) {
		/*
		 * Dont see link state interrupts initally on some switches,
		 * so directly scheduling the link state task here.
		 */
#ifdef __VMKERNEL_MODULE__
		unsigned long flags;
		spin_lock_irqsave(&nic->task_sync_lock, flags);
		if (nic->device_close_flag != TRUE) {
			nic->link_task_scheduled = 1;
#endif
		schedule_work(&nic->set_link_task);
#ifdef __VMKERNEL_MODULE__
		}
		spin_unlock_irqrestore(&nic->task_sync_lock, flags);
#endif
	}

	/* SXE-002: Initialize link and activity LED */
	subid = nic->pdev->subsystem_device;
	if (((subid & 0xFF) >= 0x07) &&
	    (nic->device_type == XFRAME_I_DEVICE)) {
		val64 = readq(&bar0->gpio_control);
		val64 |= 0x0000800000000000ULL;
		writeq(val64, &bar0->gpio_control);
		val64 = 0x0411040400000000ULL;
		writeq(val64, (void *) ((u8 *) bar0 + 0x2700));
	}

	return SUCCESS;
}

/**
 * s2io_txdlget_skb - Get the skb from txdl, unmap and return skb
 */
struct sk_buff *s2io_txdlget_skb(fifo_info_t *fifo_data, TxD_t *txdlp, int get_off)
{
	nic_t *nic = fifo_data->nic;
	struct sk_buff *skb;
	TxD_t *txds;
	u16 j, frg_cnt;

#ifdef CONFIG_AMD_FIX
	int i = fifo_data->fifo_no;
	u8 *tmp_buf;
#endif

	txds = txdlp;
#ifdef NETIF_F_UFO
	if (txds->Host_Control == (u64) nic->ufo_in_band_v) {
		pci_unmap_single(nic->pdev, (dma_addr_t)
			txds->Buffer_Pointer, sizeof(u64),
			PCI_DMA_TODEVICE);
		txds++;
	}
#endif

#ifdef CONFIG_AMD_FIX
	tmp_buf = (nic->amd_tmp[i] +
		   (AMD_BUF_ALIGN * get_off));
	/*
	 * Check whether tmp_buf was used at all or not
	 * during Xmit routine.
	 */
	if ((unsigned long)tmp_buf == (unsigned long)txds->Host_Control) {
		pci_unmap_single(nic->pdev, (dma_addr_t)
			txds->Buffer_Pointer,
			AMD_BUF_ALIGN,
		PCI_DMA_TODEVICE);
		txds++;
		memset(tmp_buf, 0, AMD_BUF_ALIGN);
	}
#endif
	skb = (struct sk_buff *) ((unsigned long)
			txds->Host_Control);
	if (!skb) {
		memset(txdlp, 0, (sizeof(TxD_t) * fifo_data->max_txds));
		return NULL;
	}
	pci_unmap_single(nic->pdev, (dma_addr_t)
			 txds->Buffer_Pointer,
			 skb->len - skb->data_len,
			 PCI_DMA_TODEVICE);
	frg_cnt = skb_shinfo(skb)->nr_frags;
	if (frg_cnt) {
		txds++;
		for (j = 0; j < frg_cnt; j++, txds++) {
			skb_frag_t *frag = &skb_shinfo(skb)->frags[j];
			if (!txds->Buffer_Pointer)
				break;
			pci_unmap_page(nic->pdev, (dma_addr_t)
					txds->Buffer_Pointer,
				       	frag->size, PCI_DMA_TODEVICE);
		}
	}
	memset(txdlp,0, (sizeof(TxD_t) * fifo_data->max_txds));
	return(skb);
}
/**
 *  free_tx_buffers - Free all queued Tx buffers
 *  @nic : device private variable.
 *  Description:
 *  Free all queued Tx buffers.
 *  Return Value: void
*/

static void free_tx_buffers(struct s2io_nic *nic)
{
	struct net_device *dev = nic->dev;
	struct sk_buff *skb;
	TxD_t *txdp;
	int i, j;
	mac_info_t *mac_control;
	struct config_param *config;
	int cnt = 0;

	mac_control = &nic->mac_control;
	config = &nic->config;

	for (i = 0; i < config->tx_fifo_num; i++) {
		unsigned long flags;

		spin_lock_irqsave(&mac_control->fifos[i].tx_lock, flags);

		for (j = 0; j < config->tx_cfg[i].fifo_len - 1; j++) {
			txdp = (TxD_t *) mac_control->fifos[i].list_info[j].
						list_virt_addr;
			skb = s2io_txdlget_skb(&mac_control->fifos[i], txdp, j);
			if (skb) {
				dev_kfree_skb(skb);
				cnt++;
			}
		}
		DBG_PRINT(INTR_DBG,
			  "%s:forcibly freeing %d skbs on FIFO%d\n",
			  dev->name, cnt, i);
		mac_control->fifos[i].tx_curr_get_info.offset = 0;
		mac_control->fifos[i].tx_curr_put_info.offset = 0;
		spin_unlock_irqrestore(&mac_control->fifos[i].tx_lock, flags);
	}
}

/**
 *   stop_nic -  To stop the nic
 *   @nic ; device private variable.
 *   Description:
 *   This function does exactly the opposite of what the start_nic()
 *   function does. This function is called to stop the device.
 *   Return Value:
 *   void.
 */

static void stop_nic(struct s2io_nic *nic)
{
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	register u64 val64 = 0;
	u16 interruptible;
	mac_info_t *mac_control;
	struct config_param *config;

	mac_control = &nic->mac_control;
	config = &nic->config;

	/*  Disable all interrupts */
	interruptible = TX_TRAFFIC_INTR | RX_TRAFFIC_INTR;
	interruptible |= TX_PIC_INTR | RX_PIC_INTR;
	interruptible |= TX_MAC_INTR | RX_MAC_INTR;

	en_dis_able_nic_intrs(nic, interruptible, DISABLE_INTRS);

	/* Clearing Adapter_En bit of ADAPTER_CONTROL Register */
	val64 = readq(&bar0->adapter_control);
	val64 &= ~(ADAPTER_CNTL_EN);
	writeq(val64, &bar0->adapter_control);
}

int fill_rxd_3buf(nic_t *nic, RxD_t *rxdp, struct sk_buff *skb)
{
	struct net_device *dev = nic->dev;
	struct sk_buff *frag_list;
	void *tmp;

	/* Buffer-1 receives L3/L4 headers */
	((RxD3_t*)rxdp)->Buffer1_ptr =
#ifdef __VMKERNEL_MODULE__
			skb->headMA;
#else
			pci_map_single
			(nic->pdev, skb->data, l3l4hdr_size + 4,
			PCI_DMA_FROMDEVICE);
#endif

	/* skb_shinfo(skb)->frag_list will have L4 data payload */
	skb_shinfo(skb)->frag_list = dev_alloc_skb(dev->mtu + ALIGN_SIZE);
	if (skb_shinfo(skb)->frag_list == NULL) {
		DBG_PRINT(INFO_DBG, "%s: dev_alloc_skb failed\n ", dev->name);
		return -ENOMEM ;
	}
	frag_list = skb_shinfo(skb)->frag_list;
	frag_list->next = NULL;
	tmp = (void *)S2IO_ALIGN((unsigned long)frag_list->data, ALIGN_SIZE+1);
	frag_list->data = tmp;
	frag_list->tail = tmp;

	/* Buffer-2 receives L4 data payload */
	((RxD3_t*)rxdp)->Buffer2_ptr =
#ifdef __VMKERNEL_MODULE__
	 		frag_list->headMA;
#else
			pci_map_single(nic->pdev,
				frag_list->data, dev->mtu,
				PCI_DMA_FROMDEVICE);
#endif
	rxdp->Control_2 |= SET_BUFFER1_SIZE_3(l3l4hdr_size + 4);
	rxdp->Control_2 |= SET_BUFFER2_SIZE_3(dev->mtu);

	return SUCCESS;
}

/**
 *  fill_rx_buffers - Allocates the Rx side skbs
 *  @nic:  device private variable
 *  @ring_no: ring number
 *  Description:
 *  The function allocates Rx side skbs and puts the physical
 *  address of these buffers into the RxD buffer pointers, so that the NIC
 *  can DMA the received frame into these locations.
 *  The NIC supports 3 receive modes, viz
 *  1. single buffer,
 *  2. three buffer and
 *  3. Five buffer modes.
 *  Each mode defines how many fragments the received frame will be split
 *  up into by the NIC. The frame is split into L3 header, L4 Header,
 *  L4 payload in three buffer mode and in 5 buffer mode, L4 payload itself
 *  is split into 3 fragments. As of now only single buffer mode is
 *  supported.
 *   Return Value:
 *  SUCCESS on success or an appropriate -ve value on failure.
 */

int fill_rx_buffers(struct s2io_nic *nic, int ring_no)
{
	struct net_device *dev = nic->dev;
	struct sk_buff *skb;
	RxD_t *rxdp;
	int off, off1, size, block_no, block_no1;
	u32 alloc_tab = 0;
	u32 alloc_cnt;
	mac_info_t *mac_control;
	struct config_param *config;
	u64 tmp;
	buffAdd_t *ba;
	RxD_t *first_rxdp = NULL;

	mac_control = &nic->mac_control;
	config = &nic->config;

	alloc_cnt = mac_control->rings[ring_no].pkt_cnt -
	    atomic_read(&mac_control->rings[ring_no].rx_bufs_left);

	block_no1 = mac_control->rings[ring_no].rx_curr_get_info.block_index;
	off1 = mac_control->rings[ring_no].rx_curr_get_info.offset;
	while (alloc_tab < alloc_cnt) {
		block_no = mac_control->rings[ring_no].rx_curr_put_info.
		    block_index;
		off = mac_control->rings[ring_no].rx_curr_put_info.offset;

		rxdp = mac_control->rings[ring_no].
				rx_blocks[block_no].rxds[off].virt_addr;

		if ((block_no == block_no1) && (off == off1) &&
					(rxdp->Host_Control)) {
			DBG_PRINT(INTR_DBG, "%s: Get and Put",
				  dev->name);
			DBG_PRINT(INTR_DBG, " info equated\n");
			goto end;
		}
		if (off && (off == rxd_count[nic->rxd_mode])) {
			mac_control->rings[ring_no].rx_curr_put_info.
			    block_index++;
			if (mac_control->rings[ring_no].rx_curr_put_info.
			    block_index == mac_control->rings[ring_no].
					block_count)
				mac_control->rings[ring_no].rx_curr_put_info.
					block_index = 0;
			block_no = mac_control->rings[ring_no].
					rx_curr_put_info.block_index;
			if (off == rxd_count[nic->rxd_mode])
				off = 0;
			mac_control->rings[ring_no].rx_curr_put_info.
				offset = off;
			rxdp = mac_control->rings[ring_no].
				rx_blocks[block_no].block_virt_addr;
			DBG_PRINT(INTR_DBG, "%s: Next block at: %p\n",
				  dev->name, rxdp);
		}

		mac_control->rings[ring_no].put_pos =
		    (block_no * (rxd_count[nic->rxd_mode] + 1)) + off;

		if ((rxdp->Control_1 & RXD_OWN_XENA) &&
			((nic->rxd_mode >= RXD_MODE_3A) &&
				(rxdp->Control_2 & BIT(0)))) {
			mac_control->rings[ring_no].rx_curr_put_info.
					offset = off;
			goto end;
		}
		/* calculate size of skb based on ring mode */
		size = dev->mtu + HEADER_ETHERNET_II_802_3_SIZE +
				HEADER_802_2_SIZE + HEADER_SNAP_SIZE;
		if (nic->rxd_mode == RXD_MODE_1)
			size += NET_IP_ALIGN;
		else if (nic->rxd_mode == RXD_MODE_3B)
			size = dev->mtu + ALIGN_SIZE + BUF0_LEN + 4;
		else
			size = l3l4hdr_size + ALIGN_SIZE + BUF0_LEN + 4;

		/* allocate skb */
		skb = dev_alloc_skb(size);
		if(!skb) {
			DBG_PRINT(INFO_DBG, "%s: Out of ", dev->name);
			DBG_PRINT(INFO_DBG, "memory to allocate SKBs\n");
			if (first_rxdp) {
				wmb();
				first_rxdp->Control_1 |= RXD_OWN_XENA;
			}
			mac_control->stats_info->sw_stat.ring_mem_fail_cnt[ring_no]++;
			return -ENOMEM ;
		}
		if (nic->rxd_mode == RXD_MODE_1) {
			/* 1 buffer mode - normal operation mode */
			skb_reserve(skb, NET_IP_ALIGN);
			((RxD1_t*)rxdp)->Buffer0_ptr =
#ifdef __VMKERNEL_MODULE__
			 		skb->headMA;
#else
					pci_map_single
			    (nic->pdev, skb->data, size - NET_IP_ALIGN,
				PCI_DMA_FROMDEVICE);
#endif
			rxdp->Control_2 = SET_BUFFER0_SIZE_1(size - NET_IP_ALIGN);

		} else if (nic->rxd_mode >= RXD_MODE_3A) {
			/*
			 * 2 or 3 buffer mode -
			 * Both 2 buffer mode and 3 buffer mode provides 128
			 * byte aligned receive buffers.
			 *
			 * 3 buffer mode provides header separation where in
			 * skb->data will have L3/L4 headers where as
			 * skb_shinfo(skb)->frag_list will have the L4 data
			 * payload
			 */
			ba = &mac_control->rings[ring_no].ba[block_no][off];
			skb_reserve(skb, BUF0_LEN);
			tmp = (u64)(unsigned long) skb->data;
			tmp += ALIGN_SIZE;
			tmp &= ~ALIGN_SIZE;
			skb->data = (void *) (unsigned long)tmp;
			skb->tail = (void *) (unsigned long)tmp;

			((RxD3_t*)rxdp)->Buffer0_ptr =
	     		pci_map_single(nic->pdev, ba->ba_0, BUF0_LEN,
					   PCI_DMA_FROMDEVICE);

			rxdp->Control_2 = SET_BUFFER0_SIZE_3(BUF0_LEN);
			if (nic->rxd_mode == RXD_MODE_3B) {
				/* Two buffer mode */

				/*
				 * Buffer2 will have L3/L4 header plus
				 * L4 payload
				 */
				((RxD3_t*)rxdp)->Buffer2_ptr =
#ifdef __VMKERNEL_MODULE__
							skb->headMA;
#else
							pci_map_single
					(nic->pdev, skb->data, dev->mtu + 4,
						PCI_DMA_FROMDEVICE);
#endif
				 /* Buffer-1 will be dummy buffer not used */
				((RxD3_t*)rxdp)->Buffer1_ptr =
				pci_map_single(nic->pdev, ba->ba_1, BUF1_LEN,
					PCI_DMA_FROMDEVICE);

				rxdp->Control_2 |= SET_BUFFER1_SIZE_3(1);
				rxdp->Control_2 |= SET_BUFFER2_SIZE_3
								(dev->mtu + 4);
			} else {
				/* 3 buffer mode */
				if (fill_rxd_3buf(nic, rxdp, skb) == -ENOMEM) {
					dev_kfree_skb_irq(skb);
					if (first_rxdp) {
						wmb();
						first_rxdp->Control_1 |=
							RXD_OWN_XENA;
					}
					return -ENOMEM ;
				}
			}
			rxdp->Control_2 |= BIT(0);
		}
		rxdp->Host_Control = (unsigned long) (skb);
		if (alloc_tab & ((1 << rxsync_frequency) - 1))
			rxdp->Control_1 |= RXD_OWN_XENA;
		off++;
		if (off == (rxd_count[nic->rxd_mode] + 1))
			off = 0;
		mac_control->rings[ring_no].rx_curr_put_info.offset = off;

		rxdp->Control_2 |= SET_RXD_MARKER;
		if (!(alloc_tab & ((1 << rxsync_frequency) - 1))) {
			if (first_rxdp) {
				wmb();
				first_rxdp->Control_1 |= RXD_OWN_XENA;
			}
			first_rxdp = rxdp;
		}
		atomic_inc(&mac_control->rings[ring_no].rx_bufs_left);
		alloc_tab++;
	}

      end:
	/* Transfer ownership of first descriptor to adapter just before
	 * exiting. Before that, use memory barrier so that ownership
	 * and other fields are seen by adapter correctly.
	 */
	if (first_rxdp) {
		wmb();
		first_rxdp->Control_1 |= RXD_OWN_XENA;
	}

	return SUCCESS;
}

static void free_rxd_blk(struct s2io_nic *sp, int ring_no, int blk)
{
	struct net_device *dev = sp->dev;
	int j;
	struct sk_buff *skb;
	RxD_t *rxdp;
	mac_info_t *mac_control;
	buffAdd_t *ba;

	mac_control = &sp->mac_control;
	for (j = 0 ; j < rxd_count[sp->rxd_mode]; j++) {
		rxdp = mac_control->rings[ring_no].
                                rx_blocks[blk].rxds[j].virt_addr;
		skb = (struct sk_buff *)
			((unsigned long) rxdp->Host_Control);
		if (!skb) {
			continue;
		}
		if (sp->rxd_mode == RXD_MODE_1) {
			pci_unmap_single(sp->pdev, (dma_addr_t)
				 ((RxD1_t*)rxdp)->Buffer0_ptr,
				 dev->mtu +
				 HEADER_ETHERNET_II_802_3_SIZE
				 + HEADER_802_2_SIZE +
				 HEADER_SNAP_SIZE,
				 PCI_DMA_FROMDEVICE);
			memset(rxdp, 0, sizeof(RxD1_t));
		} else if(sp->rxd_mode == RXD_MODE_3B) {
			ba = &mac_control->rings[ring_no].
				ba[blk][j];
			pci_unmap_single(sp->pdev, (dma_addr_t)
				 ((RxD3_t*)rxdp)->Buffer0_ptr,
				 BUF0_LEN,
				 PCI_DMA_FROMDEVICE);
			pci_unmap_single(sp->pdev, (dma_addr_t)
				 ((RxD3_t*)rxdp)->Buffer1_ptr,
				 BUF1_LEN,
				 PCI_DMA_FROMDEVICE);
			pci_unmap_single(sp->pdev, (dma_addr_t)
				 ((RxD3_t*)rxdp)->Buffer2_ptr,
				 dev->mtu + 4,
				 PCI_DMA_FROMDEVICE);
			memset(rxdp, 0, sizeof(RxD3_t));
		} else {
			pci_unmap_single(sp->pdev, (dma_addr_t)
				((RxD3_t*)rxdp)->Buffer0_ptr, BUF0_LEN,
				PCI_DMA_FROMDEVICE);
			pci_unmap_single(sp->pdev, (dma_addr_t)
				((RxD3_t*)rxdp)->Buffer1_ptr,
				l3l4hdr_size + 4,
				PCI_DMA_FROMDEVICE);
			pci_unmap_single(sp->pdev, (dma_addr_t)
				((RxD3_t*)rxdp)->Buffer2_ptr, dev->mtu,
				PCI_DMA_FROMDEVICE);
			memset(rxdp, 0, sizeof(RxD3_t));
		}
		dev_kfree_skb(skb);
		atomic_dec(&sp->mac_control.rings[ring_no].rx_bufs_left);
	}
}
/**
 *  free_rx_buffers - Frees all Rx buffers
 *  @sp: device private variable.
 *  Description:
 *  This function will free all Rx buffers allocated by host.
 *  Return Value:
 *  NONE.
 */

static void free_rx_buffers(struct s2io_nic *sp)
{
	struct net_device *dev = sp->dev;
	int i, blk = 0, buf_cnt = 0;
	mac_info_t *mac_control;
	struct config_param *config;

	mac_control = &sp->mac_control;
	config = &sp->config;

	for (i = 0; i < config->rx_ring_num; i++) {
		for (blk = 0; blk < rx_ring_sz[i]; blk++)
			free_rxd_blk(sp,i,blk);

		mac_control->rings[i].rx_curr_put_info.block_index = 0;
		mac_control->rings[i].rx_curr_get_info.block_index = 0;
		mac_control->rings[i].rx_curr_put_info.offset = 0;
		mac_control->rings[i].rx_curr_get_info.offset = 0;
		atomic_set(&mac_control->rings[i].rx_bufs_left, 0);
		DBG_PRINT(INIT_DBG, "%s:Freed 0x%x Rx Buffers on ring%d\n",
			  dev->name, buf_cnt, i);
	}
}

/**
 * s2io_poll - Rx interrupt handler for NAPI support
 * @dev : pointer to the device structure.
 * @budget : The number of packets that were budgeted to be processed
 * during  one pass through the 'Poll" function.
 * Description:
 * Comes into picture only if NAPI support has been incorporated. It does
 * the same thing that rx_intr_handler does, but not in a interrupt context
 * also It will process only a given number of packets.
 * Return value:
 * 0 on success and 1 if there are No Rx packets to be processed.
 */

#if defined(CONFIG_S2IO_NAPI)
static int s2io_poll(struct net_device *dev, int *budget)
{
	nic_t *nic = dev->priv;
	int pkt_cnt = 0, org_pkts_to_process;
	mac_info_t *mac_control;
	struct config_param *config;
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	u64 val64 = 0xFFFFFFFFFFFFFFFFULL;
	int i;

	atomic_inc(&nic->isr_cnt);

	if (unlikely(atomic_read(&nic->card_state) == CARD_DOWN)){
		atomic_dec(&nic->isr_cnt);
		return 0;
	}

	mac_control = &nic->mac_control;
	config = &nic->config;

	nic->pkts_to_process = *budget;
	if (nic->pkts_to_process > dev->quota)
		nic->pkts_to_process = dev->quota;
	org_pkts_to_process = nic->pkts_to_process;

	writeq(val64, &bar0->rx_traffic_int);
	val64 = readl(&bar0->rx_traffic_int);

	for (i = 0; i < config->rx_ring_num; i++) {
		rx_intr_handler(&mac_control->rings[i]);
		pkt_cnt = org_pkts_to_process - nic->pkts_to_process;
		if (!nic->pkts_to_process) {
			/* Quota for the current iteration has been met */
			goto no_rx;
		}
	}
	if (!pkt_cnt)
		pkt_cnt = 1;

	dev->quota -= pkt_cnt;
	*budget -= pkt_cnt;
	netif_rx_complete(dev);

	for (i = 0; i < config->rx_ring_num; i++) {
		if (fill_rx_buffers(nic, i) == -ENOMEM) {
			DBG_PRINT(INFO_DBG, "%s:Out of memory", dev->name);
			DBG_PRINT(INFO_DBG, " in Rx Poll!!\n");
			break;
		}
	}
	/* Re enable the Rx interrupts. */
	writeq(0x0, &bar0->rx_traffic_mask);
	val64 = readl(&bar0->rx_traffic_mask);
	atomic_dec(&nic->isr_cnt);
	return 0;

no_rx:
	dev->quota -= pkt_cnt;
	*budget -= pkt_cnt;

	for (i = 0; i < config->rx_ring_num; i++) {
		if (fill_rx_buffers(nic, i) == -ENOMEM) {
			DBG_PRINT(INFO_DBG, "%s:Out of memory", dev->name);
			DBG_PRINT(INFO_DBG, " in Rx Poll!!\n");
			break;
		}
	}
	atomic_dec(&nic->isr_cnt);
	return 1;
}
#endif

/**
 *  rx_intr_handler - Rx interrupt handler
 *  @nic: device private variable.
 *  Description:
 *  If the interrupt is because of a received frame or if the
 *  receive ring contains fresh as yet un-processed frames,this function is
 *  called. It picks out the RxD at which place the last Rx processing had
 *  stopped and sends the skb to the OSM's Rx handler and then increments
 *  the offset.
 *  Return Value:
 *  NONE.
 */
static void rx_intr_handler(ring_info_t *ring_data)
{
	nic_t *nic = ring_data->nic;
	struct net_device *dev = (struct net_device *) ring_data->dev;
	int get_block, put_block, put_offset;
	rx_curr_get_info_t get_info, put_info;
	RxD_t *rxdp;
	struct sk_buff *skb;
#ifndef CONFIG_S2IO_NAPI
	int pkt_cnt = 0;
#endif
	int i;

	get_info = ring_data->rx_curr_get_info;
	get_block = get_info.block_index;
	put_info = ring_data->rx_curr_put_info;
	put_block = put_info.block_index;
	rxdp = ring_data->rx_blocks[get_block].rxds[get_info.offset].virt_addr;

	put_offset = ring_data->put_pos;

	while (RXD_IS_UP2DT(rxdp)) {
		/* If your are next to put index then it's FIFO full condition */
		if ((get_block == put_block) &&
		    (get_info.offset + 1) == put_info.offset) {
			DBG_PRINT(INTR_DBG, "%s: Ring Full\n",dev->name);
			break;
		}

		skb = (struct sk_buff *) ((unsigned long)rxdp->Host_Control);
		if (skb == NULL) {
			DBG_PRINT(INFO_DBG, "%s: The skb is ",
				  dev->name);
			DBG_PRINT(INFO_DBG, "Null in Rx Intr\n");
			return;
		}
		if (nic->rxd_mode == RXD_MODE_1) {
			pci_unmap_single(nic->pdev, (dma_addr_t)
				 ((RxD1_t*)rxdp)->Buffer0_ptr,
				 dev->mtu +
				 HEADER_ETHERNET_II_802_3_SIZE +
				 HEADER_802_2_SIZE +
				 HEADER_SNAP_SIZE,
				 PCI_DMA_FROMDEVICE);
		} else if (nic->rxd_mode == RXD_MODE_3B) {
			pci_unmap_single(nic->pdev, (dma_addr_t)
				 ((RxD3_t*)rxdp)->Buffer0_ptr,
				 BUF0_LEN, PCI_DMA_FROMDEVICE);
			pci_unmap_single(nic->pdev, (dma_addr_t)
				 ((RxD3_t*)rxdp)->Buffer1_ptr,
				 BUF1_LEN, PCI_DMA_FROMDEVICE);
			pci_unmap_single(nic->pdev, (dma_addr_t)
				 ((RxD3_t*)rxdp)->Buffer2_ptr,
				 dev->mtu + 4,
				 PCI_DMA_FROMDEVICE);
		} else {
			pci_unmap_single(nic->pdev, (dma_addr_t)
					 ((RxD3_t*)rxdp)->Buffer0_ptr, BUF0_LEN,
					 PCI_DMA_FROMDEVICE);
			pci_unmap_single(nic->pdev, (dma_addr_t)
					 ((RxD3_t*)rxdp)->Buffer1_ptr,
					 l3l4hdr_size + 4,
					 PCI_DMA_FROMDEVICE);
			pci_unmap_single(nic->pdev, (dma_addr_t)
					 ((RxD3_t*)rxdp)->Buffer2_ptr,
					 dev->mtu, PCI_DMA_FROMDEVICE);
		}
		prefetch(skb->data);
		rx_osm_handler(ring_data, rxdp);
		get_info.offset++;
		ring_data->rx_curr_get_info.offset = get_info.offset;
		rxdp = ring_data->rx_blocks[get_block].
				rxds[get_info.offset].virt_addr;
		if (get_info.offset == rxd_count[nic->rxd_mode]) {
			get_info.offset = 0;
			ring_data->rx_curr_get_info.offset = get_info.offset;
			get_block++;
			if (get_block == ring_data->block_count)
				get_block = 0;
			ring_data->rx_curr_get_info.block_index = get_block;
			rxdp = ring_data->rx_blocks[get_block].block_virt_addr;
		}

#ifdef CONFIG_S2IO_NAPI
		nic->pkts_to_process -= 1;
		if (!nic->pkts_to_process)
			break;
#else
		pkt_cnt++;
		if ((indicate_max_pkts) && (pkt_cnt > indicate_max_pkts))
			break;
#endif
	}

	if (nic->lro) {
		/* Clear all LRO sessions before exiting */
		for (i=0; i<MAX_LRO_SESSIONS; i++) {
			lro_t *lro = &nic->lro0_n[i];
			if (lro->in_use) {
				update_L3L4_header(nic, lro);
				queue_rx_frame(lro->parent);
				clear_lro_session(lro);
			}
		}
	}
}

/**
 *  tx_intr_handler - Transmit interrupt handler
 *  @nic : device private variable
 *  Description:
 *  If an interrupt was raised to indicate DMA complete of the
 *  Tx packet, this function is called. It identifies the last TxD
 *  whose buffer was freed and frees all skbs whose data have already
 *  DMA'ed into the NICs internal memory.
 *  Return Value:
 *  NONE
 */

static void tx_intr_handler(fifo_info_t *fifo_data)
{
	nic_t *nic = fifo_data->nic;
	struct net_device *dev = (struct net_device *) nic->dev;
	tx_curr_get_info_t get_info, put_info;
	struct sk_buff *skb;
	TxD_t *txdlp;
	struct sk_buff *head = NULL;
	struct sk_buff **temp = NULL;
	int pkt_cnt = 0;

	if (!spin_trylock(&fifo_data->tx_lock))
		return;

	get_info = fifo_data->tx_curr_get_info;
	put_info = fifo_data->tx_curr_put_info;
	txdlp = (TxD_t *) fifo_data->list_info[get_info.offset].list_virt_addr;
	while ((!(txdlp->Control_1 & TXD_LIST_OWN_XENA)) &&
	       (get_info.offset != put_info.offset) &&
	       (txdlp->Host_Control)) {
		/* Check for TxD errors */
		if (txdlp->Control_1 & TXD_T_CODE) {
			unsigned long long err;
			err = txdlp->Control_1 & TXD_T_CODE;
			if (err & 0x1) {
				nic->mac_control.stats_info->sw_stat.
						parity_err_cnt++;
			}
			if ((err >> 48) == 0xA) {
				DBG_PRINT(TX_DBG, "TxD returned due \
						to loss of link\n");
			}
			else {
				DBG_PRINT(INFO_DBG, "***TxD error %llx\n", err);
			}
		}
		skb = s2io_txdlget_skb(fifo_data, txdlp, get_info.offset);
		if (skb == NULL) {
			spin_unlock(&fifo_data->tx_lock);
			DBG_PRINT(INFO_DBG, "%s: Null skb ",__FUNCTION__);
			DBG_PRINT(INFO_DBG, "in Tx Free Intr\n");
			return;
		}

		temp = (struct sk_buff **)&skb->cb;
		*temp = head;
		head = skb;
		pkt_cnt++;

		/* Updating the statistics block */
		nic->stats.tx_bytes += skb->len;

		get_info.offset++;
		if (get_info.offset == get_info.fifo_len + 1)
			get_info.offset = 0;
		txdlp = (TxD_t *) fifo_data->list_info
		    [get_info.offset].list_virt_addr;
		fifo_data->tx_curr_get_info.offset =
			    get_info.offset;
	}

	if (netif_queue_stopped(dev) && pkt_cnt)
		netif_wake_queue(dev);
	spin_unlock(&fifo_data->tx_lock);

	while (head)
	{
		skb = head;
		temp = (struct sk_buff **)&skb->cb;
		head = *temp;
		*temp = NULL;
		dev_kfree_skb_irq(skb);
	}
}

/**
 *  s2io_mdio_write - Function to write in to MDIO registers
 *  @mmd_type : MMD type value (PMA/PMD/WIS/PCS/PHYXS)
 *  @addr     : address value
 *  @value    : data value
 *  @dev      : pointer to net_device structure
 *  Description:
 *  This function is used to write values to the MDIO registers
 *  NONE
 */
static void s2io_mdio_write(u32 mmd_type, u64 addr, u16 value, struct net_device *dev)
{
	u64 val64 = 0x0;
	nic_t *sp = dev->priv;
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *)sp->bar0;

	//address transaction
	val64 = val64 | MDIO_MMD_INDX_ADDR(addr)
			| MDIO_MMD_DEV_ADDR(mmd_type)
			| MDIO_MMS_PRT_ADDR(0x0);
	writeq(val64, &bar0->mdio_control);
	val64 = val64 | MDIO_CTRL_START_TRANS(0xE);
	writeq(val64, &bar0->mdio_control);
	udelay(100);

	//Data transaction
	val64 = 0x0;
	val64 = val64 | MDIO_MMD_INDX_ADDR(addr)
			| MDIO_MMD_DEV_ADDR(mmd_type)
			| MDIO_MMS_PRT_ADDR(0x0)
			| MDIO_MDIO_DATA(value)
			| MDIO_OP(MDIO_OP_WRITE_TRANS);
	writeq(val64, &bar0->mdio_control);
	val64 = val64 | MDIO_CTRL_START_TRANS(0xE);
	writeq(val64, &bar0->mdio_control);
	udelay(100);

	val64 = 0x0;
	val64 = val64 | MDIO_MMD_INDX_ADDR(addr)
	| MDIO_MMD_DEV_ADDR(mmd_type)
	| MDIO_MMS_PRT_ADDR(0x0)
	| MDIO_OP(MDIO_OP_READ_TRANS);
	writeq(val64, &bar0->mdio_control);
	val64 = val64 | MDIO_CTRL_START_TRANS(0xE);
	writeq(val64, &bar0->mdio_control);
	udelay(100);

}

/**
 *  s2io_mdio_read - Function to write in to MDIO registers
 *  @mmd_type : MMD type value (PMA/PMD/WIS/PCS/PHYXS)
 *  @addr     : address value
 *  @dev      : pointer to net_device structure
 *  Description:
 *  This function is used to read values to the MDIO registers
 *  NONE
 */
static u64 s2io_mdio_read(u32 mmd_type, u64 addr, struct net_device *dev)
{
	u64 val64 = 0x0;
	u64 rval64 = 0x0;
	nic_t *sp = dev->priv;
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *)sp->bar0;

	/* address transaction */
	val64 = val64 | MDIO_MMD_INDX_ADDR(addr)
			| MDIO_MMD_DEV_ADDR(mmd_type)
			| MDIO_MMS_PRT_ADDR(0x0);
	writeq(val64, &bar0->mdio_control);
	val64 = val64 | MDIO_CTRL_START_TRANS(0xE);
	writeq(val64, &bar0->mdio_control);
	udelay(100);

	/* Data transaction */
	val64 = 0x0;
	val64 = val64 | MDIO_MMD_INDX_ADDR(addr)
			| MDIO_MMD_DEV_ADDR(mmd_type)
			| MDIO_MMS_PRT_ADDR(0x0)
			| MDIO_OP(MDIO_OP_READ_TRANS);
	writeq(val64, &bar0->mdio_control);
	val64 = val64 | MDIO_CTRL_START_TRANS(0xE);
	writeq(val64, &bar0->mdio_control);
	udelay(100);

	/* Read the value from regs */
	rval64 = readq(&bar0->mdio_control);
	rval64 = rval64 & 0xFFFF0000;
	rval64 = rval64 >> 16;
	return rval64;
}
/**
 *  s2io_chk_xpak_counter - Function to check the status of the xpak counters
 *  @counter      : couter value to be updated
 *  @flag         : flag to indicate the status
 *  @type         : counter type
 *  Description:
 *  This function is to check the status of the xpak counters value
 *  NONE
 */

static void s2io_chk_xpak_counter(u64 *counter, u64 * regs_stat, u32 index, u16 flag, u16 type)
{
	u64 mask = 0x3;
	u64 val64;
	int i;
	for(i = 0; i <index; i++)
		mask = mask << 0x2;

	if(flag > 0)
	{
		*counter = *counter + 1;
		val64 = *regs_stat & mask;
		val64 = val64 >> (index * 0x2);
		val64 = val64 + 1;
		if(val64 == 3)
		{
			switch(type)
			{
			case 1:
				DBG_PRINT(ERR_DBG, "Take Xframe NIC out of "
					  "service. Excessive temperatures may "
					  "result in premature transceiver "
					  "failure \n");
			break;
			case 2:
				DBG_PRINT(ERR_DBG, "Take Xframe NIC out of "
					  "service Excessive bias currents may "
					  "indicate imminent laser diode "
					  "failure \n");
			break;
			case 3:
				DBG_PRINT(ERR_DBG, "Take Xframe NIC out of "
					  "service Excessive laser output "
					  "power may saturate far-end "
					  "receiver\n");
			break;
			default:
				DBG_PRINT(ERR_DBG, "Incorrect XPAK Alarm "
					  "type \n");
			}
			val64 = 0x0;
		}
		val64 = val64 << (index * 0x2);
		*regs_stat = (*regs_stat & (~mask)) | (val64);

	} else {
		*regs_stat = *regs_stat & (~mask);
	}
}

/**
 *  s2io_updt_xpak_counter - Function to update the xpak counters
 *  @dev         : pointer to net_device struct
 *  Description:
 *  This function is to upate the status of the xpak counters value
 *  NONE
 */
static void s2io_updt_xpak_counter(struct net_device *dev)
{
	u16 flag  = 0x0;
	u16 type  = 0x0;
	u16 val16 = 0x0;
	u64 val64 = 0x0;
	u64 addr  = 0x0;

	nic_t *sp = dev->priv;
	StatInfo_t *stat_info = sp->mac_control.stats_info;

	/* Check the communication with the MDIO slave */
	addr = 0x0000;
	val64 = 0x0;
	val64 = s2io_mdio_read(MDIO_MMD_PMA_DEV_ADDR, addr, dev);
	if((val64 == 0xFFFF) || (val64 == 0x0000))
	{
		DBG_PRINT(ERR_DBG, "ERR: MDIO slave access failed - "
			  "Returned %llx\n", (unsigned long long)val64);
		return;
	}

	/* Check for the expecte value of 2040 at PMA address 0x0000 */
	if(val64 != 0x2040)
	{
		DBG_PRINT(ERR_DBG, "Incorrect value at PMA address 0x0000 - ");
		DBG_PRINT(ERR_DBG, "Returned: %llx- Expected: 0x2040\n",
			  (unsigned long long)val64);
		return;
	}

	/* Loading the DOM register to MDIO register */
	addr = 0xA100;
	s2io_mdio_write(MDIO_MMD_PMA_DEV_ADDR, addr, val16, dev);
	val64 = s2io_mdio_read(MDIO_MMD_PMA_DEV_ADDR, addr, dev);

	/* Reading the Alarm flags */
	addr = 0xA070;
	val64 = 0x0;
	val64 = s2io_mdio_read(MDIO_MMD_PMA_DEV_ADDR, addr, dev);

	flag = CHECKBIT(val64, 0x7);
	type = 1;
	s2io_chk_xpak_counter(&stat_info->xpak_stat.alarm_transceiver_temp_high,
				&stat_info->xpak_stat.xpak_regs_stat,
				0x0, flag, type);

	if(CHECKBIT(val64, 0x6))
		stat_info->xpak_stat.alarm_transceiver_temp_low++;

	flag = CHECKBIT(val64, 0x3);
	type = 2;
	s2io_chk_xpak_counter(&stat_info->xpak_stat.alarm_laser_bias_current_high,
				&stat_info->xpak_stat.xpak_regs_stat,
				0x2, flag, type);

	if(CHECKBIT(val64, 0x2))
		stat_info->xpak_stat.alarm_laser_bias_current_low++;

	flag = CHECKBIT(val64, 0x1);
	type = 3;
	s2io_chk_xpak_counter(&stat_info->xpak_stat.alarm_laser_output_power_high,
				&stat_info->xpak_stat.xpak_regs_stat,
				0x4, flag, type);

	if(CHECKBIT(val64, 0x0))
		stat_info->xpak_stat.alarm_laser_output_power_low++;

	/* Reading the Warning flags */
	addr = 0xA074;
	val64 = 0x0;
	val64 = s2io_mdio_read(MDIO_MMD_PMA_DEV_ADDR, addr, dev);

	if(CHECKBIT(val64, 0x7))
		stat_info->xpak_stat.warn_transceiver_temp_high++;

	if(CHECKBIT(val64, 0x6))
		stat_info->xpak_stat.warn_transceiver_temp_low++;

	if(CHECKBIT(val64, 0x3))
		stat_info->xpak_stat.warn_laser_bias_current_high++;

	if(CHECKBIT(val64, 0x2))
		stat_info->xpak_stat.warn_laser_bias_current_low++;

	if(CHECKBIT(val64, 0x1))
		stat_info->xpak_stat.warn_laser_output_power_high++;

	if(CHECKBIT(val64, 0x0))
		stat_info->xpak_stat.warn_laser_output_power_low++;
}

/**
 *  alarm_intr_handler - Alarm Interrrupt handler
 *  @nic: device private variable
 *  Description: If the interrupt was neither because of Rx packet or Tx
 *  complete, this function is called. If the interrupt was to indicate
 *  a loss of link, the OSM link status handler is invoked for any other
 *  alarm interrupt the block that raised the interrupt is displayed
 *  and a H/W reset is issued.
 *  Return Value:
 *  NONE
*/

static void alarm_intr_handler(struct s2io_nic *nic)
{
	struct net_device *dev = (struct net_device *) nic->dev;
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	register u64 val64 = 0, err_reg = 0;
	u64 cnt;
	int i;

	memset(&nic->mac_control.stats_info->sw_stat.ring_full_cnt, 0,
		sizeof(nic->mac_control.stats_info->sw_stat.ring_full_cnt));

	/* Handling the XPAK counters update */
	if(nic->mac_control.stats_info->xpak_stat.xpak_timer_count < 72000)
	{
	/* waiting for an hour */
		nic->mac_control.stats_info->xpak_stat.xpak_timer_count++;
	} else {
		s2io_updt_xpak_counter(dev);
		/* reset the count to zero */
		nic->mac_control.stats_info->xpak_stat.xpak_timer_count = 0;
	}

	 /* Handling link status change error Intr */
	if (s2io_link_fault_indication(nic) == MAC_RMAC_ERR_TIMER) {
		err_reg = readq(&bar0->mac_rmac_err_reg);
		writeq(err_reg, &bar0->mac_rmac_err_reg);
		if (err_reg & RMAC_LINK_STATE_CHANGE_INT) {
#ifdef __VMKERNEL_MODULE__
			spin_lock(&nic->task_sync_lock);
			if (nic->device_close_flag != TRUE) {
				nic->link_task_scheduled = 1;
#endif
			schedule_work(&nic->set_link_task);
#ifdef __VMKERNEL_MODULE__
			}
			spin_unlock(&nic->task_sync_lock);
#endif
		}
	}

	/* Handling Ecc errors */
	val64 = readq(&bar0->mc_err_reg);
	writeq(val64, &bar0->mc_err_reg);
	if (val64 & (MC_ERR_REG_ECC_ALL_SNG | MC_ERR_REG_ECC_ALL_DBL)) {
		if (val64 & MC_ERR_REG_ECC_ALL_DBL) {
			nic->mac_control.stats_info->sw_stat.
				double_ecc_errs++;
			DBG_PRINT(INIT_DBG, "%s: Device indicates ",
				  dev->name);
			DBG_PRINT(INIT_DBG, "double ECC error!!\n");
			if (nic->device_type != XFRAME_II_DEVICE) {
				/* Reset XframeI only if critical error */
				if (val64 & (MC_ERR_REG_MIRI_ECC_DB_ERR_0 |
					     MC_ERR_REG_MIRI_ECC_DB_ERR_1)) {
					netif_stop_queue(dev);
#ifdef __VMKERNEL_MODULE__
					spin_lock(&nic->task_sync_lock);
					if (nic->device_close_flag != TRUE) {
						nic->rst_tmr_task_scheduled = 1;
#endif
					schedule_work(&nic->rst_timer_task);
#ifdef __VMKERNEL_MODULE__
					}
					spin_unlock(&nic->task_sync_lock);
#endif
					nic->mac_control.stats_info->sw_stat.
							soft_reset_cnt++;
				}
			}
		} else {
			nic->mac_control.stats_info->sw_stat.
				single_ecc_errs++;
		}
	}

	/* In case of a serious error, the device will be Reset. */
	val64 = readq(&bar0->serr_source);
	if (val64 & SERR_SOURCE_ANY) {
		nic->mac_control.stats_info->sw_stat.serious_err_cnt++;
		DBG_PRINT(ERR_DBG, "%s: Device indicates ", dev->name);
		DBG_PRINT(ERR_DBG, "serious error %llx!!\n",
			(unsigned long long)val64);
		netif_stop_queue(dev);
#ifdef __VMKERNEL_MODULE__
		spin_lock(&nic->task_sync_lock);
		if (nic->device_close_flag != TRUE) {
			nic->rst_tmr_task_scheduled = 1;
#endif
		schedule_work(&nic->rst_timer_task);
#ifdef __VMKERNEL_MODULE__
		}
		spin_unlock(&nic->task_sync_lock);
#endif
		nic->mac_control.stats_info->sw_stat.soft_reset_cnt++;
	}

	/*
	 * Also as mentioned in the latest Errata sheets if the PCC_FB_ECC
	 * Error occurs, the adapter will be recycled by disabling the
	 * adapter enable bit and enabling it again after the device
	 * becomes Quiescent.
	 */
	val64 = readq(&bar0->pcc_err_reg);
	writeq(val64, &bar0->pcc_err_reg);
	if (val64 & PCC_FB_ECC_DB_ERR) {
		u64 ac = readq(&bar0->adapter_control);
		ac &= ~(ADAPTER_CNTL_EN);
		writeq(ac, &bar0->adapter_control);
		ac = readq(&bar0->adapter_control);
#ifdef __VMKERNEL_MODULE__
		spin_lock(&nic->task_sync_lock);
		if (nic->device_close_flag != TRUE) {
			nic->link_task_scheduled = 1;
#endif
		schedule_work(&nic->set_link_task);
#ifdef __VMKERNEL_MODULE__
		}
		spin_unlock(&nic->task_sync_lock);
#endif
	}

	/* Check for data parity error */
	val64 = readq(&bar0->pic_int_status);
	if (val64 & PIC_INT_GPIO) {
		val64 = readq(&bar0->gpio_int_reg);
		if (val64 & GPIO_INT_REG_DP_ERR_INT) {
			nic->mac_control.stats_info->sw_stat.parity_err_cnt++;
#ifdef __VMKERNEL_MODULE__
			spin_lock(&nic->task_sync_lock);
			if (nic->device_close_flag != TRUE) {
				nic->rst_tmr_task_scheduled = 1;
#endif
			schedule_work(&nic->rst_timer_task);
#ifdef __VMKERNEL_MODULE__
			}
			spin_unlock(&nic->task_sync_lock);
#endif
			nic->mac_control.stats_info->sw_stat.soft_reset_cnt++;
		}
	}

	/* Check for ring full counter */
	if (nic->device_type & XFRAME_II_DEVICE) {
		val64 = readq(&bar0->ring_bump_counter1);
		for (i=0; i<4; i++) {
			cnt = ( val64 & vBIT(0xFFFF,(i*16),16));
			cnt >>= 64 - ((i+1)*16);
			nic->mac_control.stats_info->sw_stat.ring_full_cnt[i]
				+= cnt;
		}

		val64 = readq(&bar0->ring_bump_counter2);
		for (i=0; i<4; i++) {
			cnt = ( val64 & vBIT(0xFFFF,(i*16),16));
			cnt >>= 64 - ((i+1)*16);
			nic->mac_control.stats_info->sw_stat.ring_full_cnt[i+4]
				+= cnt;
		}
	}

	/* Other type of interrupts are not being handled now,  TODO */
}

/**
 *  wait_for_cmd_complete - waits for a command to complete.
 *  @sp : private member of the device structure, which is a pointer to the
 *  s2io_nic structure.
 *  Description: Function that waits for a command to Write into RMAC
 *  ADDR DATA registers to be completed and returns either success or
 *  error depending on whether the command was complete or not.
 *  Return value:
 *   SUCCESS on success and FAILURE on failure.
 */

static int wait_for_cmd_complete(void  __iomem *addr, u64 busy_bit,
				int bit_state)
{
	int ret = FAILURE, cnt = 0;
	u64 val64;
	int delay = 1;

	if ((bit_state != S2IO_BIT_RESET) && (bit_state != S2IO_BIT_SET))
		return FAILURE;

	do{
		val64 = readq(addr);
		if (bit_state == S2IO_BIT_RESET){
			if (!(val64 & busy_bit)) {
				ret = SUCCESS;
				break;
			}
		}
		else {
			if (val64 & busy_bit) {
				ret = SUCCESS;
				break;
			}
		}

		if(in_interrupt())
			mdelay(delay);
		else
			msleep(delay);

		if(++cnt >= 10)
			delay = 50;

	}while(cnt < 20);
	return ret;
}

/*
 * check_pci_device_id - Checks if the device id is supported
 * @id : device id
 * Description: Function to check if the pci device id is supported by the driver.
 * Return value: Actual device id if supported else PCI_ANY_ID
 */
static u16 check_pci_device_id(u16 id)
{
	switch (id) {
	case PCI_DEVICE_ID_HERC_WIN:
	case PCI_DEVICE_ID_HERC_UNI:
		return XFRAME_II_DEVICE;
	case PCI_DEVICE_ID_S2IO_UNI:
	case PCI_DEVICE_ID_S2IO_WIN:
		return XFRAME_I_DEVICE;
	default:
		return PCI_ANY_ID;
	}
}

/**
 *  s2io_reset - Resets the card.
 *  @sp : private member of the device structure.
 *  Description: Function to Reset the card. This function then also
 *  restores the previously saved PCI configuration space registers as
 *  the card reset also resets the configuration space.
 *  Return value:
 *  void.
 */

void s2io_reset(nic_t * sp)
{
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64;
	u16 subid, pci_cmd;
	int i;
	u16 val16;

	DBG_PRINT(INIT_DBG,"%s - Resetting XFrame card %s\n",
	                    __FUNCTION__, sp->dev->name);

	/* Back up  the PCI-X CMD reg, dont want to lose MMRBC, OST settings */
	pci_read_config_word(sp->pdev, PCIX_COMMAND_REGISTER, &(pci_cmd));

	val64 = SW_RESET_ALL;
	writeq(val64, &bar0->sw_reset);

	if (strstr(sp->product_name, "CX4")) {
		msleep(750);
	}

	msleep(250);

	for (i = 0; i < S2IO_MAX_PCI_CONFIG_SPACE_REINIT; i++) {

		/* Restore the PCI state saved during initialization. */
		pci_restore_state(sp->pdev, sp->config_space);
		pci_read_config_word(sp->pdev, 0x2, &val16);
		if (check_pci_device_id(val16) != (u16)PCI_ANY_ID){
			break;
		}
		msleep(200);
	}

	if (check_pci_device_id(val16) == (u16)PCI_ANY_ID){
		DBG_PRINT(ERR_DBG,"%s SW_Reset failed!\n", __FUNCTION__);
	}

	pci_write_config_word(sp->pdev, PCIX_COMMAND_REGISTER,
				     pci_cmd);

	s2io_init_pci(sp);

	/* Set swapper to enable I/O register access */
	s2io_set_swapper(sp);

#ifdef CONFIG_PCI_MSI
	/* Restore the MSIX table entries from local variables */
	restore_xmsi_data(sp);
#endif

	/* Clear certain PCI/PCI-X fields after reset */
	if (sp->device_type == XFRAME_II_DEVICE) {
		/* Clear "detected parity error" bit */
		pci_write_config_word(sp->pdev, PCI_STATUS, 0x8000);

		/* Clearing PCIX Ecc status register */
		pci_write_config_dword(sp->pdev, 0x68, 0x7C);

		/* Clearing PCI_STATUS error reflected here */
		writeq(BIT(62), &bar0->txpic_int_reg);
	}

	/* Reset device statistics maintained by OS */
	memset(&sp->stats, 0, sizeof (struct net_device_stats));

	/* SXE-002: Configure link and activity LED to turn it off */
	subid = sp->pdev->subsystem_device;
	if (((subid & 0xFF) >= 0x07) &&
	    (sp->device_type == XFRAME_I_DEVICE)) {
		val64 = readq(&bar0->gpio_control);
		val64 |= 0x0000800000000000ULL;
		writeq(val64, &bar0->gpio_control);
		val64 = 0x0411040400000000ULL;
		writeq(val64, (void __iomem *)bar0 + 0x2700);
	}

	/*
	 * Clear spurious ECC interrupts that would have occured on
	 * XFRAME II cards after reset.
	 */
	if (sp->device_type == XFRAME_II_DEVICE) {
		val64 = readq(&bar0->pcc_err_reg);
		writeq(val64, &bar0->pcc_err_reg);
	}

	sp->device_initialized = FALSE;
	sp->device_enabled_once = FALSE;
}

/**
 *  s2io_set_swapper - to set the swapper controle on the card
 *  @sp : private member of the device structure,
 *  pointer to the s2io_nic structure.
 *  Description: Function to set the swapper control on the card
 *  correctly depending on the 'endianness' of the system.
 *  Return value:
 *  SUCCESS on success and FAILURE on failure.
 */

int s2io_set_swapper(nic_t * sp)
{
	struct net_device *dev = sp->dev;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64, valt, valr, sw_wr, rth_wr;

	/*
	 * Set proper endian settings and verify the same by reading
	 * the PIF Feed-back register.
	 */

	val64 = readq(&bar0->pif_rd_swapper_fb);
	if (val64 != 0x0123456789ABCDEFULL) {
		int i = 0;
		u64 value[] = { 0xC30000C3C30000C3ULL,   /* FE=1, SE=1 */
				0x8100008181000081ULL,  /* FE=1, SE=0 */
				0x4200004242000042ULL,  /* FE=0, SE=1 */
				0};                     /* FE=0, SE=0 */

		while(i<4) {
			writeq(value[i], &bar0->swapper_ctrl);
			val64 = readq(&bar0->pif_rd_swapper_fb);
			if (val64 == 0x0123456789ABCDEFULL)
				break;
			i++;
		}
		if (i == 4) {
			DBG_PRINT(ERR_DBG, "%s: Endian settings are wrong, ",
				dev->name);
			DBG_PRINT(ERR_DBG, "feedback read %llx\n",
				(unsigned long long) val64);
			return FAILURE;
		}
		valr = value[i];
	} else {
		valr = readq(&bar0->swapper_ctrl);
	}

	valt = 0x0123456789ABCDEFULL;
	writeq(valt, &bar0->xmsi_address);
	val64 = readq(&bar0->xmsi_address);

	if(val64 != valt) {
		int i = 0;
		u64 value[] = { 0x00C3C30000C3C300ULL,  /* FE=1, SE=1 */
				0x0081810000818100ULL,  /* FE=1, SE=0 */
				0x0042420000424200ULL,  /* FE=0, SE=1 */
				0};                     /* FE=0, SE=0 */

		while(i<4) {
			writeq((value[i] | valr), &bar0->swapper_ctrl);
			writeq(valt, &bar0->xmsi_address);
			val64 = readq(&bar0->xmsi_address);
			if(val64 == valt)
				break;
			i++;
		}
		if(i == 4) {
			unsigned long long x = val64;
			DBG_PRINT(ERR_DBG, "Write failed, Xmsi_addr ");
			DBG_PRINT(ERR_DBG, "reads:0x%llx\n", x);
			return FAILURE;
		}
	}
	val64 = readq(&bar0->swapper_ctrl);
	val64 &= 0xFFFF000000000000ULL;
        sw_wr = (val64 & vBIT(3, 8, 2));
        rth_wr = (sw_wr >> 2);
        val64 |= rth_wr;

#ifdef  __BIG_ENDIAN
	/*
	 * The device by default set to a big endian format, so a
	 * big endian driver need not set anything.
	 */
	val64 |= (SWAPPER_CTRL_TXP_FE |
		 SWAPPER_CTRL_TXP_SE |
		 SWAPPER_CTRL_TXD_R_FE |
		 SWAPPER_CTRL_TXD_W_FE |
		 SWAPPER_CTRL_TXF_R_FE |
		 SWAPPER_CTRL_RXD_R_FE |
		 SWAPPER_CTRL_RXD_W_FE |
		 SWAPPER_CTRL_RXF_W_FE |
		 SWAPPER_CTRL_XMSI_FE |
		 SWAPPER_CTRL_STATS_FE | SWAPPER_CTRL_STATS_SE);
	if (sp->intr_type == INTA)
		val64 |= SWAPPER_CTRL_XMSI_SE;
	writeq(val64, &bar0->swapper_ctrl);
#else
	/*
	 * Initially we enable all bits to make it accessible by the
	 * driver, then we selectively enable only those bits that
	 * we want to set.
	 */
	val64 |= (SWAPPER_CTRL_TXP_FE |
		 SWAPPER_CTRL_TXP_SE |
		 SWAPPER_CTRL_TXD_R_FE |
		 SWAPPER_CTRL_TXD_R_SE |
		 SWAPPER_CTRL_TXD_W_FE |
		 SWAPPER_CTRL_TXD_W_SE |
		 SWAPPER_CTRL_TXF_R_FE |
		 SWAPPER_CTRL_RXD_R_FE |
		 SWAPPER_CTRL_RXD_R_SE |
		 SWAPPER_CTRL_RXD_W_FE |
		 SWAPPER_CTRL_RXD_W_SE |
		 SWAPPER_CTRL_RXF_W_FE |
		 SWAPPER_CTRL_XMSI_FE |
		 SWAPPER_CTRL_STATS_FE | SWAPPER_CTRL_STATS_SE);
	if (sp->intr_type == INTA)
		val64 |= SWAPPER_CTRL_XMSI_SE;
	writeq(val64, &bar0->swapper_ctrl);
#endif
	val64 = readq(&bar0->swapper_ctrl);

	/*
	 * Verifying if endian settings are accurate by reading a
	 * feedback register.
	 */
	val64 = readq(&bar0->pif_rd_swapper_fb);
	if (val64 != 0x0123456789ABCDEFULL) {
		/* Endian settings are incorrect, calls for another dekko. */
		DBG_PRINT(ERR_DBG, "%s: Endian settings are wrong, ",
			  dev->name);
		DBG_PRINT(ERR_DBG, "feedback read %llx\n",
			  (unsigned long long) val64);
		return FAILURE;
	}

	return SUCCESS;
}

#ifdef CONFIG_PCI_MSI
int wait_for_msix_trans(nic_t *nic, int i)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
	u64 val64;
	int ret = 0, cnt = 0;

	do {
		val64 = readq(&bar0->xmsi_access);
		if (!(val64 & BIT(15)))
			break;
		mdelay(1);
		cnt++;
	} while(cnt < 5);
	if (cnt == 5) {
		DBG_PRINT(ERR_DBG, "XMSI # %d Access failed\n", i);
		ret = 1;
	}

	return ret;
}

void restore_xmsi_data(nic_t *nic)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
	u64 val64;
	int i;

	if (nic->device_type & XFRAME_I_DEVICE) {
		return;
	}

	for (i=0; i< MAX_REQUESTED_MSI_X; i++) {
		writeq(nic->msix_info[i].addr, &bar0->xmsi_address);
		writeq(nic->msix_info[i].data, &bar0->xmsi_data);
		val64 = (BIT(7) | BIT(15) | vBIT(i, 26, 6));
		writeq(val64, &bar0->xmsi_access);
		if (wait_for_msix_trans(nic, i)) {
			DBG_PRINT(ERR_DBG, "failed in %s\n", __FUNCTION__);
			continue;
		}
	}
}

void store_xmsi_data(nic_t *nic)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
	u64 val64, addr, data;
	int i;

	if (nic->device_type & XFRAME_I_DEVICE) {
		return;
	}

	/* Store and display */
	for (i=0; i< MAX_REQUESTED_MSI_X; i++) {
		val64 = (BIT(15) | vBIT(i, 26, 6));
		writeq(val64, &bar0->xmsi_access);
		if (wait_for_msix_trans(nic, i)) {
			DBG_PRINT(ERR_DBG, "failed in %s\n", __FUNCTION__);
			continue;
		}
		addr = readq(&bar0->xmsi_address);
		data = readq(&bar0->xmsi_data);
		if (addr && data) {
			nic->msix_info[i].addr = addr;
			nic->msix_info[i].data = data;
		}
	}
}

int s2io_enable_msi_x(nic_t *nic)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) nic->bar0;
	u64 tx_mat, rx_mat;
	u16 msi_control; /* Temp variable */
	int ret, i, j, msix_indx = 1;
	struct config_param *config = &(nic->config);

	nic->entries = kmalloc(MAX_REQUESTED_MSI_X * sizeof(struct msix_entry),
			       GFP_KERNEL);
	if (nic->entries == NULL) {
		DBG_PRINT(INFO_DBG, "%s: Memory allocation failed\n",
			  __FUNCTION__);
		return -ENOMEM;
	}
	memset(nic->entries, 0, MAX_REQUESTED_MSI_X * sizeof(struct msix_entry));

	nic->s2io_entries =
		kmalloc(MAX_REQUESTED_MSI_X * sizeof(struct s2io_msix_entry),
				   GFP_KERNEL);
	if (nic->s2io_entries == NULL) {
		DBG_PRINT(INFO_DBG, "%s: Memory allocation failed\n",
			  __FUNCTION__);
		kfree(nic->entries);
		return -ENOMEM;
	}
	memset(nic->s2io_entries, 0,
	       MAX_REQUESTED_MSI_X * sizeof(struct s2io_msix_entry));

	for (i=0; i< MAX_REQUESTED_MSI_X; i++) {
		nic->entries[i].entry = i;
		nic->s2io_entries[i].entry = i;
		nic->s2io_entries[i].arg = NULL;
		nic->s2io_entries[i].in_use = 0;
	}

	tx_mat = readq(&bar0->tx_mat0_n[0]);
	for (i=0; i<nic->config.tx_fifo_num; i++, msix_indx++) {
		tx_mat |= TX_MAT_SET(i, msix_indx);
		nic->s2io_entries[msix_indx].arg = &nic->mac_control.fifos[i];
		nic->s2io_entries[msix_indx].type = MSIX_FIFO_TYPE;
		nic->s2io_entries[msix_indx].in_use = MSIX_FLG;
	}
	writeq(tx_mat, &bar0->tx_mat0_n[0]);

	if (!nic->config.bimodal) {
		rx_mat = readq(&bar0->rx_mat);
		for (j=0; j<nic->config.rx_ring_num; j++, msix_indx++) {
			rx_mat |= RX_MAT_SET(j, msix_indx);
			nic->s2io_entries[msix_indx].arg =
				&nic->mac_control.rings[j];
			nic->s2io_entries[msix_indx].type = MSIX_RING_TYPE;
			nic->s2io_entries[msix_indx].in_use = MSIX_FLG;
		}
		writeq(rx_mat, &bar0->rx_mat);
	} else {
		tx_mat = readq(&bar0->tx_mat0_n[7]);
		for (j=0; j<nic->config.rx_ring_num; j++, msix_indx++) {
			tx_mat |= TX_MAT_SET(i, msix_indx);
			nic->s2io_entries[msix_indx].arg =
				&nic->mac_control.rings[j];
			nic->s2io_entries[msix_indx].type = MSIX_RING_TYPE;
			nic->s2io_entries[msix_indx].in_use = MSIX_FLG;
		}
		writeq(tx_mat, &bar0->tx_mat0_n[7]);
	}

	ret = pci_enable_msix(nic->pdev, nic->entries, MAX_REQUESTED_MSI_X);
	/* We fail init if error or we get less vectors than min required */
	if (ret >= (config->tx_fifo_num + config->rx_ring_num + 1))
		ret = pci_enable_msix(nic->pdev, nic->entries, ret);

	if (ret) {
		DBG_PRINT(ERR_DBG, "%s: Enabling MSIX failed\n", nic->dev->name);
		kfree(nic->entries);
		kfree(nic->s2io_entries);
		nic->entries = NULL;
		nic->s2io_entries = NULL;
		return -ENOMEM;
	}

	/*
	 * To enable MSI-X, MSI also needs to be enabled, due to a bug
	 * in the herc NIC. (Temp change, needs to be removed later)
	 */
	pci_read_config_word(nic->pdev, 0x42, &msi_control);
	msi_control |= 0x1; /* Enable MSI */
	pci_write_config_word(nic->pdev, 0x42, msi_control);

	return 0;
}
#endif

/* ********************************************************* *
 * Functions defined below concern the OS part of the driver *
 * ********************************************************* */

/**
 *  s2io_open - open entry point of the driver
 *  @dev : pointer to the device structure.
 *  Description:
 *  This function is the open entry point of the driver. It mainly calls a
 *  function to allocate Rx buffers and inserts them into the buffer
 *  descriptors and then enables the Rx part of the NIC.
 *  Return value:
 *  0 on success and an appropriate (-)ve integer as defined in errno.h
 *   file on failure.
 */

int s2io_open(struct net_device *dev)
{
	nic_t *sp = dev->priv;
	int err = 0;
	u16 interruptible = 0;
#ifdef __VMKERNEL_MODULE__
	unsigned long flags;
#endif

	/*
	 * Make sure you have link off by default every time
	 * Nic is initialized
	 */
	netif_carrier_off(dev);
	sp->last_link_state = 0;

	/* Initialize H/W and enable interrupts */
	err = s2io_card_up(sp);
	if (err) {
		DBG_PRINT(ERR_DBG, "%s: H/W initialization failed\n",
			  dev->name);
		if (err == -ENODEV)
			goto hw_enable_failed;
		else
			goto hw_init_failed;
	}

#ifdef CONFIG_PCI_MSI
	/* Store the values of the MSIX table in the nic_t structure */
	store_xmsi_data(sp);

	/* After proper initialization of H/W, register ISR */
	if (sp->intr_type == MSI_X) {
		int i;

		for (i=1; (sp->s2io_entries[i].in_use == MSIX_FLG); i++) {
			if (sp->s2io_entries[i].type == MSIX_FIFO_TYPE) {
				sprintf(sp->desc[i], "%s:MSI-X-%d-TX",
					dev->name, i);
				err = request_irq(sp->entries[i].vector,
					  s2io_msix_fifo_handle, 0, sp->desc[i],
						  sp->s2io_entries[i].arg);
				DBG_PRINT(ERR_DBG, "%s @ 0x%llx\n", sp->desc[i],
				(unsigned long long)sp->msix_info[i].addr);
			} else {
				sprintf(sp->desc[i], "%s:MSI-X-%d-RX",
					dev->name, i);
				err = request_irq(sp->entries[i].vector,
					  s2io_msix_ring_handle, 0, sp->desc[i],
						  sp->s2io_entries[i].arg);
				DBG_PRINT(ERR_DBG, "%s @ 0x%llx\n", sp->desc[i],
				(unsigned long long)sp->msix_info[i].addr);
			}
			if (err) {
				DBG_PRINT(ERR_DBG,"%s:MSI-X-%d registration "
					  "failed\n", dev->name, i);
				DBG_PRINT(ERR_DBG, "Returned: %d\n", err);
				goto isr_registration_failed;
			}
			sp->s2io_entries[i].in_use = MSIX_REGISTERED_SUCCESS;
		}
	}
#endif
	if (sp->intr_type == INTA) {
		err = request_irq((int) sp->pdev->irq, s2io_isr, SA_SHIRQ,
			  sp->name, dev);
		if (err) {
			DBG_PRINT(ERR_DBG, "%s: ISR registration failed\n",
				  dev->name);
			goto isr_registration_failed;
		}
	}

	if (s2io_set_mac_addr(dev, dev->dev_addr) == FAILURE) {
		DBG_PRINT(ERR_DBG, "Set Mac Address Failed\n");
		err = -ENODEV;
		goto setting_mac_address_failed;
	}

	if (sp->promisc_flg)
		sp->promisc_flg = 0;
	if (sp->m_cast_flg) {
		sp->m_cast_flg = 0;
		sp->all_multi_pos= 0;
	}
	s2io_set_multicast(dev);

#ifdef __VMKERNEL_MODULE__
	spin_lock_irqsave(&sp->task_sync_lock, flags);
	sp->device_close_flag = FALSE;
	spin_unlock_irqrestore(&sp->task_sync_lock, flags);
#endif

	/*  Enable select interrupts */
	if (sp->intr_type != INTA)
		en_dis_able_nic_intrs(sp, ENA_ALL_INTRS, DISABLE_INTRS);
	else {
		interruptible = TX_TRAFFIC_INTR | RX_TRAFFIC_INTR;
		interruptible |= TX_PIC_INTR;
		en_dis_able_nic_intrs(sp, interruptible, ENABLE_INTRS);
	}

	netif_start_queue(dev);
	return 0;

setting_mac_address_failed:
	if (sp->intr_type != MSI_X)
		free_irq(sp->pdev->irq, dev);
isr_registration_failed:
	del_timer_sync(&sp->alarm_timer);
#ifdef CONFIG_PCI_MSI
	if (sp->intr_type == MSI_X) {
		int i;
		u16 msi_control; /* Temp variable */

		for (i=1; (sp->s2io_entries[i].in_use ==
				MSIX_REGISTERED_SUCCESS); i++) {
			int vector = sp->entries[i].vector;
			void *arg = sp->s2io_entries[i].arg;

			free_irq(vector, arg);
		}
		pci_disable_msix(sp->pdev);

#ifdef __VMKERNEL_MODULE__
		/* Temp */
		pci_read_config_word(sp->pdev, 0x92, &msi_control);
		msi_control &= ~0x8000; /* Disable MSI-X */
		pci_write_config_word(sp->pdev, 0x92, msi_control);
#endif

		pci_read_config_word(sp->pdev, 0x42, &msi_control);
		msi_control &= 0xFFFE; /* Disable MSI */
		pci_write_config_word(sp->pdev, 0x42, msi_control);
	}
#endif
hw_enable_failed:
	s2io_reset(sp);
hw_init_failed:
	free_rx_buffers(sp);
#ifdef CONFIG_PCI_MSI
	if (sp->intr_type == MSI_X) {
		if (sp->entries)
			kfree(sp->entries);
		if (sp->s2io_entries)
			kfree(sp->s2io_entries);
	}
#endif

#ifdef __VMKERNEL_MODULE__
	spin_lock_irqsave(&sp->task_sync_lock, flags);
	sp->device_close_flag = TRUE;
	spin_unlock_irqrestore(&sp->task_sync_lock, flags);
#endif

	return err;
}

/**
 *  s2io_close -close entry point of the driver
 *  @dev : device pointer.
 *  Description:
 *  This is the stop entry point of the driver. It needs to undo exactly
 *  whatever was done by the open entry point,thus it's usually referred to
 *  as the close function.Among other things this function mainly stops the
 *  Rx side of the NIC and frees all the Rx buffers in the Rx rings.
 *  Return value:
 *  0 on success and an appropriate (-)ve integer as defined in errno.h
 *  file on failure.
 */

int s2io_close(struct net_device *dev)
{
	nic_t *sp = dev->priv;
	unsigned long flags;

	/* Return if the device is already closed *
	*  Can happen when change_mtu had failed. *
	*/	
	if(sp->device_close_flag == TRUE)
		return 0;

#ifndef __VMKERNEL_MODULE__
	flush_scheduled_work();
#else
	/*
	 * Unfortunately vmkernel doesn't support flush_scheduled_work.
	 * So we doing a manual wait here till any tasks scheduled earlier
	 * has been completed. FIXME in the future.
	 */
	spin_lock_irqsave(&sp->task_sync_lock, flags);
	sp->device_close_flag = TRUE;	/* Device is shut down. */
	spin_unlock_irqrestore(&sp->task_sync_lock, flags);

	while (sp->rst_tmr_task_scheduled)
		schedule();
	while (sp->link_task_scheduled)
		schedule();
#endif
	netif_stop_queue(dev);
	/* Reset card, free Tx and Rx buffers. */
	s2io_card_down(sp, 1);

#ifndef __VMKERNEL_MODULE__
	sp->device_close_flag = TRUE;	/* Device is shut down. */
#endif
	return 0;
}


/**
 *  s2io_xmit - Tx entry point of te driver
 *  @skb : the socket buffer containing the Tx data.
 *  @dev : device pointer.
 *  Description :
 *  This function is the Tx entry point of the driver. S2IO NIC supports
 *  certain protocol assist features on Tx side, namely  CSO, S/G, LSO.
 *  NOTE: when device cant queue the pkt,just the trans_start variable will
 *  not be upadted.
 *  Return value:
 *  0 on success & 1 on failure.
 */

int s2io_xmit(struct sk_buff *skb, struct net_device *dev)
{
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u16 frg_cnt, frg_len, i, queue, queue_len, put_off, get_off;
        u64 val64;
	TxD_t *txdp;
	TxFIFO_element_t __iomem *tx_fifo;
	unsigned long flags;
#ifdef NETIF_F_TSO
	int mss;
#endif
	u16 vlan_tag = 0;
	int vlan_priority = 0;
	mac_info_t *mac_control;
	struct config_param *config;
	struct fifo_info *fifo;
#ifdef CONFIG_AMD_FIX
	unsigned long buf_adr = 0;
	u8 *tmp_buf = NULL;
	int tmp_len = 0;
	int delta=0;
#endif

	mac_control = &sp->mac_control;
	config = &sp->config;

	DBG_PRINT(TX_DBG, "%s: In Neterion Tx routine\n", dev->name);

	queue = 0;
	fifo = &mac_control->fifos[queue];
	/* Get Fifo number to Transmit based on vlan priority */
	if (sp->vlgrp && vlan_tx_tag_present(skb)) {
		vlan_tag = vlan_tx_tag_get(skb);
		vlan_priority = vlan_tag >> 13;
		queue = config->fifo_mapping[vlan_priority];
	}

	spin_lock_irqsave(&fifo->tx_lock, flags);

	if (atomic_read(&sp->card_state) == CARD_DOWN) {
		DBG_PRINT(TX_DBG, "%s: Card going down for reset\n",
			  dev->name);
		spin_unlock_irqrestore(&fifo->tx_lock, flags);
		dev_kfree_skb(skb);
		return 0;
	}

	put_off = (u16) mac_control->fifos[queue].tx_curr_put_info.offset;
	get_off = (u16) mac_control->fifos[queue].tx_curr_get_info.offset;
	txdp = (TxD_t *) mac_control->fifos[queue].list_info[put_off].
		list_virt_addr;

	queue_len = mac_control->fifos[queue].tx_curr_put_info.fifo_len + 1;
	/* Avoid "put" pointer going beyond "get" pointer */
	if (txdp->Host_Control ||
		   ((put_off+1) == queue_len ? 0 : (put_off+1)) == get_off) {
		DBG_PRINT(TX_DBG, "Error in xmit, No free TXDs.\n");
		netif_stop_queue(dev);
		dev_kfree_skb(skb);
		spin_unlock_irqrestore(&fifo->tx_lock, flags);
		return 0;
	}

	/* A buffer with no data will be dropped */
	if (!skb->len) {
		DBG_PRINT(TX_DBG, "%s:Buffer has no data..\n", dev->name);
		dev_kfree_skb(skb);
		spin_unlock_irqrestore(&fifo->tx_lock, flags);
		return 0;
	}

	/* only txd0 Buffer pointer, Buffer length changes depending upon
	 * whether amd_fix is enabled, or UFO is enabled. Rest all bits of
	 * txd0 will remain same so lets set them in the beginning itself
	 * before we start processing the txds so that we can reduce number of
	 * if conditions
	 */
#ifdef NETIF_F_TSO
	mss = skb_shinfo(skb)->tso_size;
	if (mss) {
		txdp->Control_1 |= TXD_TCP_LSO_EN;
		txdp->Control_1 |= TXD_TCP_LSO_MSS(mss);
	}
#endif
	if (skb->ip_summed == CHECKSUM_HW) {
		txdp->Control_2 |=
		    (TXD_TX_CKO_IPV4_EN | TXD_TX_CKO_TCP_EN |
		     TXD_TX_CKO_UDP_EN);
	}
	txdp->Control_1 |= TXD_GATHER_CODE_FIRST;
	txdp->Control_1 |= TXD_LIST_OWN_XENA;
	txdp->Control_2 |= config->tx_intr_type;
	if (sp->vlgrp && vlan_tx_tag_present(skb)) {
		txdp->Control_2 |= TXD_VLAN_ENABLE;
		txdp->Control_2 |= TXD_VLAN_TAG(vlan_tag);
	}
	frg_len = skb->len - skb->data_len;

#ifdef NETIF_F_UFO
	if (skb_shinfo(skb)->ufo_size) {
		int ufo_size;

		ufo_size = skb_shinfo(skb)->ufo_size;
		ufo_size &= ~7;
		txdp->Control_1 |= TXD_UFO_EN;
		txdp->Control_1 |= TXD_UFO_MSS(ufo_size);
		txdp->Control_1 |= TXD_BUFFER0_SIZE(8);
#ifdef __BIG_ENDIAN
		sp->ufo_in_band_v[put_off] =
				(u64)skb_shinfo(skb)->ip6_frag_id;
#else
		sp->ufo_in_band_v[put_off] =
				(u64)skb_shinfo(skb)->ip6_frag_id << 32;
#endif
		txdp->Host_Control = (unsigned long)sp->ufo_in_band_v;
		txdp->Buffer_Pointer =
				pci_map_single(sp->pdev,
					sp->ufo_in_band_v,
					sizeof(u64), PCI_DMA_TODEVICE);
				txdp++;
	}
#endif

#ifdef CONFIG_AMD_FIX
	buf_adr = (unsigned long)skb->data;
	if (skb_shinfo(skb)->nr_frags) {
		skb_frag_t *frag = &skb_shinfo(skb)->frags[0];

		if (frag->page_offset & (AMD_BUF_ALIGN - 1)) {
			delta =  (AMD_BUF_ALIGN) -
				(frag->page_offset & (AMD_BUF_ALIGN - 1));
			delta = (delta > frag->size) ? frag->size : delta;
			__pskb_pull_tail(skb, delta);
		}
	}

	if ((buf_adr = (buf_adr & (AMD_BUF_ALIGN - 1)))) {
		tmp_len = (AMD_BUF_ALIGN - buf_adr);
		tmp_buf = (sp->amd_tmp[queue] + (AMD_BUF_ALIGN * put_off));
		memcpy(tmp_buf, skb->data, tmp_len);
		skb_pull(skb, tmp_len);
		DBG_PRINT(TX_DBG, "Tmp_len: %d ", tmp_len);
		DBG_PRINT(TX_DBG, "New skb data Ptr: 0x%p ", skb->data);
		txdp->Host_Control = (unsigned long) tmp_buf;
		txdp->Buffer_Pointer =
			pci_map_single
			(sp->pdev, tmp_buf, AMD_BUF_ALIGN, PCI_DMA_TODEVICE);
		txdp->Control_1 |= TXD_BUFFER0_SIZE(tmp_len);
		TXDS_SET_UFO_EN(txdp, skb);
		txdp++;
		frg_len -= tmp_len;
	}
#endif
	txdp->Buffer_Pointer =
#ifdef __VMKERNEL_MODULE__
		skb->headMA;
#else
		pci_map_single
	    		(sp->pdev, skb->data, frg_len, PCI_DMA_TODEVICE);
#endif

	txdp->Host_Control = (unsigned long) skb;
	txdp->Control_1 |= TXD_BUFFER0_SIZE(frg_len);
	TXDS_SET_UFO_EN(txdp, skb);

	/* For fragmented SKB. */
	frg_cnt = skb_shinfo(skb)->nr_frags;
	for (i = 0; i < frg_cnt; i++) {
		skb_frag_t *frag = &skb_shinfo(skb)->frags[i];
		/* A '0' length fragment will be ignored */
		if (!frag->size)
			continue;
		txdp++;
		txdp->Buffer_Pointer =
#ifdef __VMKERNEL_MODULE__
				page_to_phys(frag->page) +
						frag->page_offset;
#else
				(u64) pci_map_page
		    (sp->pdev, frag->page, frag->page_offset,
		     frag->size, PCI_DMA_TODEVICE);
#endif
		txdp->Control_1 = TXD_BUFFER0_SIZE(frag->size);
		TXDS_SET_UFO_EN(txdp, skb);
	}
	txdp->Control_1 |= TXD_GATHER_CODE_LAST;

#ifdef NETIF_F_UFO
	if (skb_shinfo(skb)->ufo_size)
		frg_cnt++; /* as Txd0 was used for inband header */
#endif


	tx_fifo = mac_control->tx_FIFO_start[queue];
	val64 = mac_control->fifos[queue].list_info[put_off].list_phy_addr;
	writeq(val64, &tx_fifo->TxDL_Pointer);

        wmb();

	val64 = (TX_FIFO_LAST_TXD_NUM(frg_cnt) | TX_FIFO_FIRST_LIST |
		 TX_FIFO_LAST_LIST);
#ifdef CONFIG_AMD_FIX
	if (tmp_len)
		val64 = (TX_FIFO_LAST_TXD_NUM(frg_cnt + 1) |
			 TX_FIFO_FIRST_LIST | TX_FIFO_LAST_LIST);
#endif

#ifdef NETIF_F_TSO
	if (mss)
		val64 |= TX_FIFO_SPECIAL_FUNC;
#endif
#ifdef NETIF_F_UFO
	if (skb_shinfo(skb)->ufo_size)
		val64 |= TX_FIFO_SPECIAL_FUNC;
#endif

	writeq(val64, &tx_fifo->List_Control);
	readl(&bar0->general_int_status);

	put_off++;
	if (put_off == mac_control->fifos[queue].tx_curr_put_info.fifo_len + 1)
		put_off = 0;
	mac_control->fifos[queue].tx_curr_put_info.offset = put_off;

	/* Avoid "put" pointer going beyond "get" pointer */
	if (((put_off+1) == queue_len ? 0 : (put_off+1)) == get_off) {
		sp->mac_control.stats_info->sw_stat.fifo_full_cnt[queue]++;
		DBG_PRINT(TX_DBG,
			  "No free TxDs for xmit, Put: 0x%x Get:0x%x\n",
			  put_off, get_off);
		netif_stop_queue(dev);
	}

	dev->trans_start = jiffies;
	spin_unlock_irqrestore(&fifo->tx_lock, flags);

	return 0;
}

static void
s2io_alarm_handle(unsigned long data)
{
	nic_t *sp = (nic_t *)data;

	alarm_intr_handler(sp);
	mod_timer(&sp->alarm_timer, jiffies + HZ / 2);
}

#ifdef CONFIG_PCI_MSI
static irqreturn_t
s2io_msix_ring_handle(int irq, void *dev_id, struct pt_regs *regs)
{

	ring_info_t *ring = (ring_info_t *)dev_id;
	nic_t *sp = ring->nic;
	struct net_device *dev = (struct net_device *) dev_id;

	if (test_and_set_bit(0, &ring->isr_cnt)){
		return IRQ_HANDLED;
	}

	if (unlikely(atomic_read(&sp->card_state) == CARD_DOWN)){
		atomic_dec(&ring->isr_cnt);
		return IRQ_NONE;
	}

	rx_intr_handler(ring);
	if (fill_rx_buffers(sp, ring->ring_no) == -ENOMEM) {
		DBG_PRINT(INFO_DBG, "%s:Out of memory", dev->name);
		DBG_PRINT(INFO_DBG, " in Rx Intr!!\n");
	}

	clear_bit(0, &ring->isr_cnt);
	return IRQ_HANDLED;
}

static irqreturn_t
s2io_msix_fifo_handle(int irq, void *dev_id, struct pt_regs *regs)
{
	fifo_info_t *fifo = (fifo_info_t *)dev_id;
	struct s2io_nic *sp = fifo->nic;

	atomic_inc(&fifo->isr_cnt);

	if (unlikely(atomic_read(&sp->card_state) == CARD_DOWN)){
		atomic_dec(&fifo->isr_cnt);
		return IRQ_NONE;
	}

	tx_intr_handler(fifo);
	atomic_dec(&fifo->isr_cnt);
	return IRQ_HANDLED;
}
#endif

static void s2io_txpic_intr_handle(nic_t *sp)
{
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64;

	val64 = readq(&bar0->pic_int_status);
	if (val64 & PIC_INT_GPIO) {
		val64 = readq(&bar0->gpio_int_reg);
		if ((val64 & GPIO_INT_REG_LINK_DOWN) &&
		    (val64 & GPIO_INT_REG_LINK_UP)) {
			/*
			 * This is unstable state so clear both up/down
			 * interrupt and adapter to re-evaluate the link state.
			 */
			val64 |=  GPIO_INT_REG_LINK_DOWN;
			val64 |= GPIO_INT_REG_LINK_UP;
			writeq(val64, &bar0->gpio_int_reg);
			val64 = readq(&bar0->gpio_int_mask);
			val64 &= ~(GPIO_INT_MASK_LINK_UP |
				   GPIO_INT_MASK_LINK_DOWN);
			writeq(val64, &bar0->gpio_int_mask);
		}
		else if (val64 & GPIO_INT_REG_LINK_UP) {
			/* Enable Adapter */
			val64 = readq(&bar0->adapter_control);
			val64 |= ADAPTER_CNTL_EN;
			writeq(val64, &bar0->adapter_control);
			val64 |= ADAPTER_LED_ON;
			writeq(val64, &bar0->adapter_control);
			if (!sp->device_enabled_once)
				sp->device_enabled_once = 1;

			s2io_link(sp, LINK_UP);
			/*
			 * unmask link down interrupt and mask link-up
			 * intr
			 */
			val64 = readq(&bar0->gpio_int_mask);
			val64 &= ~GPIO_INT_MASK_LINK_DOWN;
			val64 |= GPIO_INT_MASK_LINK_UP;
			writeq(val64, &bar0->gpio_int_mask);

		}else if (val64 & GPIO_INT_REG_LINK_DOWN) {
			s2io_link(sp, LINK_DOWN);
			/* Link is down so unmaks link up interrupt */
			val64 = readq(&bar0->gpio_int_mask);
			val64 &= ~GPIO_INT_MASK_LINK_UP;
			val64 |= GPIO_INT_MASK_LINK_DOWN;
			writeq(val64, &bar0->gpio_int_mask);

			/* turn off LED */
			val64 = readq(&bar0->adapter_control);
			val64 &= (~ADAPTER_LED_ON);
			writeq(val64, &bar0->adapter_control);
		}
	}
	val64 = readq(&bar0->gpio_int_mask);
}

/**
 *  s2io_isr - ISR handler of the device .
 *  @irq: the irq of the device.
 *  @dev_id: a void pointer to the dev structure of the NIC.
 *  @pt_regs: pointer to the registers pushed on the stack.
 *  Description:  This function is the ISR handler of the device. It
 *  identifies the reason for the interrupt and calls the relevant
 *  service routines. As a contongency measure, this ISR allocates the
 *  recv buffers, if their numbers are below the panic value which is
 *  presently set to 25% of the original number of rcv buffers allocated.
 *  Return value:
 *   IRQ_HANDLED: will be returned if IRQ was handled by this routine
 *   IRQ_NONE: will be returned if interrupt is not from our device
 */
static irqreturn_t s2io_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_id;
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	int i;
	u64 reason = 0;
	mac_info_t *mac_control;
	struct config_param *config;

	atomic_inc(&sp->isr_cnt);

	if (unlikely(atomic_read(&sp->card_state) == CARD_DOWN)) {
		atomic_dec(&sp->isr_cnt);
		return IRQ_NONE;
	}

	mac_control = &sp->mac_control;
	config = &sp->config;

	/*
	 * Identify the cause for interrupt and call the appropriate
	 * interrupt handler. Causes for the interrupt could be;
	 * 1. Rx of packet.
	 * 2. Tx complete.
	 * 3. Link down.
	 * 4. Error in any functional blocks of the NIC.
	 */
	reason = readq(&bar0->general_int_status);

	if (!reason) {
		/* The interrupt was not raised by us. */
		atomic_dec(&sp->isr_cnt);
		return IRQ_NONE;
	}
	else if ( unlikely(reason == S2IO_MINUS_ONE) ) {
		/* Disable device and get out */
		atomic_dec(&sp->isr_cnt);
		return IRQ_NONE;
	}

	writeq(S2IO_MINUS_ONE, &bar0->general_int_mask);

#ifdef CONFIG_S2IO_NAPI
		if (reason & GEN_INTR_RXTRAFFIC)
		{
			if ( likely (netif_rx_schedule_prep(dev)) ) {
				__netif_rx_schedule(dev);
				writeq(S2IO_MINUS_ONE, &bar0->rx_traffic_mask);
			}
			else
				writeq(S2IO_MINUS_ONE, &bar0->rx_traffic_int);
		}
#else
	/*
	 * Rx handler is called by default, without checking for the
	 * cause of interrupt.
	 * rx_traffic_int reg is an R1 register, writing all 1's
	 * will ensure that the actual interrupt causing bit get's
	 * cleared and hence a read can be avoided.
	 */
	if (reason & GEN_INTR_RXTRAFFIC)
		writeq(S2IO_MINUS_ONE, &bar0->rx_traffic_int);

	for (i = 0; i < config->rx_ring_num; i++) {
		rx_intr_handler(&mac_control->rings[i]);
	}
#endif

	/*
	 * tx_traffic_int reg is an R1 register, writing all 1's
	 * will ensure that the actual interrupt causing bit get's
	 * cleared and hence a read can be avoided.
	 */
	if (reason & GEN_INTR_TXTRAFFIC)
		writeq(S2IO_MINUS_ONE, &bar0->tx_traffic_int);

	for (i = 0; i < config->tx_fifo_num; i++)
		tx_intr_handler(&mac_control->fifos[i]);

	if (reason & GEN_INTR_TXPIC)
		s2io_txpic_intr_handle(sp);

	/*
	 * Reallocate the buffers from the interrupt handler itself
	 */
#ifndef CONFIG_S2IO_NAPI
	for (i = 0; i < config->rx_ring_num; i++) {
		if (fill_rx_buffers(sp, i) == -ENOMEM) {
			DBG_PRINT(INFO_DBG, "%s:Out of memory",
						dev->name);
			DBG_PRINT(INFO_DBG, " in Rx intr!!\n");
			break;
		}
	}
#endif
	writeq(0, &bar0->general_int_mask);
	readl(&bar0->general_int_status);
	atomic_dec(&sp->isr_cnt);
	return IRQ_HANDLED;
}


/**
 * s2io_updt_stats -
 */
static void s2io_updt_stats(nic_t *sp)
{
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64;
	int cnt = 0;

	if (atomic_read(&sp->card_state) == CARD_UP) {
		/* Apprx 30us on a 133 MHz bus */
		val64 = SET_UPDT_CLICKS(10) |
			STAT_CFG_ONE_SHOT_EN | STAT_CFG_STAT_EN;
		writeq(val64, &bar0->stat_cfg);
		do {
			udelay(100);
			val64 = readq(&bar0->stat_cfg);
			if (!(val64 & BIT(0)))
				break;
			cnt++;
			if (cnt == 5)
				break; /* Updt failed */
		} while(1);
	}
}

/**
 *  s2io_get_stats - Updates the device statistics structure.
 *  @dev : pointer to the device structure.
 *  Description:
 *  This function updates the device statistics structure in the s2io_nic
 *  structure and returns a pointer to the same.
 *  Return value:
 *  pointer to the updated net_device_stats structure.
 */

struct net_device_stats *s2io_get_stats(struct net_device *dev)
{
	nic_t *sp = dev->priv;
	mac_info_t *mac_control;
	struct config_param *config;


	mac_control = &sp->mac_control;
	config = &sp->config;

	/* Configure Stats for immediate updt */
	s2io_updt_stats(sp);

	sp->stats.tx_packets =
		le32_to_cpu(mac_control->stats_info->tmac_frms);
	sp->stats.tx_errors =
		le32_to_cpu(mac_control->stats_info->tmac_any_err_frms);
	sp->stats.rx_errors =
		le32_to_cpu(mac_control->stats_info->rmac_drop_frms);
	sp->stats.multicast =
		le32_to_cpu(mac_control->stats_info->rmac_vld_mcst_frms);
	sp->stats.rx_length_errors =
		le32_to_cpu(mac_control->stats_info->rmac_long_frms);

	return (&sp->stats);
}

/**
 *  s2io_set_multicast - entry point for multicast address enable/disable.
 *  @dev : pointer to the device structure
 *  Description:
 *  This function is a driver entry point which gets called by the kernel
 *  whenever multicast addresses must be enabled/disabled. This also gets
 *  called to set/reset promiscuous mode. Depending on the deivce flag, we
 *  determine, if multicast address must be enabled or if promiscuous mode
 *  is to be disabled etc.
 *  Return value:
 *  void.
 */

static void s2io_set_multicast(struct net_device *dev)
{
	int i, j, prev_cnt;
	struct dev_mc_list *mclist;
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64 = 0, multi_mac = 0x010203040506ULL, mask =
	    0xfeffffffffffULL;
	u64 dis_addr = 0xffffffffffffULL, mac_addr = 0;
	void __iomem *add;

	if ((dev->flags & IFF_ALLMULTI) && (!sp->m_cast_flg)) {
		/*  Enable all Multicast addresses */
		writeq(RMAC_ADDR_DATA0_MEM_ADDR(multi_mac),
		       &bar0->rmac_addr_data0_mem);
		writeq(RMAC_ADDR_DATA1_MEM_MASK(mask),
		       &bar0->rmac_addr_data1_mem);
		val64 = RMAC_ADDR_CMD_MEM_WE |
		    RMAC_ADDR_CMD_MEM_STROBE_NEW_CMD |
		    RMAC_ADDR_CMD_MEM_OFFSET(MAC_MC_ALL_MC_ADDR_OFFSET);
		writeq(val64, &bar0->rmac_addr_cmd_mem);
		/* Wait till command completes */
		wait_for_cmd_complete(&bar0->rmac_addr_cmd_mem,
				      RMAC_ADDR_CMD_MEM_STROBE_CMD_EXECUTING,
						S2IO_BIT_RESET);

		sp->m_cast_flg = 1;
		sp->all_multi_pos = MAC_MC_ALL_MC_ADDR_OFFSET;
	} else if ((dev->flags & IFF_ALLMULTI) && (sp->m_cast_flg)) {
		/*  Disable all Multicast addresses */
		writeq(RMAC_ADDR_DATA0_MEM_ADDR(dis_addr),
		       &bar0->rmac_addr_data0_mem);
		writeq(RMAC_ADDR_DATA1_MEM_MASK(0x0),
		       &bar0->rmac_addr_data1_mem);
		val64 = RMAC_ADDR_CMD_MEM_WE |
		    RMAC_ADDR_CMD_MEM_STROBE_NEW_CMD |
		    RMAC_ADDR_CMD_MEM_OFFSET(sp->all_multi_pos);
		writeq(val64, &bar0->rmac_addr_cmd_mem);
		/* Wait till command completes */
		wait_for_cmd_complete(&bar0->rmac_addr_cmd_mem,
				      RMAC_ADDR_CMD_MEM_STROBE_CMD_EXECUTING,
			  			S2IO_BIT_RESET);

		sp->m_cast_flg = 0;
		sp->all_multi_pos = 0;
	}

	if ((dev->flags & IFF_PROMISC) && (!sp->promisc_flg)) {
		/*  Put the NIC into promiscuous mode */
		add = &bar0->mac_cfg;
		val64 = readq(&bar0->mac_cfg);
		val64 |= MAC_CFG_RMAC_PROM_ENABLE;

		writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
		writel((u32) val64, add);
		writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
		writel((u32) (val64 >> 32), (add + 4));

		val64 = readq(&bar0->mac_cfg);
		sp->promisc_flg = 1;
		DBG_PRINT(INFO_DBG, "%s: entered promiscuous mode\n",
			  dev->name);
	} else if (!(dev->flags & IFF_PROMISC) && (sp->promisc_flg)) {
		/*  Remove the NIC from promiscuous mode */
		add = (void *) &bar0->mac_cfg;
		val64 = readq(&bar0->mac_cfg);
		val64 &= ~MAC_CFG_RMAC_PROM_ENABLE;

		writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
		writel((u32) val64, add);
		writeq(RMAC_CFG_KEY(0x4C0D), &bar0->rmac_cfg_key);
		writel((u32) (val64 >> 32), (add + 4));

		val64 = readq(&bar0->mac_cfg);
		sp->promisc_flg = 0;
		DBG_PRINT(INFO_DBG, "%s: left promiscuous mode\n",
			  dev->name);
	}

	/*  Update individual M_CAST address list */
	if ((!sp->m_cast_flg) && dev->mc_count) {
		if (dev->mc_count >
		    (MAX_ADDRS_SUPPORTED - MAC_MC_ADDR_START_OFFSET - 1)) {
			DBG_PRINT(ERR_DBG, "%s: No more Rx filters ",
				  dev->name);
			DBG_PRINT(ERR_DBG, "can be added, please enable ");
			DBG_PRINT(ERR_DBG, "ALL_MULTI instead\n");
			return;
		}

		prev_cnt = sp->mc_addr_count;
		sp->mc_addr_count = dev->mc_count;

		/* Clear out the previous list of Mc in the H/W. */
		for (i = 0; i < prev_cnt; i++) {
			writeq(RMAC_ADDR_DATA0_MEM_ADDR(dis_addr),
			       &bar0->rmac_addr_data0_mem);
			writeq(RMAC_ADDR_DATA1_MEM_MASK(0ULL),
				&bar0->rmac_addr_data1_mem);
			val64 = RMAC_ADDR_CMD_MEM_WE |
			    RMAC_ADDR_CMD_MEM_STROBE_NEW_CMD |
			    RMAC_ADDR_CMD_MEM_OFFSET
			    (MAC_MC_ADDR_START_OFFSET + i);
			writeq(val64, &bar0->rmac_addr_cmd_mem);

			/* Wait for command completes */
			if (wait_for_cmd_complete(&bar0->rmac_addr_cmd_mem,
				      RMAC_ADDR_CMD_MEM_STROBE_CMD_EXECUTING,
			  			S2IO_BIT_RESET)) {
				DBG_PRINT(ERR_DBG, "%s: Adding ",
					  dev->name);
				DBG_PRINT(ERR_DBG, "Multicasts failed\n");
				return;
			}
		}

		/* Create the new Rx filter list and update the same in H/W. */
		for (i = 0, mclist = dev->mc_list; i < dev->mc_count;
		     i++, mclist = mclist->next) {
			memcpy(sp->usr_addrs[i].addr, mclist->dmi_addr,
			       ETH_ALEN);
			mac_addr = 0;
			for (j = 0; j < ETH_ALEN; j++) {
				mac_addr |= mclist->dmi_addr[j];
				mac_addr <<= 8;
			}
			mac_addr >>= 8;
			writeq(RMAC_ADDR_DATA0_MEM_ADDR(mac_addr),
			       &bar0->rmac_addr_data0_mem);
			writeq(RMAC_ADDR_DATA1_MEM_MASK(0ULL),
				&bar0->rmac_addr_data1_mem);
			val64 = RMAC_ADDR_CMD_MEM_WE |
			    RMAC_ADDR_CMD_MEM_STROBE_NEW_CMD |
			    RMAC_ADDR_CMD_MEM_OFFSET
			    (i + MAC_MC_ADDR_START_OFFSET);
			writeq(val64, &bar0->rmac_addr_cmd_mem);

			/* Wait for command completes */
			if (wait_for_cmd_complete(&bar0->rmac_addr_cmd_mem,
				      RMAC_ADDR_CMD_MEM_STROBE_CMD_EXECUTING,
			  			S2IO_BIT_RESET)) {
				DBG_PRINT(ERR_DBG, "%s: Adding ",
					  dev->name);
				DBG_PRINT(ERR_DBG, "Multicasts failed\n");
				return;
			}
		}
	}
}

#if !defined(__VMKERNEL_MODULE__) || !defined(__VMKNETDDI_QUEUEOPS__)
/**
 *  s2io_set_mac_addr - Programs the Xframe mac address
 *  @dev : pointer to the device structure.
 *  @addr: a uchar pointer to the new mac address which is to be set.
 *  Description : This procedure will program the Xframe to receive
 *  frames with new Mac Address
 *  Return value: SUCCESS on success and an appropriate (-)ve integer
 *  as defined in errno.h file on failure.
 */

int s2io_set_mac_addr(struct net_device *dev, u8 * addr)
{
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	register u64 val64, mac_addr = 0;
	int i;

	/*
	 * Set the new MAC address as the new unicast filter and reflect this
	 * change on the device address registered with the OS. It will be
	 * at offset 0.
	 */
	for (i = 0; i < ETH_ALEN; i++) {
		mac_addr <<= 8;
		mac_addr |= addr[i];
	}

	writeq(RMAC_ADDR_DATA0_MEM_ADDR(mac_addr),
	       &bar0->rmac_addr_data0_mem);

	val64 =
	    RMAC_ADDR_CMD_MEM_WE | RMAC_ADDR_CMD_MEM_STROBE_NEW_CMD |
	    RMAC_ADDR_CMD_MEM_OFFSET(0);
	writeq(val64, &bar0->rmac_addr_cmd_mem);
	/* Wait till command completes */
	if (wait_for_cmd_complete(&bar0->rmac_addr_cmd_mem,
		      RMAC_ADDR_CMD_MEM_STROBE_CMD_EXECUTING,
			  S2IO_BIT_RESET)) {
		DBG_PRINT(ERR_DBG, "%s: set_mac_addr failed\n", dev->name);
		return FAILURE;
	}

	return SUCCESS;
}
#endif

/**
 * s2io_ethtool_sset - Sets different link parameters.
 * @sp : private member of the device structure, which is a pointer to the  * s2io_nic structure.
 * @info: pointer to the structure with parameters given by ethtool to set
 * link information.
 * Description:
 * The function sets different link parameters provided by the user onto
 * the NIC.
 * Return value:
 * 0 on success.
*/

#ifndef SET_ETHTOOL_OPS
#define SPEED_10000 10000
#endif
static int s2io_ethtool_sset(struct net_device *dev,
			     struct ethtool_cmd *info)
{
	nic_t *sp = dev->priv;
	if ((info->autoneg == AUTONEG_ENABLE) ||
	    (info->speed != SPEED_10000) || (info->duplex != DUPLEX_FULL))
		return -EINVAL;

	if (netif_running(dev)) {
		s2io_close(sp->dev);
		s2io_open(sp->dev);
	}

	return 0;
}

/**
 * s2io_ethtol_gset - Return link specific information.
 * @sp : private member of the device structure, pointer to the
 *      s2io_nic structure.
 * @info : pointer to the structure with parameters given by ethtool
 * to return link information.
 * Description:
 * Returns link specific information like speed, duplex etc.. to ethtool.
 * Return value :
 * return 0 on success.
 */

static int s2io_ethtool_gset(struct net_device *dev, struct ethtool_cmd *info)
{
	nic_t *sp = dev->priv;
	info->supported = (SUPPORTED_10000baseT_Full | SUPPORTED_FIBRE);
#ifdef ADVERTISED_10000baseT_Full
	info->advertising = ADVERTISED_10000baseT_Full;
#else
	info->advertising = SUPPORTED_10000baseT_Full;
#endif
#ifdef ADVERTISED_FIBRE
	info->advertising  |= ADVERTISED_FIBRE;
#else
	info->advertising  |= SUPPORTED_FIBRE;
#endif
	info->port = PORT_FIBRE;
	info->transceiver = XCVR_EXTERNAL;

	if (netif_carrier_ok(sp->dev)) {
		info->speed = 10000;
		info->duplex = DUPLEX_FULL;
	} else {
		info->speed = -1;
		info->duplex = -1;
	}

	info->autoneg = AUTONEG_DISABLE;
	return 0;
}

/**
 * s2io_ethtool_gdrvinfo - Returns driver specific information.
 * @sp : private member of the device structure, which is a pointer to the
 * s2io_nic structure.
 * @info : pointer to the structure with parameters given by ethtool to
 * return driver information.
 * Description:
 * Returns driver specefic information like name, version etc.. to ethtool.
 * Return value:
 *  void
 */

static void s2io_ethtool_gdrvinfo(struct net_device *dev,
				  struct ethtool_drvinfo *info)
{
	nic_t *sp = dev->priv;

	strncpy(info->driver, s2io_driver_name, sizeof(info->driver));
	strncpy(info->version, s2io_driver_version, sizeof(info->version));
	strncpy(info->fw_version, "", sizeof(info->fw_version));
	strncpy(info->bus_info, pci_name(sp->pdev), sizeof(info->bus_info));
	info->regdump_len = XENA_REG_SPACE;
	info->eedump_len = XENA_EEPROM_SPACE;
	info->testinfo_len = S2IO_TEST_LEN;
#ifdef ETHTOOL_GSTATS
	info->n_stats = S2IO_STAT_LEN;
#endif
}

/**
 *  s2io_ethtool_gregs - dumps the entire space of Xfame into the buffer.
 *  @sp: private member of the device structure, which is a pointer to the
 *  s2io_nic structure.
 *  @regs : pointer to the structure with parameters given by ethtool for
 *  dumping the registers.
 *  @reg_space: The input argumnet into which all the registers are dumped.
 *  Description:
 *  Dumps the entire register space of xFrame NIC into the user given
 *  buffer area.
 * Return value :
 * void .
*/

static void s2io_ethtool_gregs(struct net_device *dev,
			       struct ethtool_regs *regs, void *space)
{
	int i;
	u64 reg;
	u8 *reg_space = (u8 *) space;
	nic_t *sp = dev->priv;

	regs->len = XENA_REG_SPACE;
	regs->version = sp->pdev->subsystem_device;

	for (i = 0; i < regs->len; i += 8) {
		reg = readq((void *) (sp->bar0 + i));
		memcpy((reg_space + i), &reg, 8);
	}
}

/**
 *  s2io_phy_id  - timer function that alternates adapter LED.
 *  @data : address of the private member of the device structure, which
 *  is a pointer to the s2io_nic structure, provided as an u32.
 * Description: This is actually the timer function that alternates the
 * adapter LED bit of the adapter control bit to set/reset every time on
 * invocation. The timer is set for 1/2 a second, hence tha NIC blinks
 *  once every second.
*/
static void s2io_phy_id(unsigned long data)
{
	nic_t *sp = (nic_t *) data;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64 = 0;
	u16 subid;

	subid = sp->pdev->subsystem_device;
	if ((sp->device_type == XFRAME_II_DEVICE) ||
		   ((subid & 0xFF) >= 0x07)) {
		val64 = readq(&bar0->gpio_control);
		val64 ^= GPIO_CTRL_GPIO_0;
		writeq(val64, &bar0->gpio_control);
	} else {
		val64 = readq(&bar0->adapter_control);
		val64 ^= ADAPTER_LED_ON;
		writeq(val64, &bar0->adapter_control);
	}

	mod_timer(&sp->id_timer, jiffies + HZ / 2);
}

/**
 * s2io_ethtool_idnic - To physically identify the nic on the system.
 * @sp : private member of the device structure, which is a pointer to the
 * s2io_nic structure.
 * @id : pointer to the structure with identification parameters given by
 * ethtool.
 * Description: Used to physically identify the NIC on the system.
 * The Link LED will blink for a time specified by the user for
 * identification.
 * NOTE: The Link has to be Up to be able to blink the LED. Hence
 * identification is possible only if it's link is up.
 * Return value:
 * int , returns 0 on success
 */

static int s2io_ethtool_idnic(struct net_device *dev, u32 data)
{
	u64 val64 = 0, last_gpio_ctrl_val;
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u16 subid;

	subid = sp->pdev->subsystem_device;
	last_gpio_ctrl_val = readq(&bar0->gpio_control);
	if ((sp->device_type == XFRAME_I_DEVICE) &&
		((subid & 0xFF) < 0x07)) {
		val64 = readq(&bar0->adapter_control);
		if (!(val64 & ADAPTER_CNTL_EN)) {
			printk(KERN_ERR
			       "Adapter Link down, cannot blink LED\n");
			return -EFAULT;
		}
	}
	if (sp->id_timer.function == NULL) {
		init_timer(&sp->id_timer);
		sp->id_timer.function = s2io_phy_id;
		sp->id_timer.data = (unsigned long) sp;
	}
	mod_timer(&sp->id_timer, jiffies);
	if (data)
#ifndef __VMKERNEL_MODULE__
		msleep_interruptible(data * HZ);
#else
		msleep(data * HZ);
#endif
	else
#ifndef __VMKERNEL_MODULE__
		msleep_interruptible(MAX_FLICKER_TIME);
#else
		msleep(MAX_FLICKER_TIME);
#endif
	del_timer_sync(&sp->id_timer);

	if (CARDS_WITH_FAULTY_LINK_INDICATORS(sp->device_type, subid)) {
		writeq(last_gpio_ctrl_val, &bar0->gpio_control);
		last_gpio_ctrl_val = readq(&bar0->gpio_control);
	}

	return 0;
}

/**
 * s2io_ethtool_getpause_data -Pause frame frame generation and reception.
 * @sp : private member of the device structure, which is a pointer to the
 *	s2io_nic structure.
 * @ep : pointer to the structure with pause parameters given by ethtool.
 * Description:
 * Returns the Pause frame generation and reception capability of the NIC.
 * Return value:
 *  void
 */
static void s2io_ethtool_getpause_data(struct net_device *dev,
				       struct ethtool_pauseparam *ep)
{
	u64 val64;
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;

	val64 = readq(&bar0->rmac_pause_cfg);
	if (val64 & RMAC_PAUSE_GEN_ENABLE)
		ep->tx_pause = TRUE;
	if (val64 & RMAC_PAUSE_RX_ENABLE)
		ep->rx_pause = TRUE;
	ep->autoneg = FALSE;
}

/**
 * s2io_ethtool_setpause_data -  set/reset pause frame generation.
 * @sp : private member of the device structure, which is a pointer to the
 *      s2io_nic structure.
 * @ep : pointer to the structure with pause parameters given by ethtool.
 * Description:
 * It can be used to set or reset Pause frame generation or reception
 * support of the NIC.
 * Return value:
 * int, returns 0 on Success
 */

static int s2io_ethtool_setpause_data(struct net_device *dev,
			       struct ethtool_pauseparam *ep)
{
	u64 val64;
	nic_t *sp = dev->priv;
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) sp->bar0;

	val64 = readq(&bar0->rmac_pause_cfg);
	if (ep->tx_pause)
		val64 |= RMAC_PAUSE_GEN_ENABLE;
	else
		val64 &= ~RMAC_PAUSE_GEN_ENABLE;
	if (ep->rx_pause)
		val64 |= RMAC_PAUSE_RX_ENABLE;
	else
		val64 &= ~RMAC_PAUSE_RX_ENABLE;
	writeq(val64, &bar0->rmac_pause_cfg);
	return 0;
}

/**
 * read_eeprom - reads 4 bytes of data from user given offset.
 * @sp : private member of the device structure, which is a pointer to the
 *      s2io_nic structure.
 * @off : offset at which the data must be written
 * @data : Its an output parameter where the data read at the given
 *	offset is stored.
 * Description:
 * Will read 4 bytes of data from the user given offset and return the
 * read data.
 * NOTE: Will allow to read only part of the EEPROM visible through the
 *   I2C bus.
 * Return value:
 *  -1 on failure and 0 on success.
 */

#define S2IO_DEV_ID		5
static int read_eeprom(nic_t * sp, int off, u64 * data)
{
	int ret = -1;
	u32 exit_cnt = 0;
	u64 val64;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;

	if (sp->device_type == XFRAME_I_DEVICE) {
		val64 = I2C_CONTROL_DEV_ID(S2IO_DEV_ID) | I2C_CONTROL_ADDR(off) |
			I2C_CONTROL_BYTE_CNT(0x3) | I2C_CONTROL_READ |
			I2C_CONTROL_CNTL_START;
		SPECIAL_REG_WRITE(val64, &bar0->i2c_control, LF);

		while (exit_cnt < 5) {
			val64 = readq(&bar0->i2c_control);
			if (I2C_CONTROL_CNTL_END(val64)) {
				*data = I2C_CONTROL_GET_DATA(val64);
				ret = 0;
				break;
			}
			msleep(50);
			exit_cnt++;
		}
	}

	if (sp->device_type == XFRAME_II_DEVICE) {
		val64 = SPI_CONTROL_KEY(0x9) | SPI_CONTROL_SEL1 |
			SPI_CONTROL_BYTECNT(0x3) | SPI_CONTROL_CMD(0x3) |
			SPI_CONTROL_ADDR(off);
		SPECIAL_REG_WRITE(val64, &bar0->spi_control, LF);
		val64 |= SPI_CONTROL_REQ;
		SPECIAL_REG_WRITE(val64, &bar0->spi_control, LF);
		while (exit_cnt < 5) {
			val64 = readq(&bar0->spi_control);
			if (val64 & SPI_CONTROL_NACK) {
				ret = 1;
				break;
			} else if (val64 & SPI_CONTROL_DONE) {
				*data = readq(&bar0->spi_data);
				*data &= 0xffffff;
				ret = 0;
				break;
			}
			msleep(50);
			exit_cnt++;
		}
	}

	return ret;
}

/**
 *  write_eeprom - actually writes the relevant part of the data value.
 *  @sp : private member of the device structure, which is a pointer to the
 *       s2io_nic structure.
 *  @off : offset at which the data must be written
 *  @data : The data that is to be written
 *  @cnt : Number of bytes of the data that are actually to be written into
 *  the Eeprom. (max of 3)
 * Description:
 *  Actually writes the relevant part of the data value into the Eeprom
 *  through the I2C bus.
 * Return value:
 *  0 on success, -1 on failure.
 */

static int write_eeprom(nic_t * sp, int off, u64 data, int cnt)
{
	int exit_cnt = 0, ret = -1;
	u64 val64;
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) sp->bar0;

	if (sp->device_type == XFRAME_I_DEVICE) {
		val64 = I2C_CONTROL_DEV_ID(S2IO_DEV_ID) |
			I2C_CONTROL_ADDR(off) | I2C_CONTROL_BYTE_CNT(cnt) |
			I2C_CONTROL_SET_DATA((u32)data) |
			I2C_CONTROL_CNTL_START;
		SPECIAL_REG_WRITE(val64, &bar0->i2c_control, LF);

		while (exit_cnt < 5) {
			val64 = readq(&bar0->i2c_control);
			if (I2C_CONTROL_CNTL_END(val64)) {
				if (!(val64 & I2C_CONTROL_NACK))
					ret = 0;
				break;
			}
			msleep(50);
			exit_cnt++;
		}
	}

	if (sp->device_type == XFRAME_II_DEVICE) {
		int write_cnt = (cnt == 8) ? 0 : cnt;
		writeq(SPI_DATA_WRITE(data,(cnt<<3)), &bar0->spi_data);

		val64 = SPI_CONTROL_KEY(0x9) | SPI_CONTROL_SEL1 |
			SPI_CONTROL_BYTECNT(write_cnt) | SPI_CONTROL_CMD(0x2) |
			SPI_CONTROL_ADDR(off);
		SPECIAL_REG_WRITE(val64, &bar0->spi_control, LF);
		val64 |= SPI_CONTROL_REQ;
		SPECIAL_REG_WRITE(val64, &bar0->spi_control, LF);
		while (exit_cnt < 5) {
			val64 = readq(&bar0->spi_control);
			if (val64 & SPI_CONTROL_NACK) {
				ret = 1;
				break;
			} else if (val64 & SPI_CONTROL_DONE) {
				ret = 0;
				break;
			}
			msleep(50);
			exit_cnt++;
		}
	}

	return ret;
}
void s2io_vpd_read(nic_t *nic)
{
	u8 vpd_data[256],data;
	int i=0, cnt, fail = 0;
	int vpd_addr = 0x80;

	if (nic->device_type == XFRAME_II_DEVICE) {
		strcpy(nic->product_name, "Xframe II 10GbE network adapter");
		vpd_addr = 0x80;
	}
	else {
		strcpy(nic->product_name, "Xframe I 10GbE network adapter");
		vpd_addr = 0x50;
	}

	for (i = 0; i < 256; i +=4 ) {
		pci_write_config_byte(nic->pdev, (vpd_addr + 2), i);
		pci_read_config_byte(nic->pdev,  (vpd_addr + 2), &data);
		pci_write_config_byte(nic->pdev, (vpd_addr + 3), 0);
		for (cnt = 0; cnt <5; cnt++) {
			msleep(2);
			pci_read_config_byte(nic->pdev, (vpd_addr + 3), &data);
			if (data == 0x80)
				break;
		}
		if (cnt >= 5) {
			DBG_PRINT(ERR_DBG, "Read of VPD data failed\n");
			fail = 1;
			break;
		}
		pci_read_config_dword(nic->pdev,  (vpd_addr + 4),
				      (u32 *)&vpd_data[i]);
	}
	if ((!fail) && (vpd_data[1] < VPD_PRODUCT_NAME_LEN)) {
		memset(nic->product_name, 0, vpd_data[1]);
		memcpy(nic->product_name, &vpd_data[3], vpd_data[1]);
	}
}

/**
 *  s2io_ethtool_geeprom  - reads the value stored in the Eeprom.
 *  @sp : private member of the device structure, which is a pointer to the *       s2io_nic structure.
 *  @eeprom : pointer to the user level structure provided by ethtool,
 *  containing all relevant information.
 *  @data_buf : user defined value to be written into Eeprom.
 *  Description: Reads the values stored in the Eeprom at given offset
 *  for a given length. Stores these values int the input argument data
 *  buffer 'data_buf' and returns these to the caller (ethtool.)
 *  Return value:
 *  int  0 on success
 */

int s2io_ethtool_geeprom(struct net_device *dev,
			 struct ethtool_eeprom *eeprom, u8 * data_buf)
{
	u32 i, valid;
	u64 data;
	nic_t *sp = dev->priv;

	eeprom->magic = sp->pdev->vendor | (sp->pdev->device << 16);

	if ((eeprom->offset + eeprom->len) > (XENA_EEPROM_SPACE))
		eeprom->len = XENA_EEPROM_SPACE - eeprom->offset;

	for (i = 0; i < eeprom->len; i += 4) {
		if (read_eeprom(sp, (eeprom->offset + i), &data)) {
			DBG_PRINT(ERR_DBG, "Read of EEPROM failed\n");
			return -EFAULT;
		}
		valid = INV(data);
		memcpy((data_buf + i), &valid, 4);
	}
	return 0;
}

/**
 *  s2io_ethtool_seeprom - tries to write the user provided value in Eeprom
 *  @sp : private member of the device structure, which is a pointer to the
 *  s2io_nic structure.
 *  @eeprom : pointer to the user level structure provided by ethtool,
 *  containing all relevant information.
 *  @data_buf ; user defined value to be written into Eeprom.
 *  Description:
 *  Tries to write the user provided value in the Eeprom, at the offset
 *  given by the user.
 *  Return value:
 *  0 on success, -EFAULT on failure.
 */

static int s2io_ethtool_seeprom(struct net_device *dev,
				struct ethtool_eeprom *eeprom,
				u8 * data_buf)
{
	int len = eeprom->len, cnt = 0;
	u64 valid = 0, data;
	nic_t *sp = dev->priv;

	if (eeprom->magic != (sp->pdev->vendor | (sp->pdev->device << 16))) {
		DBG_PRINT(ERR_DBG,
			  "ETHTOOL_WRITE_EEPROM Err: Magic value ");
		DBG_PRINT(ERR_DBG, "is wrong, Its not 0x%x\n",
			  eeprom->magic);
		return -EFAULT;
	}

	while (len) {
		data = (u32) data_buf[cnt] & 0x000000FF;
		if (data) {
			valid = (u32) (data << 24);
		} else
			valid = data;

		if (write_eeprom(sp, (eeprom->offset + cnt), valid, 0)) {
			DBG_PRINT(ERR_DBG,
				  "ETHTOOL_WRITE_EEPROM Err: Cannot ");
			DBG_PRINT(ERR_DBG,
				  "write into the specified offset\n");
			return -EFAULT;
		}
		cnt++;
		len--;
	}

	return 0;
}

/**
 * s2io_register_test - reads and writes into all clock domains.
 * @sp : private member of the device structure, which is a pointer to the
 * s2io_nic structure.
 * @data : variable that returns the result of each of the test conducted b
 * by the driver.
 * Description:
 * Read and write into all clock domains. The NIC has 3 clock domains,
 * see that registers in all the three regions are accessible.
 * Return value:
 * 0 on success.
 */

static int s2io_register_test(nic_t * sp, uint64_t * data)
{
	XENA_dev_config_t *bar0 = (XENA_dev_config_t *) sp->bar0;
	u64 val64 = 0, exp_val;
	int fail = 0;

	val64 = readq(&bar0->pif_rd_swapper_fb);
	if (val64 != 0x123456789abcdefULL) {
		fail = 1;
		DBG_PRINT(INFO_DBG, "Read Test level 1 fails\n");
	}

	val64 = readq(&bar0->rmac_pause_cfg);
	if (val64 != 0xc000ffff00000000ULL) {
		fail = 1;
		DBG_PRINT(INFO_DBG, "Read Test level 2 fails\n");
	}

	val64 = readq(&bar0->rx_queue_cfg);
	if (sp->device_type == XFRAME_II_DEVICE)
		exp_val = 0x0404040404040404ULL;
	else
		exp_val = 0x0808080808080808ULL;
	if (val64 != exp_val) {
		fail = 1;
		DBG_PRINT(INFO_DBG, "Read Test level 3 fails\n");
	}

	val64 = readq(&bar0->xgxs_efifo_cfg);
	if (val64 != 0x000000001923141EULL) {
		fail = 1;
		DBG_PRINT(INFO_DBG, "Read Test level 4 fails\n");
	}

	val64 = 0x5A5A5A5A5A5A5A5AULL;
	writeq(val64, &bar0->xmsi_data);
	val64 = readq(&bar0->xmsi_data);
	if (val64 != 0x5A5A5A5A5A5A5A5AULL) {
		fail = 1;
		DBG_PRINT(ERR_DBG, "Write Test level 1 fails\n");
	}

	val64 = 0xA5A5A5A5A5A5A5A5ULL;
	writeq(val64, &bar0->xmsi_data);
	val64 = readq(&bar0->xmsi_data);
	if (val64 != 0xA5A5A5A5A5A5A5A5ULL) {
		fail = 1;
		DBG_PRINT(ERR_DBG, "Write Test level 2 fails\n");
	}

	*data = fail;
	return fail;
}

/**
 * s2io_eeprom_test - to verify that EEprom in the xena can be programmed.
 * @sp : private member of the device structure, which is a pointer to the
 * s2io_nic structure.
 * @data:variable that returns the result of each of the test conducted by
 * the driver.
 * Description:
 * Verify that EEPROM in the xena can be programmed using I2C_CONTROL
 * register.
 * Return value:
 * 0 on success.
 */

static int s2io_eeprom_test(nic_t * sp, uint64_t * data)
{
	int fail = 0;
	u64 ret_data, org_4F0, org_7F0;
	u8 saved_4F0 = 0, saved_7F0 = 0;
	struct net_device *dev = sp->dev;

	/* Test Write Error at offset 0 */
	/* Note that SPI interface allows write access to all areas
	 * of EEPROM. Hence doing all negative testing only for Xframe I.
	 */
	if (sp->device_type == XFRAME_I_DEVICE)
		if (!write_eeprom(sp, 0, 0, 3))
			fail = 1;

	/* Save current values at offsets 0x4F0 and 0x7F0 */
	if (!read_eeprom(sp, 0x4F0, &org_4F0))
		saved_4F0 = 1;
	if (!read_eeprom(sp, 0x7F0, &org_7F0))
		saved_7F0 = 1;

	/* Test Write at offset 4f0 */
	if (write_eeprom(sp, 0x4F0, 0x012345, 3))
		fail = 1;
	if (read_eeprom(sp, 0x4F0, &ret_data))
		fail = 1;

	if (ret_data != 0x012345) {
		DBG_PRINT(ERR_DBG, "%s: eeprom test error at offset 0x4F0.\
			  Data written %llx Data read %llx\n", dev->name,\
			  (unsigned long long)0x12345,\
			  (unsigned long long)ret_data);
		fail = 1;
	}

	/* Reset the EEPROM data go FFFF */
	write_eeprom(sp, 0x4F0, 0xFFFFFF, 3);

	/* Test Write Request Error at offset 0x7c */
	if (sp->device_type == XFRAME_I_DEVICE)
		if (!write_eeprom(sp, 0x07C, 0, 3))
			fail = 1;

	/* Test Write Request at offset 0x7fc */
	if (write_eeprom(sp, 0x7F0, 0x012345, 3))
		fail = 1;
	if (read_eeprom(sp, 0x7F0, &ret_data))
		fail = 1;

	if (ret_data != 0x012345) {
		DBG_PRINT(ERR_DBG, "%s: eeprom test error at offset 0x7F0.\
			  Data written %llx Data read %llx\n", dev->name,\
			  (unsigned long long)0x12345,\
			  (unsigned long long)ret_data);
                fail = 1;
	}

	/* Reset the EEPROM data go FFFF */
	write_eeprom(sp, 0x7F0, 0xFFFFFF, 3);

	if (sp->device_type == XFRAME_I_DEVICE) {
		/* Test Write Error at offset 0x80 */
		if (!write_eeprom(sp, 0x080, 0, 3))
			fail = 1;

		/* Test Write Error at offset 0xfc */
		if (!write_eeprom(sp, 0x0FC, 0, 3))
			fail = 1;

		/* Test Write Error at offset 0x100 */
		if (!write_eeprom(sp, 0x100, 0, 3))
			fail = 1;

		/* Test Write Error at offset 4ec */
		if (!write_eeprom(sp, 0x4EC, 0, 3))
			fail = 1;
	}

	/* Restore values at offsets 0x4F0 and 0x7F0 */
	if (saved_4F0)
		write_eeprom(sp, 0x4F0, org_4F0, 3);
	if (saved_7F0)
		write_eeprom(sp, 0x7F0, org_7F0, 3);

	*data = fail;
	return fail;
}

/**
 * s2io_bist_test - invokes the MemBist test of the card .
 * @sp : private member of the device structure, which is a pointer to the
 * s2io_nic structure.
 * @data:variable that returns the result of each of the test conducted by
 * the driver.
 * Description:
 * This invokes the MemBist test of the card. We give around
 * 2 secs time for the Test to complete. If it's still not complete
 * within this peiod, we consider that the test failed.
 * Return value:
 * 0 on success and -1 on failure.
 */

static int s2io_bist_test(nic_t * sp, uint64_t * data)
{
	u8 bist = 0;
	int cnt = 0, ret = -1;

	pci_read_config_byte(sp->pdev, PCI_BIST, &bist);
	bist |= PCI_BIST_START;
	pci_write_config_word(sp->pdev, PCI_BIST, bist);

	while (cnt < 20) {
		pci_read_config_byte(sp->pdev, PCI_BIST, &bist);
		if (!(bist & PCI_BIST_START)) {
			*data = (bist & PCI_BIST_CODE_MASK);
			ret = 0;
			break;
		}
		msleep(100);
		cnt++;
	}

	return ret;
}

/**
 * s2io-link_test - verifies the link state of the nic
 * @sp ; private member of the device structure, which is a pointer to the
 * s2io_nic structure.
 * @data: variable that returns the result of each of the test conducted by
 * the driver.
 * Description:
 * The function verifies the link state of the NIC and updates the input
 * argument 'data' appropriately.
 * Return value:
 * 0 on success.
 */

static int s2io_link_test(nic_t * sp, uint64_t * data)
{
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64;

	val64 = readq(&bar0->adapter_status);
	if(!(LINK_IS_UP(val64)))
		*data = 1;
	else
		*data = 0;

	return *data;
}

/**
 * s2io_rldram_test - offline test for access to the RldRam chip on the NIC
 * @sp - private member of the device structure, which is a pointer to the
 * s2io_nic structure.
 * @data - variable that returns the result of each of the test
 * conducted by the driver.
 * Description:
 *  This is one of the offline test that tests the read and write
 *  access to the RldRam chip on the NIC.
 * Return value:
 *  0 on success.
 */

static int s2io_rldram_test(nic_t * sp, uint64_t * data)
{
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	u64 val64;
	int cnt, iteration = 0, test_fail = 0;

	val64 = readq(&bar0->adapter_control);
	val64 &= ~ADAPTER_ECC_EN;
	writeq(val64, &bar0->adapter_control);

	val64 = readq(&bar0->mc_rldram_test_ctrl);
	val64 |= MC_RLDRAM_TEST_MODE;
	SPECIAL_REG_WRITE(val64, &bar0->mc_rldram_test_ctrl, LF);

	val64 = readq(&bar0->mc_rldram_mrs);
	val64 |= MC_RLDRAM_QUEUE_SIZE_ENABLE;
	SPECIAL_REG_WRITE(val64, &bar0->mc_rldram_mrs, UF);

	val64 |= MC_RLDRAM_MRS_ENABLE;
	SPECIAL_REG_WRITE(val64, &bar0->mc_rldram_mrs, UF);

	while (iteration < 2) {
		val64 = 0x55555555aaaa0000ULL;
		if (iteration == 1) {
			val64 ^= 0xFFFFFFFFFFFF0000ULL;
		}
		writeq(val64, &bar0->mc_rldram_test_d0);

		val64 = 0xaaaa5a5555550000ULL;
		if (iteration == 1) {
			val64 ^= 0xFFFFFFFFFFFF0000ULL;
		}
		writeq(val64, &bar0->mc_rldram_test_d1);

		val64 = 0x55aaaaaaaa5a0000ULL;
		if (iteration == 1) {
			val64 ^= 0xFFFFFFFFFFFF0000ULL;
		}
		writeq(val64, &bar0->mc_rldram_test_d2);

		val64 = (u64) (0x0000003ffffe0100ULL);
		writeq(val64, &bar0->mc_rldram_test_add);

		val64 = MC_RLDRAM_TEST_MODE | MC_RLDRAM_TEST_WRITE |
			MC_RLDRAM_TEST_GO;
		SPECIAL_REG_WRITE(val64, &bar0->mc_rldram_test_ctrl, LF);

		for (cnt = 0; cnt < 5; cnt++) {
			val64 = readq(&bar0->mc_rldram_test_ctrl);
			if (val64 & MC_RLDRAM_TEST_DONE)
				break;
			msleep(200);
		}

		if (cnt == 5)
			break;

		val64 = MC_RLDRAM_TEST_MODE | MC_RLDRAM_TEST_GO;
		SPECIAL_REG_WRITE(val64, &bar0->mc_rldram_test_ctrl, LF);

		for (cnt = 0; cnt < 5; cnt++) {
			val64 = readq(&bar0->mc_rldram_test_ctrl);
			if (val64 & MC_RLDRAM_TEST_DONE)
				break;
			msleep(500);
		}

		if (cnt == 5)
			break;

		val64 = readq(&bar0->mc_rldram_test_ctrl);
		if (!(val64 & MC_RLDRAM_TEST_PASS))
			test_fail = 1;

		iteration++;
	}

	*data = test_fail;

	/* Bring the adapter out of test mode */
	SPECIAL_REG_WRITE(0, &bar0->mc_rldram_test_ctrl, LF);

	return test_fail;

}

/**
 *  s2io_ethtool_test - conducts 6 tsets to determine the health of card.
 *  @sp : private member of the device structure, which is a pointer to the
 *  s2io_nic structure.
 *  @ethtest : pointer to a ethtool command specific structure that will be
 *  returned to the user.
 *  @data : variable that returns the result of each of the test
 * conducted by the driver.
 * Description:
 *  This function conducts 6 tests ( 4 offline and 2 online) to determine
 *  the health of the card.
 * Return value:
 *  void
 */

static void s2io_ethtool_test(struct net_device *dev,
			      struct ethtool_test *ethtest,
			      uint64_t * data)
{
	nic_t *sp = dev->priv;
	int orig_state = netif_running(sp->dev);

	if (ethtest->flags == ETH_TEST_FL_OFFLINE) {
		/* Offline Tests. */
		if (orig_state)
			s2io_close(sp->dev);

		if (s2io_register_test(sp, &data[0]))
			ethtest->flags |= ETH_TEST_FL_FAILED;

		s2io_reset(sp);

		if (s2io_rldram_test(sp, &data[3]))
			ethtest->flags |= ETH_TEST_FL_FAILED;

		s2io_reset(sp);

		if (s2io_eeprom_test(sp, &data[1]))
			ethtest->flags |= ETH_TEST_FL_FAILED;

		if (s2io_bist_test(sp, &data[4]))
			ethtest->flags |= ETH_TEST_FL_FAILED;

		if (orig_state)
			s2io_open(sp->dev);

		data[2] = 0;
	} else {
		/* Online Tests. */
		if (!orig_state) {
			DBG_PRINT(ERR_DBG,
				  "%s: is not up, cannot run test\n",
				  dev->name);
			data[0] = -1;
			data[1] = -1;
			data[2] = -1;
			data[3] = -1;
			data[4] = -1;
		}

		if (s2io_link_test(sp, &data[2]))
			ethtest->flags |= ETH_TEST_FL_FAILED;

		data[0] = 0;
		data[1] = 0;
		data[3] = 0;
		data[4] = 0;
	}
}

#ifdef ETHTOOL_GSTATS
static void s2io_get_ethtool_stats(struct net_device *dev,
				   struct ethtool_stats *estats,
				   u64 * tmp_stats)
{
	int i = 0, k = 0;
	nic_t *sp = dev->priv;
	StatInfo_t *stat_info = sp->mac_control.stats_info;

	s2io_updt_stats(sp);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_frms_oflow) << 32  |
		le32_to_cpu(stat_info->tmac_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_data_octets_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_data_octets);
	tmp_stats[i++] = le64_to_cpu(stat_info->tmac_drop_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_mcst_frms_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_mcst_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_bcst_frms_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_bcst_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->tmac_pause_ctrl_frms);
        tmp_stats[i++] =
                (u64)le32_to_cpu(stat_info->tmac_ttl_octets_oflow) << 32 |
                le32_to_cpu(stat_info->tmac_ttl_octets);
	tmp_stats[i++] =
                (u64)le32_to_cpu(stat_info->tmac_ucst_frms_oflow) << 32 |
                le32_to_cpu(stat_info->tmac_ucst_frms);
	tmp_stats[i++] =
                (u64)le32_to_cpu(stat_info->tmac_nucst_frms_oflow) << 32 |
                le32_to_cpu(stat_info->tmac_nucst_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_any_err_frms_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_any_err_frms);
        tmp_stats[i++] = le64_to_cpu(stat_info->tmac_ttl_less_fb_octets);
	tmp_stats[i++] = le64_to_cpu(stat_info->tmac_vld_ip_octets);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_vld_ip_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_vld_ip);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_drop_ip_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_drop_ip);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_icmp_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_icmp);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->tmac_rst_tcp_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_rst_tcp);
	tmp_stats[i++] = le64_to_cpu(stat_info->tmac_tcp);
	tmp_stats[i++] = (u64)le32_to_cpu(stat_info->tmac_udp_oflow) << 32 |
		le32_to_cpu(stat_info->tmac_udp);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_vld_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_vld_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_data_octets_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_data_octets);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_fcs_err_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_drop_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_vld_mcst_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_vld_mcst_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_vld_bcst_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_vld_bcst_frms);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_in_rng_len_err_frms);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_out_rng_len_err_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_long_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_pause_ctrl_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_unsup_ctrl_frms);
        tmp_stats[i++] =
                (u64)le32_to_cpu(stat_info->rmac_ttl_octets_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_ttl_octets);
        tmp_stats[i++] =
                (u64)le32_to_cpu(stat_info->rmac_accepted_ucst_frms_oflow)
		<< 32 | le32_to_cpu(stat_info->rmac_accepted_ucst_frms);
	tmp_stats[i++] =
                (u64)le32_to_cpu(stat_info->rmac_accepted_nucst_frms_oflow)
                 << 32 | le32_to_cpu(stat_info->rmac_accepted_nucst_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_discarded_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_discarded_frms);
        tmp_stats[i++] =
                (u64)le32_to_cpu(stat_info->rmac_drop_events_oflow)
                 << 32 | le32_to_cpu(stat_info->rmac_drop_events);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_less_fb_octets);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_usized_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_usized_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_osized_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_osized_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_frag_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_frag_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_jabber_frms_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_jabber_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_64_frms);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_65_127_frms);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_128_255_frms);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_256_511_frms);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_512_1023_frms);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_1024_1518_frms);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_ip_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_ip);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ip_octets);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_hdr_err_ip);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_drop_ip_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_drop_ip);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_icmp_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_icmp);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_tcp);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_udp_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_udp);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_err_drp_udp_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_err_drp_udp);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_xgmii_err_sym);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q0);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q1);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q2);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q3);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q4);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q5);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q6);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_frms_q7);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q0);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q1);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q2);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q3);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q4);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q5);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q6);
	tmp_stats[i++] = le16_to_cpu(stat_info->rmac_full_q7);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_pause_cnt_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_pause_cnt);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_xgmii_data_err_cnt);
        tmp_stats[i++] = le64_to_cpu(stat_info->rmac_xgmii_ctrl_err_cnt);
	tmp_stats[i++] =
		(u64)le32_to_cpu(stat_info->rmac_accepted_ip_oflow) << 32 |
		le32_to_cpu(stat_info->rmac_accepted_ip);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_err_tcp);
	tmp_stats[i++] = le32_to_cpu(stat_info->rd_req_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->new_rd_req_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->new_rd_req_rtry_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->rd_rtry_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->wr_rtry_rd_ack_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->wr_req_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->new_wr_req_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->new_wr_req_rtry_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->wr_rtry_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->wr_disc_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->rd_rtry_wr_ack_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->txp_wr_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->txd_rd_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->txd_wr_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->rxd_rd_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->rxd_wr_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->txf_rd_cnt);
	tmp_stats[i++] = le32_to_cpu(stat_info->rxf_wr_cnt);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_1519_4095_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_4096_8191_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_8192_max_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_ttl_gt_max_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_osized_alt_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_jabber_alt_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_gt_max_alt_frms);
	tmp_stats[i++] = le64_to_cpu(stat_info->rmac_vlan_frms);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_len_discard);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_fcs_discard);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_pf_discard);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_da_discard);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_red_discard);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_rts_discard);
	tmp_stats[i++] = le32_to_cpu(stat_info->rmac_ingm_full_discard);
	tmp_stats[i++] = le32_to_cpu(stat_info->link_fault_cnt);
	tmp_stats[i++] = 0;
	tmp_stats[i++] = stat_info->sw_stat.single_ecc_errs;
	tmp_stats[i++] = stat_info->sw_stat.double_ecc_errs;
	tmp_stats[i++] = stat_info->sw_stat.parity_err_cnt;
	tmp_stats[i++] = stat_info->sw_stat.serious_err_cnt;
	tmp_stats[i++] = stat_info->sw_stat.soft_reset_cnt;
	for (k = 0; k < MAX_TX_FIFOS; k++)
		tmp_stats[i++] = stat_info->sw_stat.fifo_full_cnt[k];
	for (k = 0; k < MAX_RX_RINGS; k++)
		tmp_stats[i++] = stat_info->sw_stat.ring_full_cnt[k];
	for (k = 0; k < MAX_RX_RINGS; k++)
		tmp_stats[i++] = stat_info->sw_stat.ring_mem_fail_cnt[k];
	tmp_stats[i++] = stat_info->xpak_stat.alarm_transceiver_temp_high;
	tmp_stats[i++] = stat_info->xpak_stat.alarm_transceiver_temp_low;
	tmp_stats[i++] = stat_info->xpak_stat.alarm_laser_bias_current_high;
	tmp_stats[i++] = stat_info->xpak_stat.alarm_laser_bias_current_low;
	tmp_stats[i++] = stat_info->xpak_stat.alarm_laser_output_power_high;
	tmp_stats[i++] = stat_info->xpak_stat.alarm_laser_output_power_low;
	tmp_stats[i++] = stat_info->xpak_stat.warn_transceiver_temp_high;
	tmp_stats[i++] = stat_info->xpak_stat.warn_transceiver_temp_low;
	tmp_stats[i++] = stat_info->xpak_stat.warn_laser_bias_current_high;
	tmp_stats[i++] = stat_info->xpak_stat.warn_laser_bias_current_low;
	tmp_stats[i++] = stat_info->xpak_stat.warn_laser_output_power_high;
	tmp_stats[i++] = stat_info->xpak_stat.warn_laser_output_power_low;
	tmp_stats[i++] = stat_info->sw_stat.clubbed_frms_cnt;
	tmp_stats[i++] = stat_info->sw_stat.sending_both;
	tmp_stats[i++] = stat_info->sw_stat.outof_sequence_pkts;
	tmp_stats[i++] = stat_info->sw_stat.flush_max_pkts;
	if (stat_info->sw_stat.num_aggregations) {
		u64 tmp = stat_info->sw_stat.sum_avg_pkts_aggregated;
		int count = 0;
		/*
		 * Since 64-bit divide does not work on all platforms,
		 * do repeated subtraction.
		 */
		while (tmp >= stat_info->sw_stat.num_aggregations) {
			tmp -= stat_info->sw_stat.num_aggregations;
			count++;
		}
		tmp_stats[i++] = count;
	}
	else
		tmp_stats[i++] = 0;
}
#endif

#ifndef SET_ETHTOOL_OPS
/**
 * s2io_ethtool -to support all ethtool features .
 * @dev : device pointer.
 * @ifr :   An IOCTL specefic structure, that can contain a pointer to
 * a proprietary structure used to pass information to the driver.
 * Description:
 * Function used to support all ethtool fatures except dumping Device stats
 * as it can be obtained from the util tool for now.
 * Return value:
 * 0 on success and an appropriate (-)ve integer as defined in errno.h
 * file on failure.
 */

static int s2io_ethtool(struct net_device *dev, struct ifreq *rq)
{
	nic_t *sp = dev->priv;
	void *data = rq->ifr_data;
	u32 ecmd;

	if (get_user(ecmd, (u32 *) data)) {
		return -EFAULT;
	}

	switch (ecmd) {
	case ETHTOOL_GSET:
		{
			struct ethtool_cmd info = { ETHTOOL_GSET };
			s2io_ethtool_gset(dev, &info);
			if (copy_to_user(data, &info, sizeof(info)))
				return -EFAULT;
			break;
		}
	case ETHTOOL_SSET:
		{
			struct ethtool_cmd info;

			if (copy_from_user(&info, data, sizeof(info)))
				return -EFAULT;
			if (s2io_ethtool_sset(dev, &info))
				return -EFAULT;
			break;
		}
	case ETHTOOL_GDRVINFO:
		{
			struct ethtool_drvinfo info = { ETHTOOL_GDRVINFO };

			s2io_ethtool_gdrvinfo(dev, &info);
			if (copy_to_user(data, &info, sizeof(info)))
				return -EFAULT;
			break;
		}
	case ETHTOOL_GREGS:
		{
			struct ethtool_regs regs = { ETHTOOL_GREGS };
			u8 *reg_space;
			int ret = 0;

			regs.version = sp->pdev->subsystem_device;

			reg_space = kmalloc(XENA_REG_SPACE, GFP_KERNEL);
			if (reg_space == NULL) {
				DBG_PRINT(ERR_DBG,
					  "Memory allocation to dump ");
				DBG_PRINT(ERR_DBG, "registers failed\n");
				ret = -EFAULT;
			}
			memset(reg_space, 0, XENA_REG_SPACE);
			s2io_ethtool_gregs(dev, &regs, reg_space);
			if (copy_to_user(data, &regs, sizeof(regs))) {
				ret = -EFAULT;
				goto last_gregs;
			}
			data += offsetof(struct ethtool_regs, data);
			if (copy_to_user(data, reg_space, regs.len)) {
				ret = -EFAULT;
				goto last_gregs;
			}
		      last_gregs:
			kfree(reg_space);
			if (ret)
				return ret;
			break;
		}
	case ETHTOOL_GLINK:
		{
			struct ethtool_value link = { ETHTOOL_GLINK };

			link.data = netif_carrier_ok(dev);
			if (copy_to_user(data, &link, sizeof(link)))
				return -EFAULT;
			break;
		}
	case ETHTOOL_PHYS_ID:
		{
			struct ethtool_value id;

			if (copy_from_user(&id, data, sizeof(id)))
				return -EFAULT;
			s2io_ethtool_idnic(dev, id.data);
			break;
		}
	case ETHTOOL_GPAUSEPARAM:
		{
			struct ethtool_pauseparam ep =
			    { ETHTOOL_GPAUSEPARAM };

			s2io_ethtool_getpause_data(dev, &ep);
			if (copy_to_user(data, &ep, sizeof(ep)))
				return -EFAULT;
			break;

		}
	case ETHTOOL_SPAUSEPARAM:
		{
			struct ethtool_pauseparam ep;

			if (copy_from_user(&ep, data, sizeof(ep)))
				return -EFAULT;
			s2io_ethtool_setpause_data(dev, &ep);
			break;
		}
	case ETHTOOL_GRXCSUM:
		{
			struct ethtool_value ev = { ETHTOOL_GRXCSUM };

			ev.data = sp->rx_csum;
			if (copy_to_user(data, &ev, sizeof(ev)))
				return -EFAULT;
			break;
		}
	case ETHTOOL_GTXCSUM:
		{
			struct ethtool_value ev = { ETHTOOL_GTXCSUM };
			ev.data = (dev->features & NETIF_F_IP_CSUM);

			if (copy_to_user(data, &ev, sizeof(ev)))
				return -EFAULT;
			break;
		}
	case ETHTOOL_GSG:
		{
			struct ethtool_value ev = { ETHTOOL_GSG };
			ev.data = (dev->features & NETIF_F_SG);

			if (copy_to_user(data, &ev, sizeof(ev)))
				return -EFAULT;
			break;
		}
#ifdef NETIF_F_TSO
	case ETHTOOL_GTSO:
		{
			struct ethtool_value ev = { ETHTOOL_GTSO };
			ev.data = (dev->features & NETIF_F_TSO);

			if (copy_to_user(data, &ev, sizeof(ev)))
				return -EFAULT;
			break;
		}
#endif
	case ETHTOOL_STXCSUM:
		{
			struct ethtool_value ev;

			if (copy_from_user(&ev, data, sizeof(ev)))
				return -EFAULT;

			if (ev.data)
				dev->features |= NETIF_F_IP_CSUM;
			else
				dev->features &= ~NETIF_F_IP_CSUM;
			break;
		}
	case ETHTOOL_SRXCSUM:
		{
			struct ethtool_value ev;

			if (copy_from_user(&ev, data, sizeof(ev)))
				return -EFAULT;

			if (ev.data)
				sp->rx_csum = 1;
			else
				sp->rx_csum = 0;

			break;
		}
	case ETHTOOL_SSG:
		{
			struct ethtool_value ev;

			if (copy_from_user(&ev, data, sizeof(ev)))
				return -EFAULT;

			if (ev.data)
				dev->features |= NETIF_F_SG;
			else
				dev->features &= ~NETIF_F_SG;
			break;
		}
#ifdef NETIF_F_TSO
	case ETHTOOL_STSO:
		{
			struct ethtool_value ev;

			if (copy_from_user(&ev, data, sizeof(ev)))
				return -EFAULT;

			if (ev.data)
				dev->features |= NETIF_F_TSO;
			else
				dev->features &= ~NETIF_F_TSO;
			break;
		}
#endif
	case ETHTOOL_GEEPROM:
		{
			struct ethtool_eeprom eeprom = { ETHTOOL_GEEPROM };
			char *data_buf;
			int ret = 0;

			if (copy_from_user(&eeprom, data, sizeof(eeprom)))
				return -EFAULT;

			if (eeprom.len <= 0)
				return -EINVAL;

			if (!
			    (data_buf =
			     kmalloc(XENA_EEPROM_SPACE, GFP_KERNEL)))
				return -ENOMEM;
			s2io_ethtool_geeprom(dev, &eeprom, data_buf);

			if (copy_to_user(data, &eeprom, sizeof(eeprom))) {
				ret = -EFAULT;
				goto last_geprom;
			}

			data += offsetof(struct ethtool_eeprom, data);

			if (copy_to_user
			    (data, (void *) data_buf, eeprom.len)) {
				ret = -EFAULT;
				goto last_geprom;
			}

		      last_geprom:
			kfree(data_buf);
			if (ret)
				return ret;
			break;
		}
	case ETHTOOL_SEEPROM:
		{
			struct ethtool_eeprom eeprom;
			unsigned char *data_buf;
			void *ptr;
			int ret = 0;

			if (copy_from_user(&eeprom, data, sizeof(eeprom)))
				return -EFAULT;

			if (!(data_buf = kmalloc(eeprom.len, GFP_KERNEL)))
				return -ENOMEM;
			ptr = (void *) data_buf;

			data += offsetof(struct ethtool_eeprom, data);
			if (copy_from_user(ptr, data, eeprom.len)) {
				ret = -EFAULT;
				goto last_seprom;
			}

			if ((eeprom.offset + eeprom.len) >
			    (XENA_EEPROM_SPACE)) {
				DBG_PRINT(ERR_DBG, "%s Write ", dev->name);
				DBG_PRINT(ERR_DBG, "request overshoots ");
				DBG_PRINT(ERR_DBG, "the EEPROM area\n");
				ret = -EFAULT;
				goto last_seprom;
			}
			if (s2io_ethtool_seeprom(dev, &eeprom, data_buf)) {
				ret = -EFAULT;
				goto last_seprom;
			}

		      last_seprom:
			kfree(data_buf);
			if (ret)
				return ret;
			break;
		}
	case ETHTOOL_GSTRINGS:
		{
			struct ethtool_gstrings gstrings =
			    { ETHTOOL_GSTRINGS };
			char *strings = NULL;
			int ret = 0, mem_sz;

			if (copy_from_user
			    (&gstrings, data, sizeof(gstrings)))
				return -EFAULT;

			switch (gstrings.string_set) {
			case ETH_SS_TEST:
				gstrings.len = S2IO_TEST_LEN;
				mem_sz = S2IO_STRINGS_LEN;
				strings = kmalloc(mem_sz, GFP_KERNEL);
				if (!strings)
					return -ENOMEM;
				memcpy(strings, s2io_gstrings,
				       S2IO_STRINGS_LEN);
				break;
#ifdef ETHTOOL_GSTATS
			case ETH_SS_STATS:
				gstrings.len = S2IO_STAT_LEN;
				mem_sz = S2IO_STAT_STRINGS_LEN;
				strings = kmalloc(mem_sz, GFP_KERNEL);
				if (!strings)
					return -ENOMEM;
				memcpy(strings,
				       &ethtool_stats_keys,
				       sizeof(ethtool_stats_keys));
				break;
#endif

			default:
				return -EOPNOTSUPP;
			}

			if (copy_to_user
			    (data, &gstrings, sizeof(gstrings)))
				ret = -EFAULT;
			if (!ret) {
				data +=
				    offsetof(struct ethtool_gstrings,
					     data);
				if (copy_to_user(data, strings, mem_sz))
					ret = -EFAULT;
			}
			kfree(strings);
			if (ret)
				return ret;
			break;
		}
	case ETHTOOL_TEST:
		{
			struct {
				struct ethtool_test ethtest;
				uint64_t data[S2IO_TEST_LEN];
			} test = { {
			ETHTOOL_TEST}};

			if (copy_from_user(&test.ethtest, data,
					   sizeof(test.ethtest)))
				return -EFAULT;

			s2io_ethtool_test(dev, &test.ethtest, test.data);
			if (copy_to_user(data, &test, sizeof(test)))
				return -EFAULT;

			break;
		}
#ifdef ETHTOOL_GSTATS
	case ETHTOOL_GSTATS:
		{
			struct ethtool_stats stats;
			int ret;
			u64 *stat_mem;

			if (copy_from_user(&stats, data, sizeof(stats)))
				return -EFAULT;
			stats.n_stats = S2IO_STAT_LEN;
			stat_mem =
			    kmalloc(stats.n_stats * sizeof(u64), GFP_USER);
			if (!stat_mem)
				return -ENOMEM;

			s2io_get_ethtool_stats(dev, &stats, stat_mem);
			ret = 0;
			if (copy_to_user(data, &stats, sizeof(stats)))
				ret = -EFAULT;
			data += sizeof(stats);
			if (copy_to_user(data, stat_mem,
					 stats.n_stats * sizeof(u64)))
				ret = -EFAULT;
			kfree(stat_mem);
			return ret;
		}
#endif

	default:
		return -EOPNOTSUPP;
	}

	return 0;
}
#else

int s2io_ethtool_get_regs_len(struct net_device *dev)
{
	return (XENA_REG_SPACE);
}


u32 s2io_ethtool_get_rx_csum(struct net_device * dev)
{
	nic_t *sp = dev->priv;

	return (sp->rx_csum);
}
int s2io_ethtool_set_rx_csum(struct net_device *dev, u32 data)
{
	nic_t *sp = dev->priv;

	if (data)
		sp->rx_csum = 1;
	else
		sp->rx_csum = 0;

	return 0;
}
int s2io_get_eeprom_len(struct net_device *dev)
{
	return (XENA_EEPROM_SPACE);
}

int s2io_ethtool_self_test_count(struct net_device *dev)
{
	return (S2IO_TEST_LEN);
}
void s2io_ethtool_get_strings(struct net_device *dev,
			      u32 stringset, u8 * data)
{
	switch (stringset) {
	case ETH_SS_TEST:
		memcpy(data, s2io_gstrings, S2IO_STRINGS_LEN);
		break;
	case ETH_SS_STATS:
		memcpy(data, &ethtool_stats_keys,
		       sizeof(ethtool_stats_keys));
	}
}
static int s2io_ethtool_get_stats_count(struct net_device *dev)
{
	return (S2IO_STAT_LEN);
}

int s2io_ethtool_op_set_tx_csum(struct net_device *dev, u32 data)
{
	if (data)
		dev->features |= NETIF_F_IP_CSUM;
	else
		dev->features &= ~NETIF_F_IP_CSUM;

	return 0;
}


static struct ethtool_ops netdev_ethtool_ops = {
	.get_settings = s2io_ethtool_gset,
	.set_settings = s2io_ethtool_sset,
	.get_drvinfo = s2io_ethtool_gdrvinfo,
	.get_regs_len = s2io_ethtool_get_regs_len,
	.get_regs = s2io_ethtool_gregs,
	.get_link = ethtool_op_get_link,
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,00)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,23))
	.get_eeprom_len = s2io_get_eeprom_len,
#endif
	.get_eeprom = s2io_ethtool_geeprom,
	.set_eeprom = s2io_ethtool_seeprom,
	.get_pauseparam = s2io_ethtool_getpause_data,
	.set_pauseparam = s2io_ethtool_setpause_data,
	.get_rx_csum = s2io_ethtool_get_rx_csum,
	.set_rx_csum = s2io_ethtool_set_rx_csum,
	.get_tx_csum = ethtool_op_get_tx_csum,
	.set_tx_csum = s2io_ethtool_op_set_tx_csum,
	.get_sg = ethtool_op_get_sg,
	.set_sg = ethtool_op_set_sg,
#ifdef NETIF_F_TSO
#ifndef __VMKERNEL_MODULE__
	.get_tso = ethtool_op_get_tso,
	.set_tso = ethtool_op_set_tso,
#endif
#endif
#ifdef NETIF_F_UFO
	.get_ufo = ethtool_op_get_ufo,
	.set_ufo = ethtool_op_set_ufo,
#endif
	.self_test_count = s2io_ethtool_self_test_count,
	.self_test = s2io_ethtool_test,
	.get_strings = s2io_ethtool_get_strings,
	.phys_id = s2io_ethtool_idnic,
	.get_stats_count = s2io_ethtool_get_stats_count,
	.get_ethtool_stats = s2io_get_ethtool_stats
};
#endif
/**
 *  s2io_ioctl - Entry point for the Ioctl
 *  @dev :  Device pointer.
 *  @ifr :  An IOCTL specefic structure, that can contain a pointer to
 *  a proprietary structure used to pass information to the driver.
 *  @cmd :  This is used to distinguish between the different commands that
 *  can be passed to the IOCTL functions.
 *  Description:
 *  This function has support for ethtool, adding multiple MAC addresses on
 *  the NIC and some DBG commands for the util tool.
 *  Return value:
 *  0 on success and an appropriate (-)ve integer as defined in errno.h
 *  file on failure.
 */

int s2io_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	nic_t *sp = dev->priv;

	switch (cmd) {
#ifndef SET_ETHTOOL_OPS
		case SIOCETHTOOL:
			{
				return s2io_ethtool(dev, rq);
			}
#endif
		/* Private IOCTLs used by the util tool. */
		case SIOCDEVPRIVATE + 4:
			{
				ioctlInfo_t *io = (ioctlInfo_t *) rq->ifr_data;
				struct config_param *config = &sp->config;
				int i, j;
				int lst_size = config->max_txds * sizeof(TxD_t);
				void *to_buf = io->buffer, *fr_buf;
				io->size = 0;

				DBG_PRINT(ERR_DBG, "Tx get_offset %d, "
					  "put_offset %d \n",
					  sp->mac_control.fifos[0].
					  tx_curr_get_info.offset,
					  sp->mac_control.fifos[0].
					  tx_curr_put_info.offset);

				for (i = 0; i < config->tx_fifo_num; i++) {
					int fifo_len = config->tx_cfg[i].fifo_len;
					io->size += fifo_len;
					for (j=0; j<fifo_len; j++) {
						fr_buf = sp->mac_control.fifos[i].
								list_info[j].
								list_virt_addr;
						copy_to_user(to_buf, fr_buf,
								lst_size);
						to_buf += lst_size;
					}
				}
				io->size *= lst_size;
				break;
			}
		case SIOCDEVPRIVATE + 5:
			{
				ioctlInfo_t *io = (ioctlInfo_t *) rq->ifr_data;
				int i, j, row_sz;

				io->size = sp->mac_control.stats_mem_sz;
				s2io_updt_stats(sp);

				row_sz = (io->size + 15)/16;
				printk("Stats Dump:\n");
				for (i=0; i< row_sz; i++) {
					printk("%03x: ", (i*16));
					for (j=0; j< 16; j++) {
						u8 *x = (u8 *)
						(sp->mac_control.stats_mem + (i*16)+j);
						if (j == 8)
							printk("  ");
						printk("%02x ", *x);
						if (((i*16)+j) == io->size-1)
							goto end;
					}
					printk("\n");
				}
				end:
				printk("\n");

				copy_to_user((void *)io->buffer,
					(void *)sp->mac_control.stats_mem,
					io->size);
				break;
			}
		case SIOCDEVPRIVATE + 7:
			{
				unsigned int i = 0, j = 0;
				ioctlInfo_t *io = (ioctlInfo_t *) rq->ifr_data;

				DBG_PRINT(ERR_DBG, "Rx get_offset %d, "
					  "put_offset %d \n",
					  sp->mac_control.rings[0].
					  rx_curr_get_info.offset,
					  sp->mac_control.rings[0].
					  rx_curr_put_info.offset);

				io->size = 4096 * rx_ring_sz[0];
				while (i < sp->config.rx_cfg[0].num_rxd) {
					char *c =
					    (char *) sp->mac_control.rings[0].
						rx_blocks[j].block_virt_addr;
					copy_to_user((io->buffer + (j * 4096)), c,
						     4096);
					i += (MAX_RXDS_PER_BLOCK_1 + 1);
					j++;
				}
				break;
			}

		case SIOCDEVPRIVATE + 8:
			{
				spdm_cfg_t *io=(spdm_cfg_t *)rq->ifr_data;
				spdm_data_processor(io, sp);
				break;
			}

		case SIOCDEVPRIVATE + 9:
			{
				ioctlInfo_t *io = (ioctlInfo_t *) rq->ifr_data;
				u64 val64;
				if (atomic_read(&sp->card_state) != CARD_UP) {
					DBG_PRINT(ERR_DBG,
						"%s: Device is down!!\n", __FUNCTION__);
					break;
				}

				if (io->offset == 0x9c0)
					val64 = 0;
				else
					val64 = readq((sp->bar0 + io->offset));
				io->value = val64;
				break;
			}
		case SIOCDEVPRIVATE + 10:
			{
				ioctlInfo_t *io = (ioctlInfo_t *) rq->ifr_data;
				writeq(io->value,(sp->bar0 + io->offset));
				break;
			}
		default:
			{
				return -EOPNOTSUPP;
			}
	}
	return SUCCESS;
}

/**
 *  s2io_change_mtu - entry point to change MTU size for the device.
 *   @dev : device pointer.
 *   @new_mtu : the new MTU size for the device.
 *   Description: A driver entry point to change MTU size for the device.
 *   Before changing the MTU the device must be stopped.
 *  Return value:
 *   0 on success and an appropriate (-)ve integer as defined in errno.h
 *   file on failure.
 */

int s2io_change_mtu(struct net_device *dev, int new_mtu)
{
	nic_t *sp = dev->priv;
#ifdef __VMKERNEL_MODULE__
	unsigned long flags;
#else
	int err = 0;
#endif

	if ((new_mtu < MIN_MTU) || (new_mtu > S2IO_JUMBO_SIZE)) {
		DBG_PRINT(ERR_DBG, "%s: MTU size is invalid.\n",
			  dev->name);
		return -EPERM;
	}

	dev->mtu = new_mtu;
	if (netif_running(dev)) {
#ifdef __VMKERNEL_MODULE__
		/* This is a workaround (see PR 140859) for change_mtu()
		* call resulting in a wedged NIC.
		*/
		spin_lock_irqsave(&dev->xmit_lock, flags);
		if(sp->device_close_flag == FALSE)
			s2io_close(dev);
		if(s2io_open(dev)){
			spin_unlock_irqrestore(&dev->xmit_lock, flags);
			return -ENOMEM;
		}
		spin_unlock_irqrestore(&dev->xmit_lock, flags);
#else
		netif_stop_queue(dev);
		if (atomic_read(&sp->card_state) == CARD_UP)
			s2io_card_down(sp, 0);
		err=s2io_card_up(sp);
		if (err) {
			DBG_PRINT(ERR_DBG, "%s: Device bring up failed, err %d\n",
				  __FUNCTION__, err);
			if (err==-ENODEV)
				s2io_reset(sp);
			free_rx_buffers(sp);
		}
		if (netif_queue_stopped(dev))
			netif_wake_queue(dev);
#endif
	} else { /* Device is down */
		XENA_dev_config_t __iomem *bar0 = sp->bar0;
		u64 val64 = new_mtu;

		writeq(vBIT(val64, 2, 14), &bar0->rmac_max_pyld_len);
	}

	return 0;
}

/**
 * s2io_set_link - Set the LInk status
 * @data: long pointer to device private structue
 * Description: Sets the link status for the adapter
 */

static void s2io_set_link(unsigned long data)
{
	nic_t *nic = (nic_t *) data;
	struct net_device *dev = nic->dev;
	XENA_dev_config_t __iomem *bar0 = nic->bar0;
	register u64 val64;
	u16 subid;

	if (test_and_set_bit(0, &(nic->link_state))) {
		/* The card is being reset, no point doing anything */
#ifdef __VMKERNEL_MODULE__
		nic->link_task_scheduled = 0;
#endif
		return;
	}

	subid = nic->pdev->subsystem_device;
	if (s2io_link_fault_indication(nic) == MAC_RMAC_ERR_TIMER) {
		/*
		 * Allow a small delay for the NICs self initiated
		 * cleanup to complete.
		 */
		msleep(100);
	}

	val64 = readq(&bar0->adapter_status);
	if (LINK_IS_UP(val64)) {
		if(!(readq(&bar0->adapter_control) & ADAPTER_CNTL_EN)) {
			if (verify_xena_quiescence(nic, val64, nic->device_enabled_once)) {
				val64 = readq(&bar0->adapter_control);
				val64 |= ADAPTER_CNTL_EN;
				writeq(val64, &bar0->adapter_control);

				if (CARDS_WITH_FAULTY_LINK_INDICATORS(nic->device_type,
							     subid)) {
					val64 = readq(&bar0->gpio_control);
					val64 |= GPIO_CTRL_GPIO_0;
					writeq(val64, &bar0->gpio_control);
					val64 = readq(&bar0->gpio_control);
				} else {
					val64 |= ADAPTER_LED_ON;
					writeq(val64, &bar0->adapter_control);
				}
				nic->device_enabled_once = TRUE;
			} else {
				DBG_PRINT(ERR_DBG, "%s: Error: ", dev->name);
				DBG_PRINT(ERR_DBG, "device is not Quiescent\n");
				netif_stop_queue(dev);
			}
		}
		val64 = readq(&bar0->adapter_control);
		val64 |= ADAPTER_LED_ON;
		writeq(val64, &bar0->adapter_control);
		s2io_link(nic, LINK_UP);
	} else {
		if (CARDS_WITH_FAULTY_LINK_INDICATORS(nic->device_type,
							      subid)) {
			val64 = readq(&bar0->gpio_control);
			val64 &= ~GPIO_CTRL_GPIO_0;
			writeq(val64, &bar0->gpio_control);
			val64 = readq(&bar0->gpio_control);
		}
		/* turn off LED */
		val64 = readq(&bar0->adapter_control);
		val64 = val64 &(~ADAPTER_LED_ON);
		writeq(val64, &bar0->adapter_control);
		s2io_link(nic, LINK_DOWN);
	}

	clear_bit(0, &(nic->link_state));
#ifdef __VMKERNEL_MODULE__
	nic->link_task_scheduled = 0;
#endif
}

static int set_rxd_buffer_pointer(nic_t *sp, RxD_t *rxdp, buffAdd_t *ba,
			   struct sk_buff **skb, u64 *temp0, u64 *temp1,
			   u64 *temp2, int size)
{
	struct net_device *dev = sp->dev;
	struct sk_buff *frag_list;

	if ((sp->rxd_mode == RXD_MODE_1) && (rxdp->Host_Control == 0)) {
		/* allocate skb */
		if (*skb) {
			DBG_PRINT(INFO_DBG, "SKB is not NULL\n");
			/*
			 * As Rx frame are not going to be processed,
			 * using same mapped address for the Rxd
			 * buffer pointer
			 */
			((RxD1_t*)rxdp)->Buffer0_ptr = *temp0;
		} else {
			*skb = dev_alloc_skb(size);
			if (!(*skb)) {
				DBG_PRINT(INFO_DBG, "%s: Out of ", dev->name);
				DBG_PRINT(INFO_DBG, "memory to allocate SKBs\n");
				return -ENOMEM ;
			}
			/* storing the mapped addr in a temp variable
			 * such it will be used for next rxd whose
			 * Host Control is NULL
			 */
			((RxD1_t*)rxdp)->Buffer0_ptr = *temp0 =
#ifdef __VMKERNEL_MODULE__
				(*skb)->headMA;
#else
				pci_map_single( sp->pdev, (*skb)->data,
					size - NET_IP_ALIGN,
					PCI_DMA_FROMDEVICE);
#endif
			rxdp->Host_Control = (unsigned long) (*skb);
		}
	} else if ((sp->rxd_mode == RXD_MODE_3B) && (rxdp->Host_Control == 0)) {
		/* Two buffer Mode */
		if (*skb) {
			((RxD3_t*)rxdp)->Buffer2_ptr = *temp2;
			((RxD3_t*)rxdp)->Buffer0_ptr = *temp0;
			((RxD3_t*)rxdp)->Buffer1_ptr = *temp1;
		} else {
			*skb = dev_alloc_skb(size);
			((RxD3_t*)rxdp)->Buffer2_ptr = *temp2 =
#ifdef __VMKERNEL_MODULE__
				(*skb)->headMA;
#else
				pci_map_single(sp->pdev, (*skb)->data,
					       dev->mtu + 4,
					       PCI_DMA_FROMDEVICE);
#endif
			((RxD3_t*)rxdp)->Buffer0_ptr = *temp0 =
				pci_map_single( sp->pdev, ba->ba_0, BUF0_LEN,
						PCI_DMA_FROMDEVICE);

			rxdp->Host_Control = (unsigned long) (*skb);

			/* Buffer-1 will be dummy buffer not used */
			((RxD3_t*)rxdp)->Buffer1_ptr = *temp1 =
				pci_map_single(sp->pdev, ba->ba_1, BUF1_LEN,
					       PCI_DMA_FROMDEVICE);
		}
	} else if ((rxdp->Host_Control == 0)) {
		/* Three buffer mode */
		if (*skb) {
			((RxD3_t*)rxdp)->Buffer0_ptr = *temp0;
			((RxD3_t*)rxdp)->Buffer1_ptr = *temp1;
			((RxD3_t*)rxdp)->Buffer2_ptr = *temp2;
		} else {
			*skb = dev_alloc_skb(size);

			((RxD3_t*)rxdp)->Buffer0_ptr = *temp0 =
				pci_map_single(sp->pdev, ba->ba_0, BUF0_LEN,
					       PCI_DMA_FROMDEVICE);
			/* Buffer-1 receives L3/L4 headers */
			((RxD3_t*)rxdp)->Buffer1_ptr = *temp1 =
#ifdef __VMKERNEL_MODULE__
				(*skb)->headMA;
#else
				pci_map_single( sp->pdev, (*skb)->data,
						l3l4hdr_size + 4,
						PCI_DMA_FROMDEVICE);
#endif

			/*
			 * skb_shinfo(skb)->frag_list will have L4
			 * data payload
			 */
			skb_shinfo(*skb)->frag_list = dev_alloc_skb(dev->mtu +
								   ALIGN_SIZE);
			if (skb_shinfo(*skb)->frag_list == NULL) {
				DBG_PRINT(ERR_DBG, "%s: dev_alloc_skb \
					  failed\n ", dev->name);
				return -ENOMEM ;
			}
			frag_list = skb_shinfo(*skb)->frag_list;
			frag_list->next = NULL;
			/*
			 * Buffer-2 receives L4 data payload
			 */
			((RxD3_t*)rxdp)->Buffer2_ptr = *temp2 =
#ifdef __VMKERNEL_MODULE__
				frag_list->headMA;
#else
				pci_map_single( sp->pdev, frag_list->data,
						dev->mtu, PCI_DMA_FROMDEVICE);
#endif
		}
	}
	return 0;
}
static void set_rxd_buffer_size(nic_t *sp, RxD_t *rxdp, int size)
{
	struct net_device *dev = sp->dev;
	if (sp->rxd_mode == RXD_MODE_1) {
		rxdp->Control_2 = SET_BUFFER0_SIZE_1( size - NET_IP_ALIGN);
	} else if (sp->rxd_mode == RXD_MODE_3B) {
		rxdp->Control_2 = SET_BUFFER0_SIZE_3(BUF0_LEN);
		rxdp->Control_2 |= SET_BUFFER1_SIZE_3(1);
		rxdp->Control_2 |= SET_BUFFER2_SIZE_3( dev->mtu + 4);
	} else {
		rxdp->Control_2 = SET_BUFFER0_SIZE_3(BUF0_LEN);
		rxdp->Control_2 |= SET_BUFFER1_SIZE_3(l3l4hdr_size + 4);
		rxdp->Control_2 |= SET_BUFFER2_SIZE_3(dev->mtu);
	}
}

static  int rxd_owner_bit_reset(nic_t *sp)
{
	int i, j, k, blk_cnt = 0, size;
	mac_info_t * mac_control = &sp->mac_control;
	struct config_param *config = &sp->config;
	struct net_device *dev = sp->dev;
	RxD_t *rxdp = NULL;
	struct sk_buff *skb = NULL;
	buffAdd_t *ba = NULL;
	u64 temp0_64 = 0, temp1_64 = 0, temp2_64 = 0;

	/* Calculate the size based on ring mode */
	size = dev->mtu + HEADER_ETHERNET_II_802_3_SIZE +
		HEADER_802_2_SIZE + HEADER_SNAP_SIZE;
	if (sp->rxd_mode == RXD_MODE_1)
		size += NET_IP_ALIGN;
	else if (sp->rxd_mode == RXD_MODE_3B)
		size = dev->mtu + ALIGN_SIZE + BUF0_LEN + 4;
	else
		size = l3l4hdr_size + ALIGN_SIZE + BUF0_LEN + 4;

	for (i = 0; i < config->rx_ring_num; i++) {
		blk_cnt = config->rx_cfg[i].num_rxd /
			(rxd_count[sp->rxd_mode] +1);

		for (j = 0; j < blk_cnt; j++) {
			for (k = 0; k < rxd_count[sp->rxd_mode]; k++) {
				rxdp = mac_control->rings[i].
					rx_blocks[j].rxds[k].virt_addr;
				if(sp->rxd_mode >= RXD_MODE_3A)
					ba = &mac_control->rings[i].ba[j][k];
				if (set_rxd_buffer_pointer(sp, rxdp, ba,
						       &skb,(u64 *)&temp0_64,
						       (u64 *)&temp1_64,
						(u64 *)&temp2_64, size) == -ENOMEM)
					return 0;

				set_rxd_buffer_size(sp, rxdp, size);
				wmb();
				/* flip the Ownership bit to Hardware */
				rxdp->Control_1 |= RXD_OWN_XENA;
			}
		}
	}
	return 0;

}

static void s2io_quiescent_isr(atomic_t *isr_cnt)
{
	int cnt = 0;

	do {
		msleep(10);
		if (!atomic_read(isr_cnt))
			break;
		cnt++;
	} while(cnt < 5);
}

static void s2io_card_down(nic_t * sp, int flag)
{
	int cnt = 0;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	register u64 val64 = 0;
	struct net_device *dev = sp->dev;

	del_timer_sync(&sp->alarm_timer);
	/* If s2io_set_link task is executing, wait till it completes. */
	while (test_and_set_bit(0, &(sp->link_state))) {
		msleep(50);
	}
	atomic_set(&sp->card_state, CARD_DOWN);

	/* disable Tx and Rx traffic on the NIC */
	stop_nic(sp);
	if(flag)
	{
#ifdef CONFIG_PCI_MSI
		if (sp->intr_type == MSI_X) {
		int i;
		u16 msi_control;

		for (i=1; (sp->s2io_entries[i].in_use ==
			MSIX_REGISTERED_SUCCESS); i++) {
			int vector = sp->entries[i].vector;
			void *arg = sp->s2io_entries[i].arg;

			free_irq(vector, arg);
		}
		pci_read_config_word(sp->pdev, 0x42, &msi_control);
		msi_control &= 0xFFFE; /* Disable MSI */
		pci_write_config_word(sp->pdev, 0x42, msi_control);

		pci_disable_msix(sp->pdev);
#ifdef __VMKERNEL_MODULE__
		/* Temp */
		pci_read_config_word(sp->pdev, 0x92, &msi_control);
		msi_control &= ~0x8000; /* Disable MSI-X */
		pci_write_config_word(sp->pdev, 0x92, msi_control);
#endif
		}
		else
#endif
		free_irq(sp->pdev->irq, dev);
	}

	/* Waiting till all Interrupt handlers are complete */
	if (sp->intr_type == MSI_X) {
		struct mac_info *mac_control = &sp->mac_control;
		struct config_param *config = &sp->config;
		int i;

		for (i = 0; i < config->rx_ring_num; i++)
			s2io_quiescent_isr(&mac_control->rings[i].isr_cnt);

		for (i = 0; i < config->tx_fifo_num; i++)
			s2io_quiescent_isr(&mac_control->fifos[i].isr_cnt);
	} else
		s2io_quiescent_isr(&sp->isr_cnt);

	/* Check if the device is Quiescent and then Reset the NIC */
	do {
		/* As per the HW requirement we need to replenish the
		 * receive buffer to avoid the ring bump. Since there is
		 * no intention of processing the Rx frame at this pointwe are
		 * just settting the ownership bit of rxd in Each Rx
		 * ring to HW and set the appropriate buffer size
		 * based on the ring mode
		 */
		rxd_owner_bit_reset(sp);

		val64 = readq(&bar0->adapter_status);
		if (verify_xena_quiescence(sp, val64, sp->device_enabled_once)) {
			break;
		}

		msleep(50);
		cnt++;
		if (cnt == 10) {
			DBG_PRINT(ERR_DBG,
				  "s2io_close:Device not Quiescent ");
			DBG_PRINT(ERR_DBG, "adaper status reads 0x%llx\n",
				  (unsigned long long) val64);
			break;
		}
	} while (1);
	s2io_reset(sp);

	/* Free all Tx buffers */
	free_tx_buffers(sp);

	/* Free all Rx buffers */
	free_rx_buffers(sp);
	clear_bit(0, &(sp->link_state));
}

static int s2io_card_up(nic_t * sp)
{
	int i, ret = 0;
	mac_info_t *mac_control;
	struct config_param *config;
	struct net_device *dev = (struct net_device *) sp->dev;

	/* Initialize the H/W I/O registers */
	if( (ret = init_nic(sp)) != 0) {
		DBG_PRINT(ERR_DBG, "%s: H/W initialization failed\n",
			  dev->name);
		return ret;
	}

#ifdef CONFIG_PCI_MSI
	if (sp->intr_type == MSI_X)
		ret = s2io_enable_msi_x(sp);
#endif
	if (ret) {
		DBG_PRINT(ERR_DBG, "%s: Defaulting to INTA\n", dev->name);
		sp->intr_type = INTA;
	}

#ifdef CONFIG_PCI_MSI
	/* Store the values of the MSIX table in the nic_t structure */
	store_xmsi_data(sp);
#endif

	/*
	 * Initializing the Rx buffers. For now we are considering only 1
	 * Rx ring and initializing buffers into 30 Rx blocks
	 */
	mac_control = &sp->mac_control;
	config = &sp->config;

	for (i = 0; i < config->rx_ring_num; i++) {
		if ((ret = fill_rx_buffers(sp, i))) {
			DBG_PRINT(ERR_DBG, "%s: Out of memory in Open\n",
				  dev->name);
			s2io_reset(sp);
			return -ENOMEM;
		}
		DBG_PRINT(INFO_DBG, "Buf in ring:%d is %d:\n", i,
			  atomic_read(&sp->mac_control.rings[i].rx_bufs_left));
	}

	/* Setting its receive mode */
	s2io_set_multicast(dev);

	if (sp->lro) {
		/* Initialize max aggregatable pkts per session based on MTU */
		sp->lro_max_aggr_per_sess = ((1<<16) - 1) / dev->mtu;
		/* Check if we can use(if specified) user provided value */
		if (lro_max_pkts < sp->lro_max_aggr_per_sess)
			sp->lro_max_aggr_per_sess = lro_max_pkts;
	}

	/* Enable Rx Traffic and interrupts on the NIC */
	if (start_nic(sp)) {
		DBG_PRINT(ERR_DBG, "%s: Starting NIC failed\n", dev->name);
		return -ENODEV;
	}

	S2IO_TIMER_CONF(sp->alarm_timer, s2io_alarm_handle, sp, (HZ/2));

	atomic_set(&sp->card_state, CARD_UP);
	return 0;
}

/**
 * s2io_restart_nic - Resets the NIC.
 * @data : long pointer to the device private structure
 * Description:
 * This function is scheduled to be run by the s2io_tx_watchdog
 * function after 0.5 secs to reset the NIC. The idea is to reduce
 * the run time of the watch dog routine which is run holding a
 * spin lock.
 */

static void s2io_restart_nic(unsigned long data)
{
	struct net_device *dev = (struct net_device *) data;
	nic_t *sp = dev->priv;

#ifdef __VMKERNEL_MODULE__
	unsigned long flags;

	spin_lock_irqsave(&sp->task_sync_lock, flags);
	sp->device_close_flag = TRUE;	/* Device is shut down. */
	spin_unlock_irqrestore(&sp->task_sync_lock, flags);

	/* Using the same workaround as in change_mtu()
	* call resulting in a wedged NIC.
	*/
	netif_stop_queue(dev);
	spin_lock_irqsave(&dev->xmit_lock, flags);
	/* Reset card, free Tx and Rx buffers. */
	s2io_card_down(sp, 1);
	s2io_open(dev);
	spin_unlock_irqrestore(&dev->xmit_lock, flags);
#else
	int err=0;
	if (atomic_read(&sp->card_state) == CARD_UP)
		s2io_card_down(sp, 0);
	err=s2io_card_up(sp);
	if (err) {
		DBG_PRINT(ERR_DBG, "%s: Device bring up failed, err %d \n",
			      dev->name, err);
		if (err==-ENODEV)
			s2io_reset(sp);
		free_rx_buffers(sp);
	}
	netif_wake_queue(dev);
#endif
	DBG_PRINT(ERR_DBG, "%s: was reset by Tx watchdog timer\n",
		  dev->name);

#ifdef __VMKERNEL_MODULE__
	sp->rst_tmr_task_scheduled = 0;
#endif
}

/**
 *  s2io_tx_watchdog - Watchdog for transmit side.
 *  @dev : Pointer to net device structure
 *  Description:
 *  This function is triggered if the Tx Queue is stopped
 *  for a pre-defined amount of time when the Interface is still up.
 *  If the Interface is jammed in such a situation, the hardware is
 *  reset (by s2io_close) and restarted again (by s2io_open) to
 *  overcome any problem that might have been caused in the hardware.
 *  Return value:
 *  void
 */

static void s2io_tx_watchdog(struct net_device *dev)
{
	nic_t *sp = dev->priv;

	if (netif_carrier_ok(dev)) {
#ifdef __VMKERNEL_MODULE__
		spin_lock(&sp->task_sync_lock);
		if (sp->device_close_flag != TRUE) {
			sp->rst_tmr_task_scheduled = 1;
#endif
		schedule_work(&sp->rst_timer_task);
#ifdef __VMKERNEL_MODULE__
		}
		spin_unlock(&sp->task_sync_lock);
#endif
		sp->mac_control.stats_info->sw_stat.soft_reset_cnt++;
	}
}

/**
 *   rx_osm_handler - To perform some OS related operations on SKB.
 *   @sp: private member of the device structure,pointer to s2io_nic structure.
 *   @skb : the socket buffer pointer.
 *   @len : length of the packet
 *   @cksum : FCS checksum of the frame.
 *   @ring_no : the ring from which this RxD was extracted.
 *   Description:
 *   This function is called by the Rx interrupt serivce routine to perform
 *   some OS related operations on the SKB before passing it to the upper
 *   layers. It mainly checks if the checksum is OK, if so adds it to the
 *   SKBs cksum variable, increments the Rx packet count and passes the SKB
 *   to the upper layer. If the checksum is wrong, it increments the Rx
 *   packet error count, frees the SKB and returns error.
 *   Return value:
 *   SUCCESS on success and -1 on failure.
 */
static int rx_osm_handler(ring_info_t *ring_data, RxD_t * rxdp)
{
	nic_t *sp = ring_data->nic;
	struct net_device *dev = (struct net_device *) ring_data->dev;
	struct sk_buff *skb = (struct sk_buff *)
		((unsigned long) rxdp->Host_Control);
	int ring_no = ring_data->ring_no;
	u16 l3_csum, l4_csum;
	unsigned long long err = rxdp->Control_1 & RXD_T_CODE;
	lro_t *lro;
#ifdef virt_addr_valid
	if (!virt_addr_valid(skb)) {
		DBG_PRINT(INFO_DBG, "%s: not valid SKB: 0x%llx\n",
			dev->name, (unsigned long long)skb);
		return 0;
	}
#endif
	skb->dev = dev;

	if (err) {
		/* Check for parity error */
		if (err & 0x1) {
			sp->mac_control.stats_info->sw_stat.parity_err_cnt++;
		}

		/*
		* Drop the packet if bad transfer code. Exception being
		* 0x5, which could be due to unsupported IPv6 extension header.
		* In this case, we let stack handle the packet.
		* Note that in this case, since checksum will be incorrect,
		* stack will validate the same.
		*/
		if (err && ((err >> 48) != 0x5)) {
			DBG_PRINT(ERR_DBG, "%s: Rx error Value: 0x%llx\n",
				dev->name, err);
			sp->stats.rx_crc_errors++;
			dev_kfree_skb(skb);
			atomic_dec(&sp->mac_control.rings[ring_no].rx_bufs_left);
			rxdp->Host_Control = 0;
			return 0;
		}
	}

	/* Updating statistics */
	rxdp->Host_Control = 0;
	sp->rx_pkt_count++;
	sp->stats.rx_packets++;
	if (sp->rxd_mode == RXD_MODE_1) {
		int len = RXD_GET_BUFFER0_SIZE_1(rxdp->Control_2);
		sp->stats.rx_bytes += len;
		skb_put(skb, len);
	} else if (sp->rxd_mode >= RXD_MODE_3A) {
		int get_block = ring_data->rx_curr_get_info.block_index;
		int get_off = ring_data->rx_curr_get_info.offset;
		int buf0_len = RXD_GET_BUFFER0_SIZE_3(rxdp->Control_2);
		int buf2_len = RXD_GET_BUFFER2_SIZE_3(rxdp->Control_2);
		unsigned char *buff = skb_push(skb, buf0_len);

		buffAdd_t *ba = &ring_data->ba[get_block][get_off];
		sp->stats.rx_bytes += buf0_len + buf2_len;
		memcpy(buff, ba->ba_0, buf0_len);
		if (sp->rxd_mode == RXD_MODE_3A) {
			int buf1_len = RXD_GET_BUFFER1_SIZE_3(rxdp->Control_2);
			skb_put(skb, buf1_len);
			skb->len += buf2_len;
			skb->data_len += buf2_len;
#ifndef __VMKERNEL_MODULE__
			skb->truesize += buf2_len;
#endif
			skb_put(skb_shinfo(skb)->frag_list, buf2_len);
			sp->stats.rx_bytes += buf1_len;
		} else
			skb_put(skb, buf2_len);
	}

	if ((rxdp->Control_1 & TCP_OR_UDP_FRAME) && ((!sp->lro) ||
	    (sp->lro && (!(rxdp->Control_1 & RXD_FRAME_IP_FRAG)))) &&
	    (sp->rx_csum)) {
		l3_csum = RXD_GET_L3_CKSUM(rxdp->Control_1);
		l4_csum = RXD_GET_L4_CKSUM(rxdp->Control_1);
		if ((l3_csum == L3_CKSUM_OK) && (l4_csum == L4_CKSUM_OK)) {
			skb->ip_summed = CHECKSUM_UNNECESSARY;
			if (sp->lro) {
				u32 tcp_len;
				u8 *tcp;
				int ret = 0;

				skb->ip_summed = CHECKSUM_UNNECESSARY;
				ret = s2io_club_tcp_session(skb->data, &tcp,
						&tcp_len, &lro, rxdp, sp);
				switch (ret) {
					case 3: /* Begin anew */
						lro->parent = skb;
						goto aggregate;
					case 1: /* Aggregate */
					{
						lro_append_pkt(sp, lro,
							skb, tcp_len);
						goto aggregate;
					}
					case 4: /* Flush session */
					{
						lro_append_pkt(sp, lro,
							skb, tcp_len);
						queue_rx_frame(lro->parent);
						clear_lro_session(lro);
						sp->mac_control.stats_info->
						    sw_stat.flush_max_pkts++;
						goto aggregate;
					}
					case 2: /* Flush both */
						lro->parent->data_len =
							lro->frags_len;
						sp->mac_control.stats_info->
						     sw_stat.sending_both++;
						queue_rx_frame(lro->parent);
						clear_lro_session(lro);
						goto send_up;
					case 0: /* sessions exceeded */
					case -1: /* non-TCP or not
						  * L2 aggregatable
						  */
					case 5:
						/*
						 * First pkt in session not
						 * L3/L4 aggregatable
						 */
						break;
					default:
						DBG_PRINT(ERR_DBG,
							"%s: Samadhana!!\n",
							 __FUNCTION__);
						BUG();
				}
			}
		} else {
			/*
			 * Packet with erroneous checksum, let the
			 * upper layers deal with it.
			 */
			skb->ip_summed = CHECKSUM_NONE;
		}
	} else {
		skb->ip_summed = CHECKSUM_NONE;
	}

#if defined(__VMKERNEL_MODULE__) && defined(__VMKNETDDI_QUEUEOPS__)
	if (ring_data->ring_no) {
		vmknetddi_queueops_set_skb_queueid(skb,
			VMKNETDDI_QUEUEOPS_MK_RX_QUEUEID(ring_data->ring_no));
	}
#endif

	if (!sp->lro) {
		skb->protocol = eth_type_trans(skb, dev);
#ifdef CONFIG_S2IO_NAPI
		/* Queueing the vlan frame to the upper layer */
		if (sp->vlgrp && RXD_GET_VLAN_TAG(rxdp->Control_2)) {
			vlan_hwaccel_receive_skb(skb, sp->vlgrp,
				RXD_GET_VLAN_TAG(rxdp->Control_2));
		} else
			netif_receive_skb(skb);
#else
		/* Queueing the vlan frame to the upper layer */
		if (sp->vlgrp && RXD_GET_VLAN_TAG(rxdp->Control_2)) {
			vlan_hwaccel_rx(skb, sp->vlgrp,
				RXD_GET_VLAN_TAG(rxdp->Control_2));
		} else
			netif_rx(skb);
#endif
	} else {
send_up:
		queue_rx_frame(skb);
	}

	dev->last_rx = jiffies;
aggregate:
	atomic_dec(&sp->mac_control.rings[ring_no].rx_bufs_left);
	return SUCCESS;
}

/**
 *  s2io_link - stops/starts the Tx queue.
 *  @sp : private member of the device structure, which is a pointer to the
 *  s2io_nic structure.
 *  @link : inidicates whether link is UP/DOWN.
 *  Description:
 *  This function stops/starts the Tx queue depending on whether the link
 *  status of the NIC is is down or up. This is called by the Alarm
 *  interrupt handler whenever a link change interrupt comes up.
 *  Return value:
 *  void.
 */

void s2io_link(nic_t * sp, int link)
{
	struct net_device *dev = (struct net_device *) sp->dev;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	register u64 val64 = 0;

	if (link != sp->last_link_state) {
		if(use_continuous_tx_intrs) {
			val64 = readq(&bar0->tti_data1_mem);
			/* Disable continuous Tx interrupts on link down */	
			if (link == LINK_DOWN)
				val64 &= ~TTI_DATA1_MEM_TX_TIMER_CI_EN;
			else
				val64 |= TTI_DATA1_MEM_TX_TIMER_CI_EN;
			writeq(val64, &bar0->tti_data1_mem);
			val64 = TTI_CMD_MEM_WE | TTI_CMD_MEM_STROBE_NEW_CMD;
			writeq(val64, &bar0->tti_command_mem);
		}

		if (link == LINK_DOWN) {
			DBG_PRINT(ERR_DBG, "%s: Link down\n", dev->name);
			netif_carrier_off(dev);
		} else {
			DBG_PRINT(ERR_DBG, "%s: Link Up\n", dev->name);
			netif_carrier_on(dev);
		}
	}
	sp->last_link_state = link;
}

/**
 *  get_xena_rev_id - to identify revision ID of xena.
 *  @pdev : PCI Dev structure
 *  Description:
 *  Function to identify the Revision ID of xena.
 *  Return value:
 *  returns the revision ID of the device.
 */

int get_xena_rev_id(struct pci_dev *pdev)
{
	u8 id = 0;
	int ret;
	ret = pci_read_config_byte(pdev, PCI_REVISION_ID, (u8 *) & id);
	return id;
}

/* Returns 1 if system has AMD 8131a chipset. Returns 0 otherwise */
static int has_amd_8131a(void)
{
	struct pci_dev *tdev = NULL;
	u8 id = 0;

#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,20))
	if ((tdev = pci_find_device(0x1022, 0x7450, NULL)) != NULL) {
#else
	if ((tdev = pci_get_device(0x1022, 0x7450, NULL)) != NULL) {
#endif
		int ret;
		ret = pci_read_config_byte(tdev, PCI_REVISION_ID, (u8 *) &id);
		if (id <= 0x12)
			return 1;
	}
	return 0;
}

/**
 *  s2io_init_pci -Initialization of PCI and PCI-X configuration registers .
 *  @sp : private member of the device structure, which is a pointer to the
 *  s2io_nic structure.
 *  Description:
 *  This function initializes a few of the PCI and PCI-X configuration registers
 *  with recommended values.
 *  Return value:
 *  void
 */

static void s2io_init_pci(nic_t * sp)
{
	u16 pci_cmd = 0, pcix_cmd = 0;

	/* Enable Data Parity Error Recovery in PCI-X command register. */
	pci_read_config_word(sp->pdev, PCIX_COMMAND_REGISTER,
			     &(pcix_cmd));
	pci_write_config_word(sp->pdev, PCIX_COMMAND_REGISTER,
			      (pcix_cmd | 1));
	pci_read_config_word(sp->pdev, PCIX_COMMAND_REGISTER,
			     &(pcix_cmd));

	/* Set the PErr Response bit in PCI command register. */
	pci_read_config_word(sp->pdev, PCI_COMMAND, &pci_cmd);
	pci_write_config_word(sp->pdev, PCI_COMMAND,
			      (pci_cmd | PCI_COMMAND_PARITY));
	pci_read_config_word(sp->pdev, PCI_COMMAND, &pci_cmd);

	/*
	* If system has AMD 8131a(rev12 and below) bridge set safe PCI-X
	* parameters(MMRBC=1K and max splits = 2)
	*/
	if (has_amd_8131a()) {
		DBG_PRINT(INIT_DBG, "Found AMD 8131a bridge\n");
	}

	if(mmrbc != S2IO_PCI_BIOS_DEFAULTS){
			pcix_cmd &= ~(0x0c);
			pcix_cmd |= mmrbc << 2;
			pci_write_config_word(sp->pdev, PCIX_COMMAND_REGISTER, pcix_cmd);
	}
	else {
		if (has_amd_8131a()) {
			pcix_cmd &= ~(0x0c);
			pcix_cmd |= 0x04;
			pci_write_config_word(sp->pdev, PCIX_COMMAND_REGISTER,
				pcix_cmd);
		}
	}

	if(most != S2IO_PCI_BIOS_DEFAULTS){
			pcix_cmd &= ~(0x70);
			pcix_cmd |= most << 4;
			pci_write_config_word(sp->pdev, PCIX_COMMAND_REGISTER, pcix_cmd);
	}
	else {
		if (has_amd_8131a()) {
			pcix_cmd &= ~(0x70);
			pcix_cmd |= 0x10;
			pci_write_config_word(sp->pdev, PCIX_COMMAND_REGISTER,
				pcix_cmd);
		}
	}

	if(latency != S2IO_PCI_BIOS_DEFAULTS)
		pci_write_config_byte(sp->pdev, 0x0d, latency);
}

int s2io_verify_parm(struct pci_dev *pdev, u8 *dev_intr_type)
{
	if ((mmrbc != S2IO_PCI_BIOS_DEFAULTS) && (mmrbc > 3)){
		DBG_PRINT(ERR_DBG, "s2io: Unsupported pcix mmrbc value\n");
		DBG_PRINT(ERR_DBG, "s2io: Using Bios default value for pcix mmrbc\n");
		mmrbc = S2IO_PCI_BIOS_DEFAULTS;
	}
	if ((most != S2IO_PCI_BIOS_DEFAULTS) && (most > XENA_THIRTYTWO_SPLIT_TRANSACTION)){
		DBG_PRINT(ERR_DBG, "s2io: Unsupported pcix most value\n");
		DBG_PRINT(ERR_DBG, "s2io: Using Bios default value for pcix most\n");
		most = S2IO_PCI_BIOS_DEFAULTS;
	}
	if ((latency != S2IO_PCI_BIOS_DEFAULTS) && (latency > 0xFF)){
		DBG_PRINT(ERR_DBG, "s2io: Unsupported latency timer value\n");
		DBG_PRINT(ERR_DBG, "s2io: Using Bios default value for latency timer\n");
		latency = S2IO_PCI_BIOS_DEFAULTS;
	}
	if ( tx_fifo_num > 8) {
		DBG_PRINT(ERR_DBG, "s2io: Requested number of Tx fifos not "
			 "supported\n");
		DBG_PRINT(ERR_DBG, "s2io: Default to 8 Tx fifos\n");
		tx_fifo_num = 8;
	}
	if ( rx_ring_num > 8) {
		DBG_PRINT(ERR_DBG, "s2io: Requested number of Rx rings not "
			 "supported\n");
		DBG_PRINT(ERR_DBG, "s2io: Default to 8 Rx rings\n");
		rx_ring_num = 8;
	}
#ifdef CONFIG_S2IO_NAPI
	if (*dev_intr_type != INTA) {
		DBG_PRINT(ERR_DBG, "s2io: NAPI cannot be enabled when "
			  "MSI/MSI-X is enabled. Defaulting to INTA\n");
		*dev_intr_type = INTA;
	}
#endif
#ifndef CONFIG_PCI_MSI
	if (*dev_intr_type != INTA) {
		DBG_PRINT(ERR_DBG, "s2io: This kernel does not support"
			  "MSI-X. Defaulting to INTA\n");
		*dev_intr_type = INTA;
	}
#else
	if ((*dev_intr_type != INTA) && (*dev_intr_type != MSI_X) ) {
		DBG_PRINT(ERR_DBG, "s2io: Wrong intr_type requested. "
			  "Defaulting to INTA\n");
		*dev_intr_type = INTA;
	}
#endif
	if ((*dev_intr_type == MSI_X) &&
			((pdev->device != PCI_DEVICE_ID_HERC_WIN) &&
			(pdev->device != PCI_DEVICE_ID_HERC_UNI))) {
		DBG_PRINT(ERR_DBG, "s2io: Xframe I does not support MSI_X. "
					"Defaulting to INTA\n");
		*dev_intr_type = INTA;
	}
	if (rx_ring_mode != 1) {
		DBG_PRINT(ERR_DBG, "s2io: Requested ring mode not supported\n");
		DBG_PRINT(ERR_DBG, "s2io: Defaulting to 1-buffer mode\n");
		rx_ring_mode = 1;
	}

	if (lro != 0) {
		DBG_PRINT(ERR_DBG, "s2io: LRO is not supported\n");
		DBG_PRINT(ERR_DBG, "s2io: Disabling LRO\n");
		rx_ring_mode = 1;
	}

	return SUCCESS;
}

/**
 *  s2io_init_nic - Initialization of the adapter .
 *  @pdev : structure containing the PCI related information of the device.
 *  @pre: List of PCI devices supported by the driver listed in s2io_tbl.
 *  Description:
 *  The function initializes an adapter identified by the pci_dec structure.
 *  All OS related initialization including memory and device structure and
 *  initlaization of the device private variable is done. Also the swapper
 *  control register is initialized to enable read and write into the I/O
 *  registers of the device.
 *  Return value:
 *  returns 0 on success and negative on failure.
 */

static int __devinit
s2io_init_nic(struct pci_dev *pdev, const struct pci_device_id *pre)
{
	nic_t *sp;
	struct net_device *dev;
	int i, j, ret;
	int dma_flag = FALSE;
	u32 mac_up, mac_down;
	u64 val64 = 0, tmp64 = 0;
	XENA_dev_config_t __iomem *bar0 = NULL;
	u16 subid;
	mac_info_t *mac_control;
	struct config_param *config;
#ifdef SNMP_SUPPORT
	struct timeval tm;
#endif
	int mode;
	u8 dev_intr_type;

	/*
	 * Enable MSI-X and 8 Rx rings if "enable_netq"
	 * is set.
	 */
#ifdef __VMKERNEL_MODULE__
	if (enable_netq) {
		if (!intr_type) {
			intr_type = MSI_X;
		}
		if (!rx_ring_num) {
			rx_ring_num = 8;
		}
	}
#endif

	dev_intr_type = intr_type;
	if ((ret = s2io_verify_parm(pdev, &dev_intr_type))) {
		return ret;
	}

	if ((ret = pci_enable_device(pdev))) {
		DBG_PRINT(ERR_DBG,
			  "s2io_init_nic: pci_enable_device failed\n");
		return ret;
	}

	if (!pci_set_dma_mask(pdev, DMA_64BIT_MASK)) {
		DBG_PRINT(INIT_DBG, "s2io_init_nic: Using 64bit DMA\n");
		dma_flag = TRUE;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,00)) || \
		defined (CONFIG_IA64_SGI_SN2)
		if (pci_set_consistent_dma_mask
		    (pdev, DMA_64BIT_MASK)) {
			DBG_PRINT(ERR_DBG,
				  "Unable to obtain 64bit DMA for \
					consistent allocations\n");
			pci_disable_device(pdev);
			return -ENOMEM;
		}
#endif
	} else if (!pci_set_dma_mask(pdev, DMA_32BIT_MASK)) {
		DBG_PRINT(INIT_DBG, "s2io_init_nic: Using 32bit DMA\n");
	} else {
		pci_disable_device(pdev);
		return -ENOMEM;
	}

	if (!(request_mem_region(pci_resource_start(pdev, 0),
			pci_resource_len(pdev, 0), s2io_driver_name))) {
		DBG_PRINT(ERR_DBG, "bar0 Request Regions failed\n");
		pci_disable_device(pdev);
		return -ENODEV;
	}

	if (!(request_mem_region(pci_resource_start(pdev, 2),
			pci_resource_len(pdev, 2), s2io_driver_name))) {
		DBG_PRINT(ERR_DBG, "bar1 Request Regions failed\n");
		release_mem_region(pci_resource_start(pdev, 0),
		pci_resource_len(pdev, 0));
		pci_disable_device(pdev);
		return -ENODEV;
	}

	dev = alloc_etherdev(sizeof(nic_t));
	if (dev == NULL) {
		DBG_PRINT(ERR_DBG, "Device allocation failed\n");
		release_mem_region(pci_resource_start(pdev, 0),
			pci_resource_len(pdev, 0));
		release_mem_region(pci_resource_start(pdev, 2),
			pci_resource_len(pdev, 2));
		pci_disable_device(pdev);
		return -ENODEV;
	}

	pci_set_master(pdev);
	pci_set_drvdata(pdev, dev);
	SET_MODULE_OWNER(dev);
	SET_NETDEV_DEV(dev, &pdev->dev);

	/*  Private member variable initialized to s2io NIC structure */
	sp = dev->priv;
	memset(sp, 0, sizeof(nic_t));
	sp->dev = dev;
	sp->pdev = pdev;
	sp->high_dma_flag = dma_flag;
	sp->device_enabled_once = FALSE;
	if (rx_ring_mode == 1)
		sp->rxd_mode = RXD_MODE_1;
	else if (rx_ring_mode == 2)
		sp->rxd_mode = RXD_MODE_3B;
	else if (rx_ring_mode == 3)
		sp->rxd_mode = RXD_MODE_3A;

#ifdef __VMKERNEL_MODULE__
	sprintf(dev->name, "%s", pdev->name);
#endif

	pci_save_state(sp->pdev, sp->config_space);

	sp->intr_type = dev_intr_type;

	if ((pdev->device == PCI_DEVICE_ID_HERC_WIN) ||
		(pdev->device == PCI_DEVICE_ID_HERC_UNI))
		sp->device_type = XFRAME_II_DEVICE;
	else
		sp->device_type = XFRAME_I_DEVICE;

	sp->lro = lro;

	/* Initialize some PCI/PCI-X fields of the NIC. */
	s2io_init_pci(sp);

	/*
	 * Setting the device configuration parameters.
	 * Most of these parameters can be specified by the user during
	 * module insertion as they are module loadable parameters. If
	 * these parameters are not not specified during load time, they
	 * are initialized with default values.
	 */
	mac_control = &sp->mac_control;
	config = &sp->config;


	/* Tx side parameters. */
	config->tx_fifo_num = tx_fifo_num;
	for (i = 0; i < MAX_TX_FIFOS; i++) {
		config->tx_cfg[i].fifo_len = tx_fifo_len[i];
		config->tx_cfg[i].fifo_priority = i;
		atomic_set(&mac_control->fifos[i].isr_cnt, 0);
	}

	/* mapping the QoS priority to the configured fifos */
	for (i = 0; i < MAX_TX_FIFOS; i++)
		config->fifo_mapping[i] = fifo_map[config->tx_fifo_num][i];

	config->tx_intr_type = TXD_INT_TYPE_UTILZ;
	for (i = 0; i < config->tx_fifo_num; i++) {
		config->tx_cfg[i].f_no_snoop =
		    (NO_SNOOP_TXD | NO_SNOOP_TXD_BUFFER);
		if (config->tx_cfg[i].fifo_len < 65) {
			config->tx_intr_type = TXD_INT_TYPE_PER_LIST;
			break;
		}
	}
#ifndef CONFIG_AMD_FIX
	config->max_txds = MAX_SKB_FRAGS + 1;
#else
	config->max_txds = MAX_SKB_FRAGS + 2;
#endif
#ifdef NETIF_F_UFO
	config->max_txds++;
#endif
	/* Rx side parameters. */
	config->rx_ring_num = rx_ring_num;
	for (i = 0; i < MAX_RX_RINGS; i++) {
		config->rx_cfg[i].num_rxd = rx_ring_sz[i] *
		    (rxd_count[sp->rxd_mode] + 1);
		config->rx_cfg[i].ring_priority = i;
		atomic_set(&mac_control->rings[i].isr_cnt, 0);
		atomic_set(&mac_control->rings[i].rx_bufs_left, 0);
		mac_control->rings[i].pdev = sp->pdev;
		mac_control->rings[i].dev = sp->dev;
	}

	for (i = 0; i < rx_ring_num; i++) {
		config->rx_cfg[i].ring_org = RING_ORG_BUFF1;
		config->rx_cfg[i].f_no_snoop =
		    (NO_SNOOP_RXD | NO_SNOOP_RXD_BUFFER);
	}

#if defined(__VMKERNEL_MODULE__) && defined(__VMKNETDDI_QUEUEOPS__)
	spin_lock_init(&config->netqueue_lock);

	config->n_tx_fifo_allocated = 0;
	config->n_rx_ring_allocated = 0;
	for (i = 0; i < MAX_RX_RINGS; i++) {
		config->rx_cfg[i].allocated = FALSE;
		config->rx_cfg[i].active_filter_count = 0;
		for (j=0; j<S2IO_MAX_QUEUE_RXFILTERS; j++) {
			config->rx_cfg[i].filter[j].active = FALSE;
		}
	}
	for (i = 0; i < MAX_TX_FIFOS; i++) {
		config->tx_cfg[i].allocated = FALSE;
	}
#endif

	/* RTH Support initalization */
	if (rx_steering_type > 2)
		rx_steering_type = 0;
	config->rth_enable = rx_steering_type;
	for(i=0; ((i<num_rth_ports) && rth_ports[i]); i++)
		config->steer_ports[i] = rth_ports[i];
	config->rth_fn_and_mask = rth_fn_and_mask;
	config->port_type = port_type;

	/*  Setting Mac Control parameters */
	mac_control->rmac_pause_time = rmac_pause_time;
	mac_control->mc_pause_threshold_q0q3 = mc_pause_threshold_q0q3;
	mac_control->mc_pause_threshold_q4q7 = mc_pause_threshold_q4q7;


	/* Initialize the number of ISRs currently running */
	atomic_set(&sp->isr_cnt, 0);

	/*  initialize the shared memory used by the NIC and the host */
	if (init_shared_mem(sp)) {
		DBG_PRINT(INFO_DBG, "%s: Memory allocation failed\n",
			  dev->name);
		ret = -ENOMEM;
		goto mem_alloc_failed;
	}

	sp->bar0 = (caddr_t) ioremap(pci_resource_start(pdev, 0),
				     pci_resource_len(pdev, 0));
	if (!sp->bar0) {
		DBG_PRINT(ERR_DBG, "%s: Neterion: cannot remap io mem1\n",
			  dev->name);
		ret = -ENOMEM;
		goto bar0_remap_failed;
	}

	sp->bar1 = (caddr_t) ioremap(pci_resource_start(pdev, 2),
				     pci_resource_len(pdev, 2));
	if (!sp->bar1) {
		DBG_PRINT(ERR_DBG, "%s: Neterion: cannot remap io mem2\n",
			  dev->name);
		ret = -ENOMEM;
		goto bar1_remap_failed;
	}

	dev->irq = pdev->irq;
	dev->base_addr = (unsigned long) sp->bar0;

	/* Initializing the BAR1 address as the start of the FIFO pointer. */
	for (j = 0; j < MAX_TX_FIFOS; j++) {
		mac_control->tx_FIFO_start[j] = (TxFIFO_element_t *)
		    (sp->bar1 + (j * 0x00020000));
	}

	/*  Driver entry points */
	dev->open = &s2io_open;
	dev->stop = &s2io_close;
	dev->hard_start_xmit = &s2io_xmit;
	dev->get_stats = &s2io_get_stats;
	dev->set_multicast_list = &s2io_set_multicast;
	dev->do_ioctl = &s2io_ioctl;
	dev->change_mtu = &s2io_change_mtu;
#ifdef SET_ETHTOOL_OPS
	SET_ETHTOOL_OPS(dev, &netdev_ethtool_ops);
#endif

#if defined(__VMKERNEL_MODULE__) && defined(__VMKNETDDI_QUEUEOPS__)
	VMKNETDDI_REGISTER_QUEUEOPS(dev, s2io_netqueue_ops);
#endif

	dev->features |= NETIF_F_HW_VLAN_TX | NETIF_F_HW_VLAN_RX;
	dev->vlan_rx_register = s2io_vlan_rx_register;
	dev->vlan_rx_kill_vid = (void *)s2io_vlan_rx_kill_vid;

	/*
	 * will use eth_mac_addr() for  dev->set_mac_address
	 * mac address will be set every time dev->open() is called
	 */
#if defined(CONFIG_S2IO_NAPI)
	dev->poll = s2io_poll;
	dev->weight = 32;
#endif

	dev->features |= NETIF_F_SG | NETIF_F_IP_CSUM;
	if (sp->high_dma_flag == TRUE)
		dev->features |= NETIF_F_HIGHDMA;
#ifdef NETIF_F_TSO
	dev->features |= NETIF_F_TSO;
#endif
#ifdef NETIF_F_UFO
	if (sp->device_type & XFRAME_II_DEVICE) {
		dev->features |= NETIF_F_UFO;
		dev->features |= NETIF_F_HW_CSUM;
	}
#endif

	dev->tx_timeout = &s2io_tx_watchdog;
	dev->watchdog_timeo = WATCH_DOG_TIMEOUT;
	INIT_WORK(&sp->rst_timer_task,
		  (void (*)(void *)) s2io_restart_nic, dev);
	INIT_WORK(&sp->set_link_task,
		  (void (*)(void *)) s2io_set_link, sp);

	/* Setting swapper control on the NIC, for proper reset operation */
	if (s2io_set_swapper(sp)) {
		DBG_PRINT(ERR_DBG, "%s:swapper settings are wrong\n",
			  dev->name);
		ret = -EAGAIN;
		goto set_swap_failed;
	}

	/* Verify if the Herc works on the slot its placed into */
	if (sp->device_type & XFRAME_II_DEVICE) {
		mode = s2io_verify_pci_mode(sp);
		if (mode < 0) {
			DBG_PRINT(ERR_DBG, "%s: ", __FUNCTION__);
			DBG_PRINT(ERR_DBG, " Unsupported PCI bus mode\n");
			ret = -EBADSLT;
			goto set_swap_failed;
		}
	}

	/* Not needed for Herc */
	if (sp->device_type & XFRAME_I_DEVICE) {
		/*
		 * Fix for all "FFs" MAC address problems observed on
		 * Alpha platforms
		 */
		fix_mac_address(sp);
		s2io_reset(sp);
	}

	/*
	 * MAC address initialization.
	 * For now only one mac address will be read and used.
	 */
	bar0 = sp->bar0;
	val64 = RMAC_ADDR_CMD_MEM_RD | RMAC_ADDR_CMD_MEM_STROBE_NEW_CMD |
	    RMAC_ADDR_CMD_MEM_OFFSET(0 + MAC_MAC_ADDR_START_OFFSET);
	writeq(val64, &bar0->rmac_addr_cmd_mem);
	wait_for_cmd_complete(&bar0->rmac_addr_cmd_mem,
		      RMAC_ADDR_CMD_MEM_STROBE_CMD_EXECUTING,
			  S2IO_BIT_RESET);

	tmp64 = readq(&bar0->rmac_addr_data0_mem);
	mac_down = (u32) tmp64;
	mac_up = (u32) (tmp64 >> 32);

	memset(sp->def_mac_addr[0].mac_addr, 0, sizeof(ETH_ALEN));

	sp->def_mac_addr[0].mac_addr[3] = (u8) (mac_up);
	sp->def_mac_addr[0].mac_addr[2] = (u8) (mac_up >> 8);
	sp->def_mac_addr[0].mac_addr[1] = (u8) (mac_up >> 16);
	sp->def_mac_addr[0].mac_addr[0] = (u8) (mac_up >> 24);
	sp->def_mac_addr[0].mac_addr[5] = (u8) (mac_down >> 16);
	sp->def_mac_addr[0].mac_addr[4] = (u8) (mac_down >> 24);

	/*  Set the factory defined MAC address initially   */
	dev->addr_len = ETH_ALEN;
	memcpy(dev->dev_addr, sp->def_mac_addr, ETH_ALEN);

#ifdef CONFIG_PCI_MSI
	/* Store the values of the MSIX table in the s2io_nic structure */
	store_xmsi_data(sp);
#endif

	/* reset Nic and bring it to known state */
	s2io_reset(sp);

	/*
	 * Initialize the link state flags
	 * and the card state parameter
	 */
	atomic_set(&(sp->card_state), 0);
	sp->link_state = 0;

	/* Initialize spinlocks */
	for (i = 0; i < MAX_TX_FIFOS; i++)
		spin_lock_init(&mac_control->fifos[i].tx_lock);

	/*
	 * SXE-002: Configure link and activity LED to init state
	 * on driver load.
	 */
	subid = sp->pdev->subsystem_device;
	if ((subid & 0xFF) >= 0x07) {
		val64 = readq(&bar0->gpio_control);
		val64 |= 0x0000800000000000ULL;
		writeq(val64, &bar0->gpio_control);
		val64 = 0x0411040400000000ULL;
		writeq(val64, (void __iomem *) bar0 + 0x2700);
		val64 = readq(&bar0->gpio_control);
	}

	sp->rx_csum = 1;	/* Rx chksum verify enabled by default */

#ifdef SNMP_SUPPORT
	if (!s2io_bdsnmp_init(dev))
		DBG_PRINT(INIT_DBG,
			  "Error Creating Proc directory for SNMP\n");

	sp->nLinkStatus = 1;
#ifdef NETIF_F_TSO
	sp->nFeature = 1;
#endif
	memcpy(sp->cVersion, s2io_driver_version + 8, 20);
	memcpy(sp->cName, s2io_driver_name, 20);
	do_gettimeofday(&tm);
	sp->lDate = tm.tv_sec;
#endif

	if (register_netdev(dev)) {
		DBG_PRINT(ERR_DBG, "Device registration failed\n");
		ret = -ENODEV;
		goto register_failed;
	}

#ifdef __VMKERNEL_MODULE__
	/*
	 * task_sync lock required since vmkernel doesn't support
	 * flush_schedule_tasks function. FIXME in the future.
	 */
	spin_lock_init(&sp->task_sync_lock);
#endif

	s2io_vpd_read(sp);
	DBG_PRINT(ERR_DBG, "%s: Neterion %s",dev->name, sp->product_name);
	DBG_PRINT(ERR_DBG, "(rev %d), Driver version %s\n",
				get_xena_rev_id(sp->pdev),
				s2io_driver_version);
	DBG_PRINT(ERR_DBG, "Copyright(c) 2002-2005 Neterion Inc.\n");
	DBG_PRINT(ERR_DBG, "%s: MAC ADDR: "
			  "%02x:%02x:%02x:%02x:%02x:%02x\n", dev->name,
			  sp->def_mac_addr[0].mac_addr[0],
			  sp->def_mac_addr[0].mac_addr[1],
			  sp->def_mac_addr[0].mac_addr[2],
			  sp->def_mac_addr[0].mac_addr[3],
			  sp->def_mac_addr[0].mac_addr[4],
			  sp->def_mac_addr[0].mac_addr[5]);

	if (sp->device_type & XFRAME_II_DEVICE) {
		mode = s2io_print_pci_mode(sp);
		if (mode < 0) {
			DBG_PRINT(ERR_DBG, " Unsupported PCI bus mode\n");
			ret = -EBADSLT;
			unregister_netdev(dev);
			goto set_swap_failed;
		}
	}

	switch(sp->rxd_mode) {
		case RXD_MODE_1:
		    DBG_PRINT(ERR_DBG, "%s: 1-Buffer receive mode enabled\n",
						dev->name);
		    break;
		case RXD_MODE_3B:
		    DBG_PRINT(ERR_DBG, "%s: 2-Buffer receive mode enabled\n",
						dev->name);
		    break;
		case RXD_MODE_3A:
		    DBG_PRINT(ERR_DBG, "%s: 3-Buffer receive mode enabled\n",
						dev->name);
		    break;
	}
#ifdef CONFIG_S2IO_NAPI
	DBG_PRINT(ERR_DBG, "%s: NAPI enabled\n", dev->name);
#endif
	switch(sp->intr_type) {
		case INTA:
		    DBG_PRINT(ERR_DBG, "%s: Interrupt type INTA\n", dev->name);
		    break;
		case MSI_X:
		    DBG_PRINT(ERR_DBG, "%s: Interrupt type MSI-X\n", dev->name);
		    break;
	}
	if (sp->lro)
		DBG_PRINT(ERR_DBG, "%s: Large receive offload enabled\n",
			  dev->name);

	/* Initialize device name */
	sprintf(sp->name, "%s Neterion %s", dev->name, sp->product_name);

	/* Initialize bimodal Interrupts */
	sp->config.bimodal = bimodal;
	if (!(sp->device_type & XFRAME_II_DEVICE) && bimodal) {
		sp->config.bimodal = 0;
		DBG_PRINT(ERR_DBG,"%s:Bimodal intr not supported by Xframe I\n",
			dev->name);
	}

	/*
	 * Make Link state as off at this point, when the Link change
	 * interrupt comes the state will be automatically changed to
	 * the right state.
	 */
	netif_carrier_off(dev);

	return 0;

register_failed:
set_swap_failed:
	iounmap(sp->bar1);
bar1_remap_failed:
	iounmap(sp->bar0);
bar0_remap_failed:
mem_alloc_failed:
	free_shared_mem(sp);
	pci_disable_device(pdev);
	release_mem_region(pci_resource_start(pdev, 0),
		pci_resource_len(pdev, 0));
	release_mem_region(pci_resource_start(pdev, 2),
		pci_resource_len(pdev, 2));
	pci_set_drvdata(pdev, NULL);
	free_netdev(dev);

	return ret;
}

/**
 * s2io_rem_nic - Free the PCI device
 * @pdev: structure containing the PCI related information of the device.
 * Description: This function is called by the Pci subsystem to release a
 * PCI device and free up all resource held up by the device. This could
 * be in response to a Hot plug event or when the driver is to be removed
 * from memory.
 */

static void __devexit s2io_rem_nic(struct pci_dev *pdev)
{
	struct net_device *dev =
	    (struct net_device *) pci_get_drvdata(pdev);
	nic_t *sp;

	if (dev == NULL) {
		DBG_PRINT(ERR_DBG, "Driver Data is NULL!!\n");
		return;
	}

	sp = dev->priv;

#ifdef __VMKERNEL_MODULE__
	pci_restore_state(pdev, sp->config_space);
#endif

	unregister_netdev(dev);

	free_shared_mem(sp);
	iounmap(sp->bar0);
	iounmap(sp->bar1);
	release_mem_region(pci_resource_start(pdev, 0),
		pci_resource_len(pdev, 0));
	release_mem_region(pci_resource_start(pdev, 2),
		pci_resource_len(pdev, 2));
	pci_set_drvdata(pdev, NULL);
#ifdef SNMP_SUPPORT
	s2io_bdsnmp_rem(dev);
#endif

	free_netdev(dev);
	pci_disable_device(pdev);
}

/**
 * s2io_starter - Entry point for the driver
 * Description: This function is the entry point for the driver. It verifies
 * the module loadable parameters and initializes PCI configuration space.
 */

int __init s2io_starter(void)
{
#ifdef __VMKERNEL_MODULE__
        if (!vmk_set_module_version("%s", s2io_driver_version)) {
                return -ENODEV;
        }
#endif
	return pci_module_init(&s2io_driver);
}

/**
 * s2io_closer - Cleanup routine for the driver
 * Description: This function is the cleanup routine for the driver. It unregist * ers the driver.
 */

void s2io_closer(void)
{
	pci_unregister_driver(&s2io_driver);
	DBG_PRINT(INIT_DBG, "cleanup done\n");
}

module_init(s2io_starter);
module_exit(s2io_closer);

#ifdef SNMP_SUPPORT

/**
 * fnBaseDrv - Get the driver information
 * @pBaseDrv -Pointer to Base driver structure which contains the offset
 * and length of each of the field.
 * Description
 * This function copies the driver specific information from the dev structure
 * to the pBaseDrv stucture. It calculates the number of physical adapters by
 * parsing the dev_base global variable maintained by the kernel. This
 * variable has to read locked before accesing.This function is called by
 * fnBaseReadProc function.
 *
 */

static void fnBaseDrv(struct stBaseDrv *pBaseDrv, struct net_device *dev)
{

	struct pci_dev *pdev = NULL;
	struct net_device *ndev;
	int nCount = 0;
	nic_t *sp = (nic_t *) dev->priv;

	strncpy(pBaseDrv->m_cName, sp->cName, 20);
	strncpy(pBaseDrv->m_cVersion, sp->cVersion, 20);
	pBaseDrv->m_nStatus = sp->nLinkStatus;
	pBaseDrv->m_nFeature = sp->nFeature;
	pBaseDrv->m_nMemorySize = sp->nMemorySize;
	sprintf(pBaseDrv->m_cDate, "%ld", sp->lDate);
	/*
	 * Find all the ethernet devices on the system using
	 * pci_find_class. Get the private data which will be the
	 * net_device structure assigned by the driver.
	 */
	while ((pdev =
		pci_find_class((PCI_CLASS_NETWORK_ETHERNET << 8), pdev))) {
		ndev = (struct net_device *) pci_get_drvdata(pdev);
		if (ndev == NULL)
			break;
		memcpy(pBaseDrv->m_stPhyAdap[nCount].m_cName, ndev->name,
		       20);
		pBaseDrv->m_stPhyAdap[nCount].m_nIndex = ndev->ifindex;
		nCount++;
	}
	pBaseDrv->m_nPhyCnt = nCount;
}

/*
 *  fnBaseReadProc - Read entry point for the proc file
 *  @page - Buffer pointer where the data is written
 *  @start- Pointer to buffer ptr . It is used if the data is more than a page
 *  @off- the offset to the page where data is written
 *  @count - number of bytes to write
 *  @eof - to indicate end of file
 *  @data - pointer to device structure.
 *
 * Description -
 * This function gets  Base driver specific information from the fnBaseDrv
 * function and writes into the BDInfo file. This function is called whenever
 * the user reads the file. The length of data written cannot exceed 4kb.
 * If it exceeds then use the start pointer to write multiple pages
 * Return - the length of the string written to proc file
 */
static int fnBaseReadProc(char *page, char **start, off_t off, int count,
			  int *eof, void *data)
{
	struct stBaseDrv *pBaseDrv;
	int nLength = 0;
	int nCount = 0;
	struct net_device *dev = (struct net_device *) data;
	int nIndex = 0;

	pBaseDrv = kmalloc(sizeof(struct stBaseDrv), GFP_KERNEL);
	if (pBaseDrv == NULL) {
		printk("Error allocating memory\n");
		return -ENOMEM;
	}
	fnBaseDrv(pBaseDrv, dev);
	sprintf(page + nLength, "%-30s%-20s\n", "Base Driver Name",
		pBaseDrv->m_cName);
	nLength += 51;
	if (pBaseDrv->m_nStatus == 1)
		sprintf(page + nLength, "%-30s%-20s\n", "Load Status",
			"Loaded");
	else
		sprintf(page + nLength, "%-30s%-20s\n", "Load Status",
			"UnLoaded");
	nLength += 51;
	sprintf(page + nLength, "%-30s%-20s\n", "Base Driver Version",
		pBaseDrv->m_cVersion);
	nLength += 51;

	sprintf(page + nLength, "%-30s%-20d\n", "Feature Supported",
		pBaseDrv->m_nFeature);
	nLength += 51;

	sprintf(page + nLength, "%-30s%-20d\n",
		"Base Driver Memrory in Bytes", pBaseDrv->m_nMemorySize);
	nLength += 51;

	sprintf(page + nLength, "%-30s%-20s\n", "Base Driver Date",
		pBaseDrv->m_cDate);
	nLength += 51;

	sprintf(page + nLength, "%-30s%-20d\n", "No of Phy Adapter",
		pBaseDrv->m_nPhyCnt);
	nLength += 51;
	sprintf(page + nLength, "%-20s%-20s\n\n", "Phy Adapter Index",
		"Phy Adapter Name");
	nLength += 42;

	for (nIndex = 0, nCount = pBaseDrv->m_nPhyCnt; nCount != 0;
	     nCount--, nIndex++) {
		sprintf(page + nLength, "%-20d%-20s\n",
			pBaseDrv->m_stPhyAdap[nIndex].m_nIndex,
			pBaseDrv->m_stPhyAdap[nIndex].m_cName);
		nLength += 41;
	}

	*eof = 1;
	kfree(pBaseDrv);
	return nLength;
}

/*
 *  fnPhyAdapReadProc - Read entry point for the proc file
 *  @page - Buffer pointer where the data is written
 *  @start- Pointer to buffer ptr . It is used if the data is more than a page
 *  @off- the offset to the page where data is written
 *  @count - number of bytes to write
 *  @eof - to indicate end of file
 *  @data - pointer to device structure.
 *
 * Description -
 * This function gets  physical adapter information. This function is called
 * whenever the use* r reads the file. The length of data written cannot
 * exceed 4kb. If it exceeds then use the start pointer to write multiple page
 *
 * Return - the length of the string written to proc file
 */
static int fnPhyAdapReadProc(char *page, char **start, off_t off,
			     int count, int *eof, void *data)
{

	struct stPhyData *pPhyData;
	pPhyData = kmalloc(sizeof(struct stPhyData), GFP_KERNEL);
	if (pPhyData == NULL) {
		printk("Error allocating memory\n");
		return -ENOMEM;
	}

	struct net_device *pNetDev;
	struct net_device_stats *pNetStat;
	int nLength = 0;
	unsigned char cMAC[20];

	/* Print the header in the PhyAdap proc file */
	sprintf(page + nLength,
		"%-10s%-22s%-10s%-10s%-22s%-22s%-10s%-10s%-10s%-10s%-10s%-10s%-10s%-10s%-10s%-10s%-10s%-10s%-10s%-10s\n",
		"Index", "Description", "Mode", "Type", "Speed", "MAC",
		"Status", "Slot", "Bus", "IRQ", "Colis", "Multi",
		"RxBytes", "RxDrop", "RxError", "RxPacket", "TRxBytes",
		"TRxDrop", "TxError", "TxPacket");

	/* 237 is the lenght of the above string copied in the page  */
	nLength += 237;

	struct pci_dev *pdev = NULL;
	/*
	 * pci_find_class will return to the pointer to the pdev structure
	 * for all the network devices using PCI_CLASS_NETWORK_ETHERNET.
	 * The third argument is pointer to previous pdev structure.Initailly
	 * it has to be null
	 */
	while ((pdev =
		pci_find_class((PCI_CLASS_NETWORK_ETHERNET << 8), pdev))) {
		/* Private data will point to the netdevice structure */
		pNetDev = (struct net_device *) pci_get_drvdata(pdev);
		if (pNetDev == NULL)
			continue;
		if (pNetDev->addr_len != 0) {
			pNetStat = pNetDev->get_stats(pNetDev);
			pPhyData->m_nIndex = pNetDev->ifindex;
			memcpy(pPhyData->m_cDesc, pNetDev->name, 20);
			pPhyData->m_nMode = 0;
			pPhyData->m_nType = 0;
			switch (pPhyData->m_nType) {
				/* case IFT_ETHER:
				   memcpy(pPhyData->m_cSpeed,"10000000",20);
				   break;

				   case 9:
				   memcpy(pPhyData->m_cSpeed,"4000000",20);
				   break; */
			default:
				memcpy(pPhyData->m_cSpeed, "10000000", 20);
				break;
			}
			memcpy(pPhyData->m_cPMAC, pNetDev->dev_addr,
			       ETH_ALEN);
			memcpy(pPhyData->m_cCMAC, pNetDev->dev_addr,
			       ETH_ALEN);
			pPhyData->m_nLinkStatus =
			    test_bit(__LINK_STATE_START, &pNetDev->state);
			pPhyData->m_nPCISlot = PCI_SLOT(pdev->devfn);
			pPhyData->m_nPCIBus = pdev->bus->number;
			pPhyData->m_nIRQ = pNetDev->irq;
			pPhyData->m_nCollision = pNetStat->collisions;
			pPhyData->m_nMulticast = pNetStat->multicast;

			pPhyData->m_nRxBytes = pNetStat->rx_bytes;
			pPhyData->m_nRxDropped = pNetStat->rx_dropped;
			pPhyData->m_nRxErrors = pNetStat->rx_errors;
			pPhyData->m_nRxPackets = pNetStat->rx_packets;

			pPhyData->m_nTxBytes = pNetStat->tx_bytes;
			pPhyData->m_nTxDropped = pNetStat->tx_dropped;
			pPhyData->m_nTxErrors = pNetStat->tx_errors;
			pPhyData->m_nTxPackets = pNetStat->tx_packets;

			sprintf(cMAC, "%02x:%02x:%02x:%02x:%02x:%02x",
				pPhyData->m_cPMAC[0], pPhyData->m_cPMAC[1],
				pPhyData->m_cPMAC[2], pPhyData->m_cPMAC[3],
				pPhyData->m_cPMAC[4],
				pPhyData->m_cPMAC[5]);

			sprintf(page + nLength,
				"%-10d%-22s%-10d%-10d%-22s%-22s%-10d%-10d%-10d%-10d%-10d%-10d%-10d%-10d%-10d%-10d%-10d%-10d%-10d%-10d\n",
				pPhyData->m_nIndex, pPhyData->m_cDesc,
				pPhyData->m_nMode, pPhyData->m_nType,
				pPhyData->m_cSpeed, cMAC,
				pPhyData->m_nLinkStatus,
				pPhyData->m_nPCISlot, pPhyData->m_nPCIBus,
				pPhyData->m_nIRQ, pPhyData->m_nCollision,
				pPhyData->m_nMulticast,
				pPhyData->m_nRxBytes,
				pPhyData->m_nRxDropped,
				pPhyData->m_nRxErrors,
				pPhyData->m_nRxPackets,
				pPhyData->m_nTxBytes,
				pPhyData->m_nTxDropped,
				pPhyData->m_nTxErrors,
				pPhyData->m_nTxPackets);
			nLength += 237;
		}
	}

	*eof = 1;

	kfree(pPhyData);
	return nLength;
}

/*
 * s2io_bdsnmp_init - Entry point to create proc file
 * @dev-  Pointer to net device structure passed by the driver.
 * Return
 * Success If creates all the files
 * ERROR_PROC_ENTRY /ERROR_PROC_DIR Error If could not create all the files
 * Description
 * This functon is called when the driver is loaded. It creates the S2IO
 * proc file system in the /proc/net/ directory. This directory is used to
 * store the info about the base driver afm driver, lacp, vlan  and nplus.
 * It checks if S2IO directory already exists else creates it and creates
 * the files BDInfo  files and assiciates read function to each of the files.
 */

int s2io_bdsnmp_init(struct net_device *dev)
{
	struct proc_dir_entry *S2ioDir;
	struct proc_dir_entry *BaseDrv;
	struct proc_dir_entry *PhyAdap;
	int nLength = 0;

	nLength = strlen(S2IODIRNAME);
	/* IF the directory already exists then just return */
	for (S2ioDir = proc_net->subdir; S2ioDir != NULL;
	     S2ioDir = S2ioDir->next) {
		if ((S2ioDir->namelen == nLength)
		    && (!memcmp(S2ioDir->name, S2IODIRNAME, nLength)))
			break;
	}
	if (S2ioDir == NULL)
		/* Create the s2io directory */
		if (!
		    (S2ioDir =
		     create_proc_entry(S2IODIRNAME, S_IFDIR, proc_net))) {
			DBG_PRINT(INIT_DBG,
				  "Error Creating Proc directory for SNMP\n");
			return ERROR_PROC_DIR;
		}
	/* Create the BDInfo file to store driver info and associate read funtion */
	if (!
	    (BaseDrv =
	     create_proc_read_entry(BDFILENAME, S_IFREG | S_IRUSR, S2ioDir,
				    fnBaseReadProc, (void *) dev))) {
		DBG_PRINT(INIT_DBG,
			  "Error Creating Proc File for Base Drvr\n");
		return ERROR_PROC_ENTRY;
	}
	if (!
	    (PhyAdap =
	     create_proc_read_entry(PADAPFILENAME, S_IFREG | S_IRUSR,
				    S2ioDir, fnPhyAdapReadProc,
				    (void *) dev))) {
		DBG_PRINT(INIT_DBG,
			  "Error Creating Proc File for Phys Adap\n");
		return ERROR_PROC_ENTRY;
	}


	return SUCCESS;
}

/**
 * s2io_bdsnmp_rem : Removes the proc file entry
 * @dev - pointer to netdevice structre
 * Return - void
 * Description
 * This functon is called when the driver is Unloaded. It checks if the
 * S2IO directoy exists and deletes the files in the reverse order of
 * creation.
 */

void s2io_bdsnmp_rem(struct net_device *dev)
{
	int nLength = 0;
	struct proc_dir_entry *S2ioDir;
	nLength = strlen(S2IODIRNAME);
	/*
	 * Check if the S2IO directory exists or not and then delete
	 * all the files in the S2IO Directory
	 */
	for (S2ioDir = proc_net->subdir; S2ioDir != NULL;
	     S2ioDir = S2ioDir->next) {
		if ((S2ioDir->namelen == nLength)
		    && (!memcmp(S2ioDir->name, S2IODIRNAME, nLength)))

			break;
	}
	if (S2ioDir == NULL)
		return;
	remove_proc_entry(BDFILENAME, S2ioDir);
	remove_proc_entry(PADAPFILENAME, S2ioDir);
	if (S2ioDir->subdir == NULL) {
		remove_proc_entry(S2IODIRNAME, proc_net);
	}
}
#endif

static int
check_L2_lro_capable(u8 *buffer, struct iphdr **ip, struct tcphdr **tcp,
		     RxD_t *rxdp)
{
	int ip_off;
	u8 l2_type = (u8)((rxdp->Control_1 >> 37) & 0x7), ip_len;

	if (!(rxdp->Control_1 & RXD_FRAME_PROTO_TCP)) {
		DBG_PRINT(INIT_DBG,"%s: Non-TCP frames not supported for LRO\n",
			  __FUNCTION__);
		return -1;
	}

	/* TODO:
	 * By default the VLAN field in the MAC is stripped by the card, if this
	 * feature is turned off in rx_pa_cfg register, then the ip_off field
	 * has to be shifted by a further 2 bytes
	 */
	switch (l2_type) {
		case 0: /* DIX type */
		case 4: /* DIX type with VLAN */
			ip_off = HEADER_ETHERNET_II_802_3_SIZE;
			break;
		/* LLC, SNAP etc are considered non-mergeable */
		default:
			return -1;
	}

	*ip = (struct iphdr *)((u8 *)buffer + ip_off);
	ip_len = (u8)((*ip)->ihl);
	ip_len <<= 2;
	*tcp = (struct tcphdr *)((unsigned long)*ip + ip_len);

	return 0;
}

static int check_for_socket_match(lro_t *lro, struct iphdr *ip,
				  struct tcphdr *tcp)
{
	DBG_PRINT(INFO_DBG,"%s: Been here...\n", __FUNCTION__);
	if ((lro->iph->saddr != ip->saddr) || (lro->iph->daddr != ip->daddr) ||
	   (lro->tcph->source != tcp->source) || (lro->tcph->dest != tcp->dest))
		return -1;
	return 0;
}

static inline int get_l4_pyld_length(struct iphdr *ip, struct tcphdr *tcp)
{
	return(ntohs(ip->tot_len) - (ip->ihl << 2) - (tcp->doff << 2));
}

static void initiate_new_session(lro_t *lro, u8 *l2h,
		     struct iphdr *ip, struct tcphdr *tcp, u32 tcp_pyld_len)
{
	DBG_PRINT(INFO_DBG,"%s: Been here...\n", __FUNCTION__);
	lro->l2h = l2h;
	lro->iph = ip;
	lro->tcph = tcp;
	lro->tcp_next_seq = tcp_pyld_len + ntohl(tcp->seq);
	lro->tcp_ack = ntohl(tcp->ack_seq);
	lro->sg_num = 1;
	lro->total_len = ntohs(ip->tot_len);
	lro->frags_len = 0;
	/*
	 * check if we saw TCP timestamp. Other consistency checks have
	 * already been done.
 	 */
	if (tcp->doff == 8) {
		u32 *ptr;
		ptr = (u32 *)(tcp+1);
		lro->saw_ts = 1;
		lro->cur_tsval = *(ptr+1);
		lro->cur_tsecr = *(ptr+2);
	}
	lro->in_use = 1;
}

static void update_L3L4_header(nic_t *sp, lro_t *lro)
{
	struct iphdr *ip = lro->iph;
	struct tcphdr *tcp = lro->tcph;
	u16 nchk;
	StatInfo_t *statinfo = sp->mac_control.stats_info;
	DBG_PRINT(INFO_DBG,"%s: Been here...\n", __FUNCTION__);

	/* Update L3 header */
	ip->tot_len = htons(lro->total_len);
	ip->check = 0;
	nchk = ip_fast_csum((u8 *)lro->iph, ip->ihl);
	ip->check = nchk;

	/* Update L4 header */
	tcp->ack_seq = lro->tcp_ack;
	tcp->window = lro->window;

	/* Update tsecr field if this session has timestamps enabled */
	if (lro->saw_ts) {
		u32 *ptr = (u32 *)(tcp + 1);
		*(ptr+2) = lro->cur_tsecr;
	}

	/* Update counters required for calculation of
	 * average no. of packets aggregated.
	 */
	statinfo->sw_stat.sum_avg_pkts_aggregated += lro->sg_num;
	statinfo->sw_stat.num_aggregations++;
}

static void aggregate_new_rx(lro_t *lro, struct iphdr *ip,
		struct tcphdr *tcp, u32 l4_pyld)
{
	DBG_PRINT(INFO_DBG,"%s: Been here...\n", __FUNCTION__);
	lro->total_len += l4_pyld;
	lro->frags_len += l4_pyld;
	lro->tcp_next_seq += l4_pyld;
	lro->sg_num++;

	/* Update ack seq no. and window ad(from this pkt) in LRO object */
	lro->tcp_ack = tcp->ack_seq;
	lro->window = tcp->window;

	if (lro->saw_ts) {
		u32 *ptr;
		/* Update tsecr and tsval from this packet */
		ptr = (u32 *) (tcp + 1);
		lro->cur_tsval = *(ptr + 1);
		lro->cur_tsecr = *(ptr + 2);
	}
}

static int verify_l3_l4_lro_capable(lro_t *l_lro, struct iphdr *ip,
				    struct tcphdr *tcp, u32 tcp_pyld_len)
{
	u8 *ptr;
	DBG_PRINT(INFO_DBG,"%s: Been here...\n", __FUNCTION__);

	if (!tcp_pyld_len) {
		/* Runt frame or a pure ack */
		return -1;
	}

	if (ip->ihl != 5) /* IP has options */
		return -1;

	if (tcp->urg || tcp->psh || tcp->rst || tcp->syn || tcp->fin ||
								!tcp->ack) {
		/*
		 * Currently recognize only the ack control word and
		 * any other control field being set would result in
		 * flushing the LRO session
		 */
		return -1;
	}

	/*
	 * Allow only one TCP timestamp option. Don't aggregate if
	 * any other options are detected.
	 */
	if (tcp->doff != 5 && tcp->doff != 8)
		return -1;

	if (tcp->doff == 8) {
		ptr = (u8 *)(tcp + 1);
		while (*ptr == TCPOPT_NOP)
			ptr++;
		if (*ptr != TCPOPT_TIMESTAMP || *(ptr+1) != TCPOLEN_TIMESTAMP)
			return -1;

		/* Ensure timestamp value increases monotonically */
		if (l_lro)
			if (l_lro->cur_tsval > *((u32 *)(ptr+2)))
				return -1;

		/* timestamp echo reply should be non-zero */
		if (*((u32 *)(ptr+6)) == 0)
			return -1;
	}

	return 0;
}

static int
s2io_club_tcp_session(u8 *buffer, u8 **tcp, u32 *tcp_len, lro_t **lro,
		      RxD_t *rxdp, nic_t *sp)
{
	struct iphdr *ip;
	struct tcphdr *tcph;
	int ret = 0, i;

	if (!(ret = check_L2_lro_capable(buffer, &ip, (struct tcphdr **)tcp,
					 rxdp))) {
		DBG_PRINT(INFO_DBG,"IP Saddr: %x Daddr: %x\n",
			  ip->saddr, ip->daddr);
	} else {
		return ret;
	}

	tcph = (struct tcphdr *)*tcp;
	*tcp_len = get_l4_pyld_length(ip, tcph);
	for (i=0; i<MAX_LRO_SESSIONS; i++) {
		lro_t *l_lro = &sp->lro0_n[i];
		if (l_lro->in_use) {
			if (check_for_socket_match(l_lro, ip, tcph))
				continue;
			/* Sock pair matched */
			*lro = l_lro;

			if ((*lro)->tcp_next_seq != ntohl(tcph->seq)) {
				DBG_PRINT(INFO_DBG, "%s:Out of order. expected "
					  "0x%x, actual 0x%x\n", __FUNCTION__,
					  (*lro)->tcp_next_seq,
					  ntohl(tcph->seq));

				sp->mac_control.stats_info->
				   sw_stat.outof_sequence_pkts++;
				ret = 2;
				break;
			}

			if (!verify_l3_l4_lro_capable(l_lro, ip, tcph,*tcp_len))
				ret = 1; /* Aggregate */
			else
				ret = 2; /* Flush both */
			break;
		}
	}

	if (ret == 0) {
		/* Before searching for available LRO objects,
		 * check if the pkt is L3/L4 aggregatable. If not
		 * don't create new LRO session. Just send this
		 * packet up.
		 */
		if (verify_l3_l4_lro_capable(NULL, ip, tcph, *tcp_len)) {
			return 5;
		}

		for (i=0; i<MAX_LRO_SESSIONS; i++) {
			lro_t *l_lro = &sp->lro0_n[i];
			if (!(l_lro->in_use)) {
				*lro = l_lro;
				ret = 3; /* Begin anew */
				break;
			}
		}
	}

	if (ret == 0) { /* sessions exceeded */
		DBG_PRINT(INFO_DBG,"%s:All LRO sessions already in use\n",
			  __FUNCTION__);
		*lro = NULL;
		return ret;
	}

	switch (ret) {
		case 3:
			initiate_new_session(*lro, buffer, ip, tcph, *tcp_len);
			break;
		case 2:
			update_L3L4_header(sp, *lro);
			break;
		case 1:
			aggregate_new_rx(*lro, ip, tcph, *tcp_len);
			if ((*lro)->sg_num == sp->lro_max_aggr_per_sess) {
				update_L3L4_header(sp, *lro);
				ret = 4; /* Flush the LRO */
			}
			break;
		default:
			DBG_PRINT(ERR_DBG,"%s:Dont know, can't say!!\n",
				__FUNCTION__);
			break;
	}

	return ret;
}

static void clear_lro_session(lro_t *lro)
{
	static u16 lro_struct_size = sizeof(lro_t);

	memset(lro, 0, lro_struct_size);
}

static void queue_rx_frame(struct sk_buff *skb)
{
	struct net_device *dev = skb->dev;

	skb->protocol = eth_type_trans(skb, dev);
#ifdef CONFIG_S2IO_NAPI
	netif_receive_skb(skb);
#else
	netif_rx(skb);
#endif
}

static void lro_append_pkt(nic_t *sp, lro_t *lro, struct sk_buff *skb,
			   u32 tcp_len)
{
	struct sk_buff *tmp, *first = lro->parent;

	first->len += tcp_len;
	first->data_len = lro->frags_len;
	skb_pull(skb, (skb->len - tcp_len));
	if ((tmp = skb_shinfo(first)->frag_list)) {
		while (tmp->next)
			tmp = tmp->next;
		tmp->next = skb;
	}
	else
		skb_shinfo(first)->frag_list = skb;
	sp->mac_control.stats_info->sw_stat.clubbed_frms_cnt++;
	return;
}

#if defined(__VMKERNEL_MODULE__) && defined(__VMKNETDDI_QUEUEOPS__)

static int
s2io_set_mac_addr_offset(struct net_device *dev, u8 *addr, int offset)
{
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	register u64 val64, mac_addr = 0;
	int i;

	if (offset >= MAX_ADDRS_SUPPORTED) {
		DBG_PRINT(ERR_DBG, "%s: set_mac_addr_offset failed - "
		"offset (%d) > max supported (%d)\n",
		dev->name, offset, MAX_ADDRS_SUPPORTED);
		return FAILURE;
	}

	for (i = 0; i < ETH_ALEN; i++) {
		mac_addr <<= 8;
		mac_addr |= addr[i];
	}

	writeq(RMAC_ADDR_DATA0_MEM_ADDR(mac_addr),
	       &bar0->rmac_addr_data0_mem);

	val64 =
	      RMAC_ADDR_CMD_MEM_WE | RMAC_ADDR_CMD_MEM_STROBE_NEW_CMD |
	      RMAC_ADDR_CMD_MEM_OFFSET(offset);
	writeq(val64, &bar0->rmac_addr_cmd_mem);

	/* Wait till command completes */
	if (wait_for_cmd_complete(&bar0->rmac_addr_cmd_mem,
		RMAC_ADDR_CMD_MEM_STROBE_CMD_EXECUTING,
			  S2IO_BIT_RESET)) {
		DBG_PRINT(ERR_DBG, "%s: set_mac_addr failed\n", dev->name);
		return FAILURE;
	}

	return SUCCESS;
}

int s2io_set_mac_addr(struct net_device *dev, u8 * addr)
{
	return s2io_set_mac_addr_offset(dev, addr, 0);
}

static int
s2io_rts_mac_disable(struct net_device *dev, int index)
{
	int rval;
	u8 broadcast_macaddr[ETH_ALEN] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

	if ((rval = s2io_set_mac_addr_offset(dev, broadcast_macaddr, index))
			!= SUCCESS) {
		return rval;
	}

	return SUCCESS;
}

static int
s2io_rts_mac_enable(struct net_device *dev, int index, u8 *macaddr)
{
	nic_t *sp = dev->priv;
	XENA_dev_config_t __iomem *bar0 = sp->bar0;
	register u64 val64;
	int section, rval;

	if (index >= MAX_ADDRS_SUPPORTED) {
		DBG_PRINT(ERR_DBG, "%s: s2io_rts_mac_enable failed - "
		"index (%d) > max supported (%d)\n",
		dev->name, index, MAX_ADDRS_SUPPORTED);
		return FAILURE;
	}

	if ((rval = s2io_set_mac_addr_offset(dev, macaddr, index)) != SUCCESS) {
		return rval;
	}

	section = index/32;
	DBG_PRINT(INFO_DBG, "section %d enabled\n", section);

	val64 = readq(&bar0->rts_da_cfg);

	switch (section) {
		case 0:
			val64 |= RTS_DA_SECT0_EN;
			break;
		case 1:
			val64 |= RTS_DA_SECT1_EN;
			break;
		case 2:
			val64 |= RTS_DA_SECT2_EN;
			break;
		case 3:
			val64 |= RTS_DA_SECT3_EN;
			break;
		case 4:
			val64 |= RTS_DA_SECT4_EN;
			break;
		case 5:
			val64 |= RTS_DA_SECT5_EN;
			break;
		case 6:
			val64 |= RTS_DA_SECT6_EN;
			break;
		case 7:
			val64 |= RTS_DA_SECT7_EN;
			break;
		default:
			DBG_PRINT(ERR_DBG, "invalid section %d ", section);
			break;
	}

	writeq(val64,  &bar0->rts_da_cfg);

	/* Enabled enhanced RTS steering */
	val64 = readq(&bar0->rts_ctrl);
	val64 |= RTS_CTRL_ENHANCED;
	writeq(val64, &bar0->rts_ctrl);

	return SUCCESS;
}

static int
s2io_get_netqueue_features(vmknetddi_queueop_get_features_args_t *args)
{
	args->features = VMKNETDDI_QUEUEOPS_FEATURE_NONE;
	args->features |= VMKNETDDI_QUEUEOPS_FEATURE_RXQUEUES;
	return VMKNETDDI_QUEUEOPS_OK;
}

static int
s2io_get_queue_count(vmknetddi_queueop_get_queue_count_args_t *args)
{
	nic_t *nic = args->netdev->priv;
	struct config_param *config = &nic->config;

	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_TX) {
		if (config->tx_fifo_num > 1) {
			args->count = config->tx_fifo_num - 1;
		}
		else {
			args->count = 0;
		}
		return VMKNETDDI_QUEUEOPS_OK;
	}
	else if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		if (config->rx_ring_num > 1) {
			args->count = config->rx_ring_num - 1;
		}
		else {
			args->count = 0;
		}
		return VMKNETDDI_QUEUEOPS_OK;
	}
	else {
		printk("s2io_get_channel_count: invalid queue type\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
s2io_get_filter_count(vmknetddi_queueop_get_filter_count_args_t *args)
{
	/*
	 * Tx does not support filters
	 */
	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_TX) {
		printk("s2io_get_filter_count: tx queues not supported\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	args->count = S2IO_MAX_QUEUE_RXFILTERS;
	return VMKNETDDI_QUEUEOPS_OK;
}

static int
s2io_validate_queueid(struct net_device *netdev,
			vmknetddi_queueops_queueid_t qid)
{
	nic_t *nic = netdev->priv;
	struct config_param *config = &nic->config;
	u16 cidx = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(qid);

	if (VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(qid)) {
		if (cidx > config->rx_ring_num) {
			return VMKNETDDI_QUEUEOPS_ERR;
		}
		else {
			return VMKNETDDI_QUEUEOPS_OK;
		}
	}
	else if (VMKNETDDI_QUEUEOPS_IS_TX_QUEUEID(qid)) {
		if (cidx > config->tx_fifo_num) {
			return VMKNETDDI_QUEUEOPS_ERR;
		}
		else {
			return VMKNETDDI_QUEUEOPS_OK;
		}
	}
	else {
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
s2io_validate_filterid(struct net_device *netdev,
		       vmknetddi_queueops_filterid_t filterid)
{
	u16 fidx = VMKNETDDI_QUEUEOPS_FILTERID_VAL(filterid);

	if (fidx > S2IO_MAX_QUEUE_RXFILTERS) {
		return VMKNETDDI_QUEUEOPS_ERR;
	}
	else {
		return VMKNETDDI_QUEUEOPS_OK;
	}
}

static int
s2io_alloc_tx_queue(struct net_device *netdev,
		      vmknetddi_queueops_queueid_t *p_qid)
{
	//XXX: not implemented yet
	printk("s2io_alloc_tx_queue: tx queues not supported\n");
	return VMKNETDDI_QUEUEOPS_ERR;
}

static int
s2io_alloc_rx_queue(struct net_device *netdev,
		      vmknetddi_queueops_queueid_t *p_qid)
{
	nic_t *nic = netdev->priv;
	struct config_param *config = &nic->config;

	if (config->n_rx_ring_allocated >= config->rx_ring_num) {
		printk("s2io_alloc_rx_queue: no free rx queues\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
	else {
		int i;
		spin_lock(&config->netqueue_lock);
		for (i=1; i < config->rx_ring_num; i++) {
			if (!config->rx_cfg[i].allocated) {
				config->rx_cfg[i].allocated = TRUE;
				*p_qid = VMKNETDDI_QUEUEOPS_MK_RX_QUEUEID(i);
				config->n_rx_ring_allocated++;
				spin_unlock(&config->netqueue_lock);
				return VMKNETDDI_QUEUEOPS_OK;
			}
		}
		spin_unlock(&config->netqueue_lock);

		printk("s2io_alloc_rx_queue: no free rx queues found!\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
s2io_alloc_queue(vmknetddi_queueop_alloc_queue_args_t *args)
{
	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_TX) {
		return s2io_alloc_tx_queue(args->netdev, &args->queueid);
	}
	else if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		return s2io_alloc_rx_queue(args->netdev, &args->queueid);
	}
	else {
		printk("s2io_alloc_queue: invalid queue type\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
s2io_free_tx_queue(struct net_device *netdev,
		     vmknetddi_queueops_queueid_t qid)
{
	//XXX: not implemented yet
	printk("s2io_free_tx_queue: not implemented\n");
	return VMKNETDDI_QUEUEOPS_ERR;
}

static int
s2io_free_rx_queue(struct net_device *netdev,
		     vmknetddi_queueops_queueid_t qid)
{
	nic_t *nic = netdev->priv;
	struct config_param *config = &nic->config;
	u16 queue = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(qid);


	if (!config->rx_cfg[queue].allocated) {
		printk("s2io_free_rx_queue: rx queue not allocated\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	spin_lock(&config->netqueue_lock);
	config->rx_cfg[queue].allocated = FALSE;
	config->n_rx_ring_allocated--;
	spin_unlock(&config->netqueue_lock);

	return VMKNETDDI_QUEUEOPS_OK;
}

static int
s2io_free_queue(vmknetddi_queueop_free_queue_args_t *args)
{
	if (s2io_validate_queueid(args->netdev, args->queueid)
						!= VMKNETDDI_QUEUEOPS_OK) {
		printk("s2io_free_queue: failed to validate queue id\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	if (VMKNETDDI_QUEUEOPS_IS_TX_QUEUEID(args->queueid)) {
		return s2io_free_tx_queue(args->netdev, args->queueid);
	}
	else if (VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid)) {
		return s2io_free_rx_queue(args->netdev, args->queueid);
	}
	else {
		printk("s2io_free_queue: invalid queue type\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
s2io_get_queue_vector(vmknetddi_queueop_get_queue_vector_args_t *args)
{
	//XXX: not implemented yet
	printk("s2io_get_queue_vector: not implemented\n");
	return VMKNETDDI_QUEUEOPS_ERR;
}

static int
s2io_get_default_queue(vmknetddi_queueop_get_default_queue_args_t *args)
{
	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_TX) {
		/* not implemented yet */
		printk("s2io_get_default_queue: not implemented\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
	else if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		args->queueid = VMKNETDDI_QUEUEOPS_MK_RX_QUEUEID(0);
		return VMKNETDDI_QUEUEOPS_OK;
	}
	else {
		printk("s2io_get_default_queue: invalid queue type\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
s2io_apply_rx_filter(vmknetddi_queueop_apply_rx_filter_args_t *args)
{
	nic_t *nic;
	struct config_param *config;
	s2io_rx_filter_t *queue_filter;
	rx_ring_config_t *queue_cfg;
	int status, rval, fidx;
	u8 *macaddr;
	u16 queue = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);

	if (!VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid)) {
		printk("s2io_apply_rx_filter: not an rx queue 0x%x\n",
			args->queueid);
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	if (vmknetddi_queueops_get_filter_class(&args->filter)
					!= VMKNETDDI_QUEUEOPS_FILTER_MACADDR) {
		printk("s2io_apply_rx_filter: only mac filters supported\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	queue_filter = NULL;
	nic = args->netdev->priv;
	config = &nic->config;
	queue_cfg = &config->rx_cfg[queue];

	if (!queue_cfg->allocated) {
		printk("s2io_apply_rx_filter: queue not alloacted\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	spin_lock(&config->netqueue_lock);

	if (queue_cfg->active_filter_count >= S2IO_MAX_QUEUE_RXFILTERS) {
		printk("s2io_apply_rx_filter: filter count exceeded\n");
		status = VMKNETDDI_QUEUEOPS_ERR;
		goto out;
	}

	for (fidx=0; fidx < S2IO_MAX_QUEUE_RXFILTERS; fidx++) {
		if (!queue_cfg->filter[fidx].active) {
			queue_filter = &queue_cfg->filter[fidx];
			break;
		}
	}
	if (!queue_filter) {
		printk("s2io_apply_rx_filter: all filters slots are active\n");
		status = VMKNETDDI_QUEUEOPS_ERR;
		goto out;
	}

	macaddr = vmknetddi_queueops_get_filter_macaddr(&args->filter);

	rval = s2io_rts_mac_enable(args->netdev, queue, macaddr);
	if (rval == SUCCESS) {
		queue_cfg->active_filter_count++;
		queue_filter->active = TRUE;
		memcpy(queue_filter->macaddr, macaddr, ETH_ALEN);
		args->filterid = VMKNETDDI_QUEUEOPS_MK_FILTERID(fidx);
		status = VMKNETDDI_QUEUEOPS_OK;
	}
	else {
		printk("s2io_apply_rx_filter: s2io_rts_mac_enable failed\n");
		status = VMKNETDDI_QUEUEOPS_ERR;
	}

out:
	spin_unlock(&config->netqueue_lock);
	return status;
}

static int
s2io_remove_rx_filter(vmknetddi_queueop_remove_rx_filter_args_t *args)
{
	int rval;
	nic_t *nic = args->netdev->priv;
	struct config_param *config = &nic->config;
	u16 cidx = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	u16 fidx = VMKNETDDI_QUEUEOPS_FILTERID_VAL(args->filterid);
	rx_ring_config_t *queue_cfg;
	s2io_rx_filter_t *queue_filter;

	if (s2io_validate_queueid(args->netdev, args->queueid)
						!= VMKNETDDI_QUEUEOPS_OK) {
		printk("s2io_remove_rx_filter: failed to validate queue id\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	if (s2io_validate_filterid(args->netdev, args->filterid)
						!= VMKNETDDI_QUEUEOPS_OK) {
		printk("s2io_remove_rx_filter: failed to validate filter id\n");		return VMKNETDDI_QUEUEOPS_ERR;
	}

	queue_cfg = &config->rx_cfg[cidx];
	queue_filter = &queue_cfg->filter[fidx];

	spin_lock(&config->netqueue_lock);
	rval = s2io_rts_mac_disable(args->netdev, cidx);
	if (rval == SUCCESS) {
		queue_filter->active = FALSE;
		memset(queue_filter->macaddr, 0xff, ETH_ALEN);
		queue_cfg->active_filter_count--;
		spin_unlock(&config->netqueue_lock);
		return VMKNETDDI_QUEUEOPS_OK;
	}
	else {
		spin_unlock(&config->netqueue_lock);
		printk("s2io_remove_rx_filter: s2io_rts_mac_disable failed\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
s2io_get_queue_stats(void *args)
{
	//XXX: not implemented
	printk("s2io_get_queue_stats: not implemented\n");
	return VMKNETDDI_QUEUEOPS_ERR;
}

static int
s2io_get_netqueue_version(vmknetddi_queueop_get_version_args_t *args)
{
	return vmknetddi_queueops_version(args);
}

static int
s2io_netqueue_ops(vmknetddi_queueops_op_t op, void *args)
{
	switch (op) {
		case VMKNETDDI_QUEUEOPS_OP_GET_VERSION:
		return s2io_get_netqueue_version(
			(vmknetddi_queueop_get_version_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_FEATURES:
		return s2io_get_netqueue_features(
			(vmknetddi_queueop_get_features_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_QUEUE_COUNT:
		return s2io_get_queue_count(
			(vmknetddi_queueop_get_queue_count_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_FILTER_COUNT:
		return s2io_get_filter_count(
			(vmknetddi_queueop_get_filter_count_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_ALLOC_QUEUE:
		return s2io_alloc_queue(
			(vmknetddi_queueop_alloc_queue_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_FREE_QUEUE:
		return s2io_free_queue(
			(vmknetddi_queueop_free_queue_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_QUEUE_VECTOR:
		return s2io_get_queue_vector(
			(vmknetddi_queueop_get_queue_vector_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_DEFAULT_QUEUE:
		return s2io_get_default_queue(
			(vmknetddi_queueop_get_default_queue_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_APPLY_RX_FILTER:
		return s2io_apply_rx_filter(
			(vmknetddi_queueop_apply_rx_filter_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_REMOVE_RX_FILTER:
		return s2io_remove_rx_filter(
			(vmknetddi_queueop_remove_rx_filter_args_t *)args);
		break;

		case VMKNETDDI_QUEUEOPS_OP_GET_STATS:
		return s2io_get_queue_stats(
			(vmknetddi_queueop_get_stats_args_t *)args);
		break;

		default:
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	return VMKNETDDI_QUEUEOPS_ERR;
}

#endif

/*
 *  To build the driver,
 * gcc -D__KERNEL__ -DMODULE -I/usr/src/linux-2.4/include -Wall
 * -Wstrict-prototypes -O2 -c s2io.c
 */
