/*
 *	Adaptec AAC series RAID controller driver
 *
 * Copyright (c) 2004 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdarg.h>
#include <linux/types.h>
#include <linux/wait.h>
#include <linux/spinlock.h>
#include <asm/semaphore.h>
#include <linux/kernel.h>
#include <linux/blkdev.h>
#include <linux/completion.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/version.h>
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
#include <scsi/scsi.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_host.h>
#else
#include "scsi.h"
#include "hosts.h"
#endif
#include "aacraid.h"
#include "fwdebug.h"

/*
 * Debug flags to be put into the HBA flags field when initialized
 */
const unsigned long aac_debug_flags = /* Variable to setup with above flags. */
/*			HBA_FLAGS_DBG_KERNEL_PRINT_B |		*/
			HBA_FLAGS_DBG_FW_PRINT_B |
			HBA_FLAGS_DBG_FUNCTION_ENTRY_B |
			HBA_FLAGS_DBG_FUNCTION_EXIT_B |
			HBA_FLAGS_DBG_ERROR_B |
/*			HBA_FLAGS_DBG_INIT_B |			*/
/*			HBA_FLAGS_DBG_OS_COMMANDS_B |		*/
/*			HBA_FLAGS_DBG_SCAN_B |			*/
/*			HBA_FLAGS_DBG_COALESCE_B |		*/
/*			HBA_FLAGS_DBG_IOCTL_COMMANDS_B |	*/
/*			HBA_FLAGS_DBG_SYNC_COMMANDS_B |		*/
/*			HBA_FLAGS_DBG_COMM_B |			*/
/*			HBA_FLAGS_DBG_AIF_B |			*/
/*			HBA_FLAGS_DBG_CSMI_COMMANDS_B | 	*/
/*			HBA_FLAGS_DBG_FLAGS_MASK | 		*/
0;

int aac_get_fw_debug_buffer(struct aac_dev * dev)
{
if (nblank(fwprintf(x))) {
	u32 MonDriverBufferPhysAddrLow = 0;
	u32 MonDriverBufferPhysAddrHigh = 0;
	u32 MonDriverBufferSize = 0;
	u32 MonDriverHeaderSize = 0;
	u32 ReturnStatus = 0;

	/*
	 * Initialize the firmware print buffer fields
	 */
	dev->FwDebugBuffer_P = NULL;
	dev->FwDebugFlags_P = NULL;
	dev->FwDebugStrLength_P = NULL;
	dev->FwDebugBLEDvalue_P = NULL;
	dev->FwDebugBLEDflag_P = NULL;
	dev->FwDebugBufferSize = 0;
	dev->FwDebugFlags = 0;
	dev->DebugFlags = 0;

	/*
	 * Get the firmware print buffer parameters from the firmware
	 * If the command was successful map in the address.
	 */
	if (!aac_adapter_sync_cmd(dev, GET_DRIVER_BUFFER_PROPERTIES,
	  0, 0, 0, 0, 0, 0,
	  &ReturnStatus,
	  &MonDriverBufferPhysAddrLow,
	  &MonDriverBufferPhysAddrHigh,
	  &MonDriverBufferSize,
	  &MonDriverHeaderSize) && MonDriverBufferSize) {
		unsigned long Offset = MonDriverBufferPhysAddrLow
		                     - (dev->scsi_host_ptr->base & 0xffffffff);

		/*
		 * See if the address is already mapped in and if so set it up
		 * from the base address
		 */
		if (((u32)(((u64)dev->scsi_host_ptr->base) >> 32)
		  == MonDriverBufferPhysAddrHigh)
		 && ((Offset + MonDriverBufferSize) < dev->base_size))
			dev->FwDebugBuffer_P
			  = ((unsigned char *)dev->regs.sa + Offset);

		/*
		 * If mapping went well, Set up the debug buffer fields in the
		 * HBA structure from the data returned
		 */
		if (dev->FwDebugBuffer_P != NULL) {
			dev->FwDebugFlags_P
			  = (u32 *)(dev->FwDebugBuffer_P
			  + FW_DEBUG_FLAGS_OFFSET);
			dev->FwDebugStrLength_P
			  = (u32 *)(dev->FwDebugBuffer_P
			  + FW_DEBUG_STR_LENGTH_OFFSET);
			dev->FwDebugBLEDvalue_P
			  = dev->FwDebugBuffer_P
			  + FW_DEBUG_BLED_OFFSET;
			dev->FwDebugBLEDflag_P
			  = dev->FwDebugBLEDvalue_P + 1;
			dev->FwDebugBufferSize = MonDriverBufferSize;
			dev->FwDebugBuffer_P += MonDriverHeaderSize;
			dev->FwDebugFlags = 0;
			dev->DebugFlags = aac_debug_flags;
			return 1;
		}
	}

	/*
	 * The GET_DRIVER_BUFFER_PROPERTIES command failed
	 */
}
	return 0;
}

#define PRINT_TIMEOUT (HZ/4) /* 1/4 second */

void aac_fw_printf(struct aac_dev * dev, unsigned long PrintFlags, const char * fmt, ...)
{
if (nblank(fwprintf(x))) {
	va_list args;
	u32 Count;
	unsigned long next_jiffies;
	char PrintBuffer_P[PRINT_BUFFER_SIZE];

	if ((((PrintFlags
	  & ~(HBA_FLAGS_DBG_KERNEL_PRINT_B|HBA_FLAGS_DBG_FW_PRINT_B)) != 0)
	  && (dev != NULL)
	  && ((dev->DebugFlags & PrintFlags) == 0))
	 || ((dev != NULL) && (dev->DebugFlags
	   & (HBA_FLAGS_DBG_KERNEL_PRINT_B|HBA_FLAGS_DBG_FW_PRINT_B)) == 0))
		return;
	/*
	 * Set up parameters and call sprintf function to format the data
	 */
	va_start(args, fmt);
	vsprintf(PrintBuffer_P, fmt, args);
	va_end(args);

	/*
	 * Make sure the HBA structure has been passed in for this section
	 */
	if ((dev != NULL) && (dev->FwDebugBufferSize)) {
		/*
		 * If we are set up for a Firmware print
		 */
		if ((dev->DebugFlags & HBA_FLAGS_DBG_FW_PRINT_B)
		 && ((PrintFlags
		  & (HBA_FLAGS_DBG_KERNEL_PRINT_B|HBA_FLAGS_DBG_FW_PRINT_B))
		  != HBA_FLAGS_DBG_KERNEL_PRINT_B)) {
			/*
			 * Make sure the string size is within boundaries
			 */
			Count = strlen(PrintBuffer_P);
			if (Count > dev->FwDebugBufferSize)
				Count = (u16)dev->FwDebugBufferSize;

			/*
			 * Wait for no more than PRINT_TIMEOUT for the previous
			 * message length to clear (the handshake).
			 */
			next_jiffies = jiffies + PRINT_TIMEOUT;
			while ((next_jiffies - jiffies) >= 0) {
				if (!(*dev->FwDebugStrLength_P))
					break;
				schedule();
			}

			/*
			 * If the Length is clear, copy over the message, the
			 * flags, and the length. Make sure the length is the
			 * last because that is the signal for the Firmware to
			 * pick it up.
			 */
			if (!(*dev->FwDebugStrLength_P)) {
				memcpy(dev->FwDebugBuffer_P,
				  PrintBuffer_P, Count);
				*dev->FwDebugFlags_P = cpu_to_le32(dev->FwDebugFlags);
				*dev->FwDebugStrLength_P = cpu_to_le32(Count);
			} else
				dev->DebugFlags &= ~HBA_FLAGS_DBG_FW_PRINT_B;
		}

		/*
		 * If the Kernel Debug Print flag is set, send it off to the
		 * Kernel debugger
		 */
		if ((dev->DebugFlags & HBA_FLAGS_DBG_KERNEL_PRINT_B)
		 && ((PrintFlags
		  & (HBA_FLAGS_DBG_KERNEL_PRINT_B|HBA_FLAGS_DBG_FW_PRINT_B))
		  != HBA_FLAGS_DBG_FW_PRINT_B)) {
			if (dev->FwDebugFlags & FW_DEBUG_FLAGS_NO_HEADERS_B)
				printk ("%s", PrintBuffer_P);
			else
				printk (KERN_INFO "%s: %s\n",
				  dev->scsi_host_ptr->hostt->proc_name,
				  PrintBuffer_P);
		}
	}

	/*
	 * No HBA structure passed in so it has to be for the Kernel Debugger
	 */
	else if ((PrintFlags
	  & (HBA_FLAGS_DBG_KERNEL_PRINT_B|HBA_FLAGS_DBG_FW_PRINT_B))
	  != HBA_FLAGS_DBG_FW_PRINT_B) {
		if ((dev != NULL)
		 && (dev->FwDebugFlags & FW_DEBUG_FLAGS_NO_HEADERS_B))
			printk ("%s", PrintBuffer_P);
		else if (dev != NULL)
			printk (KERN_INFO "%s: %s\n",
			  dev->scsi_host_ptr->hostt->proc_name,
			  PrintBuffer_P);
		else
			printk(KERN_INFO "%s\n", PrintBuffer_P);
	}
}
}

void aac_fw_print_mem(struct aac_dev * dev, unsigned long PrintFlags, u8 * Addr, int Count)
{
if (nblank(fwprintf(x))) {
	int Offset, i;
	u32 DebugFlags = 0;
	char Buffer[100];
	char * LineBuffer_P;

	/*
	 * If we have an HBA structure, save off the flags and set the no
	 * headers flag so we don't have garbage between our lines of data
	 */
	if (dev != NULL) {
		DebugFlags = dev->FwDebugFlags;
		dev->FwDebugFlags |= FW_DEBUG_FLAGS_NO_HEADERS_B;
	}

	Offset = 0;

	/*
	 * Loop through all the data
	 */
	while (Offset < Count) {
		/*
		 * We will format each line into a buffer and then print out
		 * the entire line so set the pointer to the beginning of the
		 * buffer
		 */
		LineBuffer_P = Buffer;

		/*
		 * Set up the address in HEX
		 */
		sprintf(LineBuffer_P, "\n%04x  ", Offset);
		LineBuffer_P += 6;

		/*
		 * Set up 16 bytes in HEX format
		 */
		for (i = 0; i < 16; ++i) {
			/*
			 * If we are past the count of data bytes to output,
			 * pad with blanks
			 */
			sprintf (LineBuffer_P,
			  (((Offset + i) >= Count) ? "   " : "%02x "),
			  Addr[Offset + i]);
			LineBuffer_P += 3;

			/*
			 * At the mid point we will put in a divider
			 */
			if (i == 7) {
				sprintf (LineBuffer_P, "- ");
				LineBuffer_P += 2;
			}
		}
		/*
		 * Now do the same 16 bytes at the end of the line in ASCII
		 * format
		 */
		sprintf (LineBuffer_P, "  ");
		LineBuffer_P += 2;
		for (i = 0; i < 16; ++i) {
			/*
			 * If all data processed, OUT-O-HERE
			 */
			if ((Offset + i) >= Count)
				break;

			/*
			 * If this is a printable ASCII character, convert it
			 */
			sprintf (LineBuffer_P,
			  (((Addr[Offset + i] > 0x1F)
			   && (Addr[Offset + i] < 0x7F))
				? "%c"
				: "."), Addr[Offset + i]);

			++LineBuffer_P;
		}
		/*
		 * The line is now formatted, so print it out
		 */
		aac_fw_printf(dev, PrintFlags, "%s", Buffer);

		/*
		 * Bump the offset by 16 for the next line
		 */
		Offset += 16;

	}

	/*
	 * Restore the saved off flags
	 */
	if (dev != NULL)
		dev->FwDebugFlags = DebugFlags;
}
}
