/*
 *	Adaptec AAC series RAID controller driver
 *	(c) Copyright 2001 Red Hat Inc.	<alan@redhat.com>
 *
 * based on the old aacraid driver that is..
 * Adaptec aacraid device driver for Linux.
 *
 * Copyright (c) 2000 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Module Name:
 *   linit.c
 *
 * Abstract: Linux Driver entry module for Adaptec RAID Array Controller
 */

#define AAC_DRIVER_VERSION		"1.1-5"
#ifndef AAC_DRIVER_BRANCH
#define AAC_DRIVER_BRANCH		""
#endif
#if defined(__VMKERNEL_MODULE__)
#define AAC_DRIVER_BUILD_DATE		__DATE__
#else
#define AAC_DRIVER_BUILD_DATE		__DATE__ " " __TIME__
#endif
#if 1 || defined(__VMWARE__)
#define AAC_DRIVERNAME			"aacraid_esx30"
#else
#define AAC_DRIVERNAME			"aacraid"
#endif

#include <linux/version.h> /* for the following test */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,3))
#include <linux/compat.h>
#endif
#include <linux/blkdev.h>
#include <linux/completion.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,3))
#include <linux/moduleparam.h>
#else
#include <linux/config.h>
#include <linux/types.h>
#include <linux/sched.h>
#endif
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,3))
#include <linux/syscalls.h>
#include <linux/ioctl32.h>
#endif
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,9)) || defined(SCSI_HAS_SSLEEP)
#include <linux/delay.h>
#endif
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
#include <linux/dma-mapping.h>
#endif
#include <asm/semaphore.h>

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
#include <scsi/scsi.h>
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,6,1)) && !defined(FAILED))
#define SUCCESS 0x2002
#define FAILED  0x2003
#endif
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,6,10)) && defined(DID_BUS_BUSY) && !defined(BLIST_NO_ULD_ATTACH))
#include <scsi/scsi_devinfo.h>	/* Pick up BLIST_NO_ULD_ATTACH? */
#endif
#include <scsi/scsi_host.h>
#include <scsi/scsi_tcq.h>
#include <scsi/scsicam.h>
#include <scsi/scsi_eh.h>
#else
#include "scsi.h"
#include "hosts.h"
#include "sd.h"
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) && defined(SCSI_HAS_DUMP))
#include "scsi_dump.h"
#endif
#include <linux/blk.h>	/* for io_request_lock definition */
#endif
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11))
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) ? defined(__x86_64__) : defined(CONFIG_COMPAT))
#if ((KERNEL_VERSION(2,4,19) <= LINUX_VERSION_CODE) && (LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,21)))
# include <asm-x86_64/ioctl32.h>
#elif (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
# include <asm/ioctl32.h>
#elif (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,3))
# include <linux/ioctl32.h>
#endif
  /* Cast the function, since sys_ioctl does not match */
# define aac_ioctl32(x,y) register_ioctl32_conversion((x), \
    (int(*)(unsigned int,unsigned int,unsigned long,struct file*))(y))
#endif
#endif
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
# include <asm/uaccess.h>
#endif
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11)) || ((LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,11)) && !defined(PCI_HAS_SHUTDOWN)))
#include <linux/reboot.h>
#endif

#include "aacraid.h"
#include "fwdebug.h"

#if (defined(AAC_DRIVER_BUILD))
#define _str(x) #x
#define str(x) _str(x)
#define AAC_DRIVER_FULL_VERSION	AAC_DRIVER_VERSION "[" str(AAC_DRIVER_BUILD) "a]" AAC_DRIVER_BRANCH
#else
#define AAC_DRIVER_FULL_VERSION	AAC_DRIVER_VERSION AAC_DRIVER_BRANCH " " AAC_DRIVER_BUILD_DATE
#endif
#if defined(__VMKERNEL_MODULE__)
spinlock_t io_request_lock;
#endif

MODULE_AUTHOR("Red Hat Inc and Adaptec");
MODULE_DESCRIPTION("Dell PERC2, 2/Si, 3/Si, 3/Di, "
		   "Adaptec Advanced Raid Products, "
		   "HP NetRAID-4M, IBM ServeRAID & ICP SCSI driver");
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,7))
MODULE_LICENSE("GPL");
#endif
#if ((LINUX_VERSION_CODE > KERNEL_VERSION(2,6,3)) || defined(MODULE_VERSION))
MODULE_VERSION(AAC_DRIVER_FULL_VERSION);
#endif

#if (defined(AAC_CSMI))
LIST_HEAD(aac_devices);
#else
static LIST_HEAD(aac_devices);
#endif
static int aac_cfg_major = -1;
char aac_driver_version[] = AAC_DRIVER_FULL_VERSION;

/*
 * Because of the way Linux names scsi devices, the order in this table has
 * become important.  Check for on-board Raid first, add-in cards second.
 *
 * Note: The last field is used to index into aac_drivers below.
 */
static struct pci_device_id aac_pci_tbl[] = {
	{ 0x1028, 0x0001, 0x1028, 0x0001, 0, 0, 0 }, /* PERC 2/Si (Iguana/PERC2Si) */
	{ 0x1028, 0x0002, 0x1028, 0x0002, 0, 0, 1 }, /* PERC 3/Di (Opal/PERC3Di) */
	{ 0x1028, 0x0003, 0x1028, 0x0003, 0, 0, 2 }, /* PERC 3/Si (SlimFast/PERC3Si */
	{ 0x1028, 0x0004, 0x1028, 0x00d0, 0, 0, 3 }, /* PERC 3/Di (Iguana FlipChip/PERC3DiF */
	{ 0x1028, 0x0002, 0x1028, 0x00d1, 0, 0, 4 }, /* PERC 3/Di (Viper/PERC3DiV) */
	{ 0x1028, 0x0002, 0x1028, 0x00d9, 0, 0, 5 }, /* PERC 3/Di (Lexus/PERC3DiL) */
	{ 0x1028, 0x000a, 0x1028, 0x0106, 0, 0, 6 }, /* PERC 3/Di (Jaguar/PERC3DiJ) */
	{ 0x1028, 0x000a, 0x1028, 0x011b, 0, 0, 7 }, /* PERC 3/Di (Dagger/PERC3DiD) */
	{ 0x1028, 0x000a, 0x1028, 0x0121, 0, 0, 8 }, /* PERC 3/Di (Boxster/PERC3DiB) */
	{ 0x9005, 0x0283, 0x9005, 0x0283, 0, 0, 9 }, /* catapult */
	{ 0x9005, 0x0284, 0x9005, 0x0284, 0, 0, 10 }, /* tomcat */
	{ 0x9005, 0x0285, 0x9005, 0x0286, 0, 0, 11 }, /* Adaptec 2120S (Crusader) */
	{ 0x9005, 0x0285, 0x9005, 0x0285, 0, 0, 12 }, /* Adaptec 2200S (Vulcan) */
	{ 0x9005, 0x0285, 0x9005, 0x0287, 0, 0, 13 }, /* Adaptec 2200S (Vulcan-2m) */
	{ 0x9005, 0x0285, 0x17aa, 0x0286, 0, 0, 14 }, /* Legend S220 (Legend Crusader) */
	{ 0x9005, 0x0285, 0x17aa, 0x0287, 0, 0, 15 }, /* Legend S230 (Legend Vulcan) */

	{ 0x9005, 0x0285, 0x9005, 0x0288, 0, 0, 16 }, /* Adaptec 3230S (Harrier) */
	{ 0x9005, 0x0285, 0x9005, 0x0289, 0, 0, 17 }, /* Adaptec 3240S (Tornado) */
	{ 0x9005, 0x0285, 0x9005, 0x028a, 0, 0, 18 }, /* ASR-2020ZCR SCSI PCI-X ZCR (Skyhawk) */
	{ 0x9005, 0x0285, 0x9005, 0x028b, 0, 0, 19 }, /* ASR-2025ZCR SCSI SO-DIMM PCI-X ZCR (Terminator) */
	{ 0x9005, 0x0286, 0x9005, 0x028c, 0, 0, 20 }, /* ASR-2230S + ASR-2230SLP PCI-X (Lancer) */
	{ 0x9005, 0x0286, 0x9005, 0x028d, 0, 0, 21 }, /* ASR-2130S (Lancer) */
	{ 0x9005, 0x0286, 0x9005, 0x029b, 0, 0, 22 }, /* AAR-2820SA (Intruder) */
	{ 0x9005, 0x0286, 0x9005, 0x029c, 0, 0, 23 }, /* AAR-2620SA (Intruder) */
	{ 0x9005, 0x0286, 0x9005, 0x029d, 0, 0, 24 }, /* AAR-2420SA (Intruder) */
	{ 0x9005, 0x0286, 0x9005, 0x029e, 0, 0, 25 }, /* ICP9024R0 (Lancer) */
	{ 0x9005, 0x0286, 0x9005, 0x029f, 0, 0, 26 }, /* ICP9014R0 (Lancer) */
	{ 0x9005, 0x0286, 0x9005, 0x02a0, 0, 0, 27 }, /* ICP9047MA (Lancer) */
	{ 0x9005, 0x0286, 0x9005, 0x02a1, 0, 0, 28 }, /* ICP9087MA (Lancer) */
	{ 0x9005, 0x0286, 0x9005, 0x02a3, 0, 0, 29 }, /* ICP5085AU (Hurricane) */
	{ 0x9005, 0x0285, 0x9005, 0x02a4, 0, 0, 30 }, /* ICP9085LI (Marauder-X) */
	{ 0x9005, 0x0285, 0x9005, 0x02a5, 0, 0, 31 }, /* ICP5085BR (Marauder-E) */
	{ 0x9005, 0x0286, 0x9005, 0x02a6, 0, 0, 32 }, /* ICP9067MA (Intruder-6) */
	{ 0x9005, 0x0287, 0x9005, 0x0800, 0, 0, 33 }, /* Themisto Jupiter Platform */
	{ 0x9005, 0x0200, 0x9005, 0x0200, 0, 0, 33 }, /* Themisto Jupiter Platform */
	{ 0x9005, 0x0286, 0x9005, 0x0800, 0, 0, 34 }, /* Callisto Jupiter Platform */
	{ 0x9005, 0x0285, 0x9005, 0x028e, 0, 0, 35 }, /* ASR-2020SA SATA PCI-X ZCR (Skyhawk) */
	{ 0x9005, 0x0285, 0x9005, 0x028f, 0, 0, 36 }, /* ASR-2025SA SATA SO-DIMM PCI-X ZCR (Terminator) */
	{ 0x9005, 0x0285, 0x9005, 0x0290, 0, 0, 37 }, /* AAR-2410SA PCI SATA 4ch (Jaguar II) */
	{ 0x9005, 0x0285, 0x1028, 0x0291, 0, 0, 38 }, /* CERC SATA RAID 2 PCI SATA 6ch (DellCorsair) */
	{ 0x9005, 0x0285, 0x9005, 0x0292, 0, 0, 39 }, /* AAR-2810SA PCI SATA 8ch (Corsair-8) */
	{ 0x9005, 0x0285, 0x9005, 0x0293, 0, 0, 40 }, /* AAR-21610SA PCI SATA 16ch (Corsair-16) */
	{ 0x9005, 0x0285, 0x9005, 0x0294, 0, 0, 41 }, /* ESD SO-DIMM PCI-X SATA ZCR (Prowler) */
	{ 0x9005, 0x0285, 0x103C, 0x3227, 0, 0, 42 }, /* AAR-2610SA PCI SATA 6ch */
	{ 0x9005, 0x0285, 0x9005, 0x0296, 0, 0, 43 }, /* ASR-2240S (SabreExpress) */
	{ 0x9005, 0x0285, 0x9005, 0x0297, 0, 0, 44 }, /* ASR-4005SAS */
	{ 0x9005, 0x0285, 0x1014, 0x02F2, 0, 0, 45 }, /* IBM 8i (AvonPark) */
	{ 0x9005, 0x0285, 0x1014, 0x0312, 0, 0, 45 }, /* IBM 8i (AvonPark Lite) */
	{ 0x9005, 0x0286, 0x1014, 0x9580, 0, 0, 46 }, /* IBM 8k/8k-l8 (Aurora) */
	{ 0x9005, 0x0286, 0x1014, 0x9540, 0, 0, 47 }, /* IBM 8k/8k-l4 (Aurora Lite) */
	{ 0x9005, 0x0285, 0x9005, 0x0298, 0, 0, 48 }, /* ASR-4000SAS (BlackBird) */
	{ 0x9005, 0x0285, 0x9005, 0x0299, 0, 0, 49 }, /* ASR-4800SAS (Marauder-X) */
	{ 0x9005, 0x0285, 0x9005, 0x029a, 0, 0, 50 }, /* ASR-4805SAS (Marauder-E) */
	{ 0x9005, 0x0286, 0x9005, 0x02a2, 0, 0, 51 }, /* ASR-4810SAS (Hurricane */

	{ 0x9005, 0x0285, 0x1028, 0x0287, 0, 0, 52 }, /* Perc 320/DC*/
	{ 0x1011, 0x0046, 0x9005, 0x0365, 0, 0, 53 }, /* Adaptec 5400S (Mustang)*/
	{ 0x1011, 0x0046, 0x9005, 0x0364, 0, 0, 54 }, /* Adaptec 5400S (Mustang)*/
	{ 0x1011, 0x0046, 0x9005, 0x1364, 0, 0, 55 }, /* Dell PERC2/QC */
	{ 0x1011, 0x0046, 0x103c, 0x10c2, 0, 0, 56 }, /* HP NetRAID-4M */

	{ 0x9005, 0x0285, 0x1028, PCI_ANY_ID, 0, 0, 57 }, /* Dell Catchall */
	{ 0x9005, 0x0285, 0x17aa, PCI_ANY_ID, 0, 0, 58 }, /* Legend Catchall */
	{ 0x9005, 0x0285, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 59 }, /* Adaptec Catch All */
	{ 0x9005, 0x0286, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 60 }, /* Adaptec Rocket Catch All */
	{ 0,}
};
MODULE_DEVICE_TABLE(pci, aac_pci_tbl);

/*
 * dmb - For now we add the number of channels to this structure.  
 * In the future we should add a fib that reports the number of channels
 * for the card.  At that time we can remove the channels from here
 */
static struct aac_driver_ident aac_drivers[] = {
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 2/Si (Iguana/PERC2Si) */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Di (Opal/PERC3Di) */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Si (SlimFast/PERC3Si */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Di (Iguana FlipChip/PERC3DiF */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Di (Viper/PERC3DiV) */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Di (Lexus/PERC3DiL) */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 1, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Di (Jaguar/PERC3DiJ) */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Di (Dagger/PERC3DiD) */
	{ aac_rx_init, "percraid", "DELL    ", "PERCRAID        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* PERC 3/Di (Boxster/PERC3DiB) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "catapult        ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* catapult */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "tomcat          ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* tomcat */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "Adaptec 2120S   ", 1, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Adaptec 2120S (Crusader) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "Adaptec 2200S   ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Adaptec 2200S (Vulcan) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "Adaptec 2200S   ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Adaptec 2200S (Vulcan-2m) */
	{ aac_rx_init, "aacraid",  "Legend  ", "Legend S220     ", 1, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Legend S220 (Legend Crusader) */
	{ aac_rx_init, "aacraid",  "Legend  ", "Legend S230     ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Legend S230 (Legend Vulcan) */

	{ aac_rx_init, "aacraid",  "ADAPTEC ", "Adaptec 3230S   ", 2 }, /* Adaptec 3230S (Harrier) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "Adaptec 3240S   ", 2 }, /* Adaptec 3240S (Tornado) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-2020ZCR     ", 2 }, /* ASR-2020ZCR SCSI PCI-X ZCR (Skyhawk) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-2025ZCR     ", 2 }, /* ASR-2025ZCR SCSI SO-DIMM PCI-X ZCR (Terminator) */
	{ aac_rkt_init, "aacraid",  "ADAPTEC ", "ASR-2230S PCI-X ", 2 }, /* ASR-2230S + ASR-2230SLP PCI-X (Lancer) */
	{ aac_rkt_init, "aacraid",  "ADAPTEC ", "ASR-2130S PCI-X ", 1 }, /* ASR-2130S (Lancer) */
	{ aac_rkt_init, "aacraid",  "ADAPTEC ", "AAR-2820SA      ", 1 }, /* AAR-2820SA (Intruder) */
	{ aac_rkt_init, "aacraid",  "ADAPTEC ", "AAR-2620SA      ", 1 }, /* AAR-2620SA (Intruder) */
	{ aac_rkt_init, "aacraid",  "ADAPTEC ", "AAR-2420SA      ", 1 }, /* AAR-2420SA (Intruder) */
	{ aac_rkt_init, "aacraid",  "ICP     ", "ICP9024R0       ", 2 }, /* ICP9024R0 (Lancer) */
	{ aac_rkt_init, "aacraid",  "ICP     ", "ICP9014R0       ", 1 }, /* ICP9014R0 (Lancer) */
	{ aac_rkt_init, "aacraid",  "ICP     ", "ICP9047MA       ", 1 }, /* ICP9047MA (Lancer) */
	{ aac_rkt_init, "aacraid",  "ICP     ", "ICP9087MA       ", 1 }, /* ICP9087MA (Lancer) */
	{ aac_rkt_init, "aacraid",  "ICP     ", "ICP5085AU       ", 1 }, /* ICP5085AU (Hurricane) */
	{ aac_rx_init, "aacraid",  "ICP     ", "ICP9085LI       ", 1 }, /* ICP9085LI (Marauder-X) */
	{ aac_rx_init, "aacraid",  "ICP     ", "ICP5085BR       ", 1 }, /* ICP5085BR (Marauder-E) */
	{ aac_rkt_init, "aacraid",  "ICP     ", "ICP9067MA       ", 1 }, /* ICP9067MA (Intruder-6) */
	{ NULL        , "aacraid",  "ADAPTEC ", "Themisto        ", 0, AAC_QUIRK_SLAVE }, /* Jupiter Platform */
	{ aac_rkt_init, "aacraid",  "ADAPTEC ", "Callisto        ", 2, AAC_QUIRK_MASTER }, /* Jupiter Platform */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-2020SA       ", 1 }, /* ASR-2020SA SATA PCI-X ZCR (Skyhawk) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-2025SA       ", 1 }, /* ASR-2025SA SATA SO-DIMM PCI-X ZCR (Terminator) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "AAR-2410SA SATA ", 1, AAC_QUIRK_17SG }, /* AAR-2410SA PCI SATA 4ch (Jaguar II) */
	{ aac_rx_init, "aacraid",  "DELL    ", "CERC SR2        ", 1, AAC_QUIRK_17SG }, /* CERC SATA RAID 2 PCI SATA 6ch (DellCorsair) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "AAR-2810SA SATA ", 1, AAC_QUIRK_17SG }, /* AAR-2810SA PCI SATA 8ch (Corsair-8) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "AAR-21610SA SATA", 1, AAC_QUIRK_17SG }, /* AAR-21610SA PCI SATA 16ch (Corsair-16) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-2026ZCR     ", 1 }, /* ESD SO-DIMM PCI-X SATA ZCR (Prowler) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "AAR-2610SA      ", 1 }, /* SATA 6Ch (Bearcat) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-2240S       ", 1 }, /* ASR-2240S (SabreExpress) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-4005SAS     ", 1 }, /* ASR-4005SAS */
	{ aac_rx_init, "ServeRAID","IBM     ", "ServeRAID 8i    ", 1 }, /* IBM 8i (AvonPark) */
	{ aac_rkt_init, "ServeRAID","IBM     ", "ServeRAID 8k-l8 ", 1 }, /* IBM 8k/8k-l8 (Aurora) */
	{ aac_rkt_init, "ServeRAID","IBM     ", "ServeRAID 8k-l4 ", 1 }, /* IBM 8k/8k-l4 (Aurora Lite) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-4000SAS     ", 1 }, /* ASR-4000SAS (BlackBird & AvonPark) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-4800SAS     ", 1 }, /* ASR-4800SAS (Marauder-X) */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "ASR-4805SAS     ", 1 }, /* ASR-4805SAS (Marauder-E) */
	{ aac_rkt_init, "aacraid",  "ADAPTEC ", "ASR-4810SAS     ", 1 }, /* ASR-4810SAS (Hurricane) */

	{ aac_rx_init, "percraid", "DELL    ", "PERC 320/DC     ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Perc 320/DC*/
	{ aac_sa_init, "aacraid",  "ADAPTEC ", "Adaptec 5400S   ", 4, AAC_QUIRK_34SG }, /* Adaptec 5400S (Mustang)*/
	{ aac_sa_init, "aacraid",  "ADAPTEC ", "AAC-364         ", 4, AAC_QUIRK_34SG }, /* Adaptec 5400S (Mustang)*/
	{ aac_sa_init, "percraid", "DELL    ", "PERCRAID        ", 4, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Dell PERC2/QC */
	{ aac_sa_init, "hpnraid",  "HP      ", "NetRAID         ", 4, AAC_QUIRK_34SG }, /* HP NetRAID-4M */

	{ aac_rx_init, "aacraid",  "DELL    ", "RAID            ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Dell Catchall */
	{ aac_rx_init, "aacraid",  "Legend  ", "RAID            ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Legend Catchall */
	{ aac_rx_init, "aacraid",  "ADAPTEC ", "RAID            ", 2, AAC_QUIRK_31BIT | AAC_QUIRK_34SG }, /* Adaptec Catch All */
	{ aac_rkt_init, "aacraid", "ADAPTEC ", "RAID            ", 2 } /* Adaptec Rocket Catch All */
};
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11))

#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) ? defined(__x86_64__) : defined(CONFIG_COMPAT))
/* 
 * Promote 32 bit apps that call get_next_adapter_fib_ioctl to 64 bit version 
 */
static int aac_get_next_adapter_fib_ioctl(unsigned int fd, unsigned int cmd, 
		unsigned long arg, struct file *file)
{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	struct fib_ioctl f;
	mm_segment_t fs;
	int retval;

	memset (&f, 0, sizeof(f));
	if (copy_from_user(&f, (void __user *)arg, sizeof(f) - sizeof(u32)))
		return -EFAULT;
	fs = get_fs();
	set_fs(get_ds());
	retval = sys_ioctl(fd, cmd, (unsigned long)&f);
	set_fs(fs);
	return retval;
#else
	struct fib_ioctl __user *f;

	f = compat_alloc_user_space(sizeof(*f));
	if (!access_ok(VERIFY_WRITE, f, sizeof(*f)))
		return -EFAULT;

	clear_user(f, sizeof(*f));
	if (copy_in_user(f, (void __user *)arg, sizeof(*f) - sizeof(u32)))
		return -EFAULT;

	return sys_ioctl(fd, cmd, (unsigned long)f);
#endif
}
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
#define sys_ioctl NULL	/* register_ioctl32_conversion defaults to this when NULL passed in as a handler */
#endif
#endif

#endif
#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2,5,0))
#if (!defined(SCSI_HAS_SCSI_IN_DETECTION) && !defined(__VMKERNEL_MODULE__))
static struct Scsi_Host * aac_dummy;
#endif

/**
 *	aac_detect	-	Probe for aacraid cards
 *	@template: SCSI driver template
 *
 *	This is but a stub to convince the 2.4 scsi layer to scan targets,
 *	the pci scan has already picked up the adapters.
 */
static int aac_detect(Scsi_Host_Template *template)
{
#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
	printk(KERN_INFO "aac_detect(%p)\n", template);
#endif
#if (!defined(SCSI_HAS_SCSI_IN_DETECTION) && !defined(__VMKERNEL_MODULE__))
	/* By changing the host list we trick a scan */
	if (aac_dummy) {
#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
		printk(KERN_INFO "scsi_host_put(%p)\n", aac_dummy);
#endif
		scsi_host_put(aac_dummy);
		aac_dummy = NULL;
	}
#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
	printk(KERN_INFO "aac_detect()=%d\n", !list_empty(&aac_devices));
#endif
	return !list_empty(&aac_devices);
#else
	return template->present = 1;
#endif
}

#endif

/**
 *	aac_queuecommand	-	queue a SCSI command
 *	@cmd:		SCSI command to queue
 *	@done:		Function to call on command completion
 *
 *	Queues a command for execution by the associated Host Adapter.
 *
 *	TODO: unify with aac_scsi_cmd().
 */ 

static int aac_queuecommand(struct scsi_cmnd *cmd, void (*done)(struct scsi_cmnd *))
{
#if (defined(AAC_DEBUG_INSTRUMENT_TIMING))
	u64 lba;
	u32 count = 0;
	struct timeval now;
	do_gettimeofday(&now);
	if ((cmd->cmnd[0] == WRITE_6)	/* 6 byte command */
	 || (cmd->cmnd[0] == READ_6)) {
		lba = ((cmd->cmnd[1] & 0x1F) << 16)
		    | (cmd->cmnd[2] << 8) | cmd->cmnd[3];
		count = cmd->cmnd[4];
		if (count == 0)
			count = 256;
#if (defined(WRITE_16))
	} else if ((cmd->cmnd[0] == WRITE_16) /* 16 byte command */
	 || (cmd->cmnd[0] == READ_16)) {
		lba = ((u64)cmd->cmnd[2] << 56)
		    | ((u64)cmd->cmnd[3] << 48)
		    | ((u64)cmd->cmnd[4] << 40)
		    | ((u64)cmd->cmnd[9] << 32)
		    | (cmd->cmnd[6] << 24)
		    | (cmd->cmnd[7] << 16)
		    | (cmd->cmnd[8] << 8) | cmd->cmnd[9];
		count = (cmd->cmnd[10] << 24)
		      | (cmd->cmnd[11] << 16)
		      | (cmd->cmnd[12] << 8) | cmd->cmnd[13];
#endif
	} else if ((cmd->cmnd[0] == WRITE_12) /* 12 byte command */
	 || (cmd->cmnd[0] == READ_12)) {
		lba = (cmd->cmnd[2] << 24)
		    | (cmd->cmnd[3] << 16)
		    | (cmd->cmnd[4] << 8) | cmd->cmnd[5];
		count = (cmd->cmnd[6] << 24)
		      | (cmd->cmnd[7] << 16)
		      | (cmd->cmnd[8] << 8) | cmd->cmnd[9];
	} else if ((cmd->cmnd[0] == WRITE_10) /* 10 byte command */
	 || (cmd->cmnd[0] == READ_10)) {
		lba = (cmd->cmnd[2] << 24)
		    | (cmd->cmnd[3] << 16)
		    | (cmd->cmnd[4] << 8) | cmd->cmnd[5];
		count = (cmd->cmnd[7] << 8) | cmd->cmnd[8];
	} else
		lba = (u64)(long)cmd;
	printk(((count)
	  ? KERN_DEBUG "%lu.%06lu q%lu %llu[%u]\n"
	  : KERN_DEBUG "%lu.%06lu q%lu 0x%llx\n"),
 	  now.tv_sec % 100, now.tv_usec,
	  (unsigned long)((struct aac_dev *)cmd->device->host->hostdata)->queues->queue[AdapNormCmdQueue].numpending,
	  lba, count);
#endif
	cmd->scsi_done = done;
	cmd->SCp.phase = AAC_OWNER_LOWLEVEL;
	return (aac_scsi_cmd(cmd) ? FAILED : 0);
} 

/**
 *	aac_info		-	Returns the host adapter name
 *	@shost:		Scsi host to report on
 *
 *	Returns a static string describing the device in question
 */

static const char *aac_info(struct Scsi_Host *shost)
{
#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2,5,0))
	struct aac_dev *dev;
#if (!defined(SCSI_HAS_SCSI_IN_DETECTION) && !defined(__VMKERNEL_MODULE__))
	if (shost == aac_dummy)
		return shost->hostt->name;
#endif
	dev = (struct aac_dev *)shost->hostdata;
	if (!dev
	 || (dev->cardtype >= (sizeof(aac_drivers)/sizeof(aac_drivers[0]))))
		return shost->hostt->name;
	if (dev->scsi_host_ptr != shost)
		return shost->hostt->name;
#else
	struct aac_dev *dev = (struct aac_dev *)shost->hostdata;
#endif
	return aac_drivers[dev->cardtype].name;
}

/**
 *	aac_get_driver_ident
 * 	@devtype: index into lookup table
 *
 * 	Returns a pointer to the entry in the driver lookup table.
 */

struct aac_driver_ident* aac_get_driver_ident(int devtype)
{
	return &aac_drivers[devtype];
}

/**
 *	aac_biosparm	-	return BIOS parameters for disk
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
 *	@disk: SCSI disk object to process
 *	@device: kdev_t of the disk in question
#endif
 *	@sdev: The scsi device corresponding to the disk
 *	@bdev: the block device corresponding to the disk
 *	@capacity: the sector capacity of the disk
 *	@geom: geometry block to fill in
 *
 *	Return the Heads/Sectors/Cylinders BIOS Disk Parameters for Disk.  
 *	The default disk geometry is 64 heads, 32 sectors, and the appropriate 
 *	number of cylinders so as not to exceed drive capacity.  In order for 
 *	disks equal to or larger than 1 GB to be addressable by the BIOS
 *	without exceeding the BIOS limitation of 1024 cylinders, Extended 
 *	Translation should be enabled.   With Extended Translation enabled, 
 *	drives between 1 GB inclusive and 2 GB exclusive are given a disk 
 *	geometry of 128 heads and 32 sectors, and drives above 2 GB inclusive 
 *	are given a disk geometry of 255 heads and 63 sectors.  However, if 
 *	the BIOS detects that the Extended Translation setting does not match 
 *	the geometry in the partition table, then the translation inferred 
 *	from the partition table will be used by the BIOS, and a warning may 
 *	be displayed.
 */
 
static int aac_biosparm(
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	struct scsi_device *sdev, struct block_device *bdev, sector_t capacity,
#else
	Scsi_Disk *disk, kdev_t dev,
#endif
	int *geom)
{
	struct diskparm *param = (struct diskparm *)geom;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	unsigned char *buf;
#else
	struct buffer_head * buf;
	sector_t capacity = disk->capacity;
#endif

	dprintk((KERN_DEBUG "aac_biosparm.\n"));

	/*
	 *	Assuming extended translation is enabled - #REVISIT#
	 */
	if (capacity >= 2 * 1024 * 1024) { /* 1 GB in 512 byte sectors */
		if(capacity >= 4 * 1024 * 1024) { /* 2 GB in 512 byte sectors */
			param->heads = 255;
			param->sectors = 63;
		} else {
			param->heads = 128;
			param->sectors = 32;
		}
	} else {
		param->heads = 64;
		param->sectors = 32;
	}

	param->cylinders = cap_to_cyls(capacity, param->heads * param->sectors);

	/* 
	 *	Read the first 1024 bytes from the disk device, if the boot
	 *	sector partition table is valid, search for a partition table
	 *	entry whose end_head matches one of the standard geometry 
	 *	translations ( 64/32, 128/32, 255/63 ).
	 */
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	buf = scsi_bios_ptable(bdev);
#else
	buf = bread(MKDEV(MAJOR(dev), MINOR(dev)&~0xf), 0, block_size(dev));
#endif
	if(buf == NULL)
		return 0;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	if(*(__le16 *)(buf + 0x40) == cpu_to_le16(0xaa55)) {
		struct partition *first = (struct partition * )buf;
#else
	if(*(unsigned short *)(buf->b_data + 0x1fe) == cpu_to_le16(0xaa55)) {
		struct partition *first = (struct partition * )(buf->b_data + 0x1be);
#endif
		struct partition *entry = first;
		int saved_cylinders = param->cylinders;
		int num;
		unsigned char end_head, end_sec;

		for(num = 0; num < 4; num++) {
			end_head = entry->end_head;
			end_sec = entry->end_sector & 0x3f;

			if(end_head == 63) {
				param->heads = 64;
				param->sectors = 32;
				break;
			} else if(end_head == 127) {
				param->heads = 128;
				param->sectors = 32;
				break;
			} else if(end_head == 254) {
				param->heads = 255;
				param->sectors = 63;
				break;
			}
			entry++;
		}

		if (num == 4) {
			end_head = first->end_head;
			end_sec = first->end_sector & 0x3f;
		}

		param->cylinders = cap_to_cyls(capacity, param->heads * param->sectors);
		if (num < 4 && end_sec == param->sectors) {
			if (param->cylinders != saved_cylinders)
				dprintk((KERN_DEBUG "Adopting geometry: heads=%d, sectors=%d from partition table %d.\n",
					param->heads, param->sectors, num));
		} else if (end_head > 0 || end_sec > 0) {
			dprintk((KERN_DEBUG "Strange geometry: heads=%d, sectors=%d in partition table %d.\n",
				end_head + 1, end_sec, num));
			dprintk((KERN_DEBUG "Using geometry: heads=%d, sectors=%d.\n",
					param->heads, param->sectors));
		}
	}
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	kfree(buf);
#else
	brelse(buf);
#endif
	return 0;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
/**
 *	aac_slave_configure		-	compute queue depths
 *	@sdev:	SCSI device we are considering
 *
 *	Selects queue depths for each target device based on the host adapter's
 *	total capacity and the queue depth supported by the target device.
 *	A queue depth of one automatically disables tagged queueing.
 */

static int aac_slave_configure(struct scsi_device *sdev)
{
#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,10)) || defined(BLIST_NO_ULD_ATTACH))
	if ((sdev->type == TYPE_DISK) && (sdev->channel != 0)) {
		struct aac_dev *aac = (struct aac_dev *)sdev->host->hostdata;
		if (!aac->raid_scsi_mode || (sdev->channel != 2))
			sdev->no_uld_attach = 1;
	}
#endif
	if (sdev->tagged_supported && (sdev->type == TYPE_DISK)
#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,10)) || defined(BLIST_NO_ULD_ATTACH))
	 && (sdev->no_uld_attach == 0)
#endif
	 && (sdev->channel == 0)) {
		struct scsi_device * dev;
		struct Scsi_Host *host = sdev->host;
		unsigned num_lsu = 0;
		unsigned num_one = 0;
		unsigned depth;

		__shost_for_each_device(dev, host) {
			if (dev->tagged_supported && (dev->type == TYPE_DISK)
#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,10)) || defined(BLIST_NO_ULD_ATTACH))
			 && (dev->no_uld_attach == 0)
#endif
			 && (dev->channel == 0))
				++num_lsu;
			else
				++num_one;
		}
		if (num_lsu == 0)
			++num_lsu;
		depth = (host->can_queue - num_one) / num_lsu;
		if (depth > 256)
			depth = 256;
		else if (depth < 2)
			depth = 2;
		scsi_adjust_queue_depth(sdev, MSG_ORDERED_TAG, depth);
		if (!(((struct aac_dev *)host->hostdata)->adapter_info.options
		  & AAC_OPT_NEW_COMM))
			blk_queue_max_segment_size(sdev->request_queue, 65536);
	} else
		scsi_adjust_queue_depth(sdev, 0, 1);

#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)) && defined(AAC_EXTENDED_TIMEOUT))
	{
		extern int extendedtimeout;

		if (extendedtimeout != -1)
			sdev->timeout = extendedtimeout * HZ;
	}
#endif
	return 0;
}
#else
/**
 *	aac_queuedepth		-	compute queue depths
 *	@host:	SCSI host in question
 *	@dev:	SCSI device we are considering
 *
 *	Selects queue depths for each target device based on the host adapter's
 *	total capacity and the queue depth supported by the target device.
 *	A queue depth of one automatically disables tagged queueing.
 */

static void aac_queuedepth(struct Scsi_Host * host, struct scsi_device * dev )
{
	struct scsi_device * dptr;
	unsigned num = 0;
	unsigned depth;

#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
	printk(KERN_INFO "aac_queuedepth(%p,%p)\n", host, dev);
#endif
	for(dptr = dev; dptr != NULL; dptr = dptr->next)
		if((dptr->host == host) && (dptr->type == 0))
			++num;

	dprintk((KERN_DEBUG "can_queue=%d num=%d\n", host->can_queue, num));
#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
	printk(KERN_INFO "can_queue=%d num=%d\n", host->can_queue, num);
#endif
	if (num == 0)
		++num;
	depth = host->can_queue / num;
	if (depth > 255)
		depth = 255;
	else if (depth < 2)
		depth = 2;
	dprintk((KERN_DEBUG "aac_queuedepth.\n"));
	dprintk((KERN_DEBUG "Device #   Q Depth   Online\n"));
	dprintk((KERN_DEBUG "---------------------------\n"));
#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
	printk(KERN_INFO "aac_queuedepth.\n");
	printk(KERN_INFO "Device #   Q Depth   Online\n");
	printk(KERN_INFO "---------------------------\n");
#endif
	for(dptr = dev; dptr != NULL; dptr = dptr->next)
	{
		if(dptr->host == host)
		{
			dptr->queue_depth = depth;
			dprintk((KERN_DEBUG "  %2d         %d        %d\n", 
				dptr->id, dptr->queue_depth, scsi_device_online(dptr)));
#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
	printk(KERN_INFO "  %2d         %d        %d\n", dptr->id, dptr->queue_depth, scsi_device_online(dptr));
#endif
		}
	}
}
#endif

static int aac_ioctl(struct scsi_device *sdev, int cmd, void __user * arg)
{
	struct aac_dev *dev = (struct aac_dev *)sdev->host->hostdata;
#if (defined(AAC_DEBUG_INSTRUMENT_IOCTL))
	int retval;
	printk("aac_ioctl(%p, %x, %p)\n", sdev, cmd, arg);
	retval = aac_do_ioctl(dev, cmd, arg);
	printk("aac_ioctl returns %d\n", retval);
	return retval;
#endif
	return aac_do_ioctl(dev, cmd, arg);
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
/**
 *	aac_eh_device_reset	-	Reset command handling
 *	@cmd:	SCSI command block causing the reset
 *
 *	Issue a reset of a SCSI device. We are ourselves not truely a SCSI
 *	controller and our firmware will do the work for us anyway. Thus this
 *	is a no-op. We just return FAILED.
 */

static int aac_eh_device_reset(struct scsi_cmnd *cmd)
{
	return FAILED;
}

/**
 *	aac_eh_bus_reset	-	Reset command handling
 *	@scsi_cmd:	SCSI command block causing the reset
 *
 *	Issue a reset of a SCSI bus. We are ourselves not truely a SCSI
 *	controller and our firmware will do the work for us anyway. Thus this
 *	is a no-op. We just return FAILED.
 */

static int aac_eh_bus_reset(struct scsi_cmnd* cmd)
{
	return FAILED;
}

#endif
/*
 *	aac_eh_reset	- Reset command handling
 *	@scsi_cmd:	SCSI command block causing the reset
 *
 */
#if (defined(__arm__))
//DEBUG
#define AAC_DEBUG_INSTRUMENT_RESET
#endif
#if (defined(AAC_DEBUG_INSTRUMENT_RESET))
# undef dprintk
# define dprintk(x)	printk x
#endif
static int aac_eh_reset(struct scsi_cmnd* cmd)
{
#if (!defined(AAC_DEBUG_INSTRUMENT_RESET) && defined(__arm__))
	return SUCCESS; /* Cause an immediate retry of the command with a ten second delay after successful tur */
#else
	struct scsi_device * dev = cmd->device;
	struct Scsi_Host * host = dev->host;
	struct scsi_cmnd * command;
	int count;
	struct aac_dev * aac;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	unsigned long flags;
#endif

	printk(KERN_ERR "%s: Host adapter reset request. SCSI hang ?\n", 
					AAC_DRIVERNAME);
	aac = (struct aac_dev *)host->hostdata;
	fwprintf((aac, HBA_FLAGS_DBG_FW_PRINT_B, "SCSI hang ?"));
	if (nblank(dprintk(x))) {
		int active = 0;
		unsigned long DebugFlags = aac->FwDebugFlags;

		active = active;
		dprintk((KERN_ERR
		  "%s: Outstanding commands on (%d,%d,%d,%d):\n",
		  AAC_DRIVERNAME,
		  host->host_no, dev->channel, dev->id, dev->lun));
		aac->FwDebugFlags |= FW_DEBUG_FLAGS_NO_HEADERS_B;
		fwprintf((aac, HBA_FLAGS_DBG_FW_PRINT_B,
		  "%s: Outstanding commands on (%d,%d,%d,%d):\n",
		  AAC_DRIVERNAME,
		  host->host_no, dev->channel, dev->id, dev->lun));
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
		spin_lock_irqsave(&dev->list_lock, flags);
		list_for_each_entry(command, &dev->cmd_list, list)
#else
		for(command = dev->device_queue; command; command = command->next)
#endif
		{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,12))
			if ((command->state != SCSI_STATE_FINISHED)
			 && (command->state != 0))
#endif
			dprintk((KERN_ERR
			  "%4d %c%c %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
			  active++,
			  (command->SCp.phase == AAC_OWNER_FIRMWARE) ? 'A' : 'C',
			  (cmd == command) ? '*' : ' ',
			  command->cmnd[0], command->cmnd[1], command->cmnd[2],
			  command->cmnd[3], command->cmnd[4], command->cmnd[5],
			  command->cmnd[6], command->cmnd[7], command->cmnd[8],
			  command->cmnd[9]));
			fwprintf((aac, HBA_FLAGS_DBG_FW_PRINT_B,
			  "%4d %c%c %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
			  active++,
			  (command->SCp.phase == AAC_OWNER_FIRMWARE) ? 'A' : 'C',
			  (cmd == command) ? '*' : ' ',
			  command->cmnd[0], command->cmnd[1], command->cmnd[2],
			  command->cmnd[3], command->cmnd[4], command->cmnd[5],
			  command->cmnd[6], command->cmnd[7], command->cmnd[8],
			  command->cmnd[9]));
		}
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
		spin_unlock_irqrestore(&dev->list_lock, flags);
#endif
		aac->FwDebugFlags = DebugFlags;
	}

	if ((count = aac_check_health(aac)))
		return count;
	/*
	 * Wait for all commands to complete to this specific
	 * target (block maximum 60 seconds).
	 */
	for (count = 60; count; --count) {
		int active = aac->in_reset;
	
		if (active == 0)
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
		__shost_for_each_device(dev, host) {
			spin_lock_irqsave(&dev->list_lock, flags);
			list_for_each_entry(command, &dev->cmd_list, list) {
				if ((command != cmd) &&
				    (command->SCp.phase == AAC_OWNER_FIRMWARE)) {
					active++;
					break;
				}
			}
			spin_unlock_irqrestore(&dev->list_lock, flags);
			if (active)
				break;

		}
#else
		for (dev = host->host_queue; dev != (struct scsi_device *)NULL; dev = dev->next) {
			for(command = dev->device_queue; command; command = command->next) {
				if ((command != cmd)
				 && (command->SCp.phase == AAC_OWNER_FIRMWARE)) {
					++active;
					break;
				}
			}
		}
#endif
		/*
		 * We can exit If all the commands are complete
		 */
		if (active == 0)
			return SUCCESS;
#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,12))
#if (defined(SCSI_HAS_HOST_LOCK) || (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)))
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,21)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,4,21)) || !defined(CONFIG_CFGNAME))
		spin_unlock_irq(host->host_lock);
#else
		spin_unlock_irq(host->lock);
#endif
#else
		spin_unlock_irq(&io_request_lock);
#endif
#endif
		ssleep(1);
#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,12))
#if (defined(SCSI_HAS_HOST_LOCK) || (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)))
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,21)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,4,21)) || !defined(CONFIG_CFGNAME))
		spin_lock_irq(host->host_lock);
#else
		spin_lock_irq(host->lock);
#endif
#else
		spin_lock_irq(&io_request_lock);
#endif
#endif
	}
	printk(KERN_ERR "%s: SCSI bus appears hung\n", AAC_DRIVERNAME);
	fwprintf((aac, HBA_FLAGS_DBG_FW_PRINT_B, "SCSI bus appears hung"));
//	return -ETIMEDOUT;
	return SUCCESS; /* Cause an immediate retry of the command with a ten second delay after successful tur */
#endif
}
#if (defined(AAC_DEBUG_INSTRUMENT_RESET))
/* We are making an assumption that dprintk was turned off */
# undef dprintk
# define dprintk(x)
#endif
#if (defined(SCSI_HAS_DUMP))
#if (defined(SCSI_HAS_DUMP_SANITY_CHECK))
static int aac_sanity_check(struct scsi_device * sdev)
{
	return 0;
}

#endif
static void aac_poll(struct scsi_device * sdev)
{
	struct Scsi_Host * shost = sdev->host;
	struct aac_dev *dev = (struct aac_dev *)shost->hostdata;
	unsigned long flags;

#if (defined(SCSI_HAS_HOST_LOCK) || (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)))
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,21)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,4,21)) || !defined(CONFIG_CFGNAME))
	spin_lock_irqsave(shost->host_lock, flags);
#else
	spin_lock_irqsave(shost->lock, flags);
#endif
#else
	spin_lock_irqsave(&io_request_lock, flags);
#endif
	aac_adapter_intr(dev);
#if (defined(SCSI_HAS_HOST_LOCK) || (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)))
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,21)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,4,21)) || !defined(CONFIG_CFGNAME))
	spin_unlock_irqrestore(shost->host_lock, flags);
#else
	spin_unlock_irqrestore(shost->lock, flags);
#endif
#else
	spin_unlock_irqrestore(&io_request_lock, flags);
#endif
}
#endif

/**
 *	aac_cfg_open		-	open a configuration file
 *	@inode: inode being opened
 *	@file: file handle attached
 *
 *	Called when the configuration device is opened. Does the needed
 *	set up on the handle and then returns
 *
 *	Bugs: This needs extending to check a given adapter is present
 *	so we can support hot plugging, and to ref count adapters.
 */

static int aac_cfg_open(struct inode *inode, struct file *file)
{
	struct aac_dev *aac;
	unsigned minor_number = iminor(inode);
	int err = -ENODEV;

	list_for_each_entry(aac, &aac_devices, entry) {
		if (aac->id == minor_number) {
			file->private_data = aac;
			err = 0;
			break;
		}
	}

	return err;
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
/**
 *	aac_cfg_release		-	close down an AAC config device
 *	@inode: inode of configuration file
 *	@file: file handle of configuration file
 *	
 *	Called when the last close of the configuration file handle
 *	is performed.
 */
 
static int aac_cfg_release(struct inode * inode, struct file * file )
{
	return 0;
}

#endif
/**
 *	aac_cfg_ioctl		-	AAC configuration request
 *	@inode: inode of device
 *	@file: file handle
 *	@cmd: ioctl command code
 *	@arg: argument
 *
 *	Handles a configuration ioctl. Currently this involves wrapping it
 *	up and feeding it into the nasty windowsalike glue layer.
 *
 *	Bugs: Needs locking against parallel ioctls lower down
 *	Bugs: Needs to handle hot plugging
 */
 
static int aac_cfg_ioctl(struct inode *inode,  struct file *file,
		unsigned int cmd, unsigned long arg)
{
#if 1 || defined(__VMWARE__)
	struct aac_dev *aac;
#endif

#if 1 || defined(__VMWARE__)
	if (!capable(CAP_SYS_ADMIN))
		return -EPERM;
#endif

#if defined(__VMKERNEL_MODULE__)
	list_for_each_entry(aac, &aac_devices, entry) {
#if 0 && defined(__VMWARE__)
		if (aac->id == iminor(inode);
#else
		if (aac->id == iminor(inode)) {
#endif
			file->private_data = aac;
			break;
		}
	}
	if (file->private_data == NULL)
		return -ENODEV;
#endif
#if (defined(AAC_DEBUG_INSTRUMENT_IOCTL))
	{
		int retval;
		if (cmd != FSACTL_GET_NEXT_ADAPTER_FIB)
			printk("aac_cfg_ioctl(%p,%p,%x,%lx)\n",
			  inode, file, cmd, arg);
		retval = aac_do_ioctl(
		  file->private_data, cmd, (void __user *)arg);
		if (cmd != FSACTL_GET_NEXT_ADAPTER_FIB)
			printk("aac_cfg_ioctl returns %d\n", retval);
		return retval;
	}
#else
	return aac_do_ioctl(file->private_data, cmd, (void __user *)arg);
#endif
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11))
#ifdef CONFIG_COMPAT
static long aac_compat_do_ioctl(struct aac_dev *dev, unsigned cmd, unsigned long arg)
{
	long ret;
	lock_kernel();
	switch (cmd) { 
	case FSACTL_MINIPORT_REV_CHECK:
	case FSACTL_SENDFIB:
	case FSACTL_OPEN_GET_ADAPTER_FIB:
	case FSACTL_CLOSE_GET_ADAPTER_FIB:
	case FSACTL_SEND_RAW_SRB:
	case FSACTL_GET_PCI_INFO:
	case FSACTL_QUERY_DISK:
	case FSACTL_DELETE_DISK:
	case FSACTL_FORCE_DELETE_DISK:
	case FSACTL_GET_CONTAINERS: 
	case FSACTL_GET_VERSION_MATCHING:
	case FSACTL_SEND_LARGE_FIB:
#if (defined(FSACTL_REGISTER_FIB_SEND))
	case FSACTL_REGISTER_FIB_SEND:
#endif
		ret = aac_do_ioctl(dev, cmd, (void __user *)arg);
		break;

	case FSACTL_GET_NEXT_ADAPTER_FIB: {
		struct fib_ioctl __user *f;
		
		f = compat_alloc_user_space(sizeof(*f));
		ret = 0;
		if (clear_user(f, sizeof(*f) != sizeof(*f)))
			ret = -EFAULT;
		if (copy_in_user(f, (void __user *)arg, sizeof(struct fib_ioctl) - sizeof(u32)))
			ret = -EFAULT;
		if (!ret)
			ret = aac_do_ioctl(dev, cmd, (void __user *)arg);
		break;
	}

	default:
#if (defined(AAC_CSMI))
		ret = aac_csmi_ioctl(dev, cmd, (void __user *)arg);
		if (ret == -ENOTTY)
#endif
		ret = -ENOIOCTLCMD; 
		break;
	} 
	unlock_kernel();
	return ret;
}

static int aac_compat_ioctl(struct scsi_device *sdev, int cmd, void __user *arg)
{
	struct aac_dev *dev = (struct aac_dev *)sdev->host->hostdata;
	return aac_compat_do_ioctl(dev, cmd, (unsigned long)arg);
}

static long aac_compat_cfg_ioctl(struct file *file, unsigned cmd, unsigned long arg)
{
#if 1 || defined(__VMWARE__)
	if (!capable(CAP_SYS_ADMIN))
		return -EPERM;
#endif

	return aac_compat_do_ioctl((struct aac_dev *)file->private_data, cmd, arg);
}
#endif
#endif
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))

#define class_device Scsi_Host
#define class_to_shost(class_dev) class_dev
#endif

static ssize_t aac_show_model(struct class_device *class_dev,
		char *buf)
{
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
	int len;

	if (dev->supplement_adapter_info.AdapterTypeText[0]) {
		char * cp = dev->supplement_adapter_info.AdapterTypeText;
		while (*cp && *cp != ' ')
			++cp;
		while (*cp == ' ')
			++cp;
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(buf, PAGE_SIZE, "%s\n", cp);
#else
		len = sprintf(buf, "%s\n", cp);
#endif
	} else
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(buf, PAGE_SIZE, "%s\n",
#else
		len = sprintf(buf, "%s\n",
#endif
		  aac_drivers[dev->cardtype].model);
	return len;
}

static ssize_t aac_show_vendor(struct class_device *class_dev,
		char *buf)
{
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
	int len;

	if (dev->supplement_adapter_info.AdapterTypeText[0]) {
		char * cp = dev->supplement_adapter_info.AdapterTypeText;
		while (*cp && *cp != ' ')
			++cp;
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(buf, PAGE_SIZE, "%.*s\n",
#else
		len = sprintf(buf, "%.*s\n",
#endif
		  (int)(cp - (char *)dev->supplement_adapter_info.AdapterTypeText),
		  dev->supplement_adapter_info.AdapterTypeText);
	} else
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(buf, PAGE_SIZE, "%s\n",
#else
		len = sprintf(buf, "%s\n",
#endif
		  aac_drivers[dev->cardtype].vname);
	return len;
}

static ssize_t aac_show_flags(struct class_device *class_dev, char *buf)
{
	int len = 0;
#if ((defined(SERVICE_ACTION_IN) && defined(SAI_READ_CAPACITY_16)) || defined(AAC_DEBUG_INSTRUMENT_PENDING))
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
#endif

	if (nblank(dprintk(x)))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(buf, PAGE_SIZE, "dprintk\n");
#else
		len = sprintf(buf, "dprintk\n");
#endif
#	if (defined(AAC_DETAILED_STATUS_INFO))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
		  "AAC_DETAILED_STATUS_INFO\n");
#else
		len += sprintf(buf + len,
		  "AAC_DETAILED_STATUS_INFO\n");
#endif
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_AAC_CONFIG))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_AAC_CONFIG\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_AIF))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_AIF\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_IOCTL))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_IOCTL\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_TIMING))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_TIMING\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_RESET))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_RESET\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_FIB))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_FIB\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_2TB))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_2TB\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_IOCTL_SENDFIB))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_IOCTL_SENDFIB\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_IO))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_IO\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_SG))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_SG\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_SG_PROBE))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_SG_PROBE\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_VM_NAMESERVE))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_VM_NAMESERVE\n");
#	endif
#if (defined(SERVICE_ACTION_IN) && defined(SAI_READ_CAPACITY_16))
	if (dev->raw_io_interface && dev->raw_io_64)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "SAI_READ_CAPACITY_16\n");
#endif
#	if (defined(SCSI_HAS_VARY_IO))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "SCSI_HAS_VARY_IO\n");
#	endif
#	if (defined(SCSI_HAS_DUMP))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "SCSI_HAS_DUMP\n");
#	endif
#	if (defined(AAC_DEBUG_INSTRUMENT_PENDING))
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len += snprintf(buf + len, PAGE_SIZE - len,
#else
		len += sprintf(buf + len,
#endif
		  "AAC_DEBUG_INSTRUMENT_PENDING=%u\n",
		  dev->queues->queue[AdapNormCmdQueue].numpending);
#	endif
	return len;
}

static ssize_t aac_show_kernel_version(struct class_device *class_dev,
		char *buf)
{
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
	int len, tmp;

	tmp = le32_to_cpu(dev->adapter_info.kernelrev);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
	len = snprintf(buf, PAGE_SIZE, "%d.%d-%d[%d]\n", 
#else
	len = sprintf(buf, "%d.%d-%d[%d]\n", 
#endif
	  tmp >> 24, (tmp >> 16) & 0xff, tmp & 0xff,
	  le32_to_cpu(dev->adapter_info.kernelbuild));
	return len;
}

static ssize_t aac_show_monitor_version(struct class_device *class_dev,
		char *buf)
{
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
	int len, tmp;

	tmp = le32_to_cpu(dev->adapter_info.monitorrev);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
	len = snprintf(buf, PAGE_SIZE, "%d.%d-%d[%d]\n", 
#else
	len = sprintf(buf, "%d.%d-%d[%d]\n", 
#endif
	  tmp >> 24, (tmp >> 16) & 0xff, tmp & 0xff,
	  le32_to_cpu(dev->adapter_info.monitorbuild));
	return len;
}

static ssize_t aac_show_bios_version(struct class_device *class_dev,
		char *buf)
{
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
	int len, tmp;

	tmp = le32_to_cpu(dev->adapter_info.biosrev);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
	len = snprintf(buf, PAGE_SIZE, "%d.%d-%d[%d]\n", 
#else
	len = sprintf(buf, "%d.%d-%d[%d]\n", 
#endif
	  tmp >> 24, (tmp >> 16) & 0xff, tmp & 0xff,
	  le32_to_cpu(dev->adapter_info.biosbuild));
	return len;
}

static ssize_t aac_show_serial_number(struct class_device *class_dev,
		char *buf)
{
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
	int len = 0;

	if (le32_to_cpu(dev->adapter_info.serial[0]) != 0xBAD0)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(buf, PAGE_SIZE, "%x\n",
#else
		len = sprintf(buf, "%x\n",
#endif
		  le32_to_cpu(dev->adapter_info.serial[0]));
	return len;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
static ssize_t aac_store_reset_adapter(struct class_device *class_dev,
		const char *buf, size_t count)
{
	struct Scsi_Host *shost;
	unsigned long flags;
	int retval = -EACCES;

	if (!capable(CAP_SYS_ADMIN))
		return retval;
	shost = class_to_shost(class_dev);
	spin_lock_irqsave(shost->host_lock, flags);
	retval = aac_reset_adapter((struct aac_dev*)shost->hostdata);
	spin_unlock_irqrestore(shost->host_lock, flags);
	if (retval >= 0)
		retval = count;
	return retval;
}

static ssize_t aac_show_reset_adapter(struct class_device *class_dev,
		char *buf)
{
	struct aac_dev *dev = (struct aac_dev*)class_to_shost(class_dev)->hostdata;
	int len, tmp;

	tmp = aac_adapter_check_health(dev);
	if ((tmp == 0) && dev->in_reset)
		tmp=-EBUSY;
	len = snprintf(buf, PAGE_SIZE, "0x%x", tmp);
	return len;
}

static struct class_device_attribute aac_model = {
	.attr = {
		.name = "model",
		.mode = S_IRUGO,
	},
	.show = aac_show_model,
};
static struct class_device_attribute aac_vendor = {
	.attr = {
		.name = "vendor",
		.mode = S_IRUGO,
	},
	.show = aac_show_vendor,
};
static struct class_device_attribute aac_flags = {
	.attr = {
		.name = "flags",
		.mode = S_IRUGO,
	},
	.show = aac_show_flags,
};
static struct class_device_attribute aac_kernel_version = {
	.attr = {
		.name = "hba_kernel_version",
		.mode = S_IRUGO,
	},
	.show = aac_show_kernel_version,
};
static struct class_device_attribute aac_monitor_version = {
	.attr = {
		.name = "hba_monitor_version",
		.mode = S_IRUGO,
	},
	.show = aac_show_monitor_version,
};
static struct class_device_attribute aac_bios_version = {
	.attr = {
		.name = "hba_bios_version",
		.mode = S_IRUGO,
	},
	.show = aac_show_bios_version,
};
static struct class_device_attribute aac_serial_number = {
	.attr = {
		.name = "serial_number",
		.mode = S_IRUGO,
	},
	.show = aac_show_serial_number,
};
static struct class_device_attribute aac_reset = {
	.attr = {
		.name = "reset_host",
		.mode = S_IWUSR|S_IRUGO,
	},
	.store = aac_store_reset_adapter,
	.show = aac_show_reset_adapter,
};

static struct class_device_attribute *aac_attrs[] = {
	&aac_model,
	&aac_vendor,
	&aac_flags,
	&aac_kernel_version,
	&aac_monitor_version,
	&aac_bios_version,
	&aac_serial_number,
	&aac_reset,
	NULL
};
#endif
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || defined(CONFIG_SCSI_PROC_FS))

/**
 *	aac_procinfo	-	Implement /proc/scsi/<drivername>/<n>
 *	@proc_buffer: memory buffer for I/O
 *	@start_ptr: pointer to first valid data
 *	@offset: offset into file
 *	@bytes_available: space left
 *	@host_no: scsi host ident
 *	@write: direction of I/O
 *
 *	Used to export driver statistics and other infos to the world outside 
 *	the kernel using the proc file system. Also provides an interface to
 *	feed the driver with information.
 *
 *		For reads
 *			- if offset > 0 return 0
 *			- if offset == 0 write data to proc_buffer and set the start_ptr to
 *			beginning of proc_buffer, return the number of characters written.
 *		For writes
 *			- writes currently not supported, return 0
 *
 *	Bugs:	Only offset zero is handled
 */
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
#define shost_to_class(shost) &shost->shost_classdev
#else
#define shost_to_class(shost) shost
#endif

static int aac_procinfo(
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	struct Scsi_Host * shost,
#endif
	char *proc_buffer, char **start_ptr,off_t offset,
	int bytes_available,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	int host_no,
#endif
	int write)
{
	struct aac_dev * dev = (struct aac_dev *)NULL;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	struct Scsi_Host * shost = (struct Scsi_Host *)NULL;
#endif
	char *buf;
	int len;
	int total_len = 0;

#if (defined(AAC_LM_SENSOR) || defined(IOP_RESET))
	if(offset > 0)
#else
	if(write || offset > 0)
#endif
		return 0;
	*start_ptr = proc_buffer;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	list_for_each_entry(dev, &aac_devices, entry) {
		shost = dev->scsi_host_ptr;
		if (shost->host_no == host_no)
			break;
	}
	if (shost == (struct Scsi_Host *)NULL)
		return 0;
#endif
	dev = (struct aac_dev *)shost->hostdata;
	if (dev == (struct aac_dev *)NULL)
		return 0;
	if (!write) {
		buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
		if (!buf)
			return 0;
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(proc_buffer, bytes_available,
		  "Adaptec Raid Controller: %s\n", aac_driver_version);
#if 1 || defined(__VMWARE__)
		len = min( len, bytes_available );
#endif
#else
		len = sprintf(proc_buffer,
		  "Adaptec Raid Controller: %s\n", aac_driver_version);
#endif
		total_len += len;
		proc_buffer += len;
		len = aac_show_vendor(shost_to_class(shost), buf);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(proc_buffer, bytes_available - total_len,
		  "Vendor: %.*s  ", len - 1, buf);
#if 1 || defined(__VMWARE__)
		len = min( len, bytes_available - total_len );
#endif
#else
		len = sprintf(proc_buffer, "Vendor: %.*s  ", len - 1, buf);
#endif
		total_len += len;
		proc_buffer += len;
		len = aac_show_model(shost_to_class(shost), buf);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(proc_buffer, bytes_available - total_len,
		  "Model: %.*s", len, buf);
#if 1 || defined(__VMWARE__)
		len = min( len, bytes_available - total_len );
#endif
#else
		len = sprintf(proc_buffer, "Model: %.*s", len, buf);
#endif
		total_len += len;
		proc_buffer += len;
		len = aac_show_flags(shost_to_class(shost), buf);
		if (len) {
			char *cp = proc_buffer;
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
			len = snprintf(cp, bytes_available - total_len,
			  "flags=%.*s", len, buf);
#if 1 || defined(__VMWARE__)
			len = min( len, bytes_available - total_len );
#endif
#else
			len = sprintf(cp, "flags=%.*s", len, buf);
#endif
			total_len += len;
			proc_buffer += len;
			while (--len > 0) {
				if (*cp == '\n')
					*cp = '+';
				++cp;
			}
		}
		len = aac_show_kernel_version(shost_to_class(shost), buf);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(proc_buffer, bytes_available - total_len,
		  "kernel: %.*s", len, buf);
#if 1 || defined(__VMWARE__)
		len = min( len, bytes_available - total_len );
#endif
#else
		len = sprintf(proc_buffer, "kernel: %.*s", len, buf);
#endif
		total_len += len;
		proc_buffer += len;
		len = aac_show_monitor_version(shost_to_class(shost), buf);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(proc_buffer, bytes_available - total_len,
		  "monitor: %.*s", len, buf);
#if 1 || defined(__VMWARE__)
		len = min( len, bytes_available - total_len );
#endif
#else
		len = sprintf(proc_buffer, "monitor: %.*s", len, buf);
#endif
		total_len += len;
		proc_buffer += len;
		len = aac_show_bios_version(shost_to_class(shost), buf);
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
		len = snprintf(proc_buffer, bytes_available - total_len,
		  "bios: %.*s", len, buf);
#if 1 || defined(__VMWARE__)
		len = min( len, bytes_available - total_len );
#endif
#else
		len = sprintf(proc_buffer, "bios: %.*s", len, buf);
#endif
		total_len += len;
		proc_buffer += len;
		len = aac_show_serial_number(shost_to_class(shost), buf);
		if (len) {
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,4))
			len = snprintf(proc_buffer, bytes_available - total_len,
			  "serial: %.*s", len, buf);
#if 1 || defined(__VMWARE__)
		len = min( len, bytes_available - total_len );
#endif
#else
			len = sprintf(proc_buffer, "serial: %.*s", len, buf);
#endif
			total_len += len;
		}
		kfree(buf);
		return total_len;
	}
#if (defined(IOP_RESET))
	{
		static char reset[] = "reset_host";
		if (strnicmp (proc_buffer, reset, sizeof(reset) - 1) == 0) {
			unsigned long flags;

			if (!capable(CAP_SYS_ADMIN))
				return bytes_available;
#if (defined(SCSI_HAS_HOST_LOCK) || (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)))
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,21)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,4,21)) || !defined(CONFIG_CFGNAME))
			spin_lock_irqsave(shost->host_lock, flags);
#else
			spin_lock_irqsave(shost->lock, flags);
#endif
#else
			spin_lock_irqsave(&io_request_lock, flags);
#endif
			(void)aac_reset_adapter(dev);
#if (defined(SCSI_HAS_HOST_LOCK) || (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)))
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,21)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,4,21)) || !defined(CONFIG_CFGNAME))
			spin_unlock_irqrestore(shost->host_lock, flags);
#else
			spin_unlock_irqrestore(shost->lock, flags);
#endif
#else
			spin_unlock_irqrestore(&io_request_lock, flags);
#endif
			return bytes_available;
		}
	}
#endif
#if (defined(AAC_LM_SENSOR))
	{
		int ret, tmp, index;
		s32 temp[5];
		static char temperature[] = "temperature=";
		if (strnicmp (proc_buffer, temperature, sizeof(temperature) - 1))
			return bytes_available;
		for (index = 0;
		  index < (sizeof(temp)/sizeof(temp[0]));
		  ++index)
			temp[index] = 0x80000000;
		ret = sizeof(temperature) - 1;
		for (index = 0;
		  index < (sizeof(temp)/sizeof(temp[0]));
		  ++index) {
			int sign, mult, c;
			if (ret >= bytes_available)
				break;
			c = proc_buffer[ret];
			if (c == '\n') {
				++ret;
				break;
			}
			if (c == ',') {
				++ret;
				continue;
			}
			sign = 1;
			mult = 0;
			tmp = 0;
			if (c == '-') {
				sign = -1;
				++ret;
			}
			for (;
			  (ret < bytes_available) && ((c = proc_buffer[ret]));
			  ++ret) {
				if (('0' <= c) && (c <= '9')) {
					tmp *= 10;
					tmp += c - '0';
					mult *= 10;
				} else if ((c == '.') && (mult == 0))
					mult = 1;
				else
					break;
			}
			if ((ret < bytes_available)
			 && ((c == ',') || (c == '\n')))
				++ret;
			if (!mult)
				mult = 1;
			if (sign < 0)
				tmp = -tmp;
			temp[index] = ((tmp << 8) + (mult >> 1)) / mult;
			if (c == '\n')
				break;
		}
		ret = index;
		if (nblank(dprintk(x))) {
			for (index = 0; index < ret; ++index) {
				int sign;
				tmp = temp[index];
				sign = tmp < 0;
				if (sign)
					tmp = -tmp;
				dprintk((KERN_DEBUG "%s%s%d.%08doC",
				  (index ? "," : ""),
				  (sign ? "-" : ""),
				  tmp >> 8, (tmp % 256) * 390625));
			}
		}
		/* Send temperature message to Firmware */
		(void)aac_adapter_sync_cmd(dev, RCV_TEMP_READINGS,
		  ret, temp[0], temp[1], temp[2], temp[3], temp[4],
		  NULL, NULL, NULL, NULL, NULL);
		return bytes_available;
	}
#endif
	return 0;
}
#endif


static struct file_operations aac_cfg_fops = {
	.owner		= THIS_MODULE,
	.ioctl		= aac_cfg_ioctl,
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11))
#ifdef CONFIG_COMPAT
	.compat_ioctl   = aac_compat_cfg_ioctl,
#endif
#endif
	.open		= aac_cfg_open,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	.release	= aac_cfg_release
#endif
};
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) && defined(SCSI_HAS_DUMP))
static struct scsi_dump_ops aac_dump_ops = {
#if (defined(SCSI_HAS_DUMP_SANITY_CHECK))
	.sanity_check	= aac_sanity_check,
#endif
	.poll		= aac_poll,
};

#define aac_driver_template aac_driver_template_dump.hostt
static struct SHT_dump aac_driver_template_dump = {
	.hostt = {
#else

static struct scsi_host_template aac_driver_template = {
#endif
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	.detect				= aac_detect,
#endif
	.module				= THIS_MODULE,
	.name           		= "AAC",
	.proc_name			= AAC_DRIVERNAME,
	.info           		= aac_info,
	.ioctl          		= aac_ioctl,
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11))
#ifdef CONFIG_COMPAT
	.compat_ioctl			= aac_compat_ioctl,
#endif
#endif
	.queuecommand   		= aac_queuecommand,
	.bios_param     		= aac_biosparm,	
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || defined(CONFIG_SCSI_PROC_FS))
	.proc_info      		= aac_procinfo,
#endif
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	.shost_attrs			= aac_attrs,
	.slave_configure		= aac_slave_configure,
#endif
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	.eh_device_reset_handler	= aac_eh_device_reset,
	.eh_bus_reset_handler		= aac_eh_bus_reset,
#endif
	.eh_host_reset_handler		= aac_eh_reset,
	.can_queue      		= AAC_NUM_IO_FIB,	
	.this_id        		= MAXIMUM_NUM_CONTAINERS,
	.sg_tablesize   		= 16,
	.max_sectors    		= 128,
#if (AAC_NUM_IO_FIB > 256)
	.cmd_per_lun			= 256,
#else		
	.cmd_per_lun    		= AAC_NUM_IO_FIB, 
#endif	
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	.use_new_eh_code		= 1, 
#endif
	.use_clustering			= ENABLE_CLUSTERING,
	.emulated                       = 1,
#if (defined(SCSI_HAS_VARY_IO))
	.vary_io			= 1,
#endif
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0))
#if (defined(SCSI_HAS_DUMP))
#if (defined(SCSI_HAS_DUMP_SANITY_CHECK))
	.dump_sanity_check		= aac_sanity_check,
#endif
	.dump_poll			= aac_poll,
#endif
#elif (defined(SCSI_HAS_DUMP))
	.disk_dump			= 1,
},
	.dump_ops			= &aac_dump_ops,
#endif
};


static int __devinit aac_probe_one(struct pci_dev *pdev,
		const struct pci_device_id *id)
{
	unsigned index = id->driver_data;
	struct Scsi_Host *shost;
	struct aac_dev *aac;
	struct list_head *insert = &aac_devices;
	int error = -ENODEV;
	int unique_id = 0;
#if (defined(__arm__) || defined(CONFIG_EXTERNAL))
	static struct pci_dev * slave = NULL;
	static int nslave = 0;
#endif

#if (defined(__arm__) || defined(CONFIG_EXTERNAL))
	if (aac_drivers[index].quirks & AAC_QUIRK_SLAVE) {
		/* detect adjoining slaves */
		if (slave) {
			if ((pci_resource_start(pdev, 0)
			  + pci_resource_len(pdev, 0))
			  == pci_resource_start(slave, 0))
				slave = pdev;
			else if ((pci_resource_start(slave, 0)
			  + (pci_resource_len(slave, 0) * nslave))
			  != pci_resource_start(pdev, 0)) {
				printk(KERN_WARNING
				  "%s: multiple sets of slave controllers discovered\n",
				  AAC_DRIVERNAME);
				nslave = 0;
				slave = pdev;
			}
		} else
			slave = pdev;
		if (pci_resource_start(slave,0)) {
			error = pci_enable_device(pdev);
			if (error) {
				printk(KERN_WARNING
				  "%s: failed to enable slave\n",
				  AAC_DRIVERNAME);
				nslave = 0;
				slave = NULL;
				return error;
			}
			++nslave;
			pci_set_master(pdev);
		} else {
			printk(KERN_WARNING
			  "%s: slave BAR0 is not set\n", AAC_DRIVERNAME);
			nslave = 0;
			slave = NULL;
			return error;
		}
		return 1;
	}
#endif
	list_for_each_entry(aac, &aac_devices, entry) {
		if (aac->id > unique_id)
			break;
		insert = &aac->entry;
		unique_id++;
	}

	error = pci_enable_device(pdev);
	if (error)
		goto out;
	error = -ENODEV;
#if (defined(__arm__) || defined(CONFIG_EXTERNAL))
	if ((aac_drivers[index].quirks & AAC_QUIRK_MASTER) && (slave)) {
		unsigned long base = pci_resource_start(pdev, 0);
		struct master_registers {
			u32 x[51];
			u32	E_CONFIG1;
			u32 y[3];
			u32	E_CONFIG2;
		} __iomem * map = ioremap(base, AAC_MIN_FOOTPRINT_SIZE);
		if (!map) {
			printk(KERN_WARNING
			  "%s: unable to map master adapter to configure slaves.\n",
			  AAC_DRIVERNAME);
		} else {
			((struct master_registers *)map)->E_CONFIG2
			  = cpu_to_le32(pci_resource_start(slave, 0));
			((struct master_registers *)map)->E_CONFIG1
			  = cpu_to_le32(0x5A000000 + nslave);
			iounmap(map);
		}
		nslave = 0;
		slave = NULL;
	}
#endif

	if (pci_set_dma_mask(pdev, DMA_32BIT_MASK) || 
			pci_set_consistent_dma_mask(pdev, DMA_32BIT_MASK))
		goto out;
	/*
	 * If the quirk31 bit is set, the adapter needs adapter
	 * to driver communication memory to be allocated below 2gig
	 */
	if (aac_drivers[index].quirks & AAC_QUIRK_31BIT) 
		if (pci_set_dma_mask(pdev, 0x7FFFFFFFULL) ||
				pci_set_consistent_dma_mask(pdev, 0x7FFFFFFFULL))
			goto out;
	
	pci_set_master(pdev);

	aac_driver_template.name = aac_drivers[index].name;
#if defined(__VMKERNEL_MODULE__)
	shost = vmk_scsi_register(&aac_driver_template, sizeof(struct aac_dev), pdev->bus->number, pdev->devfn);
#else
	shost = scsi_host_alloc(&aac_driver_template, sizeof(struct aac_dev));
#endif
#if (defined(AAC_DEBUG_INSTRUMENT_INIT))
	printk(KERN_INFO "scsi_host_alloc(%p,%d)=%p\n",
	  &aac_driver_template, sizeof(struct aac_dev), shost);
#endif
	if (!shost)
		goto out_disable_pdev;

	shost->irq = pdev->irq;
	shost->base = pci_resource_start(pdev, 0);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	scsi_set_pci_device(shost, pdev);
#elif (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,9))
	scsi_set_device(shost, &pdev->dev);
#endif
	shost->unique_id = unique_id;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	/*
	 *	This function is called after the device list
	 *	has been built to find the tagged queueing
	 *	depth supported for each device.
	 */
	shost->select_queue_depths = aac_queuedepth;
#endif
#if ((LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0)) || defined(SERVICE_ACTION_IN))
	shost->max_cmd_len = 16;
#endif

	aac = (struct aac_dev *)shost->hostdata;
	aac->scsi_host_ptr = shost;	
	aac->pdev = pdev;
	aac->name = aac_driver_template.name;
	aac->id = shost->unique_id;
	aac->cardtype =  index;
	INIT_LIST_HEAD(&aac->entry);
	aac->fibs = kmalloc(sizeof(struct fib) * (shost->can_queue + AAC_NUM_MGT_FIB), GFP_KERNEL);
	if (!aac->fibs)
		goto out_free_host;
	spin_lock_init(&aac->fib_lock);

	/*
	 *	Map in the registers from the adapter.
	 */
	aac->base_size = AAC_MIN_FOOTPRINT_SIZE;
	if ((aac->regs.sa = ioremap(
	  (unsigned long)aac->scsi_host_ptr->base, AAC_MIN_FOOTPRINT_SIZE))
	  == NULL) {	
		printk(KERN_WARNING "%s: unable to map adapter.\n",
		  AAC_DRIVERNAME);
		goto out_free_fibs;
	}
	if ((*aac_drivers[index].init)(aac))
		goto out_unmap;

	aac_get_fw_debug_buffer(aac);
	/*
	 *	Start any kernel threads needed
	 */
	aac->thread_pid = kernel_thread((int (*)(void *))aac_command_thread,
	  aac, 0);
	if (aac->thread_pid < 0) {
		printk(KERN_ERR "aacraid: Unable to create command thread.\n");
		goto out_deinit;
	}

	/*
	 * If we had set a smaller DMA mask earlier, set it to 4gig
	 * now since the adapter can dma data to at least a 4gig
	 * address space.
	 */
	if (aac_drivers[index].quirks & AAC_QUIRK_31BIT)
		if (pci_set_dma_mask(pdev, DMA_32BIT_MASK))
			goto out_deinit;
 
	aac->maximum_num_channels = aac_drivers[index].channels;
	error = aac_get_adapter_info(aac);
	if (error < 0)
		goto out_deinit;

	/*
 	 * Lets override negotiations and drop the maximum SG limit to 34
 	 */
 	if ((aac_drivers[index].quirks & AAC_QUIRK_34SG) && 
			(aac->scsi_host_ptr->sg_tablesize > 34)) {
 		aac->scsi_host_ptr->sg_tablesize = 34;
 		aac->scsi_host_ptr->max_sectors
 		  = (aac->scsi_host_ptr->sg_tablesize * 8) + 112;
 	}

 	if ((aac_drivers[index].quirks & AAC_QUIRK_17SG) && 
			(aac->scsi_host_ptr->sg_tablesize > 17)) {
 		aac->scsi_host_ptr->sg_tablesize = 17;
 		aac->scsi_host_ptr->max_sectors
 		  = (aac->scsi_host_ptr->sg_tablesize * 8) + 112;
 	}

	/*
	 * Firware printf works only with older firmware.
	 */
	if (aac_drivers[index].quirks & AAC_QUIRK_34SG) 
		aac->printf_enabled = 1;
	else
		aac->printf_enabled = 0;
 
 	/*
	 * max channel will be the physical channels plus 1 virtual channel
	 * all containers are on the virtual channel 0
	 * physical channels are address by their actual physical number+1
	 */
	if (aac->nondasd_support == 1)
		shost->max_channel = aac->maximum_num_channels + 1;
	else
		shost->max_channel = 1;

	aac_get_config_status(aac);
	aac_get_containers(aac);
	list_add(&aac->entry, insert);

	shost->max_id = aac->maximum_num_containers;
	if (shost->max_id < aac->maximum_num_physicals)
		shost->max_id = aac->maximum_num_physicals;
	if (shost->max_id < MAXIMUM_NUM_CONTAINERS)
		shost->max_id = MAXIMUM_NUM_CONTAINERS;
	else
		shost->this_id = shost->max_id;

	/*
	 * dmb - we may need to move the setting of these parms somewhere else once
	 * we get a fib that can report the actual numbers
	 */
	shost->max_lun = AAC_MAX_LUN;

#if defined(__VMKERNEL_MODULE__)
	vmk_scsi_register_uinfo(shost, pdev->bus->number, pdev->devfn, (void *)aac);
#endif
	pci_set_drvdata(pdev, shost);

	error = scsi_add_host(shost, &pdev->dev);
	if (error)
		goto out_deinit;
	fwprintf((aac, HBA_FLAGS_DBG_FW_PRINT_B,
	  "Linux %s driver (%s)",
	  AAC_DRIVERNAME, aac_driver_version));
	scsi_scan_host(shost);

	return 0;

 out_deinit:
#if defined(__VMKERNEL_MODULE__)
	if (aac->thread_pid > 0) {
		aac->thread_die = 1;
		up(&aac->queues->queue[HostNormCmdQueue].cmdready);
		wait_for_completion(&aac->aif_completion);
	}
#else
	kill_proc(aac->thread_pid, SIGKILL, 0);
	wait_for_completion(&aac->aif_completion);
#endif

	aac_send_shutdown(aac);
	aac_adapter_disable_int(aac);
	free_irq(pdev->irq, aac);
 out_unmap:
	aac_fib_map_free(aac);
	pci_free_consistent(aac->pdev, aac->comm_size, aac->comm_addr, aac->comm_phys);
	kfree(aac->queues);
	iounmap(aac->regs.sa);
 out_free_fibs:
	kfree(aac->fibs);
	kfree(aac->fsa_dev);
#if defined(__VMKERNEL_MODULE__)
	spin_lock_destroy(&aac->fib_lock);
#endif
 out_free_host:
	scsi_host_put(shost);
 out_disable_pdev:
	pci_disable_device(pdev);
 out:
	return error;
}

#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,6,11)) || defined(PCI_HAS_SHUTDOWN)))
static void aac_shutdown(struct pci_dev *dev)
{
	struct Scsi_Host *shost = pci_get_drvdata(dev);
	struct aac_dev *aac = (struct aac_dev *)shost->hostdata;
	aac_send_shutdown(aac);
}

#endif
static void __devexit aac_remove_one(struct pci_dev *pdev)
{
	struct Scsi_Host *shost = pci_get_drvdata(pdev);
	struct aac_dev *aac = (struct aac_dev *)shost->hostdata;

	scsi_remove_host(shost);

#if defined(__VMKERNEL_MODULE__)
	if (aac->thread_pid > 0) {
		aac->thread_die = 1;
		up(&aac->queues->queue[HostNormCmdQueue].cmdready);
		wait_for_completion(&aac->aif_completion);
	}
#else
	kill_proc(aac->thread_pid, SIGKILL, 0);
	wait_for_completion(&aac->aif_completion);
#endif

	aac_send_shutdown(aac);
	aac_adapter_disable_int(aac);
	aac_fib_map_free(aac);
	pci_free_consistent(aac->pdev, aac->comm_size, aac->comm_addr,
			aac->comm_phys);
	kfree(aac->queues);

	free_irq(pdev->irq, aac);
	iounmap(aac->regs.sa);
	
	kfree(aac->fibs);
	kfree(aac->fsa_dev);
	
#if defined(__VMKERNEL_MODULE__)
	spin_lock_destroy(&aac->fib_lock);
#endif
	list_del(&aac->entry);
	scsi_host_put(shost);
	pci_disable_device(pdev);
	if (list_empty(&aac_devices)) {
		unregister_chrdev(aac_cfg_major, "aac");
		aac_cfg_major = -1;
	}
}

static struct pci_driver aac_pci_driver = {
	.name		= AAC_DRIVERNAME,
	.id_table	= aac_pci_tbl,
	.probe		= aac_probe_one,
	.remove		= __devexit_p(aac_remove_one),
#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,11)) && ((LINUX_VERSION_CODE > KERNEL_VERSION(2,6,11)) || defined(PCI_HAS_SHUTDOWN)))
	.shutdown 	= aac_shutdown,
#endif
};

#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11)) || ((LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,11)) && !defined(PCI_HAS_SHUTDOWN)))
static int aac_reboot_event(struct notifier_block * n, ulong code, void *p)
{
	if ((code == SYS_RESTART)
	 || (code == SYS_HALT)
	 || (code == SYS_POWER_OFF)) {
		struct aac_dev *aac;

		list_for_each_entry(aac, &aac_devices, entry)
			aac_send_shutdown(aac);
	}
	return NOTIFY_DONE;
}

static struct notifier_block aac_reboot_notifier =
{
	aac_reboot_event,
	NULL,
	0
};
 
#endif
static int __init aac_init(void)
{
	int error;
	
#if defined(__VMKERNEL_MODULE__)
	if (!vmk_set_module_version(AAC_DRIVER_FULL_VERSION)) {
		return 0;
	}

	spin_lock_init(&io_request_lock);
	aac_driver_template.driverLock = &io_request_lock;
#endif
	printk(KERN_INFO "Adaptec %s driver (%s)\n",
	  AAC_DRIVERNAME, aac_driver_version);
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) && defined(SCSI_HAS_SCSI_IN_DETECTION))
	scsi_register_module(MODULE_SCSI_HA,&aac_driver_template);
	/* Reverse 'detect' action */
	aac_driver_template.present = 0;
#endif

	error = pci_register_driver(&aac_pci_driver);
	if (error < 0 || list_empty(&aac_devices)) {
		if (error >= 0) {
			pci_unregister_driver(&aac_pci_driver);
			error = -ENODEV;
		}
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) && defined(SCSI_HAS_SCSI_IN_DETECTION))
		scsi_unregister_module(MODULE_SCSI_HA,&aac_driver_template);
#endif
#if defined(__VMKERNEL_MODULE__)
		spin_lock_destroy(&io_request_lock);
#endif
		return error;
	}

	aac_cfg_major = register_chrdev( 0, "aac", &aac_cfg_fops);
	if (aac_cfg_major < 0) {
		printk(KERN_WARNING
		       "aacraid: unable to register \"aac\" device.\n");
	}
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11))
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) ? defined(__x86_64__) : defined(CONFIG_COMPAT))
	aac_ioctl32(FSACTL_MINIPORT_REV_CHECK, sys_ioctl);
	aac_ioctl32(FSACTL_SENDFIB, sys_ioctl);
	aac_ioctl32(FSACTL_OPEN_GET_ADAPTER_FIB, sys_ioctl);
	aac_ioctl32(FSACTL_GET_NEXT_ADAPTER_FIB,
	  aac_get_next_adapter_fib_ioctl);
	aac_ioctl32(FSACTL_CLOSE_GET_ADAPTER_FIB, sys_ioctl);
	aac_ioctl32(FSACTL_SEND_RAW_SRB, sys_ioctl);
	aac_ioctl32(FSACTL_GET_PCI_INFO, sys_ioctl);
	aac_ioctl32(FSACTL_QUERY_DISK, sys_ioctl);
	aac_ioctl32(FSACTL_DELETE_DISK, sys_ioctl);
	aac_ioctl32(FSACTL_FORCE_DELETE_DISK, sys_ioctl);
	aac_ioctl32(FSACTL_GET_CONTAINERS, sys_ioctl);
#if (defined(FSACTL_REGISTER_FIB_SEND))
	aac_ioctl32(FSACTL_REGISTER_FIB_SEND, sys_ioctl);
#endif
	aac_ioctl32(FSACTL_GET_VERSION_MATCHING, sys_ioctl);
	aac_ioctl32(FSACTL_SEND_LARGE_FIB, sys_ioctl);
#if (defined(AAC_CSMI))
	aac_csmi_register_ioctl32_conversion();
#endif
#endif
#endif
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11)) || ((LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,11)) && !defined(PCI_HAS_SHUTDOWN)))
	register_reboot_notifier(&aac_reboot_notifier);
#endif
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) && !defined(SCSI_HAS_SCSI_IN_DETECTION))
#if !defined(__VMKERNEL_MODULE__)
	/* Trigger a target scan in the 2.4 tree */
	if (!aac_dummy) {
		aac_dummy = scsi_host_alloc(&aac_driver_template,0);
	}
#endif /* !__VMKERNEL_MODULE__ */
	scsi_register_module(MODULE_SCSI_HA,&aac_driver_template);
#endif

	return 0;
}

static void __exit aac_exit(void)
{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11))
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) ? defined(__x86_64__) : defined(CONFIG_COMPAT))
	unregister_ioctl32_conversion(FSACTL_MINIPORT_REV_CHECK);
	unregister_ioctl32_conversion(FSACTL_SENDFIB);
	unregister_ioctl32_conversion(FSACTL_OPEN_GET_ADAPTER_FIB);
	unregister_ioctl32_conversion(FSACTL_GET_NEXT_ADAPTER_FIB);
	unregister_ioctl32_conversion(FSACTL_CLOSE_GET_ADAPTER_FIB);
	unregister_ioctl32_conversion(FSACTL_SEND_RAW_SRB);
	unregister_ioctl32_conversion(FSACTL_GET_PCI_INFO);
	unregister_ioctl32_conversion(FSACTL_QUERY_DISK);
	unregister_ioctl32_conversion(FSACTL_DELETE_DISK);
	unregister_ioctl32_conversion(FSACTL_FORCE_DELETE_DISK);
	unregister_ioctl32_conversion(FSACTL_GET_CONTAINERS);
#if (defined(FSACTL_REGISTER_FIB_SEND))
	unregister_ioctl32_conversion(FSACTL_REGISTER_FIB_SEND);
#endif
	unregister_ioctl32_conversion(FSACTL_GET_VERSION_MATCHING);
	unregister_ioctl32_conversion(FSACTL_SEND_LARGE_FIB);
#if (defined(AAC_CSMI))
	aac_csmi_unregister_ioctl32_conversion();
#endif
#endif
#endif
	if (aac_cfg_major > -1)
		unregister_chrdev(aac_cfg_major, "aac");
#if ((LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,11)) && !defined(PCI_HAS_SHUTDOWN))
	unregister_reboot_notifier(&aac_reboot_notifier);
#endif
	pci_unregister_driver(&aac_pci_driver);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	scsi_unregister_module(MODULE_SCSI_HA,&aac_driver_template);
#endif
#if defined(__VMKERNEL_MODULE__)
	spin_lock_destroy(&io_request_lock);
#endif
}

module_init(aac_init);
module_exit(aac_exit);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
EXPORT_NO_SYMBOLS;
#endif
