/*
Copyright (C) 2004  Kenney He

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#define ADDR32 __GFP_DMA

////////////////////////////////////////////////////////////////

#include <linux/version.h>
#include <linux/ioctl.h>	/* For SCSI-Passthrough */
#include <asm/uaccess.h>
#include <linux/stat.h>

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,6)
#include <linux/slab.h>
#else
#include <linux/malloc.h>	/* for kmalloc() */
#endif

#include <linux/config.h>	/* for CONFIG_PCI */
#include <linux/pci.h>		/* for PCI support */
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/delay.h>	/* for udelay */
#include <linux/tqueue.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>	/* for printk */
#include <linux/sched.h>
#include <linux/smp_lock.h> 
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/stat.h>
#include <linux/time.h>
#include <linux/notifier.h>	/* System Shutdown or Reboot Notification */
#include <linux/reboot.h>
#include <linux/wait.h>
#include <asm/processor.h>	/* for boot_cpu_data */
#include <asm/pgtable.h>
#include <asm/io.h>		/* for virt_to_bus, etc. */

#if defined(MODVERSIONS)
#include <linux/modversions.h>
#endif
#include <linux/tqueue.h>
#include <linux/signal.h>
#include <asm/semaphore.h>

#include "scsi.h"
#include "hosts.h"
#include "sd.h"
#include "asa72xx_sig.h"
#include "asa72xx_osm.h"
#include "uerrdef.h"
#include "uerrmsg.h"
#include "asa72xx_debug.h"

#ifdef __VMKERNEL_MODULE__
#include "vmklinux_dist.h"
#endif //__VMKERNEL_MODULE__

#ifdef MODULE
#include <linux/module.h>

MODULE_AUTHOR("Adaptec, Inc.");
MODULE_DESCRIPTION("Adaptec ASA-72xx 1Gb iSCSI HBA Linux Driver");

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,8)
MODULE_LICENSE("GPL");
#endif

static S8 *asa72xxc = 0;
MODULE_PARM(asa72xxc, "s");

S8 kernel_version[] = UTS_RELEASE;
#endif

/* Create a binary signature */
static dpt_sig_S DPTI_sig = {
	{'d', 'P', 't', 'S', 'i', 'G'}, SIG_VERSION,
#ifdef i386
	PROC_INTEL, PROC_386 | PROC_486 | PROC_PENTIUM | PROC_SEXIUM,
#elif defined __ia64
	PROC_INTEL, PROC_IA64,
#elif defined(_solaris)
	PROC_ULTRASPARC,
#elif defined(_alpha)
	PROC_ALPHA ,
#else
#error You must add your Processor Family/Type to this source
#endif
	 FT_HBADRVR, 0, OEM_DPT, OS_LINUX, CAP_OVERLAP, DEV_ALL,
	ADF_ALL_SC5, 0, 0, ASA72XX_VERSION, ASA72XX_REVISION, ASA72XX_SUBREVISION,
	ASA72XX_MONTH, ASA72XX_DAY, ASA72XX_YEAR, "Adaptec ASA-72xx iSCSI HBA Driver"
};

/*
 * Locking 
 */  
DECLARE_MUTEX(adpt_configuration_lock);

/* Initialize lock */
static inline void adpt_init_lock(adpt_hba *pHba)
{
	spin_lock_init(&pHba->spinlock);
}
	
/* Acquire lock */
static inline void 
adpt_lock(adpt_hba *pHba, U32_64 flags, U8 irqsave)
{
	if (irqsave) {
		spin_lock_irqsave(&pHba->spinlock, flags);
	} else {
		spin_lock_irq(&pHba->spinlock);
	}
}

/* Release lock */
static inline void 
adpt_unlock(adpt_hba *pHba, U32_64 flags, U8 irqsave)
{
	if (irqsave) {
		spin_unlock_irqrestore(&pHba->spinlock, flags);
	} else {
		spin_unlock_irq(&pHba->spinlock);
	}
}

static struct i2o_sys_tbl 	*sys_tbl = NULL;
static S32 			sys_tbl_ind = 0;
static S32 			sys_tbl_len = 0;

static adpt_hba			*hbas[DPTI_MAX_HBA];
static adpt_hba			*hba_chain = NULL;
static adpt_setup		*pSetup;
static S32 			setup_done = 0;
static S32 			reg_drv_done = 0;
static S32			event_registered = 0;
static S8 			hba_count = 0;

/* 
 * Universal Error Message declarations 
 */
static U32_64		uerrno = 0;
static U32_64		transport = UT_ISCSI;

/*
 * Debug switch level , refer to asa72xx.h for the debug level.
 */  
#ifdef ASA72XX_DEBUG
U32 	debug_level = DBG_INIT | DBG_UNLOAD;
/*U32 	debug_level = DBG_LCT | CRIT_Q | DBG_ERR_HNDL; */
#else
U32 	debug_level = DBG_NONE;
#endif

struct pci_device_id	asa72xx_pci_id_table[] = 
{
	{0x9005, 0x564a, 0x9005, 0x7211, 0, 0, 0},
	{0x9005, 0x564a, 0x9005, 0x7210, 0, 0, 0},
	{0}
};

MODULE_DEVICE_TABLE(pci, asa72xx_pci_id_table);

struct pci_driver asa72xx_pci_driver = 
{
	name:		"asa72xx",
	probe:		adpt_pci_dev_probe,
	remove:		adpt_pci_dev_remove,
	id_table:	asa72xx_pci_id_table
};

#define NUMBER(arr)	(sizeof(arr) / sizeof(arr[0]))

/*
 * Array of supported HBA 
 * The format is as following: 
 * {PCI_VENDOR_ID, PCI_DEVICE_ID}
 */  
struct
{
	U16	vendor_id;
	U16	device_id;
} const hba_supported[] =
{
	{0x9004, 0x5647},
	{0x9004, 0x564A}
};	

static struct file_operations adpt_fops = 
{
	ioctl:		adpt_ioctl,
	open:		adpt_open,
	release:	adpt_close
};

static struct notifier_block adpt_shutdown_notifier =
{
	adpt_shutdown_event,
	NULL,
	0
};


/*
 * Macros
 */  
#define MAX(A,B)	((A) > (B) ? (A) : (B))
#define MIN(A,B)	((A) < (B) ? (A) : (B))

#define	POST_WAIT_CONTEXT  0x00800000

inline U32 
adpt_getBlinkLED(adpt_hba *host)
{
	return (host->FwDebugBLEDflag_P && 
	       (volatile U32)(*(host->FwDebugBLEDflag_P)) == 0xbc) ? 
			host->FwDebugBLEDvalue_P[0] : 0;
}

inline adpt_device *
adpt_find_device(adpt_hba *pHba, U32 chan, U32 id, U32 lun)
{
	adpt_target 	*pTarget;
	adpt_device 	*pDev;
	
	BEGIN;

	if (pHba->channel[chan].target == NULL) {
		PDEBUG(DBG_DEVICE, "asa72xx: Trying to find device before "
				"they are allocated\n");

		ERROR_END;

		return NULL;
	}

	if (pHba->channel[chan].target[id] != NULL) {
		pTarget = pHba->channel[chan].target[id];
	} else {
		PDEBUG(DBG_DEVICE, "No Target found !!\n");

		ERROR_END;

		return NULL;
	}

	pDev = pTarget->device;
	
	if (!pDev || (pDev->tid == 0)) {
		return NULL;
	}

	if (pDev->scsi_lun == lun) {
		return pDev;
	}

	for (pDev = pDev->next_lun ; pDev ; pDev = pDev->next_lun) {
		if (pDev->scsi_lun == lun) {
			return pDev;
		}
	}

	PDEBUG(DBG_DEVICE, "No Target found !!\n");
	
	END;

	return NULL;
}

/*
 * adpt_get_target()
 *
 * Description :
 * This function gets the target pointer that the driver
 * has stored for the BT entry.
 */
inline adpt_target * adpt_get_target(adpt_hba *pHba, int channel, int target)
{
   return (pHba->channel[channel].target[target]);
}

///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////
//				Functions                            //
///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////
#ifdef ASA72XX_VARYIO_SHT_EXT
Scsi_Host_Template_Ext driver_template = ASA72XX_SHT;
#else
Scsi_Host_Template driver_template = ASA72XX_SHT;
#endif

#include "scsi_module.c"


/*
 * adpt_i2o_systab_send()
 *
 * Description:
 *	Send the ExecSysTabSet request to each IOP of the identity (location)
 *	of the other IOPs in the system, as well as declarations of memory
 * 	and I/O for private space. This event also takes the IOP from the
 *	HOLD state to the READY state.
 */	 			   
static S32 
adpt_i2o_systab_send(adpt_hba *pHba)
{
	 U32 	msg[12];
	 S32 	ret;

	BEGIN;
       	
	msg[0] = I2O_MESSAGE_SIZE(12) | SGL_OFFSET_6;
	msg[1] = I2O_CMD_SYS_TAB_SET << 24 | HOST_TID << 12 | ADAPTER_TID;
	msg[2] = 0;
	msg[3] = 0;
	/* Host 0 IOP ID (unit + 2) */
	msg[4] = (0<<16) | ((pHba->unit + 2) << 12); 
 	/* Segment 0 */
	msg[5] = 0;	  

	/* 
	 * Provide three SGL-elements:
	 * System table (SysTab), Private memory space declaration and 
	 * Private i/o space declaration  
	 */
	msg[6] = 0x54000000 | sys_tbl_len;
	msg[7] = virt_to_phys(sys_tbl);
	msg[8] = 0x54000000 | 0;
	msg[9] = 0;
	msg[10] = 0xD4000000 | 0;
	msg[11] = 0;

	if ((ret = adpt_i2o_post_wait(pHba, msg, sizeof(msg), 120))) {
		PDEBUG(DBG_POST, "%s: Unable to set SysTab (status=%#10x).\n", 
			pHba->name, ret);
	} else {
		PDEBUG(DBG_POST, "%s: SysTab set.\n", pHba->name);
	}

	END;

	return ret;	
}


/*
 * adpt_i2o_sys_shutdown()
 *
 * Description: 
 * 	Shutdown I2O system
 */
void 
adpt_i2o_sys_shutdown(void)
{
	adpt_hba 	*pHba = NULL;
	U32		event_mask;	

	BEGIN;
	
	PDEBUG(DBG_UNLOAD, "Shutting down Adaptec iSCSI HBA controller !\n");
	 
	/*
	 * Set Event Mask to 0 to get back the pending Async Event 
	 * Notification message.
	 */   
	event_mask = 0;
	
	/* 
	 * Delete all Host Adapter from the controller chain 
	 */
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		pHba->state |= ADPT_STATE_SHUTDOWN;

		if (adpt_async_register_event(pHba, event_mask) != 0) {
			PDEBUG(DBG_UNLOAD, "Failed to unregister "
					"Async Event !\n");
		}

		adpt_destroy_cmd_pool(pHba);

		adpt_i2o_delete_hba(pHba);
	}
	
	PDEBUG(DBG_UNLOAD, "Adaptec iSCSI Host Adapter has been shut down !\n");

	END;
}


/*
 * adpt_shutdown_event()
 * 
 * Description:
 *	This routine will be called when the system shutdown or reboot.
 * 	Perform clean up.
 */
static S32
adpt_shutdown_event(struct notifier_block *nb, U32_64 code, void *unused)
{
	adpt_hba 	*pHba = NULL;
	U32		msg[MAX_MESSAGE_SIZE/4];
	U32		msg_size;
	S32		rcode = 0;
	
	BEGIN;

	PDEBUG(DBG_UNLOAD, "received shutdown event!\n");

	if ((code == SYS_RESTART) || (code == SYS_HALT) ||
		(code == SYS_POWER_OFF)) {
		/*
		 * Flush all the IOs that remain in the SendQueue 
		 * to the device.
		 */   
		for (pHba = hba_chain; pHba; pHba = pHba->next) {
			if (pHba->state & ADPT_STATE_LINK_UP) {
				adpt_start_io(pHba);
			}

	                pHba->state |= ADPT_STATE_SHUTDOWN;
		
			/*
			 * Send a Private Shutdown Message to the FW to perform
			 * Logout for all the targets. 
			 */
			msg[0] = I2O_MESSAGE_SIZE(6) | SGL_OFFSET_0;
			msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
			msg[2] = 0x23000000;	/* Initiator Context */
			msg[3] = (U32) msg;	/* Target Context */
			msg[4] = (DPT_ORGANIZATION_ID << 16) | 
					I2O_PRVT_SYS_SHUTDOWN;
			msg[5] = 0x00010000;	/* Interpret bit */

			/* Message size in U32 words */
			msg_size = 6;
			
			/* Message size in U8 bytes */
			msg_size = 6 << 2;

			rcode = adpt_i2o_post_wait(pHba, msg, msg_size,  
					TMOUT_SYS_SHUTDOWN);

			if (rcode != 0) {
				PDEBUG(DBG_EVENT, "Failed to issue Logout "
						"to the targets !\n");
			}
		}	

		/*
		 * Shutdown Host Adapter and perform House Cleaning. 
		 */  
		adpt_i2o_sys_shutdown();
	}

	END;
	
	return NOTIFY_DONE;
}


/*
 * adpt_install_hba()
 *
 * Description:
 *	Installing the controller.
 */    
S32 
adpt_install_hba(struct pci_dev *pDev) 
{
	adpt_hba 	*pHba = NULL;
	adpt_hba 	*p = NULL;
	U32		m;
	U16 		command = 0;
	U8 		i;
	
	/* 
	 * i960 ATU or Power PC debug window physical address and 
	 * virtual address variables
	 */
	caddr_t base_addr_phys = NULL;
	caddr_t base_page_phys = NULL;
	off_t 	base_offset_phys = 0;
	caddr_t base_addr_virt = NULL;
	U32 	hba_map_area_size = 0;

	BEGIN;

	if (pci_enable_device(pDev)) {
		ERROR_END;

		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_LINUX | 
					UC_NO_EXT | U_PCI_ENABLE_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);

		return -ENODEV;
	}
	
	/*
	 * Read in the command register and make sure that the device is 
	 * enabled and set up for bus master.
	 */                              
	pci_read_config_word(pDev, PCI_COMMAND, &command);
	
	if (!((command & PCI_COMMAND_MEMORY) && 
			(command & PCI_COMMAND_MASTER))) {
		PDEBUG(DBG_INIT, "HBA %d: Device not memory enable or "
				"bus master.\n",
				hba_count);
	
		ERROR_END;

		return -EINVAL;
	}

	base_addr_phys = (caddr_t) pci_resource_start(pDev, 0);

	hba_map_area_size = pci_resource_len(pDev, 0);

	if (hba_map_area_size > 0x400000) {
		/* Only give them 4M */
		hba_map_area_size = 0x400000;
	}	
	
	/* 
	 * i960 ATU or Power PC Debug window virtual address space
	 */ 
	base_page_phys = (caddr_t) ((U32) base_addr_phys & PAGE_MASK);
	
	base_offset_phys = base_addr_phys - base_page_phys;

	base_addr_virt = ioremap((S32_64) base_page_phys, 
				(S32_64) base_offset_phys + hba_map_area_size);

	if (base_addr_virt == NULL) {
		PERROR("adpt_install_hba: ioremap failed for "
				"base_addr_virt !\n");
	
		ERROR_END;

		return -EINVAL;
	}

	/* 
	 * Allocate and zero the data structure
	 */ 
	if ((pHba = kmalloc(sizeof (adpt_hba), GFP_ATOMIC)) == NULL) {
		iounmap(base_addr_virt);

		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN |
				UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
				uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

		return -ENOMEM;
	}
	
	memset(pHba, 0, sizeof(adpt_hba));

	down(&adpt_configuration_lock);

	for (i = 0; i < DPTI_MAX_HBA; i++) {
		if (hbas[i] == NULL) {
			hbas[i] = pHba;

			break;
		}
	}
	
	if (hba_chain != NULL) {
		for (p = hba_chain; p->next; p = p->next) {
		}
		p->next = pHba;
	} else {
		hba_chain = pHba;
	}

	pHba->next = NULL;
	
	pHba->unit = hba_count;

	sprintf(pHba->name, "asa72xx");
	
	hba_count++;
	
	up(&adpt_configuration_lock);

	pHba->base_addr_phys = base_addr_phys;
	
	/*
	 * Set the DMA bit to 32-bit addresses.
	 * In the future, when we do support 64-bit addressing, 
	 * the dma mask bit need to be changed accordingly.
	 */    
	if (pci_dma_supported(pDev, 0xffffffff)) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,3)
		pci_set_dma_mask(pDev, 0xffffffff);
#else		
		pDev->dma_mask = 0xffffffff;
#endif
	}
		
	/* 
	 * Set up the Virtual Base Address of the I2O Device
	 */ 
	pHba->base_addr_virt = base_addr_virt;

	pHba->irq_status = (U32 *) (base_addr_virt + I2O_INTR_STATUS_REG);
	pHba->irq_mask = (U32 *) (base_addr_virt + I2O_INTR_MASK_REG);
	pHba->ToIopFifo = (U32 *) (base_addr_virt + I2O_INBOUND_QUEUE_REG);
	pHba->FromIopFifo = (U32 *) (base_addr_virt + I2O_OUTBOUND_QUEUE_REG);

	pHba->hrt = NULL;
	pHba->lct = NULL;
	pHba->lct_size = 0;
	pHba->status_block = NULL;
	pHba->state = ADPT_STATE_IOP_RESET; 
	pHba->pDev = pDev;
	pHba->active_cmds = 0;
	pHba->aborted_cmd_count = 0;
	pHba->in_use = 0;

	/*
	 * Initialize the fields required by the worker thread
	 * */
	spin_lock_init(&pHba->protect_io_cmd);
	init_waitqueue_head(&(pHba->interrupt_triggered_event));
	init_waitqueue_head(&(pHba->wake_up_worker));
	init_waitqueue_head(&(pHba->wake_up_sr));
	init_waitqueue_head(&(pHba->lct_processing_complete));
#ifdef __VMKERNEL_MODULE__
	init_MUTEX(&pHba->wake_up_worker_sem);
	init_MUTEX_LOCKED(&pHba->process_io_sem);
#endif
	/* 
	 *  Start the Timer tick
	 */
        adpt_timer((U32_64) pHba);

	/*
	 * Adpt start processing the requests with the new thread
	 * */
	if(adpt_start_worker_thread(pHba)){
		PDEBUG(DBG_INIT, "%s: Couldn't Start Worker Thread\n", 
				pHba->name);
		
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN |
				UC_NO_EXT | U_ISR_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
				uni_err(uerrno), (U32) uerrno);
		
		adpt_i2o_delete_hba(pHba);
		
		ERROR_END;

		return -EINVAL;
	}
	
	/*
	 * clear Interrupt Status Register and Unmask Post Queue interrupts
	 */ 
	PDEBUG(DBG_INIT, "ISR = %x \n", *pHba->irq_status);
	
	PDEBUG(DBG_INIT, "IMR = %x\n", *pHba->irq_mask);

	/*
	 * clear any outstanding interrupts
	 */ 
	m = (volatile U32) readl(pHba->irq_status);

	writel(m, pHba->irq_status);

	/*
	 * Initializing the spinlocks
	 */ 
	adpt_init_lock(pHba);

	PWARN("Adaptec 1Gb iSCSI Host Bus Adapter on PCI bus %d device %d "
		"irq %d \n", 
		pHba->pDev->bus->number,
		PCI_SLOT(pHba->pDev->devfn),
		pHba->pDev->irq);

	if (request_irq(pDev->irq, adpt_isr, SA_SHIRQ, pHba->name, pHba)) {
		PDEBUG(DBG_INIT, "%s: Couldn't register IRQ %d\n", 
				pHba->name, pDev->irq);
		
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN |
				UC_NO_EXT | U_ISR_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
				uni_err(uerrno), (U32) uerrno);
		
		adpt_i2o_delete_hba(pHba);
		
		ERROR_END;

		return -EINVAL;
	}

	/* 
	 * unmask the post queue fifo interrupt (bit 3)
	 */  
	m = (volatile U32) readl(pHba->irq_mask);
	
	m &= ~I2O_INTERRUPT_PENDING_B;
	
	writel(m, pHba->irq_mask);

	END;

	return 0;
}

/*
 * adpt_i2o_delete_hba()
 *
 * Description:
 *	Free all the resources allocated for each HBA.
 */	    
void 
adpt_i2o_delete_hba(adpt_hba *pHba)
{
	adpt_hba		*p1;
	adpt_hba		*p2;
	adpt_device		*pDev;
	adpt_device		*pNext;
	adpt_target		*pTarget;
	U32			m;
	U16 			i;
	U16 			j;

	BEGIN;

	down(&adpt_configuration_lock);
	
	if (pHba->host) {
		/*
		 * Disable IRQ line.
		 */
	      	m = (volatile U32) readl(pHba->irq_mask);

		m &= ~I2O_INTERRUPT_PENDING_B;

		writel(m, pHba->irq_mask);
		
		/*
		 * Free the IRQ.
		 */ 
		free_irq(pHba->host->irq, pHba);

		scsi_unregister(pHba->host);
	}
	
	for (i=0; i < DPTI_MAX_HBA; i++) {
		if (hbas[i] == pHba) {
			hbas[i] = NULL;
		}
	}
	
	p2 = NULL;
	
	for (p1 = hba_chain; p1; p2 = p1, p1 = p1->next) {
		if (p1 == pHba) {
			if (p2) {
				p2->next = p1->next;
			} else {
				hba_chain = p1->next;
			}

			break;
		}
	}

	hba_count--;
	
	up(&adpt_configuration_lock);

	iounmap(pHba->base_addr_virt);

	if (pHba->hrt) {
		pci_free_consistent(pHba->pDev, pHba->hrt_size, pHba->hrt,
				pHba->hrt_bus_addr);
	}
	
	if (pHba->lct) {
		pci_free_consistent(pHba->pDev, pHba->lct_size,
				pHba->lct, pHba->lct_bus_addr);
	}
	
	if (pHba->status_block) {
		pci_free_consistent(pHba->pDev, sizeof(i2o_status_block),
				pHba->status_block,
				pHba->status_block_bus_addr);
	}
	
	if (pHba->FromIop_Pool) {
		kfree(pHba->FromIop_Pool);
	}
	
	for (i = 0 ; i < pHba->top_scsi_channel ; i++) {
		for (j = 0; j < MAX_ID; j++) {
			if (pHba->channel[i].target[j] != NULL) {
				pTarget = pHba->channel[i].target[j];

				if (pTarget->device != NULL) {
					for (pDev = pTarget->device; 
						pDev; pDev = pNext) {
						pNext = pDev->next_lun;
					
						kfree(pDev);
					}
				}

				if (pTarget != NULL) {
					kfree(pTarget);
				}
			}
		}
	}

	if(pHba->pdevice_info_ioctl){
		pci_free_consistent(pHba->pDev, 
				sizeof(get_device_list),
				pHba->pdevice_info_ioctl,
				pHba->pdevice_list_bus_addr);
	}

	if(pHba->ptarget_info_ioctl){
		pci_free_consistent(pHba->pDev, 
				sizeof(get_target_list),
				pHba->ptarget_info_ioctl,
				pHba->ptarget_list_bus_addr);
	}
	adpt_stop_worker_thread(pHba);

	kfree(pHba);

	END;
}

/*
 * adpt_start_worker_thread()
 *
 * Description:
 * 	This function will start the worker thread to process IOCTL requests
 */	    
S32
adpt_start_worker_thread(adpt_hba *pHba)
{
	BEGIN;

	/*
	 * Start the LCT thread now
	 */
	pHba->lct_thread_id.arg=(void *)pHba;
	start_kthread(adpt_lct_thread,&pHba->lct_thread_id);

	/* Pass on the argument to store the pointer to HBA structure */
	pHba->thread_id.arg=(void *)pHba;
	start_kthread(adpt_io_thread,&pHba->thread_id);

	END;
	return 0;
}


/*
 * adpt_lct_thread()
 *
 * Description:
 * 	This thread calls the rescan at any given time when the link goes down
 *      The thread is woken up from the adpt_timer
 */	    
void adpt_lct_thread(kthread_t *kthread)
{
	adpt_hba *pHba=NULL;

	BEGIN;
	pHba = (adpt_hba *)kthread->arg;

        /* setup the thread environment */
        init_kthread(kthread, "SR thread");

	PDEBUG(DBG_I2O_POST,"\nIn the SR Thread");

        /* an endless loop in which we are doing our work */
        for(;;)
        {
		/*
		 * Check to see if there are any pending requests for SR
		 * If so process it else wait for an event from adpt_timer to
		 * wake us up for processing
		 * */
    		if ((pHba->state & ADPT_STATE_LCT_GET) && (pHba->state & ADPT_STATE_RESCAN_IN_PROGRESS) ) {
			adpt_process_lct_async_msg((void *)pHba);
		}else
		{
			/*
			 * This is woken up either by a new IO request or 
			 * by a shutdown request
			 **/
			PDEBUG(DBG_I2O_POST,"\nWaiting for a new SR request");
			/* 
			* Just to catch any signal we might miss on, we wait 
			* for an event for a tenth of second and then wake up
			*/
#ifdef __VMKERNEL_MODULE__
			vmk_thread_wait_event(&pHba->wake_up_sr, NULL);
#else
			interruptible_sleep_on_timeout(&pHba->wake_up_sr,HZ/10);
#endif //__VMKERNEL_MODULE__
			PDEBUG(DBG_I2O_POST,"\nGot a SR request");
		}

                /* We need to do a memory barrier here to be sure that
                   the flags are visible on all CPUs. 
                */
                 mb();
                
                /* here we are back from sleep, either due to the timeout
                   (one second), or because we caught a signal.
                */
                if (kthread->terminate)
                {
                        /* we received a request to terminate ourself */
                        break;    
                }
                
        }
        /* here we go only in case of termination of the thread */

        /* cleanup the thread, leave */
        exit_kthread(kthread);

	END;
	/* returning from the thread here calls the exit functions */
	return;
}

/*
 * adpt_io_thread()
 *
 * Description:
 * 	This is the thread function that loops in and processes
 * 	each IOCTL serially 
 */	    
void adpt_io_thread(kthread_t *kthread)
{
	adpt_hba *pHba=NULL;
	adpt_command_request *padpt_cmd_req=NULL;

	BEGIN;
	pHba = (adpt_hba *)kthread->arg;

        /* setup the thread environment */
        init_kthread(kthread, "worker thread");

	PDEBUG(DBG_I2O_POST,"\nIn the worker Thread");

        /* an endless loop in which we are doing our work */
        for(;;)
        {
		/*
		 * Check to see if there are any pending requests in the
		 * IO queue. If so process it else wait for an event to
		 * wake us up for processing
		 * */
		if((padpt_cmd_req = get_cmd_from_io_queue(pHba))){
			process_io_request(pHba,padpt_cmd_req);
		}else
		{
			/*
			 * This is woken up either by a new IO request or 
			 * by a shutdown request
			 **/
			PDEBUG(DBG_I2O_POST,"\nWaiting for a new I2O request");
			/* 
			* Just to catch any signal we might miss on, we wait for an event for a second
			* and then wake up
			*/
#ifdef __VMKERNEL_MODULE__
			/* vmkernel does not implement
			 *  interruptible_sleep_on_timeout */
			down(&pHba->wake_up_worker_sem);
#else
			interruptible_sleep_on_timeout(&pHba->wake_up_worker,HZ/10);
#endif //__VMKERNEL_MODULE__
			PDEBUG(DBG_I2O_POST,"\nGot a IO request");
		}

                /* We need to do a memory barrier here to be sure that
                   the flags are visible on all CPUs. 
                */
                 mb();
                
                /* here we are back from sleep, either due to the timeout
                   (one second), or because we caught a signal.
                */
                if (kthread->terminate)
                {
                        /* we received a request to terminate ourself */
                        break;    
                }
                
        }
        /* here we go only in case of termination of the thread */

        /* cleanup the thread, leave */
#ifndef __VMKERNEL_MODULE__
        exit_kthread(kthread);
#endif
	END;
	/* returning from the thread here calls the exit functions */
	return;
}

/*
 * adpt_stop_worker_thread()
 *
 * Description:
 * 	This function will stop the thread and is called befor the HBA instance
 * 	is deleted out
 * */
S32
adpt_stop_worker_thread(adpt_hba *pHba)
{
	BEGIN;

	PDEBUG(DBG_I2O_POST,"\nIn the Stop Thread");
#ifdef __VMKERNEL_MODULE__
	(&pHba->thread_id)->terminate = 1;
	up(&pHba->wake_up_worker_sem);
#else
	stop_kthread(&pHba->thread_id);
#endif
	stop_kthread(&pHba->lct_thread_id);

	END;
	return 0;
}

/*
 * process_io_request()
 *
 * Description:
 * 	The i2o command is sent to adpt_post_this function first
 * 	Once the Command is submitted, we block and wait to 
 * 	receive the event from the interrupt
 * 	If the event is generated, then just go and process the event
 * 	Note that, we should check if the timeout for the command has already 
 * 	occured, Since the queue depth is set to 1, the requester will set
 * 	that we drop the IO response and dont generate any interrupts
 * */
S32
process_io_request(adpt_hba *pHba,adpt_command_request *pio_request)
{
	BEGIN;

	PDEBUG(DBG_I2O_POST,"\nIn Process IO request");
	atomic_set(&pio_request->fw_started_processing,1);
	/*
	 * Just update the flag, if this is set to 1, indicates a timeout 
	 * */
	atomic_set(&pHba->fw_drop_current_io,0);
	atomic_set(&pHba->fw_completed_request,0);
	pHba->cmd_status = adpt_i2o_post_this(pHba, pio_request->msg, pio_request->len);

#ifdef __VMKERNEL_MODULE__
	/*  We do not support sleep_on and it's variants in the
	 *  vmkernel. We cannot use vmk_thread_wait_event here either
	 *  because of a potential race condition between the if
	 *  check and the wait that causes a deadlock. Hence the semaphore. */
	down(&pHba->process_io_sem);
#else 
	while(1){
		if((atomic_read(&pHba->fw_drop_current_io) == 1) || (atomic_read(&pHba->fw_completed_request)==1)){
			break;
		}
		// Sleep for one second or till some one wakes us up
		interruptible_sleep_on_timeout(&pHba->interrupt_triggered_event,HZ/10);
	}	   
#endif
	PDEBUG(DBG_I2O_POST,"\nIn Process IO request after waking up command");
	/*
	 * When we are here, we are for sure that the I2O message 
	 * has been completed
	 * */
	if(atomic_read(&pHba->fw_drop_current_io) != 1){
		PDEBUG(DBG_I2O_POST,"\nGoing to wake up the I2O post wait call");
		atomic_set(&pio_request->fw_completed_processing,1);
#ifdef __VMKERNEL_MODULE__
		up(&pio_request->io_done_sem);
#else	   
		wake_up(&pio_request->thread_processing_queue);
#endif

	}	
	//If we are timed out just ignore and continue
	END;
	return 0;
}

/*
 * Just trigger an event to wake up the worker thread*/
S32
adpt_i2o_post_event_wake_up(adpt_hba *pHba)
{
	BEGIN;

	atomic_set(&pHba->fw_completed_request,1);
#ifdef __VMKERNEL_MODULE__
	up(&pHba->process_io_sem);
#else
	wake_up(&pHba->interrupt_triggered_event);
#endif //__VMKERNEL_MODULE__
	END;
	return 0;
}

/*
 * adpt_destroy_cmd_pool()
 * 
 * Description:
 *	Destroy the Free Queue cmd pool.
 */
static void
adpt_destroy_cmd_pool(adpt_hba *pHba)
{
	adpt_cmnd	*pCmnd;
	adpt_cmnd	*pNextCmnd;
	
	BEGIN;

	pCmnd = pHba->FreeQueueHead;

	while (pCmnd != NULL) {
		pNextCmnd = pCmnd->next;

		kfree(pCmnd);

		pCmnd = pNextCmnd;
	}

	END;
}


/*
 * adpt_i2o_enable_hba()
 *
 * Description:   
 * 	Send the ExecSysEnable message which enables the  IOP and allows 
 * 	the IOP to resume external operations.
 */
S32 
adpt_i2o_enable_hba(adpt_hba *pHba)
{
	U32 msg[4];
	S32 ret;

	BEGIN;
	
	adpt_i2o_status_get(pHba);

	if (!pHba->status_block) {
		ERROR_END;

		return -ENOMEM;
	}

	/* 
	 * Enable only allowed on READY state 
	 */
	if (pHba->status_block->iop_state == ADAPTER_STATE_OPERATIONAL) {	
		return 0;
	}

	if (pHba->status_block->iop_state != ADAPTER_STATE_READY) {
		ERROR_END;

		return -EINVAL;
	}

	msg[0] = FOUR_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = I2O_CMD_SYS_ENABLE << 24 | HOST_TID << 12 | ADAPTER_TID;
	msg[2] = 0;
	msg[3] = 0;

	/* 
	 * How long of a timeout do we need? 
	*/
	if ((ret = adpt_i2o_post_wait(pHba, msg, sizeof(msg), 240))) {
		PDEBUG(DBG_POST, "%s: Could not enable (status=%#10x).\n", 
			pHba->name, ret);
	} else {
		PDEBUG(DBG_POST, "%s: Enabled.\n", pHba->name);
	}

	adpt_i2o_status_get(pHba);

	END;

	return ret;
}


/*
 * adpt_i2o_online_hba
 *
 * Description:  
 * 	Bring a controller online into OPERATIONAL state. 
 */
S32 
adpt_i2o_online_hba(adpt_hba *pHba)
{
	BEGIN;
	
	if (adpt_i2o_systab_send(pHba) < 0) {
		adpt_i2o_delete_hba(pHba);

		ERROR_END;

		return -1;
	}

	/* 
	 * In READY state 
	 */
	if (adpt_i2o_enable_hba(pHba) < 0) {
		adpt_i2o_delete_hba(pHba);
		
		ERROR_END;
		
		return -1;
	}

	/* 
	 * At this point, our HBA should be in OPERATIONAL state  
	 */

	END;

	return 0;
}


/*
 * adpt_detect()
 *
 * Description:
 *    This function is called by the OS at initalization time. The system 
 *    is scanned for HBAs. Once an HBA is found various structures are
 *    set up for the HBA and a list of all devices on the HBA is generated.
 *    Once all HBAs are found, a device lookup table is set up and then 
 *    each HBA is enabled and presented to the OS.
 */
S32 
adpt_detect(Scsi_Host_Template *sht)
{
	adpt_hba	*pHba;
	U32		events_mask = 0;
	S32		rcode = 0;
	U16		opblk[OP_BLOCK_SIZE/2];
	
	BEGIN;

#ifdef __VMKERNEL_MODULE__
	if (!vmk_set_module_version("#ASA72XX_VERSION")) {
	   return 0;
	}
        /* In the vmkernel, we do not hold the io_request lock during init,
         * so leave it unlocked and do not lock it before returning. */
#else
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	spin_unlock_irq(&io_request_lock);
#endif
#endif //__VMKERNEL_MODULE__

#ifdef ASA72XX_DEBUG
        __beeponce();
#endif
	adpt_init();
	
	sht->use_new_eh_code = 1;
	
	PWARN("Detecting Adaptec ASA-72xx iSCSI Host Bus Adapters !\n");

	/*
	 * Check if PCI bus is present on the system.
	 */  
	if (pci_present() == 0) {
		uerrno = (UL_ERROR | UP_INIT | US_HW | transport | UO_CMN |
				UC_NO_EXT | U_BUS_ARCH_NOT_SUPP);

		PERROR("ERROR: %s [%x] \n", uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

                goto ERROR_UNLOAD_DRIVER;
	}

	/* 
	 * Register our PCI Device Structure to the OS. 
	 */
        if (reg_drv_done == 0 )
        {
	   pci_register_driver(&asa72xx_pci_driver);
           reg_drv_done=1;
        }

	if (hba_count == 0) {
		uerrno = (UL_ERROR | UP_INIT | US_HW | transport | UO_CMN |
				UC_NO_EXT | U_NO_SUPP_ADAPTER);

		PERROR("ERROR: %s [%x] \n", uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

                goto ERROR_UNREGISTER_DRIVER;
	}
	
	/* 
	 * In INIT state, Activate IOPs 
	 */
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		/*
		 * Activate does get status , init outbound, and get hrt
		 */ 
		if (adpt_i2o_activate_hba(pHba) != 0) {
			adpt_i2o_delete_hba(pHba);

			ERROR_END;

                        goto ERROR_UNREGISTER_DRIVER;

		}
		/*
	 	 * Register an Asynchronous Notification Message to the FW.
	 	 * Initially, we want Link State Change Event and 
	 	 * LCT Notify Events.  Register our state before LCT get
	 	 */  
		events_mask = I2O_EVT_LINKSTATE_CHANGE | I2O_EVT_DO_LCT_NOTIFY;	
		
		PDEBUG(DBG_EVENT, "Events to be registered : 0x%x !\n",
				events_mask);

		if (adpt_async_register_event(pHba, events_mask) == 0) {
			PDEBUG(DBG_INIT, 
				"Async. Event Messages Registration is "
				"successful !\n");

			adpt_async_send_notification(pHba, IMMEDIATE_REPLY);
		} else {
			PDEBUG(DBG_INIT, 
				"Async. Event Messages Registration is "
				"failed !\n");
		}
	}


	/* Active IOPs in HOLD state */

rebuild_sys_tab:
	if (hba_chain == NULL) {	
		ERROR_END;

                goto ERROR_UNREGISTER_DRIVER;
	}

	/*
	 * If build_sys_table fails, we kill everything and bail
	 * as we can't init the IOPs w/o a system table
	 */	
	if (adpt_i2o_build_sys_table() < 0) {
		adpt_i2o_sys_shutdown();
	
		ERROR_END;

                goto ERROR_UNREGISTER_DRIVER;
	}

	/* 
	 * If IOP don't get online, we need to rebuild the System table 
	 */
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		if (adpt_i2o_online_hba(pHba) < 0) {
			adpt_i2o_delete_hba(pHba);	

			goto rebuild_sys_tab;
		}
	}

	/* 
	 * Active IOPs now in OPERATIONAL state 
	 */
	PDEBUG(DBG_INIT, "HBA IS IN OPERATIONAL STATE \n");

	/*
	 * let the firmware know that we're coming up
	 * from a fresh boot. Power management routines will not
	 * invoke a probe, so we can do this here...
	 */
                                                                                
	for(pHba = hba_chain;pHba; pHba = pHba->next) {
		if(adpt_set_driver_status(pHba)) {
                                                                                
			PDEBUG(DBG_INIT,
				"Could not tell driver status to firmware."
				"FW will login to all discovered targets !\n");
		}
	}

	/*
	 * parse command line parameters here
	 */ 
	if ((asa72xxc) && (!setup_done)) {	
		parse_cmdline_params(asa72xxc, (S32 *) 0);
	}

	/*
	 * Validate the iSCSI HBA configuration.
	 */
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		if ((rcode  = validatehostconfiguration(pHba)) != 0) {
			PDEBUG(DBG_INIT, "Failed to validate host "
				"configuration !\n");
		}	
	}
	
	/*
	 * Only if the MachineName matched with the one stored in EEPROM,
	 * then we shall get the LCT table from FW.
	 */   
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
#if 0 /* Moving up the Registration as requested by the FW */
		/*
	 	 * Register an Asynchronous Notification Message to the FW.
	 	 * Initially, we want Link State Change Event and 
	 	 * LCT Notify Events.  Register our state before LCT get
	 	 */  
		events_mask = I2O_EVT_LINKSTATE_CHANGE | I2O_EVT_DO_LCT_NOTIFY;	
		
		PDEBUG(DBG_EVENT, "Events to be registered : 0x%x !\n",
				events_mask);

		if (adpt_async_register_event(pHba, events_mask) == 0) {
			PDEBUG(DBG_INIT, 
				"Async. Event Messages Registration is "
				"successful !\n");

			/*
			 * Send a Private Async. Event Notification
		pDev = pDev->next_lun;
			 * to check if the Link is up.
			 */     
			adpt_async_send_notification(pHba, IMMEDIATE_REPLY);
		} else {
			PDEBUG(DBG_INIT, 
				"Async. Event Messages Registration is "
				"failed !\n");
		}
#endif
    	       if (pHba->hostverifydoneflag) {
	              if (pHba->state & ADPT_STATE_LINK_UP) {
                        /*
                         * Above link state check eliminates the LCT timeout
                         */

                      	adpt_rescan(pHba, TRUE);

                      }
		}
	} 
		
	/*
	 * Register the driver with scsi core.
	 */ 	
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		/*
	 	 * Initialize command pool for the devices.
	 	 */
		if (adpt_init_cmd_pool(pHba) == 0) {
			PDEBUG(DBG_INIT, "Failed to allocate cmd pool !\n");

			pci_unregister_driver(&asa72xx_pci_driver);

			adpt_i2o_delete_hba(pHba);
				
			ERROR_END;

                        goto ERROR_NO_MEM;

		}

		if (adpt_scsi_register(pHba, sht) < 0) {
			adpt_i2o_delete_hba(pHba);

			continue;
		}
		
		pHba->initialized = TRUE;

		pHba->state &= ~ADPT_STATE_IOP_RESET;
	}


	/*
	 * Register our control device node
	 * nodes will need to be created in /dev to access this
	 * the nodes can not be created from within the driver
	 */ 
	if (hba_count && register_chrdev(ASA72XX_I2O_MAJOR, 
				ASA72XX_DRIVER, &adpt_fops)) {
		for (pHba = hba_chain; pHba; pHba = pHba->next) {
			if (pHba->host) {
				scsi_unregister(pHba->host);
			}
		}
		
		pci_unregister_driver(&asa72xx_pci_driver);

		adpt_i2o_sys_shutdown();

		ERROR_END;

                goto ERROR_UNREGISTER_DRIVER;
	}

	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		/*
		 * Enable Interrupt Batching.
		 */
		 /* Operation Code */
		opblk[0] = I2O_PARAMS_FIELD_SET; 
		 /* Group Number */
		opblk[1] = I2O_DPT_HBA_OPER_PARAM_GROUP_NO; 
		/* Field Count */
		opblk[2] = 0x0001;  
		/* FieldIdx[0] */	
		opblk[3] = I2O_DPT_HBA_OPER_PARAM_INT_BATCH_FIELD_NO; 
		/* Parameter */
		opblk[4] = pSetup->interrupt_batching;	
		/* Padding */
		opblk[5] = 0x0000;
		
		PDEBUG(DBG_INIT, "Set Interrupt Batching to %d !\n",
				opblk[4]);

		rcode = adpt_i2o_issue_params(I2O_CMD_UTIL_PARAMS_SET, pHba, 0, 
				    opblk, sizeof(opblk), 
				    NULL, 0);

		if (rcode != 0) {
			PDEBUG(DBG_INIT, "Failed to enable "
					 "Interrupt Batching !\n");
		}
	}
		
	/*
	 * Register our shutdown/reboot notifier to the kernel.
	 */  	
	register_reboot_notifier(&adpt_shutdown_notifier);
#ifndef __VMKERNEL_MODULE__	
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	spin_lock_irq(&io_request_lock);
#endif
#endif //__VMKERNEL_MODULE__
	END; 

	return hba_count;

ERROR_UNREGISTER_DRIVER:
        pci_unregister_driver(&asa72xx_pci_driver);

ERROR_UNLOAD_DRIVER:
#ifndef __VMKERNEL_MODULE__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	spin_lock_irq(&io_request_lock);
#endif
#endif //__VMKERNEL_MODULE__
#ifdef ASA72XX_DEBUG
        __beeponce();
        __beeponce();
#endif

        return (0);

ERROR_NO_MEM:
#ifndef __VMKERNEL_MODULE__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
        spin_lock_irq(&io_request_lock);
#endif
#endif //__VMKERNEL_MODULE__
        return -ENOMEM;

}

/*
 * adpt_init()
 *
 * Description:
 *	Initialize the HBAs array and command line setups.
 */	    
S32 
adpt_init()
{
	U8 	i;

	BEGIN;
	
	PWARN("Loading ASA-7211x - 1GB iSCSI HBA Linux Driver Version: " 
			ASA72XX_I2O_VERSION "\n");
	
	for (i = 0; i < DPTI_MAX_HBA; i++) {
		hbas[i] = NULL;
	}

	pSetup = (adpt_setup *) kmalloc(sizeof(adpt_setup), GFP_ATOMIC);
	
	if (pSetup == NULL) {
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN |
				UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("ERROR: %s [%x] \n", uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

		return -ENOMEM;
	}

	pSetup->device_q_depth = (ASA72XX_MAX_DEV_Q_DEPTH / 2);

	pSetup->command_timeout = TMOUT_SCSI;

	pSetup->interrupt_batching = IB_DEFAULT;

	pSetup->linkdown_timeout = TMOUT_LINKDOWN;

	END;

	return 0;
}


/*
 * adpt_pci_dev_probe()
 *
 * Description:
 *	Probe for all supported Host Bus Adapters.
 */
static S32
adpt_pci_dev_probe(struct pci_dev *pDev, const struct pci_device_id *id)
{
	BEGIN;

	if (adpt_install_hba(pDev)) {
		ERROR_END;

		return -ENODEV;
	}

	END;

	return 0;
}

/*
 * adpt_pci_dev_remove()
 *
 * Description:
 *	For now, do nothing and just return. 
 *	In the future when we support Hot Plug or Hot Remove , we might need
 *	to do perform clean up in here.
 */			     
static void 
adpt_pci_dev_remove(struct pci_dev *pDev)
{
	BEGIN;

	END;

	return;
}
		

/*
 * adpt_async_register_event()
 * 
 * Description:   
 *	This routine is to send request to the FW to turn the 
 *	event notification on or off.
 */		 
static S32 
adpt_async_register_event(adpt_hba *pHba, U32 events_mask)
{
	U32	msg[MAX_MESSAGE_SIZE/4];
	U32	msg_size;
	S32	rcode = 0;

	BEGIN;

	msg[0] = I2O_MESSAGE_SIZE(7) | SGL_OFFSET_0;
	msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
	msg[2] = 0x21000000;	/* Initiator Context (Private context) */
	msg[3] = (U32) msg;	/* Target Context (we want the msg back) */
	msg[4] = (DPT_ORGANIZATION_ID << 16) | I2O_PRVT_EVT_REGISTERATION;
	msg[5] = 0x00010000;	/* Interpret bit set to indicate as IOCTL */
	msg[6] = events_mask;	/* Events that we want to be notified */
	
	/* Message size in U32 words */
	msg_size = 7;
	
	/* Message size in U8 bytes */
	msg_size = 7 << 2;

	rcode = adpt_i2o_post_wait(pHba, msg, msg_size, TMOUT_ASYNC_EVENT);

	if ((rcode != 0) || (!event_registered)) {
		PDEBUG(DBG_EVENT, "Failed to post the message or "
				"register event !\n");

		ERROR_END;

		return 1;
	}
	       	
	END;

	return 0;
}	


/*
 * adpt_async_send_notification()
 * 
 * Description:
 * 	Send a Private Async. Event to FW to report back of any notification
 *	that needs to be communicated to driver.
 * 	Reply Indicator could be either :
 *	 (a) IMMEDIATE_REPLY : FW will respond to this message immediately.
 *	 (b) NORMAL_REPLY: FW will only respond if an event is happening.	 */
static void
adpt_async_send_notification(adpt_hba *pHba, U8 reply_indicator)
{
	U32	msg[MAX_MESSAGE_SIZE/4];
	S32	msg_size = 0;
	S32	rcode = 0;

	BEGIN;

	msg[0] = I2O_MESSAGE_SIZE(7) | SGL_OFFSET_0;
	msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
	msg[2] = 0x22000000;	/* Initiator Context (Private context) */
	msg[3] = (U32) msg;	/* Target Context (we want the msg back) */
	msg[4] = (DPT_ORGANIZATION_ID << 16) | I2O_PRVT_EVT_NOTIFICATION;
	msg[5] = 0x00010000;	/* Interpret bit set to indicate as IOCTL */
	msg[6] = reply_indicator;	/* Reply Indicator */

	/* Message size in U32 words */
	msg_size = 7;
	
	/* Message size in U8 bytes */
	msg_size = 7 << 2;

	/*
	 * If we want the reslut back immediately from FW, we will stay and 
	 * wait. If not, just post and leave.
	 */
     	if (reply_indicator == IMMEDIATE_REPLY) {
		PDEBUG(DBG_EVENT, "Post Async Event Msg with IMD_REPLY !\n");

		rcode = adpt_i2o_post_wait(pHba, msg, msg_size, 
				TMOUT_ASYNC_EVENT);
	} else {	
		PDEBUG(DBG_EVENT, "Post Async Event Msg with NORMAL_REPLY !\n");

		rcode = adpt_i2o_post_this(pHba, msg, msg_size);
	}

	if (rcode != 0) {
		PDEBUG(DBG_EVENT, "Failed to post event to FW !\n");

	}

	END;

	return;
}


/*
 * adpt_init_cmd_pool()
 *
 * Description:
 * 	Initialize cmd pool for each device found (3 cmds per device)
 */ 	   
static U32
adpt_init_cmd_pool(adpt_hba *pHba)
{
	adpt_cmnd	*pCmnd;
	adpt_cmnd	**pFQHead;
	adpt_cmnd	**pFQTail;
	U16		total_cmd_needed = 0;
	U16		i;
	
	BEGIN;

	pFQHead = (adpt_cmnd **)&pHba->FreeQueueHead;

	pFQTail = (adpt_cmnd **)&pHba->FreeQueueTail;

	pHba->total_cmds = 0;

        total_cmd_needed = ASA72XX_MAX_INTERNAL_CMD + 2;	

	for (i = 0; i < total_cmd_needed; i++) {
		pCmnd = (adpt_cmnd *) kmalloc(sizeof(adpt_cmnd), GFP_ATOMIC);
		
		if (pCmnd == NULL) {
			continue;
		}

		memset((void *) pCmnd, 0 ,sizeof(adpt_cmnd));
		
		if (*pFQHead == NULL) {
			*pFQHead = pCmnd;	
		} else {
			(*pFQTail)->next = pCmnd;

			pCmnd->prev = *pFQTail;
		}

		*pFQTail = pCmnd;

		pHba->total_cmds++;
	}

	PDEBUG(DBG_INIT, "Allocated %d cmds in Free Queue !\n\n",
			pHba->total_cmds);

	END;

	return pHba->total_cmds;
}


/* 
 * adpt_i2o_post_wait_data() 
 *
 * Description: 
 * 	Structures and definitions for synchronous message posting.
 * 	See adpt_i2o_post_wait() for description
 */
struct 
adpt_i2o_post_wait_data
{
	S32 				status;
	U32 				id;
	adpt_wait_queue_head_t 		*wq;
	struct adpt_i2o_post_wait_data 	*next;
};

/*
 * adpt_i2o_post_wait()
 *
 * Description:
 *	Insert the current task to the command wait queue.
 *	Start a timer routine to watch out for the time out
 *	Wait for an event.
 *	There are 3 possible outcomes
 *	1)If the timeout occurs, then check if the entry has been
 *	submitted to the FW. If this is submitted to the FW, then 
 *	just return with timeout value
 *	2)If timeout occurs and the command has not yet been submitted
 *	to the FW, then just remove the entry from the linked list
 *	3)If FW triggered an event, then just return the status in pHBA
 */  
S32 
adpt_i2o_post_wait(adpt_hba *pHba, U32 *msg, S32 len, S32 timeout)
{
	S32 status;
	padpt_command_request pio_request=NULL;

	BEGIN;
	
	/*
	 * Note that this memory will be deallocated at the end of this
	 * function for all cases
	 * */
	pio_request = kmalloc(sizeof(adpt_command_request),
				GFP_ATOMIC);

	if (!pio_request) {
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN |
				UC_NO_EXT | U_MALLOC_FAIL);
		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
				uni_err(uerrno), (U32) uerrno);
		ERROR_END;
		return -ENOMEM;
	}
	memset(pio_request, 0, sizeof(adpt_command_request));

	pio_request->msg=msg;
	pio_request->len=len;
	pio_request->timeout=timeout;

	init_waitqueue_head(&pio_request->thread_processing_queue);
	atomic_set(&pio_request->fw_completed_processing,0);
	atomic_set(&pio_request->fw_started_processing,0);
	atomic_set(&pio_request->fw_timed_out,0);
#ifdef __VMKERNEL_MODULE__
        init_MUTEX_LOCKED(&pio_request->io_done_sem);	
#endif

	/*
	 * Add the command to the queue and wake it up for processing
	 * */
	*(msg+2) |= POST_WAIT_CONTEXT;
	add_cmd_to_io_queue(pHba,pio_request);
#ifdef __VMKERNEL_MODULE__
	up(&pHba->wake_up_worker_sem);
#else
	wake_up(&pHba->wake_up_worker);
#endif //__VMKERNEL_MODULE__

	/*
	 * If timeout value is specified, then just add the timer
	 * */
	if(timeout){
		init_timer(&pio_request->cmd_timeout_timer);
		pio_request->cmd_timeout_timer.data = (U32) pio_request;
		pio_request->cmd_timeout_timer.expires = jiffies + (HZ*timeout);
		pio_request->cmd_timeout_timer.function = (void (*) (U32_64))cmd_timeout_trigger;
		add_timer(&pio_request->cmd_timeout_timer);
	}

	/*
	 * Wait for either the FW/Worker thread or the timer to wake us up
	 * */
	PDEBUG(DBG_I2O_POST,"\nWaiting for the worker to wake us up");
#if 0
	if(atomic_read(&pio_request->fw_completed_processing) == 0){
		sleep_on(&pio_request->thread_processing_queue);
	}
#endif

#ifdef __VMKERNEL_MODULE__
	/*  We do not support sleep_on and it's variants in the
	 *  vmkernel. We cannot use vmk_thread_wait_event here either,
	 *  because of a potential race condition between the if
	 *  check and the wait that causes a deadlock. Hence we spin. */
	down(&pio_request->io_done_sem);
#else
	while(1){
		if((atomic_read(&pio_request->fw_completed_processing) == 0) && (atomic_read(&pio_request->fw_timed_out) == 0)){
			interruptible_sleep_on_timeout(&pio_request->thread_processing_queue,HZ/10);
		}else{
			break;
		}

	}
#endif
	PDEBUG(DBG_I2O_POST,"\nEither the worker/timeout woke us up");

	if(atomic_read(&pio_request->fw_completed_processing) != 0){
		if(timeout){
			PDEBUG(DBG_I2O_POST,"\nFW completed the request");
			del_timer_sync(&pio_request->cmd_timeout_timer);
		}

		/*The FW has processed this request*/
		status = pHba->cmd_status;
		END;
	}else{
		/*
		 * We might end up here even when we get a signal here
		 * if we sleep_interruptible. For now ignore this case
		 * Even in that case we return a timeout*/
		if(atomic_read(&pio_request->fw_started_processing) != 0){
			/*Just make sure that the event does not get
			 * called for the deleted io command
			 * */
			atomic_set(&pHba->fw_drop_current_io,1);
		}
		status = -ETIMEDOUT;
		PDEBUG(DBG_I2O_POST,"\nCommand Timeout occured");
		ERROR_END;
	}
	delete_cmd_from_io_queue(pHba,pio_request);
	kfree(pio_request);
	return status;
}

/*
 * cmd_timeout_trigger()
 *
 * Description:
 *	Wake up the timeout routine. If the FW returned a response
 *	before a timeout occured, then this will not be called as 
 *	the timer routine is deleted
 * */
void cmd_timeout_trigger(U32 dataptr)
{
	padpt_command_request pio_request=(adpt_command_request *)dataptr;
	BEGIN;

	if(dataptr){
		atomic_set(&pio_request->fw_timed_out,1);
#ifdef __VMKERNEL_MODULE__
		up(&pio_request->io_done_sem);
#else
		wake_up(&pio_request->thread_processing_queue);
#endif //__VMKERNEL_MODULE__
	}
	END;
}

/*
 * delete_cmd_from_io_queue()
 *
 * Description:
 * 	This command will delete the node for the given data pointer
 * 	Returns SUCCESS if the deletion went through smoothly or returns an
 * 	error if the NODE is missing
 * */
S32
delete_cmd_from_io_queue(adpt_hba *pHba,adpt_command_request *pio_request)
{
	U32	flags=0;
	U8	dbg_match_found=0;
	adpt_command_request *preq=NULL;

	BEGIN;
	/*
	 * Hold the Spinlock
	 * */
	spin_lock_irqsave(&pHba->protect_io_cmd,flags);

	// Get the matching Node
	preq = pHba->pio_cmd_base_ptr;
	while(preq){
		if(preq == pio_request){
			if(preq->prev == NULL){
				pHba->pio_cmd_base_ptr = preq->next;
			}
			dbg_match_found=1;
			break;
		}
		preq = preq->next;
	}
	if(!dbg_match_found){
		spin_unlock_irqrestore(&pHba->protect_io_cmd,flags);
		PDEBUG(DBG_I2O_POST,"\nUnable to delete the Node");
		ERROR_END;
		return -1;
	}

	// We need to delete this node from the list
	if(preq->next){
		preq->next->prev = preq->prev;
	}
	if(preq->prev){
		preq->prev->next = preq->next;
	}

	/*
	 * Release the Spinlock
	 * */
	spin_unlock_irqrestore(&pHba->protect_io_cmd,flags);

	END;

	return 0;
}

/*
 * add_cmd_to_io_queue()
 *
 * Description:
 * 	This command will add the node to the end of the list
 * */
S32
add_cmd_to_io_queue(adpt_hba *pHba,adpt_command_request *pio_request)
{
	U32	flags=0;
	adpt_command_request *preq=NULL;

	BEGIN;

	PDEBUG(DBG_I2O_POST,"\nCommand added to the queue");
	/*
	 * Hold the Spinlock
	 * */
	spin_lock_irqsave(&pHba->protect_io_cmd,flags);

	if(!pHba->pio_cmd_base_ptr){
		pHba->pio_cmd_base_ptr=pio_request;
	}else{
		preq=pHba->pio_cmd_base_ptr;
		while(preq->next != NULL){
			preq = preq->next;
		}
		preq->next=pio_request;
		pio_request->prev = preq;
	}

	/*
	 * Release the Spinlock
	 * */
	spin_unlock_irqrestore(&pHba->protect_io_cmd,flags);

	END;

	return 0;
}

/*
 * get_cmd_from_io_queue()
 *
 * Description:
 * 	This command will return the first available node for processing 
 * */
adpt_command_request *
get_cmd_from_io_queue(adpt_hba *pHba)
{
	U32	flags=0;
	adpt_command_request *preq=NULL;

	/*
	 * Hold the Spinlock
	 * */
	spin_lock_irqsave(&pHba->protect_io_cmd,flags);

	preq=pHba->pio_cmd_base_ptr;
	while(preq){
		if(atomic_read(&preq->fw_completed_processing) != 0){
			preq = preq->next;
		}else{
			break;
		}
	}
	/*
	 * Release the Spinlock
	 * */
	spin_unlock_irqrestore(&pHba->protect_io_cmd,flags);

	return preq;
}

/*
 * adpt_i2o_post_this()
 *
 * Description: 
 *	Post the message to outbound queue.  
 */  
S32 
adpt_i2o_post_this(adpt_hba *pHba, U32 *data, S32 len)
{
	U32 	m;
	U32 	*msg;
	U32 	timeout;

	timeout = jiffies + (TMOUT_POST * HZ);
	
	do {
		rmb();

		m = readl(pHba->ToIopFifo);
	
		if (m != EMPTY_QUEUE) {
			break;
		}
		
		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "%s: Timeout waiting for message "
					"frame!\n", 
					pHba->name);
		
			ERROR_END;

			return -ETIMEDOUT;
		}
	} while (m == EMPTY_QUEUE);

	
	msg = (U32 *) (pHba->base_addr_virt + m);
		
	memcpy_toio(msg, data, len);
	
	wmb();
	
	/* post message */
	writel(m, pHba->ToIopFifo);

	wmb();

	return 0;
}

/*
 * adpt_i20_activate_hba()
 *
 * Description:
 *	Bring an I2O controller into HOLD state. See the spec.
 */
S32 
adpt_i2o_activate_hba(adpt_hba *pHba)
{
	S32 rcode;
	
	BEGIN;
	
	if (pHba->initialized) {
		if (adpt_i2o_status_get(pHba) < 0) {
			if ((rcode = adpt_i2o_reset_hba(pHba) != 0)) {
				PDEBUG(DBG_INIT, "%s: Could NOT reset.\n", 
						pHba->name);

				ERROR_END;
				
				return rcode;
			}
			
			if (adpt_i2o_status_get(pHba) < 0) {
				PDEBUG(DBG_INIT, "HBA not responding.\n");

				ERROR_END;

				return -1;
			}
		}

		if (pHba->status_block->iop_state == ADAPTER_STATE_FAULTED) {
			PDEBUG(DBG_INIT, "%s: hardware fault\n", pHba->name);

			ERROR_END;

			return -1;
		}

		if ((pHba->status_block->iop_state == ADAPTER_STATE_READY) ||
			(pHba->status_block->iop_state == 
				ADAPTER_STATE_OPERATIONAL) ||
			(pHba->status_block->iop_state == 
				ADAPTER_STATE_HOLD) ||
			(pHba->status_block->iop_state == 
				ADAPTER_STATE_FAILED)) {
			adpt_i2o_reset_hba(pHba);			
		
			if (adpt_i2o_status_get(pHba) < 0 || 
				pHba->status_block->iop_state != 
				ADAPTER_STATE_RESET) {
				PDEBUG(DBG_INIT, "%s: Failed to initialize.\n",
					       	pHba->name);

				ERROR_END;

				return -1;
			}
		}
	} else {
		if ((rcode = adpt_i2o_reset_hba(pHba) != 0)) {
			PDEBUG(DBG_INIT, "%s: Could NOT reset.\n", 
					pHba->name);

			ERROR_END;

			return rcode;
		}

	}

	if (adpt_i2o_init_outbound_q(pHba) < 0) {
		ERROR_END;

		return -1;
	}

	/* In HOLD state */
	
	if (adpt_i2o_hrt_get(pHba) < 0) {
		ERROR_END;

		return -1;
	}
	
	END;

	return 0;
}


/*
 * adpt_i2o_reset_hba()
 */  
S32 
adpt_i2o_reset_hba(adpt_hba *pHba)			
{
	U32 		msg[8];
	U32 		m = EMPTY_QUEUE;
	U32 		timeout = jiffies + (TMOUT_IOP_RESET * HZ);
	U8  		*status;
	dma_addr_t	status_bus_addr;	

	BEGIN;
	
	if (pHba->initialized  == FALSE) {	
		/* First time reset should be quick */
		timeout = jiffies + (TMOUT_IOP_FIRST_RESET * HZ);
	} else {
		adpt_i2o_quiesce_hba(pHba);
	}

	do {
		rmb();
	
		m = readl(pHba->ToIopFifo);
	
		if (m != EMPTY_QUEUE) {
			break;
		}
		
		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "Timeout waiting for message!\n");

			ERROR_END;

			return -ETIMEDOUT;
		}
	} while (m == EMPTY_QUEUE);

	status = pci_alloc_consistent(pHba->pDev, 4, &status_bus_addr);
			
	if (status == NULL) {
		adpt_send_nop(pHba, m);
		
		PDEBUG(DBG_INIT, "IOP reset failed - no free memory.\n");
		
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN |
				UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
				uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

		return -ENOMEM;
	}
	
	memset(status, 0, 4);

	msg[0] = EIGHT_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = I2O_CMD_IOP_RESET << 24 | HOST_TID << 12 | ADAPTER_TID;
	msg[2] = 0;		/* Reserved */
	msg[3] = 0;		/* Reserved */
	msg[4] = 0;		/* Reserved */
	msg[5] = 0;		/* Reserved */
	msg[6] = status_bus_addr;
	msg[7] = 0;     

	memcpy_toio(pHba->base_addr_virt + m, msg, sizeof(msg));
	
	wmb();
	
	writel(m, pHba->ToIopFifo);
	
	wmb();

	while ((volatile U8) *status == 0) {
		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "%s: IOP Reset Timeout\n", 
					pHba->name);
			
			pci_free_consistent(pHba->pDev, 4, status, 
					status_bus_addr);
			
			ERROR_END;

			return -ETIMEDOUT;
		}
		
		wmb();
	}

	if ((volatile U8) *status == IN_PROGRESS) {
		PDEBUG(DBG_INIT, "%s: Reset in progress...\n",
				pHba->name);
		/* 
		 * Here we wait for message frame to become available
		 * indicated that reset has finished
		 */ 
		do { 
			rmb();
		
			m = readl(pHba->ToIopFifo);
			
			if (m != EMPTY_QUEUE) {
				break;
			}
			
			if (time_after(jiffies, timeout)) {
				PDEBUG(DBG_TIMEOUT, "%s:Timeout waiting for "
						"IOP Reset.\n", pHba->name);
				
				ERROR_END;

				return -ETIMEDOUT;
			}
		} while (m == EMPTY_QUEUE);
		
		/* 
		 * Flush the offset
		 */
		adpt_send_nop(pHba, m);
	}
	
	adpt_i2o_status_get(pHba);
	
	if (*status == REJECTED ||
		(pHba->status_block->iop_state != ADAPTER_STATE_RESET)) {
		PDEBUG(DBG_INIT, "%s: Reset reject, trying to clear\n",
				pHba->name);
	} else {
		PDEBUG(DBG_INIT, "%s: Reset completed.\n",
				pHba->name);
	}

	pci_free_consistent(pHba->pDev, 4, status, status_bus_addr);
	
	END;

	return 0;
}


/*
 * adpt_i2o_quiesce_hba()
 */  
S32 
adpt_i2o_quiesce_hba(adpt_hba* pHba)
{
	U32  	msg[4];
	S32 	ret;

	BEGIN;
	
	adpt_i2o_status_get(pHba);

	/* SysQuiesce discarded if IOP not in READY or OPERATIONAL state */

	if ((pHba->status_block->iop_state != ADAPTER_STATE_READY) &&
   	   (pHba->status_block->iop_state != ADAPTER_STATE_OPERATIONAL)) {
		return 0;
	}

	msg[0] = FOUR_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = I2O_CMD_SYS_QUIESCE << 24 | HOST_TID << 12 | ADAPTER_TID;
	msg[2] = 0;
	msg[3] = 0;

	if ((ret = adpt_i2o_post_wait(pHba, msg, sizeof(msg), 240))) {
		PDEBUG(DBG_POST, "%s: Unable to quiesce (status=%#x).\n",
					pHba->name, -ret);
	} else {
		PDEBUG(DBG_POST, "%s: Quiesced.\n", pHba->name);
	}

	adpt_i2o_status_get(pHba);
	
	END;

	return ret;
}


/*
 * adpt_i2o_status_get()
 *
 * Description:
 *	Send the ExecStatusGet message to the IOP to place its status block
 *	in the buffer specified by the host.
 */  
S32 
adpt_i2o_status_get(adpt_hba *pHba)
{
	U32 		timeout;
	U32 		m;
	U32 		msg[9];
	volatile U8 	*status_block = NULL;

	BEGIN;
	
	if (pHba->status_block == NULL) {
		pHba->status_block = (i2o_status_block *)
				pci_alloc_consistent(pHba->pDev, 
					sizeof(i2o_status_block),
					&pHba->status_block_bus_addr);
		
		if (pHba->status_block == NULL) {
			PDEBUG(DBG_INIT, "%s: Get Status Block failed; "
					 "Out of memory. \n", pHba->name);
			
			uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | 
					UO_CMN | UC_NO_EXT | U_MALLOC_FAIL);

			PERROR("%s: ERROR: %s [%x] \n", pHba->name,
					uni_err(uerrno), (U32) uerrno);
		
			ERROR_END;

			return -ENOMEM;
		}
	}
	
	memset(pHba->status_block, 0, sizeof (i2o_status_block));
	
	status_block = (U8 *)(pHba->status_block);
	
	timeout = jiffies + (TMOUT_GET_STATUS * HZ);
	
	do {
		rmb();

		m = readl(pHba->ToIopFifo);
	
		if (m != EMPTY_QUEUE) {
			break;
		}
		
		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "%s: Timeout waiting for "
					"message !\n",
					pHba->name);

			ERROR_END;

			return -ETIMEDOUT;
		}
	} while (m == EMPTY_QUEUE);

	msg[0] = NINE_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = I2O_CMD_STATUS_GET << 24 | HOST_TID << 12 | ADAPTER_TID;
	msg[2] = 1;
	msg[3] = 0;
	msg[4] = 0;
	msg[5] = 0;
	msg[6] = pHba->status_block_bus_addr & 0xffffffff;
	msg[7] = 0;
	msg[8] = sizeof(i2o_status_block); /* 88 bytes */
	
	memcpy_toio(pHba->base_addr_virt + m, msg, sizeof(msg));

	wmb();

	writel(m, pHba->ToIopFifo);
	
	wmb();

	/* 
	 * Keep polling until all information is written to status_block.
	 * If the last byte of status block is 0xff, then it is done.
	 */  
	while (status_block[CMD_STATUS] != 0xff) {
		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "%s: Get status timeout.\n",
				pHba->name);
			
			ERROR_END;

			return -ETIMEDOUT;
		}
		
		wmb();
	}

	/* 
	 * Set up our number of outbound and inbound messages
	 */
 	pHba->ToIop_Fifo_Size = pHba->status_block->max_inbound_frames;

	if (pHba->ToIop_Fifo_Size > MAX_TO_IOP_MESSAGES) {
		pHba->ToIop_Fifo_Size = MAX_TO_IOP_MESSAGES;
	}

	pHba->FromIop_Fifo_Size = pHba->status_block->max_outbound_frames;
	
	if (pHba->FromIop_Fifo_Size > MAX_FROM_IOP_MESSAGES) {
		pHba->FromIop_Fifo_Size = MAX_FROM_IOP_MESSAGES;
	}

	/* 
	 * Calculate the Scatter Gather list size
	 */
 	 pHba->sg_tablesize = (pHba->status_block->inbound_frame_size * 4 - 40)
		 		/ sizeof(struct sg_simple_element);
	 
	if (pHba->sg_tablesize > SG_LIST_ELEMENTS) {
		pHba->sg_tablesize = SG_LIST_ELEMENTS;
	}

#ifdef ASA72XX_DEBUG	
	PDEBUG(DBG_INIT, "%s: State = ", pHba->name);

	switch(pHba->status_block->iop_state) {
		case 0x01:
			PDEBUG(DBG_INIT, "INIT\n");
			break;
			
		case 0x02:
			PDEBUG(DBG_INIT, "RESET\n");
			break;
			
		case 0x04:
			PDEBUG(DBG_INIT, "HOLD\n");
			break;
			
		case 0x05:
			PDEBUG(DBG_INIT, "READY\n");
			break;
			
		case 0x08:
			PDEBUG(DBG_INIT, "OPERATIONAL\n");
			break;
			
		case 0x10:
			PDEBUG(DBG_INIT, "FAILED\n");
			break;
			
		case 0x11:
			PDEBUG(DBG_INIT, "FAULTED\n");
			break;

		default:
			PDEBUG(DBG_INIT, "%x (unknown!!)\n",
				pHba->status_block->iop_state);
	}
#endif	

	END;

	return 0;
}


/*
 * adpt_i2o_hrt_get()
 *
 * Description:
 *	Send the ExecHrtGet message to the host to notify the IOP's hardware
 *	resource table. HRT lists all adapters and location that the IOP
 *	can control.			   
 */  
S32 
adpt_i2o_hrt_get(adpt_hba *pHba)
{
	U32 	msg[6];
	U32 	new_hrt_size = 0;
	S32 	ret;

	BEGIN;

	pHba->hrt_size = sizeof(i2o_hrt);

	do {
		if (pHba->hrt == NULL) {
			pHba->hrt = pci_alloc_consistent(pHba->pDev, 
					pHba->hrt_size,
					&pHba->hrt_bus_addr);
			
			if (pHba->hrt == NULL) {
				PDEBUG(DBG_INIT, "%s: Hrt Get failed; "
						"Out of memory.\n", 
						pHba->name);
				
				uerrno = (UL_ERROR | UP_INIT | US_SYS | 
						transport | UO_CMN | 
						UC_NO_EXT | U_MALLOC_FAIL);

				PERROR("%s: ERROR: %s [%x] \n", pHba->name,
						uni_err(uerrno), 
						(U32) uerrno);
		
				ERROR_END;

				return -ENOMEM;
			}
		}

		msg[0] = SIX_WORD_MSG_SIZE| SGL_OFFSET_4;
		msg[1] = I2O_CMD_HRT_GET << 24 | HOST_TID << 12 | ADAPTER_TID;
		msg[2] = 0;
		msg[3] = 0;
		msg[4] = (0xD0000000 | pHba->hrt_size); /* Simple transaction */
		msg[5] = pHba->hrt_bus_addr;

		if ((ret = adpt_i2o_post_wait(pHba, msg, sizeof(msg), 120))) {
			PDEBUG(DBG_POST, "%s: Unable to get HRT "
					"(status=%#10x)\n", 
					pHba->name, ret);

			ERROR_END;

			return ret;
		}

		if (pHba->hrt->num_entries * pHba->hrt->entry_len << 2 > 
				pHba->hrt_size) {
			new_hrt_size = pHba->hrt->num_entries * 
						pHba->hrt->entry_len << 2;
			
			pci_free_consistent(pHba->pDev, pHba->hrt_size, 
					pHba->hrt,
					pHba->hrt_bus_addr);
			
			pHba->hrt = NULL;

			pHba->hrt_size = new_hrt_size;
		}
	} while (pHba->hrt == NULL);

	END;

	return 0;
}                                                                                                                                       

/*
 * adpt_i2o_query_scalar()
 *
 * Description:
 *	 Query one scalar group value or a whole scalar group.
 */		    	
S32 
adpt_i2o_query_scalar(adpt_hba *pHba, S32 tid, S32 group, S32 field, 
			void *buf, S32 buflen)
{
	S32 	size;
#if 0
	U16 	opblk[] = { 1, 0, I2O_PARAMS_FIELD_GET, group, 1, field };
	U8  	resblk[8+buflen]; /* 8 bytes for header */
#endif
	U16	*opblk=NULL;
	U8	*resblk=NULL;

	BEGIN;

	opblk = (U16 *)kmalloc(OP_BLOCK_SIZE, GFP_ATOMIC | GFP_DMA);
	if(!opblk){
		END;
		return 0;
	}

	resblk = (U8 *)kmalloc((8+buflen), GFP_ATOMIC | GFP_DMA);
	if(!resblk){
		kfree(opblk);
		END;
		return 0;
	}
	memset(opblk, 0, OP_BLOCK_SIZE);
	memset(resblk, 0, (8+buflen));

	opblk[0]=1;
	opblk[1]=0;
	opblk[2]=I2O_PARAMS_FIELD_GET;
	opblk[3]=group;
	opblk[4]=1;
	opblk[5]=field;

	if (field == -1) {	
		/* whole group */
		opblk[4] = -1;
	}

	size = adpt_i2o_issue_params(I2O_CMD_UTIL_PARAMS_GET, pHba, tid, 
		opblk, sizeof(opblk), resblk, sizeof(resblk));
			
	memcpy(buf, resblk+8, buflen);  /* cut off header */

	kfree(opblk);
	kfree(resblk);

	if (size < 0) {
		return size;	
	}

	END;

	return buflen;
}


/*
 * adpt_i2o_lct_get()
 * 
 * Description: 
 * 	Send the ExecLctNotify message to get the IOP's 
 * 	Logical Configuration Table
 */
S32 
adpt_i2o_lct_get(adpt_hba *pHba)
{
	U32 	msg[8];
	U32	new_lct_size = 0;
	S32 	ret;
	
	BEGIN;
	
	if ((pHba->lct_size == 0) || (pHba->lct == NULL)) {
		pHba->lct_size = pHba->status_block->expected_lct_size;
	}

	do {
		if (pHba->lct == NULL) {
			pHba->lct = pci_alloc_consistent(pHba->pDev,
					pHba->lct_size, &pHba->lct_bus_addr);

			if (pHba->lct == NULL) {
				PDEBUG(DBG_LCT, "%s: Lct Get failed. "
						 "Out of memory.\n",
						 pHba->name);
				
				uerrno = (UL_ERROR | UP_INIT | US_SYS | 
						transport | UO_CMN | 
						UC_NO_EXT | U_MALLOC_FAIL);

				PERROR("%s: ERROR: %s [%x] \n", pHba->name,
						uni_err(uerrno), 
						(U32) uerrno);
		
				ERROR_END;

				return -ENOMEM;
			}
		}
		
		memset(pHba->lct, 0, pHba->lct_size);

		msg[0] = EIGHT_WORD_MSG_SIZE | SGL_OFFSET_6;
		msg[1] = I2O_CMD_LCT_NOTIFY<<24 | HOST_TID<<12 | ADAPTER_TID;
		msg[2] = 0;
		msg[3] = 0;
		msg[4] = 0xFFFFFFFF;	/* All devices */
		msg[5] = 0x00000000;	/* Report now */
		msg[6] = 0xD0000000 | pHba->lct_size;
		msg[7] = pHba->lct_bus_addr;

		if ((ret = adpt_i2o_post_wait(pHba, msg, sizeof(msg), 
						TMOUT_LCTGET))) {
			PDEBUG(DBG_LCT, "%s: LCT Get failed (status=%#10x) \n", 
				pHba->name, ret);	
		
			PDEBUG(DBG_LCT, "%s: Error Reading LCT.\n", pHba->name);
		
			ERROR_END;

			return ret;
		}
		
		PDEBUG(DBG_LCT, "\nLogical Configuration Table read !\n");

		if (pHba->lct->table_size << 2 > pHba->lct_size) {
			new_lct_size = pHba->lct->table_size << 2;
			
			pci_free_consistent(pHba->pDev, pHba->lct_size,
					pHba->lct, pHba->lct_bus_addr);
			
			pHba->lct = NULL;

			pHba->lct_size = new_lct_size;
		}
	} while (pHba->lct == NULL);

	END;

	return 0;
}


/*
 * adpt_get_device()
 *
 * Description:
 * 	Find if the device belongs to the target.
 * 	Return the device if found. Otherwise, return NULL. 	 	    
 */
adpt_device *
adpt_get_device(adpt_target *pTarget, U16 tid, U16 scsi_lun)
{
	adpt_device	*pDev = NULL;
	
	BEGIN;

	if (pTarget->device == NULL) {
		return NULL;
	}
	
	for (pDev = pTarget->device; pDev; pDev = pDev->next_lun) {
		// TID is not guaranteed to be the same across
		// LCT notifies. But if its a LUN we're looking for within
		// a target, LUN # will be the same.
			 
		// if (pDev->tid == tid) {
		if (pDev->scsi_lun == scsi_lun) {
		   PDEBUG(DBG_LCT,"Tid found %d\n", tid);
			return pDev;
		}
	}

	END;
	
	return NULL;
}


/*
 * adpt_i2o_get_class_name()
 *
 * Description:   
 *	Do i2o class name lookup
 */
const S8 
*adpt_i2o_get_class_name(S32 class)
{
	S32 idx = 16;
	
	static S8 *i2o_class_name[] = {
		"Executive",
		"Device Driver Module",
		"Block Device",
		"Tape Device",
		"LAN Interface",
		"WAN Interface",
		"Fibre Channel Port",
		"Fibre Channel Device",
		"SCSI Device",
		"ATE Port",
		"ATE Device",
		"Floppy Controller",
		"Floppy Device",
		"Secondary Bus Port",
		"Peer Transport Agent",
		"Peer Transport",
		"Unknown"
	};
	
	switch(class & 0xFFF) {
		case I2O_CLASS_EXECUTIVE:
			idx = 0; break;
		case I2O_CLASS_DDM:
			idx = 1; break;
		case I2O_CLASS_RANDOM_BLOCK_STORAGE:
			idx = 2; break;
		case I2O_CLASS_SEQUENTIAL_STORAGE:
			idx = 3; break;
		case I2O_CLASS_LAN:
			idx = 4; break;
		case I2O_CLASS_WAN:
			idx = 5; break;
		case I2O_CLASS_FIBRE_CHANNEL_PORT:
			idx = 6; break;
		case I2O_CLASS_FIBRE_CHANNEL_PERIPHERAL:
			idx = 7; break;
		case I2O_CLASS_SCSI_PERIPHERAL:
			idx = 8; break;
		case I2O_CLASS_ATE_PORT:
			idx = 9; break;
		case I2O_CLASS_ATE_PERIPHERAL:
			idx = 10; break;
		case I2O_CLASS_FLOPPY_CONTROLLER:
			idx = 11; break;
		case I2O_CLASS_FLOPPY_DEVICE:
			idx = 12; break;
		case I2O_CLASS_BUS_ADAPTER_PORT:
			idx = 13; break;
		case I2O_CLASS_PEER_TRANSPORT_AGENT:
			idx = 14; break;
		case I2O_CLASS_PEER_TRANSPORT:
			idx = 15; break;
	}
	
	return i2o_class_name[idx];
}


/*
 * adpt_open()
 *
 * Description: 
 *	This routine is called when a user open our module device node
 *	for IOCTL purpose.    
 */  
S32 
adpt_open(struct inode *inode, struct file *file)
{
	adpt_hba	*pHba;
	S32 		minor;

	minor = MINOR(inode->i_rdev);
	
	if (minor >= hba_count) {
		ERROR_END;
		
		return -ENXIO;
	}

	down(&adpt_configuration_lock);
	
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		if (pHba->unit == minor) {		
			break;	/* found adapter */
		}
	}
	
	if (pHba == NULL) {
		up(&adpt_configuration_lock);

		ERROR_END;
		
		return -ENXIO;
	}

	if (pHba->in_use > hba_count) {
		up(&adpt_configuration_lock);

		return -EBUSY;
	}

	pHba->in_use++;

	/*
	 * Make a call to MOD_INC_USE_COUNT so that the module system 
	 * knows not to unload it while the device is open.
	 */   
	if (pHba->host->hostt->module) {
		__MOD_INC_USE_COUNT(pHba->host->hostt->module);
	}	
	
	up(&adpt_configuration_lock);

	return 0;
}


/*
 * adpt_close()
 *
 * Description: 
 *	This routine is called when a user close our module device node.    
 */  
S32 
adpt_close(struct inode *inode, struct file *file)
{
	adpt_hba	*pHba;
	S32 		minor;

	minor = MINOR(inode->i_rdev);
	
	if (minor >= hba_count) {
		return -ENXIO;
	}
	
	down(&adpt_configuration_lock);
	
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		if (pHba->unit == minor) {
			break;	/* found adapter */
		}
	}
	
	up(&adpt_configuration_lock);
	
	if (pHba == NULL) {
		return -ENXIO;
	}

	pHba->in_use--;

	/*
	 * call MOD_DEC_USE_COUNT to indicate that the module system 
	 * can now be unloaded
	 */    
	if (pHba->host->hostt->module) {
		__MOD_DEC_USE_COUNT(pHba->host->hostt->module);
	}

	return 0;
}


/*
 * adpt_i2o_passthru()
 */  
S32 
adpt_i2o_passthru(adpt_hba *pHba, U32 *arg)
{
	U32 	msg[MAX_MESSAGE_SIZE/4];
	U32	*reply = NULL;
	U32 	size = 0;
	U32 	reply_size = 0;
	U32	*user_msg = (U32*)arg;
	U32	*user_reply = NULL;
	U32 	sg_offset = 0;
	U32 	sg_count = 0;
	U32 	i = 0;
	U32 	rcode = 0;
	U32_64 	flags = 0;
	S32 	sg_index = 0;
	caddr_t p = NULL;
	caddr_t sg_list[pHba->sg_tablesize];
	
	BEGIN;

	memset(&msg, 0, MAX_MESSAGE_SIZE);
	
	/*
	 * get user msg size in U32s 
	 */ 
	get_user(size, &user_msg[0]);
	size = size >> 16;

	user_reply = &user_msg[size];
	size *= 4; // Convert to bytes
	
	if (size > MAX_MESSAGE_SIZE) {
		ERROR_END;
		
		return -EFAULT;
	}

	/* 
	 * Copy in the user's I2O command 
	 */
	if (copy_from_user ((void *) msg, (void *) user_msg, size)) {
		ERROR_END;
		
		return -EFAULT;
	}
	
	get_user(reply_size, &user_reply[0]);

	reply_size = reply_size >> 16;
	
	if (reply_size > REPLY_FRAME_SIZE) {
		reply_size = REPLY_FRAME_SIZE;
	}
	
	reply_size *= 4;

	reply = kmalloc(REPLY_FRAME_SIZE * 4, GFP_KERNEL);
	
	if (reply == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate reply buffer\n",
					pHba->name);

		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | 
				UO_CMN | UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
					uni_err(uerrno), 
					(U32) uerrno);
		
		ERROR_END;

		return -ENOMEM;
	}
	
	memset(reply, 0, REPLY_FRAME_SIZE * 4);
	
	sg_offset = (msg[0]>>4) & 0xf;
	msg[2] = 0x40000000; // IOCTL context
	msg[3] = (U32)reply;
	
	memset(sg_list, 0, sizeof(sg_list[0]) * pHba->sg_tablesize);
	
	if (sg_offset) {
		struct sg_simple_element *sg = (struct sg_simple_element*) 
				(msg + sg_offset);
		
		sg_count = (size - sg_offset * 4) / (sizeof(struct 
					sg_simple_element));
		
		if (sg_count > pHba->sg_tablesize) {
			PDEBUG(DBG_IOCTL, "%s:IOCTL SG List too large (%u)\n",
				       		pHba->name,sg_count);
			
			kfree (reply);
			
			ERROR_END;

			return -EINVAL;
		}

		for (i = 0; i < sg_count; i++) {
			U32 	physical;
			caddr_t virtual;
			S32 	residual, length, sub_length;

			 /*I2O_SGL_FLAGS_SIMPLE_ADDRESS_ELEMENT*/
			if (!(sg[i].flag_count & 0x10000000)) {
				PDEBUG(DBG_IOCTL, "%s:Bad SG element %d - "
							"not simple (%x)\n",
							pHba->name,i, 
							sg[i].flag_count);
			
				rcode = -EINVAL;

				goto cleanup;
			}
			
			/* Allocate memory for the transfer */
			p = (caddr_t)kmalloc(sg[i].flag_count & 0xffffff, 
					GFP_KERNEL | ADDR32);
			
			if (p == NULL) {
				PDEBUG(DBG_IOCTL, "%s: Could not allocate SG buffer\n", 
							pHba->name);
				
				uerrno = (UL_ERROR | UP_INIT | US_SYS |
						       	transport | 
							UO_CMN | UC_NO_EXT | 
							U_MALLOC_FAIL);

				PERROR("%s: ERROR: %s [%x] \n", 
							pHba->name,
							uni_err(uerrno), 
							(U32) uerrno);
		
				rcode = -ENOMEM;

				goto cleanup;
			}
			
			/* 
			 * sglist indexed with input frame, not our internal 
			 * frame.
			 */ 
			sg_list[sg_index++] = p; 
			
			/* Copy in the user's SG buffer if necessary */
			if (sg[i].flag_count & 0x04000000) {
				if (copy_from_user(p,
					(caddr_t)(sg[i].addr_low32),
					sg[i].flag_count&0xffffff)) {
					PDEBUG(DBG_IOCTL, "%s: Could not copy "
							"FROM user\n",
							pHba->name);

					rcode = -EFAULT;

					goto cleanup;
				}
			}
			
			sg[i].addr_low32 = (U32)virt_to_bus(p);
			
			/*
			 * Need to split requests should the buffer not be 
			 * physically contiguous
			 * It gets ugly from here on in ...
			 */ 
			physical = sg[i].addr_low32;
			virtual = p;
			residual = (sg[i].flag_count & 0xffffff);
			length = 0;
			
			while ((sub_length = (((physical + PAGE_SIZE) & 
					PAGE_MASK) - physical)) < residual) {
				/*
				 * split this request?
				 */ 
				length += sub_length;
				residual -= sub_length;
				physical += sub_length;
				virtual += sub_length;
				
				if (physical != (U32)virt_to_bus(virtual)) {
					S32 j;
				
					if (++sg_count > pHba->sg_tablesize) {
						PDEBUG(DBG_IOCTL, "%s:IOCTL SG"
							      "List too large "
							      "(%u)\n", 
							      pHba->name,
							      sg_count);
						
						rcode = -EINVAL;
						
						goto cleanup;
					}
					
					if ((size += sizeof(struct 
						sg_simple_element)) > 
						MAX_MESSAGE_SIZE) {
						PDEBUG(DBG_IOCTL, "%s:IOCTL "
							"Message too large " 
							"(%u)\n", 
							pHba->name,
							size);
						
						rcode = -EINVAL;

						goto cleanup;
						
					}
					
					/*
					 * fixup message and scatter gather 
					 * list.
					 */
					msg[0] &= 0xffff;
					
					/* 
					 * shift upwards the scatter gather 
					 * list.
					 */ 
					msg[0] |= (size / sizeof(U32)) << 16;
					
					sg[i].flag_count &= ~0xffffff;
					
					for (j = sg_count; --j > i;) {
						sg[j] = sg[j-1];
					}
					
					/* 
					 * length &= 0xffffff;
					 */ 
					sg[i].flag_count |= length;
					length = 0;
					sg[++i].addr_low32 = physical = 
						(U32)virt_to_bus(virtual);

					/*
					 * residual &= 0xffffff;
					 */
					sg[i].flag_count |= residual;
				}
			}
		}
	}

	adpt_lock(pHba, flags, 1);
	
	/* 
	 * This state stops any new commands from enterring the
	 * controller while processing the ioctl
	 * pHba->state |= ADPT_STATE_IOCTL;
	 * We can't set this now - The scsi subsystem sets host_blocked and
	 * the queue empties and stops.  We need a way to restart the queue.
	 */ 
	rcode = adpt_i2o_post_wait(pHba, msg, size, FOREVER);

	//pHba->state &= ~ADPT_STATE_IOCTL;

	adpt_unlock(pHba, flags, 1);

	if (!rcode && sg_offset) {
		/* 
		 * Copy back the Scatter Gather buffers back to user space 
		 */
		U32 			 j;
		struct sg_simple_element *sg;

		/* 
		 * re-acquire the original message to handle correctly the sg 
		 * copy operation
		 */ 
		memset(&msg, 0, MAX_MESSAGE_SIZE);
		
		/* 
		 * get user msg size in U32s 
		 */ 
		get_user(size, &user_msg[0]);
		size = size >> 16;
		size *= 4;
		
		/* 
		 * Copy in the user's I2O command 
		 */
		if (copy_from_user ((void*)msg, (void*)user_msg, size)) {
			rcode = -EFAULT;

			goto cleanup;
		}
		
		sg_count = (size - (sg_offset * 4)) / sizeof(struct 
				sg_simple_element);

		sg = (struct sg_simple_element *)(msg + sg_offset);
		
		for (j = 0; j < sg_count; j++) {
			/* 
			 * Copy out the SG list to user's buffer if necessary 
			*/
			if (!(sg[j].flag_count & 0x4000000)) {
				if (copy_to_user((caddr_t)(sg[j].addr_low32),
					sg_list[j], 
					sg[j].flag_count & 0xffffff)) {
					PDEBUG(DBG_IOCTL, "%s: Could not "
						"copy %p TO user %p\n",
						pHba->name, sg_list[j],
						(caddr_t)sg[j].addr_low32);
					
					rcode = -EFAULT;

					goto cleanup;
				}
			}
		}
	} 

	/* 
	 * Copy back the reply to user space 
	 */
	if (reply_size) {
		/* 
		 * we wrote our own values for context - now restore the user 
		 * supplied ones.
		 */ 
		if (copy_from_user(reply + 2, user_msg + 2, sizeof(U32) * 2)) {
			PDEBUG(DBG_IOCTL, "%s: Could not copy message context "
					"FROM user\n", pHba->name);
			
			rcode = -EFAULT;
		}
		
		if (copy_to_user(user_reply, reply, reply_size)) {
			PDEBUG(DBG_IOCTL, "%s: Could not copy reply TO user\n",
					pHba->name);
			
			rcode = -EFAULT;
		}
	}

cleanup:
	kfree (reply);
	/*
	 * Now try and drain the mid-level queue if any commands have been
	 * inserted.  Check to see whether the queue even has anything in
	 * it first, as otherwise this is useless overhead.
	 */
	i = sg_index; 
	
	while (i) {
		if (sg_list[--i]) {
			kfree(sg_list[i]);
		}
	}
	
	return rcode;
}

/***********************************
* Handle all IP Storage IOCTL's here
************************************/

/*
 * adpt_mgmt_ioctl()
 * 
 * Description:
 *	This is a function to handle our private IOCTL requests.   
 */  
S32 
adpt_mgmt_ioctl(adpt_hba *pHba, U32 *arg)
{
	U32 		msg[MAX_MESSAGE_SIZE/4];
	U32		*user_msg = (U32 *) arg;
	U32		*msg_buffer;
	U32 		msg_size, reply_size, size;
	U32		arg1, arg2;
	U32 		sg_count = 0;
	U32 		rcode = 0;
	U8 		direction;
	controller_info	CtlrInfo;
	S8		*driver_version;
	S8		*driver_name;
	dma_addr_t	msg_buffer_bus_addr;

	BEGIN;

	memset(&msg, 0, MAX_MESSAGE_SIZE);

	/*
	 * get user msg size in U32s 
	 */ 
	get_user(arg1, &user_msg[0]);

	/* 
	 * get user msg size in next U32s 
	 */ 
	get_user(arg2, &user_msg[1]);

	PDEBUG(DBG_IOCTL, "arg1 = 0x%x !\n", (arg1 & 0x0000007F)); 
	
	/**********************************************************
	* Handle all Driver Specific IOCTL's here.
	* Assumes user has previously allocated enough memory
	* before calling these ioctls 
	***********************************************************/

	if ((arg1 & 0x0000007F) == DRIVER_SPECIFIC_IOCTL) {
		if (((arg1 & 0x0000FF00) >> 8) == GET_NUM_CONTROLLERS) {
			
			if (copy_to_user((S8*)(user_msg + (sizeof(
					ioctl_header)/4)), (S8 *) &hba_count,
					sizeof(hba_count))) {
				ERROR_END;

				return -EFAULT;
			} else {
				END;

				return rcode;
			}
		}
		
		if (((arg1 & 0x0000FF00) >> 8) == GET_CONTROLLER_INFO) {
			memset(&CtlrInfo, 0, sizeof(CtlrInfo));
		
			CtlrInfo.pciBusNum =  pHba->pDev->bus->number;
			CtlrInfo.pciDeviceNum = PCI_SLOT(pHba->pDev->devfn); 
			CtlrInfo.pciFunctionNum = PCI_FUNC(pHba->pDev->devfn); 
			CtlrInfo.pciBaseAddr = (ulong) pHba->base_addr_phys;
			CtlrInfo.pciIntrVector = pHba->pDev->irq; 
		
			driver_version = ASA72XX_I2O_VERSION;
			driver_name = ASA72XX_DRIVER_NAME;

			strcpy(CtlrInfo.driverVersion, driver_version);
			strcpy(CtlrInfo.driverName, driver_name);

			if (copy_to_user((S8 *) (user_msg+(sizeof(
					ioctl_header)+4)/4), (S8 *) &CtlrInfo,
					sizeof(CtlrInfo))) {
				ERROR_END;

				return -EFAULT;
			} else {
				END;

				return rcode;
			}
		}
		
		if (((arg1 & 0x0000FF00) >> 8) == GET_BTL_MAPPING) {
			rcode = getbtlmapping(pHba,user_msg);
			
			END;

			return rcode;
		}
	}			

	/**********************************************************
	* Handle all  F/W Specific IOCTL's here
	***********************************************************/

	/* 
	 * get page length and check for non zero.if it is zero
	 * then get the size from expected length.
	 */ 
	size = ((arg1 & 0xFFFF0000) >> 16);
	
	if (!size) {
		size = arg2 & 0x0000FFFF;
	}

	/* 
	 * sg_count is the number of bytes to be transferred i.e. pagelength
	 */ 
	sg_count = size;

	/*
	 * find out the direction and reply size
	 */
	direction = ((arg1 & 0x80000000) >> 31);	
	
	if (direction == IOCTL_SET_DATA) {	
		reply_size = 16;	//size of header
	} else {
		reply_size = size;
	}

	PDEBUG(DBG_IOCTL, "\nPage Length = %x reply_size = %x "
			"direction = %x\n", 
			size, reply_size, direction);

	/*
	 * allocate kernel space for user message IOCTL
	 */ 
	msg_buffer = pci_alloc_consistent(pHba->pDev, size, 
				&msg_buffer_bus_addr);
	
	if (msg_buffer == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate message buffer\n",
					pHba->name);

		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | 
					UO_CMN | UC_NO_EXT | 
					U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
					uni_err(uerrno), 
					(U32) uerrno);
		
		rcode = -ENOMEM;
		
		goto MGMTIOCTL_cleanup;
	}
	
	memset(msg_buffer, 0, size);

	/* 
	 * copy in the user's IOCTL request packet 
	 */
	if (copy_from_user((void *) msg_buffer, (void *) user_msg, size)) {
		ERROR_END;

		return -EFAULT;
	}

	/* 
	 * fill I2O message
	 */ 
	msg[0] = I2O_MESSAGE_SIZE(11) | SGL_OFFSET_9;
	msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
	msg[2] = 0x40000000;	// Initiator Context (IOCTL context)
	msg[3] = (U32)msg;	// Target Context (we want the msg back)
	msg[4] = (DPT_ORGANIZATION_ID << 16) | ASA72XX_MGMT_IOCTL_CMD;
	msg[5] = 0x00010000;	//interpret bit set to indicate as IOCTL
	msg[6] = size;		//number of bytes to be transferred
	msg[7] = 0xCD000000 | (1 * 8); // one SG element always present
	msg[8] = 0;		// buffer context

	/* 
	 * fill up the SG list
	 */
	msg[9] = msg_buffer_bus_addr;
	msg[10] = sg_count;
	
	/*
	 * message size in U32 words
	 */ 
	msg_size = 11;

	/* 
	 * message size in U8 bytes
	 */
	msg_size = 11 << 2;

	//pHba->state |= ADPT_STATE_IOCTL;

	rcode = adpt_i2o_post_wait(pHba, msg, msg_size, FOREVER);  

	if (!rcode) {
		if (direction == IOCTL_SET_DATA) {
			if (copy_to_user((void *)user_msg, (void *)msg_buffer, 
					reply_size)) {
				PDEBUG(DBG_IOCTL, "%s: Could not copy TO "
							"user\n", 
							pHba->name);
				
				rcode = -EFAULT;
				
				goto MGMTIOCTL_cleanup;
			}
		} else {
			if (copy_to_user((void *)user_msg, (void *)msg_buffer,						reply_size)) {
				PDEBUG(DBG_IOCTL, "%s: Could not copy TO "
							"user\n",
							pHba->name);
				
				rcode = -EFAULT;
				
				goto MGMTIOCTL_cleanup;
			}
		}
	}

MGMTIOCTL_cleanup:
	//pHba->state &= ~ADPT_STATE_IOCTL;

	pci_free_consistent(pHba->pDev, size, msg_buffer, msg_buffer_bus_addr);
	
	END;

	return rcode;
}

/*
 * adpt_system_info()
 */  
S32 
adpt_system_info(void *buffer)
{
	S32 		i;
	S32 		j;
	caddr_t 	c_addr;
	sysInfo_S 	si;

	memset(&si, 0, sizeof(si));

	si.osType = OS_LINUX;
	si.osMajorVersion = (U8) (LINUX_VERSION_CODE >> 16);
	si.osMinorVersion = (U8) (LINUX_VERSION_CODE >> 8 & 0x0ff);
	si.osRevision =     (U8) (LINUX_VERSION_CODE & 0x0ff);
	si.busType = SI_PCI_BUS;
	si.processorFamily = DPTI_sig.dsProcessorFamily;

#ifdef i386
#ifdef __VMKERNEL_MODULE__
	/* boot_cpu_data is not an exported symbol from
	 * vmkernel. Since ESX is probably not going to be used with 
	 * 386/486, we fix this */
	si.processorType = PROC_PENTIUM;
#else
	switch (boot_cpu_data.x86) {
		case CPU_386:
			si.processorType = PROC_386;
			break;
		case CPU_486:
			si.processorType = PROC_486;
			break;
		case CPU_586:
			si.processorType = PROC_PENTIUM;
			break;
	}
#endif //__VMKERNEL_MODULE__
	outb(0x70, 0x12);
	i = inb(0x71);
	j = i >> 4;
	
	if (i == 0x0f) {
		outb(0x70, 0x19);
		j = inb(0x71);
	}
	
	si.drive0CMOS = j;

	j = i & 0x0f;
	
	if (i == 0x0f) {
		outb(0x70, 0x1a);
		j = inb(0x71);
	}
	
	si.drive1CMOS = j;

	si.numDrives = *((S8 *) phys_to_virt(0x475));

	si.flags = SI_CMOS_Valid | SI_NumDrivesValid |
			    SI_OSversionValid | SI_BusTypeValid;
	
	/* 
	 * Go Out And Look For SmartROM 
	 */
	for (i = 0; i < 3; ++i) {
		S32 k;

		if (i == 0) {
			j = 0xC8000;
		} else if (i == 1) {
			j = 0xD8000;
		} else {
			j = 0xDC000;
		}
		
		c_addr = phys_to_virt(j);
		
		if (*((U16 *) c_addr) != 0xAA55) {
			continue;
		}
		
		if (*((U32_64 *) (c_addr + 6)) != 0x202053) {
			continue;
		}
		
		if (*((U32_64 *) (c_addr + 10)) != 0x545044) {
			continue;
		}
		
		c_addr += 0x24;
		
		for (k = 0; k < 64; ++k) {
			if ((*((U8 *) (c_addr++)) == ' ') &&
			    (*((U8 *) (c_addr))   == 'v')) {
				break;
			}
		}
		
		if (k < 64) {
			si.smartROMMajorVersion = *((U8 *)(c_addr +=
					       	3)) - '0';
			
			si.smartROMMinorVersion = *((U8 *)(c_addr +=
					       	2));
			
			si.smartROMRevision = *((U8 *)(++c_addr));
			si.flags |= SI_SmartROMverValid;
			
			break;
		}
	}
	
	if (i >= 3) {
		si.flags |= SI_NO_SmartROM;
	}

	if (si.numDrives > 0) {
		/*
		 * Get The Pointer From Int 41 For The First
		 * Drive Parameters
		 */
		j =  ((unsigned)(*((U16 *)phys_to_virt(0x104 + 2))) << 4)
		    + (unsigned)(*((U16 *)phys_to_virt(0x104 + 0)));
		/*
		 * It appears that SmartROM's Int41/Int46 pointers
		 * use memory that gets stepped on by the kernel
		 * loading. We no longer have access to this
		 * geometry information but try anyways (!?)
		 */
		si.drives[0].cylinders = *((U8 *)phys_to_virt(j));
		++j;
		si.drives[0].cylinders += ((S32)*((U8 *)phys_to_virt(
						j))) << 8;
		
		++j;
		si.drives[0].heads = *((U8 *) phys_to_virt(j));
		j += 12;
		si.drives[0].sectors = *((U8 *) phys_to_virt(j));
		si.flags |= SI_DriveParamsValid;
		
		if ((si.drives[0].cylinders == 0)
		    || (si.drives[0].heads == 0)
		    || (si.drives[0].sectors == 0)) {
			si.flags &= ~SI_DriveParamsValid;
		}
		
		if (si.numDrives > 1) {
			/*
			 * Get The Pointer From Int 46 For The
			 * Second Drive Parameters
			 */
			j = ((unsigned)(*((U16 *)phys_to_virt(0x118 +
				2))) << 4) + (unsigned)(*((U16 *)
				phys_to_virt(0x118  + 0)));
			
			si.drives[1].cylinders = *((U8 *)
					phys_to_virt(j));
			
			++j;
			
			si.drives[1].cylinders += ((S32)*((U8*)
						phys_to_virt(j)))<< 8;
			
			++j;
			
			si.drives[1].heads =  *((U8 *)
					phys_to_virt(j));
			
			j += 12;
			
			si.drives[1].sectors = *((U8 *)
				       	phys_to_virt(j));
			
			if ((si.drives[1].cylinders == 0) || 
			    (si.drives[1].heads == 0) || 
			    (si.drives[1].sectors ==  0)) {
				si.flags &=  ~SI_DriveParamsValid;
			}
		}
	}

#elif (defined __ia64)
	si.processorType = PROC_IA64;
	//TODO more info
#else
	si.processorType = 0xff ;
#endif
	copy_to_user(buffer, &si, sizeof(si));

	return 0;
}


/*
 * adpt_ioctl()
 *
 * Description:
 *	Entry point for IOCTL call to the driver.    
 */  
S32 
adpt_ioctl(struct inode *inode, struct file *file, U32 cmd, U32_64 arg)
{
	S32 		minor;
	S32 		error = 0;
	adpt_hba	*pHba= NULL;

	BEGIN;

	minor = MINOR(inode->i_rdev);
	
	if (minor >= DPTI_MAX_HBA) {
		ERROR_END;

                goto IOCTL_SYS_ERROR;
	}
	
	down(&adpt_configuration_lock);
	
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		if (pHba->unit == minor) {
			break;	/* found adapter */
		}
	}
	
	up(&adpt_configuration_lock);
	
	if (pHba == NULL) {
		ERROR_END;

                goto IOCTL_SYS_ERROR;
	}

	while ((volatile U32) pHba->state & ADPT_STATE_IOP_RESET) {
		schedule();
	}

        /*
         *  If there is no pending IOs, lets send down IOCTL and lock it.
         *  Else, lets give it 10 tries to see if IOs are blocked.
        {
	U8 		retry = 0;

        while ( !adpt_check_queues_empty(pHba) ){

                if (retry > 10) {
		  ERROR_END;

                  goto IOCTL_ERROR;
                }
		PDEBUG(DBG_IOCTL, "IOCTL blocked by pending IOs (%d)\n",cmd);
		retry ++;
		schedule();
        }
	PDEBUG(DBG_IOCTL, "IOCTL passed (%d)\n",cmd);
        }

         */


	switch (cmd) {
		case DPT_SIGNATURE:
		if (copy_to_user((S8 *) arg, &DPTI_sig, sizeof(DPTI_sig))) {
			ERROR_END;

                        goto IOCTL_ERROR;
		}
		
		break;
		
	case I2OUSRCMD:
		return adpt_i2o_passthru(pHba, (U32 *) arg);
		
	case ASA72XX_NUM_CTRLS:
		if (copy_to_user((S8 *) arg, &hba_count, sizeof(hba_count))) {
			ERROR_END;

                        goto IOCTL_ERROR;
		}
		
		break;

	case DPT_CTRLINFO:
	{
		drvrHBAinfo_S HbaInfo;

#define FLG_OSD_PCI_VALID 0x0001
#define FLG_OSD_DMA	  0x0002
#define FLG_OSD_I2O	  0x0004
	
		memset(&HbaInfo, 0, sizeof(HbaInfo));
		HbaInfo.drvrHBAnum = pHba->unit;
		HbaInfo.baseAddr = (ulong) pHba->base_addr_phys;
		HbaInfo.blinkState = adpt_getBlinkLED(pHba);
		HbaInfo.pciBusNum =  pHba->pDev->bus->number;
		HbaInfo.pciDeviceNum=PCI_SLOT(pHba->pDev->devfn); 
		HbaInfo.Interrupt = pHba->pDev->irq; 
		HbaInfo.hbaFlags = FLG_OSD_PCI_VALID | FLG_OSD_DMA | 
				   FLG_OSD_I2O;
		
		copy_to_user((void *) arg, &HbaInfo, sizeof(HbaInfo));
		
		break;
	}

	case DPT_SYSINFO:
		return adpt_system_info((void *) arg);
		
	case DPT_BLINKLED:
	{	
		U32 value;
		
		value = adpt_getBlinkLED(pHba);
	
		if (copy_to_user((S8 *) arg, &value, sizeof(value))) {
			ERROR_END;

                        goto IOCTL_ERROR;
		}
	
		break;
	}

	case I2ORESETCMD:
		break;

	case I2ORESCANCMD:
		adpt_rescan(pHba, TRUE);
		
		break;
		
	case DPT_TARGET_BUSY & 0xFFFF:
	case DPT_TARGET_BUSY:
	{	
		TARGET_BUSY_T 		busy;
		adpt_device		*d;

		if (copy_from_user((void *)&busy, (void *) arg, 
					sizeof(TARGET_BUSY_T))) {
			ERROR_END;
                        goto IOCTL_ERROR;

		}

		d = adpt_find_device(pHba, busy.channel, busy.id, busy.lun);
		
		if (d == NULL) {
			ERROR_END;
                        pHba->state &= ~ADPT_STATE_IOCTL;

			return -ENODEV;
		}
		
		busy.isBusy = ((d->pScsi_dev) && 
			(0 != d->pScsi_dev->access_count)) ? 1 : 0;
		
		if (copy_to_user ((S8 *) arg, &busy, sizeof(busy))) {
			ERROR_END;

                        goto IOCTL_ERROR;
		}

		break;
	}

	case ASA72XX_MGMT_IOCTL_CMD:
                pHba->state &= ~ADPT_STATE_IOCTL;
		return	adpt_mgmt_ioctl(pHba, (U32 *) arg);
		
	default:
		PDEBUG(DBG_IOCTL, "Unknown IOCTL cmd (%d)\n",cmd);
	
		ERROR_END;
		goto IOCTL_IO_ERROR;	
	}

        /* ASSERT */
        ASSERT(0);
        pHba->state &= ~ADPT_STATE_IOCTL;
	return error;

IOCTL_ERROR:
        pHba->state &= ~ADPT_STATE_IOCTL;
	return -EFAULT;

IOCTL_SYS_ERROR:
        pHba->state &= ~ADPT_STATE_IOCTL;
	return -ENXIO;

IOCTL_IO_ERROR:
        pHba->state &= ~ADPT_STATE_IOCTL;
	return -EINVAL;

}

/*
 * adpt_info()
 *
 * Description:
 * 	Return information describing the driver.     
 */  
const S8 *
adpt_info(struct Scsi_Host *host)
{
	adpt_hba *pHba;

        BEGIN;
	pHba = (adpt_hba *) host->hostdata[0];

	sprintf(pHba->detail, "Adaptec ASA-72xx 1Gb iSCSI Host Controller");

	END;
	return (S8 *) (pHba->detail);
}


/*
 * adpt_get_device_state()
 * 
 * Description:
 * 	Return the string that represent the current device state.
 */
const S8
*adpt_get_device_state(U8 state)
{
	U8	array_id;

#if LINUX_VERSION_CODE == KERNEL_VERSION(2,4,9)
	static 	S8 *device_state[] = {
		"ON",
		"OFF",
		"ABS"
	};	
#else
	static 	S8 *device_state[] = {
		"ONLINE",
		"OFFLINE",
		"ABSENT"
	};	
#endif

	switch (state & 0xFF) {
		case ADPT_DEV_OFFLINE:
			array_id = 1;
			break;

		case ADPT_DEV_ABSENT:
			array_id = 2;
			break;

		//case ADPT_DEV_ONLINE:
		default:
			array_id = 0;
			break;
	}

	return (device_state[array_id]);
}


/*
 * adpt_proc_info()
 *
 * Description:
 *	The /proc filesystem support.   
 */ 
S32 adpt_proc_info(S8 *buffer, S8 **start, off_t offset,
		  S32 length, S32 hostno, S32 inout)
{
	adpt_hba		*pHba;
	adpt_target 		*pTarget;
	adpt_device		*pDev;
	struct Scsi_Host 	*host;
	S32 			id;
	S32 			chan;
	S32 			len = 0;
	S32 			begin = 0;
	S32 			pos = 0;
	U16			SubDeviceId;
	S8			*format_string;

	BEGIN;

	*start = buffer;
	
	/*
	 * We have returned buffer to (buffer + offset) in previous calls
	 * Set start to where we want the beginning of the new data to be.
	 */
	*start = buffer + offset;

	/*
	 * Find HBA (host bus adapter) we are looking for.
	 */
	down(&adpt_configuration_lock);
	
	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		if (pHba->host->host_no == hostno) {
			break;	/* found adapter */
		}
	}
	
	up(&adpt_configuration_lock);
	
	if (pHba == NULL) {
		return (-ENXIO);
	}
	
	host = pHba->host;

	if (inout) {
		/*
		 * The user has done a write and wants us to take the
		 * data in the buffer and do something with it.
		 * proc_scsiwrite calls us with inout = 1
		 *
		 * Read data from buffer (writing to us)
		 */

		buffer[length] = '\0';

		/*
		 * Check at least we have 'asa72xx" to start with.
		 */  
		if (!buffer || length < 15 || strncmp("asa72xx", buffer, 7)) {
			ERROR_END;

			return (-EINVAL);
		}

		/*
		 * Request for Device Table Rescan.
		 * Usage : echo "asa72xx rescan" > /proc/scsi/asa72xx
		 */  	
		if (!strncmp("rescan", buffer + 8, 6)) {
			adpt_rescan(pHba, TRUE);

			END;

			return length;
		}

		/*
		 * Request for HBA shutdown in error/shutdown condition.
                 * reboot notifier is not fast enough for most cases.
		 * Usage : echo "asa72xx shutdown" > /proc/scsi/asa72xx
		 */  	
		if (!strncmp("shutdown", buffer + 8, 8)) {

	                pHba->state |= ADPT_STATE_SHUTDOWN;
	                PDEBUG(DBG_UNLOAD, "shutdown \n");

			END;

			return length;
		}

               /*
	        * Lets dynamically adjust the driver's verbosity level
		* Usage : echo "asa72xx dbgflag 1" > /proc/scsi/asa72xx
		if (!strncmp("dbgall", buffer + 8, 6)) {
	                PDEBUG(DBG_ERR_HNDL, "debug_level :  %d !\n",
						    debug_level);
		        debug_level = DBG_ALL;

			END;

			return length;
	        }
                else
		        debug_level = DBG_NONE;
	        */

		ERROR_END;
		
		return (-EINVAL);
	}

	/*
	 * inout = 0 means the user has done a read and wants information
	 * returned, so we write information about the cards into the buffer
	 * proc_scsiread() calls us with inout = 0
	 */

	pci_read_config_word(pHba->pDev, PCI_DEVICE_ID, &SubDeviceId);
			
	switch (SubDeviceId) {
		case 0x7211:
			format_string = 
				"\nAdaptec Storage Accelerator 7211C - 1Gb iSCSI HBA #%d\n"
				"Driver Version %s "
				"on PCI bus %02x device %02x irq %d\n\n";
			
			break;
			
		case 0x7210:
			format_string = 
				"\nAdaptec Storage Accelerator 7211F - 1Gb iSCSI HBA #%d\n"
				"Driver Version %s "
				"on PCI bus %02x device %02x irq %d\n\n";
			
			break;

		case 0x0:
			format_string = 
				"\nAdaptec Storage Accelerator 7211 - 1Gb iSCSI HBA #%d\n"
				"Driver Version %s "
				"on PCI bus %02x device %02x irq %d\n\n";
			
			break;

		default:
			format_string = 
				"\nAdaptec ASA-7211 - 1Gb iSCSI HBA #%d\n"
				"Driver Version %s "
				"on PCI bus %02x device %02x irq %d\n\n";
			
			break;
	}

	len  += sprintf(buffer, format_string, pHba->unit, ASA72XX_I2O_VERSION,
			pHba->pDev->bus->number,
			PCI_SLOT(pHba->pDev->devfn),
			pHba->pDev->irq);

	len += sprintf(buffer + len, "SCSI Host No. %d. Device "
			"Node =/dev/%s%d\n",
		       	pHba->host->host_no, pHba->name, pHba->unit);

	len += sprintf(buffer + len, "Firmware HostName=[%s], System"
                         " HostName=[%s]\n\n",
			 pHba->machine_name,pSetup->machine_name);
	
	len += sprintf(buffer + len, "\tPost FIFO Size = %d\n",
			host->can_queue); 

	len += sprintf(buffer + len, "\tReply FIFO Size = %d\n",
			(S32) pHba->FromIop_Fifo_Size);
	
	len += sprintf(buffer + len, "\tCurrent Active Cmds = %d\n"
			"\tCurrent Active Abort Count = %d\n",
			pHba->active_cmds, pHba->aborted_cmd_count);

	pos = begin + len;

	for (chan = 0; chan < MAX_CHANNEL; chan++) {
		for (id = 0; id < MAX_ID; id++) {
			pTarget =  pHba->channel[chan].target[id];
			
			if (pTarget == NULL) {
				continue;
			}

			if ((pDev = pTarget->device) == NULL) {
                                /* Target has no LUN */
				continue;
			}

#if LINUX_VERSION_CODE == KERNEL_VERSION(2,4,9)
                           if (len > 2048) goto SKIP_VPRINT; 
#else
                           if (len >= 3072) goto SKIP_VPRINT;
#endif
			len +=  sprintf(buffer + len, 
					"\nSCSI Devices for Target: %s\n",
					pTarget->iscsi_name);
	
			while (pDev) {
#if LINUX_VERSION_CODE == KERNEL_VERSION(2,4,9)
   if (!(pDev->state & ADPT_DEV_ABSENT)){
    if (len > 2048) goto SKIP_VPRINT;
                        /*
                         * There is currently a bug in AS 2.1
                         * that causes a panic in vsprintf 
                         */

				len += sprintf(buffer+len, " Ch = %d, "
					"Id = %d, Lun = %d:%s\n",
					(U8) pDev->scsi_channel, 
					(U8) pDev->scsi_id, 
					(U8) pDev->scsi_lun,
					adpt_get_device_state(pDev->state));
     		
   }
#else
   if (!(pDev->state & ADPT_DEV_ABSENT)){
    if (len >= 3072) goto SKIP_VPRINT;
			       /*
				* works for everything other than AS2.1 kernel.
				* I am seeing length as always 3072. No point doing 
				* more. Using goto for compiler optimization.
				*/
				len += sprintf(buffer+len, "\tChannel = %d, "
					"Id = %d, Lun = %d, Status : %s,%s\n",
					(U8) pDev->scsi_channel, 
					(U8) pDev->scsi_id, 
					(U8) pDev->scsi_lun,
					adpt_get_device_state(pDev->state),
                                        (pHba->state & 0x20) ? "CONN": 
                                        "DISC");
    
   }
#endif
				pos = begin + len;

				pDev = pDev->next_lun;
			}
		}
		
SKIP_VPRINT:
		if (pos < offset) {
			/*
			 * If we haven't even written to where we last left
			 * off (the last time we were called), reset the 
			 * beginning pointer.
			 */
			len = 0;
			begin = pos;
		}
		
		if (pos > offset + length) {
			/*
			 * We have written past the end of the buffer 
			 * (proc_read/writescsi gives us 1K slop
			 *
			 * stop the output and calculate the correct length
			 */
			goto stop_output;
		}
	}
	
	/*
	 * begin is where we last checked our position with regards to offset
	 * begin is always less than offset.  len is relative to begin.  It
	 * is the number of bytes written past begin.
	 */
	
stop_output:
	*(buffer + len) = '\0';

	len -= (offset - begin);
	
	if (len > length) {
		len = length;
	}
	
	END;

	return len;
}


/*
 * adpt_i2o_build_sys_table()
 *
 * Description:   
 * 	I2O System Table.  Contains information about
 * 	all the IOPs in the system.  Used to inform IOPs
 * 	about each other's existence.
 *
 *	sys_tbl_ver is the CurrentChangeIndicator that is
 * 	used by IOPs to track changes.
 */
S32 
adpt_i2o_build_sys_table(void)
{
	adpt_hba	*pHba = NULL;
	S32 		count = 0;

	sys_tbl_len = sizeof(struct i2o_sys_tbl) +	/* Header + IOPs */
			((hba_count) * sizeof(struct i2o_sys_tbl_entry));

	BEGIN;
	
	if (sys_tbl) {
		kfree(sys_tbl);
	}

	sys_tbl = kmalloc(sys_tbl_len, GFP_ATOMIC | ADDR32);
	
	if (!sys_tbl) {
		PDEBUG(DBG_DEVICE, "SysTab Set failed. Out of memory.\n");	

		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | 
				UO_CMN | UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name,
				uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

		return -ENOMEM;
	}
	
	memset(sys_tbl, 0, sys_tbl_len);

	sys_tbl->num_entries = hba_count;
	sys_tbl->version = I2OVERSION;
	sys_tbl->change_ind = sys_tbl_ind++;

	for (pHba = hba_chain; pHba; pHba = pHba->next) {
		/* 
		 * Get updated Status Block so we have the latest information
		 */
		if (adpt_i2o_status_get(pHba)) {
			sys_tbl->num_entries--;

			continue; /* try next one */	
		}

		sys_tbl->iops[count].org_id = pHba->status_block->org_id;
		sys_tbl->iops[count].iop_id = pHba->unit + 2;
		sys_tbl->iops[count].seg_num = 0;
		sys_tbl->iops[count].i2o_version = 
			pHba->status_block->i2o_version;
		
		sys_tbl->iops[count].iop_state = pHba->status_block->iop_state;
		sys_tbl->iops[count].msg_type = pHba->status_block->msg_type;
		sys_tbl->iops[count].frame_size = 
			pHba->status_block->inbound_frame_size;
		
		sys_tbl->iops[count].last_changed = sys_tbl_ind - 1; 
		sys_tbl->iops[count].iop_capabilities = 
			pHba->status_block->iop_capabilities;
		
		sys_tbl->iops[count].inbound_low = 
			(U32)virt_to_bus(pHba->ToIopFifo);
		
		sys_tbl->iops[count].inbound_high = 
			(U32)((U64)virt_to_bus(pHba->ToIopFifo) >> 32);

		count++;
	}

	PDEBUG(DBG_DEVICE, "sys_tbl_len=%d in 32bit words\n", 
			(sys_tbl_len >>2));
	
	END;

	return 0;
}

/*
 * adpt_i2o_init_outbound_q()
 *
 * Description:
 * 	Initialize the outbound message queue. 
 *	i.e. empty both FIFOs. 	
 */  
S32 
adpt_i2o_init_outbound_q(adpt_hba *pHba)
{
	U32 		m;
	U32 		msg[8];
	U32 		timeout = jiffies + (TMOUT_INIT_OUTBOUND);
	U32		*ptr;
	U32 		outbound_frame;
	S32 		i;
	U8 		*status;
	dma_addr_t	status_bus_addr;

	BEGIN;
		
	do {
		rmb();

		m = readl(pHba->ToIopFifo);
	
		if (m != EMPTY_QUEUE) {
			break;
		}

		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "%s: Timeout waiting for message "
						"frame\n",
						pHba->name);
			
			ERROR_END;

			return -ETIMEDOUT;
		}
	} while (m == EMPTY_QUEUE);

	status = pci_alloc_consistent(pHba->pDev, 4, &status_bus_addr);
	
	if (status == NULL) {
		adpt_send_nop(pHba, m);

		PDEBUG(DBG_INIT, "%s: IOP reset failed - no free memory.\n",
					pHba->name);
	
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | 
					UO_CMN | UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

		return -ENOMEM;
	}
	
	memset(status, 0, 4);

	msg[0] = EIGHT_WORD_MSG_SIZE | SGL_OFFSET_6;
	msg[1] = I2O_CMD_OUTBOUND_INIT << 24 | HOST_TID << 12 | ADAPTER_TID;
	msg[2] = 0;
	msg[3] = 0x0106;
	msg[4] = I2O_PAGE_SIZE;
	msg[5] = (REPLY_FRAME_SIZE) << 16 | 0x80;
	msg[6] = 0xD0000004;
	msg[7] = status_bus_addr;

	memcpy_toio(pHba->base_addr_virt + m, msg, sizeof(msg));
	
	wmb();
	
	writel(m, pHba->ToIopFifo);
	
	wmb();

	/* 
	 * Wait for the reply status to come back
  	 */
 	do {
		if ((volatile U32) *status) {
			if ((volatile U32) *status != IN_PROGRESS) {	
				break;
			}
		}
		
		wmb();
		
		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "%s: Timeout Initializing\n", 
						pHba->name);
			
			pci_free_consistent(pHba->pDev, 4, status, 
					status_bus_addr);
			
			ERROR_END;

			return -ETIMEDOUT;
		}
	} while (1);

	/* 
	 * If the command was successful, fill the fifo with our reply
	 * message packets
	 */ 
	if ((volatile U32) (*status) != COMPLETE) {	
		pci_free_consistent(pHba->pDev, 4, status, status_bus_addr);

		ERROR_END;

		return -2;
	}

	pci_free_consistent(pHba->pDev, 4, status, status_bus_addr);

	if (pHba->FromIop_Pool != NULL) {
		kfree(pHba->FromIop_Pool);
	}

	pHba->FromIop_Pool = (U32 *)kmalloc(pHba->FromIop_Fifo_Size * 
			REPLY_FRAME_SIZE * 4, GFP_ATOMIC | ADDR32);
	
	if (!pHba->FromIop_Pool) {
		PDEBUG(DBG_INIT, "%s: Could not allocate FromIop_Pool\n",
					pHba->name);

		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | 
					UO_CMN | UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

		return -1;
	}
	
	memset(pHba->FromIop_Pool, 0 , (pHba->FromIop_Fifo_Size * 
			REPLY_FRAME_SIZE * 4));

	ptr = pHba->FromIop_Pool;
	
	m = 0; 
	
	for (i = 0; i < pHba->FromIop_Fifo_Size; i++) {
		outbound_frame = (U32) virt_to_bus(ptr);
	
		wmb();

		writel(outbound_frame, pHba->FromIopFifo);
				
		wmb();
		
		ptr += REPLY_FRAME_SIZE;
	}
	
	adpt_i2o_status_get(pHba);
	
	END;

	return 0;
}


/*
 * adpt_isr()
 *
 * Description:   
 *	Interrupt handler routine. 
 */  
void 
adpt_isr(S32 irq, void *dev_id, struct pt_regs *regs)
{
	adpt_cmnd	*pCmnd;
	adpt_hba	*pHba = NULL;
	adpt_target	*pTarget = NULL;
	U32 		m;
	U32 		status = 0;
	U32 		context;
	U32_64 		reply; 
        PI2O_SINGLE_REPLY_MESSAGE_FRAME reply_P;
	U32_64			flags = 0;
	
	pHba = (adpt_hba *) dev_id;
	adpt_lock(pHba, flags, 1);

	while (readl(pHba->irq_status) & I2O_INTERRUPT_PENDING_B) {
		PDEBUG(DBG_I2O_POST,"\nGot Interrupt");
		m = readl(pHba->FromIopFifo);
		
		if (m == EMPTY_QUEUE) {
			/* 
			 * Try twice then give up
			 */ 
			rmb();

			m = readl(pHba->FromIopFifo);
			
			if (m == EMPTY_QUEUE) {
				/* This really should not happen
				 * printk(KERN_ERR"%s: Could not get reply 
				 * frame\n",pHba->name);
				 */ 
				PERROR("Out of Reply Frame !\n");
				adpt_unlock(pHba, flags, 1);

				return;
			}
		}
		
		reply = (ulong) bus_to_virt(m);
		reply_P = (PI2O_SINGLE_REPLY_MESSAGE_FRAME) reply;

		if (readl(reply) & MSG_FAIL) {

			U32 	old_m = readl(reply + 28); 
			ulong 	msg;
			U32 	old_context;
			
			PDEBUG(DBG_ISR, "%s: Failed message\n", pHba->name);
			
			if (old_m >= 0x100000) {
				PDEBUG(DBG_ISR, "%s: Bad preserved MFA (%x)- "
						"dropping frame\n",
						pHba->name,old_m);
				
				writel(m, pHba->FromIopFifo);
				
				continue;
			}
			
			/* 
			 * Transaction context is 0 in failed reply frame
			 */ 
			msg = (ulong)(pHba->base_addr_virt + old_m);
			old_context = readl(msg + 12);
			writel(old_context, reply + 12);
			adpt_send_nop(pHba, old_m);
		} // if MSG_FAIL
	#ifdef ASA72XX_DEBUG	
         {
                U32 i=0;
		/* Lets see what is returning */
                for (i=0; i<29; i=i+4) {
		  PDEBUG(DBG_ISR,"%d) 0x%x!!\n", i, readl(reply + i));
                }
         }
        #endif
		
		/*
		 * Get the value of initiator context 
		 */  
		context = reply_P->StdMessageFrame.initiator_context;
/*
		PDEBUG(DBG_I2O_POST,"\nContext is 0x%x",context);
       if ( (context & 0x80000000) || 
            (context & 0x20000000) ){
		  PDEBUG(DBG_ERR_HNDL," context %x!!\n", context);
       } 
*/
		
		/* 
		 * DATA OUT 
		 * For Regular IOCTL or Private Message
		 */
		if (context & 0x80000000) { 
			/* 
			 * Post wait message  
			 */ 
			status = reply_P->DetailedStatusCode;
 
			/*
			 * Get the value of ReqStatus 
			 */  
			if (reply_P->ReqStatus) { 
				status = -(status & 0xffff);
			} else {
				status = I2O_POST_WAIT_OK;
			}
			
			/* Private Event Message */
			if (context & 0x20000000) {
				/*
				 * Private Event Message completion
				 */
				adpt_process_event_completion(pHba, reply);
			}

		} else if ((context & 0x22000000) && 
				(!(context & 0x80000000))) {
			/*
			 * Private Event Message completion
			 */
			adpt_process_event_completion(pHba, reply);
		} 
		if (context & POST_WAIT_CONTEXT/*0x1*/) {
			PDEBUG(DBG_I2O_POST,"FW ACKED our request");
			adpt_i2o_post_event_wake_up(pHba);
		}

		if (context == 0x0) { 
			/* 
			 * SCSI message
			 */ 
			pCmnd =  (adpt_cmnd *) reply_P->TransactionContext;
			if (readl(reply+12) == 0xffff0001) {
				PDEBUG(DBG_ERR_HNDL, "Session Recovery done\n");
				/*
				 * This is an aborted i2o msg.
				 * Just throw it away.
				 */  	
				goto next;
			}

			if (readl(reply+12) == 0xffff0000) {
				PDEBUG(DBG_ERR_HNDL, "Abort done\n");
				pHba->aborted_cmd_count--;
				/*
				 * This is an aborted i2o msg.
				 * Just throw it away.
				 */  	
				goto next;
			}

			if ((pCmnd != NULL) && (pCmnd->command != NULL)) {

				pTarget=pCmnd->target;
				if (pTarget == NULL) {
			       PDEBUG(DBG_ERR_HNDL, "pTarget is NULL!\n");
				}

				if ( (pCmnd->cmd_state & CMD_REQ_ABORT) || 
					  (pCmnd->cmd_state == 0)) {
					
					PDEBUG(DBG_ERR_HNDL, "pCmnd : %p %x!\n", pCmnd, 
												readl(reply + 8));
         
					if (adpt_remove_cmd_fr_queue(
						(adpt_cmnd **)&pTarget->PendingQueueHead,
						(adpt_cmnd **)&pTarget->PendingQueueTail,
						pCmnd) == 0) {
		
							PDEBUG(DBG_ERR_HNDL, "Removed Aborted pCmnd = %x !\n",
														pCmnd);

							pCmnd->cmd_state &= ~CMD_REQ_ACTIVE;
							pHba->active_cmds--;	
		
					}
					else {
						PDEBUG(DBG_ERR_HNDL, "%s: Failed to remove Aborted "
													"cmd=%x !\n",
													pHba->name, pCmnd);
			
					}
					/*
					 * This is an aborted command.
					 * Just throw it away.
					 */  	
					adpt_free_cmd(pHba, pCmnd);
					
					goto next;
				}
#ifdef ASA72XX_DEBUG
				{			
					U16 		detailed_status;
					detailed_status = readl(reply + 16) & 0xFFFF;	
      	  		if (detailed_status) {
						PDEBUG(DBG_ERR_HNDL, "pCmnd : %p %x!\n",
													pCmnd, detailed_status);
					}
				}
#endif
				/* Debug kernel oops here when rebooting */
				if (pTarget == NULL) {

					PDEBUG(DBG_ERR_HNDL,"Invalid interrupt given!!\n");
					/*
					 * attempt to recover from a bogus
					 *  command; invalid or shared interrupt,
					 *  throw away
					 */
				}


				if (adpt_remove_cmd_fr_queue(
				   (adpt_cmnd **)&pTarget->PendingQueueHead,
		       		   (adpt_cmnd **)&pTarget->PendingQueueTail,
				    pCmnd) == 0) {
					pCmnd->cmd_state &= ~CMD_REQ_ACTIVE;

					pHba->active_cmds--;	
				}

				adpt_insert_cmd_to_queue(
						(adpt_cmnd **)&pHba->DoneQueueHead,
						(adpt_cmnd **)&pHba->DoneQueueTail,
						pCmnd); 

				pCmnd->cmd_state |= CMD_REQ_DONE;

				pCmnd = pHba->DoneQueueHead;
	
				pHba->DoneQueueHead = NULL;
				pHba->DoneQueueTail = NULL;
			
				if (pCmnd->command->serial_number != 0) { 
					/* 
					 * If not Timed Out, process
					 * SCSI command completion.  
					 */  
					adpt_done(pHba, reply, pCmnd);
				}
			}
		}

next:		
		writel(m, pHba->FromIopFifo);

		wmb();
	}
	adpt_unlock(pHba, flags, 1);

}

/*
 * adpt_send_nop()
 *
 * Description:
 *	Send the "UtilNOP" message to IOP. Upon receiving the message,
 *	the IOP return the message frame to the free list.	    
 */ 
S32 
adpt_send_nop(adpt_hba *pHba, U32 m)
{
	U32 	msg[3];
	U32 	timeout;

	timeout = jiffies + (5 * HZ);

	while (m == EMPTY_QUEUE) {	
		wmb();
	
		m = readl(pHba->ToIopFifo);
	
		if (m != EMPTY_QUEUE) {
			break;
		}
			
		if (time_after(jiffies, timeout)) {
			PDEBUG(DBG_TIMEOUT, "%s: Timeout waiting for "
						"message frame!\n",
						pHba->name);
				
			ERROR_END;
				
			return 2;
		}
	}
	
	msg[0] = THREE_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = I2O_CMD_UTIL_NOP << 24 | HOST_TID << 12 | 0;
	msg[2] = 0;
		
	memcpy_toio(pHba->base_addr_virt + m, msg, sizeof(msg));

	wmb();
	
	writel(m, pHba->ToIopFifo);
	
	wmb();
	
	return 0;
}


/*
 * adpt_process_lct_async_msg()
 * 
 * Description:
 * 	This routine is scheduled during interrupt context and to be run 
 *	when we are not in interrupt mode so that we can sleep, schedule, etc.
 * 	Perform rescan and other necessary session recovery actions after
 *	receiving I2O_EVENT_DO_LCT_NOTIFY by the FW. 		 	    
 */
static void
adpt_process_lct_async_msg(void *data)
{
	adpt_hba	*pHba; 

	BEGIN;

	pHba = (adpt_hba *) data;

        pHba->state &=~ADPT_STATE_LCT_GET;
        adpt_rescan(pHba, FALSE);
        pHba->state &= ~ADPT_STATE_RESCAN_IN_PROGRESS;

	if(pHba->wakeup_flag){
#ifdef __VMKERNEL_MODULE__
	   vmk_thread_wakeup(&pHba->lct_processing_complete);
#else
		wake_up(&pHba->lct_processing_complete);
#endif //__VMKERNEL_MODULE__
		pHba->wakeup_flag=0;
	}

	END;
}

/*
 * adpt_process_event_completion()
 * 
 * Description:
 * 	Callback function to process private event completion.
 */
static void
adpt_process_event_completion(adpt_hba *pHba, U32_64 reply)
{
	U32		event_indicator = 0;
	U32		changed_event_mask = 0;
	U8		linkstatus_byte = 0;
	U8		event_code;
	U8		req_status;
	
	BEGIN;

	/*
	 * See what event code is returning.
	 */  
	event_code = (readl(reply + 8) >> 24) & 0x0F;

	PDEBUG(DBG_EVENT, "Event Type = 0x%x \n", event_code);

	switch (event_code) {
		case I2O_PRVT_EVT_REGISTERATION:
			req_status = (readl(reply + 16) >> 24) & 0xFF;
			
			PDEBUG(DBG_EVENT, "Event Registration is %s !\n",
					req_status == 0 ? "SUCCESSFUL " :
				       "FAILED ");	

			changed_event_mask = readl(reply + 24) & 0xFFFF;
			
			PDEBUG(DBG_EVENT, "Current Event Mask = 0x%x !\n",
					changed_event_mask);

			event_registered = 1;

			break;

		case I2O_PRVT_EVT_NOTIFICATION:
			event_indicator = readl(reply + 20) & 0xFFFF;
			
			PDEBUG(DBG_EVENT, "Event Indicator = 0x%x !\n",
					event_indicator);

			if (event_indicator & 
				BIT(I2O_EVENT_LINKSTATE_CHANGE)) {
				/*
				 * Link State Event has occured.
				 * Check the Status Btye.
				 */
				linkstatus_byte = readl(reply + 24) & 
					LINK_STATUS_MASK;

				if (linkstatus_byte & BIT(0)) {
					/*
					 * We got a LINK UP Event.
					 */  
					if (pHba->state & 
						ADPT_STATE_LINK_DOWN) {
						pHba->state &= 
							~ADPT_STATE_LINK_DOWN;
					}

					pHba->state |= ADPT_STATE_LINK_UP;

					PDEBUG(DBG_EVENT, "Link is UP !\n");
				} else {
					/*
					 * We got a LINK DOWN Event.
					 */  
					if (pHba->state & ADPT_STATE_LINK_UP) {
						pHba->state &= 
							~ADPT_STATE_LINK_UP;
					}

					pHba->state |= ADPT_STATE_LINK_DOWN;

					PDEBUG(DBG_EVENT, "Link is DOWN !\n");
				}
			} else if (event_indicator & 
					BIT(I2O_EVENT_DO_LCT_NOTIFY)) {
                 pHba->state |=ADPT_STATE_LCT_GET;
					PDEBUG(DBG_EVENT, "LCT GET !\n");
			} else if (event_indicator & 
					BIT(I2O_EVENT_NEWERROR_TO_LOG)) {
				pHba->fw_error_code = readl(reply + 28); 

				PDEBUG(DBG_EVENT, "FW Error Code = 0x%x !\n",
						pHba->fw_error_code);
			} else {
				PDEBUG(DBG_EVENT, "Unregister Async Event !\n");
			}	

			break;

		case I2O_PRVT_SYS_SHUTDOWN:
			PDEBUG(DBG_EVENT, "Issued Logout to the FW !\n");
			
			break;

		default:
			PDEBUG(DBG_EVENT, "Unknown Event Code !!\n");

			break;
	}
	
	/*
	 * Send Async. Event Notification to the FW for future async. events
	 * notification.
	 */   	
	if ((event_code == I2O_PRVT_EVT_NOTIFICATION) && 
		(!(pHba->state & ADPT_STATE_SHUTDOWN))) {
		PDEBUG(DBG_EVENT, "Repost Async Notify Message !\n");

		adpt_async_send_notification(pHba, NORMAL_REPLY);
	}
	
	END;

	return;
}


/*
 * adpt_done()
 *
 * Description:
 *	Process completed commands in the done queue.
 */
static void
adpt_done(adpt_hba *pHba, U32_64 reply, adpt_cmnd *pCmnd)
{
	adpt_cmnd	*pCmnd_curr;
	adpt_cmnd	*pCmnd_next;
	Scsi_Cmnd	*cmd;
	
	BEGIN;

	pCmnd_curr = pCmnd;
	
	while (pCmnd_curr) {
		pCmnd_next = pCmnd_curr->next;

		if (adpt_i2o_to_scsi(reply, pCmnd_curr) == 0) {
			cmd = pCmnd_curr->command;

			/*
			 * Calls the OS completion done function.
			 */  
			if (cmd->scsi_done != NULL) {
				cmd->scsi_done(cmd);
			}
		}
		adpt_free_cmd(pHba, pCmnd_curr);

		pCmnd_curr = pCmnd_next;
	}	

	END;
}	


/*
 * adpt_scsi_to_i2o()
 *
 * Description:
 *	Convert scsi to I2O message. Setup the private message structure
 * 	for ScsiScbExec.	  
 */ 
S32 
adpt_scsi_to_i2o(adpt_hba *pHba, adpt_cmnd *pCmnd, adpt_device *pDev)
{
	U32			target_context = pDev->target_context;
	Scsi_Cmnd		*cmd;
	U32 			msg[MAX_MESSAGE_SIZE/4];
	U32 			*mptr;
	U32 			*lenptr; 
	U32 			*es_bufcntxt_ptr;
	U32 			len;
	U32 			msg_size;
	S32 			rcode;
	S32			dma_dir;
	S32 			scsidir;
	S32 			i;

	BEGIN;

	cmd = pCmnd->command;

	memset(msg, 0, sizeof(msg));
	
	len = cmd->request_bufflen;
	
	scsidir = 0x00000000;	/*  DATA NO XFER */

	if (len) {
		/*
		 * Set SCBFlags to indicate if data is being transferred
		 * in or out, or no data transfer
		 */
		switch (cmd->sc_data_direction) {
			case SCSI_DATA_READ:
				/* DATA IN  (iop<--dev) */
				scsidir = 0x40000000;	

				if (cmd->cmnd[0] == 0x0) {
					/*  DATA NO XFER for TUR */
					scsidir = 0x00000000;	
				}
					
				break;

			case SCSI_DATA_WRITE:
				/* DATA OUT (iop-->dev) */
				scsidir = 0x80000000;
				break;

			case SCSI_DATA_NONE:
				break;

			case SCSI_DATA_UNKNOWN:
				/* Assume DATA IN */
				scsidir  = 0x40000000;	
				break;

			default:
				PDEBUG(DBG_SCSI, "Unsupported scsi opcode ="
						" 0x%x", 
						cmd->cmnd[0]);
				
				cmd->result = (DID_OK << 16) | 
					(INITIATOR_ERROR << 8);

			      	cmd->scsi_done(cmd);	

				return 0;
		}		
	}
	
	/* 
	 * msg[0] will be set later because we still don't know the 
	 * MessageSize. To find out the MessageSize, we need to know 
	 * if we will be having data or not data transfer and also
	 * how many SGL elements if we are using them.   
	 */ 
	
	msg[1] = ((0xff << 24) | (HOST_TID << 12) | pDev->tid);
	msg[2] = 0;
	msg[3] = (U32) pCmnd;	/* We want the SCSI control block back */
	
	/* 
	 * Our cards use the transaction context as the tag for queueing
	 * Adaptec/DPT Private stuff 
	 */ 
	msg[4] = I2O_CMD_SCSI_EXEC | (DPT_ORGANIZATION_ID << 16);
	msg[5] = pDev->tid;

	/* Direction, Allow disconnect | sense data | simple queue , CDBLen
	 * I2O_SCB_FLAG_ENABLE_DISCONNECT | 
	 * I2O_SCB_FLAG_SIMPLE_QUEUE_TAG | 
	 * I2O_SCB_FLAG_SENSE_DATA_IN_REPLY_MESSAGE;
	 */ 
	if ((cmd->device->tagged_supported) || 
			(cmd->device->type == TYPE_DISK)) {
		msg[6] = scsidir | 0x20a00000 | cmd->cmd_len;
	} else {
		/* 
		 * For non direct access devices, disable tagged-queuing and 
		 * disconnect feature.
		 */   
		msg[6] = scsidir | 0x00200000 | cmd->cmd_len;
	}

	mptr = msg + 7;

	/* 
	 * Write SCSI command into the message - always 16 byte block 
	 */ 
	memset(mptr, 0, 16);
	
	memcpy(mptr, cmd->cmnd, cmd->cmd_len);
	
	mptr += 4;
	
	lenptr = mptr++;	/* Remember me - fill in when we know */

	/*
	 * msg_size is set to 16 if we have only 1 SGL element. 
	 */   
	msg_size = 16;		
	
	es_bufcntxt_ptr = mptr++;

 	/* 
	 * Fill in the buffer context with target context value
	 */
	//*mptr++ = pI2Odevice->lct_data.target_context;
	*mptr++ = target_context;
	
	/* Now fill in the SGList and command */
	if (cmd->use_sg) {
		struct scatterlist 	*sg;
		U32			num_sg;
		
		pCmnd->xfer_len = 0;

		sg = (struct scatterlist *) cmd->request_buffer;
		
		dma_dir = scsi_to_pci_dma_dir(cmd->sc_data_direction);
#ifdef __VMKERNEL_MODULE__
		num_sg = cmd->use_sg;
#else
		num_sg = pci_map_sg(pHba->pDev, sg, cmd->use_sg, dma_dir);
#endif //__VMKERNEL_MODULE__

		for (i = 0 ; i < num_sg; i++, sg++) {
			/*
			 * SGL element consists of addr (4 bytes) and 
			 * length (4 bytes) for 32-bit platform.
			 */   
			*mptr++ = sg_dma_address(sg);
			*mptr++ = sg_dma_len(sg);
			pCmnd->xfer_len += sg_dma_len(sg);
		}
		
		/* 
		 * Make this an end of list. 
		 * Setting the last 4 bit to value 8 will mark it as end of
		 * the list.
		 */
		mptr[-1] = 0x80000000 | (sg-1)->length;
		
		msg_size = mptr - msg;
		
		*lenptr = pCmnd->xfer_len;
		
		/* 
		 * flags field and byte count 
		 */
		*es_bufcntxt_ptr = ((0xCD << 24) | (cmd->use_sg * 8));
		
#ifdef ASA72XX_DEBUG		
		if ((pCmnd->xfer_len != cmd->underflow) & (cmd->underflow)) {
			PDEBUG(DBG_SCSI, "Cmd len %08x Cmd underflow %08x\n",
					pCmnd->xfer_len, 
					cmd->underflow);
		}
#endif
	} else {
		dma_addr_t	bus_addr;
		
		*lenptr = pCmnd->xfer_len = cmd->request_bufflen;
	
		dma_dir = scsi_to_pci_dma_dir(cmd->sc_data_direction);

		if (pCmnd->xfer_len == 0) {
			msg_size = 14;
			
			/* flags field and byte count */
			*es_bufcntxt_ptr = ((0xCD << 24) | 0 );	
		} else {
#ifdef __VMKERNEL_MODULE__
		   /* We already have the machine address */
		   bus_addr = cmd->request_bufferMA;
		   vmk_verify_memory_for_io(bus_addr, cmd->request_bufflen);
#else
			bus_addr = pci_map_single(pHba->pDev, 
						cmd->request_buffer,
						cmd->request_bufflen,
						dma_dir);
#endif //__VMKERNEL_MODULE__
			pCmnd->bus_addr = bus_addr;

			*mptr++ = bus_addr;

			*mptr++ = 0x80000000 | cmd->request_bufflen;
			
			/* flags field and byte count */
			*es_bufcntxt_ptr = ((0xCD << 24) | (1 * 8));
		}
	}

	msg[0] = (msg_size << 16) | 
			((msg_size > 14) ? SGL_OFFSET_14 : SGL_OFFSET_0);
	
	/* Send it on it's way */
	rcode = adpt_i2o_post_this(pHba, msg, msg_size << 2);

	END;

	return rcode;
}


/*
 * adpt_initiate_session_recovery()
 * 
 * Description:
 * 	Issue a session recovery to the particular target    
 */
static S32
adpt_initiate_session_recovery(adpt_hba *pHba, adpt_target *pTarget)
{
	U32	msg[MAX_MESSAGE_SIZE/4];
	U32	msg_size;
	S32	rcode = 0;

	BEGIN;

	msg[0] = I2O_MESSAGE_SIZE(6) | SGL_OFFSET_0;
	msg[1] = (0xff << 24 | HOST_TID << 12 | pTarget->tid);
	msg[2] = 0;		/* Initiator Context */
	msg[3] = (U32) 0xffff0001;	/* Target Context */
	msg[4] = (DPT_ORGANIZATION_ID << 16) | I2O_PRVT_START_SESSION_RECOVERY;
	msg[5] = 0x00010000;	/* Interpret bit */

	/* Message size in U32 words */
	msg_size = 6;
			
	/* Message size in U8 bytes */
	msg_size = 6 << 2;

	rcode = adpt_i2o_post_this(pHba, msg, msg_size);

	END;

	return rcode;
}

/*
 * adpt_abort()
 *
 * Description:
 *	Aborting an outstanding command   
 *	This abort eh_handler is called by the OS.	
 */  
S32 
adpt_abort(Scsi_Cmnd *cmd)
{
	adpt_hba		*pHba = NULL;	
	adpt_target		*pTarget = NULL;
	adpt_device		*pDev = NULL;	
	adpt_cmnd		*pCmnd = NULL;
	U32 			msg[5];
        U32_64                  flags=0;
	S32			rcode;		
	U8			cmd_found = FALSE;
	
	BEGIN;
	
	if (cmd->serial_number == 0) {
		ERROR_END;

		return DRV_FAILED;
	}

	/* 
	 * If the serial number information is not correct, it is
	 * a different command.  
	 */
	if (cmd->serial_number != cmd->serial_number_at_timeout) {
		PDEBUG(DBG_ERR_HNDL, "%s: Abort called with finished cmd\n",
					pHba->name);
		
		cmd->result = DID_ABORT << 16;
		
		goto abort_success;
	}

        cmd->result = DID_ABORT << 16;
	pHba = (adpt_hba *) cmd->host->hostdata[0];

//	pDev = (void *) (cmd->device->hostdata); 
	pDev = adpt_find_device(pHba, cmd->channel, cmd->target,
				cmd->lun);
	
	pTarget = adpt_get_target(pHba, cmd->channel, cmd->target);

	if (pTarget == NULL) {

		PDEBUG(DBG_ERR_HNDL, "%s: Unable to abort: "
				"No Target in cmnd \n",
				pHba->name);

		ERROR_END;

		return DRV_FAILED;
	}

		/*
		 * Find the cmd in the Pending Queue.
		 */  
		if ((pCmnd = adpt_find_cmd_in_queue(
			(adpt_cmnd **)&pTarget->PendingQueueHead,
			(adpt_cmnd **)&pTarget->PendingQueueTail,
		       	cmd)) == NULL) {

			PDEBUG(DBG_ERR_HNDL, "Could not find cmd in the "
					"Pending Queue !!\n");
		}
        else {

		adpt_lock(pHba, flags, 1);
		PDEBUG(DBG_ERR_HNDL, "%s: Aborting cmd = 0x%02x with serial # %ld "
			"for Device with TID = %d\n", 
			pHba->name, cmd->cmnd[0], cmd->serial_number,
			pDev->tid);

		pCmnd->cmd_state |= CMD_REQ_ABORT;

		memset(msg, 0, sizeof(msg));

		msg[0] = FIVE_WORD_MSG_SIZE | SGL_OFFSET_0;
		msg[1] = I2O_CMD_SCSI_ABORT << 24 | HOST_TID << 12 | pDev->tid;
		msg[2] = 0;		/* Initiator context */
		msg[3] = 0xffff0000;	/* Transaction context */
		msg[4] = (U32) pCmnd;	/* Transaction context to be aborted */
	
		spin_unlock_irq(&io_request_lock);
	
			rcode = adpt_i2o_post_wait(pHba, msg, sizeof(msg), TMOUT_ABORT);

		spin_lock_irq(&io_request_lock);
		
		if (rcode != 0) {
			PDEBUG(DBG_ERR_HNDL, "%s: Failed to abort cmd=%ld !\n",
					pHba->name, cmd->serial_number);
	
			PDEBUG(DBG_ERR_HNDL, "Status returned is 0x%x !!\n", rcode);

			ERROR_END;
			adpt_unlock(pHba, flags, 1);
			return DRV_FAILED;
		} else {
			pHba->aborted_cmd_count++;

			if (adpt_remove_cmd_fr_queue(
				(adpt_cmnd **)&pTarget->PendingQueueHead,
				(adpt_cmnd **)&pTarget->PendingQueueTail,
				pCmnd) == 0) {
				PDEBUG(DBG_ERR_HNDL, "Removed Aborted Cmd = %ld !\n",
					cmd->serial_number);
			
				pCmnd->cmd_state &= ~CMD_REQ_ACTIVE;

				pHba->active_cmds--;	
			}
			adpt_unlock(pHba, flags, 1);
		}

		goto abort_success;
        }
	
	/*
	 * Check if the command is still in the SendQueue.
	 * If yes, abort it and return to OS.
	 */   
	if ((pHba->SendQueueHead != NULL) && (pHba->SendQueueTail != NULL)) {
		if ((pCmnd = adpt_find_cmd_in_queue(
				(adpt_cmnd **)&pHba->SendQueueHead,
				(adpt_cmnd **)&pHba->SendQueueTail,
		       		cmd)) != NULL) {
			PDEBUG(DBG_ERR_HNDL, "Found the cmd in the "
						"SendQueue !\n");

			if (adpt_remove_cmd_fr_queue(
					(adpt_cmnd **)&pHba->SendQueueHead,
		       			(adpt_cmnd **)&pHba->SendQueueTail,
				        pCmnd) == 0) {		
				cmd_found = TRUE;
			}
		}
	}

	/*
	 * Check if the command is already in the DoneQueue.
	 * If yes, discard it and return to OS.
	 */   
	if ((pHba->DoneQueueHead != NULL) && (pHba->DoneQueueTail != NULL)) {
		if ((pCmnd = adpt_find_cmd_in_queue(
				(adpt_cmnd **)&pHba->DoneQueueHead,
				(adpt_cmnd **)&pHba->DoneQueueTail,
		       		cmd)) != NULL) {
			PDEBUG(DBG_ERR_HNDL, "Found the cmd in the "
						"DoneQueue !\n");

			if (adpt_remove_cmd_fr_queue(
					(adpt_cmnd **)&pHba->DoneQueueHead,
		       			(adpt_cmnd **)&pHba->DoneQueueTail,
				        pCmnd) == 0) {		
				cmd_found = TRUE;
			}
		}
	}	

	if (cmd_found == TRUE) {
		PDEBUG(DBG_ERR_HNDL, "%s: Success in aborting cmd=%ld !\n",
					pHba->name, 
					cmd->serial_number);
                                                                        
		pCmnd->cmd_state &= ~CMD_REQ_ACTIVE;

		pHba->active_cmds--;	
				
		cmd->result = DID_ABORT << 16;

		if (cmd->done != NULL) {
			cmd->scsi_done(cmd);
		}
			
		adpt_free_cmd(pHba, pCmnd);

		goto abort_success;
	}
abort_success:	
	
	END;

	return  DRV_SUCCESS;
}


/* 
 * adpt_device_reset()
 *
 * Description:   
 *	Issue a target reset to the device that timed-out or failed.
 *	This device reset eh_handler is called by the OS.	
 */	
S32 
adpt_device_reset(Scsi_Cmnd *cmd)
{
	adpt_hba 		*pHba = NULL;
	adpt_target 		*pTarget = NULL; 
	adpt_device 		*pDev = NULL; 
	U32 			msg[4];
	S32			rcode;

	BEGIN;
	
	pHba = (void *) cmd->host->hostdata[0];

	//pDev = (void *) cmd->device->hostdata;
	pDev = adpt_find_device(pHba, cmd->channel, cmd->target,
				cmd->lun);

	if ((pDev->state & ADPT_DEV_OFFLINE)  ||
			(pDev->state & ADPT_DEV_ABSENT)) {
		PDEBUG(DBG_ERR_HNDL, "Device is currently offline !\n");
	
		return DRV_FAILED;
	}
		
	PDEBUG(DBG_ERR_HNDL, "Reset device for TID = %d !\n", 
			pDev->parent_tid);

	pTarget = adpt_get_target(pHba, cmd->channel, cmd->target);

	if (pTarget == NULL) {
		PDEBUG(DBG_ERR_HNDL, "No Target found for TID = %d !\n",
				pDev->parent_tid);

		ERROR_END;
		
		return DRV_FAILED;
	}

	memset(msg, 0, sizeof(msg));

	msg[0] = FOUR_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = (I2O_CMD_SCSI_DEVICE_RESET << 24 | HOST_TID << 12 | 
			pTarget->tid);
	msg[2] = 0;		/* Initiator Context */
	msg[3] = 0;		/* Transaction Context */

	pTarget->state |= ADPT_TARGET_RESET;

	spin_unlock_irq(&io_request_lock);
	
	rcode = adpt_i2o_post_wait(pHba, (void *) msg, sizeof(msg), 
			TMOUT_DEV_RESET);

	spin_lock_irq(&io_request_lock);

	if (rcode != 0) {
		PDEBUG(DBG_ERR_HNDL, "Reset failed for device with "
				"TID = %d !! \n", pTarget->tid);

		ERROR_END;
		
		return DRV_FAILED;		
	} 
		
	pTarget->state &= ~ADPT_TARGET_RESET;
	
	PDEBUG(DBG_ERR_HNDL, "Reset was successful for Target with "
			"TID = %d !\n", pTarget->tid);
	
	END;
	
	return DRV_SUCCESS;
}


/*
 * adpt_initiate_bus_reset()
 *
 * Description:
 * 	Perform Bus Reset for a specific channel.
 *	Bus Reset is simply a Device Reset for all the targets found on 
 *	the specific channel.	 	
 */ 	    
static S32
adpt_initiate_bus_reset(adpt_hba *pHba, U8 channel_no)
{
	adpt_target	*pTarget = NULL;
	U32		msg[(I2O_HBR_MAX_TID_COUNT/2) + 5];
	U32		tid_count = 0;
	S32		rcode = 0;
	U8		scsi_id = 0;
	U8		n;
	
	BEGIN;

	pHba->state |= ADPT_STATE_BUS_RESET;

	memset(msg, 0, sizeof(msg));

	n = 5;

	for (scsi_id = 0; scsi_id < pHba->top_scsi_id; scsi_id++) {
		if ((pTarget = pHba->channel[channel_no].target[scsi_id]) !=
					NULL) {
			if ((tid_count % 2) == 1) {
				msg[n] = (pTarget->tid << 16) | msg[n];

				n++;
			} else {
				msg[n] = pTarget->tid;
			}

			tid_count++;
		}

	}
		
	PDEBUG(DBG_ERR_HNDL, "HBA Bus Reset for %d Target(s) on "
			"Channel %d !\n",
			tid_count,
			channel_no);	

	/*
	 * Fill in the message once we know the TID Count and have the 
	 * ResetTID filled.
	 */  		
	msg[0] = FOUR_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = (I2O_HBA_BUS_RESET << 24 | HOST_TID << 12);
	msg[2] = 0; 		/* Initiator Context */
	msg[3] = 0;		/* Transaction Context */
	msg[4] = tid_count;	/* TID Count */
	
	spin_unlock_irq(&io_request_lock);
	
	rcode = adpt_i2o_post_wait(pHba, (void *) msg, sizeof(msg), 
				TMOUT_DEV_RESET);

	spin_lock_irq(&io_request_lock);

	if (rcode != 0) {
		PDEBUG(DBG_ERR_HNDL, "Failed to perform HBA Bus Reset on "
				"Channel %d !\n ",
				channel_no);
	} else {
		PDEBUG(DBG_ERR_HNDL, "HBA Bus Reset is successful on "
				"Channel %d !\n",
				channel_no);
	}

	pHba->state &= ~ADPT_STATE_BUS_RESET;

	END;

	return 0;
}


/*
 * adpt_bus_reset()
 *
 * Description:  
 *   	Perform BUS Reset for the corresponding channel.
 *	This bus reset eh_handler is called by the OS.   	
 */
S32 
adpt_bus_reset(Scsi_Cmnd *cmd)
{
	adpt_hba 	*pHba;
	adpt_device	*pDev;
	U8		channel_no;	
	S32		status = 0;
	
	BEGIN;
	
	/*
	 * The FW doesn't support BUS Reset. What it does for BUS Reset is 
	 * simply performs a Device Reset for each target found on the 
	 * corresponding channel.
	 */    
	pHba = (adpt_hba *) cmd->host->hostdata[0];

//	pDev = (adpt_device *) (cmd->device->hostdata);
	pDev = adpt_find_device(pHba, cmd->channel, cmd->target,
				cmd->lun);
	
	if ((pDev->state & ADPT_DEV_OFFLINE)  ||
			(pDev->state & ADPT_DEV_ABSENT)) {
		PDEBUG(DBG_ERR_HNDL, "Device is currently offline in %s !\n",
			__FUNCTION__);
	
		return DRV_FAILED;
	} 
		
	channel_no = pDev->scsi_channel;

	status = adpt_initiate_bus_reset(pHba, channel_no);
	
	if (status != 0) {
		PDEBUG(DBG_ERR_HNDL, "Unable to initiate Bus Reset for "
				"SCSI Host %d Channel %d !\n", 
				pHba->host->host_no, channel_no);

		ERROR_END;

		return DRV_FAILED;
	}

	END;

	return DRV_SUCCESS;
}


/*
 * adpt_release()
 *
 * Description:   
 * 	When adpt_release is called (from scsi_unregister_host, etc.), 
 * 	there are no longer any scsi command or device structures for this host
 * 	in the upper-level scsi code - cannot send any commands up
 *
 * 	scsi_unregister will be called AFTER we return. 
 */
S32 
adpt_release(struct Scsi_Host *host)
{
	adpt_hba  	*pHba; 
	U32		event_mask; 	
	
	BEGIN;

	pHba = (adpt_hba *) host->hostdata[0];

	pHba->state |= ADPT_STATE_SHUTDOWN;

        /*
         *  Remove the timer so timer tick stops
         */
        del_timer_sync(&pHba->hba_timer);

	/*
	 * Set Event Mask to 0 to have FW returned the pending Async Event 
	 * Notification message.
	 */   
	event_mask = 0;

	if (adpt_async_register_event(pHba, event_mask) != 0) {
		PDEBUG(DBG_UNLOAD, "Failed to unregister Async Event !\n");
	}

	unregister_reboot_notifier(&adpt_shutdown_notifier);

	unregister_chrdev(ASA72XX_I2O_MAJOR, ASA72XX_DRIVER); 

	adpt_destroy_cmd_pool(pHba);

	adpt_i2o_delete_hba(pHba);

        if (reg_drv_done == 1)
        {
	  pci_unregister_driver(&asa72xx_pci_driver);
	  PDEBUG(DBG_UNLOAD, "Unregister device !\n");
          reg_drv_done=0;
        }
#ifdef __VMKERNEL_MODULE__
	destroy_waitqueue_head(&(pHba->interrupt_triggered_event));
	destroy_waitqueue_head(&(pHba->wake_up_worker));
	destroy_waitqueue_head(&(pHba->wake_up_sr));
	destroy_waitqueue_head(&(pHba->lct_processing_complete));
	spin_lock_destroy(&pHba->protect_io_cmd);
#endif //__VMKERNEL_MODULE__
	END;

	return 0;
}


/*
 * adpt_select_queue_depths()
 *
 * Description:  
 *	Set the queue depth for each device that supports tagged
 *	queueing
 */	 
static void 
adpt_select_queue_depths(struct Scsi_Host *host, Scsi_Device *devicelist)
{
	Scsi_Device 	*device;   /* scsi layer per device information */
	adpt_hba	*pHba;

	BEGIN;
	
	pHba = (adpt_hba *) host->hostdata[0];

	for (device = devicelist; device != NULL; device = device->next) {
		if (device->host != host) {
			continue;
		}

		if ((host->can_queue) && (device->tagged_supported)) {
			device->queue_depth = pSetup->device_q_depth;

			device->tagged_queue = 1;
		} else {
			device->queue_depth = 2;
		}
	}

	END;
}


/*
 * adpt_queue_done_command()
 * 
 * Description :
 * 	Schedule a command to return for OS completion.
 */ 	    
static inline void
adpt_queue_done_command(Scsi_Cmnd *cmd, void (*scsi_done) (Scsi_Cmnd *))
{
	struct adpt_queue_info	*qinfop;
	struct timer_list	*ptimerlist;

	BEGIN;

	/*
	 * We are currently not using SCp for anything.  
	 * Besides, we are done with this command anyway.
	 */
	qinfop = (struct adpt_queue_info *) &cmd->SCp;
	
	qinfop->cmd = cmd;
	
	qinfop->scsi_done = scsi_done;
	
	ptimerlist = &qinfop->timerlist;
	
	PDEBUG(DBG_SCSI, "Cmd returned:cmd = %x, bus = %d, target = %d , " \
	                 "lun = %d, status = %d\n",
			 cmd->cmnd[0], cmd->channel, cmd->target, cmd->lun,
			 cmd->result >> 16);

	/*
	 * Trying to buy us some times before returning to the OS.
	 */   
	switch (cmd->result >> 16) {
		case DID_ERROR:
			ptimerlist->expires = jiffies;

			break;

		case DID_TIME_OUT:
		case DID_NO_CONNECT:
			ptimerlist->expires = jiffies + 10;

                     break;

		case DID_BUS_BUSY:	
			ptimerlist->expires = jiffies;

			break;

		case DID_SOFT_ERROR: 
			/* 
			 * We enter in only when firmware hasn't released the 
			 * resources. Let the OS try after 30 sec and hopefully 
			 * the FW should have freed the resources 
			 */
			ptimerlist->expires = jiffies + (30 * HZ);
	   		cmd->result = (DID_BUS_BUSY << 16);
			break;

		case DID_BUS_LINKDOWN: 
		default:
			ptimerlist->expires = jiffies+(TMOUT_LINKDOWN *HZ);

			break;
	}		
	if(cmd->retries == (cmd->allowed-1)){
		PDEBUG(DBG_ERR_HNDL, "\nMultiple Command Retries by the OS, Retry count = %d",cmd->retries);
	}
		
	ptimerlist->data = (U32_64) qinfop;
	
	ptimerlist->function = adpt_process_done_queue;
	
	add_timer(ptimerlist);

	END;
}


/*
 * adpt_process_done_queue()
 * 
 * Description:
 *	Call the OS command completion routine.
 */	    
static inline void
adpt_process_done_queue(U32_64 arg)
{
	adpt_hba		*pHba = NULL;
	struct adpt_queue_info	*qinfop;
	U32_64			flags = 0;

	BEGIN;

	qinfop = (struct adpt_queue_info *) arg;

	pHba = (adpt_hba *) qinfop->cmd->host->hostdata[0];

	PDEBUG(DBG_ERR_HNDL, "cmd %x !\n",qinfop->cmd);

	adpt_lock(pHba, flags, 1);
	
	qinfop->scsi_done(qinfop->cmd);
	
	adpt_unlock(pHba, flags, 1);

	END;
}

/*
 * 
 * adpt_cmd_timeout()
 * 
 * Description:
 * 	Internal timeout watchdog routine.
 *	Send an Abort to the specific command. 	
 */
static void 
adpt_cmd_timeout(U32_64 arg)
{
	adpt_hba	*pHba = NULL;
	adpt_target	*pTarget = NULL;
	adpt_device	*pDev = NULL;	
	adpt_cmnd	*pCmnd;
	Scsi_Cmnd	*cmd;	
	U32 		msg[5];
	S32		rcode = 0;

	BEGIN;

	pCmnd = (adpt_cmnd *) arg;
	PDEBUG(DBG_SCSI, "Command Timed out %x\n", pCmnd);

	if (pCmnd == NULL) {
		PDEBUG(DBG_ERR_HNDL, "pCmnd is NULL!\n");
            if (pCmnd->cmd_state & CMD_REQ_DONE) {
		PDEBUG(DBG_ERR_HNDL, "pCmnd Completed !\n");

		ERROR_END;
		
                goto finish_cmd;
             }
	}

	if (pCmnd->cmd_timer.function != NULL) {

		pCmnd->cmd_timer.data = (U32_64) NULL;

		pCmnd->cmd_timer.function = NULL;

	}


	cmd = pCmnd->command;

	pHba = (adpt_hba *) cmd->host->hostdata[0];

	//pDev = (void *) (cmd->device->hostdata); 
	pDev = adpt_find_device(pHba, cmd->channel, cmd->target,
				cmd->lun);

	pTarget = adpt_get_target(pHba, cmd->channel, cmd->target);

	if (pTarget == NULL) {
		PDEBUG(DBG_ERR_HNDL, "%s: Unable to abort: No Target found "
				"for the cmnd\n",
				pHba->name);

		ERROR_END;

                goto finish_cmd;
	}

	/*
	 * Just to be sure if the command is still in the Pending Queue.
	 */  
	if ((pCmnd = adpt_find_cmd_in_queue(
			(adpt_cmnd **)&pTarget->PendingQueueHead,
			(adpt_cmnd **)&pTarget->PendingQueueTail,
		       	cmd)) == NULL) {
		PDEBUG(DBG_ERR_HNDL, "Could not find cmd in the Pending "
					"Queue !!\n");

		ERROR_END;

                goto finish_cmd;
	}	
	
	/*
	 * DC:	We still have RTTH left, send abort and have the command
	 *	retried again.
	 */	   
	PDEBUG(DBG_ERR_HNDL, "Aborting cmd = 0x%02x with pCmnd %p \n", 
				cmd->cmnd[0], pCmnd);
	
	pCmnd->cmd_state |= CMD_REQ_ABORT;

	memset(msg, 0, sizeof(msg));

	msg[0] = FIVE_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = I2O_CMD_SCSI_ABORT << 24 | HOST_TID << 12 | pDev->tid;
	msg[2] = 0;		/* Initiator context */
	msg[3] = (U32) 0xffff0000; /* Transaction context */
	msg[4] = (U32) pCmnd;	/* Transaction context to be aborted */
	
	PDEBUG(DBG_ERR_HNDL, "Abort !\n");
	rcode = adpt_i2o_post_this(pHba, msg, sizeof(msg));
	PDEBUG(DBG_ERR_HNDL, "Abort Complete!\n");

	if (rcode != 0) {
		PDEBUG(DBG_ERR_HNDL, "Failed to perform Abort !\n");
		ERROR_END;
                goto finish_cmd;

                 
	} else {

                   pTarget->state |= ADPT_TARGET_SESS_RECOVERY; 
                   pTarget->state |= ADPT_DEV_OFFLINE; 
//                   pTarget->state &= ~ADPT_DEV_ONLINE; 

		pHba->aborted_cmd_count++;


	   cmd->result = (DID_BUS_BUSY << 16);
			
           adpt_queue_done_command(cmd, cmd->scsi_done);


		   	
			/*
			 * Retry the IO again (FTO)
	   		 * Need to check if it is disk IO, though
	   		adpt_retry_io(pHba, pNewCmnd);
	   		 */  
        }	
finish_cmd:
        END;
        return;
}

/*
 * adpt_queue()
 * 
 * Description:
 *	Queue a command to the controller. 	
 */    
S32 
adpt_queue(Scsi_Cmnd *cmd, void (*scsi_done)(Scsi_Cmnd *))
{
	adpt_hba		*pHba = NULL;
	adpt_target		*pTarget = NULL;
	adpt_device		*pDev = NULL;	
	adpt_cmnd		*pCmnd;
	U32 			timeout = 0;
	U32_64			flags = 0;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,17)
	adpt_device		*pPrevDev = NULL;
	U32			prev_lun = 0;
#endif	
        BEGIN;

	cmd->scsi_done = scsi_done;
	cmd->result = (DID_ERROR << 16);
	/*
	 * SCSI REQUEST_SENSE commands will be executed automatically by the 
	 * Host Adapter for any errors, so they should not be executed 
	 * explicitly unless the Sense Data is zero indicating that no error 
	 * occurred.
	 */
	if ((cmd->cmnd[0] == REQUEST_SENSE) && (cmd->sense_buffer[0] != 0)) {
		PDEBUG(DBG_ERR_HNDL,"Request Sense\n");
		cmd->result = (DID_OK << 16);

		adpt_queue_done_command(cmd, scsi_done);

		ERROR_END;
	
		return 0;
	}

	pHba = (adpt_hba *) cmd->host->hostdata[0];

	/*
	 * We are still on the Bus Reset Stage.
	 * Return the IO to be retried later on.
	 */   
	if (pHba->state & ADPT_STATE_BUS_RESET) {
		cmd->result = (DID_ERROR << 16);

		adpt_queue_done_command(cmd, scsi_done);

		ERROR_END;

		return 0;
	}

	if ((pDev = adpt_find_device(pHba, cmd->channel, 
				cmd->target, cmd->lun)) == NULL) {
			//printk("adpt_find_device failed for BTL %d %d %d\n",
			//	cmd->channel, cmd->target, cmd->lun);
			cmd->result = (DID_NO_CONNECT << 16);

			adpt_queue_done_command(cmd, scsi_done);

			ERROR_END;
			
			return 0;
	}
#if 0		
	pDev = (adpt_device *) (cmd->device->hostdata);
	
	if (pDev == NULL) {
		/*
		 * First command request for this device.  Set up a pointer
		 * to the device structure.  This should be a TEST_UNIT_READY
		 * command from scan_scsis_single.
		 */
		if ((pDev = adpt_find_device(pHba, cmd->channel, 
				cmd->target, cmd->lun)) == NULL) {
			cmd->result = (DID_NO_CONNECT << 16);

			adpt_queue_done_command(cmd, scsi_done);

			ERROR_END;
			
			return 0;
		}
		
		(adpt_device *) (cmd->device->hostdata) = pDev;
	}
#endif
	
	if ((pTarget = adpt_get_target(pHba, cmd->channel, cmd->target)) == NULL) {	
		/*
		 * No Target attached 
		 */  
		cmd->result = (DID_NO_CONNECT << 16);

		adpt_queue_done_command(cmd, scsi_done);

		ERROR_END;
		
		return 0;
	}

	/*
	 * If we are being called from when the device is being reset, 
	 * session recovery or while being rescanned,
	 * delay processing of the command until later.
         * Link is up we are in session recovery
	 */

	if (pTarget->state & ADPT_TARGET_RESET) { 

		cmd->result = (DID_ERROR << 16);

		adpt_queue_done_command(cmd, scsi_done);

		ERROR_END;

		return 0;
	}

	pDev->pScsi_dev = cmd->device;

	/*
	 * This is a workround fix for the scsi_scan issue (scsi_level for 
	 * previous INQUIRY didn't get saved) when it sets the wrong byte 1 
	 * field for INQUIRY CDB for multi-lun devices.
	 * Latest kernel seems to fix this issue. 
	 */   

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,17)
	if ((cmd->cmnd[0] == 0x12) && (cmd->lun > 0)) {
		prev_lun = cmd->lun - 1;

		if ((pPrevDev = adpt_find_device(pHba, cmd->channel, 
					cmd->target, prev_lun)) != NULL) {
			if (pPrevDev->pScsi_dev->scsi_level > SCSI_2) {
				cmd->cmnd[1] = 0;
			}
		}
	}	
#endif	
        /* 60 seconds added so if debug messages from serial port are holding
         * the driver from returning in process_done_queue
         */
        if (pTarget->flexible_timeout < 120)         
        {
/*
	    timeout = cmd->eh_timeout.expires + ((pSetup->command_timeout + 60) * HZ); 
*/
	    timeout = cmd->eh_timeout.expires + (120 * HZ); 
        }
        else
        {
	    timeout = cmd->eh_timeout.expires + (pTarget->flexible_timeout *HZ);
        }

	if (cmd->eh_state != SCSI_STATE_QUEUED) {
#ifdef __VMKERNEL_MODULE__
	   if(cmd->eh_timeout.function != NULL)
#endif	      
		mod_timer(&cmd->eh_timeout, timeout); 
	}
			
	/*
	 * Returned the IO back to the OS for that device that is gone for 
	 * good.  
	 */
	if ( (pTarget->state & ADPT_DEV_OFFLINE)  && 
             (!(pTarget->state & ADPT_TARGET_SESS_RECOVERY)) ) 
	{
		if (cmd->device != NULL) {
			cmd->device->online = FALSE;
		}

		cmd->result = (DID_NO_CONNECT << 16);

		adpt_queue_done_command(cmd, scsi_done);

		ERROR_END;
		
		return 0;
	}

	// if the LUN has been marked absent in our list, do not start 
	// IO on that
	if(pDev->state & ADPT_DEV_ABSENT) {

      if (cmd->device != NULL) {
         cmd->device->online = FALSE;
      }
      
		cmd->result = (DID_NO_CONNECT << 16);
                                                                                                                             
      adpt_queue_done_command(cmd, scsi_done);
                                                                                                                             
      ERROR_END;
                                                                                                                             
      return 0;

	}
	/*
	 * Check if we can send more IOs down to the  Adapter.
	if ( pHba->aborted_cmd_count >ASA72XX_MAX_Q_DEPTH
	     || pHba->active_cmds > ASA72XX_MAX_Q_DEPTH ) {
	 */ 
	if(pHba->active_cmds >= ASA72XX_MAX_Q_DEPTH ) {
           if (!( (pTarget->state & ADPT_TARGET_SESS_RECOVERY) )){ 
		PDEBUG(DBG_ERR_HNDL,"Reached MAX IOs limit\n");
					     
/*		cmd->result = (DID_BUS_BUSY << 16); */
		cmd->result = (DID_SOFT_ERROR << 16); 
		
		adpt_queue_done_command(cmd, scsi_done);
                ERROR_END;
												return 0;
           }
        }


	adpt_lock(pHba, flags, 1);
	/*
	 * Get a free scb.
	 */  
	if ((pCmnd = adpt_get_free_cmd(pHba)) == NULL) {
		cmd->result = (DID_BUS_BUSY << 16);

		adpt_queue_done_command(cmd, scsi_done);

		ERROR_END;
	        adpt_unlock(pHba, flags, 1);

		return 0;
	}

	adpt_unlock(pHba, flags, 1);
	
	pCmnd->target = pTarget;
	
	pCmnd->command = cmd;
/*
	pCmnd->cmd_timeout = cmd->eh_timeout.expires - jiffies;
*/

	/*
	 * If the device is offline, perform Session Recovery for that device
	 * until FTO expires or the device is back online again in timer.
	 */   
	if ( (pTarget->state & ADPT_DEV_OFFLINE)  && 
             (pTarget->state & ADPT_TARGET_SESS_RECOVERY) ){ 
		/*
		 * Return DID_BUS_BUSY to have the command to be retried 
		 * again if the SR limit has been exceeded.    
                 * Return the cmd at the end of SR timeout 
                 * It's not a good idea for many error & busy because it 
                 * fills up /var/log.  Recall cmd_timeout is called 5s 
                 * before timing out.
		 */  
               
		PDEBUG(DBG_SCSI,"Command retried on No nexus cmd/(%x/)\n", pCmnd);
                pCmnd->cmd_state |= CMD_REQ_EXTENDED_COMPLETION;
                 
                if ( (pTarget->flexible_timeout == 0) &&
                     (pTarget->reset_target_threshold) ) {

                      /*
                       * Default setting exists. User forgot to set iconfig. 
                       * Driver now must use the default values.
                       */
                       pTarget->flexible_timeout = TMOUT_SESS_RECOVERY;
                    

                }
                else {
                      /*
                       * User's setting exists. 
                       * Driver now must use those values.
                       */

                }


		cmd->result = (DID_BUS_LINKDOWN << 16);

                /* lets add a timer to return it */
		PDEBUG(DBG_SCSI, "pCmd = %p \n ", pCmnd);
		/*
		* We do not need another thread running in for SR, and returning
		* Let adpt_timer handle this
               	adpt_queue_done_command_for_sr(cmd, cmd->scsi_done, pTarget);
		*/

	adpt_lock(pHba, flags, 1);
		adpt_insert_cmd_to_queue((adpt_cmnd **)&pHba->LinkdownQueueHead,
		       (adpt_cmnd **)&pHba->LinkdownQueueTail, pCmnd); 
	adpt_unlock(pHba, flags, 1);


		ERROR_END;

		return 0;
        }

	/*
	 * Set a timer to execute adpt_cmd_timeout 5 secs before the 
	 * command timeout.       
	 */     
	if ((cmd->eh_state != SCSI_STATE_QUEUED) && 
			(cmd->device->type == TYPE_DISK)) {
		init_timer(&pCmnd->cmd_timer);
		
		pCmnd->cmd_timer.data = (U32_64) pCmnd;

	        /* Driver should timeout earlier because added 60 s */	
		pCmnd->cmd_timer.expires = adpt_get_jiffies() + (60*HZ);
		
		pCmnd->cmd_timer.function = adpt_cmd_timeout;
		
		add_timer(&pCmnd->cmd_timer);
	}
	
	adpt_lock(pHba, flags, 1);

	adpt_insert_cmd_to_queue((adpt_cmnd **)&pHba->SendQueueHead,
		       (adpt_cmnd **)&pHba->SendQueueTail, pCmnd); 

	adpt_start_io(pHba);

	adpt_unlock(pHba, flags, 1);
        END;
	return 0;
}


/*
 * adpt_insert_cmd_to_queue()
 *  
 * Description:
 *  	Insert the command into the queue.
 */  
static inline void
adpt_insert_cmd_to_queue(adpt_cmnd **pQHead, 
			adpt_cmnd **pQTail, 
			adpt_cmnd *pCmnd)
{
	BEGIN;

	if (*pQHead == NULL) {
		*pQHead = pCmnd;
	} else {
		(*pQTail)->next = pCmnd;
		pCmnd->prev = *pQTail;
		pCmnd->next = NULL;
	}

	*pQTail = pCmnd;

	END;
}


/*
 * adpt_find_cmd_in_queue()
 *
 * Description:
 *	Search if the corresponding command is in the Queue.
 */	    
static inline adpt_cmnd *
adpt_find_cmd_in_queue(adpt_cmnd **pQHead, adpt_cmnd **pQTail,
			Scsi_Cmnd *cmd)
{
	adpt_cmnd 	*pCmnd;

	BEGIN;

	if (*pQHead == NULL) {
		goto end;
	}

	pCmnd = *pQHead;

	while (pCmnd) {
		if (pCmnd->command == cmd) {
			PDEBUG(DBG_ERR_HNDL, "Command = 0x%p found in the "
					"Queue !\n", pCmnd->command);

			return pCmnd;
		}

		if (pCmnd == *pQTail) {
			goto end;
		} else {
			pCmnd = pCmnd->next;
		}
	}

end:	
	END;

	return NULL;
}


/*
 * adpt_remove_cmd_fr_queue() 
 *
 * Description:
 * 	Remove the command from queue if exists.
 */
inline U8 
adpt_remove_cmd_fr_queue(adpt_cmnd **pQHead, 
			adpt_cmnd **pQTail, 
			adpt_cmnd *pCmnd)
{
	adpt_cmnd	*pCmnd_curr;
	
	BEGIN;

	if (*pQHead == NULL) {
		PDEBUG(DBG_ERR_HNDL, "The Queue Head is NULL !\n");

		return 1;
	}

	if (*pQTail == NULL) {
		PDEBUG(DBG_ERR_HNDL, "The Queue Tail is NULL !\n");

		return 1;
	}

	if (pCmnd == NULL) {
		PDEBUG(DBG_ERR_HNDL, "The command is NULL !\n");

		return 1;
	}

/*
	if (pCmnd->command == NULL) {
		PDEBUG(DBG_ERR_HNDL, "Bogus command !\n");

		return 1;
	}
*/
			
	pCmnd_curr = *pQHead;

	while (pCmnd_curr) {	
/*
		if (pCmnd_curr->command == pCmnd->command) {
*/
		if (pCmnd_curr == pCmnd) {
			break;
		}
/*
		if (pCmnd_curr == *pQTail) {
			PDEBUG(DBG_ERR_HNDL, "Command not found in "
					"the Queue !\n");

			return 1;
		}
*/

		pCmnd_curr = pCmnd_curr->next;
	}
        if (pCmnd_curr == NULL) {
		PDEBUG(DBG_ERR_HNDL, "Couldn't find pCmnd in Q!\n");
		return 1;
        }

	if ((pCmnd_curr == *pQHead) && (pCmnd_curr == *pQTail)) {
		*pQHead = NULL;
		*pQTail = NULL;
	} else if ((pCmnd_curr == *pQHead) && (pCmnd_curr != *pQTail)) {
		*pQHead = pCmnd_curr->next;
		pCmnd_curr->next->prev = NULL;
	} else if ((pCmnd_curr != *pQHead) && (pCmnd_curr != *pQTail)) {
		pCmnd_curr->prev->next = pCmnd_curr->next;
		pCmnd_curr->next->prev = pCmnd_curr->prev;
	} else {
		*pQTail = pCmnd_curr->prev;
		pCmnd_curr->prev->next = NULL;
	}
	pCmnd->next = pCmnd->prev = NULL;

	END;

	return 0;
}


#ifdef ASA72XX_DEBUG
/*
 * adpt_print_queue()
 *
 * Description:
 *	Print out the content of the Queue.
 */
static void 
adpt_print_queue(adpt_cmnd **pQueueHead)
{
	adpt_cmnd	*pCmnd_curr;
	adpt_cmnd	*pCmnd_next;
	U8		count = 0;

	BEGIN;

	pCmnd_curr = *pQueueHead;

	if (pCmnd_curr == NULL) {
		PDEBUG(DBG_ERR_HNDL, "The Queue is empty !\n");
	}

	while (pCmnd_curr) {
		pCmnd_next = pCmnd_curr->next;

		PDEBUG(DBG_ERR_HNDL, "The content # %d of the Queue is %p !\n",
				count, pCmnd_curr->command);

		pCmnd_curr = pCmnd_next;

		count++;
	}

	END;
}

#endif

/*
 * adpt_start_io()
 *
 * Description:
 *	Start the execution of IO.
 *	If fail to start the IO, put the command back to the queue.
 */	    
static void 
adpt_start_io(adpt_hba *pHba)
{
	adpt_cmnd		*pCmnd;
	adpt_cmnd		*pNextCmnd;
	adpt_target		*pTarget;
	adpt_device		*pDev;
	S32			rcode;
	
	BEGIN;

	pCmnd = pHba->SendQueueHead;
	
	pHba->SendQueueHead = NULL;

	pHba->SendQueueTail = NULL;

	while (pCmnd) {
		pNextCmnd = pCmnd->next;

		pCmnd->prev = NULL;

		pCmnd->next = NULL;

		pCmnd->cmd_state = CMD_REQ_ACTIVE;
			
		//pDev = (adpt_device *) (pCmnd->command->device->hostdata);
		pDev = adpt_find_device(pHba, pCmnd->command->channel, 
					pCmnd->command->target,
					pCmnd->command->lun);
	
		if ((rcode = adpt_scsi_to_i2o(pHba, pCmnd, pDev)) == 0) {
			pTarget = pCmnd->target;

			adpt_insert_cmd_to_queue(
				(adpt_cmnd **) &pTarget->PendingQueueHead,
		       	    	(adpt_cmnd **) &pTarget->PendingQueueTail, 
			    	pCmnd); 

			pHba->active_cmds++;
		} else {
			/*
			 * Failed to start the IO.
			 * Put the command back into the queue.
			 */    
			PDEBUG(DBG_ERR_HNDL, "Failed to start the IO !\n");

			pCmnd->cmd_state &= ~CMD_REQ_ACTIVE;

			adpt_insert_cmd_to_queue(
					(adpt_cmnd **)&pHba->SendQueueHead,
		       			(adpt_cmnd **)&pHba->SendQueueTail, 
					pCmnd); 
		}

		pCmnd = pNextCmnd;
	}

	END;
}


/*
 * adpt_scsi_register()
 *
 * Description:
 *	Register the Host Adapter's template to the OS.
 */	    
S32 
adpt_scsi_register(adpt_hba *pHba, Scsi_Host_Template *sht)
{
	struct Scsi_Host 	*host = NULL;

	BEGIN;
	
#ifndef __VMKERNEL_MODULE__
	adpt_lock(pHba, 0, 0);

	host = scsi_register(sht, sizeof(adpt_hba *));
#else

	host = vmk_scsi_register(sht, sizeof(adpt_hba *), pHba->pDev->bus->number, pHba->pDev->devfn);
#endif //__VMKERNEL_MODULE__
	
	if (host == NULL) {
		PDEBUG(DBG_INIT, "%s:  scsi_register returned NULL\n", 
				pHba->name);

#ifndef __VMKERNEL_MODULE__NO_BUG_FIX__
#ifndef __VMKERNEL_MODULE__		
		adpt_unlock(pHba, 0, 0);
#endif
#endif
		ERROR_END;
		
		return -1;
	}

#ifndef __VMKERNEL_MODULE__
        /*
         * Usually the driver can get scsi host # 2-5 
         * if 0,1 are used... ie. driver gets x,x+1,x+2,x+3
         * where x is the maximum number of adapter.
         */
        if (pSetup->persistent_host_no != host->host_no &&
            pSetup->persistent_host_no > host->host_no ) {
             {
	        struct Scsi_Host 	*tmphost = NULL;
	        struct Scsi_Host 	*tmphost1 = NULL;
	        struct Scsi_Host 	*tmphost2 = NULL;
	        struct Scsi_Host 	*default_host = NULL;

	        tmphost = scsi_register(sht, sizeof(adpt_hba *));

                if (tmphost != NULL) {
                     default_host=tmphost;
                     if (pSetup->persistent_host_no == tmphost->host_no) 
                     goto host_done_1;
                } 
                else { 
	             scsi_unregister(tmphost);
                     goto host_done_0;
                }

	        tmphost1 = scsi_register(sht, sizeof(adpt_hba *));
                if (tmphost1 != NULL) {
                     default_host=tmphost1;
                     if (pSetup->persistent_host_no == tmphost1->host_no) 
                     goto host_done_2;
                } 
                else { 
	             scsi_unregister(tmphost1);
                     goto host_done_1;
                }

	        tmphost2 = scsi_register(sht, sizeof(adpt_hba *));
                if (tmphost2 != NULL) {
                     default_host=tmphost2;
                } 
                else { 
	             scsi_unregister(tmphost2);
                     goto host_done_2;
                }

	        scsi_unregister(tmphost1);
host_done_2:
	        scsi_unregister(tmphost);
host_done_1:
	        scsi_unregister(host);
host_done_0:
                host=default_host;
             }
        }
#endif //___VMKERNEL_MODULE__
	(adpt_hba *)(host->hostdata[0]) = pHba;

	pHba->host = host;

	host->hostt->use_new_eh_code = 1;
	
	host->irq = pHba->pDev->irq;
	
	/* 
	 * no IO ports, so don't have to set host->io_port and 
	 * host->n_io_port
	 */
	host->io_port = 0;
	
	host->n_io_port = 0;
	
	/* see comments in hosts.h */
	host->max_id = pHba->top_scsi_id + 1;
	
	host->max_lun = pHba->top_scsi_lun + 1;
	
	host->max_channel = pHba->top_scsi_channel;

	host->cmd_per_lun = ASA72XX_CMD_PER_LUN;
	
	host->unique_id = (U32) pHba;
	
	host->sg_tablesize = pHba->sg_tablesize;
	
	host->can_queue = ASA72XX_MAX_Q_DEPTH;
	
	host->select_queue_depths = adpt_select_queue_depths;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,7)
	/*
	 * Maximum number of sectors allowed in a single SCSI command.
	 */  
	host->max_sectors = 512;
#endif	
	
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,4)
	scsi_set_pci_device(host, pHba->pDev);
#endif		

#ifdef __VMKERNEL_MODULE__
	vmk_scsi_register_uinfo(host, pHba->pDev->bus->number, pHba->pDev->devfn, pHba);
#else
	adpt_unlock(pHba, 0, 0);
#endif //__VMKERNEL_MODULE__

	END;

	return 0;
}

/* 
 * adpt_bios_param
 * 
 * Description:
 *	Reports the disk geometry for the given SCSI devices.
 */	   
S32 
adpt_bios_param(Disk *disk, kdev_t dev, S32 geom[])
{
	S32 heads = -1;
	S32 sectors = -1;
	S32 cylinders = -1;

	BEGIN;

	if ((disk->capacity >> 11) >= 1024) {
		heads = 255;
		sectors = 63;
	} else {
		heads = 64;
		sectors = 32;
	}	
	
	cylinders = disk->capacity / (heads * sectors);

	geom[0] = heads;
	geom[1] = sectors;
	geom[2] = cylinders;
	
	END;

	return 0;
}


/*
 * adpt_unmap_sg_mapping()
 *
 * Description:
 *	Unmap the scatter gather list DMA mapping.
 */	    
static inline void
adpt_unmap_sg_mapping(adpt_hba *pHba, adpt_cmnd *pCmnd)
{
	Scsi_Cmnd	*cmd;
	dma_addr_t	bus_addr;
	S32		dma_dir;

	BEGIN;
	
	cmd = pCmnd->command;

	dma_dir = scsi_to_pci_dma_dir(cmd->sc_data_direction);

	if (cmd->use_sg) {
		struct scatterlist	*sg;

		sg = (struct scatterlist *) cmd->request_buffer;

		pci_unmap_sg(pHba->pDev, sg, cmd->use_sg, dma_dir);
	} else {
		if (cmd->request_bufflen != 0) {
			bus_addr = pCmnd->bus_addr;
			
			pci_unmap_single(pHba->pDev, bus_addr, 
					cmd->request_bufflen, dma_dir);
		}
	}

	END;
}	


/*
 * adpt_i2o_to_scsi()
 *
 * Description:
 * 	Check I2O Reply frame status and report to OS for IO completion
 * 	with or without errors.	     
 */  
S32 
adpt_i2o_to_scsi(U32_64 reply, adpt_cmnd *pCmnd)
{
	adpt_hba	*pHba;
	adpt_device	*pDev;
	Scsi_Cmnd	*cmd;
	U32 		hba_status;
	U32 		dev_status;
	/* Leave it shifted up 8 bits  */
	U32 		reply_flags = readl(reply) & 0xff00; 
	U32		len = 0;
	U16 		detailed_status;
	U8 		req_status;

	BEGIN;

	cmd = pCmnd->command;

	pHba = (adpt_hba *) cmd->host->hostdata[0];

//	pDev = (adpt_device *) (cmd->device->hostdata);
	pDev = adpt_find_device(pHba, pCmnd->command->channel, 
				pCmnd->command->target,
				pCmnd->command->lun);
	
	if (pCmnd->cmd_timer.function != NULL) {
		/*
		 * To avoid race condition issue when deleting timer,
		 * it is better to use del_timer_sync.
		 */    

		pCmnd->cmd_timer.data = (U32_64) NULL;

		pCmnd->cmd_timer.function = NULL;

		del_timer_sync(&pCmnd->cmd_timer);
	}
	
	/*
	 * Unmapped the scatter-gather mapping.
	 */  
	adpt_unmap_sg_mapping(pHba, pCmnd);
	
	req_status = (readl(reply + 16) >> 24) & 0xFF;
	detailed_status = readl(reply + 16) & 0xFFFF;	
	
	dev_status = detailed_status & 0xff;
	hba_status = detailed_status >> 8;
	
	if (!(reply_flags & MSG_FAIL)) {
		switch (detailed_status & I2O_SCSI_DSC_MASK) {
			case I2O_SCSI_DSC_SUCCESS:
				cmd->result = (DID_OK << 16);
				break;
				
			case I2O_SCSI_DSC_REQUEST_ABORTED:
			case I2O_SCSI_DSC_COMMAND_ABORTED:
				cmd->result = (DID_ABORT << 16);
				break;
				
			case I2O_SCSI_DSC_DATA_OVERRUN:
				cmd->result = (DID_OK << 16);
				break;
				
			case I2O_SCSI_DSC_NO_NEXUS:
					pCmnd->target->state |= 
						ADPT_TARGET_SESS_RECOVERY;
					pCmnd->target->state |= 
						ADPT_DEV_OFFLINE;
					//pCmnd->target->state &= ~ADPT_DEV_ONLINE;
			case I2O_SCSI_DSC_UNABLE_TO_ABORT:
			case I2O_SCSI_DSC_COMPLETE_WITH_ERROR:
			case I2O_SCSI_DSC_UNABLE_TO_TERMINATE:
			case I2O_SCSI_DSC_MR_MESSAGE_RECEIVED:
			case I2O_SCSI_DSC_AUTOSENSE_FAILED:
			case I2O_SCSI_DSC_UNEXPECTED_BUS_FREE:
			case I2O_SCSI_DSC_SEQUENCE_FAILURE:
			case I2O_SCSI_DSC_REQUEST_LENGTH_ERROR:
			case I2O_SCSI_DSC_PROVIDE_FAILURE:
			case I2O_SCSI_DSC_REQUEST_TERMINATED:
			case I2O_SCSI_DSC_IDE_MESSAGE_SENT:
			case I2O_SCSI_DSC_UNACKNOWLEDGED_EVENT:
			case I2O_SCSI_DSC_MESSAGE_RECEIVED:
			case I2O_SCSI_DSC_INVALID_CDB:
			case I2O_SCSI_DSC_LUN_INVALID:
			case I2O_SCSI_DSC_SCSI_TID_INVALID:
			case I2O_SCSI_DSC_FUNCTION_UNAVAILABLE:
			case I2O_SCSI_DSC_SCSI_IID_INVALID:
			case I2O_SCSI_DSC_CDB_RECEIVED:
			case I2O_SCSI_DSC_LUN_ALREADY_ENABLED:
			case I2O_SCSI_DSC_QUEUE_FROZEN:
				cmd->result = (DID_ERROR << 16);
				break;
				
			case I2O_SCSI_DSC_ADAPTER_BUSY:
			case I2O_SCSI_DSC_BUS_BUSY:
				cmd->result = (DID_BUS_BUSY << 16);
				break;
				
			case I2O_SCSI_DSC_REQUEST_INVALID:
				cmd->result = (DID_TIME_OUT << 16);
				break;
				
			case I2O_SCSI_DSC_PATH_INVALID:
				cmd->result = (DID_TIME_OUT << 16);
				break;
				
			case I2O_SCSI_DSC_DEVICE_NOT_PRESENT:
				cmd->result = (DID_TIME_OUT << 16);
				break;
				
			case I2O_SCSI_DSC_SELECTION_TIMEOUT:
			case I2O_SCSI_DSC_COMMAND_TIMEOUT:
			case I2O_SCSI_DSC_NO_ADAPTER:
			case I2O_SCSI_DSC_RESOURCE_UNAVAILABLE:
				cmd->result = (DID_TIME_OUT << 16);
				break;
				
			case I2O_SCSI_DSC_SCSI_BUS_RESET:
			case I2O_SCSI_DSC_BDR_MESSAGE_SENT:
				cmd->result = (DID_RESET << 16);
				break;
				
			case I2O_SCSI_DSC_PARITY_ERROR_FAILURE:
				cmd->result = (DID_ERROR << 16);
				break;
				
			default:
				cmd->result = (DID_TIME_OUT << 16);
				break;
		}

		/* 
		 * If there is a SCSI target status, save it off in the
		 * OS request packet and copy over the request sense 
		 * data if it was a check condition status
		 */ 
		if (dev_status == 0x02) { /* Check Condition */
		        /*
		         * Sometimes target has DSC0000, success but
			 * still report check condition, so lets report error
			 * as default
			 */
                        cmd->result = (DID_ERROR << 16);
			len = sizeof(cmd->sense_buffer);

			len = (len > 256) ?  256 : len;
	
			memset(cmd->sense_buffer, 0, len);

			/* Copy over the sense data */
			memcpy(cmd->sense_buffer, (void *)(reply + 28), 
					len);

			/* Check Sense data first */
			if (((cmd->sense_buffer[0] & 0x70) >> 4) == 7) {
				PDEBUG(DBG_ERR_HNDL,"Error Code : 0x%x\n"
						"Sense Key  : 0x%x\n",
						cmd->sense_buffer[0],
						cmd->sense_buffer[2]);

				PDEBUG(DBG_ERR_HNDL,"ASC        : 0x%x\n"
						"ASCQ       : 0x%x\n",
						cmd->sense_buffer[12],
						cmd->sense_buffer[13]);
				
				/*
				 * If the hba_status is I2O_SCSI_DSC_NO_NEXUS
				 * (0x3b) and the sense buffer[0] is 0x70,
				 * sense buffer[2]=0xb, then the FW is 
				 * doing Session Recovery initiated by the 
				 * target.    
				 */
			    	if ((hba_status == I2O_SCSI_DSC_NO_NEXUS)
					&& (cmd->sense_buffer[0] == 0x70)
					&& (cmd->sense_buffer[2] == 0xb)) {
					
				PDEBUG(DBG_SCSI,"No Nexus cmd/(%x/)\n", cmd);
		 
		                        pCmnd->target->sess_recovery_by_fw = 1;
				}

				/*
				 * Some special error conditions need to be 
				 * treated differently.
				 */   
				if (((cmd->cmnd[0] == 0x12) || 
					(cmd->cmnd[0] == 0x0)) && 
					(cmd->sense_buffer[2] == 0xb)) {	
					cmd->result = (DID_NO_CONNECT << 16);
				}

				/*
				 * Unit Attention.
				 */  
				if (cmd->sense_buffer[2] == 0x6) {
					/*
					 * Third party reset.
					 * Return DID_ERROR for a retry. 
					 */  
					if ((cmd->sense_buffer[12] == 0x29) &&
					    (cmd->sense_buffer[13] == 0x0)) {
						cmd->result = (DID_ERROR << 16);
					}	
					/*
					 * Above code is questionable.. Driver
					 * should have error in 0x6 case
					 */
				}	
			}
		}
	} else {
		cmd->result = (DID_TIME_OUT << 16);
	}

	if (cmd->result == (DID_ERROR << 16)) {
		PDEBUG(DBG_ERR_HNDL, "SCSI error - Device (%d,%d,%d) tid=%d "
	               "hba_status = 0x%x, dev_status = 0x%x, cmd = 0x%x\n", 
		       (U32)cmd->channel, (U32)cmd->target, 
		       (U32)cmd-> lun,
		       //((adpt_device *)(cmd->device->hostdata))->tid,
			pDev->tid,
		       hba_status, dev_status, cmd->cmnd[0]);
	}
       	
	if (cmd->result == (DID_TIME_OUT << 16)) {
		PDEBUG(DBG_ERR_HNDL, "%s: SCSI timeout - Device (%d,%d,%d) "
			"tid=%d "
			"hba_status = 0x%x, dev_status = 0x%x, cmd = 0x%x\n",
			pHba->name, (U32)cmd->channel, (U32)cmd->target, 
			(U32)cmd-> lun,
			//((adpt_device *)(cmd->device->hostdata))->tid,
			pDev->tid,
			hba_status, dev_status, cmd->cmnd[0]);
	} 
	
	cmd->result |= (dev_status);

#ifdef ASA72XX_DEBUG	
        if (req_status ||  detailed_status) {
        if ( cmd->cmnd[0] == WRITE_10) {
	          PDEBUG(DBG_SCSI, "pCmd = %p \n ", pCmnd);
	          PDEBUG(DBG_SCSI, "CDB ="
			 " 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n",
                         cmd->cmnd[0],
			 cmd->cmnd[1],
			 cmd->cmnd[2],
			 cmd->cmnd[3],
                         cmd->cmnd[4],
			 cmd->cmnd[5],
			 cmd->cmnd[6],
			 cmd->cmnd[7],
			 cmd->cmnd[8],
			 cmd->cmnd[9]);
        }
	PDEBUG(DBG_SCSI, "Target ID          = 0x%x \n", 
//			((adpt_device *)(cmd->device->hostdata))->tid);
			pDev->tid);
	PDEBUG(DBG_SCSI, "ReplyFlags         = 0x%04x \n", reply_flags);
	PDEBUG(DBG_SCSI, "ReqStatus          = 0x%02x \n", req_status);
	PDEBUG(DBG_SCSI, "DetailedStatusCode = 0x%04x \n", detailed_status);
	PDEBUG(DBG_SCSI, "AdapterStatus      = 0x%02x \n", hba_status);
	PDEBUG(DBG_SCSI, "DeviceStatus       = 0x%02x \n", dev_status);
	PDEBUG(DBG_SCSI, "Cmnd State         = 0x%x   \n", pCmnd->cmd_state);
	PDEBUG(DBG_SCSI, "Cmnd Result         = 0x%x   \n", cmd->result);
        }
#endif

	END;

	return 0;
}

/*
 * adpt_rescan()
 *
 * Description:
 *	This function can be called with bus_scan_needed
 *	flag set to a 0 or non-zero. A non-zero value
 *	will cause a HBA_BUS_SCAN I2O message to be sent
 *	down. Then it waits until LCT notify gets called
 *	and the LCT table is parsed completely.
 *	If called with a zero value, the LCT table is
 *	is got and the table is parsed.
 */  
static void
adpt_rescan(adpt_hba *pHba, U8 bus_scan_needed)
{
	S32 		rcode;
	struct 		Scsi_Host *host = NULL;

	BEGIN;
	
	if (pHba->hostverifydoneflag != 1) {
		ERROR_END;

		return;
	}
	
	/*
	 * In session recovery, we do not need to issue HBA_BUS_SCAN.
	 */  
	if (bus_scan_needed) {

		if ((rcode = adpt_i2o_hba_scan(pHba)) < 0) {
			PDEBUG(DBG_LCT, "Failed to Rescan Host Adapter !\n");

			ERROR_END;

			return;
		}
		// If the rescan was successfully issued, wait for
		// the Rescan to give back a LCT notification. We will
		// wait until the lct notification comes back and we've
		// processed the LCT table.
		// 
		pHba->wakeup_flag=1;
#ifdef __VMKERNEL_MODULE__
		vmk_thread_wait_event(&pHba->lct_processing_complete, NULL);
#else
		sleep_on(&pHba->lct_processing_complete);
#endif //__VMKERNEL_MODULE
	}
	else {
	
		if ((rcode = adpt_i2o_lct_get(pHba)) < 0) {
			PDEBUG(DBG_LCT, "Failed to issue LCT Get when "
				"rescanning HBA !\n");	

			ERROR_END;
	
			return;
		}


		if ((rcode = adpt_i2o_parse_lct(pHba)) < 0) {
			PDEBUG(DBG_LCT, "Failed to issue parse LCT when "
				"rescanning HBA !\n");	

			ERROR_END;

			return;
		}

		if(pHba->host) {

			host = pHba->host;
			if(pHba->top_scsi_channel > host->max_channel)
				host->max_channel = pHba->top_scsi_channel;

			if(pHba->top_scsi_id >= host->max_id)
				host->max_id = pHba->top_scsi_id + 1;
		
			if(pHba->top_scsi_lun >= host->max_lun)
				host->max_lun = pHba->top_scsi_lun + 1;
	
			PDEBUG(DBG_LCT,"Update Max Channel," 
					"Max Id and Max Lun %d %d %d\n",
					host->max_channel,
					host->max_id,
					host->max_lun);

		}
	}

	END;
}


/*
 * adpt_i2o_hba_scan()
 *
 * Description:
 * 	Issue HBA BUS SCAN.
 */
S32
adpt_i2o_hba_scan(adpt_hba *pHba)
{
	U32	msg[4];
	S32	ret = 0;

	BEGIN;

	msg[0] = FOUR_WORD_MSG_SIZE | SGL_OFFSET_0;
	msg[1] = I2O_HBA_BUS_SCAN << 24 | HOST_TID << 12 | ADAPTER_TID;
	msg[2] = 0;
	msg[3] = 0;

	if ((ret = adpt_i2o_post_wait(pHba, msg, sizeof(msg), 
				TMOUT_HBA_SCAN))) {
		PDEBUG(DBG_LCT, "%s: HBA Bus Scan times out !\n", pHba->name);

		return (-1);
	}

	END;

	return 0;
}
/*
 * adpt_i2o_parse_lct()
 *
 * Description:
 *	The logical configuration table tells us what we can talk to
 *	on the board. Most of the stuff isn't interesting to us. 
 *	Remap our target mapping if any changes in the device table.	     
 */  
/*
 * New parse.
 * - Merges parse and reparse
 * - implements peristent binding
 * - cleaned up (removed) struct i2o_device
 */
S32 
adpt_i2o_parse_lct(adpt_hba *pHba)
{
	adpt_device	*pDev = NULL;
	adpt_target	*pTarget = NULL;
	i2o_lct 	*lct = pHba->lct;
	S32			status = 0;
	U32			i;
	U32			max;
	S32			tid = 0;
	S32			parent_tid = 0;
	U16			scsi_lun = 0;
	U8				iscsi_name[256];

	U8 			channel_no = 0;
	U16         scsi_id = 0;
	U16			no_of_devs = 0;
	
	BEGIN;
	
#define DEFAULT_VALUE 0	
#define INITIAL_SCAN 1	
#define RESCAN 2	
	if (lct == NULL) {
		PDEBUG(DBG_LCT, "%s: LCT is empty !\n", pHba->name);
		
		ERROR_END;
		
		return -1;
	}
	
	max = lct->table_size;	
	max -= 3;
	max /= 9;

#ifdef ASA72XX_DEBUG
	PDEBUG(DBG_LCT, "%s: LCT has %d entries.\n", pHba->name, max);
	
	for (i = 0; i < max; i++) {
		//PDEBUG(DBG_LCT, "%d | %d | %x | %x | %x | %x | %x\n",
		PDEBUG(DBG_LCT, "%d | %x | %x | %x | %x | %x\n",
				i,
				//lct->lct_entry[i].change_ind,
		      lct->lct_entry[i].class_id,
				lct->lct_entry[i].sub_class,
				lct->lct_entry[i].tid,
				lct->lct_entry[i].parent_tid,
				lct->lct_entry[i].device_flags);
	}
#endif

	/*
	 * Here's our scheme. We go over LCT table 2 times for targets. 
	 * First time we take care of all static mappings that were 
	 * already given by the firmware. If there's a clash for a slot
	 * we cannot map the target to the same slot, we will move it to 
	 * a new slot.

	 * Next we go over LCT again, this time mapping all those targets 
	 * that did not have any mapping earlier. We check our target list 
	 * to see if we had mapped it earlier and try to map it to the same 
	 * location. If we cannot, we will assign them elsewhere.
	 *
	 * In both the above scans we ignore targets that dont have atleast
	 * 1 LUN behind it.

	 * By this time LCT table scanning is through for targets.

	 * Now we have to validate entries in static list. If some entries
	 * have been deleted, then remove from our target list. This loop
	 * iterates over our target list and marks targets that were unscanned
	 * as absent. It is not yet safe to delete these targets because we
	 * would have reported the LUNs behind these targets to the OS. So the
	 * OS has a pointer possibly pointing to our LUN entry. So just mark
	 * the target absent. Before initiating session recovery, we ignore
	 * the targets which are absent.

	 * Target list is thru, now go over LCT table again, resolving
	 * LUNs. Luns are already assigned, we just need to assign them to
	 * proper target list and modify LUN list based on targets that have
	 * gone off.
	 * 
	 * Now go over our list one last time, marking unscanned LUN entries 
	 * absent.
	 *
	 */

	channel_no = 0;
	scsi_id = 0;
	// mark all the target and lun entries as unscanned.
	// if at the end of 2nd iteration some target entry is still
	// unscanned, then it means that target is not present anymore

	// One optimization here is that we can skip this for
	// the boot scan, since we dont have any target setup at all.

	//if(!(setup_done)) {
		PDEBUG(DBG_LCT,"Marking all offlines...\n");
		for(channel_no = 0;channel_no < MAX_CHANNEL; channel_no++) {
			for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
				if((pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {
					pTarget->state |= ADPT_DEV_UNSCANNED;
					if(pTarget->device) {
						for(pDev = pTarget->device; pDev; pDev=pDev->next_lun)
							pDev->state |= ADPT_DEV_UNSCANNED;
					} // lun entry valid
				} // target entry valid
			} // end for scsi_id
		} // end for channel no
//	}

	for (i = 0; i < max; i++) {
		if (lct->lct_entry[i].class_id  == I2O_CLASS_BUS_ADAPTER_PORT ||
		    lct->lct_entry[i].class_id  == I2O_CLASS_FIBRE_CHANNEL_PORT) {
			
			if(lct->lct_entry[i].device_flags & I2O_LCT_DEVICE_FLAGS_ABSENT) {

				continue;
			}

			if( (lct->lct_entry[i].sub_class & 0xFFFF) == 0xFFFF) {
				
				// no assignments yet, take care in 2nd iteration
				continue;
			}

			if(!(adpt_check_target_has_lun(pHba, &lct->lct_entry[i]))) {

				continue;
			}

			tid = lct->lct_entry[i].tid;
			pTarget = NULL;

			PDEBUG(DBG_LCT, "BUS ADAPTER PORT CLASS FOUND "
					"WITH TID = %d \n", 
					tid);

			channel_no = ( (lct->lct_entry[i].sub_class & 0x0000FF00) >> 8);
			scsi_id = (lct->lct_entry[i].sub_class & 0x000000FF);
			
			// sanity check these values 
			if((channel_no>MAX_CHANNEL) || (scsi_id>MAX_ID_PER_CHANNEL)) {

				/* This was earlier configured in a different OS??
				 * We cannot assign the BT its requesting, we wont
				 * drop it either, but we'll remap the BT for this target.
				 * In this iteration, set it to FFFF, it will be
				 * taken care in the next iteration.
				 */

				PDEBUG(DBG_LCT,"Channel # %d, SCSI Id %d invalid."
							"Resettings IDs ...\n", channel_no, scsi_id);
				
				lct->lct_entry[i].sub_class |= 0xFFFF;
				continue;
			}
			
			PDEBUG(DBG_LCT,"Channel # %d, SCSI Id %d found in LCT, checking "
							"internal table ...\n", channel_no, scsi_id);
			//
			//  This target has a mapping in LCT,  see
			// if its an existing one, or a new target that is trying
			// to unseat an old one...
			//
			if (pHba->channel[channel_no].target[scsi_id] == NULL) {

				// new target. Assign adpt_target structure to it
				PDEBUG(DBG_LCT,"target has BT assigned, but no entry in "
								"our list, mapping it now...\n");	

				pHba->channel[channel_no].target[scsi_id] = pTarget = 
										adpt_map_new_target(pHba,&lct->lct_entry[i],
																channel_no,
																scsi_id);

				if(pTarget == NULL) {

					PDEBUG(DBG_LCT,"Could not map target at Bus %d Target %d\n",
														channel_no, scsi_id);

					continue;
				}

			} // if new target
			else
			{
		
				// we're here because either this is a target that is already
				// existing, or a target that is overwriting a possibly
				// deleted target
				// compare names to determine this.

				pTarget = pHba->channel[channel_no].target[scsi_id];

				status = adpt_get_iscsi_name(pHba, tid, &iscsi_name[0]);

				if(status) {
					
					uerrno = (UL_ERROR | UP_INIT | US_SYS |
						       	transport | 
									UO_CMN | UC_NO_EXT);

					PERROR("%s: ERROR: %s [%x] \n", 
							pHba->name,
							uni_err(uerrno), 
							(U32) uerrno);
					
					ERROR_END;

					return -1;
				}

				if(strcmp(pTarget->iscsi_name, iscsi_name) == 0) {

					// an already existing target, update target params 
					// to reflect any change in its status

					status = adpt_initialize_target_params(pHba,pTarget,
															channel_no, scsi_id,
															&lct->lct_entry[i]);

					if(status) {
					
						uerrno = (UL_ERROR | UP_INIT | US_SYS |
						       	transport | 
									UO_CMN | UC_NO_EXT);

						PERROR("%s: ERROR: %s [%x] \n", 
							pHba->name,
							uni_err(uerrno), 
							(U32) uerrno);
					
						ERROR_END;

						return -1;
					}
				} // iscsi name matches
				else
				{
					// This is a very unlikely case to enter. In the same
					// bootup session, there cannot be multiple targets
					// that have come up with the same BT mapping.
					// Not sure if the firmware remembers deleted targets, but
					// lets handle this gracefully. We have to move this target
					// elsewhere.

					// CR: TODO: assert here for checked builds
					// panic()

					PDEBUG(DBG_LCT,"LCT gave a BT map, but there's a "
									"different target sitting there. "
									" Mapping this target elsewhere...\n");

					pTarget = adpt_map_new_target(pHba, &lct->lct_entry[i],
									0xFF,0xFF);

					if(pTarget == NULL) {
						
						PDEBUG(DBG_LCT,"Could not map new target after a clash, "
											" dropping the new target...\n");
						// lets try another one, this target will be dropped.
						continue;	
					}
				}

			} // if not new target
				
			if (channel_no > pHba->top_scsi_channel) {
				pHba->top_scsi_channel = channel_no;
			}
			
			if (scsi_id > pHba->top_scsi_id) {
				pHba->top_scsi_id = scsi_id;
			}
		} // if lct entry is for target
	} // end for all entries in lct

	PDEBUG(DBG_LCT,"First scan over, scanning 2nd time...\n");

	// 1st looping done. Start 2nd iteration over LCT.
	// Take care of LCT entries that have FFFFs in BT

	for (i = 0; i < max; i++) {
		if (lct->lct_entry[i].class_id  == I2O_CLASS_BUS_ADAPTER_PORT ||
		    lct->lct_entry[i].class_id  == I2O_CLASS_FIBRE_CHANNEL_PORT) {
			if(lct->lct_entry[i].device_flags & I2O_LCT_DEVICE_FLAGS_ABSENT) {

				continue;
			}

			// take care of only unassigned BTs
			if(!( (lct->lct_entry[i].sub_class & 0xFFFF) == 0xFFFF))
			{
				// assignments done already, skip now
				continue;
			}

			if(!(adpt_check_target_has_lun(pHba, &lct->lct_entry[i]))) {

				continue;
			}

			tid = lct->lct_entry[i].tid;
			channel_no = ( (lct->lct_entry[i].sub_class & 0x0000FF00) >> 8);
			scsi_id = (lct->lct_entry[i].sub_class & 0x000000FF);

			// check iSCSI name to see if there was an entry earlier
			status = adpt_get_iscsi_name(pHba,tid,&iscsi_name[0]);

			if(status) {

				PDEBUG(DBG_LCT,"Could not get iSCSI name, quitting\n");
					
				uerrno = (UL_ERROR | UP_INIT | US_SYS |
						       	transport | 
									UO_CMN | UC_NO_EXT);

				PERROR("%s: ERROR: %s [%x] \n", 
							pHba->name,
							uni_err(uerrno), 
							(U32) uerrno);
					
				ERROR_END;

				return -1;
			}

			if(is_existing_target(pHba,iscsi_name,&pTarget))
			{
				status = adpt_initialize_target_params(pHba, pTarget,
														pTarget->channel_no, 
														pTarget->scsi_id,
														&lct->lct_entry[i] );

				if(status) {

					PDEBUG(DBG_LCT,"Could not initialize target, quitting\n");
					
					uerrno = (UL_ERROR | UP_INIT | US_SYS |
					       	transport | 
								UO_CMN | UC_NO_EXT);

					PERROR("%s: ERROR: %s [%x] \n", 
						pHba->name,
						uni_err(uerrno), 
						(U32) uerrno);
					
					ERROR_END;
					return -1;
				}
			} // if isexisting target
			else {

				PDEBUG(DBG_LCT, "new target, mapping to a free slot\n");
	
				// target is new, assign it to a free slot
				pTarget = adpt_map_new_target(pHba, &lct->lct_entry[i],
												0xFF,0xFF);

				if(pTarget == NULL) {
				
					PDEBUG(DBG_LCT,"Could not map new target\n");

					return -1;
				}
			}
		} // if this lct entry is for a target
	}// end for all entries in lct

	PDEBUG(DBG_LCT,"Second scan over \n");
	
	// now we have to go check our internal list and remove items we
	// did not see this time. 

	channel_no = 0;
	scsi_id = 0;

	for(channel_no = 0;channel_no < MAX_CHANNEL; channel_no++) {
		for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
			if((pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {

				// there is a valid entry in our target list, check
				// the flag to see if we're ok to remove it
				if(pTarget->state & ADPT_DEV_UNSCANNED) {

					PDEBUG(DBG_LCT,"Marking target %s offline",
								pTarget->iscsi_name);
			
					// mark this target deleted,
					// how to inform the OS about this??
					// CR: mark them different from OFFLINE - i.e ABSENT. 
					// If absent, dont initiate session recovery on this one.

					// pTarget->state |= ADPT_DEV_OFFLINE;
					pTarget->state |= ADPT_DEV_ABSENT;

					// mark all LUNs behind this target as offline
					if(pTarget->device) {

						for(pDev=pTarget->device;pDev;pDev=pDev->next_lun) {
							pDev->state |= ADPT_DEV_OFFLINE;
							pDev->state |= ADPT_DEV_ABSENT;
							pDev->state &= ~ADPT_DEV_UNSCANNED;
						}
					}
				} // if unscanned
			} // if target is present
		} // end for scsi id
	} // end for channel no

	channel_no = 0;
	no_of_devs = 0;
	
	/*
	 *  Lets examine LUNs
	 */
	// CR: Handle case where target is offline and LUN is online.
	for (i = 0; i < max; i++) {
		if (lct->lct_entry[i].class_id  == I2O_CLASS_RANDOM_BLOCK_STORAGE ||
		   lct->lct_entry[i].class_id  == I2O_CLASS_SCSI_PERIPHERAL ||
		   lct->lct_entry[i].class_id  == I2O_CLASS_FIBRE_CHANNEL_PERIPHERAL)
		{
			if(lct->lct_entry[i].device_flags & I2O_LCT_DEVICE_FLAGS_ABSENT) {

				continue;
			}

			tid = lct->lct_entry[i].tid;

			parent_tid = lct->lct_entry[i].parent_tid;

 			if (lct->lct_entry[i].class_id == I2O_CLASS_SCSI_PERIPHERAL) {
				PDEBUG(DBG_LCT, "SCSI PERIPHERAL CLASS FOUND "
						" WITH TID = %x, PARENT TID = "
						"%x SUB CLASS = %d Flags %x\n", 
						tid, parent_tid,
						lct->lct_entry[i].sub_class,
						lct->lct_entry[i].device_flags);
			}

			scsi_lun = lct->lct_entry[i].sub_class & I2O_LCT_SUBCLASS_MASK;
			
			pTarget = adpt_get_target_from_lun_lct(pHba, &lct->lct_entry[i]);
		
			if (pTarget == NULL) {

				PDEBUG(DBG_LCT, "Could not find target that "
						"match the LUN device's "
						"Parent TID !!\n");

				PDEBUG(DBG_LCT,"Ignoring this LUN...\n");

				continue;

			}

			// From CR: If LUN is offline and target is online, then only this LUN
			// has to be marked absent.

			if(lct->lct_entry[i].device_flags & I2O_LCT_DEVICE_FLAGS_OFFLINE) {
				if(!(pTarget->state & ADPT_DEV_OFFLINE))  {
					
					// dont consider this LUN, it will be marked absent and session
					// recovery code will not trigger because of this LUN not being
					// present

					PDEBUG(DBG_LCT,"Skipping LUN with TID %x since its parent "
										"is present. This LUN will be marked ABSENT "
										"%d\n", lct->lct_entry[i].tid);
					continue;
				}
			}
					
			status = adpt_initialize_lun_params(pHba, pTarget, 
												&lct->lct_entry[i], &pDev);

			if(status) {

				PDEBUG(DBG_LCT,"Could not initialize LUN params "
								"for LUN tid %d target tid %d\n",
									lct->lct_entry[i].tid, parent_tid);
				ERROR_END;
				
				return status;
			}

			no_of_devs++; 
			
			if (scsi_lun > pHba->top_scsi_lun) {
				pHba->top_scsi_lun = scsi_lun;
			}

			// from CR: check LUN count, not LUN #
			if(no_of_devs > MAX_LUN) {

				PDEBUG(DBG_LCT,"Cannot add anymore LUN entries, breaking out "
									"of reparse\n");
				break;
			}
		}
	}

	pHba->no_of_devs = no_of_devs;

	if (pHba->no_of_devs == 0) {
		PWARN("\nNo device(s) found !\n");
	}

	// mark any LUNs that were not scanned as offline
	// CR: mark unscanned entries as absent
	adpt_rescan_device(pHba);
#ifdef ASA72XX_DEBUG
	// just traverse the list we have for debug...
	for(channel_no = 0; channel_no< MAX_CHANNEL; channel_no++) {
		for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
			if( (pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {
				PDEBUG(DBG_REPARSE,"		Target Details	\n");
				PDEBUG(DBG_REPARSE,"Name\t: %s\n",pTarget->iscsi_name);
				PDEBUG(DBG_REPARSE,"B T\t: %d %d\n",pTarget->channel_no, 
																pTarget->scsi_id);
				PDEBUG(DBG_REPARSE,"TID: %d\n",pTarget->tid);
				PDEBUG(DBG_REPARSE,"Flags\t: %x\n",pTarget->state);
			
				if(pTarget->device) {
					for(pDev=pTarget->device;pDev;pDev=pDev->next_lun) {
						PDEBUG(DBG_REPARSE,"		LUN Details	\n");
						PDEBUG(DBG_REPARSE,"BTL\t: %d %d %d\n",pDev->scsi_channel,
											 	pDev->scsi_id, pDev->scsi_lun);
						PDEBUG(DBG_REPARSE,"tid: %d\n",pDev->tid);
						PDEBUG(DBG_REPARSE,"parent tid\t: %d\n",pDev->parent_tid);
						PDEBUG(DBG_REPARSE,"Flags\t: %x\n",pTarget->state);
					}
				}
			}
		}
	}
#endif
	/*
	 * Let the FW know the latest BTL MAP
	 * */

    // firmware fix for B25, safe to enable

	if(sync_up_bt_with_fw(pHba)){
		PDEBUG(DBG_REPARSE, "SYNC UP BT WITH FW FAILED");
		ERROR_END;
		return -1;
	}

	END;

	return 0;
}
			


/* 
 * set_btl_into_fw()
 *
 * Description : This will set the BTL information on to the firmware
 */ 		  
S32	
set_btl_into_fw(adpt_hba *pHba)
{
	U32 				msg[MAX_MESSAGE_SIZE/4];
	U32 				msg_size;
	U32 				sg_count = 0;
	U32 				rcode = 0;
	U32				status;
	U16 				size;

	BEGIN;
	
	/* allocate kernel space for message IOCTL if this was not done
	 * If we have already obtained this list, then dont do this again*/
	if(pHba->pdevice_info_ioctl){
		pHba->pdevice_info_ioctl->ioctlheader.ioctlresponseheader.pagecode = GET_DEVICE_LIST;
		pHba->pdevice_info_ioctl->ioctlheader.ioctlresponseheader.rwbit = IOCTL_SET_DATA;
		pHba->pdevice_info_ioctl->ioctlheader.ioctlresponseheader.pagelength = sizeof(ioctl_request_header) + sizeof(pHba->pdevice_info_ioctl->device_list_info.number_of_devices) + (pHba->pdevice_info_ioctl->device_list_info.number_of_devices * sizeof(lun_entry));
		pHba->pdevice_info_ioctl->ioctlheader.ioctlresponseheader.devicehandle = HBA_HANDLE;
		size = pHba->pdevice_info_ioctl->ioctlheader.ioctlresponseheader.pagelength;  
/*		printk("\nThe page length passed is %d",size); */
		
		sg_count = size;

		/* fill I2O message */
		msg[0] = I2O_MESSAGE_SIZE(11) | SGL_OFFSET_9;
		msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
		msg[2] = 0x40000000;	/* Initiator Context (IOCTL context) */
		msg[3] = (U32)msg;	/* Target Context (we want the msg back) */
		msg[4] = (DPT_ORGANIZATION_ID << 16) | ASA72XX_MGMT_IOCTL_CMD;
		msg[5] = 0x00010000;	/* interpret bit set to indicate as IOCTL */
		msg[6] = size;			/* number of bytes to be transferred */
		msg[7] = 0xCD000000 | (1*8); 	/* one SG element always present */
		msg[8] = 0;		/* buffer context */

		/* fill up the SG list */
		msg[9] = pHba->pdevice_list_bus_addr;
		msg[10] = sg_count;
	
		/* message size in U32 words */
		msg_size = 11;

		/* message size in U8 bytes */
		msg_size = 11 << 2;

		if ((rcode = adpt_i2o_post_wait(pHba, msg, msg_size, FOREVER))) {
			PDEBUG(DBG_REPARSE, "%s: Unable to get Target Info "
				"(status=%#10x).\n", 
				pHba->name, rcode);
		
			ERROR_END;
			return (-1);
		}
	
		if ((status = pHba->pdevice_info_ioctl->ioctlheader.ioctlresponseheader.status)) {	
			PDEBUG(DBG_REPARSE, "The IOCTL Failed to SET the DATA");
			ERROR_END;
			return (-1); 
		}else{
			PDEBUG(DBG_REPARSE, "The IOCTL Succeded to SET the DATA");
		}
	
	}

	END;
	return(0);
}

/* 
 * sync_up_bt_with_fw()
 *
 * Description : This will get the list of LUNs. If the B and T are
 * set to 0xffff, then we try to parse up the BT table for a LUN information
 * If we match up a TID, then appropriate Bus and Target are replaced
 *
 */ 		  
S32
sync_up_bt_with_fw(adpt_hba *pHba)
{
	U32 				msg[MAX_MESSAGE_SIZE/4];
	U32 				msg_size=0;
	U32 				sg_count = 0;
	U32 				rcode = 0;
	U32				status;
	U16 				size=0;
	U16				i=0;
	adpt_device			*pDev = NULL; 
	adpt_target			*pTarget = NULL; 
	U32				tid = 0;
	U8  				tid_found = 0;
	U8				channel_no = 0;
	U16				scsi_id = 0;

	BEGIN;

	/* allocate kernel space for message IOCTL if this was not done
	 * If we have already obtained this list, then dont do this again
	 * Note that this memory is deallocated while we delete the node*/
	if(!pHba->pdevice_info_ioctl){
		pHba->pdevice_info_ioctl = pci_alloc_consistent(pHba->pDev, 
				sizeof(get_device_list),
				&pHba->pdevice_list_bus_addr);
			
		if (pHba->pdevice_info_ioctl == NULL) {
			PDEBUG(DBG_REPARSE, "%s: Could not allocate message buffer\n",
					pHba->name);
			uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN | 
					UC_NO_EXT | U_MALLOC_FAIL);
			PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);

			rcode = -ENOMEM;
			ERROR_END;
			return (rcode);
		}
	}

	/* clear memory */
	memset(pHba->pdevice_info_ioctl, 0, sizeof(get_device_list));

	pHba->pdevice_info_ioctl->ioctlheader.ioctlrequestheader.pagecode = GET_DEVICE_LIST;
	pHba->pdevice_info_ioctl->ioctlheader.ioctlrequestheader.rwbit = IOCTL_GET_DATA;
	pHba->pdevice_info_ioctl->ioctlheader.ioctlrequestheader.expectedlength = (U16) sizeof(get_device_list);
	pHba->pdevice_info_ioctl->ioctlheader.ioctlrequestheader.devicehandle = HBA_HANDLE;

	size = pHba->pdevice_info_ioctl->ioctlheader.ioctlrequestheader.expectedlength;
	sg_count = size;

	/* fill I2O message */
	msg[0] = I2O_MESSAGE_SIZE(11) | SGL_OFFSET_9;
	msg[1] = (0xff<<24 | HOST_TID<<12 | ADAPTER_TID);
	msg[2] = 0x40000000;	/* Initiator Context (IOCTL context) */
	msg[3] = (U32)msg;	/* Target Context (we want the msg back) */
	msg[4] = (DPT_ORGANIZATION_ID<<16) | ASA72XX_MGMT_IOCTL_CMD; 
	msg[5] = 0x00010000;	/* interpret bit set to indicate as IOCTL */
	msg[6] = size;		/* number of bytes to be transferred */
	msg[7] = 0xCD000000 | (1*8); /* one SG element always present */
	msg[8] = 0;		/* buffer context */

	/* fill up the SG list */
	msg[9] = pHba->pdevice_list_bus_addr;
	msg[10] = sg_count;
	
	/* message size in U32 words */
	msg_size = 11;

	/* message size in U8 bytes */
	msg_size = 11 << 2;

	//pHba->state |= ADPT_STATE_IOCTL;
	
	rcode = adpt_i2o_post_wait(pHba, msg, msg_size, FOREVER);  
	
	if (rcode) {
		PDEBUG(DBG_REPARSE, "%s: Unable to get device list "
				"(status=%#10x).\n", 
				pHba->name, rcode);
		
		pci_free_consistent(pHba->pDev, 
				sizeof(get_device_list), 
				pHba->pdevice_info_ioctl,
				pHba->pdevice_list_bus_addr);

		ERROR_END;
		
		return(-1);
	}
	
	if ((status = pHba->pdevice_info_ioctl->ioctlheader.ioctlresponseheader.status)) {
		pci_free_consistent(pHba->pDev, 
				sizeof(get_device_list), 
				pHba->pdevice_info_ioctl,
				pHba->pdevice_list_bus_addr);

		ERROR_END;
		
		return(-1);
	} 

	/*
	 * Go thru the device mapping table and get the info.
	 * Devices that are offline are skipped. 
	 */ 
#ifdef ASA72XX_DEBUG
	PDEBUG(DBG_REPARSE,"Device Listi from a GET_DATA...\n");
	for (i=0; i < pHba->pdevice_info_ioctl->device_list_info.number_of_devices; ++i) {
		PDEBUG(DBG_REPARSE,"TID\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].tid);
		PDEBUG(DBG_REPARSE,"Port\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].port_number);
		PDEBUG(DBG_REPARSE,"Target: %s\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].target_name);
		PDEBUG(DBG_REPARSE,"Bus\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_bus);
		PDEBUG(DBG_REPARSE,"Target\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_tgt);
		PDEBUG(DBG_REPARSE,"Lun\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_lun);
	}
#endif

	if(pHba->pdevice_info_ioctl->device_list_info.number_of_devices == 0) {
		PDEBUG(DBG_REPARSE,"GET_DEVICE_LIST: Did not find any devices, returning...\n");
		return 0;
	}

	for (i=0; i < pHba->pdevice_info_ioctl->device_list_info.number_of_devices; ++i) {
		tid = pHba->pdevice_info_ioctl->device_list_info.device_list[i].tid;
		tid_found = 0;
		pTarget = NULL;
		for(channel_no = 0; (channel_no < MAX_CHANNEL && !tid_found); channel_no++) {
			for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
				if((pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {
					if(strcmp(pTarget->iscsi_name,
						pHba->pdevice_info_ioctl->device_list_info.device_list[i].target_name) == 0) {
						tid_found = 1;
						break;
					}
				}
			}
		}

		if(!tid_found) {

			PDEBUG(DBG_REPARSE,"!!ERROR!! target found in device list. "
									"But not present in adpt_target list!!\n");
			continue;
		}

		for (pDev = pTarget->device; pDev; pDev = pDev->next_lun) {
			if ((pDev->tid == tid) && 
					!(pDev->state & ADPT_DEV_OFFLINE)) {
					pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_bus = pDev->scsi_channel;
					pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_tgt = pDev->scsi_id;
					// dont overwrite LUN, its already given by the firmware
					//pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_lun = pDev->scsi_lun;
					break;
			}
		}
	}
#ifdef ASA72XX_DEBUG
	PDEBUG(DBG_REPARSE,"Device List for SET_DATA...\n");
	for (i=0; i < pHba->pdevice_info_ioctl->device_list_info.number_of_devices; ++i) {
		PDEBUG(DBG_REPARSE,"TID\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].tid);
		PDEBUG(DBG_REPARSE,"Port\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].port_number);
		PDEBUG(DBG_REPARSE,"Target: %s\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].target_name);
		PDEBUG(DBG_REPARSE,"Bus\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_bus);
		PDEBUG(DBG_REPARSE,"Target\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_tgt);
		PDEBUG(DBG_REPARSE,"Lun\t: %d\n", pHba->pdevice_info_ioctl->device_list_info.device_list[i].os_lun);
	}
#endif
	if(set_btl_into_fw(pHba)){
		PDEBUG(DBG_REPARSE,"\nFailed to set the BTL info on to the Firmware"); 
		ERROR_END;
		return -1;
	}

	END;
	return 0;
}


/*
 * adpt_rescan_device()
 * 
 * Description :
 * 	Scan all the devices for the HBA if any device is offline.
 */
static void
adpt_rescan_device(adpt_hba *pHba)
{
	adpt_target		*pTarget = NULL;
	adpt_device		*pDev = NULL;
	U8			channel_no = 0;
	U16			scsi_id = 0;

	BEGIN;

	for(channel_no = 0; channel_no < MAX_CHANNEL; channel_no++) {
		for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
			if( (pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {
				
				if(pTarget->state & ADPT_DEV_UNSCANNED) {
					
					PDEBUG(DBG_LCT, "%s: Target(%d,%d) absent\n",
											pHba->name, 
											pTarget->channel_no,
											pTarget->scsi_id);

					pTarget->state |= ADPT_DEV_ABSENT;
					pTarget->state |= ADPT_DEV_OFFLINE;
					pTarget->state &= ~ADPT_DEV_UNSCANNED;
				}

				for(pDev = pTarget->device; pDev; pDev=pDev->next_lun) {

					if(pDev->state & ADPT_DEV_UNSCANNED) {
					
						PDEBUG(DBG_LCT, "%s: Device (%d,%d,%d)	absent\n",
											pHba->name, 
											pDev->scsi_channel,
											pDev->scsi_id, 
											pDev->scsi_lun);

						pDev->state |= ADPT_DEV_ABSENT;
						pDev->state |= ADPT_DEV_OFFLINE;
						pDev->state &= ~ADPT_DEV_UNSCANNED;
					
						if (pDev->pScsi_dev) {
							pDev->pScsi_dev->online = FALSE;
			
							PWARN("Device at Channel %d "
										"Id %d Lun %d is "
										"absent. \n",
										pDev->scsi_channel,
										pDev->scsi_id,
										pDev->scsi_lun);
						}
					}
				}
			}
		}
	}

	END;
}


/*	
 * adpt_i2o_issue_parms()
 * 
 * Description: 
 *	Issue UTIL_PARAMS_GET or UTIL_PARAMS_SET
 *
 *	This function can be used for all UtilParamsGet/Set operations.
 *	The OperationBlock is given in opblk-buffer, 
 *	and results are returned in resblk-buffer.
 *	Note that the minimum sized resblk is 8 bytes and contains
 *	ResultCount, ErrorInfoSize, BlockStatus and BlockSize.
 */
S32 
adpt_i2o_issue_params(S32 cmd, adpt_hba *pHba, S32 tid, void *opblk, 
				S32 oplen, void *resblk, S32 reslen)
{
	U32	msg[(cmd == I2O_CMD_UTIL_PARAMS_GET) ? 9 : 7];
	S32 	rcode = 0;

	BEGIN;
	
	if (cmd == I2O_CMD_UTIL_PARAMS_GET) {		
		msg[0] = NINE_WORD_MSG_SIZE | SGL_OFFSET_5;
		msg[1] = cmd << 24 | HOST_TID << 12 | tid; 
		msg[2] = 0;			/* Initiator Context */
		msg[3] = 0;			/* Transaction Context */
		msg[4] = 0;			/* Operation Flags */
		msg[5] = 0x54000000 | oplen;
		msg[6] = virt_to_bus(opblk); 	/* Operation Block */
		msg[7] = 0xD0000000 | reslen;
		msg[8] = virt_to_bus(resblk);	/* Result Block */
	} else {
		msg[0] = SEVEN_WORD_MSG_SIZE | SGL_OFFSET_5;
		msg[1] = cmd << 24 | HOST_TID << 12; 
		msg[2] = 0;			/* Initiator Context */
		msg[3] = 0;			/* Transaction Context */
		msg[4] = 0;			/* Operation Flags */
		msg[5] = 0xD0000000 | oplen;
		msg[6] = virt_to_bus(opblk); 	/* Operation Block */
	}
		
	rcode = adpt_i2o_post_wait(pHba, msg, sizeof(msg), TMOUT_UTIL_PARAMS);

	if (rcode != 0) {
		PDEBUG(DBG_LCT, "Status returned is 0x%x !!\n", rcode);

   		return rcode; 
	}

	if (cmd == I2O_CMD_UTIL_PARAMS_GET) { 	
		U32 	*res = (U32 *)resblk;
		
		if (res[1] & 0x00FF0000) {
			/* BlockStatus != SUCCESS */
			PDEBUG(DBG_DEVICE, "%s: Error:\n  ErrorInfoSize = "
					"0x%02x,"
					"BlockStatus = 0x%02x, "
					"BlockSize = 0x%04x\n",
					pHba->name,
					res[1] >> 24, (res[1] >> 16) & 0xFF, 
					res[1] & 0xFFFF);
		}
	}

	END;
	
	return 0;
}


/*
 * adpt_fail_posted_scbs()
 *
 * Description:
 * 	Fail all the commands in the Device Queue to upper layer with status
 *	of QUEUE FULL.  	  
 */  
void 
adpt_fail_posted_scbs(adpt_hba *pHba)
{
	Scsi_Cmnd	*cmd = NULL;
	Scsi_Device	*d = NULL;

	BEGIN;
	
	if (pHba->host->host_queue != NULL) {
		if ((d = pHba->host->host_queue) == NULL) {
			return;
		}
		
		while (d->next != NULL) {
			for (cmd = d->device_queue; cmd ; cmd = cmd->next) {
				if (cmd->serial_number == 0) {
					continue;
				}
				
				cmd->result = (DID_OK << 16) | (QUEUE_FULL <<1);

				cmd->scsi_done(cmd);
			}

			d = d->next;
		}
	}

	END;
}


/*
 * adpt_delay()
 */  
static void 
adpt_delay(S32 millisec)
{
	S32 i;

	for (i = 0; i < millisec; i++) {
		udelay(1000);	/* delay for one millisecond */
	}
}
	

/*
 * parse_cmdline_params()
 *
 * Description:   
 * 	This function parses command line parameters that 
 *  	are passed during the module loading.
 *
 * 	The 'insmod' command line format is:
 *    		insmod driver.o driver="boot parameters"
 *
 *  	For Etherstorage drivers currently we pass machine name
 *
 *    	Example: insmod asa72xx.o asa72xxc="MachineName:$(HOSTNAME)"
 */
void 
parse_cmdline_params(S8 *str, S32 *ints)
{
	S32	base = 0;
	S32 	val;
	S8 	*cur = str;
	S8 	*pc, *pv;
	S8 	arg_sep = '\0';

	BEGIN;

	PDEBUG(DBG_INIT, "Cmdline param = %s\n", str);
	
	if (asa72xxc)
	{
		arg_sep = ' ';
	}

	while (*cur == ' ') {	
		cur++;
	}
		
	while (cur != NULL && (pc = strchr(cur, ':')) != NULL) {
		val = 0;
		pv = pc;
		pv++;
		val = (S32) simple_strtoul(pv, NULL, base);
			
		if (!strncmp(cur, "MachineName:", 
				sizeof("MachineName"))) {
				strcpy(pSetup->machine_name, pv);
		
			PDEBUG(DBG_INIT, "MachineName:%s\n", 
						pSetup->machine_name);
		} else if (!strncmp(cur, "DeviceQueueDepth:", 
					sizeof("DeviceQueueDepth"))) {
			if (val > ASA72XX_MAX_DEV_Q_DEPTH) {
				/* Device Queue Depth */
				pSetup->device_q_depth = 
					ASA72XX_MAX_DEV_Q_DEPTH;
			} else {
					pSetup->device_q_depth = val;

			}

			PDEBUG(DBG_INIT, "DeviceQueueDepth:%d\n",
					pSetup->device_q_depth);
		} else if (!strncmp(cur, "Bind:", 
				sizeof("Bind"))) {
                 {
                       {
	S8 	*token = '\0';

			PDEBUG(DBG_INIT,"Bind:%s\n",
					pv);
                        token=strtok(pv,"-");

			strcpy(pSetup->persistent_name, token);
			PDEBUG(DBG_INIT,"Bind:iscsiname %s\n",
					token);

                        /* Lets skip -host< */
                        token=strtok(NULL,"<");
                        token=strtok(NULL,">");
                        pSetup->persistent_host_no = (U8) 
		                simple_strtoul(token, NULL, base);
			PDEBUG(DBG_INIT,"Bind:host # %d\n",
                               pSetup->persistent_host_no);

                        /* Lets skip -chan< */
                        token=strtok(NULL,"<");
                        token=strtok(NULL,">");
                        pSetup->persistent_channel_no = (U8) 
		                simple_strtoul(token, NULL, base);
			PDEBUG(DBG_INIT,"Bind:channel # %d\n",
                               pSetup->persistent_channel_no);

                        /* Lets skip -scsi< */
                        token=strtok(NULL,"<");
                        token=strtok(NULL,">");
                        pSetup->persistent_scsi_id = (U8) 
		                simple_strtoul(token, NULL, base);
			PDEBUG(DBG_INIT,"Bind:targetid # %d\n",
                               pSetup->persistent_scsi_id);
                       }
           }
		} else if (!strncmp(cur, "DiskTimeout:", 
					sizeof("DiskTimeout"))) {
			if ((val > TMOUT_SCSI) || (val < 0)) {
				pSetup->command_timeout = TMOUT_SCSI;
			} else {
				pSetup->command_timeout = val;
			}

			PDEBUG(DBG_INIT, "DiskTimeout:%d\n",
					pSetup->command_timeout);
		} else if (!strncmp(cur, "IB:", sizeof("IB"))) {
			if ((val > 32) || (val < 0)) {
				pSetup->interrupt_batching = IB_DEFAULT;
			} else {
				pSetup->interrupt_batching = val;
			}

			PDEBUG(DBG_INIT, "InterruptBatching: %d \n",
					pSetup->interrupt_batching);
		} else {
			PWARN("asa72xx: boot option not recognized"
					" '%.*s'\n",
					(S32)(pc - cur + 1), cur);
		}				 	

		if ((cur = strchr(cur, arg_sep))  != NULL) {
			++cur;
		}
	}
	
	setup_done = 1;

	END;
}


/* 
 * getcontrollerattributes()
 *
 * Description :
 * 	Set up an I2O message to get the controller atrributes , such as:
 * 	machine name, PCI vendor/device id and etc from the HBA.
 */ 		  
S32	
getcontrollerattributes(adpt_hba *pHba)
{
	U32 				msg[MAX_MESSAGE_SIZE/4];
	U32 				msg_size;
	U32 				sg_count = 0;
	U32 				rcode = 0;
	U32				status;
	U16 				size;
	pget_controller_attributes	pgetctlrattrib;
	dma_addr_t			pgetctlrattrib_bus_addr;

	BEGIN;

	/* allocate kernel space for message IOCTL */
	pgetctlrattrib = pci_alloc_consistent(pHba->pDev, 
				sizeof(get_controller_attributes),
				&pgetctlrattrib_bus_addr);
			
	if (pgetctlrattrib == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate message buffer\n",
					pHba->name);
		
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN | 
					UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);

		rcode = -ENOMEM;
		
		return (rcode);
	}
	
	/* clear memory */
	memset(pgetctlrattrib, 0, sizeof(get_controller_attributes));

	pgetctlrattrib->ioctlheader.ioctlrequestheader.pagecode = 
		CONTROLLER_ATTRIBUTES;
	
	pgetctlrattrib->ioctlheader.ioctlrequestheader.rwbit = IOCTL_GET_DATA;
	
	pgetctlrattrib->ioctlheader.ioctlrequestheader.expectedlength = 
		(U16) sizeof(get_controller_attributes);
	
	pgetctlrattrib->ioctlheader.ioctlrequestheader.devicehandle = 
		HBA_HANDLE;

	size = pgetctlrattrib->ioctlheader.ioctlrequestheader.expectedlength;

	sg_count = size;

	/* fill I2O message */
	msg[0] = I2O_MESSAGE_SIZE(11) | SGL_OFFSET_9;
	msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
	msg[2] = 0x40000000;	/* Initiator Context (IOCTL context) */
	msg[3] = (U32) msg;	/* Target Context (we want the msg back) */
	msg[4] = (DPT_ORGANIZATION_ID << 16) | ASA72XX_MGMT_IOCTL_CMD;
	msg[5] = 0x00010000;	/* interpret bit set to indicate as IOCTL */
	msg[6] = size;			/* number of bytes to be transferred */
	msg[7] = 0xCD000000 | (1*8); 	/* one SG element always present */
	msg[8] = 0;		/* buffer context */

	/* fill up the SG list */
	msg[9] = pgetctlrattrib_bus_addr;
	msg[10] = sg_count;
	
	/* message size in U32 words */
	msg_size = 11;

	/* message size in U8 bytes */
	msg_size = 11 << 2;

	//pHba->state |= ADPT_STATE_IOCTL;
	
	if ((rcode = adpt_i2o_post_wait(pHba, msg, msg_size, 
					TMOUT_GET_CTRL_ATTRIB))) {
		PDEBUG(DBG_POST, "%s: Unable to get controller attributes "
				"(status=%#10x).\n", 
				pHba->name, rcode);
		
		pci_free_consistent(pHba->pDev, 
				sizeof(get_controller_attributes),
				pgetctlrattrib,
				pgetctlrattrib_bus_addr);
	
		return (-1);
	}
	
	if ((status = pgetctlrattrib->ioctlheader.ioctlresponseheader.status)) {
		return (-1); 
	}

	strcpy(pHba->machine_name, 
			pgetctlrattrib->getctlrattribreply.userassignedname);
	
	PDEBUG(DBG_IOCTL, "\n user assigned name : %s\n", 
			pgetctlrattrib->getctlrattribreply.userassignedname);
	PDEBUG(DBG_IOCTL, "\n serial number: %s\n", 
			pgetctlrattrib->getctlrattribreply.controllerserialnumber);

	pci_free_consistent(pHba->pDev, sizeof(get_controller_attributes),
				pgetctlrattrib,
				pgetctlrattrib_bus_addr);
	
	if (getsettime(pHba) < 0) {
		PDEBUG(DBG_IOCTL, "Unable to send current system time "
				"to FW !\n");
	}

	//pHba->state &= ~ADPT_STATE_IOCTL;

	END;

	return(0);
}

/*
 * getsettime()
 *
 * Description:
 * 	Send IOCTL_GET_SET_TIME to report the current system time to the FW,
 * 	so that proper time stamp will be used for Event Log in iConfig. 	
 */
S32
getsettime(adpt_hba *pHba)
{
	pioctl_get_set_time	pgetsettime;
	struct timeval		*tv;
	U32 			msg[MAX_MESSAGE_SIZE/4];
	U32 			msg_size;
	U32 			sg_count = 0;
	U32 			rcode = 0;
	U32			status;
	U32			TimeInSecs = 0;
	U16 			size = 0;
	dma_addr_t		pgetsettime_bus_addr;		
	
	BEGIN;

	pgetsettime = pci_alloc_consistent(pHba->pDev, 
				sizeof(ioctl_get_set_time), 
				&pgetsettime_bus_addr);

	if (pgetsettime == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate message buffer\n",
					pHba->name);
		
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN | 
					UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);

		rcode = -ENOMEM;
		
		return (rcode);
	}
	
	/* clear memory */
	memset(pgetsettime, 0, sizeof(ioctl_get_set_time));
	
	tv = kmalloc(sizeof(struct timeval), GFP_ATOMIC | ADDR32);

	if (tv == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate message buffer\n",
					pHba->name);

		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN | 
					UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);

		rcode = -ENOMEM;

		return (rcode);
	}

	memset(tv, 0, sizeof(struct timeval));

	/* 
	 * get the current system time in secs 
	 */  
	do_gettimeofday(tv);

	TimeInSecs = (U32) tv->tv_sec;
	
	PDEBUG(DBG_IOCTL, "Current System Time in Secs = %d\n", TimeInSecs);
	
	pgetsettime->ioctlheader.ioctlresponseheader.pagecode = 
		GET_SET_TIME;
	
	pgetsettime->ioctlheader.ioctlresponseheader.rwbit = IOCTL_SET_DATA;
	
	pgetsettime->ioctlheader.ioctlresponseheader.pagelength = 
		sizeof(ioctl_get_set_time);
	
	pgetsettime->ioctlheader.ioctlresponseheader.devicehandle = 
		HBA_HANDLE;

	pgetsettime->time = TimeInSecs;

	size = pgetsettime->ioctlheader.ioctlresponseheader.pagelength;  
		
	sg_count = size;

	/* fill I2O message */
	msg[0] = I2O_MESSAGE_SIZE(11) | SGL_OFFSET_9;
	msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
	msg[2] = 0x40000000;	/* Initiator Context (IOCTL context) */
	msg[3] = (U32)msg;	/* Target Context (we want the msg back) */
	msg[4] = (DPT_ORGANIZATION_ID << 16) | ASA72XX_MGMT_IOCTL_CMD;
	msg[5] = 0x00010000;	/* interpret bit set to indicate as IOCTL */
	msg[6] = size;			/* number of bytes to be transferred */
	msg[7] = 0xCD000000 | (1*8); 	/* one SG element always present */
	msg[8] = 0;		/* buffer context */

	/* fill up the SG list */
	msg[9] = pgetsettime_bus_addr;
	msg[10] = sg_count;
	
	/* message size in U32 words */
	msg_size = 11;

	/* message size in U8 bytes */
	msg_size = 11 << 2;

	if ((rcode = adpt_i2o_post_wait(pHba, msg, msg_size, FOREVER))) {
		PDEBUG(DBG_POST, "%s: Unable to set current time "
				"(status=%#10x).\n", 
				pHba->name, rcode);
		
		kfree(tv);

		pci_free_consistent(pHba->pDev, sizeof(ioctl_get_set_time), 
				pgetsettime, pgetsettime_bus_addr);

		return (-1);
	}
	
	if ((status = pgetsettime->ioctlheader.ioctlresponseheader.status)) {	
		return (-1); 
	}
	
	kfree(tv);

	pci_free_consistent(pHba->pDev, sizeof(ioctl_get_set_time), 
			pgetsettime, pgetsettime_bus_addr);
	
	END;

	return (0);
}	
	

/*
 * sendhostverifydone()
 *
 * Description:
 *	Send a IOCTL message to FW to notify if the MachineName is matched or
 *	not.	   
 */  
S32	
sendhostverifydone(adpt_hba *pHba, U8 mach_name_mismatched)
{
	U32 				msg[MAX_MESSAGE_SIZE/4];
	U32 				msg_size;
	U32 				sg_count = 0;
	U32 				rcode = 0;
	U32				status;
	U16 				size;
	phost_config_verification	phostconfigverify;
	dma_addr_t			phostconfigverify_bus_addr;

	BEGIN;

	phostconfigverify = pci_alloc_consistent(pHba->pDev, 
				sizeof(host_config_verification), 
				&phostconfigverify_bus_addr);
	
	if (phostconfigverify == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate message buffer\n",
					pHba->name);
		
		uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | UO_CMN | 
					UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);

		ERROR_END;

		return (-ENOMEM);
	}
	
	memset(phostconfigverify, 0, sizeof(host_config_verification));

	phostconfigverify->ioctlheader.ioctlrequestheader.pagecode = 
			HOST_CONFIG_VERIFY;
	
	phostconfigverify->ioctlheader.ioctlrequestheader.rwbit = 
			IOCTL_SET_DATA;
	
	phostconfigverify->ioctlheader.ioctlresponseheader.pagelength = 
			sizeof(host_config_verification);
	
	phostconfigverify->ioctlheader.ioctlrequestheader.devicehandle = 
			HBA_HANDLE;

	phostconfigverify->VerificationMismatched = mach_name_mismatched;
		
	size = phostconfigverify->ioctlheader.ioctlresponseheader.pagelength;
	
	sg_count = size;

	/* fill I2O message */
	msg[0] = I2O_MESSAGE_SIZE(11) | SGL_OFFSET_9;
	msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
	msg[2] = 0x40000000;	/* Initiator Context (IOCTL context) */
	msg[3] = (U32) msg;	/* Target Context (we want the msg back) */
	msg[4] = (DPT_ORGANIZATION_ID<<16) | ASA72XX_MGMT_IOCTL_CMD;
	msg[5] = 0x00010000;	/* interpret bit set to indicate as IOCTL */
	msg[6] = size;		/* number of bytes to be transferred */
	msg[7] = 0xCD000000 | (1*8); /* one SG element always present */
	msg[8] = 0;		/* buffer context */

	/* fill up the SG list */
	msg[9] = phostconfigverify_bus_addr;
	msg[10] = sg_count;
	
	/* message size in U32 words */
	msg_size = 11;

	/* message size in U8 bytes */
	msg_size = 11 << 2;
	
	//pHba->state |= ADPT_STATE_IOCTL;
	
	if ((rcode = adpt_i2o_post_wait(pHba, msg, msg_size, FOREVER))) {
		PDEBUG(DBG_POST, "%s: Unable to send host config verification "
				"(status=%#10x).\n", 
				pHba->name, rcode);
		
		pci_free_consistent(pHba->pDev, 
				sizeof(host_config_verification), 
				phostconfigverify,
				phostconfigverify_bus_addr);
		
		ERROR_END;

		return (-1);
	}
	
	pHba->verifydone = 
		phostconfigverify->ioctlheader.ioctlresponseheader.status;
	
	PDEBUG(DBG_INIT, "Verifydoneflag %x\n", pHba->verifydone);
	
	status = phostconfigverify->ioctlheader.ioctlresponseheader.status;
	
	if ((status != 0) && (status != 0x87)) {
		pci_free_consistent(pHba->pDev, 
				sizeof(host_config_verification), 
				phostconfigverify,
				phostconfigverify_bus_addr);

		ERROR_END;

		return (-1); 
	}
	
	pci_free_consistent(pHba->pDev, sizeof(host_config_verification), 
			phostconfigverify, phostconfigverify_bus_addr);

	END;

	return (0);
}


/*
 * validatehostconfiguration()
 *
 * Description:  
 * 	This function validates the host configuration by
 * 	comparing the host name that has been passed as
 * 	command line string with what is there on the EEPROM.
 * 	If the string matches then it will sendhostconfigverify()
 * 	command to the f/w to initialize the f/w.
*/
S32 
validatehostconfiguration(adpt_hba *pHba)
{
	U32	oldstate;	
	U8	mach_name_mismatched;
	
	BEGIN;
	
	oldstate = pHba->state;

	if (getcontrollerattributes(pHba) < 0) {	
		ERROR_END;

		return -1;
	}

	PDEBUG(DBG_IOCTL, "Firmware Name \"%s\" ?= Machine Name \"%s\" %d\n",
	                     pHba->machine_name,pSetup->machine_name,
		!strcmp(pSetup->machine_name, pHba->machine_name)?TRUE:FALSE);

        #if BOOTDISK
                        /* for OS installation skip the check and say host ok */ 
			mach_name_mismatched = 0;
			pHba->hostverifydoneflag = 1;
	                PINFO("BOOTDISK mode\n");
			if ((sendhostverifydone(pHba, mach_name_mismatched)) 
					< 0) {
				ERROR_END;

				return -1;
                        }
	pHba->state = oldstate;
	END;

	return (0);
        #endif

	if ((pHba->machine_name[0]) && (pSetup->machine_name[0])) {
		if (!strcmp(pSetup->machine_name, pHba->machine_name)) {
			/*
			 * MachineName passed in by the user matched with
			 * the one stored in HBA's EEPROM.
			 */   
			mach_name_mismatched = 0;

			if ((sendhostverifydone(pHba, mach_name_mismatched)) 
					< 0) {
				ERROR_END;

				return -1;
			}
			
			pHba->hostverifydoneflag = 1;
		} else {
			uerrno = (UL_WARN | UP_INIT | US_HW | transport | 
					UO_LINUX | UC_NO_EXT | 
					U_HA_MACH_NAME_MISMATCH);

			PERROR("%s: ERROR: %s for Host Adapter %d [%x] \n", 
					pHba->name, uni_err(uerrno), 
					pHba->unit + 1, (U32) uerrno);

			/*
			 * MachineName passed in by the user mismatched with
			 * the one stored in HBA's EEPROM.
			 */   
			mach_name_mismatched = 1;

			if ((sendhostverifydone(pHba, mach_name_mismatched)) 
					< 0) {
				ERROR_END;

				return -1;
			}
		}
	} else {
		if (!(pHba->machine_name[0])) {
			uerrno = (UL_WARN | UP_INIT | US_HW | transport | 
					UO_LINUX | UC_NO_EXT | 
					U_HA_NO_MACH_NAME);

			PERROR("%s: ERROR: %s for Host Adapter %d [%x] \n", 
					pHba->name, uni_err(uerrno), 
					pHba->unit + 1, (U32) uerrno);
		} else {
			if (!(pSetup->machine_name[0])) {
				uerrno = (UL_WARN | UP_INIT | US_DRV | 
						transport | UO_LINUX | 
						UC_NO_EXT | 
						U_LX_INVALID_OPTION);

				PERROR("%s: ERROR: %s for Host "
						"Adapter %d [%x] \n", 
					pHba->name, uni_err(uerrno), 
					pHba->unit + 1, (U32) uerrno);
			}
		}
	}

	pHba->state = oldstate;

	if (pHba->verifydone != 87) {
		adpt_delay(TEN_SECONDS);
	}
		
	END;

	return (0);
}


/*
 * getbtlmapping()
 *
 * Description:
 *	Send out "GET_DEVICE_LIST" ioctl to get the Bus, Target and Lun
 * 	mapping of target devices.	  
 */  
S32	
getbtlmapping(adpt_hba *pHba, U32 *user_msg)
{
	ptarget_lun_list		ptargetlunlist;
	pioctl_get_device_list_reply	pgetdevlist;
	adpt_device			*pDev = NULL; 
	adpt_target			*pTarget = NULL; 
	U32 				msg[MAX_MESSAGE_SIZE/4];
	U32 				msg_size;
	U32 				sg_count = 0;
	U32				rcode = 0;
	U32				status;
	U32				tid = 0;
	U32				no_of_devs_found = 0;
	U16 				size;
	U16				numluns = 0;
	U16				i;
	U8  				tid_found = 0;
	dma_addr_t			pgetdevlist_bus_addr;
	U8				channel_no = 0;
	U16				scsi_id = 0;
	
	BEGIN;

	/* 
	 * allocate kernel space for get device list message IOCTL 
	 */
	pgetdevlist = pci_alloc_consistent(pHba->pDev, 
				sizeof(ioctl_get_device_list_reply), 
				&pgetdevlist_bus_addr);
	
	if (pgetdevlist == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate message buffer\n",
					pHba->name);
		
		uerrno = (UL_ERROR | UP_RUN | US_SYS | transport | UO_CMN | 
					UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);
		
		ERROR_END;

		return (-ENOMEM);
	}
	
	/* clear memory */
	memset(pgetdevlist, 0, sizeof(ioctl_get_device_list_reply));

	pgetdevlist->ioctlheader.ioctlrequestheader.pagecode = GET_DEVICE_LIST;
	pgetdevlist->ioctlheader.ioctlrequestheader.rwbit = IOCTL_GET_DATA;
	pgetdevlist->ioctlheader.ioctlrequestheader.expectedlength = 
		(U16) sizeof(ioctl_get_device_list_reply);
	pgetdevlist->ioctlheader.ioctlrequestheader.devicehandle = HBA_HANDLE;

	size = pgetdevlist->ioctlheader.ioctlrequestheader.expectedlength;
	sg_count = size;

	/* fill I2O message */
	msg[0] = I2O_MESSAGE_SIZE(11) | SGL_OFFSET_9;
	msg[1] = (0xff<<24 | HOST_TID<<12 | ADAPTER_TID);
	msg[2] = 0x40000000;	/* Initiator Context (IOCTL context) */
	msg[3] = (U32)msg;	/* Target Context (we want the msg back) */
	msg[4] = (DPT_ORGANIZATION_ID<<16) | ASA72XX_MGMT_IOCTL_CMD; 
	msg[5] = 0x00010000;	/* interpret bit set to indicate as IOCTL */
	msg[6] = size;		/* number of bytes to be transferred */
	msg[7] = 0xCD000000 | (1*8); /* one SG element always present */
	msg[8] = 0;		/* buffer context */

	/* fill up the SG list */
	msg[9] = pgetdevlist_bus_addr;
	msg[10] = sg_count;
	
	/* message size in U32 words */
	msg_size = 11;

	/* message size in U8 bytes */
	msg_size = 11 << 2;

	//pHba->state |= ADPT_STATE_IOCTL;
	
	rcode = adpt_i2o_post_wait(pHba, msg, msg_size, FOREVER);  
	
	if (rcode) {
		PDEBUG(DBG_POST, "%s: Unable to get device list "
				"(status=%#10x).\n", 
				pHba->name, rcode);
		
		pci_free_consistent(pHba->pDev, 
				sizeof(ioctl_get_device_list_reply), 
				pgetdevlist,
				pgetdevlist_bus_addr);

		ERROR_END;
		
		return(-1);
	}
	
	if ((status = pgetdevlist->ioctlheader.ioctlresponseheader.status)) {
		pci_free_consistent(pHba->pDev, 
				sizeof(ioctl_get_device_list_reply), 
				pgetdevlist,
				pgetdevlist_bus_addr);

		ERROR_END;
		
		return(-1);
	} 
	
	/******************************
	* Fill Get BTL Mapping Structure
	*******************************/
	
	/* allocate kernel space for target lun list IOCTL */
	ptargetlunlist = kmalloc(sizeof(target_lun_list), GFP_KERNEL | ADDR32);	

	if (ptargetlunlist == NULL) {
		PDEBUG(DBG_IOCTL, "%s: Could not allocate message buffer\n",
					pHba->name);

		uerrno = (UL_ERROR | UP_RUN | US_SYS | transport | UO_CMN | 
					UC_NO_EXT | U_MALLOC_FAIL);

		PERROR("%s: ERROR: %s [%x] \n", pHba->name, 
					uni_err(uerrno), (U32) uerrno);

		pci_free_consistent(pHba->pDev, 
				sizeof(ioctl_get_device_list_reply), 
				pgetdevlist,
				pgetdevlist_bus_addr);

		ERROR_END;
		
		return (-ENOMEM);
	}
	
	memset(ptargetlunlist, 0, sizeof(target_lun_list));
	
	PDEBUG(DBG_IOCTL, "Num devices found = %d\n", 
			pgetdevlist->tgtdevlist.numdevices);

	for (i = 0; i < pgetdevlist->tgtdevlist.numdevices; i++) {	
		PDEBUG(DBG_IOCTL, "DevEntry[%d].tid = %d\n", 
				i, pgetdevlist->tgtdevlist.deventry[i].tid);
	}	
	
	/*
	 * Go thru the device mapping table and get the info.
	 * Devices that are offline are skipped. 
	 */ 
	for (i=0; i < pgetdevlist->tgtdevlist.numdevices; i++) {
		tid = pgetdevlist->tgtdevlist.deventry[i].tid;
      tid_found = 0;
      pTarget = NULL;
      for(channel_no = 0; (channel_no < MAX_CHANNEL && !tid_found); channel_no++) {
         for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
            if((pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {
               if(strcmp(pTarget->iscsi_name,
						pgetdevlist->tgtdevlist.deventry[i].TargetName) == 0) {
                  tid_found = 1;
                  break;
               }
            }
         }
      }
      if(!tid_found) {
                                                                                                                             
         PDEBUG(DBG_REPARSE,"!!ERROR!! target found in device list. "
                           "But not present in adpt_target list!!\n");
         continue;
      }


		if (pTarget == NULL) {
			PDEBUG(DBG_IOCTL, "Bogus TID = %d returned "
					"by FW !\n",tid);
			
			continue;
		}

		if (pTarget->state & ADPT_DEV_OFFLINE) {
			PDEBUG(DBG_IOCTL, "The Target (TID=%d) is "
					"OFFLINE !\n", tid);

			continue;
		}
		no_of_devs_found++;

		for (pDev = pTarget->device; pDev; pDev = pDev->next_lun) {
			if ((pDev->tid == tid) && 
					!(pDev->state & ADPT_DEV_OFFLINE)) {
				ptargetlunlist->btlentry[numluns].busno =  
					pDev->scsi_channel;
		
				ptargetlunlist->btlentry[numluns].targetid =  
					pDev->scsi_id;
				
				ptargetlunlist->btlentry[numluns].lun =  
					pDev->scsi_lun;
				ptargetlunlist->tgtdevlist.deventry[numluns].osBus = pDev->scsi_channel;
				ptargetlunlist->tgtdevlist.deventry[numluns].osTgt = pDev->scsi_id;
				ptargetlunlist->tgtdevlist.deventry[numluns].osLun = pDev->scsi_lun;
		
				memcpy(ptargetlunlist->tgtdevlist.deventry[numluns].sra, 
				pgetdevlist->tgtdevlist.deventry[i].sra,8);
		
				memcpy(&ptargetlunlist->tgtdevlist.deventry[numluns].ipaddress,
			       	&pgetdevlist->tgtdevlist.deventry[i].ipaddress,
				sizeof(ip_address_format));

				ptargetlunlist->tgtdevlist.deventry[numluns].portnumber =  
				pgetdevlist->tgtdevlist.deventry[i].portnumber;
		
				ptargetlunlist->tgtdevlist.deventry[numluns].devstatus =  
				pgetdevlist->tgtdevlist.deventry[i].devstatus;
		
				numluns++;
			}
		}

		tid_found = 0;
	}
	
	ptargetlunlist->tgtdevlist.numdevices = no_of_devs_found;
	
	if (copy_to_user((S8 *)(user_msg + (sizeof(ioctl_header) / 4)),
			(S8 *) ptargetlunlist, sizeof(target_lun_list))) {	
		ERROR_END;

		pci_free_consistent(pHba->pDev, 
				sizeof(ioctl_get_device_list_reply), 
				pgetdevlist, pgetdevlist_bus_addr);

		kfree(ptargetlunlist);
	
		return -EFAULT;
	}

	pci_free_consistent(pHba->pDev, sizeof(ioctl_get_device_list_reply), 
			pgetdevlist, pgetdevlist_bus_addr);

	kfree(ptargetlunlist);
	
	//pHba->state &= ~ADPT_STATE_IOCTL;

	END;

	return (0);
}

/*
 * adpt_timer()
 *
 * Description:
 *	Timer tick for the hba. It is recursive because this kernel timer
 *      reschedule itself every 1 sec. It stops registering itself when
 *      shutdown is detected. The timer's responsibility is being incremented
 *      as more functionalities are added.
 */
static void
adpt_timer (U32_64 arg)
{
	struct timer_list	*ptimerlist;
	adpt_hba 	        *pHba;
	adpt_target		*pTarget = NULL;
	U8			channel_no = 0;
	U16			scsi_id = 0;

	BEGIN;

	pHba = (adpt_hba *) arg;

    if (pHba->state & ADPT_STATE_LCT_GET) {
		if(!(pHba->state & ADPT_STATE_RESCAN_IN_PROGRESS) ) {

		    PDEBUG(DBG_ERR_HNDL, "Jiffies : LCT get  !\n");
		    /*
	 	     * Should parse lct here first before doing SR.
	 	     * lets skip the current SR because we haven't
	 	     * parsed that latest LCT, thus, risking 
	 	     * redundant SR when the device is offline.
	 	     */   	
		    /* Have a dedicated tasklet for every thread instead of 
		     * using the same and memset. 
		     */
            /* Ran into issues when we give control to SR. The schedule
             * task can be invoked after all the high priority tasks are
             * are completed. However we need more like a fixed time out in a
             * non interrupt context. Hence moved to threads
             */
             
#if 0
		    memset(&pHba->process_lct_async_msg_bh, 0,sizeof(struct tq_struct));
		    pHba->process_lct_async_msg_bh.data = (void *) pHba;
		    pHba->process_lct_async_msg_bh.routine = adpt_process_lct_async_msg;
       	            pHba->state |= ADPT_STATE_RESCAN_IN_PROGRESS;

		    /* 
		     *  According to Linux Device Driver 2nd Ed.
		     *  the process will be called in 10ms after timer. 
		     */
		    schedule_task(&pHba->process_lct_async_msg_bh);
#endif
		    /* Get Rid of the Schedule_task for now */	
       	            pHba->state |= ADPT_STATE_RESCAN_IN_PROGRESS;
#ifdef __VMKERNEL_MODULE__
		    vmk_thread_wakeup(&pHba->wake_up_sr);
#else
		    wake_up(&pHba->wake_up_sr);
#endif //__VMKERNEL_MODULE__
#if 0
                    ptsk_tmr = kmalloc(sizeof(task_timer),GFP_ATOMIC);
		    if(ptsk_tmr == NULL){
			PDEBUG(DBG_ERR_HNDL,"\nUnable to Allocate Memory");
			goto skip_SR;
		    }	

		    memset(ptsk_tmr, 0,sizeof(task_timer));

		    ptsk_tmr->process_lct_async_msg_bh.data = (void *) ptsk_tmr;
		    ptsk_tmr->process_lct_async_msg_bh.routine = adpt_process_lct_async_msg;
		    ptsk_tmr->pHba = pHba;
 	
       	            pHba->state |= ADPT_STATE_RESCAN_IN_PROGRESS;

		    /* 
		     *  According to Linux Device Driver 2nd Ed.
		     *  the process will be called in 10ms after timer. 
		     */
		    schedule_task(&ptsk_tmr->process_lct_async_msg_bh);
#endif
        }

		goto skip_SR;
                       

	}

	/*
	 *  Session Recovery is done here to reduce time spent on other OS
	 *  Entry points.
	 */

	for(channel_no = 0; channel_no < MAX_CHANNEL; channel_no++) {
		for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
			if((pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {
			
				if (pTarget == NULL) {
            	/*
              PDEBUG(DBG_EVENT, "Jiffies : No Luns in target" \
                                             " %d !\n",tid);
            	*/

			    	/* Target has no lun */
			    	continue;
				}


				/*
				 *  The configured but unconnected target is always 
				 *  offline, need not do SR 
				 */ 
			
				if (pTarget->device == NULL) {
					/*
					PDEBUG(DBG_EVENT, "Jiffies : No Luns in target" \
                                             " %d !\n",tid);
					*/

					/* Target has no lun */
					continue;
				}

				/*
				 * If driver is about to be shutdown, then no need to 
				 * call adpt_timer
				 */
				if (pHba->state & ADPT_STATE_SHUTDOWN) {
					PDEBUG(DBG_EVENT, "Jiffies : target %d" \
											" shutting down!\n",pTarget->tid);
					pTarget->state &= ~ADPT_TARGET_SESS_RECOVERY;
					continue;
				}
                        
				if ( pTarget->current_sr_timeout > pTarget->flexible_timeout)  {

					/* Target did not come back after SRTO */
					PDEBUG(DBG_EVENT, "Jiffies : target %d" \
											" exceed SRTO !\n",pTarget->tid);
					pTarget->state &= ~ADPT_TARGET_SESS_RECOVERY;

					continue;
				}
				else
				if ( pTarget->flexible_timeout == 0 &&
						pTarget->reset_target_threshold == 1)  {
					/* Target is using default setting for SRTO 
					 * which works for Microsoft but not Linux
					 */
					pTarget->flexible_timeout=TMOUT_SESS_RECOVERY; 
				}


				//if (pTarget->state & ADPT_DEV_ONLINE) {
				if(! (pTarget->state & ADPT_DEV_OFFLINE) ) {
				
					pTarget->state &= ~ADPT_TARGET_SESS_RECOVERY;
					pTarget->state &= ~ADPT_DEV_OFFLINE;
				} 

				/* 
				 * LinkDown has higher precedence than session recovery
				if ( pHba->state & ADPT_STATE_LINK_DOWN ) {
					pTarget->state &= ~ADPT_TARGET_SESS_RECOVERY;
					break;
				}
				*/

				if (pTarget->state & ADPT_TARGET_SESS_RECOVERY) { 
					pTarget->current_sr_timeout++;
					PDEBUG(DBG_ERR_HNDL, "Jiffies : SR %d %d !\n",
												pTarget->tid,
												pTarget->current_sr_timeout);
				}

				if ((pTarget->state & ADPT_TARGET_SESS_RECOVERY) && 
					((pTarget->current_sr_timeout %10) == 0) ) { 

					/*
					 * Issue Session Recovery to the FW.
					 */   	

					adpt_initiate_session_recovery(pHba, pTarget);

					PDEBUG(DBG_ERR_HNDL, "Session Recovery was " \
												"issued for Target : %s \n",
												pTarget->iscsi_name);


					/* current_sr_timeout is zeroed when target 
					 * become online 
					 */

				} 
			}
		}
	}

	/*
	 * Time to wake up all commands that have 
	 * extended timeout. Should not be NULL if there
	 *  is commands waiting to be completed 
	 */

	if (pHba->LinkdownQueueHead != NULL){
		if (!(pHba->state & ADPT_STATE_RESCAN_IN_PROGRESS)) { 
			adpt_complete_linkdown_queue(pHba);
				/* No Need to remove from Queue since 
             * process_done_queue
             *  and adpt_i2o_to_scsi will handle it. 
             */
           }
         }
skip_SR:

	/*
	 * Lets schedule timer 1 seconds later.
    */
	ptimerlist = &pHba->hba_timer;

	ptimerlist->expires = adpt_get_jiffies() + (INTERNAL_TICK * HZ);

	ptimerlist->data = (U32_64)pHba;

	ptimerlist->function = adpt_timer;

	add_timer(ptimerlist);

	END;
	return;
}

unsigned long
adpt_get_jiffies( void)
{
	return jiffies;
}

/*
 * This functions answers the questions if driver has any active commands
 * pending with OS or pending with FW.  Return TRUE if none are pending.
 * Returns FALSE if there are pending commands.   
unsigned int
adpt_check_queues_empty(adpt_hba *pHba) 
{
    if ( pHba->active_cmds == 0 && 
         pHba->SendQueueHead == NULL)  { 
        return TRUE;
    }

    return FALSE;
}
 */

/*
 * adpt_complete_linkdown_queue()
 *
 * Description:
 *	Modifies timeout for all done commands in the Queue that has this flag set.
 */
static void 
adpt_complete_linkdown_queue(adpt_hba *pHba)
{
	BEGIN;

	if (pHba->LinkdownQueueHead == NULL) 
           return;

	PDEBUG(DBG_ERR_HNDL, "Linkdown Queue isn't empty !\n");
        adpt_complete_queue(pHba,  
                             CMD_REQ_EXTENDED_COMPLETION);
	END;
}

/*
 * adpt_complete_queue()
 *
 * Description:
 *	Modifies timeout for all commands in the Queue that has this flag set.
 */
static void 
adpt_complete_queue(adpt_hba *pHba, U8 cmd_state)
{
	adpt_cmnd	*pCmnd_curr;
	adpt_cmnd	*pCmnd_next;
	adpt_cmnd	*pCmnd_prev;
        adpt_target     *pTarget=NULL;
	U8		count = 0;
        Scsi_Cmnd       *cmd;
	U32_64			flags = 0;

	struct adpt_queue_info	*qinfop;
/*	struct timer_list	*ptimerlist; */

	BEGIN;
	adpt_lock(pHba, flags, 1);

	pCmnd_curr = pHba->LinkdownQueueHead;

	if (pCmnd_curr == NULL) {
		PDEBUG(DBG_ERR_HNDL, "The Queue is empty !\n");
		adpt_unlock(pHba, flags, 1);
                return;
	}

	while (pCmnd_curr) {
		pCmnd_next = pCmnd_curr->next;
                pCmnd_prev = pCmnd_curr->prev;


                /*  Return all commands with the HBA flag in the next jiffy */

                   pTarget = pCmnd_curr->target;

                   if (pTarget == NULL) {
	              PDEBUG(DBG_ERR_HNDL, "Target %p is NULL, %x!\n",
                                         pTarget,pCmnd_curr);
		      pCmnd_curr = pCmnd_next;
                      continue;
                   }
                      
                   if (pTarget->state & ADPT_TARGET_SESS_RECOVERY) {
	              PDEBUG(DBG_ERR_HNDL, "Target %p is still in LinkQ for %p %x!\n",
                                         pTarget,pCmnd_curr, pCmnd_curr->command);

		      pCmnd_curr = pCmnd_next;
                      continue;
                   }

	           PDEBUG(DBG_ERR_HNDL, "Complete Queue returns %p %x!\n",
                                         pCmnd_curr, pCmnd_curr->command);

                   cmd = pCmnd_curr->command;
                   cmd->result = (DID_BUS_BUSY << 16);

	           /*
	            * We are currently not using SCp for anything.  
	            * Besides, we are done with this command anyway.
	            */

			if (cmd->scsi_done != NULL) {
			    cmd->scsi_done(cmd);
			}
                        else {
	                    PDEBUG(DBG_ERR_HNDL, "scsi_done is NULL!\n");
                        }

                   qinfop = (struct adpt_queue_info *) &cmd->SCp;
	
                   qinfop->cmd = cmd;

#if 0    /* We got away with another timer here */
	           ptimerlist = &qinfop->timerlist;

	           ptimerlist->data = (U32_64) NULL;
	
	           ptimerlist->function = NULL;

                   del_timer_sync(ptimerlist);
#endif

		   PCRIT(CRIT_Q,"<d%d> The content is %p state is %x in Q(H) !</d%d>\n",
				count, pCmnd_curr->command, 
                                pCmnd_curr->cmd_state,
                                count);
		   count++;

	       if (adpt_remove_cmd_fr_queue(
				    &pHba->LinkdownQueueHead,
		       		    &pHba->LinkdownQueueTail,
				    pCmnd_curr))
                {
	            PDEBUG(DBG_ERR_HNDL, "Failed to remove command !\n");
                }
                else
                {
		    adpt_free_cmd(pHba, pCmnd_curr); 
                }
	      pCmnd_curr = pCmnd_next;

	}

	adpt_unlock(pHba, flags, 1);
	END;
}

/*
 * adpt_queue_done_command_for_sr()
 * 
 * Description :
 * 	Schedule a command with to return for OS completion.
 */ 	    
static inline void
adpt_queue_done_command_for_sr(Scsi_Cmnd *cmd, void (*scsi_done) (Scsi_Cmnd *), adpt_target *pTarget)
{
	struct adpt_queue_info	*qinfop;
	struct timer_list	*ptimerlist;

	BEGIN;

	/*
	 * We are currently not using SCp for anything.  
	 * Besides, we are done with this command anyway.
	 */
	qinfop = (struct adpt_queue_info *) &cmd->SCp;
	
	qinfop->cmd = cmd;
	
	qinfop->scsi_done = scsi_done;
	
	ptimerlist = &qinfop->timerlist;

        if (pTarget == NULL) {
	   PDEBUG(DBG_ERR_HNDL, "No target, returned!\n");
           return;
        }
	
	PDEBUG(DBG_SCSI, "delayed Cmd returned:cmd = %x %x", cmd, pTarget);

	switch (cmd->result >> 16) {
		case DID_BUS_LINKDOWN:
			cmd->result = (DID_BUS_BUSY << 16);

                        /* Tell queue to return this command at later SR Time */
                        if ( (pTarget->flexible_timeout-
                              pTarget->current_sr_timeout)> 0) {
		              ptimerlist->expires = jiffies + 
                              ((pTarget->flexible_timeout-5) *HZ);
                        }
                        break;

                case DID_BUS_BUSY:
			ptimerlist->expires = jiffies +INTERNAL_TICK*HZ;
                        break;

		default:

	                PDEBUG(DBG_ERR_HNDL, "wrong returned status was returned!\n");
			cmd->result = (DID_ERROR << 16);
			ptimerlist->expires = jiffies;

			break;
	}		
		
	ptimerlist->data = (U32_64) qinfop;
	
	ptimerlist->function = adpt_process_done_queue;
	
	add_timer(ptimerlist);

	END;
}

static adpt_cmnd *
adpt_get_free_cmd(adpt_hba *pHba)
{
	adpt_cmnd	*pCmnd;

	pCmnd=pHba->FreeQueueHead;

        if (adpt_remove_cmd_fr_queue(
		    &pHba->FreeQueueHead,
		    &pHba->FreeQueueTail,
		    pCmnd)) {
                   return NULL;
        }

        return pCmnd;

}

static void
adpt_free_cmd(adpt_hba *pHba, adpt_cmnd *pCmnd)
{
        BEGIN;
        
	memset(pCmnd, 0, sizeof(adpt_cmnd));
	adpt_insert_cmd_to_queue(
			(adpt_cmnd **)&pHba->FreeQueueHead,
		       	(adpt_cmnd **)&pHba->FreeQueueTail, 
			pCmnd); 
        END;
}

/*
 * adpt_set_driver_status()
 *
 * Description :
 *      Sets the status about the driver to firmware that this is a
 *      fresh boot and not a power management resume bootup
 */
S32
adpt_set_driver_status(adpt_hba *pHba)
{
        U32 msg[MAX_MESSAGE_SIZE/4];
        U32     msg_size;
        S32     rcode = 0;
                                                                                
        BEGIN;
                                                                                
        msg[0] = I2O_MESSAGE_SIZE(7) | SGL_OFFSET_0;
        msg[1] = (0xff << 24 | HOST_TID << 12 | ADAPTER_TID);
        msg[2] = 0x0;           /* Initiator Context */
        msg[3] = (U32) msg;     /* Target Context (we want the msg back) */
        msg[4] = (DPT_ORGANIZATION_ID << 16) | I2O_PRVT_SET_DRIVER_STATUS;
        msg[5] = 0x00010000;    /* Interpret bit set to 1 */
        msg[6] = DRIVER_STATUS_FRESH_START;   
        /* Message size in U32 words */
        msg_size = 7;
                                                                                
        /* Message size in U8 bytes */
        msg_size = 7 << 2;
                                                                                
        /* Poll for max 90 secs - taken from Windows driver */
        if ((rcode = adpt_i2o_post_wait(pHba, msg, msg_size, 90))) {

                PDEBUG(DBG_POST, "%s: failed setting driver status "
                        "(status=%#10x).\n",
                        pHba->name, rcode);
        } else {

                PDEBUG(DBG_POST, "%s: driver status set.\n", pHba->name);
        }
	
        return rcode;
}

/* private functions */
static void kthread_launcher(void *data)
{
        kthread_t *kthread = data;
        kernel_thread((int (*)(void *))kthread->function, (void *)kthread, 0);
        
}

/* public functions */
/* create a new kernel thread. Called by the creator. */
void start_kthread(void (*func)(kthread_t *), kthread_t *kthread)
{
        /* initialize the semaphore:
           we start with the semaphore locked. The new kernel
           thread will setup its stuff and unlock it. This
           control flow (the one that creates the thread) blocks
           in the down operation below until the thread has reached
           the up() operation.
         */
        init_MUTEX_LOCKED(&kthread->startstop_sem);

        /* store the function to be executed in the data passed to
           the launcher */
        kthread->function=func;
        
        /* create the new thread my running a task through keventd */

        /* initialize the task queue structure */
        kthread->tq.sync = 0;
        INIT_LIST_HEAD(&kthread->tq.list);
        kthread->tq.routine =  kthread_launcher;
        kthread->tq.data = kthread;

        /* and schedule it for execution */
        schedule_task(&kthread->tq);

        /* wait till it has reached the setup_thread routine */
        down(&kthread->startstop_sem);
               
}

/* stop a kernel thread. Called by the removing instance */
void stop_kthread(kthread_t *kthread)
{
        if (kthread->thread == NULL)
        {
                printk("stop_kthread: killing non existing thread!\n");
                return;
        }

        /* this function needs to be protected with the big
	   kernel lock (lock_kernel()). The lock must be
           grabbed before changing the terminate
	   flag and released after the down() call. */
        lock_kernel();
        
        /* initialize the semaphore. We lock it here, the
           leave_thread call of the thread to be terminated
           will unlock it. As soon as we see the semaphore
           unlocked, we know that the thread has exited.
	*/
        init_MUTEX_LOCKED(&kthread->startstop_sem);

        /* We need to do a memory barrier here to be sure that
           the flags are visible on all CPUs. 
        */
        mb();

        /* set flag to request thread termination */
        kthread->terminate = 1;

        /* We need to do a memory barrier here to be sure that
           the flags are visible on all CPUs. 
        */
        mb();
        kill_proc(kthread->thread->pid, SIGKILL, 1);
       
        /* block till thread terminated */
        down(&kthread->startstop_sem);

        /* release the big kernel lock */
        unlock_kernel();

        /* now we are sure the thread is in zombie state. We
           notify keventd to clean the process up.
        */
        kill_proc(2, SIGCHLD, 1);

}

/* initialize new created thread. Called by the new thread. */
void init_kthread(kthread_t *kthread, char *name)
{
        /* lock the kernel. A new kernel thread starts without
           the big kernel lock, regardless of the lock state
           of the creator (the lock level is *not* inheritated)
        */
        lock_kernel();

        /* fill in thread structure */
        kthread->thread = current;

        /* set signal mask to what we want to respond */
        siginitsetinv(&current->blocked, sigmask(SIGKILL)|sigmask(SIGINT)|sigmask(SIGTERM));

        /* initialise wait queue */
        init_waitqueue_head(&kthread->queue);

        /* initialise termination flag */
        kthread->terminate = 0;

        /* set name of this process (max 15 chars + 0 !) */
        sprintf(current->comm, name);
        
        /* let others run */
        unlock_kernel();

        /* tell the creator that we are ready and let him continue */
        up(&kthread->startstop_sem);

}

/* cleanup of thread. Called by the exiting thread. */
void exit_kthread(kthread_t *kthread)
{
        /* we are terminating */

	/* lock the kernel, the exit will unlock it */
        lock_kernel();

        kthread->thread = NULL;
        mb();

        /* notify the stop_kthread() routine that we are terminating. */
	up(&kthread->startstop_sem);
	/* the kernel_thread that called clone() does a do_exit here. */

	/* there is no race here between execution of the "killer" and real termination
	   of the thread (race window between up and do_exit), since both the
	   thread and the "killer" function are running with the kernel lock held.
	   The kernel lock will be freed after the thread exited, so the code
	   is really not executed anymore as soon as the unload functions gets
	   the kernel lock back.
	   The init process may not have made the cleanup of the process here,
	   but the cleanup can be done safely with the module unloaded.
	*/

}


/* 
 * adpt_initialize_target_params
 *
 * Gets target parameters like recovery timeout and initializes
 * target structure.
 */

S32 adpt_initialize_target_params(adpt_hba *pHba,
													adpt_target *pTarget,
													U8 channel_no,
													U16 scsi_id,
													i2o_lct_entry *lct_entry)
{
/*	
	U16	opblk[OP_BLOCK_SIZE/2];
	U32	resblk[RES_BLOCK_SIZE/4];
*/
	S32	status = 0;
	U16	*opblk=NULL;
	U32	*resblk=NULL;

	if( !pTarget || !pHba || !lct_entry)
			return -1;

	opblk = (U16 *)kmalloc(OP_BLOCK_SIZE, GFP_ATOMIC | GFP_DMA);
	if(!opblk){
		END;
		return -1;
	}

	resblk = (U32 *)kmalloc(RES_BLOCK_SIZE, GFP_ATOMIC | GFP_DMA);
	if(!resblk){
		kfree(opblk);
		END;
		return -1;
	}

	memset(opblk, 0, OP_BLOCK_SIZE);
	memset(resblk, 0, RES_BLOCK_SIZE);

	/* Operation Count */
	opblk[0] = 0x0001;
	/* Reserved */
	opblk[1] = 0xffff;
	/* Operation Code */
	opblk[2] = I2O_PARAMS_FIELD_GET;
	/* Group Number */
	opblk[3] = I2O_DPT_ISCSIDEV_INFO_GROUP_NO;
	/* Field Count */
	opblk[4] = 0xffff;
	/* Padding */
	opblk[5] = 0x0000;
                                                                                
	status = adpt_i2o_issue_params(
					I2O_CMD_UTIL_PARAMS_GET, pHba, 
					//d->lct_data.tid,
					lct_entry->tid,
/*					opblk, sizeof(opblk),
					resblk, sizeof(resblk));*/
					opblk, OP_BLOCK_SIZE,
					resblk, RES_BLOCK_SIZE);

	if (status != 0) {
		kfree(opblk);
		kfree(resblk);
		ERROR_END;
		return -1;
	}
                                                                                
	memcpy(pTarget->iscsi_name, resblk + 3, 256);
                                                                                
	memcpy(&pTarget->reset_target_threshold, resblk + 67, 4);
                                                                                
	memcpy(&pTarget->flexible_timeout, resblk + 68, 4);
	/*
	 * Get RTTH value.
	 */
	pTarget->reset_target_threshold =
               MIN(pTarget->reset_target_threshold,
                  MAX_RTTH);
                                                                                
	/*
	* Get FTO value.
	*/
	pTarget->flexible_timeout = MIN(pTarget->flexible_timeout,
                  						MAX_FTO);
                                                                                
	//pTarget->type = d->lct_data.class_id;
	//pTarget->tid = d->lct_data.tid;
                                                                                
	pTarget->type = lct_entry->class_id;
	pTarget->tid = lct_entry->tid;

	pTarget->channel_no = channel_no;
                                                                                
	pTarget->scsi_id = scsi_id;
                                                                                
	/*
	 * Check the device flags to determine the
	 * current status of the device.
	 */
	//if (d->lct_data.device_flags & I2O_LCT_DEVICE_FLAGS_OFFLINE) {
	if (lct_entry->device_flags & I2O_LCT_DEVICE_FLAGS_OFFLINE) {

		// if we're going from online to offline state, setup state
		// for session recovery
		if(!(pTarget->state & ADPT_DEV_OFFLINE))
			pTarget->state |= ADPT_TARGET_SESS_RECOVERY;
		
		pTarget->state |= ADPT_DEV_OFFLINE;
		pTarget->state &= ~ADPT_DEV_ABSENT;
                                                                                
		PDEBUG(DBG_LCT, "Tgt(TID=%d) "
      		"off\n",
      		pTarget->tid);
	//} else if(d->lct_data.device_flags & I2O_LCT_DEVICE_FLAGS_ABSENT) {
	} else if(lct_entry->device_flags & I2O_LCT_DEVICE_FLAGS_ABSENT) {
		
		pTarget->state |= ADPT_DEV_ABSENT;
                                                                                
		PDEBUG(DBG_LCT, "Tgt(TID=%d) "
                     "absent\n",
                     pTarget->tid);
 		} else {
			//pTarget->state = ADPT_DEV_ONLINE;
			pTarget->state &= ~ADPT_DEV_OFFLINE;
			pTarget->state &= ~ADPT_DEV_ABSENT;
			pTarget->state &= ~ADPT_TARGET_SESS_RECOVERY;
			pTarget->current_sr_timeout=0;
                                                                                
			PDEBUG(DBG_LCT, "Tgt(TID=%d) "
                     "on\n",
                     pTarget->tid);
		}

		pTarget->state &= ~ADPT_DEV_UNSCANNED;
                                                                                
		PDEBUG(DBG_LCT, "Mapping ClassID of 80h (%s)"
                  " with TID = %d to "
						" SCSI Bus = %d "
                  "SCSI ID = %d "
						"State = %x\n",
                  pTarget->iscsi_name,
                  pTarget->tid,
						pTarget->channel_no,
                  pTarget->scsi_id,
						pTarget->state);
                                                                                
		PDEBUG(DBG_LCT, "RTTH = %d times and "
                  "FTO = %d secs !\n",
               pTarget->reset_target_threshold,
               pTarget->flexible_timeout);

		kfree(opblk);
		kfree(resblk);
		return 0;
}


/*
 * adpt_get_iscsi_name()
 *
 * This gets the iSCSI name for a given TID
 */

S32 adpt_get_iscsi_name(adpt_hba *pHba, int tid, U8 *iscsi_name)
{
/*	U16	opblk[OP_BLOCK_SIZE/2];
	U32	resblk[RES_BLOCK_SIZE/4];*/
	S32 	status = 0;

	U16	*opblk=NULL;
	U32	*resblk=NULL;

	BEGIN;

	opblk = (U16 *)kmalloc(OP_BLOCK_SIZE, GFP_ATOMIC | GFP_DMA);
	if(!opblk){
		ERROR_END;
		return 0;
	}

	resblk = (U32 *)kmalloc(RES_BLOCK_SIZE, GFP_ATOMIC | GFP_DMA);
	if(!resblk){
		kfree(opblk);
		ERROR_END;
		return 0;
	}
	memset(opblk, 0, OP_BLOCK_SIZE);
	memset(resblk, 0, RES_BLOCK_SIZE);

	/* Operation Count */
	opblk[0] = 0x0001;
	/* Reserved */
	opblk[1] = 0xffff;
	/* Operation Code */
	opblk[2] = I2O_PARAMS_FIELD_GET;
	/* Group Number */
	opblk[3] = I2O_DPT_ISCSIDEV_INFO_GROUP_NO;
	/* Field Count */
	opblk[4] = 0xffff;
	/* Padding */
	opblk[5] = 0x0000;
                                                                                
	status = adpt_i2o_issue_params( I2O_CMD_UTIL_PARAMS_GET, 
												pHba, tid,
                                 /*   opblk, sizeof(opblk),
                                    resblk, sizeof(resblk));*/
					opblk, OP_BLOCK_SIZE,
					resblk, RES_BLOCK_SIZE);
	if (status != 0) {
		kfree(opblk);
		kfree(resblk);
		ERROR_END;
		return -1;
	}
                                                                                
	memcpy(iscsi_name, resblk + 3, 256);
	
	kfree(opblk);
	kfree(resblk);

	return 0;
		
}


/*
 * is_existing_target()
 *
 * This function loops over the target list and checks if there
 * is a target entry with the same name as iscsi_name. If so it
 * returns the pointer to target entry. Else it returns NULL.
 */

S32 is_existing_target(adpt_hba *pHba, U8 *iscsi_name, 
								adpt_target **pTarget)
{
	int channel_no = 0, scsi_id = 0;

	if(!pHba || !iscsi_name || !pTarget)
		return -1;

	for(channel_no = 0; channel_no < MAX_CHANNEL; channel_no++) {
		for(scsi_id = 0; scsi_id < MAX_ID_PER_CHANNEL; scsi_id++) {
			if( (*pTarget = pHba->channel[channel_no].target[scsi_id]) != NULL) {
				if(strcmp((*pTarget)->iscsi_name,iscsi_name) == 0)
					return 1;
			}
		}
	}
		
	*pTarget = NULL;
	return 0;
}
/*
 * get_free_bt()
 *
 * This will loop over the target list and get the first slot
 * that is available. The function remembers previous assignment
 * and the next assignment starts off from where it left the 
 * last time
 */
S32 get_free_bt(adpt_hba *pHba, U8 *pChannel_no, U16 *pScsi_id)
{
	static U8 assigned_channel_no = 0;
	static U16 assigned_scsi_id = -1;
	int max_ids = MAX_CHANNEL * MAX_ID_PER_CHANNEL;
	int traversed_ids = 0;

	while(1) {

		assigned_scsi_id++;

		if(assigned_scsi_id >= MAX_ID_PER_CHANNEL) {
		
			assigned_scsi_id = 0;
			assigned_channel_no++;
			if(assigned_channel_no >= MAX_CHANNEL) 
				assigned_channel_no = 0;
		
		}

		traversed_ids++;
		if(traversed_ids >= max_ids)
			// CR: TODO : print to make sure we catch corner case
			return -1;

		if(pHba->channel[assigned_channel_no].target[assigned_scsi_id]) {

			// check to make sure this slot can be given
			// some flag will be needed
			// TODO
			PDEBUG(DBG_LCT,"BT %d %d already taken\n", assigned_channel_no, assigned_scsi_id);
			continue;
		}

		// this is a free slot, take it!
		*pChannel_no = assigned_channel_no;
		*pScsi_id = assigned_scsi_id;
		return 0;
	}

	return 1;

}


/*
 * adpt_map_new_target()
 *
 * This function allocates a adpt_target structure and initializes
 * it. This target is inserted at the position specified by the
 * channel_no and scsi_id. If these 2 values are FFs then it means
 * its a new assignment and this target is inserted at the free slot
 */

adpt_target * adpt_map_new_target(adpt_hba *pHba, i2o_lct_entry *lct_entry,
							U8 channel_no, U16 scsi_id)
{

	S32 status;
	adpt_target *pTarget = NULL; 

	if(channel_no == 0xFF && scsi_id == 0xFF) {
		
		// find a free slot for this target
		PDEBUG(DBG_LCT,"Getting free slot...\n");
		status = get_free_bt(pHba, &channel_no, &scsi_id);

		if(status) {

			PDEBUG(DBG_LCT,"Cannot get free slot for target\n");

			ERROR_END;
			return NULL;

		}
	}

	PDEBUG(DBG_LCT,"Got free slot at B = %d T = %d...\n",
						channel_no, scsi_id);

	pHba->channel[channel_no].target[scsi_id] = pTarget =
                              kmalloc(sizeof(adpt_target),
                                       GFP_ATOMIC);
                                                                                
	if (pTarget == NULL) {
  		uerrno = (UL_ERROR | UP_INIT | US_SYS |
                           transport |
                           UO_CMN | UC_NO_EXT |
                           U_MALLOC_FAIL);
                                                                                
		PERROR("%s: ERROR: %s [%x] \n",
                     pHba->name,
                     uni_err(uerrno),
                     (U32) uerrno);
                                                                                
		ERROR_END;
                                                                                
		return NULL;
	}

	memset(pTarget, 0, sizeof(adpt_target));

	PDEBUG(DBG_LCT,"Initializing target at %d %d\n", channel_no, scsi_id);

	status = adpt_initialize_target_params(pHba, pTarget, channel_no,
                                                   scsi_id,
                                                   lct_entry);
	if(status) {
                                                                                
		kfree(pTarget);
		pHba->channel[channel_no].target[scsi_id] = NULL;
                                                                                
		uerrno = (UL_ERROR | UP_INIT | US_SYS |
                           transport |
                           UO_CMN | UC_NO_EXT);
                                                                                
		PERROR("%s: ERROR: %s [%x] \n",
                     pHba->name,
                     uni_err(uerrno),
                     (U32) uerrno);

		ERROR_END;
		return NULL;

	}

	if(channel_no > pHba->top_scsi_channel)
		pHba->top_scsi_channel = channel_no;

	if(scsi_id > pHba->top_scsi_id)
		pHba->top_scsi_id = scsi_id;

	return pTarget;
}

/* 
 * adpt_initialize_lun_params()
 * This function allocates memory for a lun, initializes the
 * values and links it to the target entry. If the lct entry
 * indicates that the LUN was already present, then it just
 * sets the adpt_device structure appropriately for that LUN.
 * This function also clears the unscanned flag for LUN entry.
 */

S32 adpt_initialize_lun_params(adpt_hba *pHba, adpt_target *pTarget,
				i2o_lct_entry * lct_entry,
				adpt_device **pLun)
{
	adpt_device *pDev = NULL ;
	int lun_present = 0;

	if(!pHba || !pTarget || !lct_entry || !pLun)
		return -1;
		

	if(pTarget->device) {
		lun_present = 0;

		pDev = pTarget->device;
		while(1) {
			if((pDev->scsi_lun == (lct_entry->sub_class & I2O_LCT_SUBCLASS_MASK)))
		   {
				lun_present = 1;
				break;
			}
			if(pDev->next_lun == NULL) {
				lun_present = 0;
				break;
			}
			pDev = pDev->next_lun;
		}
	}
	PDEBUG(DBG_REPARSE,"LunPresent = %d\n", lun_present);
	PDEBUG(DBG_REPARSE,"init_lun_parms for LUN %x Flag %x\n", 
								lct_entry->tid, lct_entry->device_flags);

	if(!lun_present) {                                                                                
		PDEBUG(DBG_REPARSE,"Allocating for a new LUN...\n");

		// need to allocate space for the new lun
		*pLun = kmalloc(sizeof(adpt_device),
											GFP_ATOMIC);
                                                                                
		if (*pLun == NULL) {
			uerrno = (UL_ERROR | UP_INIT | US_SYS |
						transport |
							UO_CMN | UC_NO_EXT |
							U_MALLOC_FAIL);
                                                                                
			PERROR("%s: ERROR: %s [%x] \n",
						pHba->name,
						uni_err(uerrno),
						(U32) uerrno);

			ERROR_END;
			return -ENOMEM;
		}
	} // if !(lun_present)

	// setup pDev for different cases
	if(pTarget->device == NULL) {
		PDEBUG(DBG_REPARSE,"First time setup for pTarget->device\n");
		pTarget->device = *pLun;
		pDev = *pLun;
	}
	else if(!lun_present) {
		PDEBUG(DBG_REPARSE,"setup for pTarget->device\n");
		pDev->next_lun = *pLun;
		pDev = pDev->next_lun;
	}
	else
		PDEBUG(DBG_REPARSE,"No setup for pTarget->device\n");

	
	if(!lun_present) {
		memset(pDev, 0, sizeof(adpt_device));

		pDev->scsi_channel = pTarget->channel_no;
		pDev->scsi_id = pTarget->scsi_id;
		pDev->scsi_lun = lct_entry->sub_class & I2O_LCT_SUBCLASS_MASK;
		pDev->target_context = lct_entry->target_context;
		pDev->flags = 0xC4;
		pDev->state = 0x01;
		pDev->next_lun = NULL;
   }                                                                               
    // update tid and parent tid always
	pDev->tid = lct_entry->tid;
	pDev->parent_tid = lct_entry->parent_tid;
	
	/*
 	 * Check the device flags to determine the
 	 * current status of the device.
 	 */
	if (lct_entry->device_flags & I2O_LCT_DEVICE_FLAGS_OFFLINE) {
		
		// this case will arise if there are multiple LUNs in the LCT
		// that have the same LUN, but one of them is online, rest
		// are offlines. If we've already seen this LUN before, then
		// dont mark it's state as offline.
		if(pDev->state & ADPT_DEV_UNSCANNED) {
	   	pDev->state |= ADPT_DEV_OFFLINE;
	   	pDev->state &= ~ADPT_DEV_ABSENT;
                                                                                
   		PDEBUG(DBG_LCT, "Device at Channel %d Id %d "
      	   	"Lun %d is offline !\n",
         		pDev->scsi_channel,
         		pDev->scsi_id,
         		pDev->scsi_lun);
		}
		else {

			PDEBUG(DBG_LCT,"Not modifying LUN with tid %d state "
								"as its been scanned before.\n",pDev->tid);

		}
	} else 
	if (lct_entry->device_flags & I2O_LCT_DEVICE_FLAGS_ABSENT) {

		pDev->state |= ADPT_DEV_ABSENT;
                                                                                
		PDEBUG(DBG_LCT, "Device at Channel %d Id %d "
      		"Lun %d is absent !\n",
      		pDev->scsi_channel,
      		pDev->scsi_id,
      		pDev->scsi_lun);
	} else {
		//pDev->state = ADPT_DEV_ONLINE;
		pDev->state &= ~ADPT_DEV_OFFLINE;
		pDev->state &= ~ADPT_DEV_ABSENT;
                                                                                
		PDEBUG(DBG_LCT, "Device at Channel %d Id %d "
    			"Lun %d is online !\n",
     			pDev->scsi_channel,
     			pDev->scsi_id,
     			pDev->scsi_lun);			
	}

	pDev->state &= ~ADPT_DEV_UNSCANNED;
	return 0;
}

/*
 * adpt_check_target_has_lun()
 *
 * This function traverses the LCT entry and checks if the LCT entry pointed by
 * lct_entry has a LUN attached to it. If there is a LUN present, then this 
 * function returns a 1, else it returns a 0
 */

S32 adpt_check_target_has_lun(adpt_hba *pHba, i2o_lct_entry *lct_entry)
{

	i2o_lct 		*lct = pHba->lct;
	U32			max = 0;
	U32			i = 0;

	if(!pHba || !lct_entry || !lct)
		return 0;

	max = lct->table_size;	
	max -= 3;
	max /= 9;

	for(i = 0; i < max; i++) {
		if ( (lct->lct_entry[i].class_id  == I2O_CLASS_RANDOM_BLOCK_STORAGE ||
		   lct->lct_entry[i].class_id  == I2O_CLASS_SCSI_PERIPHERAL ||
		   lct->lct_entry[i].class_id  == I2O_CLASS_FIBRE_CHANNEL_PERIPHERAL) &&
			(lct->lct_entry[i].parent_tid == lct_entry->tid) 
		) {
				return 1;
		}
	}

	return 0;

}

/*
 * adpt_get_target_from_lun_lct()
 *
 * Description :
 * This function gets the target pointer that the driver
 * has stored for the lun lct entry.
 */
adpt_target *
adpt_get_target_from_lun_lct(adpt_hba *pHba,  
                          i2o_lct_entry *lun_lct_entry)
{
   // first loop through the lct table looking for the
   // tid that matches the parent tid of lun_lct_entry.
   // Having found the entry in the lct, get iscsi name
   // of this target and search for the target in driver
   // list.
   i2o_lct  *lct = pHba->lct;
   U32            max = 0;
   U32            i = 0;
   int            target_tid = 0, target_found = 0;
   U8             iscsi_name[256];
   S32            status = 0;
   adpt_target*   pTarget = NULL;

   if(!pHba || !lun_lct_entry || !lct)
      return 0;

   max = lct->table_size;	
   max -= 3;
   max /= 9;

   for(i = 0; i < max; i++) {
      if(lct->lct_entry[i].device_flags & I2O_LCT_DEVICE_FLAGS_ABSENT)
         continue;

      if((lct->lct_entry[i].class_id  == I2O_CLASS_BUS_ADAPTER_PORT ||
          lct->lct_entry[i].class_id  == I2O_CLASS_FIBRE_CHANNEL_PORT) && 
         (lct->lct_entry[i].tid == lun_lct_entry->parent_tid)) {

         target_tid = lct->lct_entry[i].tid;
         target_found = 1;
         break;
      }
   }

   if(!target_found) {
#ifdef ASA72XX_DEBUG
      PDEBUG(DBG_LCT,"Did not find target in LCT for the LUN!!!\n");
      asm("int $3");
#endif
      return NULL;
   }
       
   status = adpt_get_iscsi_name(pHba,target_tid,&iscsi_name[0]);

   if(status) {

      PDEBUG(DBG_LCT,"Could not get iSCSI name, quitting\n");
					
      uerrno = (UL_ERROR | UP_INIT | US_SYS | transport | 
                UO_CMN | UC_NO_EXT);

      PERROR("%s: ERROR: %s [%x] \n", pHba->name, uni_err(uerrno), 
                                      (U32) uerrno);
      ERROR_END;
      return NULL;
   }
	
   if(is_existing_target(pHba,iscsi_name,&pTarget))
      return pTarget;
   else {
#ifdef ASA72XX_DEBUG
      PDEBUG(DBG_LCT,"Target name not found in internal list!!!\n");
      asm("int $3");
#endif
      return NULL;
   }

}
