/*
 *    Disk Array driver for HP SA 5xxx and 6xxx Controllers, SCSI version
 *    Copyright 2000, 2008 Hewlett-Packard Development Company, L.P. 
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *    NON INFRINGEMENT.  See the GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *    Questions/Comments/Bugfixes to iss_storagedev@hp.com
 *
 */
#include <linux/config.h>	/* CONFIG_PROC_FS */
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/major.h>
#include <linux/fs.h>
#include <linux/blkpg.h>
#include <linux/timer.h>
#include <linux/proc_fs.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/init.h> 
#include <linux/hdreg.h>
#include <linux/reboot.h>
#include <linux/spinlock.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/smp_lock.h>
#include <linux/completion.h>
#include <linux/blk.h>
#include <linux/blkdev.h>
#include <linux/genhd.h>
#include "scsi.h"
#include <linux/cciss_ioctl.h>
#include "hpsa_cmd.h"
#include "hpsa.h"

#ifdef __VMKERNEL_MODULE__
#include "vmklinux_dist.h"
spinlock_t io_request_lock;
#endif

#define HPSA_DRIVER_VERSION(maj,min,submin) ((maj<<16)|(min<<8)|(submin))
#define DRIVER_NAME "HP HPSA Driver (v 2.4.66-95vmw)"
#define DRIVER_VERSION HPSA_DRIVER_VERSION(2,4,66)

/* Embedded module documentation macros - see modules.h */
MODULE_AUTHOR("Hewlett-Packard Company");
MODULE_DESCRIPTION("Driver for HP Smart Array Controllers version 2.4.66-95vmw");
MODULE_SUPPORTED_DEVICE("HP P700M and Smart Array G2 Series"); 
MODULE_LICENSE("GPL");

// define HPSA_MAJORS using COMPAQ_CCISS_MAJORs
#define COMPAQ_HPSA_MAJOR	COMPAQ_CISS_MAJOR  
#define COMPAQ_HPSA_MAJOR1	COMPAQ_CISS_MAJOR1  
#define COMPAQ_HPSA_MAJOR2	COMPAQ_CISS_MAJOR2  
#define COMPAQ_HPSA_MAJOR3	COMPAQ_CISS_MAJOR3  
#define COMPAQ_HPSA_MAJOR4	COMPAQ_CISS_MAJOR4  
#define COMPAQ_HPSA_MAJOR5	COMPAQ_CISS_MAJOR5  
#define COMPAQ_HPSA_MAJOR6	COMPAQ_CISS_MAJOR6  
#define COMPAQ_HPSA_MAJOR7	COMPAQ_CISS_MAJOR7  

/* define the PCI info for the cards we can control */
/* remove when PCI_DEVICE_ID_COMPAQ_HPSAC is in pci_ids.h */
#ifndef PCI_DEVICE_ID_COMPAQ_HPSAC
#define PCI_DEVICE_ID_COMPAQ_HPSAC 0x46
#endif
#ifndef PCI_DEVICE_ID_HP_HPSA
#define PCI_DEVICE_ID_HP_HPSA 0x3210
#endif
#ifndef PCI_DEVICE_ID_HP_HPSAA
#define PCI_DEVICE_ID_HP_HPSAA 0x3220
#endif
#ifndef PCI_DEVICE_ID_HP_HPSAC
#define PCI_DEVICE_ID_HP_HPSAC 0x3230
#endif
#ifndef PCI_DEVICE_ID_HP_HPSAD
#define PCI_DEVICE_ID_HP_HPSAD 0x3238
#endif
#ifndef PCI_DEVICE_ID_HP_CISSE
#define PCI_DEVICE_ID_HP_CISSE 0x323A
#endif

const struct pci_device_id hpsa_pci_device_id[] = {
/*Support P700M, and for debug only, P800 controller */
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_HPSAC, 0x103C, 0x323D, 0, 0, 0},
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_CISSE, 0x103C, 0x3241, 0, 0, 0},
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_CISSE, 0x103C, 0x3243, 0, 0, 0},
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_CISSE, 0x103C, 0x3245, 0, 0, 0},
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_CISSE, 0x103C, 0x3247, 0, 0, 0},
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_CISSE, 0x103C, 0x3249, 0, 0, 0},
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_CISSE, 0x103C, 0x324A, 0, 0, 0},
	{ PCI_VENDOR_ID_HP, PCI_DEVICE_ID_HP_CISSE, 0x103C, 0x324B, 0, 0, 0},
	{0,}
};

 
MODULE_DEVICE_TABLE(pci, hpsa_pci_device_id);

#define NR_PRODUCTS (sizeof(products)/sizeof(struct board_type))

/*  board_id = Subsystem Device ID & Vendor ID
 *  product = Marketing Name for the board
 *  access = Address of the struct of function pointers 
 */
static struct board_type products[] = {
 	{ 0x323D103c, "Smart Array P700M", &SA5_access},
	{ 0x3241103C, "Smart Array P212", &SA5_access},
	{ 0x3243103C, "Smart Array P410", &SA5_access},
	{ 0x3245103C, "Smart Array P410i", &SA5_access},
	{ 0x3247103C, "Smart Array P411", &SA5_access},
	{ 0x3249103C, "Smart Array P812", &SA5_access},
	{ 0x324A103C, "Smart Array P712m", &SA5_access},
	{ 0x324B103C, "Smart Array P711m", &SA5_access},
};

/* How long to wait (in millesconds) for board to go into simple mode */
#define MAX_CONFIG_WAIT 30000 
#define MAX_IOCTL_CONFIG_WAIT 1000

/*define how many times we will try a command because of bus resets */
#define MAX_CMD_RETRIES 6

#define READ_AHEAD 	 128
#define NR_CMDS          128 /* Set upper limit of Qdepth when not in ZMR mode */

#if defined (CONFIG_VMNIX) || defined (__VMKERNEL_MODULE__)
#define MAX_CTLR	 8	//vmkernel only supports 16 total (block+scsi)
#else
#define MAX_CTLR	 32
#endif

/* No sense in giving up our preallocated major numbers */
#if MAX_CTLR < 8
#error"hpsa.c: MAX_CTLR must be 8 or greater"
#endif

/* Originally hpsa driver only supports 8 major number */
#define MAX_CTLR_ORIG  COMPAQ_HPSA_MAJOR7 - COMPAQ_HPSA_MAJOR + 1

#define HPSA_DMA_MASK	0xFFFFFFFF	/* 32 bit DMA */

static void print_cmd(CommandList_struct *cp);

static int sendcmd(
        __u8    cmd,
        int     ctlr,
        void    *buff,
        size_t  size,
        unsigned int use_unit_num,
        unsigned int log_unit,
        __u8    page_code,
        unsigned char *scsi3addr,
        int block_nr);
static int add_sendcmd_reject(__u8 cmd, int ctlr, unsigned long complete);
static void do_hpsa_intr(int irq, void *dev_id, struct pt_regs *regs);
static inline unsigned long get_next_completion(ctlr_info_t *h);


static ctlr_info_t *hba[MAX_CTLR];
static devfs_handle_t de_arr[MAX_CTLR][NWD*NUM_GENDISK];
static int map_major_to_ctlr[MAX_BLKDEV] = {0};	/* gets ctlr num from maj num */

static int notify_count=0;

static struct proc_dir_entry *proc_hpsa;

static int register_new_disk(int cltr, int opened_vol, __u64 requested_lun);
static int hpsa_rescan_disk(int cltr, int logvol);

static void __devexit hpsa_remove_one(struct pci_dev *pdev);

static inline void addQ(CommandList_struct **Qptr, CommandList_struct *c);
static void start_io( ctlr_info_t *h);
static int hpsa_notify_reboot(struct notifier_block *this,
                            unsigned long code, void *x);
static int hpsa_ioctl(Scsi_Device *dev, int cmd, void *arg);

#ifdef CONFIG_PROC_FS
static int hpsa_proc_get_info(char *buffer, char **start, off_t offset, 
		int length, int *eof, void *data);
static void hpsa_procinit(int i);
#else
static int hpsa_proc_get_info(char *buffer, char **start, off_t offset, 
		int length, int *eof, void *data) { return 0;}
static void hpsa_procinit(int i) {}
#endif /* CONFIG_PROC_FS */

#if defined (__VMKERNEL_MODULE__)
/*
 * The File Operations structure for the serial/ioctl interface of the driver
 */
long hpsa_char_ioctl(struct inode *inode, struct file *f,
                                    unsigned cmd, unsigned long arg);
int hpsa_char_open(struct inode *inode, struct file *filep);

static const struct file_operations hpsa_char_fops = {
	.owner		= THIS_MODULE,
	.open		= hpsa_char_open,
	.ioctl		= hpsa_char_ioctl,
};
int char_major[MAX_CTLR][2]; 	// Holds chrdev major number by controller, 
				// probably won't need the second element
#endif

#include "hpsa_scsi.c"		/* For SCSI support */

#define ENG_GIG	1000000000
#define ENG_GIG_FACTOR (ENG_GIG/512)
#define	RAID_UNKNOWN 6
static const char *raid_label[] = {"0","4","1(1+0)","5","5+1","ADG",
				   "UNKNOWN"};
#if defined (__VMKERNEL_MODULE__)

/**
 * hpsa_char_open()
 * @inode - unused
 * @filep - unused
 *
 * Routines for the character/ioctl interface to the driver. Find out if this
 * is a valid open.
 */
int hpsa_char_open (struct inode *inode, struct file *filep)
{
	/*
	 * Only allow superuser to access private ioctl interface
	 */
	if( !capable(CAP_SYS_ADMIN) ) return -EACCES;

	return 0;
}
long hpsa_char_ioctl(struct inode *inode, struct file *f,
                                    unsigned cmd, unsigned long arg)
{
	int i;
	int found = 0;
	int chrmajor = 0;
	Scsi_Device dev;
	Scsi_Device *p_dev = &dev;

	chrmajor = MAJOR(inode->i_rdev);
	if (!chrmajor) {
		printk(KERN_WARNING "hpsa: Invalid Char Device major: %d",
			MAJOR(inode->i_rdev));
		return -ENODEV;
	}
	for (i = 0; i < MAX_CTLR; i++) {
		if (char_major[i][0] == chrmajor) {
			found = 1;
			break;
		}
	}
	if (!found)
	{
		printk(KERN_WARNING "hpsa: No matching char dev found for major number %d\n", chrmajor);
		return -ENODEV;
	}
	// i is now the index into hba struct
	p_dev->host = ((struct hpsa_scsi_adapter_data_t *)hba[i]->scsi_ctlr)->scsi_host;

	return hpsa_ioctl(p_dev, cmd, (void *)arg);

}
#endif /*  defined(__VMKERNEL_MODULE__)  */
/*
 * Report information about this controller.
 */
#ifdef CONFIG_PROC_FS
static int hpsa_proc_get_info(char *buffer, char **start, off_t offset, 
		int length, int *eof, void *data)
{
	off_t pos = 0;
	off_t len = 0;
	int size, i, ctlr;
	ctlr_info_t *h = (ctlr_info_t*)data;
	drive_info_struct *drv;
	unsigned long flags;
	unsigned int vol_sz, vol_sz_frac;

	spin_lock_irqsave(&io_request_lock, flags);
	if (h->busy_configuring) {
		spin_unlock_irqrestore(&io_request_lock, flags);
		return -EBUSY;
	}
	h->busy_configuring = 1;
	spin_unlock_irqrestore(&io_request_lock, flags);
		
	ctlr = h->ctlr;
	size = sprintf(buffer, "%s: HP %s Controller\n"
 		"Board ID: 0x%08lx\n"
		"Firmware Version: %c%c%c%c\n"
 		"IRQ: %d\n"
 		"Logical drives: %d\n"
 		"Current Q depth: %d\n"
 		"Current # commands on controller: %d\n"
 		"Max Q depth since init: %d\n"
		"Max # commands on controller since init: %d\n"
		"Max SG entries since init: %d\n"
		MONITOR_PERIOD_PATTERN 
		MONITOR_DEADLINE_PATTERN
		MONITOR_STATUS_PATTERN 
		"\n",
  		h->devname,
  		h->product_name,
  		(unsigned long)h->board_id,
  		h->firm_ver[0], h->firm_ver[1], h->firm_ver[2], h->firm_ver[3],
  		(unsigned int)h->intr,
  		h->num_luns, 
  		h->Qdepth, h->commands_outstanding,
		h->maxQsinceinit, h->max_outstanding, h->maxSG,
		MONITOR_PERIOD_VALUE(h),
		MONITOR_DEADLINE_VALUE(h),
		CTLR_STATUS(h));
  
	pos += size; len += size;
	hpsa_proc_tape_report(ctlr, buffer, &pos, &len);
	for(i=0; i<=h->highest_lun; i++) {
		drv = &h->drv[i];
		if (drv->nr_blocks == 0)
			continue;
		vol_sz = drv->nr_blocks/ENG_GIG_FACTOR; 
		vol_sz_frac = (drv->nr_blocks%ENG_GIG_FACTOR)*100/ENG_GIG_FACTOR;

		if (drv->raid_level > 5)
			drv->raid_level = RAID_UNKNOWN;
		size = sprintf(buffer+len, "hpsa/c%dd%d:"
				"\t%4d.%02dGB\tRAID %s\n",
		       		 ctlr, i, vol_sz,vol_sz_frac,
				 raid_label[drv->raid_level]);
		pos += size, len += size;
        }

	*eof = 1;
	*start = buffer+offset;
	len -= offset;
	if (len>length)
		len = length;
	h->busy_configuring = 0;
	return len;
}

static int
hpsa_proc_write(struct file *file, const char *buffer,
			unsigned long count, void *data)
{
	unsigned char cmd[20];
	int len;
	ctlr_info_t *h = (ctlr_info_t *) data;
	int rc;


	if (count > sizeof(cmd)-1) 
		return -EINVAL;
	if (copy_from_user(cmd, buffer, count)) 
		return -EFAULT;
	cmd[count] = '\0';
	len = strlen(cmd);	
	if (cmd[len-1] == '\n')
		cmd[--len] = '\0';

#if defined CONFIG_SCSI_HPSA || defined CONFIG_SCSI_HPSA_MODULE || defined __VMKERNEL_MODULE__
		if (strcmp("engage scsi", cmd)==0) {
			rc = hpsa_engage_scsi(h->ctlr);
			if (rc != 0) 
				return -rc;
			return count;
		}
#endif


	return -EINVAL;
}

static struct notifier_block hpsa_notifier = {
	.notifier_call  = hpsa_notify_reboot,
};

static int hpsa_notify_reboot(struct notifier_block *this,
                            unsigned long code, void *x)
{
	int i, return_code;
	char flush_buf[4];

	if ((code == SYS_DOWN) || (code == SYS_HALT) || 
					(code == SYS_POWER_OFF)) {
		printk(KERN_INFO "hpsa: stopping all hpsa devices.\n");
		/* double check that all controller entrys have been removed */
		for (i = 0; i < MAX_CTLR; i++) {
			if (hba[i] != NULL) {
				printk(KERN_WARNING "hpsa: removing "
						"controller %d\n", i);
				/* To write all data in the battery backed cache to disks */
				memset(flush_buf, 0, 4);
				return_code = sendcmd(HPSA_CACHE_FLUSH, i, flush_buf, 4, 0, 0, 0, NULL, 0);
				if(return_code != IO_OK) {
					printk(KERN_WARNING "Error Flushing cache on controller  %d\n", i);
				}
			}
		}
	}
	return NOTIFY_DONE;
}

/*
 * Get us a file in /proc/hpsa that says something about each controller.
 * Create /proc/hpsa if it doesn't exist yet.
 */
static void __init hpsa_procinit(int i)
{
	struct proc_dir_entry *pde;

	if (proc_hpsa == NULL) {
		proc_hpsa = proc_mkdir("hpsa", proc_root_driver);
		if (!proc_hpsa) {
			printk(KERN_ERR "hpsa:  proc_mkdir failed\n");
			return;
		}
	}

	pde = create_proc_read_entry(hba[i]->devname,
		S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH,
		proc_hpsa, hpsa_proc_get_info, hba[i]);
	pde->write_proc = hpsa_proc_write;
}
#endif /* CONFIG_PROC_FS */

/* 
 * For operations that cannot sleep, a command block is allocated at init, 
 * and managed by cmd_alloc() and cmd_free() using a simple bitmap to track
 * which ones are free or in use.  For operations that can wait for kmalloc 
 * to possible sleep, this routine can be called with get_from_pool set to 0. 
 * cmd_free() MUST be called with a got_from_pool set to 0 if cmd_alloc was. 
 */ 
static CommandList_struct * cmd_alloc(ctlr_info_t *h, int get_from_pool)
{
	CommandList_struct *c;
	int i; 
	u64bit temp64;
	dma_addr_t cmd_dma_handle, err_dma_handle;

	if (!get_from_pool) {
		c = (CommandList_struct *) pci_alloc_consistent(
			h->pdev, sizeof(CommandList_struct), &cmd_dma_handle); 
        	if (c==NULL)
                 	return NULL;
		memset(c, 0, sizeof(CommandList_struct));

		c->err_info = (ErrorInfo_struct *)pci_alloc_consistent(
					h->pdev, sizeof(ErrorInfo_struct), 
					&err_dma_handle);
	
		if (c->err_info == NULL)
		{
			pci_free_consistent(h->pdev, 
				sizeof(CommandList_struct), c, cmd_dma_handle);
			return NULL;
		}
		memset(c->err_info, 0, sizeof(ErrorInfo_struct));
	} else /* get it out of the controllers pool */ 
	{
	     	do {
                	i = find_first_zero_bit(h->cmd_pool_bits, h->nr_cmds);
                        if (i == h->nr_cmds)
                                return NULL;
                } while(test_and_set_bit(i%32, h->cmd_pool_bits+(i/32)) != 0);
                c = h->cmd_pool + i;
		memset(c, 0, sizeof(CommandList_struct));
		cmd_dma_handle = h->cmd_pool_dhandle 
					+ i*sizeof(CommandList_struct);
		c->err_info = h->errinfo_pool + i;
		memset(c->err_info, 0, sizeof(ErrorInfo_struct));
		err_dma_handle = h->errinfo_pool_dhandle 
					+ i*sizeof(ErrorInfo_struct);
                h->nr_allocs++;
        }

	c->busaddr = (__u32) cmd_dma_handle;
	temp64.val = (__u64) err_dma_handle;	
	c->ErrDesc.Addr.lower = temp64.val32.lower;
	c->ErrDesc.Addr.upper = temp64.val32.upper;
	c->ErrDesc.Len = sizeof(ErrorInfo_struct);
	
	c->ctlr = h->ctlr;
        return c;


}

/* 
 * Frees a command block that was previously allocated with cmd_alloc(). 
 */
static void cmd_free(ctlr_info_t *h, CommandList_struct *c, int got_from_pool)
{
	int i;
	u64bit temp64;

	if (!got_from_pool) { 
		temp64.val32.lower = c->ErrDesc.Addr.lower;
		temp64.val32.upper = c->ErrDesc.Addr.upper;
		pci_free_consistent(h->pdev, sizeof(ErrorInfo_struct), 
			c->err_info, (dma_addr_t) temp64.val);
		pci_free_consistent(h->pdev, sizeof(CommandList_struct), 
			c, (dma_addr_t) c->busaddr);
	} else 
	{
		i = c - h->cmd_pool;
		clear_bit(i%32, h->cmd_pool_bits+(i/32));
                h->nr_frees++;
        }
}

#if NUM_GENDISK > 2
#error this code is meant to use 2 gendisks
#endif

#if defined (CONFIG_X86_64) || defined (CONFIG_IA32E)
/* for AMD 64 bit kernel compatibility with 32-bit userland ioctls */
extern int sys_ioctl(unsigned int fd, unsigned int cmd, unsigned long arg);

extern int 
register_ioctl32_conversion(unsigned int cmd, int (*handler)(unsigned int,
      unsigned int, unsigned long, struct file *));
extern int unregister_ioctl32_conversion(unsigned int cmd);

static int hpsa_ioctl32_passthru(unsigned int fd, unsigned int cmd, unsigned long arg, struct file *file);
static int hpsa_ioctl32_big_passthru(unsigned int fd, unsigned int cmd, unsigned long arg, struct file *file);

typedef int (*handler_type)(unsigned int, unsigned int, unsigned long,
				struct file *);

static struct ioctl32_map {
	unsigned int cmd; 
	handler_type handler;
	int registered;
} hpsa_ioctl32_map[] = {
	{ CCISS_GETPCIINFO,	(handler_type)sys_ioctl, 0 },
	{ CCISS_GETINTINFO,	(handler_type)sys_ioctl, 0 },
	{ CCISS_SETINTINFO,	(handler_type)sys_ioctl, 0 },
	{ CCISS_GETNODENAME,	(handler_type)sys_ioctl, 0 },
	{ CCISS_SETNODENAME,	(handler_type)sys_ioctl, 0 },
	{ CCISS_GETHEARTBEAT,	(handler_type)sys_ioctl, 0 },
	{ CCISS_GETBUSTYPES,	(handler_type)sys_ioctl, 0 },
	{ CCISS_GETFIRMVER,	(handler_type)sys_ioctl, 0 },
	{ CCISS_GETDRIVVER,	(handler_type)sys_ioctl, 0 },
	{ CCISS_REVALIDVOLS,	(handler_type)sys_ioctl, 0 },
	{ CCISS_PASSTHRU32,	hpsa_ioctl32_passthru, 0 },
	{ CCISS_DEREGDISK,	(handler_type)sys_ioctl, 0 },
	{ CCISS_REGNEWDISK,	(handler_type)sys_ioctl, 0 },
	{ CCISS_REGNEWD,	(handler_type)sys_ioctl, 0 },
	{ CCISS_RESCANDISK,	(handler_type)sys_ioctl, 0 },
	{ CCISS_GETLUNINFO,	(handler_type)sys_ioctl, 0 },
	{ CCISS_BIG_PASSTHRU32,	hpsa_ioctl32_big_passthru, 0 },
};
#define NHPSA_IOCTL32_ENTRIES (sizeof(hpsa_ioctl32_map) / sizeof(hpsa_ioctl32_map[0]))
static void register_hpsa_ioctl32(void)
{
	int i, rc;

	for (i=0; i < NHPSA_IOCTL32_ENTRIES; i++) {
		rc = register_ioctl32_conversion(
			hpsa_ioctl32_map[i].cmd,
			hpsa_ioctl32_map[i].handler);
		if (rc != 0) {
			printk(KERN_WARNING "hpsa: failed to register "
				"32 bit compatible ioctl 0x%08x\n", 
				hpsa_ioctl32_map[i].cmd);
			hpsa_ioctl32_map[i].registered = 0;
		} else
			hpsa_ioctl32_map[i].registered = 1;
	}
}
static void unregister_hpsa_ioctl32(void)
{
	int i, rc;

	for (i=0; i < NHPSA_IOCTL32_ENTRIES; i++) {
		if (!hpsa_ioctl32_map[i].registered)
			continue;
		rc = unregister_ioctl32_conversion(
			hpsa_ioctl32_map[i].cmd);
		if (rc == 0) {
			hpsa_ioctl32_map[i].registered = 0;
			continue;
		}
		printk(KERN_WARNING "hpsa: failed to unregister "
			"32 bit compatible ioctl 0x%08x\n",
			hpsa_ioctl32_map[i].cmd);
	}
}
int hpsa_ioctl32_passthru(unsigned int fd, unsigned int cmd, unsigned long arg, 
	struct file *file)
{
	IOCTL32_Command_struct *arg32 = 
		(IOCTL32_Command_struct *) arg;
	IOCTL_Command_struct arg64;
	mm_segment_t old_fs; 
	int err;
	unsigned long cp;

	err = 0;
	err |= copy_from_user(&arg64.LUN_info, &arg32->LUN_info, sizeof(arg64.LUN_info));
	err |= copy_from_user(&arg64.Request, &arg32->Request, sizeof(arg64.Request));
	err |= copy_from_user(&arg64.error_info, &arg32->error_info, sizeof(arg64.error_info));
	err |= get_user(arg64.buf_size, &arg32->buf_size);
	err |= get_user(cp, &arg32->buf);
	arg64.buf = (BYTE *)cp;
	if (err) 
		return -EFAULT; 

	old_fs = get_fs();
	set_fs(KERNEL_DS);
	err = sys_ioctl(fd, CCISS_PASSTHRU, (unsigned long) &arg64);
	set_fs(old_fs);
	if (err)
		return err;
	err |= copy_to_user(&arg32->error_info, &arg64.error_info, sizeof(arg32->error_info));
	if (err) 
		return -EFAULT; 
	return err;
}
int hpsa_ioctl32_big_passthru(unsigned int fd, unsigned int cmd, unsigned long arg, 
	struct file *file)
{
	BIG_IOCTL32_Command_struct *arg32 = 
		(BIG_IOCTL32_Command_struct *) arg;
	BIG_IOCTL_Command_struct arg64;
	mm_segment_t old_fs; 
	int err;
	unsigned long cp;

	err = 0;
	err |= copy_from_user(&arg64.LUN_info, &arg32->LUN_info, sizeof(arg64.LUN_info));
	err |= copy_from_user(&arg64.Request, &arg32->Request, sizeof(arg64.Request));
	err |= copy_from_user(&arg64.error_info, &arg32->error_info, sizeof(arg64.error_info));
	err |= get_user(arg64.buf_size, &arg32->buf_size);
	err |= get_user(arg64.malloc_size, &arg32->malloc_size);
	err |= get_user(cp, &arg32->buf);
	arg64.buf = (BYTE *)cp;

	if (err) 
		return -EFAULT; 

	old_fs = get_fs();
	set_fs(KERNEL_DS);
	err = sys_ioctl(fd, CCISS_BIG_PASSTHRU, (unsigned long) &arg64);
	set_fs(old_fs);
	if (err)
		return err;
	err |= copy_to_user(&arg32->error_info, &arg64.error_info, sizeof(arg32->error_info));
	if (err) 
		return -EFAULT; 
	return err;
}
#else 
static inline void register_hpsa_ioctl32(void) {}
static inline void unregister_hpsa_ioctl32(void) {}
#endif

#if NUM_GENDISK > 2
#error this code is meant to use 2 gendisks
#endif

/*
 * ioctl 
 */
static int hpsa_ioctl(Scsi_Device *dev, int cmd, void *arg)
{
	int ctlr, rc;
	ctlr_info_t *ci;
	
#ifdef HPSA_DEBUG
	printk(KERN_DEBUG "hpsa_ioctl: Called with cmd=%x %lx\n", cmd, arg);
#endif /* HPSA_DEBUG */ 

	ci = (ctlr_info_t *) dev->host->hostdata[0];
	if (ci == NULL) {
		printk(KERN_ERR "hpsa: hpsa_ioctl hostdata is NULL for host %d.\n", 
			dev->host->host_no);
		return -EINVAL;
	}
	ctlr = ci->ctlr;	/* Get our index into the hba[] array */
#ifdef HPSA_DEBUG
	printk(KERN_INFO "hpsa: hpsa_ioctl for hba %d with hostno %d.\n",
			ctlr, dev->host->host_no);
#endif /* HPSA_DEBUG */ 

	switch(cmd) {
	
	/* IOCTL 3 is generated during rescans, and was the only signal found 
	 * to trigger so that we could recognize newly added or removed drives.
	 */
	case 3:
	{
		printk(KERN_ERR "hpsa: rescanning for configuration changes.\n");
		hpsa_update_scsi_devices(ctlr, dev->host->host_no);
		return 0;
	}

	case CCISS_GETPCIINFO:
	{
		hpsa_pci_info_struct pciinfo;

		if (!arg) 
			return -EINVAL;
		pciinfo.bus = hba[ctlr]->pdev->bus->number;
		pciinfo.dev_fn = hba[ctlr]->pdev->devfn;
		pciinfo.board_id = hba[ctlr]->board_id;
		if (copy_to_user((void *) arg, &pciinfo,  sizeof( hpsa_pci_info_struct )))
			return  -EFAULT;
		return 0;
	}	
        case CCISS_GETDRIVVER:
        {
		DriverVer_type DriverVer = DRIVER_VERSION;

                if (!arg) 
			return -EINVAL;

                if (copy_to_user((void *) arg, &DriverVer, sizeof( DriverVer_type) ))
                	return -EFAULT;
                return 0;
        }
	case CCISS_DEREGDISK:
	{
		printk(KERN_ERR "hpsa: deregdisk: rescanning for configuration changes.\n");
		hpsa_update_scsi_devices(ctlr, dev->host->host_no);
		return 0;
	}
	case CCISS_REGNEWD:
	{
		printk(KERN_ERR "hpsa: regnewd: rescanning for configuration changes.\n");
		hpsa_update_scsi_devices(ctlr, dev->host->host_no);
		return 0;
	}
        case CCISS_REGNEWDISK:
	{
		printk(KERN_ERR "hpsa: regnewdisk: rescanning for configuration changes.\n");
		hpsa_update_scsi_devices(ctlr, dev->host->host_no);
		return 0;
	}
	case CCISS_PASSTHRU:
	{
		IOCTL_Command_struct iocommand;
		ctlr_info_t *h = hba[ctlr];
		CommandList_struct *c;
		char 	*buff = NULL;
		u64bit	temp64;
		unsigned long flags;
#ifdef __VMKERNEL_MODULE__
		struct completion *wait;
#else
		DECLARE_COMPLETION(wait);
#endif
		if (!arg) {
			return -EINVAL;
		}	
		if (!capable(CAP_SYS_RAWIO))  {
			return -EPERM;
		}


		if (copy_from_user(&iocommand, (void *) arg, sizeof( IOCTL_Command_struct) ))
		{
			printk(KERN_WARNING "hpsa: hpsa_ioctl: copy_from_user failed.\n");
			return -EFAULT;
		}

		if ((iocommand.buf_size < 1) && 
				(iocommand.Request.Type.Direction 
				 	!= XFER_NONE)) {	
			return -EINVAL;
		} 

		/* Check kmalloc limits */
		if (iocommand.buf_size > 128000)
			return -EINVAL;
		if (iocommand.buf_size > 0) {
			buff =  kmalloc(iocommand.buf_size, GFP_KERNEL);
			if (buff == NULL) 
				return -ENOMEM;
		}

		if (iocommand.Request.Type.Direction == XFER_WRITE) {
			/* Copy the data into the buffer we created */ 

			if (copy_from_user(buff, iocommand.buf, iocommand.buf_size)) {
				kfree(buff);
				printk(KERN_WARNING "hpsa: hpsa_ioctl: error copying iocommand.buf to buff\n");
				return -EFAULT;
			}
		} else {
			memset(buff, 0, iocommand.buf_size);
		}
		if ((c = cmd_alloc(h , 0)) == NULL) {
			kfree(buff);
			printk(KERN_WARNING "hpsa: hpsa_ioctl: error on cmd_alloc\n");
			return -ENOMEM;
		}
#ifdef __VMKERNEL_MODULE__
		wait = (struct completion *) kmalloc(sizeof(struct completion),
						GFP_ATOMIC);
		if (wait == NULL)
		{
			cmd_free(h, c, 0);
			kfree(buff);
			return -ENOMEM;
		}
#endif
		/* Fill in the command type */
		c->cmd_type = CMD_IOCTL_PEND;
		/* Fill in Command Header */
		c->Header.ReplyQueue = 0;  /* unused in simple mode */
		if (iocommand.buf_size > 0) { 	/* buffer to fill */
			c->Header.SGList = 1;
			c->Header.SGTotal= 1;
		} else	{  /* no buffers to fill  */
			c->Header.SGList = 0;
                	c->Header.SGTotal= 0;
		}
		c->Header.LUN = iocommand.LUN_info;
		c->Header.Tag.lower = c->busaddr;  /* use the kernel address */
						/* the cmd block for tag */
		
		/* Fill in Request block */
		c->Request = iocommand.Request; 

		/* Fill in the scatter gather information */
		if (iocommand.buf_size > 0 ) {
#ifndef __VMKERNEL_MODULE__
			temp64.val = pci_map_single( h->pdev, buff,
                                        iocommand.buf_size, 
                	                PCI_DMA_BIDIRECTIONAL);	
#else
			temp64.val = virt_to_phys(buff);

#endif
			c->SG[0].Addr.lower = temp64.val32.lower;
			c->SG[0].Addr.upper = temp64.val32.upper;
			c->SG[0].Len = iocommand.buf_size;
			c->SG[0].Ext = 0;  /* we are not chaining */
			
		}
#ifdef __VMKERNEL_MODULE__
		init_completion(wait);
		c->waiting = wait;
#else
		c->waiting = &wait;
#endif

		/* Put the request on the tail of the request queue */
		spin_lock_irqsave(&io_request_lock, flags);
		addQ(&h->reqQ, c);
		h->Qdepth++;
		start_io(h);
		spin_unlock_irqrestore(&io_request_lock, flags);

#ifdef __VMKERNEL_MODULE__
		wait_for_completion(wait);
		kfree(wait);
#else
		wait_for_completion(&wait);
#endif

		/* unlock the buffers from DMA */
		if (iocommand.buf_size > 0 ) {
			temp64.val32.lower = c->SG[0].Addr.lower;
                	temp64.val32.upper = c->SG[0].Addr.upper;
#ifndef __VMKERNEL_MODULE__
                	pci_unmap_single( h->pdev, (dma_addr_t) temp64.val,
                		iocommand.buf_size, PCI_DMA_BIDIRECTIONAL);
#endif
		}

		/* Copy the error information out */ 
		iocommand.error_info = *(c->err_info);
		if (copy_to_user((void *) arg, &iocommand, 
				sizeof( IOCTL_Command_struct) ) ) {
			kfree(buff);
			cmd_free(h, c, 0);
			printk(KERN_WARNING "hpsa: hpsa_ioctl: error copying out\n");
			return( -EFAULT);
		} 	

		if (iocommand.Request.Type.Direction == XFER_READ) {
                        /* Copy the data out of the buffer we created */

                        rc= copy_to_user(iocommand.buf, buff, 
						iocommand.buf_size);
                        if (rc!=0) {
				printk(KERN_WARNING "hpsa: hpsa_ioctl: Could not copy %d bytes.\n", rc);
                        	kfree(buff);
				cmd_free(h, c, 0);
				return -EFAULT;
			}
                }
                kfree(buff);
		cmd_free(h, c, 0);
                return 0;
	} 
	case CCISS_BIG_PASSTHRU:
	{
		BIG_IOCTL_Command_struct iocommand;
		ctlr_info_t *h = hba[ctlr];
		CommandList_struct *c;
		char 	*buff[MAXSGENTRIES] = {NULL,};
		int	buff_size[MAXSGENTRIES] = {0,};
		u64bit	temp64;
		unsigned long flags;
		BYTE sg_used = 0;
		int status = 0;
		int i;
#ifdef __VMKERNEL_MODULE__
		struct completion *wait = NULL;
#else
		DECLARE_COMPLETION(wait);
#endif
		if (!arg) 
			return -EINVAL;
		
		if (!capable(CAP_SYS_RAWIO)) 
			return -EPERM;

		if (copy_from_user(&iocommand, (void *) arg, sizeof( BIG_IOCTL_Command_struct) ))
			return -EFAULT;
		if ((iocommand.buf_size < 1) && 
			(iocommand.Request.Type.Direction != XFER_NONE)) {
			return -EINVAL;
		} 
		/* Check kmalloc limits  using all SGs */
		if (iocommand.malloc_size > MAX_KMALLOC_SIZE)
			return -EINVAL;
		if (iocommand.buf_size > iocommand.malloc_size * MAXSGENTRIES)
			return -EINVAL;
		if (iocommand.buf_size > 0) {
			__u32   size_left_alloc = iocommand.buf_size;
			BYTE    *data_ptr = (BYTE *) iocommand.buf;
			while (size_left_alloc > 0) {
				buff_size[sg_used] = (size_left_alloc 
							> iocommand.malloc_size)
					? iocommand.malloc_size : size_left_alloc;
				buff[sg_used] = kmalloc( buff_size[sg_used], 
						GFP_KERNEL);
				if (buff[sg_used] == NULL) {
					status = -ENOMEM;
					goto cleanup1;
				}
				if (iocommand.Request.Type.Direction == 
						XFER_WRITE) {
				   /* Copy the data into the buffer created */
				   if (copy_from_user(buff[sg_used], data_ptr, 
						buff_size[sg_used])) {
					status = -ENOMEM;
					goto cleanup1;			
				   }	
				   } else
					memset(buff[sg_used], 0, buff_size[sg_used]);
				size_left_alloc -= buff_size[sg_used];
				data_ptr += buff_size[sg_used];
				sg_used++;
			}
			
		}
		if ((c = cmd_alloc(h , 0)) == NULL) {
			status = -ENOMEM;
			goto cleanup1;	
		}
#ifdef __VMKERNEL_MODULE__
		wait = (struct completion *)
		kmalloc(sizeof(struct completion), GFP_ATOMIC);
		if (!wait) {
			cmd_free(h, c, 0);
			status = -ENOMEM;
			goto cleanup1;
		}
#endif
		/* Fill in the command type */
		c->cmd_type = CMD_IOCTL_PEND;
		/* Fill in Command Header */
		c->Header.ReplyQueue = 0;  /* unused in simple mode */
		
		if (iocommand.buf_size > 0) { 	/* buffer to fill */
			c->Header.SGList = sg_used;
			c->Header.SGTotal= sg_used;
		} else	{	/* no buffers to fill */
			c->Header.SGList = 0;
			c->Header.SGTotal= 0;
		}
		c->Header.LUN = iocommand.LUN_info;
		c->Header.Tag.lower = c->busaddr;  /* use the kernel address */
						/* the cmd block for tag */
		
	/* Fill in Request block */
	c->Request = iocommand.Request; 
	/* Fill in the scatter gather information */
	if (iocommand.buf_size > 0 ) {
		int i;
		for(i=0; i< sg_used; i++) {
#ifndef __VMKERNEL_MODULE__
			temp64.val = pci_map_single( h->pdev, buff[i], 
					buff_size[i], 
					PCI_DMA_BIDIRECTIONAL);
#else
			temp64.val = virt_to_phys(buff[i]);
#endif

			c->SG[i].Addr.lower = temp64.val32.lower;
			c->SG[i].Addr.upper = temp64.val32.upper;
			c->SG[i].Len = buff_size[i];
			c->SG[i].Ext = 0;  /* we are not chaining */
		}
	}
#ifdef __VMKERNEL_MODULE__
	init_completion(wait);
	c->waiting = wait;
#else
	c->waiting = &wait;
#endif
	/* Put the request on the tail of the request queue */
	spin_lock_irqsave(&io_request_lock, flags);
	addQ(&h->reqQ, c);
	h->Qdepth++;
	start_io(h);
	spin_unlock_irqrestore(&io_request_lock, flags);
#ifdef __VMKERNEL_MODULE__
	wait_for_completion(wait);
	kfree(wait);
#else
	wait_for_completion(&wait);
#endif
	/* unlock the buffers from DMA */
	for(i=0; i< sg_used; i++) {
		temp64.val32.lower = c->SG[i].Addr.lower;
		temp64.val32.upper = c->SG[i].Addr.upper;
#ifndef __VMKERNEL_MODULE__
		pci_unmap_single( h->pdev, (dma_addr_t) temp64.val,
				buff_size[i], PCI_DMA_BIDIRECTIONAL);
#endif
	}
	/* Copy the error information out */
		iocommand.error_info = *(c->err_info);
		if (copy_to_user((void *) arg, &iocommand, 
					sizeof( IOCTL_Command_struct) ) ) {
				cmd_free(h, c, 0);
				status = -EFAULT;
				goto cleanup1;
		}
		if (iocommand.Request.Type.Direction == XFER_READ) {
		/* Copy the data out of the buffer we created */
			BYTE *ptr = (BYTE  *) iocommand.buf;
	        	for(i=0; i< sg_used; i++) {
				if (copy_to_user(ptr, buff[i], buff_size[i])) {
					cmd_free(h, c, 0);
					status = -EFAULT;
					goto cleanup1;

				}
				ptr += buff_size[i];
			}
		}
		cmd_free(h, c, 0);
		status = 0;
		

cleanup1:
		for(i=0; i< sg_used; i++) {
			if (buff[i] != NULL)
				kfree(buff[i]);
		}
		return status;
	}
	default:
	{
		printk(KERN_WARNING "hpsa: Unknown IOCTL %d\n", cmd);
					
		return -EBADRQC;
	}
	}
}


/*
 *   Wait polling for a command to complete.
 *   The memory mapped FIFO is polled for the completion.
 *   Used only at init time, interrupts disabled.
 */
static unsigned long pollcomplete(int ctlr)
{
	unsigned long done;
	int i;

	/* Wait (up to 20 seconds) for a command to complete */

        for (i = 20 * HZ; i > 0; i--) {
		done = hba[ctlr]->access.command_completed(hba[ctlr]);
		if (done == FIFO_EMPTY) {
#ifdef __VMKERNEL_MODULE__
			//VMware Porting guide says to use mdelay for 'short' timeouts instead of schedule_timeout:
			mdelay(1);
#else
			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(1);
#endif

		} else
		{
			return done;
		}
	}
	/* Invalid address to tell caller we ran out of time */
	return 1;
}
/*
 * Send a command to the controller, and wait for it to complete.  
 * Only used at init time. 
 */
static int sendcmd(
	__u8	cmd,
	int	ctlr,
	void	*buff,
	size_t	size,
	unsigned int use_unit_num, /* 0: address the controller,
				      1: address logical volume log_unit,
				      2: periph device address is scsi3addr */
	unsigned int log_unit,
	__u8	page_code,
	unsigned char *scsi3addr,
        int block_nr)
{
	CommandList_struct *c;
	CommandList_struct *c2=buff;
	ErrorInfo_struct *err_ptr;
	dma_addr_t tmp_cmd_dma_handle;
	u64bit tmp_err_dma_handle;
	int i;
	unsigned long complete;
	ctlr_info_t *info_p= hba[ctlr];
	u64bit buff_dma_handle;
	int status = IO_OK;
	u64bit temp64;
	int temp;
	int nr_blocks;
	int done=0;			// done processing pollcomplete loop

	c = cmd_alloc(info_p, 1);

	if (c == NULL) {
		printk(KERN_WARNING "hpsa: unable to get memory");
		return IO_ERROR;
	}

	/* Fill in Command Header */
	c->Header.ReplyQueue = 0;  /* unused in simple mode */
	//if (buff != NULL) { 	/* buffer to fill */
	if ((buff != NULL) && (size > 0))  { 	/* buffer to fill */
		c->Header.SGList = 1;
		c->Header.SGTotal= 1;
	} else	{	/* no buffers to fill  */
		c->Header.SGList = 0;
                c->Header.SGTotal= 0;
	}
	c->Header.Tag.lower = c->busaddr;  /* use the kernel address */
					   /* the cmd block for tag */
	/* Fill in Request block */
	switch (cmd) {
		case  HPSA_INQUIRY:
			/* If the logical unit number is 0 then, this is going
				to controller so It's a physical command
				mode = 0 target = 0.
				So we have nothing to write. 
				otherwise, if use_unit_num == 1,
				mode = 1(volume set addressing) target = LUNID
				otherwise, if use_unit_num == 2,
				mode = 0(periph dev addr) target = scsi3addr
			*/
			if (use_unit_num == 1) {
				c->Header.LUN.LogDev.VolId=
                                	hba[ctlr]->drv[log_unit].LunID;
                        	c->Header.LUN.LogDev.Mode = 1;
			}
			else if (use_unit_num == 2) {
				memcpy(c->Header.LUN.LunAddrBytes,scsi3addr,8);
				c->Header.LUN.LogDev.Mode = 0; 
							/* phys dev addr */
			}

			/* are we trying to read a vital product page */
			if (page_code != 0) {
				c->Request.CDB[1] = 0x01;
				c->Request.CDB[2] = page_code;
			}
			c->Request.CDBLen = 6;
			c->Request.Type.Type =  TYPE_CMD; /* It is a command. */
			c->Request.Type.Attribute = ATTR_SIMPLE;  
			c->Request.Type.Direction = XFER_READ; /* Read */
			c->Request.Timeout = 0; /* Don't time out */
			c->Request.CDB[0] =  HPSA_INQUIRY;
			c->Request.CDB[4] = size  & 0xFF;  
		break;
		case HPSA_REPORT_LOG:
		case HPSA_REPORT_PHYS:
                        /* Talking to controller so It's a physical command
                                mode = 00 target = 0.
                                So we have nothing to write.
                        */
                        c->Request.CDBLen = 12;
                        c->Request.Type.Type =  TYPE_CMD; /* It is a command. */
                        c->Request.Type.Attribute = ATTR_SIMPLE; 
                        c->Request.Type.Direction = XFER_READ; /* Read */
                        c->Request.Timeout = 0; /* Don't time out */
			c->Request.CDB[0] = cmd;
                        c->Request.CDB[6] = (size >> 24) & 0xFF;  /* MSB */
                        c->Request.CDB[7] = (size >> 16) & 0xFF;
                        c->Request.CDB[8] = (size >> 8) & 0xFF;
                        c->Request.CDB[9] = size & 0xFF;
                break;

		case HPSA_READ_CAPACITY:
			c->Header.LUN.LogDev.VolId= 
				hba[ctlr]->drv[log_unit].LunID;
			c->Header.LUN.LogDev.Mode = 1;
			c->Request.CDBLen = 10;
                        c->Request.Type.Type =  TYPE_CMD; /* It is a command. */
                        c->Request.Type.Attribute = ATTR_SIMPLE; 
                        c->Request.Type.Direction = XFER_READ; /* Read */
                        c->Request.Timeout = 0; /* Don't time out */
                        c->Request.CDB[0] = HPSA_READ_CAPACITY;
		break;
		case HPSA_CACHE_FLUSH:
			c->Request.CDBLen = 12;
			c->Request.Type.Type =  TYPE_CMD; /* It is a command. */
			c->Request.Type.Attribute = ATTR_SIMPLE;
			c->Request.Type.Direction = XFER_WRITE; /* No data */
			c->Request.Timeout = 0; /* Don't time out */
			c->Request.CDB[0] = BMIC_WRITE;  /* BMIC Passthru */
			c->Request.CDB[6] = BMIC_CACHE_FLUSH;
		break;
		case HPSA_READ:
			if (use_unit_num == 1) {
				c->Header.LUN.LogDev.VolId =
					hba[ctlr]->drv[log_unit].LunID;
				c->Header.LUN.LogDev.Mode = 1;
			}
			c->ctlr = ctlr;
			c->cmd_type = CMD_RWREQ;
			c->Request.CDBLen = 10;
			c->Request.Type.Type = TYPE_CMD;
			c->Request.Type.Attribute = ATTR_SIMPLE;
			c->Request.Type.Direction = XFER_READ;
			c->Request.Timeout = 0;
			c->Request.CDB[0] = HPSA_READ;
			c->Request.CDB[2] = (block_nr >> 24) & 0xff;
			c->Request.CDB[3] = (block_nr >> 16) & 0xff;
			c->Request.CDB[4] = (block_nr >> 8) & 0xff;
			c->Request.CDB[5] = block_nr & 0xff;
			c->Request.CDB[6] = 0;
			nr_blocks = size/hba[ctlr]->drv[log_unit].block_size;
			c->Request.CDB[7] = (nr_blocks >> 8) & 0xff;
			c->Request.CDB[8] = nr_blocks & 0xff;
			c->Request.CDB[9] = 0;
		break;
		case HPSA_WRITE:
			if (use_unit_num == 1) {
				c->Header.LUN.LogDev.VolId =
					hba[ctlr]->drv[log_unit].LunID;
				c->Header.LUN.LogDev.Mode = 1;
			}
			c->ctlr = ctlr;
			c->cmd_type = CMD_RWREQ;
			c->Request.CDBLen = 10;
			c->Request.Type.Type = TYPE_CMD;
			c->Request.Type.Attribute = ATTR_SIMPLE;
			c->Request.Type.Direction = XFER_WRITE;
			c->Request.Timeout = 0;
			c->Request.CDB[0] = HPSA_WRITE;
			c->Request.CDB[2] = (block_nr >> 24) & 0xff;
			c->Request.CDB[3] = (block_nr >> 16) & 0xff;
			c->Request.CDB[4] = (block_nr >> 8) & 0xff;
			c->Request.CDB[5] = block_nr & 0xff;
			c->Request.CDB[6] = 0;
			nr_blocks = size/hba[ctlr]->drv[log_unit].block_size;
			c->Request.CDB[7] = (nr_blocks >> 8) & 0xff;
			c->Request.CDB[8] = nr_blocks & 0xff;
			c->Request.CDB[9] = 0;
		break;
		case  HPSA_DEVICE_RESET_MSG:
			/* If the logical unit number is 0 then, this is going
				to controller so It's a physical command
				mode = 0 target = 0.
				So we have nothing to write. 
				otherwise, if use_unit_num == 1,
				mode = 1(volume set addressing) target = LUNID
				otherwise, if use_unit_num == 2,
				mode = 0(periph dev addr) target = scsi3addr
			*/
			if (use_unit_num == 1) {
				c->Header.LUN.LogDev.VolId=
                                	hba[ctlr]->drv[log_unit].LunID;
                        	c->Header.LUN.LogDev.Mode = 1;
			}
			else if (use_unit_num == 2) {
				memcpy(c->Header.LUN.LunAddrBytes,scsi3addr,8);
				c->Header.LUN.LogDev.Mode = 0; 
							/* phys dev addr */
			}

			c->Request.CDBLen = 12;
			c->Request.Type.Type =  1; /* It is a MSG as opposed to a CMD */
			c->Request.Type.Attribute = ATTR_SIMPLE;  
			c->Request.Type.Direction = XFER_WRITE; /* Write */
			c->Request.Timeout = 0; /* Don't time out */
			c->Request.CDB[0] =  0x01; // RESET_MSG is 0x01
			c->Request.CDB[1] = 0x04;  // Reset LunID above
			c->Request.CDB[4] = 0x00;  // If bytes 4-7 are zero, it means reset the LunID device
			c->Request.CDB[5] = 0x00;  
			c->Request.CDB[6] = 0x00; 
			c->Request.CDB[7] = 0x00;
		break;

		case  HPSA_ABORT_MSG:
			/* Emulating aborts with a bus reset */
		case  HPSA_BUS_RESET_MSG:
			/* If the logical unit number is 0 then, this is going
				to controller so It's a physical command
				mode = 0 target = 0.
				So we have nothing to write. 
				otherwise, if use_unit_num == 1,
				mode = 1(volume set addressing) target = LUNID
				otherwise, if use_unit_num == 2,
				mode = 0(periph dev addr) target = scsi3addr
			*/
			if (use_unit_num == 1) {
				c->Header.LUN.LogDev.VolId=
                                	hba[ctlr]->drv[log_unit].LunID;
                        	c->Header.LUN.LogDev.Mode = 1;
			}
			else if (use_unit_num == 2) {
				memcpy(c->Header.LUN.LunAddrBytes,scsi3addr,8);
				c->Header.LUN.LogDev.Mode = 0; 
							/* phys dev addr */
			}

			c->Request.CDBLen = 12;
			c->Request.Type.Type =  1; /* It is a MSG as opposed to a CMD */
			c->Request.Type.Attribute = ATTR_SIMPLE;  
			c->Request.Type.Direction = XFER_WRITE; /* Write */
			c->Request.Timeout = 0; /* Don't time out */
			c->Request.CDB[0] =  0x01; // RESET_MSG is 0x01
			c->Request.CDB[1] = 0x04;  // Reset Bus of LunID above
			c->Request.CDB[7] = 0xff;  // If bytes 4-7 are non-zero it means reset the bus associated with the specified LunID
		break;
		case  HPSA_HOST_RESET_MSG:
			/* If the logical unit number is 0 then, this is going
				to controller so It's a physical command
				mode = 0 target = 0.
				So we have nothing to write. 
				otherwise, if use_unit_num == 1,
				mode = 1(volume set addressing) target = LUNID
				otherwise, if use_unit_num == 2,
				mode = 0(periph dev addr) target = scsi3addr
			*/
			if (use_unit_num == 1) {
				c->Header.LUN.LogDev.VolId=
                                	hba[ctlr]->drv[log_unit].LunID;
                        	c->Header.LUN.LogDev.Mode = 1;
			}
			else if (use_unit_num == 2) {
				memcpy(c->Header.LUN.LunAddrBytes,scsi3addr,8);
				c->Header.LUN.LogDev.Mode = 0; 
							/* phys dev addr */
			}

			c->Request.CDBLen = 12;
			c->Request.Type.Type =  1; /* It is a MSG as opposed to a CMD */
			c->Request.Type.Attribute = ATTR_SIMPLE;  
			c->Request.Type.Direction = XFER_WRITE; /* Write */
			c->Request.Timeout = 0; /* Don't time out */
			c->Request.CDB[0] =  0x01; // RESET_MSG is 0x01
			c->Request.CDB[1] = 0x00;  // Reset Host
		break;
		default:
			printk(KERN_WARNING
				"hpsa:  Unknown Command 0x%x sent attempted\n",
				  cmd);
			cmd_free(info_p, c, 1);
			return IO_ERROR;
	};
	/* Fill in the scatter gather information */
	if (size > 0) {
#ifndef __VMKERNEL_MODULE__
		buff_dma_handle.val = (__u64) pci_map_single( info_p->pdev, 
			buff, size, PCI_DMA_BIDIRECTIONAL);
#else
		buff_dma_handle.val = (__u64)virt_to_phys(buff);
#endif
		c->SG[0].Addr.lower = buff_dma_handle.val32.lower;
		c->SG[0].Addr.upper = buff_dma_handle.val32.upper;
		c->SG[0].Len = size;
		c->SG[0].Ext = 0;  /* we are not chaining */
	}
resend_cmd1:
	/*
         * Disable interrupt
         */
        info_p->access.set_intr_mask(info_p, HPSA_INTR_OFF);
	
	/* Make sure there is room in the command FIFO */
        /* Actually it should be completely empty at this time. */
        for (i = 200000; i > 0; i--) {
		/* if fifo isn't full go */
                if (!(info_p->access.fifo_full(info_p))) {
			
                        break;
                }
                udelay(10);
#ifdef HPSA_DEBUG
                printk(KERN_WARNING "hpsa hpsa%d: SendCmd FIFO full,"
                        " waiting!\n", ctlr);
#endif /* HPSA_DEBUG */ 
        }
        /*
         * Send the cmd
         */

        info_p->access.submit_command(info_p, c);

	done=0;

	//Find 'our' command completion, put others in reject list to be reprocessed later.
	do {
        	complete = pollcomplete(ctlr);


		if (complete == 1) {
			printk(KERN_WARNING
			"hpsa hpsa%d: SendCmd Timeout, "
			"No command list address returned!\n",
			ctlr);
			status = IO_ERROR;
			done=1;
			break;
		}


		/* This will need changing for direct lookup completions */
		if ( (complete & ~HPSA_ERROR_BIT) != c->busaddr) {
			// Save unexpected completions for processing later.
			if (add_sendcmd_reject(cmd, ctlr, complete) != 0) {
				BUG();  /* we are pretty much hosed if we get here. */
			}
			continue;
		} else
			done = 1;
		
		if ( (complete & HPSA_ERROR_BIT)
		     && (complete & ~HPSA_ERROR_BIT) == c->busaddr) {
			/* if data overrun or underun on Report command 
				ignore it 
			*/
			if (((c->Request.CDB[0] == HPSA_REPORT_LOG) ||
			     (c->Request.CDB[0] == HPSA_REPORT_PHYS) ||
			     (c->Request.CDB[0] == HPSA_INQUIRY)) &&
				((c->err_info->CommandStatus == 
					CMD_DATA_OVERRUN) || 
				 (c->err_info->CommandStatus == 
					CMD_DATA_UNDERRUN)
			 	)) {
				complete = c->busaddr;
			} else {
				if (c->err_info->CommandStatus == 
						CMD_UNSOLICITED_ABORT) {
					printk(KERN_WARNING "hpsa%d: "
						"cmd aborted due "
					"to an unsolicited abort \n", ctlr); 
					if (c->retry_count < MAX_CMD_RETRIES) {
						printk(KERN_WARNING
						   "retrying cmd\n");
						c->retry_count++;
						/* erase the old error */
						/* information */
						memset(c->err_info, 0, 
						   sizeof(ErrorInfo_struct));
						goto resend_cmd1;
					} else {
						printk(KERN_WARNING
						   "retried too many times\n");
						status = IO_ERROR;
						goto cleanup1;
					}
				}
				printk(KERN_WARNING "hpsa hpsa%d: sendcmd"
				" Error %x \n", ctlr, 
					c->err_info->CommandStatus); 
				printk(KERN_WARNING "hpsa hpsa%d: sendcmd"
				" offensive info\n"
				"  size %x\n   num %x   value %x\n", ctlr,
				  c->err_info->MoreErrInfo.Invalid_Cmd.offense_size,
				  c->err_info->MoreErrInfo.Invalid_Cmd.offense_num,
				  c->err_info->MoreErrInfo.Invalid_Cmd.offense_value);
				status = IO_ERROR;
				goto cleanup1;
			}
		}
			

		
	} while (!done);


cleanup1:	
	if (status == IO_ERROR)
		print_cmd(c);

#ifndef __VMKERNEL_MODULE__
	/* unlock the data buffer from DMA */
	pci_unmap_single(info_p->pdev, (dma_addr_t) buff_dma_handle.val,
                                size, PCI_DMA_BIDIRECTIONAL);
#endif

	/* if we saved some commands for later, process them now. */
	if (info_p->scsi_rejects.ncompletions > 0) {
		do_hpsa_intr(0, info_p, 0);
	}
	cmd_free(info_p, c, 1);

        return status;
} 
static int add_sendcmd_reject(__u8 cmd, int ctlr, unsigned long complete)
{
        /* We get in here if sendcmd() is polling for completions
           and gets some command back that it wasn't expecting --
           something other than that which it just sent down.
           Ordinarily, that shouldn't happen, but it can happen when
           the scsi stuff gets into error handling mode, and
           starts using sendcmd() to try to abort commands and
           reset drives.  In that case, sendcmd may pick up
           completions of commands that were sent to logical drives.
           In that case, we need to save those completions for later
           processing by the interrupt handler.
         */

        struct sendcmd_reject_list *srl = &hba[ctlr]->scsi_rejects;

        /* If it's not the scsi stuff doing error handling, (abort, inquiry, report phys and logical, */
        /* or reset or cache flush) then we don't expect anything weird. */
        if (	cmd != HPSA_DEVICE_RESET_MSG &&  
		cmd != HPSA_BUS_RESET_MSG && 
		cmd != HPSA_HOST_RESET_MSG && 
		cmd != HPSA_ABORT_MSG && 
		cmd != HPSA_REPORT_PHYS && 
		cmd != HPSA_INQUIRY && 
		cmd != HPSA_CACHE_FLUSH) {

                printk(KERN_WARNING "hpsa hpsa%d: add_sendcmd_reject "
                       "Invalid command type (%lx)\n",
                       ctlr, complete);
                /* not much we can do. */
                return 1;
        }

        if (srl->ncompletions >= (NR_CMDS + 2)) {
                /* Uh oh.  No room to save it for later... */
                printk(KERN_WARNING "hpsa%d: add_sendcmd_reject: "
                       "reject list overflow, command lost!\n", ctlr);
                return 1;
        }
        // We've sent down an abort or reset, but something else
        //  has completed. Save it for later processing.
        srl->complete[srl->ncompletions] = complete;
        srl->ncompletions++;
        return 0;
}

/*
 * Map (physical) PCI mem into (virtual) kernel space
 */
static ulong remap_pci_mem(ulong base, ulong size)
{
        ulong page_base        = ((ulong) base) & PAGE_MASK;
        ulong page_offs        = ((ulong) base) - page_base;
        ulong page_remapped    = (ulong) ioremap(page_base, page_offs+size);

        return (ulong) (page_remapped ? (page_remapped + page_offs) : 0UL);
}

/*
 * Enqueuing and dequeuing functions for cmdlists.
 */
static inline void addQ(CommandList_struct **Qptr, CommandList_struct *c)
{
        if (*Qptr == NULL) {
                *Qptr = c;
                c->next = c->prev = c;
        } else {
                c->prev = (*Qptr)->prev;
                c->next = (*Qptr);
                (*Qptr)->prev->next = c;
                (*Qptr)->prev = c;
        }
}

static inline CommandList_struct *removeQ(CommandList_struct **Qptr, 
						CommandList_struct *c)
{
        if (c && c->next != c) {
                if (*Qptr == c) *Qptr = c->next;
                c->prev->next = c->next;
                c->next->prev = c->prev;
        } else {
                *Qptr = NULL;
        }
        return c;
}

/* 
 * Takes jobs of the Q and sends them to the hardware, then puts it on 
 * the Q to wait for completion. 
 */ 
static void start_io( ctlr_info_t *h)
{
	CommandList_struct *c;
	
	while(( c = h->reqQ) != NULL ) {
		/* can't do anything if fifo is full */
		if ((h->access.fifo_full(h))) {
			printk(KERN_WARNING "hpsa: fifo full \n");
			return;
		}
		/* Get the frist entry from the Request Q */ 
		removeQ(&(h->reqQ), c);
		h->Qdepth--;
	
		/* Tell the controller execute command */ 
		h->access.submit_command(h, c);
		
		/* Put job onto the completed Q */ 
		addQ (&(h->cmpQ), c); 
	}
}

static inline void complete_buffers( struct request *req, int status)
{
	struct buffer_head *xbh;
	
	while(req->bh)
	{
		xbh = req->bh->b_reqnext; 
		req->bh->b_reqnext = NULL;


	#ifdef HPQ_HPSA_2_PARAMS
		blk_finished_io(req, req->bh->b_size >> 9);
	#else
		blk_finished_io(req->bh->b_size >> 9);
	#endif
		req->bh->b_end_io(req->bh, status);
		req->bh = xbh;
	}
}
 
/* This code assumes io_request_lock is already held */
/* Zeros out the error record and then resends the command back */
/* to the controller */ 
static inline void resend_hpsa_cmd( ctlr_info_t *h, CommandList_struct *c)
{
	/* erase the old error information */
	memset(c->err_info, 0, sizeof(ErrorInfo_struct));

	/* add it to software queue and then send it to the controller */
	addQ(&(h->reqQ),c);
	h->Qdepth++;
	if (h->Qdepth > h->maxQsinceinit)
		h->maxQsinceinit = h->Qdepth; 

	start_io(h);
}
/* checks the status of the job and calls complete buffers to mark all 
 * buffers for the completed job. 
 */ 
static inline void complete_command( ctlr_info_t *h, CommandList_struct *cmd, 
		int timeout)
{
	int status = 1;
	int retry_cmd = 0;
	int i, ddir;
	int ctlr = h->ctlr;
	u64bit temp64;
		
	if (timeout)
		status = 0; 

	if (cmd->err_info->CommandStatus != 0) { 
		/* an error has occurred */ 
		switch (cmd->err_info->CommandStatus) {
			unsigned char sense_key;
			case CMD_TARGET_STATUS:
				status = 0;
			
				if (cmd->err_info->ScsiStatus == 0x02) {
					printk(KERN_WARNING "hpsa%d: cmd "
						"has CHECK CONDITION,"
						" sense key = 0x%x\n", ctlr,
						cmd->err_info->SenseInfo[2]);
					/* check the sense key */
					sense_key = 0xf & 
						cmd->err_info->SenseInfo[2];
					/* recovered error */
					if ( sense_key == 0x1)
						status = 1;
				} else {
					printk(KERN_WARNING "hpsa%d: cmd "
						"has SCSI Status 0x%x\n",
						ctlr, cmd->err_info->ScsiStatus);
				}
			break;
			case CMD_DATA_UNDERRUN:
				printk(KERN_WARNING "hpsa%d: cmd has"
					" completed with data underrun "
					"reported\n", ctlr);
			break;
			case CMD_DATA_OVERRUN:
				printk(KERN_WARNING "hpsa%d: cmd has"
					" completed with data overrun "
					"reported\n", ctlr);
			break;
			case CMD_INVALID:
				printk(KERN_WARNING "hpsa%d: cmd is "
					"reported invalid\n", ctlr);
				status = 0;
			break;
			case CMD_PROTOCOL_ERR:
                                printk(KERN_WARNING "hpsa%d: cmd has "
					"protocol error \n", ctlr);
                                status = 0;
                        break;
			case CMD_HARDWARE_ERR:
                                printk(KERN_WARNING "hpsa%d: cmd had " 
                                        " hardware error\n", ctlr);
                                status = 0;
                        break;
			case CMD_CONNECTION_LOST:
				printk(KERN_WARNING "hpsa%d: cmd had "
					"connection lost\n", ctlr);
				status=0;
			break;
			case CMD_ABORTED:
				printk(KERN_WARNING "hpsa%d: cmd was "
					"aborted\n", ctlr);
				status=0;
			break;
			case CMD_ABORT_FAILED:
				printk(KERN_WARNING "hpsa%d: cmd reports "
					"abort failed\n", ctlr);
				status=0;
			break;
			case CMD_UNSOLICITED_ABORT:
				printk(KERN_WARNING "hpsa%d: cmd aborted do "
					"to an unsolicited abort \n",
				       	ctlr);
				if (cmd->retry_count < MAX_CMD_RETRIES) {
					retry_cmd=1;
					printk(KERN_WARNING
						"retrying cmd\n");
					cmd->retry_count++;
				} else {
					printk(KERN_WARNING
					"retried to many times\n");
				}
				status=0;
			break;
			case CMD_TIMEOUT:
				printk(KERN_WARNING "hpsa%d: cmd timed out\n",
					ctlr);
				if (cmd->retry_count < MAX_CMD_RETRIES) {
					retry_cmd=1;
					printk(KERN_WARNING
						"retrying cmd\n");
					cmd->retry_count++;
				} else {
					printk(KERN_WARNING
					"retried to many times\n");
				}
				status=0;
			break;
			default:
				printk(KERN_WARNING "hpsa%d: cmd returned "
					"unknown status %x\n", ctlr, 
						cmd->err_info->CommandStatus); 
				status=0;
		}
	}
	/* We need to return this command */
	if (retry_cmd) {
		resend_hpsa_cmd(h,cmd);
		return;
	}	

	if (status == 0)
		print_cmd(cmd);

	/* command did not need to be retried */
	/* unmap the DMA mapping for all the scatter gather elements */
	if (cmd->Request.Type.Direction == XFER_READ)
		ddir = PCI_DMA_FROMDEVICE;
	else
		ddir = PCI_DMA_TODEVICE;
	for(i=0; i<cmd->Header.SGList; i++) {
		temp64.val32.lower = cmd->SG[i].Addr.lower;
		temp64.val32.upper = cmd->SG[i].Addr.upper;
#ifndef __VMKERNEL_MODULE__
		pci_unmap_page(hba[cmd->ctlr]->pdev,
			temp64.val, cmd->SG[i].Len, ddir);
#endif
	}
	complete_buffers(cmd->rq, status);
	end_that_request_last(cmd->rq);
	cmd_free(h,cmd,1);
}


static inline int cpq_new_segment(request_queue_t *q, struct request *rq,
                                  int max_segments)
{
        if (rq->nr_segments < MAXSGENTRIES) {
                rq->nr_segments++;
                return 1;
        }
        return 0;
}

/*
 * can we merge the two segments, or do we need to start a new one?
 *
 * This function mimics the kernel function blk_seg_merge_ok.  It has
 * been defined here because some versions of linux (i.e. rhel 2.1 and
 * redhat 7.3 professional) use a different API for blk_seg_merge_ok that
 * has 3 parameters rather than 2.  However, the third parameter is never
 * used and has been removed from later kernels.  In order to ensure that 
 * this driver will work across all supported distributions this function
 * has been created to only use two parameters but do the same work.  
 * A custom compiler flag would not work here because in some distributions
 * the base kernel and errata kernels differ in the number of parameters
 * passed to the API.  The kernel function blk_seg_merge_ok should be checked
 * in each new errata to make sure that no changes have been made.
 */
inline int hpq_blk_seg_merge_ok(struct buffer_head *bh, struct buffer_head *nxt)
{
	/*
	 * if bh and nxt are contigous and don't cross a 4g boundary, it's ok
	 */
	if (BH_CONTIG(bh, nxt) && BH_PHYS_4G(bh, nxt))
		return 1;

	return 0;
}
static int cpq_back_merge_fn(request_queue_t *q, struct request *rq,
                             struct buffer_head *bh, int max_segments)
{
	if (hpq_blk_seg_merge_ok(rq->bhtail, bh))
                return 1;
        return cpq_new_segment(q, rq, max_segments);
}

static int cpq_front_merge_fn(request_queue_t *q, struct request *rq,
                             struct buffer_head *bh, int max_segments)
{
	if (hpq_blk_seg_merge_ok(bh, rq->bh))
                return 1;
        return cpq_new_segment(q, rq, max_segments);
}

static int cpq_merge_requests_fn(request_queue_t *q, struct request *rq,
                                 struct request *nxt, int max_segments)
{
        int total_segments = rq->nr_segments + nxt->nr_segments;

	if (hpq_blk_seg_merge_ok(rq->bhtail, nxt->bh))
                total_segments--;

        if (total_segments > MAXSGENTRIES)
                return 0;

        rq->nr_segments = total_segments;
        return 1;
}


static void do_hpsa_intr(int irq, void *dev_id, struct pt_regs *regs)
{
	ctlr_info_t *h = dev_id;
	CommandList_struct *c;
	unsigned long flags = 0;
	__u32 a, a1;
	int i;
	int next_to_run = h->start_index;
	int got_lock=0;


	/* Is this interrupt for us? */
	if (((h->access.intr_pending(h) == 0) || (h->interrupts_enabled == 0)) && (h->scsi_rejects.ncompletions == 0))
		return;

	// We need to get lock unless there are rejected commands,
	// in which case SCSI layer already holds the lock.
	if (h->scsi_rejects.ncompletions == 0) {
		got_lock=1;
		spin_lock_irqsave(&io_request_lock, flags);
	}

	/*
	 * If there are completed commands in the completion queue,
	 * we had better do something about it.
	 */
	while( h->access.intr_pending(h) || h->scsi_rejects.ncompletions > 0) {

		while((a = get_next_completion(h)) != FIFO_EMPTY) {
			a1 = a;
			a &= ~3;
			if ((c = h->cmpQ) == NULL) {  
				printk(KERN_WARNING "hpsa: Completion of %08lx ignored\n", (unsigned long)a1);
				continue;	
			} 
			while(c->busaddr != a) {
				c = c->next;
				if (c == h->cmpQ) 
					break;
			}
			/*
			 * If we've found the command, take it off the
			 * completion Q and free it
			 */
			 if (c->busaddr == a) {
				removeQ(&h->cmpQ, c);
				if (c->cmd_type == CMD_RWREQ) {
					complete_command(h, c, 0);

				} else if (c->cmd_type == CMD_IOCTL_PEND) {
					complete(c->waiting);
				}
#if defined CONFIG_SCSI_HPSA || defined CONFIG_SCSI_HPSA_MODULE || defined __VMKERNEL_MODULE__
				else if (c->cmd_type == CMD_SCSI) {
					complete_scsi_command(c, 0, a1);
				}
#endif
				continue;
			}
		}
	}
	h->start_index = (h->start_index + 1) % NUM_GENDISK;
	if ( got_lock ) 
		spin_unlock_irqrestore(&io_request_lock, flags);
}

static inline unsigned long get_next_completion(ctlr_info_t *h)
{
        /* Any rejects from sendcmd() lying around? Process them first */
        if (h->scsi_rejects.ncompletions == 0) 
                return h->access.command_completed(h);
        else {
                struct sendcmd_reject_list *srl;
                int n;
                srl = &h->scsi_rejects;
                n = --srl->ncompletions;
                printk("hpsa%d: processing saved reject\n", h->ctlr); 
                return srl->complete[n];
        }
}

/* 
 *  We cannot read the structure directly, for portablity we must use 
 *   the io functions.
 *   This is for debug only. 
 */
#ifdef HPSA_DEBUG
static void print_cfg_table( CfgTable_struct *tb)
{
	int i;
	char temp_name[17];

	printk("Controller Configuration information\n");
	printk("------------------------------------\n");
	for(i=0;i<4;i++)
		temp_name[i] = readb(&(tb->Signature[i]));
	temp_name[4]='\0';
	printk("   Signature = %s\n", temp_name); 
	printk("   Spec Number = %d\n", readl(&(tb->SpecValence)));
	printk("   Transport methods supported = 0x%x\n", 
				readl(&(tb-> TransportSupport)));
	printk("   Transport methods active = 0x%x\n", 
				readl(&(tb->TransportActive)));
	printk("   Requested transport Method = 0x%x\n", 
			readl(&(tb->HostWrite.TransportRequest)));
	printk("   Coalese Interrupt Delay = 0x%x\n", 
			readl(&(tb->HostWrite.CoalIntDelay)));
	printk("   Coalese Interrupt Count = 0x%x\n", 
			readl(&(tb->HostWrite.CoalIntCount)));
	printk("   Max outstanding commands = 0x%d\n", 
			readl(&(tb->CmdsOutMax)));
	printk("   Bus Types = 0x%x\n", readl(&(tb-> BusTypes)));
	for(i=0;i<16;i++)
		temp_name[i] = readb(&(tb->ServerName[i]));
	temp_name[16] = '\0';
	printk("   Server Name = %s\n", temp_name);
	printk("   Heartbeat Counter = 0x%x\n\n\n", 
			readl(&(tb->HeartBeat)));
}
#endif /* HPSA_DEBUG */ 

static void release_io_mem(ctlr_info_t *c)
{
	/* if IO mem was not protected do nothing */
	if (c->io_mem_addr == 0)
		return;
	release_region(c->io_mem_addr, c->io_mem_length);
	c->io_mem_addr = 0;
	c->io_mem_length = 0;
}

static int find_PCI_BAR_index(struct pci_dev *pdev,
               unsigned long pci_bar_addr)
{
	int i, offset, mem_type, bar_type;
	if (pci_bar_addr == PCI_BASE_ADDRESS_0) /* looking for BAR zero? */
		return 0;
	offset = 0;
	for (i=0; i<DEVICE_COUNT_RESOURCE; i++) {
		bar_type = pdev->resource[i].flags &
			PCI_BASE_ADDRESS_SPACE; 
		if (bar_type == PCI_BASE_ADDRESS_SPACE_IO)
			offset += 4;
		else {
			mem_type = pdev->resource[i].flags &
				PCI_BASE_ADDRESS_MEM_TYPE_MASK; 
			switch (mem_type) {
				case PCI_BASE_ADDRESS_MEM_TYPE_32:
				case PCI_BASE_ADDRESS_MEM_TYPE_1M:
					offset += 4; /* 32 bit */
					break;
				case PCI_BASE_ADDRESS_MEM_TYPE_64:
					offset += 8;
					break;
				case PCI_BASE_ADDRESS_MEM_PREFETCH:
				break;
			}
		}
		if (offset == pci_bar_addr - PCI_BASE_ADDRESS_0)
			return i+1;
	}
	return -1;
}
			
static int hpsa_pci_init(ctlr_info_t *c, struct pci_dev *pdev)
{
	ushort subsystem_vendor_id, subsystem_device_id, command;
	unchar ready = 0;
	__u32 board_id, scratchpad;
	__u64 cfg_offset;
	__u32 cfg_base_addr;
	__u64 cfg_base_addr_index;
	int i;

	/* check to see if controller has been disabled */
	/* BEFORE we try to enable it */
	(void) pci_read_config_word(pdev, PCI_COMMAND,&command);
	if (!(command & 0x02)) {
		printk(KERN_WARNING "hpsa: controller appears to be disabled\n");
		return -1;
	}
	if (pci_enable_device(pdev)) {
		printk(KERN_ERR "hpsa: Unable to Enable PCI device\n");
		return -1;
	}
	if (pci_set_dma_mask(pdev, HPSA_DMA_MASK ) != 0) {
		printk(KERN_ERR "hpsa:  Unable to set DMA mask\n");
		return -1;
	}

	subsystem_vendor_id = pdev->subsystem_vendor;
	subsystem_device_id = pdev->subsystem_device;
	board_id = (((__u32) (subsystem_device_id << 16) & 0xffff0000) |
					subsystem_vendor_id );
	
	
	/* search for our IO range so we can protect it */
	for (i=0; i<DEVICE_COUNT_RESOURCE; i++) {
		/* is this an IO range */
		if (pdev->resource[i].flags & 0x01) {
			c->io_mem_addr = pdev->resource[i].start;
			c->io_mem_length = pdev->resource[i].end -
				pdev->resource[i].start +1; 
			/* register the IO range */
			if (!request_region( c->io_mem_addr,
                                        c->io_mem_length, "hpsa")) {
				printk(KERN_WARNING 
					"hpsa I/O memory range already in "
					"use addr=%lx length=%ld\n",
				c->io_mem_addr, c->io_mem_length);
				c->io_mem_addr= 0;
				c->io_mem_length = 0;
			}
			break;
		}
	}


	c->intr = pdev->irq;

	/*
	 * Memory base addr is first addr , the second points to the config
         *   table
	 */

	c->paddr = pdev->resource[0].start; /* addressing mode bits already removed */	
	c->vaddr = remap_pci_mem(c->paddr, 0x250);

	/* Wait for the board to become ready.  (PCI hotplug needs this.)
	 * We poll for up to 120 secs, once per 100ms. */
	for (i=0; i < 1200; i++) {
		scratchpad = readl(c->vaddr + SA5_SCRATCHPAD_OFFSET);
		if (scratchpad == 0xffff0000) {
			ready = 1;
			break;
		}
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(HZ / 10); /* wait 100ms */
	}
	if (!ready) {
		printk(KERN_WARNING "hpsa: Board not ready.  Timed out.\n");
		return -1;
	}

	/* get the address index number */
	cfg_base_addr = readl(c->vaddr + SA5_CTCFG_OFFSET);
	/* I am not prepared to deal with a 64 bit address value */
	cfg_base_addr &= (__u32) 0x0000ffff;
	cfg_base_addr_index =
		find_PCI_BAR_index(pdev, cfg_base_addr);
	if (cfg_base_addr_index == -1) {
		printk(KERN_WARNING "hpsa: Cannot find cfg_base_addr_index\n");
		release_io_mem(c);
		return -1;
	}

	cfg_offset = readl(c->vaddr + SA5_CTMEM_OFFSET);
	c->cfgtable = (CfgTable_struct *) 
		remap_pci_mem(pdev->resource[cfg_base_addr_index].start
				+ cfg_offset, sizeof(CfgTable_struct));
	c->board_id = board_id;


	// Query controller for max supported commands:
	c->max_commands = readl(&(c->cfgtable->CmdsOutMax));

	//Place an upper limit on depth no matter what controller supports
	c->max_commands = c->max_commands < NR_CMDS ? c->max_commands : NR_CMDS;
	
	for (i=0; i<NR_PRODUCTS; i++) {
		if (board_id == products[i].board_id) {
			c->product_name = products[i].product_name;
			c->access = *(products[i].access);
			//allow room for some ioctls
			c->nr_cmds = c->max_commands - 4;
			break;
		}
	}
	if (  (readb(&c->cfgtable->Signature[0]) != 'C') ||
	      (readb(&c->cfgtable->Signature[1]) != 'I') ||
	      (readb(&c->cfgtable->Signature[2]) != 'S') ||
	      (readb(&c->cfgtable->Signature[3]) != 'S') ) {
		printk("Does not appear to be a valid HPSA config table\n");
		return -1;
	}
	/* We didn't find the controller in our list. We know the
	 * signature is valid. If it's an HP device let's try to
	 * bind to the device and fire it up. Otherwise we bail.
	 */
	if (i == NR_PRODUCTS) {
		if (subsystem_vendor_id == 0x103C) {
			c->product_name = products[NR_PRODUCTS-1].product_name;
			c->access = *(products[NR_PRODUCTS-1].access);
			c->nr_cmds = c->max_commands - 4; // allow for some ioctls
			printk(KERN_WARNING "hpsa: This is an unknown "
				"Smart Array controller.\n"
				"hpsa: Please update to the latest driver "
				"available from www.hp.com.\n");
		} else {
			printk(KERN_WARNING "hpsa: Sorry, I don't know how"
				" to access the Smart Array controller %08lx\n"
					, (unsigned long)board_id);
			return -1;
		}
	}

#ifdef CONFIG_X86
	{
	/* Need to enable prefetch in the SCSI core for 6400 in x86 */
	__u32 prefetch;
	prefetch = readl(&(c->cfgtable->SCSI_Prefetch));
	prefetch |= 0x100;
	writel(prefetch, &(c->cfgtable->SCSI_Prefetch));
	}
#endif

	c->max_commands = readl(&(c->cfgtable->CmdsOutMax));
	//Place an upper limit on depth no matter what controller supports
	c->max_commands = c->max_commands < NR_CMDS ? c->max_commands : NR_CMDS;
	/* Update the field, and then ring the doorbell */ 
	writel( CFGTBL_Trans_Simple, 
		&(c->cfgtable->HostWrite.TransportRequest));
	writel( CFGTBL_ChangeReq, c->vaddr + SA5_DOORBELL);

	/* Here, we wait, possibly for a long time, (4 secs or more). 
	 * In some unlikely cases, (e.g. A failed 144 GB drive in a 
	 * RAID 5 set was hot replaced just as we're coming in here) it 
	 * can take that long.  Normally (almost always) we will wait 
	 * less than 1 sec. */
	for(i=0;i<MAX_CONFIG_WAIT;i++) {
		if (!(readl(c->vaddr + SA5_DOORBELL) & CFGTBL_ChangeReq))
			break;
		/* delay and try again */
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(10);
	}	


	if (!(readl(&(c->cfgtable->TransportActive)) & CFGTBL_Trans_Simple)) {
		printk(KERN_WARNING "hpsa: unable to get board into"
					" simple mode\n");
		return -1;
	}
	printk(KERN_INFO "hpsa:%s is at irq %u\n",
			c->product_name, pdev->irq);
	return 0;

}


/* Function to find the first free pointer into our hba[] array */
/* Returns -1 if no free entries are left.  */
static int alloc_hpsa_hba(void)
{
	int i;
	for(i=0; i< MAX_CTLR; i++) {
		if (hba[i] == NULL) {
			hba[i] = kmalloc(sizeof(ctlr_info_t), GFP_KERNEL);
			if (hba[i]==NULL) {
				printk(KERN_ERR "hpsa: out of memory.\n");
				return -1;
			}
			return i;
		}
	}
	printk(KERN_WARNING 
		"hpsa: This driver supports a maximum of %d controllers.\n"
		, MAX_CTLR);
	return -1;
}

static void free_hba(int i)
{
	kfree(hba[i]);
	hba[i]=NULL;
}

/*
 *  This is it.  Find all the controllers and register them.  I really hate
 *  stealing all these major device numbers.
 *  returns the number of block devices registered.
 */
static int __devinit hpsa_init_one(struct pci_dev *pdev,
	const struct pci_device_id *ent)
{
	request_queue_t *q;
	int i;
	int j;
/*	int rc; */
	int k;
	
	i = alloc_hpsa_hba();
	if (i < 0 ) 
		return -1;
	memset(hba[i], 0, sizeof(ctlr_info_t));

	hba[i]->busy_initializing = 1;

	if (hpsa_pci_init(hba[i], pdev) != 0) {
		free_hba(i);
		return -1;
	}
	sprintf(hba[i]->devname, "hpsa%d", i);
	hba[i]->ctlr = i;
	
	/* 
	 * register with the major number, or get a dynamic major number
	 * by passing 0 as argument 
	 */
	if (i < MAX_CTLR_ORIG)
		hba[i]->major[0] = MAJOR_NR + i;
	
	hba[i]->pdev = pdev;
	ASSERT_CTLR_ALIVE(hba[i]);

	/* configure PCI DMA stuff */
        /* When this flag is passed on the compile line we will */
	/* not attempt to set a 64bit DMA mask */
#if 1
	if (!pci_set_dma_mask(pdev, 0xffffffff))
		printk(KERN_ERR "hpsa: not using DAC cycles for hba %d\n", i);
	else {
		printk(KERN_ERR "hpsa: no suitable DMA available for hba %d\n", i);
		free_hba(i);
		return -ENODEV;
	}
#else
	if (!pci_set_dma_mask(pdev, (u64) 0xffffffffffffffff))
		printk(KERN_ERR "hpsa: not using DAC cycles\n");
	else {
		printk(KERN_ERR "hpsa: no suitable DMA available\n");
		free_hba(i);
		return -ENODEV;
	}
#endif

	/* make sure the board interrupts are off */
	hba[i]->access.set_intr_mask(hba[i], HPSA_INTR_OFF);
	if( request_irq(hba[i]->intr, do_hpsa_intr, 
		SA_INTERRUPT | SA_SHIRQ | SA_SAMPLE_RANDOM, 
			hba[i]->devname, hba[i]))
	{
		printk(KERN_ERR "hpsa: Unable to get irq %d for %s\n",
			hba[i]->intr, hba[i]->devname);
		for (k=0; k<NUM_GENDISK; k++){
#ifdef __VMKERNEL_MODULE__
			unregister_blkdev(hba[i]->major[k], hba[i]->devname);
#else
	 		devfs_unregister_blkdev(hba[i]->major[k], hba[i]->devname);
#endif
	 		map_major_to_ctlr[hba[i]->major[k]] = 0;
		}
		release_io_mem(hba[i]);
		free_hba(i);
		return(-1);
	}

	
	hba[i]->cmd_pool_bits = (__u32*)kmalloc(
        	(((hba[i]->nr_cmds)+31)/32)*sizeof(__u32), GFP_KERNEL);
	hba[i]->cmd_pool = (CommandList_struct *)pci_alloc_consistent(
		hba[i]->pdev, (hba[i]->nr_cmds) * sizeof(CommandList_struct), 
		&(hba[i]->cmd_pool_dhandle));
	hba[i]->errinfo_pool = (ErrorInfo_struct *)pci_alloc_consistent(
		hba[i]->pdev, (hba[i]->nr_cmds) * sizeof( ErrorInfo_struct), 
		&(hba[i]->errinfo_pool_dhandle));

	//Allocate mem for add_sendcmd_rejects.
        hba[i]->scsi_rejects.complete =
            kmalloc(sizeof(hba[i]->scsi_rejects.complete[0]) *
                    (NR_CMDS + 5), GFP_KERNEL);

	if ((hba[i]->cmd_pool_bits == NULL) 
		|| (hba[i]->cmd_pool == NULL)
		|| (hba[i]->errinfo_pool == NULL)
		|| (hba[i]->scsi_rejects.complete == NULL )) {

		if (hba[i]->scsi_rejects.complete)
                	kfree(hba[i]->scsi_rejects.complete);
		if (hba[i]->cmd_pool_bits)
                	kfree(hba[i]->cmd_pool_bits);
                if (hba[i]->cmd_pool)
                	pci_free_consistent(hba[i]->pdev,  
				(hba[i]->nr_cmds) * sizeof(CommandList_struct), 
				hba[i]->cmd_pool, hba[i]->cmd_pool_dhandle);	
		if (hba[i]->errinfo_pool)
			pci_free_consistent(hba[i]->pdev,
				(hba[i]->nr_cmds) * sizeof( ErrorInfo_struct),
				hba[i]->errinfo_pool, 
				hba[i]->errinfo_pool_dhandle);
                free_irq(hba[i]->intr, hba[i]);
		for (k=0; k<NUM_GENDISK; k++){
#ifdef __VMKERNEL_MODULE__
			unregister_blkdev(hba[i]->major[k], hba[i]->devname);
#else
	                devfs_unregister_blkdev(hba[i]->major[k], hba[i]->devname);
#endif
 			map_major_to_ctlr[hba[i]->major[k]] = 0;
		}
		release_io_mem(hba[i]);
		free_hba(i);
                printk( KERN_ERR "hpsa: out of memory");
		return -1;
	}


	/* Initialize the pdev driver private data. 
		have it point to hba[i].  */
	pci_set_drvdata(pdev, hba[i]);
	/* command and error info recs zeroed out before 
			they are used */
        memset(hba[i]->cmd_pool_bits, 0, (((hba[i]->nr_cmds)+31)/32)*sizeof(__u32));

 	if (notify_count == 0) {
		register_reboot_notifier(&hpsa_notifier);
		notify_count=1;
	}


	hpsa_find_non_disk_devices(i);	/* find our tape drives, if any */

	/* Turn the interrupts on so we can service requests */
	hba[i]->access.set_intr_mask(hba[i], HPSA_INTR_ON);

	hpsa_procinit(i);

	hpsa_register_scsi(i, 1);  /* hook ourself into SCSI subsystem */

	hba[i]->busy_initializing = 0;

	return 1;
}

static void __devexit hpsa_remove_one (struct pci_dev *pdev)
{
	ctlr_info_t *tmp_ptr;
	int i, j, k;
	char flush_buf[4];
	int return_code; 

	if (pci_get_drvdata(pdev) == NULL) {
		printk( KERN_ERR "hpsa: Unable to remove device \n");
		return;
	}
	tmp_ptr = pci_get_drvdata(pdev);
	i = tmp_ptr->ctlr;
	if (hba[i] == NULL) {
		printk(KERN_ERR "hpsa: device appears to "
			"already be removed \n");
		return;
	}
	/* no sense in trying to flush a dead board's cache. */
	if (CTLR_IS_ALIVE(hba[i])) {
		/* Turn board interrupts off and flush the cache */
		/* write all data in the battery backed cache to disks */
		memset(flush_buf, 0, 4);
                return_code = sendcmd(HPSA_CACHE_FLUSH, i, flush_buf,
                                        4, 0, 0, 0, NULL, 0);
		if (return_code != IO_OK)
			printk(KERN_WARNING 
				"hpsa%d: Error flushing cache\n", i);
 	}
	free_irq(hba[i]->intr, hba[i]);
	pci_set_drvdata(pdev, NULL);
	iounmap((void*)hba[i]->vaddr);
#if !defined(CONFIG_VMNIX)
	for (j = 0; j < NWD; j++) {
            char txt[16];
            sprintf(txt, "hpsa/c%dd%d", i, j);
            devfs_unregister(devfs_find_handle(NULL, txt, 0, 0, 0, 0));
	}
#endif

	hpsa_unregister_scsi(i);  /* unhook from SCSI subsystem */

	for (k=0; k<NUM_GENDISK; k++){
		map_major_to_ctlr[hba[i]->major[k]] = 0;
	
	}
	remove_proc_entry(hba[i]->devname, proc_hpsa);	

	pci_free_consistent(hba[i]->pdev, (hba[i]->nr_cmds) * sizeof(CommandList_struct), 
		hba[i]->cmd_pool, hba[i]->cmd_pool_dhandle);
	pci_free_consistent(hba[i]->pdev, (hba[i]->nr_cmds) * sizeof( ErrorInfo_struct),
		hba[i]->errinfo_pool, hba[i]->errinfo_pool_dhandle);
	kfree(hba[i]->cmd_pool_bits);
	kfree(hba[i]->scsi_rejects.complete);
	release_io_mem(hba[i]);
	free_hba(i);
}	

static struct pci_driver hpsa_pci_driver = {
	 name:   "hpsa",
	probe:  hpsa_init_one,
	remove:  __devexit(hpsa_remove_one),
	id_table:  hpsa_pci_device_id, /* id_table */
};

/*
*  This is it.  Register the PCI driver information for the cards we control
*  the OS will call our registered routines when it finds one of our cards. 
*/
int __init hpsa_init(void)
{
	printk(KERN_INFO DRIVER_NAME "\n");
	/* Register for out PCI devices */
	return pci_module_init(&hpsa_pci_driver);
 }

 				
EXPORT_NO_SYMBOLS;
static int __init init_hpsa_module(void)
{

#ifdef __VMKERNEL_MODULE__
	int status;
	int i;
	for (i = 0; i < MAX_CTLR; i++)
	{
		char_major[i][0] = -1;
		char_major[i][1] = -1;
	}


	if (!vmk_set_module_version("%s(%d)", DRIVER_NAME, DRIVER_VERSION)) {
		return -ENODEV;
	}

	spin_lock_init(&io_request_lock);
	status=hpsa_init();
	if (status) {
		spin_lock_destroy(&io_request_lock);
		printk(KERN_WARNING "hpsa: no controllers found, quitting.\n");
	}
	else {
		register_hpsa_ioctl32();
	}
	return status;
#else
	register_hpsa_ioctl32();
	return hpsa_init();
#endif
}

static void __exit cleanup_hpsa_module(void)
{
	int i, j;

	unregister_hpsa_ioctl32();
	pci_unregister_driver(&hpsa_pci_driver);
	unregister_reboot_notifier(&hpsa_notifier);
	/* double check that all controller entrys have been removed */
	for (i=0; i< MAX_CTLR; i++) {
		if (hba[i] != NULL) {
			printk(KERN_WARNING "hpsa: had to remove"
					" controller %d\n", i);
			hpsa_remove_one(hba[i]->pdev);
		}

#ifdef __VMKERNEL_MODULE__
                if (char_major[i][0] != -1)
                {
                        printk(KERN_WARNING "hpsa: DEBUG: Unregistering char device with major number of %d, name of %s\n", char_major[i][0], hba[i]->devname);
                        unregister_chrdev(char_major[i][0], hba[i]->devname);
                }
#endif

#if !defined(CONFIG_VMNIX)
		/* clean up any devfs handle */
		for (j=0; j<NWD*NUM_GENDISK; j++) {
			if (de_arr[i][j]) {
				devfs_unregister(de_arr[i][j]);
				de_arr[i][j] = NULL;
			}
		}
#endif 
	}
	remove_proc_entry("hpsa", proc_root_driver);

#ifdef __VMKERNEL_MODULE__
	spin_lock_destroy(&io_request_lock);
#endif
}

module_init(init_hpsa_module);
module_exit(cleanup_hpsa_module);
