/*
 *    Disk Array driver for HP SA 5xxx and 6xxx Controllers, SCSI version
 *    Copyright 2001, 2008 Hewlett-Packard Development Company, L.P.
 *    
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *    NON INFRINGEMENT.  See the GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *    Questions/Comments/Bugfixes to iss_storagedev@hp.com
 */
#if defined CONFIG_SCSI_HPSA || defined CONFIG_SCSI_HPSA_MODULE || defined __VMKERNEL_MODULE__

/* Here we have code to present the driver as a scsi driver */

#include "scsi.h"
#include "hosts.h"
#include <asm/atomic.h>
#include <linux/timer.h>

#include "hpsa_scsi.h"
#ifdef __VMKERNEL_MODULE__
#include "scsi_transport_sas.h"
#endif

/* some prototypes... */
static int sendcmd(
	__u8	cmd,
	int	ctlr,
	void	*buff,
	size_t	size,
	unsigned int use_unit_num, /* 0: address the controller,
				      1: address logical volume log_unit,
				      2: address is in scsi3addr */
	unsigned int log_unit,
	__u8	page_code,
	unsigned char *scsi3addr,
        int block_nr);

static void cmd_free(ctlr_info_t *h, CommandList_struct *c, int got_from_pool); // TSL
static CommandList_struct * cmd_alloc(ctlr_info_t *h, int get_from_pool); // TSL
int hpsa_scsi_detect(Scsi_Host_Template *tpnt);
int hpsa_scsi_release(struct Scsi_Host *sh);
const char *hpsa_scsi_info(struct Scsi_Host *sa);

int hpsa_scsi_proc_info(
		char *buffer, /* data buffer */
		char **start, 	   /* where data in buffer starts */
		off_t offset,	   /* offset from start of imaginary file */
		int length, 	   /* length of data in buffer */
		int hostnum, 	   /* which host adapter (always zero for me) */
		int func);	   /* 0 == read, 1 == write */

int hpsa_scsi_queue_command (Scsi_Cmnd *cmd, void (* done)(Scsi_Cmnd *));
static int  hpsa_eh_abort_handler(struct scsi_cmnd *scsicmd);
static int hpsa_eh_device_reset_handler(struct scsi_cmnd *scsicmd);
static int hpsa_eh_bus_reset_handler(struct scsi_cmnd *scsicmd);
static int hpsa_eh_host_reset_handler(struct scsi_cmnd *scsicmd);

static struct hpsa_scsi_hba_t hpsascsi[MAX_CTLR];

/* We need one Scsi_Host_Template *per controller* instead of
   the usual one Scsi_Host_Template per controller *type*. This
   is so PCI hot plug could have a remote possibility of still
   working even with the SCSI system.  It's so
   scsi_unregister_module will differentiate the controllers.
   When register_scsi_module is called, each host template is
   customized (name change) in hpsa_register_scsi()
   called from hpsa.c:hpsa_init_one()) 
*/

static
Scsi_Host_Template driver_template[MAX_CTLR];

struct hpsa_scsi_adapter_data_t {
	struct Scsi_Host *scsi_host;
	int registered;
	spinlock_t lock; // to protect hpsascsi[ctlr];
};

#ifdef __VMKERNEL_MODULE__
/*
 * Transport functions
 */
static void
hpsa_get_target_sas_identifier(struct scsi_target *starget);

static struct sas_function_template hpsa_transport_functions = {
        .get_starget_sas_address = hpsa_get_target_sas_identifier,
};
#endif

/* scsi_device_types comes from scsi.h */
#define DEVICETYPE(n) (n<0 || n>MAX_SCSI_DEVICE_CODE) ? \
			"Unknown" : scsi_device_types[n]

static int xmargin=8;
static int amargin=60;

static void
print_bytes (unsigned char *c, int len, int hex, int ascii)
{

	int i;
	unsigned char *x;

	if (hex)
	{
		x = c;
		for (i=0;i<len;i++)
		{
			if ((i % xmargin) == 0 && i>0) printk("\n");
			if ((i % xmargin) == 0) printk("0x%04x:", i);
			printk(" %02x", *x);
			x++;
		}
		printk("\n");
	}
	if (ascii)
	{
		x = c;
		for (i=0;i<len;i++)
			{
			if ((i % amargin) == 0 && i>0) printk("\n");
			if ((i % amargin) == 0) printk("0x%04x:", i);
			if (*x > 26 && *x < 128) printk("%c", *x);
			else printk(".");
			x++;
		}
		printk("\n");
	}
}

static void
print_cmd(CommandList_struct *cp)
{
	printk("queue:%d\n", cp->Header.ReplyQueue);
	printk("sglist:%d\n", cp->Header.SGList);
	printk("sgtot:%d\n", cp->Header.SGTotal);
	printk("Tag:0x%08x/0x%08x\n", cp->Header.Tag.upper,
		cp->Header.Tag.lower);
	printk("LUN:0x%02x%02x%02x%02x%02x%02x%02x%02x\n",
		cp->Header.LUN.LunAddrBytes[0],
		cp->Header.LUN.LunAddrBytes[1],
		cp->Header.LUN.LunAddrBytes[2],
		cp->Header.LUN.LunAddrBytes[3],
		cp->Header.LUN.LunAddrBytes[4],
		cp->Header.LUN.LunAddrBytes[5],
		cp->Header.LUN.LunAddrBytes[6],
		cp->Header.LUN.LunAddrBytes[7]);
	printk("CDBLen:%d\n", cp->Request.CDBLen);
	printk("Type:%d\n",cp->Request.Type.Type);
	printk("Attr:%d\n",cp->Request.Type.Attribute);
	printk(" Dir:%d\n",cp->Request.Type.Direction);
	printk("Timeout:%d\n",cp->Request.Timeout);
	printk( "CDB: %02x %02x %02x %02x %02x %02x %02x %02x"
		" %02x %02x %02x %02x %02x %02x %02x %02x\n",
		cp->Request.CDB[0], cp->Request.CDB[1],
		cp->Request.CDB[2], cp->Request.CDB[3],
		cp->Request.CDB[4], cp->Request.CDB[5],
		cp->Request.CDB[6], cp->Request.CDB[7],
		cp->Request.CDB[8], cp->Request.CDB[9],
		cp->Request.CDB[10], cp->Request.CDB[11],
		cp->Request.CDB[12], cp->Request.CDB[13],
		cp->Request.CDB[14], cp->Request.CDB[15]),
	printk("edesc.Addr: 0x%08x/0%08x, Len  = %d\n",
		cp->ErrDesc.Addr.upper, cp->ErrDesc.Addr.lower,
		cp->ErrDesc.Len);
	printk("sgs..........Errorinfo:\n");
	printk("scsistatus:%d\n", cp->err_info->ScsiStatus);
	printk("senselen:%d\n", cp->err_info->SenseLen);
	printk("SenseInfo: ");
	print_bytes(cp->err_info->SenseInfo, cp->err_info->SenseLen, 1, 0); 
	printk("cmd status:%d\n", cp->err_info->CommandStatus);
	printk("resid cnt:%d\n", cp->err_info->ResidualCnt);
	printk("offense size:%d\n", cp->err_info->MoreErrInfo.Invalid_Cmd.offense_size);
	printk("offense byte:%d\n", cp->err_info->MoreErrInfo.Invalid_Cmd.offense_num);
	printk("offense value:%d\n", cp->err_info->MoreErrInfo.Invalid_Cmd.offense_value);

}

static int
hpsa_scsi_add_entry(int ctlr, int hostno,
		unsigned char *scsi3addr, int devtype,
		int bus, int target, int lun)
{
	/* assumes hba[ctlr]->scsi_ctlr->lock is held */
	int n = hpsascsi[ctlr].ndevices;
	struct hpsa_scsi_dev_t *sd;

	if (n >= MAX_DEVS) {
		printk("hpsa%d: Too many devices, "
			"some will be inaccessible.\n", ctlr);
		return -1;
	}
	sd = &hpsascsi[ctlr].dev[n];
	memcpy(&sd->scsi3addr[0], scsi3addr, 8);
	sd->devtype = devtype;

	// bus, target, and lun are set to -1 for tape operations.
	if ( bus != -1 ) { 
		sd->bus = bus;
		sd->target = target;
		sd->lun = lun;
	}
	hpsascsi[ctlr].ndevices++;

	/* initially, (before registering with scsi layer) we don't
	   know our hostno and we don't want to print anything first
	   time anyway (the scsi layer's inquiries will show that info) */
	if (hostno != -1)
		printk("hpsa%d: %s device c%db%dt%dl%d added.\n",
			ctlr, DEVICETYPE(sd->devtype), hostno,
			sd->bus, sd->target, sd->lun);
	return 0;
}

static void
hpsa_scsi_remove_entry(int ctlr, int hostno, int entry)
{
	/* assumes hba[ctlr]->scsi_ctlr->lock is held */
	int i;
	struct hpsa_scsi_dev_t sd;

	if (entry < 0 || entry >= MAX_DEVS) return;
	sd = hpsascsi[ctlr].dev[entry];
	for (i=entry;i<hpsascsi[ctlr].ndevices-1;i++)
		hpsascsi[ctlr].dev[i] = hpsascsi[ctlr].dev[i+1];
	hpsascsi[ctlr].ndevices--;
	printk("hpsa%d: %s device c%db%dt%dl%d removed.\n",
		ctlr, DEVICETYPE(sd.devtype), hostno,
			sd.bus, sd.target, sd.lun);
}


#define SCSI3ADDR_EQ(a,b) ( \
	(a)[7] == (b)[7] && \
	(a)[6] == (b)[6] && \
	(a)[5] == (b)[5] && \
	(a)[4] == (b)[4] && \
	(a)[3] == (b)[3] && \
	(a)[2] == (b)[2] && \
	(a)[1] == (b)[1] && \
	(a)[0] == (b)[0])

static int
adjust_hpsa_scsi_table(int ctlr, int hostno,
	struct hpsa_scsi_dev_t sd[], int nsds)
{
	/* sd contains scsi3 addresses and devtypes, but
	   bus target and lun are not filled in.  This funciton
	   takes what's in sd to be the current and adjusts
	   hpsascsi[] to be in line with what's in sd. */

	int i,j, found, changes=0;
	struct hpsa_scsi_dev_t *csd;
	unsigned long flags;

	spin_lock_irqsave( &(((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->lock), flags);

	/* find any devices in hpsascsi[] that are not in
	   sd[] and remove them from hpsascsi[] */

	i = 0;
	while(i<hpsascsi[ctlr].ndevices) {
		csd = &hpsascsi[ctlr].dev[i];
		found=0;
		for (j=0;j<nsds;j++) {
			if (SCSI3ADDR_EQ(sd[j].scsi3addr,
				csd->scsi3addr)) {
				if (sd[j].devtype == csd->devtype)
					found=2;
				else
					found=1;
				break;
			}
		}

		if (found == 0) { /* device no longer present. */
			changes++;
			printk("hpsa%d: %s device c%db%dt%dl%d removed.\n",
				ctlr, DEVICETYPE(csd->devtype), hostno,
					csd->bus, csd->target, csd->lun);
			hpsa_scsi_remove_entry(ctlr, hostno, i);
			/* note, i not incremented */
		}
		else if (found == 1) { /* device is different kind */
			changes++;
			printk("hpsa%d: device c%db%dt%dl%d type changed "
				"(device type now %s).\n",
				ctlr, hostno, csd->bus, csd->target, csd->lun,
					DEVICETYPE(csd->devtype));
			csd->devtype = sd[j].devtype;
			i++;	/* so just move along. */
		} else 		/* device is same as it ever was, */
			i++;	/* so just move along. */
	}

	/* Now, make sure every device listed in sd[] is also
 	   listed in hpsascsi[], adding them if they aren't found */

	for (i=0;i<nsds;i++) {
		found=0;
		for (j=0;j<hpsascsi[ctlr].ndevices;j++) {
			csd = &hpsascsi[ctlr].dev[j];
			if (SCSI3ADDR_EQ(sd[i].scsi3addr,
				csd->scsi3addr)) {
				if (sd[i].devtype == csd->devtype) {
					found=2;	/* found device */
				} 
				else {
					found=1; 	/* found a bug. */
				}
				break;
			}
		}
		if (!found) {
			changes++;
			if (hpsa_scsi_add_entry(ctlr, hostno,
				&sd[i].scsi3addr[0], sd[i].devtype, sd[i].bus, sd[i].target, sd[i].lun) != 0)
				break;
		} else if (found == 1) {
			/* should never happen... */
			changes++;
			printk("hpsa%d: device unexpectedly changed type\n",
				ctlr);
			/* but if it does happen, we just ignore that device */
		}
	}
	spin_unlock_irqrestore( &(((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->lock), flags);

	if (!changes)
		printk("hpsa%d: No device changes detected.\n", ctlr);

	return 0;
}

#ifdef __VMKERNEL_MODULE__

static u64
hpsa_get_initiator_sas_identifier(struct Scsi_Host *sh)
{
   /*
    * SAS initiator information should be available at transport layer and
    * this workaround should be removed when SAS tranport support is ready.
    */
   u64 sas_id=0;
   struct pci_dev *pDev;

   pDev = sh->pci_dev;

   if(pDev == NULL) 
	return 0;

   /*
    * An initiator id has a form of 0x1XXXXXXXXXXXX which is a combination of
    * (1 << 48) | (pci device ID << 32) | (pci bus number << 8 ) | pci devfn.
    * Choosing this format is to avoid any potential conflict with MPTSAS
    * fw-generated RAID target ID with a form of 0xXXXXXXXXXXXXXX.
    */
   sas_id = ((((u64)pDev->device) << 32) | (((u64)pDev->bus->number) << 8) | pDev->devfn) | 0x1000000000000;
   return sas_id;
}

static void
hpsa_get_target_sas_identifier(struct scsi_target *starget)
{
   
   u64 sas_id;
   /*
    * This workaround needs to be replaced with real implementation when
    * SAS transport implementation is ready.
    */
   if (!starget) {
      return FAILED;
   }
   /*
    * A target id has a form of 0x2XXXXXXXXXXXX which is a combination of
    * (channel << 32) | id.
    * This is to avoid any potential conflict with MPTSAS fw-generated
    * RAID target ID with a form of 0xXXXXXXXXXXXXXX.
    */
   sas_id = ((((u64)starget->channel) << 32) + starget->id) | 0x2000000000000;

   sas_starget_sas_address(starget) = sas_id;

   return;
}

#endif

static int
lookup_scsi3addr(int ctlr, int bus, int target, int lun, char *scsi3addr)
{
	int i;
	struct hpsa_scsi_dev_t *sd;
	unsigned long flags;

	spin_lock_irqsave( &(((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->lock), flags);
	for (i=0;i<hpsascsi[ctlr].ndevices;i++) {
		sd = &hpsascsi[ctlr].dev[i];
		if (sd->bus == bus &&
		    sd->target == target &&
		    sd->lun == lun) {
			memcpy(scsi3addr, &sd->scsi3addr[0], 8);
			spin_unlock_irqrestore( &(((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->lock), flags);
			return 0;
		}
	}
	spin_unlock_irqrestore( &(((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->lock), flags);
	return -1;
}

static void
hpsa_find_non_disk_devices(int cntl_num)
{
	ReportLunData_struct *ld_buff;
	InquiryData_struct *inq_buff;
	int return_code;
	int i;
	int listlength = 0;
	int nphysicals;
	unsigned char scsi3addr[8];
	unsigned long flags;
	int reportlunsize = sizeof(*ld_buff) + HPSA_MAX_PHYS_LUN * 8;

	hba[cntl_num]->scsi_ctlr = (void *)
		kmalloc(sizeof(struct hpsa_scsi_adapter_data_t),
			GFP_KERNEL);
	if (hba[cntl_num]->scsi_ctlr == NULL)
		return;

	((struct hpsa_scsi_adapter_data_t *)
		hba[cntl_num]->scsi_ctlr)->scsi_host = NULL;
	((struct hpsa_scsi_adapter_data_t *)
		hba[cntl_num]->scsi_ctlr)->lock = SPIN_LOCK_UNLOCKED;
	((struct hpsa_scsi_adapter_data_t *)
		hba[cntl_num]->scsi_ctlr)->registered = 0;


	ld_buff = kmalloc(reportlunsize, GFP_KERNEL);
	if (ld_buff == NULL) {
		printk(KERN_ERR "hpsa: out of memory\n");
		return;
	}
	memset(ld_buff, 0, sizeof(ReportLunData_struct));
	inq_buff = kmalloc(sizeof( InquiryData_struct), GFP_KERNEL);
	if (inq_buff == NULL) {
		printk(KERN_ERR "hpsa: out of memory\n");
		kfree(ld_buff);
		return;
	}

	/* Get the physical luns */
	return_code = sendcmd(HPSA_REPORT_PHYS, cntl_num, ld_buff,
			reportlunsize, 0, 0, 0, NULL, 0);

	if( return_code == IO_OK) {
		unsigned char *c = &ld_buff->LUNListLength[0];
		listlength = (c[0] << 24) | (c[1] << 16) | (c[2] << 8) | c[3];
	}
	else {  /* getting report of physical luns failed */
		printk(KERN_WARNING "hpsa: report physical luns"
			" command failed\n");
		listlength = 0;
	}

	spin_lock_irqsave( &(((struct hpsa_scsi_adapter_data_t *) hba[cntl_num]->scsi_ctlr)->lock), flags);
	hpsascsi[cntl_num].ndevices = 0;
	nphysicals = listlength / 8; // 8 bytes per entry

	if (nphysicals > HPSA_MAX_PHYS_LUN)
	{
		printk(KERN_WARNING
			"hpsa: Maximum physical LUNs (%d) exceeded.  "
			"%d LUNs ignored.\n", HPSA_MAX_PHYS_LUN,
			nphysicals - HPSA_MAX_PHYS_LUN);
		nphysicals = HPSA_MAX_PHYS_LUN;
	}

	for(i=0; i<nphysicals; i++) {
		/* Execute an inquiry to figure the device type */
		/* Skip over masked devices */
		if (ld_buff->LUN[i][3] & 0xC0) continue;
		memset(inq_buff, 0, sizeof(InquiryData_struct));
		memcpy(scsi3addr, ld_buff->LUN[i], 8); /* ugly... */
		return_code = sendcmd(HPSA_INQUIRY, cntl_num, inq_buff,
			sizeof(InquiryData_struct), 2, 0 ,0, scsi3addr, 0);
	  	if (return_code == IO_OK) {
			if(inq_buff->data_byte[8] == 0xFF)
			{
			   printk(KERN_WARNING "hpsa: inquiry failed\n");
			} else {
			   int devtype;

			   devtype = (inq_buff->data_byte[0] & 0x1f);

			   switch (devtype)
			   {
			    case 0x01: /* A sequential access (tape) device */
			    case 0x08: /* A medium changer device */
					  /* These are the only kind of dev */
					  /* we want to expose here. */

				// added bus, target, and lun params as -1 for tape.
				if (hpsa_scsi_add_entry(cntl_num, -1,
					(unsigned char *) ld_buff->LUN[i],
					devtype, -1, -1, -1) != 0)
						i=nphysicals; // leave loop
				break;
			    default:
				break;
			   }

			}
		}
		else printk(KERN_ERR "hpsa: inquiry failed.\n");
	}
	spin_unlock_irqrestore( &(((struct hpsa_scsi_adapter_data_t *) hba[cntl_num]->scsi_ctlr)->lock), flags);
	kfree(ld_buff);
	kfree(inq_buff);
	return;
}

static void
complete_scsi_command( CommandList_struct *cp, int timeout, __u32 tag)
{
	Scsi_Cmnd *cmd;
	ctlr_info_t *ctlr;
	u64bit addr64;
	ErrorInfo_struct *ei;
	unsigned char sense_key;
	unsigned char asc;	//additional sense code 
	unsigned char ascq;	//additional sense code qualifier

	ei = cp->err_info;

	/* First, see if it was a message rather than a command */
	if (cp->Request.Type.Type == TYPE_MSG)  {
		cp->cmd_type = CMD_MSG_DONE;
		return;
	}

	/* we stored ptr to scsi cmd in the buffer head pointer */
	cmd = (Scsi_Cmnd *) cp->scsi_cmd;
	ctlr = hba[cp->ctlr];

	/* undo the DMA mappings */

#ifndef __VMKERNEL_MODULE__
	if (cmd->use_sg) {
		pci_unmap_sg(ctlr->pdev,
			cmd->buffer, cmd->use_sg,
				scsi_to_pci_dma_dir(cmd->sc_data_direction));
	}
	else if (cmd->request_bufflen) {
		addr64.val32.lower = cp->SG[0].Addr.lower;
		addr64.val32.upper = cp->SG[0].Addr.upper;
		pci_unmap_single(ctlr->pdev, (dma_addr_t) addr64.val,
			cmd->request_bufflen,
				scsi_to_pci_dma_dir(cmd->sc_data_direction));
	}
#endif

	cmd->result = (DID_OK << 16); 		/* host byte */
	cmd->result |= (COMMAND_COMPLETE << 8);	/* msg byte */
	/* cmd->result |= (GOOD < 1); */	/* status byte */

	cmd->result |= (ei->ScsiStatus);

	/* copy the sense data whether we need to or not. */

	memcpy(cmd->sense_buffer, ei->SenseInfo,
		ei->SenseLen > SCSI_SENSE_BUFFERSIZE ?
			SCSI_SENSE_BUFFERSIZE :
			ei->SenseLen);
	cmd->resid = ei->ResidualCnt;

	if(ei->CommandStatus != 0)
	{ /* an error has occurred */
		switch(ei->CommandStatus)
		{
			case CMD_TARGET_STATUS:
				if ( ei->ScsiStatus  ) { 
					sense_key = 0xf & ei->SenseInfo[2];	//Get sense key
					asc = ei->SenseInfo[12]; 		//Get additional sense code
					ascq = ei->SenseInfo[13];		//Get additional sense code qualifier
				}

				if ( ei->ScsiStatus == SAM_STAT_CHECK_CONDITION ) { 


					if (sense_key == ILLEGAL_REQUEST ) { 

						cmd->resid = cmd->request_bufflen;

						// If ASC/ASCQ indicate Logical Unit Not Supported condition, 
						// Report this as NO CONNECT for VMware multipathing to work.
						if ( ( asc == 0x25)  && (ascq == 0x0) ) {
							cmd->result = DID_NO_CONNECT << 16;
							printk(KERN_WARNING "hpsa: cp %p has check condition: "
									"path problem: "
									"returning no connection\n", cp);
							break;
						}
					}
						
					if (sense_key == NOT_READY ) {

						// If Sense is Not Ready, Logical Unit Not ready, Manual Intervention required,
						// probably need to report this as NO CONNECT also for multipathing to work.
						if ( ( asc == 0x04)  && (ascq == 0x03) ) {
							cmd->result = DID_NO_CONNECT << 16;
							printk(KERN_WARNING "hpsa: cp %p has check condition: "
									"unit not ready, manual intervention required: "
									"returning no connection\n", cp);
							break;
						}
					}				


					//Must be some other type of check condition
					cmd->result |= (ei->ScsiStatus < 1); 
					printk(KERN_WARNING "hpsa: cp %p has check condition: "
							"unknown type: "
							"Sense: 0x%x, ASC: 0x%x, ASCQ: 0x%x, "
							"Returning result: 0x%x\n",
							cp, sense_key, asc, ascq,
							cmd->result); 
					break;
				}
				

				
				//Problem was not a check condition.
				/* Pass it up to the upper layers... */
				if( ei->ScsiStatus) {

					cmd->result |= (ei->ScsiStatus < 1); 
					printk(KERN_WARNING "hpsa: cp %p has status 0x%x "
							"Sense: 0x%x, ASC: 0x%x, ASCQ: 0x%x, "
							"Returning result: 0x%x\n",
							cp, ei->ScsiStatus, 
							sense_key, asc, ascq,
							cmd->result); 
				}
				else {  /* scsi status is zero??? How??? */
					printk(KERN_WARNING "hpsa: cp %p SCSI status was 0. "
							"Returning no connection.\n", cp),

	/* Ordinarily, this case should never happen, but there is a bug
	   in some released firmware revisions that allows it to happen
	   if, for example, a 4100 backplane loses power and the tape
	   drive is in it.  We assume that it's a fatal error of some
	   kind because we can't show that it wasn't. We will make it
	   look like selection timeout since that is the most common
	   reason for this to occur, and it's severe enough. */

					cmd->result = DID_NO_CONNECT << 16;
				}
			break;
			case CMD_DATA_UNDERRUN: /* let mid layer handle it. */
			break;
			case CMD_DATA_OVERRUN:
				printk(KERN_WARNING "hpsa: cp %p has"
					" completed with data overrun "
					"reported\n", cp);
			break;
			case CMD_INVALID: {
				printk(KERN_WARNING "hpsa: cp %p has command invalid\n", cp);

				/* print_bytes(cp, sizeof(*cp), 1, 0);
				print_cmd(cp); */
     /* We get CMD_INVALID if you address a non-existent device instead
	of a selection timeout (no response).  You will see this if you yank
	out a drive, then try to access it. This is kind of a shame
	because it means that any other CMD_INVALID (e.g. driver bug) will
	get interpreted as a missing target. */
				cmd->result = DID_NO_CONNECT << 16;
				}
			break;
			case CMD_PROTOCOL_ERR:
				printk(KERN_WARNING "hpsa: cp %p has "
					"protocol error \n", cp);
			break;
			case CMD_HARDWARE_ERR:
				cmd->result = DID_ERROR << 16;
				printk(KERN_WARNING "hpsa: cp %p had "
					" hardware error\n", cp);
			break;
			case CMD_CONNECTION_LOST:
				cmd->result = DID_ERROR << 16;
				printk(KERN_WARNING "hpsa: cp %p had "
					"connection lost\n", cp);
			break;
			case CMD_ABORTED:
				cmd->result = DID_ABORT << 16;
				printk(KERN_WARNING "hpsa: cp %p was "
					"aborted with status 0x%x\n", cp, ei->ScsiStatus);
			break;
			case CMD_ABORT_FAILED:
				cmd->result = DID_ERROR << 16;
				printk(KERN_WARNING "hpsa: cp %p reports "
					"abort failed\n", cp);
			break;
			case CMD_UNSOLICITED_ABORT:
				cmd->result = DID_ABORT << 16;
				printk(KERN_WARNING "hpsa: cp %p aborted "
					"do to an unsolicited abort\n", cp);
			break;
			case CMD_TIMEOUT:
				cmd->result = DID_TIME_OUT << 16;
				printk(KERN_WARNING "hpsa: cp %p timedout\n",
					cp);
			break;
			default:
				cmd->result = DID_ERROR << 16;
				printk(KERN_WARNING "hpsa: cp %p returned "
					"unknown status %x\n", cp,
						ei->CommandStatus);
		}
	}
	cmd->scsi_done(cmd);

	cmd_free(ctlr, cp, 1);	// how do we know we got it from the pool ?
}

/* hpsa_scsi_detect is called from the scsi mid layer.
   The scsi mid layer (scsi_register_module) is
   called from hpsa.c:hpsa_init_one().  */

int 
hpsa_scsi_detect(Scsi_Host_Template *tpnt)
{
	int i;
	struct Scsi_Host *sh;

	i = simple_strtol((char *)&tpnt->name[4], NULL, 10);
	if (i<0 || i>=MAX_CTLR || hba[i] == NULL) {
		/* we didn't find ourself... we shouldn't get here. */
		return 0;
	}


	/* Tell the kernel we want to be a SCSI driver... */
#ifdef __VMKERNEL_MODULE__
	sh = vmk_scsi_register(tpnt, sizeof(struct ctlr_info *),
				hba[i]->pdev->bus->number,
				hba[i]->pdev->devfn);

#else
	sh = scsi_register(tpnt, sizeof(struct ctlr_info *));
#endif
	if (sh == NULL) {
		printk("hpsa_scsi_detect: scsi_register failed.\n");
		return 0;
	}

#ifdef __VMKERNEL_MODULE__
	vmk_scsi_register_uinfo(sh, 
				hba[i]->pdev->bus->number,
				hba[i]->pdev->devfn,
				hba[i]
				);
	char_major[i][0] = register_chrdev(0, hba[i]->devname, &hpsa_char_fops);
	if (!char_major[i][0]) {
		printk(KERN_WARNING "hpsa: failed to register char device %s\n", hba[i]->devname);
	
		return 0;
	}
	else
		printk(KERN_WARNING "hpsa: Created character device %s, major %d\n", hba[i]->devname, char_major[i][0]);
#endif

	sh->io_port = 0;	// not used
	sh->n_io_port = 0;	// I don't think we use these two...

#if defined(CONFIG_VMNIX) && !defined(__VMKERNEL_MODULE__)
	sh->bus = hba[i]->pdev->bus->number;	// SST:
	sh->devfn = hba[i]->pdev->devfn;	// not so sure about these three. They are shown in 
	sh->devid = (void *)sh->hostdata;	// a SCSI porting example, but...
#endif
	sh->this_id = SELF_SCSI_ID;
	sh->max_lun = MAX_DEVS;
	sh->max_id = MAX_DEVS;

	((struct hpsa_scsi_adapter_data_t *)
		hba[i]->scsi_ctlr)->scsi_host = (void *) sh;
	sh->hostdata[0] = (unsigned long) hba[i];
	sh->irq = hba[i]->intr;
	sh->unique_id = sh->irq;
	scsi_set_pci_device(sh, hba[i]->pdev);

#ifdef __VMKERNEL_MODULE__
	if(sh) {
		// Start the SAS transport
		if (vmk_sas_attach_transport(sh, &hpsa_transport_functions) != SUCCESS) {
			printk("hpsa_scsi_detect: vmk_sas_attach_transport: failed!\n");
			scsi_unregister(sh);
			return 0;
		}
		sas_host_sas_address(sh) = hpsa_get_initiator_sas_identifier(sh);
	}
#endif
	return 1;	/* Say we have 1 scsi adapter, this will be */
			/* called multiple times, once for each adapter */
			/* from hpsa.c:hpsa_init_one().  We do it this */
			/* way for PCI-hot plug reasons. (we don't know how */
			/* many adapters we have total, so we say we have */
			/* 1, each of a unique type.) */
}

static void __exit cleanup_hpsa_module(void);
int
hpsa_scsi_release(struct Scsi_Host *sh)
{
	return 0;
}

static void
hpsa_unmap_one(struct pci_dev *pdev,
		CommandList_struct *cp,
		size_t buflen,
		int data_direction)
{
	u64bit addr64;

#ifndef __VMKERNEL_MODULE__
	addr64.val32.lower = cp->SG[0].Addr.lower;
	addr64.val32.upper = cp->SG[0].Addr.upper;
	pci_unmap_single(pdev, (dma_addr_t) addr64.val, buflen, data_direction);
#endif
}

static void
hpsa_map_one(struct pci_dev *pdev,
		CommandList_struct *cp,
		unsigned char *buf,
		size_t buflen,
		int data_direction)
{
	__u64 addr64;

#ifndef __VMKERNEL_MODULE__
	addr64 = (__u64) pci_map_single(pdev, buf, buflen, data_direction);
#else
	addr64 = (__u64) virt_to_phys(buf);
	//vmk_verify_memory_for_io(buf, buflen);
#endif
	cp->SG[0].Addr.lower =
	  (__u32) (addr64 & (__u64) 0x00000000FFFFFFFF);
	cp->SG[0].Addr.upper =
	  (__u32) ((addr64 >> 32) & (__u64) 0x00000000FFFFFFFF);
	cp->SG[0].Len = buflen;
	cp->Header.SGList = (__u8) 1;   /* no. SGs contig in this cmd */
	cp->Header.SGTotal = (__u16) 1; /* total sgs in this cmd list */
}

static int
hpsa_scsi_do_simple_cmd(ctlr_info_t *c,
			CommandList_struct *cp,
			unsigned char *scsi3addr,
			unsigned char *cdb,
			unsigned char cdblen,
			unsigned char *buf, int bufsize,
			int direction)
{
	unsigned long flags;
	DECLARE_COMPLETION(wait);

	cp->cmd_type = CMD_IOCTL_PEND;		// treat this like an ioctl
	cp->scsi_cmd = NULL;
	cp->Header.ReplyQueue = 0;  // unused in simple mode
	memcpy(&cp->Header.LUN, scsi3addr, sizeof(cp->Header.LUN));

	cp->Header.Tag.lower = cp->busaddr;  // Use k. address of cmd as tag
	// Fill in the request block...

	memset(cp->Request.CDB, 0, sizeof(cp->Request.CDB));
	memcpy(cp->Request.CDB, cdb, cdblen);
	cp->Request.Timeout = 0;	// No timeout
	cp->Request.CDBLen = cdblen;
	cp->Request.Type.Type = TYPE_CMD;
	cp->Request.Type.Attribute = ATTR_SIMPLE;
	cp->Request.Type.Direction = direction;

	/* Fill in the SG list and do dma mapping */
	hpsa_map_one(c->pdev, cp,
			(unsigned char *) buf, bufsize,
			scsi_to_pci_dma_dir(SCSI_DATA_READ));

	cp->waiting = &wait;

	/* Put the request on the tail of the request queue */
	spin_lock_irqsave(&io_request_lock, flags);
	addQ(&c->reqQ, cp);
	c->Qdepth++;
	start_io(c);
	spin_unlock_irqrestore(&io_request_lock, flags);
	wait_for_completion(&wait);

	/* undo the dma mapping */
#ifndef __VMKERNEL__MODULE__
	hpsa_unmap_one(c->pdev, cp, bufsize,
				scsi_to_pci_dma_dir(SCSI_DATA_READ));
#endif

	return(0);
}

static void
hpsa_scsi_interpret_error(CommandList_struct *cp)
{
	ErrorInfo_struct *ei;

	ei = cp->err_info;
	switch(ei->CommandStatus)
	{
		case CMD_TARGET_STATUS:
			printk(KERN_WARNING "hpsa: cmd %p has "
				"completed with errors\n", cp);
			printk(KERN_WARNING "hpsa: cmd %p "
				"has SCSI Status = %x\n",
					cp,
					ei->ScsiStatus);
			if ( ei->ScsiStatus == SAM_STAT_CHECK_CONDITION ) {
				printk(KERN_WARNING "hpsa: cp %p has check condition.\n");
				unsigned char sense_key;
				sense_key = 0xf & ei->SenseInfo[2];
				if (sense_key == ILLEGAL_REQUEST ) {
					printk(KERN_WARNING "hpsa: cp %p has illegal request.\n");
					unsigned char asc;      
					unsigned char ascq;    
			
					asc = ei->SenseInfo[12]; 
					ascq = ei->SenseInfo[13];
			
					// If additional Sense Code and qualifier indicate
					// Logical Unit Not Supported condition,
					// This is interpreted as a path failure.
					if ( (asc == 0x25) && (ascq == 0x0) ) {
						printk(KERN_WARNING "hpsa: cp %p has path problem.\n");
						break;
					} 		
				} 			
			} 				

			if (ei->ScsiStatus == 0)
				printk(KERN_WARNING
				"hpsa:SCSI status is abnormally zero.  "
				"(probably indicates selection timeout "
				"reported incorrectly due to a known "
				"firmware bug, circa July, 2001.)\n");
		break;
		case CMD_DATA_UNDERRUN: /* let mid layer handle it. */
			printk("UNDERRUN\n");
		break;
		case CMD_DATA_OVERRUN:
			printk(KERN_WARNING "hpsa: cp %p has"
				" completed with data overrun "
				"reported\n", cp);
		break;
		case CMD_INVALID: {
			/* controller unfortunately reports SCSI passthru's */
			/* to non-existent targets as invalid commands. */
			printk(KERN_WARNING "hpsa: cp %p is "
				"reported invalid (probably means "
				"target device no longer present)\n",
				cp);
			/* print_bytes((unsigned char *) cp, sizeof(*cp), 1, 0);
			print_cmd(cp);  */
			}
		break;
		case CMD_PROTOCOL_ERR:
			printk(KERN_WARNING "hpsa: cp %p has "
				"protocol error \n", cp);
		break;
		case CMD_HARDWARE_ERR:
			/* cmd->result = DID_ERROR << 16; */
			printk(KERN_WARNING "hpsa: cp %p had "
				" hardware error\n", cp);
		break;
		case CMD_CONNECTION_LOST:
			printk(KERN_WARNING "hpsa: cp %p had "
				"connection lost\n", cp);
		break;
		case CMD_ABORTED:
			printk(KERN_WARNING "hpsa: cp %p was "
				"aborted\n", cp);
		break;
		case CMD_ABORT_FAILED:
			printk(KERN_WARNING "hpsa: cp %p reports "
				"abort failed\n", cp);
		break;
		case CMD_UNSOLICITED_ABORT:
			printk(KERN_WARNING "hpsa: cp %p aborted "
				"do to an unsolicited abort\n", cp);
		break;
		case CMD_TIMEOUT:
			printk(KERN_WARNING "hpsa: cp %p timedout\n",
				cp);
		break;
		default:
			printk(KERN_WARNING "hpsa: cp %p returned "
				"unknown status %x\n", cp,
					ei->CommandStatus);
	}
}

static int
hpsa_scsi_do_inquiry(ctlr_info_t *c, unsigned char *scsi3addr,
		unsigned char *buf, unsigned char bufsize)
{
	int rc;
	CommandList_struct *cp;
	char cdb[6];
	ErrorInfo_struct *ei;

	cp = cmd_alloc(c, 0);
	ei = cp->err_info;

	if (cp == NULL) {			/* trouble... */
		printk("cmd_alloc returned NULL!\n");
		return -1;
	}

	cdb[0] = HPSA_INQUIRY;
	cdb[1] = 0;
	cdb[2] = 0;
	cdb[3] = 0;
	cdb[4] = bufsize;
	cdb[5] = 0;
	rc = hpsa_scsi_do_simple_cmd(c, cp, scsi3addr, cdb,
				6, buf, bufsize, XFER_READ);

	if (rc != 0) return rc; /* something went wrong */

	if (ei->CommandStatus != 0 &&
	    ei->CommandStatus != CMD_DATA_UNDERRUN) {
		hpsa_scsi_interpret_error(cp);
		cmd_free(c, cp, 0);
		return -1;
	}
	cmd_free(c, cp, 0);
	return 0;
}

static int
hpsa_scsi_do_report_luns(ctlr_info_t *c, int logical,
		ReportLunData_struct *buf, int bufsize)
{
	int rc;
	CommandList_struct *cp;
	unsigned char cdb[12];
	unsigned char scsi3addr[8];
	ErrorInfo_struct *ei;

	cp = cmd_alloc(c, 0);
	if (cp == NULL) {			/* trouble... */
		printk("cmd_alloc returned NULL!\n");
		return -1;
	}

	memset(&scsi3addr[0], 0, 8); /* address the controller */
	cdb[0] = logical ? HPSA_REPORT_LOG : HPSA_REPORT_PHYS;
	cdb[1] = 0;
	cdb[2] = 0;
	cdb[3] = 0;
	cdb[4] = 0;
	cdb[5] = 0;
	cdb[6] = (bufsize >> 24) & 0xFF;  //MSB
	cdb[7] = (bufsize >> 16) & 0xFF;
	cdb[8] = (bufsize >> 8) & 0xFF;
	cdb[9] = bufsize & 0xFF;
	cdb[10] = 0;
	cdb[11] = 0;

	rc = hpsa_scsi_do_simple_cmd(c, cp, scsi3addr,
				cdb, 12,
				(unsigned char *) buf,
				bufsize, XFER_READ);

	if (rc != 0) return rc; /* something went wrong */

	ei = cp->err_info;
	if (ei->CommandStatus != 0 &&
	    ei->CommandStatus != CMD_DATA_UNDERRUN) {
		hpsa_scsi_interpret_error(cp);
		cmd_free(c, cp, 0);
		return -1;
	}
	cmd_free(c, cp, 0);
	return 0;
}

static inline int
hpsa_scsi_do_report_phys_luns(ctlr_info_t *c,
               ReportLunData_struct *buf, int bufsize)
{
       return hpsa_scsi_do_report_luns(c, 0, buf, bufsize);
}

static inline int
hpsa_scsi_do_report_log_luns(ctlr_info_t *c,
               ReportLunData_struct *buf, int bufsize)
{
       return hpsa_scsi_do_report_luns(c, 1, buf, bufsize);
}


static void
hpsa_update_scsi_devices(int cntl_num, int hostno)
{
	/* 
           The idea here is we could get notified 
	   that some devices have changed, so we do a report
	   physical luns cmd, and adjust our list of devices
	   accordingly.  (We can't rely on the scsi-mid layer just
	   doing inquiries, because the "busses" that the scsi
	   mid-layer probes are totally fabricated by this driver,
	   so new devices wouldn't show up.

	   the scsi3addr's of devices won't change so long as the
	   adapter is not reset.  That means we can rescan and
	   tell which devices we already know about, vs. new
	   devices, vs.  disappearing devices.
	 */
#define OBDR_TAPE_INQ_SIZE 49
#define OBDR_TAPE_SIG "$DR-10"
	ReportLunData_struct *device_list = NULL;
	ReportLunData_struct *logdev_list = NULL;
	unsigned char *inq_buff = NULL;
	unsigned char scsi3addr[8];
	ctlr_info_t *c;
	__u32 nphysicals=0;
	unsigned char *ch;
	struct hpsa_scsi_dev_t *currentsd;
	int ncurrent=0;
	int reportlunsize = sizeof(*device_list) + HPSA_MAX_PHYS_LUN * 8;
	int nlogicals = 0;
	int i;

	int lun;
	int target;


	char lun0_processing[MAX_CTLR][16];

	int hpsa_targets=0;	//number of targets registered 

	memset(&lun0_processing[0][0], 0, sizeof(lun0_processing));

	currentsd = kmalloc(sizeof(*currentsd) * MAX_DEVS, GFP_KERNEL);
	if (currentsd == NULL)
		goto out;



	c = (ctlr_info_t *) hba[cntl_num];
	device_list = kmalloc(reportlunsize * 2, GFP_KERNEL);
	if (device_list == NULL) {
		printk(KERN_ERR "hpsa: out of memory\n");
		goto out;
	}
	logdev_list = (ReportLunData_struct *) (((char *) device_list) + reportlunsize);

	
	inq_buff = kmalloc(OBDR_TAPE_INQ_SIZE, GFP_KERNEL);
	if (inq_buff == NULL) {
		printk(KERN_ERR "hpsa: out of memory\n");
		goto out;
	}

       if (hpsa_scsi_do_report_phys_luns(c, device_list, reportlunsize) == 0) {               
		ch = &device_list->LUNListLength[0];

		nphysicals = ((ch[0]<<24) | (ch[1]<<16) | (ch[2]<<8) | ch[3]) / 8;
		if (nphysicals > HPSA_MAX_PHYS_LUN) {
			printk(KERN_WARNING
				"hpsa: Maximum physical LUNs (%d) exceeded.  "
				"%d LUNs ignored.\n", HPSA_MAX_PHYS_LUN,
				nphysicals - HPSA_MAX_PHYS_LUN);
			nphysicals = HPSA_MAX_PHYS_LUN;
		}
	}
	else {
		printk(KERN_ERR  "hpsa: Report physical LUNs failed.\n");
		goto out;
	}

       if (hpsa_scsi_do_report_log_luns(c, logdev_list, reportlunsize) == 0) {                
		char temp[MAX_DEVS * 8];
		ch = &logdev_list->LUNListLength[0];
		nlogicals = ((ch[0]<<24) | (ch[1]<<16) | (ch[2]<<8) | ch[3]) / 8;

		// Reject Logicals in excess of our max capability.
		if (nlogicals > HPSA_MAX_LUN ) {
                       printk(KERN_WARNING
				"hpsa: Maximum logical LUNs (%d) exceeded.  "
				"%d LUNs ignored.\n", HPSA_MAX_LUN,
				nlogicals  - HPSA_MAX_LUN);
                       nlogicals = HPSA_MAX_LUN;
		}

		// If Logicals + Physicals is greater than our max combined capability,
		// reject enough logicals to reduce total to within supported maximum.
               if (nlogicals + nphysicals > HPSA_MAX_PHYS_LUN) {
                       printk(KERN_WARNING
                               "hpsa: Maximum logical + physical LUNs (%d) exceeded.  "
                               "%d LUNs ignored.\n", HPSA_MAX_PHYS_LUN,
                               nphysicals + nlogicals  - HPSA_MAX_PHYS_LUN);
                       nlogicals = HPSA_MAX_PHYS_LUN - nphysicals;
               }

		// append logical list to physical list. 
		ch = &device_list->LUN[nphysicals][0];
		memcpy(temp,  &logdev_list->LUN[0][0], 8 * nlogicals);
		memcpy(ch, temp, 8 * nlogicals);

       } else {
               printk(KERN_ERR  "hpsa: Report logical LUNs failed.\n");
               goto out;
       }
	c->num_luns = nlogicals; // TSL

	// Get the firmware version and save it off for this cntl_num
	memset(inq_buff, 0, OBDR_TAPE_INQ_SIZE);
	memset(&scsi3addr[0], 0, 8); /* address the controller */
	
	if (hpsa_scsi_do_inquiry(hba[cntl_num], scsi3addr, inq_buff,
		(unsigned char) OBDR_TAPE_INQ_SIZE) != 0) {
		/* Inquiry failed (msg printed already) */
		printk("hpsa_update_scsi_devices: inquiry for firmware version failed\n");
	} else {
		// Save off firmware version
		hba[cntl_num]->firm_ver[0] = inq_buff[32];
		hba[cntl_num]->firm_ver[1] = inq_buff[33];
		hba[cntl_num]->firm_ver[2] = inq_buff[34];
		hba[cntl_num]->firm_ver[3] = inq_buff[35];
#ifdef HPSA_DEBUG
	printk(KERN_INFO "hpsa: Saved off Firmware Version of %c%c%c%c for controller number %d\n",
		hba[cntl_num]->firm_ver[0], hba[cntl_num]->firm_ver[1], hba[cntl_num]->firm_ver[2],
		hba[cntl_num]->firm_ver[3], cntl_num);
#endif
	}

	/* adjust our table of devices */
	int logical_drive_number = 0;
	for(i=0; i<nphysicals + nlogicals; i++)
	{
		int devtype;
		unsigned int lunid = 0;

		/* for each physical lun, do an inquiry */
		if (device_list->LUN[i][3] & 0xC0 && i < nphysicals)
                       continue; /* Skip "masked" physical devices. */

		lunid = (0xff & (unsigned int)(device_list->LUN[i][3])) << 24;
		lunid |= (0xff & (unsigned int)(device_list->LUN[i][2])) << 16;
		lunid |= (0xff & (unsigned int)(device_list->LUN[i][1])) << 8;
		lunid |= (0xff & (unsigned int)(device_list->LUN[i][0]));

		lun = lunid & 0x00ff;
		target = ( lunid >> 16) & 0x3fff;

		if (lun0_processing[cntl_num][target] == 0)
		{
	
			// If this is first LUN of a target, and LUN id is not 0,
			// add a LUN id of 0, which is enclosure on MSA2012sa.
			// This allows scanning code to work properly, as it requires
			// a LUN 0 on every target.	
			if ( lun != 0 ) {
				currentsd[ncurrent].bus        =  0;
				currentsd[ncurrent].target     = target;
				currentsd[ncurrent].lun        = 0;
				currentsd[ncurrent].devtype    = 0x0D; /*enclosure device */
				memset(currentsd[ncurrent].scsi3addr, 0, 8); 
				currentsd[ncurrent].scsi3addr[3] = target;
				ncurrent++;
				hpsa_targets++;
			}
			lun0_processing[cntl_num][target] = 1;
		}


		c->drv[logical_drive_number].LunID = lunid;
		logical_drive_number++;
		memset(inq_buff, 0, OBDR_TAPE_INQ_SIZE);
		memcpy(&scsi3addr[0], &device_list->LUN[i][0], 8);
			

		if (hpsa_scsi_do_inquiry(hba[cntl_num],
			scsi3addr, inq_buff, OBDR_TAPE_INQ_SIZE) != 0)
		{
			/* Inquiry failed (msg printed already) */
			printk("hpsa_update_scsi_devices: inquiry failed\n");
			devtype = 0; /* so we will skip this device. */
		} else /* what kind of device is this? */
			devtype = (inq_buff[0] & 0x1f);


		switch (devtype)
		{
		  case 0x05: /* CD-ROM */ {

			/* We don't *really* support actual CD-ROM devices,
			 * just "One Button Disaster Recovery" tape drive
			 * which temporarily pretends to be a CD-ROM drive.
			 * So we check that the device is really an OBDR tape 
			 * device by checking for "$DR-10" in bytes 43-48 of 
			 * the inquiry data.
			 */ 
				char obdr_sig[7]; 

				strncpy(obdr_sig, &inq_buff[43], 6);
				obdr_sig[6] = '\0';
				if (strncmp(obdr_sig, OBDR_TAPE_SIG, 6) != 0) 
					/* Not OBDR device, ignore it. */
					break;
			}
			/* fall through . . . */
                 case 0x00: /* direct access device, aka a disk */
                       if (i < nphysicals)
                               break; /* skip the physical disks, only expose logical disks. */


		  case 0x0C: /* RAID controller device */
			if (devtype == 0x0C)
				printk(KERN_ERR "hpsa: ncurrent = %d, RAID Controller found.\n", ncurrent);
		  case 0x01: /* sequential access, (tape) */
		  case 0x08: /* medium changer */
			//Too many devices
			if ( ncurrent >= MAX_DEVS) {
				printk(KERN_INFO "hpsa%d: %s ignored, "
					"too many devices.\n", cntl_num,
					DEVICETYPE(devtype));
				break;
			}
			//Too many LUNs
			if (( ncurrent - hpsa_targets) >= HPSA_MAX_LUN) {
				printk(KERN_INFO "hpsa%d: %s ignored, "
					"too many LUNs.\n", cntl_num,
					DEVICETYPE(devtype));
				break;
			}
			memcpy(&currentsd[ncurrent].scsi3addr[0],
				&scsi3addr[0], 8);
			currentsd[ncurrent].devtype = devtype;
			currentsd[ncurrent].bus = 0;
			currentsd[ncurrent].target = target;
			currentsd[ncurrent].lun = lun;
			ncurrent++;
			break;
		  default:
			printk(KERN_WARNING "hpsa: Unknown device type %d.\n", devtype);
			break;
		}
	}

	adjust_hpsa_scsi_table(cntl_num, hostno, currentsd, ncurrent);
out:
	kfree(currentsd);
	kfree(inq_buff);
	kfree(device_list);
	return;
}

static int
is_keyword(char *ptr, int len, char *verb)  // Thanks to ncr53c8xx.c
{
	int verb_len = strlen(verb);
	if (len >= verb_len && !memcmp(verb,ptr,verb_len))
		return verb_len;
	else
		return 0;
}

static int
hpsa_scsi_user_command(int ctlr, int hostno, char *buffer, int length)
{
	int arg_len;

	if ((arg_len = is_keyword(buffer, length, "rescan")) != 0)
		hpsa_update_scsi_devices(ctlr, hostno);
	else
		return -EINVAL;
	return length;
}

/* It's a pity that we need this, but, we do... */
extern struct Scsi_Host *scsi_hostlist;  /* from ../scsi/hosts.c */

int
hpsa_scsi_proc_info(char *buffer, /* data buffer */
		char **start, 	   /* where data in buffer starts */
		off_t offset,	   /* offset from start of imaginary file */
		int length, 	   /* length of data in buffer */
		int hostnum, 	   /* which host adapter (always zero for me) */
		int func)	   /* 0 == read, 1 == write */
{

	int buflen, datalen;
	struct Scsi_Host *sh;
	int found;
	ctlr_info_t *ci;
	int i;
	int cntl_num;

	/* Lets see if we can find our Scsi_Host...
	   this might be kind of "bad", searching scsi_hostlist this way
	   but how else can we find the scsi host?  I think I've seen
	   this coded both ways, (circular list and null terminated list)
	*/

	sh = scsi_hostlist;
	found=0;
	do {
		if (sh == NULL) break;
		if (sh->host_no == hostnum) {
			found++;
			break;
		}
		sh = sh->next;
	} while (sh != scsi_hostlist && sh != NULL);

	if (sh == NULL || found == 0) /* This really shouldn't ever happen. */
		return -EINVAL;

	ci = (ctlr_info_t *) sh->hostdata[0];
	if (ci == NULL)  /* This really shouldn't ever happen. */
		return -EINVAL;

	cntl_num = ci->ctlr;	/* Get our index into the hba[] array */

	if (func == 0) {	/* User is reading from /proc/scsi/hpsa*?/?*  */

	buflen = sprintf(buffer, "hpsa%d: SCSI host: %d\n", cntl_num, sh->host_no); 	
			/* this information is needed by apps to know which hpsa device corresponds to
			   which scsi host number without having to open a scsi target device node.
			   The device information is not a duplicate of /proc/scsi/scsi because the
			   two may be out of sync due to scsi hotplug, rather this info is for an app
			   to be able to use to know how to get them back in sync. */

			/*  Printing controller/bus/target/lun/devtype for every lun
				causes proc node to exceed 1 page when there are lots of LUNs.
				VMware has a 1 page limit.
			*/

		for (i=0;i<hpsascsi[cntl_num].ndevices;i++) {
			struct hpsa_scsi_dev_t *sd = &hpsascsi[cntl_num].dev[i];

#if 1 || defined(__VMKERNEL_MODULE__)
                        /*
                         * The -80 is done so that we can append the vmkadapter
                         * info in the last line.
                         */
			int len = length - buflen - 80;
			if (len <= 0) {
				break;
			}
			buflen += snprintf(&buffer[buflen], len,
					   "c%db%dt%dl%d %02d "
#else
			buflen += sprintf(&buffer[buflen], "c%db%dt%dl%d %02d "
#endif
				"0x%02x%02x%02x%02x%02x%02x%02x%02x\n",
				sh->host_no, sd->bus, sd->target, sd->lun, sd->devtype,
				sd->scsi3addr[0], sd->scsi3addr[1], sd->scsi3addr[2], sd->scsi3addr[3],
				sd->scsi3addr[4], sd->scsi3addr[5], sd->scsi3addr[6], sd->scsi3addr[7]);
		}
		buflen = min(buflen, length - 80);
		datalen = buflen - offset;
		if (datalen < 0) { 	/* they're reading past EOF. */
			datalen = 0;
			*start = buffer+buflen;
		} else
			*start = buffer + offset;
		return(datalen);
	} else 	/* User is writing to /proc/scsi/hpsa*?/?*  ... */
		return hpsa_scsi_user_command(cntl_num, hostnum,
			buffer, length);
}

/* this is via the generic proc support */
const char *
hpsa_scsi_info(struct Scsi_Host *sa)
{
	static char buf[300];
	ctlr_info_t *ci;

	/* probably need to work on putting a bit more info in here... */
	/* this is output via the /proc filesystem. */

	ci = (ctlr_info_t *) sa->hostdata[0];

	sprintf(buf, "%s %c%c%c%c\n",
		ci->product_name,
		ci->firm_ver[0],
		ci->firm_ver[1],
		ci->firm_ver[2],
		ci->firm_ver[3]);

	return buf;
}


/* hpsa_scatter_gather takes a Scsi_Cmnd, (cmd), and does the pci
   dma mapping  and fills in the scatter gather entries of the
   hpsa command, cp. */

static void
hpsa_scatter_gather(struct pci_dev *pdev,
		CommandList_struct *cp,
		Scsi_Cmnd *cmd)
{
	unsigned int use_sg, nsegs=0, len;
	struct scatterlist *scatter = (struct scatterlist *) cmd->buffer;
	__u64 addr64;

	/* is it just one virtual address? */
	if (!cmd->use_sg) {
		if (cmd->request_bufflen) {	/* anything to xfer? */

#ifndef __VMKERNEL_MODULE__
			addr64 = (__u64) pci_map_single(pdev,
				cmd->request_buffer,
				cmd->request_bufflen,
				scsi_to_pci_dma_dir(cmd->sc_data_direction));
#else
			addr64 = cmd->request_bufferMA;
	//		addr64 = (__u64 )virt_to_phys(cmd->request_buffer);
	//		vmk_verify_memory_for_io(cmd->request_buffer, cmd->request_bufflen);
#endif

			cp->SG[0].Addr.lower =
			  (__u32) (addr64 & (__u64) 0x00000000FFFFFFFF);
			cp->SG[0].Addr.upper =
			  (__u32) ((addr64 >> 32) & (__u64) 0x00000000FFFFFFFF);
			cp->SG[0].Len = cmd->request_bufflen;
			nsegs=1;
		}
	} /* else, must be a list of virtual addresses.... */
	else if (cmd->use_sg <= MAXSGENTRIES) {	/* not too many addrs? */

#ifndef __VMKERNEL_MODULE__
		use_sg = pci_map_sg(pdev, cmd->buffer, cmd->use_sg,
			scsi_to_pci_dma_dir(cmd->sc_data_direction));
#else
		use_sg = cmd->use_sg;
		//use_sg = virt_to_phys(cmd->use_sg);
#endif

		for (nsegs=0; nsegs < use_sg; nsegs++) {
			addr64 = (__u64) sg_dma_address(&scatter[nsegs]);
			len  = sg_dma_len(&scatter[nsegs]);
			cp->SG[nsegs].Addr.lower =
			  (__u32) (addr64 & (__u64) 0x00000000FFFFFFFF);
			cp->SG[nsegs].Addr.upper =
			  (__u32) ((addr64 >> 32) & (__u64) 0x00000000FFFFFFFF);
			cp->SG[nsegs].Len = len;
			cp->SG[nsegs].Ext = 0;  // we are not chaining
		}
	} else BUG();

	cp->Header.SGList = (__u8) nsegs;   /* no. SGs contig in this cmd */
	cp->Header.SGTotal = (__u16) nsegs; /* total sgs in this cmd list */
	return;
}


int
hpsa_scsi_queue_command (Scsi_Cmnd *cmd, void (* done)(Scsi_Cmnd *))
{
	ctlr_info_t **c;
	int ctlr, rc;
	unsigned char scsi3addr[8];
	CommandList_struct *cp;

	// Get the ptr to our adapter structure (hba[i]) out of cmd->host.
	// We violate cmd->host privacy here.  (Is there another way?)
	c = (ctlr_info_t **) &cmd->host->hostdata[0];
	ctlr = (*c)->ctlr;

	rc = lookup_scsi3addr(ctlr, cmd->channel, cmd->target, cmd->lun,
			scsi3addr);
	if (rc != 0) {
		/* the scsi nexus does not match any that we presented... */
		/* pretend to mid layer that we got selection timeout */
		cmd->result = DID_NO_CONNECT << 16;
		done(cmd);
		/* we might want to think about registering controller itself
		   as a processor device on the bus so sg binds to it. */
		return 0;
	}


	/* Ok, we have a reasonable scsi nexus, so send the cmd down, and
		see what the device thinks of it. */

	cp = cmd_alloc(*c, 1);
	if (cp == NULL) {			/* trouble... */
		printk("scsi_cmd_alloc returned NULL!\n");
		cmd->result = DID_NO_CONNECT << 16;
		done(cmd);
		return 0;
	}

	// Fill in the command list header

	cmd->scsi_done = done;    // save this for use by completion code

	// save cp in case we have to abort it
	cmd->host_scribble = (unsigned char *) cp;

	cp->cmd_type = CMD_SCSI;
	cp->scsi_cmd = cmd;
	cp->Header.ReplyQueue = 0;  // unused in simple mode
	memcpy(&cp->Header.LUN.LunAddrBytes[0], &scsi3addr[0], 8);
	cp->Header.Tag.lower = cp->busaddr;  // Use k. address of cmd as tag

	// Fill in the request block...

	cp->Request.Timeout = 0; // No timeout 
	memset(cp->Request.CDB, 0, sizeof(cp->Request.CDB));
	if (cmd->cmd_len > sizeof(cp->Request.CDB)) BUG();
	cp->Request.CDBLen = cmd->cmd_len;
	memcpy(cp->Request.CDB, cmd->cmnd, cmd->cmd_len);
	cp->Request.Type.Type = TYPE_CMD;
	cp->Request.Type.Attribute = ATTR_SIMPLE;
	switch(cmd->sc_data_direction)
	{
	  case SCSI_DATA_WRITE: cp->Request.Type.Direction = XFER_WRITE; break;
	  case SCSI_DATA_READ: cp->Request.Type.Direction = XFER_READ; break;
	  case SCSI_DATA_NONE: cp->Request.Type.Direction = XFER_NONE; break;

	  case SCSI_DATA_UNKNOWN:
		// This can happen if a buggy application does a scsi passthru
		// and sets both inlen and outlen to non-zero. ( see
		// ../scsi/scsi_ioctl.c:scsi_ioctl_send_command() )

		cp->Request.Type.Direction = XFER_RSVD;
		// This is technically wrong, and hpsa controllers should
		// reject it with CMD_INVALID, which is the most correct
		// response, but non-fibre backends appear to let it
		// slide by, and give the same results as if this field
		// were set correctly.  Either way is acceptable for
		// our purposes here.

		break;

	  default:
		printk(KERN_ERR "hpsa: unknown data direction: %d\n",
			cmd->sc_data_direction);
		BUG();
		break;
	}

	hpsa_scatter_gather((*c)->pdev, cp, cmd); // Fill the SG list

	/* Put the request on the tail of the request queue */

	addQ(&(*c)->reqQ, cp);
	(*c)->Qdepth++;
	start_io(*c);

	/* the cmd'll come back via intr handler in complete_scsi_command()  */
	return 0;
}

static void
init_driver_template(int ctlr)
{
	memset(&driver_template[ctlr], 0, sizeof(driver_template[ctlr]));
	driver_template[ctlr].name = hpsascsi[ctlr].name;
	driver_template[ctlr].proc_name = hpsascsi[ctlr].name;
	driver_template[ctlr].detect = hpsa_scsi_detect;
	driver_template[ctlr].release = hpsa_scsi_release;
	driver_template[ctlr].proc_info = hpsa_scsi_proc_info;
	driver_template[ctlr].queuecommand = hpsa_scsi_queue_command;
	driver_template[ctlr].eh_abort_handler = hpsa_eh_abort_handler;
	driver_template[ctlr].eh_device_reset_handler = hpsa_eh_device_reset_handler; 
	driver_template[ctlr].eh_bus_reset_handler = hpsa_eh_bus_reset_handler; 
	driver_template[ctlr].eh_host_reset_handler = hpsa_eh_host_reset_handler; 
	driver_template[ctlr].bios_param = scsicam_bios_param;
	driver_template[ctlr].can_queue = 512;
	driver_template[ctlr].this_id = SELF_SCSI_ID;
	driver_template[ctlr].sg_tablesize = MAXSGENTRIES;
	driver_template[ctlr].cmd_per_lun = 512;
	driver_template[ctlr].use_new_eh_code = 1;
	driver_template[ctlr].use_clustering = DISABLE_CLUSTERING;
	driver_template[ctlr].module = THIS_MODULE;
	driver_template[ctlr].ioctl = hpsa_ioctl;
#ifdef __VMKERNEL_MODULE__
	driver_template[ctlr].driverLock = &io_request_lock; 
#endif

	/* set scsi_host to NULL so our detect routine will
	   find us on register */

	((struct hpsa_scsi_adapter_data_t *)
		hba[ctlr]->scsi_ctlr)->scsi_host = NULL;

}

static void
hpsa_unregister_scsi(int ctlr)
{
	unsigned long flags;

	/* we are being forcibly unloaded, and may not refuse. */

	spin_lock_irqsave(&io_request_lock, flags);

	/* if we weren't ever actually registered, don't unregister */
	if (((struct hpsa_scsi_adapter_data_t *)
		hba[ctlr]->scsi_ctlr)->registered) {
		spin_unlock_irqrestore(&io_request_lock, flags);
		scsi_unregister_module(MODULE_SCSI_HA, &driver_template[ctlr]);
		spin_lock_irqsave(&io_request_lock, flags);
	}
	init_driver_template(ctlr);
	//scsi_cmd_stack_free(ctlr);
	kfree(hba[ctlr]->scsi_ctlr);
	spin_unlock_irqrestore(&io_request_lock, flags);
}

static int
hpsa_register_scsi(int ctlr, int this_is_init_time)
{
	unsigned long flags;
	int	rc;

	spin_lock_irqsave(&io_request_lock, flags);

	if (((struct hpsa_scsi_adapter_data_t *)
		hba[ctlr]->scsi_ctlr)->registered) {
		spin_unlock_irqrestore(&io_request_lock, flags);
		return ENXIO;
	}
	spin_unlock_irqrestore(&io_request_lock, flags);
	hpsa_update_scsi_devices(ctlr, -1);

	spin_lock_irqsave( &(((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->lock), flags);

	sprintf( hpsascsi[ctlr].name, "hpsa%d", ctlr );
	
	init_driver_template(ctlr);
	
	/* Register controller whether or not there are any logicals configured */
	((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->registered = 1;
	spin_unlock_irqrestore( &(((struct hpsa_scsi_adapter_data_t *) hba[ctlr]->scsi_ctlr)->lock), flags);
	rc = scsi_register_module(MODULE_SCSI_HA, &driver_template[ctlr]);
	if (rc) {
       		printk(KERN_ERR "hpsa: hpsa_register_scsi: failed scsi_register_module()\n");
	}
	return rc; 
}

static int
hpsa_engage_scsi(int ctlr)
{
	unsigned long flags;


	spin_lock_irqsave(&io_request_lock, flags);

	if (((struct hpsa_scsi_adapter_data_t *)
		hba[ctlr]->scsi_ctlr)->registered) {
		printk("hpsa%d: SCSI subsystem already engaged.\n", ctlr);
		spin_unlock_irqrestore(&io_request_lock, flags);
		return ENXIO;
	}
	spin_unlock_irqrestore(&io_request_lock, flags);
	hpsa_update_scsi_devices(ctlr, -1);
	hpsa_register_scsi(ctlr, 0);
	return 0;
}

static void
hpsa_proc_tape_report(int ctlr, unsigned char *buffer, off_t *pos, off_t *len)
{
	int size;
	unsigned long flags;

}

/* Need at least one of these error handlers to keep ../scsi/hosts.c from 
 * complaining.  Doing a host- or bus-reset can't do anything good here. 
*/

static int hpsa_eh_device_reset_handler(struct scsi_cmnd *scsicmd)
{
	int rc;
	CommandList_struct *cmd_in_trouble;
	ctlr_info_t **c;
	int ctlr;
	unsigned char scsi3addr[8];

	/* find the controller to which the command to be aborted was sent */
	c = (ctlr_info_t **) &scsicmd->device->host->hostdata[0];	
	if (c == NULL) /* paranoia */
		return FAILED;
	ctlr = (*c)->ctlr;
	printk(KERN_WARNING "hpsa%d: resetting drive\n", ctlr);

	rc = lookup_scsi3addr(ctlr, scsicmd->channel, scsicmd->target, scsicmd->lun,
			scsi3addr);
	if (rc != 0) {
		printk("hpsa_eh_device_reset_handler: lookup_scsi3addr failed\n");
		return FAILED;
	}
	/* send a reset to the SCSI LUN which the command was sent to */
	rc = sendcmd(HPSA_DEVICE_RESET_MSG, ctlr, NULL, 0, 1, scsicmd->target, 0, 
		(unsigned char *) NULL, 
		0);

	/* sendcmd turned off interrputs on the board, turn 'em back on. */
	(*c)->access.set_intr_mask(*c, HPSA_INTR_ON);
	if (rc == 0)
	{
		return SUCCESS;
	}
	printk(KERN_WARNING "hpsa%d: resetting device failed.\n", ctlr);
	return FAILED;
}
static int hpsa_eh_bus_reset_handler(struct scsi_cmnd *scsicmd)
{
	int rc;
	CommandList_struct *cmd_in_trouble;
	ctlr_info_t **c;
	int ctlr;
	unsigned char scsi3addr[8];

	/* find the controller to which the command to be aborted was sent */
	c = (ctlr_info_t **) &scsicmd->device->host->hostdata[0];	
	if (c == NULL) /* paranoia */
		return FAILED;
	ctlr = (*c)->ctlr;
	printk(KERN_WARNING "hpsa%d: resetting bus\n", ctlr);

	// May not need this lookup_scsi3addr for sendcmd - TSL
	rc = lookup_scsi3addr(ctlr, scsicmd->channel, scsicmd->target, scsicmd->lun,
			scsi3addr);
	if (rc != 0) {
		printk("hpsa_eh_bus_reset_handler: lookup_scsi3addr failed\n");
		return FAILED;
	}
	/* send a bus reset to the SCSI LUN which the command was sent to */
	rc = sendcmd(HPSA_BUS_RESET_MSG, ctlr, NULL, 0, 1, scsicmd->target, 0, 
		(unsigned char *) NULL, 
		0);
	/* sendcmd turned off interrputs on the board, turn 'em back on. */
	(*c)->access.set_intr_mask(*c, HPSA_INTR_ON);
	if (rc == 0)
	{
		return SUCCESS;
	}
	printk(KERN_WARNING "hpsa%d: resetting device failed.\n", ctlr);
	return FAILED;
}
static int hpsa_eh_host_reset_handler(struct scsi_cmnd *scsicmd)
{
	int rc;
	CommandList_struct *cmd_in_trouble;
	ctlr_info_t **c;
	int ctlr;
	unsigned char scsi3addr[8];

	/* find the controller to which the command to be aborted was sent */
	c = (ctlr_info_t **) &scsicmd->device->host->hostdata[0];	
	if (c == NULL) /* paranoia */
		return FAILED;
	ctlr = (*c)->ctlr;
	printk(KERN_WARNING "hpsa%d: resetting host\n", ctlr);

	// May not need this lookup_scsi3addr for sendcmd - TSL
	rc = lookup_scsi3addr(ctlr, scsicmd->channel, scsicmd->target, scsicmd->lun,
			scsi3addr);
	if (rc != 0) {
		printk("hpsa_eh_host_reset_handler: lookup_scsi3addr failed\n");
		return FAILED;
	}
	/* send a host reset to the SCSI LUN which the command was sent to */
	rc = sendcmd(HPSA_HOST_RESET_MSG, ctlr, NULL, 0, 1, scsicmd->target, 0, 
		(unsigned char *) NULL, 
		0);
	/* sendcmd turned off interrputs on the board, turn 'em back on. */
	(*c)->access.set_intr_mask(*c, HPSA_INTR_ON);
	if (rc == 0)
	{
		return SUCCESS;
	}
	printk(KERN_WARNING "hpsa%d: resetting device failed.\n", ctlr);
	return FAILED;
}

static int  hpsa_eh_abort_handler(struct scsi_cmnd *scsicmd)
{
	int rc;
	CommandList_struct *cmd_to_abort;
	ctlr_info_t **c;
	int ctlr;

	/* find the controller to which the command to be aborted was sent */
	c = (ctlr_info_t **) &scsicmd->device->host->hostdata[0];	
	if (c == NULL) /* paranoia */
		return FAILED;
	ctlr = (*c)->ctlr;
	printk(KERN_WARNING "hpsa%d: aborting tardy SCSI cmd\n", ctlr);

	/* find the command to be aborted */
	cmd_to_abort = (CommandList_struct *) scsicmd->host_scribble;
	if (cmd_to_abort == NULL) /* paranoia */
	{
		printk(KERN_WARNING "hpsa%d: aborting tardy SCSI cmd failed, no outstanding command to abort\n", ctlr);
		return FAILED;
	}
	// Send scsicmd->target since we're going to emulate abort with a bus reset.
	//rc = sendcmd(HPSA_ABORT_MSG, ctlr, cmd_to_abort, 
	//	0, 2, 0, 0, 
	//	(unsigned char *) &cmd_to_abort->Header.LUN.LunAddrBytes[0], 
	//	TYPE_MSG);
	rc = sendcmd(HPSA_ABORT_MSG, ctlr, cmd_to_abort, 0, 1, scsicmd->target, 0, 
		(unsigned char *) NULL, 
		0);
	/* sendcmd turned off interrputs on the board, turn 'em back on. */
	(*c)->access.set_intr_mask(*c, HPSA_INTR_ON);
	if (rc == 0)
		return SUCCESS;
	return FAILED;

}

#else /* no CONFIG_SCSI_HPSA */

/* If no tape support, then these become defined out of existence */

#define hpsa_find_non_disk_devices(cntl_num)
#define hpsa_unregister_scsi(ctlr)
#define hpsa_register_scsi(ctlr, this_is_init_time)
#define hpsa_proc_tape_report(ctlr, buffer, pos, len)

#endif /* CONFIG_SCSI_HPSA */
