/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Fibre Channel Host Bus Adapters.                                *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003-2008 Emulex.  All rights reserved.           *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

/*
 * $Id: lpfc_els.c 1.7 2005/07/08 19:29:44EDT sf_support Exp  $
 */
#include <linux/version.h>
#include <linux/config.h>
#ifdef __VMKERNEL_MODULE__
#include "linux_skbuff.h"
#endif
#include <linux/kernel.h>
#include <linux/blk.h>
#include <linux/spinlock.h>
#include <linux/pci.h>

#ifdef __VMKERNEL_MODULE__
#include "vmklinux_dist.h"
#include "npiv_api_dist.h"
#endif

#include "lpfc_hw.h"
#include "lpfc_cfgparm.h"
#include "lpfc_dfc.h"
#include "lpfc_sli.h"
#include "lpfc_mem.h"
#include "lpfc_sched.h"
#include "lpfc_disc.h"
#include "lpfc.h"
#include "lpfc_logmsg.h"
#include "lpfc_scsi.h"
#include "hbaapi.h"
#include "lpfc_crtn.h"

int lpfc_els_rcv_rscn(lpfc_vport_t *, LPFC_IOCBQ_t *, LPFC_NODELIST_t *, uint8_t);
int lpfc_els_rcv_flogi(lpfc_vport_t *, LPFC_IOCBQ_t *, LPFC_NODELIST_t *, uint8_t);
int lpfc_els_rcv_rrq(lpfc_vport_t *, LPFC_IOCBQ_t *, LPFC_NODELIST_t *);
int lpfc_els_rcv_rnid(lpfc_vport_t *, LPFC_IOCBQ_t *, LPFC_NODELIST_t *);
int lpfc_els_rcv_farp(lpfc_vport_t *, LPFC_IOCBQ_t *, LPFC_NODELIST_t *);
int lpfc_els_rcv_farpr(lpfc_vport_t *, LPFC_IOCBQ_t *, LPFC_NODELIST_t *);
int lpfc_els_rcv_fan(lpfc_vport_t *, LPFC_IOCBQ_t *, LPFC_NODELIST_t *);
static void lpfc_cmpl_fabric_iocb(lpfcHBA_t *phba, LPFC_IOCBQ_t *cmdiocb,
				  LPFC_IOCBQ_t *rspiocb);
int lpfc_max_els_tries = 3;

int
lpfc_initial_flogi(lpfc_vport_t * vport)
{
	LPFC_NODELIST_t *ndlp;
	lpfcHBA_t *phba = vport->phba;

	/* First look for Fabric ndlp on the unmapped list */

	if ((ndlp =
	     lpfc_findnode_did(vport, (NLP_SEARCH_UNMAPPED | NLP_SEARCH_DEQUE),
			       Fabric_DID)) == 0) {
		/* Cannot find existing Fabric ndlp, so allocate a new one */
		if ((ndlp = lpfc_nlp_alloc(phba, 0)) == 0) {
			return (0);
		}
		memset(ndlp, 0, sizeof (LPFC_NODELIST_t));
		ndlp->nlp_DID = Fabric_DID;
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
			lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
	}
	if (lpfc_issue_els_flogi(vport, ndlp, 0)) {

		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
			lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);

		lpfc_nlp_free(phba, ndlp);
	}
	return (1);
}

int
lpfc_issue_els_flogi(lpfc_vport_t * vport, LPFC_NODELIST_t * ndlp, uint8_t retry)
{
	lpfcHBA_t *phba = vport->phba;
	SERV_PARM *sp;
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = (sizeof (uint32_t) + sizeof (SERV_PARM));
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry,
				     ndlp, ELS_CMD_FLOGI);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	/* For FLOGI request, remainder of payload is service parameters */
	*((uint32_t *) (pCmd)) = ELS_CMD_FLOGI;
	pCmd += sizeof (uint32_t);
	memcpy(pCmd, &vport->fc_sparam, sizeof (SERV_PARM));
	sp = (SERV_PARM *) pCmd;

	/* Setup CSPs accordingly for Fabric */
	sp->cmn.e_d_tov = 0;
	sp->cmn.w2.r_a_tov = 0;
	sp->cls1.classValid = 0;
	sp->cls2.seqDelivery = 1;
	sp->cls3.seqDelivery = 1;
	if (sp->cmn.fcphLow < FC_PH3)
		sp->cmn.fcphLow = FC_PH3;
	if (sp->cmn.fcphHigh < FC_PH3)
		sp->cmn.fcphHigh = FC_PH3;

	if (phba->max_vpi) {
		sp->cmn.request_multiple_NPort = 1;
		elsiocb->iocb.ulpCt_h = 1;
		elsiocb->iocb.ulpCt_l = 0;
	}

	/*
	 * Clear the S_ID field and set SP to 1 so that the port takes the S_ID
	 * from word4. This approach maintains the driver's DID change 
	 * detection logic, FCP-2 support and leaves the els iocb prep clean.
	 */
	if (phba->fc_topology != TOPOLOGY_LOOP) {
		icmd->un.elsreq64.myID = 0;
		icmd->un.elsreq64.fl = 1;
	}

	lpfc_set_disctmo(vport);

	phba->fc_stat.elsXmitFLOGI++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_flogi;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
				IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return (1);
	}
	return (0);
}

void
lpfc_cmpl_els_flogi(lpfcHBA_t * phba, LPFC_IOCBQ_t * cmdiocb,
		    LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;
	DMABUF_t *pCmd, *pRsp;
	SERV_PARM *sp;
	uint32_t *lp;
	LPFC_MBOXQ_t *mbox;
	LPFC_SLI_t *psli;
	LPFC_NODELIST_t *ndlp;
	lpfcCfgParam_t *clp;
	uint32_t rc;
	int stat;
	lpfc_vport_t *vport = cmdiocb->vport;

	psli = &phba->sli;
	irsp = &(rspiocb->iocb);
	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;
	pCmd = (DMABUF_t *) cmdiocb->context2;

	clp = &phba->config[0];

	/* Return to default values since first FLOGI has completed */
	phba->fc_edtov = FF_DEF_EDTOV;
	phba->fc_ratov = FF_DEF_RATOV;
	phba->fcp_timeout_offset = 2 * phba->fc_ratov +
		clp[LPFC_CFG_EXTRA_IO_TMO].a_current;

	/* Check to see if link went down during discovery */
	if (lpfc_els_chk_latt(phba, rspiocb))
		goto out;

	if (irsp->ulpStatus) {
		/* Check for retry */
		if (lpfc_els_retry(phba, cmdiocb, rspiocb)) {
			/* ELS command is being retried */
			goto out;
		}
		/*
		 * FLOGI failed, so there is no fabric.  Plus, the initial
		 * implementation of NPIV is fabric only. 
		 */
		vport->fc_flag &= ~(FC_FABRIC | FC_PUBLIC_LOOP);
		phba->max_vpi = 0;

		/* If private loop, then allow max outstandting els to be
		 * LPFC_MAX_DISC_THREADS (32). Scanning in the case of no 
		 * alpa map would take too long otherwise. 
		 */
		if (phba->alpa_map[0] == 0) {
			clp[LPFC_CFG_DISC_THREADS].a_current =
			    LPFC_MAX_DISC_THREADS;
		}

		/* FLOGI failure */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0100,
				lpfc_mes0100,
				lpfc_msgBlk0100.msgPreambleStr,
				irsp->ulpStatus, irsp->un.ulpWord[4]);
	} else {
		/* The FLogI succeeded.  Sync the data for the CPU before
		 * accessing it. 
		 */
		pRsp = (DMABUF_t *) pCmd->list.next;
		lp = (uint32_t *) pRsp->virt;

		sp = (SERV_PARM *) ((uint8_t *) lp + sizeof (uint32_t));

		/* FLOGI completes successfully */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0101,
				lpfc_mes0101, lpfc_msgBlk0101.msgPreambleStr,
				irsp->un.ulpWord[4], sp->cmn.e_d_tov,
				sp->cmn.w2.r_a_tov, sp->cmn.edtovResolution);

		if (vport->port_state == LPFC_FLOGI) {
			/* If Common Service Parameters indicate Nport
			 * we are point to point, if Fport we are Fabric.
			 */
			if (sp->cmn.fPort) {
				vport->fc_flag |= FC_FABRIC;
				if (sp->cmn.edtovResolution) {
					/* E_D_TOV ticks are in nanoseconds */
					phba->fc_edtov =
					    (be32_to_cpu(sp->cmn.e_d_tov) +
					     999999) / 1000000;
				} else {
					/* E_D_TOV ticks are in milliseconds */
					phba->fc_edtov =
					    be32_to_cpu(sp->cmn.e_d_tov);
				}
				phba->fc_ratov =
				    (be32_to_cpu(sp->cmn.w2.r_a_tov) +
				     999) / 1000;
				phba->fcp_timeout_offset =
				    2 * phba->fc_ratov +
				    clp[LPFC_CFG_EXTRA_IO_TMO].a_current;

				if (phba->fc_topology == TOPOLOGY_LOOP) {
					vport->fc_flag |= FC_PUBLIC_LOOP;
				} else {
					/* 
					 * Initiator is not connected to a loop.
					 * Set the altBbCredit to enable logins
					 * to devices on remote loops work.
					 */
					vport->fc_sparam.cmn.altBbCredit = 1;
				}

				vport->fc_myDID = irsp->un.ulpWord[4] & Mask_DID;
				memcpy(&ndlp->nlp_portname, &sp->portName,
					sizeof (NAME_TYPE));
				memcpy(&ndlp->nlp_nodename, &sp->nodeName,
					sizeof (NAME_TYPE));
				memcpy(&phba->fc_fabparam, sp,
					sizeof (SERV_PARM));

				/*
				 * The HBA has been configured to support NPIV.
				 * If the fabric does not support NPIV, reset
				 * the number available vpis.
				 */
				if (phba->max_vpi) {
					if (!sp->cmn.response_multiple_NPort) {
						phba->npiv_capable = LPFC_NPIV_UNSUPPORTED_FAB;
					} else {
						phba->npiv_capable = LPFC_NPIV_SUPPORTED;
					}
				}

				lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0132,
						lpfc_mes0132,
						lpfc_msgBlk0132.msgPreambleStr,
						sp->cmn.response_multiple_NPort,
						phba->max_vpi);

				/*
				 * If the NportID has change, unregister the
				 * previous vpi<->DID binding before registering
				 * the new one.  Note that the prevDID is
				 * initialized to Mask_DID so that the code can
				 * correctly detect the first FLOGI completion.
				 * Note that the unreg_vpi, reg_vpi mailbox
				 * callbacks are not provided here because this
				 * is the physical port and its requirement to
				 * do config link and clear la handles discovery
				 * start.
				 */
				if ((vport->fc_myDID != vport->fc_prevDID) &&
				    (vport->fc_prevDID != Mask_DID)) {
					/* Since the driver's Nport ID changed,
					 * all remaining remote Nports mapped
					 * to the previous ID must get their
					 * rpi binding unregistered as well.
					 */
					lpfc_unreg_rpi_cleanup(vport);
					mbox = lpfc_mbox_alloc(phba, MEM_PRI);
					if (mbox == NULL)
						goto flogifail;

					lpfc_unreg_vpi(phba, vport->vpi, mbox);
					mbox->vport = vport;
					rc = lpfc_sli_issue_mbox(phba, mbox,
								 (MBX_NOWAIT |
								  MBX_STOP_IOCB));
					if (rc == MBX_NOT_FINISHED) {
						lpfc_mbox_free(phba, mbox);
						goto flogifail;
					}
				}

				ndlp->nlp_SID = irsp->un.ulpWord[4] & Mask_DID;
				mbox = lpfc_mbox_alloc(phba, 0);
				if (mbox == NULL)
					goto flogifail;

				vport->port_state = LPFC_FABRIC_CFG_LINK;
				lpfc_config_link(phba, mbox);
				mbox->vport = vport;
				if (lpfc_sli_issue_mbox
				    (phba, mbox, (MBX_NOWAIT | MBX_STOP_IOCB))
				    == MBX_NOT_FINISHED) {
					lpfc_mbox_free(phba, mbox);
					goto flogifail;
				}

				if (phba->max_vpi) {
					mbox = lpfc_mbox_alloc(phba, 0);
					if (mbox == NULL)
						goto flogifail;

					lpfc_reg_vpi(phba, vport->vpi,
						vport->fc_myDID, mbox);
					mbox->vport = vport;
					if (lpfc_sli_issue_mbox(phba, mbox,
						(MBX_NOWAIT | MBX_STOP_IOCB))
					    == MBX_NOT_FINISHED) {
						lpfc_mbox_free(phba, mbox);
						goto flogifail;
					}
				}

				mbox = lpfc_mbox_alloc(phba, 0);
				if (mbox == NULL)
					goto flogifail;

				 rc = lpfc_reg_login(phba, vport->vpi,
						     Fabric_DID,
						     (uint8_t *) sp, mbox, 0);
				 if (rc) {
					lpfc_mbox_free(phba, mbox);
					goto flogifail;
				 }

				/* set_slim mailbox command needs to execute
				 * first, queue this command to be processed
				 * later.
				 */
				mbox->mbox_cmpl = lpfc_mbx_cmpl_fabric_reg_login;
				mbox->context2 = ndlp;
				mbox->vport = vport;
				if (lpfc_sli_issue_mbox(phba, mbox,
				     (MBX_NOWAIT | MBX_STOP_IOCB)) == 
				    MBX_NOT_FINISHED) {
					DMABUF_t *mp;
					mp = (DMABUF_t *)(mbox->context1);
					lpfc_mbuf_free(phba, mp->virt, mp->phys);
					kfree(mp);
					lpfc_mbox_free(phba, mbox);
					goto flogifail;
				}
			} else {
				/* We FLOGIed into an NPort, initiate pt2pt
				   protocol */
				phba->max_vpi = 0;
				vport->fc_flag &= ~(FC_FABRIC | FC_PUBLIC_LOOP);
				phba->fc_edtov = FF_DEF_EDTOV;
				phba->fc_ratov = FF_DEF_RATOV;
				phba->fcp_timeout_offset = 2 * phba->fc_ratov +
				    clp[LPFC_CFG_EXTRA_IO_TMO].a_current;
				if ((rc =
				     lpfc_geportname((NAME_TYPE *) & vport->
						     fc_portname,
						     (NAME_TYPE *) & sp->
						     portName))) {
					/* This side will initiate the PLOGI */
					vport->fc_flag |= FC_PT2PT_PLOGI;

					/* N_Port ID cannot be 0, set our to
					 * LocalID the other side will be
					 * RemoteID.
					 */

					/* not equal */
					if (rc == 1)
						vport->fc_myDID = PT2PT_LocalID;
					rc = 0;

					mbox = lpfc_mbox_alloc(phba, 0);
					if (mbox == NULL)
						goto flogifail;
					lpfc_config_link(phba, mbox);
					mbox->vport = vport;
					if (lpfc_sli_issue_mbox
					    (phba, mbox,
					     (MBX_NOWAIT | MBX_STOP_IOCB))
					    == MBX_NOT_FINISHED) {
						lpfc_mbox_free(phba, mbox);
						goto flogifail;
					}
					lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
						lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
						__func__, ndlp->nlp_DID, vport->vpi);
					lpfc_nlp_free(phba, ndlp);

					if ((ndlp =
					     lpfc_findnode_did(vport,
							       NLP_SEARCH_ALL,
							       PT2PT_RemoteID))
					    == 0) {
						/* Cannot find existing Fabric
						   ndlp, so allocate a new
						   one */
						if ((ndlp = (LPFC_NODELIST_t *)
						     lpfc_nlp_alloc(phba,
								    0)) == 0) {
							goto flogifail;
						}
						memset(ndlp, 0,
						       sizeof
						       (LPFC_NODELIST_t));
						ndlp->nlp_DID = PT2PT_RemoteID;
						lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
							lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
							__func__, ndlp->nlp_DID, vport->vpi);
					}
					memcpy(&ndlp->nlp_portname,
					       &sp->portName,
					       sizeof (NAME_TYPE));
					memcpy(&ndlp->nlp_nodename,
					       &sp->nodeName,
					       sizeof (NAME_TYPE));
					lpfc_nlp_plogi(vport, ndlp);
				} else {
					/* This side will wait for the PLOGI */
					lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
						lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
						__func__, ndlp->nlp_DID, vport->vpi);
					lpfc_nlp_free(phba, ndlp);
				}

				vport->fc_flag |= FC_PT2PT;
				lpfc_set_disctmo(vport);

				/* Start discovery - this should just do
				   CLEAR_LA */
				lpfc_disc_start(vport);
			}
			goto out;
		}
	}

      flogifail:
	lpfc_nlp_remove(vport, ndlp);
	phba->max_vpi = 0;

	/* Do not start discovery on a driver initialed FLOGI abort */
	if((irsp->ulpStatus != IOSTAT_LOCAL_REJECT) ||
		(irsp->un.ulpWord[4] != IOERR_SLI_ABORTED)) {

		/* FLOGI failed, so just use loop map to make discovery list */
		lpfc_disc_list_loopmap(vport);

		/* Start discovery */
		lpfc_disc_start(vport);
	}
	else if (vport->port_state < LPFC_CLEAR_LA) {
		/*
		 * Fix for CR 75243 so that link attention interrupts
		 * can be handled after a FLOGI timeout.
		 */
		mbox = lpfc_mbox_alloc(phba, MEM_PRI);
		if (mbox) {
			vport->port_state = LPFC_CLEAR_LA;
			lpfc_clear_la(phba, mbox);
			mbox->mbox_cmpl = lpfc_mbx_cmpl_clear_la;
			mbox->vport = vport;
			stat = lpfc_sli_issue_mbox(phba, mbox, 
			   (MBX_NOWAIT | MBX_STOP_IOCB));
			if (stat == MBX_NOT_FINISHED) {
				/* CLEAR_LA failure */
				lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0133,
					lpfc_mes0133,
					lpfc_msgBlk0133.msgPreambleStr, stat);
				lpfc_mbox_free(phba, mbox);
				lpfc_disc_flush_list(vport);
			}
		}
		else {
			/* No mbox resources */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0133,
				lpfc_mes0133,
				lpfc_msgBlk0133.msgPreambleStr, 0);
			lpfc_disc_flush_list(vport);
		}
	}

      out:
	/* if myDID ndlp exists, remove it */
	if ((ndlp = lpfc_findnode_did(vport,
	     NLP_SEARCH_ALL, vport->fc_myDID))) {
		lpfc_nlp_remove(vport, ndlp);
	}

	if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		(irsp->un.ulpWord[4] == IOERR_SLI_ABORTED))
		lpfc_free_delayed_iocbs(phba, cmdiocb);
	else
		lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

int
lpfc_els_abort_flogi(lpfcHBA_t *phba)
{
	LPFC_SLI_t *psli;
	LPFC_SLI_RING_t *pring;
	LPFC_IOCBQ_t *iocb, *next_iocb;
	LPFC_NODELIST_t *ndlp;
	IOCB_t *icmd;
	struct list_head *curr, *next;

	/* Abort outstanding I/O to the Fabric */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0242,
			lpfc_mes0242,
			lpfc_msgBlk0242.msgPreambleStr,
			Fabric_DID);

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];

	/* check the txcmplq */
	list_for_each_safe(curr, next, &pring->txcmplq) {
		next_iocb = list_entry(curr, LPFC_IOCBQ_t, list);
		iocb = next_iocb;
		/* Check to see if iocb matches the nport we are
		   looking for */
		icmd = &iocb->iocb;
		if (icmd->ulpCommand == CMD_ELS_REQUEST64_CR) {
			ndlp = (LPFC_NODELIST_t *)(iocb->context1);
			if(ndlp && (ndlp->nlp_DID == Fabric_DID)) {
				/* It matches, so deque and call compl
				   with an error */
				list_del(&iocb->list);
				pring->txcmplq_cnt--;

				if ((icmd->un.elsreq64.bdl.ulpIoTag32)) {
					lpfc_sli_issue_abort_iotag32
					    (phba, pring, iocb);
				}
				if (iocb->iocb_cmpl) {
					icmd->ulpStatus =
					    IOSTAT_LOCAL_REJECT;
					icmd->un.ulpWord[4] =
					    IOERR_SLI_ABORTED;
					(iocb->iocb_cmpl) (phba, iocb, iocb);
				} else {
					lpfc_iocb_free(phba, iocb);
				}
			}
		}
	}
	return (0);
}

int
lpfc_issue_els_plogi(lpfc_vport_t * vport, LPFC_NODELIST_t * ndlp, uint8_t retry)
{
	SERV_PARM *sp;
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = (sizeof (uint32_t) + sizeof (SERV_PARM));
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry,
				     ndlp, ELS_CMD_PLOGI);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	/* For PLOGI request, remainder of payload is service parameters */
	*((uint32_t *) (pCmd)) = ELS_CMD_PLOGI;
	pCmd += sizeof (uint32_t);
	memcpy(pCmd, &vport->fc_sparam, sizeof (SERV_PARM));
	sp = (SERV_PARM *) pCmd;

	if (sp->cmn.fcphLow < FC_PH_4_3)
		sp->cmn.fcphLow = FC_PH_4_3;

	if (sp->cmn.fcphHigh < FC_PH3)
		sp->cmn.fcphHigh = FC_PH3;

	phba->fc_stat.elsXmitPLOGI++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_plogi;
	ndlp->nlp_flag |= NLP_PLOGI_SND;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ)
	    == IOCB_ERROR) {
		ndlp->nlp_flag &= ~NLP_PLOGI_SND;
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}
	return 0;
}

void
lpfc_cmpl_els_plogi(lpfcHBA_t *phba, LPFC_IOCBQ_t * cmdiocb,
		    LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;
	LPFC_SLI_t *psli;
	LPFC_NODELIST_t *ndlp;
	int disc;
	lpfc_vport_t *vport = cmdiocb->vport;

	psli = &phba->sli;

	/* we pass cmdiocb to state machine which needs rspiocb as well */
	cmdiocb->context_un.rsp_iocb = rspiocb;
	irsp = &rspiocb->iocb;

	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;
	ndlp->nlp_flag &= ~NLP_PLOGI_SND;

	/* Since ndlp can be freed in the disc state machine, note if this node
	 * is being used during discovery.
	 */
	disc = (ndlp->nlp_flag & NLP_DISC_NODE);
	ndlp->nlp_flag &= ~NLP_DISC_NODE;

	/* PLOGI completes to NPort <nlp_DID> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0102,
			lpfc_mes0102, lpfc_msgBlk0102.msgPreambleStr,
			ndlp->nlp_DID, vport->vpi, irsp->ulpStatus, irsp->un.ulpWord[4],
			disc, vport->num_disc_nodes);

	/* Check to see if link went down during discovery */
	if (lpfc_els_chk_latt(phba, rspiocb))
		goto out;

	if (irsp->ulpStatus) {
		/* Check for retry */
		if (lpfc_els_retry(phba, cmdiocb, rspiocb)) {
			if (disc) 
				ndlp->nlp_flag |= NLP_DISC_NODE;
			goto out;
		}

		/*
		 * If the driver is aborting the PLOGI, the DSM has nothing
		 * further to do except release resources.
		 */
		if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		    ((irsp->un.ulpWord[4] == IOERR_SLI_ABORTED) ||
		    (irsp->un.ulpWord[4] == IOERR_LINK_DOWN) ||
		    (irsp->un.ulpWord[4] == IOERR_SLI_DOWN))) {
			goto out;
		}
	}

	/* Call the DSM to transition the ndlp state. */
	lpfc_disc_state_machine(vport, ndlp, cmdiocb, NLP_EVT_CMPL_PLOGI);

	if (disc && vport->num_disc_nodes) {
		/* Check to see if there are more PLOGIs to be sent */
		lpfc_more_plogi(vport);

		if (vport->num_disc_nodes == 0) {
			vport->fc_flag &= ~FC_NDISC_ACTIVE;

			lpfc_can_disctmo(vport);
			if (vport->fc_flag & FC_RSCN_MODE) {
				/*
				 * Check to see if more RSCNs came in while we were
				 * processing this one.
				 */
				if ((vport->fc_rscn_id_cnt == 0) &&
				    (!(vport->fc_flag & FC_RSCN_DISCOVERY))) {
					vport->fc_flag &= ~FC_RSCN_MODE;
				} else {
					lpfc_els_handle_rscn(vport);
				}
			}
		}
	}

 out:
	if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		(irsp->un.ulpWord[4] == IOERR_SLI_ABORTED))
		lpfc_free_delayed_iocbs(phba, cmdiocb);
	else
		lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

int
lpfc_issue_els_prli(lpfc_vport_t *vport, LPFC_NODELIST_t * ndlp, uint8_t retry)
{
	PRLI *npr;
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = (sizeof (uint32_t) + sizeof (PRLI));
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry, ndlp, 
				     ELS_CMD_PRLI);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	/* For PRLI request, remainder of payload is service parameters */
	memset(pCmd, 0, (sizeof (PRLI) + sizeof (uint32_t)));
	*((uint32_t *) (pCmd)) = ELS_CMD_PRLI;
	pCmd += sizeof (uint32_t);

	/* For PRLI, remainder of payload is PRLI parameter page */
	npr = (PRLI *) pCmd;
	/*
	 * If our firmware version is 3.20 or later, 
	 * set the following bits for FC-TAPE support.
	 */
	if (phba->vpd.rev.feaLevelHigh >= 0x02) {
		npr->ConfmComplAllowed = 1;
		npr->Retry = 1;
		npr->TaskRetryIdReq = 1;
	}
	npr->estabImagePair = 1;
	npr->readXferRdyDis = 1;

	/* For FCP support */
	npr->prliType = PRLI_FCP_TYPE;
	npr->initiatorFunc = 1;

	phba->fc_stat.elsXmitPRLI++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_prli;
	ndlp->nlp_flag |= NLP_PRLI_SND;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
	    IOCB_ERROR) {
		ndlp->nlp_flag &= ~NLP_PRLI_SND;
		lpfc_els_free_iocb(phba, elsiocb);
		return (1);
	}
	phba->fc_prli_sent++;
	return (0);
}

void
lpfc_cmpl_els_prli(lpfcHBA_t *phba, LPFC_IOCBQ_t * cmdiocb,
		   LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;
	LPFC_SLI_t *psli;
	LPFC_NODELIST_t *ndlp;
	lpfc_vport_t *vport = cmdiocb->vport;

	psli = &phba->sli;
	cmdiocb->context_un.rsp_iocb = rspiocb;

	irsp = &(rspiocb->iocb);
	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;
	ndlp->nlp_flag &= ~NLP_PRLI_SND;

	/* PRLI completes to NPort <nlp_DID> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0103,
			lpfc_mes0103, lpfc_msgBlk0103.msgPreambleStr,
			ndlp->nlp_DID, irsp->ulpStatus, irsp->un.ulpWord[4],
			vport->num_disc_nodes);

	phba->fc_prli_sent--;
	if (lpfc_els_chk_latt(phba, rspiocb))
		goto out;

	if (irsp->ulpStatus) {
		/* Check for retry */
		if (lpfc_els_retry(phba, cmdiocb, rspiocb))
			goto out;

		/*
		 * If the driver is aborting the PRLI, the DSM has nothing
		 * further to do except release resources.
		 */
		if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		    ((irsp->un.ulpWord[4] == IOERR_SLI_ABORTED) ||
		    (irsp->un.ulpWord[4] == IOERR_LINK_DOWN) ||
		    (irsp->un.ulpWord[4] == IOERR_SLI_DOWN))) {
			goto out;
 		}
	}
 
	/* Call the DSM to handle the state change. */
	lpfc_disc_state_machine(vport, ndlp, cmdiocb, NLP_EVT_CMPL_PRLI);

 out:
	if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		(irsp->un.ulpWord[4] == IOERR_SLI_ABORTED))
		lpfc_free_delayed_iocbs(phba, cmdiocb);
	else
		lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

int
lpfc_issue_els_adisc(lpfc_vport_t *vport, LPFC_NODELIST_t * ndlp, uint8_t retry)
{
	ADISC *ap;
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = (sizeof (uint32_t) + sizeof (ADISC));
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry, ndlp, 
				     ELS_CMD_ADISC);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	/* For ADISC request, remainder of payload is service parameters */
	*((uint32_t *) (pCmd)) = ELS_CMD_ADISC;
	pCmd += sizeof (uint32_t);

	/* Fill in ADISC payload */
	ap = (ADISC *) pCmd;
	ap->hardAL_PA = phba->fc_pref_ALPA;
	memcpy(&ap->portName, &vport->fc_portname, sizeof (NAME_TYPE));
	memcpy(&ap->nodeName, &vport->fc_nodename, sizeof (NAME_TYPE));
	ap->DID = be32_to_cpu(vport->fc_myDID);

	phba->fc_stat.elsXmitADISC++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_adisc;
	ndlp->nlp_flag |= NLP_ADISC_SND;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
							IOCB_ERROR) {
		ndlp->nlp_flag &= ~NLP_ADISC_SND;
		lpfc_els_free_iocb(phba, elsiocb);
		return (1);
	}
	return 0;
}

/* lpfc_rscn_disc is only called by lpfc_cmpl_els_adisc below */
static void
lpfc_rscn_disc(lpfc_vport_t *vport)
{
	LPFC_NODELIST_t *ndlp;
	lpfcCfgParam_t *clp;
	struct list_head *pos;
	lpfcHBA_t *phba = vport->phba;

	clp = &phba->config[0];

	/* RSCN discovery */
	/* go thru PLOGI list and issue ELS PLOGIs */
	if (vport->fc_plogi_cnt) {
		list_for_each(pos, &vport->fc_plogi_list) {
			ndlp = list_entry(pos, LPFC_NODELIST_t, nlp_listp);
			if ((ndlp->nlp_DID & Fabric_DID_MASK) == Fabric_DID_MASK)
				continue;
			if ((ndlp->nlp_state == NLP_STE_UNUSED_NODE) && 
			    (lpfc_node_teardown(ndlp)==0)) {
				ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
				lpfc_issue_els_plogi(vport, ndlp, 0);
				ndlp->nlp_flag |= NLP_DISC_NODE;
				vport->num_disc_nodes++;
				if (vport->num_disc_nodes >=
				    clp[LPFC_CFG_DISC_THREADS].a_current) {
					if (vport->fc_plogi_cnt >
					    vport->num_disc_nodes)
						vport->fc_flag |= FC_NLP_MORE;
					break;
				}
			}
		}
	} else {
		if (vport->fc_flag & FC_RSCN_MODE) {
			/* Check to see if more RSCNs came in while we were
			 * processing this one.
			 */
			if ((vport->fc_rscn_id_cnt == 0) &&
			    (!(vport->fc_flag & FC_RSCN_DISCOVERY))) {
				lpfc_els_flush_rscn(vport);
			} else {
				lpfc_els_handle_rscn(vport);
			}
		}
	}
}

void
lpfc_cmpl_els_adisc(lpfcHBA_t *phba, LPFC_IOCBQ_t * cmdiocb,
		    LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;
	LPFC_SLI_t *psli;
	LPFC_NODELIST_t *ndlp;
	LPFC_MBOXQ_t *mbox;
	int disc;
	lpfcCfgParam_t *clp;
	lpfc_vport_t *vport = cmdiocb->vport;
	struct list_head *pos;

	clp = &phba->config[0];
	psli = &phba->sli;

	cmdiocb->context_un.rsp_iocb = rspiocb;

	irsp = &(rspiocb->iocb);
	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;
	ndlp->nlp_flag &= ~NLP_ADISC_SND;

	if ((irsp->ulpStatus ) &&
	    (irsp->un.ulpWord[4] == IOERR_SLI_DOWN))
		goto out;

	/*
	 * Since ndlp can be freed in the disc state machine, note if this node
	 * is being used during discovery.
	 */
	disc = (ndlp->nlp_flag & NLP_DISC_NODE);
	ndlp->nlp_flag &= ~NLP_DISC_NODE;

	/* ADISC completes to NPort <nlp_DID> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0104,
			lpfc_mes0104, lpfc_msgBlk0104.msgPreambleStr,
			ndlp->nlp_DID, irsp->ulpStatus, irsp->un.ulpWord[4],
			disc, vport->num_disc_nodes);

	if (lpfc_els_chk_latt(phba, rspiocb))
		goto out;

	if (irsp->ulpStatus) {
		if (lpfc_els_retry(phba, cmdiocb, rspiocb)) {
			if (disc) {
				ndlp->nlp_flag |= NLP_DISC_NODE;
				lpfc_set_disctmo(vport);
			}

			goto out;
		}

		/*
		 * If the driver is aborting the ADISC, the DSM has nothing
		 * further to do except release resources.
		 */
		if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		    ((irsp->un.ulpWord[4] == IOERR_SLI_ABORTED) ||
		    (irsp->un.ulpWord[4] == IOERR_LINK_DOWN)    ||
		    (irsp->un.ulpWord[4] == IOERR_SLI_DOWN)))
			goto next_adisc;
	}

	/* Call the DSM and process this adisc completion. */
	lpfc_disc_state_machine(vport, ndlp, cmdiocb, NLP_EVT_CMPL_ADISC);

 next_adisc:

	if (disc && vport->num_disc_nodes) {
		/* Check to see if there are more ADISCs to be sent */
		lpfc_more_adisc(vport);

		/* Check to see if we are done with ADISC authentication */
		if (vport->num_disc_nodes == 0) {
			/* If we get here, there is nothing left to wait for */
			if ((vport->port_state < LPFC_HBA_READY) &&
			    (vport->port_state != LPFC_CLEAR_LA)) {
				if (vport->port_type == LPFC_PHYSICAL_PORT) {
					/* Link up discovery */
					mbox = lpfc_mbox_alloc(phba, MEM_PRI);
					if (mbox == NULL) 
						goto out;
					vport->port_state = LPFC_CLEAR_LA;
					lpfc_clear_la(phba, mbox);
					mbox->mbox_cmpl =
					lpfc_mbx_cmpl_clear_la;
					mbox->vport = vport;
					if (lpfc_sli_issue_mbox
					    (phba, mbox,
					     (MBX_NOWAIT | MBX_STOP_IOCB))
					    == MBX_NOT_FINISHED) {
						lpfc_mbox_free(phba, mbox);
						lpfc_disc_flush_list(vport);
						psli->ring[(psli->extra_ring)].
						flag &=
						~LPFC_STOP_IOCB_EVENT;
						psli->ring[(psli->fcp_ring)].
						flag &=
						~LPFC_STOP_IOCB_EVENT;
						psli->ring[(psli->next_ring)].
						flag &=
						~LPFC_STOP_IOCB_EVENT;
						vport->port_state =
						LPFC_HBA_READY;
					}
				} else {
					vport->num_disc_nodes = 0;
					/* go thru PLOGI list and issue ELS PLOGIs */
					if (vport->fc_plogi_cnt) {
						list_for_each(pos, &vport->fc_plogi_list) {
							ndlp = list_entry(pos, LPFC_NODELIST_t, nlp_listp);
							if ((ndlp->nlp_DID & Fabric_DID_MASK) == Fabric_DID_MASK)
								continue;
							if ((ndlp->nlp_state == NLP_STE_UNUSED_NODE) &&
							    (lpfc_node_teardown(ndlp)==0)) {
								ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
								lpfc_issue_els_plogi(vport, ndlp, 0);
								ndlp->nlp_flag |= NLP_DISC_NODE;
								vport->num_disc_nodes++;
								if (vport->num_disc_nodes >=
								    clp[LPFC_CFG_DISC_THREADS].a_current) {
									if (vport->fc_plogi_cnt >
									    vport->num_disc_nodes)
										vport->fc_flag |= FC_NLP_MORE;
									break;
								}
							}
						}
					}
					
					vport->port_state = LPFC_HBA_READY;
				
					/* Device Discovery completes */
					lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0225,
							lpfc_mes0225, lpfc_msgBlk0225.msgPreambleStr, vport->vpi);
				
					if (vport->fc_flag & FC_ESTABLISH_LINK) {
						vport->fc_flag &= ~FC_ESTABLISH_LINK;
					}
				
					lpfc_can_disctmo(vport);
				
				      
					/* If there are mapped FCP nodes still running, restart the scheduler 
					 * to get any pending IOCBs out.
					 */

					if (vport->fc_map_cnt) {
						lpfc_sched_check(vport);
					} 				
 				}
			} else {
				lpfc_rscn_disc(vport);
			}
		}
	}

 out:
	if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		(irsp->un.ulpWord[4] == IOERR_SLI_ABORTED))
		lpfc_free_delayed_iocbs(phba, cmdiocb);
	else
		lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

int
lpfc_issue_els_logo(lpfc_vport_t * vport, LPFC_NODELIST_t * ndlp, uint8_t retry)
{
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;
	int rc;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];

	cmdsize = (2 * sizeof (uint32_t)) + sizeof (NAME_TYPE);
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry,
				     ndlp, ELS_CMD_LOGO);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);
	*((uint32_t *) (pCmd)) = ELS_CMD_LOGO;
	pCmd += sizeof (uint32_t);

	/* Fill in LOGO payload */
	*((uint32_t *) (pCmd)) = be32_to_cpu(vport->fc_myDID);
	pCmd += sizeof (uint32_t);
	memcpy(pCmd, &vport->fc_portname, sizeof (NAME_TYPE));

	phba->fc_stat.elsXmitLOGO++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_logo;
	ndlp->nlp_flag |= NLP_LOGO_SND;

	rc = lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ);
	if (rc == IOCB_ERROR) {
		ndlp->nlp_flag &= ~NLP_LOGO_SND;
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}
	return 0;
}

void
lpfc_cmpl_els_logo(lpfcHBA_t *phba, LPFC_IOCBQ_t * cmdiocb,
		   LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;
	LPFC_SLI_t *psli;
	LPFC_NODELIST_t *ndlp;
	lpfc_vport_t *vport = cmdiocb->vport;

	psli = &phba->sli;
	cmdiocb->context_un.rsp_iocb = rspiocb;

	irsp = &(rspiocb->iocb);
	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;
	ndlp->nlp_flag &= ~NLP_LOGO_SND;

	/* LOGO completes to NPort <nlp_DID> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0105,
			lpfc_mes0105, lpfc_msgBlk0105.msgPreambleStr,
			ndlp->nlp_DID, irsp->ulpStatus, irsp->un.ulpWord[4],
			vport->num_disc_nodes);

	if (lpfc_els_chk_latt(phba, rspiocb))
		goto out;

	if (irsp->ulpStatus) {
		if (lpfc_els_retry(phba, cmdiocb, rspiocb))
			goto out;

		/*
		 * If the driver is aborting the LOGO, the DSM has nothing
		 * further to do except release resources.
		 */
		if((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		   ((irsp->un.ulpWord[4] == IOERR_SLI_ABORTED) ||
		   (irsp->un.ulpWord[4] == IOERR_LINK_DOWN) ||
		   (irsp->un.ulpWord[4] == IOERR_SLI_DOWN))) {
 			goto out;
 		}
	} else if ((ndlp->nlp_flag & NLP_TARGET_REMOVE) == 0) {
		/* The target can be recovered via the dsm. */
		lpfc_disc_state_machine(vport, ndlp, cmdiocb, NLP_EVT_CMPL_LOGO);
	} else {
	        /*
		 * Good status, but the driver doesn't want to recover the
		 * target.  Call the DSM to remove the node.
		 */
		lpfc_disc_state_machine(vport, ndlp, cmdiocb, NLP_EVT_DEVICE_RM);
	}

 out:
	if ((irsp->ulpStatus == IOSTAT_LOCAL_REJECT) &&
		(irsp->un.ulpWord[4] == IOERR_SLI_ABORTED))
		lpfc_free_delayed_iocbs(phba, cmdiocb);
	else
		lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

int
lpfc_issue_els_scr(lpfc_vport_t *vport, uint32_t nportid, uint8_t retry)
{
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	LPFC_NODELIST_t *ndlp;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */
	cmdsize = (sizeof (uint32_t) + sizeof (SCR));
	if ((ndlp = lpfc_nlp_alloc(phba, 0)) == 0) {
		return (1);
	}

	memset(ndlp, 0, sizeof (LPFC_NODELIST_t));
	ndlp->nlp_DID = nportid;

	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
		lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
		__func__, ndlp->nlp_DID, vport->vpi);
	
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry, ndlp, 
				     ELS_CMD_SCR);
	if (elsiocb == NULL) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
			lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
		lpfc_nlp_free(phba, ndlp);
		return 1;
	}

	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	*((uint32_t *) (pCmd)) = ELS_CMD_SCR;
	pCmd += sizeof (uint32_t);

	/* For SCR, remainder of payload is SCR parameter page */
	memset(pCmd, 0, sizeof (SCR));
	((SCR *) pCmd)->Function = SCR_FUNC_FULL;

	phba->fc_stat.elsXmitSCR++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_cmd;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
	    IOCB_ERROR) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
			lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
		lpfc_nlp_free(phba, ndlp);
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
		lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
		__func__, ndlp->nlp_DID, vport->vpi);
	lpfc_nlp_free(phba, ndlp);
	return 0;
}

int
lpfc_issue_els_farp(lpfc_vport_t *vport, uint8_t * arg,
		    LPFC_FARP_ADDR_TYPE argFlag)
{
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	FARP *fp;
	uint8_t *pCmd;
	uint32_t *lp;
	uint16_t cmdsize;
	LPFC_NODELIST_t *ndlp;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = (sizeof (uint32_t) + sizeof (FARP));
	if ((ndlp = lpfc_nlp_alloc(phba, 0)) == 0) {
		return 1;
	}
	memset(ndlp, 0, sizeof (LPFC_NODELIST_t));
	ndlp->nlp_DID = Bcast_DID;

	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
		lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
		__func__, ndlp->nlp_DID, vport->vpi);

	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, 0,
				     ndlp, ELS_CMD_RNID);
	if (elsiocb == NULL) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
			lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
		lpfc_nlp_free(phba, ndlp);
		return 1;
	}

	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);
	*((uint32_t *) (pCmd)) = ELS_CMD_FARP;
	pCmd += sizeof (uint32_t);

	/* Provide a timeout value, function, and context.  If the IP node on
	 * far end never responds, this FARP and all IP bufs must be timed out.
	 */
	icmd = &elsiocb->iocb;
	icmd->ulpTimeout = phba->fc_ipfarp_timeout;
	icmd->ulpContext = (uint16_t) ELS_CMD_FARP;

	/* Fill in FARP payload */

	fp = (FARP *) (pCmd);
	memset(fp, 0, sizeof (FARP));
	lp = (uint32_t *) pCmd;
	*lp++ = be32_to_cpu(vport->fc_myDID);
	fp->Mflags = FARP_MATCH_PORT;
	fp->Rflags = FARP_REQUEST_PLOGI;
	memcpy(&fp->OportName, &vport->fc_portname, sizeof (NAME_TYPE));
	memcpy(&fp->OnodeName, &vport->fc_nodename, sizeof (NAME_TYPE));
	switch (argFlag) {
	case LPFC_FARP_BY_IEEE:
		fp->Mflags = FARP_MATCH_PORT;
		fp->RportName.nameType = NAME_IEEE;	/* IEEE name */
		fp->RportName.IEEEextMsn = 0;
		fp->RportName.IEEEextLsb = 0;
		memcpy(fp->RportName.IEEE, arg, 6);
		fp->RnodeName.nameType = NAME_IEEE;	/* IEEE name */
		fp->RnodeName.IEEEextMsn = 0;
		fp->RnodeName.IEEEextLsb = 0;
		memcpy(fp->RnodeName.IEEE, arg, 6);
		break;
	case LPFC_FARP_BY_WWPN:
		fp->Mflags = FARP_MATCH_PORT;
		memcpy(&fp->RportName, arg, sizeof (NAME_TYPE));
		break;
	case LPFC_FARP_BY_WWNN:
		fp->Mflags = FARP_MATCH_NODE;
		memcpy(&fp->RnodeName, arg, sizeof (NAME_TYPE));
		break;
	}

	phba->fc_stat.elsXmitFARP++;

	elsiocb->iocb_cmpl = lpfc_cmpl_els_cmd;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
								IOCB_ERROR) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
			lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
		lpfc_nlp_free(phba, ndlp);
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}

	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0610,
			lpfc_mes0610,
			lpfc_msgBlk0610.msgPreambleStr,
			vport->fc_nodename.IEEE[0], vport->fc_nodename.IEEE[1],
			vport->fc_nodename.IEEE[2], vport->fc_nodename.IEEE[3],
			vport->fc_nodename.IEEE[4], vport->fc_nodename.IEEE[5]);
	return 0;
}

int
lpfc_issue_els_farpr(lpfc_vport_t * vport, uint32_t nportid, uint8_t retry)
{
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	FARP *fp;
	uint8_t *pCmd;
	uint32_t *lp;
	uint16_t cmdsize;
	LPFC_NODELIST_t *ondlp;
	LPFC_NODELIST_t *ndlp;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */
	cmdsize = (sizeof (uint32_t) + sizeof (FARP));
	if ((ndlp = lpfc_nlp_alloc(phba, 0)) == 0) {
		return 1;
	}
	memset(ndlp, 0, sizeof (LPFC_NODELIST_t));
	ndlp->nlp_DID = nportid;

	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
		lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
		__func__, ndlp->nlp_DID, vport->vpi);
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry,
				     ndlp, ELS_CMD_RNID);
	if (elsiocb == NULL) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
			lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
		lpfc_nlp_free(phba, ndlp);
		return 1;
	}

	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	*((uint32_t *) (pCmd)) = ELS_CMD_FARPR;
	pCmd += sizeof (uint32_t);

	/* Fill in FARPR payload */
	fp = (FARP *) (pCmd);
	memset(fp, 0, sizeof (FARP));
	lp = (uint32_t *) pCmd;
	*lp++ = be32_to_cpu(nportid);
	*lp++ = be32_to_cpu(vport->fc_myDID);
	fp->Rflags = 0;
	fp->Mflags = (FARP_MATCH_PORT | FARP_MATCH_NODE);

	memcpy(&fp->RportName, &vport->fc_portname, sizeof (NAME_TYPE));
	memcpy(&fp->RnodeName, &vport->fc_nodename, sizeof (NAME_TYPE));
	if ((ondlp = lpfc_findnode_did(vport, NLP_SEARCH_ALL, nportid))) {
		memcpy(&fp->OportName, &ondlp->nlp_portname,
		       sizeof (NAME_TYPE));
		memcpy(&fp->OnodeName, &ondlp->nlp_nodename,
		       sizeof (NAME_TYPE));
	}

	phba->fc_stat.elsXmitFARPR++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_cmd;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
								IOCB_ERROR) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
			lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
		lpfc_nlp_free(phba, ndlp);
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
		lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
		__func__, ndlp->nlp_DID, vport->vpi);
	lpfc_nlp_free(phba, ndlp);
	return 0;
}

void
lpfc_cmpl_els_cmd(lpfcHBA_t *phba, LPFC_IOCBQ_t * cmdiocb,
		  LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;

	irsp = &rspiocb->iocb;

	/* ELS cmd tag <ulpIoTag> completes */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0106,
			lpfc_mes0106,
			lpfc_msgBlk0106.msgPreambleStr,
			irsp->ulpIoTag, irsp->ulpStatus, irsp->un.ulpWord[4]);

	/* Check to see if link went down during discovery */
	lpfc_els_chk_latt(phba, rspiocb);
	lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

void
lpfc_els_retry_delay(unsigned long ptr)
{
	lpfcHBA_t *phba;
	lpfc_vport_t *vport;
	LPFC_NODELIST_t *ndlp;
	uint32_t cmd;
	uint32_t did;
	uint8_t retry;
	struct clk_data *clkData;
	unsigned long iflag;

	clkData = (struct clk_data *)ptr;
	phba = clkData->phba;
	vport = (lpfc_vport_t *)clkData->clData3;

	LPFC_DRVR_LOCK(phba, iflag);
       	if (clkData->flags & TM_CANCELED) {
		list_del((struct list_head *)clkData);
		kfree(clkData);	
		goto out;
	}

	did = (uint32_t) (unsigned long)(clkData->clData1);
	cmd = (uint32_t) (unsigned long)(clkData->clData2);
	clkData->timeObj->function = 0;
	list_del((struct list_head *)clkData);
	kfree(clkData);

	if ((ndlp = lpfc_findnode_did(vport, NLP_SEARCH_ALL, did)) == 0) {
		if ((ndlp = lpfc_nlp_alloc(phba, 0)) == 0) {
			LPFC_DRVR_UNLOCK(phba, iflag);
			return;
		}
		memset(ndlp, 0, sizeof (LPFC_NODELIST_t));
		ndlp->nlp_DID = did;

		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
			lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
	}

	ndlp->nlp_flag &= ~NLP_DELAY_TMO;
	ndlp->nlp_tmofunc.function = 0;
	retry = ndlp->nlp_retry;

	switch (cmd) {
	case ELS_CMD_FLOGI:
		lpfc_issue_els_flogi(vport, ndlp, retry);
		break;
	case ELS_CMD_FDISC:
		lpfc_issue_els_fdisc(vport, ndlp, retry);
		break;
	case ELS_CMD_PLOGI:
		if (lpfc_node_teardown(ndlp))
			break;
		ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
		lpfc_nlp_plogi(vport, ndlp);
		lpfc_issue_els_plogi(vport, ndlp, retry);
		break;
	case ELS_CMD_ADISC:
		lpfc_issue_els_adisc(vport, ndlp, retry);
		break;
	case ELS_CMD_PRLI:
		lpfc_issue_els_prli(vport, ndlp, retry);
		break;
	case ELS_CMD_LOGO:
		lpfc_issue_els_logo(vport, ndlp, retry);
		break;
	}
out:
	LPFC_DRVR_UNLOCK(phba, iflag);
	return;
}

int
lpfc_els_retry(lpfcHBA_t *phba, LPFC_IOCBQ_t *cmdiocb, LPFC_IOCBQ_t *rspiocb)
{
	IOCB_t *irsp;
	DMABUF_t *pCmd;
	LPFC_NODELIST_t *ndlp;
	uint32_t *elscmd;
	lpfcCfgParam_t *clp;
	LS_RJT stat;
	int retry, maxretry;
	int delay;
	uint32_t cmd;
	lpfc_vport_t *vport = cmdiocb->vport;

	clp = &phba->config[0];
	retry = 0;
	delay = 0;
	maxretry = lpfc_max_els_tries;
	irsp = &rspiocb->iocb;
	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;

	pCmd = (DMABUF_t *) cmdiocb->context2;
	cmd = 0;

	/* Note: context2 may be 0 for internal driver abort 
	 * of delays ELS command.
	 */

	if (pCmd && pCmd->virt) {
		elscmd = (uint32_t *) (pCmd->virt);
		cmd = *elscmd++;
	}

	switch (irsp->ulpStatus) {
	case IOSTAT_FCP_RSP_ERROR:
	case IOSTAT_REMOTE_STOP:
		break;

	case IOSTAT_LOCAL_REJECT:
		if ((irsp->un.ulpWord[4] & 0xff) == IOERR_LINK_DOWN)
			break;

		if (irsp->un.ulpWord[4] == IOERR_SLI_DOWN)
			break;

		if ((irsp->un.ulpWord[4] & 0xff) == IOERR_LOOP_OPEN_FAILURE) {
			if (cmd == ELS_CMD_PLOGI) {
				if (cmdiocb->retry == 0) {
					delay = 1;
				}
			}
			retry = 1;
			break;
		}
		if ((irsp->un.ulpWord[4] & 0xff) == IOERR_SEQUENCE_TIMEOUT) {
			retry = 1;
			if ((cmd == ELS_CMD_FLOGI || cmd == ELS_CMD_FDISC)
			    && (phba->fc_topology != TOPOLOGY_LOOP)) {
				delay = 1;
				maxretry = 48;
			}
			break;
		}
		if ((irsp->un.ulpWord[4] & 0xff) == IOERR_NO_RESOURCES) {
			if (cmd == ELS_CMD_PLOGI) {
				delay = 1;
			}
			retry = 1;
			break;
		}
		if ((irsp->un.ulpWord[4] & 0xff) == IOERR_INVALID_RPI) {
			break;
		}

		if ((irsp->un.ulpWord[4] & 0xff) == IOERR_ABORT_REQUESTED) {
			if (cmd == ELS_CMD_PRLI) {
				if (cmdiocb->retry == 0) {
					delay = 1;
				}
			}
			retry = 1;
			break;
		}
		break;

	case IOSTAT_NPORT_RJT:
	case IOSTAT_FABRIC_RJT:
		if (irsp->un.ulpWord[4] & RJT_UNAVAIL_TEMP) {
			retry = 1;
			break;
		}
		break;

	case IOSTAT_NPORT_BSY:
	case IOSTAT_FABRIC_BSY:
		retry = 1;
		break;

	case IOSTAT_LS_RJT:
		stat.un.lsRjtError = be32_to_cpu(irsp->un.ulpWord[4]);
		/* Added for Vendor specifc support
		 * Just keep retrying for these Rsn / Exp codes
		 */
		switch (stat.un.b.lsRjtRsnCode) {
		case LSRJT_UNABLE_TPC:
			if (stat.un.b.lsRjtRsnCodeExp ==
			    LSEXP_CMD_IN_PROGRESS) {
				if (cmd == ELS_CMD_PLOGI) {
					delay = 1;
					maxretry = 48;
				}
				retry = 1;
				break;
			}
			if (cmd == ELS_CMD_PLOGI) {
				delay = 1;
				maxretry = lpfc_max_els_tries + 1;
				retry = 1;
				break;
			}
			break;

		case LSRJT_LOGICAL_BSY:
			if (cmd == ELS_CMD_PLOGI) {
				delay = 1;
				maxretry = 48;
			} else if (cmd == ELS_CMD_FDISC) {
				maxretry = 48;
			}
			retry = 1;
			break;
		}
		break;

	case IOSTAT_INTERMED_RSP:
	case IOSTAT_BA_RJT:
		break;

	default:
		break;
	}

	if (ndlp->nlp_DID == FDMI_DID) {
		retry = 1;
	}

	if ((cmd == ELS_CMD_FLOGI) && 
	    (phba->fc_topology != TOPOLOGY_LOOP) &&
	    (!(irsp->ulpStatus == IOSTAT_LOCAL_REJECT &&
	       (irsp->un.ulpWord[4] == IOERR_SLI_ABORTED ||
		irsp->un.ulpWord[4] == IOERR_LINK_DOWN ||
		irsp->un.ulpWord[4] == IOERR_SLI_DOWN)))) {
		/* FLOGI retry policy */
		retry = 1;
		maxretry = 48;
		if (cmdiocb->retry >= 32)
			delay = 1000;
	}

	if ((++cmdiocb->retry) >= maxretry) {
		phba->fc_stat.elsRetryExceeded++;
		retry = 0;
	}

	if (retry) {

		/* Retry ELS command <elsCmd> to remote NPORT <did> */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0107,
				lpfc_mes0107,
				lpfc_msgBlk0107.msgPreambleStr,
				cmd, ndlp->nlp_DID, cmdiocb->retry, delay);

		if ((cmd == ELS_CMD_PLOGI) || (cmd == ELS_CMD_ADISC)) {
			/* If discovery / RSCN timer is running, reset it */
			if ((vport->fc_disctmo.function)
			    || (vport->fc_flag & FC_RSCN_MODE)) {
				lpfc_set_disctmo(vport);
			}
		}

		phba->fc_stat.elsXmitRetry++;
		if (delay) {
			phba->fc_stat.elsDelayRetry++;
			if (ndlp->nlp_tmofunc.function) {
				ndlp->nlp_flag &= ~NLP_DELAY_TMO;
				lpfc_stop_timer((struct clk_data *)
						ndlp->nlp_tmofunc.data);
			}
			if (cmd == ELS_CMD_PLOGI) {
				lpfc_nlp_plogi(vport, ndlp);
				ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
			}
			ndlp->nlp_flag |= NLP_DELAY_TMO;
			ndlp->nlp_retry = cmdiocb->retry;

			lpfc_start_timer(phba, 1, &ndlp->nlp_tmofunc,
					 lpfc_els_retry_delay,
					 (unsigned long) ndlp->nlp_DID,
					 (unsigned long) cmd,
					 (unsigned long) vport);

			return (1);
		}
		switch (cmd) {
		case ELS_CMD_FLOGI:
			lpfc_issue_els_flogi(vport, ndlp, cmdiocb->retry);
			return (1);
		case ELS_CMD_FDISC:
			if (vport->vport_logo_state > LPFC_VPORT_LOGO_NONE)
				break;
			lpfc_issue_els_fdisc(vport, ndlp, cmdiocb->retry);
			return (1);
		case ELS_CMD_PLOGI:
			if (vport->vport_logo_state > LPFC_VPORT_LOGO_NONE)
				break;
			if (lpfc_node_teardown(ndlp))
				break;
			lpfc_issue_els_plogi(vport, ndlp, cmdiocb->retry);
			return (1);
		case ELS_CMD_ADISC:
			if (vport->vport_logo_state > LPFC_VPORT_LOGO_NONE)
				break;
			lpfc_issue_els_adisc(vport, ndlp, cmdiocb->retry);
			return (1);
		case ELS_CMD_PRLI:
			if (vport->vport_logo_state > LPFC_VPORT_LOGO_NONE)
				break;
			lpfc_issue_els_prli(vport, ndlp, cmdiocb->retry);
			return (1);
		case ELS_CMD_LOGO:
			if (vport->vport_logo_state > LPFC_VPORT_LOGO_NONE)
				break;
			lpfc_issue_els_logo(vport, ndlp, cmdiocb->retry);
			return (1);
		}
	}

	/* No retry ELS command <elsCmd> to remote NPORT <did> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0108,
			lpfc_mes0108,
			lpfc_msgBlk0108.msgPreambleStr,
			cmd, ndlp->nlp_DID, cmdiocb->retry, ndlp->nlp_flag);

	return (0);
}

LPFC_IOCBQ_t *
lpfc_prep_els_iocb(lpfc_vport_t * vport, uint8_t expectRsp,
		   uint16_t cmdSize, uint8_t retry, 
		   LPFC_NODELIST_t * ndlp, uint32_t elscmd)
{
	LPFC_SLI_t *psli;
	LPFC_SLI_RING_t *pring;
	LPFC_IOCBQ_t *elsiocb;
	DMABUF_t *pCmd, *pRsp, *pBufList;
	ULP_BDE64 *bpl;
	IOCB_t *icmd;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	if (vport->port_state < LPFC_LINK_UP) {
		return (0);
	}

	/* Allocate buffer for  command iocb */
	elsiocb = lpfc_iocb_alloc(phba, MEM_PRI);
	if (elsiocb == NULL)
		return NULL;
	icmd = &elsiocb->iocb;

	/* fill in BDEs for command */
	/* Allocate buffer for command payload */
	if (((pCmd = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC)) == NULL) ||
	    ((pCmd->virt = lpfc_mbuf_alloc(phba,
					   MEM_PRI, &(pCmd->phys))) == NULL)) {
		if (pCmd != NULL)
			kfree(pCmd);
		lpfc_iocb_free(phba, elsiocb);
		return NULL;
	}

	INIT_LIST_HEAD(&pCmd->list);

	/* Allocate buffer for response payload */
	if (expectRsp) {
		pRsp = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC);
		if (pRsp != NULL)
			pRsp->virt = lpfc_mbuf_alloc(phba, MEM_PRI, 
						     &(pRsp->phys));
		if (pRsp == NULL || pRsp->virt == NULL) {
			if (pRsp != NULL)
				kfree(pRsp);
			lpfc_mbuf_free(phba, pCmd->virt, pCmd->phys);
			kfree(pCmd);
			lpfc_iocb_free(phba, elsiocb);
			return NULL;
		}
		INIT_LIST_HEAD(&pRsp->list);
	} else {
		pRsp = NULL;
	}

	/* Allocate buffer for Buffer ptr list */
	pBufList = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC);
	if (pBufList)
		pBufList->virt = lpfc_mbuf_alloc(phba, MEM_PRI,
						 &(pBufList->phys));
	if (pBufList == NULL || pBufList->virt == NULL) {
		lpfc_iocb_free(phba, elsiocb);
		lpfc_mbuf_free(phba, pCmd->virt, pCmd->phys);
		if (pRsp != NULL) {
			lpfc_mbuf_free(phba, pRsp->virt, pRsp->phys);
			kfree(pRsp);
		}
		kfree(pCmd);
		if (pBufList != NULL)
			kfree(pBufList);
		return NULL;
	}

	INIT_LIST_HEAD(&pBufList->list);

	icmd->un.elsreq64.bdl.addrHigh = putPaddrHigh(pBufList->phys);
	icmd->un.elsreq64.bdl.addrLow = putPaddrLow(pBufList->phys);
	icmd->un.elsreq64.bdl.bdeFlags = BUFF_TYPE_BDL;
	icmd->un.elsreq64.remoteID = ndlp->nlp_DID;
	if (expectRsp) {
		icmd->un.elsreq64.bdl.bdeSize = (2 * sizeof (ULP_BDE64));
		icmd->ulpCommand = CMD_ELS_REQUEST64_CR;
	} else {
		icmd->un.elsreq64.bdl.bdeSize = sizeof (ULP_BDE64);
		icmd->ulpCommand = CMD_XMIT_ELS_RSP64_CX;
	}
	icmd->ulpBdeCount = 1;
	icmd->ulpLe = 1;
	icmd->ulpClass = CLASS3;

	if (phba->max_vpi) {
		icmd->un.elsreq64.myID = vport->fc_myDID;
		icmd->ulpContext = vport->vpi;
		icmd->ulpCt_h = 0;
		/* The CT field must be 0=INVALID_RPI for the ECHO cmd */
		if (elscmd == ELS_CMD_ECHO)
			icmd->ulpCt_l = 0; /* context = invalid RPI */
		else
			icmd->ulpCt_l = 1; /* context = VPI */
	}

	bpl = (ULP_BDE64 *) pBufList->virt;
	bpl->addrLow = le32_to_cpu(putPaddrLow(pCmd->phys));
	bpl->addrHigh = le32_to_cpu(putPaddrHigh(pCmd->phys));
	bpl->tus.f.bdeSize = cmdSize;
	bpl->tus.f.bdeFlags = 0;
	bpl->tus.w = le32_to_cpu(bpl->tus.w);

	if (expectRsp) {
		bpl++;
		bpl->addrLow = le32_to_cpu(putPaddrLow(pRsp->phys));
		bpl->addrHigh = le32_to_cpu(putPaddrHigh(pRsp->phys));
		bpl->tus.f.bdeSize = FCELSSIZE;
		bpl->tus.f.bdeFlags = BUFF_USE_RCV;
		bpl->tus.w = le32_to_cpu(bpl->tus.w);
	}

	/* Save for completion so we can release these resources */
	if (elscmd != ELS_CMD_LS_RJT)
		elsiocb->context1 = (uint8_t *) ndlp;
	elsiocb->context2 = (uint8_t *) pCmd;
	elsiocb->context3 = (uint8_t *) pBufList;
	elsiocb->retry = retry;
	elsiocb->vport = vport;
	elsiocb->drvrTimeout = (phba->fc_ratov << 1) + LPFC_DRVR_TIMEOUT;

	if (pRsp) {
		list_add(&pRsp->list, &pCmd->list);
	}

	if (expectRsp) {
		/* Xmit ELS command <elsCmd> to remote NPORT <did> */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0116, lpfc_mes0116,
				lpfc_msgBlk0116.msgPreambleStr, elscmd,
				icmd->un.elsreq64.myID, ndlp->nlp_DID, vport->vpi, 
				vport->port_state);
	} else {
		/* Xmit ELS response <elsCmd> to remote NPORT <did> */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0117, lpfc_mes0117,
				lpfc_msgBlk0117.msgPreambleStr, elscmd,
				icmd->un.elsreq64.myID, ndlp->nlp_DID, vport->vpi, 
				cmdSize);
	}

	return elsiocb;
}

/*
 * This function is called without lock held. This function will free
 * all the delayed iocb resources.
 */
void
lpfc_free_all_delayed_iocbs(lpfcHBA_t *phba)
{
	LPFC_IOCBQ_t *iocb, *next_iocb;
	unsigned long iflag;

	LPFC_DRVR_LOCK(phba, iflag);	
	list_for_each_entry_safe(iocb, next_iocb, &phba->delayed_iocbs, list) {
		list_del(&iocb->list);
		lpfc_els_free_iocb(phba, iocb);
	}
	phba->delayed_iocb_count = 0;
	phba->next_delayed_timer = 0;
	LPFC_DRVR_UNLOCK(phba, iflag);
}

/*
 * This function is called from the timer context to free the delayed iocbs
 */
void
lpfc_free_delayed_iocbs_tmo(unsigned long ptr)
{
	lpfcHBA_t *phba = (lpfcHBA_t *) ptr;
	LPFC_IOCBQ_t *iocb, *next_iocb;
	unsigned long iflag;
	unsigned long iocb_time_out;

	LPFC_DRVR_LOCK(phba, iflag);
	phba->next_delayed_timer = 0;
	list_for_each_entry_safe(iocb, next_iocb, &phba->delayed_iocbs, list) {
		iocb_time_out = (unsigned long) iocb->context1;
		/* Free all the iocbs timing out within next second */
		if (time_before_eq(iocb_time_out, jiffies + HZ)) {
			list_del(&iocb->list);
			phba->delayed_iocb_count--;
			lpfc_els_free_iocb(phba, iocb);
		} else {
			mod_timer(&phba->delayed_iocb_tmo,
				iocb_time_out);
			phba->next_delayed_timer = iocb_time_out;
			break;
		}
	}
	LPFC_DRVR_UNLOCK(phba, iflag);
}

/*
 * This function is called with host_lock held to free iocb resource after
 * a delay to allow all DMA to the buffer to complete.
 */
void
lpfc_free_delayed_iocbs(lpfcHBA_t *phba, LPFC_IOCBQ_t *elsiocb)
{
	unsigned long iocb_time_out = jiffies + ELS_IOCB_DELAY_TIME * HZ;
	elsiocb->context1 = (void *) iocb_time_out;
	list_add_tail(&elsiocb->list, &phba->delayed_iocbs);
	phba->delayed_iocb_count++;
	if (!phba->next_delayed_timer) {
		phba->next_delayed_timer = iocb_time_out;
		mod_timer(&phba->delayed_iocb_tmo, iocb_time_out);
	}
}



int
lpfc_els_free_iocb(lpfcHBA_t *phba, LPFC_IOCBQ_t *elsiocb)
{
	DMABUF_t *buf_ptr, *buf_ptr1;

	/* context2  = cmd,  context2->next = rsp, context3 = bpl */
	if (elsiocb->context2) {
		buf_ptr1 = (DMABUF_t *) elsiocb->context2;
		/* Free the response before processing the command.  */
		if (!list_empty(&buf_ptr1->list)) {
			buf_ptr = list_entry(buf_ptr1->list.next,
					     DMABUF_t, list);
			lpfc_mbuf_free(phba, buf_ptr->virt, buf_ptr->phys);
			kfree(buf_ptr);
		}
		lpfc_mbuf_free(phba, buf_ptr1->virt, buf_ptr1->phys);
		kfree(buf_ptr1);
	}

	if (elsiocb->context3) {
		buf_ptr = (DMABUF_t *) elsiocb->context3;
		lpfc_mbuf_free(phba, buf_ptr->virt, buf_ptr->phys);
		kfree(buf_ptr);
	}

	lpfc_iocb_free(phba, elsiocb);
	return 0;
}

void
lpfc_cmpl_els_logo_acc(lpfcHBA_t *phba, LPFC_IOCBQ_t *cmdiocb,
			LPFC_IOCBQ_t *rspiocb)
{
	LPFC_NODELIST_t *ndlp;
	lpfcCfgParam_t *clp;
	int delay;
	lpfc_vport_t *vport = cmdiocb->vport;

	clp = &phba->config[0];
	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;

	/* ACC to LOGO completes to NPort <nlp_DID> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0109,
			lpfc_mes0109,
			lpfc_msgBlk0109.msgPreambleStr,
			ndlp->nlp_DID, ndlp->nlp_flag, ndlp->nlp_state,
			ndlp->nlp_rpi);

	delay = 1;
	switch (ndlp->nlp_state) {
	case NLP_STE_UNUSED_NODE:	/* node is just allocated */
	case NLP_STE_PLOGI_ISSUE:	/* PLOGI was sent to NL_PORT */
/* AAA
	case NLP_STE_ADISC_ISSUE:
*/
	case NLP_STE_REG_LOGIN_ISSUE:	/* REG_LOGIN was issued for NL_PORT */
		break;
	case NLP_STE_PRLI_ISSUE:	/* PRLI was sent to NL_PORT */
		/* dequeue, cancel timeout, unreg login */
		lpfc_freenode(vport, ndlp);

		/* put back on plogi list and send a new plogi */
		lpfc_nlp_plogi(vport, ndlp);
		ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
		lpfc_issue_els_plogi(vport, ndlp, 0);
		break;

	case NLP_STE_UNMAPPED_NODE:	/* Identified as a FCP Target */
	case NLP_STE_MAPPED_NODE:	/* Identified as a FCP Target */

		lpfc_set_failmask(vport, ndlp, LPFC_DEV_DISCONNECTED,
				  LPFC_SET_BITMASK);
		if (ndlp->nlp_flag & NLP_ADISC_SND) {
			/* dequeue, cancel timeout, unreg login */
			lpfc_freenode(vport, ndlp);

			/* put back on plogi list and send a new plogi */
			lpfc_nlp_plogi(vport, ndlp);
			ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
			lpfc_issue_els_plogi(vport, ndlp, 0);
		} else {
			/* dequeue, cancel timeout, unreg login */
			lpfc_freenode(vport, ndlp);

			if (ndlp->nlp_tmofunc.function) {
				ndlp->nlp_flag &= ~NLP_DELAY_TMO;
				lpfc_stop_timer((struct clk_data *)
						ndlp->nlp_tmofunc.data);
			}
			lpfc_nlp_plogi(vport, ndlp);
			ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
			ndlp->nlp_flag |= NLP_DELAY_TMO;
			ndlp->nlp_retry = 0;

			lpfc_start_timer(phba, delay, &ndlp->nlp_tmofunc,
					 lpfc_els_retry_delay,
					 (unsigned long)ndlp->nlp_DID,
					 (unsigned long)ELS_CMD_PLOGI,
					 (unsigned long)vport);

		}
		break;
	}
	if(ndlp)
		ndlp->nlp_flag &= ~NLP_LOGO_ACC;
	lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

void
lpfc_cmpl_els_rsp(lpfcHBA_t *phba, LPFC_IOCBQ_t *cmdiocb,
		  LPFC_IOCBQ_t *rspiocb)
{
	LPFC_NODELIST_t *ndlp;
	LPFC_MBOXQ_t *mbox = 0;
	lpfc_vport_t *vport = cmdiocb->vport;

	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;
	if (!ndlp) {

		/* Check to see if link went down during discovery */
		lpfc_els_chk_latt(phba, rspiocb);

		lpfc_els_free_iocb(phba, cmdiocb);
		return;
	}

	/* ELS response tag <ulpIoTag> completes */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0110,
			lpfc_mes0110,
			lpfc_msgBlk0110.msgPreambleStr,
			cmdiocb->iocb.ulpIoTag, rspiocb->iocb.ulpStatus,
			rspiocb->iocb.un.ulpWord[4], ndlp->nlp_DID,
			ndlp->nlp_flag, ndlp->nlp_state, ndlp->nlp_rpi);

	if (cmdiocb->context_un.mbox)
		mbox = cmdiocb->context_un.mbox;

	if (mbox) {
		if ((rspiocb->iocb.ulpStatus == 0) &&
		    ((ndlp->nlp_flag & NLP_ACC_REGLOGIN) ||
		     (ndlp->nlp_flag & NLP_RM_DFLT_RPI))) {
			mbox->context2 = ndlp;
			mbox->vport = vport;
			if (ndlp->nlp_flag & NLP_RM_DFLT_RPI) {
				mbox->mbox_cmpl = lpfc_mbx_cmpl_dflt_rpi;
				ndlp->nlp_flag &= ~NLP_RM_DFLT_RPI;
			}
			else {
				mbox->mbox_cmpl = lpfc_mbx_cmpl_reg_login;
				ndlp->nlp_state = NLP_STE_REG_LOGIN_ISSUE;
				ndlp->nlp_flag &= ~NLP_ACC_REGLOGIN;
			}

			if (lpfc_sli_issue_mbox(phba, mbox, MBX_NOWAIT)
			    != MBX_NOT_FINISHED) {
				goto out;
			}
			else {
				/* NOTE: we should have messages for unsuccessful
				   reglogin */
				DMABUF_t *mp;
				mp = (DMABUF_t *)(mbox->context1);
				lpfc_mbuf_free(phba, mp->virt, mp->phys);
				kfree(mp);
				lpfc_mbox_free(phba, mbox);
			}
		} else {
			lpfc_mbox_free(phba, mbox);
		}
	}

 out:

	/* Check to see if link went down during discovery */
	lpfc_els_chk_latt(phba, rspiocb);
	lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

int
lpfc_els_rsp_acc(lpfc_vport_t *vport, uint32_t flag, LPFC_IOCBQ_t * oldiocb,
		 LPFC_NODELIST_t *ndlp, LPFC_MBOXQ_t *mbox, uint8_t newnode)
{
	IOCB_t *icmd;
	IOCB_t *oldcmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */
	oldcmd = &oldiocb->iocb;

	switch (flag) {
	case ELS_CMD_ACC:
		cmdsize = sizeof (uint32_t);
		elsiocb = lpfc_prep_els_iocb(vport, FALSE, cmdsize, 
					     oldiocb->retry, ndlp, ELS_CMD_ACC);
		if (elsiocb == NULL)
			return 1;
		
		icmd = &elsiocb->iocb;
		icmd->ulpContext = oldcmd->ulpContext;	/* Xri */
		pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);
		*((uint32_t *) (pCmd)) = ELS_CMD_ACC;
		pCmd += sizeof (uint32_t);
		break;
	case ELS_CMD_PLOGI:
		cmdsize = (sizeof (SERV_PARM) + sizeof (uint32_t));
		elsiocb = lpfc_prep_els_iocb(vport, FALSE, cmdsize, 
					     oldiocb->retry, ndlp, ELS_CMD_ACC);
		if (elsiocb == NULL)
			return 1;
		
		icmd = &elsiocb->iocb;
		icmd->ulpContext = oldcmd->ulpContext;	/* Xri */
		pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

		if (mbox)
			elsiocb->context_un.mbox = mbox;

		*((uint32_t *) (pCmd)) = ELS_CMD_ACC;
		pCmd += sizeof (uint32_t);
		memcpy(pCmd, &vport->fc_sparam, sizeof (SERV_PARM));
		break;
	case ELS_CMD_PRLO:
		cmdsize = sizeof(uint32_t) + sizeof(PRLO);
		elsiocb = lpfc_prep_els_iocb(vport, FALSE, cmdsize, 
					     oldiocb->retry, ndlp, ELS_CMD_ACC);
		if (elsiocb == NULL)
			return 1;

		icmd = &elsiocb->iocb;
		icmd->ulpContext = oldcmd->ulpContext; /* Xri */
		pCmd = (((struct lpfc_dmabuf *) elsiocb->context2)->virt);

		memcpy(pCmd, ((struct lpfc_dmabuf *) oldiocb->context2)->virt,
		       sizeof(uint32_t) + sizeof(PRLO));
		*((uint32_t *) (pCmd)) = ELS_CMD_PRLO_ACC;
		((ELS_PKT *) pCmd)->un.prlo.acceptRspCode = PRLO_REQ_EXECUTED;
		break;
	default:
		return (1);
	}

	if (newnode)
		elsiocb->context1 = 0;

	if (ndlp->nlp_flag & NLP_LOGO_ACC)
		elsiocb->iocb_cmpl = lpfc_cmpl_els_logo_acc;
	else 
		elsiocb->iocb_cmpl = lpfc_cmpl_els_rsp;

	phba->fc_stat.elsXmitACC++;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
								IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return (1);
	}
	return (0);
}

int
lpfc_els_rsp_reject(lpfc_vport_t * vport, uint32_t rejectError,
		    LPFC_IOCBQ_t * oldiocb, LPFC_NODELIST_t * ndlp,
		    LPFC_MBOXQ_t *mbox)
{
	IOCB_t *icmd;
	IOCB_t *oldcmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = 2 * sizeof (uint32_t);
	elsiocb = lpfc_prep_els_iocb(vport, FALSE, cmdsize, oldiocb->retry,
				     ndlp, ELS_CMD_LS_RJT);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	oldcmd = &oldiocb->iocb;
	icmd->ulpContext = oldcmd->ulpContext;	/* Xri */
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	*((uint32_t *) (pCmd)) = ELS_CMD_LS_RJT;
	pCmd += sizeof (uint32_t);
	*((uint32_t *) (pCmd)) = rejectError;

	if (mbox) {
		elsiocb->context_un.mbox = mbox;
		elsiocb->context1 = ndlp;
	}

	phba->fc_stat.elsXmitLSRJT++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_rsp;

	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
								IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}
	return 0;
}

int
lpfc_els_rsp_adisc_acc(lpfc_vport_t *vport, LPFC_IOCBQ_t *oldiocb, LPFC_NODELIST_t *ndlp)
{
	ADISC *ap;
	IOCB_t *icmd;
	IOCB_t *oldcmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = sizeof (uint32_t) + sizeof (ADISC);
	elsiocb = lpfc_prep_els_iocb(vport, FALSE, cmdsize, oldiocb->retry,
				     ndlp, ELS_CMD_ACC);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	oldcmd = &oldiocb->iocb;
	icmd->ulpContext = oldcmd->ulpContext;	/* Xri */
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	*((uint32_t *) (pCmd)) = ELS_CMD_ACC;
	pCmd += sizeof (uint32_t);

	ap = (ADISC *) (pCmd);
	ap->hardAL_PA = phba->fc_pref_ALPA;
	memcpy(&ap->portName, &vport->fc_portname, sizeof (NAME_TYPE));
	memcpy(&ap->nodeName, &vport->fc_nodename, sizeof (NAME_TYPE));
	ap->DID = be32_to_cpu(vport->fc_myDID);

	phba->fc_stat.elsXmitACC++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_rsp;

	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
								IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return (1);
	}
	return (0);
}

int
lpfc_els_rsp_prli_acc(lpfc_vport_t * vport,
		      LPFC_IOCBQ_t * oldiocb, LPFC_NODELIST_t * ndlp)
{
	PRLI *npr;
	lpfc_vpd_t *vpd;
	IOCB_t *icmd;
	IOCB_t *oldcmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = sizeof (uint32_t) + sizeof (PRLI);
	elsiocb = lpfc_prep_els_iocb(vport, FALSE, cmdsize, oldiocb->retry,
				     ndlp, (ELS_CMD_ACC | 
					    (ELS_CMD_PRLI & ~ELS_RSP_MASK)));
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	oldcmd = &oldiocb->iocb;
	icmd->ulpContext = oldcmd->ulpContext;	/* Xri */
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	*((uint32_t *) (pCmd)) = (ELS_CMD_ACC | (ELS_CMD_PRLI & ~ELS_RSP_MASK));
	pCmd += sizeof (uint32_t);

	/* For PRLI, remainder of payload is PRLI parameter page */
	memset(pCmd, 0, sizeof (PRLI));

	npr = (PRLI *) pCmd;
	vpd = &phba->vpd;
	/*
	 * If our firmware version is 3.20 or later, 
	 * set the following bits for FC-TAPE support.
	 */
	if (vpd->rev.feaLevelHigh >= 0x02) {
		npr->ConfmComplAllowed = 1;
		npr->Retry = 1;
		npr->TaskRetryIdReq = 1;
	}

	npr->acceptRspCode = PRLI_REQ_EXECUTED;
	npr->estabImagePair = 1;
	npr->readXferRdyDis = 1;
	npr->ConfmComplAllowed = 1;

	npr->prliType = PRLI_FCP_TYPE;
	npr->initiatorFunc = 1;

	phba->fc_stat.elsXmitACC++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_rsp;

	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
								IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return (1);
	}
	return (0);
}

int
lpfc_els_rsp_rnid_acc(lpfc_vport_t *vport, uint8_t format, LPFC_IOCBQ_t *oldiocb,
		      LPFC_NODELIST_t *ndlp)
{
	RNID *rn;
	IOCB_t *icmd;
	IOCB_t *oldcmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;
	lpfcCfgParam_t *clp;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];
	clp = &phba->config[0];

	cmdsize =
	    sizeof (uint32_t) + sizeof (uint32_t) + (2 * sizeof (NAME_TYPE));
	if (format)
		cmdsize += sizeof (RNID_TOP_DISC);

	elsiocb = lpfc_prep_els_iocb(vport, FALSE, cmdsize, oldiocb->retry,
				     ndlp, ELS_CMD_ACC);
	if (elsiocb == NULL)
		return (1);

	icmd = &elsiocb->iocb;
	oldcmd = &oldiocb->iocb;
	icmd->ulpContext = oldcmd->ulpContext;	/* Xri */
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	*((uint32_t *) (pCmd)) = ELS_CMD_ACC;
	pCmd += sizeof (uint32_t);

	memset(pCmd, 0, sizeof (RNID));
	rn = (RNID *) (pCmd);
	rn->Format = format;
	rn->CommonLen = (2 * sizeof (NAME_TYPE));
	memcpy(&rn->portName, &vport->fc_portname, sizeof (NAME_TYPE));
	memcpy(&rn->nodeName, &vport->fc_nodename, sizeof (NAME_TYPE));
	switch (format) {
	case 0:
		rn->SpecificLen = 0;
		break;
	case RNID_TOPOLOGY_DISC:
		rn->SpecificLen = sizeof (RNID_TOP_DISC);
		memcpy(&rn->un.topologyDisc.portName,
		       &vport->fc_portname, sizeof (NAME_TYPE));
		rn->un.topologyDisc.unitType = RNID_HBA;
		rn->un.topologyDisc.physPort = 0;
		rn->un.topologyDisc.attachedNodes = 0;
		break;
	default:
		rn->CommonLen = 0;
		rn->SpecificLen = 0;
		break;
	}

	phba->fc_stat.elsXmitACC++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_rsp;

	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, SLI_IOCB_USE_TXQ) ==
								IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return (1);
	}
	return (0);
}

static int
lpfc_els_rcv_lirr(lpfc_vport_t *vport, LPFC_IOCBQ_t *cmdiocb,
		  LPFC_NODELIST_t *ndlp)
{
	LS_RJT stat;

	/* For now, unconditionally reject this command */
	stat.un.b.lsRjtRsvd0 = 0;
	stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
	stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
	stat.un.b.vendorUnique = 0;
	lpfc_els_rsp_reject(vport, stat.un.lsRjtError, cmdiocb, ndlp, NULL);
	return 0;
}

static void
lpfc_els_rsp_rps_acc(lpfcHBA_t *phba, LPFC_MBOXQ_t *pmb)
{
	LPFC_SLI_t *psli = &phba->sli;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_ELS_RING];
	MAILBOX_t *mb;
	IOCB_t *icmd;
	RPS_RSP *rps_rsp;
	uint8_t *pcmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_NODELIST_t *ndlp;
	uint16_t xri, status;
	uint32_t cmdsize;

	mb = &pmb->mb;

	ndlp = (LPFC_NODELIST_t *) pmb->context2;
	xri = (uint16_t) ((unsigned long)(pmb->context1));
	pmb->context1 = NULL;
	pmb->context2 = NULL;

	if (mb->mbxStatus) {
		lpfc_mbox_free(phba, pmb);
		return;
	}

	cmdsize = sizeof(RPS_RSP) + sizeof(uint32_t);
	lpfc_mbox_free(phba, pmb);
	elsiocb = lpfc_prep_els_iocb(phba->pport, 0, cmdsize,
				     lpfc_max_els_tries, ndlp, ELS_CMD_ACC);
	if (elsiocb == NULL)
		return;

	icmd = &elsiocb->iocb;
	icmd->ulpContext = xri;

	pcmd = (uint8_t *) (((struct lpfc_dmabuf *) elsiocb->context2)->virt);
	*((uint32_t *) (pcmd)) = ELS_CMD_ACC;
	pcmd += sizeof(uint32_t); /* Skip past command */
	rps_rsp = (RPS_RSP *)pcmd;

	if (phba->fc_topology != TOPOLOGY_LOOP)
		status = 0x10;
	else
		status = 0x8;
	if (phba->pport->fc_flag & FC_FABRIC)
		status |= 0x4;

	rps_rsp->rsvd1 = 0;
	rps_rsp->portStatus = be16_to_cpu(status);
	rps_rsp->linkFailureCnt = be32_to_cpu(mb->un.varRdLnk.linkFailureCnt);
	rps_rsp->lossSyncCnt = be32_to_cpu(mb->un.varRdLnk.lossSyncCnt);
	rps_rsp->lossSignalCnt = be32_to_cpu(mb->un.varRdLnk.lossSignalCnt);
	rps_rsp->primSeqErrCnt = be32_to_cpu(mb->un.varRdLnk.primSeqErrCnt);
	rps_rsp->invalidXmitWord = be32_to_cpu(mb->un.varRdLnk.invalidXmitWord);
	rps_rsp->crcCnt = be32_to_cpu(mb->un.varRdLnk.crcCnt);

	elsiocb->iocb_cmpl = lpfc_cmpl_els_rsp;
	phba->fc_stat.elsXmitACC++;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, 0) == IOCB_ERROR)
		lpfc_els_free_iocb(phba, elsiocb);
	return;
}

static int
lpfc_els_rcv_rps(lpfc_vport_t *vport, LPFC_IOCBQ_t *cmdiocb,
		 LPFC_NODELIST_t *ndlp)
{
	lpfcHBA_t *phba = vport->phba;
	uint32_t *lp;
	uint8_t flag;
	LPFC_MBOXQ_t *mbox;
	struct lpfc_dmabuf *pcmd;
	RPS *rps;
	LS_RJT stat;

	if ((ndlp->nlp_state != NLP_STE_UNMAPPED_NODE) &&
	    (ndlp->nlp_state != NLP_STE_MAPPED_NODE)) {
		stat.un.b.lsRjtRsvd0 = 0;
		stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
		stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
		stat.un.b.vendorUnique = 0;
		lpfc_els_rsp_reject(vport, stat.un.lsRjtError, cmdiocb, ndlp,
				    NULL);
	}

	pcmd = (struct lpfc_dmabuf *) cmdiocb->context2;
	lp = (uint32_t *) pcmd->virt;
	flag = (be32_to_cpu(*lp++) & 0xf);
	rps = (RPS *) lp;

	if ((flag == 0) ||
	    ((flag == 1) && (be32_to_cpu(rps->un.portNum) == 0)) ||
	    ((flag == 2) && (memcmp(&rps->un.portName, &vport->fc_portname,
				    sizeof(NAME_TYPE)) == 0))) {
		mbox = lpfc_mbox_alloc(phba, MEM_PRI);
		if (mbox != NULL) {
			lpfc_read_lnk_stat(phba, mbox);
			mbox->context1 =
			    (void *)((unsigned long) cmdiocb->iocb.ulpContext);
			mbox->context2 = ndlp;
			mbox->vport = vport;
			mbox->mbox_cmpl = lpfc_els_rsp_rps_acc;
			if (lpfc_sli_issue_mbox (phba, mbox,
			    (MBX_NOWAIT | MBX_STOP_IOCB)) != MBX_NOT_FINISHED)
				/* Mbox completion will send ELS Response */
				return 0;

			lpfc_mbox_free(phba, mbox);
		}
	}
	stat.un.b.lsRjtRsvd0 = 0;
	stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
	stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
	stat.un.b.vendorUnique = 0;
	lpfc_els_rsp_reject(vport, stat.un.lsRjtError, cmdiocb, ndlp, NULL);
	return 0;
}

static int
lpfc_els_rsp_rpl_acc(lpfc_vport_t *vport, uint16_t cmdsize,
		     LPFC_IOCBQ_t *oldiocb, LPFC_NODELIST_t *ndlp)
{
	lpfcHBA_t *phba = vport->phba;
	IOCB_t *icmd, *oldcmd;
	RPL_RSP rpl_rsp;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_t *psli = &phba->sli;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_ELS_RING];
	uint8_t *pcmd;

	elsiocb = lpfc_prep_els_iocb(vport, 0, cmdsize, oldiocb->retry, ndlp,
				     ELS_CMD_ACC);
	if (elsiocb == NULL)
		return 1;

	icmd = &elsiocb->iocb;
	oldcmd = &oldiocb->iocb;
	icmd->ulpContext = oldcmd->ulpContext;	/* Xri */

	pcmd = (((struct lpfc_dmabuf *) elsiocb->context2)->virt);
	*((uint32_t *) (pcmd)) = ELS_CMD_ACC;
	pcmd += sizeof(uint16_t);
	*((uint16_t *)(pcmd)) = be16_to_cpu(cmdsize);
	pcmd += sizeof(uint16_t);

	/* Setup the RPL ACC payload */
	rpl_rsp.listLen = be32_to_cpu(1);
	rpl_rsp.index = 0;
	rpl_rsp.port_num_blk.portNum = 0;
	rpl_rsp.port_num_blk.portID = be32_to_cpu(vport->fc_myDID);
	memcpy(&rpl_rsp.port_num_blk.portName, &vport->fc_portname,
	       sizeof(NAME_TYPE));

	memcpy(pcmd, &rpl_rsp, cmdsize - sizeof(uint32_t));

	/* Xmit ELS RPL ACC response tag <ulpIoTag> */
	elsiocb->iocb_cmpl = lpfc_cmpl_els_rsp;

	phba->fc_stat.elsXmitACC++;
	if (lpfc_sli_issue_iocb(phba, pring, elsiocb, 0) == IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}
	return 0;
}

static int
lpfc_els_rcv_rpl(lpfc_vport_t *vport, LPFC_IOCBQ_t *cmdiocb,
		 LPFC_NODELIST_t *ndlp)
{
	struct lpfc_dmabuf *pcmd;
	uint32_t *lp;
	uint32_t maxsize;
	uint16_t cmdsize;
	RPL *rpl;
	LS_RJT stat;

	if ((ndlp->nlp_state != NLP_STE_UNMAPPED_NODE) &&
	    (ndlp->nlp_state != NLP_STE_MAPPED_NODE)) {
		stat.un.b.lsRjtRsvd0 = 0;
		stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
		stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
		stat.un.b.vendorUnique = 0;
		lpfc_els_rsp_reject(vport, stat.un.lsRjtError, cmdiocb, ndlp,
				    NULL);
	}

	pcmd = (struct lpfc_dmabuf *) cmdiocb->context2;
	lp = (uint32_t *) pcmd->virt;
	rpl = (RPL *) (lp + 1);

	maxsize = be32_to_cpu(rpl->maxsize);

	/* We support only one port */
	if ((rpl->index == 0) &&
	    ((maxsize == 0) ||
	     ((maxsize * sizeof(uint32_t)) >= sizeof(RPL_RSP)))) {
		cmdsize = sizeof(uint32_t) + sizeof(RPL_RSP);
	} else {
		cmdsize = sizeof(uint32_t) + maxsize * sizeof(uint32_t);
	}
	lpfc_els_rsp_rpl_acc(vport, cmdsize, cmdiocb, ndlp);

	return 0;
}

void
lpfc_els_unsol_buffer(lpfcHBA_t * phba, LPFC_SLI_RING_t * pring, 
		      lpfc_vport_t *vport, LPFC_IOCBQ_t *elsiocb)
{
	LPFC_NODELIST_t *ndlp;
	uint32_t *lp;
	LS_RJT stat;
	uint32_t cmd, rej_unsol_els_cmd = 0;
	uint32_t did;
	uint32_t newnode;
	struct list_head *curr, *next;
	IOCB_t *icmd = &elsiocb->iocb;
	uint8_t rjt_reason = 0;

	if ((elsiocb->context2 == NULL) || (vport == NULL))
		goto dropit;

	newnode = 0;
	lp = ((DMABUF_t *)elsiocb->context2)->virt;
	cmd = *lp++;
	if (!(phba->sli3_options & LPFC_SLI3_HBQ_ENABLED))
		lpfc_post_buffer(phba, pring, 1, 1);

	if (icmd->ulpStatus) {
		goto dropit;
	}

	/* Check to see if link went down during discovery */
	if (lpfc_els_chk_latt(phba, elsiocb)) {
		goto dropit;
	}

	if (phba->max_vpi && icmd->ulpPU == PARM_NPIV_DID) {
		list_for_each_safe(curr, next, &phba->port_list) {
			vport = list_entry(curr, lpfc_vport_t, listentry);
			if (vport->fc_myDID == icmd->un.rcvels.parmRo)
				break;
		}

		if (!vport) {
			goto dropit;
		}
	} else {
		vport = phba->pport;     
	}
		
	did = icmd->un.rcvels.remoteID;
	if ((ndlp = lpfc_findnode_did(vport, NLP_SEARCH_ALL, did)) == 0) {
		/* Cannot find existing Fabric ndlp, so allocate a new one */
		if ((ndlp = lpfc_nlp_alloc(phba, 0)) == 0) {
			goto dropit;
		}

		newnode = 1;
		memset(ndlp, 0, sizeof (LPFC_NODELIST_t));
		ndlp->nlp_DID = did;
		if ((did & Fabric_DID_MASK) == Fabric_DID_MASK) {
			ndlp->nlp_type |= NLP_FABRIC;
		}
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
			lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
			__func__, ndlp->nlp_DID, vport->vpi);
	}

	phba->fc_stat.elsRcvFrame++;
	elsiocb->context1 = ndlp;
	elsiocb->vport = vport;

	if ((cmd & ELS_CMD_MASK) == ELS_CMD_RSCN) {
		cmd &= ELS_CMD_MASK;
	}

	/* ELS command <elsCmd> received from NPORT <did> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0112,
			lpfc_mes0112,
			lpfc_msgBlk0112.msgPreambleStr,
			cmd, vport->vpi, did, vport->port_state);


	/* 
	 * If node is being torn down, don't except incoming
	 * els commands.
	 */
	if (lpfc_node_teardown(ndlp)) {
		rej_unsol_els_cmd++;
		rjt_reason = LSRJT_UNABLE_TPC;
		goto reject;
	}

	switch (cmd) {
	case ELS_CMD_PLOGI:
		phba->fc_stat.elsRcvPLOGI++;

		/* If either the vport or target initiated a LOGO and it's inflight.... */

		if ((vport->vport_logo_state > LPFC_VPORT_LOGO_NONE) ||
		   (ndlp->nlp_flag == NLP_LOGO_ACC)) {
			rej_unsol_els_cmd++;
			rjt_reason = LSRJT_UNABLE_TPC;
		}
		else
			lpfc_disc_state_machine(vport, ndlp, elsiocb, NLP_EVT_RCV_PLOGI);
		break;
	case ELS_CMD_FLOGI:
		phba->fc_stat.elsRcvFLOGI++;
		lpfc_els_rcv_flogi(vport, elsiocb, ndlp, newnode);
		if (newnode) {
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
				lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
				__func__, ndlp->nlp_DID, vport->vpi);
			lpfc_nlp_free(phba, ndlp);
		}
		break;
	case ELS_CMD_LOGO:
		phba->fc_stat.elsRcvLOGO++;
		if (ndlp->nlp_Target)
			lpfc_set_npr_tmo(vport, ndlp->nlp_Target, ndlp);
		lpfc_disc_state_machine(vport, ndlp, elsiocb, NLP_EVT_RCV_LOGO);
		break;
	case ELS_CMD_PRLO:
		phba->fc_stat.elsRcvPRLO++;
		if (ndlp->nlp_Target)
			lpfc_set_npr_tmo(vport, ndlp->nlp_Target, ndlp);
		lpfc_disc_state_machine(vport, ndlp, elsiocb, NLP_EVT_RCV_PRLO);
		break;
	case ELS_CMD_RSCN:
		phba->fc_stat.elsRcvRSCN++;
		lpfc_els_rcv_rscn(vport, elsiocb, ndlp, newnode);
		if (newnode) {
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
				lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
				__func__, ndlp->nlp_DID, vport->vpi);
			lpfc_nlp_free(phba, ndlp);
		}
		break;
	case ELS_CMD_ADISC:
		phba->fc_stat.elsRcvADISC++;
		if (vport->vport_logo_state > LPFC_VPORT_LOGO_NONE) {
			rej_unsol_els_cmd++;
			rjt_reason = LSRJT_UNABLE_TPC;
		}
		else
			lpfc_disc_state_machine(vport, ndlp, elsiocb, NLP_EVT_RCV_ADISC);
		break;
	case ELS_CMD_PDISC:
		phba->fc_stat.elsRcvPDISC++;
		lpfc_disc_state_machine(vport, ndlp, elsiocb, NLP_EVT_RCV_PDISC);
		break;
	case ELS_CMD_FARPR:
		phba->fc_stat.elsRcvFARPR++;
		lpfc_els_rcv_farpr(vport, elsiocb, ndlp);
		break;
	case ELS_CMD_FARP:
		phba->fc_stat.elsRcvFARP++;
		lpfc_els_rcv_farp(vport, elsiocb, ndlp);
		break;
	case ELS_CMD_FAN:
		phba->fc_stat.elsRcvFAN++;
		lpfc_els_rcv_fan(vport, elsiocb, ndlp);
		break;
	case ELS_CMD_RRQ:
		phba->fc_stat.elsRcvRRQ++;
		lpfc_els_rcv_rrq(vport, elsiocb, ndlp);
		break;
	case ELS_CMD_PRLI:
		phba->fc_stat.elsRcvPRLI++;
		lpfc_disc_state_machine(vport, ndlp, elsiocb, NLP_EVT_RCV_PRLI);
		break;
	case ELS_CMD_LIRR:
		phba->fc_stat.elsRcvLIRR++;
		lpfc_els_rcv_lirr(vport, elsiocb, ndlp);
		break;
	case ELS_CMD_RPS:
		phba->fc_stat.elsRcvRPS++;
		lpfc_els_rcv_rps(vport, elsiocb, ndlp);
		break;
	case ELS_CMD_RPL:
		phba->fc_stat.elsRcvRPL++;
		lpfc_els_rcv_rpl(vport, elsiocb, ndlp);
		break;
	case ELS_CMD_RNID:
		phba->fc_stat.elsRcvRNID++;
		lpfc_els_rcv_rnid(vport, elsiocb, ndlp);
		break;
	default:
		rej_unsol_els_cmd = 1;
		rjt_reason = LSRJT_CMD_UNSUPPORTED;

		/* Unknown ELS command <elsCmd> received from NPORT <did> */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0115,
				lpfc_mes0115,
				lpfc_msgBlk0115.msgPreambleStr, cmd, did);
		break;
	}
reject:
	/* Check if the unsol ELS command should get rejected. */
	if (rej_unsol_els_cmd) {
		stat.un.b.lsRjtRsvd0 = 0;
		stat.un.b.lsRjtRsnCode = rjt_reason;
		stat.un.b.lsRjtRsnCodeExp = LSEXP_NOTHING_MORE;
		stat.un.b.vendorUnique = 0;
		lpfc_els_rsp_reject(vport, stat.un.lsRjtError, elsiocb, ndlp,
				    NULL);

		if (newnode) {
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0934,
				lpfc_mes0934, lpfc_msgBlk0934.msgPreambleStr,
				__func__, ndlp->nlp_DID, vport->vpi);
			lpfc_nlp_free(phba, ndlp);
		}
	}
	return;

dropit:
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0111,
			lpfc_mes0111,
			lpfc_msgBlk0111.msgPreambleStr,
			icmd->ulpStatus, icmd->un.ulpWord[4]);
	phba->fc_stat.elsRcvDrop++;
}

static struct lpfc_vport *
lpfc_find_vport_by_vpid(lpfcHBA_t *phba, uint16_t vpi)
{
	struct lpfc_vport *vport;

	list_for_each_entry(vport, &phba->port_list, listentry) {
		if (vport->vpi == vpi)
			return vport;
	}
	return NULL;
}

void
lpfc_els_unsol_event(lpfcHBA_t * phba,
		     LPFC_SLI_RING_t * pring, LPFC_IOCBQ_t * elsiocb)
{
	DMABUF_t *mp = NULL;
	dma_addr_t paddr;
	IOCB_t *icmd = &elsiocb->iocb;
	lpfc_vport_t *vport = phba->pport;
	DMABUF_t *bdeBuf1 = elsiocb->context2;
	DMABUF_t *bdeBuf2 = elsiocb->context3;
	
	elsiocb->context2 = NULL;
	elsiocb->context3 = NULL;

	if (phba->max_vpi &&
	    (icmd->ulpCommand == CMD_IOCB_RCV_ELS64_CX ||
	     icmd->ulpCommand == CMD_IOCB_RCV_SEQ64_CX)) {
		if (icmd->unsli3.rcvsli3.vpi == 0xffff)
			vport = phba->pport;
		else {
			uint16_t vpi = icmd->unsli3.rcvsli3.vpi;
			vport = lpfc_find_vport_by_vpid(phba, vpi);
		}
	}
	if (icmd->ulpStatus == IOSTAT_NEED_BUFFER) {
		lpfc_sli_hbqbuf_fill_hbqs(phba, ELS_HBQ, 5);
	}
	/*
	 * If there are no BDEs associated
	 * with this IOCB, there is nothing to do.
	 */
	if (icmd->ulpBdeCount == 0)
		return;

	/* type of ELS cmd is first 32bit word in packet */
	if (phba->sli3_options & LPFC_SLI3_HBQ_ENABLED) {
		elsiocb->context2 = bdeBuf1;
		lpfc_els_unsol_buffer(phba, pring, vport, elsiocb);
	} else {
		paddr = getPaddr(icmd->un.cont64[0].addrHigh,
				 icmd->un.cont64[0].addrLow);
		mp = lpfc_sli_ringpostbuf_get(phba, pring, paddr);
		elsiocb->context2 = mp;
		lpfc_els_unsol_buffer(phba, pring, vport, elsiocb);
	}

	
	if (elsiocb->context2) {
		if (phba->sli3_options & LPFC_SLI3_HBQ_ENABLED)
			lpfc_hbq_free(phba, ((DMABUF_t *)elsiocb->context2)->virt, 
				       ((DMABUF_t *)elsiocb->context2)->phys);
		else {
			lpfc_mbuf_free(phba, ((DMABUF_t *)elsiocb->context2)->virt, 
				       ((DMABUF_t *)elsiocb->context2)->phys);
		}
		kfree(elsiocb->context2);
		elsiocb->context2 = NULL;
	}

	/* RCV_ELS64_CX provide for 2 BDEs - process 2nd if included */
	if ((phba->sli3_options & LPFC_SLI3_HBQ_ENABLED) &&
	    icmd->ulpBdeCount == 2) {
		elsiocb->context2 = bdeBuf2;
		lpfc_els_unsol_buffer(phba, pring, vport, elsiocb);
		/* free mp if we are done with it */
		if (elsiocb->context2) {
			lpfc_hbq_free(phba, ((DMABUF_t *)elsiocb->context2)->virt, 
				      ((DMABUF_t *)elsiocb->context2)->phys);
			kfree(elsiocb->context2);
			elsiocb->context2 = NULL;

		}
	}

	return;
}

void
lpfc_more_adisc(lpfc_vport_t *vport)
{
	int sentadisc;
	LPFC_NODELIST_t *ndlp;
	struct list_head *pos;
	lpfcHBA_t *phba = vport->phba;

	if (vport->num_disc_nodes)
		vport->num_disc_nodes--;

	/* Continue discovery with <num_disc_nodes> ADISCs to go */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0210,
			lpfc_mes0210,
			lpfc_msgBlk0210.msgPreambleStr,
			vport->num_disc_nodes, vport->fc_adisc_cnt, vport->fc_flag,
			vport->port_state);

	/* Check to see if there are more ADISCs to be sent */
	if (vport->fc_flag & FC_NLP_MORE) {
		sentadisc = 0;
		lpfc_set_disctmo(vport);

		/* go thru ADISC list and issue any remaining ELS ADISCs */
		list_for_each(pos, &vport->fc_adisc_list) {
			ndlp = list_entry(pos, LPFC_NODELIST_t, nlp_listp);
			if (!(ndlp->nlp_flag & NLP_ADISC_SND)) {
				/* If we haven't already sent an ADISC for this
				 * node, send it.
				 */
				lpfc_issue_els_adisc(vport, ndlp, 0);
				vport->num_disc_nodes++;
				ndlp->nlp_flag |= NLP_DISC_NODE;
				sentadisc = 1;
				break;
			}
		}

		if (sentadisc == 0) {
			vport->fc_flag &= ~FC_NLP_MORE;
		}
	}
	return;
}

void
lpfc_more_plogi(lpfc_vport_t * vport)
{
	int sentplogi;
	LPFC_NODELIST_t *ndlp;
	struct list_head *pos;
	lpfcHBA_t *phba = vport->phba;

	if (vport->num_disc_nodes)
		vport->num_disc_nodes--;

	/* Continue discovery with <num_disc_nodes> PLOGIs to go */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0232,
			lpfc_mes0232,
			lpfc_msgBlk0232.msgPreambleStr,
			vport->num_disc_nodes, vport->fc_plogi_cnt, vport->fc_flag,
			vport->port_state);

	/* Check to see if there are more PLOGIs to be sent */
	if (vport->fc_flag & FC_NLP_MORE) {
		sentplogi = 0;

		/* go thru PLOGI list and issue any remaining ELS PLOGIs */
		list_for_each(pos, &vport->fc_plogi_list) {
			ndlp = list_entry(pos, LPFC_NODELIST_t, nlp_listp);
			if ((ndlp->nlp_DID & Fabric_DID_MASK) == Fabric_DID_MASK)
				continue;
			if ((!(ndlp->nlp_flag & NLP_PLOGI_SND)) &&
			    (ndlp->nlp_state == NLP_STE_UNUSED_NODE) &&
			    (lpfc_node_teardown(ndlp)==0)) {
				/* If we haven't already sent an PLOGI for this
				 * node, send it.
				 */
				ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
				lpfc_issue_els_plogi(vport, ndlp, 0);
				vport->num_disc_nodes++;
				ndlp->nlp_flag |= NLP_DISC_NODE;
				sentplogi = 1;
				break;
			}
		}
		if (sentplogi == 0) {
			vport->fc_flag &= ~FC_NLP_MORE;
		}
	}
	return;
}

void
lpfc_els_flush_rscn(lpfc_vport_t * vport)
{
	DMABUF_t *mp;
	int i;
	lpfcHBA_t *phba = vport->phba;

	for (i = 0; i < vport->fc_rscn_id_cnt; i++) {
		mp = vport->fc_rscn_id_list[i];
		if (phba->sli3_options & LPFC_SLI3_HBQ_ENABLED) {
			lpfc_hbq_free(phba, mp->virt, mp->phys);
		} else {
			lpfc_mbuf_free(phba, mp->virt, mp->phys);
		}
		vport->fc_rscn_id_list[i] = NULL;
		kfree(mp);
	}
	vport->fc_rscn_id_cnt = 0;
	vport->fc_flag &= ~(FC_RSCN_MODE | FC_RSCN_DISCOVERY);
	lpfc_can_disctmo(vport);
}

int
lpfc_rscn_payload_check(lpfc_vport_t * vport, uint32_t did)
{
	D_ID ns_did;
	D_ID rscn_did;
	uint32_t *lp;
	uint32_t payload_len, cmd, i, match;
	lpfcHBA_t *phba = vport->phba;

	ns_did.un.word = did;
	match = 0;

	/* If we are doing a FULL RSCN rediscovery, match everything */
	if (!(vport->fc_flag & FC_RSCN_MODE) || 
	    (vport->fc_flag & FC_RSCN_DISCOVERY)) {
		return (did);
	}

	for (i = 0; i < vport->fc_rscn_id_cnt; i++) {
		lp = vport->fc_rscn_id_list[i]->virt;
		cmd = *lp++;
		payload_len = be32_to_cpu(cmd & ~ELS_CMD_MASK); /* payload len*/
		payload_len -= sizeof (uint32_t);	/* take off word 0 */
		while (payload_len) {
			rscn_did.un.word = *lp++;
			rscn_did.un.word = be32_to_cpu(rscn_did.un.word);
			payload_len -= sizeof (uint32_t);
			switch (rscn_did.un.b.resv) {
			case 0:	/* Single N_Port ID effected */
				if (ns_did.un.word == rscn_did.un.word) {
					match = did;
				}
				break;
			case 1:	/* Whole N_Port Area effected */
				if ((ns_did.un.b.domain == rscn_did.un.b.domain)
				    && (ns_did.un.b.area == rscn_did.un.b.area))
					{
						match = did;
					}
				break;
			case 2:	/* Whole N_Port Domain effected */
				if (ns_did.un.b.domain == rscn_did.un.b.domain)
					{
						match = did;
					}
				break;
			case 3:	/* Whole Fabric effected */
				match = did;
				break;
			default:
				/* Unknown Identifier in RSCN list */
				lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0217,
						lpfc_mes0217,
						lpfc_msgBlk0217.msgPreambleStr,
						rscn_did.un.word);
				break;
			}
			if (match) {
				break;
			}
		}
	}
	return (match);
}


static int
lpfc_rscn_recovery_check(lpfc_vport_t * vport)
{
	LPFC_NODELIST_t *ndlp = NULL, *new_ndlp;
	struct list_head *pos, *next, *listp;
	struct list_head *node_list[4];
	LPFCSCSITARGET_t *targetp;
	int i;

	/* Make all effected nodes LPFC_DEV_DISCONNECTED */
	node_list[0] = &vport->fc_plogi_list;
	node_list[1] = &vport->fc_adisc_list;
	node_list[2] = &vport->fc_nlpunmap_list;
	node_list[3] = &vport->fc_nlpmap_list;
	for (i = 0; i < 4; i++) {
		listp = node_list[i];
		if (list_empty(listp))
			continue;

		list_for_each_safe(pos, next, listp) {
			new_ndlp = list_entry(pos, LPFC_NODELIST_t, nlp_listp);
			ndlp = new_ndlp;

			if ((lpfc_rscn_payload_check(vport, ndlp->nlp_DID))) {
				lpfc_set_failmask(vport, ndlp,
						  LPFC_DEV_DISCONNECTED,
						  LPFC_SET_BITMASK);

				targetp = ndlp->nlp_Target;
				if(targetp)
					lpfc_set_npr_tmo(vport, targetp, ndlp);
			}
		}
	}
	return (0);
}

int
lpfc_els_rcv_rscn(lpfc_vport_t *vport,  LPFC_IOCBQ_t *cmdiocb,
		  LPFC_NODELIST_t * ndlp, uint8_t newnode)
{
	DMABUF_t *pCmd;
	uint32_t *lp;
	IOCB_t *icmd;
	uint32_t payload_len, *cmd, length;
	
	icmd = &cmdiocb->iocb;
	pCmd = (DMABUF_t *) cmdiocb->context2;
	lp = (uint32_t *) pCmd->virt;
	lpfcHBA_t *phba = vport->phba;

	payload_len = be32_to_cpu(*lp++ & ~ELS_CMD_MASK); /* payload length */
	payload_len -= sizeof (uint32_t);                 /* take off word 0 */

	/* RSCN received */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0214,
			lpfc_mes0214,
			lpfc_msgBlk0214.msgPreambleStr,
			vport->fc_flag, payload_len, *lp, vport->fc_rscn_id_cnt);

	/* if we are already processing an RSCN, save the received
	 * RSCN payload buffer, cmdiocb->context2 to process later.
	 * If we zero, cmdiocb->context2, the calling routine will
	 * not try to free it.
	 */
	if (vport->fc_flag & FC_RSCN_MODE) {
		if ((vport->fc_rscn_id_cnt < FC_MAX_HOLD_RSCN) &&
		    !(vport->fc_flag & FC_RSCN_DISCOVERY)) {
			if (vport->fc_rscn_id_cnt) {
				cmd = vport->fc_rscn_id_list[vport->fc_rscn_id_cnt - 1]->virt;
				length = be32_to_cpu(*cmd & ~ELS_CMD_MASK);
			}
			if ((vport->fc_rscn_id_cnt) &&
			    (payload_len + length <= LPFC_BPL_SIZE)) {
				*cmd &= ELS_CMD_MASK;
				*cmd |= be32_to_cpu(payload_len + length);
				memcpy(((uint8_t *)cmd) + length, 
				       lp, payload_len);
			} else {
				vport->fc_rscn_id_list[vport->fc_rscn_id_cnt++] = pCmd;
				cmdiocb->context2 = NULL;
			}
			/* Deferred RSCN */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0235,
					lpfc_mes0235,
					lpfc_msgBlk0235.msgPreambleStr,
					vport->fc_rscn_id_cnt, vport->fc_flag,
					vport->port_state);
		} else {
			vport->fc_flag |= FC_RSCN_DISCOVERY;
			/* ReDiscovery RSCN */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0234,
					lpfc_mes0234,
					lpfc_msgBlk0234.msgPreambleStr,
					vport->fc_rscn_id_cnt, vport->fc_flag,
					vport->port_state);
		}
		/* Send back ACC */
		lpfc_els_rsp_acc(vport, ELS_CMD_ACC, cmdiocb, ndlp, 0, newnode);
		lpfc_rscn_recovery_check(vport);
		return (0);
	}

	vport->fc_flag |= FC_RSCN_MODE;
	vport->fc_rscn_id_list[vport->fc_rscn_id_cnt++] = pCmd;
	/*
	 * If we zero, cmdiocb->context2, the calling routine will
	 * not try to free it.
	 */
	cmdiocb->context2 = 0;

	lpfc_set_disctmo(vport);

	/* Send back ACC */
	lpfc_els_rsp_acc(vport, ELS_CMD_ACC, cmdiocb, ndlp, 0, newnode);
	lpfc_rscn_recovery_check(vport);

	return (lpfc_els_handle_rscn(vport));
}

int
lpfc_els_handle_rscn(lpfc_vport_t * vport)
{
	LPFC_NODELIST_t *ndlp;
	lpfcHBA_t *phba = vport->phba;

	lpfc_hba_put_event(phba, HBA_EVENT_RSCN, vport->fc_myDID,
			  vport->fc_myDID, 0, 0);
	lpfc_put_event(phba, FC_REG_RSCN_EVENT, vport->fc_myDID, 0, 0);

	/* RSCN processed */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0215,
			lpfc_mes0215,
			lpfc_msgBlk0215.msgPreambleStr,
			vport->fc_flag, 0, vport->fc_rscn_id_cnt,
			vport->port_state);

	/* To process RSCN, first compare RSCN data with NameServer */
	vport->fc_ns_retry = 0;
	if ((ndlp = lpfc_findnode_did(vport, NLP_SEARCH_UNMAPPED,
				      NameServer_DID))) {
		/* Good ndlp, issue CT Request to NameServer */
		if (lpfc_ns_cmd(vport, SLI_CTNS_GID_FT, 0) == 0) {
			/* Wait for NameServer query cmpl before we can
			   continue */
			return (1);
		}
	} else {
		/* If login to NameServer does not exist, issue one */
		/* Good status, issue PLOGI to NameServer */
		if ((ndlp =
		     lpfc_findnode_did(vport, NLP_SEARCH_ALL, NameServer_DID))) {
			/* Wait for NameServer login cmpl before we can
			   continue */
			return (1);
		}
		if ((ndlp = lpfc_nlp_alloc(phba, 0)) == 0) {
			lpfc_els_flush_rscn(vport);
			return (0);
		} else {
			memset(ndlp, 0, sizeof (LPFC_NODELIST_t));
			ndlp->nlp_type |= NLP_FABRIC;
			ndlp->nlp_DID = NameServer_DID;

			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0933,
				lpfc_mes0933, lpfc_msgBlk0933.msgPreambleStr,
				__func__, ndlp->nlp_DID, vport->vpi);

			ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
			lpfc_issue_els_plogi(vport, ndlp, 0);
			/* Wait for NameServer login cmpl before we can
			   continue */
			return (1);
		}
	}

	lpfc_els_flush_rscn(vport);
	return (0);
}

int
lpfc_els_rcv_flogi(lpfc_vport_t * vport,
		   LPFC_IOCBQ_t * cmdiocb,
		   LPFC_NODELIST_t * ndlp, uint8_t newnode)
{
	DMABUF_t *pCmd;
	uint32_t *lp;
	IOCB_t *icmd;
	SERV_PARM *sp;
	LPFC_MBOXQ_t *mbox;
	LPFC_SLI_t *psli;
	lpfcCfgParam_t *clp;
	LS_RJT stat;
	uint32_t cmd, did;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	clp = &phba->config[0];
	icmd = &cmdiocb->iocb;
	pCmd = (DMABUF_t *) cmdiocb->context2;
	lp = (uint32_t *) pCmd->virt;

	cmd = *lp++;
	sp = (SERV_PARM *) lp;

	/* FLOGI received */

	lpfc_set_disctmo(vport);

	if (phba->fc_topology == TOPOLOGY_LOOP) {
		/* We should never receive a FLOGI in loop mode, ignore it */
		did = icmd->un.elsreq64.remoteID;

		/* An FLOGI ELS command <elsCmd> was received from DID <did> in
		   Loop Mode */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0113,
				lpfc_mes0113,
				lpfc_msgBlk0113.msgPreambleStr, cmd, did);
		return 1;
	}

	did = Fabric_DID;

	if ((lpfc_check_sparm(vport, ndlp, sp, CLASS3))) {
		/* For a FLOGI we accept, then if our portname is greater
		 * then the remote portname we initiate Nport login. 
		 */
		int rc;

		rc = lpfc_geportname((NAME_TYPE *) & vport->fc_portname,
				     (NAME_TYPE *) & sp->portName);

		if (rc == 2) {
			mbox = lpfc_mbox_alloc(phba, 0);
			if (mbox == NULL)
				return 1;

			lpfc_linkdown(phba);
			lpfc_init_link(phba, mbox,
				       clp[LPFC_CFG_TOPOLOGY].a_current,
				       clp[LPFC_CFG_LINK_SPEED].a_current);
			mbox->mb.un.varInitLnk.lipsr_AL_PA = 0;
			mbox->vport = vport;
			if (lpfc_sli_issue_mbox
			    (phba, mbox, (MBX_NOWAIT | MBX_STOP_IOCB))
			    == MBX_NOT_FINISHED) {
				lpfc_mbox_free(phba, mbox);
			}
			return 1;
		}

		if (rc == 1) {	/* greater than */
			vport->fc_flag |= FC_PT2PT_PLOGI;
		}
		vport->fc_flag |= FC_PT2PT;
		vport->fc_flag &= ~(FC_FABRIC | FC_PUBLIC_LOOP);
	} else {
		/* Reject this request because invalid parameters */
		stat.un.b.lsRjtRsvd0 = 0;
		stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
		stat.un.b.lsRjtRsnCodeExp = LSEXP_SPARM_OPTIONS;
		stat.un.b.vendorUnique = 0;
		lpfc_els_rsp_reject(vport, stat.un.lsRjtError, cmdiocb, ndlp,
				    NULL);
		return 1;
	}

	/* Send back ACC */
	lpfc_els_rsp_acc(vport, ELS_CMD_PLOGI, cmdiocb, ndlp, 0, newnode);

	return 0;
}

int
lpfc_els_rcv_rnid(lpfc_vport_t *vport, LPFC_IOCBQ_t * cmdiocb,
		  LPFC_NODELIST_t * ndlp)
{
	DMABUF_t *pCmd;
	uint32_t *lp;
	IOCB_t *icmd;
	RNID *rn;
	LS_RJT stat;
	uint32_t cmd, did;

	icmd = &cmdiocb->iocb;
	did = icmd->un.elsreq64.remoteID;
	pCmd = (DMABUF_t *) cmdiocb->context2;
	lp = (uint32_t *) pCmd->virt;

	cmd = *lp++;
	rn = (RNID *) lp;

	/* RNID received */

	switch (rn->Format) {
	case 0:
	case RNID_TOPOLOGY_DISC:
		/* Send back ACC */
		lpfc_els_rsp_rnid_acc(vport, rn->Format, cmdiocb, ndlp);
		break;
	default:
		/* Reject this request because format not supported */
		stat.un.b.lsRjtRsvd0 = 0;
		stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
		stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
		stat.un.b.vendorUnique = 0;
		lpfc_els_rsp_reject(vport, stat.un.lsRjtError, cmdiocb, ndlp,
				    NULL);
	}
	return (0);
}

int
lpfc_els_rcv_rrq(lpfc_vport_t *vport, LPFC_IOCBQ_t *cmdiocb,
		 LPFC_NODELIST_t *ndlp)
{
	DMABUF_t *pCmd;
	uint32_t *lp;
	IOCB_t *icmd;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	RRQ *rrq;
	uint32_t cmd, did;
	lpfcHBA_t *phba = vport->phba;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_FCP_RING];
	icmd = &cmdiocb->iocb;
	did = icmd->un.elsreq64.remoteID;
	pCmd = (DMABUF_t *) cmdiocb->context2;
	lp = (uint32_t *) pCmd->virt;

	cmd = *lp++;
	rrq = (RRQ *) lp;

	/* RRQ received */
	/* Get oxid / rxid from payload and abort it */
	if ((rrq->SID == be32_to_cpu(vport->fc_myDID))) {
		lpfc_sli_abort_iocb_ctx(phba, pring, rrq->Oxid);
	} else {
		lpfc_sli_abort_iocb_ctx(phba, pring, rrq->Rxid);
	}
	/* ACCEPT the rrq request */
	lpfc_els_rsp_acc(vport, ELS_CMD_ACC, cmdiocb, ndlp, 0, 0);

	return (0);
}

int
lpfc_els_rcv_farp(lpfc_vport_t * vport,
		  LPFC_IOCBQ_t * cmdiocb, LPFC_NODELIST_t * ndlp)
{
	DMABUF_t *pCmd;
	uint32_t *lp;
	IOCB_t *icmd;
	FARP *fp;
	uint32_t cmd, cnt, did;
	lpfcHBA_t *phba = vport->phba;

	icmd = &cmdiocb->iocb;
	did = icmd->un.elsreq64.remoteID;
	pCmd = (DMABUF_t *) cmdiocb->context2;
	lp = (uint32_t *) pCmd->virt;

	cmd = *lp++;
	fp = (FARP *) lp;

	/* FARP-REQ received from DID <did> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0601,
			lpfc_mes0601, lpfc_msgBlk0601.msgPreambleStr, did);

	/* We will only support match on WWPN or WWNN */
	if (fp->Mflags & ~(FARP_MATCH_NODE | FARP_MATCH_PORT)) {
		return (0);
	}

	cnt = 0;
	/* If this FARP command is searching for my portname */
	if (fp->Mflags & FARP_MATCH_PORT) {
		if (lpfc_geportname(&fp->RportName, &vport->fc_portname) == 2)
			cnt = 1;
	}

	/* If this FARP command is searching for my nodename */
	if (fp->Mflags & FARP_MATCH_NODE) {
		if (lpfc_geportname(&fp->RnodeName, &vport->fc_nodename) == 2)
			cnt = 1;
	}

	if (cnt) {
		if ((ndlp->nlp_failMask == 0) &&
		    (!(ndlp->nlp_flag & NLP_ELS_SND_MASK))) {
			/* Log back into the node before sending the FARP. */
			if (fp->Rflags & FARP_REQUEST_PLOGI) {
				lpfc_nlp_plogi(vport, ndlp);
				ndlp->nlp_state = NLP_STE_PLOGI_ISSUE;
				lpfc_issue_els_plogi(vport, ndlp, 0);
			}

			/* Send a FARP response to that node */
			if (fp->Rflags & FARP_REQUEST_FARPR) {
				lpfc_issue_els_farpr(vport, did, 0);
			}
		}
	}
	return (0);
}

int
lpfc_els_rcv_farpr(lpfc_vport_t *vport, LPFC_IOCBQ_t *cmdiocb,
		   LPFC_NODELIST_t * ndlp)
{
	DMABUF_t *pCmd;
	uint32_t *lp;
	IOCB_t *icmd;
	uint32_t cmd, did;
	lpfcHBA_t *phba = vport->phba;

	icmd = &cmdiocb->iocb;
	did = icmd->un.elsreq64.remoteID;
	pCmd = (DMABUF_t *) cmdiocb->context2;
	lp = (uint32_t *) pCmd->virt;

	cmd = *lp++;
	/* FARP-RSP received from DID <did> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0600,
			lpfc_mes0600, lpfc_msgBlk0600.msgPreambleStr, did);

	/* ACCEPT the Farp resp request */
	lpfc_els_rsp_acc(vport, ELS_CMD_ACC, cmdiocb, ndlp, 0, 0);

	return (0);
}

int
lpfc_els_rcv_fan(lpfc_vport_t *vport, LPFC_IOCBQ_t *cmdiocb,
		  LPFC_NODELIST_t * ndlp)
{
	lpfcHBA_t *phba = vport->phba;
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0261,
			lpfc_mes0261,
			lpfc_msgBlk0261.msgPreambleStr);
	return (0);
}

int
lpfc_els_chk_latt(lpfcHBA_t * phba, LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;
	LPFC_MBOXQ_t *mbox;
	uint32_t ha_copy;
	
	if ((phba->pport->port_state < LPFC_HBA_READY) &&
		(phba->pport->port_state != LPFC_LINK_DOWN)) {
		uint32_t tag, stat, wd4;

		/* Read the HBA Host Attention Register */
		ha_copy = readl(phba->HAregaddr);

		if (ha_copy & HA_LATT) {	/* Link Attention interrupt */
			if (rspiocb) {
				irsp = &(rspiocb->iocb);
				tag = irsp->ulpIoTag;
				stat = irsp->ulpStatus;
				wd4 = irsp->un.ulpWord[4];
				irsp->ulpStatus = IOSTAT_LOCAL_REJECT;
				irsp->un.ulpWord[4] = IOERR_SLI_ABORTED;
			} else {
				tag = 0;
				stat = 0;
				wd4 = 0;
			}
			/* Pending Link Event during Discovery */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0237,
					lpfc_mes0237,
					lpfc_msgBlk0237.msgPreambleStr,
					phba->pport->port_state, tag, stat, wd4);

			lpfc_linkdown(phba);

			if (phba->pport->port_state != LPFC_CLEAR_LA) {
				mbox = lpfc_mbox_alloc(phba, MEM_PRI);
				if (mbox != NULL) {
					phba->pport->port_state = LPFC_CLEAR_LA;
					lpfc_clear_la(phba, mbox);
					mbox->mbox_cmpl =
					    lpfc_mbx_cmpl_clear_la;
					mbox->vport = phba->pport;
					if (lpfc_sli_issue_mbox
					    (phba, mbox,
					     (MBX_NOWAIT | MBX_STOP_IOCB))
					    == MBX_NOT_FINISHED) {
						lpfc_mbox_free(phba, mbox);
						phba->hba_state =
						    LPFC_HBA_ERROR;
					}
				}
			}
			return 1;
		}
	}

	return 0;
}

void
lpfc_els_timeout_handler(unsigned long ptr)
{
	lpfcHBA_t *phba;
	lpfc_vport_t *vport;
	LPFC_SLI_RING_t *pring;
	LPFC_IOCBQ_t *piocb;
	LPFC_NODELIST_t *ndlp;
	IOCB_t *cmd = 0;
	DMABUF_t *pCmd;
	uint32_t els_command;
	uint32_t timeout;
	uint32_t next_timeout;
	uint32_t remote_ID;
	unsigned long iflag;
	struct clk_data *elsClkData;
	struct list_head *curr, *next;

	elsClkData = (struct clk_data *)ptr;
	phba = elsClkData->phba;
	LPFC_DRVR_LOCK(phba, iflag);
       	if (elsClkData->flags & TM_CANCELED) {
		list_del((struct list_head *)elsClkData);
		kfree(elsClkData);	
		goto out;
	}

	timeout = (uint32_t) (unsigned long)(elsClkData->clData1);
	phba->els_tmofunc.function = 0;
	list_del((struct list_head *)elsClkData);
	kfree(elsClkData);

	pring = &phba->sli.ring[LPFC_ELS_RING];
	next_timeout = phba->fc_ratov << 1;
	els_command = ELS_CMD_MASK;

	list_for_each_safe(curr, next, &pring->txcmplq) {
		piocb = list_entry(curr, LPFC_IOCBQ_t, list);
		cmd = &piocb->iocb;

		if (piocb->iocb_flag & LPFC_IO_LIBDFC)
			continue;

		pCmd = (DMABUF_t *) piocb->context2;
		if (pCmd)
			els_command = *((uint32_t *) (pCmd->virt));

		if ((els_command == ELS_CMD_FARP) ||
		    (els_command == ELS_CMD_FARPR))
			continue;

		if (piocb->drvrTimeout > 0) {
			if (piocb->drvrTimeout >= timeout)
				piocb->drvrTimeout -= timeout;
			else
				piocb->drvrTimeout = 0;
			continue;
		}

		vport = piocb->vport;
		list_del(&piocb->list);
		pring->txcmplq_cnt--;

		if (cmd->ulpCommand == CMD_GEN_REQUEST64_CR) {
			ndlp = lpfc_findnode_rpi(vport, cmd->ulpContext);
			remote_ID = ndlp->nlp_DID;
		} else {
			remote_ID = cmd->un.elsreq64.remoteID;
		}

		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0127, lpfc_mes0127,
				lpfc_msgBlk0127.msgPreambleStr, els_command,
				remote_ID, cmd->ulpCommand, cmd->ulpIoTag);

		/*
		 * The iocb has timed out.  Presume the FW has lost the command
		 * and just complete it.
		 */
		cmd->ulpStatus = IOSTAT_LOCAL_REJECT;
		cmd->un.ulpWord[4] = IOERR_SLI_ABORTED;

		if ((els_command == ELS_CMD_PLOGI) ||
		    (els_command == ELS_CMD_PRLI))
			cmd->un.ulpWord[4] = IOERR_ABORT_REQUESTED;

		if (piocb->iocb_cmpl)
			(piocb->iocb_cmpl) (phba, piocb, piocb);
		else
			lpfc_iocb_free(phba, piocb);
	}

	lpfc_start_timer(phba, next_timeout, &phba->els_tmofunc,
			 lpfc_els_timeout_handler,
			 (unsigned long) next_timeout, (unsigned long) 0, (unsigned long) 0);
out:
	LPFC_DRVR_UNLOCK(phba, iflag);
}

void
lpfc_els_flush_cmd(lpfc_vport_t * vport)
{
	LPFC_SLI_RING_t *pring;
	LPFC_IOCBQ_t *next_iocb;
	LPFC_IOCBQ_t *piocb;
	IOCB_t *cmd = 0;
	DMABUF_t *pCmd;
	uint32_t *elscmd;
	uint32_t els_command;
	uint32_t remote_ID;
	struct list_head *curr, *next;
	lpfcHBA_t *phba = vport->phba;

	lpfc_fabric_abort_vport(vport);

	pring = &phba->sli.ring[LPFC_ELS_RING];

	list_for_each_safe(curr, next, &pring->txq) {
		next_iocb = list_entry(curr, LPFC_IOCBQ_t, list);
		piocb = next_iocb;
		cmd = &piocb->iocb;

		if (piocb->iocb_flag & LPFC_IO_LIBDFC) {
			continue;
		}

		/* Do not flush out the QUE_RING and ABORT/CLOSE iocbs */
		if ((cmd->ulpCommand == CMD_QUE_RING_BUF_CN) ||
		    (cmd->ulpCommand == CMD_QUE_RING_BUF64_CN) ||
		    (cmd->ulpCommand == CMD_CLOSE_XRI_CN) ||
		    (cmd->ulpCommand == CMD_ABORT_XRI_CN)) {
			continue;
		}

		pCmd = (DMABUF_t *) piocb->context2;
		elscmd = (uint32_t *) (pCmd->virt);
		els_command = *elscmd;

		if (piocb->vport != vport)
			continue;

		if (cmd->ulpCommand == CMD_GEN_REQUEST64_CR) {
			LPFC_NODELIST_t *ndlp;

			ndlp = lpfc_findnode_rpi(vport, cmd->ulpContext);
			remote_ID = ndlp->nlp_DID;
			if (vport->port_state == LPFC_HBA_READY) {
				continue;
			}
		} else {
			remote_ID = cmd->un.elsreq64.remoteID;
		}

		list_del(&piocb->list);
		pring->txq_cnt--;

		cmd->ulpStatus = IOSTAT_LOCAL_REJECT;
		cmd->un.ulpWord[4] = IOERR_SLI_ABORTED;

		if (piocb->iocb_cmpl) {
			(piocb->iocb_cmpl) (phba, piocb, piocb);
		} else {
			lpfc_iocb_free(phba, piocb);
		}
	}

	list_for_each_safe(curr, next, &pring->txcmplq) {
		next_iocb = list_entry(curr, LPFC_IOCBQ_t, list);
		piocb = next_iocb;
		cmd = &piocb->iocb;

		if (piocb->iocb_flag & LPFC_IO_LIBDFC) {
			continue;
		}
		pCmd = (DMABUF_t *) piocb->context2;
		elscmd = (uint32_t *) (pCmd->virt);
		els_command = *elscmd;

		if (cmd->ulpCommand == CMD_GEN_REQUEST64_CR) {
			LPFC_NODELIST_t *ndlp;

			ndlp = lpfc_findnode_rpi(vport, cmd->ulpContext);
			remote_ID = ndlp->nlp_DID;
			if (vport->port_state == LPFC_HBA_READY) {
				continue;
			}
		} else {
			remote_ID = cmd->un.elsreq64.remoteID;
		}

		list_del(&piocb->list);
		pring->txcmplq_cnt--;

		cmd->ulpStatus = IOSTAT_LOCAL_REJECT;
		cmd->un.ulpWord[4] = IOERR_SLI_ABORTED;

		if (piocb->iocb_cmpl) {
			(piocb->iocb_cmpl) (phba, piocb, piocb);
		} else {
			lpfc_iocb_free(phba, piocb);
		}
	}
	return;
}
void
lpfc_cmpl_els_fdisc(lpfcHBA_t * phba,
		    LPFC_IOCBQ_t * cmdiocb, LPFC_IOCBQ_t * rspiocb)
{
	lpfc_vport_t *vport = cmdiocb->vport;
	IOCB_t *irsp;
	DMABUF_t *pCmd;
	LPFC_MBOXQ_t *mbox;
	LPFC_SLI_t *psli;
	LPFC_NODELIST_t *ndlp;
	lpfcCfgParam_t *clp;
	uint32_t rc;

	psli = &phba->sli;
	irsp = &(rspiocb->iocb);
	ndlp = (LPFC_NODELIST_t *) cmdiocb->context1;
	pCmd = (DMABUF_t *) cmdiocb->context2;

	clp = &phba->config[0];
	if (irsp->ulpStatus) {
		if (lpfc_els_retry(phba, cmdiocb, rspiocb))
			goto out;
		vport->port_state = LPFC_FDISC_ERR;
		goto fdisc_failed;
	} else {
		vport->fc_flag |= FC_FABRIC;
		if (vport->phba->fc_topology == TOPOLOGY_LOOP)
			vport->fc_flag |= FC_PUBLIC_LOOP;

		vport->fc_myDID = irsp->un.ulpWord[4] & Mask_DID;
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0129,
				lpfc_mes0129,
				lpfc_msgBlk0129.msgPreambleStr,
				vport->fc_myDID);

		/*
		 * If the NportID has change, unregister the
		 * previous vpi<->DID binding before registering
		 * the new one.  Note that the prevDID is
		 * initialized to Mask_DID so that the code can
		 * correctly detect the first FLOGI completion.
		 */
		if ((vport->fc_myDID != vport->fc_prevDID) &&
		    (vport->fc_prevDID != Mask_DID)) {
			/* Since the driver's Nport ID changed,
			 * all remaining remote Nports mapped
			 * to the previous ID must get their
			 * rpi binding unregistered as well.
			 */
			lpfc_unreg_rpi_cleanup(vport);
			mbox = lpfc_mbox_alloc(phba, MEM_PRI);
			if (mbox == NULL)
				goto fdisc_failed;
			lpfc_unreg_vpi(phba, vport->vpi, mbox);
			mbox->mbox_cmpl = lpfc_mbx_cmpl_unreg_vpi;
			mbox->vport = vport;
			rc = lpfc_sli_issue_mbox(phba, mbox,
						 (MBX_NOWAIT |
						  MBX_STOP_IOCB));
			if (rc == MBX_NOT_FINISHED) {
				lpfc_mbox_free(phba, mbox);
				goto fdisc_failed;
			}
		}
			
		mbox = lpfc_mbox_alloc(phba, 0);
		if (mbox == NULL)
			goto fdisc_failed;
		
		lpfc_reg_vpi(phba, vport->vpi, vport->fc_myDID, mbox);
		mbox->mbox_cmpl = lpfc_mbx_cmpl_reg_vpi;
		mbox->vport = vport;
		rc = lpfc_sli_issue_mbox(phba, mbox,
						(MBX_NOWAIT | MBX_STOP_IOCB));
		if (rc == MBX_NOT_FINISHED) {
			lpfc_mbox_free(phba, mbox);
			goto fdisc_failed;
		}

	}
	goto out;

fdisc_failed:
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0128,
			lpfc_mes0128,
			lpfc_msgBlk0128.msgPreambleStr);
out:
	lpfc_els_free_iocb(phba, cmdiocb);
	return;
}

int
lpfc_issue_els_fdisc(lpfc_vport_t * vport, LPFC_NODELIST_t * ndlp, int retry)
{
	lpfcHBA_t *phba = vport->phba;
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];	/* ELS ring */

	cmdsize = (sizeof (uint32_t) + sizeof (SERV_PARM));
	elsiocb = lpfc_prep_els_iocb(vport, TRUE, cmdsize, retry,
				     ndlp, ELS_CMD_FDISC);
	if (elsiocb == NULL)
		return 1;

	/*
	 * Clear the S_ID field and set SP to 1 so that the port takes the S_ID
	 * from word4. This approach maintains the driver's DID change 
	 * detection logic, FCP-2 support and leaves the els iocb prep clean.
	 */
	icmd = &elsiocb->iocb;
	icmd->un.elsreq64.myID = 0;
	icmd->un.elsreq64.fl = 1;
	icmd->ulpCt_h = 1;
	icmd->ulpCt_l = 0;

	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);
	*((uint32_t *) (pCmd)) = ELS_CMD_FDISC;

	pCmd += sizeof (uint32_t); /* CSP Word 1 */
	memcpy((char*)pCmd, (char*)&vport->phba->pport->fc_sparam,
					sizeof(SERV_PARM));

	pCmd += sizeof (uint32_t); /* CSP Word 2 */
	pCmd += sizeof (uint32_t); /* CSP Word 3 */
	pCmd += sizeof (uint32_t); /* CSP Word 4 */
	pCmd += sizeof (uint32_t); /* N_Port Name  */

	memcpy(pCmd, (char*)&vport->fc_portname, 8);
	pCmd += sizeof(uint32_t); /* Node Name */
	pCmd += sizeof(uint32_t); /* Node Name */
	memcpy(pCmd, (char*)&vport->fc_nodename, 8);	

	lpfc_set_disctmo(vport);

	vport->phba->fc_stat.elsXmitFDISC++;
	elsiocb->iocb_cmpl = lpfc_cmpl_els_fdisc;
	if (lpfc_issue_fabric_iocb(phba, elsiocb) == IOCB_ERROR) {
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}
	return 0;
}

#ifdef __VMKERNEL_MODULE__
void
lpfc_cmpl_els_fabric_logo(lpfcHBA_t * phba,
		    LPFC_IOCBQ_t * cmdiocb, LPFC_IOCBQ_t * rspiocb)
{
	IOCB_t *irsp;
	int rc = 0;
	LPFC_MBOXQ_t *mbx_did, *mbx_login;
	lpfc_vport_t *vport = cmdiocb->vport;

	irsp = &(rspiocb->iocb);

	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0131, lpfc_mes0131,
			lpfc_msgBlk0131.msgPreambleStr, vport->fc_myDID,
			vport->vpi, irsp->ulpStatus, irsp->un.ulpWord[4]);

	/* Call FW to clean up all default rpi's */
	mbx_did = lpfc_mbox_alloc(phba, MEM_PRI);
	if (mbx_did == NULL) {
		vport->vport_logo_state = LPFC_VPORT_LOGO_FAILED;
		return;
	}

	/* Cleanup all default dids opened by the FW. */
	lpfc_unreg_did(phba, vport->vpi, 0xffffffff, mbx_did);
	mbx_did->vport = vport;
	rc = lpfc_sli_issue_mbox(phba, mbx_did,
				(MBX_NOWAIT | MBX_STOP_IOCB));
	if (rc == MBX_NOT_FINISHED) {
		lpfc_mbox_free(phba, mbx_did);
		vport->vport_logo_state = LPFC_VPORT_LOGO_FAILED;
		goto out;
	}

	/*
	 * Cleanup all remote port rpis for this vport.  The implicit logo
	 * is useful here since remote initiators will not immediately
	 * reissue a PLOGI.
	 */
	mbx_login = lpfc_mbox_alloc(phba, MEM_PRI);	
	if (mbx_login == NULL) {
		vport->vport_logo_state = LPFC_VPORT_LOGO_FAILED;
		return;
	}

	lpfc_unreg_login(phba, vport->vpi, 0xffff, mbx_login);
	mbx_login->mbox_cmpl = lpfc_mbx_cmpl_unreg_login_all;
	mbx_login->vport = vport;
	rc = lpfc_sli_issue_mbox(phba, mbx_login, MBX_NOWAIT | MBX_STOP_IOCB);
	if (rc == MBX_NOT_FINISHED) {
		lpfc_mbox_free(phba, mbx_login);
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1803,
				lpfc_mes1803,
				lpfc_msgBlk1803.msgPreambleStr,
				vport->fc_myDID);
	}

out:
	lpfc_els_free_iocb(phba, cmdiocb);
}


int
lpfc_issue_els_fabric_logo(lpfc_vport_t *vport, LPFC_NODELIST_t * ndlp)
{
	lpfcHBA_t *phba = vport->phba;
	IOCB_t *icmd;
	LPFC_IOCBQ_t *elsiocb;
	LPFC_SLI_RING_t *pring;
	LPFC_SLI_t *psli;
	uint8_t *pCmd;
	uint16_t cmdsize;

	if (vport->fc_myDID == 0)
		return 1;

	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];

	/* A LOGO is 16 bytes, Command, DID, PortName. */
	cmdsize = (2 * sizeof (uint32_t)) + sizeof (NAME_TYPE); 
	elsiocb = lpfc_prep_els_iocb(vport, 1, cmdsize, 0, ndlp, ELS_CMD_LOGO);
	if (elsiocb == NULL)
		return 1;

	/* Fill in LOGO payload */
	icmd = &elsiocb->iocb;
	pCmd = (uint8_t *) (((DMABUF_t *) elsiocb->context2)->virt);

	/* First word is the command. */
	*((uint32_t *) (pCmd)) = ELS_CMD_LOGO;
	pCmd += sizeof (uint32_t);

	/* Second is the Nport DID. */
	*((uint32_t *) (pCmd)) = be32_to_cpu(vport->fc_myDID);
	pCmd += sizeof (uint32_t);

	/* Third and fourth are the portname. */
	memcpy(pCmd, &vport->fc_portname, sizeof (NAME_TYPE));

	elsiocb->iocb_cmpl = lpfc_cmpl_els_fabric_logo;
	ndlp->nlp_flag |= NLP_LOGO_SND;
	if (lpfc_sli_issue_iocb(phba,  pring, elsiocb, 0) == IOCB_ERROR) {
		ndlp->nlp_flag &= ~NLP_LOGO_SND;
		lpfc_els_free_iocb(phba, elsiocb);
		return 1;
	}

	/* Xmit ELS command <elsCmd> to remote NPORT <did> */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0130, lpfc_mes0130,
			lpfc_msgBlk0130.msgPreambleStr, vport->fc_myDID,
			vport->vpi, vport->port_state);
	return 0;
}
#endif

static void
lpfc_resume_fabric_iocbs(lpfcHBA_t *phba)
{
	LPFC_IOCBQ_t *iocb;
	int ret;
	LPFC_SLI_RING_t *pring = &phba->sli.ring[LPFC_ELS_RING];
	IOCB_t *cmd;

	iocb = NULL;
	/* Post any pending iocb to the SLI layer */
	if (atomic_read(&phba->fabric_iocb_count) == 0) {
		if (!list_empty(&phba->fabric_iocb_list)) {
		    iocb = list_entry(phba->fabric_iocb_list.next, 
				      LPFC_IOCBQ_t, list);
		    list_del_init(&iocb->list);
		}
	}
	if (iocb != NULL) {
		atomic_inc(&phba->fabric_iocb_count);
		iocb->fabric_iocb_cmpl = iocb->iocb_cmpl;
		iocb->iocb_cmpl = lpfc_cmpl_fabric_iocb;

		ret = lpfc_sli_issue_iocb(phba, pring, iocb, 0);

		if (ret == IOCB_ERROR) {
			iocb->iocb_cmpl = iocb->fabric_iocb_cmpl;
			iocb->fabric_iocb_cmpl = NULL;
			cmd = &iocb->iocb;
			cmd->ulpStatus = IOSTAT_LOCAL_REJECT;
			cmd->un.ulpWord[4] = IOERR_SLI_ABORTED;
			iocb->iocb_cmpl(phba, iocb, iocb);

			atomic_dec(&phba->fabric_iocb_count);
			lpfc_resume_fabric_iocbs(phba);
		}
	}

	return;
}

void
lpfc_unblock_fabric_iocbs(lpfcHBA_t *phba)
{
	clear_bit(FABRIC_COMANDS_BLOCKED, &phba->bit_flags);

	lpfc_resume_fabric_iocbs(phba);
	return;
}

void
lpfc_fabric_block_timeout(unsigned long ptr)
{
	struct clk_data *clkData = (struct clk_data *)ptr;
	lpfcHBA_t *phba = clkData->phba;

	if (clkData->flags & TM_CANCELED) {
		list_del((struct list_head *)clkData);
		kfree(clkData);	
		return;
	}

	phba->fabric_block_timer.function = NULL;
	list_del((struct list_head *)clkData);
	kfree(clkData);
	lpfc_unblock_fabric_iocbs(phba);
}

static void
lpfc_block_fabric_iocbs(lpfcHBA_t *phba)
{
	/* Start a timer to unblock fabric iocbs after 100ms */
	set_bit(FABRIC_COMANDS_BLOCKED, &phba->bit_flags);
	if (phba->fabric_block_timer.function == NULL)
		lpfc_start_timer_ms(phba, HZ/10, &phba->fabric_block_timer,
				    lpfc_fabric_block_timeout,
				    (unsigned long)phba, 0, 0);

	return;
}

static void
lpfc_cmpl_fabric_iocb(lpfcHBA_t *phba, LPFC_IOCBQ_t *cmdiocb,
	LPFC_IOCBQ_t *rspiocb)
{
	LS_RJT stat;

	switch (rspiocb->iocb.ulpStatus) {
		case IOSTAT_NPORT_RJT:
		case IOSTAT_FABRIC_RJT:
			if (rspiocb->iocb.un.ulpWord[4] & RJT_UNAVAIL_TEMP) {
				lpfc_block_fabric_iocbs(phba);
			}
			break;

		case IOSTAT_NPORT_BSY:
		case IOSTAT_FABRIC_BSY:
			lpfc_block_fabric_iocbs(phba);
			break;

		case IOSTAT_LS_RJT:
			stat.un.lsRjtError =
				be32_to_cpu(rspiocb->iocb.un.ulpWord[4]);
			if ((stat.un.b.lsRjtRsnCode == LSRJT_UNABLE_TPC) ||
				(stat.un.b.lsRjtRsnCode == LSRJT_LOGICAL_BSY))
				lpfc_block_fabric_iocbs(phba);
			break;
	}

	if (atomic_read(&phba->fabric_iocb_count) == 0)
		BUG();

	cmdiocb->iocb_cmpl = cmdiocb->fabric_iocb_cmpl;
	cmdiocb->fabric_iocb_cmpl = NULL;
	cmdiocb->iocb_cmpl(phba, cmdiocb, rspiocb);
	atomic_dec(&phba->fabric_iocb_count);
	if (!test_bit(FABRIC_COMANDS_BLOCKED, &phba->bit_flags)) {
				/* Post any pending iocbs to HBA */
		    lpfc_resume_fabric_iocbs(phba);
	}
}

int
lpfc_issue_fabric_iocb(lpfcHBA_t *phba, LPFC_IOCBQ_t *iocb)
{
	LPFC_SLI_RING_t *pring = &phba->sli.ring[LPFC_ELS_RING];
	int ready;
	int ret;

	if (atomic_read(&phba->fabric_iocb_count) > 1)
		BUG();

	ready = ((atomic_read(&phba->fabric_iocb_count) == 0) &&
		 !(test_bit(FABRIC_COMANDS_BLOCKED, &phba->bit_flags)));

	if (ready) {
		iocb->fabric_iocb_cmpl = iocb->iocb_cmpl;
		iocb->iocb_cmpl = lpfc_cmpl_fabric_iocb;
		atomic_inc(&phba->fabric_iocb_count);
		ret = lpfc_sli_issue_iocb(phba, pring, iocb, 0);

		if (ret == IOCB_ERROR) {
			iocb->iocb_cmpl = iocb->fabric_iocb_cmpl;
			iocb->fabric_iocb_cmpl = NULL;
			atomic_dec(&phba->fabric_iocb_count);
		}
	} else {
		list_add_tail(&iocb->list, &phba->fabric_iocb_list);
		ret = IOCB_SUCCESS;
	}
	return ret;
}

void lpfc_fabric_abort_vport(lpfc_vport_t *vport)
{
	lpfcHBA_t  *phba = vport->phba;
	LPFC_IOCBQ_t *tmp_iocb, *piocb;

	list_for_each_entry_safe(piocb, tmp_iocb, &phba->fabric_iocb_list,
				 list) {
		if (piocb->vport != vport)
			continue;
		list_del(&piocb->list);

		piocb->iocb.ulpStatus = IOSTAT_LOCAL_REJECT;
		piocb->iocb.un.ulpWord[4] = IOERR_SLI_ABORTED;
		(piocb->iocb_cmpl) (phba, piocb, piocb);
	}
}

void lpfc_fabric_abort_nport(lpfcHBA_t *phba, LPFC_NODELIST_t *ndlp)
{
	LPFC_IOCBQ_t *tmp_iocb, *piocb;
	LPFC_SLI_RING_t *pring = &phba->sli.ring[LPFC_ELS_RING];

	list_for_each_entry_safe(piocb, tmp_iocb, &phba->fabric_iocb_list,
				 list) {
		if ((lpfc_check_sli_ndlp(phba, pring, piocb, ndlp))) {
			list_del(&piocb->list);
			
			piocb->iocb.ulpStatus = IOSTAT_LOCAL_REJECT;
			piocb->iocb.un.ulpWord[4] = IOERR_SLI_ABORTED;
			(piocb->iocb_cmpl) (phba, piocb, piocb);
		}
	}
}

void lpfc_fabric_abort_hba(lpfcHBA_t *phba)
{
	LPFC_IOCBQ_t *piocb, *tmp_iocb;

	list_for_each_entry_safe(piocb, tmp_iocb, &phba->fabric_iocb_list,
				 list) {
		list_del(&piocb->list);

		piocb->iocb.ulpStatus = IOSTAT_LOCAL_REJECT;
		piocb->iocb.un.ulpWord[4] = IOERR_SLI_ABORTED;
		(piocb->iocb_cmpl) (phba, piocb, piocb);
	}
}

void lpfc_fabric_abort_flogi(lpfcHBA_t *phba)
{
	LPFC_IOCBQ_t *tmp_iocb, *piocb;
	LPFC_NODELIST_t *ndlp;

	list_for_each_entry_safe(piocb, tmp_iocb, &phba->fabric_iocb_list,
				 list) {
		ndlp = (struct lpfc_nodelist *) piocb->context1;
		if (piocb->iocb.ulpCommand == CMD_ELS_REQUEST64_CR &&
		    ndlp != NULL && ndlp->nlp_DID == Fabric_DID) {
			list_del(&piocb->list);

			piocb->iocb.ulpStatus = IOSTAT_LOCAL_REJECT;
			piocb->iocb.un.ulpWord[4] = IOERR_SLI_ABORTED;
			(piocb->iocb_cmpl) (phba, piocb, piocb);

		}
	}
}

void
lpfc_loopback_event(lpfcHBA_t *phba, LPFC_SLI_RING_t *pring, LPFC_IOCBQ_t *piocbq)
{
	IOCB_t *icmd = &piocbq->iocb;
	LPFC_IOCBQ_t *next_piocbq;
	struct list_head head;
	struct lpfc_dmabuf *pnext;
	int i;

	if (icmd->ulpStatus) {
		return;
	}

	if (icmd->ulpBdeCount == 0) {
		phba->fc_loopback_rxxri = icmd->ulpContext;
		return;
	}

	phba->fc_loopback_data = NULL;

	INIT_LIST_HEAD(&head);
	list_add_tail(&head, &piocbq->list);
	list_for_each_entry_safe(piocbq, next_piocbq, &head, list) {
		icmd = &piocbq->iocb;

		for (i = 0; i < icmd->ulpBdeCount; i++) {
			pnext = lpfc_sli_ringpostbuf_get(phba, pring,
							getPaddr(icmd->un.
								cont64[i].
								addrHigh,
								icmd->un.
								cont64[i].
								addrLow));
			if (!pnext)
				return;
			if (!phba->fc_loopback_data) {
				phba->fc_loopback_data = pnext;
				INIT_LIST_HEAD(&phba->fc_loopback_data->list);
			} else { 
				list_add_tail(&pnext->list, 
						&phba->fc_loopback_data->list);
			}
		}
	}
	list_del(&head);
}

