/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Fibre Channel Host Bus Adapters.                                *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003-2008 Emulex.  All rights reserved.           *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

/*
 * $Id: lpfc_fcp.c 1.8 2005/05/03 11:21:36EDT jsullivan Exp  $
 */


/* This is to export entry points needed for IP interface */
#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB
#endif
#include <linux/version.h>
#include <linux/config.h>

#ifdef __VMKERNEL_MODULE__
#include "linux_skbuff.h"
#include "vmklinux_dist.h"
#include "npiv_api_dist.h"
#endif

#include <linux/kernel.h>
#include <linux/blk.h>
#include <linux/utsname.h>
#include <linux/pci.h>
#include <linux/timer.h>
#include <linux/if_arp.h>
#include <linux/spinlock.h>

#ifdef CONFIG_VMNIX
#define INFO_BUF    8192
#endif

/* From drivers/scsi */
#include <sd.h>
#include <hosts.h>
#include <scsi.h>
#include <linux/ctype.h>

#include "lpfc_hw.h"

#define LPFC_DEF_ICFG
#include "lpfc_dfc.h"
#include "lpfc_cfgparm.h"

#include "lpfc_sli.h"
#include "lpfc_mem.h"
#include "lpfc_sched.h"
#include "lpfc_disc.h"
#include "lpfc.h"
#include "lpfc_logmsg.h"
#include "lpfc_fcp.h"
#include "lpfc_scsi.h"
#include "lpfc_version.h"

#include <linux/rtnetlink.h>
#include <asm/byteorder.h>
#include <linux/module.h>

/* Configuration parameters defined */
#include "lpfc_module_param.h"
#include "lpfc.conf"
#include "lpfc_compat.h"
#include "lpfc_crtn.h"
#include "hbaapi.h"

char *lpfc_drvr_name = LPFC_DRIVER_NAME;
char *lpfc_release_version = LPFC_DRIVER_VERSION;

MODULE_DESCRIPTION("Emulex LightPulse Fibre Channel driver - Open Source");
MODULE_AUTHOR("Emulex Corporation - tech.support@emulex.com");

#define FC_EXTEND_TRANS_A 1
#define ScsiResult(host_code, scsi_code) (((host_code) << 16) | scsi_code)

int lpfc_detect(Scsi_Host_Template *);
int lpfc_detect_instance(int, struct pci_dev *, uint32_t, Scsi_Host_Template *);
int lpfc_linux_attach(int, Scsi_Host_Template *, struct pci_dev *);
int lpfc_get_bind_type(lpfcHBA_t *);
void lpfc_ns_queue_purge_vport(lpfc_vport_t *);

int lpfc_release(struct Scsi_Host *host);
int lpfc_linux_detach(int);

void lpfc_select_queue_depth(struct Scsi_Host *, struct scsi_device *);

const char *lpfc_info(struct Scsi_Host *);

int lpfc_device_queue_depth(lpfcHBA_t *, struct scsi_device *);
int lpfc_reset_bus_handler(struct scsi_cmnd *cmnd);

int lpfc_memmap(lpfcHBA_t *);
int lpfc_unmemmap(lpfcHBA_t *);
int lpfc_pcimap(lpfcHBA_t *);

int lpfc_config_setup(lpfcHBA_t *);
int lpfc_bind_setup(lpfc_vport_t *);
int lpfc_sli_setup(lpfcHBA_t *);
int lpfc_bind_wwpn(lpfc_vport_t *, char **, u_int);
int lpfc_bind_wwnn(lpfc_vport_t *, char **, u_int);
int lpfc_bind_did(lpfc_vport_t *, char **, u_int);
LPFCSCSILUN_t *lpfc_tran_find_lun(LPFC_SCSI_BUF_t *);

extern int lpfc_biosparam(Disk *, kdev_t, int[]);

int lpfc_ioctl(Scsi_Device *dev, int cmd, void *arg);

#ifdef __VMKERNEL_MODULE__

int lpfc_npiv_api_handler(int cmd, void *arg);

#include "lpfc_cdev.c"
#include "scsi_transport_fc.h"

/*
 * Transport functions
 */
static void
lpfc_get_starget_port_id(struct scsi_target *starget);
static void
lpfc_get_starget_node_name(struct scsi_target *starget);
static void
lpfc_get_starget_port_name(struct scsi_target *starget);
static void
lpfc_get_host_port_id(struct Scsi_Host *shost);
static void
lpfc_get_port_state(struct Scsi_Host *shost);
static void
lpfc_get_port_type(struct Scsi_Host *shost);
static void
lpfc_get_host_speed(struct Scsi_Host *shost);

static struct fc_function_template lpfc_transport_functions = {
	.get_starget_port_id  = lpfc_get_starget_port_id,
	.get_starget_node_name = lpfc_get_starget_node_name,
	.get_starget_port_name = lpfc_get_starget_port_name,
	.get_host_port_id  = lpfc_get_host_port_id,
	.get_host_port_type  = lpfc_get_port_type,
	.get_host_port_state  = lpfc_get_port_state,
 	.get_host_speed = lpfc_get_host_speed,       
};

#endif

/* Binding Definitions: Max string size */
#define FC_MAX_DID_STRING       6
#define FC_MAX_WW_NN_PN_STRING 16


#if VARYIO == 20
#define VARYIO_ENTRY .can_do_varyio = 1,
#elif VARYIO == 3
#define VARYIO_ENTRY .vary_io =1,
#else
#define VARYIO_ENTRY
#endif

#if defined CONFIG_HIGHMEM
#if USE_HIGHMEM_IO ==2		/* i386 & Redhat 2.1 */
#define HIGHMEM_ENTRY .can_dma_32 = 1,
#define SINGLE_SG_OK .single_sg_ok = 1,
#else
#if USE_HIGHMEM_IO ==3		/* Redhat 3.0, Suse */
#define HIGHMEM_ENTRY .highmem_io = 1,
#define SINGLE_SG_OK
#else				/* any other */
#define HIGHMEM_ENTRY
#define SINGLE_SG_OK
#endif
#endif
#else				/* no highmem config */
#define HIGHMEM_ENTRY
#define SINGLE_SG_OK
#endif

static Scsi_Host_Template driver_template = {
	.module = THIS_MODULE,
	.name = LPFC_DRIVER_NAME,
	.info = lpfc_info,
	.queuecommand = lpfc_queuecommand,

	.eh_abort_handler = lpfc_abort_handler,
	.eh_device_reset_handler = lpfc_reset_lun_handler,
	.eh_bus_reset_handler = lpfc_reset_bus_handler,

	.detect = lpfc_detect,
	.release = lpfc_release,
	.use_new_eh_code = 1,
	VARYIO_ENTRY
	HIGHMEM_ENTRY
	SINGLE_SG_OK

	.bios_param = lpfc_biosparam,
	.proc_info = lpfc_proc_info,
	.proc_name = LPFC_DRIVER_NAME,

	.this_id = -1,

	.sg_tablesize = SG_ALL,
	.cmd_per_lun = 30,
	.use_clustering = DISABLE_CLUSTERING,
	.max_sectors = MAX_SCSI_SECTORS,

#ifdef __VMKERNEL_MODULE__
	.ioctl = lpfc_ioctl,
#endif
};



int lpfc_nethdr = 1;

uint16_t lpfc_num_nodes = 128;	/* default number of NPort structs to alloc */

lpfcDRVR_t lpfcDRVR;

extern char *lpfc_fcp_bind_WWPN[];
extern char *lpfc_fcp_bind_WWNN[];
extern char *lpfc_fcp_bind_DID[];


static struct pci_device_id lpfc_id_table[] = {
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_VIPER,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_THOR,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_PEGASUS,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_CENTAUR,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_DRAGONFLY,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_SUPERFLY,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_RFLY,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_PFLY,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_NEPTUNE,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_NEPTUNE_SCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_NEPTUNE_DCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_HELIOS,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_HELIOS_SCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_HELIOS_DCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_BMID,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_BSMB,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_ZEPHYR,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_HORNET,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_ZEPHYR_SCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_ZEPHYR_DCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_ZMID,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_ZSMB,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_TFLY,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_LP101,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_LP10000S,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_LP11000S,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_LPE11000S,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_SAT,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_SAT_MID,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_SAT_SMB,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_SAT_DCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_SAT_SCSP,
		PCI_ANY_ID, PCI_ANY_ID, },
	{PCI_VENDOR_ID_EMULEX, PCI_DEVICE_ID_LPE12000S,
		PCI_ANY_ID, PCI_ANY_ID, },
	{ 0 }
};
MODULE_DEVICE_TABLE(pci, lpfc_id_table);

#ifdef __VMKERNEL_MODULE__
int
lpfc_npiv_api_handler(int cmd, void *arg)
{
	int rc;
	vport_arg *p = (vport_arg *)arg;
	struct vport_data data;
	struct vport_info info;

	if (arg == NULL)
		return VPORT_PARAMETER_ERR;

	switch (cmd) {
	case NPIV_VPORT_CREATE:
		memset(&data, 0, sizeof(struct vport_data));
		memcpy(data.node_name, p->vport_create.cinfo.node_name,
			sizeof(data.node_name));
		memcpy(data.port_name, p->vport_create.cinfo.port_name,
			sizeof(data.port_name));

		rc = lpfc_vport_create(p->vport_create.physhost, &data);
		if (rc == VPORT_OK)
			p->vport_create.cinfo.vport_shost = data.vport_shost;
		break;

	case NPIV_VPORT_DELETE:
		rc = lpfc_vport_delete(p->vport_delete.virthost);
		break;

	case NPIV_VPORT_INFO:
		rc = lpfc_vport_getinfo(p->vport_getinfo.virthost, &info);
		if (rc == VPORT_OK)
			memcpy(p->vport_getinfo.info, &info, sizeof(info));
		break;

	case NPIV_VPORT_TGT_REMOVE:
		rc = lpfc_vport_tgt_remove(p->vport_tgt_remove.virthost,
					   p->vport_tgt_remove.channel,
					   p->vport_tgt_remove.tgtid);
		break;
		
	default:
		printk("%s: lpfcdd_732 Received unknown vport command 0x%x on "
			"arg %p- ignoring\n",  __FUNCTION__, cmd, arg);
		return VPORT_INVAL;
	}

	return rc;
}
#endif

int
lpfc_ioctl(Scsi_Device *dev, int cmd, void *arg)
{
#ifdef __VMKERNEL_MODULE__
	return lpfc_npiv_api_handler(cmd, arg);
#else
	return -EINVAL;
#endif
}


int
lpfc_detect(Scsi_Host_Template * tmpt)
{
	struct pci_dev *pdev = 0;
	int instance = 0;
	int i;

#ifdef __VMKERNEL_MODULE__
        if (!vmk_set_module_version(LPFC_DRIVER_VERSION)) {
                return 0;
        }   
#endif

#if VARYIO == 21
#ifdef SCSI_HOST_VARYIO
	SCSI_HOST_VARYIO(tmpt) = 1;
#endif
#endif
	printk(LPFC_MODULE_DESC "\n");

#ifndef __VMKERNEL_MODULE__
	spin_unlock_irq(&io_request_lock);
#endif

	memset((char *)&lpfcDRVR, 0, sizeof (lpfcDRVR_t));
	lpfcDRVR.loadtime = jiffies;

	INIT_LIST_HEAD(&lpfcDRVR.hba_list_head);

	i = 0;
	while (lpfc_id_table[i].vendor) {
		instance = lpfc_detect_instance(instance, pdev, lpfc_id_table[i].device, tmpt );
		i++;
	}

#ifdef __VMKERNEL_MODULE__
	if (instance)
		lpfc_cdev_init();
#endif
#ifndef __VMKERNEL_MODULE__
	spin_lock(&io_request_lock);
#endif
	return (instance);
}

int
lpfc_detect_instance(int instance,
		     struct pci_dev *pdev, uint type, Scsi_Host_Template * tmpt)
{
	/* Find all Emulex HBAs and attach to them.  */
	while ((pdev = pci_find_subsys(PCI_VENDOR_ID_EMULEX, type,
				       PCI_ANY_ID, PCI_ANY_ID, pdev))) {
		if (pci_enable_device(pdev)) {
			continue;
		}
		if (pci_request_regions(pdev, LPFC_DRIVER_NAME)) {
			printk("lpfc pci I/O region is already in use. \n");
			printk("a driver for lpfc is already loaded on this "
			       "system\n");
			continue;
		}

		if (lpfc_linux_attach(instance, tmpt, pdev)) {
			pci_release_regions(pdev);

			/* Failed to attach to lpfc adapter: bus <bus>
			   device <device> irq <irq> */
			lpfc_printf_log(instance, &lpfc_msgBlk0443,
					lpfc_mes0443,
					lpfc_msgBlk0443.msgPreambleStr,
					pdev->bus->number,
					PCI_SLOT(pdev->devfn),
					pdev->irq);

			continue;
		}
		instance++;
	}

	return (instance);
}


int
lpfc_release(struct Scsi_Host *host)
{
	lpfcHBA_t *phba;
	int instance;
	lpfc_vport_t *vport, *n_vport;

#ifdef __VMKERNEL_MODULE__
        vmk_fc_release_transport(host);
#endif
	phba  = ((lpfc_vport_t*) host->hostdata[0])->phba;
	list_for_each_entry_safe(vport, n_vport, &phba->port_list, listentry) {
		if (vport->scsihost == host) {
			goto found;
		}
	}

	return (0);

found:
    
	phba->port_cnt--;
	if (phba->port_cnt)
		return (0);

 	instance = phba->brd_no;

	/* detach the board */
	lpfc_linux_detach(instance);

#ifdef __VMKERNEL_MODULE__
	if (lpfc_major)
		lpfc_cdev_exit();
#endif
	return (0);
}


int
lpfc_linux_attach(int instance, Scsi_Host_Template *tmpt, struct pci_dev *pdev)
{
	lpfcHBA_t *phba;
	lpfcCfgParam_t *clp;
	struct clk_data *clkData;
	int rc = 0, i;
	unsigned long iflag;
	lpfc_vport_t *vport;

	if (!pdev)
		return 1;

	if (!(phba = kmalloc(sizeof (lpfcHBA_t), GFP_ATOMIC)))
		return 1;

	memset(phba, 0, sizeof (lpfcHBA_t));
	INIT_LIST_HEAD(&phba->timerList);
	lpfc_drvr_init_lock(phba);
	INIT_LIST_HEAD(&phba->task_disc);

	/* Initialize default values for configuration parameters */
	if (!(phba->config = kmalloc(sizeof (lpfc_icfgparam), GFP_ATOMIC)))
		goto error_1;

	memset(phba->config, 0, sizeof (lpfc_icfgparam));
	memcpy(&phba->config[0], (uint8_t *) & lpfc_icfgparam[0],
	       sizeof (lpfc_icfgparam));

	clp = &phba->config[0];

	/* Set everything to the defaults */
	for (i = 0; i < LPFC_TOTAL_NUM_OF_CFG_PARAM; i++)
		clp[i].a_current = clp[i].a_default;
	
	phba->brd_no = instance;
	phba->tmpt = tmpt;

	/* Add adapter structure to list */
	list_add_tail(&phba->hba_list, &lpfcDRVR.hba_list_head);

	/* Initialize all internally managed lists. */
	INIT_LIST_HEAD(&phba->port_list);
	INIT_LIST_HEAD(&phba->hbq_buffer_list);
	INIT_LIST_HEAD(&phba->ns_cmd_list);
	INIT_LIST_HEAD(&phba->fabric_iocb_list);
	INIT_LIST_HEAD(&phba->delayed_iocbs);
	phba->num_ns_cmds = 0;

	/* Initialize plxhba - LINUX specific */
	phba->pcidev = pdev;
	init_waitqueue_head(&phba->linkevtwq);
	init_waitqueue_head(&phba->rscnevtwq);
	init_waitqueue_head(&phba->ctevtwq);
	init_waitqueue_head(&phba->tempevtwq);

	/* Initialize delayed iocb timer */
	init_timer(&phba->delayed_iocb_tmo);
	phba->delayed_iocb_tmo.function = lpfc_free_delayed_iocbs_tmo;
	phba->delayed_iocb_tmo.data = (unsigned long)phba;

	if (lpfc_pcimap(phba))
		goto error_2;

	if (lpfc_memmap(phba))
		goto error_2;

	lpfcDRVR.num_devs++;
	lpfc_config_setup(phba);

	/* Let administrator know if hba heartbeat is disabled */
	if (!clp[LPFC_CFG_ENABLE_HEARTBEAT].a_current) {
	        lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0406,
                                       lpfc_mes0406,
                                       lpfc_msgBlk0406.msgPreambleStr);
	}

	/* Let administrator know if hba reset is disabled */
        if (!clp[LPFC_CFG_ENABLE_RESET].a_current) {
                lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0407,
                                       lpfc_mes0407,
                                       lpfc_msgBlk0407.msgPreambleStr);
        }

	/* If the t.o value is not set, set it to 30 */
	if (clp[LPFC_CFG_SCSI_REQ_TMO].a_current == 0)
		clp[LPFC_CFG_SCSI_REQ_TMO].a_current = 30;

	if (clp[LPFC_CFG_DISC_THREADS].a_current) {
		/*
		 * Set to FC_NLP_REQ if automap is set to 0 since order of
		 * discovery does not matter if everything is persistently
		 * bound. 
		 */
		if (clp[LPFC_CFG_AUTOMAP].a_current == 0) {
			clp[LPFC_CFG_DISC_THREADS].a_current =
			    LPFC_MAX_DISC_THREADS;
		}
	}

	/* Create and initialize the physical port instance. */
	vport = lpfc_create_port(phba, instance);
	if (vport == NULL)
		goto error_3;
		
	phba->pport = vport;
	vport->port_type = LPFC_PHYSICAL_PORT;
 
	/*
	 * Adjust the number of id's
	 * Although max_id is an int, target id's are unsined chars
	 * Do not exceed 255, otherwise the device scan will wrap around
	 */
	if (clp[LPFC_CFG_MAX_TARGET].a_current > LPFC_MAX_TARGET)
		clp[LPFC_CFG_MAX_TARGET].a_current = LPFC_DFT_MAX_TARGET;

	if (clp[LPFC_CFG_MAX_VPI].a_current > LPFC_MAX_VPI)
		clp[LPFC_CFG_MAX_VPI].a_current = LPFC_DFT_MAX_VPI;

	phba->max_vpi = clp[LPFC_CFG_MAX_VPI].a_current;

	if (clp[LPFC_CFG_MAX_LUN].a_current > LPFC_MAX_LUN)
		clp[LPFC_CFG_MAX_LUN].a_current = LPFC_DFT_MAX_LUN;

	/* Setup SLI Layer to run over lpfc HBAs and then initialize it. */
	lpfc_sli_setup(phba);
	
	if (lpfc_sli_queue_setup(phba) == 0)
		goto error_4;

	if (lpfc_mem_alloc(phba) == 0)
		goto error_5;

	lpfc_bind_setup(phba->pport);

	/*
	 * Establish default FC SAN timeouts.  These are overwritten once
	 * Fabric or loop is established.
	 */
	phba->fc_edtov = FF_DEF_EDTOV;
	phba->fc_ratov = FF_DEF_RATOV;
	phba->fc_altov = FF_DEF_ALTOV;
	phba->fc_arbtov = FF_DEF_ARBTOV;

	/*
	 * Set the FARP and XRI timeout values now since they depend on
	 * fc_ratov.
	 */
	phba->fc_ipfarp_timeout = (3 * phba->fc_ratov);
	phba->fc_ipxri_timeout = (3 * phba->fc_ratov);

	tasklet_init(&phba->task_run, (void *)lpfc_tasklet, (unsigned long) phba);

	/*
	 * Set the watchdog timer up for LPFC_IOCB_WATCHDOG_TMO second
	 * cycle time.
	 */
	lpfc_start_timer(phba, LPFC_IOCB_WATCHDOG_TMO,
			&phba->iocb_tmofunc, lpfc_iocb_timeout_handler,
			(HZ * LPFC_IOCB_WATCHDOG_TMO), 0, 0);

	LPFC_DRVR_LOCK(phba, iflag);
	if ((rc = lpfc_sli_hba_setup(phba)) != 0) {
		LPFC_DRVR_UNLOCK(phba, iflag);
		tasklet_kill(&phba->task_run);
		goto error_6;
	}

	LPFC_DRVR_UNLOCK(phba, iflag);

#if defined(CONFIG_VMNIX) && !defined(__VMKERNEL_MODULE__)
        vport->scsihost->bus = pdev->bus->number;
        vport->scsihost->devfn = pdev->devfn;
        vport->scsihost->devid = phba;
#endif
#ifdef __VMKERNEL_MODULE__
        /* have wait for the the chip to come up before we can get the wwns */
	fc_host_node_name(vport->scsihost) = wwn_to_u64((uint8_t *)&phba->pport->fc_nodename);
	fc_host_port_name(vport->scsihost) = wwn_to_u64((uint8_t *)&phba->pport->fc_portname);
#endif
	goto out;

 error_6:
	LPFC_DRVR_LOCK(phba, iflag);
	lpfc_sli_hba_down(phba);
	while (!list_empty(&phba->timerList)) {
		clkData = (struct clk_data *)(phba->timerList.next);
		if (clkData)
			lpfc_stop_timer(clkData);
	}
	
	LPFC_DRVR_UNLOCK(phba, iflag);
	lpfc_tasklet((unsigned long)phba);
	lpfc_mem_free(phba);

 error_5:
	/* Release all ring resources allocated to this phba. */
	lpfc_sli_queue_free(phba);

 error_4:
	lpfc_destroy_port(vport);

 error_3:
	lpfc_unmemmap(phba);
	lpfcDRVR.num_devs--;

 error_2:
	kfree(phba->config);

 error_1:
	list_del_init(&phba->hba_list);
	kfree(phba);
	rc = 1;

 out:
	return rc;
}

lpfc_vport_t *
lpfc_create_port(lpfcHBA_t *phba, int instance)  
{
	unsigned long iflag;
	lpfc_vport_t *vport = NULL;
	struct Scsi_Host *scsihost;
	lpfcCfgParam_t *clp;

#ifdef __VMKERNEL_MODULE__
	if (instance >= 100) {
		scsihost = vmk_scsi_register(phba->tmpt, sizeof(unsigned long), 0xfffe, 0xfffe);
		if (scsihost)
			vmk_scsi_register_uinfo(scsihost, 0xfffe,  0xfffe, phba);
	} else {
		scsihost = vmk_scsi_register(phba->tmpt, sizeof(unsigned long), phba->pcidev->bus->number,
					phba->pcidev->devfn);
		if (scsihost)
			vmk_scsi_register_uinfo(scsihost, phba->pcidev->bus->number,  phba->pcidev->devfn, phba);
	}

	if (scsihost) {
		/* start the fc transport */
		if (vmk_fc_attach_transport(scsihost, &lpfc_transport_functions) != SUCCESS) {
			printk("lpfc_create_port: vmk_fc_attach_transport: failed!\n");
			scsi_unregister(scsihost);
			goto out;
		}
	}
#else
	scsihost = scsi_register(phba->tmpt, sizeof (unsigned long));
#endif
	if (scsihost == NULL) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1805,
				lpfc_mes1805,
				lpfc_msgBlk1805.msgPreambleStr,
				instance);
		goto out;
	}
	
	if (!(vport = kmalloc(sizeof (lpfc_vport_t), GFP_ATOMIC))) {
#ifdef __VMKERNEL_MODULE__
                vmk_fc_release_transport(scsihost);
#endif
		scsi_unregister(scsihost);
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1806,
				lpfc_mes1806,
				lpfc_msgBlk1806.msgPreambleStr,
				instance);
		goto out;
	}
	memset(vport, 0, sizeof (lpfc_vport_t));

	clp = &phba->config[0];
	vport->phba = phba;
	vport->scsihost = scsihost;
	vport->fc_prevDID = Mask_DID;
	scsihost->hostdata[0] = (unsigned long)vport;

	INIT_LIST_HEAD(&vport->fc_nlpmap_list);
	INIT_LIST_HEAD(&vport->fc_nlpunmap_list);
	INIT_LIST_HEAD(&vport->fc_plogi_list);
	INIT_LIST_HEAD(&vport->fc_adisc_list);
	INIT_LIST_HEAD(&vport->fc_nlpbind_list);
	INIT_LIST_HEAD(&vport->delay_list);

	/* Set up the HBA specific LUN device lookup routine */
	vport->lpfc_tran_find_lun = lpfc_tran_find_lun;

	/* Adapter ID - tell midlayer not to reserve an ID for us */
	scsihost->this_id = -1;

	/*
	 * Scsi command can exceed 12 bytes.  Set the maximum command length
	 * to guarantee 12 bytes is not the limit.
	 */
	scsihost->max_cmd_len = 16;
	scsihost->max_lun = clp[LPFC_CFG_MAX_LUN].a_current;

	/* Configure the lun queue depth. */
	scsihost->cmd_per_lun = clp[LPFC_CFG_DFT_LUN_Q_DEPTH].a_current;
	scsihost->select_queue_depths = lpfc_select_queue_depth;   
	scsihost->can_queue = LPFC_SET_CAN_QUEUE;
	scsihost->max_id = clp[LPFC_CFG_MAX_TARGET].a_current;
	scsihost->unique_id = instance; 

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,4)
	scsi_set_pci_device(vport->scsihost, phba->pcidev);
#endif

#if defined(CONFIG_VMNIX) && !defined(__VMKERNEL_MODULE__)
        vport->scsihost->bus = phba->pcidev->bus->number;
        vport->scsihost->devfn = phba->pcidev->devfn;
        vport->scsihost->devid = phba;
#endif

	/* Configure the scheduler with the max xri reported by the HBA. */
	lpfc_sched_hba_init(vport, LPFC_SET_CAN_QUEUE);
	LPFC_DRVR_LOCK(phba, iflag);
	list_add_tail(&vport->listentry, &phba->port_list);
	phba->port_cnt++;
	LPFC_DRVR_UNLOCK(phba, iflag);
 out:
	return vport;	      
}

void 
lpfc_destroy_port(lpfc_vport_t *vport)  
{
	lpfcHBA_t *phba;
	LPFC_SCSI_BUF_t *lpfc_cmd;
	LPFC_SLI_RING_t *pring;
	uint32_t i;
	unsigned long iflag;

	phba = vport->phba;
	LPFC_DRVR_LOCK(phba, iflag);

	/* Cleanup any outstanding RSCN activity */
	lpfc_els_flush_rscn(vport);
       
	/* remove any name server requests that are queued */
	lpfc_ns_queue_purge_vport(vport);
	
	/* Complete any scheduled delayed i/o completions */
	while (!list_empty(&vport->delay_list)) {
		lpfc_cmd = list_entry(vport->delay_list.next, LPFC_SCSI_BUF_t, listentry);
		list_del(&lpfc_cmd->listentry);
		lpfc_iodone(vport, lpfc_cmd, SCSI_DONE_LOCK_HELD);
	}

	/* Remove any iocbs originated by this host */
	for (i = 0; i < phba->sli.sliinit.num_rings; i++) {
		pring = &phba->sli.ring[i];
		lpfc_vport_sli_abort_iocb_ring(phba, vport, pring, LPFC_SLI_ABORT_IMED);
	}

	LPFC_DRVR_UNLOCK(phba, iflag);
#ifdef __VMKERNEL_MODULE__
        vmk_fc_release_transport(vport->scsihost);
#endif
	scsi_unregister(vport->scsihost);
	LPFC_DRVR_LOCK(phba, iflag);

	vport->scsihost = NULL;
	list_del(&vport->listentry);

	lpfc_flush_disc_evtq(phba, vport);
	
	lpfc_cleanup(vport, 0);
	lpfc_scsi_free(vport);
	phba->port_cnt--;
	LPFC_DRVR_UNLOCK(phba, iflag);
	kfree(vport);
	return;     
}

void 
lpfc_destroy_host_all(lpfcHBA_t *phba)  
{
	lpfc_vport_t *vport, *n_vport;
		
	list_for_each_entry_safe(vport, n_vport, &phba->port_list, listentry) {
		lpfc_cleanup(vport, 0);  
		lpfc_scsi_free(vport);	      
		list_del(&vport->listentry);
		kfree(vport);
	}

	return;
}

/*
 * Retrieve instance (board no) matching phba
 * If valid hba return 1
 * If invalid hba return 0 
 */
int
lpfc_check_valid_phba(lpfcHBA_t * phba)
{
	struct list_head * pos;
	lpfcHBA_t        * tphba = NULL;
	int    found = 0;

	list_for_each(pos, &lpfcDRVR.hba_list_head) {
		tphba = list_entry(pos, lpfcHBA_t, hba_list);
		if (tphba == phba) {
			found = 1;
			break;
		}
	}
	
	return found;
}

/*
 * Retrieve instance (board no) matching phba
 * If found return board number
 * If not found return -1 
 */
int
lpfc_get_inst_by_phba(lpfcHBA_t * phba)
{
	struct list_head * pos;
	lpfcHBA_t        * tphba = NULL;
	int    found = 0;

	list_for_each(pos, &lpfcDRVR.hba_list_head) {
		tphba = list_entry(pos, lpfcHBA_t, hba_list);
		if (tphba == phba) {
			found = 1;
			break;
		}
	}
	
	if (!found) 
		return(-1);
	else
		return(tphba->brd_no);
	
}

/*
 * Retrieve lpfcHBA * matching instance (board no)
 * If found return lpfcHBA *
 * If not found return NULL 
 */
lpfcHBA_t *
lpfc_get_phba_by_inst(int inst)
{
	struct list_head * pos;
	lpfcHBA_t * phba;
	int    found = 0;

	phba = NULL;
	list_for_each(pos, &lpfcDRVR.hba_list_head) {
		phba = list_entry(pos, lpfcHBA_t, hba_list);
		if (phba->brd_no == inst) {
			found = 1;
			break;
		}
	}
	
	if (!found) 
		return(NULL);
	else
		return(phba);
	
}

int
lpfc_linux_detach(int instance)
{
	lpfcHBA_t        *phba;
	LPFC_SLI_t       *psli;
	struct clk_data  *clkData, *n_clkData;
	unsigned long      iflag;
	LPFC_SCSI_BUF_t  *lpfc_cmd;
	struct timer_list *ptimer;
	lpfc_vport_t *vport, *n_vport;

	if ((phba = lpfc_get_phba_by_inst(instance)) == NULL)
		return 0;

	psli = &phba->sli;

	LPFC_DRVR_LOCK(phba, iflag);
	/* First stop all scheduled timers. */
	list_for_each_entry_safe(clkData, n_clkData, &phba->timerList, listLink) {
		ptimer = clkData->timeObj;
		lpfc_stop_timer(clkData);
	}

	/* Traverse the timerlist again and kill the remaining stopped timers. */
	list_for_each_entry_safe(clkData, n_clkData, &phba->timerList, listLink) {
		list_del(&clkData->listLink);
		kfree(clkData);
	}

	/* Bring down the SLI Layer */
	lpfc_sli_hba_down(phba);
	if (phba->intr_inited) {
		/* Clear all interrupt enable conditions */
		writel(0, phba->HCregaddr);
		readl(phba->HCregaddr); /* flush */
		/* Clear all pending interrupts */
		writel(0xffffffff, phba->HAregaddr);
		readl(phba->HAregaddr); /* flush */
		LPFC_DRVR_UNLOCK(phba, iflag);
		free_irq(phba->pcidev->irq, phba);
		LPFC_DRVR_LOCK(phba, iflag);
		phba->intr_inited = 0;
	}

	if (phba->pcidev) {
		pci_release_regions(phba->pcidev);
		pci_disable_device(phba->pcidev);
	}

	list_for_each_entry_safe(vport, n_vport, &phba->port_list, listentry) {
		/* Complete any scheduled delayed i/o completions */
		while(!list_empty(&vport->delay_list)) {
			lpfc_cmd = list_entry(vport->delay_list.next, LPFC_SCSI_BUF_t, listentry);
			list_del(&lpfc_cmd->listentry);
			lpfc_iodone(vport, lpfc_cmd, SCSI_DONE_DEFAULT);
		}
	}

	LPFC_DRVR_UNLOCK(phba, iflag);
	lpfc_tasklet((unsigned long)phba);
	tasklet_kill(&phba->task_run);
	LPFC_DRVR_LOCK(phba, iflag);

	lpfc_destroy_host_all(phba);
	
	LPFC_DRVR_UNLOCK(phba, iflag);
	lpfc_mem_free(phba);
	lpfc_unmemmap(phba);
	LPFC_DRVR_LOCK(phba, iflag);

	if (phba->config)
		kfree(phba->config);

	LPFC_DRVR_UNLOCK(phba, iflag);

	/* Remove from list of active HBAs */
	list_del(&phba->hba_list);

	kfree(phba);

	lpfcDRVR.num_devs--;

	return (0);
}

/*
 * Issue a READ_CONFIG mailbox command to get the port
 * configuration values for rpi, vpi, and xri. The caller
 * is expected to initialize the memory for max_rpi and
 * avail_rpis to some known value before calling this routine.
 * Note this routine polls the mailbox command for completion.
 * The driver lock must not be held by the caller. 
 */
void
lpfc_get_hba_info(lpfcHBA_t *phba, uint32_t *mrpi, uint32_t *arpi,
		  uint32_t *mvpi, uint32_t *avpi)
{
	LPFC_MBOXQ_t *pmb;
	int rc;
	unsigned long iflag;

	pmb = lpfc_mbox_alloc(phba, MEM_PRI);
	if (pmb) {
		lpfc_read_config(phba, pmb);
		pmb->vport = phba->pport;
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_sli_issue_mbox_wait(phba, pmb, phba->fc_ratov * 2);
		LPFC_DRVR_UNLOCK(phba, iflag);
		if (rc == MBX_SUCCESS) {
			if (mrpi)
				*mrpi = pmb->mb.un.varRdConfig.max_rpi;
			if (arpi)
				*arpi = pmb->mb.un.varRdConfig.avail_rpi;
			if (mvpi)
				*mvpi = pmb->mb.un.varRdConfig.max_vpi;
			if (avpi)
				*avpi = pmb->mb.un.varRdConfig.avail_vpi;
		}
	}
	lpfc_mbox_free(phba, pmb);
}


const char *
lpfc_info(struct Scsi_Host *host)
{
	lpfcHBA_t *phba;
	lpfc_vport_t *vport;
	int len;
	static char  lpfcinfobuf[320];
	
        memset(lpfcinfobuf,0,320);

	vport = (lpfc_vport_t*)host->hostdata[0];
	phba = vport->phba;
	if (phba && phba->pcidev){
		strncpy(lpfcinfobuf, phba->ModelDesc, 320);
		len = strlen(lpfcinfobuf);	
		snprintf(lpfcinfobuf + len,
			320-len,
	       		" on PCI bus %02x device %02x irq %d",
			phba->pcidev->bus->number,
		 	phba->pcidev->devfn,
			phba->pcidev->irq);
	}

	return lpfcinfobuf;
}

int
lpfc_proc_info(char *buf,
	       char **start, off_t offset, int length, int hostnum, int rw)
{

	lpfcHBA_t *phba = NULL;
	lpfc_vport_t *vport = NULL;
	struct pci_dev *pdev;
	char fwrev[32];
	lpfc_vpd_t *vp;
	struct list_head *pos;
	LPFC_NODELIST_t *ndlp;
	int  i, j, incr;
	char hdw[9];
	int len = 0;
	uint32_t mrpi, arpi, mvpi, avpi;
	unsigned long iflag;
#ifdef CONFIG_VMNIX
        static char info_buf[INFO_BUF];
	char *save;
	int retval;
#define MIN(a, b) (((a) < (b)) ? (a) : (b))
#endif
	/* Sufficient bytes to hold a port or node name. */
	uint8_t name[sizeof (NAME_TYPE)];

	/* If rw = 0, then read info
	 * If rw = 1, then write info (NYI)
	 */
	if (rw)
		return -EINVAL;

	/* Find the physical port instance matching the caller's scsi host num. */
	list_for_each_entry(phba, &lpfcDRVR.hba_list_head, hba_list) {
		if (phba->pport->scsihost->host_no == hostnum) {
			goto found;
		}
	}

	return sprintf(buf, "Cannot find adapter for requested host number.\n");

found:
#ifdef CONFIG_VMNIX
	save = buf;
	buf = info_buf;
#endif

	vp = &phba->vpd;
	pdev = phba->pcidev;

	len += snprintf(buf, PAGE_SIZE, LPFC_MODULE_DESC "\n");

	len += snprintf(buf + len, PAGE_SIZE-len, "%s\n",
			lpfc_info(phba->pport->scsihost));

	len += snprintf(buf + len, PAGE_SIZE-len, "BoardNum: %d\n",
			phba->brd_no);

	len += snprintf(buf + len, PAGE_SIZE-len, "SerialNum: %s\n",
			phba->SerialNumber);

	lpfc_decode_firmware_rev(phba, fwrev, 1);
	len += snprintf(buf + len, PAGE_SIZE-len, "Firmware Version: %s\n",
			fwrev);

	len += snprintf(buf + len, PAGE_SIZE-len, "Hdw: ");
	/* Convert JEDEC ID to ascii for hardware version */
	incr = vp->rev.biuRev;
	for (i = 0; i < 8; i++) {
		j = (incr & 0xf);
		if (j <= 9)
			hdw[7 - i] = (char)((uint8_t) 0x30 + (uint8_t) j);
		else
			hdw[7 - i] =
			    (char)((uint8_t) 0x61 + (uint8_t) (j - 10));
		incr = (incr >> 4);
	}
	hdw[8] = 0;
	len += snprintf(buf + len, PAGE_SIZE-len, hdw);

	len += snprintf(buf + len, PAGE_SIZE-len, "\nVendorId: 0x%x\n",
		       ((((uint32_t) pdev->device) << 16) |
			(uint32_t) (pdev->vendor)));

	/* A Fibre Channel node or port name is 8 octets long and delimited by 
	 * colons.
	 */
	len += snprintf(buf + len, PAGE_SIZE-len, "Portname: ");
	memcpy (&name[0], &phba->pport->fc_portname, sizeof (NAME_TYPE));
	len += snprintf(buf + len, PAGE_SIZE-len,
			"%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x",
			name[0], name[1], name[2], name[3], name[4], name[5],
			name[6], name[7]);

	len += snprintf(buf + len, PAGE_SIZE-len, "   Nodename: ");
	memcpy (&name[0], &phba->pport->fc_nodename, sizeof (NAME_TYPE));
	len += snprintf(buf + len, PAGE_SIZE-len,
			"%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x\n",
			name[0], name[1], name[2], name[3], name[4], name[5],
			name[6], name[7]);

	if (phba->sli_rev == 3) {
		mrpi = arpi = mvpi = avpi = 0;
		lpfc_get_hba_info(phba, &mrpi, &arpi, &mvpi, &avpi);
		len += snprintf(buf+len, PAGE_SIZE-len, "\nSLI Rev: %d\n",
				phba->sli_rev);

		switch (phba->npiv_capable) {
		case LPFC_NPIV_SUPPORTED:
			len += snprintf(buf+len, PAGE_SIZE-len,
					"   NPIV Supported: ");
			len += snprintf(buf+len, PAGE_SIZE-len,
					"VPIs max %d  ", mvpi);
			len += snprintf(buf+len, PAGE_SIZE-len,
					"VPIs used %d\n", mvpi - avpi);
			break;
		case LPFC_NPIV_UNSUPPORTED_FW:
			len += snprintf(buf+len, PAGE_SIZE-len,
					"   NPIV Unsupported by HBA FW\n");
			break;
		case LPFC_NPIV_UNSUPPORTED_FAB:
			len += snprintf(buf+len, PAGE_SIZE-len,
					"   NPIV Unsupported by Fabric\n");
			break;
		default:
			len += snprintf(buf+len, PAGE_SIZE-len,
					"   Unknown NPIV State %d\n", phba->npiv_capable);
			break;
		}

		len += snprintf(buf+len, PAGE_SIZE-len, "   RPIs max %d  ",
				mrpi);
		len += snprintf(buf+len, PAGE_SIZE-len, "RPIs used %d\n\n",
				mrpi - arpi);

		/* Grab the vports on this physical port instance. */
		len += snprintf(buf+len, PAGE_SIZE-len, "   Vports list on this physical port:\n");

		/*
		 * This operation needs to be synchronized as a vport could
		 * be getting destroyed while a proc script is running.
		 */
		LPFC_DRVR_LOCK(phba, iflag);
		list_for_each_entry(vport, &phba->port_list, listentry) {
			if (vport->port_type == LPFC_PHYSICAL_PORT)
				continue;
			
			len += snprintf(buf+len, PAGE_SIZE-len, "   Vport DID 0x%x, vpi %d, state 0x%x\n", 
					vport->fc_myDID, vport->vpi, vport->port_state);
			len += snprintf(buf+len, PAGE_SIZE-len, "      Portname: ");
			memcpy (&name[0], &vport->fc_portname, sizeof (NAME_TYPE));
			len += snprintf(buf+len, PAGE_SIZE-len,
				"%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x",
				name[0], name[1], name[2], name[3], name[4], name[5],
				name[6], name[7]);

			len += snprintf(buf+len, PAGE_SIZE-len, "  Nodename: ");
			memcpy (&name[0], &vport->fc_nodename, sizeof (NAME_TYPE));
			len += snprintf(buf+len, PAGE_SIZE-len,
				"%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x\n",
				name[0], name[1], name[2], name[3], name[4], name[5],
				name[6], name[7]);			
		}
		LPFC_DRVR_UNLOCK(phba, iflag);
	}

	switch (phba->pport->port_state) {
	case LPFC_INIT_START:
	case LPFC_INIT_MBX_CMDS:
	case LPFC_LINK_DOWN:
		len += snprintf(buf + len, PAGE_SIZE-len, "\nLink Down\n");
		break;
	case LPFC_LINK_UP:
	case LPFC_LOCAL_CFG_LINK:
		len += snprintf(buf + len, PAGE_SIZE-len, "\nLink Up\n");
		break;
	case LPFC_FLOGI:
	case LPFC_FABRIC_CFG_LINK:
	case LPFC_NS_REG:
	case LPFC_NS_QRY:
	case LPFC_BUILD_DISC_LIST:
	case LPFC_DISC_AUTH:
	case LPFC_CLEAR_LA:
		len += snprintf(buf + len, PAGE_SIZE-len,
				"\nLink Up - Discovery\n");
		break;
	case LPFC_HBA_READY:
		len += snprintf(buf + len, PAGE_SIZE-len,
				"\nLink Up - Ready:\n");
		len += snprintf(buf + len, PAGE_SIZE-len, "   PortID 0x%x\n",
				phba->pport->fc_myDID);
		if (phba->fc_topology == TOPOLOGY_LOOP) {
			if (phba->pport->fc_flag & FC_PUBLIC_LOOP)
				len += snprintf(buf + len, PAGE_SIZE-len,
						"   Public Loop\n");
			else
				len += snprintf(buf + len, PAGE_SIZE-len,
						"   Private Loop\n");
		} else {
			if (phba->pport->fc_flag & FC_FABRIC)
				len += snprintf(buf + len, PAGE_SIZE-len,
						"   Fabric\n");
			else
				len += snprintf(buf + len, PAGE_SIZE-len,
						"   Point-2-Point\n");
		}

		if (phba->fc_linkspeed == LA_8GHZ_LINK)
			len += snprintf(buf + len, PAGE_SIZE-len,
					"   Current speed 8G\n\n");
		else
		if (phba->fc_linkspeed == LA_4GHZ_LINK)
			len += snprintf(buf + len, PAGE_SIZE-len,
					"   Current speed 4G\n\n");
		else
		if (phba->fc_linkspeed == LA_2GHZ_LINK)
			len += snprintf(buf + len, PAGE_SIZE-len,
					"   Current speed 2G\n\n");
		else
		if (phba->fc_linkspeed == LA_1GHZ_LINK)
			len += snprintf(buf + len, PAGE_SIZE-len,
					"   Current speed 1G\n\n");
		else
			len += snprintf(buf + len, PAGE_SIZE-len,
					"   Current speed unknown\n\n");

		/* Loop through the list of mapped nodes and dump the known node
		   information. */
		len += snprintf(buf + len, PAGE_SIZE-len, "Current Mapped Nodes on Physical Port: \n");
		list_for_each(pos, &phba->pport->fc_nlpmap_list) {
			ndlp = list_entry(pos, LPFC_NODELIST_t, nlp_listp);
			if (ndlp->nlp_state == NLP_STE_MAPPED_NODE) {
				len += snprintf(buf + len, PAGE_SIZE -len, 
						"lpfc%dt%02x DID %06x WWPN ",
						ndlp->nlp_Target->pHba->brd_no,
						ndlp->nlp_scsi_id, ndlp->nlp_DID);

				/* A Fibre Channel node or port name is 8 octets
				 * long and delimited by colons.
				 */
				memcpy (&name[0], &ndlp->nlp_portname,
					sizeof (NAME_TYPE));
				len += snprintf(buf + len, PAGE_SIZE-len,
						"%02x:%02x:%02x:%02x:%02x:%02x:"
						"%02x:%02x",
						name[0], name[1], name[2],
						name[3], name[4], name[5],
						name[6], name[7]);

				len += snprintf(buf + len, PAGE_SIZE-len,
						" WWNN ");
				memcpy (&name[0], &ndlp->nlp_nodename,
					sizeof (NAME_TYPE));
				len += snprintf(buf + len, PAGE_SIZE-len,
						"%02x:%02x:%02x:%02x:%02x:%02x:"
						"%02x:%02x\n",
						name[0], name[1], name[2],
						name[3], name[4], name[5],
						name[6], name[7]);

			}
			if(PAGE_SIZE - len < 90)
				break;
		}

		if (pos != &phba->pport->fc_nlpmap_list) 
			len += snprintf(buf+len, PAGE_SIZE-len, "...\n");

	}

#ifdef CONFIG_VMNIX
	*start = save;
	retval = MIN(len - offset, length);
	len = 0;
	if (retval > 0) {
		memcpy(save, info_buf + offset, retval);
		len = retval;
	}
#endif

	return (len);
}


int
lpfc_reset_bus_handler(struct scsi_cmnd *cmnd)
{
	lpfcHBA_t *phba;
	LPFCSCSITARGET_t *ptarget;
	unsigned long iflag;
	int ret, i, errcnt;
	lpfc_vport_t *vport; 

	spin_unlock_irq(&io_request_lock);

	vport = (lpfc_vport_t *) cmnd->host->hostdata[0];
	phba = vport->phba;
	errcnt = 0;

	LPFC_DRVR_LOCK(phba, iflag);
	for (i = 0; i < LPFC_MAX_TARGET; i++) {
		ptarget = vport->device_queue_hash[i];
		if (ptarget) {
			ret = lpfc_scsi_tgt_reset(vport, 0, i);
			if (ret != 1)
				errcnt++;
		}
	}
	LPFC_DRVR_UNLOCK(phba, iflag);

	ret = SUCCESS;
	if (errcnt)
		ret = FAILED;

	/* SCSI layer issued Bus Reset */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0714,
		        lpfc_mes0714,
		        lpfc_msgBlk0714.msgPreambleStr,
		        cmnd->target, cmnd->lun, ret);

	spin_lock_irq(&io_request_lock);	
	return ret;
}


void
lpfc_select_queue_depth(struct Scsi_Host *host, struct scsi_device *scsi_devs)
{
	struct scsi_device *device;
	lpfcHBA_t *phba;

	phba = ((lpfc_vport_t *) host->hostdata[0])->phba;
	for (device = scsi_devs; device != 0; device = device->next) {
		if (device->host == host)
			lpfc_device_queue_depth(phba, device);
	}
}

int
lpfc_device_queue_depth(lpfcHBA_t * phba, struct scsi_device *device)
{
	lpfcCfgParam_t *clp;

	clp = &phba->config[0];

	if (device->tagged_supported) {
		device->tagged_queue = 1;
		device->current_tag = 0;
		device->queue_depth = clp[LPFC_CFG_DFT_LUN_Q_DEPTH].a_current;
	} else {
		device->queue_depth = 16;
	}
	return (device->queue_depth);
}



int
lpfc_memmap(lpfcHBA_t * phba)
{
	unsigned long bar0map_len, bar2map_len;

	if (phba->pcidev == 0)
		return (1);

	/* Configure DMA attributes. */
	if (pci_set_dma_mask(phba->pcidev, 0xffffffffffffffffULL)) {
		if (pci_set_dma_mask(phba->pcidev, 0xffffffffULL)) {
			return (1);
		}
	}

	/* 
	 * Get the physical address of Bar0 and Bar2 and the number of bytes
	 * required by each mapping.
	 */
	phba->pci_bar0_map = pci_resource_start(phba->pcidev, 0);
	bar0map_len        = pci_resource_len(phba->pcidev, 0);
	
	phba->pci_bar2_map = pci_resource_start(phba->pcidev, 2);
	bar2map_len        = pci_resource_len(phba->pcidev, 2);

	/* Map HBA SLIM and Control Registers to a kernel virtual address. */
	phba->slim_memmap_p      = ioremap(phba->pci_bar0_map, bar0map_len);
	phba->ctrl_regs_memmap_p = ioremap(phba->pci_bar2_map, bar2map_len);

	/* Setup SLI2 interface */
	if (phba->slim2p.virt == 0) {
		/*
		 * Allocate memory for SLI-2 structures
		 */
		phba->slim2p.virt = pci_alloc_consistent(phba->pcidev,
							 SLI2_SLIM_SIZE,
							 &(phba->slim2p.phys));
		
		if (phba->slim2p.virt == 0) {
			/* Cleanup adapter SLIM and Control Register
			   mappings. */
			iounmap(phba->ctrl_regs_memmap_p);
			iounmap(phba->slim_memmap_p);
			return (1);
		}

		/* The SLIM2 size is stored in the next field */ 
		phba->slim_size = SLI2_SLIM_SIZE;
		memset((char *)phba->slim2p.virt, 0, SLI2_SLIM_SIZE);
		
		phba->hbqslimp.virt = pci_alloc_consistent(phba->pcidev,
							   lpfc_sli_hbq_size(),
							   &(phba->hbqslimp.phys));
		if (!phba->hbqslimp.virt) {
			/* Cleanup adapter SLIM and Control Register
			   mappings. */
			pci_free_consistent(phba->pcidev, 
					    phba->slim_size, 
					    phba->slim2p.virt, 
					    phba->slim2p.phys);
			iounmap(phba->ctrl_regs_memmap_p);
			iounmap(phba->slim_memmap_p);
			return (1);
		}		
		memset((char *)phba->hbqslimp.virt, 0, lpfc_sli_hbq_size());
	}
	return (0);
}

int
lpfc_unmemmap(lpfcHBA_t * phba)
{
	struct pci_dev *pdev;

	pdev = phba->pcidev;

	/* unmap adapter SLIM and Control Registers */
	iounmap(phba->ctrl_regs_memmap_p);
	iounmap(phba->slim_memmap_p);
	
	if (phba->hbqslimp.virt) {
		pci_free_consistent(pdev, lpfc_sli_hbq_size(), 
				    phba->hbqslimp.virt,
				    phba->hbqslimp.phys);
	}
	
	/* Free resources associated with SLI2 interface */
	if (phba->slim2p.virt) {

		
		pci_free_consistent(pdev, 
				    phba->slim_size, 
				    phba->slim2p.virt, 
				    phba->slim2p.phys);

	}
	return (0);
}

int
lpfc_pcimap(lpfcHBA_t * phba)
{
	struct pci_dev *pdev;
	int ret_val;

	/*
	 * PCI for board
	 */
	pdev = phba->pcidev;
	if (!pdev)
		return (1);

	/* The LPFC HBAs are bus-master capable.  Call the kernel and have this
	 * functionality enabled.  Note that setting pci bus master also sets
	 * the latency value as well.  Also turn on MWI so that the cache line
	 * size is set to match the host pci bridge.
	 */

	pci_set_master (pdev);
	ret_val = pci_set_mwi (pdev);
	if (ret_val != 0) {
		/* The mwi set operation failed.  This is not a fatal error
		 * so don't return an error.
		 */
	}

	return (0);
}

void
lpfc_setup_slim_access(lpfcHBA_t * arg)
{
	lpfcHBA_t *phba;

	phba = (lpfcHBA_t *) arg;
	phba->MBslimaddr = phba->slim_memmap_p;
	phba->HAregaddr = (uint32_t *) (phba->ctrl_regs_memmap_p) +
		HA_REG_OFFSET;
	phba->HCregaddr = (uint32_t *) (phba->ctrl_regs_memmap_p) +
		HC_REG_OFFSET;
	phba->CAregaddr = (uint32_t *) (phba->ctrl_regs_memmap_p) +
		CA_REG_OFFSET;
	phba->HSregaddr = (uint32_t *) (phba->ctrl_regs_memmap_p) +
		HS_REG_OFFSET;
	return;
}


uint32_t
lpfc_intr_prep(lpfcHBA_t * phba)
{
	LPFC_SLI_t *psli;
	uint32_t ha_copy;

	/* Ignore all interrupts during initialization. */
	if (phba->pport->port_state < LPFC_LINK_DOWN) {
		return (0);
	}

	psli = &phba->sli;
	/* Read host attention register to determine interrupt source */
	ha_copy = readl(phba->HAregaddr);

	/* Clear Attention Sources, except ERATT (to preserve status) & LATT
	 *    (ha_copy & ~(HA_ERATT | HA_LATT));
	 */
	writel((ha_copy & ~(HA_LATT | HA_ERATT)), phba->HAregaddr);
	readl(phba->HAregaddr);
	return (ha_copy);
}				/* lpfc_intr_prep */

int
lpfc_valid_lun(LPFCSCSITARGET_t * targetp, uint64_t lun)
{
	uint32_t rptLunLen;
	uint32_t *datap32;
	uint32_t lunvalue, i;

	if (targetp->rptLunState != REPORT_LUN_COMPLETE) {
		return 1;
	}

	if (targetp->RptLunData) {
		datap32 = (uint32_t *) targetp->RptLunData->virt;
		rptLunLen = be32_to_cpu(*datap32);

		for (i = 0; i < rptLunLen; i += 8) {
			datap32 += 2;
			lunvalue = (((*datap32) >> FC_LUN_SHIFT) & 0xff);
			if (lunvalue == (uint32_t) lun)
				return 1;
		}
		return 0;
	} else {
		return 1;
	}
}

void
lpfc_nodev_unsol_event(lpfcHBA_t * phba,
		      LPFC_SLI_RING_t * pring, LPFC_IOCBQ_t * piocbq)
{
	return;
}

void
lpfc_sli_async_event_handler(lpfcHBA_t *phba, LPFC_SLI_RING_t *pring,
			     LPFC_IOCBQ_t *piocbq)
{
	IOCB_t *icmd;
	uint16_t evt_code;
	unsigned long temp;
	uint32_t evt_type = 0;

	icmd = &piocbq->iocb;
	evt_code = icmd->un.asyncstat.evt_code;
	temp = icmd->ulpContext;

	if ((evt_code != ASYNC_TEMP_WARN) &&
		(evt_code != ASYNC_TEMP_SAFE)) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0329,
				lpfc_mes0329,
			        lpfc_msgBlk0329.msgPreambleStr,
				pring->ringno, icmd->un.asyncstat.evt_code);
		return;
	}

	if (evt_code == ASYNC_TEMP_WARN) {
	 	evt_type = LPFC_THRESHOLD_TEMP;
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0339,
				lpfc_mes0339,
			        lpfc_msgBlk0339.msgPreambleStr,
				temp);
	}

	if (evt_code == ASYNC_TEMP_SAFE) {
		evt_type = LPFC_NORMAL_TEMP;
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0340,
				lpfc_mes0340,
			        lpfc_msgBlk0340.msgPreambleStr,
				temp);
	}

	/* Send temperature change event to applications */
	lpfc_put_event(phba, HBA_EVENT_TEMP, evt_type,
			(void *)temp, 0);
}

int
lpfc_sli_setup(lpfcHBA_t * phba)
{
	int i, totiocbsize = 0;
	LPFC_SLI_t *psli;
	LPFC_RING_INIT_t *pring;
	lpfcCfgParam_t *clp;

	psli = &phba->sli;
	psli->sliinit.num_rings = MAX_CONFIGURED_RINGS;
	psli->fcp_ring = LPFC_FCP_RING;
	psli->next_ring = LPFC_FCP_NEXT_RING;
	psli->extra_ring = LPFC_EXTRA_RING;

	clp = &phba->config[0];

	for (i = 0; i < psli->sliinit.num_rings; i++) {
		pring = &psli->sliinit.ringinit[i];
		switch (i) {
		case LPFC_FCP_RING:	/* ring 0 - FCP */
			/* numCiocb and numRiocb are used in config_port */
			pring->numCiocb = SLI2_IOCB_CMD_R0_ENTRIES;
			pring->numRiocb = SLI2_IOCB_RSP_R0_ENTRIES;
			pring->numCiocb += SLI2_IOCB_CMD_R1XTRA_ENTRIES;
			pring->numRiocb += SLI2_IOCB_RSP_R1XTRA_ENTRIES;
			pring->numCiocb += SLI2_IOCB_CMD_R3XTRA_ENTRIES;
			pring->numRiocb += SLI2_IOCB_RSP_R3XTRA_ENTRIES;

 			pring->sizeCiocb = (phba->sli_rev == 3) ? SLI3_IOCB_CMD_SIZE : SLI2_IOCB_CMD_SIZE;
			pring->sizeRiocb = (phba->sli_rev == 3) ? SLI3_IOCB_RSP_SIZE : SLI2_IOCB_RSP_SIZE;

			pring->iotag_ctr = 0;
			pring->iotag_max =
			    (clp[LPFC_CFG_DFT_HBA_Q_DEPTH].a_current * 2);
			pring->fast_iotag = pring->iotag_max;
			pring->num_mask = 0;
			break;
		case LPFC_EXTRA_RING:	/* ring 1 - IP */
			/* numCiocb and numRiocb are used in config_port */
			pring->numCiocb = SLI2_IOCB_CMD_R1_ENTRIES;
			pring->numRiocb = SLI2_IOCB_RSP_R1_ENTRIES;

			pring->sizeCiocb = (phba->sli_rev == 3) ? SLI3_IOCB_CMD_SIZE : SLI2_IOCB_CMD_SIZE;
			pring->sizeRiocb = (phba->sli_rev == 3) ? SLI3_IOCB_RSP_SIZE : SLI2_IOCB_RSP_SIZE;

			pring->num_mask = 0;
			pring->iotag_ctr = 0;
			pring->iotag_max = clp[LPFC_CFG_XMT_Q_SIZE].a_current;
			pring->fast_iotag = 0;
			break;
		case LPFC_ELS_RING:	/* ring 2 - ELS / CT */
			/* numCiocb and numRiocb are used in config_port */
			pring->numCiocb = SLI2_IOCB_CMD_R2_ENTRIES;
			pring->numRiocb = SLI2_IOCB_RSP_R2_ENTRIES;

			pring->sizeCiocb = (phba->sli_rev == 3) ? SLI3_IOCB_CMD_SIZE : SLI2_IOCB_CMD_SIZE;
			pring->sizeRiocb = (phba->sli_rev == 3) ? SLI3_IOCB_RSP_SIZE : SLI2_IOCB_RSP_SIZE;

			pring->fast_iotag = 0;
			pring->iotag_ctr = 0;
			pring->iotag_max = 4096;
			pring->lpfc_sli_rcv_async_status =
				lpfc_sli_async_event_handler;
			pring->num_mask = 5;
			pring->prt[0].profile = 0;	/* Mask 0 */
			pring->prt[0].rctl = FC_ELS_REQ;
			pring->prt[0].type = FC_ELS_DATA;
			pring->prt[0].lpfc_sli_rcv_unsol_event =
			    lpfc_els_unsol_event;
			pring->prt[1].profile = 0;	/* Mask 1 */
			pring->prt[1].rctl = FC_ELS_RSP;
			pring->prt[1].type = FC_ELS_DATA;
			pring->prt[1].lpfc_sli_rcv_unsol_event =
			    lpfc_els_unsol_event;
			pring->prt[2].profile = 0;	/* Mask 2 */
			/* NameServer Inquiry */
			pring->prt[2].rctl = FC_UNSOL_CTL;
			/* NameServer */
			pring->prt[2].type = FC_COMMON_TRANSPORT_ULP;
			pring->prt[2].lpfc_sli_rcv_unsol_event =
			    lpfc_ct_unsol_event;
			pring->prt[3].profile = 0;	/* Mask 3 */
			/* NameServer response */
			pring->prt[3].rctl = FC_SOL_CTL;
			/* NameServer */
			pring->prt[3].type = FC_COMMON_TRANSPORT_ULP;
			pring->prt[3].lpfc_sli_rcv_unsol_event =
			    lpfc_ct_unsol_event;
			pring->prt[4].profile = 0;      /* Mask 4 */
			pring->prt[4].rctl = FC_UNSOL_DATA;
			pring->prt[4].type = FC_VENDOR_SPECIFIC;
			pring->prt[4].lpfc_sli_rcv_unsol_event =
				lpfc_loopback_event;
			break;
		}

		totiocbsize += (pring->numCiocb * pring->sizeCiocb) + (pring->numRiocb * pring->sizeRiocb);
	}

	if (totiocbsize > MAX_SLIM_IOCB_SIZE) {
		/* Too many cmd / rsp ring entries in SLI2 SLIM */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0462, lpfc_mes0462,
			        lpfc_msgBlk0462.msgPreambleStr, totiocbsize,
				MAX_SLIM_IOCB_SIZE);
	}

	psli->sliinit.sli_flag = 0;

	return (0);
}

irqreturn_t
lpfc_intr_handler(int irq, void *dev_id, struct pt_regs * regs)
{
	lpfcHBA_t *phba;
	unsigned long iflag;

	/* Sanity check dev_id parameter */
	phba = (lpfcHBA_t *) dev_id;
	if (!phba) {
		return IRQ_NONE;
	}

	/* More sanity checks on dev_id parameter.
	 * We have seen our interrupt service routine being called
	 * with the dev_id of another PCI card in the system.
	 * Here we verify the dev_id is really ours!
	 */
	if(!lpfc_check_valid_phba(phba))
		return IRQ_NONE;

	LPFC_DRVR_LOCK(phba, iflag);

	/* Call SLI Layer to process interrupt */
	lpfc_sli_intr(phba);

	LPFC_DRVR_UNLOCK(phba, iflag);
	return IRQ_HANDLED;
} /* lpfc_intr_handler */

int
lpfc_bind_setup(lpfc_vport_t *vport)
{
	lpfcCfgParam_t *clp;
	char **arrayp = 0;
	u_int cnt = 0;
	lpfcHBA_t *phba = vport->phba;

	/* 
	 * Check if there are any WWNN / scsid bindings
	 */
	clp = &phba->config[0];

	lpfc_get_bind_type(phba);

	switch (phba->fcp_mapping) {
	case FCP_SEED_WWNN:
		arrayp = lpfc_fcp_bind_WWNN;
		cnt = 0;
		while (arrayp[cnt] != 0)
			cnt++;
		if (cnt && (*arrayp != 0)) {
			lpfc_bind_wwnn(vport, arrayp, cnt);
		}
		break;

	case FCP_SEED_WWPN:
		arrayp = lpfc_fcp_bind_WWPN;
		cnt = 0;
		while (arrayp[cnt] != 0)
			cnt++;
		if (cnt && (*arrayp != 0)) {
			lpfc_bind_wwpn(vport, arrayp, cnt);
		}
		break;

	case FCP_SEED_DID:
		if (clp[LPFC_CFG_BINDMETHOD].a_current != 4) {
			arrayp = lpfc_fcp_bind_DID;
			cnt = 0;
			while (arrayp[cnt] != 0)
				cnt++;
			if (cnt && (*arrayp != 0)) {
				lpfc_bind_did(vport, arrayp, cnt);
			}
		}
		break;
	}

	if (cnt && (*arrayp != 0) &&
	    (clp[LPFC_CFG_BINDMETHOD].a_current == 4)) {
		/* Using ALPA map with Persistent binding - ignoring ALPA map */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0411,
			        lpfc_mes0411, lpfc_msgBlk0411.msgPreambleStr,
			        clp[LPFC_CFG_BINDMETHOD].a_current,
				phba->fcp_mapping);
	}

	if (clp[LPFC_CFG_SCAN_DOWN].a_current > 1) {
		/* Scan-down is out of range - ignoring scan-down */
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0412,
			        lpfc_mes0412, lpfc_msgBlk0412.msgPreambleStr,
			        clp[LPFC_CFG_BINDMETHOD].a_current,
				phba->fcp_mapping);
		clp[LPFC_CFG_SCAN_DOWN].a_current = 0;
	}
	return (0);
}

/******************************************************************************
* Function name : lpfc_config_setup
*
* Description   : Called from attach to setup configuration parameters for 
*                 adapter 
*                 The goal of this routine is to fill in all the a_current 
*                 members of the CfgParam structure for all configuration 
*                 parameters.
* Example:
* clp[LPFC_CFG_XXX].a_current = (uint32_t)value;
* value might be a define, a global variable, clp[LPFC_CFG_XXX].a_default,
* or some other enviroment specific way of initializing config parameters.
******************************************************************************/

int
lpfc_config_setup(lpfcHBA_t * phba)
{
	lpfcCfgParam_t *clp;
	LPFC_SLI_t *psli;
	int i;
	int brd;

	clp = &phba->config[0];
	psli = &phba->sli;
	brd = phba->brd_no;

	/*
	 * Read the configuration parameters. Also set to default if
	 * parameter value is out of allowed range.
	 */
	for (i = 0; i < LPFC_TOTAL_NUM_OF_CFG_PARAM; i++) {
		clp[i].a_current = fc_get_cfg_param(brd, i);

		if ((clp[i].a_current >= clp[i].a_low) &&
		    (clp[i].a_current <= clp[i].a_hi)) {
			/*
			 * LPFC_CFG_TOPOLOGY has holes in the value range and
			 * LPFC_CFG_FCP_CLASS needs to readjusted when it does
			 * satisfy the range check
			 */
			if (i == LPFC_CFG_TOPOLOGY) {
				/* odd values 1,3,5 are out */
				if (!(clp[i].a_current & 1))
					continue;
			} else if (i == LPFC_CFG_FCP_CLASS) {
				switch (clp[i].a_current) {
				case 2:
					/* CLASS2 = 1 */
					clp[i].a_current = CLASS2;
					break;
				case 3:
					/* CLASS3 = 2 */
					clp[i].a_current = CLASS3;
					break;
				}
				continue;
			} else
				continue;
		}

		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0413,
				lpfc_mes0413,
				lpfc_msgBlk0413.msgPreambleStr,
				clp[i].a_string, clp[i].a_low, clp[i].a_hi,
				clp[i].a_default);

		clp[i].a_current = clp[i].a_default;

		/*
		 * The PCI MAX READ value has to default back to 
		 * runtime behavior because PCI-X HBAs are limited
		 * to 1024 DMA Read bytes. 
		 */
		if (i == LPFC_CFG_PCI_MAX_READ)
		        clp[i].a_current = 0;
	}

	phba->sli.ring[LPFC_EXTRA_RING].txq_max = clp[LPFC_CFG_XMT_Q_SIZE].a_current;

	return 0;
}

int
lpfc_bind_wwpn(lpfc_vport_t * vport, char **arrayp, u_int cnt)
{
	uint8_t *datap, *np;
	LPFC_BINDLIST_t *blp;
	NAME_TYPE pn;
	int i, entry, lpfc_num, rstatus;
	unsigned int sum;
	lpfcHBA_t *phba = vport->phba;

	phba->fcp_mapping = FCP_SEED_WWPN;
	np = (uint8_t *) & pn;

	for (entry = 0; entry < cnt; entry++) {
		datap = (uint8_t *) arrayp[entry];
		if (datap == 0)
			break;
		/* Determined the number of ASC hex chars in WWNN & WWPN */
		for (i = 0; i < FC_MAX_WW_NN_PN_STRING; i++) {
			if (!isxdigit(datap[i]))
				break;
		}
		if ((rstatus = lpfc_parse_binding_entry(phba, datap, np,
							i, sizeof (NAME_TYPE),
							LPFC_BIND_WW_NN_PN,
							&sum, entry,
							&lpfc_num)) > 0) {
			if (rstatus == LPFC_SYNTAX_OK_BUT_NOT_THIS_BRD)
				continue;

			/* For syntax error code definitions see
			   LPFC_SYNTAX_ERR_ defines. */
			/* WWPN binding entry <num>: Syntax error code <code> */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0430,
					lpfc_mes0430,
					lpfc_msgBlk0430.msgPreambleStr,
					entry, rstatus);
			goto out;
		}

		/* Loop through all BINDLIST entries and find
		 * the next available entry.
		 */
		if ((blp = lpfc_bind_alloc(phba, 0)) == 0) {
			/* WWPN binding entry: node table full */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0432,
					lpfc_mes0432,
					lpfc_msgBlk0432.msgPreambleStr);
			goto out;
		}
		memset(blp, 0, sizeof (LPFC_BINDLIST_t));
		blp->nlp_bind_type = FCP_SEED_WWPN;
		blp->nlp_scsi_id = (sum & 0xff);
		memcpy(&blp->nlp_portname, (uint8_t *) & pn,
		       sizeof (NAME_TYPE));

		lpfc_nlp_bind(vport, blp);

	      out:
		np = (uint8_t *) & pn;
	}
	return (0);
}				/* lpfc_bind_wwpn */

int
lpfc_get_bind_type(lpfcHBA_t * phba)
{
	int bind_type;
	lpfcCfgParam_t *clp;

	clp = &phba->config[0];

	bind_type = clp[LPFC_CFG_BINDMETHOD].a_current;

	switch (bind_type) {
	case 1:
		phba->fcp_mapping = FCP_SEED_WWNN;
		break;

	case 2:
		phba->fcp_mapping = FCP_SEED_WWPN;
		break;

	case 3:
		phba->fcp_mapping = FCP_SEED_DID;
		break;

	case 4:
		phba->fcp_mapping = FCP_SEED_DID;
		break;
	}

	return 0;
}

int
lpfc_bind_wwnn(lpfc_vport_t * vport, char **arrayp, u_int cnt)
{
	uint8_t *datap, *np;
	LPFC_BINDLIST_t *blp;
	NAME_TYPE pn;
	int i, entry, lpfc_num, rstatus;
	unsigned int sum;
	lpfcHBA_t * phba = vport->phba;

	phba->fcp_mapping = FCP_SEED_WWNN;
	np = (uint8_t *) & pn;

	for (entry = 0; entry < cnt; entry++) {
		datap = (uint8_t *) arrayp[entry];
		if (datap == 0)
			break;
		/* Determined the number of ASC hex chars in WWNN & WWPN */
		for (i = 0; i < FC_MAX_WW_NN_PN_STRING; i++) {
			if (!isxdigit(datap[i]))
				break;
		}
		if ((rstatus = lpfc_parse_binding_entry(phba, datap, np,
							i, sizeof (NAME_TYPE),
							LPFC_BIND_WW_NN_PN,
							&sum, entry,
							&lpfc_num)) > 0) {
			if (rstatus == LPFC_SYNTAX_OK_BUT_NOT_THIS_BRD) {
				continue;
			}

			/* For syntax error code definitions see
			   LPFC_SYNTAX_ERR_ defines. */
			/* WWNN binding entry <num>: Syntax error code <code> */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0431,
					lpfc_mes0431,
					lpfc_msgBlk0431.msgPreambleStr,
					entry, rstatus);
			goto out;
		}

		/* Loop through all BINDLIST entries and find
		 * the next available entry.
		 */
		if ((blp = lpfc_bind_alloc(phba, 0)) == 0) {
			/* WWNN binding entry: node table full */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0433,
					lpfc_mes0433,
					lpfc_msgBlk0433.msgPreambleStr);
			goto out;
		}
		memset(blp, 0, sizeof (LPFC_BINDLIST_t));
		blp->nlp_bind_type = FCP_SEED_WWNN;
		blp->nlp_scsi_id = (sum & 0xff);
		memcpy(&blp->nlp_nodename, (uint8_t *) & pn,
		       sizeof (NAME_TYPE));
		lpfc_nlp_bind(vport, blp);

	      out:
		np = (uint8_t *) & pn;
	}			/* for loop */
	return (0);
}				/* lpfc_bind_wwnn */

int
lpfc_bind_did(lpfc_vport_t * vport, char **arrayp, u_int cnt)
{
	uint8_t *datap, *np;
	LPFC_BINDLIST_t *blp;
	D_ID ndid;
	int i, entry, lpfc_num, rstatus;
	unsigned int sum;
	lpfcHBA_t * phba = vport->phba;

	phba->fcp_mapping = FCP_SEED_DID;
	ndid.un.word = 0;
	np = (uint8_t *) & ndid.un.word;

	for (entry = 0; entry < cnt; entry++) {
		datap = (uint8_t *) arrayp[entry];
		if (datap == 0)
			break;
		/* Determined the number of ASC hex chars in DID */
		for (i = 0; i < FC_MAX_DID_STRING; i++) {
			if (!isxdigit(datap[i]))
				break;
		}
		if ((rstatus = lpfc_parse_binding_entry(phba, datap, np,
							i, sizeof (D_ID),
							LPFC_BIND_DID, &sum,
							entry,
							&lpfc_num)) > 0) {
			if (rstatus == LPFC_SYNTAX_OK_BUT_NOT_THIS_BRD)
				continue;

			/* For syntax error code definitions see
			   LPFC_SYNTAX_ERR_ defines. */
			/* DID binding entry <num>: Syntax error code <code> */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0434,
				        lpfc_mes0434,
				        lpfc_msgBlk0434.msgPreambleStr,
				        entry, rstatus);
			goto out;
		}

		/* Loop through all BINDLIST entries and find
		 * the next available entry.
		 */
		if ((blp = lpfc_bind_alloc(phba, 0)) == 0) {
			/* DID binding entry: node table full */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0435,
				        lpfc_mes0435,
				        lpfc_msgBlk0435.msgPreambleStr);
			goto out;
		}
		memset(blp, 0, sizeof (LPFC_BINDLIST_t));
		blp->nlp_bind_type = FCP_SEED_DID;
		blp->nlp_scsi_id = (sum & 0xff);
		blp->nlp_DID = be32_to_cpu(ndid.un.word);

		lpfc_nlp_bind(vport, blp);

	      out:

		np = (uint8_t *) & ndid.un.word;
	}
	return (0);
}

void
lpfc_wakeup_event(lpfcHBA_t * phba, fcEVTHDR_t * ep)
{
	ep->e_mode &= ~E_SLEEPING_MODE;
	switch (ep->e_mask) {
	case FC_REG_LINK_EVENT:
		wake_up_interruptible(&phba->linkevtwq);
		break;
	case FC_REG_RSCN_EVENT:
		wake_up_interruptible(&phba->rscnevtwq);
		break;
	case FC_REG_CT_EVENT:
		wake_up_interruptible(&phba->ctevtwq);
		break;
	case FC_REG_TEMPERATURE_EVENT:
		wake_up_interruptible(&phba->tempevtwq);
		break;
	}
	
	return;
}

int
lpfc_put_event(lpfcHBA_t * phba,
	      uint32_t evcode, uint32_t evdata0, void *evdata1, void *evdata2)
{
	fcEVT_t *ep;
	fcEVT_t *oep;
	fcEVTHDR_t *ehp = 0;
	int found;
	DMABUF_t *mp;
	void *fstype;
	SLI_CT_REQUEST *ctp;

	ehp = (fcEVTHDR_t *) phba->fc_evt_head;
	fstype = 0;
	switch (evcode) {
	case FC_REG_CT_EVENT:
		mp = (DMABUF_t *) evdata1;
		ctp = (SLI_CT_REQUEST *) mp->virt;
		fstype = (void *)(ulong) (ctp->FsType);
		break;
	}

	while (ehp) {
		if ((ehp->e_mask == evcode) && (ehp->e_type == fstype))
			break;
		
		ehp = (fcEVTHDR_t *) ehp->e_next_header;
	}

	if (!ehp) {
		return (0);
	}

	ep = ehp->e_head;
	oep = 0;
	found = 0;

	while (ep && !(found)) {
		if (ep->evt_sleep) {
			switch (evcode) {
			case FC_REG_CT_EVENT:
				if ((ep->evt_type ==
				     (void *)(ulong) FC_FSTYPE_ALL)
				    || (ep->evt_type == fstype)) {
					found++;
					ep->evt_data0 = evdata0; /* tag */
					ep->evt_data1 = evdata1; /* buffer
								    ptr */
					ep->evt_data2 = evdata2; /* count */
					ep->evt_sleep = 0;
					if (ehp->e_mode & E_SLEEPING_MODE) {
						ehp->e_flag |=
						    E_GET_EVENT_ACTIVE;
						lpfc_wakeup_event(phba, ehp);
					}
					/* For FC_REG_CT_EVENT just give it to
					   first one found */
				}
				break;
			default:
				found++;
				ep->evt_data0 = evdata0;
				ep->evt_data1 = evdata1;
				ep->evt_data2 = evdata2;
				ep->evt_sleep = 0;
				if ((ehp->e_mode & E_SLEEPING_MODE)
				    && !(ehp->e_flag & E_GET_EVENT_ACTIVE)) {
					ehp->e_flag |= E_GET_EVENT_ACTIVE;
					lpfc_wakeup_event(phba, ehp);
				}
				/* For all other events, give it to every one
				   waiting */
				break;
			}
		}
		oep = ep;
		ep = ep->evt_next;
	}
	return (found);
}

int
lpfc_hba_put_event(lpfcHBA_t * phba, uint32_t evcode, uint32_t evdata1,
		  uint32_t evdata2, uint32_t evdata3, uint32_t evdata4)
{
	HBAEVT_t *rec;

	rec = &phba->hbaevt[phba->hba_event_put];
	rec->fc_eventcode = evcode;

	rec->fc_evdata1 = evdata1;
	rec->fc_evdata2 = evdata2;
	rec->fc_evdata3 = evdata3;
	rec->fc_evdata4 = evdata4;
	phba->hba_event_put++;
	if (phba->hba_event_put >= MAX_HBAEVT) {
		phba->hba_event_put = 0;
	}
	if (phba->hba_event_put == phba->hba_event_get) {
		phba->hba_event_missed++;
		phba->hba_event_get++;
		if (phba->hba_event_get >= MAX_HBAEVT) {
			phba->hba_event_get = 0;
		}
	}

	return (0);
}


LPFCSCSILUN_t *
lpfc_tran_find_lun(LPFC_SCSI_BUF_t * lpfc_cmd)
{
	lpfc_vport_t *vport;
	LPFCSCSILUN_t *lunp;

	vport = lpfc_cmd->vport;
	lunp = lpfc_find_lun(vport, lpfc_cmd->scsi_target, lpfc_cmd->scsi_lun, 1);
	return (lunp);
}

int
lpfc_utsname_nodename_check(void)
{
#ifndef __VMKERNEL_MODULE__
	if (system_utsname.nodename[0] == '\0')
		return (1);
#endif

	return (0);
}



#define HBA_SPECIFIC_CFG_PARAM(hba)                                           \
	switch (param) {                                                      \
	case LPFC_CFG_LOG_VERBOSE:	/* log-verbose */                     \
		value = lpfc_log_verbose;                                     \
		if (lpfc##hba##_log_verbose != -1)                            \
			value = lpfc##hba##_log_verbose;                      \
		break;                                                        \
	case LPFC_CFG_AUTOMAP:	/* automap */                                 \
		value = lpfc_automap;                                         \
		if (lpfc##hba##_automap != -1)                                \
			value = lpfc##hba##_automap;                          \
		break;                                                        \
	case LPFC_CFG_BINDMETHOD:	/* bind-method */                     \
		value = lpfc_fcp_bind_method;                                 \
		if (lpfc##hba##_fcp_bind_method != -1)                        \
			value = lpfc##hba##_fcp_bind_method;                  \
		break;                                                        \
	case LPFC_CFG_CR_DELAY:	/* cr_delay */                                \
		value = lpfc_cr_delay;                                        \
		if (lpfc##hba##_cr_delay != -1)                               \
			value = lpfc##hba##_cr_delay;                         \
		break;                                                        \
	case LPFC_CFG_CR_COUNT:	/* cr_count */                                \
		value = lpfc_cr_count;                                        \
		if (lpfc##hba##_cr_count != -1)                               \
			value = lpfc##hba##_cr_count;                         \
		break;                                                        \
	case LPFC_CFG_DFT_TGT_Q_DEPTH:	/* tgt_queue_depth */                 \
		value = lpfc_tgt_queue_depth;                                 \
		if (lpfc##hba##_tgt_queue_depth != -1)                        \
			value = lpfc##hba##_tgt_queue_depth;                  \
		break;                                                        \
	case LPFC_CFG_DFT_LUN_Q_DEPTH:	/* lun_queue_depth */                 \
		value = lpfc_lun_queue_depth;                                 \
		if (lpfc##hba##_lun_queue_depth != -1)                        \
			value = lpfc##hba##_lun_queue_depth;                  \
		break;                                                        \
	case LPFC_CFG_EXTRA_IO_TMO:	/* fcpfabric-tmo */                   \
		value = lpfc_extra_io_tmo;                                    \
		if (lpfc##hba##_extra_io_tmo != -1)                           \
			value = lpfc##hba##_extra_io_tmo;                     \
		break;                                                        \
	case LPFC_CFG_FCP_CLASS:	/* fcp-class */                       \
		value = lpfc_fcp_class;                                       \
		if (lpfc##hba##_fcp_class != -1)                              \
			value = lpfc##hba##_fcp_class;                        \
		break;                                                        \
	case LPFC_CFG_USE_ADISC:	/* use-adisc */                       \
		value = lpfc_use_adisc;                                       \
		if (lpfc##hba##_use_adisc != -1)                              \
			value = lpfc##hba##_use_adisc;                        \
		break;                                                        \
	case LPFC_CFG_NO_DEVICE_DELAY:	/* no-device-delay */                 \
		value = lpfc_no_device_delay;                                 \
		if (lpfc##hba##_no_device_delay != -1)                        \
			value = lpfc##hba##_no_device_delay;                  \
		break;                                                        \
	case LPFC_CFG_XMT_Q_SIZE:	/* xmt-que-size */                    \
		value = lpfc_xmt_que_size;                                    \
		if (lpfc##hba##_xmt_que_size != -1)                           \
			value = lpfc##hba##_xmt_que_size;                     \
		break;                                                        \
	case LPFC_CFG_ACK0:	/* ack0 */                                    \
		value = lpfc_ack0;                                            \
		if (lpfc##hba##_ack0 != -1)                                   \
			value = lpfc##hba##_ack0;                             \
		break;                                                        \
	case LPFC_CFG_TOPOLOGY:	/* topology */                                \
		value = lpfc_topology;                                        \
		if (lpfc##hba##_topology != -1)                               \
			value = lpfc##hba##_topology;                         \
		break;                                                        \
	case LPFC_CFG_SCAN_DOWN:	/* scan-down */                       \
		value = lpfc_scan_down;                                       \
		if (lpfc##hba##_scan_down != -1)                              \
			value = lpfc##hba##_scan_down;                        \
		break;                                                        \
	case LPFC_CFG_LINKDOWN_TMO:	/* linkdown-tmo */                    \
		value = lpfc_linkdown_tmo;                                    \
		if (lpfc##hba##_linkdown_tmo != -1)                           \
			value = lpfc##hba##_linkdown_tmo;                     \
		break;                                                        \
	case LPFC_CFG_HOLDIO:	/* nodev-holdio */                            \
		value = lpfc_nodev_holdio;                                    \
		if (lpfc##hba##_nodev_holdio != -1)                           \
			value = lpfc##hba##_nodev_holdio;                     \
		break;                                                        \
	case LPFC_CFG_DELAY_RSP_ERR:	/* delay-rsp-err */                   \
		value = lpfc_delay_rsp_err;                                   \
		if (lpfc##hba##_delay_rsp_err != -1)                          \
			value = lpfc##hba##_delay_rsp_err;                    \
		break;                                                        \
	case LPFC_CFG_CHK_COND_ERR:	/* check-cond-err */                  \
		value = lpfc_check_cond_err;                                  \
		if (lpfc##hba##_check_cond_err != -1)                         \
			value = lpfc##hba##_check_cond_err;                   \
		break;                                                        \
	case LPFC_CFG_NODEV_TMO:	/* nodev-tmo */                       \
		value = lpfc_nodev_tmo;                                       \
		if (lpfc##hba##_nodev_tmo != -1)                              \
			value = lpfc##hba##_nodev_tmo;                        \
		break;                                                        \
	case LPFC_CFG_LINK_SPEED:	/* link-speed */                      \
		value = lpfc_link_speed;                                      \
		if (lpfc##hba##_link_speed != -1)                             \
			value = lpfc##hba##_link_speed;                       \
		break;                                                        \
	case LPFC_CFG_DQFULL_THROTTLE_UP_TIME:	/* dqfull-throttle-up-time */ \
		value = lpfc_dqfull_throttle_up_time;                         \
		if (lpfc##hba##_dqfull_throttle_up_time != -1)                \
			value = lpfc##hba##_dqfull_throttle_up_time;          \
		break;                                                        \
	case LPFC_CFG_DQFULL_THROTTLE_UP_INC:	/* dqfull-throttle-up-inc */  \
		value = lpfc_dqfull_throttle_up_inc;                          \
		if (lpfc##hba##_dqfull_throttle_up_inc != -1)                 \
			value = lpfc##hba##_dqfull_throttle_up_inc;           \
		break;                                                        \
	case LPFC_CFG_FDMI_ON:	/* fdmi-on */                                 \
		value = lpfc_fdmi_on;                                         \
		if (lpfc##hba##_fdmi_on != -1)                                \
			value = lpfc##hba##_fdmi_on;                          \
		break;                                                        \
	case LPFC_CFG_MAX_LUN:	/* max-lun */                                 \
		value = lpfc_max_lun;                                         \
		if (lpfc##hba##_max_lun != -1)                                \
			value = lpfc##hba##_max_lun;                          \
		break;                                                        \
	case LPFC_CFG_DISC_THREADS:	/* discovery-threads */               \
		value = lpfc_discovery_threads;                               \
		if (lpfc##hba##_discovery_threads != -1)                      \
			value = lpfc##hba##_discovery_threads;                \
		break;                                                        \
	case LPFC_CFG_MAX_TARGET:	/* max-target */                      \
		value = lpfc_max_target;                                      \
		if (lpfc##hba##_max_target != -1)                             \
			value = lpfc##hba##_max_target;                       \
		break;                                                        \
	case LPFC_CFG_SCSI_REQ_TMO:	/* scsi-req-tmo */                    \
		value = lpfc_scsi_req_tmo;                                    \
		if (lpfc##hba##_scsi_req_tmo != -1)                           \
			value = lpfc##hba##_scsi_req_tmo;                     \
		break;                                                        \
	case LPFC_CFG_LUN_SKIP:	/* lun-skip */                                \
		value = lpfc_lun_skip;                                        \
		if (lpfc##hba##_lun_skip != -1)                               \
			value = lpfc##hba##_lun_skip;                         \
		break;                                                        \
	case LPFC_CFG_PCI_MAX_READ:                                           \
		value = lpfc_pci_max_read;                                    \
		if (lpfc##hba##_pci_max_read != -1)                           \
			value = lpfc##hba##_pci_max_read;                     \
		break;                                                        \
	case LPFC_CFG_MAX_VPI:	/* NPIV */                                    \
		value = lpfc_max_vpi;                                         \
		if (lpfc##hba##_max_vpi != -1)                                \
			value = lpfc##hba##_max_vpi;                          \
		break;                                                        \
	case LPFC_CFG_PEER_VPORT_LOGIN:                                       \
		value = lpfc_peer_vport_login;                                \
		if (lpfc##hba##_peer_vport_login != -1)                       \
			value = lpfc##hba##_peer_vport_login;                 \
		break;                                                        \
	case LPFC_CFG_INITIATOR_LOGIN:                                        \
		value = lpfc_initiator_login;                                 \
		if (lpfc##hba##_initiator_login != -1)                        \
			value = lpfc##hba##_initiator_login;                  \
		break;                                                        \
	case LPFC_CFG_NS_THREADS:                                             \
		value = lpfc_ns_threads;                                      \
		if (lpfc##hba##_ns_threads != -1)                             \
			value = lpfc##hba##_ns_threads;                       \
		break;                                                        \
	case LPFC_CFG_WATCHDOG_TMO:  	                                      \
		value = lpfc_iocb_wdog_tmo;                                   \
		if (lpfc##hba##_iocb_wdog_tmo != -1)                          \
			value = lpfc##hba##_iocb_wdog_tmo;    		      \
		break;                                                        \
        case LPFC_CFG_DFT_HBA_Q_DEPTH:                                        \
                value = lpfc_hba_queue_depth;                                 \
                if (lpfc##hba##_hba_queue_depth != -1)                        \
                        value = lpfc##hba##_hba_queue_depth;                  \
                break;                                                        \
        case LPFC_CFG_ENABLE_RESET:                                           \
                value = lpfc_enable_hba_reset;                                \
                if (lpfc##hba##_enable_hba_reset != -1)                       \
                        value = lpfc##hba##_enable_hba_reset;                 \
                break;                                                        \
        case LPFC_CFG_ENABLE_HEARTBEAT:                                       \
                value = lpfc_enable_hba_heartbeat;                            \
                if (lpfc##hba##_enable_hba_heartbeat != -1)                   \
                        value = lpfc##hba##_enable_hba_heartbeat;             \
                break;                                                        \
        case LPFC_CFG_SLI_MODE:                                               \
                value = lpfc_sli_mode;                                        \
                if (lpfc##hba##_sli_mode != -1)                               \
                        value = lpfc##hba##_sli_mode;                         \
                break;                                                        \
	default:                                                              \
		break;                                                        \
	}

uint32_t
fc_get_cfg_param(int brd, int param)
{
	uint32_t value = (uint32_t)-1;

	switch (brd) {
	case 0:		/* HBA 0 */
		HBA_SPECIFIC_CFG_PARAM(0);
		break;
	case 1:		/* HBA 1 */
		HBA_SPECIFIC_CFG_PARAM(1);
		break;
	case 2:		/* HBA 2 */
		HBA_SPECIFIC_CFG_PARAM(2);
		break;
	case 3:		/* HBA 3 */
		HBA_SPECIFIC_CFG_PARAM(3);
		break;
	case 4:		/* HBA 4 */
		HBA_SPECIFIC_CFG_PARAM(4);
		break;
	case 5:		/* HBA 5 */
		HBA_SPECIFIC_CFG_PARAM(5);
		break;
	case 6:		/* HBA 6 */
		HBA_SPECIFIC_CFG_PARAM(6);
		break;
	case 7:		/* HBA 7 */
		HBA_SPECIFIC_CFG_PARAM(7);
		break;
	case 8:		/* HBA 8 */
		HBA_SPECIFIC_CFG_PARAM(8);
		break;
	case 9:		/* HBA 9 */
		HBA_SPECIFIC_CFG_PARAM(9);
		break;
	case 10:	/* HBA 10 */
		HBA_SPECIFIC_CFG_PARAM(10);
		break;
	case 11:	/* HBA 11 */
		HBA_SPECIFIC_CFG_PARAM(11);
		break;
	case 12:	/* HBA 12 */
		HBA_SPECIFIC_CFG_PARAM(12);
		break;
	case 13:	/* HBA 13 */
		HBA_SPECIFIC_CFG_PARAM(13);
		break;
	case 14:	/* HBA 14 */
		HBA_SPECIFIC_CFG_PARAM(14);
		break;
	case 15:	/* HBA 15 */
		HBA_SPECIFIC_CFG_PARAM(15);
		break;
	case 16:	/* HBA 16 */
		HBA_SPECIFIC_CFG_PARAM(16);
		break;
	case 17:	/* HBA 17 */
		HBA_SPECIFIC_CFG_PARAM(17);
		break;
	case 18:	/* HBA 18 */
		HBA_SPECIFIC_CFG_PARAM(18);
		break;
	case 19:	/* HBA 19 */
		HBA_SPECIFIC_CFG_PARAM(19);
		break;
	case 20:	/* HBA 20 */
		HBA_SPECIFIC_CFG_PARAM(20);
		break;
	case 21:	/* HBA 21 */
		HBA_SPECIFIC_CFG_PARAM(21);
		break;
	case 22:	/* HBA 22 */
		HBA_SPECIFIC_CFG_PARAM(22);
		break;
	case 23:	/* HBA 23 */
		HBA_SPECIFIC_CFG_PARAM(23);
		break;
	case 24:	/* HBA 24 */
		HBA_SPECIFIC_CFG_PARAM(24);
		break;
	case 25:	/* HBA 25 */
		HBA_SPECIFIC_CFG_PARAM(25);
		break;
	case 26:	/* HBA 26 */
		HBA_SPECIFIC_CFG_PARAM(26);
		break;
	case 27:	/* HBA 27 */
		HBA_SPECIFIC_CFG_PARAM(27);
		break;
	case 28:	/* HBA 28 */
		HBA_SPECIFIC_CFG_PARAM(28);
		break;
	case 29:	/* HBA 29 */
		HBA_SPECIFIC_CFG_PARAM(29);
		break;
	case 30:	/* HBA 30 */
		HBA_SPECIFIC_CFG_PARAM(30);
		break;
	case 31:	/* HBA 31 */
		HBA_SPECIFIC_CFG_PARAM(31);
		break;
	}
	return (value);
}



void
lpfc_sleep_ms(lpfcHBA_t * phba, int cnt)
{
	if (in_interrupt())
		mdelay(cnt);
	else {
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout((cnt * HZ / 1000) + 1);
	}
	return;
}

void
lpfc_drvr_init_lock(lpfcHBA_t * phba)
{

	spin_lock_init(&phba->drvrlock);
	return;
}

void
lpfc_drvr_lock(lpfcHBA_t * phba, unsigned long *iflag)
{
	unsigned long flag;

	flag = 0;
	spin_lock_irqsave(&phba->drvrlock, flag);
	*iflag = flag;
	phba->iflag = flag;
	return;
}

void
lpfc_drvr_unlock(lpfcHBA_t * phba, unsigned long *iflag)
{

	unsigned long flag;

	flag = phba->iflag;
	spin_unlock_irqrestore(&phba->drvrlock, flag);
	return;
}

int
lpfc_biosparam(Disk * disk, kdev_t n, int ip[])
{
	int size = disk->capacity;

	ip[0] = 64;
	ip[1] = 32;
	ip[2] = size >> 11;
	if (ip[2] > 1024) {
		ip[0] = 255;
		ip[1] = 63;
		ip[2] = size / (ip[0] * ip[1]);
#ifndef FC_EXTEND_TRANS_A
		if (ip[2] > 1023)
			ip[2] = 1023;
#endif
	}
	return (0);
}


void
lpfc_nodev(unsigned long l)
{
	return;
}


int
lpfc_sleep(lpfcHBA_t * phba, void *wait_q_head, long tmo)
{
	wait_queue_t wq_entry;
	unsigned long iflag = phba->iflag;
	int rc = 1;
	long left = 0;

	init_waitqueue_entry(&wq_entry, current);
	/* start to sleep before we wait, to avoid races */
	set_current_state(TASK_INTERRUPTIBLE);
	add_wait_queue((wait_queue_head_t *) wait_q_head, &wq_entry);
	if (tmo > 0) {
		LPFC_DRVR_UNLOCK(phba, iflag);
		left = schedule_timeout(tmo * HZ);
		LPFC_DRVR_LOCK(phba, iflag);
	} else {
		LPFC_DRVR_UNLOCK(phba, iflag);
		schedule();
		LPFC_DRVR_LOCK(phba, iflag);
	}
	remove_wait_queue((wait_queue_head_t *) wait_q_head, &wq_entry);

	if (signal_pending(current))
		return (EINTR);
	if (rc > 0)
		return (0);
	else
		return (ETIMEDOUT);
}

void
lpfc_discq_tasklet(lpfcHBA_t * phba, LPFC_DISC_EVT_t * evtp)
{

	/* Queue the cmnd to the iodone tasklet to be scheduled later */
	list_add_tail(&evtp->evt_listp, &phba->task_disc);
	phba->task_discq_cnt++;
	tasklet_schedule(&phba->task_run);
	return;
}

int
lpfc_discq_post_event(lpfcHBA_t * phba, void *arg1, void *arg2, uint32_t evt)
{
	LPFC_DISC_EVT_t  *evtp;

	/* All Mailbox completions and LPFC_ELS_RING rcv ring events will be
	 * queued to tasklet for processing
	 */
	evtp = (LPFC_DISC_EVT_t *) kmalloc(sizeof(LPFC_DISC_EVT_t), GFP_ATOMIC);
	if(evtp == 0) {
		return (0);
	}
	evtp->evt_arg1  = arg1;
	evtp->evt_arg2  = arg2;
	evtp->evt       = evt;
	evtp->evt_listp.next = 0;
	evtp->evt_listp.prev = 0;
	lpfc_discq_tasklet(phba, evtp);
	return (1);
}

void
lpfc_flush_disc_evtq(lpfcHBA_t * phba, lpfc_vport_t *vport)
{
	struct list_head * cur, * next;
	LPFC_DISC_EVT_t  * evtp, * next_evtp;
	LPFC_IOCBQ_t     * cmdiocbp;
	LPFC_IOCBQ_t     * rspiocbp;
	LPFC_IOCBQ_t     * saveq;
	LPFC_MBOXQ_t	 * pmb;
	LPFC_RING_MASK_t * func;

	list_for_each_entry_safe(evtp, next_evtp, &phba->task_disc, evt_listp) {
		switch(evtp->evt) {
		case LPFC_EVT_MBOX:
			pmb = (LPFC_MBOXQ_t *)(evtp->evt_arg1);
			if (vport != NULL && vport != pmb->vport)
				continue;
			(pmb->mbox_cmpl) (phba, pmb);
			break;
		case LPFC_EVT_SOL_IOCB:
			cmdiocbp = (LPFC_IOCBQ_t *)(evtp->evt_arg1);
			if (vport != NULL && vport != cmdiocbp->vport)
				continue;
			saveq = (LPFC_IOCBQ_t *)(evtp->evt_arg2);
			cmdiocbp->iocb.ulpStatus = IOSTAT_LOCAL_REJECT;
			cmdiocbp->iocb.un.ulpWord[4] = IOERR_SLI_ABORTED;
			(cmdiocbp->iocb_cmpl) (phba, cmdiocbp, saveq);

			list_for_each_safe(cur, next, &saveq->list) {
				rspiocbp = list_entry(cur, LPFC_IOCBQ_t, list);
				list_del(&rspiocbp->list);
				lpfc_iocb_free(phba, rspiocbp);
			}

			lpfc_iocb_free(phba, saveq);
			break;
		case LPFC_EVT_UNSOL_IOCB:
                        func = (LPFC_RING_MASK_t *)(evtp->evt_arg1);
                        saveq = (LPFC_IOCBQ_t *)(evtp->evt_arg2);
			if (vport != NULL && vport != saveq->vport)
				continue;
                        (func->lpfc_sli_rcv_unsol_event) (phba,
					&phba->sli.ring[LPFC_ELS_RING], saveq);

                        list_for_each_safe(cur, next, &saveq->list) {
                                rspiocbp = list_entry(cur, LPFC_IOCBQ_t, list);
                                list_del(&rspiocbp->list);
                                lpfc_iocb_free(phba, rspiocbp);
                        }

                        lpfc_iocb_free(phba, saveq);
                        break;	
                default:
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0251,
					lpfc_mes0251,
					lpfc_msgBlk0251.msgPreambleStr,
					evtp->evt,
					(uint32_t) evtp->evt_arg1,
					(uint32_t) evtp->evt_arg2);
			break;
		}

		list_del(&evtp->evt_listp);
		phba->task_discq_cnt--;
		kfree(evtp);
	}
}

#ifdef __VMKERNEL_MODULE__
static lpfc_vport_t*
lpfc_find_vport_context(lpfcHBA_t *phba, LPFC_DISC_EVT_t *evtp,
			uint32_t *prev_rscn_flag, uint32_t *prev_state)
{
	LPFC_MBOXQ_t *pmb;
	LPFC_IOCBQ_t *cmdiocbp;
	LPFC_IOCBQ_t *saveq;
	lpfc_vport_t *vport = NULL;
	int match = 0;

	switch(evtp->evt) {
		case LPFC_EVT_MBOX:
			pmb = evtp->evt_arg1;
			vport = pmb->vport;
			break;
		case LPFC_EVT_SOL_IOCB:
			cmdiocbp = evtp->evt_arg1;
			vport = cmdiocbp->vport;
			break;
		case LPFC_EVT_UNSOL_IOCB:
			saveq = evtp->evt_arg2;
			list_for_each_entry(vport, &phba->port_list, listentry) {
				if (vport->fc_myDID == saveq->iocb.un.rcvels.parmRo) {
					match++;
					break;
				}
			}

			if (match == 0)
				vport = NULL;

			break;
		default:
			break;
	}
	
	if (vport) {
		*prev_rscn_flag = vport->fc_flag & FC_RSCN_MODE;
		*prev_state = vport->port_state;
	}
	
	return vport;
}

static void
lpfc_vmk_notify_state_change(lpfc_vport_t *vport, uint32_t prev_rscn_flag,
			     uint32_t prev_state)
{
	/*
	 * Notifying vmk scsi of a state change occurs each time one of the 
	 * following is true - if the port has entered RSCN mode, if the port
	 * state has changed from READY to NOT_READY, or from NOT_READY to
	 * READY.  Vmkernel uses this upcall notification to probe the known
	 * storage paths.
	 */
	if ((prev_rscn_flag == 0) && (vport->fc_flag & FC_RSCN_MODE)) {
		vport->vmk_state_upcall = LPFC_VMK_UPCALL_READY;
	} else if ((prev_rscn_flag == FC_RSCN_MODE) &&
		   ((vport->fc_flag & FC_RSCN_MODE) == 0)) {
		vport->vmk_state_upcall = LPFC_VMK_UPCALL_READY;
	} else if ((prev_state == LPFC_HBA_READY) &&
		   (vport->port_state != LPFC_HBA_READY)) {
		vport->vmk_state_upcall = LPFC_VMK_UPCALL_READY;
	} else if ((prev_state != LPFC_HBA_READY) &&
		   (vport->port_state == LPFC_HBA_READY)) {
		vport->vmk_state_upcall = LPFC_VMK_UPCALL_PENDING;
		if ((vport->fc_plogi_cnt == 0) && (vport->fc_adisc_cnt == 0)) {
			if (vport->fc_map_cnt > 0)
				vport->vmk_state_upcall = LPFC_VMK_UPCALL_READY;
		}
	} else if (vport->vmk_state_upcall == LPFC_VMK_UPCALL_PENDING) {
		if ((vport->fc_plogi_cnt == 0) && (vport->fc_adisc_cnt == 0)) {
			if (vport->fc_map_cnt > 0)
				vport->vmk_state_upcall = LPFC_VMK_UPCALL_READY;
		}
	} else {
		vport->vmk_state_upcall = LPFC_VMK_UPCALL_NONE;
	}

	if (vport->vmk_state_upcall == LPFC_VMK_UPCALL_READY) {
		vmk_scsi_state_change(vport->scsihost);
		vport->vmk_state_upcall = LPFC_VMK_UPCALL_NONE;
	}
}
#endif

void
lpfc_tasklet(unsigned long p)
{
	lpfcHBA_t        * phba = (lpfcHBA_t *)p;
	LPFC_SLI_t       * psli;
	LPFC_DISC_EVT_t  * evtp;
	struct list_head * cur, * next;
	LPFC_MBOXQ_t     * pmb;
	LPFC_IOCBQ_t     * cmdiocbp;
	LPFC_IOCBQ_t     * rspiocbp;
	LPFC_IOCBQ_t     * saveq;
	LPFC_RING_MASK_t * func;
	unsigned long      flags;
#ifdef __VMKERNEL_MODULE__
        uint32_t prev_rscn_flag, prev_state, rc;
	lpfc_vport_t *vport = NULL;

	prev_rscn_flag = prev_state = rc = 0;
#endif

	psli = &phba->sli;
	LPFC_DRVR_LOCK(phba, flags);
	/* check discovery event list */
	while (!list_empty(&phba->task_disc)) {
		evtp = list_entry(phba->task_disc.next, LPFC_DISC_EVT_t, evt_listp);
		list_del(&evtp->evt_listp);
		phba->task_discq_cnt--;

#ifdef __VMKERNEL_MODULE__
		vport = lpfc_find_vport_context(phba, evtp, &prev_rscn_flag, &prev_state);
#endif
		switch(evtp->evt) {
		case LPFC_EVT_MBOX:
			pmb = (LPFC_MBOXQ_t *)(evtp->evt_arg1);
			(pmb->mbox_cmpl) (phba, pmb);
			break;

		case LPFC_EVT_SOL_IOCB:
			cmdiocbp = (LPFC_IOCBQ_t *)(evtp->evt_arg1);
			saveq = (LPFC_IOCBQ_t *)(evtp->evt_arg2);
			(cmdiocbp->iocb_cmpl) (phba, cmdiocbp, saveq);

			list_for_each_safe(cur, next, &saveq->list) {
				rspiocbp = list_entry(cur, LPFC_IOCBQ_t, list);
				list_del(&rspiocbp->list);
				lpfc_iocb_free(phba, rspiocbp);
			}

			lpfc_iocb_free(phba, saveq);
			break;

		case LPFC_EVT_UNSOL_IOCB:
			func = (LPFC_RING_MASK_t *)(evtp->evt_arg1);
			saveq = (LPFC_IOCBQ_t *)(evtp->evt_arg2);

			(func->lpfc_sli_rcv_unsol_event) (phba, 
	 		&psli->ring[LPFC_ELS_RING], saveq);

			list_for_each_safe(cur, next, &saveq->list) {
				rspiocbp = list_entry(cur, LPFC_IOCBQ_t, list);
				list_del(&rspiocbp->list);
				lpfc_iocb_free(phba, rspiocbp);
			}
			lpfc_iocb_free(phba, saveq);
			break;
		default:
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk0252,
					lpfc_mes0252,
					lpfc_msgBlk0252.msgPreambleStr,
					evtp->evt,
					(uint32_t) evtp->evt_arg1,
					(uint32_t) evtp->evt_arg2);
			break;
		}

		kfree(evtp);

#ifdef __VMKERNEL_MODULE__
		/*
		 * Determine whether or not vmkernel needs to be notified of a 
		 * fibre channel fabric change.
		 */
		if (vport)
			lpfc_vmk_notify_state_change(vport, prev_rscn_flag, prev_state);
#endif
	}

	LPFC_DRVR_UNLOCK(phba, flags);
	return;
}


#ifdef  __VMKERNEL_MODULE__


static void
lpfc_get_starget_port_id(struct scsi_target *starget)
{
	LPFC_NODELIST_t *ndlp = NULL;
	struct Scsi_Host *shost = starget->sh;
	lpfc_vport_t *vport = (lpfc_vport_t *) shost->hostdata[0];
	lpfcHBA_t *phba = vport->phba;
	uint32_t did = 0;
	unsigned long flags;

	LPFC_DRVR_LOCK(phba, flags);

	/* Search the mapped list for this target ID and return its DID. */
	list_for_each_entry(ndlp, &phba->pport->fc_nlpmap_list, nlp_listp) {
		if (starget->id == ndlp->nlp_scsi_id) {
			did = ndlp->nlp_DID;
			break;
		}
	}

	fc_starget_port_id(starget) = did;
	LPFC_DRVR_UNLOCK(phba, flags);
}

static void
lpfc_get_starget_node_name(struct scsi_target *starget)
{
	LPFC_NODELIST_t *ndlp = NULL;
	struct Scsi_Host *shost = starget->sh;
        lpfc_vport_t *vport = (lpfc_vport_t *) shost->hostdata[0];
        lpfcHBA_t *phba = vport->phba;
	uint64_t node_name = 0;
	unsigned long flags;

	LPFC_DRVR_LOCK(phba, flags);

	/*
	 * Search the mapped list for this target ID and retrieve the
         * node name.
	 */
	list_for_each_entry(ndlp, &phba->pport->fc_nlpmap_list, nlp_listp) {
		if (starget->id == ndlp->nlp_scsi_id) {
			memcpy(&node_name, &ndlp->nlp_nodename,
				sizeof(NAME_TYPE));
			break;
		}
	}

	fc_starget_node_name(starget) = be64_to_cpu(node_name);
	LPFC_DRVR_UNLOCK(phba, flags);
}

static void
lpfc_get_starget_port_name(struct scsi_target *starget)
{
	LPFC_NODELIST_t *ndlp = NULL;
	struct Scsi_Host *shost = starget->sh;
	lpfc_vport_t *vport = (lpfc_vport_t *) shost->hostdata[0];
	lpfcHBA_t *phba = vport->phba;
	uint64_t port_name = 0;
	unsigned long flags;

	LPFC_DRVR_LOCK(phba, flags);

	/* Search the mapped list for this target ID */
	list_for_each_entry(ndlp, &phba->pport->fc_nlpmap_list, nlp_listp) {
		if (starget->id == ndlp->nlp_scsi_id) {
			memcpy(&port_name, &ndlp->nlp_portname,
				sizeof(NAME_TYPE));
			break;
		}
	}

	fc_starget_port_name(starget) = be64_to_cpu(port_name);
	LPFC_DRVR_UNLOCK(phba, flags);
}

static void
lpfc_get_host_port_id(struct Scsi_Host *shost)
{
	lpfc_vport_t *vport = (lpfc_vport_t *) shost->hostdata[0];

	fc_host_port_id(shost) = vport->fc_myDID;
}

static void
lpfc_get_host_speed(struct Scsi_Host *shost)
{
	lpfc_vport_t *vport = (lpfc_vport_t *) shost->hostdata[0];
	lpfcHBA_t *phba = vport->phba;

	switch(phba->fc_linkspeed) {
	case LA_1GHZ_LINK:
		fc_host_speed(shost) = FC_PORTSPEED_1GBIT;
		break;
	case LA_2GHZ_LINK:
		fc_host_speed(shost) = FC_PORTSPEED_2GBIT;
		break;
	case LA_4GHZ_LINK:
		fc_host_speed(shost) = FC_PORTSPEED_4GBIT;
		break;
	case LA_8GHZ_LINK: /* We need to add this to template */
	default:
		fc_host_speed(shost) = FC_PORTSPEED_UNKNOWN;
		break;
	}

}

static void
lpfc_get_port_type(struct Scsi_Host *shost)
{
	lpfc_vport_t *vport = (lpfc_vport_t *) shost->hostdata[0];
	lpfcHBA_t *phba = vport->phba;

	if (phba->fc_topology == TOPOLOGY_LOOP) {
		if (vport->fc_flag & FC_PUBLIC_LOOP) {
                        fc_host_port_type(shost) = FC_PORTTYPE_NLPORT;
		} else {
                        fc_host_port_type(shost) = FC_PORTTYPE_LPORT;
		}
	} else {
		if (vport->fc_flag & FC_FABRIC) {
                        fc_host_port_type(shost) = FC_PORTTYPE_NPORT;
		} else {
                        fc_host_port_type(shost) = FC_PORTTYPE_PTP;
		}
	}
}

static void
lpfc_get_port_state(struct Scsi_Host *shost)
{
	lpfc_vport_t *vport = (lpfc_vport_t *) shost->hostdata[0];

	if (vport->fc_flag & FC_BYPASSED_MODE) {
                fc_host_port_state(shost) = FC_PORTSTATE_BYPASSED;
	} else if (vport->fc_flag & FC_OFFLINE_MODE) {
                fc_host_port_state(shost) = FC_PORTSTATE_DIAGNOSTICS;
	} else {
		switch (vport->port_state) {
		case LPFC_INIT_START:
		case LPFC_INIT_MBX_CMDS:
                        fc_host_port_state(shost) = FC_PORTSTATE_UNKNOWN;
			break;
		case LPFC_LINK_DOWN:
		case LPFC_LINK_UP:
		case LPFC_LOCAL_CFG_LINK:
		case LPFC_FLOGI:
		case LPFC_FABRIC_CFG_LINK:
		case LPFC_NS_REG:
		case LPFC_NS_QRY:
		case LPFC_BUILD_DISC_LIST:
		case LPFC_DISC_AUTH:
		case LPFC_CLEAR_LA:
                        fc_host_port_state(shost) = FC_PORTSTATE_LINKDOWN;
			break;
		case LPFC_HBA_READY:
                        fc_host_port_state(shost) = FC_PORTSTATE_ONLINE;
			break;
		case LPFC_HBA_ERROR:
		default:
                        fc_host_port_state(shost) = FC_PORTSTATE_ERROR;
			break;
		}
	}
}
#endif 

#include <scsi_module.c>
MODULE_LICENSE("GPL");

/*
 * Note: PPC64 architecture has function descriptors,
 * so insmod on 2.4 does not automatically export all symbols.
 */
EXPORT_SYMBOL(lpfc_add_bind);
EXPORT_SYMBOL(lpfc_block_requests);
EXPORT_SYMBOL(lpfc_build_scsi_cmd);
EXPORT_SYMBOL(lpfc_decode_firmware_rev);
EXPORT_SYMBOL(lpfc_del_bind);
EXPORT_SYMBOL(lpfcDRVR);
EXPORT_SYMBOL(lpfc_drvr_name);
EXPORT_SYMBOL(lpfc_drvr_lock);
EXPORT_SYMBOL(lpfc_drvr_unlock);
EXPORT_SYMBOL(lpfc_els_free_iocb);
EXPORT_SYMBOL(lpfc_find_lun);
EXPORT_SYMBOL(lpfc_findnode_did);
EXPORT_SYMBOL(lpfc_findnode_scsiid);
EXPORT_SYMBOL(lpfc_findnode_wwnn);
EXPORT_SYMBOL(lpfc_findnode_wwpn);
EXPORT_SYMBOL(lpfc_free_scsi_buf);
EXPORT_SYMBOL(lpfc_geportname);
EXPORT_SYMBOL(lpfc_get_hba_sym_node_name);
EXPORT_SYMBOL(lpfc_get_scsi_buf);
EXPORT_SYMBOL(lpfc_init_link);
EXPORT_SYMBOL(lpfc_iocb_alloc);
EXPORT_SYMBOL(lpfc_iocb_free);
EXPORT_SYMBOL(lpfc_issue_ct_rsp);
EXPORT_SYMBOL(lpfc_issue_els_adisc);
EXPORT_SYMBOL(lpfc_issue_els_logo);
EXPORT_SYMBOL(lpfc_issue_els_plogi);
EXPORT_SYMBOL(lpfc_mbuf_alloc);
EXPORT_SYMBOL(lpfc_mbuf_free);
EXPORT_SYMBOL(lpfc_page_alloc);
EXPORT_SYMBOL(lpfc_page_free);
EXPORT_SYMBOL(lpfc_nlp_bind);
EXPORT_SYMBOL(lpfc_nlp_plogi);
EXPORT_SYMBOL(lpfc_offline);
EXPORT_SYMBOL(lpfc_online);
EXPORT_SYMBOL(lpfc_prep_els_iocb);
EXPORT_SYMBOL(lpfc_release_version);
EXPORT_SYMBOL(lpfc_scsi_lun_reset);
EXPORT_SYMBOL(lpfc_scsi_tgt_reset);
EXPORT_SYMBOL(lpfc_sleep);
EXPORT_SYMBOL(lpfc_sleep_ms);
EXPORT_SYMBOL(lpfc_sli_brdreset);
EXPORT_SYMBOL(lpfc_sli_issue_iocb_wait);
EXPORT_SYMBOL(lpfc_sli_issue_mbox);
EXPORT_SYMBOL(lpfc_sli_issue_mbox_wait);
EXPORT_SYMBOL(lpfc_sli_pcimem_bcopy);
EXPORT_SYMBOL(lpfc_unblock_requests);
EXPORT_SYMBOL(lpfc_get_phba_by_inst);

#ifdef __VMKERNEL_MODULE__
EXPORT_SYMBOL(lpfc_vport_create);
EXPORT_SYMBOL(lpfc_vport_delete);
EXPORT_SYMBOL(lpfc_vport_getinfo);
EXPORT_SYMBOL(lpfc_vport_tgt_remove);
#endif

EXPORT_SYMBOL(lpfc_sched_continue_hba);
EXPORT_SYMBOL(lpfc_sched_pause_hba);

