/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Fibre Channel Host Bus Adapters.                                *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003-2007 Emulex.  All rights reserved.           *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

/*
 * $Id: lpfc_sched.c 1.3 2005/05/03 11:22:04EDT sf_support Exp  $
 */

#include <linux/version.h>
#include <linux/spinlock.h>

#ifdef __VMKERNEL_MODULE__
#include "linux_skbuff.h"
#include "vmklinux_dist.h"
#include "npiv_api_dist.h"
#endif

#include <linux/blk.h>
#include <scsi.h>

#include "lpfc_hw.h"
#include "lpfc_cfgparm.h"
#include "lpfc_mem.h"
#include "lpfc_dfc.h"
#include "lpfc_sli.h"
#include "lpfc_sched.h"
#include "lpfc_disc.h"
#include "lpfc.h"
#include "lpfc_logmsg.h"
#include "lpfc_fcp.h"
#include "lpfc_scsi.h"
#include "lpfc_crtn.h"

struct lpfcHBA;

/* *********************************************************************
**
**    Forward declaration of internal routines to SCHED
**
** ******************************************************************** */

static void lpfc_sched_internal_check(lpfc_vport_t *vport);

/* ***************************************************************
**
**  Initialize HBA, TARGET and LUN SCHED structures
**  Basically clear them, set MaxQueue Depth
** and mark them ready to go
**
** **************************************************************/

void
lpfc_sched_hba_init(lpfc_vport_t * vport, uint16_t maxOutstanding)
{
	memset(&vport->hbaSched, 0, sizeof (vport->hbaSched));
	vport->hbaSched.maxOutstanding = maxOutstanding;
	vport->hbaSched.status = LPFC_SCHED_STATUS_OKAYTOSEND;
	INIT_LIST_HEAD(&vport->hbaSched.highPriorityCmdList);
	INIT_LIST_HEAD(&vport->hbaSched.targetRing);
}

void
lpfc_sched_target_init(LPFCSCSITARGET_t * target, uint16_t maxOutstanding)
{
	memset(&target->targetSched, 0, sizeof (target->targetSched));
	target->targetSched.maxOutstanding = maxOutstanding;
	target->targetSched.status = LPFC_SCHED_STATUS_OKAYTOSEND;
	/* initlize the queue */
	INIT_LIST_HEAD(&target->listentry);
	INIT_LIST_HEAD(&target->targetSched.lunRing);
}

void
lpfc_sched_lun_init(LPFCSCSILUN_t * lun, uint16_t maxOutstanding)
{
	memset(&lun->lunSched, 0, sizeof (lun->lunSched));
	lun->lunSched.maxOutstanding = maxOutstanding;
	lun->fcp_lun_queue_depth = maxOutstanding;
	lun->lunSched.status = LPFC_SCHED_STATUS_OKAYTOSEND;
	/* initialize list */
	INIT_LIST_HEAD(&lun->lunSched.commandList);
	INIT_LIST_HEAD(&lun->listentry);
}

uint32_t
lpfc_sched_set_hba_qdepth(struct lpfcHBA *phba, uint32_t new_qdepth)
{
	uint32_t return_qdepth;
	lpfc_vport_t *vport;
	/*
	 * Do not exceed HBA queue depth
	*/
	if (new_qdepth >= phba->hba_max_qdepth) {
		return_qdepth = phba->hba_max_qdepth;
	}
	else {
		return_qdepth = new_qdepth;
	}
	list_for_each_entry(vport, &phba->port_list, listentry) {
		lpfc_sched_pause_hba(vport);
		vport->hbaSched.maxOutstanding = (uint16_t) return_qdepth;
		lpfc_sched_continue_hba(vport);
	}
	return return_qdepth;
}

uint32_t
lpfc_sched_set_tgt_qdepth(struct lpfcHBA *phba, uint32_t new_qdepth)
{
	uint32_t tgt = 0;
	lpfc_vport_t *vport;
	LPFCSCSITARGET_t *target;
	uint32_t return_qdepth;
	lpfcCfgParam_t *clp = &phba->config[0];
	/*
	 * Do not exceed driver's HBA queue depth.
	 */ 
	if (new_qdepth >= LPFC_SET_CAN_QUEUE) {
		return_qdepth = LPFC_SET_CAN_QUEUE;
	}
	else {
		return_qdepth = new_qdepth;
	}

	list_for_each_entry(vport, &phba->port_list, listentry) {
		for (tgt = 0; tgt < vport->sid_cnt; tgt++) {
			target = vport->device_queue_hash[tgt];
			lpfc_sched_pause_target(target);
			target->targetSched.maxOutstanding = (uint16_t) return_qdepth;
			lpfc_sched_continue_target(target);
		}
	}
	return return_qdepth;
}

uint32_t
lpfc_sched_set_lun_qdepth(struct lpfcHBA *phba, uint32_t new_qdepth)
{
	uint32_t tgt = 0;	
	lpfc_vport_t *vport;
	LPFCSCSITARGET_t *target;
	LPFCSCSILUN_t *lun;
	uint32_t return_qdepth;
	lpfcCfgParam_t *clp = &phba->config[0];

        /*
         * Do not exceed driver's target queue depth.
         */
        if (new_qdepth >= clp[LPFC_CFG_DFT_TGT_Q_DEPTH].a_current) {
                return_qdepth = clp[LPFC_CFG_DFT_TGT_Q_DEPTH].a_current;
        }
        else {
                return_qdepth = new_qdepth;
        }


	list_for_each_entry(vport, &phba->port_list, listentry) {
		for (tgt = 0; tgt < vport->sid_cnt; tgt++) {
			target = vport->device_queue_hash[tgt];
			lpfc_sched_pause_target(target);
			list_for_each_entry(lun, &target->lunlist, list) {
				lun->lunSched.maxOutstanding = (uint16_t) return_qdepth;
				lun->fcp_lun_queue_depth = (uint16_t) return_qdepth;
			}
			lpfc_sched_continue_target(target);
		}
	}
	return return_qdepth;
}

void
lpfc_sched_pause_target(LPFCSCSITARGET_t * target)
{
	target->targetSched.status = LPFC_SCHED_STATUS_PAUSED;
}

void
lpfc_sched_pause_hba(lpfc_vport_t * vport)
{
	vport->hbaSched.status = LPFC_SCHED_STATUS_PAUSED;
}

void
lpfc_sched_continue_target(LPFCSCSITARGET_t * target)
{
	target->targetSched.status = LPFC_SCHED_STATUS_OKAYTOSEND;
	lpfc_sched_internal_check(target->vport);
}

void
lpfc_sched_continue_hba(lpfc_vport_t * vport)
{
	vport->hbaSched.status = LPFC_SCHED_STATUS_OKAYTOSEND;
	lpfc_sched_internal_check(vport);
}

void
lpfc_sched_sli_done(lpfcHBA_t * pHba,
		    LPFC_IOCBQ_t * pIocbIn, LPFC_IOCBQ_t * pIocbOut)
{
	lpfc_vport_t *vport = pIocbIn->vport;
	LPFC_SCSI_BUF_t *pCommand = (LPFC_SCSI_BUF_t *) pIocbIn->context1;
	LPFCSCSILUN_t *plun = pCommand->pLun;
	static int doNotCheck = 0;
	lpfcCfgParam_t *clp;
	FCP_RSP *fcprsp;

	plun->lunSched.currentOutstanding--;
	plun->pTarget->targetSched.currentOutstanding--;

	pCommand->result = pIocbOut->iocb.un.ulpWord[4];
	if ((pCommand->status = pIocbOut->iocb.ulpStatus) ==
	    IOSTAT_LOCAL_REJECT) {
		if(pCommand->result & IOERR_DRVR_MASK) {
			pCommand->status = IOSTAT_DRIVER_REJECT;
		}
	}

	if (pCommand->status)
		plun->errorcnt++;

	plun->iodonecnt++;

	vport->hbaSched.currentOutstanding--;

	fcprsp = pCommand->fcp_rsp;
	if ((pCommand->status == IOSTAT_FCP_RSP_ERROR) &&
	    (fcprsp->rspStatus3 == SCSI_STAT_QUE_FULL)) {

		/* Scheduler received Queue Full status from FCP device (tgt>
		   <lun> */
		lpfc_printf_log(pHba->brd_no, &lpfc_msgBlk0738,
				lpfc_mes0738,
				lpfc_msgBlk0738.msgPreambleStr,
				pCommand->scsi_target, pCommand->scsi_lun,
				pCommand->qfull_retry_count,
				plun->qfull_retries,
				plun->lunSched.currentOutstanding,
				plun->lunSched.maxOutstanding);

		if (((plun->qfull_retries > 0) &&
		     (pCommand->qfull_retry_count < plun->qfull_retries)) || 
		    (plun->lunSched.currentOutstanding + plun->lunSched.q_cnt == 0)){
			clp = &pHba->config[0];
			if (clp[LPFC_CFG_DQFULL_THROTTLE_UP_TIME].a_current) {
				lpfc_scsi_lower_lun_qthrottle(vport, pCommand);
			}
			lpfc_sched_queue_command(vport, pCommand);
			plun->qcmdcnt++;
			pCommand->qfull_retry_count++;
			goto skipcmpl;
		}
	}

	(pCommand->cmd_cmpl) (vport, pCommand);

 skipcmpl:

	if (!doNotCheck) {
		doNotCheck = 1;
		lpfc_sched_internal_check(vport);
		doNotCheck = 0;
	}
	return;
}

void
lpfc_sched_check(lpfc_vport_t *vport)
{
	lpfc_sched_internal_check(vport);
	return;
}


static void    
lpfc_sched_internal_check(lpfc_vport_t  *vport)
{
	lpfcHBA_t *hba = vport->phba;
	LPFC_SCHED_HBA_t  * hbaSched           = &vport->hbaSched;
	LPFC_SLI_t        * psli;
	LPFC_NODELIST_t   * ndlp;
	int                numberOfFailedTargetChecks = 0;
	int                didSuccessSubmit   = 0;    /* SLI optimization for Port signals */
	int                stopSched          = 0;    /* used if SLI rejects on interloop */

	/* get the sli struct from phba */
	psli = &hba->sli;
   
	/* Service the High Priority Queue first */
	if (vport->hbaSched.q_cnt)
		lpfc_sched_service_high_priority_queue(vport);
   
	/* If targetCount is identically 0 then there are no Targets on the ring therefore
	   no pending commands on any LUN           
	*/
	if ( (hbaSched->targetCount == 0) ||
	     (hbaSched->status == LPFC_SCHED_STATUS_PAUSED) )   
		return;
   
	/* We are going to cycle through the Targets
	   on a round robin basis until we make a pass through
	   with nothing to schedule. 
	*/

	while ( (stopSched == 0)                                            &&
		(hbaSched->currentOutstanding < hbaSched->maxOutstanding) &&
		(numberOfFailedTargetChecks < hbaSched->targetCount) ) {
		LPFCSCSITARGET_t      *target      = hbaSched->nextTargetToCheck;
		LPFC_SCHED_TARGET_t   *targetSched = &target->targetSched;
		LPFCSCSITARGET_t      *newNext     = list_entry(target->listentry.next,
								LPFCSCSITARGET_t,
								listentry);
		int                   numberOfFailedLunChecks = 0;
      
		if (target->listentry.next == &hbaSched->targetRing) {
			newNext = list_entry( hbaSched->targetRing.next,
					      LPFCSCSITARGET_t,
					      listentry);
		}

		if (( targetSched->currentOutstanding  < targetSched->maxOutstanding) &&
		    ( targetSched->status != LPFC_SCHED_STATUS_PAUSED)) {
			while ( numberOfFailedLunChecks < targetSched->lunCount ) {
				LPFCSCSILUN_t      *lun        = target->targetSched.nextLunToCheck;
				LPFC_SCHED_LUN_t   *lunSched   = &lun->lunSched;
				LPFCSCSILUN_t      *newNextLun = list_entry(lun->listentry.next,
									    LPFCSCSILUN_t,
									    listentry);
			   
				if ( lun->listentry.next == &target->targetSched.lunRing ) {
					newNextLun = list_entry(target->targetSched.lunRing.next,
								LPFCSCSILUN_t,
								listentry);
				}

				if (( lunSched->currentOutstanding < lunSched->maxOutstanding ) &&
				    ( !(list_empty(&lunSched->commandList))) &&
				    ( lunSched->status != LPFC_SCHED_STATUS_PAUSED)) {
					LPFC_SCSI_BUF_t   *command;
					int               sliStatus;
					LPFC_IOCBQ_t      *pIocbq;
					struct list_head *head;

					head = lunSched->commandList.next;
					command = list_entry(head,
							     LPFC_SCSI_BUF_t,
							     listentry);
					list_del(head);
					--lunSched->q_cnt;
				   
					if (!command) {
						numberOfFailedLunChecks++;
						targetSched->nextLunToCheck  = newNextLun;
						continue;
					}
				   
					ndlp = command->pLun->pnode;
					if(ndlp == 0) {
						numberOfFailedLunChecks++;
						lpfc_sched_queue_command(vport,command);
						targetSched->nextLunToCheck  = newNextLun;
						continue;
					}
	       
					pIocbq = &command->cur_iocbq;
					pIocbq->context1  = command;
					pIocbq->iocb_cmpl = lpfc_sched_sli_done;
					pIocbq->vport = vport;
				   
					/* put the RPI number and NODELIST info in the IOCB command */
					pIocbq->iocb.ulpContext = ndlp->nlp_rpi;
					if (ndlp->nlp_fcp_info & NLP_FCP_2_DEVICE) {
						pIocbq->iocb.ulpFCP2Rcvy = 1;
					}

					pIocbq->iocb.ulpClass = (ndlp->nlp_fcp_info & 0x0f); 
					sliStatus = lpfc_sli_issue_iocb(hba,
									&psli->ring[ psli->fcp_ring],
									pIocbq, SLI_IOCB_RET_IOCB);
					switch (sliStatus) {
					case  IOCB_ERROR:   
						stopSched = 1;
						list_add(&command->listentry,
							 &lunSched->commandList);
						++lunSched->q_cnt;
						break;
		    
					case  IOCB_BUSY: 
					case  IOCB_SUCCESS:
						didSuccessSubmit = 1;
						lunSched->currentOutstanding++;
						targetSched->currentOutstanding++;
						hbaSched->currentOutstanding++;
						targetSched->nextLunToCheck = newNextLun;
						break;
		     
					default:
						break;
					}      /* End of Switch */				   

					/* 
					 * Check if there is any pending command on the lun. If not 
					 * remove the lun. If this is the last lun in the target, the
					 * target also will get removed from the scheduler ring.
					 */
					if (list_empty(&lunSched->commandList))		   
						lpfc_sched_remove_lun_from_ring(vport,lun);

					/* Either we shipped or SLI refused the operation. In any chase
					 * the driver is done with this LUN/Target!. 
					 */
					break;

					/* This brace ends LUN window open */
				}   
				else {
					numberOfFailedLunChecks++;
					targetSched->nextLunToCheck = newNextLun;
				}
				/* This brace ends While looping through LUNs on a Target */
			}
	 
			if ( numberOfFailedLunChecks >= targetSched->lunCount )  
				numberOfFailedTargetChecks++;
			else 
				numberOfFailedTargetChecks = 0;
		}   /* if Target isn't pended */
		else 
			numberOfFailedTargetChecks++;
      
		hbaSched->nextTargetToCheck = newNext;
	}   /* While looping through Targets on HBA */
  
	return;
}


void
lpfc_sched_service_high_priority_queue(lpfc_vport_t *vport)
{
	LPFC_SLI_t *psli;
	LPFC_NODELIST_t *ndlp;
	lpfcHBA_t *hba = vport->phba;
	LPFC_IOCBQ_t *pIocbq;
	LPFC_SCSI_BUF_t *command;
	int sliStatus;
	struct list_head *pos, *pos_tmp;

	psli = &hba->sli;

	/* 
	 * Iterate through highprioritycmdlist if any cmds waiting on it
	 * dequeue first cmd from highPriorityCmdList
	 * 
	 */
	list_for_each_safe(pos, pos_tmp, &vport->hbaSched.highPriorityCmdList) {
		command =
			list_entry(pos,
				   LPFC_SCSI_BUF_t,
				   listentry);
		list_del(pos);
		--vport->hbaSched.q_cnt;

		if (!command) {
			continue;
		}

		if ((command->pLun) && (command->pLun->pnode)) {

			ndlp = command->pLun->pnode;
			if (ndlp == 0) {

			} else {
				/* put the RPI number and NODELIST info in the
				   IOCB command */
				pIocbq = &command->cur_iocbq;
				pIocbq->iocb.ulpContext = ndlp->nlp_rpi;
				if (ndlp->nlp_fcp_info & NLP_FCP_2_DEVICE) {
					pIocbq->iocb.ulpFCP2Rcvy = 1;
				}
				pIocbq->iocb.ulpClass =
					(ndlp->nlp_fcp_info & 0x0f);
			}
		}

		pIocbq = &command->cur_iocbq;
		pIocbq->context1 = command;
		pIocbq->iocb_cmpl = lpfc_sli_wake_iocb_high_priority;
		pIocbq->vport = vport;
		sliStatus = lpfc_sli_issue_iocb(hba,
						&psli->ring[psli->fcp_ring],
						pIocbq,
						SLI_IOCB_HIGH_PRIORITY |
						SLI_IOCB_RET_IOCB);
		switch (sliStatus) {
		case IOCB_ERROR:
			/* We'll put it back to the head of the q and try
			   again */
			list_add(&command->listentry,
				 &vport->hbaSched.highPriorityCmdList);
			++vport->hbaSched.q_cnt;
			break;

		case IOCB_BUSY:
		case IOCB_SUCCESS:
			vport->hbaSched.currentOutstanding++;
			break;

		default:

			break;
		}

		break;
	}

	return;
}

uint32_t
lpfc_sched_dequeue(lpfc_vport_t *vport, LPFC_SCSI_BUF_t * ourCommand)
{
        uint32_t found;
        LPFC_SCSI_BUF_t *currentCommand;
        LPFC_SCHED_LUN_t *pLunSched;
        struct list_head *cur, *next;

        pLunSched = &ourCommand->pLun->lunSched;
        found = 0;
        list_for_each_safe (cur, next, &pLunSched->commandList) {
                currentCommand = list_entry(cur, LPFC_SCSI_BUF_t, listentry);
                if (currentCommand == ourCommand) {
                        found++;
                        list_del(cur);
                        --pLunSched->q_cnt;
                        if (!pLunSched->q_cnt){
                                lpfc_sched_remove_lun_from_ring(vport,
                                                                ourCommand->pLun);
                        }
                        break;
                }
        }

        return found;
}

uint32_t
lpfc_sched_flush_command(lpfc_vport_t *vport,
                         LPFC_SCSI_BUF_t * command,
                         uint8_t iocbStatus, uint32_t word4)
{
	lpfcHBA_t *phba = vport->phba;
        uint32_t foundCommand = lpfc_sched_dequeue(vport, command);
        uint32_t ret = 0;

        if (foundCommand) {
                IOCB_t *pIOCB = (IOCB_t *) & (command->cur_iocbq.iocb);
                pIOCB->ulpStatus = iocbStatus;
                command->status = iocbStatus;
                if (word4) {
                        pIOCB->un.ulpWord[4] = word4;
                        command->result = word4;
                }

                if (command->status) {
                        command->pLun->errorcnt++;
                }

                command->pLun->iodonecnt++;
                (command->cmd_cmpl) (vport, command);
        } else {
                ret = lpfc_sli_abort_iocb_context1(phba,
                        &phba->sli.ring[phba->sli.fcp_ring], command);
        }

        return ret;
}

uint32_t
lpfc_sched_flush_lun(lpfc_vport_t *vport,
		     LPFCSCSILUN_t * lun, uint8_t iocbStatus, uint32_t word4)
{
	struct list_head *cur, *next;
	lpfcHBA_t * phba = vport->phba;

	int numberFlushed = 0;

	list_for_each_safe(cur, next, &lun->lunSched.commandList) {
		IOCB_t *pIOCB;
		LPFC_SCSI_BUF_t *command =
			list_entry(cur,
				   LPFC_SCSI_BUF_t,
				   listentry);
		list_del(cur);
		--lun->lunSched.q_cnt;

		pIOCB = (IOCB_t *) & (command->cur_iocbq.iocb);
		pIOCB->ulpStatus = iocbStatus;
		command->status = iocbStatus;
		if (word4) {
			pIOCB->un.ulpWord[4] = word4;
			command->result = word4;
		}

		if (command->status) {
			lun->errorcnt++;
		}
		lun->iodonecnt++;

		(command->cmd_cmpl) (vport, command);

		numberFlushed++;
	}
	lpfc_sched_remove_lun_from_ring(vport, lun);

	/* flush the SLI layer also */
	lpfc_sli_abort_iocb_lun(vport, &phba->sli.ring[phba->sli.fcp_ring],
				lun->pTarget->scsi_id, lun->lun_id);
	return (numberFlushed);
}

uint32_t
lpfc_sched_flush_target(lpfc_vport_t *vport,
			LPFCSCSITARGET_t * target,
			uint8_t iocbStatus, uint32_t word4)
{
	LPFCSCSILUN_t *lun;
	int numberFlushed = 0;
	struct list_head *cur_h, *next_h;
	struct list_head *cur_l, *next_l;
	lpfcHBA_t * phba = vport->phba;

	if (target->rptlunfunc.function) {
		lpfc_stop_timer((struct clk_data *) target->rptlunfunc.data);
		target->targetFlags &= ~FC_RETRY_RPTLUN;
	}

	/* walk the list of LUNs on this target and flush each LUN.  We
	   accomplish this by pulling the first LUN off the head of the
	   queue until there aren't any LUNs left */
	list_for_each_safe(cur_h, next_h, &target->targetSched.lunRing) {
		lun = list_entry(cur_h,
				 LPFCSCSILUN_t,
				 listentry);
		
		list_for_each_safe(cur_l, next_l, &lun->lunSched.commandList) {
			IOCB_t *pIOCB;
			LPFC_SCSI_BUF_t *command =
				list_entry(cur_l,
					   LPFC_SCSI_BUF_t,
					   listentry);
			list_del(cur_l);
			--lun->lunSched.q_cnt;

			pIOCB = (IOCB_t *) & (command->cur_iocbq.iocb);
			pIOCB->ulpStatus = iocbStatus;
			command->status = iocbStatus;
			if (word4) {
				pIOCB->un.ulpWord[4] = word4;
				command->result = word4;
			}

			if (command->status) {
				lun->errorcnt++;
			}
			lun->iodonecnt++;

			(command->cmd_cmpl) (vport, command);

			numberFlushed++;
		}

		lpfc_sched_remove_lun_from_ring(vport, lun);
	}
	lpfc_sched_remove_target_from_ring(vport, target);

	/* flush the SLI layer also */
	lpfc_sli_abort_iocb_tgt(vport, &phba->sli.ring[phba->sli.fcp_ring],
				target->scsi_id);
	return (numberFlushed);
}

uint32_t
lpfc_sched_flush_hba(lpfc_vport_t *vport, uint8_t iocbStatus, uint32_t word4)
{
	int numberFlushed = 0;
	LPFCSCSITARGET_t *target;
	LPFCSCSILUN_t *lun;
	struct list_head *cur_h, *next_h;
	struct list_head *cur_l, *next_l;
	struct list_head *cur_c, *next_c;
	lpfcHBA_t * phba = vport->phba;

	list_for_each_safe(cur_h, next_h, &vport->hbaSched.targetRing) {
		target = list_entry(cur_h,
				    LPFCSCSITARGET_t,
				    listentry);
		list_for_each_safe(cur_l, next_l, &target->targetSched.lunRing)
			{
				lun = list_entry(cur_l, LPFCSCSILUN_t, listentry);
				list_for_each_safe(cur_c, next_c,
						   &lun->lunSched.commandList) {
					IOCB_t *pIOCB;
					LPFC_SCSI_BUF_t *command =
						list_entry(cur_c,
							   LPFC_SCSI_BUF_t,
							   listentry);
					list_del(cur_c);
					--lun->lunSched.q_cnt;

					pIOCB = (IOCB_t *) & (command->cur_iocbq.iocb);
					pIOCB->ulpStatus = iocbStatus;
					command->status = iocbStatus;
					if (word4) {
						pIOCB->un.ulpWord[4] = word4;
						command->result = word4;
					}

					if (command->status) {
						lun->errorcnt++;
					}
					lun->iodonecnt++;

					(command->cmd_cmpl) (vport, command);

					numberFlushed++;
				}

				lpfc_sched_remove_lun_from_ring(vport, lun);
			}
		lpfc_sched_remove_target_from_ring(vport, target);
	}

	/* flush the SLI layer also */
	lpfc_sli_abort_iocb_hba(phba, &phba->sli.ring[phba->sli.fcp_ring]);
	return (numberFlushed);
}

int
lpfc_sched_submit_command(lpfc_vport_t *vport, LPFC_SCSI_BUF_t * command)
{
	LPFC_NODELIST_t *ndlp;
	uint16_t okayToSchedule = 1;
	lpfcHBA_t *phba = vport->phba;

	/* If we have a command see if we can cut through */
	if (command != 0) {
		/* Just some short cuts */
		LPFC_SCHED_HBA_t *hbaSched = &vport->hbaSched;
		LPFC_SCHED_LUN_t *lunSched = &command->pLun->lunSched;
		LPFC_SCHED_TARGET_t *targetSched =
			&command->pLun->pTarget->targetSched;
		LPFC_IOCBQ_t *pIocbq = &command->cur_iocbq;
		LPFC_SLI_t *psli = &phba->sli;

		/*    Set it up so SLI calls us when it is done       */

		ndlp = command->pLun->pnode;
		if (ndlp == 0) {
			if(!(command->pLun->pTarget->targetFlags &
				FC_NPR_ACTIVE)) {
				return (1);
			}
			/* To be filled in later */
			pIocbq->iocb.ulpContext = 0;
			pIocbq->iocb.ulpFCP2Rcvy = 0;
			pIocbq->iocb.ulpClass = CLASS3;
		}
		else {
			/* put RPI number and NODELIST info in IOCB command */
			pIocbq->iocb.ulpContext = ndlp->nlp_rpi;
			if (ndlp->nlp_fcp_info & NLP_FCP_2_DEVICE) {
				pIocbq->iocb.ulpFCP2Rcvy = 1;
			}
			pIocbq->iocb.ulpClass = (ndlp->nlp_fcp_info & 0x0f);
		}

		pIocbq->context1 = command;
		pIocbq->iocb_cmpl = lpfc_sched_sli_done;
		pIocbq->vport = vport;
		if (ndlp &&
		    (hbaSched->currentOutstanding < hbaSched->maxOutstanding) &&
		    (hbaSched->status & LPFC_SCHED_STATUS_OKAYTOSEND) &&
		    (targetSched->lunCount == 0) &&
		    (targetSched->currentOutstanding <
		     targetSched->maxOutstanding)
		    && (targetSched->status & LPFC_SCHED_STATUS_OKAYTOSEND)
		    && (lunSched->currentOutstanding < lunSched->maxOutstanding)
		    && (lunSched->status & LPFC_SCHED_STATUS_OKAYTOSEND)
		    ) {

			/* The scheduler, target and lun are all in a position
			 * to accept a send operation.  Call the SLI layer and
			 * issue the IOCB.
			 */
			int sliStatus;
			sliStatus =
				lpfc_sli_issue_iocb(phba,
						    &psli->ring[psli->fcp_ring],
						    pIocbq, SLI_IOCB_RET_IOCB);

			switch (sliStatus) {
			case IOCB_ERROR:
				okayToSchedule = 0;
				lpfc_sched_queue_command(vport, command);
				break;
			case IOCB_BUSY:
			case IOCB_SUCCESS:
				lunSched->currentOutstanding++;
				targetSched->currentOutstanding++;
				hbaSched->currentOutstanding++;
				break;
			default:
				break;
			}

			/* Remove this state to cause a scan of queues if submit
			   worked. */
			okayToSchedule = 0;
		} else {
			/* This clause is execute only if there are outstanding
			 * commands in the scheduler.
			 */
			lpfc_sched_queue_command(vport, command);
		}
	}

	/* We either queued something or someone called us to schedule
	   so now go schedule. */
	if (okayToSchedule)
		lpfc_sched_internal_check(vport);
	return (0);
}

void
lpfc_sched_queue_command(lpfc_vport_t *vport, LPFC_SCSI_BUF_t * command)
{
	LPFCSCSILUN_t *lun = command->pLun;
	LPFC_SCHED_LUN_t *lunSched = &lun->lunSched;
	struct list_head *head;
	
	head = (struct list_head *)&lunSched->commandList;
	
	list_add_tail(&command->listentry, head);
	lunSched->q_cnt++;

	lpfc_sched_add_lun_to_ring(vport, lun);

	return;
}

void
lpfc_sched_add_target_to_ring(lpfc_vport_t *vport, LPFCSCSITARGET_t * target)
{
	LPFC_SCHED_TARGET_t *targetSched = &target->targetSched;
	LPFC_SCHED_HBA_t *hbaSched = &vport->hbaSched;

	if (!list_empty(&target->listentry) ||	/* Already on list */
	    (targetSched->lunCount == 0))	/* nothing to schedule */
		return;

	list_add_tail(&target->listentry, &hbaSched->targetRing);
	if ( hbaSched->targetCount == 0 ) {
		hbaSched->nextTargetToCheck = target;
	}
	hbaSched->targetCount++;
	return;
}

void
lpfc_sched_add_lun_to_ring(lpfc_vport_t *vport, LPFCSCSILUN_t * lun)
{
	LPFC_SCHED_LUN_t *lunSched = &lun->lunSched;
	LPFCSCSITARGET_t *target = lun->pTarget;
	LPFC_SCHED_TARGET_t *targetSched = &target->targetSched;

	if (!list_empty(&lun->listentry) ||	/* Already on list */
	    (lunSched->q_cnt == 0))	/* nothing to schedule */
		return;

	list_add_tail(&lun->listentry, &targetSched->lunRing);
	if ( targetSched->lunCount == 0 ) {
		targetSched->nextLunToCheck = lun;
	}

	targetSched->lunCount++;
	lpfc_sched_add_target_to_ring(vport, target);
	return;
}

void
lpfc_sched_remove_target_from_ring(lpfc_vport_t *vport, LPFCSCSITARGET_t * target)
{

	LPFC_SCHED_HBA_t *hbaSched = &vport->hbaSched;

	if (list_empty(&target->listentry))
		return;		/* Not on Ring */
	hbaSched->targetCount--;

	if ( hbaSched->targetCount ) {
		if ( hbaSched->nextTargetToCheck == target ) {
			if (target->listentry.next == &hbaSched->targetRing) {
				hbaSched->nextTargetToCheck  = list_entry( hbaSched->targetRing.next,
									   LPFCSCSITARGET_t,
									   listentry);
			} else {
				hbaSched->nextTargetToCheck = list_entry ( target->listentry.next,
									   LPFCSCSITARGET_t,
									   listentry);
			}
		}
	} else {
		hbaSched->nextTargetToCheck = 0;
	}

	list_del_init(&target->listentry);
	return;
}

void
lpfc_sched_remove_lun_from_ring(lpfc_vport_t *vport, LPFCSCSILUN_t * lun)
{
	LPFCSCSITARGET_t *target = lun->pTarget;
	LPFC_SCHED_TARGET_t *targetSched = &target->targetSched;

	if (list_empty(&lun->listentry))
		return;

	targetSched->lunCount--;

	if (targetSched->lunCount) {
		if ( targetSched->nextLunToCheck == lun ) {

			if (lun->listentry.next == &target->targetSched.lunRing) {
				targetSched->nextLunToCheck = 
					list_entry(target->targetSched.lunRing.next,
						   LPFCSCSILUN_t,
						   listentry);
			} else {
				targetSched->nextLunToCheck = 
					list_entry(lun->listentry.next,
						   LPFCSCSILUN_t,
						   listentry);
			}
		}
	} else
		targetSched->nextLunToCheck = 0; /*   Ring is empty */
	
	list_del_init(&lun->listentry);

	if (!targetSched->lunCount)
		lpfc_sched_remove_target_from_ring(vport, target);

	return;
}
