/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Fibre Channel Host Bus Adapters.                                *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003-2008 Emulex.  All rights reserved.           *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

/*
 * $Id: ioctls/lpfc_util_ioctl.c 1.9 2005/05/03 11:21:00EDT jsullivan Exp  $
 */
#include <linux/version.h>
#ifdef __VMKERNEL_MODULE__
#include "linux_skbuff.h"
#include "npiv_api_dist.h"
#endif // __VMKERNEL_MODULE__

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/blkdev.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/unistd.h>
#include <linux/timex.h>
#include <linux/timer.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/irq.h>
#include <sd.h>			/* From drivers/scsi */
#include <hosts.h>

#include "lpfc_hw.h"
#include "lpfc_cfgparm.h"
#include "lpfc_dfc.h"
#include "lpfc_sli.h"
#include "lpfc_mem.h"
#include "lpfc_sched.h"
#include "lpfc_disc.h"
#include "lpfc.h"
#include "lpfc_logmsg.h"
#include "lpfc_fcp.h"
#include "lpfc_scsi.h"
#include "lpfc_diag.h"
#include "lpfc_ioctl.h"
#include "lpfc_diag.h"
#include "lpfc_crtn.h"
#include "hbaapi.h"
#include "lpfc_util_ioctl.h"

#define LPFC_MAX_EVENT 4 /* Default events we can queue before dropping them */

extern void lpfc_check_menlo_cfg(lpfcHBA_t *phba);
extern int lpfc_mbox_tmo_val(lpfcHBA_t *phba, int cmd);

extern char *lpfc_release_version;
extern char *lpfc_drvr_name;
extern lpfcDRVR_t lpfcDRVR;

uint32_t lpfc_diag_state = DDI_ONDI;

int
dfc_rsp_data_copy_to_buf(lpfcHBA_t * phba,
		  uint8_t * outdataptr, DMABUFEXT_t * mlist, uint32_t size);

int
lpfc_ioctl_send_menlo_mgmt_cmd(lpfcHBA_t * phba,
			 LPFCCMDINPUT_t * cip, void *dataout);

DMABUFEXT_t *
dfc_fcp_cmd_data_alloc(lpfcHBA_t * phba,
                   char *indataptr, ULP_BDE64 * bpl,
                   uint32_t size, struct lpfc_dmabuf *bmp_list);

int
lpfc_ioctl_initboard(lpfcHBA_t * phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	struct pci_dev *pdev;
	struct dfc_info *di;
	char lpfc_fwrevision[32];

	pdev = phba->pcidev;
	if (!pdev)
		return 1;

	di = (struct dfc_info *) dataout;
	di->a_onmask = (ONDI_MBOX | ONDI_RMEM | ONDI_RPCI | ONDI_RCTLREG |
			ONDI_IOINFO | ONDI_LNKINFO | ONDI_NODEINFO |
			ONDI_CFGPARAM | ONDI_CT | ONDI_HBAAPI);
	di->a_offmask = (OFFDI_MBOX | OFFDI_RMEM | OFFDI_WMEM | OFFDI_RPCI |
			 OFFDI_WPCI | OFFDI_RCTLREG | OFFDI_WCTLREG);

	if (phba->fc_flag & FC_OFFLINE_MODE)
		di->a_offmask |= OFFDI_OFFLINE;

	if (lpfc_diag_state == DDI_ONDI)
		di->a_onmask |= ONDI_SLI2;
	else
		di->a_onmask |= ONDI_SLI1;

	/* set endianness of driver diagnotic interface */
#if __BIG_ENDIAN
	di->a_onmask |= ONDI_BIG_ENDIAN;
#else	/*  __LITTLE_ENDIAN */
	di->a_onmask |= ONDI_LTL_ENDIAN;
#endif

	di->a_pci = ((((uint32_t) pdev->device) << 16) |
		     (uint32_t) (pdev->vendor));
	di->a_ddi = phba->brd_no;

	if (pdev->bus)
		di->a_busid = (uint32_t) (pdev->bus->number);
	else
		di->a_busid = 0;

	di->a_devid = (uint32_t) (pdev->devfn);

	sprintf(di->a_drvrid, "%s", lpfc_release_version);
	lpfc_decode_firmware_rev(phba, lpfc_fwrevision, 1);
	sprintf(di->a_fwname, "%s",  lpfc_fwrevision);
	memcpy(di->a_wwpn, &phba->pport->fc_portname, sizeof(NAME_TYPE));
	cip->lpfc_outsz = sizeof (struct dfc_info);
	return (0);
}
/* Routine Declaration - Local */

int
lpfc_process_ioctl_util(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip)
{
	int rc = -1;
	uint32_t total_mem;
	void   *dataout;
	unsigned long iflag;

	if (cip->lpfc_outsz >= 4096) {
		/*
		 * Allocate memory for ioctl data. If buffer is bigger than 64k, then we
		 * allocate 64k and re-use that buffer over and over to xfer the whole 
		 * block. This is because Linux kernel has a problem allocating more than
		 * 120k of kernel space memory. Saw problem with GET_FCPTARGETMAPPING...
		 */
		if (cip->lpfc_outsz <= (64 * 1024))
			total_mem = cip->lpfc_outsz;
		else
			total_mem = 64 * 1024;		
	} else {
		/* Allocate memory for ioctl data */
		total_mem = 4096;
	}

	dataout = kmalloc(total_mem, GFP_KERNEL);
	memset (dataout, 0, total_mem);

	if (!dataout)
		return ENOMEM;

	switch (cip->lpfc_cmd) {
	case LPFC_WRITE_PCI:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_write_pci(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_READ_PCI:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_read_pci(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_WRITE_MEM:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_write_mem(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_READ_MEM:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_read_mem(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_WRITE_CTLREG:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_write_ctlreg(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_READ_CTLREG:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_read_ctlreg(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_GET_DFC_REV:
		((DfcRevInfo *) dataout)->a_Major = DFC_MAJOR_REV;
		((DfcRevInfo *) dataout)->a_Minor = DFC_MINOR_REV;
		cip->lpfc_outsz = sizeof (DfcRevInfo);
		rc = 0;
		break;

	case LPFC_INITBRDS:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_initboard(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_SETDIAG:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_setdiag(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_HBA_SEND_SCSI:
	case LPFC_HBA_SEND_FCP:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_send_scsi_fcp(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_SEND_ELS:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_send_els(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
 		break;
 
	case LPFC_HBA_SEND_MGMT_RSP:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_send_mgmt_rsp(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_HBA_SEND_MGMT_CMD:
	case LPFC_CT:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_send_mgmt_cmd(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_MENLO:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_send_menlo_mgmt_cmd(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_MBOX:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_mbox(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_LINKINFO:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_linkinfo(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_IOINFO:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_ioinfo(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_NODEINFO:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_nodeinfo(phba, cip, dataout, total_mem);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_GETCFG:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_getcfg(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_SETCFG:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_setcfg(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_HBA_GET_EVENT:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_hba_get_event(phba, cip, dataout, total_mem);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_TEMP_SENSOR_SUPPORT:
		rc = (phba->temp_sensor_support) ? 0 : ENODEV;
		break;

	case LPFC_HBA_SET_EVENT:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_hba_set_event(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_ADD_BIND:
		/* Adding bindings is unsupported. */
		rc = EPERM;
		break;
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_add_bind(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_DEL_BIND:
		/* Deleting bindings is unsupported. */
		rc = EPERM;
		break;
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_del_bind(phba, cip);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_LIST_BIND:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_list_bind(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_GET_VPD:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_get_vpd(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

        case LPFC_VPORT_GET_LIST:
                LPFC_DRVR_LOCK(phba, iflag);
                rc = lpfc_ioctl_vport_list(phba, cip, dataout);
                LPFC_DRVR_UNLOCK(phba, iflag);
                break;

	case LPFC_VPORT_GET_ATTRIB:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_vport_attrib(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_VPORT_GET_RESRC:
		/* Driver Lock is not needed */
		rc = lpfc_ioctl_vport_resrc(phba, cip, dataout);
		break;

	case LPFC_NPIV_READY:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_npiv_ready(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_VPORT_GET_NODE_INFO:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_vport_nodeinfo(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_GET_DUMPREGION:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_get_dumpregion(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_GET_LPFCDFC_INFO:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_get_lpfcdfc_info(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
 		break;

	case LPFC_LOOPBACK_MODE:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_loopback_mode(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
		break;

	case LPFC_LOOPBACK_TEST:
		LPFC_DRVR_LOCK(phba, iflag);
		rc = lpfc_ioctl_loopback_test(phba, cip, dataout);
		LPFC_DRVR_UNLOCK(phba, iflag);
 		break;
	}

	/* 
	 *Copy data to user space config method.  If return
	 * code is E2BIG, data length has been truncated
	 */
	if (rc == 0 || rc == E2BIG) {
		if (cip->lpfc_outsz) {
			if (copy_to_user((uint8_t *) cip->lpfc_dataout,
			     (uint8_t *) dataout, (int)cip->lpfc_outsz)) {
				rc = EIO;
			}
		}
	}

        kfree(dataout);
	return rc;
}

int
lpfc_ioctl_write_pci(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip)
{
	uint32_t offset, cnt;
	int i;
	unsigned long iflag;
	uint32_t *buffer;
	lpfc_vport_t *vport = phba->pport;

	offset = (ulong) cip->lpfc_arg1;
	cnt = (ulong) cip->lpfc_arg2;

	if (!(vport->fc_flag & FC_OFFLINE_MODE))
		return EPERM;

	if ((cnt + offset) > 256)
		return ERANGE;

	buffer = kmalloc(4096, GFP_ATOMIC);
	if (!buffer)
		return ENOMEM;

	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user(buffer, cip->lpfc_dataout, cnt)) {
		LPFC_DRVR_LOCK(phba, iflag);
		kfree(buffer);
		return EIO;
	}

	LPFC_DRVR_LOCK(phba, iflag);	
	if (!(phba->fc_flag & FC_OFFLINE_MODE))
		return EPERM;

	for (i = offset; i < (offset + cnt); i += 4) {
		pci_write_config_dword(phba->pcidev, i, *buffer);
		buffer++;
	}

	kfree(buffer);
	return 0;
}

int
lpfc_ioctl_read_pci(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip, void *dataout)
{
	uint32_t offset, cnt;
	uint32_t *destp;
	int i;

	offset = (ulong) cip->lpfc_arg1;
	cnt = (ulong) cip->lpfc_arg2;
	destp = (uint32_t *) dataout;

	if ((cnt + offset) > 4096)
		return ERANGE;

	for (i = offset; i < (offset + cnt); i += 4) {
		pci_read_config_dword(phba->pcidev, i, destp);
		destp++;
	}

	return 0;
}

int
lpfc_ioctl_write_mem(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip)
{
	uint32_t offset, cnt;
	LPFC_SLI_t *psli;
	unsigned long iflag;
	uint8_t *buffer;
	lpfc_vport_t *vport = phba->pport;

	psli = &phba->sli;
	offset = (ulong) cip->lpfc_arg1;
	cnt = (ulong) cip->lpfc_arg2;

	if (!(vport->fc_flag & FC_OFFLINE_MODE)) {
		if (offset != 256)
			return EPERM;

		/* Allow writing of first 128 bytes after mailbox in online mode */
		if (cnt > 128)
			return EPERM;
	}

	if (offset >= 4096)
		return ERANGE;

	cnt = (ulong) cip->lpfc_arg2;
	if ((cnt + offset) > 4096)
		return ERANGE;

	buffer =  kmalloc(4096, GFP_ATOMIC);
	if (!buffer)
		return ENOMEM;

	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user((uint8_t *) buffer, (uint8_t *) cip->lpfc_dataout,
			   (ulong) cnt)) {
		LPFC_DRVR_LOCK(phba, iflag);
		kfree(buffer);
		return EIO;
	}
	LPFC_DRVR_LOCK(phba, iflag);

	if (psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE) {
		/* copy into SLIM2 */
		lpfc_sli_pcimem_bcopy((uint32_t *) buffer,
				     ((uint32_t *) phba->slim2p.virt + offset),
				     cnt >> 2);
	} else {
		/* First copy command data */
		lpfc_memcpy_to_slim( phba->MBslimaddr, (void *)buffer, cnt);
	}
	kfree(buffer);
	return 0;
}

int
lpfc_ioctl_read_mem(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip, void *dataout)
{
	uint32_t offset, cnt;
	LPFC_SLI_t *psli;
	int i;

	psli = &phba->sli;
	offset = (ulong) cip->lpfc_arg1;
	cnt = (ulong) cip->lpfc_arg2;

	if (psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE) {
		/* The SLIM2 size is stored in the next field */
		i = phba->slim_size;
	} else {
		i = 4096;
	}

	if (offset >= i)
		return ERANGE;

	if ((cnt + offset) > i) {
		/* Adjust cnt instead of error ret */
		cnt = (i - offset);
	}

	if (psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE) {
		/* copy results back to user */
		lpfc_sli_pcimem_bcopy((uint32_t *) psli->MBhostaddr,
				     (uint32_t *) dataout, cnt);
	} else {
		/* First copy command data from SLIM */
		lpfc_memcpy_from_slim( dataout,
			       phba->MBslimaddr,
			       MAILBOX_CMD_SIZE );		
	}
	return 0;
}

int
lpfc_ioctl_write_ctlreg(lpfcHBA_t * phba,
			LPFCCMDINPUT_t * cip)
{
	uint32_t offset, incr;
	LPFC_SLI_t *psli;
	lpfc_vport_t *vport = phba->pport;

	psli = &phba->sli;
	offset = (ulong) cip->lpfc_arg1;
	incr = (ulong) cip->lpfc_arg2;

	if (!(vport->fc_flag & FC_OFFLINE_MODE))
		return EPERM;

	if (offset > 255)
		return ERANGE;

	if (offset % 4)
		return EINVAL;

	writel(incr, (phba->ctrl_regs_memmap_p) + offset);
	return 0;
}

int
lpfc_ioctl_read_ctlreg(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip, void *dataout)
{
	uint32_t offset, incr;

	offset = (ulong) cip->lpfc_arg1;

	if (offset > 255)
		return ERANGE;

	if (offset % 4)
		return EINVAL;

	incr = readl((phba->ctrl_regs_memmap_p) + offset);
	*((uint32_t *) dataout) = incr;

	return 0;
}

int
lpfc_ioctl_setdiag(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip, void *dataout)
{
	uint32_t offset;
	int rc = 0;
	lpfc_vport_t *vport = phba->pport;
	offset = (ulong) cip->lpfc_arg1;
        unsigned long iflag = 0;

	switch (offset) {
	case DDI_ONDI:
	case DDI_OFFDI:
	case DDI_SHOW:
		rc = ENXIO;
		break;

	case DDI_BRD_ONDI:
		if (vport->port_state < LPFC_LINK_DOWN) {
			if (lpfc_online(phba))
				rc = EIO;
		}
		*((uint32_t *) (dataout)) = DDI_ONDI;
		break;

	case DDI_BRD_OFFDI:
                if (phba->config[LPFC_CFG_ENABLE_RESET].a_current) {
			if (!(vport->fc_flag & FC_OFFLINE_MODE))
				lpfc_offline(phba);
			*((uint32_t *) (dataout)) = DDI_OFFDI;
		}

                lpfc_sli_brdrestart(phba); 
                LPFC_DRVR_UNLOCK(phba, iflag);
                lpfc_sleep_ms(phba, 2500);
                LPFC_DRVR_LOCK(phba, iflag);

                rc = lpfc_sli_brdready(phba, HS_FFRDY | HS_MBRDY);

		break;

	case DDI_BRD_WARMDI:
		if (vport->port_state >= LPFC_LINK_DOWN)
			lpfc_offline(phba);

		lpfc_reset_barrier(phba);
		lpfc_sli_brdreset(phba, 0);
 		lpfc_hba_down_post(phba);

 		rc = lpfc_sli_brdready(phba, HS_MBRDY);

		*((uint32_t *) (dataout)) = DDI_WARMDI;
		break;

	case DDI_BRD_DIAGDI:
		if (vport->port_state >= LPFC_LINK_DOWN) {
			phba->hba_state = LPFC_HBA_ERROR;
			lpfc_offline(phba);
		}

 		lpfc_sli_brdkill(phba); 

		*((uint32_t *) (dataout)) = DDI_DIAGDI;
		break;

	case DDI_BRD_SHOW:
		if (phba->hba_state == LPFC_HBA_ERROR) 
			*((uint32_t *) (dataout)) = DDI_DIAGDI;
		else switch (vport->port_state) {
			case LPFC_HBA_ERROR:
				*((uint32_t *) (dataout)) = DDI_DIAGDI;
				break;
			case LPFC_WARM_START:
				*((uint32_t *) (dataout)) = DDI_WARMDI;
				break;
			case LPFC_INIT_START:
				*((uint32_t *) (dataout)) = DDI_OFFDI;
				break;
			default:
				*((uint32_t *) (dataout)) = DDI_ONDI;
				break;
		}
		break;

	default:
		rc = ERANGE;
		break;
	}

	return rc;
}

void
lpfc_ioctl_timeout_iocb_cmpl(lpfcHBA_t *phba, LPFC_IOCBQ_t *cmd_iocb_q, 
			     LPFC_IOCBQ_t *rsp_iocb_q)
{
	struct lpfc_timedout_iocb_ctxt *iocb_ctxt = cmd_iocb_q->context1;
	struct list_head *curr, *next;
	struct lpfc_dmabuf *mlast;

	if (!iocb_ctxt) {
		if (cmd_iocb_q->context2) {
			lpfc_els_free_iocb(phba, cmd_iocb_q);
		} else {
			lpfc_iocb_free(phba, cmd_iocb_q);
		}
		return;
	}

	if (iocb_ctxt->outdmp)
		dfc_cmd_data_free(phba, iocb_ctxt->outdmp);

	if (iocb_ctxt->indmp)
		dfc_cmd_data_free(phba, iocb_ctxt->indmp);

	if (iocb_ctxt->mp) {
		lpfc_mbuf_free(phba, 
			       iocb_ctxt->mp->virt, 
			       iocb_ctxt->mp->phys);
		kfree(iocb_ctxt->mp);
	}

	if (iocb_ctxt->bmp) {
		list_for_each_safe(curr, next, &iocb_ctxt->bmp->list) {
			mlast = list_entry(curr, struct lpfc_dmabuf, list);
			list_del(&mlast->list);
			lpfc_mbuf_free(phba, mlast->virt, mlast->phys);
			kfree(mlast);
		}

		lpfc_mbuf_free(phba, 
			       iocb_ctxt->bmp->virt, 
			       iocb_ctxt->bmp->phys);
		kfree(iocb_ctxt->bmp);
	}

	if (iocb_ctxt->lpfc_cmd)
		lpfc_free_scsi_buf(iocb_ctxt->lpfc_cmd);

	lpfc_iocb_free(phba, cmd_iocb_q);

	if (iocb_ctxt->rspiocbq)
		lpfc_iocb_free(phba, iocb_ctxt->rspiocbq);

	kfree(iocb_ctxt);
}


int
lpfc_ioctl_send_scsi_fcp(lpfcHBA_t * phba,
			 LPFCCMDINPUT_t * cip)
{

	LPFC_SLI_t *psli = &phba->sli;
	lpfcCfgParam_t *clp;
	int reqbfrcnt;
	int snsbfrcnt;
	int j = 0;
	FCP_CMND *fcpcmd;
	FCP_RSP *fcprsp;
	ULP_BDE64 *bpl;
	LPFC_NODELIST_t *pndl;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_FCP_RING];
	LPFC_IOCBQ_t *cmdiocbq = 0;
	LPFC_IOCBQ_t *rspiocbq = 0;
	DMABUFEXT_t *outdmp = 0;
	IOCB_t *cmd = 0;
	IOCB_t *rsp = 0;
	DMABUF_t *mp = 0;
	DMABUF_t *bmp = 0;
	char *outdta;
	uint32_t clear_count;
	int rc = 0;
	unsigned long iflag;
	uint32_t iocb_wait_timeout = cip->lpfc_arg5;
	uint32_t iocb_retries;
	lpfc_vport_t *vport=0;
	struct lpfc_timedout_iocb_ctxt *iocb_ctxt;
	struct list_head *curr, *next;
	struct lpfc_dmabuf *mlast;
	IOargUn arg3;
	int found = 0;


	/*
	 * Rspcnt is really data buffer size
	 * Snscnt is sense count in case of LPFC_HBA_SEND_SCSI or
	 * it is fcp response size in case of LPFC_HBA_SEND_FCP
	 */
	struct {
		uint32_t rspcnt;
		uint32_t snscnt;
	} count;

	clp = &phba->config[0];
	reqbfrcnt = cip->lpfc_arg4;
	snsbfrcnt = cip->lpfc_flag;

	if ((reqbfrcnt + cip->lpfc_outsz) > (80 * 4096)) {
		rc = ERANGE;
		goto sndsczout;
	}

	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user((uint8_t *) &arg3, (uint8_t *) cip->lpfc_arg3,
			   (ulong) (sizeof (IOargUn)))) {
		rc = EIO;
		LPFC_DRVR_LOCK(phba, iflag);
		goto sndsczout;
	}
	LPFC_DRVR_LOCK(phba, iflag);


	/*
	 * If user passed a vport wwpn, use it.
	 * Else, use the phba's physical port.
	 */

	vport = phba->pport;
	if ((uint64_t)arg3.Iarg.vport_wwpn.wwn[0]) {
		list_for_each_entry(vport, &phba->port_list, listentry) {
			found = memcmp(&arg3.Iarg.vport_wwpn,
					&vport->fc_portname, sizeof (HBA_WWN));
			if (found)
				break;
		}

		if (found == 0) {
			rc = ENOENT;
			goto sndsczout;
                }
	}

	/*
	 * Using the target wwpn, get the target node pointer
	 */
	pndl = lpfc_findnode_wwpn(vport, NLP_SEARCH_MAPPED, 
				  (NAME_TYPE *) &arg3.Iarg.targ_wwpn);
	if (!pndl) {
		pndl = lpfc_findnode_wwpn(vport, NLP_SEARCH_UNMAPPED,
					   (NAME_TYPE *)&arg3.Iarg.targ_wwpn);
		if (!pndl || !(pndl->nlp_flag & NLP_TGT_NO_SCSIID))
			pndl = (LPFC_NODELIST_t *) 0;
	}

	if (!pndl || !(psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE)) {
		rc = EACCES;
		goto sndsczout;
	}

	if (pndl->nlp_flag & NLP_ELS_SND_MASK) {
		rc = ENODEV;
		goto sndsczout;
	}

	/* Allocate buffer for command iocb */
	if ((cmdiocbq = lpfc_iocb_alloc(phba, MEM_PRI)) == 0) {
		rc = ENOMEM;
		goto sndsczout;
	}

	cmdiocbq->vport = vport;
	cmd = &(cmdiocbq->iocb);

	/* Allocate buffer for response iocb */
	if ((rspiocbq = lpfc_iocb_alloc(phba, MEM_PRI)) == 0) {
		rc = ENOMEM;
		goto sndsczout;
	}

	rsp = &(rspiocbq->iocb);
	rspiocbq->vport = vport;

	/* Allocate buffer for Buffer ptr list */
	if (((bmp = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC)) == 0) ||
	    ((bmp->virt = lpfc_mbuf_alloc(phba, 0, &(bmp->phys))) == 0)) {
		if (bmp)
			kfree(bmp);
		bmp = NULL;
		rc = ENOMEM;
		goto sndsczout;
	}
	INIT_LIST_HEAD(&bmp->list);
	bpl = (ULP_BDE64 *) bmp->virt;

	/* Allocate buffer for FCP CMND / FCP RSP */
	if (((mp = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC)) == 0) ||
	    ((mp->virt = lpfc_mbuf_alloc(phba, MEM_PRI, &(mp->phys))) == 0)) {
		if (mp)
			kfree(mp);
		mp = NULL;
		rc = ENOMEM;
		goto sndsczout;
	}
	INIT_LIST_HEAD(&mp->list);
	fcpcmd = (FCP_CMND *) mp->virt;
	fcprsp = (FCP_RSP *) ((uint8_t *) mp->virt + sizeof (FCP_CMND));
	memset((void *)fcpcmd, 0, sizeof (FCP_CMND) + sizeof (FCP_RSP));

	/* Setup FCP CMND and FCP RSP */
	bpl->addrHigh = le32_to_cpu( putPaddrHigh(mp->phys) );
	bpl->addrLow = le32_to_cpu( putPaddrLow(mp->phys) );
	bpl->tus.f.bdeSize = sizeof (FCP_CMND);
	bpl->tus.f.bdeFlags = BUFF_USE_CMND;
	bpl->tus.w = le32_to_cpu(bpl->tus.w);
	bpl++;
	bpl->addrHigh = le32_to_cpu(putPaddrHigh(mp->phys + sizeof (FCP_CMND)));
	bpl->addrLow = le32_to_cpu(putPaddrLow(mp->phys + sizeof (FCP_CMND)));
	bpl->tus.f.bdeSize = sizeof (FCP_RSP);
	bpl->tus.f.bdeFlags = (BUFF_USE_CMND | BUFF_USE_RCV);
	bpl->tus.w = le32_to_cpu(bpl->tus.w);
	bpl++;

	/*
	 * Copy user data into fcpcmd buffer at this point to see if its a read
	 * or a write.
	 */
	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user((uint8_t *) fcpcmd, (uint8_t *) cip->lpfc_arg1,
			   (ulong) (reqbfrcnt))) {
		rc = EIO;
		LPFC_DRVR_LOCK(phba, iflag);
		goto sndsczout;
	}
	LPFC_DRVR_LOCK(phba, iflag);

	outdta = (fcpcmd->fcpCntl3 == WRITE_DATA ? cip->lpfc_dataout : 0);

	/* Allocate data buffer, and fill it if its a write */
	if (cip->lpfc_outsz == 0) {
		outdmp = dfc_fcp_cmd_data_alloc(phba, outdta, bpl, 512, bmp);
	} else {
		outdmp = dfc_fcp_cmd_data_alloc(phba, outdta, bpl, cip->lpfc_outsz, bmp);
	}
	if (outdmp == 0) {
		rc = ENOMEM;
		goto sndsczout;
	}

	cmd->un.fcpi64.bdl.ulpIoTag32 = 0;
	cmd->un.fcpi64.bdl.addrHigh = putPaddrHigh(bmp->phys);
	cmd->un.fcpi64.bdl.addrLow = putPaddrLow(bmp->phys);
	cmd->un.fcpi64.bdl.bdeSize = (3 * sizeof (ULP_BDE64));
	cmd->un.fcpi64.bdl.bdeFlags = BUFF_TYPE_BDL;
	cmd->ulpBdeCount = 1;
	cmd->ulpContext = pndl->nlp_rpi;
	cmd->ulpClass = pndl->nlp_fcp_info & 0x0f;
	cmd->ulpOwner = OWN_CHIP;
	cmd->ulpTimeout =
	    clp[LPFC_CFG_SCSI_REQ_TMO].a_current + phba->fcp_timeout_offset;
	cmd->ulpLe = 1;

	if (pndl->nlp_fcp_info & NLP_FCP_2_DEVICE)
		cmd->ulpFCP2Rcvy = 1;

	if ((fcpcmd->fcpCntl3) && (fcpcmd->fcpDl == 0))
		fcpcmd->fcpCntl3 = 0;

	switch (fcpcmd->fcpCntl3) {
	case READ_DATA:
		cmd->ulpCommand = CMD_FCP_IREAD64_CR;
		cmd->ulpPU = PARM_READ_CHECK;
		cmd->un.fcpi.fcpi_parm = cip->lpfc_outsz;
		cmd->un.fcpi64.bdl.bdeSize =
		    ((outdmp->flag + 2) * sizeof (ULP_BDE64));
		break;
	case WRITE_DATA:
		cmd->ulpCommand = CMD_FCP_IWRITE64_CR;
		cmd->un.fcpi64.bdl.bdeSize =
		    ((outdmp->flag + 2) * sizeof (ULP_BDE64));
		break;
	default:
		cmd->ulpCommand = CMD_FCP_ICMND64_CR;
		cmd->un.fcpi64.bdl.bdeSize = (2 * sizeof (ULP_BDE64));
		break;
	}
	cmdiocbq->context1 = (uint8_t *) 0;
	cmdiocbq->iocb_flag |= LPFC_IO_LIBDFC;

	/* Set up the timeout value for the iocb wait command. */
	if (iocb_wait_timeout == 0) {
	        iocb_wait_timeout = clp[LPFC_CFG_SCSI_REQ_TMO].a_current +
				    phba->fcp_timeout_offset + LPFC_DRVR_TIMEOUT;
		iocb_retries = 4;
	} else {
		iocb_retries = 1;
	}

	cmdiocbq->vport = vport;

	rc = lpfc_sli_issue_iocb_wait(phba, pring, cmdiocbq, SLI_IOCB_USE_TXQ,
				      rspiocbq, iocb_wait_timeout);
	if (rc == IOCB_TIMEDOUT) {
		iocb_ctxt = kmalloc(sizeof(struct lpfc_timedout_iocb_ctxt),
				    GFP_ATOMIC);
		if (!iocb_ctxt)
			return EIO;

		cmdiocbq->context1 = iocb_ctxt;
		cmdiocbq->context2 = NULL;
		iocb_ctxt->rspiocbq = rspiocbq;
		iocb_ctxt->mp = mp;
		iocb_ctxt->bmp = bmp;
		iocb_ctxt->outdmp = outdmp;
		iocb_ctxt->lpfc_cmd = NULL;
		iocb_ctxt->indmp = NULL;
		
		cmdiocbq->iocb_cmpl = lpfc_ioctl_timeout_iocb_cmpl;
		return EIO;
 	}

	if (rc != IOCB_SUCCESS) {
		rc = EIO;
		goto sndsczout;
	}

	/*
	 * For LPFC_HBA_SEND_FCP, just return FCP_RSP unless we got
	 * an IOSTAT_LOCAL_REJECT.
	 *
	 * For SEND_FCP case, snscnt is really FCP_RSP length. In the
	 * switch statement below, the snscnt should not get destroyed.
	 */
	if (cmd->ulpCommand == CMD_FCP_IWRITE64_CX) {
		clear_count = (rsp->ulpStatus == IOSTAT_SUCCESS ? 1 : 0);
	} else {
		clear_count = cmd->un.fcpi.fcpi_parm;
	}
	if ((cip->lpfc_cmd == LPFC_HBA_SEND_FCP) &&
	    (rsp->ulpStatus != IOSTAT_LOCAL_REJECT)) {
		if (snsbfrcnt < sizeof (FCP_RSP)) {
			count.snscnt = snsbfrcnt;
		} else {
			count.snscnt = sizeof (FCP_RSP);
		}
		LPFC_DRVR_UNLOCK(phba, iflag);
		if (copy_to_user((uint8_t *) cip->lpfc_arg2, (uint8_t *) fcprsp,
				 count.snscnt)) {
			rc = EIO;
			LPFC_DRVR_LOCK(phba, iflag);
			goto sndsczout;
		}
		LPFC_DRVR_LOCK(phba, iflag);
	}
	switch (rsp->ulpStatus) {
	case IOSTAT_SUCCESS:
	      cpdata:
		if (cip->lpfc_outsz < clear_count) {
			cip->lpfc_outsz = 0;
			rc = ERANGE;
			break;
		}
		cip->lpfc_outsz = clear_count;
		if (cip->lpfc_cmd == LPFC_HBA_SEND_SCSI) {
			count.rspcnt = cip->lpfc_outsz;
			count.snscnt = 0;
		} else {
			/* For LPFC_HBA_SEND_FCP, snscnt is already set */
			count.rspcnt = cip->lpfc_outsz;
		}
		LPFC_DRVR_UNLOCK(phba, iflag);
		/* Return data length */
		if (copy_to_user((uint8_t *)cip->lpfc_arg3, 
				(uint8_t *)&count,
				 (2 * sizeof (uint32_t)))) {
			rc = EIO;
			LPFC_DRVR_LOCK(phba, iflag);
			break;
		}
		LPFC_DRVR_LOCK(phba, iflag);
		cip->lpfc_outsz = 0;
		if (count.rspcnt) {
			if (dfc_rsp_data_copy
			    (phba, (uint8_t *) cip->lpfc_dataout, outdmp,
			     count.rspcnt)) {
				rc = EIO;
				break;
			}
		}
		break;
	case IOSTAT_LOCAL_REJECT:
		cip->lpfc_outsz = 0;
		if (rsp->un.grsp.perr.statLocalError == IOERR_SEQUENCE_TIMEOUT) {
			rc = ETIMEDOUT;
			break;
		}
		rc = EFAULT;

		/* count.rspcnt and count.snscnt is already 0 */
		goto sndsczout;
	case IOSTAT_FCP_RSP_ERROR:
		/* 
		 * At this point, clear_count is the residual count. 
		 * We are changing it to the amount actually xfered.
		 */
		if (fcpcmd->fcpCntl3 == READ_DATA) {
			if ((fcprsp->rspStatus2 & RESID_UNDER)
			    && (fcprsp->rspStatus3 == SCSI_STAT_GOOD)) {
				goto cpdata;
			}
		} else {
			clear_count = 0;
		}

		count.rspcnt = (uint32_t) clear_count;
		cip->lpfc_outsz = 0;
		if (fcprsp->rspStatus2 & RSP_LEN_VALID)
			j = be32_to_cpu(fcprsp->rspRspLen);

		if (fcprsp->rspStatus2 & SNS_LEN_VALID) {
			if (cip->lpfc_cmd == LPFC_HBA_SEND_SCSI) {
				if (snsbfrcnt < be32_to_cpu(fcprsp->rspSnsLen))
					count.snscnt = snsbfrcnt;
				else
					count.snscnt = be32_to_cpu(fcprsp->rspSnsLen);

				/* Return sense info from rsp packet */
				LPFC_DRVR_UNLOCK(phba, iflag);
				if (copy_to_user((uint8_t *) cip->lpfc_arg2,
						 ((uint8_t *) & fcprsp->rspInfo0) + j,
						 count.snscnt)) {
					rc = EIO;
					LPFC_DRVR_LOCK(phba, iflag);
					break;
				}
				LPFC_DRVR_LOCK(phba, iflag);
			}
		} else {
			rc = EFAULT;
			break;
		}

		LPFC_DRVR_UNLOCK(phba, iflag);
		if (copy_to_user((uint8_t *)cip->lpfc_arg3, 
				(uint8_t *)&count,
				 (2 * sizeof(uint32_t)))) {
			rc = EIO;
			LPFC_DRVR_LOCK(phba, iflag);
			break;
		}

		LPFC_DRVR_LOCK(phba, iflag);
		/* return data for read */
		if (count.rspcnt) {
			if (dfc_rsp_data_copy(phba, (uint8_t *) cip->lpfc_dataout,
					      outdmp, count.rspcnt)) {
				rc = EIO;
				break;
			}
		}
		break;
	default:
		cip->lpfc_outsz = 0;
		rc = EFAULT;
		break;
	}

 sndsczout:
	dfc_cmd_data_free(phba, outdmp);
	if (mp) {
		lpfc_mbuf_free(phba, mp->virt, mp->phys);
		kfree(mp);
	}

	if (bmp) {
		list_for_each_safe(curr, next, &bmp->list) {
			mlast = list_entry(curr, struct lpfc_dmabuf, list);
			list_del(&mlast->list);
			lpfc_mbuf_free(phba, mlast->virt, mlast->phys);
			kfree(mlast);
		}
		lpfc_mbuf_free(phba, bmp->virt, bmp->phys);
		kfree(bmp);
	}

	if (cmdiocbq)
		lpfc_iocb_free(phba, cmdiocbq);

	if (rspiocbq)
		lpfc_iocb_free(phba, rspiocbq);

	return rc;
}

int
lpfc_ioctl_send_els(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip)
{
	LPFC_SLI_t *psli = &phba->sli;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_ELS_RING];
	LPFC_IOCBQ_t *cmdiocbq, *rspiocbq;
	DMABUFEXT_t *pcmdext = 0, *prspext = 0;
	LPFC_NODELIST_t *pndl;
	ULP_BDE64 *bpl;
	IOCB_t *rsp;
	DMABUF_t *pcmd, *prsp, *pbuflist = 0;
	unsigned long iflag;
	uint16_t rpi = 0;
	DestID destID;
	int rc = 0;
	uint32_t cmdsize;
	uint32_t rspsize;
	uint32_t elscmd;

	cmdsize = cip->lpfc_arg4;
	rspsize = cip->lpfc_outsz;

        LPFC_DRVR_UNLOCK(phba, iflag);
        if (copy_from_user((uint8_t *)&elscmd, (uint8_t *)cip->lpfc_arg2, 
				min(sizeof(elscmd),cmdsize))) {
		LPFC_DRVR_LOCK(phba, iflag);
		return EIO;
        }
	if (copy_from_user((uint8_t *)&destID, (uint8_t *)cip->lpfc_arg1,
			   (ulong)(sizeof(DestID)))) {
		LPFC_DRVR_LOCK(phba, iflag);
		return EIO;
	}
	LPFC_DRVR_LOCK(phba, iflag);

	if ((rspiocbq = lpfc_iocb_alloc(phba, 0)) == 0)
		return ENOMEM;

	memset(rspiocbq, 0, sizeof (LPFC_IOCBQ_t));
	rsp = &rspiocbq->iocb;

	if (destID.idType == 0) {
		pndl = lpfc_findnode_wwpn(phba->pport, NLP_SEARCH_ALL,
					  (NAME_TYPE *)&destID.wwpn);
	} else {
		destID.d_id = (destID.d_id & Mask_DID);
		pndl = lpfc_findnode_did(phba->pport, NLP_SEARCH_ALL, destID.d_id);
	}

	if (pndl == 0) {
		if (destID.idType == 0) {
			lpfc_iocb_free(phba, rspiocbq);
			return ENODEV;
		}
		pndl = kmalloc(sizeof (LPFC_NODELIST_t), GFP_ATOMIC);
		if (!pndl) {
			lpfc_iocb_free(phba, rspiocbq);
			return ENODEV;
		}
		/* lpfc_nlp_init(phba, pndl, destID.d_id); */
	} else {
		rpi = pndl->nlp_rpi;
	}

	cmdiocbq = lpfc_prep_els_iocb(phba->pport, 1, cmdsize, 0, pndl, elscmd);
	if (rpi == 0)
		kfree(pndl);

	if (cmdiocbq == 0) {
		lpfc_iocb_free(phba, rspiocbq);
		return EIO;
	}

	pcmd = (DMABUF_t *) cmdiocbq->context2;
	prsp = (DMABUF_t *) pcmd->list.next;

	/* If we exceed the size of the allocated mbufs we need to */
	/* free them and allocate our own. */
	if ((cmdsize > LPFC_BPL_SIZE) || (rspsize > LPFC_BPL_SIZE)) {
		lpfc_mbuf_free(phba, pcmd->virt, pcmd->phys);
		kfree(pcmd);
		lpfc_mbuf_free(phba, prsp->virt, prsp->phys);
		kfree(prsp);
		cmdiocbq->context2 = 0;

		pbuflist = (DMABUF_t *) cmdiocbq->context3;
		bpl = (ULP_BDE64 *) pbuflist->virt;
		pcmdext = dfc_cmd_data_alloc(phba, cip->lpfc_arg2, bpl, cmdsize);
		if (!pcmdext) {
			lpfc_els_free_iocb(phba, cmdiocbq);
			lpfc_iocb_free(phba, rspiocbq);
			return ENOMEM;
		}
		bpl += pcmdext->flag; 
		prspext = dfc_cmd_data_alloc(phba, 0, bpl, rspsize);
		if (!prspext) {
			dfc_cmd_data_free(phba, pcmdext);
			lpfc_els_free_iocb(phba, cmdiocbq);
			lpfc_iocb_free(phba, rspiocbq);
			return ENOMEM;
		}
	} else {
		/* Copy the command from user space */
		LPFC_DRVR_UNLOCK(phba, iflag);
		if (copy_from_user((uint8_t *) pcmd->virt,
				   (uint8_t *) cip->lpfc_arg2,
				   cmdsize)) {
			LPFC_DRVR_LOCK(phba, iflag);
			lpfc_els_free_iocb(phba, cmdiocbq);
			lpfc_iocb_free(phba, rspiocbq);
			return EIO;
		}
		LPFC_DRVR_LOCK(phba, iflag);
	}

	cmdiocbq->iocb.ulpContext = rpi;
	cmdiocbq->iocb_flag |= LPFC_IO_LIBDFC;
	cmdiocbq->context1 = 0;
	cmdiocbq->context2 = 0;

	rc = lpfc_sli_issue_iocb_wait(phba, pring, cmdiocbq, SLI_IOCB_USE_TXQ,
				      rspiocbq, (phba->fc_ratov*2) + LPFC_DRVR_TIMEOUT);
	if (rc == IOCB_SUCCESS) {
		if (rsp->ulpStatus == IOSTAT_SUCCESS) {
			if (pbuflist) {
				if (dfc_rsp_data_copy(
					phba,
					(uint8_t *) cip->lpfc_dataout,
					prspext,
					rspsize)) {
					rc = EIO;
				} else {
					cip->lpfc_outsz = 0;
				}
			} else {
				LPFC_DRVR_UNLOCK(phba, iflag);
				if (copy_to_user(
					(uint8_t *) cip->lpfc_dataout,
					(uint8_t *) prsp->virt,
					rspsize)) {
					rc = EIO;
				} else {
					cip->lpfc_outsz = 0;
				}
				LPFC_DRVR_LOCK(phba, iflag);
			}
		} else if (rsp->ulpStatus == IOSTAT_LS_RJT) {
			uint8_t ls_rjt[8];

			/* construct the LS_RJT payload */
			ls_rjt[0] = 0x01;
			ls_rjt[1] = 0x00;
			ls_rjt[2] = 0x00;
			ls_rjt[3] = 0x00;
			memcpy(&ls_rjt[4], (uint8_t *) &rsp->un.ulpWord[4],
			       sizeof(uint32_t));

			if (rspsize < 8) {
				rc = ERANGE;
			} else {
				rspsize = 8;
			}

			if (rspsize) {
				LPFC_DRVR_UNLOCK(phba, iflag);
				if (copy_to_user((uint8_t *) cip->lpfc_dataout,
						ls_rjt, rspsize)) {
					rc = EIO;
				}
				else {
					cip->lpfc_outsz = 0;
				}
				LPFC_DRVR_LOCK(phba, iflag);
			}
		} else {
			rc = EIO;
		}

		LPFC_DRVR_UNLOCK(phba, iflag);
		if (copy_to_user((uint8_t *)cip->lpfc_arg3,
				 (uint8_t *)&rspsize, sizeof(uint32_t))) {
			rc = EIO;
		}
		LPFC_DRVR_LOCK(phba, iflag);
	} else {
		rc = EIO;
	}

	if (pbuflist) {
		dfc_cmd_data_free(phba, pcmdext);
		dfc_cmd_data_free(phba, prspext);
	} else {
		cmdiocbq->context2 = (uint8_t *) pcmd;
	}

	lpfc_els_free_iocb(phba, cmdiocbq);
	lpfc_iocb_free(phba, rspiocbq);
	return rc;
}

int
lpfc_ioctl_send_mgmt_rsp(lpfcHBA_t * phba,
			 LPFCCMDINPUT_t * cip)
{
	ULP_BDE64 *bpl;
	DMABUF_t *bmp;
	DMABUFEXT_t *indmp;
	uint32_t tag;
	int reqbfrcnt;
	int rc = 0;
	lpfc_vport_t *vport = phba->pport;

	tag = (uint32_t) cip->lpfc_flag;	/* XRI for XMIT_SEQUENCE */
	reqbfrcnt = (ulong) cip->lpfc_arg2;

	if ((reqbfrcnt == 0) || (reqbfrcnt > (80 * 4096)))
		return ERANGE;

	if (((bmp = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC)) == 0) ||
	    ((bmp->virt = lpfc_mbuf_alloc(phba, 0, &(bmp->phys))) == 0)) {
		if (bmp)
			kfree(bmp);
		return ENOMEM;
	}

	INIT_LIST_HEAD(&bmp->list);
	bpl = (ULP_BDE64 *) bmp->virt;

	if ((indmp = dfc_cmd_data_alloc(phba, (char *)cip->lpfc_arg1, bpl,
					reqbfrcnt)) == 0) {
		lpfc_mbuf_free(phba, bmp->virt, bmp->phys);
		kfree(bmp);
		return ENOMEM;
	}

	/* flag contains total number of BPLs for xmit */
	if ((rc = lpfc_issue_ct_rsp(vport, tag, bmp, indmp))) {
		if (rc == IOCB_TIMEDOUT)
			rc = ETIMEDOUT;
		else if (rc == IOCB_ERROR)
			rc = EACCES;
	}

	dfc_cmd_data_free(phba, indmp);
	lpfc_mbuf_free(phba, bmp->virt, bmp->phys);
	kfree(bmp);

	return rc;
}

typedef struct lpfc_build_iocb {
	LPFC_IOCBQ_t *cmdiocbq;
	LPFC_IOCBQ_t *rspiocbq;
	int    cmdsize;
	int    rspsize;
	struct lpfc_dmabuf *bmp;
	DMABUFEXT_t *indmp; 
	DMABUFEXT_t *outdmp;
	uint8_t   *datain_ptr;
}lpfc_build_iocb_t;

void
lpfc_iocb_cleanup(lpfcHBA_t *phba,
	struct lpfc_build_iocb *piocb)
{
	if (piocb->outdmp)
		dfc_cmd_data_free(phba, piocb->outdmp);
	if (piocb->indmp)
		dfc_cmd_data_free(phba, piocb->indmp);
	if (piocb->bmp && piocb->bmp->virt && piocb->bmp->phys)
		lpfc_mbuf_free(phba, piocb->bmp->virt, piocb->bmp->phys);
	if (piocb->bmp)
		kfree(piocb->bmp);
	if (piocb->rspiocbq)
        	lpfc_iocb_free(phba, piocb->rspiocbq);
	if (piocb->cmdiocbq)
		lpfc_iocb_free(phba, piocb->cmdiocbq);
}

/*
 * This routine is menlo specific, it will always set the DID to 
 * 0xFC0E (menlo DID).
 */
int
lpfc_prep_iocb(lpfcHBA_t *phba,
	struct lpfc_build_iocb *piocb)
{
	ULP_BDE64 *bpl = NULL;
	IOCB_t *cmd = NULL, *rsp = NULL;
	int rc = 0;

	if (!piocb->cmdsize || !piocb->rspsize 
		|| (piocb->cmdsize + piocb->rspsize > 80 * 4096)) {
		rc = ERANGE;
		goto prep_iocb_exit;
	}

	if ((piocb->cmdiocbq = lpfc_iocb_alloc(phba, MEM_PRI)) == 0) {
		rc = ENOMEM;
		goto prep_iocb_exit;
	}

	memset(piocb->cmdiocbq, 0, sizeof (LPFC_IOCBQ_t));
	cmd = &piocb->cmdiocbq->iocb;

	if ((piocb->rspiocbq = lpfc_iocb_alloc(phba, MEM_PRI)) == 0) {
		rc = ENOMEM;
		goto prep_iocb_err;
	}

	memset(piocb->rspiocbq, 0, sizeof (LPFC_IOCBQ_t));
	rsp = &piocb->rspiocbq->iocb;

	piocb->bmp = kmalloc(sizeof (struct lpfc_dmabuf), GFP_ATOMIC);
	if (!piocb->bmp) {
		rc = ENOMEM;
		goto prep_iocb_err;
	}
	piocb->bmp->virt = lpfc_mbuf_alloc(phba, 0, &piocb->bmp->phys);
	if (!piocb->bmp->virt) {
		rc = ENOMEM;
		goto prep_iocb_err;
	}

	INIT_LIST_HEAD(&piocb->bmp->list);
	bpl = (ULP_BDE64*) piocb->bmp->virt;

	piocb->indmp = dfc_cmd_data_alloc(phba, piocb->datain_ptr, bpl, piocb->cmdsize);

	if (!piocb->indmp) {
		rc = ENOMEM;
		goto prep_iocb_err;
	}

	/* flag contains total number of BPLs for xmit */
	bpl += piocb->indmp->flag; 

	piocb->outdmp = dfc_cmd_data_alloc(phba, 0, bpl, piocb->rspsize);
	if (!piocb->outdmp) {
		rc = ENOMEM;
		goto prep_iocb_err;
	}

	cmd->un.genreq64.bdl.ulpIoTag32 = 0;
	cmd->un.genreq64.bdl.addrHigh = putPaddrHigh(piocb->bmp->phys);
	cmd->un.genreq64.bdl.addrLow = putPaddrLow(piocb->bmp->phys);
	cmd->un.genreq64.bdl.bdeFlags = BUFF_TYPE_BDL;
	cmd->un.genreq64.bdl.bdeSize =
	    (piocb->outdmp->flag + piocb->indmp->flag) * sizeof (ULP_BDE64);
	cmd->ulpCommand = CMD_GEN_REQUEST64_CR;
	cmd->un.genreq64.w5.hcsw.Fctl = (SI | LA);
	cmd->un.genreq64.w5.hcsw.Dfctl = 0;
	cmd->un.genreq64.w5.hcsw.Rctl = FC_FCP_CMND;
	cmd->un.genreq64.w5.hcsw.Type = MENLO_TRANSPORT_TYPE; /* 0xfe */
	cmd->un.ulpWord[4] = MENLO_DID; /* 0x0000FC0E */
	cmd->ulpBdeCount = 1;
	cmd->ulpLe = 1;
	cmd->ulpPU = MENLO_PU;
	cmd->ulpClass = CLASS3;
	cmd->ulpOwner = OWN_CHIP;
	piocb->cmdiocbq->context1 = (uint8_t *) 0;
	piocb->cmdiocbq->context2 = (uint8_t *) 0;
	piocb->cmdiocbq->iocb_flag |= LPFC_IO_LIBDFC;

	cmd->ulpTimeout = 25;

prep_iocb_exit:
	return rc;
prep_iocb_err:
	lpfc_iocb_cleanup(phba,piocb);
	return rc;
}

int
lpfc_ioctl_send_menlo_mgmt_cmd(lpfcHBA_t * phba,
			 LPFCCMDINPUT_t * cip, void *dataout)
{
	IOCB_t *cmd = NULL, *rsp = NULL;
	LPFC_SLI_t *psli = &phba->sli;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_ELS_RING];
	int rc = 0;
	unsigned long iflag;
	uint32_t cmdcode;
	struct lpfc_timedout_iocb_ctxt *iocb_ctxt;
	struct lpfc_build_iocb iocb;
	struct lpfc_build_iocb *piocb = &iocb;
	uint16_t  ulpCtxt;

	if ((phba->hba_state == LPFC_LINK_DOWN) || 
	    (phba->pport->port_state == LPFC_LINK_DOWN)) {
		if (!(phba->sli.sliinit.sli_flag & LPFC_MENLO_MAINT)) {
			rc = EACCES;
			goto send_menlo_mgmt_cmd_exit;
		}
	}

	if (!(psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE)) {
		rc = EACCES;
		goto send_menlo_mgmt_cmd_exit;
	}
	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user((uint8_t *)&cmdcode, (uint8_t *)cip->lpfc_arg1,
			min(sizeof(cmdcode), (uint32_t)cip->lpfc_arg2))) {
		LPFC_DRVR_LOCK(phba, iflag);
		return EIO;
	}

	LPFC_DRVR_LOCK(phba, iflag);

	if (cmdcode == MENLO_CMD_FW_DOWNLOAD) {
		piocb->cmdsize = MENLO_CMD_HDR_SIZE;
		piocb->rspsize = 4;
	} else {
		piocb->cmdsize = (uint32_t)(unsigned long)cip->lpfc_arg2;
		piocb->rspsize = (int)cip->lpfc_outsz;
	}

	piocb->datain_ptr = (uint8_t *) cip->lpfc_arg1;
	rc = lpfc_prep_iocb(phba, piocb);
	if (rc) {
		rc = ENOMEM;
		goto send_menlo_mgmt_cmd_exit;
	}
	cmd = &piocb->cmdiocbq->iocb;
issueIocb:
	rsp = &piocb->rspiocbq->iocb;
	rc = lpfc_sli_issue_iocb_wait(phba, pring, piocb->cmdiocbq,
			 SLI_IOCB_USE_TXQ, piocb->rspiocbq, 30);
	if (rc == IOCB_TIMEDOUT) {
		lpfc_iocb_free(phba, piocb->rspiocbq);
		iocb_ctxt = kmalloc(sizeof(struct lpfc_timedout_iocb_ctxt),
				    GFP_ATOMIC);
		if (!iocb_ctxt) {
			return EACCES;
		}
		
		piocb->cmdiocbq->context1 = iocb_ctxt;
		piocb->cmdiocbq->context2 = NULL;
		iocb_ctxt->rspiocbq = NULL;
		iocb_ctxt->mp = NULL;
		iocb_ctxt->bmp = piocb->bmp;
		iocb_ctxt->outdmp = piocb->outdmp;
		iocb_ctxt->lpfc_cmd = NULL;
		iocb_ctxt->indmp = piocb->indmp;
		
		piocb->cmdiocbq->iocb_cmpl = lpfc_ioctl_timeout_iocb_cmpl;
		return EACCES;			
	}

	if (rc != IOCB_SUCCESS) {
		rc = EACCES;
		goto send_menlo_mgmt_cmd_free_piocb;
	}

	if (rsp->ulpStatus) {
		if (rsp->ulpStatus == IOSTAT_LOCAL_REJECT) {
			switch (rsp->un.ulpWord[4] & 0xff) {
			case IOERR_SEQUENCE_TIMEOUT:
				rc = ETIMEDOUT;
				break;
			case IOERR_INVALID_RPI:
				rc = EFAULT;
				break;
			default:
				rc = EACCES;
				break;
			}
			goto send_menlo_mgmt_cmd_free_piocb;
		}
	} else {
		piocb->outdmp->flag = rsp->un.genreq64.bdl.bdeSize;
	}

	/* Copy back response data */
	if (piocb->outdmp->flag > piocb->rspsize) {
		rc = ERANGE;
		goto send_menlo_mgmt_cmd_free_piocb;
	}

	if (cmdcode == MENLO_CMD_FW_DOWNLOAD 
		&& piocb->cmdiocbq->iocb.ulpCommand == CMD_GEN_REQUEST64_CR) {
		ulpCtxt = rsp->ulpContext;	
		lpfc_iocb_cleanup(phba,piocb);
		piocb->cmdsize = (uint32_t)(unsigned long)cip->lpfc_arg2 
				- MENLO_CMD_HDR_SIZE;
		piocb->rspsize = (int)cip->lpfc_outsz;
		piocb->datain_ptr = (uint8_t *) cip->lpfc_arg1
				+ MENLO_CMD_HDR_SIZE;
		rc = lpfc_prep_iocb(phba, piocb);
		if (rc) {
			rc = ENOMEM;
			goto send_menlo_mgmt_cmd_exit;
		}
		cmd = &piocb->cmdiocbq->iocb;
		cmd->ulpContext = ulpCtxt;
		cmd->ulpCommand = CMD_GEN_REQUEST64_CX;
		cmd->un.ulpWord[4] = 0;
		cmd->ulpPU = 1;
		goto issueIocb;
	}

	/* copy back size of response, and response itself */
	memcpy(dataout, &piocb->outdmp->flag, sizeof (int));

	rc = 0;
	rc = dfc_rsp_data_copy_to_buf (phba, dataout, piocb->outdmp, piocb->outdmp->flag);
	if (rc)
		rc = EIO;

send_menlo_mgmt_cmd_free_piocb:

	lpfc_iocb_cleanup(phba,piocb);
send_menlo_mgmt_cmd_exit:
	return rc;
}

int
lpfc_ioctl_send_mgmt_cmd(lpfcHBA_t * phba,
			 LPFCCMDINPUT_t * cip, void *dataout)
{
	LPFC_NODELIST_t *pndl;
	ULP_BDE64 *bpl;
	HBA_WWN findwwn;
	uint32_t finddid;
	LPFC_IOCBQ_t *cmdiocbq = 0;	/* Initialize the command iocb queue to a 0 default. */
	LPFC_IOCBQ_t *rspiocbq = 0;	/* Initialize the response iocb queue to a 0 default. */
	DMABUFEXT_t *indmp = 0;
	DMABUFEXT_t *outdmp = 0;
	IOCB_t *cmd = 0;
	IOCB_t *rsp = 0;
	DMABUF_t *bmp = 0;
	LPFC_SLI_t *psli = &phba->sli;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_ELS_RING];	/* els ring */
	int i0 = 0, rc = 0;
	int reqbfrcnt;
	int snsbfrcnt;
	uint32_t timeout;
	unsigned long iflag;
	lpfc_vport_t *vport = phba->pport;
	struct lpfc_timedout_iocb_ctxt *iocb_ctxt;

	/*
	 * CR 82786: We cannot access the device if the
	 * HBA is not ready.
	 */
	if (phba->pport->port_state != LPFC_HBA_READY)
		return EACCES;

	if (!(psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE))
		return EACCES;

	reqbfrcnt = cip->lpfc_arg4;
	snsbfrcnt = cip->lpfc_arg5;

	if (!reqbfrcnt || !snsbfrcnt || (reqbfrcnt + snsbfrcnt) > (80 * 4096)) {
		rc = ERANGE;
		goto sndmgtqwt;
	}

	if (cip->lpfc_cmd == LPFC_HBA_SEND_MGMT_CMD) {
		LPFC_DRVR_UNLOCK(phba, iflag);
		if (copy_from_user((uint8_t *) & findwwn, (uint8_t *) cip->lpfc_arg3,
				   (ulong) (sizeof (HBA_WWN)))) {
			rc = EIO;
			LPFC_DRVR_LOCK(phba, iflag);
			goto sndmgtqwt;
		}
		LPFC_DRVR_LOCK(phba, iflag);

		pndl = lpfc_findnode_wwpn(vport,
				          NLP_SEARCH_MAPPED | NLP_SEARCH_UNMAPPED,
				          (NAME_TYPE *) & findwwn);

		if (!pndl) {
			rc = ENODEV;
			goto sndmgtqwt;
		}
	} else {
		finddid = (uint32_t)((unsigned long)cip->lpfc_arg3);
		if (!(pndl = lpfc_findnode_did(vport, 
					       NLP_SEARCH_MAPPED |
					       NLP_SEARCH_UNMAPPED, finddid))) {
			if (vport->fc_flag & FC_FABRIC) {
				if ((pndl = lpfc_nlp_alloc(phba, 0)) == 0) {
					rc = ENODEV;
					goto sndmgtqwt;
				}

				memset(pndl, 0, sizeof (LPFC_NODELIST_t));
				pndl->nlp_DID = finddid;
				pndl->nlp_state = NLP_STE_PLOGI_ISSUE;
				lpfc_nlp_plogi(vport, pndl);

				if (lpfc_issue_els_plogi(vport, pndl, 0)) {
					lpfc_nlp_free(phba, pndl);
					rc = ENODEV;
					goto sndmgtqwt;
				}

				/* Allow the node to complete discovery */
				while ((i0++ < 4) &&
				       !(pndl = lpfc_findnode_did(vport,
								  NLP_SEARCH_MAPPED |
								  NLP_SEARCH_UNMAPPED, finddid))) {
					LPFC_DRVR_UNLOCK(phba, iflag);
					lpfc_sleep_ms(phba, 500);
					LPFC_DRVR_LOCK(phba, iflag);
				}

				if (i0 == 4) {
					rc = ENODEV;
					goto sndmgtqwt;
				}
			}
			else {
				rc = ENODEV;
				goto sndmgtqwt;
			}
		}
	}

	if (!pndl) {
		rc = EACCES;
		goto sndmgtqwt;
	}

	if (pndl->nlp_flag & NLP_ELS_SND_MASK) {
		rc = ENODEV;
		goto sndmgtqwt;
	}

	if ((cmdiocbq = lpfc_iocb_alloc(phba, MEM_PRI)) == 0) {
		rc = ENOMEM;
		goto sndmgtqwt;
	}

	cmd = &(cmdiocbq->iocb);
	cmdiocbq->vport = vport;

	if ((rspiocbq = lpfc_iocb_alloc(phba, MEM_PRI)) == 0) {
		rc = ENOMEM;
		goto sndmgtqwt;
	}

	rsp = &(rspiocbq->iocb);
	rspiocbq->vport = vport;

	if (((bmp = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC)) == 0) ||
	    ((bmp->virt = lpfc_mbuf_alloc(phba, 0, &(bmp->phys))) == 0)) {
		if (bmp)
			kfree(bmp);
		bmp = NULL;
		rc = ENOMEM;
		goto sndmgtqwt;
	}

	INIT_LIST_HEAD(&bmp->list);
	bpl = (ULP_BDE64 *) bmp->virt;
	if ((indmp = dfc_cmd_data_alloc(phba, (char *)cip->lpfc_arg1, bpl,
					reqbfrcnt)) == 0) {
		lpfc_mbuf_free(phba, bmp->virt, bmp->phys);
		kfree(bmp);
		bmp = NULL;
		rc = ENOMEM;
		goto sndmgtqwt;
	}
	bpl += indmp->flag;	/* flag contains total number of BPLs for xmit */
	if ((outdmp = dfc_cmd_data_alloc(phba, 0, bpl, snsbfrcnt)) == 0) {
		dfc_cmd_data_free(phba, indmp);
		indmp = NULL;
		lpfc_mbuf_free(phba, bmp->virt, bmp->phys);
		kfree(bmp);
		bmp = NULL;
		rc = ENOMEM;
		goto sndmgtqwt;
	}

	cmd->un.genreq64.bdl.ulpIoTag32 = 0;
	cmd->un.genreq64.bdl.addrHigh = putPaddrHigh(bmp->phys);
	cmd->un.genreq64.bdl.addrLow = putPaddrLow(bmp->phys);
	cmd->un.genreq64.bdl.bdeFlags = BUFF_TYPE_BDL;
	cmd->un.genreq64.bdl.bdeSize =
	    (outdmp->flag + indmp->flag) * sizeof (ULP_BDE64);
	cmd->ulpCommand = CMD_GEN_REQUEST64_CR;
	cmd->un.genreq64.w5.hcsw.Fctl = (SI | LA);
	cmd->un.genreq64.w5.hcsw.Dfctl = 0;
	cmd->un.genreq64.w5.hcsw.Rctl = FC_UNSOL_CTL;
	cmd->un.genreq64.w5.hcsw.Type = FC_COMMON_TRANSPORT_ULP;
	cmd->ulpTimeout = cip->lpfc_flag;
	cmd->ulpBdeCount = 1;
	cmd->ulpLe = 1;
	cmd->ulpClass = CLASS3;
	cmd->ulpContext = pndl->nlp_rpi;
	cmd->ulpOwner = OWN_CHIP;
	cmdiocbq->context1 = (uint8_t *) 0;
	cmdiocbq->context2 = (uint8_t *) 0;
	cmdiocbq->iocb_flag |= LPFC_IO_LIBDFC;

        if (cip->lpfc_flag == 0 )
                timeout = phba->fc_ratov * 2 ;
        else
                timeout = cip->lpfc_flag;

        cmd->ulpTimeout = timeout;

	cmdiocbq->vport = vport;
	rc = lpfc_sli_issue_iocb_wait(phba, pring, cmdiocbq, SLI_IOCB_USE_TXQ,
				      rspiocbq, timeout + LPFC_DRVR_TIMEOUT);
	if (rc == IOCB_TIMEDOUT) {
		lpfc_iocb_free(phba, rspiocbq);
		iocb_ctxt = kmalloc(sizeof(struct lpfc_timedout_iocb_ctxt),
				    GFP_ATOMIC);
		if (!iocb_ctxt){
			return EACCES;
		}
		
		cmdiocbq->context1 = iocb_ctxt;
		cmdiocbq->context2 = NULL;
		iocb_ctxt->rspiocbq = NULL;
		iocb_ctxt->mp = NULL;
		iocb_ctxt->bmp = bmp;
		iocb_ctxt->outdmp = outdmp;
		iocb_ctxt->lpfc_cmd = NULL;
		iocb_ctxt->indmp = indmp;
		
		cmdiocbq->iocb_cmpl = lpfc_ioctl_timeout_iocb_cmpl;
		return EACCES;			
 	}

	if (rc != IOCB_SUCCESS) {
		rc = EACCES;
		goto sndmgtqwt;
	}

	if (rsp->ulpStatus) {
		if (rsp->ulpStatus == IOSTAT_LOCAL_REJECT) {
			switch (rsp->un.ulpWord[4] & 0xff) {
			case IOERR_SEQUENCE_TIMEOUT:
				rc = ETIMEDOUT;
				break;
			case IOERR_INVALID_RPI:
				rc = EFAULT;
				break;
			default:
				rc = EACCES;
				break;
			}

			goto sndmgtqwt;
		}
	} else {
		outdmp->flag = rsp->un.genreq64.bdl.bdeSize;
	}
	if (outdmp->flag > snsbfrcnt) {
		rc = ERANGE;
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1208,
			       lpfc_mes1208,
			       lpfc_msgBlk1208.msgPreambleStr,
			       outdmp->flag, 4096);
		goto sndmgtqwt;
	}

	/* Copy back size of response, and response itself */
	memcpy(dataout, (char *)&outdmp->flag, sizeof (int));
	if (dfc_rsp_data_copy(phba, (uint8_t *) cip->lpfc_arg2, outdmp,
			      outdmp->flag)) {
		rc = EIO;
		goto sndmgtqwt;
	}

 sndmgtqwt:
	if (indmp)
		dfc_cmd_data_free(phba, indmp);
	if (outdmp)
		dfc_cmd_data_free(phba, outdmp);
	if (bmp) {
		lpfc_mbuf_free(phba, bmp->virt, bmp->phys);
		kfree(bmp);
	}
	if (cmdiocbq)
		lpfc_iocb_free(phba, cmdiocbq);

	if (rspiocbq)
		lpfc_iocb_free(phba, rspiocbq);
	return rc;
}

static void
lpfc_sli_def_mbox_cmpl(lpfcHBA_t *phba, LPFC_MBOXQ_t *pmb)
{
	struct lpfc_dmabuf *mp1, *mp2;

	mp1 = (struct lpfc_dmabuf *) (pmb->context1);
	mp2 = (struct lpfc_dmabuf *) (pmb->context2);

	if (mp1) {
		lpfc_mbuf_free(phba, mp1->virt, mp1->phys);
		kfree(mp1);
	}

	if (mp2) {
		lpfc_mbuf_free(phba, mp2->virt, mp2->phys);
		kfree(mp2);
	}
	lpfc_mbox_free(phba, pmb);
	return;
}

int
lpfc_ioctl_mbox(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	MAILBOX_t *pmbox;
	MAILBOX_t *pmb;
	LPFC_SLI_t *psli = &phba->sli;
	uint32_t size;
	dma_addr_t lptr;
	LPFC_MBOXQ_t *pmboxq = NULL;
	DMABUF_t *pbfrnfo = NULL;
	struct lpfc_dmabuf *pxmitbuf = NULL;
	unsigned long iflag;
	int count = 0;
	int rc = 0;
	int mbxstatus = 0;
	lpfc_vport_t *vport = phba->pport;
	uint32_t incnt = (uint32_t)(unsigned long) cip->lpfc_arg2;
	int wait_4_menlo_maint = 0;
	struct ioctl_mailbox_ext_data *ext_data = 0;

	/* Allocate mbox structure */
	pmbox = (MAILBOX_t *)lpfc_mbox_alloc(phba, MEM_PRI);
	if (pmbox == NULL)
		return ENOMEM;

	pxmitbuf = kmalloc(sizeof (struct lpfc_dmabuf), GFP_ATOMIC);
	if (pxmitbuf == NULL) {
		lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);
		return ENOMEM;
	}

	if ((pxmitbuf->virt = lpfc_mbuf_alloc(phba, 0, &(pxmitbuf->phys))) == 0) {
		kfree(pxmitbuf);
		lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);
		return ENOMEM;
        }

	/* Allocate mbox extension */
	if (cip->lpfc_arg3) {
		ext_data = kmalloc(sizeof(struct ioctl_mailbox_ext_data), GFP_KERNEL);
		if (!ext_data) {
			lpfc_mbuf_free(phba, pxmitbuf->virt, pxmitbuf->phys);
			kfree(pxmitbuf);
			lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);
			return ENOMEM;
		}
		memset (ext_data, 0, sizeof(ext_data));
		LPFC_DRVR_UNLOCK(phba, iflag);
		if ((copy_from_user(ext_data, cip->lpfc_arg3,
				sizeof(struct ioctl_mailbox_ext_data)))) {
			LPFC_DRVR_LOCK(phba, iflag);
			kfree(ext_data);
			lpfc_mbuf_free(phba, pxmitbuf->virt, pxmitbuf->phys);
			kfree(pxmitbuf);
			lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);
			return EIO;
		}
		LPFC_DRVR_LOCK(phba, iflag);
	}
	INIT_LIST_HEAD(&pxmitbuf->list);

	/* Allocate mboxq structure */
	pmboxq = lpfc_mbox_alloc(phba, MEM_PRI);
	if (pmboxq == NULL) {
                lpfc_mbuf_free(phba, pxmitbuf->virt, pxmitbuf->phys);
                kfree(pxmitbuf);
                lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);
		if (ext_data)
			kfree(ext_data);
		return ENOMEM;
	}

	/* Allocate mbuf structure */
	if (((pbfrnfo = kmalloc(sizeof (DMABUF_t), GFP_ATOMIC)) == 0) ||
	    ((pbfrnfo->virt = lpfc_mbuf_alloc(phba,
					      0, &(pbfrnfo->phys))) == 0)) {
		if (pbfrnfo)
			kfree(pbfrnfo);
		lpfc_mbuf_free(phba, pxmitbuf->virt, pxmitbuf->phys);
		kfree(pxmitbuf);
		lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);
		lpfc_mbox_free(phba, pmboxq);
		if (ext_data)
			kfree(ext_data);
		return ENOMEM;
	}
	INIT_LIST_HEAD(&pbfrnfo->list);

	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user((uint8_t *) pmbox, (uint8_t *) cip->lpfc_arg1,
			min(incnt,MAILBOX_CMD_SIZE))) {
		LPFC_DRVR_LOCK(phba, iflag);
		lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);
		lpfc_mbox_free(phba, pmboxq);
		lpfc_mbuf_free(phba, pbfrnfo->virt, pbfrnfo->phys);
		lpfc_mbuf_free(phba, pxmitbuf->virt, pxmitbuf->phys);
		kfree(pbfrnfo);
		kfree(pxmitbuf);
		if (ext_data)
			kfree(ext_data);
		return EIO;
	}

	LPFC_DRVR_LOCK(phba, iflag);

        while (phba->di_fc_flag & DFC_MBOX_ACTIVE) {
                LPFC_DRVR_UNLOCK(phba, iflag);
                lpfc_sleep_ms(phba, 5);
                LPFC_DRVR_LOCK(phba, iflag);
                if (count++ == 200)
                        break;
        }

        if (count >= 200) {
                pmbox->mbxStatus = MBXERR_ERROR;
                rc = EAGAIN;
                goto mbout_err;
        } else {
#ifdef _LP64
		if ((pmbox->mbxCommand == MBX_READ_SPARM) ||
		    (pmbox->mbxCommand == MBX_READ_RPI) ||
		    (pmbox->mbxCommand == MBX_REG_LOGIN) ||
		    (pmbox->mbxCommand == MBX_READ_LA)) {
			/* Must use 64 bit versions of these mbox cmds */
			pmbox->mbxStatus = MBXERR_ERROR;
			rc = ENODEV;
			goto mbout_err;
		}
#endif
	}
        phba->di_fc_flag |= DFC_MBOX_ACTIVE;
	lptr = 0;
	size = 0;

	switch (pmbox->mbxCommand) {
	/* Offline only */
	case MBX_INIT_LINK:
	case MBX_DOWN_LINK:
	case MBX_CONFIG_LINK:
	case MBX_CONFIG_RING:
	case MBX_RESET_RING:
	case MBX_UNREG_LOGIN:
	case MBX_CLEAR_LA:
	case MBX_DUMP_CONTEXT:
	case MBX_RUN_DIAGS:
	case MBX_RESTART:
	case MBX_SET_MASK:
	case MBX_SET_DEBUG:
		if (!(vport->fc_flag & FC_OFFLINE_MODE)) {
			pmbox->mbxStatus = MBXERR_ERROR;
			phba->di_fc_flag &= ~DFC_MBOX_ACTIVE;
			goto mbout_err;
		}
		break;

	/* Online / Offline */
	case MBX_WRITE_VPARMS:
	case MBX_WRITE_WWN:
	case MBX_WRITE_NV:
	case MBX_LOAD_SM:
	case MBX_READ_NV:
	case MBX_READ_CONFIG:
	case MBX_READ_RCONFIG:
	case MBX_READ_STATUS:
	case MBX_READ_XRI:
	case MBX_READ_REV:
	case MBX_READ_LNK_STAT:
	case MBX_DUMP_MEMORY:
	case MBX_DOWN_LOAD:
	case MBX_UPDATE_CFG:
	case MBX_LOAD_AREA:
	case MBX_LOAD_EXP_ROM:
	case MBX_BEACON:
	case MBX_DEL_LD_ENTRY:
	case MBX_WRITE_EVENT_LOG:
	case MBX_READ_EVENT_LOG_STATUS:
		break;
	case MBX_SET_VARIABLE:
		if ((pmbox->un.varWords[0] == SETVAR_MLOMNT)
			&& (pmbox->un.varWords[1] == 1)) {

			if ((phba->sli.sliinit.sli_flag & LPFC_MENLO_MAINT) == 0) {
				wait_4_menlo_maint = 1;
				phba->wait_4_mlo_maint_flg = 1;

			}
		}	
		break;

	/* Online / Offline - with DMA */
	case MBX_READ_EVENT_LOG:
		lptr = getPaddr(pmbox->un.varRdEventLog.rcv_bde64.addrHigh,
				pmbox->un.varRdEventLog.rcv_bde64.addrLow);
		size = (int)pmbox->un.varRdEventLog.rcv_bde64.tus.f.bdeSize;

		if (size > LPFC_BPL_SIZE) {
			/* user bde too big */
			pmbox->mbxStatus = MBX_STATUS_INTERNAL_DRIVER_ERROR;
			phba->di_fc_flag &= ~DFC_MBOX_ACTIVE;
			goto mbout_err;
		}

		if (lptr) {
			pmbox->un.varRdEventLog.rcv_bde64.addrHigh =
				putPaddrHigh(pbfrnfo->phys);
			pmbox->un.varRdEventLog.rcv_bde64.addrLow =
				putPaddrLow(pbfrnfo->phys);
		}
		break;

	case MBX_READ_SPARM64:
		lptr = getPaddr(pmbox->un.varRdSparm.un.sp64.addrHigh,
				pmbox->un.varRdSparm.un.sp64.addrLow);
		size = (int)pmbox->un.varRdSparm.un.sp64.tus.f.bdeSize;
		if (lptr) {
			pmbox->un.varRdSparm.un.sp64.addrHigh =
			    putPaddrHigh(pbfrnfo->phys);
			pmbox->un.varRdSparm.un.sp64.addrLow =
			    putPaddrLow(pbfrnfo->phys);
		}
		break;

	case MBX_READ_RPI64:
		/* This is only allowed when online is SLI2 mode */
		lptr = getPaddr(pmbox->un.varRdRPI.un.sp64.addrHigh,
				pmbox->un.varRdRPI.un.sp64.addrLow);
		size = (int)pmbox->un.varRdRPI.un.sp64.tus.f.bdeSize;
		if (lptr) {
			pmbox->un.varRdRPI.un.sp64.addrHigh =
			    putPaddrHigh(pbfrnfo->phys);
			pmbox->un.varRdRPI.un.sp64.addrLow =
			    putPaddrLow(pbfrnfo->phys);
		}
		break;

        case MBX_RUN_BIU_DIAG64:
		lptr = getPaddr(pmbox->un.varBIUdiag.un.s2.xmit_bde64.addrHigh,
				pmbox->un.varBIUdiag.un.s2.xmit_bde64.addrLow);
		if (lptr) {
			LPFC_DRVR_UNLOCK(phba, iflag);
			rc = copy_from_user((uint8_t *)pxmitbuf->virt,
					(uint8_t *)(unsigned long)lptr,
					pmbox->un.varBIUdiag.un.s2.xmit_bde64.tus.f.bdeSize);
			LPFC_DRVR_LOCK(phba, iflag);

			if (rc) {
				rc = EIO;
				goto mbout_err;
			}

			pmbox->un.varBIUdiag.un.s2.xmit_bde64.addrHigh =
				putPaddrHigh(pxmitbuf->phys);
			pmbox->un.varBIUdiag.un.s2.xmit_bde64.addrLow =
				putPaddrLow(pxmitbuf->phys);
		}

		lptr = getPaddr(pmbox->un.varBIUdiag.un.s2.rcv_bde64.addrHigh,
				pmbox->un.varBIUdiag.un.s2.rcv_bde64.addrLow);
		size = pmbox->un.varBIUdiag.un.s2.rcv_bde64.tus.f.bdeSize;
		if (lptr) {
			pmbox->un.varBIUdiag.un.s2.rcv_bde64.addrHigh =
				putPaddrHigh(pbfrnfo->phys);
			pmbox->un.varBIUdiag.un.s2.rcv_bde64.addrLow =
				putPaddrLow(pbfrnfo->phys);
		}
		break;

	case MBX_READ_LA:
	case MBX_READ_LA64:
	case MBX_REG_LOGIN:
	case MBX_REG_LOGIN64:
	case MBX_CONFIG_PORT:
	case MBX_RUN_BIU_DIAG:
		/* Do not allow SLI-2 commands */
		pmbox->mbxStatus = MBXERR_ERROR;
		phba->di_fc_flag &= ~DFC_MBOX_ACTIVE;
		goto mbout_err;
	default:
		/*
		 * Offline only.  Let firmware return error for unsupported
		 * commands
		 */
		if (!(vport->fc_flag & FC_OFFLINE_MODE)) {
			pmbox->mbxStatus = MBXERR_ERROR;
                        phba->di_fc_flag &= ~DFC_MBOX_ACTIVE;
			goto mbout_err;
		}
		break;
	} /* switch pmbox->mbxCommand */

	memset((void *)pmboxq, 0, sizeof (LPFC_MBOXQ_t));
	pmb = &pmboxq->mb;
	pmb->mbxCommand = pmbox->mbxCommand;
	pmb->mbxOwner = pmbox->mbxOwner;
	pmb->un = pmbox->un;
	pmb->us = pmbox->us;
	pmboxq->context1 = (uint8_t *) 0;
	pmboxq->vport = vport;

	if (ext_data) {
		pmboxq->context2 = &ext_data->mbox_extension_data[0];
		pmboxq->in_ext_byte_len = ext_data->in_ext_byte_len;
		pmboxq->out_ext_byte_len = ext_data->out_ext_byte_len;
		pmboxq->mbox_offset_word = ext_data->mbox_offset_word;
	}

	if ((vport->fc_flag & FC_OFFLINE_MODE) ||
	    (!(psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE))){
		LPFC_DRVR_UNLOCK(phba, iflag);
		mbxstatus = lpfc_sli_issue_mbox(phba, pmboxq, MBX_POLL);
		LPFC_DRVR_LOCK(phba, iflag);
	} else {
		mbxstatus = lpfc_sli_issue_mbox_wait(phba, pmboxq, 
				lpfc_mbox_tmo_val(phba, pmbox->mbxCommand));
	}
        phba->di_fc_flag &= ~DFC_MBOX_ACTIVE;

	if (mbxstatus == MBX_TIMEOUT) {
		rc = EBUSY;
		goto mbout;
	} else if (mbxstatus != MBX_SUCCESS) {
		rc = ENODEV;
		goto mbout;
	} else if (wait_4_menlo_maint) {
		int waiting = 30;

		/*
		 * Wait up to 30 seconds for
		 * Menlo Maintenance Link UP
		 */
		set_current_state(TASK_INTERRUPTIBLE);
		LPFC_DRVR_UNLOCK(phba, iflag);

		do {
			schedule_timeout(HZ);
		}while (--waiting && phba->wait_4_mlo_maint_flg);

		LPFC_DRVR_LOCK(phba, iflag);
		set_current_state(TASK_RUNNING);
		if (phba->wait_4_mlo_maint_flg != 0) {
			rc =  ETIME;
			goto mbout;
		}
	}
	
	/* Copy the extended data to user space memory */
	if (ext_data && ext_data->out_ext_byte_len) {
		LPFC_DRVR_UNLOCK(phba, iflag);	
		rc = copy_to_user(cip->lpfc_arg3,
				ext_data,
				sizeof(struct ioctl_mailbox_ext_data));
		LPFC_DRVR_LOCK(phba, iflag);

		if (rc) {
			rc = EIO;
			goto mbout_err;
		}
	}

	if (lptr) {
		LPFC_DRVR_UNLOCK(phba, iflag);
		if ((copy_to_user((void *)(unsigned long)lptr, 
				(uint8_t *)pbfrnfo->virt,
				(ulong) size))) {
			rc = EIO;
		}

		LPFC_DRVR_LOCK(phba, iflag);
	}

 mbout:
	cip->lpfc_outsz = min(cip->lpfc_outsz ,MAILBOX_CMD_SIZE);
	memcpy(dataout, (char *)pmb, cip->lpfc_outsz);
	goto mbout_freemem;

 mbout_err:
	/* Jump here only if there is an error and copy the status */
	memcpy(dataout, (char *)pmbox, MAILBOX_CMD_SIZE);

 mbout_freemem:
	/* Free allocated mbox extension */
	if (ext_data)
		kfree(ext_data);

	/* Free allocated mbox memory */
	if (pmbox)
		lpfc_mbox_free(phba, (LPFC_MBOXQ_t *) pmbox);

	if (pxmitbuf) { 
		lpfc_mbuf_free(phba, pxmitbuf->virt, pxmitbuf->phys);
		kfree(pxmitbuf);
	}

	/* Free allocated mboxq memory */
	if (pmboxq) {
		if (mbxstatus == MBX_TIMEOUT) {
			/*
			 * Let SLI layer to release mboxq if mbox command
			 * completed after timeout.
			 */
			pmboxq->mbox_cmpl = 0;
		} else {
			lpfc_mbox_free(phba, pmboxq);
		}
	}

	/* Free allocated mbuf memory */
	if (pbfrnfo) {
		lpfc_mbuf_free(phba, pbfrnfo->virt, pbfrnfo->phys);
		kfree(pbfrnfo);
	}
	return rc;
}

int
lpfc_ioctl_linkinfo(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip, void *dataout)
{
	LinkInfo *linkinfo;
	int rc = 0;
	lpfc_vport_t *vport = phba->pport;

	linkinfo = (LinkInfo *) dataout;
	linkinfo->a_linkEventTag = phba->fc_eventTag;
	linkinfo->a_linkUp = phba->fc_stat.LinkUp;
	linkinfo->a_linkDown = phba->fc_stat.LinkDown;
	linkinfo->a_linkMulti = phba->fc_stat.LinkMultiEvent;
	linkinfo->a_DID = vport->fc_myDID;
	if (phba->sli.sliinit.sli_flag & LPFC_MENLO_MAINT)
		linkinfo->a_topology = LPFC_LNK_MENLO_MAINT;
	else if (phba->fc_topology == TOPOLOGY_LOOP) {
		if (vport->fc_flag & FC_PUBLIC_LOOP) {
			linkinfo->a_topology = LNK_PUBLIC_LOOP;
			memcpy((uint8_t *) linkinfo->a_alpaMap,
			       (uint8_t *) phba->alpa_map, 128);
			linkinfo->a_alpaCnt = phba->alpa_map[0];
		} else {
			linkinfo->a_topology = LNK_LOOP;
			memcpy((uint8_t *) linkinfo->a_alpaMap,
			       (uint8_t *) phba->alpa_map, 128);
			linkinfo->a_alpaCnt = phba->alpa_map[0];
		}
	} else {
		memset((uint8_t *) linkinfo->a_alpaMap, 0, 128);
		linkinfo->a_alpaCnt = 0;
		if (vport->fc_flag & FC_FABRIC) {
			linkinfo->a_topology = LNK_FABRIC;
		} else {
			linkinfo->a_topology = LNK_PT2PT;
		}
	}
	linkinfo->a_linkState = 0;
	switch (vport->port_state) {
	case LPFC_INIT_START:

	case LPFC_LINK_DOWN:
		linkinfo->a_linkState = LNK_DOWN;
		memset((uint8_t *) linkinfo->a_alpaMap, 0, 128);
		linkinfo->a_alpaCnt = 0;
		break;
	case LPFC_LINK_UP:

	case LPFC_LOCAL_CFG_LINK:
		linkinfo->a_linkState = LNK_UP;
		break;
	case LPFC_FLOGI:
		linkinfo->a_linkState = LNK_FLOGI;
		break;
	case LPFC_DISC_AUTH:
	case LPFC_FABRIC_CFG_LINK:
	case LPFC_NS_REG:
	case LPFC_NS_QRY:

	case LPFC_CLEAR_LA:
		linkinfo->a_linkState = LNK_DISCOVERY;
		break;
	case LPFC_HBA_READY:
		linkinfo->a_linkState = LNK_READY;
		break;
	}
	linkinfo->a_alpa = (uint8_t) (vport->fc_myDID & 0xff);
	memcpy((uint8_t *) linkinfo->a_wwpName,
	       (uint8_t *) & vport->fc_portname, 8);
	memcpy((uint8_t *) linkinfo->a_wwnName,
	       (uint8_t *) & vport->fc_nodename, 8);

	return rc;
}

int
lpfc_ioctl_ioinfo(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip, void *dataout)
{
	IOinfo *ioinfo;
	LPFC_SLI_t *psli;
	int rc = 0;

	psli = &phba->sli;
	ioinfo = (IOinfo *) dataout;
	memset((void *)ioinfo, 0, sizeof (IOinfo));

	ioinfo->a_mbxCmd = psli->slistat.mboxCmd;
	ioinfo->a_mboxCmpl = psli->slistat.mboxEvent;
	ioinfo->a_mboxErr = psli->slistat.mboxStatErr;
	ioinfo->a_iocbCmd = psli->slistat.iocbCmd[cip->lpfc_ring];
	ioinfo->a_iocbRsp = psli->slistat.iocbRsp[cip->lpfc_ring];
	ioinfo->a_adapterIntr = (psli->slistat.linkEvent +
				 psli->slistat.iocbRsp[cip->lpfc_ring] +
				 psli->slistat.mboxEvent);
	ioinfo->a_fcpCmd = phba->fc_stat.fcpCmd;
	ioinfo->a_fcpCmpl = phba->fc_stat.fcpCmpl;
	ioinfo->a_fcpErr = phba->fc_stat.fcpRspErr +
	    phba->fc_stat.fcpRemoteStop + phba->fc_stat.fcpPortRjt +
	    phba->fc_stat.fcpPortBusy + phba->fc_stat.fcpError +
	    phba->fc_stat.fcpLocalErr;
	ioinfo->a_bcastRcv = phba->fc_stat.frameRcvBcast;
	ioinfo->a_RSCNRcv = phba->fc_stat.elsRcvRSCN;

	return rc;
}

int
lpfc_ioctl_nodeinfo(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout, int size)
{
	NodeInfo *np;
	LPFC_NODELIST_t *pndl;
	LPFC_BINDLIST_t *pbdl;
	uint32_t cnt;
	int rc = 0;
	uint32_t total_mem = size;
	struct list_head *listp;
	struct list_head *node_list[4];
	int i;
	lpfc_vport_t *vport = phba->pport;

	np = (NodeInfo *) dataout;
	cnt = 0;

	/*
	 * Since the size of bind & others are different,
	 * get the node list of bind first
	 */
	total_mem -= sizeof (LPFC_BINDLIST_t);

	list_for_each_entry(pbdl, &vport->fc_nlpbind_list, nlp_listp) {
		if (total_mem <= 0)
			break;

		memset((uint8_t *) np, 0, sizeof (LPFC_BINDLIST_t));
		if (pbdl->nlp_bind_type & FCP_SEED_WWPN)
			np->a_flag |= NODE_SEED_WWPN;
		if (pbdl->nlp_bind_type & FCP_SEED_WWNN)
			np->a_flag |= NODE_SEED_WWNN;
		if (pbdl->nlp_bind_type & FCP_SEED_DID)
			np->a_flag |= NODE_SEED_DID;
		if (pbdl->nlp_bind_type & FCP_SEED_AUTO)
			np->a_flag |= NODE_AUTOMAP;
		np->a_state = NODE_SEED;
		np->a_did = pbdl->nlp_DID;
		np->a_targetid = pbdl->nlp_scsi_id;
		memcpy(np->a_wwpn, &pbdl->nlp_portname, 8);
		memcpy(np->a_wwnn, &pbdl->nlp_nodename, 8);
		total_mem -= sizeof (LPFC_BINDLIST_t);
		np++;
		cnt++;
	}

	/* Get the node list of unmap, map, plogi and adisc */
	total_mem -= sizeof (LPFC_NODELIST_t);

	node_list[0] = &vport->fc_plogi_list;
	node_list[1] = &vport->fc_adisc_list;
	node_list[2] = &vport->fc_nlpunmap_list;
	node_list[3] = &vport->fc_nlpmap_list;
	for (i = 0; i < 4; i++) {
		listp = node_list[i];
		if (list_empty(listp)) 
			continue;	

		list_for_each_entry(pndl, listp, nlp_listp) {
			if (total_mem <= 0)
				break;

			memset((uint8_t *) np, 0, sizeof (LPFC_NODELIST_t));
			if (pndl->nlp_flag & NLP_ADISC_LIST) {
				np->a_flag |= NODE_ADDR_AUTH;
				np->a_state = NODE_LIMBO;
			}
			if (pndl->nlp_flag & NLP_PLOGI_LIST) {
				np->a_state = NODE_PLOGI;
			}
			if (pndl->nlp_flag & NLP_MAPPED_LIST) {
				np->a_state = NODE_ALLOC;
			}
			if (pndl->nlp_flag & NLP_UNMAPPED_LIST) {
				np->a_state = NODE_PRLI;
			}
			if (pndl->nlp_type & NLP_FABRIC)
				np->a_flag |= NODE_FABRIC;
			if (pndl->nlp_type & NLP_FCP_TARGET)
				np->a_flag |= NODE_FCP_TARGET;
			if (pndl->nlp_flag & NLP_ELS_SND_MASK)	/* Sent ELS mask  -- Check this */
				np->a_flag |= NODE_REQ_SND;
			if (pndl->nlp_flag & NLP_FARP_SND)
				np->a_flag |= NODE_FARP_SND;
			if (pndl->nlp_flag & NLP_SEED_WWPN)
				np->a_flag |= NODE_SEED_WWPN;
			if (pndl->nlp_flag & NLP_SEED_WWNN)
				np->a_flag |= NODE_SEED_WWNN;
			if (pndl->nlp_flag & NLP_SEED_DID)
				np->a_flag |= NODE_SEED_DID;
			if (pndl->nlp_flag & NLP_AUTOMAP)
				np->a_flag |= NODE_AUTOMAP;
			np->a_did = pndl->nlp_DID;
			np->a_targetid = pndl->nlp_scsi_id;
			memcpy(np->a_wwpn, &pndl->nlp_portname, 8);
			memcpy(np->a_wwnn, &pndl->nlp_nodename, 8);
			total_mem -= sizeof (LPFC_NODELIST_t);
			np++;
			cnt++;
		}
	}

	cip->lpfc_outsz = (uint32_t) (cnt * sizeof (NodeInfo));
	return rc;
}

int
lpfc_ioctl_getcfg(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	CfgParam *cp;
	iCfgParam *icp;
	uint32_t cnt;
	int i, rc = 0, astringi;

	/* First uint32_t word will be count */
	cp = (CfgParam *) dataout;
	cnt = 0;
	for (i = 0; i < LPFC_TOTAL_NUM_OF_CFG_PARAM; i++) {
		icp = (iCfgParam *) &phba->config[i];
		if (!(icp->a_flag & CFG_EXPORT))
			continue;
		cp->a_low = icp->a_low;
		cp->a_hi = icp->a_hi;
		cp->a_flag = icp->a_flag;
		cp->a_default = icp->a_default;
		if (i == LPFC_CFG_FCP_CLASS) {
			switch (icp->a_current) {
			case CLASS1:
				cp->a_current = 1;
				break;
			case CLASS2:
				cp->a_current = 2;
				break;
			case CLASS3:
				cp->a_current = 3;
				break;
			}
		} else {
			cp->a_current = icp->a_current;
		}

		cp->a_changestate = icp->a_changestate;
		memcpy(cp->a_string, icp->a_string, 32);

		/* Translate all "_" to "-" to preserve backwards compatibility
		with older drivers that used "_" */
		astringi=0;
		while(cp->a_string[astringi++])
			if(cp->a_string[astringi] == '_')
				cp->a_string[astringi] = '-';

		memcpy(cp->a_help, icp->a_help, 80);
		cp++;
		cnt++;
	}

	if (cnt)
		cip->lpfc_outsz = (uint32_t) (cnt * sizeof (CfgParam));

	return rc;
}

int
lpfc_ioctl_setcfg(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip)
{
	iCfgParam *icp;
	uint32_t offset, cnt;
	lpfcCfgParam_t *clp;
	LPFC_SLI_t *psli;
	int i, j;

	psli = &phba->sli;
	clp = &phba->config[0];
	offset = (ulong) cip->lpfc_arg1;
	cnt = (ulong) cip->lpfc_arg2;
	if (offset >= LPFC_TOTAL_NUM_OF_CFG_PARAM)
		return ERANGE;

	j = offset;
	for (i = 0; i < LPFC_TOTAL_NUM_OF_CFG_PARAM; i++) {
		icp = (iCfgParam *) & clp[i];
		if (!(icp->a_flag & CFG_EXPORT))
			continue;
		if (j == 0)
			break;
		j--;
	}

	if (icp->a_changestate != CFG_DYNAMIC)
		return EPERM;

	if (((icp->a_low != 0) && (cnt < icp->a_low)) || (cnt > icp->a_hi))
		return ERANGE;

	if (!(icp->a_flag & CFG_EXPORT))
		return EPERM;

	switch (offset) {
	case LPFC_CFG_FCP_CLASS:
		switch (cnt) {
		case 1:
			clp[LPFC_CFG_FCP_CLASS].a_current = CLASS1;
			break;
		case 2:
			clp[LPFC_CFG_FCP_CLASS].a_current = CLASS2;
			break;
		case 3:
			clp[LPFC_CFG_FCP_CLASS].a_current = CLASS3;
			break;
		}
		break;

	case LPFC_CFG_LINKDOWN_TMO:
		icp->a_current = cnt;
		break;

 	case LPFC_CFG_DFT_LUN_Q_DEPTH:
		/*
		 * Update the LUN Queue Depth
		 */
		icp->a_current = lpfc_sched_set_lun_qdepth(phba, cnt);
 		break;
 
 	case LPFC_CFG_DFT_TGT_Q_DEPTH:
		/*
		 * Update the Target Queue Depth and then
		 * adjust the LUN Queue Depth.
		 */
		icp->a_current = lpfc_sched_set_tgt_qdepth(phba, cnt);
                clp[LPFC_CFG_DFT_LUN_Q_DEPTH].a_current = 
		    lpfc_sched_set_lun_qdepth(phba, clp[LPFC_CFG_DFT_LUN_Q_DEPTH].a_current);
 		break;
 
 	case LPFC_CFG_DFT_HBA_Q_DEPTH:
		/*
		 * Update the HBA Queue Depth, 
		 * adjust the Target Queue Depth and then
		 * adjust the LUN Queue Depth.
		 */
 		icp->a_current = lpfc_sched_set_hba_qdepth(phba, cnt);
                clp[LPFC_CFG_DFT_TGT_Q_DEPTH].a_current = 
		    lpfc_sched_set_tgt_qdepth(phba, clp[LPFC_CFG_DFT_TGT_Q_DEPTH].a_current);
		clp[LPFC_CFG_DFT_LUN_Q_DEPTH].a_current =
		    lpfc_sched_set_lun_qdepth(phba, clp[LPFC_CFG_DFT_LUN_Q_DEPTH].a_current);
 		break;

	default:
		icp->a_current = cnt;
	}

	return 0;
}

int
lpfc_ioctl_hba_get_event(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, 
			 void *dataout, int data_size)
{
	fcEVT_t *ep;
	fcEVT_t *oep;
	fcEVTHDR_t *ehp;
	uint8_t *cp;
	void *type;
	uint32_t offset, incr, size, cnt, i, gstype;
	DMABUF_t *mm;
	int no_more;
	int rc = 0;
	uint32_t total_mem = data_size;
	unsigned long iflag;
	struct list_head head, *pos, *tmp_pos;
	temp_event_info_t temp_event_info;

	no_more = 1;

	/* event mask */
	offset = ((ulong) cip->lpfc_arg3 &  FC_REG_EVENT_MASK);
	/* event id */
	incr = (uint32_t) cip->lpfc_flag;
	/* process requesting evt */
	size = (uint32_t) cip->lpfc_iocb;

	type = 0;
	if (offset == FC_REG_CT_EVENT) {
		LPFC_DRVR_UNLOCK(phba, iflag);
		rc = copy_from_user((uint8_t *) & gstype,
				    (uint8_t *) cip->lpfc_arg2,
				    (ulong) (sizeof (uint32_t)));
		LPFC_DRVR_LOCK(phba, iflag);
		if (rc)
			return EIO;

		type = (void *)(ulong) gstype;
	}

	ehp = (fcEVTHDR_t *) phba->fc_evt_head;
	while (ehp) {
		if ((ehp->e_mask == offset) && (ehp->e_type == type))
			break;
		ehp = (fcEVTHDR_t *) ehp->e_next_header;
	}

	if (!ehp)
		return ENOENT;

	ep = ehp->e_head;
	oep = 0;
	while (ep) {
		/* Find an event that matches the event mask */
		if (ep->evt_sleep == 0) {
			/* dequeue event from event list */
			if (oep == 0)
				ehp->e_head = ep->evt_next;
			else
				oep->evt_next = ep->evt_next;

			if (ehp->e_tail == ep)
				ehp->e_tail = oep;

			switch (offset) {
			case FC_REG_LINK_EVENT:
				break;
			case FC_REG_TEMPERATURE_EVENT:
				temp_event_info.event_type = ep->evt_data0;
				temp_event_info.temp = (unsigned long) ep->evt_data1;
				memcpy(dataout, (char *)&temp_event_info,
				       sizeof (temp_event_info));
				cip->lpfc_outsz = sizeof (temp_event_info);
				break;
			case FC_REG_RSCN_EVENT:
				/* Return data length */
				cnt = sizeof (uint32_t);
				LPFC_DRVR_UNLOCK(phba, iflag);
				if (copy_to_user
				    ((uint8_t *) cip->lpfc_arg1,
				     (uint8_t *) & cnt, sizeof (uint32_t))) {
					rc = EIO;
				}
				LPFC_DRVR_LOCK(phba, iflag);
				memcpy(dataout, (char *)&ep->evt_data0,
				       cnt);
				cip->lpfc_outsz = (uint32_t) cnt;
				break;
			case FC_REG_CT_EVENT:
				/* Return data length */
				cnt = (ulong) (ep->evt_data2);
				LPFC_DRVR_UNLOCK(phba, iflag);
				if (copy_to_user
				    ((uint8_t *) cip->lpfc_arg1,
				     (uint8_t *) & cnt, sizeof (uint32_t))) {
					rc = EIO;
				} else {
					if (copy_to_user
					    ((uint8_t *) cip->lpfc_arg2,
					     (uint8_t *) & ep->evt_data0,
					     sizeof (uint32_t))) {
						rc = EIO;
					}
				}
				LPFC_DRVR_LOCK(phba, iflag);

				cip->lpfc_outsz = (uint32_t) cnt;
				i = cnt;
				mm = (DMABUF_t *) ep->evt_data1;
				cp = (uint8_t *) dataout;
				list_add_tail(&head, &mm->list);
				list_for_each_safe(pos, tmp_pos, &head) {
					mm = list_entry(pos, DMABUF_t, list);

					if (cnt > FCELSSIZE)
						i = FCELSSIZE;
					else
						i = cnt;

					if (total_mem > 0) {
						memcpy(cp, (char *)mm->virt, i);
						total_mem -= i;
					}

					cp += i;
					if (phba->sli3_options & LPFC_SLI3_HBQ_ENABLED) {
						lpfc_hbq_free(phba, mm->virt, mm->phys);
					} else {
						lpfc_mbuf_free(phba, mm->virt, mm->phys);
					}
					list_del(pos);
					kfree(mm);
				}
				list_del(&head);
				break;
			}

			if ((offset == FC_REG_CT_EVENT) && (ep->evt_next) &&
			    (((fcEVT_t *) (ep->evt_next))->evt_sleep == 0)) {
				ep->evt_data0 |= 0x80000000;	/* More event r waiting */
				LPFC_DRVR_UNLOCK(phba, iflag);
				if (copy_to_user
				    ((uint8_t *) cip->lpfc_arg2,
				     (uint8_t *) & ep->evt_data0,
				     sizeof (uint32_t))) {
					rc = EIO;
				}
				LPFC_DRVR_LOCK(phba, iflag);
				no_more = 0;
			}

			/* Requeue event entry */
			ep->evt_next = 0;
			ep->evt_data0 = 0;
			ep->evt_data1 = 0;
			ep->evt_data2 = 0;
			ep->evt_sleep = 1;
			ep->evt_flags = 0;

			if (ehp->e_head == 0) {
				ehp->e_head = ep;
				ehp->e_tail = ep;
			} else {
				ehp->e_tail->evt_next = ep;
				ehp->e_tail = ep;
			}

			if (offset == FC_REG_LINK_EVENT) {
				ehp->e_flag &= ~E_GET_EVENT_ACTIVE;
				rc = lpfc_ioctl_linkinfo(phba, cip, dataout);
				return rc;
			}

			if (no_more)
				ehp->e_flag &= ~E_GET_EVENT_ACTIVE;
			return rc;
		}

		oep = ep;
		ep = ep->evt_next;
	}

	if (ep == 0)
		rc = ENOENT;

	return rc;
}

int
lpfc_sleep_event(lpfcHBA_t *phba, fcEVTHDR_t *ep)
{
	ep->e_mode |= E_SLEEPING_MODE;
	switch (ep->e_mask) {
	case FC_REG_LINK_EVENT:
		return (lpfc_sleep(phba, &phba->linkevtwq, 0));
	case FC_REG_RSCN_EVENT:
		return (lpfc_sleep(phba, &phba->rscnevtwq, 0));
	case FC_REG_CT_EVENT:
		return (lpfc_sleep(phba, &phba->ctevtwq, 0));
	case FC_REG_DUMP_EVENT:
		return (lpfc_sleep(phba, &phba->dumpevtwq, 0));
	case FC_REG_TEMPERATURE_EVENT:
		return (lpfc_sleep(phba, &phba->tempevtwq, 0));
	}

	return 0;
}

int
lpfc_ioctl_hba_set_event(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip)
{
	fcEVT_t *evp;
	fcEVT_t *ep;
	fcEVT_t *oep;
	fcEVTHDR_t *ehp;
	fcEVTHDR_t *oehp;
	int found;
	void *type;
	uint32_t offset, incr;
	int rc = 0;

	/* event mask */
	offset = (((ulong) cip->lpfc_arg3) & FC_REG_EVENT_MASK);

	/* event id */
	incr = (uint32_t) cip->lpfc_flag;

	/*
	 * Number of events we can queue up + 1, before dropping events for this
	 * event id.
	 */
	switch (offset) {
	case FC_REG_CT_EVENT:
		type = cip->lpfc_arg2;
		found = LPFC_MAX_EVENT;	
		break;
	case FC_REG_RSCN_EVENT:
		type = (void *)0;
		found = LPFC_MAX_EVENT;
		break;
	case FC_REG_LINK_EVENT:
		type = (void *)0;
		found = 2;
		break;
	case FC_REG_DUMP_EVENT:
		type  = (void *)0;
		found = 1;
		break;
	case FC_REG_TEMPERATURE_EVENT:
		type = (void *)0;
		found = LPFC_MAX_EVENT;
		break;
	default:
		found = 0;
		return EINTR;
	}

	/*
	 * find the fcEVT_t header for this Event, allocate a header
	 * if not found.
	 */
	oehp = 0;
	ehp = (fcEVTHDR_t *) phba->fc_evt_head;
	while (ehp) {
		if ((ehp->e_mask == offset) && (ehp->e_type == type)) {
			found = 0;
			break;
		}
		oehp = ehp;
		ehp = (fcEVTHDR_t *) ehp->e_next_header;
	}

	if (!ehp) {
		ehp = kmalloc (sizeof (fcEVTHDR_t), GFP_ATOMIC);
		if (ehp == 0 )
			return EINTR;

		memset((char *)ehp, 0, sizeof (fcEVTHDR_t));
		if (phba->fc_evt_head == 0) {
			phba->fc_evt_head = ehp;
			phba->fc_evt_tail = ehp;
		} else {
			((fcEVTHDR_t *) (phba->fc_evt_tail))->e_next_header =
			    ehp;
			phba->fc_evt_tail = (void *)ehp;
		}

		ehp->e_handle = incr;
		ehp->e_mask = offset;
		ehp->e_type = type;
		ehp->e_refcnt++;
	} else {
		ehp->e_refcnt++;
	}

	while (found) {
		/* Save event id for C_GET_EVENT */
		oep = kmalloc (sizeof (fcEVT_t), GFP_ATOMIC);
		if (oep == 0) {
			rc = EINTR;
			break;
		}

		memset((char *)oep, 0, sizeof (fcEVT_t));
		oep->evt_sleep = 1;
		oep->evt_handle = incr;
		oep->evt_mask = offset;
		oep->evt_type = type;

		if (ehp->e_head == 0) {
			ehp->e_head = oep;
			ehp->e_tail = oep;
		} else {
			ehp->e_tail->evt_next = (void *)oep;
			ehp->e_tail = oep;
		}

		oep->evt_next = 0;
		found--;
	}

	switch (offset) {
	case FC_REG_CT_EVENT:
	case FC_REG_RSCN_EVENT:
	case FC_REG_LINK_EVENT:
	case FC_REG_DUMP_EVENT:
	case FC_REG_TEMPERATURE_EVENT:
		if (rc || lpfc_sleep_event(phba, ehp)) {
			rc = EINTR;
			ehp->e_mode &= ~E_SLEEPING_MODE;
			ehp->e_refcnt--;
			if (ehp->e_refcnt)
				goto setout;

			/* Remove all eventIds from queue */
			ep = ehp->e_head;
			oep = 0;
			found = 0;
			while (ep) {
				if (ep->evt_handle == incr) {
					/* dequeue event from event list */
					if (oep == 0)
						ehp->e_head = ep->evt_next;
					else
						oep->evt_next = ep->evt_next;

					if (ehp->e_tail == ep)
						ehp->e_tail = oep;
					evp = ep;
					ep = ep->evt_next;
					kfree(evp);
				} else {
					oep = ep;
					ep = ep->evt_next;
				}
			}

			/*
			 * No more fcEVT_t pointer under this fcEVTHDR_t
			 * Free the fcEVTHDR_t
			 */
			if (ehp->e_head == 0) {
				oehp = 0;
				ehp = (fcEVTHDR_t *) phba->fc_evt_head;
				while (ehp) {
					if ((ehp->e_mask == offset) &&
					    (ehp->e_type == type)) {
						found = 0;
						break;
					}
					oehp = ehp;
					ehp = (fcEVTHDR_t *) ehp->e_next_header;
				}
				if (oehp == 0) {
					phba->fc_evt_head = ehp->e_next_header;
				} else {
					oehp->e_next_header = ehp->e_next_header;
				}

				if (phba->fc_evt_tail == ehp)
					phba->fc_evt_tail = oehp;

				kfree(ehp);
			}
			goto setout;
		}
		ehp->e_refcnt--;
		break;
	}

 setout:
	return rc;
}

int
lpfc_ioctl_add_bind(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip)
{

	bind_ctl_t bind_ctl;
	void *bind_id = 0;
	uint8_t bind_type = FCP_SEED_WWNN;
	int rc = 0;
	unsigned long iflag;
	lpfc_vport_t *vport = phba->pport;

	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user((uint8_t *) & bind_ctl, (uint8_t *) cip->lpfc_arg1,
			   (ulong) (sizeof (bind_ctl)))) {
		rc = EIO;
		LPFC_DRVR_LOCK(phba, iflag);
		return rc;
	}
	LPFC_DRVR_LOCK(phba, iflag);

	switch (bind_ctl.bind_type) {
	case LPFC_WWNN_BIND:
		bind_type = FCP_SEED_WWNN;
		bind_id = &bind_ctl.wwnn[0];
		break;
	case LPFC_WWPN_BIND:
		bind_type = FCP_SEED_WWPN;
		bind_id = &bind_ctl.wwpn[0];
		break;
	case LPFC_DID_BIND:
		bind_type = FCP_SEED_DID;
		bind_id = &bind_ctl.did;
		break;
	default:
		rc = EIO;
		break;
	}

	if (rc)
		return rc;

	rc = lpfc_add_bind(vport, bind_type, bind_id, bind_ctl.scsi_id);
	return rc;
}

int
lpfc_ioctl_del_bind(lpfcHBA_t * phba, LPFCCMDINPUT_t * cip)
{

	bind_ctl_t bind_ctl;
	void *bind_id = 0;
	uint8_t bind_type = FCP_SEED_WWNN;
	int rc = 0;
	unsigned long iflag;
	lpfc_vport_t *vport = phba->pport;

	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user((uint8_t *) & bind_ctl, (uint8_t *) cip->lpfc_arg1,
			   (ulong) (sizeof (bind_ctl)))) {
		LPFC_DRVR_LOCK(phba, iflag);
		rc = EIO;
		return rc;
	}
	LPFC_DRVR_LOCK(phba, iflag);

	switch (bind_ctl.bind_type) {

	case LPFC_WWNN_BIND:
		bind_type = FCP_SEED_WWNN;
		bind_id = &bind_ctl.wwnn[0];
		break;

	case LPFC_WWPN_BIND:
		bind_type = FCP_SEED_WWPN;
		bind_id = &bind_ctl.wwpn[0];
		break;

	case LPFC_DID_BIND:
		bind_type = FCP_SEED_DID;
		bind_id = &bind_ctl.did;
		break;

	case LPFC_SCSI_ID:
		bind_id = 0;
		break;

	default:
		rc = EIO;
		break;
	}

	if (rc)
		return rc;

	rc = lpfc_del_bind(vport, bind_type, bind_id, bind_ctl.scsi_id);

	return rc;
}

int
lpfc_ioctl_list_bind(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	unsigned long next_index = 0;
	unsigned long max_index = (unsigned long)cip->lpfc_arg1;
	HBA_BIND_LIST *bind_list;
	HBA_BIND_ENTRY *bind_array;
	LPFC_BINDLIST_t *pbdl;
	LPFC_NODELIST_t *pndl;
	lpfc_vport_t *vport = phba->pport;
	int rc = 0;

	bind_list = (HBA_BIND_LIST *) dataout;
	bind_array = &bind_list->entry[0];

	/* Iterate through the mapped list */
	list_for_each_entry(pndl, &vport->fc_nlpmap_list, nlp_listp) {
		if (next_index >= max_index) {
			rc = E2BIG;
			next_index++;
			continue;
		}


		memset(&bind_array[next_index], 0, sizeof (HBA_BIND_ENTRY));
		bind_array[next_index].scsi_id = pndl->nlp_scsi_id;
		bind_array[next_index].did = pndl->nlp_DID;
		memcpy(&bind_array[next_index].wwpn, &pndl->nlp_portname,
		       sizeof (HBA_WWN));
		memcpy(&bind_array[next_index].wwnn, &pndl->nlp_nodename,
		       sizeof (HBA_WWN));
		if (pndl->nlp_flag & NLP_AUTOMAP)
			bind_array[next_index].flags |= HBA_BIND_AUTOMAP;
		if (pndl->nlp_flag & NLP_SEED_WWNN)
			bind_array[next_index].bind_type = BIND_WWNN;
		if (pndl->nlp_flag & NLP_SEED_WWPN)
			bind_array[next_index].bind_type = BIND_WWPN;
		if (pndl->nlp_flag & NLP_SEED_ALPA)
			bind_array[next_index].bind_type = BIND_ALPA;
		else if (pndl->nlp_flag & NLP_SEED_DID)
			bind_array[next_index].bind_type = BIND_DID;

		bind_array[next_index].flags |= HBA_BIND_MAPPED;
		if (pndl && pndl->nlp_Target) {
			if (pndl->nlp_Target->targetFlags & FC_NPR_ACTIVE)
				bind_array[next_index].flags |= HBA_BIND_NODEVTMO;
			if (pndl->nlp_Target->rptLunState == REPORT_LUN_COMPLETE)
				bind_array[next_index].flags |= HBA_BIND_RPTLUNST;
		}

		next_index++;
	}

	/* Iterate through the unmapped list */
	list_for_each_entry(pndl, &vport->fc_nlpunmap_list, nlp_listp) {
		if (next_index >= max_index) {
			rc = E2BIG;
			next_index++;
			continue;
		}

		memset(&bind_array[next_index], 0, sizeof (HBA_BIND_ENTRY));
		bind_array[next_index].did = pndl->nlp_DID;
		memcpy(&bind_array[next_index].wwpn, &pndl->nlp_portname,
		       sizeof (HBA_WWN));
		memcpy(&bind_array[next_index].wwnn, &pndl->nlp_nodename,
		       sizeof (HBA_WWN));
		bind_array[next_index].flags |= HBA_BIND_UNMAPPED;
		if (pndl->nlp_flag & NLP_TGT_NO_SCSIID)
			bind_array[next_index].flags |= HBA_BIND_NOSCSIID;
		if (pndl && pndl->nlp_Target) {
			if (pndl->nlp_Target->targetFlags & FC_NPR_ACTIVE)
				bind_array[next_index].flags |= HBA_BIND_NODEVTMO;
			if (pndl->nlp_Target->rptLunState == REPORT_LUN_COMPLETE)
				bind_array[next_index].flags |= HBA_BIND_RPTLUNST;
		}

		next_index++;
	}

	/* Iterate through the bind list */
	list_for_each_entry(pbdl, &vport->fc_nlpbind_list, nlp_listp) {
		if (next_index >= max_index) {
			rc = E2BIG;
			next_index++;
			continue;
		}

		memset(&bind_array[next_index], 0, sizeof (HBA_BIND_ENTRY));
		bind_array[next_index].scsi_id = pbdl->nlp_scsi_id;
		if (pbdl->nlp_bind_type & FCP_SEED_DID) {
			bind_array[next_index].bind_type = BIND_DID;
			bind_array[next_index].did = pbdl->nlp_DID;
		}

		if (pbdl->nlp_bind_type & FCP_SEED_WWPN) {
			bind_array[next_index].bind_type = BIND_WWPN;
			memcpy((uint8_t *) & bind_array[next_index].wwpn,
			       &pbdl->nlp_portname, sizeof (HBA_WWN));
		}

		if (pbdl->nlp_bind_type & FCP_SEED_WWNN) {
			bind_array[next_index].bind_type = BIND_WWNN;
			memcpy((uint8_t *) & bind_array[next_index].wwnn,
			       &pbdl->nlp_nodename, sizeof (HBA_WWN));
		}

		bind_array[next_index].flags |= HBA_BIND_BINDLIST;
		next_index++;
	}

	bind_list->NumberOfEntries = next_index;
	return rc;
}

int
lpfc_ioctl_get_vpd(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	struct vpd *dp;
	int rc = 0;

	dp = (struct vpd *) dataout;

	if (cip->lpfc_arg4 != VPD_VERSION1)
		rc = EINVAL;

	dp->version = VPD_VERSION1;

	memset(dp->ModelDescription, 0, 256);
	memset(dp->Model, 0, 80);
	memset(dp->ProgramType, 0, 256);
	memset(dp->PortNum, 0, 20);

	if (phba->vpd_flag & VPD_MASK) {
		if (phba->vpd_flag & VPD_MODEL_DESC) {
			memcpy(dp->ModelDescription, phba->ModelDesc, 256);
		}
		if (phba->vpd_flag & VPD_MODEL_NAME) {
			memcpy(dp->Model, phba->ModelName, 80);
		}
		if (phba->vpd_flag & VPD_PROGRAM_TYPE) {
			memcpy(dp->ProgramType, phba->ProgramType, 256);
		}
		if (phba->vpd_flag & VPD_PORT) {
			memcpy(dp->PortNum, phba->Port, 20);
		}
	}

	return rc;
}

int
lpfc_ioctl_get_dumpregion(lpfcHBA_t *phba, LPFCCMDINPUT_t  *cip, void *dataout)
{
	uint32_t identifier = (uint32_t)(unsigned long) cip->lpfc_arg1;
	uint32_t size = cip->lpfc_outsz;
	uint32_t *bufp = (uint32_t *) dataout;
	int rc = 0;

	switch (identifier) {
	case 0:		/* SLI Registers */
		if (size < 16) {
			/*
			 * App is hunting for size requirements.
			 * Return minimum buffer size that
			 * is required and return ENOMEM.
			 */
			rc = ENOMEM;
			size = 16;
			cip->lpfc_outsz = size;
		}
		else {
			/*
			 * App's buffer is large enough to
			 * hold SLI registers
			 */
			*bufp++ = readl((phba->ctrl_regs_memmap_p) + 0);
			*bufp++ = readl((phba->ctrl_regs_memmap_p) + 4);
			*bufp++ = readl((phba->ctrl_regs_memmap_p) + 8);
			*bufp++ = readl((phba->ctrl_regs_memmap_p) + 12);
                        size = 16;
			cip->lpfc_outsz = size;
		}

		/*
		 * Return buffer size in arg 2.
		 */
		if (copy_to_user((uint8_t *)cip->lpfc_arg2,
			(uint8_t *)&size, sizeof(uint32_t))) {
			rc = EIO;
		}
		break;

	case 1:		/* Board SLIM */
	case 2:		/* Port Control Block */
	case 3:		/* Mailbox in Host Memory */
	case 4:		/* Host Get/Put pointer array */
	case 5:		/* Port Get/Put pointer array */
	case 6:		/* Command/Response Ring */
	case 7:		/* DriverInternal Structures */
		rc = ENOENT;
		break;
	default:
		rc = EINVAL;
		break;
	}

	return rc;
}

int
lpfc_ioctl_get_lpfcdfc_info(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip,
			    void *dataout)
{
	LPFCDFCDrvInfo *info;

	info = (LPFCDFCDrvInfo *) dataout;
	sprintf(info->version, "%s", lpfc_release_version);
	sprintf(info->name, "%s", lpfc_drvr_name);
	info->sliMode = phba->sli_rev;
	info->featureList = FEATURE_LIST;
	return 0;
}

int
lpfc_ioctl_loopback_mode(lpfcHBA_t *phba,  LPFCCMDINPUT_t  *cip,
			 void *dataout)
{
	LPFC_SLI_t *psli = &phba->sli;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_FCP_RING];
	uint32_t link_flags = cip->lpfc_arg4;
	uint32_t timeout = cip->lpfc_arg5 * 100;
        lpfc_vport_t *vport = phba->pport;
	LPFC_MBOXQ_t *pmboxq;
	int mbxstatus;
	int i = 0;
	int rc = 0;
	unsigned long iflag = 0;
	if ((phba->hba_state == LPFC_HBA_ERROR) ||
	    (!(psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE))) {
		return EACCES;
	}

	if ((pmboxq = lpfc_mbox_alloc(phba, 0)) == 0) {
		return ENOMEM;
}

	scsi_block_requests(phba->pport->scsihost);
	while (pring->txcmplq_cnt) {
		if (i++ > 500)
			break;
		mdelay(10);
	}

	/*
	 * Check if diagnostic sli2 mode override is needed.
	 * is used for loopback testing since the IOCBs 
	 * for loopback are not supported in SLI-3.
	 */
	if (phba->sli_rev == 3) {
		phba->diag_sli2_mode = TRUE;
                lpfc_offline(phba);
                lpfc_reset_barrier(phba);
                lpfc_sli_brdreset(phba, 1);
                lpfc_hba_down_post(phba);
                rc = lpfc_sli_brdready(phba, HS_MBRDY);
		phba->sli_rev = 2;
                lpfc_online(phba);
        }

	memset((void *)pmboxq, 0, sizeof (LPFC_MBOXQ_t));
	pmboxq->mb.mbxCommand = MBX_DOWN_LINK;
	pmboxq->mb.mbxOwner = OWN_HOST;


	mbxstatus = lpfc_sli_issue_mbox_wait(phba, pmboxq, LPFC_MBOX_TMO);
	if ((mbxstatus == MBX_SUCCESS) && (pmboxq->mb.mbxStatus == 0)) {
		/* wait for link down before proceeding */
		i = 0;
		while (vport->port_state != LPFC_LINK_DOWN) {
			if (i++ > timeout) {
				rc = ETIMEDOUT;
				goto loopback_mode_exit;
			}
	                /* Can't sleep holding locks */
        	        LPFC_DRVR_UNLOCK(phba, iflag);
                	lpfc_sleep_ms(phba, 10);
	                LPFC_DRVR_LOCK(phba, iflag);
		}

		memset((void *)pmboxq, 0, sizeof (LPFC_MBOXQ_t));
		pmboxq->mb.un.varInitLnk.link_flags = link_flags;
		pmboxq->mb.mbxCommand = MBX_INIT_LINK;
		pmboxq->mb.mbxOwner = OWN_HOST;

		mbxstatus = lpfc_sli_issue_mbox_wait(phba, pmboxq, LPFC_MBOX_TMO);
		if ((mbxstatus != MBX_SUCCESS) || (pmboxq->mb.mbxStatus)) {
			rc = ENODEV;
		} else {
			/* wait for the link attention interrupt */
	                /* Can't sleep holding locks */
        	        LPFC_DRVR_UNLOCK(phba, iflag);
	                lpfc_sleep_ms(phba, 100);
        	        LPFC_DRVR_LOCK(phba, iflag);

			i = 0;
			while (vport->port_state != LPFC_HBA_READY) {
				if (i++ > timeout) {
					rc = ETIMEDOUT;
					break;
				}
		                /* Can't sleep holding locks */
                		LPFC_DRVR_UNLOCK(phba, iflag);
		                lpfc_sleep_ms(phba, 10);
                		LPFC_DRVR_LOCK(phba, iflag);
			}
		}
	} else {
		rc = ENODEV;
	}
loopback_mode_exit:
	scsi_unblock_requests(phba->pport->scsihost);
	/*
	 * Let SLI layer release mboxq if mbox command completed after timeout.
	 */
	if (mbxstatus == MBX_TIMEOUT)
		pmboxq->mbox_cmpl = lpfc_sli_def_mbox_cmpl;
	else
		lpfc_mbox_free(phba, pmboxq);

	return rc;
}

int
lpfc_ioctl_loopback_test(lpfcHBA_t *phba, LPFCCMDINPUT_t  *cip, void *dataout)
{
	LPFC_SLI_t *psli = &phba->sli;
	LPFC_SLI_RING_t *pring = &psli->ring[LPFC_ELS_RING];
	struct lpfc_timedout_iocb_ctxt *iocb_ctxt;
	uint32_t size = cip->lpfc_outsz;
	LPFC_MBOXQ_t *pmboxq;
	int mbxstatus;
	uint16_t rpi;
	LPFC_IOCBQ_t *cmdiocbq, *rspiocbq;
	IOCB_t *cmd, *rsp;
	DMABUF_t *txbmp, *rxbmp;
	ULP_BDE64 *txbpl, *rxbpl;
	DMABUFEXT_t *txbuffer, *rxbuffer;
	struct list_head head, *curr, *next;
	DMABUF_t *dmp;
	DMABUF_t *mp[2] = {0, 0};
	uint16_t txxri, rxxri;
	uint32_t num_bde;
	uint8_t *ptr;
	ULP_BDE64 *bpl;
	int rc = 0;
	int i = 0;
	unsigned long iflag = 0;

	if ((phba->hba_state == LPFC_HBA_ERROR) ||
	    (!(psli->sliinit.sli_flag & LPFC_SLI2_ACTIVE))) {
		rc = EACCES;
		goto loopback_exit;
	}

	if ((size == 0) || (size > 80 * 4096)) {
		rc = ERANGE;
		goto loopback_exit;
	}

	/* Allocate mboxq structure */
	if ((pmboxq = lpfc_mbox_alloc(phba, 0)) == 0) {
		rc = ENOMEM;
		goto loopback_exit;
	}

	/* Acquire a rpi for the HBA */
	if (lpfc_reg_login(phba, phba->pport->vpi, phba->pport->fc_myDID,
			   (uint8_t *) &phba->pport->fc_sparam, pmboxq, 0)) {
		rc = ENOMEM;
		goto loopback_free_mbox;
	}

	dmp = (DMABUF_t *) pmboxq->context1;
	pmboxq->context1 = NULL;

	mbxstatus = lpfc_sli_issue_mbox_wait(phba, pmboxq, LPFC_MBOX_TMO);
	if ((mbxstatus != MBX_SUCCESS) || (pmboxq->mb.mbxStatus)) {
		/* If mailbox timedout let SLI free the resources */
		if ( mbxstatus == MBX_TIMEOUT) {
			pmboxq->context1 = dmp;
			pmboxq->mbox_cmpl = lpfc_sli_def_mbox_cmpl;
			goto loopback_exit;
		} else {
			lpfc_mbuf_free(phba, dmp->virt, dmp->phys);
			kfree(dmp);
			rc = ENODEV;
			goto loopback_free_mbox;
		}
	}

	rpi = pmboxq->mb.un.varWords[0];

	lpfc_mbuf_free(phba, dmp->virt, dmp->phys);
	kfree(dmp);

	if ((cmdiocbq = lpfc_iocb_alloc(phba, 0)) == 0) {
		rc = ENOMEM;
		goto loopback_unreg_login;
	}

	memset((void *)cmdiocbq, 0, sizeof (LPFC_IOCBQ_t));
	cmd = &cmdiocbq->iocb;

	if ((rspiocbq = lpfc_iocb_alloc(phba, 0)) == 0) {
		rc = ENOMEM;
		goto loopback_free_cmdiocbq;
	}

	memset((void *)rspiocbq, 0, sizeof (LPFC_IOCBQ_t));
	rsp = &rspiocbq->iocb;

	if ((txbmp = kmalloc(sizeof (DMABUF_t), GFP_KERNEL)) == 0) {
		rc = ENOMEM;
		goto loopback_free_rspiocbq;
	}

	if ((txbmp->virt = lpfc_mbuf_alloc(phba, 0, &txbmp->phys)) == 0) {
		rc = ENOMEM;
		goto loopback_free_txbmp;
	}

	INIT_LIST_HEAD(&txbmp->list);
	txbpl = (ULP_BDE64 *) txbmp->virt;

	txbuffer = dfc_cmd_data_alloc(phba, cip->lpfc_arg1, txbpl, size);
	if (!txbuffer) {
		rc = ENOMEM;
		goto loopback_free_txvirt;
	}

	if ((rxbmp = kmalloc(sizeof (DMABUF_t), GFP_KERNEL)) == 0) {
		rc = ENOMEM;
		goto loopback_free_txbuffer;
	}

	if ((rxbmp->virt = lpfc_mbuf_alloc(phba, 0, &rxbmp->phys)) == 0) {
		rc = ENOMEM;
		goto loopback_free_rxbmp;
	}

	INIT_LIST_HEAD(&rxbmp->list);
	rxbpl = (ULP_BDE64 *) rxbmp->virt;

	rxbuffer = dfc_cmd_data_alloc(phba, 0, rxbpl, size);
	if (!rxbuffer) {
		rc = ENOMEM;
		goto loopback_free_rxvirt;
	}

	phba->fc_loopback_rxxri = 0;

	/* Send initial XMIT_SEQUENCE64_CR to obtain an xri */

	if ((dmp = kmalloc(sizeof (DMABUF_t), GFP_KERNEL)) == 0) {
		dfc_cmd_data_free(phba, rxbuffer);
		rc = ENOMEM;
		goto loopback_free_rxvirt;
	}

	if ((dmp->virt = lpfc_mbuf_alloc(phba, 0, &dmp->phys)) == 0) {
		kfree(dmp);
		dfc_cmd_data_free(phba, rxbuffer);
		rc = ENOMEM;
		goto loopback_free_rxvirt;
	}

	INIT_LIST_HEAD(&dmp->list);
	bpl = (ULP_BDE64 *) dmp->virt;
	memset(bpl, 0, sizeof (ULP_BDE64));

	memset((void *)cmdiocbq, 0, sizeof (LPFC_IOCBQ_t));
	memset((void *)rspiocbq, 0, sizeof (LPFC_IOCBQ_t));

	cmd->un.xseq64.bdl.addrHigh = putPaddrHigh(dmp->phys);
	cmd->un.xseq64.bdl.addrLow = putPaddrLow(dmp->phys);
	cmd->un.xseq64.bdl.bdeFlags = BUFF_TYPE_BDL;
	cmd->un.xseq64.bdl.bdeSize = sizeof(ULP_BDE64);

	cmd->un.xseq64.w5.hcsw.Fctl = LA;
	cmd->un.xseq64.w5.hcsw.Dfctl = 0;
	cmd->un.xseq64.w5.hcsw.Rctl = FC_UNSOL_DATA;
	cmd->un.xseq64.w5.hcsw.Type = FC_VENDOR_SPECIFIC;

	cmd->ulpCommand = CMD_XMIT_SEQUENCE64_CR;
	cmd->ulpBdeCount = 1;
	cmd->ulpLe = 1;
	cmd->ulpClass = CLASS3;
	cmd->ulpContext = rpi;

	cmdiocbq->iocb_flag |= LPFC_IO_LIBDFC;
	rc = lpfc_sli_issue_iocb_wait(phba, pring, cmdiocbq, SLI_IOCB_USE_TXQ,
				      rspiocbq, (phba->fc_ratov * 2) + LPFC_DRVR_TIMEOUT);

	lpfc_mbuf_free(phba, dmp->virt, dmp->phys);
	kfree(dmp);

	if (rc == IOCB_TIMEDOUT) {
		dfc_cmd_data_free(phba, rxbuffer);
		lpfc_iocb_free(phba, rspiocbq);
		iocb_ctxt = kmalloc(sizeof(struct lpfc_timedout_iocb_ctxt),
				    GFP_ATOMIC);
		if (!iocb_ctxt) {
			rc = EACCES;
			goto loopback_unreg_login;
		}
		
		cmdiocbq->context1 = iocb_ctxt;
		cmdiocbq->context2 = NULL;
		iocb_ctxt->rspiocbq = NULL;
		iocb_ctxt->mp = txbmp;
		iocb_ctxt->bmp = rxbmp;
		iocb_ctxt->outdmp = NULL;
		iocb_ctxt->lpfc_cmd = NULL;
		iocb_ctxt->indmp = txbuffer;
		
		cmdiocbq->iocb_cmpl = lpfc_ioctl_timeout_iocb_cmpl;
		rc = EACCES;			
		goto loopback_unreg_login;
	}

	if ((rc != IOCB_SUCCESS) || (rsp->ulpStatus != IOCB_SUCCESS)) {
		dfc_cmd_data_free(phba, rxbuffer);
		rc = EIO;
		goto loopback_free_rxvirt;
	}

        LPFC_DRVR_UNLOCK(phba, iflag);
	while ((phba->fc_loopback_rxxri == 0) && (i++ < 10)) {
		lpfc_sleep_ms(phba, 10);
	}
        LPFC_DRVR_LOCK(phba, iflag);

	txxri = rsp->ulpContext;
	rxxri = phba->fc_loopback_rxxri;

	if (rxxri == 0) {
		dfc_cmd_data_free(phba, rxbuffer);
		rc = EIO;
		goto loopback_free_rxvirt;
	}

	/* Queue buffers for the receive exchange */
	num_bde = (uint32_t)rxbuffer->flag;
	dmp = &rxbuffer->dma;

	memset((void *)cmdiocbq, 0, sizeof (LPFC_IOCBQ_t));
	i = 0;

	INIT_LIST_HEAD(&head);
	list_add_tail(&head, &dmp->list);
	list_for_each_safe(curr, next, &head) {
		mp[i] = list_entry(curr, DMABUF_t, list);
		list_del(curr);

		cmd->un.cont64[i].addrHigh = putPaddrHigh(mp[i]->phys);
		cmd->un.cont64[i].addrLow = putPaddrLow(mp[i]->phys);
		cmd->un.cont64[i].tus.f.bdeSize = ((DMABUFEXT_t *)mp[i])->size;
		cmd->ulpBdeCount = ++i;

		if ((--num_bde > 0) && (i < 2)) {
			continue;
		}

		cmd->ulpCommand = CMD_QUE_XRI_BUF64_CX;
		cmd->ulpLe = 1;
		cmd->ulpClass = CLASS3;
		cmd->ulpContext = rxxri;

		rc = lpfc_sli_issue_iocb(phba, pring, cmdiocbq, 0);
		if (rc == IOCB_ERROR) {
			dfc_cmd_data_free(phba, (DMABUFEXT_t *)mp[0]);
			if (mp[1])
				dfc_cmd_data_free(phba, (DMABUFEXT_t *)mp[1]);
			dmp = list_entry(next, DMABUF_t, list);
			rc = EIO;
			goto loopback_free_dmp;
		}

		lpfc_sli_ringpostbuf_put(phba, pring, mp[0]);
		if (mp[1]) {
			lpfc_sli_ringpostbuf_put(phba, pring, mp[1]);
			mp[1] = NULL;
		}

		/* The iocb was freed by lpfc_sli_issue_iocb */
		if ((cmdiocbq = lpfc_iocb_alloc(phba, 0)) == 0) {
			dmp = list_entry(next, DMABUF_t, list);
			rc = EIO;
			goto loopback_free_dmp;
		}

		memset((void *)cmdiocbq, 0, sizeof (LPFC_IOCBQ_t));
		cmd = &cmdiocbq->iocb;
		i = 0;
	}
	list_del(&head);

	phba->fc_loopback_data = NULL;

	/* Build the XMIT_SEQUENCE iocb */
	memset((void *)cmdiocbq, 0, sizeof (LPFC_IOCBQ_t));
	memset((void *)rspiocbq, 0, sizeof (LPFC_IOCBQ_t));

	num_bde = (uint32_t)txbuffer->flag;

	cmd->un.xseq64.bdl.addrHigh = putPaddrHigh(txbmp->phys);
	cmd->un.xseq64.bdl.addrLow = putPaddrLow(txbmp->phys);
	cmd->un.xseq64.bdl.bdeFlags = BUFF_TYPE_BDL;
	cmd->un.xseq64.bdl.bdeSize = (num_bde * sizeof(ULP_BDE64));

	cmd->un.xseq64.w5.hcsw.Fctl = (LS | LA);
	cmd->un.xseq64.w5.hcsw.Dfctl = 0;
	cmd->un.xseq64.w5.hcsw.Rctl = FC_UNSOL_DATA;
	cmd->un.xseq64.w5.hcsw.Type = FC_VENDOR_SPECIFIC;

	cmd->ulpCommand = CMD_XMIT_SEQUENCE64_CX;
	cmd->ulpBdeCount = 1;
	cmd->ulpLe = 1;
	cmd->ulpClass = CLASS3;
	cmd->ulpContext = txxri;

	cmdiocbq->iocb_flag |= LPFC_IO_LIBDFC;
	rc = lpfc_sli_issue_iocb_wait(phba, pring, cmdiocbq, SLI_IOCB_USE_TXQ,
				      rspiocbq, (phba->fc_ratov * 2) + LPFC_DRVR_TIMEOUT);

	if (rc == IOCB_TIMEDOUT) {
		lpfc_iocb_free(phba, rspiocbq);
		iocb_ctxt = kmalloc(sizeof(struct lpfc_timedout_iocb_ctxt),
				    GFP_ATOMIC);
		if (!iocb_ctxt) {
			rc = EACCES;
			goto loopback_unreg_login;
		}
		
		cmdiocbq->context1 = iocb_ctxt;
		cmdiocbq->context2 = NULL;
		iocb_ctxt->rspiocbq = NULL;
		iocb_ctxt->mp = txbmp;
		iocb_ctxt->bmp = rxbmp;
		iocb_ctxt->outdmp = NULL;
		iocb_ctxt->lpfc_cmd = NULL;
		iocb_ctxt->indmp = txbuffer;
		
		cmdiocbq->iocb_cmpl = lpfc_ioctl_timeout_iocb_cmpl;
		rc = EACCES;			
		goto loopback_unreg_login;
	}

	if ((rc != IOCB_SUCCESS) || (rsp->ulpStatus != IOCB_SUCCESS)) {
		rc = EIO;
		goto loopback_free_rxvirt;
	}

	i = 0;
        LPFC_DRVR_UNLOCK(phba, iflag);
	while ((phba->fc_loopback_data == 0) && (i++ < 10)) {
		lpfc_sleep_ms(phba, 10);
	}
        LPFC_DRVR_LOCK(phba, iflag);

	/* copy the received payload */
	ptr = dataout;
	dmp = phba->fc_loopback_data;
	phba->fc_loopback_data = NULL;

	if (dmp == NULL) {
		rc = EIO;
		goto loopback_free_rxvirt;
	}

	INIT_LIST_HEAD(&head);
	list_add_tail(&head, &dmp->list);
	list_for_each_safe(curr, next, &head) {
		dmp = list_entry(curr, DMABUF_t, list);
		memcpy(ptr, dmp->virt, ((DMABUFEXT_t *)dmp)->size);
		ptr += ((DMABUFEXT_t *)dmp)->size;
	}
	list_del(&head);

loopback_free_dmp:
	dfc_cmd_data_free(phba, (DMABUFEXT_t *)dmp);
loopback_free_rxvirt:
	lpfc_mbuf_free(phba, rxbmp->virt, rxbmp->phys);
loopback_free_rxbmp:
	kfree(rxbmp);
loopback_free_txbuffer:
	dfc_cmd_data_free(phba, txbuffer);
loopback_free_txvirt:
	lpfc_mbuf_free(phba, txbmp->virt, txbmp->phys);
loopback_free_txbmp:
	kfree(txbmp);
loopback_free_rspiocbq:
	lpfc_iocb_free(phba, rspiocbq);
loopback_free_cmdiocbq:
	if (cmdiocbq)
		lpfc_iocb_free(phba, cmdiocbq);
loopback_unreg_login:
	lpfc_unreg_login(phba, phba->pport->vpi, rpi, pmboxq);
	mbxstatus = lpfc_sli_issue_mbox_wait(phba, pmboxq, LPFC_MBOX_TMO);
	if ((mbxstatus != MBX_SUCCESS) || (pmboxq->mb.mbxStatus)) {
		rc = EIO;
	}

	/* If mailbox timedout let SLI free the resources */
	if ( mbxstatus == MBX_TIMEOUT) {
		pmboxq->mbox_cmpl = lpfc_sli_def_mbox_cmpl;
		goto loopback_exit;
	}
loopback_free_mbox:
	lpfc_mbox_free(phba, pmboxq);
loopback_exit:
	return rc;
}

int
dfc_rsp_data_copy_to_buf(lpfcHBA_t * phba,
		  uint8_t * outdataptr, DMABUFEXT_t * mlist, uint32_t size)
{
	DMABUFEXT_t *mlast = 0;
	int cnt, offset = 0;
	struct list_head head, *curr, *next;

	if (!mlist)
		return(0);

	list_add_tail(&head, &mlist->dma.list);

	list_for_each_safe(curr, next, &head) {
		mlast = list_entry(curr, DMABUFEXT_t , dma.list);
		if (!size)
			break;
		if (!mlast)
			break;

		/* We copy chucks of 4K */
		if (size > 4096)
			cnt = 4096;
		else
			cnt = size;

		if (outdataptr) {
			/* Copy data to user space */
			memcpy
			    ((uint8_t *) (outdataptr + offset),
			     (uint8_t *) mlast->dma.virt, (ulong) cnt);
		}
		offset += cnt;
		size -= cnt;
	}
	list_del(&head);
	return (0);
}

int
dfc_rsp_data_copy(lpfcHBA_t * phba,
		  uint8_t * outdataptr, DMABUFEXT_t * mlist, uint32_t size)
{
	DMABUFEXT_t *mlast = 0;
	int cnt, offset = 0;
	unsigned long iflag;
	struct list_head head, *curr, *next;

	if (!mlist)
		return 0;

	list_add_tail(&head, &mlist->dma.list);

	list_for_each_safe(curr, next, &head) {
		mlast = list_entry(curr, DMABUFEXT_t, dma.list);
		if (!size)
			break;

		/* We copy chucks of 4K */
		if (size > 4096)
			cnt = 4096;
		else
			cnt = size;

		if (outdataptr) {
			pci_dma_sync_single(phba->pcidev, mlast->dma.phys,
						LPFC_BPL_SIZE, PCI_DMA_TODEVICE);
			/* Copy data to user space */
			LPFC_DRVR_UNLOCK(phba, iflag);
			if (copy_to_user
			    ((uint8_t *) (outdataptr + offset),
			     (uint8_t *) mlast->dma.virt, (ulong) cnt)) {
				LPFC_DRVR_LOCK(phba, iflag);
				return 1;
			}
			LPFC_DRVR_LOCK(phba, iflag);
		}
		offset += cnt;
		size -= cnt;
	}

	list_del(&head);
	return 0;
}

DMABUFEXT_t *
dfc_cmd_data_alloc(lpfcHBA_t *phba, char *indataptr, ULP_BDE64 *bpl,
		   uint32_t size)
{
	DMABUFEXT_t *mlist = 0;
	DMABUFEXT_t *dmp;
	int cnt, offset = 0, i = 0;
	unsigned long iflag;
	struct pci_dev *pcidev;

	pcidev = phba->pcidev;

	while (size) {
		/* We get chucks of 4K */
		if (size > 4096)
			cnt = 4096;
		else
			cnt = size;

		/* allocate DMABUFEXT_t buffer header */
		dmp = kmalloc(sizeof (DMABUFEXT_t), GFP_ATOMIC);
		if ( dmp == 0 )
			goto out;

		INIT_LIST_HEAD(&dmp->dma.list);

		/* Queue it to a linked list */
		if (mlist)
			list_add_tail(&dmp->dma.list, &mlist->dma.list);
		else
			mlist = dmp;

		/* allocate buffer */
		dmp->dma.virt = pci_alloc_consistent(pcidev, 
						     cnt, 
						     &(dmp->dma.phys));

		if (dmp->dma.virt == 0)
			goto out;

		dmp->size = cnt;
		if (indataptr) {
			/* Copy data from user space in */
			LPFC_DRVR_UNLOCK(phba, iflag);
			if (copy_from_user
			    ((uint8_t *) dmp->dma.virt,
			     (uint8_t *) (indataptr + offset), (ulong) cnt)) {
				LPFC_DRVR_LOCK(phba, iflag);
				goto out;
			}

			LPFC_DRVR_LOCK(phba, iflag);
			bpl->tus.f.bdeFlags = 0;

			pci_dma_sync_single(phba->pcidev, dmp->dma.phys, 
						LPFC_BPL_SIZE, PCI_DMA_TODEVICE);
		} else {
			bpl->tus.f.bdeFlags = BUFF_USE_RCV;
		}

		/* build buffer ptr list for IOCB */
		bpl->addrLow = le32_to_cpu( putPaddrLow(dmp->dma.phys) );
		bpl->addrHigh = le32_to_cpu( putPaddrHigh(dmp->dma.phys) );
		bpl->tus.f.bdeSize = (ushort) cnt;
		bpl->tus.w = le32_to_cpu(bpl->tus.w);
		bpl++;

		i++;
		offset += cnt;
		size -= cnt;
	}

	mlist->flag = i;
	return mlist;
 
 out:
	dfc_cmd_data_free(phba, mlist);
	return 0;
}

DMABUFEXT_t *
dfc_fcp_cmd_data_alloc(lpfcHBA_t * phba,
		   char *indataptr, ULP_BDE64 * bpl,
		   uint32_t size, struct lpfc_dmabuf *bmp_list)
{
	DMABUFEXT_t *mlist = 0;
	DMABUFEXT_t *dmp;
	int cnt, offset = 0, i = 0;
	struct pci_dev *pcidev;
	uint32_t num_bmps, num_bde, max_bde;
	uint32_t top_num_bdes = 0;
	struct list_head *curr, *next;
	struct lpfc_dmabuf *mlast;
	struct lpfc_dmabuf *bmp;
	ULP_BDE64 *topbpl = NULL;
	num_bmps = 1;
	num_bde = 0;
	

	pcidev = phba->pcidev;

	/*
	 * 3 bdes are reserved for a fcp_command, a fcp_reponse and
	 * a continuation bde.
	 */
	max_bde = (LPFC_BPL_SIZE / sizeof(ULP_BDE64))-3;


	while (size) {
		/* We get chucks of 4K */
		if (size > 8192)
			cnt = 8192;
		else
			cnt = size;

		if (num_bde == max_bde) {
			bmp = kmalloc(sizeof (struct lpfc_dmabuf),
				GFP_KERNEL);
			if (bmp == 0) {
				goto out;
			}
			memset(bmp, 0, sizeof (struct lpfc_dmabuf));
			bmp->virt =
				lpfc_mbuf_alloc(phba, 0, &bmp->phys);
			if (!bmp->virt) {
				kfree(bmp);
				goto out;
			}
			max_bde = ((LPFC_BPL_SIZE / sizeof(ULP_BDE64))-3);
			/* Fill in continuation entry to next bpl */
			bpl->addrHigh =
				le32_to_cpu(putPaddrHigh(bmp->phys));
			bpl->addrLow =
				le32_to_cpu(putPaddrLow(bmp->phys));
			bpl->tus.f.bdeFlags = BPL64_SIZE_WORD;
			num_bde++;
			if (num_bmps == 1) {
				top_num_bdes = num_bde;
			} else {
				topbpl->tus.f.bdeSize = (num_bde *
					sizeof (ULP_BDE64));
				topbpl->tus.w =
					le32_to_cpu(topbpl->tus.w);
			}
			topbpl = bpl;
			bpl = (ULP_BDE64 *) bmp->virt;
			list_add_tail(&bmp->list, &bmp_list->list);
			num_bde = 0;
			num_bmps++;
		}

		/* allocate DMABUFEXT_t buffer header */
		dmp = kmalloc(sizeof (DMABUFEXT_t), GFP_KERNEL);
		if ( dmp == 0 ) {
			goto out;
		}

		INIT_LIST_HEAD(&dmp->dma.list);

		/* Queue it to a linked list */
		if (mlist)
			list_add_tail(&dmp->dma.list, &mlist->dma.list);
		else
			mlist = dmp;

		/* allocate buffer */
		dmp->dma.virt = pci_alloc_consistent(pcidev,
						     cnt,
						     &(dmp->dma.phys));


		if (dmp->dma.virt == 0) {
			goto out;
		}
		dmp->size = cnt;

		if (indataptr) {
			/* Copy data from user space in */
			if (copy_from_user
			    ((uint8_t *) dmp->dma.virt,
			     (uint8_t *) (indataptr + offset), (ulong) cnt)) {
				goto out;
			}
			bpl->tus.f.bdeFlags = 0;

			pci_dma_sync_single(phba->pcidev, dmp->dma.phys,
					    LPFC_BPL_SIZE, PCI_DMA_TODEVICE);

		} else {
			memset((uint8_t *)dmp->dma.virt, 0, cnt);
			bpl->tus.f.bdeFlags = BUFF_USE_RCV;
		}

		/* build buffer ptr list for IOCB */
		bpl->addrLow = le32_to_cpu( putPaddrLow(dmp->dma.phys) );
		bpl->addrHigh = le32_to_cpu( putPaddrHigh(dmp->dma.phys) );
		bpl->tus.f.bdeSize = (ushort) cnt;
		bpl->tus.w = le32_to_cpu(bpl->tus.w);
		bpl++;
		num_bde++;
		i++;
		offset += cnt;
		size -= cnt;
	}

	if (num_bmps == 1)
		mlist->flag = i;
	else {
		mlist->flag = top_num_bdes;
		topbpl->tus.f.bdeSize = (num_bde *
			sizeof (ULP_BDE64));
		topbpl->tus.w =
			le32_to_cpu(topbpl->tus.w);
	}

	return (mlist);
out:
	dfc_cmd_data_free(phba, mlist);
	list_for_each_safe(curr, next, &bmp_list->list) {
		mlast = list_entry(curr, struct lpfc_dmabuf, list);
		list_del(&mlast->list);
		lpfc_mbuf_free(phba, mlast->virt, mlast->phys);
		kfree(mlast);
	}
	return (0);
}


int
dfc_cmd_data_free(lpfcHBA_t *phba, DMABUFEXT_t *mlist)
{
	DMABUFEXT_t *mlast;
	struct pci_dev *pcidev;
	struct list_head head, *curr, *next;

	if (!mlist)
		return 0;

	pcidev = phba->pcidev;
	list_add_tail(&head, &mlist->dma.list);

	list_for_each_safe(curr, next, &head) {
		mlast = list_entry(curr, DMABUFEXT_t , dma.list);
		if (mlast->dma.virt) {
			pci_free_consistent(pcidev, 
					    mlast->size, 
					    mlast->dma.virt, 
					    mlast->dma.phys);
		}

		kfree(mlast);
	}
	return 0;
}

uint32_t 
dfc_npiv_checklist (lpfc_vport_t *vport) {
	uint32_t checklist = 0;

	checklist |= CHECKLIST_BIT_NPIV;
	if (vport->phba->sli_rev == 3) {
		checklist |= CHECKLIST_BIT_SLI3;

		/* dfc_npiv spec says:
		 * Bit 2 is based on featurelevel high value 
		 * returned from READ_REV mailbox, if it is equal 
		 * to or greater than 9, then HBA has NPIV support.
		 */
		if (vport->phba->vpd.rev.feaLevelHigh >= 0x09)
			checklist |= CHECKLIST_BIT_HBA;
		/*
		 * dfc_npiv spec says:
		 * Bit 3 will only be valid if bit 0-2 are all '1's
		 */
		if ((vport->phba->max_vpi - vport->phba->vpi_cnt) > 0) 
			checklist |= CHECKLIST_BIT_RSRC;
	}
	/*
	 * Use the physical link's port state
	 */
	switch (vport->phba->pport->port_state) {
	case LPFC_LINK_UP:
	case LPFC_LOCAL_CFG_LINK:
	case LPFC_FLOGI:
	case LPFC_FABRIC_CFG_LINK:
	case LPFC_NS_REG:
	case LPFC_NS_QRY:
	case LPFC_BUILD_DISC_LIST:
	case LPFC_DISC_AUTH:
	case LPFC_CLEAR_LA:
	case LPFC_HBA_READY:
		/*
		 * dfc_npiv spec says:
		 * Bit 0, 1, 2, 4 will be reported regardless
		 * of the driver parameter setting for enable NPIV
		 */
		checklist |= CHECKLIST_BIT_LINK;
		/*
		 * dfc_npiv spec says:
		 * Bit 5 -7 should be ignored, 
		 * if any one of the bits (bit 0 - 4) is '0'
		 */
		if   ((CHECKLIST_BIT_NPIV + CHECKLIST_BIT_SLI3 + 
		       CHECKLIST_BIT_HBA + CHECKLIST_BIT_RSRC) == 
		      (checklist & 
			(CHECKLIST_BIT_NPIV + CHECKLIST_BIT_SLI3 +
			 CHECKLIST_BIT_HBA + CHECKLIST_BIT_RSRC))) {

			if (vport->phba->fc_topology == TOPOLOGY_PT_PT) {
        	               	checklist |= CHECKLIST_BIT_FBRC;
			if (vport->phba->npiv_capable == LPFC_NPIV_SUPPORTED)
					checklist |= CHECKLIST_BIT_FSUP + 
						     CHECKLIST_BIT_NORSRC; 
			}
	      	}
	}
	return checklist;
}


int
lpfc_ioctl_vport_attrib(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	int rc = 0;
	int found = 0;
	struct list_head *vport_pos;
	lpfc_vport_t *vport = NULL;
	HBA_WWN vport_wwpn;
        DFC_VPAttrib *pAttrib = dataout;
	unsigned long iflag;

	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user(&vport_wwpn, cip->lpfc_arg1, sizeof (vport_wwpn))) {
		LPFC_DRVR_LOCK(phba, iflag);
		return EIO;
	}
	LPFC_DRVR_LOCK(phba, iflag);

	/*
	 * Search for user's vport_wwpn 
	 */
	list_for_each(vport_pos, &phba->port_list) {
		vport = list_entry(vport_pos, lpfc_vport_t, listentry);

		if (0 == memcmp(&vport_wwpn, 
				&vport->fc_portname, sizeof (HBA_WWN))) {
			if (vport->port_type == LPFC_PHYSICAL_PORT)
				continue;
			else
				found=1;
			break;
		}
        }
	if (!found) {
		cip->lpfc_outsz = 0;
		rc = ENOENT;
	}
	else {
		/*
		 * Here to fill in vport attributes
		 */
		pAttrib->ver = 2;
		pAttrib->portFcId = vport->fc_myDID;
		memcpy (&pAttrib->wwpn, &vport->fc_portname, 
			sizeof(pAttrib->wwpn));
		memcpy (&pAttrib->wwnn, &vport->fc_nodename,
			sizeof(pAttrib->wwnn));
		if (strlen(&vport->vmname[0]) > 0)
			strcpy(pAttrib->name, vport->vmname);
		memcpy (&pAttrib->fabricName, &phba->fc_fabparam.nodeName,
			sizeof(pAttrib->fabricName));

		switch (vport->port_state) {
		case LPFC_INIT_START:
		case LPFC_INIT_MBX_CMDS:
			pAttrib->state = ATTRIB_STATE_UNKNOWN;
			break;
		case LPFC_LINK_DOWN:
			pAttrib->state = ATTRIB_STATE_LINKDOWN;
			break;
		case LPFC_LINK_UP:
		case LPFC_LOCAL_CFG_LINK:
		case LPFC_FLOGI:
		case LPFC_FABRIC_CFG_LINK:
		case LPFC_NS_REG:
		case LPFC_NS_QRY:
		case LPFC_BUILD_DISC_LIST:
		case LPFC_DISC_AUTH:
		case LPFC_CLEAR_LA:
			pAttrib->state = ATTRIB_STATE_INIT;
			break;
		case LPFC_HBA_READY:
			pAttrib->state = ATTRIB_STATE_ACTIVE;
			break;
		case LPFC_HBA_ERROR:
		default:
			pAttrib->state = ATTRIB_STATE_FAILED;
			break;
		}
		pAttrib->checklist = dfc_npiv_checklist(vport);
	}
        cip->lpfc_outsz = min(cip->lpfc_outsz, sizeof(DFC_VPAttrib));
	return rc;
}

int
lpfc_ioctl_npiv_ready(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	uint32_t *pchecklist = dataout;
	int rc = 0;

	if (cip->lpfc_outsz >= sizeof(uint32_t)) {
		*pchecklist = dfc_npiv_checklist(phba->pport);
		cip->lpfc_outsz = sizeof(uint32_t); 
	}
	else {
		rc = E2BIG;
	}
	return rc;
}

int
lpfc_ioctl_vport_resrc(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
        int rc = 0;
	uint32_t arpi, avpi;
	DFC_VPResource *pResrc = dataout;

	lpfc_get_hba_info(phba, &pResrc->rpi_max, &arpi, &pResrc->vlinks_max, &avpi);
	pResrc->vlinks_inuse = pResrc->vlinks_max - avpi;
	pResrc->rpi_inuse = pResrc->rpi_max - arpi;
        cip->lpfc_outsz = min(cip->lpfc_outsz, sizeof(DFC_VPResource));
	return rc;
}

int
lpfc_ioctl_vport_list(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{
	int rc = 0;
	struct list_head *vport_pos;
	lpfc_vport_t *vport = NULL;
	int entry= 0;
	uint32_t numberOfEntries = 0;
	uint32_t len = sizeof(numberOfEntries);
	unsigned long iflag;
	DFC_VPEntryList *vplist = dataout;
	DFC_VPEntry *vpentry = vplist->vpentry;

	/*
	 * Copy user data to get number of host entries available
	 */
	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user(&numberOfEntries, cip->lpfc_arg1, sizeof(numberOfEntries))) {
		LPFC_DRVR_LOCK(phba, iflag);
		return EIO;
        }
	LPFC_DRVR_LOCK(phba, iflag);

	/*
	 * Find all the vports on this pport 
	 */
	list_for_each(vport_pos, &phba->port_list) {
		vport = list_entry(vport_pos, lpfc_vport_t, listentry);
		if (vport->port_type == LPFC_PHYSICAL_PORT)
			continue;

		/*
		 * Fill the buffer with the vport entries if
		 * there is room.
		 */
		if (entry < numberOfEntries) {
			memcpy (&vpentry->wwnn, 
				&vport->fc_nodename,
       				sizeof (HBA_WWN));
			memcpy (&vpentry->wwpn,
				&vport->fc_portname,
				sizeof (HBA_WWN));
			vpentry->PortFcId = vport->fc_myDID;
			len += sizeof(DFC_VPEntry);
			vpentry++;
		}
		else {
			/*
			 * Not enough space in the user buffer but
			 * we'll continue to count our vport entries.
			 */ 
			rc = E2BIG;
		}
		entry++;		
	}
	vplist->numberOfEntries = entry;
	cip->lpfc_outsz = min(len, cip->lpfc_outsz);
	return rc;
}




int
lpfc_ioctl_vport_nodeinfo(lpfcHBA_t *phba, LPFCCMDINPUT_t *cip, void *dataout)
{

	int rc = 0;
	struct list_head *vport_pos;
	lpfc_vport_t *vport = NULL;
	int entry = 0, found = 0;
	uint32_t numberOfEntries = 0;
	unsigned long iflag;
	DFC_GetNodeInfo *vplist = dataout;
        DFC_NodeInfoEntry *np = vplist->nodeInfo;
        HBA_WWN vport_wwpn;

	LPFC_NODELIST_t *pndl;
	LPFC_BINDLIST_t *pbdl;
	struct list_head *listp;
	struct list_head *node_list[4];
	int i;


	/* 
	 * Read user data to get vport WWPN (arg1)
	 */
	LPFC_DRVR_UNLOCK(phba, iflag);
	if (copy_from_user(&vport_wwpn, cip->lpfc_arg1, sizeof (vport_wwpn))) {
		LPFC_DRVR_LOCK(phba, iflag);
		return EIO;
	}

	/*
	 * Read user data to get number of host entries available (arg2)
	 */
	if (copy_from_user(&numberOfEntries, cip->lpfc_arg2, sizeof(numberOfEntries))) {
		LPFC_DRVR_LOCK(phba, iflag);
		return EIO;
	}
        LPFC_DRVR_LOCK(phba, iflag);



	/*
	 * Search for user's vport_wwpn
	*/
	list_for_each(vport_pos, &phba->port_list) {
		vport = list_entry(vport_pos, lpfc_vport_t, listentry);

		if (0 == memcmp(&vport_wwpn,
			&vport->fc_portname, sizeof (HBA_WWN))) {
			found=1;
			break;
		}
	}
	if (!found) {
		cip->lpfc_outsz = 0;
		rc = ENOENT;
	}

	/*
	 * Since the size of bind & others are different,
	 * get the node list of bind first
	 */

	list_for_each_entry(pbdl, &vport->fc_nlpbind_list, nlp_listp) {
		if (entry < numberOfEntries) {
			np->nodeState = NODE_INFO_STATE_EXIST;
			switch (vport->port_state) {
			case LPFC_LINK_DOWN:
				np->nodeState |= NODE_INFO_STATE_LINKDOWN;
				break;
			case LPFC_LINK_UP:
			case LPFC_HBA_READY:
				np->nodeState |= NODE_INFO_STATE_READY;
				break;
			}

			np->type = 0;
			if (pbdl->nlp_bind_type & FCP_SEED_WWPN)
				np->type |= NODE_INFO_TYPE_WWPN;
			if (pbdl->nlp_bind_type & FCP_SEED_WWNN)
				np->type |= NODE_INFO_TYPE_WWNN;
			if (pbdl->nlp_bind_type & FCP_SEED_DID)
				np->type |= NODE_INFO_TYPE_DID;
			if (pbdl->nlp_bind_type & FCP_SEED_AUTO)
				np->type |= NODE_INFO_TYPE_AUTOMAP;

			np->fcpId.FcId = pbdl->nlp_DID;
			np->scsiId.ScsiTargetNumber = pbdl->nlp_scsi_id;
			memset((uint8_t *) np, 0, sizeof (LPFC_BINDLIST_t));
			memcpy(&np->fcpId.PortWWN, &pbdl->nlp_portname, 8);
			memcpy(&np->fcpId.NodeWWN, &pbdl->nlp_nodename, 8);
		}
		else {
			/*
			 * Not enough user space memory
			 */
			rc = E2BIG;
		}
		np++;
		entry++;
	}

	/* Get the node list of unmap, map, plogi and adisc */

	node_list[0] = &vport->fc_plogi_list;
	node_list[1] = &vport->fc_adisc_list;
	node_list[2] = &vport->fc_nlpunmap_list;
	node_list[3] = &vport->fc_nlpmap_list;
	for (i = 0; i < 4; i++) {
		listp = node_list[i];
		if (list_empty(listp)) 
			continue;	

		list_for_each_entry(pndl, listp, nlp_listp) {
			if (entry < numberOfEntries) {
				if (pndl->nlp_type & NLP_FABRIC) {
					continue;
				}
                        	if (pndl->nlp_flag & NLP_MAPPED_LIST) {
                                	np->nodeState = NODE_INFO_STATE_EXIST;
                        	}
				if (pndl->nlp_flag & NLP_UNMAPPED_LIST) {
					np->nodeState = NODE_INFO_STATE_EXIST |
							NODE_INFO_STATE_UNMAPPED;
				}
				switch (vport->port_state) {
        			case LPFC_LINK_DOWN:
					np->nodeState |= NODE_INFO_STATE_LINKDOWN;
					break;
				case LPFC_LINK_UP:
				case LPFC_HBA_READY:
					np->nodeState |= NODE_INFO_STATE_READY;
					break;
				}
				np->type = 0;
				if (pndl->nlp_type & FCP_SEED_WWPN)
					np->type |= NODE_INFO_TYPE_WWPN;
				if (pndl->nlp_type & FCP_SEED_WWNN)
					np->type |= NODE_INFO_TYPE_WWNN;
				if (pndl->nlp_type & FCP_SEED_DID)
					np->type |= NODE_INFO_TYPE_DID;
				if (pndl->nlp_type & FCP_SEED_AUTO)
					np->type |= NODE_INFO_TYPE_AUTOMAP;

				np->fcpId.FcId = pndl->nlp_DID;
				np->scsiId.ScsiTargetNumber = pndl->nlp_scsi_id;
				memcpy(&np->fcpId.PortWWN, &pndl->nlp_portname, 8);
				memcpy(&np->fcpId.NodeWWN, &pndl->nlp_nodename, 8);
			}
			else {
				/*
				 * Not enough user space memory
				 */
				rc = E2BIG;
			}
			np++;
			entry++;
		}
	}

	vplist->numberOfEntries = entry;
	cip->lpfc_outsz = min ((uint32_t)(entry * sizeof(DFC_NodeInfoEntry)), cip->lpfc_outsz);
	return rc;
}
