/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Fibre Channel Host Bus Adapters.                                *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2005-2008 Emulex.  All rights reserved.           *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

/*
 * $Id: lpfc_vlink.c 1.22 2005/10/07 13:59:10EDT cavery Exp  $
 */

#include <linux/version.h>
#include <linux/version.h>

#ifdef __VMKERNEL_MODULE__
#include "linux_skbuff.h"
#include "vmklinux_dist.h"
#include "npiv_api_dist.h"
#endif

#include <linux/spinlock.h>
#include <linux/pci.h>
#include <linux/blk.h>

#include <scsi.h>
#include <hosts.h>

#include "lpfc_hw.h"
#include "lpfc_cfgparm.h"
#include "lpfc_dfc.h"
#include "lpfc_sli.h"
#include "lpfc_mem.h"
#include "lpfc_sched.h"
#include "lpfc_disc.h"
#include "lpfc.h"
#include "lpfc_logmsg.h"
#include "lpfc_fcp.h"
#include "lpfc_scsi.h"
#include "lpfc_crtn.h"
#include "lpfc_compat.h"

#ifdef __VMKERNEL_MODULE__
// must be after scsi.h!
#include "scsi_transport_fc.h"
#endif



/**
 * lpfc_discovery_wait: Wait for driver discovery to quiesce.
 * @vport: The virtual port for which this call is being executed.
 *
 * This driver calls this routine specifically from lpfc_vport_create
 * to enforce a synchronous execution of vport create relative to 
 * discovery activities.  The lpfc_vport_create routine should not 
 * return to vmkernel until it can reasonably guarantee that discovery
 * has quiesced.  This behavior is required because vmkernel can call
 * lpfc_vport_delete within 1 - 5 seconds of vport_create's return to 
 * vmkernel based on its vport scsi scan policy.
 * 
 * This routine does not require any locks held.
 **/
static void lpfc_discovery_wait(struct lpfc_vport *vport)
{
	lpfcHBA_t *phba = vport->phba;
	uint32_t wait_flags = FC_RSCN_MODE | FC_RSCN_DISCOVERY |
			      FC_NLP_MORE | FC_NDISC_ACTIVE;
	uint32_t wait_loop = (phba->fc_ratov * 3) + 3;
	uint32_t wait_time = 0;
	uint32_t fc_map_cnt = 0;

	/*
	 * The time constraint is based on the 3x multiplier of the
	 * fabric RA_TOV value. 
	 */

        while (wait_time++ < wait_loop) {
		if ((vport->fc_plogi_cnt > 0)     ||
		    (vport->fc_adisc_cnt > 0)     ||
		    (vport->fc_map_cnt == 0)      ||
		    (vport->fc_flag & wait_flags) || 
		    (vport->port_state < LPFC_HBA_READY)) {

			lpfc_sleep_ms(phba,1000);
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1814,
                               	lpfc_mes1814,
                               	lpfc_msgBlk1814.msgPreambleStr,
				vport->vpi, vport->port_state,
				vport->fc_flag, vport->num_disc_nodes,
				vport->fc_plogi_cnt,vport->fc_adisc_cnt,
				vport->fc_map_cnt);

		} else {
			/* Base wait variants satisfied. This clause
			 * is here to wait for the mapped node count
			 * to stablize.
			 */
			if (fc_map_cnt != vport->fc_map_cnt) {
				fc_map_cnt = vport->fc_map_cnt;
				continue;
			}
			/* All wait variants satisfied. */
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1815,
				lpfc_mes1815,
				lpfc_msgBlk1815.msgPreambleStr,	
				vport->vpi, vport->port_state,
				vport->fc_map_cnt);
			return;
		}
	}

	/*
	 * Discovery timed out.  This is not an error.
	 * Log it and continue.
	 */
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1816,
		lpfc_mes1816,
		lpfc_msgBlk1816.msgPreambleStr,
		vport->vpi, vport->port_state,
		vport->fc_flag, vport->num_disc_nodes,
		vport->fc_plogi_cnt,vport->fc_adisc_cnt,
		vport->fc_map_cnt);
}


static int
lpfc_alloc_vpi(lpfcHBA_t *phba)
{
	int  vpi;
	unsigned long iflag;

	/* Start at bit 1 because vpi zero is reserved for the physical port */
	LPFC_DRVR_LOCK(phba, iflag);
	vpi = find_next_zero_bit(phba->vpi_bmask, (phba->max_vpi + 1), 1);
	if (vpi > phba->max_vpi)
		vpi = 0;
	else
		set_bit(vpi, phba->vpi_bmask);
	LPFC_DRVR_UNLOCK(phba, iflag);
	return vpi;
}


static void
lpfc_free_vpi(lpfcHBA_t *phba, int vpi)
{
	clear_bit(vpi, phba->vpi_bmask);
}

static void
lpfc_vport_stop_timers(lpfc_vport_t *vport)
{
	lpfcHBA_t *phba;
	lpfc_vport_t *tmp_vport;
	struct list_head *curr, *next;
	struct clk_data *clkData;
	struct timer_list *ptimer;

	phba = vport->phba;

	/* Make sure all timers have been stopped. */
	list_for_each_safe(curr, next, &phba->timerList) {
		clkData = list_entry(curr, struct clk_data, listLink);
		if ((clkData) && (clkData->clData3)) {
			tmp_vport = (lpfc_vport_t *) clkData->clData3;
			if ((tmp_vport) && (tmp_vport == vport)) {
				ptimer = clkData->timeObj;
				if (timer_pending(ptimer))
					lpfc_stop_timer(clkData);	
			}
		}
	}
}

int
lpfc_vport_create(struct Scsi_Host *physhost, struct vport_data *vdata)
{
	LPFC_NODELIST_t *ndlp;
	lpfc_vport_t *pport;
	lpfcHBA_t *phba;
	lpfc_vport_t *vport;
	static int instance = 100;
	unsigned long iflag;
	int rc = VPORT_ERROR;
	int vpi;
	uint32_t mrpi, arpi, mvpi, avpi;

	pport = (lpfc_vport_t*)physhost->hostdata[0];
	phba = pport->phba;
	mrpi = arpi = mvpi = avpi = 0;

	if (phba->sli_rev < 3) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1801,
				lpfc_mes1801,
				lpfc_msgBlk1801.msgPreambleStr,
				phba->sli_rev);
		return VPORT_ERROR;
	}

	vpi = lpfc_alloc_vpi(phba);
	if (vpi == 0) {
		lpfc_get_hba_info(phba, &mrpi, &arpi, &mvpi, &avpi);
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1800,
				lpfc_mes1800,
				lpfc_msgBlk1800.msgPreambleStr,
				vpi, mvpi, avpi);
		return VPORT_NORESOURCES;
	}
	
	vport = lpfc_create_port(phba, instance++);
	if (!vport) {
		lpfc_free_vpi(phba, vpi);
		return VPORT_NORESOURCES;
	}

	/*
	 * Any other failures at this point are not vport creation
	 * failures so this routine should return VPORT_OK.  Initialize
	 * vdata to success values and let the error handling adjust
	 * the attributes depending on the conditions.
	 */
	rc = VPORT_OK;
	vport->vpi = vpi;
	vport->npiv_fail_reason = VPORT_FAIL_UNKNOWN;
	vdata->vport_shost = vport->scsihost;
	memcpy(&vport->fc_sparam, &pport->fc_sparam, sizeof (SERV_PARM));
	memcpy(&vport->fc_nodename, &vdata->node_name, 8);
	memcpy(&vport->fc_portname, &vdata->port_name, 8);
	memcpy(&vport->fc_sparam.portName, &vport->fc_portname, 8);
	memcpy(&vport->fc_sparam.nodeName, &vport->fc_nodename, 8);
#ifdef __VMKERNEL_MODULE__
	fc_host_node_name(vport->scsihost) = wwn_to_u64((uint8_t *)&vport->fc_nodename);
	fc_host_port_name(vport->scsihost) = wwn_to_u64((uint8_t *)&vport->fc_portname);
#endif

	/*
	 * Did vmkernel pass a VM name string?  If so, set the driver'
	 * symbolic node name string to include this information.
	 */
	if (strlen(&vdata->symname[0]) > 0)
		strcpy(vport->vmname, vdata->symname);

	vport->port_state = LPFC_FDISC;
	vport->port_type = LPFC_VIRTUAL_PORT;

	LPFC_DRVR_LOCK(phba, iflag);
	lpfc_bind_setup(vport);
	ndlp = lpfc_findnode_did(phba->pport, NLP_SEARCH_UNMAPPED, Fabric_DID);
	if (!ndlp) {
		vport->port_state = LPFC_LINK_DOWN;
		vport->npiv_fail_reason = VPORT_FAIL_LINKDOWN;
		LPFC_DRVR_UNLOCK(phba, iflag);
		goto out;
	}

	lpfc_issue_els_fdisc(vport, ndlp, 0);
	
	LPFC_DRVR_UNLOCK(phba, iflag);

	/* Wait for vport node discovery to quiesce */
	lpfc_discovery_wait(vport);


 out:
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1808, lpfc_mes1808,
			lpfc_msgBlk1808.msgPreambleStr,
			vport->vpi, vport->fc_myDID, vport->port_type,
			vport->port_state);
	return rc;
}
int
lpfc_vport_delete(struct Scsi_Host *shost)
{
	LPFC_NODELIST_t *ndlp;
	lpfc_vport_t *vport;
	lpfcHBA_t *phba;
	unsigned long iflag;
	int rc = VPORT_OK;
	uint32_t loopcnt;

	vport = (lpfc_vport_t*)shost->hostdata[0];
	phba = vport->phba;

	if (vport->port_type == LPFC_PHYSICAL_PORT) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1809, lpfc_mes1809,
				lpfc_msgBlk1809.msgPreambleStr,
				vport->fc_myDID);
		return VPORT_ERROR;
	}

	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1811, lpfc_mes1811,
			lpfc_msgBlk1811.msgPreambleStr, vport->fc_myDID,
			vport->vpi, vport->port_state, vport->fc_plogi_cnt,
			vport->fc_adisc_cnt, vport->fc_unmap_cnt,
			vport->fc_map_cnt);

	LPFC_DRVR_LOCK(phba, iflag);

	if (vport->port_state == LPFC_HBA_ERROR && !vport->fc_myDID)
		goto cleanup_vport; 
	
	ndlp = lpfc_findnode_did(phba->pport, NLP_SEARCH_ALL, Fabric_DID);
	if (!ndlp)
		goto cleanup_vport; 

	/* Logout of the fabric port now that all vport resourcs are released. */
	if (phba->pport->port_state < LPFC_LINK_UP)
		goto cleanup_vport; 

	vport->vport_logo_state = LPFC_VPORT_LOGO_ISSUED;
	if (lpfc_issue_els_fabric_logo(vport, ndlp)) {
		rc = VPORT_ERROR;
		goto cleanup_vport;
	}

	set_current_state(TASK_INTERRUPTIBLE);
	LPFC_DRVR_UNLOCK(phba, iflag);
        /*
         * The time constraint is based on the 3x multiplier of the
         * fabric RA_TOV value.
         */
	loopcnt = (phba->fc_ratov * 3) + 3;
	while (vport->vport_logo_state != LPFC_VPORT_LOGO_CMPL) {
		lpfc_sleep_ms(phba,1000);
		if (--loopcnt == 0) {
			lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1812,
				lpfc_mes1812,
				lpfc_msgBlk1812.msgPreambleStr,
				vport->fc_myDID);
			break;
		}
	}

	LPFC_DRVR_LOCK(phba, iflag);

 cleanup_vport:
	/* Kill all timers running on this vport. */
	lpfc_vport_stop_timers(vport);

	LPFC_DRVR_UNLOCK(phba, iflag);

	/*	
	 * Clean up the rscn list, the delay io list, and abort all IOCBs
	 * on all rings.  Unregister the scsi host, cleanup any other scsi
	 * resources and delete the vport.
	 */
	lpfc_free_vpi(phba, vport->vpi);
	lpfc_destroy_port(vport);

	return rc;
}

int
lpfc_vport_getinfo(struct Scsi_Host *shost, struct vport_info *vinfo)
{
	int ret = VPORT_OK;
	lpfc_vport_t *vport = NULL;
	lpfcHBA_t *phba = NULL;
	uint32_t mrpi, arpi, mvpi, avpi;

	vport = (lpfc_vport_t*)shost->hostdata[0];
	phba = vport->phba;

	memset(vinfo, 0, sizeof (struct vport_info));
	vinfo->api_versions = VPORT_API_VERSION;

	/*
	 * Only support getinfo requests on when the hba port and the
	 * switch port support NPIV.  In all other cases, fail the get
	 * request and update the fail reason.
	 */
	if (phba->sli_rev < 3) {
		vinfo->vports_max = VPORT_CNT_INVALID;
		vinfo->vports_inuse = VPORT_CNT_INVALID;
		vinfo->state = VPORT_STATE_OFFLINE;
		vinfo->linktype = VPORT_TYPE_PHYSICAL;
		vinfo->prev_fail_reason = vport->npiv_fail_reason;
		vport->npiv_fail_reason = VPORT_FAIL_ADAP_NORESOURCES;
		vinfo->fail_reason = VPORT_FAIL_ADAP_NORESOURCES;
		goto out;
	}

	if (phba->npiv_capable == LPFC_NPIV_UNSUPPORTED_FAB) {
		mrpi = arpi = mvpi = avpi = 0;
		lpfc_get_hba_info(phba, &mrpi, &arpi, &mvpi, &avpi);
		vinfo->vports_max = phba->max_vpi;
		vinfo->vports_inuse = mvpi - avpi;
		vinfo->state = VPORT_STATE_OFFLINE;
		vinfo->linktype = VPORT_TYPE_PHYSICAL;
		vinfo->prev_fail_reason = vport->npiv_fail_reason;
		vport->npiv_fail_reason = VPORT_FAIL_FAB_UNSUPPORTED;
		vinfo->fail_reason = VPORT_FAIL_FAB_UNSUPPORTED;
		goto out;
	}

	vinfo->linktype = VPORT_TYPE_VIRTUAL;
	vinfo->prev_fail_reason = vport->npiv_fail_reason;
	vport->npiv_fail_reason = VPORT_FAIL_UNKNOWN;
	vinfo->fail_reason = VPORT_FAIL_UNKNOWN;
	if (vport->port_type == LPFC_PHYSICAL_PORT) {
		mrpi = arpi = mvpi = avpi = 0;
		lpfc_get_hba_info(phba, &mrpi, &arpi, &mvpi, &avpi);
		vinfo->vports_max = phba->max_vpi;
		vinfo->vports_inuse = mvpi - avpi;
		vinfo->rpi_max = mrpi;
		vinfo->rpi_inuse = mrpi - arpi;

		vinfo->linktype = VPORT_TYPE_PHYSICAL;
	} else {
		vinfo->vports_max = VPORT_CNT_INVALID;
		vinfo->vports_inuse = VPORT_CNT_INVALID;
	}

	switch (vport->port_state) {
	case LPFC_HBA_ERROR:
		vinfo->state = VPORT_STATE_FAILED;
		break;
	case LPFC_LINK_DOWN:
		vinfo->state = VPORT_STATE_OFFLINE;
		break;
	default:
		vinfo->state = VPORT_STATE_ACTIVE;
		break;
	}

	memcpy(&vinfo->node_name, &vport->fc_nodename, sizeof (NAME_TYPE));
	memcpy(&vinfo->port_name, &vport->fc_portname, sizeof (NAME_TYPE));

 out:
	lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1810, lpfc_mes1810,
			lpfc_msgBlk1810.msgPreambleStr, vport->vpi,
			vport->fc_myDID, vinfo->fail_reason,
			vinfo->prev_fail_reason, vinfo->vports_max,
			vinfo->vports_inuse, vinfo->linktype, vinfo->state);

	return ret;
}


int
lpfc_vport_tgt_remove(struct Scsi_Host *shost,
			unsigned int channel,
			unsigned int tgtid)
{
	lpfc_vport_t *vport = (lpfc_vport_t*) shost->hostdata[0];
	lpfcHBA_t *phba = vport->phba;
	struct lpfc_nodelist *ndlp;
	unsigned long iflag;
	int rc = VPORT_OK;

	LPFC_DRVR_LOCK(phba, iflag);
	ndlp = lpfc_findnode_scsiid(vport, tgtid);
	if (!ndlp) {
		lpfc_printf_log(phba->brd_no, &lpfc_msgBlk1813, lpfc_mes1813,
				lpfc_msgBlk1813.msgPreambleStr, vport->vpi,
				vport->fc_myDID, tgtid);
		rc = VPORT_INVAL;
		goto tgt_id_error;
	}

	ndlp->nlp_flag |= NLP_TARGET_REMOVE;
	lpfc_issue_els_logo(vport, ndlp, 0);

 tgt_id_error:
	LPFC_DRVR_UNLOCK(phba, iflag);
	return rc;
}
